/******************************* battery.c *******************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2006
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 *
 * Description: Implements functions that make is easier to support
 *              Battery Operated Nodes
 *
 * Author:   Jonas Roum-Mller
 *
 * Last Changed By:  $Author: efh $
 * Revision:         $Revision: 21390 $
 * Last Changed:     $Date: 2011-09-27 16:27:19 +0200 (Tue, 27 Sep 2011) $
 *
 ****************************************************************************/

/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
/* Enhanced Slave - needed for battery operation (RTC timer) on 100 series */
/* 200 Series have WUT */
#ifdef ZW_SLAVE_32
  #include <ZW_slave_32_api.h>
#else
  #ifdef  ZW_SLAVE
    #include <ZW_slave_api.h>
  #endif
#endif

/* ASIC power management functionality */
#if defined (ZW020x) || defined(ZW030x)
  #include <ZW_power_api.h>
#endif

#include <ZW_sysdefs.h>

/* Allows data storage of application data even after reset */
#if defined (ZW020x) || defined(ZW030x)
  #include <ZW_non_zero.h>
#endif

#include <eeprom.h>
#include <ZW_uart_api.h>
#include <ZW_pindefs.h>
#include <ZW_evaldefs.h>

#include <self_heal.h>
#include <slave_learn.h>
#include <battery.h>

#include <ZW_TransportLayer.h>

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/
BYTE sleepSecondsMSB = 0;
BYTE sleepSecondsM = 0;
BYTE sleepSecondsLSB = 0;


/****************************************************************************/
/*                              EXPORTED DATA                               */
/****************************************************************************/
/* Wakeup timeout - number of seconds before wakeup */
XDWORD sleepPeriod = 0;
/* Keep alive time between power downs */
BYTE powerDownTimeout;
BYTE powerDownTimerHandle = 0xFF;

/****************************************************************************/
/*                            PRIVATE FUNCTIONS                             */
/****************************************************************************/

/*============================   WakeupNotificationCallback   ======================
**    Callback function for sending wakeup notification
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
WakeupNotificationCallback( BYTE txStatus ) /*IN   Transmission result        */
{
  /* We did not get in contact with the Wakeup Node, dont expect no more information frame */
  if(txStatus != TRANSMIT_COMPLETE_OK || masterNodeID == 0xFF)
  {
    ZW_DEBUG_SEND_BYTE(' ');
    ZW_DEBUG_SEND_BYTE('A');
    ZW_DEBUG_SEND_BYTE('F');
    currentState = STATE_APPL_IDLE;
  }
  if(masterNodeID != 0xFF)
  {
#ifdef ZW_SELF_HEAL
    UpdateLostCounter(txStatus);
#endif
  }

  ZW_DEBUG_SEND_BYTE('A');
  ZW_DEBUG_SEND_BYTE('7');

}


/*============================   WakeupNotification   ======================
**    Function sends off the Wakeup notification command
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
WakeupNotification( void )
{
  currentState    = STATE_WAKEUP_NOTIFICATION; /* Prevent sleeping */
  /* Only send wakeup notifiers when sensor is node in a network */
  /* and a recovery operation is not in progress */
  ZW_DEBUG_SEND_BYTE('W');
  ZW_DEBUG_SEND_BYTE('n');
#ifdef ZW_SELF_HEAL
  if ((myNodeID != 0) && (currentHealMode == HEAL_NONE))
#else
  if ((myNodeID != 0))
#endif
  {
    ZW_DEBUG_SEND_BYTE('A');
    ZW_DEBUG_SEND_BYTE('6');
    txBuf.ZW_WakeUpNotificationFrame.cmdClass = COMMAND_CLASS_WAKE_UP;
    txBuf.ZW_WakeUpNotificationFrame.cmd = WAKE_UP_NOTIFICATION;

	if (!Transport_SendRequest((masterNodeID == 0xFF ? NODE_BROADCAST : masterNodeID),
						(BYTE *)&txBuf, sizeof(ZW_WAKE_UP_NOTIFICATION_FRAME),
        				(masterNodeID != 0xFF ? (TRANSMIT_OPTION_RETURN_ROUTE|TRANSMIT_OPTION_ACK|TRANSMIT_OPTION_EXPLORE) : 0),
		 				WakeupNotificationCallback,
						(masterNodeID == 0xFF ? TRUE : FALSE)))
      {
        ZW_DEBUG_SEND_BYTE('W');
        ZW_DEBUG_SEND_BYTE('f');
        currentState = STATE_APPL_IDLE;
        WakeupNotificationCallback(TRANSMIT_COMPLETE_FAIL);
      }
      SetSleepPeriod();
  }
  else
  {
    /* We are not in any network, go idle */
    currentState = STATE_APPL_IDLE;
    SetSleepPeriod();
  }
}


/*============================   PowerDownTimeoutFunction   ======================
**    When this function is called, it's time to power down
**    the sensor.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
PowerDownTimeoutFunction( void )
{
  if ((learnState == FALSE) && (currentState == STATE_APPL_IDLE || currentState == STATE_CHECK_INPUT))
  {
    /* The timer is activated once every second */
    /* When PowerDownTimeout reaches 0, it's time */
    /* to power down. */
    ZW_DEBUG_SEND_NUM(powerDownTimeout);
    if (powerDownTimeout != 0 && --powerDownTimeout == 0)
    {
      keepAliveActive = FALSE;

      ZW_SetWutTimeout(15);
      ZW_SetSleepMode(ZW_WUT_MODE,ZW_INT_MASK_EXT1,0);
    }
  }
}


/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/

/*============================   SetSleepPeriod   ======================
**    This function converts the 24-bit timeout value to a
**    big endian DWORD value that determines how long the sensor sleeps.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
SetSleepPeriod( void )
{
  ZW_DEBUG_SEND_BYTE('A');
  ZW_DEBUG_SEND_BYTE('9');

#if defined (ZW020x) || defined(ZW030x)

  ((BYTE_P)&sleepPeriod)[0] = 0;
  ((BYTE_P)&sleepPeriod)[1] = sleepSecondsMSB;
  ((BYTE_P)&sleepPeriod)[2] = sleepSecondsM;
  ((BYTE_P)&sleepPeriod)[3] = sleepSecondsLSB;


#endif /* ZW020x */
}



/*============================   StartPowerDownTimer   ======================
**    This function manages the startup of the timer that, when it
**    expires, turns off the sensor. Timer is only started if it is
**    not already running.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
StartPowerDownTimer( void )
{
  StopPowerDownTimer();

  if (powerDownTimerHandle == 0xFF)
  {
    powerDownTimerHandle = ZW_TIMER_START(PowerDownTimeoutFunction,
                           TIMER_ONE_SECOND,
                           TIMER_FOREVER);
  }
}


/*============================   StopPowerDownTimer   ======================
**    This function stops the power down timer if it's running.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
StopPowerDownTimer( void )
{
  if (powerDownTimerHandle != 0xFF)
  {
    if (ZW_TIMER_CANCEL(powerDownTimerHandle) == TRUE)
    {
      powerDownTimerHandle = 0xFF;
    }
  }
}

/*============================   SetDefaultBatteryConfiguration   ======================
**    Function resets configuration to default values.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
SetDefaultBatteryConfiguration( void )
{

  powerDownTimeout = DEFAULT_POWERDOWNTIMEOUT;
  masterNodeID = DEFAULT_MASTER_NODEID;

  sleepSecondsMSB = DEFAULT_SLEEP_SECONDS_MSB;
  sleepSecondsM = DEFAULT_SLEEP_SECONDS_M;
  sleepSecondsLSB = DEFAULT_SLEEP_SECONDS_LSB;
  sleepPeriod = 0;

  SetSleepPeriod();

}


/*============================   SaveBatteryConfiguration   ======================
**    This function saves the current configuration to EEPROM
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
SaveBatteryConfiguration( void )
{
  ZW_MEM_PUT_BYTE(EEOFFSET_MASTER_NODEID, masterNodeID);
  ZW_MEM_PUT_BYTE(EEOFFSET_POWERDOWNTIMEOUT, powerDownTimeout);
  ZW_MEM_PUT_BYTE(EEOFFSET_SLEEP_PERIOD_1, sleepSecondsMSB);
  ZW_MEM_PUT_BYTE(EEOFFSET_SLEEP_PERIOD_2, sleepSecondsM);
  ZW_MEM_PUT_BYTE(EEOFFSET_SLEEP_PERIOD_3, sleepSecondsLSB);
}


/*============================   LoadConfiguration   ======================
**    This function loads the application settings from EEPROM.
**    If no settings are found, default values are used and saved.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /* RET  Nothing      */
LoadBatteryConfiguration( void )
{
    masterNodeID = ZW_MEM_GET_BYTE(EEOFFSET_MASTER_NODEID);
    powerDownTimeout = ZW_MEM_GET_BYTE(EEOFFSET_POWERDOWNTIMEOUT);
    sleepSecondsMSB = ZW_MEM_GET_BYTE(EEOFFSET_SLEEP_PERIOD_1);
    sleepSecondsM = ZW_MEM_GET_BYTE(EEOFFSET_SLEEP_PERIOD_2);
    sleepSecondsLSB = ZW_MEM_GET_BYTE(EEOFFSET_SLEEP_PERIOD_3);
}


void
WakeupCallback(BYTE txStatus)
{
  if(txStatus != TRANSMIT_COMPLETE_OK){
   currentState = STATE_APPL_IDLE;
  } else {
   currentState = STATE_WAKEUP_NOTIFICATION;
  }


}

/*============================   HandleWakeupFrame   ======================
**
**--------------------------------------------------------------------------*/
void                   /* RET  Nothing      */
HandleWakeupFrame(ZW_APPLICATION_TX_BUFFER *pCmd, BYTE txOption, BYTE sourceNode)
{
  BYTE param1 = ((BYTE_P)pCmd)[OFFSET_PARAM_1];
  BYTE param2 = ((BYTE_P)pCmd)[OFFSET_PARAM_2];
  BYTE param3 = ((BYTE_P)pCmd)[OFFSET_PARAM_3];
  BYTE param4 = ((BYTE_P)pCmd)[OFFSET_PARAM_4];
  BYTE param5 = ((BYTE_P)pCmd)[OFFSET_PARAM_5];
  BYTE param6 = ((BYTE_P)pCmd)[OFFSET_PARAM_6];

  if (pCmd->ZW_Common.cmd == WAKE_UP_INTERVAL_SET)
  {
    ZW_DEBUG_SEND_BYTE('W');
    ZW_DEBUG_SEND_BYTE('0');
    sleepSecondsMSB = param1;
    sleepSecondsM = param2;
    sleepSecondsLSB = param3;
    masterNodeID = param4;
    SetSleepPeriod();
    SaveBatteryConfiguration();
  }
  else if(pCmd->ZW_Common.cmd == WAKE_UP_INTERVAL_GET)
  {
    ZW_DEBUG_SEND_BYTE('W');
    ZW_DEBUG_SEND_BYTE('1');
    txBuf.ZW_WakeUpIntervalReportFrame.cmdClass = pCmd->ZW_Common.cmdClass;
    txBuf.ZW_WakeUpIntervalReportFrame.cmd = WAKE_UP_INTERVAL_REPORT;
    txBuf.ZW_WakeUpIntervalReportFrame.seconds1 = sleepSecondsMSB;
    txBuf.ZW_WakeUpIntervalReportFrame.seconds2 = sleepSecondsM;
    txBuf.ZW_WakeUpIntervalReportFrame.seconds3 = sleepSecondsLSB;
    txBuf.ZW_WakeUpIntervalReportFrame.nodeid = masterNodeID;

	Transport_SendReport(sourceNode,
						(BYTE *)&txBuf, sizeof(ZW_WAKE_UP_INTERVAL_REPORT_FRAME),
        				(TRANSMIT_OPTION_RETURN_ROUTE | TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE),
		 				WakeupCallback, FALSE);


//    ZW_SendData(sourceNode, (BYTE *)&txBuf, sizeof(ZW_WAKE_UP_INTERVAL_REPORT_FRAME),
//                (TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE), WakeupCallback);
  }
  else if(pCmd->ZW_Common.cmd == WAKE_UP_NO_MORE_INFORMATION)
  {
    ZW_DEBUG_SEND_BYTE('W');
    ZW_DEBUG_SEND_BYTE('2');
    /* Delay powerdown just a little while to ensure "End Of Communication" */
    powerDownTimeout = DEFAULT_POWERDOWNTIMEOUT;
    currentState = STATE_APPL_IDLE;
  }
  else
  {
    ZW_DEBUG_SEND_BYTE('W');
    ZW_DEBUG_SEND_BYTE('3');
  }
}

