/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/

#include <config_app.h>

/* Enhanced Slave - needed for battery operation (RTC timer) on 100 series */
/* 200 Series have WUT */
#ifdef ZW_SLAVE_32
  #include <ZW_slave_32_api.h>
#else
  #ifdef  ZW_SLAVE
    #include <ZW_slave_api.h>
  #endif
#endif

#ifdef ZW_SELF_HEAL
  /* Lost functionality */
  #include <self_heal.h>
#endif

  /* ASIC power management functionality */
#if defined(ZW020x) || defined(ZW030x)
  #include <ZW_power_api.h>
#endif
/* Support functions for battery operation */
#include "mybattery.h"

/* Z-Wave libraries */
#include <ZW_sysdefs.h>
#include <ZW_pindefs.h>
#include <ZW_classcmd.h>
#include <ZW_uart_api.h>
#include <ZW_non_zero.h>

#include <myStromMeter.h>
#include <eeprom.h>
#include <slave_learn.h>

/* Support functions for association */
#include <association.h>
/* Support functions for button press detection */
#include <one_button.h>

#include <ZW_TransportLayer.h>

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/

#define DEFAULT_CVALUE 0

#define LED_STATUS_ON() PIN_ON(TRIACpin)
#define LED_STATUS_OFF() PIN_OFF(TRIACpin)


#define BUTTON_KEEPALIVECOUNT     0xFE  /* x 10msec 2seconds for KeepAlive */

BYTE wakeUpReason;
BYTE currentGroupIndex;
BYTE currentGroupNodeIndex;
BYTE stateWakeUPTimerHandle = 0xFF;

BYTE stateWatchdogTimerHandle = 0xFF;
BYTE stateWatchdogTimeout = WATCHDOG_TIMEOUT;
BYTE quarterSecondTimerHandle = 0xFF;

/* Function Prototypes */
BOOL AssociationSendMeterReport(BYTE broadcast);
void SendCompleted(BYTE txStatus);
void AssociationSendToGroup(void);

BYTE myNodeID = 0;
BYTE myHomeID[4];
BYTE currentState    = STATE_APPL_IDLE;
BYTE nextState       = STATE_APPL_IDLE;
BYTE watchState      = STATE_APPL_IDLE;

#define METER_MODE_ANALOG  1
#define METER_MODE_DIGITAL 2
BYTE  meterMode = 0;
WORD  cValue = DEFAULT_CVALUE;
WORD  currentSeconds = 0;

// TODO: Store in non-zero SRAM
DWORD currentMeterPulses = 0;
DWORD previousMeterPulses = 0; 
WORD  currentMeterDuration = 0;

BYTE uartLastCmd = 0;
BYTE uartToRecv = 0;
BYTE uartRecvBuffer[8];
BYTE uartRecvBufferPos = 0;

BYTE currentGroupSize;

BOOL keepAliveActive = FALSE;

/* MasterNodeID specifies the node the sensor will try to contact on wakeup */
/* and when lost. Defined extern, currenly used by "battery" and "self-heal"*/
BYTE masterNodeID = 0xFF;

/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/

/* A list of the known command classes. Except the basic class which allways */
/* should be supported. Used when node info is send */
/* TO#1938 fix - COMMAND_CLASS_BASIC need never be mentioned here, as it is mandatory */
static code t_nodeInfo nodeInfo = {
                       COMMAND_CLASS_METER,
                       COMMAND_CLASS_MANUFACTURER_SPECIFIC,
                       COMMAND_CLASS_VERSION,
                       COMMAND_CLASS_ASSOCIATION,
                       COMMAND_CLASS_CONFIGURATION,
                       COMMAND_CLASS_WAKE_UP,
#ifdef SECURITY
  COMMAND_CLASS_SECURITY,
#endif
};

#ifdef SECURITY
extern BYTE nodeSecure;
extern BYTE secureCmdHandling;

#define TSEC_CMD_HANDLING_SECURE    1

static code BYTE nodeInfoAfterIncluded[] = {
  COMMAND_CLASS_VERSION,
  COMMAND_CLASS_MANUFACTURER_SPECIFIC,
  COMMAND_CLASS_SECURITY
};

static code BYTE nodeInfoForTransport[] = {
  COMMAND_CLASS_METER,
  COMMAND_CLASS_MANUFACTURER_SPECIFIC,
  COMMAND_CLASS_VERSION,
  COMMAND_CLASS_ASSOCIATION,
  COMMAND_CLASS_CONFIGURATION,
  COMMAND_CLASS_WAKE_UP,
};
#else
#define nodeInfoForTransport nodeInfo
#define nodeInfoAfterIncluded nodeInfo
#endif

/****************************************************************************/
/*                              EXPORTED DATA                               */
/****************************************************************************/

ZW_APPLICATION_TX_BUFFER txBuf;
BYTE frameSize;

/****************************************************************************/
/*                              IMPORTED DATA                               */
/****************************************************************************/



/****************************************************************************/
/*                            PRIVATE FUNCTIONS                             */
/****************************************************************************/

BOOL meterModeSet(BYTE t);
void sendCmdToPIC(BYTE cmd, BYTE length);


void                   /*RET  Nothing       */
CheckWakeupCount( void )
{
  sendCmdToPIC('R', 7);
}

void
UpdateWakeupCount( DWORD secondsCount ) {
  if (secondsCount >= sleepPeriod) {
    currentState = STATE_WAKEUP_NOTIFICATION_START;
    sendCmdToPIC('C', 1);
  }
}

void
powerDownNow() {
  // Send request to read to Z-Wave module
  //sendCmdToPIC('Q', 0);
  /* Sleep for 5 seconds */
  ZW_SetWutTimeout(5);
  ZW_SetSleepMode(ZW_WUT_MODE,ZW_INT_MASK_EXT1,0);
}


/*============================   WatchdogCallback  ==============================
**    Check to see if the current state have run longer than WATCHDOG_TIMEOUT
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
WakeUPCallbackStart(void)
{
  powerDownTimeout = DEFAULT_POWERDOWNTIMEOUT;

  currentState = STATE_BASIC_CMD_START;

  StartPowerDownTimer();
}

/*============================   StartWakeUP  ================================
**    WakeUP timer
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
StartWakeUP(void)
{
  if(stateWakeUPTimerHandle != 0xFF)
  {
    ZW_TIMER_CANCEL(stateWakeUPTimerHandle);
  }
  stateWakeUPTimerHandle = ZW_TIMER_START(WakeUPCallbackStart,
                                            TIMER_ONE_SECOND/2,
                                            TIMER_ONE_TIME);
}

/*============================   WatchdogCallback  ==============================
**    Check to see if the current state have run longer than WATCHDOG_TIMEOUT
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
WatchdogCallback( void )
{
  stateWatchdogTimeout--;
  if(stateWatchdogTimeout <= 0)
  {
    stateWatchdogTimeout = WATCHDOG_TIMEOUT;
    // State not changed
    if(watchState == currentState)
    {
      // Force into IDLE state
      currentState = STATE_APPL_IDLE;
    }
  }
}

/*============================   StartWatchdog  ==============================
**    WatchdogTimer, make sure no state runs forever
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
StartWatchdog( void )
{
  watchState = currentState;
  if(stateWatchdogTimerHandle != 0xFF)
  {
    ZW_TIMER_CANCEL(stateWatchdogTimerHandle);
  }
  stateWatchdogTimerHandle = ZW_TIMER_START(WatchdogCallback,
                                            TIMER_ONE_SECOND,
                                            TIMER_FOREVER);
}

/*============================   StopWatchdog  ==============================
**    WatchdogTimer, stop the timer
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
StopWatchdog( void )
{
  watchState = STATE_APPL_IDLE;
  if(stateWatchdogTimerHandle != 0xFF)
  {
    ZW_TIMER_CANCEL(stateWatchdogTimerHandle);
    stateWatchdogTimerHandle = 0xFF;
  }
}

BOOL blinkingLedOn = 0;
void                   /*RET  Nothing       */
QuarterSecondCallback( void )
{
  if (myNodeID == 0) {
//  if (currentState == STATE_LEARN_MODE) {
    if (blinkingLedOn) {
      LED_STATUS_OFF();
      blinkingLedOn = 0;
    } else {
      LED_STATUS_ON();
      blinkingLedOn = 1;
    }
  } else {
    LED_STATUS_OFF();
  }
}

/*============================   StartQuarterSecondTimer  ====================
**    WakeUP timer
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
StartQuarterSecondTimer(void)
{
  if(quarterSecondTimerHandle != 0xFF)
  {
    ZW_TIMER_CANCEL(quarterSecondTimerHandle);
  }
  quarterSecondTimerHandle = ZW_TIMER_START(QuarterSecondCallback,
                                            TIMER_ONE_SECOND/4,
                                            TIMER_FOREVER);
}

/*============================   SaveConfiguration   ======================
**    This function saves the current configuration to EEPROM
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
SaveConfiguration( void )
{
  SaveBatteryConfiguration();

  ZW_MEM_PUT_BYTE(EEOFFSET_METER_MODE, meterMode);
  ZW_MEM_PUT_BYTE(EEOFFSET_METER_CVALUE_1, (cValue>>8)&0xff);
  ZW_MEM_PUT_BYTE(EEOFFSET_METER_CVALUE_2, cValue&0xff);
  ZW_MEM_PUT_BYTE(EEOFFSET_LOST_COUNTER, 0);

  /* Mark stored configuration as OK */
  ZW_MEM_PUT_BYTE(EEOFFSET_MAGIC, MAGIC_VALUE);
}

/*============================   SetDefaultConfiguration   ======================
**    Function resets configuration to default values.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
SetDefaultConfiguration( void )
{
  SetDefaultBatteryConfiguration();
#ifdef ZW_SELF_HEAL
  SetDefaultNetworkUpdateConfiguration();
#endif
  meterModeSet(METER_MODE_ANALOG);
  cValue = DEFAULT_CVALUE;
}


/*============================   LoadConfiguration   ======================
**    This function loads the application settings from EEPROM.
**    If no settings are found, default values are used and saved.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /* RET  Nothing      */
LoadConfiguration( void )
{
  /* Get this sensors identification on the network */
  MemoryGetID(myHomeID, &myNodeID);

  /* Check to see, if any valid configuration is stored in the EEPROM */
  if (ZW_MEM_GET_BYTE(EEOFFSET_MAGIC) == MAGIC_VALUE)
  {
    /* There is a configuration stored, so load it */
    LoadBatteryConfiguration();
    meterModeSet(ZW_MEM_GET_BYTE(EEOFFSET_METER_MODE));
	cValue = ZW_MEM_GET_BYTE(EEOFFSET_METER_CVALUE_1)<<8 | 
	         ZW_MEM_GET_BYTE(EEOFFSET_METER_CVALUE_2);
  }
  else
  {
    /* Apparently there is no valid configuration in EEPROM, so load */
    /* default values and save them to EEPROM. */
    SetDefaultConfiguration();
    SaveConfiguration();
  }
}

/*=====================   HasAssociatedNode   ====================
**    Checks if any nodes have been associated with this node.
**
**    Side effects:
**
**---------------------------------------------------------------*/
BOOL                   /*RET TRUE if any nodes have been associated */
HasAssociatedNode ( void )
{
  BYTE nodeIndex;

  for (nodeIndex = 0; nodeIndex < MAX_ASSOCIATION_IN_GROUP; nodeIndex++)
  {
    if (groups[currentGroupIndex].nodeID[nodeIndex])
    {
      return TRUE;
    }
  }
  return FALSE;
}

/*=========================   PrepareMeterReport   ====================
**    Prepares the transmit buffer with data for a Meter Report frame.
**
**    Side effects:
**
**---------------------------------------------------------------*/
void                    /*RET  Nothing       */
PrepareMeterReport ( BYTE scale )
{
	WORD deltaTime = currentMeterDuration / 100;

	DWORD currentValue = currentMeterPulses;
	DWORD previousValue = previousMeterPulses;
	BYTE precision = 0;
	if (scale == 0) {
		precision = 2;
		currentValue = currentValue * 100 / cValue;
		previousValue = previousValue * 100 / cValue;
	} else if (scale == 2) {
		deltaTime = 0;
	}
	// TODO: Calculate current Watt energy usage

	if (currentValue > 0x7fff) {
		/* Send 4 byte size report */
		txBuf.ZW_MeterReport4byteV2Frame.cmdClass = COMMAND_CLASS_METER;
		txBuf.ZW_MeterReport4byteV2Frame.cmd = METER_REPORT_V2;
		txBuf.ZW_MeterReport4byteV2Frame.properties1 = 0x40 | 0x01; /* Rate Type: Import(consumed), Meter Type: Electric meter */
		txBuf.ZW_MeterReport4byteV2Frame.properties2 =
			precision << 5 /* precision */ | 
			scale << 3 /* scale */ |
			4 /* Size: 4 bytes */;
		txBuf.ZW_MeterReport4byteV2Frame.meterValue1 = (currentValue >> 24) & 0xff;
		txBuf.ZW_MeterReport4byteV2Frame.meterValue2 = (currentValue >> 16) & 0xff;
		txBuf.ZW_MeterReport4byteV2Frame.meterValue3 = (currentValue >> 8) & 0xff;
		txBuf.ZW_MeterReport4byteV2Frame.meterValue4 = currentValue & 0xff;
		txBuf.ZW_MeterReport4byteV2Frame.deltaTime1 = (deltaTime >> 8) & 0xff;
		txBuf.ZW_MeterReport4byteV2Frame.deltaTime2 = deltaTime & 0xff;
		if (scale != 2) {
			txBuf.ZW_MeterReport4byteV2Frame.previousMeterValue1 = (previousValue >> 24) & 0xff;
			txBuf.ZW_MeterReport4byteV2Frame.previousMeterValue2 = (previousValue >> 16) & 0xff;
			txBuf.ZW_MeterReport4byteV2Frame.previousMeterValue3 = (previousValue >> 8) & 0xff;
			txBuf.ZW_MeterReport4byteV2Frame.previousMeterValue4 = previousValue & 0xff;
		}
		frameSize = sizeof(ZW_METER_REPORT_4BYTE_V2_FRAME);
	} else if (currentValue > 0x7f) {
		/* Send 2 byte size report */
		txBuf.ZW_MeterReport2byteV2Frame.cmdClass = COMMAND_CLASS_METER;
		txBuf.ZW_MeterReport2byteV2Frame.cmd = METER_REPORT_V2;
		txBuf.ZW_MeterReport2byteV2Frame.properties1 = 0x40 | 0x01; /* Rate Type: Import(consumed), Meter Type: Electric meter */
		txBuf.ZW_MeterReport2byteV2Frame.properties2 =
			precision << 5 /* precision */ | 
			scale << 3 /* scale */ |
			2 /* Size: 2 bytes */;
		txBuf.ZW_MeterReport2byteV2Frame.meterValue1 = (currentValue >> 8) & 0xff;
		txBuf.ZW_MeterReport2byteV2Frame.meterValue2 = currentValue & 0xff;
		txBuf.ZW_MeterReport2byteV2Frame.deltaTime1 = (deltaTime >> 8) & 0xff;
		txBuf.ZW_MeterReport2byteV2Frame.deltaTime2 = deltaTime & 0xff;
		if (scale != 2) {
			txBuf.ZW_MeterReport2byteV2Frame.previousMeterValue1 = (previousValue >> 8) & 0xff;
			txBuf.ZW_MeterReport2byteV2Frame.previousMeterValue2 = previousValue & 0xff;
		}
		frameSize = sizeof(ZW_METER_REPORT_2BYTE_V2_FRAME);
	} else {
		/* Send 1 byte size report */
		txBuf.ZW_MeterReport1byteV2Frame.cmdClass = COMMAND_CLASS_METER;
		txBuf.ZW_MeterReport1byteV2Frame.cmd = METER_REPORT_V2;
		txBuf.ZW_MeterReport1byteV2Frame.properties1 = 0x40 | 0x01;	/* Rate Type: Import(consumed), Meter Type: Electric meter */
		txBuf.ZW_MeterReport1byteV2Frame.properties2 =
			precision << 5 /* precision */ | 
			scale << 3 /* scale */ |
			1 /* Size: 1 byte */;
		txBuf.ZW_MeterReport1byteV2Frame.meterValue1 = currentValue;
		txBuf.ZW_MeterReport1byteV2Frame.deltaTime1 = (deltaTime >> 8) & 0xff;
		txBuf.ZW_MeterReport1byteV2Frame.deltaTime2 = deltaTime & 0xff;
		if (scale != 2) {
			txBuf.ZW_MeterReport1byteV2Frame.previousMeterValue1 = previousValue;
		}
		frameSize = sizeof(ZW_METER_REPORT_1BYTE_V2_FRAME);
	}
	 
	previousMeterPulses = currentMeterPulses;
	currentMeterDuration = 0;
}

/*============================   BasicCmd   ======================
**    Send an unsolicited Meter Report command to associated nodes
**    if any, otherwise broadcast to all.
**
**    Side effects:
**
**---------------------------------------------------------------*/
void                   /*RET  Nothing       */
BasicCmd( void )
{
  currentState = STATE_BASIC_CMD;
  /* An unsolicited meter report is only sent when the node is member of a network */
  if (myNodeID != 0)
  {
  	// If C-value is known send kWh, otherwise pulse count
    PrepareMeterReport(cValue == 0?3:0);
    if (HasAssociatedNode())
    {
      AssociationSendToGroup();
    }
    else
    {
      ZW_SendData(NODE_BROADCAST, (BYTE*)&txBuf, frameSize,
        (TRANSMIT_OPTION_RETURN_ROUTE | TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE), SendCompleted);
      currentState = STATE_APPL_IDLE;
    }
  }
  else
  {
    currentState = STATE_APPL_IDLE;
  }
}



/*============================   AssociationStoreAll   ======================
**    Function description
**      Stores all groups in external Nonvolatile memory. Used by association.c
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
AssociationStoreAll( void )
{
  MemoryPutBuffer(EEOFFSET_ASSOCIATION_START, (BYTE *)&groups[0], ASSOCIATION_SIZE, NULL);
}

/*============================   AssociationClearAll   ======================
**    Function description
**      Clears the Association area in Nonvolatile memory. Used by association.c
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
AssociationClearAll( void )
{
  MemoryPutBuffer(EEOFFSET_ASSOCIATION_START, NULL, ASSOCIATION_SIZE, NULL);
  ZW_MEM_PUT_BYTE(EEOFFSET_ASSOCIATION_MAGIC, MAGIC_VALUE); /* Now ASSOCIATION should be OK */
}

/*============================   AssociationInit   ==========================
**    Function description
**      Reads the groups stored in the Nonvolatile memory
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
AssociationInit(void)
{
  if (ZW_MEM_GET_BYTE(EEOFFSET_ASSOCIATION_MAGIC) != MAGIC_VALUE)
  {
    /* Clear it */
    AssociationClearAll();
  }
  MemoryGetBuffer(EEOFFSET_ASSOCIATION_START, (BYTE *)&groups[0], ASSOCIATION_SIZE);
}

/*============================   AssociationSendNext   ======================
**    Function description
**      Callback used when sending to association command
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
AssociationSendNext(
  BYTE bStatus)
{
  VerifyAssociatedTransmit(bStatus, groups[currentGroupIndex].nodeID[currentGroupNodeIndex]);
  currentGroupNodeIndex++;
  if (!AssociationSendMeterReport(FALSE))
  {
    SendCompleted(bStatus);  /* Now done... */
  }
}


/*============================   AssociationSendMeterReport   ======================
**    Function description
**      Sends Meter Report frame to the associated node currentGroupNodeIndex (if exist)
**      in the current currentGroupIndex group.
**    Side effects:
**
**--------------------------------------------------------------------------*/
BOOL            /*RET TRUE if there may be more associated nodes to send to */
AssociationSendMeterReport(
BYTE broadcast)
{
  for (; currentGroupNodeIndex < MAX_ASSOCIATION_IN_GROUP; currentGroupNodeIndex++)
  {
    if (groups[currentGroupIndex].nodeID[currentGroupNodeIndex])
    {
      Transport_SendRequest(groups[currentGroupIndex].nodeID[currentGroupNodeIndex], (BYTE*)&txBuf, frameSize,
        (TRANSMIT_OPTION_RETURN_ROUTE | TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE), AssociationSendNext, FALSE);
      return TRUE;
    }
  }
  if (broadcast)
  {
#ifdef ZW030x
    Transport_SendRequest(NODE_BROADCAST, (BYTE*)&txBuf, frameSize, (TRANSMIT_OPTION_RETURN_ROUTE | TRANSMIT_OPTION_ACK), SendCompleted, FALSE);
#else
    Transport_SendRequest(NODE_BROADCAST, (BYTE*)&txBuf, frameSize, 0, SendCompleted, FALSE);
#endif
    return TRUE;
  }
  return FALSE;
}

/*============================   SendCompleted   ======================
**    Callback for association sends...
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                    /*RET  Nothing       */
SendCompleted(
BYTE txStatus)          /*IN   Transmission result           */
{
  currentState = STATE_WAKEUP_NOTIFICATION_START;
}

/*============================   AssociationSendToGroup   ====================
**    Function description
**      Initiate transmission of a preprepared Basic Set frame to nodes in
**      first association group.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
AssociationSendToGroup(void)
{
  /* Setup to start transmit to first group (for now we have only one group!) */
  currentGroupIndex = 0;
  /* Setup to start with first node */
  currentGroupNodeIndex = 0;
  if (!AssociationSendMeterReport(TRUE))
  {
    SendCompleted(TRANSMIT_COMPLETE_OK);  /* Now done... */
  }
}

/*============================   LearnCompleted   ========================
**    Callback which is called on learnmode completed
**  Application specific handling of LearnModeCompleted - called from
**  slave_learn.c
**--------------------------------------------------------------------------*/
void									/*RET	Nothing */
LearnCompleted(
  BYTE bNodeID)					/* IN resulting nodeID */
{
  myNodeID = bNodeID;
  if (myNodeID == 0)
  {
#ifdef ZW_SELF_HEAL
  	UpdateNetworkUpdateCount(TRUE); // Reset networkupdatecount
#endif
    SetDefaultConfiguration();
    SaveConfiguration();
    AssociationClearAll();
    AssociationInit();
  }

  currentState = STATE_APPL_IDLE;

  Transport_OnLearnCompleted(bNodeID);
}

/*============================   CheckButtonPress   ==========================
**    Check and handle button press from ApplicationPoll
**
**
**--------------------------------------------------------------------------*/
void
CheckButtonPress( void )
{
  register BYTE bStatus;

  bStatus = OneButtonLastAction();
#if 0
  if(bStatus == BUTTON_WAS_RELEASED) // Long press
  {
    // TODO Send Node Information Frame
#if 0
#ifdef ZW_SELF_HEAL
    currentHealMode = HEAL_NONE;
#endif

    currentState = STATE_WAKEUP_NOTIFICATION_START;

    if(lostCount != 0 && currentHealMode != HEAL_NONE)
    {
      HealComplete(FALSE);
      currentHealMode = HEAL_NONE;
      //powerDownTicks = COMMAND_POWERDOWNTIMEOUT*2;
      CancelRediscoveryTimer();
    }

    nextState = STATE_BASIC_CMD_START;
#endif
  }
  else 
#endif
  if (bStatus == BUTTON_WAS_PRESSED) // Short press
  {
    currentState = STATE_LEARN_MODE_START;
    nextState = STATE_KEEP_ALIVE;
    if(!keepAliveActive) /* Time to activate keepalive */
    {
      keepAliveActive = TRUE;
      powerDownTimeout = LEARN_KEEPALIVETIMEOUT;
    }
    // No button press
    if(currentState == STATE_CHECK_INPUT)
    {
      currentState = STATE_APPL_IDLE;
    }
  }

}

/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/


/*============================   ApplicationInitHW   ========================
**    Initialization of non Z-Wave module hardware
**
**    Side effects:
**       Returning FALSE from this function will force the API into
**       production test mode.
**--------------------------------------------------------------------------*/

BYTE                       /* RET TRUE        */
ApplicationInitHW(
BYTE bWakeupReason
)  /* IN  Nothing     */
{
  IBYTE i;

#ifdef APPL_PROD_TEST
#ifdef __C51__
  SET_PRODUCTIONTEST_PIN;
  for (i = 0; i < 10; i++) ;  /* Short delay... */
  if (IN_PRODUCTIONTEST) /* Anyone forcing it into productiontest mode ? */
  {
    /* Return FALSE to enter production test mode */
    return(FALSE);
  }
#endif /* __C51__ */
#endif /* APPL_PROD_TEST */

  /* hardware initialization */
  PIN_IN(Button, 1); // SW1
  for (i = 0; i < 10; i++) ;  /* Short delay... */

  PIN_OUT(TRIACpin); // LED
  PIN_IN(SSN,0); // TX_PHOTO
  PIN_IN(PWM,0); // RX_PHOTO
  PIN_IN(ZEROXpin,0); // POWER_USB

#if 0 
  if (BUTTON_PRESSED()) {
  	/* Force reset to factory settings */
	ZW_SetDefault();
	/* Mark stored configuration as not-OK */
    ZW_MEM_PUT_BYTE(EEOFFSET_MAGIC, 0x00);
  }
#endif

  LED_STATUS_ON();
 
#if defined(ZW020x) || defined(ZW030x)
  IT1 = 0; /*level triggered*/
#endif /* ZW020x */

#if defined(ZW020x) || defined(ZW030x)
  wakeUpReason = bWakeupReason;
#endif

  ZW_UART_ENABLE;

  Transport_OnApplicationInitHW(bWakeupReason);

  /* Return true to continue normal operating mode */
  return(TRUE);
}

/*===========================   ApplicationInitSW   =========================
**    Initialization of the Application Software variables and states
**    100 series: Not called on RTC wakeup
**    200 series: Called on WUT wakeup
**
**--------------------------------------------------------------------------*/
BYTE                      /*RET  TRUE       */
ApplicationInitSW( void ) /* IN   Nothing   */
{
  ZW_UART_INIT(96);

  LoadConfiguration();

  /* Initialize button detection */
  OneButtonInit();

  AssociationInit();

  Transport_OnApplicationInitSW(nodeInfoForTransport, sizeof(nodeInfoForTransport),
    FALSE, EEOFFSET_TRANSPORT_SETTINGS_START, EEOFFSET_TRANSPORT_SETTINGS_SIZE, NULL);

  if(currentState != STATE_WAKEUP_NOTIFICATION_START)
  {
    currentState = STATE_APPL_IDLE;
  }

  StartQuarterSecondTimer();

  // Send request to read to PIC
  //sendCmdToPIC('R', 7);

  powerDownTimeout = 3; // ???
  StartPowerDownTimer();

  return(TRUE);
}

/*============================   ApplicationTestPoll   ======================
**    Function description
**      This function is called when the slave enters test mode.
**
**    Side effects:
**       Code will not exit until it is reset
**--------------------------------------------------------------------------*/
void ApplicationTestPoll(void)
{
}

static void
readUARTMessage() {
	BYTE dat = ZW_UART_REC_BYTE;
	if (uartToRecv == 0) {
		return;
	}
	if (uartRecvBufferPos == 0 && dat != uartLastCmd) {
		return;
	}
	uartRecvBuffer[uartRecvBufferPos++] = dat;
	if (uartRecvBufferPos < uartToRecv) {
		return;
	}

	switch (uartRecvBuffer[0]) {
		case 'R': 
		{
			currentMeterPulses += uartRecvBuffer[1]<<8 + uartRecvBuffer[2];
			currentMeterDuration += uartRecvBuffer[3]<<8 + uartRecvBuffer[4];
			currentSeconds += uartRecvBuffer[5]<<8 + uartRecvBuffer[6];

			// Send report to associated nodes or as broadcast
			currentState = STATE_BASIC_CMD_START;
		} break;
		case 'C':
		{
			UpdateWakeupCount(uartRecvBuffer[1]<<8 + uartRecvBuffer[2]);
		} break;
	}
	if (currentState == STATE_PIC_COMM_BUSY) {
		currentState = STATE_APPL_IDLE;
	}
}

/*=============================  ApplicationPoll   =========================
**    Application poll function
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                    /*RET  Nothing                  */
ApplicationPoll( void ) /* IN  Nothing                  */
{
  if (ZW_UART_REC_STATUS == TRUE) {
  	/* Byte to read from UART */
	readUARTMessage();
  }

  switch (currentState)
  {

    case STATE_APPL_IDLE:
    {
      /* Stop the watchdog and set the watchState to APPL_IDLE */
      StopWatchdog();
      /* If something else is in queue do that */
      if(nextState != STATE_APPL_IDLE)
      {
        currentState = nextState;
        nextState = STATE_APPL_IDLE;
      } else { /* When theres nothing else to do, check button and sensor */
        currentState = STATE_CHECK_INPUT;
      }
      break;
    }
    case STATE_CHECK_INPUT:
    {
      CheckButtonPress();
      break;
    }
    case STATE_EXECUTE:
    {
      /* Do nothing but check button press - we are waiting for an operation to finish */
      CheckButtonPress();
      break;
    }
	case STATE_PIC_COMM_BUSY:
	{
      if (ZW_UART_REC_STATUS == TRUE) {
  	    /* Byte to read from UART */
	    readUARTMessage();
      }
	  break;
	}
    case STATE_COMMAND_HANDLER:
    {
      /* Do nothing  */
      break;
    }
    case STATE_HEAL_LOST:
    {
     // Do nothing, we are currently searching for a SUC node. - Accept button input
     currentState = STATE_CHECK_INPUT;
     nextState = STATE_HEAL_LOST;
     break;
    }
    case STATE_WAKEUP_NOTIFICATION_START:
    {
      currentState = STATE_WAKEUP_NOTIFICATION;

      WakeupNotification();
      StartWatchdog();
      break;
    }
    case STATE_WAKEUP_NOTIFICATION:
    {
      // Do nothing, wait for the controller to finish talking to us
      break;
    }
    case STATE_BASIC_CMD_START:
    {

      BasicCmd();
      /* Make sure we don't get stuck */
      StartWatchdog();
      break;
    }
    case STATE_BASIC_CMD:
    {
      // Do nothing but wait for basic set/report to complete
      break;
    }
    case STATE_LEARN_MODE:
      if (OneButtonLastAction()==BUTTON_WAS_PRESSED)
      {
        /* Stop learn mode */
        StartLearnModeNow(FALSE);
        currentState = STATE_APPL_IDLE;
      }
      break;

    case STATE_LEARN_MODE_START:
    {
      currentState = STATE_LEARN_MODE;
      StartWatchdog();
      StartLearnModeNow(ZW_SET_LEARN_MODE_CLASSIC);

	  StartWakeUP();


      break;
    }
    default:
    {
      /* BTW: Something is terribly wrong if 'default' is ever executed */
      currentState = STATE_APPL_IDLE;
      break;
    }

  }
}

/*========================   ApplicationCommandHandler   ====================
**    Handling of a received application commands and requests
**
**
**--------------------------------------------------------------------------*/
static void
handleCmdClassVersion(BYTE sourceNode, ZW_APPLICATION_TX_BUFFER *pCmd, BYTE cmdLength, BYTE txOption) {
    if (pCmd->ZW_Common.cmd == VERSION_GET)
    {
      txBuf.ZW_VersionReportFrame.cmdClass = COMMAND_CLASS_VERSION;
      txBuf.ZW_VersionReportFrame.cmd = VERSION_REPORT;
      txBuf.ZW_VersionReportFrame.zWaveLibraryType = ZW_TYPE_LIBRARY();
      txBuf.ZW_VersionReportFrame.zWaveProtocolVersion = ZW_VERSION_MAJOR;
      txBuf.ZW_VersionReportFrame.zWaveProtocolSubVersion = ZW_VERSION_MINOR;
      txBuf.ZW_VersionReportFrame.applicationVersion = APP_VERSION;
      txBuf.ZW_VersionReportFrame.applicationSubVersion = APP_REVISION;
      Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_VersionReportFrame),
                 txOption | TRANSMIT_OPTION_EXPLORE, NULL, FALSE);
    }
    else if (pCmd->ZW_Common.cmd == VERSION_COMMAND_CLASS_GET)
    {
      txBuf.ZW_VersionCommandClassReportFrame.cmdClass = COMMAND_CLASS_VERSION;
      txBuf.ZW_VersionCommandClassReportFrame.cmd = VERSION_COMMAND_CLASS_REPORT;
      txBuf.ZW_VersionCommandClassReportFrame.requestedCommandClass = *((BYTE*)pCmd + OFFSET_PARAM_1);
		switch(txBuf.ZW_VersionCommandClassReportFrame.requestedCommandClass) {
		case COMMAND_CLASS_BASIC:
        txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = BASIC_VERSION;
		  break;
      case COMMAND_CLASS_METER_V2:
        txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = METER_VERSION_V2;
		  break;
      case COMMAND_CLASS_CONFIGURATION:
        txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = CONFIGURATION_VERSION;
		  break;
      case COMMAND_CLASS_WAKE_UP:
        txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = WAKE_UP_VERSION;
		  break;
      case COMMAND_CLASS_MANUFACTURER_SPECIFIC:
        txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = MANUFACTURER_SPECIFIC_VERSION;
		  break;
      case COMMAND_CLASS_VERSION:
        txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = VERSION_VERSION;
		  break;
      default:
        txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = UNKNOWN_VERSION;
      }
      Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_VersionCommandClassReportFrame),
                  txOption | TRANSMIT_OPTION_EXPLORE, NULL, FALSE);
    }
}

static void
handleCmdClassManufacturerSpecific(BYTE sourceNode, ZW_APPLICATION_TX_BUFFER *pCmd, BYTE cmdLength, BYTE txOption) {
    if (pCmd->ZW_Common.cmd == MANUFACTURER_SPECIFIC_GET)
    {
      txBuf.ZW_ManufacturerSpecificReportFrame.cmdClass = COMMAND_CLASS_MANUFACTURER_SPECIFIC;
      txBuf.ZW_ManufacturerSpecificReportFrame.cmd = MANUFACTURER_SPECIFIC_REPORT;
      txBuf.ZW_ManufacturerSpecificReportFrame.manufacturerId1 = 0x00; /* myStrom/Swisscom manufacturer ID TBD !!! ???*/
      txBuf.ZW_ManufacturerSpecificReportFrame.manufacturerId2 = 0x00; /* myStrom/Swisscom manufacturer ID TBD !!! ???*/
      txBuf.ZW_ManufacturerSpecificReportFrame.productTypeId1 = 0x00; /* Assigned by manufacturer */
      txBuf.ZW_ManufacturerSpecificReportFrame.productTypeId2 = 0x01;
      txBuf.ZW_ManufacturerSpecificReportFrame.productId1 = 0x00; /* Assigned by manufacturer */
      txBuf.ZW_ManufacturerSpecificReportFrame.productId2 = 0x01;
      Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_ManufacturerSpecificReportFrame),
                  txOption | TRANSMIT_OPTION_EXPLORE, NULL, FALSE);
    }
}

static void
handleCmdClassAssociation(BYTE sourceNode, ZW_APPLICATION_TX_BUFFER *pCmd, BYTE cmdLength, BYTE txOption) {
	BYTE param1 = ((BYTE_P)pCmd)[OFFSET_PARAM_1];
    switch (pCmd->ZW_Common.cmd)
    {
      case ASSOCIATION_GET:
        AssociationSendReport(param1, sourceNode, txOption);
        break;

      case ASSOCIATION_SET:
        /* If node only support one group. Ignore the group ID requested */
        /* (As defined in Device class specification */
        AssociationAdd(1, (BYTE*)pCmd + OFFSET_PARAM_2, cmdLength - OFFSET_PARAM_2);
		AssociationStoreAll();
        break;

      case ASSOCIATION_REMOVE:
        /*If node only support one group. Ignore the group ID requested (As defined in Device class specification*/
        AssociationRemove(1, (BYTE*)pCmd + OFFSET_PARAM_2, cmdLength - OFFSET_PARAM_2);
        break;

      case ASSOCIATION_GROUPINGS_GET:
        AssociationSendGroupings(sourceNode, txOption);
        break;
    }
}

static void
handleCmdClassMeter(BYTE sourceNode, ZW_APPLICATION_TX_BUFFER *pCmd, BYTE cmdLength, BYTE txOption) {
	BYTE scale;
	switch (pCmd->ZW_Common.cmd){
		case METER_GET_V2:
			if (cmdLength == 3) {
				/* Version 2 */
				scale = (((BYTE_P)pCmd)[OFFSET_PARAM_1] & METER_GET_PROPERTIES1_SCALE_MASK_V2) >> 3;
				if (scale == 1 || (scale != 0x03 && cValue==0)) {
					/* Note! A device receiving a Meter Get Command containing a non-supported Scale must ignore the command. */
					return;
				}
			} else {
				scale = 0x03;
			}
			PrepareMeterReport(scale);
			Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_MeterReport1byteV2Frame),
				txOption | TRANSMIT_OPTION_EXPLORE, StartPowerDownTimer, FALSE);
			break;
		case METER_SUPPORTED_GET_V2:
    		txBuf.ZW_MeterSupportedReportV2Frame.cmdClass = COMMAND_CLASS_METER;
    		txBuf.ZW_MeterSupportedReportV2Frame.cmd = METER_SUPPORTED_REPORT_V2;
    		txBuf.ZW_MeterSupportedReportV2Frame.properties1 = METER_SUPPORTED_REPORT_PROPERTIES1_METER_RESET_BIT_MASK_V2 | 0x01; /* Meter type: Electric meter;  */
    		txBuf.ZW_MeterSupportedReportV2Frame.properties2 = cValue==0 ? 0x08 /* Pulse count	*/ : 0x0d; /* kWh, W, Pulse count */
    		Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_MeterSupportedReportV2Frame),
                txOption | TRANSMIT_OPTION_EXPLORE, StartPowerDownTimer, FALSE);
			break;
		case METER_RESET_V2:
			// TODO: Reset
		break;
	}
}

static void sendCmdToPIC(BYTE cmd, BYTE length) {
	currentState = STATE_PIC_COMM_BUSY;
	uartLastCmd = cmd;
	ZW_UART_SEND_BYTE(cmd);
	uartRecvBufferPos = 0;
	uartToRecv = length;
}

static BOOL
meterModeSet(BYTE t) {
	if (meterMode == t) {
		return FALSE;
	}
	meterMode = t;
	sendCmdToPIC(meterMode?'D':'A', 1);
	return TRUE;
}

static BOOL
meterCValueSet(BYTE v) {
	if (cValue == v) {
		return FALSE;
	}
	cValue = v;
	return TRUE;
}

static void
meterModeConfigure(BYTE t) {
	if (meterModeSet(t)) {
		ZW_MEM_PUT_BYTE(EEOFFSET_METER_MODE, meterMode);
	}
}

static void
meterCValueConfigure(WORD v) {
	if (meterCValueSet(v)) {
		ZW_MEM_PUT_BYTE(EEOFFSET_METER_CVALUE_1, (v>>8)&0xff);
		ZW_MEM_PUT_BYTE(EEOFFSET_METER_CVALUE_2, v&0xff);
	}
}

static BYTE
meterModeGet() {
	return meterMode;
}

static void
paramterResetToDeafult(BYTE parameterNumber)	{
	switch (parameterNumber) {
		case 1: /* Meter type 0 - analog, 1 - digital*/
			meterModeConfigure(METER_MODE_ANALOG);
			break;
		case 2: /* Meter C-value */
			meterCValueConfigure(DEFAULT_CVALUE);
			break;
	}
}

static void
paramterSet(BYTE parameterNumber, DWORD value) {
	switch (parameterNumber) {
		case 1: /* Meter type 0 - analog, 1 - digital*/
			meterModeConfigure((BYTE)(value&0xff));
			break;
		case 2: /* Meter C-value */
			meterCValueConfigure(value);
			break;
	}
}

static void
paramterGet(BYTE sourceNode, BYTE parameterNumber, BYTE txOption) {
	DWORD value;
	switch (parameterNumber) {
		case 1: /* Meter type 0 - analog, 1 - digital*/
			value = meterModeGet();
			break;
		case 2: /* Meter C-value*/
			value = cValue;
			break;
		default:
			return;
	}

	if (value < 0x80) {
		txBuf.ZW_ConfigurationReport1byteFrame.cmdClass = COMMAND_CLASS_CONFIGURATION;
		txBuf.ZW_ConfigurationReport1byteFrame.cmd = CONFIGURATION_REPORT;
		txBuf.ZW_ConfigurationReport1byteFrame.parameterNumber = parameterNumber;
		txBuf.ZW_ConfigurationReport1byteFrame.level = 1;
		txBuf.ZW_ConfigurationReport1byteFrame.configurationValue1 = value & 0xff;
    	Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_ConfigurationReport1byteFrame),
		  txOption | TRANSMIT_OPTION_EXPLORE, StartPowerDownTimer, FALSE);
	}
	else if (value < 0x8000) {
		txBuf.ZW_ConfigurationReport2byteFrame.cmdClass = COMMAND_CLASS_CONFIGURATION;
		txBuf.ZW_ConfigurationReport2byteFrame.cmd = CONFIGURATION_REPORT;
		txBuf.ZW_ConfigurationReport2byteFrame.parameterNumber = parameterNumber;
		txBuf.ZW_ConfigurationReport2byteFrame.level = 2;
		txBuf.ZW_ConfigurationReport2byteFrame.configurationValue1 = (value >> 8)& 0xff;
		txBuf.ZW_ConfigurationReport2byteFrame.configurationValue2 = value & 0xff;
    	Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_ConfigurationReport1byteFrame),
		  txOption | TRANSMIT_OPTION_EXPLORE, StartPowerDownTimer, FALSE);
	} else {
		txBuf.ZW_ConfigurationReport4byteFrame.cmdClass = COMMAND_CLASS_CONFIGURATION;
		txBuf.ZW_ConfigurationReport4byteFrame.cmd = CONFIGURATION_REPORT;
		txBuf.ZW_ConfigurationReport4byteFrame.parameterNumber = parameterNumber;
		txBuf.ZW_ConfigurationReport4byteFrame.level = 4;
		txBuf.ZW_ConfigurationReport4byteFrame.configurationValue1 = (value >> 24) & 0xff;
		txBuf.ZW_ConfigurationReport4byteFrame.configurationValue2 = (value >> 16) & 0xff;
		txBuf.ZW_ConfigurationReport4byteFrame.configurationValue3 = (value >> 8) & 0xff;
		txBuf.ZW_ConfigurationReport4byteFrame.configurationValue4 = value & 0xff;
    	Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_ConfigurationReport1byteFrame),
		  txOption | TRANSMIT_OPTION_EXPLORE, StartPowerDownTimer, FALSE);
	}
}

static void
handleCmdClassConfiguration(BYTE sourceNode, ZW_APPLICATION_TX_BUFFER *pCmd, BYTE cmdLength, BYTE txOption) {
	switch (pCmd->ZW_Common.cmd){
		case CONFIGURATION_GET:
		{
			BYTE parameterNumber = ((BYTE_P)pCmd)[OFFSET_PARAM_1];
			paramterGet(sourceNode, parameterNumber, txOption);
		} break;
		case CONFIGURATION_SET:
		{
			BYTE parameterNumber = ((BYTE_P)pCmd)[OFFSET_PARAM_1];
			BYTE size = ((BYTE_P)pCmd)[OFFSET_PARAM_2];
			DWORD value;
			switch (size & 0x7) {
				case 1:
					value = ((BYTE_P)pCmd)[OFFSET_PARAM_3];
					break;
				case 2:
					value = ((BYTE_P)pCmd)[OFFSET_PARAM_3];
					value <<= 8;
					value |= ((BYTE_P)pCmd)[OFFSET_PARAM_4];
					break;
				case 4:
					value = ((BYTE_P)pCmd)[OFFSET_PARAM_3];
					value <<= 8;
					value |= ((BYTE_P)pCmd)[OFFSET_PARAM_4];
					value <<= 8;
					value |= ((BYTE_P)pCmd)[OFFSET_PARAM_5];
					value <<= 8;
					value |= ((BYTE_P)pCmd)[OFFSET_PARAM_6];
					break;
			}
			if (size & 0x80) {
				/* Reset to default value */
				paramterResetToDeafult(parameterNumber);
			} else {
				paramterSet(parameterNumber, value);
			}
		} break;
	}
}

static void
handleCmdClassBasic(BYTE sourceNode, ZW_APPLICATION_TX_BUFFER *pCmd, BYTE cmdLength, BYTE rxStatus) {

	if (pCmd->ZW_Common.cmd == BASIC_GET){
    	/* Controller wants the meter value */
    	txBuf.ZW_BasicReportFrame.cmdClass = pCmd->ZW_Common.cmdClass;
    	txBuf.ZW_BasicReportFrame.cmd = BASIC_REPORT;
    	txBuf.ZW_BasicReportFrame.value = 0x01; // Sensor type = Electric meter

		Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(ZW_BASIC_REPORT_FRAME),
		  ((rxStatus & RECEIVE_STATUS_LOW_POWER) ? TRANSMIT_OPTION_LOW_POWER : 0) | TRANSMIT_OPTION_ACK, 
		  StartPowerDownTimer, FALSE);
    }
}

void                              /*RET Nothing                  */
Transport_ApplicationCommandHandler(
  BYTE  rxStatus,                 /* IN Frame header info */
  BYTE  sourceNode,               /* IN Command sender Node ID */
  ZW_APPLICATION_TX_BUFFER *pCmd, /* IN Payload from the received frame, the union */
                                  /*    should be used to access the fields */
  BYTE   cmdLength)               /* IN Number of command bytes including the command */
{

  BYTE txOption = ((rxStatus & RECEIVE_STATUS_LOW_POWER) ?
               TRANSMIT_OPTION_LOW_POWER : 0) | TRANSMIT_OPTION_ACK;

  /* We were about to sleep, prevent it */
  if(currentState == STATE_APPL_IDLE)
  {
    currentState = STATE_COMMAND_HANDLER;
    StartWatchdog();
  }

  powerDownTimeout = COMMAND_POWERDOWNTIMEOUT;

  switch(pCmd->ZW_Common.cmdClass)
  {
    case COMMAND_CLASS_VERSION:
		handleCmdClassVersion(sourceNode, pCmd, cmdLength, txOption);
	break;
    case COMMAND_CLASS_MANUFACTURER_SPECIFIC:
		handleCmdClassManufacturerSpecific(sourceNode, pCmd, cmdLength, txOption);
    	break;
  }

#ifdef SECURITY
	if(nodeSecure && secureCmdHandling != TSEC_CMD_HANDLING_SECURE)
	{
		if(currentState == STATE_COMMAND_HANDLER)
		{
			currentState = STATE_APPL_IDLE;
		}
  	 	return;
	}
#endif

	switch(pCmd->ZW_Common.cmdClass)
	{
		case COMMAND_CLASS_CONFIGURATION:
			handleCmdClassConfiguration(sourceNode, pCmd, cmdLength, txOption);
			break;
		case COMMAND_CLASS_METER:
			handleCmdClassMeter(sourceNode, pCmd, cmdLength, txOption);
			break;
    	case COMMAND_CLASS_BASIC:
			handleCmdClassBasic(sourceNode, pCmd, cmdLength, rxStatus);
			break;
    	case COMMAND_CLASS_WAKE_UP:
    		HandleWakeupFrame(pCmd, txOption, sourceNode);
    		break;
    	case COMMAND_CLASS_ASSOCIATION:
			handleCmdClassAssociation(sourceNode, pCmd, cmdLength, txOption);
    		break;
	}

	if(currentState == STATE_COMMAND_HANDLER)
	{
    	currentState = STATE_APPL_IDLE;
	}
}




/*==========================   ApplicationSlaveUpdate   =======================
**   Inform a slave application that a node information is received.
**   Called from the slave command  when a node information frame
**   is received and the Z-Wave protocol is not in a state where it is needed.
**
**--------------------------------------------------------------------------*/
void
ApplicationSlaveUpdate(
  BYTE bStatus,     /*IN  Status event */
  BYTE bNodeID,     /*IN  Node id of the node that send node info */
  BYTE* pCmd,       /*IN  Pointer to Application Node information */
  BYTE bLen)       /*IN  Node info length                        */
{
}


/*======================   ApplicationNodeInformation   =====================
**    Request Node information and current status
**    Called by the the Z-Wave application layer before transmitting a
**    "Node Information" frame.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
extern  void                  /*RET  Nothing */
ApplicationNodeInformation(
  BYTE   *deviceOptionsMask,      /*OUT Bitmask with application options     */
  APPL_NODE_TYPE  *nodeType,  /*OUT  Device type Generic and Specific   */
  BYTE       **nodeParm,      /*OUT  Device parameter buffer pointer    */
  BYTE       *parmLength)     /*OUT  Number of Device parameter bytes   */
{
  /* this is a NON listening node and it supports optional CommandClasses*/
  *deviceOptionsMask = APPLICATION_NODEINFO_NOT_LISTENING|APPLICATION_NODEINFO_OPTIONAL_FUNCTIONALITY;
  nodeType->generic = GENERIC_TYPE_METER; /* Generic Device Type */
  nodeType->specific = SPECIFIC_TYPE_SIMPLE_METER; /* Specific Device Type */
  *nodeParm = (BYTE *)&nodeInfo;        /* Send list of known command classes. */
  *parmLength = sizeof(nodeInfo);       /* Set length*/

#ifdef SECURITY
  MemoryGetID(myHomeID, &myNodeID);
  nodeSecure = GetNodeSecure(EEOFFSET_TRANSPORT_SETTINGS_START);
  if (myNodeID)
	if (nodeSecure)
	{
  		*nodeParm = nodeInfoAfterIncluded;        /* Send list of known command classes. */
  		*parmLength = sizeof(nodeInfoAfterIncluded);       /* Set length*/
	}
#endif

}

#ifdef SECURITY
void
FuncZWSecure(BYTE txStatus)
{
	if (txStatus == TRANSPORT_WORK_END)
	{
	   StartPowerDownTimer();
	}
	else if (txStatus == TRANSPORT_WORK_START)
	{
	   StopPowerDownTimer();
	}
	else if (txStatus == TRANSPORT_WORK_ERROR)
	{
	   StartPowerDownTimer();
	}
}
#endif

