/********************************  LEDDIM.C  ********************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2001
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 * Copyright Zensys A/S, 2001
 *
 * Description: Slave application for evaluation kit LED dimmer
 *
 * Author:   Peter Shorty
 *
 * Last Changed By:  $Author: efh $
 * Revision:         $Revision: 18852 $
 * Last Changed:     $Date: 2010-09-29 12:18:01 +0200 (on, 29 sep 2010) $
 *
 ****************************************************************************/
/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
#include <config_app.h>
#ifdef ZW_CONTROLLER
#include <ZW_controller_api.h>
#else
#include <ZW_slave_api.h>
#endif
#include <ZW_pindefs.h>
#include <ZW_evaldefs.h>
#include <ZW_debug_api.h>
#include <LEDdim.h>
#include <eeprom.h>
#include <slave_learn.h>
#include <one_button.h>
#include <ZW_uart_api.h>

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/

#ifdef JP_DK
/* Define RSSI threshold so JP can be tested in DK*/
#define JP_DK_RSSI_THRESHOLD      20
#endif

/* SetLightLevel() command defines */
enum
{
  CURRENT_LEVEL = 0,
  TOGGLE,
  ON,
  OFF,
  LEVEL
};

/* SetLightLevel level defines */
#define LEVEL_DONTCARE    0 /* Level is don't care - the command says it all */


/* Poll function button state defines */
#define POLLIDLE          0
#define POLLDIMMING       1 /* we are dimming dimming */
#define POLLLEARNMODE     3 /* We're in learnmode */


/* Debounce count on button press detection. How many times must we see button */

#define DEBOUNCECOUNT       100


/* Timer repeat define */
#define FOREVER             TIMER_FOREVER
#define ONE_SHOT            TIMER_ONE_TIME

/* DIM direction defines */
#define DIM_TOGGLE          0xFF

/* DIM defines */
#define LEVEL_MAX           99  /* Maximum light level */
#define LEVEL_MIN           1   /* Minimum light level */
#define DIM_SPEED           3   /* DIM timer routine inc/dec with this value */

#define DIM_TIMER_FREQ      10  /* DIM timer routine is called every 100ms */

#define DIM_DIRECTION_DOWN  0
#define DIM_DIRECTION_UP    1

/* Define for setting the time which button must be pressed before starting DIM */
/* and when in childprotect mode its the max time between button presses */
#define DIM_PROTECT         50 /* 1/2 second */
#define CHILD_PROTECT       50 /* max 500ms between button presses */

#define CHILD_PROTECT_COUNT 3  /* number of button press before toggle */
                               /* when in childprotect mode */


#define LIGHTIDLE           0x00
#define LIGHTTOGGLE         0x01
#define LIGHTDIM            0x02
#define LIGHTNOP            0x04

#ifdef APPL_PROD_TEST
/* Production test pin definition */
/* PIN P14 now input */
#define SET_PRODUCTIONTEST_PIN  PIN_IN(P14, 0)
#define IN_PRODUCTIONTEST       (!PIN_GET(P14))
#endif /* APPL_PROD_TEST */

/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/

/* node status */
volatile t_nodeStatus dimmerNodeStatus;

/* A list of the known command classes. Except the basic class which allways */
/* should be supported. Used when node info is send */
t_nodeInfo nodeInfo = {COMMAND_CLASS_SWITCH_MULTILEVEL,
                       COMMAND_CLASS_SWITCH_ALL,
                       COMMAND_CLASS_MANUFACTURER_SPECIFIC,
                       COMMAND_CLASS_VERSION,
                       COMMAND_CLASS_POWERLEVEL};

BYTE timerDimHandle = 0;
BYTE timerPollHandle = 0;
BYTE pollState = POLLIDLE;
BYTE pollDimProtect = FALSE;
BYTE toggleDone = 0;
BYTE buttonPress = 0;
ZW_APPLICATION_TX_BUFFER txBuf;

BYTE ignoreAllOnOff = SWITCH_ALL_ENABLE_ON_OFF; /* Default we do allow ALL ON/OFF */
                                                /* commands to be executed */
/* ignoreAllOnOff = SWITCH_ALL_EXCLUDE_ON_OFF   - Ignore CMD_ALL_ON/CMD_ALL_OFF */
/* ignoreAllOnOff = SWITCH_ALL_EXCLUDE_ONLY_ON  - Ignore CMD_ALL_ON */
/* ignoreAllOnOff = SWITCH_ALL_EXCLUDE_ONLY_OFF - Ignore CMD_ALL_OFF */
/* ignoreAllOnOff = SWITCH_ALL_ENABLE_ON_OFF    - Accept CMD_ALL_ON/CMD_ALL_OFF */

BYTE txOption;

BYTE timerPowerLevelHandle = 0;
BYTE timerPowerLevelSec = 0;
IBYTE testNodeID = ZW_TEST_NOT_A_NODEID;
IBYTE testSourceNodeID;
IBYTE testPowerLevel;
IWORD testFrameCount;
IWORD testFrameSuccessCount;
IBYTE testState = 0;
IBYTE currentPower = normalPower;

#ifdef DEBUG_STRESS
#define DEBUG_STRESS_TIMEOUT_START 250
#define DEBUG_STRESS_TIMEOUT       30
BOOL debugStress = 0;
BYTE debugStressHandle = 0;
BYTE currentStressValue = 0;
#endif

#ifdef METADATA_DUT
IWORD metaframecount = 0;
IWORD metaframerecount = 0;
IWORD timestamp = 0;
IWORD timespan = 0;
BOOL metarecount = FALSE;
BOOL metadocount = FALSE;
BYTE metaDataReportNode;
#endif

BYTE bNWIStartup;
BYTE bMyNodeID;

/****************************************************************************/
/*                              EXPORTED DATA                               */
/****************************************************************************/

/****************************************************************************/
/*                            PRIVATE FUNCTIONS                             */
/****************************************************************************/
static void DIMStop(BYTE saveInE2PROM);

#ifdef DEBUG_STRESS
void
SendBasicSet( void )
{
  if (!debugStressHandle)
  {
    debugStressHandle = ZW_TIMER_START(SendBasicSet, DEBUG_STRESS_TIMEOUT, ONE_SHOT);
  }
  else
  {
    debugStressHandle = 0;
    txBuf.ZW_BasicSetFrame.cmdClass = COMMAND_CLASS_BASIC;
    txBuf.ZW_BasicSetFrame.cmd      = BASIC_SET;
    txBuf.ZW_BasicSetFrame.value    = currentStressValue;
    if (currentStressValue == 10)
    {
      currentStressValue = 50;
    }
    else
    {
      if (currentStressValue == 50)
      {
        currentStressValue = 0;
      }
      else
      {
        currentStressValue = 10;
      }
    }
    ZW_SEND_DATA(2, (BYTE *)&txBuf, sizeof(txBuf.ZW_BasicSetFrame), (TRANSMIT_OPTION_ACK), SendBasicSet);
  }
}
#endif


/*=============================   LEDallON   ===============================
**    Turn all LED ON
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
static void                   /*RET  Nothing                  */
LEDallON ( void )             /* IN  Nothing                  */
{
  LED_ON(1);
  LED_ON(2);
  LED_ON(3);
  LED_ON(4);
}


/*=============================   LEDallOFF   ==============================
**    Turn all LED OFF
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
static void                   /*RET  Nothing                  */
LEDallOFF( void )             /* IN  Nothing                  */
{
  LED_OFF(1);
  LED_OFF(2);
  LED_OFF(3);
  LED_OFF(4);
}


/*==============================   LEDUpdate   ==============================
**    Update LEDs according to current level
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
static void                   /*RET  Nothing                  */
LEDUpdate( void )             /* IN  Nothing                  */
{
  if (dimmerNodeStatus.status & M_DS_ON_OFF)
  {
    BYTE n;
    n = dimmerNodeStatus.level / (100 / 4);
    switch (n)
    {
      case 0:
        LED_ON(1);
        LED_OFF(2);
        LED_OFF(3);
        LED_OFF(4);
        break;

      case 1:
        LED_ON(1);
        LED_ON(2);
        LED_OFF(3);
        LED_OFF(4);
        break;

      case 2:
        LED_ON(1);
        LED_ON(2);
        LED_ON(3);
        LED_OFF(4);
        break;

      case 3:
        LED_ON(1);
        LED_ON(2);
        LED_ON(3);
        LED_ON(4);
        break;

      default:
        LEDallON();
        break;
    }
  }
  else
  {
    LEDallOFF();
  }
}


/*===========================   ValidateLevel   =============================
**    Validate current level value
**
**    Side effects
**
**--------------------------------------------------------------------------*/
static void              /*RET  Nothing                  */
ValidateLevel( void )    /* IN  Nothing                  */
{
  if (dimmerNodeStatus.level > LEVEL_MAX)
  {
    dimmerNodeStatus.level = LEVEL_MAX;
  }
  else
  {
    if (dimmerNodeStatus.level < LEVEL_MIN)
    {
      dimmerNodeStatus.level = LEVEL_MIN;
    }
  }
}


/*============================   TimerChangeLevel   =========================
**    Change level
**
**    Side effects
**
**--------------------------------------------------------------------------*/
static void                /*RET  Nothing                  */
TimerChangeLevel( void )   /* IN  Nothing                  */
{
  if (dimmerNodeStatus.status & M_DS_CHANGING_DIM) /* Are we still DIMming? */
  {
    if ((dimmerNodeStatus.status & M_DS_UP_DOWN) == 0)  /* Dim UP - more light */
    {
      if (dimmerNodeStatus.level < LEVEL_MAX)
      {
        dimmerNodeStatus.level += DIM_SPEED;
        if (dimmerNodeStatus.level >= LEVEL_MAX)
        {
          if (dimmerNodeStatus.status & M_DS_ROLLOVER)
          {
            dimmerNodeStatus.status ^= M_DS_UP_DOWN;
          }
        }
      }
      else
      {
        if (dimmerNodeStatus.status & M_DS_ROLLOVER)
        {
          dimmerNodeStatus.status ^= M_DS_UP_DOWN;
        }
      }
    }
    else  /* Dim DOWN - less light */
    {
      if (dimmerNodeStatus.level > LEVEL_MIN)
      {
        dimmerNodeStatus.level -= DIM_SPEED;
        if (dimmerNodeStatus.level <= LEVEL_MIN)
        {
          if (dimmerNodeStatus.status & M_DS_ROLLOVER)
          {
            dimmerNodeStatus.status ^= M_DS_UP_DOWN;
          }
        }
      }
      else
      {
        if (dimmerNodeStatus.status & M_DS_ROLLOVER)
        {
          dimmerNodeStatus.status ^= M_DS_UP_DOWN;
        }
      }
    }
    LEDUpdate();  /* Update LEDs */
  }
}


/*===========================   TimerPollCancel   ============================
**    Function description
**      cancels timer, if still valid, used in CHILD_PROTECT
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
static void             /*RET  Nothing                  */
TimerPollCancel( void ) /*IN  Nothing                   */
{
  if (timerPollHandle)
  {
    ZW_TIMER_CANCEL(timerPollHandle);
    timerPollHandle = 0;
  }
}


/*===========================   TimerDimCancel   =============================
**    Function description
**      cancels timer, if still valid, used in DIM
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
static void             /*RET  Nothing                  */
TimerDimCancel( void )  /*IN  Nothing                   */
{
  if (timerDimHandle)
  {
    ZW_TIMER_CANCEL(timerDimHandle);
    timerDimHandle = 0;
  }
}


/*=========================   TimerPollDimProtect   =========================
**    DIM_PROTECT time has past
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
static void                 /*RET  Nothing                  */
TimerPollDimProtect( void ) /* IN  Nothing                  */
{
  pollDimProtect = TRUE;
}


/*===============================   DIMStart   ==============================
**    Start DIMming
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
static void         /*RET  Nothing                */
DIMStart(
  BYTE level,         /* IN  starting DIM level     */
  BYTE dimBits)       /* IN  DIMming config bits    */
{
   /* are we dimming allready ? */
  if (dimmerNodeStatus.status & M_DS_CHANGING_DIM)
  {
    DIMStop(FALSE);
  }
  dimmerNodeStatus.level = level;
  ValidateLevel();
  dimmerNodeStatus.status |= M_DS_ON_OFF;    /* Now ON */
  /* Save new status in EEPROM */
  ZW_MEM_PUT_BYTE(EEOFFSET_STATUS, dimmerNodeStatus.status);

  if (dimBits == DIM_TOGGLE)
  {
    dimmerNodeStatus.status ^= M_DS_UP_DOWN;   /* Toggle DIM direction */
    dimmerNodeStatus.status |= M_DS_ROLLOVER;  /* Make sure its set */
  }
  else
  {
    dimmerNodeStatus.status &= ~(M_DS_UP_DOWN | M_DS_ROLLOVER);
    dimmerNodeStatus.status |= dimBits & (M_DS_UP_DOWN | M_DS_ROLLOVER);
  }
  dimmerNodeStatus.status |= M_DS_CHANGING_DIM;  /* Indicate DIMming... */
  TimerDimCancel(); /* Cancel timer if needed... */
  /* Allocate timer for calling TimerChangeLevel every DIM_TIMER_FREQ, forever */
  timerDimHandle = ZW_TIMER_START(TimerChangeLevel, DIM_TIMER_FREQ, FOREVER);
}


/*===============================   DIMStop   ==============================
**    Stop DIMming
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
static void           /*RET  Nothing            */
DIMStop(
  BYTE saveInE2PROM)    /* IN  Should the current level be saved in EEPROM */
{
  TimerDimCancel(); /* Cancel timer if needed... */
  dimmerNodeStatus.status &= ~(M_DS_CHANGING_DIM); /* STOP DIMming */
  ValidateLevel();
  if (saveInE2PROM)
  {
    ZW_MEM_PUT_BYTE(EEOFFSET_LEVEL, dimmerNodeStatus.level); /* Save the last value */
//    ZW_MEM_PUT_BYTE(EEOFFSET_STATUS, dimmerNodeStatus.status);
  }
}


/*=============================   SetLightLevel   ============================
**    Set the light level to:
**
**    CURRENT_LEVEL - The current light level stored in eeprom
**    LEVEL         - Use parameter bLevel as light level
**    TOGGLE        - Toggle the light level
**    ON            - Full on
**    OFF           - Off
**
**--------------------------------------------------------------------------*/
static void       /*RET  Nothing           */
SetLightLevel(
  BYTE bAction,     /* IN  Action to be done */
  BYTE bLevel)      /* IN  Lightlevel to set */
{
  if (dimmerNodeStatus.status & M_DS_CHANGING_DIM)  /* If we're dimming */
  {
    DIMStop(FALSE);                                 /* then STOP that */
  }
  switch (bAction)
  {
    case TOGGLE:
      if ((dimmerNodeStatus.status & M_DS_ON_OFF) == 0)   /* Is switch OFF ? */
			{
      	dimmerNodeStatus.status |= M_DS_ON_OFF;
      	LEDUpdate();
			}
      else
      {
      	dimmerNodeStatus.status &= ~(M_DS_ON_OFF);
      	LEDallOFF(); /* Just do it! */
      }
      break;

    case LEVEL:
      if (bLevel == SWITCHED_ON)
			{
      	dimmerNodeStatus.level = LEVEL_MAX;
      	dimmerNodeStatus.status |= M_DS_ON_OFF;
      	ZW_MEM_PUT_BYTE(EEOFFSET_LEVEL, dimmerNodeStatus.level);
      	LEDallON(); /* Just do it! */
			}
      else if (bLevel == SWITCHED_OFF)
			{
      	dimmerNodeStatus.status &= ~(M_DS_ON_OFF);
      	LEDallOFF(); /* Just do it! */
			}
      else
      {
        dimmerNodeStatus.level = bLevel;
        dimmerNodeStatus.status |= M_DS_ON_OFF;
        ZW_MEM_PUT_BYTE(EEOFFSET_LEVEL, dimmerNodeStatus.level);
        LEDUpdate();
      }
      break;

    case CURRENT_LEVEL:
      dimmerNodeStatus.status |= M_DS_ON_OFF;
      LEDUpdate();
      break;

    case ON:
      dimmerNodeStatus.level = LEVEL_MAX;
      dimmerNodeStatus.status |= M_DS_ON_OFF;
      ZW_MEM_PUT_BYTE(EEOFFSET_LEVEL, dimmerNodeStatus.level);
      LEDallON(); /* Just do it! */
      break;

    case OFF:
      dimmerNodeStatus.status &= ~(M_DS_ON_OFF);
      LEDallOFF(); /* Just do it! */
      break;
  }
  /* Save new status in EEPROM */
  ZW_MEM_PUT_BYTE(EEOFFSET_STATUS, dimmerNodeStatus.status);
}

/*=============================  ApplicationPoll   =========================
**    Application poll function for LED dimmer
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                    /*RET  Nothing                  */
ApplicationPoll( void ) /* IN  Nothing                  */
{
  BYTE lastAction = OneButtonLastAction();
  ZW_DEBUG_CMD_POLL();

  if (bNWIStartup)
  {
    StartLearnModeNow(ZW_SET_LEARN_MODE_NWI);
    bNWIStartup = FALSE;
    pollState = POLLLEARNMODE;
  }

  switch (pollState)
  {
    case POLLIDLE:
      if (lastAction==BUTTON_WAS_PRESSED)
      {
        if (toggleDone == LIGHTIDLE)
        {
          SetLightLevel(TOGGLE, LEVEL_DONTCARE);
          // toggleDone = LIGHTTOGGLE;
        }
      }
      else if (lastAction==BUTTON_IS_HELD)
      {
         toggleDone = LIGHTDIM;  /* Now dimming has been started */
         DIMStart(dimmerNodeStatus.level, DIM_TOGGLE);
        pollState = POLLDIMMING;
      }
      else if (lastAction==BUTTON_TRIPLE_PRESS)
      {
        StartLearnModeNow(ZW_SET_LEARN_MODE_CLASSIC);
        pollState = POLLLEARNMODE;
      }
      break;

    case POLLDIMMING:
      if (lastAction==BUTTON_WAS_RELEASED)
      {
        DIMStop(TRUE);
        toggleDone = LIGHTIDLE;
        pollState = POLLIDLE;
      }
      break;

    case POLLLEARNMODE:
      /* Just wait... */
      if (lastAction==BUTTON_WAS_PRESSED)
      {
        /* Stop learn mode */
        StartLearnModeNow(FALSE);
        pollState = POLLIDLE;
      }
      break;
  }
}


/*==============================   SendReport   ==============================
**
**  Function:  SendReport
**
**    Sends a report frame to source node.
**
**  Side effects: None
**
**--------------------------------------------------------------------------*/
void              /*RET Nothing */
SendReport(
BYTE sourceNode,  /* IN Source node - to whom report is to be send */
BYTE classcmd,    /* IN Command class, which the report belongs to */
BYTE reportparam) /* IN paramter to include in report frame */
{
  txBuf.ZW_BasicReportFrame.cmdClass = classcmd;
/* Report command definition - BASIC/ALL/PROTECTION/MULTI report use the same value ;-) */
  txBuf.ZW_BasicReportFrame.cmd      = BASIC_REPORT;
  txBuf.ZW_BasicReportFrame.value    = reportparam;
/* Size of Report (BASIC/ALL/PROTECTION/MULTI GET response) frame are equal for BASIC/ALL/PROTECTION/MULTI */
  ZW_SEND_DATA(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_BasicReportFrame), txOption, NULL);
}


/*===========================   SendTestReport   ============================
**    Send current Powerlevel test results
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void
SendTestReport( void )
{
  txBuf.ZW_PowerlevelTestNodeReportFrame.cmdClass = COMMAND_CLASS_POWERLEVEL;
  txBuf.ZW_PowerlevelTestNodeReportFrame.cmd = POWERLEVEL_TEST_NODE_REPORT;
  txBuf.ZW_PowerlevelTestNodeReportFrame.testNodeid = testNodeID;
  txBuf.ZW_PowerlevelTestNodeReportFrame.statusOfOperation = testState;
  txBuf.ZW_PowerlevelTestNodeReportFrame.testFrameCount1 = (BYTE)(*((BYTE*)&testFrameSuccessCount));
  txBuf.ZW_PowerlevelTestNodeReportFrame.testFrameCount2 = (BYTE)*(&testFrameSuccessCount);
  /* Send Report - we do not care if it gets there or not - if needed report can be requested again */
  ZW_SEND_DATA(testSourceNodeID, (BYTE *)&txBuf, sizeof(txBuf.ZW_PowerlevelTestNodeReportFrame), txOption, NULL);
}


/*=============================   SendTestDone   ============================
**    Test frame has been transmitted to DUT and the result is noted for
**    later reporting. If not finished then another Test frame is transmitted.
**    If all test frames has been transmitted then the test is stopped
**    and the final result is reported to the PowerlevelTest initiator.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void
SendTestDone(
  BYTE bStatus)
{
  if (bStatus == TRANSMIT_COMPLETE_OK)
  {
    testFrameSuccessCount++;
  }
  if (testFrameCount && (--testFrameCount))
  {
    ZW_SEND_TEST_FRAME(testNodeID, testPowerLevel, SendTestDone);
  }
  else
  {
    if (testFrameSuccessCount)
    {
      testState = ZW_TEST_SUCCES;
    }
    else
    {
      testState = ZW_TEST_FAILED;
    }
    ZW_RF_POWERLEVEL_SET(currentPower);  /* Back to previous setting */
    SendTestReport();
  }
}


/*=======================   PowerLevelTimerCancel   ==========================
**    Cancels PowerLevel timer
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void
PowerLevelTimerCancel( void )
{
  ZW_TIMER_CANCEL(timerPowerLevelHandle);
  timerPowerLevelHandle = 0;
}


/*===============================   StartTest   ==============================
**    Timer callback which maked sure that the RF transmit powerlevel is set
**    back to normalPower after the designated time period.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void
PowerLevelTimeout( void )
{
  if (!--timerPowerLevelSec)
  {
    ZW_RF_POWERLEVEL_SET(normalPower);  /* Reset powerlevel to normalPower */
    PowerLevelTimerCancel();
  }
}


/*===============================   StartTest   ==============================
**    Start the powerlevel test run
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void
StartTest(void)
{
  testState = ZW_TEST_INPROGRESS;
  currentPower = ZW_RF_POWERLEVEL_GET();  /* Get current (normalPower) */
  ZW_RF_POWERLEVEL_SET(testPowerLevel);
  ZW_SEND_TEST_FRAME(testNodeID, testPowerLevel, SendTestDone);
}


#ifdef METADATA_DUT
void
SendMetaTestReportComplete(
  BYTE bStatus)
{
  if (metadocount && !metarecount)
  {
    metarecount = TRUE;
    timestamp = getTickTime();
  }
}


void
SendMetaTestReport(
  BYTE sourceNode)
{
  txBuf.ZW_ManufacturerProprietary2ByteFrame.cmdClass = COMMAND_CLASS_MANUFACTURER_PROPRIETARY;
  txBuf.ZW_ManufacturerProprietary2ByteFrame.manufacturerId1 = 0x00;
  txBuf.ZW_ManufacturerProprietary2ByteFrame.manufacturerId2 = 0x00;
  txBuf.ZW_ManufacturerProprietary2ByteFrame.value1 = 0x03;
  txBuf.ZW_ManufacturerProprietary2ByteFrame.value2 = 0x81;
  *((&txBuf.ZW_ManufacturerProprietary2ByteFrame.value2) + 1) = (BYTE)*((BYTE*)&timespan);
  *((&txBuf.ZW_ManufacturerProprietary2ByteFrame.value2) + 2) = (BYTE)*(&timespan);
  *((&txBuf.ZW_ManufacturerProprietary2ByteFrame.value2) + 3) = (BYTE)*((BYTE*)&metaframecount);
  *((&txBuf.ZW_ManufacturerProprietary2ByteFrame.value2) + 4) = (BYTE)*(&metaframecount);
  if (metarecount)
  {
    *((&txBuf.ZW_ManufacturerProprietary2ByteFrame.value2) + 5) = (BYTE)*((BYTE*)&metaframerecount);
    *((&txBuf.ZW_ManufacturerProprietary2ByteFrame.value2) + 6) = (BYTE)*(&metaframerecount);
  }
  ZW_SEND_DATA(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_ManufacturerProprietary2ByteFrame)+((metarecount) ? 6 : 4), txOption, SendMetaTestReportComplete);
#if 0
  txBuf.ZW_MetaDataReportFrame.cmdClass = COMMAND_CLASS_META_DATA;
  txBuf.ZW_MetaDataReportFrame.cmd      = META_DATA_REPORT;
  /* We use the MetaData report to report our findings */
  txBuf.ZW_MetaDataReportFrame.lastAndReportNo1 = (BYTE)*((BYTE*)&timespan);
  txBuf.ZW_MetaDataReportFrame.reportNo2 = (BYTE)*(&timespan);
  txBuf.ZW_MetaDataReportFrame.data1 = (BYTE)*((BYTE*)&metaframecount);
  *((&txBuf.ZW_MetaDataReportFrame.data1) + 1) = (BYTE)*(&metaframecount);
  if (metarecount)
  {
    *((&txBuf.ZW_MetaDataReportFrame.data1) + 2) = (BYTE)*((BYTE*)&metaframerecount);
    *((&txBuf.ZW_MetaDataReportFrame.data1) + 3) = (BYTE)*(&metaframerecount);
  }
  ZW_SEND_DATA(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_MetaDataReportFrame)+((metarecount) ? 3 : 1), txOption, SendMetaTestReportComplete);
#endif
}
#endif


/*========================   ApplicationCommandHandler   ====================
**    Handling of a received application commands and requests
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void                              /*RET Nothing                  */
ApplicationCommandHandler(
  BYTE  rxStatus,                 /* IN Frame header info */
  BYTE  sourceNode,               /* IN Command sender Node ID */
  ZW_APPLICATION_TX_BUFFER *pCmd, /* IN Payload from the received frame, the union */
                                  /*    should be used to access the fields */
  BYTE   cmdLength)               /* IN Number of command bytes including the command */
{
  BYTE classcmd = pCmd->ZW_Common.cmdClass;
  BYTE cmd = pCmd->ZW_Common.cmd;
  BYTE param1 = *((BYTE*)pCmd + OFFSET_PARAM_1);
  BYTE param2 = *((BYTE*)pCmd + OFFSET_PARAM_2);
  BYTE param3 = *((BYTE*)pCmd + OFFSET_PARAM_3);
  BYTE param4 = *((BYTE*)pCmd + OFFSET_PARAM_4);
  BYTE status = dimmerNodeStatus.status;
#ifdef METADATA_DUT
  BYTE frametype = rxStatus & RECEIVE_STATUS_TYPE_MASK;
#endif
 	txOption = ((rxStatus & RECEIVE_STATUS_LOW_POWER) ?
 	            TRANSMIT_OPTION_LOW_POWER : 0) | TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE;
  if ((classcmd == COMMAND_CLASS_SWITCH_MULTILEVEL) ||
      (classcmd == COMMAND_CLASS_BASIC))
  {
    /* Single switch command */
    switch(cmd)
    {
      /* Start dimming */
      case SWITCH_MULTILEVEL_START_LEVEL_CHANGE:
        DIMStart(param2, (param1 & (M_DS_UP_DOWN | M_DS_ROLLOVER)));
        break;

      /* Stop dimming */
      case SWITCH_MULTILEVEL_STOP_LEVEL_CHANGE:
        DIMStop(TRUE);
        break;

      /* Controller wants the level */
      case SWITCH_MULTILEVEL_GET:
        {
          BYTE level = (status & M_DS_ON_OFF) ? dimmerNodeStatus.level : SWITCHED_OFF;
          SendReport(sourceNode, classcmd, level);
        }
        break;

      /* Jump to the specified level param1 */
      case SWITCH_MULTILEVEL_SET:
        if (param1 == SWITCHED_ON)
        {
          SetLightLevel(CURRENT_LEVEL, LEVEL_DONTCARE);
        }
        else if (param1<100) /* Check for valid level */
        {
          SetLightLevel(LEVEL, param1);
        }
        break;
    } /* switch */
  } /* if */
#ifdef ZW_CONTROLLER
  else if (classcmd == COMMAND_CLASS_CONTROLLER_REPLICATION)
  {
    /* Handle replication frames */
    /* We are only interested in group replication here */
    /* but all other replication can be handled in the same way. */
    /* In this application we ignore everything else and just */
    /* send the command complete reply */
#if 0
    if (pCmd->ZW_Common.cmd == CTRL_REPLICATION_TRANSFER_GROUP)
    {
      /* We are only interested in group 1 */
      if (pCmd->ZW_TransferGroupFrame.groupID == 1)
      {
        /* Add the node ID to the group */
        AddNodeToGroup(pCmd->ZW_TransferGroupFrame.nodeID);
      }
    }
#endif
  }
#endif
  else if (classcmd == COMMAND_CLASS_SWITCH_ALL)
  {
    switch(cmd)
    {
      case SWITCH_ALL_ON:
        if ((ignoreAllOnOff != SWITCH_ALL_EXCLUDE_ON_OFF) && (ignoreAllOnOff != SWITCH_ALL_EXCLUDE_ONLY_ON))
        {
          SetLightLevel(ON, LEVEL_DONTCARE);
        }
        break;

      case SWITCH_ALL_OFF:
#ifdef METADATA_DUT
        if (frametype == RECEIVE_STATUS_TYPE_BROAD)
        {
          /* We received the broadcasted "ALL OFF" - start counting and sampling */
          timestamp = getTickTime();
          metarecount = FALSE;  /* This is the initial metadata frame count */
          metadocount = TRUE; /* Now we count the received metadata frames */
          metaframecount = 0; /* Initially no frames received */
          metaframerecount = 0; /* Initially no frames received */
        }
        else if (metadocount && !metarecount && (frametype == RECEIVE_STATUS_TYPE_SINGLE))
        {
          timespan = getTickTime() - timestamp;
          metaDataReportNode = sourceNode;
#if 1
          SendMetaTestReportComplete(TRUE);
#else
          SendMetaTestReport(sourceNode);
#endif
        }
#endif
        if ((ignoreAllOnOff != SWITCH_ALL_EXCLUDE_ON_OFF) && (ignoreAllOnOff != SWITCH_ALL_EXCLUDE_ONLY_OFF))
        {
          SetLightLevel(OFF, LEVEL_DONTCARE);
        }
        break;

      case SWITCH_ALL_SET:
        ignoreAllOnOff = param1;
        /* Store it in the EEPROM */
        ZW_MEM_PUT_BYTE(EEOFFSET_IGNORE_ALL_ON_OFF, ignoreAllOnOff);
        break;

      case SWITCH_ALL_GET:
        SendReport(sourceNode, COMMAND_CLASS_SWITCH_ALL, ignoreAllOnOff);
        break;

        default:
        break;

    } /* switch */
  } /* else if == COMMAND_CLASS_SWITCH_ALL */
  else if (classcmd == COMMAND_CLASS_POWERLEVEL)
  {
    switch (cmd)
    {
      case POWERLEVEL_SET:
	  	if(( param1==normalPower)||(param1==minus1dBm)||(param1==minus2dBm)||
		(param1==minus3dBm)||(param1==minus4dBm)||(param1==minus5dBm)||
		(param1==minus6dBm)||(param1==minus7dBm)||(param1==minus8dBm)||(param1==minus9dBm))
		{
	        if (param2 || (param1 == normalPower))  /* Only if timeout defined or powerlevel is normalPower */
	        {
	          param1 = ZW_RF_POWERLEVEL_SET(param1);  /* Set powerlevel according to command */
	          if (timerPowerLevelHandle)
	          {
	            PowerLevelTimerCancel();
	            timerPowerLevelSec = 0;
	          }
	          if (param1 != normalPower)  /* If normalPower then no timer is started */
	          {
	            timerPowerLevelSec = param2;
	            timerPowerLevelHandle = ZW_TIMER_START(PowerLevelTimeout, TIMER_ONE_SECOND, FOREVER);
	          }
	        }
		}
        break;

      case POWERLEVEL_GET:
        txBuf.ZW_PowerlevelReportFrame.cmdClass = COMMAND_CLASS_POWERLEVEL;
        txBuf.ZW_PowerlevelReportFrame.cmd = POWERLEVEL_REPORT;
        txBuf.ZW_PowerlevelReportFrame.powerLevel = ZW_RF_POWERLEVEL_GET();
        txBuf.ZW_PowerlevelReportFrame.timeout = timerPowerLevelSec;
        ZW_SEND_DATA(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_PowerlevelReportFrame), txOption, NULL);
        break;

      case POWERLEVEL_TEST_NODE_SET:
        if (testState != ZW_TEST_INPROGRESS)
        {
          testSourceNodeID = sourceNode;
          testNodeID = param1;
          testPowerLevel = param2;
          testFrameCount = (WORD)((WORD)param3 << 8) + param4;
          testFrameSuccessCount = 0;
          if (testFrameCount)
          {
            StartTest();
          }
          else
          {
            testState = ZW_TEST_FAILED;
            SendTestReport();
          }
        }
        break;

      case POWERLEVEL_TEST_NODE_GET:
        SendTestReport();
        break;
        default:
        break;

    } /* switch */
  } /* else if == COMMAND_CLASS_POWERLEVEL */
  else if (classcmd == COMMAND_CLASS_MANUFACTURER_SPECIFIC)
  {
    switch (cmd)
    {
      case MANUFACTURER_SPECIFIC_GET:
        txBuf.ZW_ManufacturerSpecificReportFrame.cmdClass = COMMAND_CLASS_MANUFACTURER_SPECIFIC;
        txBuf.ZW_ManufacturerSpecificReportFrame.cmd = MANUFACTURER_SPECIFIC_REPORT;
        txBuf.ZW_ManufacturerSpecificReportFrame.manufacturerId1 = 0x00; // Zensys manufacturer ID
        txBuf.ZW_ManufacturerSpecificReportFrame.manufacturerId2 = 0x00;
        txBuf.ZW_ManufacturerSpecificReportFrame.productTypeId1 = 0x00; // Assigned by manufacturer
        txBuf.ZW_ManufacturerSpecificReportFrame.productTypeId2 = 0x00;
        txBuf.ZW_ManufacturerSpecificReportFrame.productId1 = 0x00; // Assigned by manufacturer
        txBuf.ZW_ManufacturerSpecificReportFrame.productId2 = 0x00;
        ZW_SEND_DATA(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_ManufacturerSpecificReportFrame), txOption, NULL);
        break;
    }
  }
  else if (classcmd == COMMAND_CLASS_VERSION)
  {
    switch (cmd)
    {
      case VERSION_GET:
        txBuf.ZW_VersionReportFrame.cmdClass = COMMAND_CLASS_VERSION;
        txBuf.ZW_VersionReportFrame.cmd = VERSION_REPORT;
        txBuf.ZW_VersionReportFrame.zWaveLibraryType = ZW_TYPE_LIBRARY();
#ifdef ZW_VERSION_BETA
        txBuf.ZW_VersionReportFrame.zWaveProtocolVersion = ZW_VERSION_MAJOR+(ZW_VERSION_BETA-0x60)*10;
#else
        txBuf.ZW_VersionReportFrame.zWaveProtocolVersion = ZW_VERSION_MAJOR;
#endif
        txBuf.ZW_VersionReportFrame.zWaveProtocolSubVersion = ZW_VERSION_MINOR;
        txBuf.ZW_VersionReportFrame.applicationVersion = APP_VERSION;
        txBuf.ZW_VersionReportFrame.applicationSubVersion = APP_REVISION;
        ZW_SEND_DATA(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_VersionReportFrame), txOption, NULL);
        break;

      case VERSION_COMMAND_CLASS_GET:
        txBuf.ZW_VersionCommandClassReportFrame.cmdClass = COMMAND_CLASS_VERSION;
        txBuf.ZW_VersionCommandClassReportFrame.cmd = VERSION_COMMAND_CLASS_REPORT;
        txBuf.ZW_VersionCommandClassReportFrame.requestedCommandClass = param1;
        switch (param1)
        {
          case COMMAND_CLASS_BASIC:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = BASIC_VERSION;
            break;

          case COMMAND_CLASS_SWITCH_MULTILEVEL:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = SWITCH_MULTILEVEL_VERSION;
            break;

          case COMMAND_CLASS_SWITCH_ALL:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = SWITCH_ALL_VERSION;
            break;

          case COMMAND_CLASS_MANUFACTURER_SPECIFIC:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = MANUFACTURER_SPECIFIC_VERSION;
            break;

          case COMMAND_CLASS_VERSION:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = VERSION_VERSION;
            break;

          case COMMAND_CLASS_PROTECTION:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = PROTECTION_VERSION;
            break;

          case COMMAND_CLASS_POWERLEVEL:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = POWERLEVEL_VERSION;
            break;

          default:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = UNKNOWN_VERSION;
            break;
        }
        ZW_SEND_DATA(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_VersionCommandClassReportFrame), txOption, NULL);
        break;
    }
  }
#ifdef METADATA_DUT
  else if (classcmd == COMMAND_CLASS_MANUFACTURER_PROPRIETARY)
  {
    cmd = pCmd->ZW_ManufacturerProprietary2ByteFrame.value1;
    switch (cmd)
    {
      case 0x03:
        break;

      case 0x04:
        if (metadocount)
        {
          if (metarecount)
          {
            metaframerecount++;
            if (timespan <= getTickTime() - timestamp)
            {
              SendMetaTestReport(metaDataReportNode);
              metarecount = FALSE;
              metadocount = FALSE;  /* We are done... */
            }
          }
          else
          {
            metaframecount++;
          }
        }
        break;

      default:
        break;
    }
  }
#endif
}

/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/

/*============================   ApplicationInitHW   ========================
**    Non Z-Wave hardware initialization
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/

BYTE                       /*RET  TRUE        */
ApplicationInitHW( BYTE bStatus )  /* IN  Nothing     */
{
#ifdef APPL_PROD_TEST
  IBYTE i;

  SET_PRODUCTIONTEST_PIN;
  for (i = 0; i < 10; i++) ;  /* Short delay... */
  if (IN_PRODUCTIONTEST) /* Anyone forcing it into productiontest mode ? */
  {
    return(FALSE);  /*Enter production test mode*/
  }
#endif /* APPL_PROD_TEST */

  /* hardware initialization */
  PIN_IN(Button, 1);

  /* Setup specifics for current dim module */
  PIN_OUT(LED1);
  PIN_OUT(LED2);
  PIN_OUT(LED3);
  PIN_OUT(LED4);
  return(TRUE);
}


/*===========================   ApplicationInitSW   =========================
**    Initialization of the Application Software
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
BYTE                      /*RET  TRUE       */
ApplicationInitSW( void ) /* IN   Nothing   */
{
  /* get stored values */
  if (ZW_MEM_GET_BYTE(EEOFFSET_MAGIC) == MAGIC_VALUE)
  {
    dimmerNodeStatus.level = ZW_MEM_GET_BYTE(EEOFFSET_LEVEL);
    if (dimmerNodeStatus.level > LEVEL_MAX)
    {
      dimmerNodeStatus.level = LEVEL_MAX;
    }
    dimmerNodeStatus.status = ZW_MEM_GET_BYTE(EEOFFSET_STATUS);
    /* Get Ignore all On/Off frames state */
    ignoreAllOnOff = ZW_MEM_GET_BYTE(EEOFFSET_IGNORE_ALL_ON_OFF);
  }
  else
  { /* Its alive */
    ZW_MEM_PUT_BYTE(EEOFFSET_LEVEL, LEVEL_MAX); /* Full light */
    ZW_MEM_PUT_BYTE(EEOFFSET_STATUS, 0);  /* Switch OFF */
    ZW_MEM_PUT_BYTE(EEOFFSET_IGNORE_ALL_ON_OFF, SWITCH_ALL_ENABLE_ON_OFF);
    ZW_MEM_PUT_BYTE(EEOFFSET_MAGIC, MAGIC_VALUE); /* Now EEPROM should be OK */
    dimmerNodeStatus.status = SWITCHED_OFF;
    dimmerNodeStatus.level  = LEVEL_MAX;          /* Set level to full light */
    /* ignoreAllOnOff are initialized on startup */
  }
  // ZW_DEBUG_CMD_INIT(1152);
  LEDUpdate();
  OneButtonInit();

  /* Check if we have a node id */
  MemoryGetID(NULL, &bMyNodeID);

  /* Set Nwtwork wide inclusion active if we dont have aa node ID */
  if (bMyNodeID)
    bNWIStartup = FALSE;
  else
    bNWIStartup = TRUE;

#ifdef JP_DK
  /* Set RSSI threshold to be able to test JP frequency in EU */
  ZW_SetListenBeforeTalkThreshold(JP_DK_RSSI_THRESHOLD);
#endif

  return(TRUE);
}


/*============================   ApplicationTestPoll   ======================
**    Function description
**      This function is called when the dimmer enters test mode.
**    Side effects:
**       Code will not exit until it is reset
**--------------------------------------------------------------------------*/
void ApplicationTestPoll(void)
{
#ifdef APPL_PROD_TEST
  /*Send constant out*/
  ZW_SEND_CONST();
  while(1);
#endif
}


/*======================   ApplicationNodeInformation   =====================
**    Request Node information and current status
**    Called by the the Z-Wave application layer before transmitting a
**    "Node Information" frame.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
extern void                  /*RET  Nothing */
ApplicationNodeInformation(
  BYTE   *deviceOptionsMask,      /*OUT Bitmask with application options     */
  APPL_NODE_TYPE  *nodeType,  /*OUT  Device type Generic and Specific   */
  BYTE       **nodeParm,      /*OUT  Device parameter buffer pointer    */
  BYTE       *parmLength)     /*OUT  Number of Device parameter bytes   */
{
  /* this is a listening node and it supports optional CommandClasses */
  *deviceOptionsMask = APPLICATION_NODEINFO_LISTENING|APPLICATION_NODEINFO_OPTIONAL_FUNCTIONALITY;
  nodeType->generic = GENERIC_TYPE_SWITCH_MULTILEVEL; /* Generic device type */
  nodeType->specific = SPECIFIC_TYPE_POWER_SWITCH_MULTILEVEL; /* Specific class */
  *nodeParm = (BYTE *)&nodeInfo;        /* Send list of known command classes. */
  *parmLength = sizeof(nodeInfo);       /* Set length*/
}


#ifdef ZW_SLAVE
/*==========================   ApplictionSlaveUpdate   =======================
**   Inform a slave application that a node information is received.
**   Called from the slave command handler when a node information frame
**   is received and the Z-Wave protocol is not in a state where it is needed.
**
**--------------------------------------------------------------------------*/
void
ApplicationSlaveUpdate(
  BYTE bStatus,     /*IN  Status event */
  BYTE bNodeID,     /*IN  Node id of the node that send node info */
  BYTE* pCmd,       /*IN  Pointer to Application Node information */
  BYTE bLen)       /*IN  Node info length                        */
{
}
#endif


#ifdef ZW_CONTROLLER
/*=====================   ApplicationControllerUpdate   =====================
**   Inform the controller of node information update done through
**   the network managment.
**
**--------------------------------------------------------------------------*/
void
ApplicationControllerUpdate(
  BYTE bStatus,     /*IN  Status event */
  BYTE bNodeID,     /*IN  Node id of the node that send node info */
  BYTE* pCmd,       /*IN  Pointer to Application Node information */
  BYTE bLen)       /*IN  Node info length                        */
{
}
#endif


/*============================   LearnCompleted   ========================
**    Function description
**      Called from learnmode.c when learnmode completed
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void									/*RET	Nothing */
LearnCompleted(
BYTE nodeID)					/* IN resulting nodeID */
{
  if (nodeID == 0)		/* Was it reset? */
  {
    ignoreAllOnOff = SWITCH_ALL_ENABLE_ON_OFF;
    /* Store it in the EEPROM */
    ZW_MEM_PUT_BYTE(EEOFFSET_IGNORE_ALL_ON_OFF, ignoreAllOnOff);
  }
#ifdef DEBUG_STRESS
  if (!debugStress)
  {
    debugStress = TRUE;
    debugStressHandle = ZW_TIMER_START(SendBasicSet, DEBUG_STRESS_TIMEOUT_START, ONE_SHOT);
  }
#endif
}
