/******************************* ZW_uart_api.h  ****************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Products that speak Z-Wave work together better
 *
 *              Copyright (c) 2008
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 *
 * Description: Interface driver for the 400 series built-in UART's.
 *
 * Author:   Morten Vested Olesen
 *
 * Last Changed By:  $Author: jsi $
 * Revision:         $Revision: 21456 $
 * Last Changed:     $Date: 2011-09-30 10:24:08 +0200 (Fri, 30 Sep 2011) $
 *
 ****************************************************************************/
#ifndef _ZW_UART_API_H_
#define _ZW_UART_API_H_
/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
#include <zw_typedefs.h>

/****************************************************************************/
/*                     EXPORTED TYPES and DEFINITIONS                       */
/****************************************************************************/
#ifdef ZW040x

/* Macroes for debug output */
#define ZW_DEBUG_BAUD_RATE    2304

#ifdef ZW_DEBUG
#define ZW_DEBUG_INIT(baud)       ZW_UART1_init(baud, TRUE, FALSE)
#define ZW_DEBUG_SEND_BYTE(bData) ZW_UART1_tx_send_byte(bData)
#define ZW_DEBUG_SEND_NUM(bData)  ZW_UART1_tx_send_num(bData)
#define ZW_DEBUG_SEND_WORD_NUM(bData)  ZW_UART1_tx_send_w_num(bData)
#define ZW_DEBUG_SEND_NL()        ZW_UART1_tx_send_nl()
#define ZW_DEBUG_SEND_STR(STR)    ZW_UART1_tx_send_str(STR)
#define ZW_DEBUG_TX_STATUS()      ZW_UART1_tx_active_get()
#else
#define ZW_DEBUG_INIT(baud)
#define ZW_DEBUG_SEND_BYTE(bData)
#define ZW_DEBUG_SEND_NUM(bData)
#define ZW_DEBUG_SEND_WORD_NUM(bData)
#define ZW_DEBUG_SEND_NL()
#define ZW_DEBUG_SEND_STR(STR)
#define ZW_DEBUG_TX_STATUS()
#endif /* ZW_DEBUG */

#define ZW_UART0_INIT(baud)        ZW_UART0_init(baud, TRUE, TRUE)
#define ZW_UART0_SEND_BYTE(bData)  ZW_UART0_tx_send_byte(bData)
#define ZW_UART0_SEND_NUM(bData)   ZW_UART0_tx_send_num(bData)
#define ZW_UART0_REC_STATUS        (ZW_UART0_rx_int_get())
#define ZW_UART0_REC_BYTE          (ZW_UART0_rx_data_wait_get())
#define ZW_UART0_SEND_NL()         ZW_UART0_tx_send_nl()
#define ZW_UART0_SEND_STATUS       (ZW_UART0_tx_active_get())
#define ZW_UART0_SEND_STR(STR)     (ZW_UART0_tx_send_str(STR))

#define ZW_UART_INIT(baud)        ZW_UART1_init(baud, TRUE, FALSE)
#define ZW_UART_SEND_BYTE(bData)  ZW_UART1_tx_send_byte(bData)
#define ZW_UART_SEND_NUM(bData)   ZW_UART1_tx_send_num(bData)
#define ZW_UART_REC_STATUS        (ZW_UART1_rx_int_get())
#define ZW_UART_REC_BYTE          (ZW_UART1_rx_data_wait_get())
#define ZW_UART_SEND_NL()         ZW_UART1_tx_send_nl()
#define ZW_UART_SEND_STATUS       (ZW_UART1_tx_int_get())
#define ZW_UART_SEND_STR(STR)     (ZW_UART1_tx_send_str(STR))


/*===============================   ZW_UART0_init   =============================
**  Initializes UART0.
**  Optionally enables transmit and/or receive, clears the rx and tx interrupt
**  flags, and sets the specified baudrate.
**
**  Side effects:
**--------------------------------------------------------------------------*/
void             /*RET Nothing */
ZW_UART0_init(
  WORD bBaudRate,  /* IN  Baud rate / 100 (e.g. 96 => 9600baud/s, 1152 => 115200baud/s) */
  BOOL bEnableTx,  /* IN  TRUE: Tx enabled, FALSE: Tx disabled */
  BOOL bEnableRx); /* IN  TRUE: Rx enabled, FALSE: Rx disabled */

/*===============================   ZW_UART0_zm4102_mode_enable   ===============================
**  Enable\ Disable UART0 ZM4102 compatibility mode.
**  When compatibility mode is enabled Tx0 is mapped to IO P35 and Rx0 is mapped to IO P3.4
**  When compatibility mode is disabled Tx0 is mapped to IO P21 and Rx0 is mapped to IO P20
**    Side effects: none.
**---------------------------------------------------------------------------------------------*/
void
ZW_UART0_zm4102_mode_enable(
  BYTE bState);  /* IN, TRUE to set the UART0 in ZM4102 compatibility mode,*/
                /*     FALSE to disable ZM4102 compatibility mode*/

/*===============================   ZW_UART0_rx_int_clear   ========================
**  Clear the UART0 Rx interrupt flag
**
**    Side effects:
**--------------------------------------------------------------------------*/
void
ZW_UART0_rx_int_clear(void);

/*=============================   ZW_UART0_tx_int_clear   ========================
**  Clear the UART0 Tx interrupt flag
**
**    Side effects:
**--------------------------------------------------------------------------*/
void
ZW_UART0_tx_int_clear(void);

/*===============================   ZW_UART0_rx_data_wait_get   ========================
**  Read the content of the UART0 receive register
**
**    Side effects:
**--------------------------------------------------------------------------*/
BYTE    /* RET the content of the receive register*/
ZW_UART0_rx_data_wait_get(void);

/*===============================   ZW_UART0_rx_data_get   ========================
**  Read the content of the UART0 receive register
**
**    Side effects:
**--------------------------------------------------------------------------*/

BYTE    /* RET the content of the receive register*/
ZW_UART0_rx_data_get(void);

/*===============================   ZW_UART0_tx_data_set   ========================
**  Write data byte to UART0 transmit register
**
**    Side effects:
**--------------------------------------------------------------------------*/
void
ZW_UART0_tx_data_set(
  BYTE txByte);  /* IN a byte to written to the UART transmit register.*/

/*===============================   ZW_UART0_tx_send_byte   ========================
**  Wait until UART0 Tx is idle, then write data byte to UART0 transmit register
**
**    Side effects: waits until UART0 Tx is idle
**--------------------------------------------------------------------------*/
void
ZW_UART0_tx_send_byte(
  BYTE bData);  /* IN a byte to written to the UART transmit register.*/

/*===============================   ZW_UART0_rx_enable   ========================
**  Enable the UART receiver and reserve IO.
**
**    Side effects:
**--------------------------------------------------------------------------*/
void
ZW_UART0_rx_enable(
  BYTE bState); /* IN  TRUE: enables UART0 rx function, FALSE: disables UART0 rx function */

/*===============================   ZW_UART0_tx_enable   ========================
**  Enable the UART transmitter and reserve IO.
**
**    Side effects:
**--------------------------------------------------------------------------*/
void
ZW_UART0_tx_enable(
  BYTE bState); /* IN  TRUE: enables UART0 tx function, FALSE: disables UART0 tx function */


/*===============================   ZW_UART0_tx_int_get   ========================
**  This function checks if the UART0 has sent a byte.
**
**    Side effects:
**--------------------------------------------------------------------------*/

BYTE    /* RET zero (0x00): tx interrupt flag is not set,
         *     non-zero:  : tx interrupt flag is set      */
ZW_UART0_tx_int_get(void);

/*===============================   ZW_UART0_tx_active_get   ========================
**  This function checks if the UART0 is sending.
**
**    Side effects:
**--------------------------------------------------------------------------*/

BYTE    /* RET zero (0x00): UART0 tx is idle,
         *     non-zero:  : UART0 tx is active     */
ZW_UART0_tx_active_get(void);

/*===============================   ZW_UART0_rx_int_get   ========================
**  This function checks if the UART0 has received a byte.
**
**    Side effects:
**--------------------------------------------------------------------------*/
BYTE    /* RET zero (0x00): rx interrupt flag is not set,
         *     non-zero:  : rx interrupt flag is set      */
ZW_UART0_rx_int_get(void);


/*===========================  ZW_UART0_tx_send_num  ==========================
**  Converts a byte to a two-byte hexadecimal ASCII representation,
**  and transmits it over UART0.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void            /*RET Nothing */
ZW_UART0_tx_send_num(
  BYTE bData);   /* IN data to send */

/*============================   ZW_UART0_tx_send_str   ========================
**  Transmit a null terminated string over UART0.
**  The null data is not transmitted.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void          /*RET Nothing */
ZW_UART0_tx_send_str(
  BYTE_P str); /* IN String pointer */

/*=============================   ZW_UART0_send_nl   =========================
**  Transmit CR + LF over UART0.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                /*RET Nothing */
ZW_UART0_tx_send_nl( void ); /*IN Nothing */


/*===============================   ZW_UART1_init   =============================
**  Initializes UART1.
**  Optionally enables transmit and/or receive, clears the rx and tx interrupt
**  flags, and sets the specified baudrate.
**
**  Side effects:
**--------------------------------------------------------------------------*/
void             /*RET Nothing */
ZW_UART1_init(
  WORD bBaudRate,  /* IN  Baud rate / 100 (e.g. 96 => 9600baud/s, 1152 => 115200baud/s) */
  BOOL bEnableTx,  /* IN  TRUE: Tx enabled, FALSE: Tx disabled */
  BOOL bEnableRx); /* IN  TRUE: Rx enabled, FALSE: Rx disabled */

/*===============================   ZW_UART1_rx_int_clear   ========================
**  Clear the UART1 Rx interrupt flag
**
**    Side effects:
**--------------------------------------------------------------------------*/
void
ZW_UART1_rx_int_clear(void);

/*=============================   ZW_UART1_tx_int_clear   ========================
**  Clear the UART1 Tx interrupt flag
**
**    Side effects:
**--------------------------------------------------------------------------*/
void
ZW_UART1_tx_int_clear(void);

/*===============================   ZW_UART1_rx_data_wait_get   ========================
**  Read the content of the UART1 receive register
**
**    Side effects:
**--------------------------------------------------------------------------*/
BYTE    /* RET the content of the receive register*/
ZW_UART1_rx_data_wait_get(void);

/*===============================   ZW_UART1_rx_data_get   ========================
**  Read the content of the UART1 receive register
**
**    Side effects:
**--------------------------------------------------------------------------*/

BYTE    /* RET the content of the receive register*/
ZW_UART1_rx_data_get(void);

/*===============================   ZW_UART1_tx_data_set   ========================
**  Write data byte to UART1 transmit register
**
**    Side effects:
**--------------------------------------------------------------------------*/
void
ZW_UART1_tx_data_set(
  BYTE txByte);  /* IN a byte to written to the UART transmit register.*/

/*===============================   ZW_UART1_tx_send_byte   ========================
**  Wait until UART1 Tx is idle, then write data byte to UART1 transmit register
**
**    Side effects: waits until UART1 Tx is idle
**--------------------------------------------------------------------------*/
void
ZW_UART1_tx_send_byte(
  BYTE bData);  /* IN a byte to written to the UART transmit register.*/

/*===============================   ZW_UART1_rx_enable   ========================
**  Enable the UART receiver and reserve IO.
**
**    Side effects:
**--------------------------------------------------------------------------*/
void
ZW_UART1_rx_enable(
  BYTE bState); /* IN  TRUE: enables UART1 rx function, FALSE: disables UART1 rx function */

/*===============================   ZW_UART1_tx_enable   ========================
**  Enable the UART transmitter and reserve IO.
**
**    Side effects:
**--------------------------------------------------------------------------*/
void
ZW_UART1_tx_enable(
  BYTE bState); /* IN  TRUE: enables UART1 tx function, FALSE: disables UART1 tx function */


/*===============================   ZW_UART1_tx_int_get   ========================
**  This function checks if the UART1 has sent a byte.
**
**    Side effects:
**--------------------------------------------------------------------------*/

BYTE    /* RET zero (0x00): tx interrupt flag is not set,
         *     non-zero:  : tx interrupt flag is set      */
ZW_UART1_tx_int_get(void);

/*===============================   ZW_UART1_tx_active_get   ========================
**  This function checks if the UART1 is sending.
**
**    Side effects:
**--------------------------------------------------------------------------*/

BYTE    /* RET zero (0x00): UART1 tx is idle,
         *     non-zero:  : UART1 tx is active     */
ZW_UART1_tx_active_get(void);


/*===============================   ZW_UART1_rx_int_get   ========================
**  This function checks if the UART1 has received a byte.
**
**    Side effects:
**--------------------------------------------------------------------------*/
BYTE    /* RET zero (0x00): rx interrupt flag is not set,
         *     non-zero:  : rx interrupt flag is set      */
ZW_UART1_rx_int_get(void);


/*===========================  ZW_UART1_tx_send_num  ==========================
**  Converts a byte to a two-byte hexadecimal ASCII representation,
**  and transmits it over UART1.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void            /*RET Nothing */
ZW_UART1_tx_send_num(
  BYTE bData);   /* IN data to send */

/*===========================  ZW_UART1_tx_send_num  ==========================
**  Converts a WORD to a 4-byte hexadecimal ASCII representation,
**  and transmits it over UART1.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void            /*RET Nothing */
ZW_UART1_tx_send_w_num(
  WORD bData);   /* IN data to send */

/*============================   ZW_UART1_tx_send_str   ========================
**  Transmit a null terminated string over UART1.
**  The null data is not transmitted.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void          /*RET Nothing */
ZW_UART1_tx_send_str(
  BYTE_P str); /* IN String pointer */

/*=============================   ZW_UART1_send_nl   =========================
**  Transmit CR + LF over UART1.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                /*RET Nothing */
ZW_UART1_tx_send_nl( void ); /*IN Nothing */


#endif /*ZW040x*/

/*===============================   nibble2hex   ============================
**    Convert bit 0-3 to a ASCII hex value.
**
**
**--------------------------------------------------------------------------*/
static BYTE     /*RET ASCII hax value */
nibble2hex(
  BYTE bData);   /* IN Data nibble */



#endif /* _ZW_UART_API_H_ */
