using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.Programmer.Classes;
using Zensys.ZWave.Enums;
using System.IO;
using Zensys.Framework;
using Zensys.ZWave.Devices;

namespace Zensys.ZWave.Programmer.Models
{
    /// <summary>
    /// DocumentModel class. Implements application model class.
    /// </summary>
    public class DocumentModel
    {
        /// <summary>
        /// Occurs when Document model state changed.
        /// </summary>
        public event EventHandler<EventArgs> DocumentModelStateChanged;

        /// <summary>
        /// Initializes a new instance of the <see cref="DocumentModel"/> class.
        /// </summary>
        public DocumentModel()
        {
            InitEmptyHexFileNameDictionaries();
        }
        private Win32PnPEntityClass mPortInfo;
        /// <summary>
        /// Gets or sets the Port Info.
        /// </summary>
        /// <value>The port info.</value>
        public Win32PnPEntityClass PortInfo
        {
            get { return mPortInfo; }
            set
            {
                mPortInfo = value;
                SetDocumentModelStateChanged();
            }
        }

        private bool mIsAsic = true;
        /// <summary>
        /// Gets or sets a value indicating whether application is in ASIC mode.
        /// </summary>
        /// <value><c>true</c> if application is in ASIC mode; otherwise, <c>false</c>.</value>
        public bool IsAsic
        {
            get { return mIsAsic; }
            set { mIsAsic = value; }
        }

        private ProgrammerSettings mSettings;
        /// <summary>
        /// Gets or sets the Settings.
        /// </summary>
        /// <value>The Settings.</value>
        public ProgrammerSettings Settings
        {
            get
            {
                if (mSettings == null)
                {
                    mSettings = new ProgrammerSettings();
                    mSettings.Load();

                }
                return mSettings;
            }
            set
            {
                mSettings = value;
            }
        }
        private byte mSelectedChipType;
        /// <summary>
        /// Gets or sets the type of the Selected chip.
        /// </summary>
        /// <value>The type of the Selected chip.</value>
        public byte SelectedChipType
        {
            get { return mSelectedChipType; }
            set
            {
                mSelectedChipType = value;
                SetDocumentModelStateChanged();
            }
        }

        private IDevice mDevice = null;
        public IDevice Device
        {
            get { return mDevice; }
			set { mDevice = value; }
        }

		/// <summary>
		/// Is set to true by ControllerManager.DeviceOpen to indicate, that device is busy by some operation.
		/// </summary>
		public bool DeviceBusy = false;

		/// <summary>
		/// Is set to true by button press event handler if Program operation requestet. Reseted to false by Program operation.
		/// </summary>
		public bool DeviceProgramButtonPressed = false;


        private void InitEmptyHexFileNameDictionaries()
        {
            mEepromHexFileNames = new Dictionary<ChipTypes, string>();
            mFlashHexFileNames = new Dictionary<ChipTypes, string>();
            foreach (byte i in Enum.GetValues(typeof(ChipTypes)))
            {
                switch ((ChipTypes)i)
                {
                    case ChipTypes.ZW010x:
                        {
                            mEepromHexFileNames.Add((ChipTypes)i, Properties.Settings.Default.ZW010xEepromHexFileName);
                            mFlashHexFileNames.Add((ChipTypes)i, Properties.Settings.Default.ZW010xFlashHexFileName);
                        } break;
                    case ChipTypes.ZW020x:
                        {
                            mEepromHexFileNames.Add((ChipTypes)i, Properties.Settings.Default.ZW020xEepromHexFileName);
                            mFlashHexFileNames.Add((ChipTypes)i, Properties.Settings.Default.ZW020xFlashHexFileName);
                        } break;
                    case ChipTypes.ZW030x:
                        {
                            mEepromHexFileNames.Add((ChipTypes)i, Properties.Settings.Default.ZW030xEepromHexFileName);
                            mFlashHexFileNames.Add((ChipTypes)i, Properties.Settings.Default.ZW030xFlashHexFileName);
                        } break;
                    case ChipTypes.ZW040x:
                        {
                            mEepromHexFileNames.Add((ChipTypes)i, Properties.Settings.Default.ZW040xEepromHexFileName);
                            mFlashHexFileNames.Add((ChipTypes)i, Properties.Settings.Default.ZW040xFlashHexFileName);
                            mSramHexFileNames.Add((ChipTypes)i, Properties.Settings.Default.ZW040xSramHexFileName);
							mMtpHexFileNames.Add((ChipTypes)i, Properties.Settings.Default.ZW040xMtpHexFileName);
                        } break;
                    default:
                        {
                            mEepromHexFileNames.Add((ChipTypes)i, "");
                            mFlashHexFileNames.Add((ChipTypes)i, "");
                        } break;
                }

            }
        }

        private Dictionary<ChipTypes, string> mEepromHexFileNames = new Dictionary<ChipTypes, string>();
        private Dictionary<ChipTypes, string> mFlashHexFileNames = new Dictionary<ChipTypes, string>();
		private Dictionary<ChipTypes, string> mSramHexFileNames = new Dictionary<ChipTypes, string>();
		private Dictionary<ChipTypes, string> mMtpHexFileNames = new Dictionary<ChipTypes, string>();

        internal void SetFlashHexFileName(ChipTypes chipType, string flashHexFileName)
        {
            if (!String.IsNullOrEmpty(flashHexFileName))
            {
                mFlashHexFileNames[chipType] = flashHexFileName;
                switch (chipType)
                {
                    case ChipTypes.ZW010x:
                        {
                            Properties.Settings.Default.ZW010xFlashHexFileName = flashHexFileName;
                        } break;
                    case ChipTypes.ZW020x:
                        {
                            Properties.Settings.Default.ZW020xFlashHexFileName = flashHexFileName;
                        } break;
                    case ChipTypes.ZW030x:
                        {
                            Properties.Settings.Default.ZW030xFlashHexFileName = flashHexFileName;
                        } break;
                    case ChipTypes.ZW040x:
                        {
                            Properties.Settings.Default.ZW040xFlashHexFileName = flashHexFileName;
                        } break;
                }
                Properties.Settings.Default.Save();
                SetDocumentModelStateChanged();
            }
        }
        internal string GetFlashHexFilePath(ChipTypes chipType)
        {
            return mFlashHexFileNames[chipType];
        }
        internal string GetFlashHexFileName(ChipTypes chipType)
        {
            if (Properties.Settings.Default.ShowHexFilesPath != true)
            {
                return Path.GetFileName(mFlashHexFileNames[chipType]);
            }
            else
            {
                return mFlashHexFileNames[chipType];
            }

        }

        internal void SetEepromHexFileName(ChipTypes chipType, string eepromHexFileName)
        {
            if (!String.IsNullOrEmpty(eepromHexFileName))
            {
                mEepromHexFileNames[chipType] = eepromHexFileName;
                switch (chipType)
                {
                    case ChipTypes.ZW010x:
                        {
                            Properties.Settings.Default.ZW010xEepromHexFileName = eepromHexFileName;
                        } break;
                    case ChipTypes.ZW020x:
                        {
                            Properties.Settings.Default.ZW020xEepromHexFileName = eepromHexFileName;
                        } break;
                    case ChipTypes.ZW030x:
                        {
                            Properties.Settings.Default.ZW030xEepromHexFileName = eepromHexFileName;
                        } break;
                    case ChipTypes.ZW040x:
                        {
                            Properties.Settings.Default.ZW040xEepromHexFileName = eepromHexFileName;
                        } break;
                }
                Properties.Settings.Default.Save();
                SetDocumentModelStateChanged();
            }
        }
        internal string GetEepromHexFilePath(ChipTypes chipType)
        {
            return mEepromHexFileNames[chipType];
        }
        internal string GetEepromHexFileName(ChipTypes chipType)
        {
            if (Properties.Settings.Default.ShowHexFilesPath != true)
            {
                return Path.GetFileName(mEepromHexFileNames[chipType]);
            }
            else
            {
                return mEepromHexFileNames[chipType];
            }
        }

        internal void SetSramHexFileName(ChipTypes chipType, string sramHexFileName)
        {
            if (!String.IsNullOrEmpty(sramHexFileName))
            {
                mSramHexFileNames[chipType] = sramHexFileName;
                switch (chipType)
                {
                    case ChipTypes.ZW040x:
                        {
                            Properties.Settings.Default.ZW040xSramHexFileName = sramHexFileName;
                        } break;
                }
                Properties.Settings.Default.Save();
                SetDocumentModelStateChanged();
            }
        }
		internal string GetSramHexFilePath(ChipTypes chipType)
		{
			return mSramHexFileNames[chipType];
		}
		internal string GetSramHexFileName(ChipTypes chipType)
        {
            if (Properties.Settings.Default.ShowHexFilesPath != true)
            {
                return Path.GetFileName(mSramHexFileNames[chipType]);
            }
            else
            {
                return mSramHexFileNames[chipType];
            }
        }

		internal void SetMtpHexFileName(ChipTypes chipType, string mtpHexFileName)
		{
			if (!String.IsNullOrEmpty(mtpHexFileName))
			{
				mMtpHexFileNames[chipType] = mtpHexFileName;
				switch (chipType)
				{
					case ChipTypes.ZW040x:
						{
							Properties.Settings.Default.ZW040xMtpHexFileName = mtpHexFileName;
						} break;
				}
				Properties.Settings.Default.Save();
				SetDocumentModelStateChanged();
			}
		}
		internal string GetMtpHexFilePath(ChipTypes chipType)
		{
			return mMtpHexFileNames[chipType];
		}
		internal string GetMtpHexFileName(ChipTypes chipType)
		{
			if (Properties.Settings.Default.ShowHexFilesPath != true)
			{
				return Path.GetFileName(mMtpHexFileNames[chipType]);
			}
			else
			{
				return mMtpHexFileNames[chipType];
			}
		}

		private void OnDocumentModelStateChanged(object sender, EventArgs e)
        {
            SetDocumentModelStateChanged();
        }

        private void SetDocumentModelStateChanged()
        {
            if (DocumentModelStateChanged != null)
                DocumentModelStateChanged(this, EventArgs.Empty);
        }



        internal bool FlashHexFileSelected(ChipTypes chipType)
        {
            return !String.IsNullOrEmpty(GetFlashHexFileName(chipType));
        }

        internal bool EepromHexFileSelected(ChipTypes chipType)
        {
            return !String.IsNullOrEmpty(GetEepromHexFileName(chipType));
        }

		internal bool SramHexFileSelected(ChipTypes chipType)
		{
			return !String.IsNullOrEmpty(GetSramHexFileName(chipType));
		}

		internal bool MtpHexFileSelected(ChipTypes chipType)
		{
			return !String.IsNullOrEmpty(GetMtpHexFileName(chipType));
		}
	}
}
