using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using System.Collections;
using Zensys.Framework;
using Zensys.ZWave.Logging;
using Zensys.ZWave.Logging.Events;

namespace Zensys.ZWave.Programmer.Controls
{
    /// <summary>
    /// ListDataView class. Implements the data bound ListView control.
    /// </summary>
    public class ListDataView : ListView
    {
        private LogDataSource dataSource;
        private void BuildList()
        {
            this.Items.Clear();
            if (dataSource.SelectedSession != null)
            {
                foreach (LogPacket topRow in dataSource.SelectedSession.Packets)
                {
                    ListViewItem topNode = new ListViewItem();
                    topNode.Tag = topRow;
                    topNode.Text = Tools.ToHexString(topRow.Payload, " ");
                    if (topRow.IsPut)
                    {
                        topNode.ImageIndex = 2;
                        topNode.StateImageIndex = 2;
                    }
                    else
                    {
                        topNode.ImageIndex = 3;
                        topNode.StateImageIndex = 3;
                    }
                    topNode.SubItems.Add(new ListViewItem.ListViewSubItem(topNode, String.Format("{0} {1}", DateTime.Now.ToShortDateString(), DateTime.Now.ToShortTimeString())));
                    Items.Add(topNode);
                }
            }
        }

        /// <summary>
        /// Gets the row.
        /// </summary>
        /// <param name="item">The item.</param>
        /// <returns></returns>
        public LogSession GetRow(ListViewItem item)
        {
            return item.Tag as LogSession;
        }

        /// <summary>
        /// Selects the row.
        /// </summary>
        /// <param name="UID">The UID.</param>
        public void SelectNode(Guid UID)
        {
            foreach (ListViewItem treeNode in Items)
            {
                if (GetRow(treeNode).Id == UID)
                {
                    if (SelectedItems.Count == 0 || treeNode != SelectedItems[0])
                    {
                        treeNode.Selected = true;
                        break;
                    }
                }
            }
        }

        /// <summary>
        /// Gets or sets the Data Source.
        /// </summary>
        /// <value>The Data Source.</value>
        [Category("Data")]
        public LogDataSource DataSource
        {
            get
            {
                return dataSource;
            }
            set
            {
                if(value!=null)
                {
                    dataSource = value;
                    dataSource.PacketDataChanged += new EventHandler<PacketDataChangedEventArgs>(dataSource_PacketDataChanged);
                    dataSource.SelectedSessionChanged += new EventHandler<SelectedSessionChangedEventArgs>(dataSource_SelectedSessionChanged);
                }
            }
        }

        void dataSource_SelectedSessionChanged(object sender, SelectedSessionChangedEventArgs e)
        {
            ReBuildList();
        }

        void dataSource_PacketDataChanged(object sender, PacketDataChangedEventArgs e)
        {
            ReBuildList();
        }

        private delegate void ReBuildListDelegate();
        private void ReBuildList()
        {
            if (InvokeRequired)
            {
                Invoke(new ReBuildListDelegate(ReBuildList));
            }
            else
            {
                BuildList();
            }
        }
    }
}
