using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.Programmer.Controllers;
using Zensys.ZWave.Programmer.Properties;
using Zensys.ZWave.Enums;
using Zensys.ZWave.Devices;
using Zensys.ZWave.Programmer.UI;
using Zensys.Framework;
using System.Globalization;
using System.IO;
using System.Diagnostics;
using Zensys.ZWave.SerialPortApplication.Devices;

namespace Zensys.ZWave.Programmer.Classes
{
    /// <summary>
    /// ProgrammerConsole class. Contains methods for parsing command line arguments.
    /// </summary>
    public class ProgrammerConsole
    {
        private ControllerManager mControllerManager;
        /// <summary>
        /// Initializes a new instance of the <see cref="ProgrammerConsole"/> class.
        /// </summary>
        /// <param name="controller">The controller.</param>
        public ProgrammerConsole(ControllerManager controller)
        {
            mControllerManager = controller;
            //NativeMethods.AttachConsole(NativeMethods.ATTACH_PARENT_PROCESS);
        }

        private void ShowCaption()
        {
            Console.WriteLine(String.Format("{0} v.{1}", AboutBox.AssemblyProduct, AboutBox.AssemblyVersion));
        }

        /// <summary>
        /// Usage method. Show help in console window.
        /// </summary>
        public void Usage()
        {                  //         1         2         3         4         5         6         7         8         9        10        11     
            //12345678901234567890123456789012345678901234567890123456789012345678901234567890123456789012345678901234567890
            Console.WriteLine(@"USAGE: ZWaveProgrammer.exe -c COMPORT 
  [-t chipType] 
  [-pf frequency] 
  [-p filename] 
  [-f filename] 
  [-r filename] 
  [-v filename] 
  [-e] 
  [-ls lockbitsbyte]
  [-lg]
  [-ro]
  [-so normalpowerbyte lowpowerbyte]
  [-sr filename] 
  [-sw filename] 
  [-sc filename] 
  [-swrd filename] 
  [-sweo filename] 
  [-sro]
  [-sso normalpowerch0byte lowpowerch0byte normalpowerch1byte lowpowerch1byte normalpowerch2byte lowpowerch2byte]
  [-mp filename] 
  [-mp 0] 
  [-mr filename] 
  [-pe filename] 
  [-pe 0] 
  [-pr filename] 
  [-ph HomeID]");
            Console.WriteLine(" filename must be supplied where noted");
            Console.WriteLine("");
            Console.WriteLine(" -c  - The comport to use in the format COMx: where x is number.");
            Console.WriteLine(" -t  - The Chip Type (ZW010x, ZW020x, ZW030x, ZW040x).");
            Console.WriteLine("");
            Console.WriteLine(" Program Memory Programming options:");
            Console.WriteLine(" -p  - Erase chip (if supported), Program and verify pages that are not empty.");
            Console.WriteLine(" -f  - Erase chip (if supported), Program chip");
            Console.WriteLine(" -r  - Reads the flash contents and writes it to the file specified.");
            Console.WriteLine(" -v  - Verify chip against file supplied.");
            Console.WriteLine(" -e  - Erase chip (if supported).");
            Console.WriteLine(" -ls - Set lockbits.");
            Console.WriteLine(" -lg - Get lockbits.");
            Console.WriteLine(" -pf - Set the frequency (EU, US, ANZ, HK, MY, IN, ...) (used after -p or -f).");
            Console.WriteLine(" -ro - Read options.");
            Console.WriteLine(" -so - Set options.");
            Console.WriteLine("");
            Console.WriteLine(" ZW040x specific programming options:");
            Console.WriteLine(" -sr  - Read the SRAM contents and writes it to the file specified.");
            Console.WriteLine(" -sw  - Write the SRAM contents with the file specified.");
            Console.WriteLine(" -sc  - Compare the SRAM contents with the file specified.");
            Console.WriteLine(@" -swrd  - Write the SRAM contents with the file specified 
          and run Development mode.");
            Console.WriteLine(@" -sweo  - Write the SRAM contents with the file specified 
          and Exec out of SRAM mode.");
            Console.WriteLine(" -sro - SRAM Read options.");
            Console.WriteLine(@" -sso - SRAM Set options (options: 
          [normal power ch0 byte] [low power ch0 byte] [normal power ch1 byte] [low power ch1 byte] [normal power ch2 byte] [low power ch2 byte]).");
            Console.WriteLine("");
            Console.WriteLine(@" -mp - Programs the MTP memory with the file specified. 
       Zero fill empty areas.");
            Console.WriteLine(@" -mp 0 - Erase entire MTP memory.");
            Console.WriteLine(" -mr - Reads MTTP memory to file specified.");
            Console.WriteLine("");
            Console.WriteLine(" External Non-Volatile Memory (NVM) Programming options:");
            Console.WriteLine(@" -pe - Programs the external NVM with the file specified. 
       Zero fill empty areas.");
            Console.WriteLine(@" -pe 0 - Erase entire external NVM.");
            Console.WriteLine(" -pr - Reads extern NVM to file specified.");
            Console.WriteLine(" -ph - Set HomeID of external NVM with the supplied address: e.g. DEADBEEF.");
            Console.WriteLine("");
            Console.WriteLine(Resources.MsgMtpAutoEraseNote);

            //#if DEBUG
            //Console.WriteLine("");
            //Console.WriteLine("Press any key to continue...");

            //Console.ReadKey();
            //#endif
        }





        internal void ProcessCommandLine(string[] args)
        {
            ShowCaption();

            Dictionary<string, string> commands = new Dictionary<string, string>();

            string key = null;
            foreach (string cmd in args)
            {
                if (cmd.StartsWith("-"))
                {
                    key = cmd;
                    commands.Add(cmd, null);
                }
                else
                {
                    if (!String.IsNullOrEmpty(key))
                    {
                        commands[key] += " " + cmd;
                        commands[key] = commands[key].Trim();
                    }
                }
            }
            if (commands.ContainsKey("-c"))
            {
                if (commands.Count == 1)
                {
                    if (ProcessSerialPort(commands["-c"]))
                    {
                        ControllerManager.RunUI();
                    }
                    else
                    {
                        Usage();
                    }
                }
                else if (commands.Count > 1)
                {
                    if (ProcessSerialPort(commands["-c"]))
                    {
                        if (commands.Count == 2 && commands.ContainsKey("-t"))
                        {
                            if (ProcessChipType(commands["-t"]))
                            {
                                ControllerManager.RunUI();
                            }
                            else
                            {
                                Usage();
                            }
                        }
                        else
                        {
                            mControllerManager.DocumentModel.Settings.WriteOnPcbButton = false;
                            if (mControllerManager.Actions.ProgrammerActions.UpgradeLatestFirmware(true) >= 0)
                            {
                                if (commands.ContainsKey("-t"))
                                {
                                    if (ProcessChipType(commands["-t"]))
                                    {
                                        ProcessCommandLine(commands);
                                    }
                                    else
                                    {
                                        Usage();
                                    }
                                }
                                else
                                {
                                    if (mControllerManager.Actions.ProgrammerActions.DetectDevice(true))
                                    {
                                        ProcessCommandLine(commands);
                                    }
                                }
                            }
                        }
                    }
                    else
                    {
                        Usage();
                    }
                }
            }
            else
            {
                Usage();
            }
        }

        private void ProcessCommandLine(Dictionary<string, string> commands)
        {
            foreach (KeyValuePair<string, string> cmd in commands)
            {
                if (cmd.Key != "-c" && cmd.Key != "-t")
                {
                    ProcessCommandArgs(cmd);
                }
            }
            //#if DEBUG
            //Console.WriteLine("Press any key to continue...");
            //Console.ReadKey();
            //#endif
        }

        private void ProcessCommandArgs(KeyValuePair<string, string> command)
        {
            if (String.IsNullOrEmpty(command.Value) &&
                command.Key != "-e" && command.Key != "-lg" && command.Key != "-ro" &&
                command.Key != "-sro")
            {
                ShowParameterNotSpecifiedMessage(command.Key);
                return;
            }
            if (command.Key == "-pf")
            {
                int frequency = -1;
                if (command.Value == "US") frequency = MemoryLayout.RF_US;
                if (command.Value == "EU") frequency = MemoryLayout.RF_EU;
                if (command.Value == "ANZ") frequency = MemoryLayout.RF_ANZ;
                if (command.Value == "HK") frequency = MemoryLayout.RF_HK;
                if (command.Value == "MY") frequency = MemoryLayout.RF_MY;
                if (command.Value == "IN") frequency = MemoryLayout.RF_IN;
                if (command.Value == "RU") frequency = MemoryLayout.RF_RU;
                if (command.Value == "866") frequency = MemoryLayout.RF_866;
                if (command.Value == "870") frequency = MemoryLayout.RF_870;
                if (command.Value == "906") frequency = MemoryLayout.RF_906;
                if (command.Value == "910") frequency = MemoryLayout.RF_910;
                if (frequency != -1)
                {
                    IFlashSettings flashSettings = new FlashSettings();
                    flashSettings.Frequency = (byte)frequency;
                    mControllerManager.Actions.ProgrammerActions.FlashWriteOptions(true, flashSettings);
                }
            }
            else if (command.Key == "-p")
            {
                if (Path.GetExtension(command.Value).ToLower() == ".hex")
                {
                    mControllerManager.Actions.ProgrammerActions.FlashProgram(true, command.Value, false);
                }
                else
                {
                    Console.WriteLine(String.Format("Invalid file extension in file '{0}'.", command.Value));
                }
            }
            else if (command.Key == "-f")
            {
                mControllerManager.Actions.ProgrammerActions.FlashWrite(true, command.Value, false);
            }
            else if (command.Key == "-r")
            {
                mControllerManager.Actions.ProgrammerActions.FlashRead(true, command.Value);
            }
            else if (command.Key == "-v")
            {
                mControllerManager.Actions.ProgrammerActions.FlashCompare(true, command.Value);
            }
            else if (command.Key == "-e")
            {
                mControllerManager.Actions.ProgrammerActions.FlashErase(true);
            }
            else if (command.Key == "-pe")
            {
                if (command.Value == "0")
                {
                    mControllerManager.Actions.ProgrammerActions.EEPROMErase(true);
                }
                else
                {
                    mControllerManager.Actions.ProgrammerActions.EEPROMWrite(true, command.Value);
                }
            }
            else if (command.Key == "-pr")
            {
                mControllerManager.Actions.ProgrammerActions.EEPROMRead(true, command.Value);
            }
            else if (command.Key == "-ph")
            {
                mControllerManager.Actions.ProgrammerActions.WriteHomeId(true, command.Value);
            }
            else if (command.Key == "-sr")
            {
                mControllerManager.Actions.ZW040xFormActions.SramRead(true, command.Value, WorkingModes.Development);
            }
            else if (command.Key == "-sw")
            {
                mControllerManager.Actions.ZW040xFormActions.SramWrite(true, command.Value, WorkingModes.Development);
            }
            else if (command.Key == "-sc")
            {
                mControllerManager.Actions.ZW040xFormActions.SramCompare(true, command.Value, WorkingModes.Development);
            }
            else if (command.Key == "-swrd")
            {
                mControllerManager.Actions.ZW040xFormActions.SramWriteAndRunMode(true, command.Value, WorkingModes.Development);
            }
            else if (command.Key == "-sweo")
            {
                mControllerManager.Actions.ZW040xFormActions.SramWriteAndRunMode(true, command.Value, WorkingModes.ExecuteOutOfSram);
            }
            else if (command.Key == "-ls")
            {
                mControllerManager.Actions.ProgrammerActions.LockBitsSet(byte.Parse(command.Value, NumberStyles.HexNumber), true);
            }
            else if (command.Key == "-lg")
            {
                Console.WriteLine(String.Format("LockBits: {0}",
                    mControllerManager.Actions.ProgrammerActions.LockBitsRead(true).ToString("X2")));
            }
            else if (command.Key == "-mp")
            {
                if (command.Value == "0")
                {
                    mControllerManager.Actions.ZW040xFormActions.MtpErase(true);
                }
                else
                {
                    mControllerManager.Actions.ZW040xFormActions.MtpProgram(true, command.Value);
                }
            }
            else if (command.Key == "-mr")
            {
                mControllerManager.Actions.ZW040xFormActions.MtpRead(true, command.Value);
            }
            else if (command.Key == "-ro")
            {
                ChipTypes deviceChipType = ChipTypes.UNKNOWN;
                IFlashSettings flashSettings = mControllerManager.Actions.ProgrammerActions.FlashReadOptions(true, out deviceChipType);
                if (flashSettings != null)
                {
                    switch (deviceChipType)
                    {
                        case (ChipTypes.ZW020x):
                        case (ChipTypes.ZW030x):
                            {
                                Console.WriteLine("");
                                Console.WriteLine(String.Format("Chip: {0}", deviceChipType.ToString()));
                                Console.WriteLine("------------------");
                                Console.WriteLine(String.Format("Normal Power:{0}", flashSettings.NormalPower.ToString("X2")));
                                Console.WriteLine(String.Format("Low Power:{0}", flashSettings.LowPower.ToString("X2")));
                            } break;
                        case (ChipTypes.ZW040x):
                            {
                                Console.WriteLine("");
                                Console.WriteLine(String.Format("Chip: {0}", deviceChipType.ToString()));
                                Console.WriteLine("------------------");
                                Console.WriteLine(String.Format("Normal Power (ch0):{0}", flashSettings.NormalPowerCh0.ToString("X2")));
                                Console.WriteLine(String.Format("Low Power (ch0):{0}", flashSettings.LowPowerCh0.ToString("X2")));
                                Console.WriteLine(String.Format("Normal Power (ch1):{0}", flashSettings.NormalPowerCh1.ToString("X2")));
                                Console.WriteLine(String.Format("Low Power (ch1):{0}", flashSettings.LowPowerCh1.ToString("X2")));
                                Console.WriteLine(String.Format("Normal Power (ch2):{0}", flashSettings.NormalPowerCh2.ToString("X2")));
                                Console.WriteLine(String.Format("Low Power (ch2):{0}", flashSettings.LowPowerCh2.ToString("X2")));
                            } break;
                    }
                }
            }
            else if (command.Key == "-sro")
            {
                IFlashSettings sramSettings = null;
                ChipTypes deviceChipType = ChipTypes.UNKNOWN;
                mControllerManager.Actions.ZW040xFormActions.ReadSRAMRfOptions(ref sramSettings, ref deviceChipType, false);
                if (deviceChipType == ChipTypes.ZW040x && sramSettings != null)
                {
                    Console.WriteLine("");
                    Console.WriteLine(String.Format("Chip: {0}", deviceChipType.ToString()));
                    Console.WriteLine("------------------");
                    Console.WriteLine(String.Format("Normal Power (ch0):{0}", sramSettings.NormalPowerCh0.ToString("X2")));
                    Console.WriteLine(String.Format("Low Power (ch0):{0}", sramSettings.LowPowerCh0.ToString("X2")));
                    Console.WriteLine(String.Format("Normal Power (ch1):{0}", sramSettings.NormalPowerCh1.ToString("X2")));
                    Console.WriteLine(String.Format("Low Power (ch1):{0}", sramSettings.LowPowerCh1.ToString("X2")));
                    Console.WriteLine(String.Format("Normal Power (ch2):{0}", sramSettings.NormalPowerCh2.ToString("X2")));
                    Console.WriteLine(String.Format("Low Power (ch2):{0}", sramSettings.LowPowerCh2.ToString("X2")));
                }
                else
                {
                    Console.WriteLine(String.Format("Chip: {0}. SRAM options not supported.", deviceChipType.ToString()));
                }
            }
            else if (command.Key == "-so")
            {
                string[] arr = command.Value.Split(' ');
                if (arr.Length == 2)
                {
                    byte npValue = Tools.FromHexString(arr[0])[0];
                    byte lpValue = Tools.FromHexString(arr[1])[0];

                    IFlashSettings flashSettings = new FlashSettings();
                    flashSettings.NormalPower = npValue;
                    flashSettings.LowPower = lpValue;
                    mControllerManager.Actions.ProgrammerActions.FlashWriteOptions(true, flashSettings);
                }

            }
            else if (command.Key == "-sso")
            {
                string[] arr = command.Value.Split(' ');
                if (arr.Length == 6)
                {
                    byte npValueCh0 = 0;
                    byte lpValueCh0 = 0;
                    byte npValueCh1 = 0;
                    byte lpValueCh1 = 0;
                    byte npValueCh2 = 0;
                    byte lpValueCh2 = 0;

                    npValueCh0 = Tools.FromHexString(arr[0])[0];
                    lpValueCh0 = Tools.FromHexString(arr[1])[0];
                    npValueCh1 = Tools.FromHexString(arr[2])[0];
                    lpValueCh1 = Tools.FromHexString(arr[3])[0];
                    npValueCh2 = Tools.FromHexString(arr[4])[0];
                    lpValueCh2 = Tools.FromHexString(arr[5])[0];

                    IFlashSettings flashSettings = new FlashSettings();
                    flashSettings.NormalPowerCh0 = npValueCh0;
                    flashSettings.LowPowerCh0 = lpValueCh0;
                    flashSettings.NormalPowerCh1 = npValueCh1;
                    flashSettings.LowPowerCh1 = lpValueCh1;
                    flashSettings.NormalPowerCh2 = npValueCh2;
                    flashSettings.LowPowerCh2 = lpValueCh2;
                    mControllerManager.Actions.ZW040xFormActions.SramWriteOptions(true, flashSettings);
                }
            }
            else
            {
                Usage();
            }
        }

        private void ShowParameterNotSpecifiedMessage(string command)
        {
            Console.WriteLine(String.Format("{0} Command: {1}.",
                Resources.MsgParameterNotSpecified, command));
            Console.WriteLine("");
            Usage();
        }

        private bool ProcessSerialPort(string serialPort)
        {
            bool result = false;

            Win32PnPEntityClass serialPortInfo = ComputerSystemHardwareHelper.GetWin32PnPEntityClassSerialPortDevice(serialPort);
            if (serialPortInfo != null)
            {
                mControllerManager.DocumentModel.PortInfo = serialPortInfo;
                //Settings.Default.LastUsedDevice = serialPortInfo.DeviceID;
                //Settings.Default.Save();
                result = true;
            }
            else
            {
                Console.WriteLine(Resources.MsgIncorrectSerialPort);
            }

            return result;
        }

        private bool ProcessChipType(string chipType)
        {
            bool result = false;

            if (!String.IsNullOrEmpty(chipType))
            {
                foreach (byte i in Enum.GetValues(typeof(ChipTypes)))
                {
                    if (chipType == Enum.GetName(typeof(ChipTypes), i))
                    {
                        mControllerManager.DocumentModel.SelectedChipType = i;
                        //Settings.Default.SelectedChipType = i;
                        //Settings.Default.Save();
                        result = true;
                        break;
                    }
                }
            }

            return result;
        }
    }
}
