using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.Programmer.Classes;
using Zensys.ZWave.Programmer.Controllers;
using Zensys.ZWave.Devices;
using Zensys.ZWave.Programmer.Properties;
using Zensys.ZWave.Enums;
using System.IO;
using Zensys.Framework;
using Zensys.ZWave.Programmer.UI;
using System.Globalization;
using Zensys.ZWave.Programmer.Controls;
using System.Windows.Forms;
using System.Threading;

namespace Zensys.ZWave.Programmer.Actions
{
    /// <summary>
    /// ProgrammerActions class. Contains main application functions.
    /// </summary>
    public class ProgrammerActions : BaseAction
    {
        /// <summary>
        /// Initializes a new instance of the <see cref="ProgrammerActions"/> class.
        /// </summary>
        /// <param name="controller">The controller.</param>
        public ProgrammerActions(ControllerManager controller)
            : base(controller)
        {

        }
        /// <summary>
        /// Called when Flash Erase click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnFlashEraseClick(object sender, EventArgs e)
        {
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    FlashErase(false);
                }), Resources.MsgEraseFlashProgress, false, 0x00);
            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }
        }
        /// <summary>
        /// Erase the Flash.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        public void FlashErase(bool isCommandPrompt)
        {
            if (isCommandPrompt)
            {
                ControllerManager.ShowMessage(Resources.MsgEraseFlashProgress, false, isCommandPrompt);
            }

            IDevice device = ControllerManager.DeviceOpen();
            try
            {
                if (device.SetProgrammingMode(true))
                {
                    device.ReadSignatureBits();
                    if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
                    {
                        IDeviceLed busyLed = device.Leds.GetByType(LedTypes.Busy);
                        if (busyLed != null)
                        {
                            busyLed.SetState(true);
                            device.Flash.Erase(ControllerManager.DocumentModel.IsAsic);
                            busyLed.SetState(false);
                        }
                    }
                    else
                    {
                        ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
                    }
                    if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
                        device.SetProgrammingMode(false);
                    ControllerManager.ShowMessage(Resources.MsgEraseROMDone, false, isCommandPrompt, !isCommandPrompt);
                }
                else
                {
                    ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
                }
            }
            catch
            {
                ControllerManager.ShowMessage(Resources.ErrorCantEraseFlash, true, isCommandPrompt);
            }
            finally
            {
                ControllerManager.DeviceClose();
            }
        }

        /// <summary>
        /// Called when Flash Read click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnFlashReadClick(object sender, EventArgs e)
        {
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                string flashReadData = "";
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    flashReadData = FlashRead(false, null);
                }), Resources.MsgReadFlashProgress, false, 0x00);

                ControllerManager.ShowConsoleInfo(flashReadData);
            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }

        }

        private string ProgrammingStateByteToString(byte state)
        {
            string result = "";
            if ((state & (byte)ZW040xProgrammingState.CrcBusy) != 0)
                result += "CRC busy,";
            if ((state & (byte)ZW040xProgrammingState.CrcDone) != 0)
                result += "CRC done,";
            if ((state & (byte)ZW040xProgrammingState.CrcFailed) != 0)
                result += "CRC failed,";
            if ((state & (byte)ZW040xProgrammingState.WriteOperationBusy) != 0)
                result += "Write Operation busy,";
            if ((state & (byte)ZW040xProgrammingState.WriteOperationFailed) != 0)
                result += "Write Operation failed,";
            if ((state & (byte)ZW040xProgrammingState.ContOperationRefused) != 0)
                result += "Cont operation refused,";
            if ((state & (byte)ZW040xProgrammingState.DevelopmentModeEnabled) != 0)
                result += "Development mode enabled,";
            if ((state & (byte)ZW040xProgrammingState.ExecSramModeEnabled) != 0)
                result += "Exec SRAM mode enabled,";
            if (result.Length != 0)
            {
                result = result.Insert(0, "(");
                result = result.Remove(result.Length - 1);
                result += ")";
            }
            return result;
        }

        private string ReadProgrammingStatusesString(IDevice device)
        {
            string result = "";
            try
            {
                if (device.ChipType == (byte)ChipTypes.ZW040x)
                {
                    //read the statuses:
                    byte state = device.ReadState();
                    short num = device.ReadWriteOtpStats();
                    result = string.Format(
                        "\nWrite OTP Stats = {0}, State = 0x{1:X2} ", num, state)
                        + ProgrammingStateByteToString(state);
                }
            }
            catch
            {
            }
            return result;
        }

        /// <summary>
        /// Read the flash.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        /// <param name="fileName">Name of the file.</param>
        /// <returns></returns>
        public string FlashRead(bool isCommandPrompt, string fileName)
        {
            string strStatuses = "";
            _isCommandPrompt = isCommandPrompt;
            bool result = false;
            if (isCommandPrompt)
            {
                ControllerManager.ShowMessage(Resources.MsgReadFlashProgress, false, isCommandPrompt);
            }
            string flashReadData = "";
            IDevice device = ControllerManager.DeviceOpen();

            ControllerManager.ChangeProgressInfo(ProgressStatuses.None, 0, 0, _isCommandPrompt);
            try
            {
                if (device.SetProgrammingMode(true))
                {
                    device.ReadSignatureBits();
                    if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
                    {

                        IDeviceLed busyLed = device.Leds.GetByType(LedTypes.Busy);
                        if (busyLed != null)
                        {
                            busyLed.SetState(true);

                            device.Flash.ProgressChanged += new Zensys.ZWave.Events.ProgressChangedEventHandler(OnDeviceMemoryProgressChanged);
                            device.Flash.Read(ControllerManager.DocumentModel.IsAsic);
                            device.Flash.ProgressChanged -= new Zensys.ZWave.Events.ProgressChangedEventHandler(OnDeviceMemoryProgressChanged);

                            strStatuses = ReadProgrammingStatusesString(device);

                            byte[] flashData = device.Flash.GetBytes();
                            flashReadData = HexFileHelper.WriteIntelHexFile(flashData, 0, flashData.Length, 0x10, Constants.BLANK_VALUE);
                            busyLed.SetState(false);
                            result = true;
                        }
                    }
                    else
                    {
                        ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
                    }
                    if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
                        device.SetProgrammingMode(false);
                }
                else
                {
                    ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
                }
            }
            catch
            {
                ControllerManager.ShowMessage(Resources.ErrorCantReadFlash, true, isCommandPrompt);
            }
            finally
            {
                ControllerManager.DeviceClose();
            }
            if (isCommandPrompt)
            {
                if (!String.IsNullOrEmpty(fileName))
                {
                    try
                    {
                        TextWriter writer = new StreamWriter(fileName);
                        writer.Write(flashReadData);
                        writer.Close();
                    }
                    catch (Exception ex)
                    {
                        Console.WriteLine(ex.Message);
                        result = false;
                    }
                }
            }
            if (result)
            {
                ControllerManager.ShowMessage(Resources.MsgReadROMDone + strStatuses, false, isCommandPrompt);
            }
            return flashReadData;
        }

        /// <summary>
        /// Called when Flash Write click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnFlashWriteClick(object sender, EventArgs e)
        {
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    FlashWrite(false, null, false);
                }), Resources.MsgWriteFlashProgress, false, 0x00);

            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }
        }
        /// <summary>
        /// Writes the Flash.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        /// <param name="fileName">Name of the file.</param>
        public void FlashWrite(bool isCommandPrompt, string fileName, bool calibrate)
        {
            WriteFlash(false, isCommandPrompt, fileName, calibrate);
        }

        /// <summary>
        /// Called when Flash Program click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnFlashProgramClick(object sender, EventArgs e)
        {
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    FlashProgram(false, null, false);
                }), Resources.MsgWriteFlashProgress, false, 0x00);

            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }

        }

        /// <summary>
        /// Called when Calibrate and program click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnFlashCalibrateAndProgramClick(object sender, EventArgs e)
        {
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    FlashProgram(false, null, true);
                }), Resources.MsgWriteFlashProgress, false, 0x00);

            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }

        }

        /// <summary>
        /// Programs the Flash.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        /// <param name="fileName">Name of the file.</param>
        public void FlashProgram(bool isCommandPrompt, string fileName, bool calibrate)
        {
            WriteFlash(true, isCommandPrompt, fileName, calibrate);
        }

        /// <summary>
        /// Called when Flash Compare click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/>, application is in command prompt mode.</param>
        public void OnFlashCompareClick(object sender, EventArgs e)
        {
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    FlashCompare(false, null);
                }), Resources.MsgCompareFlashProgress, false, 0x00);

            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }
        }
        /// <summary>
        /// Compare the Flash.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        /// <param name="fileName">Name of the file.</param>
        public void FlashCompare(bool isCommandPrompt, string fileName)
        {
            if (isCommandPrompt)
            {
                ControllerManager.ShowMessage(Resources.MsgCompareFlashProgress, false, isCommandPrompt);
            }
            bool result = false;
            IDevice device = ControllerManager.DeviceOpen();
            try
            {
                if (device.SetProgrammingMode(true))
                {
                    device.ReadSignatureBits();
                    if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
                    {
                        IDeviceLed busyLed = device.Leds.GetByType(LedTypes.Busy);
                        if (busyLed != null)
                        {
                            busyLed.SetState(true);
                            device.Flash.Read(ControllerManager.DocumentModel.IsAsic);
                            string hexFileName;
                            IFlashSettings flashSettings;
                            if (isCommandPrompt)
                            {
                                hexFileName = fileName;
                                //We need to read the flash setting from the Program Memory,
                                // because flash settings can be changed relative the settings in the hex file.
                                // If we not read them, we will always get the error about the Program memory of the
                                // chip is not the same as program memory in the hex.
                                // We read the flash settings from the Program memory of the chip, and not from the
                                // command line, because currently command line command write the settings to the chip
                                // before writting the hex file to the chip. Therefore Program Memory of the chip always
                                // have updated flash setting. TODO: may be this should be changed, because OTP can be
                                // written only once.
                                try
                                {	//flash setting are not exist in empty chip, this would give the exception.
                                    flashSettings = device.Flash.ReadRfOptions();
                                }
                                catch
                                {
                                    //If flash settings cant be read from the chip - just ignore it.
                                    flashSettings = null;
                                }
                            }
                            else
                            {
                                hexFileName = ControllerManager.DocumentModel.GetFlashHexFilePath((ChipTypes)device.ChipType);
                                flashSettings = GetUserFlashSettings((ChipTypes)device.ChipType);
                            }
                            if (!String.IsNullOrEmpty(hexFileName) && File.Exists(hexFileName))
                            {
                                byte[] flashDataRaw = HexFileHelper.BlankArray((int)device.Flash.FlashSize, Constants.BLANK_VALUE);
                                if (HexFileHelper.ReadIntelHexFile(hexFileName, flashDataRaw, (int)device.Flash.FlashSize) == HexFileHelper.FILE_READ)
                                {
                                    IDevice _device = ControllerManager.ZWaveManager.ApplicationLayer.CreateDevice();
                                    _device.ChipType = device.ChipType;
                                    result = _device.Flash.Compare(ControllerManager.DocumentModel.IsAsic, device.Flash, flashSettings, flashDataRaw);
                                }
                            }
                            else if (String.IsNullOrEmpty(hexFileName))
                            {
                                ControllerManager.ShowMessage("Hex file is not selected.", true, isCommandPrompt);
                            }
                            else
                            {
                                ControllerManager.ShowMessage(String.Format("{0} does not exists.", hexFileName), true, isCommandPrompt);
                            }
                            busyLed.SetState(false);
                            if (result)
                            {
                                ControllerManager.ShowMessage(Resources.MsgCompareFlashSuccesful, false, isCommandPrompt);
                            }
                            else
                            {
                                ControllerManager.ShowMessage(Resources.MsgCompareFlashFailed, true, isCommandPrompt);
                            }
                        }

                    }
                    else
                    {
                        ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
                    }
                    if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
                        device.SetProgrammingMode(false);
                }
                else
                {
                    ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
                }
            }
            catch
            {
                ControllerManager.ShowMessage(Resources.ErrorCantCompareFlash, true, isCommandPrompt);
            }
            finally
            {
                ControllerManager.DeviceClose();
            }
        }

        /// <summary>
        /// Called when Flash Read options click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnFlashReadOptionsClick(object sender, EventArgs e)
        {
            IFlashSettings flashSettings = null;
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    ChipTypes deviceChipType = ChipTypes.UNKNOWN;
                    flashSettings = FlashReadOptions(false, out deviceChipType);
                }), Resources.MsgReadFlashOptionsProgress, false, 0x00);
            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }
        }

        public IFlashSettings FlashReadOptions(bool isCommandPrompt, out ChipTypes outChipType)
        {
            IFlashSettings flashSettings = null;
            ChipTypes deviceChipType = ChipTypes.UNKNOWN;
            outChipType = ChipTypes.UNKNOWN;
            IDevice device = ControllerManager.DeviceOpen();
            try
            {
                if (device.SetProgrammingMode(true))
                {
                    device.ReadSignatureBits();
                    if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
                    {

                        deviceChipType = (ChipTypes)device.ChipType;
                        outChipType = deviceChipType;
                        flashSettings = device.Flash.ReadRfOptions();

                        IFlashPage page = device.Flash.ReadPage(ControllerManager.DocumentModel.IsAsic, 0);
                        if (page.Buffer.Length > Constants.CALIBRATION_VALUE_OTP_POS)
                        {
                            flashSettings.CalibrationValue = (sbyte)page.Buffer[Constants.CALIBRATION_VALUE_OTP_POS];
                        }
                    }
                    else
                    {
                        ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
                    }
                    if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
                        device.SetProgrammingMode(false);
                }
                else
                {
                    ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
                }
                if (flashSettings != null && deviceChipType != ChipTypes.UNKNOWN)
                {
                    if (!isCommandPrompt)
                    {
                        UpdateUserFlashSettings(deviceChipType, flashSettings);
                    }
                    ControllerManager.ShowMessage(Resources.MsgFinished, false, isCommandPrompt, true);
                }
            }
            catch
            {
                ControllerManager.ShowMessage(Resources.ErrorCantReadFlashOptions, true, isCommandPrompt);
            }
            finally
            {
                ControllerManager.DeviceClose();
            }

            return flashSettings;
        }

        /// <summary>
        /// Called when Flash Write options click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnFlashWriteOptionsClick(object sender, EventArgs e)
        {
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    FlashWriteOptions(false, null);

                }), Resources.MsgWriteFlashOptionsProgress, false, 0x00);
            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }
        }
        /// <summary>
        /// Writes the Flash options.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        /// <param name="flashSettings">The flash RF Options.</param>
        public void FlashWriteOptions(bool isCommandPrompt, IFlashSettings flashSettings)
        {
            WriteRfOptionsStatuses result = WriteRfOptionsStatuses.None;
            if (isCommandPrompt)
            {
                ControllerManager.ShowMessage(Resources.MsgWriteFlashOptionsProgress, false, isCommandPrompt);
            }
            IDevice device = ControllerManager.DeviceOpen();
            try
            {
                if (device.SetProgrammingMode(true))
                {
                    device.ReadSignatureBits();
                    if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
                    {
                        IFlashSettings flashSettingsInHex = null;
                        if (isCommandPrompt)
                        {
                            flashSettingsInHex = flashSettings;
                        }
                        else
                        {
                            flashSettingsInHex = GetUserFlashSettings((ChipTypes)device.ChipType);
                        }
                        result = device.Flash.WriteRfOptions(flashSettingsInHex);
                        if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
                            device.SetProgrammingMode(false);
                    }
                    else
                    {
                        ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
                    }
                }
                else
                {
                    ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
                }
            }
            catch
            {
                ControllerManager.ShowMessage(Resources.ErrorCantWriteFlashOptions, true, isCommandPrompt);
            }
            finally
            {
                ControllerManager.DeviceClose();
            }

            switch (result)
            {
                case WriteRfOptionsStatuses.CantReadAppRfSettings:
                    {
                        ControllerManager.ShowMessage(Resources.ErrorCantReadAppRfSettings, true, isCommandPrompt);
                    } break;
                case WriteRfOptionsStatuses.CantReadGeneralRfSettings:
                    {
                        ControllerManager.ShowMessage(Resources.ErrorCantReadGeneralRfSettings, true, isCommandPrompt);
                    } break;
                case WriteRfOptionsStatuses.CantWriteAppRfSettings:
                    {
                        ControllerManager.ShowMessage(Resources.ErrorCantWriteAppRfSettings, true, isCommandPrompt);
                    } break;
                case WriteRfOptionsStatuses.CantWriteGeneralRfSettings:
                    {
                        ControllerManager.ShowMessage(Resources.ErrorCantWriteGeneralRfSettings, true, isCommandPrompt);
                    } break;
                case WriteRfOptionsStatuses.NoErrors:
                    {
                        ControllerManager.ShowMessage(Resources.MsgWriteRfSettingsNoErrors, false, isCommandPrompt);
                    } break;
                case WriteRfOptionsStatuses.None:
                    {
                        ControllerManager.ShowMessage(Resources.MsgWriteRfSettingsNone, false, isCommandPrompt);
                    } break;
                case WriteRfOptionsStatuses.RfFrequencyNotSelected:
                    {
                        ControllerManager.ShowMessage(Resources.ErrorRfFrequencyNotSelected, true, isCommandPrompt);
                    } break;
                case WriteRfOptionsStatuses.UndefinedRfSettings:
                    {
                        ControllerManager.ShowMessage(Resources.ErrorUndefinedRfSettings, true, isCommandPrompt);
                    } break;
            }
        }

        /// <summary>
        /// Updates the flash settings.
        /// </summary>
        /// <param name="chipType">Type of the chip.</param>
        /// <param name="fileName">Name of the file.</param>
        //public void UpdateFlashSettings(ChipTypes chipType, string fileName)
        //{
        //    IDevice device = ControllerManager.ZWaveManager.ApplicationLayer.CreateDevice();
        //    device.ChipType = (byte)chipType;
        //    IFlashSettings flashSettings = ControllerManager.ZWaveManager.ApplicationLayer.CreateFlashSettings();
        //    flashSettings = ReadFlashRFSettings(device, fileName);
        //    if (flashSettings != null)
        //    {
        //        UpdateFlashSettings(chipType, flashSettings);
        //    }
        //}
        /// <summary>
        /// Updates the flash settings.
        /// </summary>
        /// <param name="chipType">Type of the chip.</param>
        /// <param name="flashSettings">The flash settings.</param>
        public void UpdateUserFlashSettings(ChipTypes chipType, IFlashSettings flashSettings)
        {
            if (ControllerManager.ActiveView != null)
            {
                if (chipType == ChipTypes.ZW010x && ControllerManager.ActiveView is ZW010xForm)
                {

                    ZW010xForm form = ControllerManager.ActiveView as ZW010xForm;
                    form.BeginInvoke(new EventHandler(delegate
                    {
                        switch (flashSettings.Frequency)
                        {
                            case MemoryLayout.RF_US:
                                {
                                    form.UsRadioButton.Checked = true;
                                } break;
                            case MemoryLayout.RF_EU:
                                {
                                    form.EuRadioButton.Checked = true;
                                } break;
                            default:
                                {
                                    form.DefaultRadioButton.Checked = true;
                                } break;
                        }
                        form.LowTxPowerTextBox.Text = flashSettings.LowPower.ToString("X");
                        form.NormalTxPowerTextBox.Text = flashSettings.NormalPower.ToString("X");
                        form.RxMatchTextBox.Text = flashSettings.RxMatch.ToString("X");
                        form.TxMatchTextBox.Text = flashSettings.TxMatch.ToString("X");
                    }));
                }
                else if (chipType == ChipTypes.ZW020x && ControllerManager.ActiveView is ZW020xForm)
                {
                    ZW020xForm form = ControllerManager.ActiveView as ZW020xForm;
                    form.BeginInvoke(new EventHandler(delegate
                    {
                        form.FrequencyComboBox.SelectedIndex = 0;
                        foreach (FrequencyWrapper freqWrapper in form.FrequencyComboBox.Items)
                        {
                            if (freqWrapper.Frequency == flashSettings.Frequency)
                            {
                                form.FrequencyComboBox.SelectedItem = freqWrapper;
                            }
                        }
                        form.LowTxPowerTextBox.Text = flashSettings.LowPower.ToString("X");
                        form.NormalTxPowerTextBox.Text = flashSettings.NormalPower.ToString("X");
                    }));
                }
                else if (chipType == ChipTypes.ZW030x && ControllerManager.ActiveView is ZW030xForm)
                {
                    ZW030xForm form = ControllerManager.ActiveView as ZW030xForm;
                    form.BeginInvoke(new EventHandler(delegate
                    {
                        form.FrequencyComboBox.SelectedIndex = 0;
                        foreach (FrequencyWrapper freqWrapper in form.FrequencyComboBox.Items)
                        {
                            if (freqWrapper.Frequency == flashSettings.Frequency)
                            {
                                form.FrequencyComboBox.SelectedItem = freqWrapper;
                            }
                        }
                        form.LowTxPowerTextBox.Text = flashSettings.LowPower.ToString("X");
                        form.NormalTxPowerTextBox.Text = flashSettings.NormalPower.ToString("X");
                    }));
                }
                else if (chipType == ChipTypes.ZW040x && ControllerManager.ActiveView is ZW040xForm)
                {
                    ZW040xForm form = ControllerManager.ActiveView as ZW040xForm;
                    form.BeginInvoke(new EventHandler(delegate
                    {

                        form.LowTxPowerCh0TextBox.Text = flashSettings.LowPowerCh0.ToString("X2");
                        form.NormalTxPowerCh0TextBox.Text = flashSettings.NormalPowerCh0.ToString("X2");

                        form.LowTxPowerCh1TextBox.Text = flashSettings.LowPowerCh1.ToString("X2");
                        form.NormalTxPowerCh1TextBox.Text = flashSettings.NormalPowerCh1.ToString("X2");

                        form.LowTxPowerCh2TextBox.Text = flashSettings.LowPowerCh2.ToString("X2");
                        form.NormalTxPowerCh2TextBox.Text = flashSettings.NormalPowerCh2.ToString("X2");
                        form.CalibrationValueTextBox.Text = flashSettings.CalibrationValue.ToString("X2");
                    }));

                }
            }
        }

        /// <summary>
        /// Updates the flash settings to Default.
        /// </summary>
        /// <param name="chipType">Type of the chip.</param>
        public void UpdateUserFlashSettingsToDefault(ChipTypes chipType)
        {
            if (ControllerManager.ActiveView != null)
            {
                if (chipType == ChipTypes.ZW010x && ControllerManager.ActiveView is ZW010xForm)
                {
                    ZW010xForm form = ControllerManager.ActiveView as ZW010xForm;
                    form.DefaultRadioButton.Checked = true;
                    form.LowTxPowerTextBox.Text = "";
                    form.NormalTxPowerTextBox.Text = "";
                    form.RxMatchTextBox.Text = "";
                    form.TxMatchTextBox.Text = "";
                }
                else if (chipType == ChipTypes.ZW020x && ControllerManager.ActiveView is ZW020xForm)
                {
                    ZW020xForm form = ControllerManager.ActiveView as ZW020xForm;
                    form.FrequencyComboBox.SelectedIndex = 0;
                    form.LowTxPowerTextBox.Text = "";
                    form.NormalTxPowerTextBox.Text = "";
                }
                else if (chipType == ChipTypes.ZW030x && ControllerManager.ActiveView is ZW030xForm)
                {
                    ZW030xForm form = ControllerManager.ActiveView as ZW030xForm;
                    form.FrequencyComboBox.SelectedIndex = 0;
                    form.LowTxPowerTextBox.Text = "";
                    form.NormalTxPowerTextBox.Text = "";
                }
                else if (chipType == ChipTypes.ZW040x && ControllerManager.ActiveView is ZW040xForm)
                {
                    ZW040xForm form = ControllerManager.ActiveView as ZW040xForm;
                    form.LowTxPowerCh0TextBox.Text = "";
                    form.NormalTxPowerCh0TextBox.Text = "";

                    form.LowTxPowerCh1TextBox.Text = "";
                    form.NormalTxPowerCh1TextBox.Text = "";

                    form.LowTxPowerCh2TextBox.Text = "";
                    form.NormalTxPowerCh2TextBox.Text = "";
                    form.CalibrationValueTextBox.Text = "";
                }
            }
        }



        bool _isCommandPrompt = false;
        private void WriteFlash(bool verify, bool isCommandPrompt, string fileName, bool calibrate)
        {
            string strStatuses = "";
            sbyte calibrationValue = -128;
            _isCommandPrompt = isCommandPrompt;
            bool writtenOk = false;
            if (isCommandPrompt)
            {
                ControllerManager.ShowMessage(Resources.MsgWriteFlashProgress, false, isCommandPrompt);
            }
            IDevice device = ControllerManager.DeviceOpen();

            ControllerManager.ChangeProgressInfo(ProgressStatuses.None, 0, 0, _isCommandPrompt);
            try
            {
                byte currentCalibrationValue = 0;
                if (device.SetProgrammingMode(true))
                {
                    device.ReadSignatureBits();
                    if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
                    {
                        IDeviceLed busyLed = device.Leds.GetByType(LedTypes.Busy);
                        bool writeChip = true;

                        if (busyLed != null)
                        {
                            busyLed.SetState(true);

                            if (device.ChipType == (byte)ChipTypes.ZW040x)
                            {
                                if (calibrate)
                                {
                                    calibrationValue = PerformCalibration(isCommandPrompt, calibrationValue, device);
                                }

                                if (IsEmpty(device.Flash, 1, out currentCalibrationValue))
                                {
                                    if (!isCommandPrompt)
                                    {
                                        if (currentCalibrationValue != 0)
                                        {
                                            ControllerManager.ShowProgrammerStatus("Chip is empty and calibrated. Calibration Value: 0x" + Tools.ToHexString((byte)calibrationValue));
                                            //calibrationValue = currentCalibrationValue;
                                        }
                                        else
                                            ControllerManager.ShowProgrammerStatus("Chip is empty and not calibrated.");
                                    }
                                    if (!isCommandPrompt &&
                                        !(ControllerManager.DocumentModel.Settings.WriteOnPcbButton
                                            && ControllerManager.DocumentModel.DeviceProgramButtonPressed))
                                    {
                                        if (ControllerManager.ShowConfirmMessage(Resources.MsgWriteFlashAreYouSureToProgram, false) != DialogResult.Yes)
                                        {
                                            writeChip = false;
                                        }
                                    }
                                }
                                else
                                {
                                    if (isCommandPrompt)
                                    {
                                        ControllerManager.ShowMessage(Resources.MsgWriteFlashAlreadyProgrammed, true, isCommandPrompt);
                                        writeChip = false;
                                    }
                                    else
                                    {
                                        if (ControllerManager.ShowConfirmMessage(Resources.MsgWriteFlashAlreadyProgrammedIsOverwrite, false) != DialogResult.Yes)
                                        {
                                            writeChip = false;
                                        }
                                    }
                                }
                            }
                            else
                            {
                                device.Flash.Erase(true);
                            }

                            bool writeCalibrationValue = false;
                            //Legal calibration value -127 <= x <= 127 returned. Insert value in OTP hex file address 0x0006. Calculate CRC.

                            if (calibrationValue >= -127 && calibrationValue <= 127)
                            {
                                writeCalibrationValue = true;
                                ControllerManager.ShowMessage(String.Format("Calibration performed. Calibration Value: '{0:X2}' \r\n Press 'OK' to continue ...", calibrationValue), false, isCommandPrompt);
                            }
                            else
                            {
                                //if (!isCommandPrompt && device.ChipType == (byte)ChipTypes.ZW040x && ControllerManager.ShowConfirmMessage(String.Format("Calibration not performed. Calibration Value: '{0:X2}' \r\n Do you want to continue?", calibrationValue), false) == DialogResult.No)
                                //{
                                //    writeChip = false;
                                //}
                            }

                            if (writeChip)
                            {
                                ControllerManager.SetProgressInfo(Resources.MsgWriteFlashProgress, isCommandPrompt);
                                string hexFileName = null;
                                IFlashSettings flashSettings = null;
                                if (isCommandPrompt)
                                {
                                    hexFileName = fileName;
                                    flashSettings = null;			//the flashSettings will be used from the .hex file
                                }
                                else
                                {
                                    hexFileName = ControllerManager.DocumentModel.GetFlashHexFilePath((ChipTypes)device.ChipType);
                                    flashSettings = GetUserFlashSettings((ChipTypes)device.ChipType);
                                }
                                if (!String.IsNullOrEmpty(hexFileName) && File.Exists(hexFileName))
                                {
                                    byte[] flashDataRaw = HexFileHelper.BlankArray((int)device.Flash.FlashSize, Constants.BLANK_VALUE);
                                    if (HexFileHelper.ReadIntelHexFile(hexFileName, flashDataRaw, (int)device.Flash.FlashSize) == HexFileHelper.FILE_READ)
                                    {
                                        //flashDataRaw = CalculateCrc(flashDataRaw);
                                        if (device.ChipType == (byte)ChipTypes.ZW040x && flashDataRaw.Length > Constants.CALIBRATION_VALUE_OTP_POS)
                                        {
                                            if (writeCalibrationValue)
                                                flashDataRaw[Constants.CALIBRATION_VALUE_OTP_POS] = (byte)calibrationValue;
                                            else
                                                flashDataRaw[Constants.CALIBRATION_VALUE_OTP_POS] = (byte)currentCalibrationValue;

                                            //flashDataRaw = CalculateCrc(flashDataRaw);
                                        }


                                        device.Flash.ProgressChanged += new Zensys.ZWave.Events.ProgressChangedEventHandler(OnDeviceMemoryProgressChanged);
                                        writtenOk = device.Flash.Write(ControllerManager.DocumentModel.IsAsic, flashDataRaw, flashSettings, verify);
                                        device.Flash.ProgressChanged -= new Zensys.ZWave.Events.ProgressChangedEventHandler(OnDeviceMemoryProgressChanged);

                                        strStatuses = ReadProgrammingStatusesString(device);

                                        if (writtenOk)
                                        {
                                            if (device.ChipType == (byte)ChipTypes.ZW040x)
                                            {
                                                //Erace the MTP Memory of the ZW040x chip
                                                WorkingModes wMode = ControllerManager.Actions.ZW040xFormActions.GetSelectedWorkingMode();
                                                if (wMode == WorkingModes.ExecuteOutOfSram)
                                                {
                                                    ControllerManager.Actions.ZW040xFormActions.MtpAutoErase(device);
                                                }
                                            }
                                        }
                                        else
                                        {
                                            ControllerManager.ShowMessage(Resources.ErrorProgramROMFailed + strStatuses, true, isCommandPrompt);
                                        }

                                    }
                                }
                                else if (String.IsNullOrEmpty(hexFileName))
                                {
                                    ControllerManager.ShowMessage("Hex file is not selected.", true, isCommandPrompt);
                                }
                                else
                                {
                                    ControllerManager.ShowMessage(String.Format("{0} does not exists.", hexFileName), true, isCommandPrompt);
                                }
                            }
                            busyLed.SetState(false);
                        }
                        if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
                            device.SetProgrammingMode(false);
                    }
                    else
                    {
                        ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
                    }
                }
                else
                {
                    ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
                }
            }
            catch (Exception exc)
            {
                ControllerManager.ShowMessage(Resources.ErrorCantWriteFlash, true, isCommandPrompt);
            }
            finally
            {
                ControllerManager.DeviceClose();
            }
            if (writtenOk)
            {
                if (verify)
                {
                    ControllerManager.ShowMessage(Resources.MsgProgramROMDone + strStatuses, false, isCommandPrompt, !isCommandPrompt);
                }
                else
                {
                    ControllerManager.ShowMessage(Resources.MsgWriteROMDone + strStatuses, false, isCommandPrompt, !isCommandPrompt);
                }
            }

            ControllerManager.DocumentModel.DeviceProgramButtonPressed = false;
        }

        private sbyte PerformCalibration(bool isCommandPrompt, sbyte calibrationValue, IDevice device)
        {
            ControllerManager.SetProgressInfo("Reading calibration value from OTP....", isCommandPrompt);

            //device.SetProgrammingMode(false);
            //Thread.Sleep(3000);
            device.SetProgrammingMode(true);
            //device.SetMode(WorkingModes.Normal);

            IFlashPage page = device.Flash.ReadPage(ControllerManager.DocumentModel.IsAsic, 0);
            if (page.Buffer.Length > Constants.CALIBRATION_VALUE_OTP_POS)
            {
                calibrationValue = (sbyte)page.Buffer[Constants.CALIBRATION_VALUE_OTP_POS];
            }
            if (calibrationValue == 0)
            {
                //if (!String.IsNullOrEmpty(ControllerManager.DocumentModel.Settings.CalibrationHexPath) &&
                //    File.Exists(ControllerManager.DocumentModel.Settings.CalibrationHexPath))
                //{
                ControllerManager.SetProgressInfo(Resources.MsgCalibrationProgress, isCommandPrompt);
                //, Resources.MsgWriteFlashProgress, false, 0x00);
                //

                if (ControllerManager.Actions.ZW040xFormActions.WirteHEXToSRAM(isCommandPrompt, device,
                    String.IsNullOrEmpty(ControllerManager.DocumentModel.Settings.CalibrationHexPath) ?
                    System.Windows.Forms.Application.StartupPath + "\\calibration.hex"
                    :
                    ControllerManager.DocumentModel.Settings.CalibrationHexPath, null))
                {
                    //if (device.SetMode(WorkingModes.ExecuteOutOfSram))
                    //{

                    device.SetProgrammingMode(true);
                    device.SetMode(WorkingModes.ExecuteOutOfSram);
                    device.Calibration(true);
                    Thread.Sleep(3000);
                    device.Calibration(false);
                    device.SetProgrammingMode(false);

                    ControllerManager.ShowMessage(Resources.MsgFinished, false, isCommandPrompt, !isCommandPrompt);
                    device.SetProgrammingMode(true);
                    device.SetMode(WorkingModes.Normal);

                    int pageIndex = (int)(Constants.CALIBRATION_VALUE_SRAM_POS / Constants.SRAM_PAGE_SIZE);
                    byte[] pageBuffer = device.Memory.ReadSRAMPage(pageIndex);
                    int calibrationValuePosInPage = (int)(Constants.CALIBRATION_VALUE_SRAM_POS - ((pageIndex) * Constants.SRAM_PAGE_SIZE));


                    if (pageBuffer != null && pageBuffer.Length > calibrationValuePosInPage)
                    {
                        byte _calibrationValue = pageBuffer[calibrationValuePosInPage];
                        if (_calibrationValue == 0x80)
                        {
                            calibrationValue = -128;
                            ControllerManager.ShowMessage("Calibration fail. Calibration Value: 0x80", false, isCommandPrompt, !isCommandPrompt);
                        }
                        else
                        {
                            calibrationValue = (sbyte)_calibrationValue;
                        }
                        //System.Diagnostics.Debug.WriteLine("Calibration value: " + Tools.ToHexString(device.Memory.Buffer[CALIBRATION_VALUE_POS]));
                        //File.AppendAllText(@"C:\calibration.txt", "Calibration value: " + Tools.ToHexString(device.Memory.Buffer[CALIBRATION_VALUE_POS]) + "\r\n");
                    }

                    //}
                    //else
                    //{
                    //    ControllerManager.ShowMessage(Resources.ErrorCantSetChipWorkingMode, true, isCommandPrompt);
                    //}
                }
                //}
            }
            return calibrationValue;
        }

        private byte[] CalculateCrc(byte[] flashDataRaw)
        {
            byte[] result = new byte[flashDataRaw.Length];

            //CRCTool crcTool = new CRCTool();
            //crcTool.Init(CRCTool.CRCCode.CRC32);

            //List<byte> dataRaw = new List<byte>(flashDataRaw);
            //dataRaw.RemoveRange(flashDataRaw.Length - 5, 4);

            //ushort crc = (ushort)crcTool.crctablefast(dataRaw.ToArray());


            File.WriteAllText(@"C:\myLed.hex", HexFileHelper.WriteIntelHexFile(flashDataRaw, 0, flashDataRaw.Length));
            //byte[] data = HexFileHelper.ReadIntelHexFile(

            return result;
        }
        //private IFlashSettings ReadFlashRFSettings(IDevice device, string fileName)
        //{
        //    byte[] flashDataRaw = HexFileHelper.BlankArray((int)device.Flash.FlashSize, Constants.BLANK_VALUE);
        //    IFlashSettings rfSettings = ControllerManager.ZWaveManager.ApplicationLayer.CreateFlashSettings();
        //    if (File.Exists(fileName))
        //    {
        //        if (HexFileHelper.ReadIntelHexFile(fileName, flashDataRaw, (int)device.Flash.FlashSize) == HexFileHelper.FILE_READ)
        //        {
        //            rfSettings.Frequency = flashDataRaw[(int)MemoryLayout.FlashLayout.FLASH_APPL_TABLE_OFFSET + (int)MemoryLayout.FlashLayout.FLASH_APPL_FREQ_OFFS];
        //            rfSettings.NormalPower = flashDataRaw[(int)MemoryLayout.FlashLayout.FLASH_APPL_TABLE_OFFSET + (int)MemoryLayout.FlashLayout.FLASH_APPL_NORM_OFFS];
        //            rfSettings.LowPower = flashDataRaw[(int)MemoryLayout.FlashLayout.FLASH_APPL_TABLE_OFFSET + (int)MemoryLayout.FlashLayout.FLASH_APPL_LOW_OFFS];
        //            rfSettings.RxMatch = flashDataRaw[(int)MemoryLayout.FlashLayout.FLASH_APPL_TABLE_OFFSET + (int)MemoryLayout.FlashLayout.FLASH_APPL_RX_MATCH_OFFS];
        //            rfSettings.TxMatch = flashDataRaw[(int)MemoryLayout.FlashLayout.FLASH_APPL_TABLE_OFFSET + (int)MemoryLayout.FlashLayout.FLASH_APPL_TX_MACTH_OFFS];
        //        }
        //    }
        //    return rfSettings;
        //}
        private IFlashSettings GetUserFlashSettings(ChipTypes chipType)
        {
            IFlashSettings result = null;
            if (ControllerManager.ActiveView != null)
            {
                if (chipType == ChipTypes.ZW010x && ControllerManager.ActiveView is ZW010xForm)
                {
                    result = ControllerManager.ZWaveManager.ApplicationLayer.CreateFlashSettings();
                    ZW010xForm form = ControllerManager.ActiveView as ZW010xForm;

                    if (form.UsRadioButton.Checked)
                        result.Frequency = MemoryLayout.RF_US;
                    if (form.EuRadioButton.Checked)
                        result.Frequency = MemoryLayout.RF_EU;

                    if (form.LowTxPowerTextBox.Text != "")
                        result.LowPower = byte.Parse(form.LowTxPowerTextBox.Text, NumberStyles.HexNumber);
                    if (form.NormalTxPowerTextBox.Text != "")
                        result.NormalPower = byte.Parse(form.NormalTxPowerTextBox.Text, NumberStyles.HexNumber);
                    if (form.RxMatchTextBox.Text != "")
                        result.RxMatch = byte.Parse(form.RxMatchTextBox.Text, NumberStyles.HexNumber);
                    if (form.TxMatchTextBox.Text != "")
                        result.TxMatch = byte.Parse(form.TxMatchTextBox.Text, NumberStyles.HexNumber);

                }
                else if (chipType == ChipTypes.ZW020x && ControllerManager.ActiveView is ZW020xForm)
                {
                    result = ControllerManager.ZWaveManager.ApplicationLayer.CreateFlashSettings();
                    ZW020xForm form = ControllerManager.ActiveView as ZW020xForm;

                    form.Invoke(new EventHandler(delegate
                    {
                        if (form.FrequencyComboBox.SelectedItem != null && form.FrequencyComboBox.SelectedIndex != 0)
                        {
                            result.Frequency = ((FrequencyWrapper)form.FrequencyComboBox.SelectedItem).Frequency;
                        }
                    }));

                    if (form.LowTxPowerTextBox.Text != "")
                        result.LowPower = byte.Parse(form.LowTxPowerTextBox.Text, NumberStyles.HexNumber);
                    if (form.NormalTxPowerTextBox.Text != "")
                        result.NormalPower = byte.Parse(form.NormalTxPowerTextBox.Text, NumberStyles.HexNumber);
                }
                else if (chipType == ChipTypes.ZW030x && ControllerManager.ActiveView is ZW030xForm)
                {
                    result = ControllerManager.ZWaveManager.ApplicationLayer.CreateFlashSettings();
                    ZW030xForm form = ControllerManager.ActiveView as ZW030xForm;

                    form.Invoke(new EventHandler(delegate
                    {
                        if (form.FrequencyComboBox.SelectedItem != null && form.FrequencyComboBox.SelectedIndex != 0)
                        {
                            result.Frequency = ((FrequencyWrapper)form.FrequencyComboBox.SelectedItem).Frequency;
                        }
                    }));

                    if (form.LowTxPowerTextBox.Text != "")
                        result.LowPower = byte.Parse(form.LowTxPowerTextBox.Text, NumberStyles.HexNumber);
                    if (form.NormalTxPowerTextBox.Text != "")
                        result.NormalPower = byte.Parse(form.NormalTxPowerTextBox.Text, NumberStyles.HexNumber);
                }
                else if (chipType == ChipTypes.ZW040x && ControllerManager.ActiveView is ZW040xForm)
                {
                    result = ControllerManager.ZWaveManager.ApplicationLayer.CreateFlashSettings();
                    ZW040xForm form = ControllerManager.ActiveView as ZW040xForm;

                    if (form.LowTxPowerCh0TextBox.Text != "")
                        result.LowPowerCh0 = byte.Parse(form.LowTxPowerCh0TextBox.Text, NumberStyles.HexNumber);
                    if (form.NormalTxPowerCh0TextBox.Text != "")
                        result.NormalPowerCh0 = byte.Parse(form.NormalTxPowerCh0TextBox.Text, NumberStyles.HexNumber);

                    if (form.LowTxPowerCh1TextBox.Text != "")
                        result.LowPowerCh1 = byte.Parse(form.LowTxPowerCh1TextBox.Text, NumberStyles.HexNumber);
                    if (form.NormalTxPowerCh1TextBox.Text != "")
                        result.NormalPowerCh1 = byte.Parse(form.NormalTxPowerCh1TextBox.Text, NumberStyles.HexNumber);

                    if (form.LowTxPowerCh2TextBox.Text != "")
                        result.LowPowerCh2 = byte.Parse(form.LowTxPowerCh2TextBox.Text, NumberStyles.HexNumber);
                    if (form.NormalTxPowerCh2TextBox.Text != "")
                        result.NormalPowerCh2 = byte.Parse(form.NormalTxPowerCh2TextBox.Text, NumberStyles.HexNumber);
                }
            }
            return result;
        }

        private string NVMChipTypeToString(NVMTypes NVMChipType)
        {
            string str = "";
            if (NVMChipType != NVMTypes.NVMInvalid)
            {
                str += Resources.MsgNVMChipType;
                switch (NVMChipType & NVMTypes.NVMTypeMask)
                {
                    case NVMTypes.NVMSerialEEPROM:
                        str += "Serial EEPROM";
                        break;

                    case NVMTypes.NVMSerialFlash:
                        str += "Serial Flash";
                        break;

                    case NVMTypes.NVMInvalid:
                        str += "Invalid";
                        break;

                    default:
                        str += "Unknown";
                        break;
                }

                switch (NVMChipType)
                {
                    case NVMTypes.NVMInvalid:
                    case NVMTypes.NVMUnknown:
                    case NVMTypes.NVMSerialEEPROM:
                    case NVMTypes.NVMSerialFlash:
                        break;
                    case NVMTypes.NVMNumonyxM25PE10:
                        str += " Numonyx M25PE10";
                        break;
                    case NVMTypes.NVMNumonyxM25PE20:
                        str += " Numonyx M25PE20";
                        break;
                    case NVMTypes.NVMNumonyxM25PE40:
                        str += " Numonyx M25PE40";
                        break;
                }
                str += ". ";
            }
            return str;
        }

        /// <summary>
        /// Called when EEPROM Erase click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnEEPROMEraseClick(object sender, EventArgs e)
        {
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    EEPROMErase(false);
                }), Resources.MsgEraseEEPROMProgress, false, 0x00);
            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }
        }
        /// <summary>
        /// Erase the EEPROM.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        public void EEPROMErase(bool isCommandPrompt)
        {
            if (isCommandPrompt)
            {
                ControllerManager.ShowMessage(Resources.MsgEraseEEPROMProgress, false, isCommandPrompt);
            }
            IDevice device = ControllerManager.DeviceOpen();
            NVMTypes nvm_type = NVMTypes.NVMInvalid;
            try
            {
                if (device.SetProgrammingMode(true))
                {
                    device.ReadSignatureBits();
                    if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
                    {

                        if ((nvm_type = device.Memory.SwitchEEPROM(true)) != NVMTypes.NVMInvalid)
                        {
                            IDeviceLed busyLed = device.Leds.GetByType(LedTypes.Busy);
                            if (busyLed != null)
                            {
                                busyLed.SetState(true);
                                byte[] eepromDataRaw = HexFileHelper.BlankArray((int)device.Memory.EepromSize, Constants.EEPROM_BLANK_VALUE);
                                device.Memory.ClearEEPROM(eepromDataRaw);
                                busyLed.SetState(false);
                                ControllerManager.ShowMessage(NVMChipTypeToString(nvm_type) + Resources.MsgNVMEraseFinished, false, isCommandPrompt, !isCommandPrompt);
                            }
                            else
                            {
                                ControllerManager.ShowMessage(Resources.ErrorCantEraseEEPROM, true, isCommandPrompt);
                            }
                            device.Memory.SwitchEEPROM(false);
                        }
                        else
                        {
                            ControllerManager.ShowMessage(Resources.ErrorCantInitEEPROM, true, isCommandPrompt);
                        }
                    }
                    else
                    {
                        ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
                    }
                }
                else
                {
                    ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
                }
            }
            catch
            {
                device.Memory.SwitchEEPROM(false);
                ControllerManager.ShowMessage(Resources.ErrorCantEraseEEPROM, true, isCommandPrompt);
            }
            finally
            {
                if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
                    device.SetProgrammingMode(false);
                ControllerManager.DeviceClose();
            }
        }

        /// <summary>
        /// Called when EEPROM Read click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnEEPROMReadClick(object sender, EventArgs e)
        {


            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    EEPROMRead(false, null);

                }), Resources.MsgReadEEPROMProgress, false, 0x00);

            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }

        }

        /// <summary>
        /// Reads the EEPROM.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        /// <param name="fileName">Name of the file.</param>
        internal void EEPROMRead(bool isCommandPrompt, string fileName)
        {
            _isCommandPrompt = isCommandPrompt;
            if (isCommandPrompt)
            {
                ControllerManager.ShowMessage(Resources.MsgReadEEPROMProgress, false, isCommandPrompt);
            }

            byte[] eepromBuffer = null;
            NVMTypes nvm_type = NVMTypes.NVMInvalid;
            IDevice device = ControllerManager.DeviceOpen();
            ControllerManager.ChangeProgressInfo(ProgressStatuses.None, 0, 0, _isCommandPrompt);
            try
            {

                if (device.SetProgrammingMode(true))
                {
                    device.ReadSignatureBits();
                    if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
                    {
                        if ((nvm_type = device.Memory.SwitchEEPROM(true)) != NVMTypes.NVMInvalid)
                        {
                            IDeviceLed busyLed = device.Leds.GetByType(LedTypes.Busy);
                            if (busyLed != null)
                            {
                                busyLed.SetState(true);
                                device.Memory.ProgressChanged += new Zensys.ZWave.Events.ProgressChangedEventHandler(OnDeviceMemoryProgressChanged);
                                device.Memory.ReadEEPROM();
                                device.Memory.ProgressChanged -= new Zensys.ZWave.Events.ProgressChangedEventHandler(OnDeviceMemoryProgressChanged);
                                eepromBuffer = device.Memory.Buffer;
                                busyLed.SetState(false);
                            }
                            else
                            {
                                ControllerManager.ShowMessage(Resources.ErrorCantReadEEPROM, true, isCommandPrompt);
                            }
                            device.Memory.SwitchEEPROM(false);
                        }
                        else
                        {
                            ControllerManager.ShowMessage(Resources.ErrorCantInitEEPROM, true, isCommandPrompt);
                        }
                    }
                    else
                    {
                        ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
                    }
                }
                else
                {
                    ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
                }
            }
            catch
            {
                device.Memory.SwitchEEPROM(false);
                ControllerManager.ShowMessage(Resources.ErrorCantReadEEPROM, true, isCommandPrompt);
            }
            finally
            {
                if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
                    device.SetProgrammingMode(false);
                ControllerManager.DeviceClose();
            }
            if (eepromBuffer != null)
            {
                string nvm_type_str = NVMChipTypeToString(nvm_type);
                if (isCommandPrompt)
                {
                    try
                    {
                        TextWriter writer = new StreamWriter(fileName);
                        writer.Write(HexFileHelper.WriteIntelHexFile(eepromBuffer, 0, eepromBuffer.Length));
                        writer.Close();
                        ControllerManager.ShowMessage(nvm_type_str + Resources.MsgNVMReadFinished, false, isCommandPrompt);
                    }
                    catch (Exception ex)
                    {
                        Console.WriteLine(ex.Message);
                    }
                }
                else
                {
                    ControllerManager.ShowConsoleInfo(HexFileHelper.WriteIntelHexFile(eepromBuffer, 0, eepromBuffer.Length));
                    ControllerManager.ShowMessage(nvm_type_str + Resources.MsgNVMReadFinished, false, false, true);
                }
            }
        }

        private void OnDeviceMemoryProgressChanged(Zensys.ZWave.Events.ProgressChangedEventArgs args)
        {
            ControllerManager.ChangeProgressInfo(args.ProgressStatus, args.Current, args.Total, _isCommandPrompt);
        }

        /// <summary>
        /// Called when EEPROM Write click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnEEPROMWriteClick(object sender, EventArgs e)
        {
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    EEPROMWrite(false, null);
                }), Resources.MsgWriteEEPROMProgress, false, 0x00);

            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }

        }
        /// <summary>
        /// Write the EEPROM.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        /// <param name="fileName">Name of the file.</param>
        public void EEPROMWrite(bool isCommandPrompt, string fileName)
        {
            _isCommandPrompt = isCommandPrompt;
            bool result = false;
            if (isCommandPrompt)
            {
                ControllerManager.ShowMessage(Resources.MsgWriteEEPROMProgress, false, isCommandPrompt);
            }
            IDevice device = ControllerManager.DeviceOpen();
            ControllerManager.ChangeProgressInfo(ProgressStatuses.None, 0, 0, _isCommandPrompt);
            NVMTypes nvm_type = NVMTypes.NVMInvalid;
            try
            {
                if (device.SetProgrammingMode(true))
                {
                    device.ReadSignatureBits();
                    if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
                    {
                        string hexFileName;
                        if (isCommandPrompt)
                        {
                            hexFileName = fileName;
                        }
                        else
                        {
                            hexFileName = ControllerManager.DocumentModel.GetEepromHexFilePath((ChipTypes)device.ChipType);
                        }
                        if (!String.IsNullOrEmpty(hexFileName) && File.Exists(hexFileName))
                        {
                            byte[] eepromDataRaw = HexFileHelper.BlankArray((int)device.Memory.EepromSize, Constants.EEPROM_BLANK_VALUE);
                            if (HexFileHelper.ReadIntelHexFile(hexFileName, eepromDataRaw, (int)device.Memory.EepromSize) == HexFileHelper.FILE_READ)
                            {
                                if ((nvm_type = device.Memory.SwitchEEPROM(true)) != NVMTypes.NVMInvalid)
                                {
                                    device.Memory.ProgressChanged += new Zensys.ZWave.Events.ProgressChangedEventHandler(OnDeviceMemoryProgressChanged);
                                    IDeviceLed busyLed = device.Leds.GetByType(LedTypes.Busy);
                                    if (busyLed != null)
                                    {
                                        busyLed.SetState(true);
                                        byte[] eepromEmptyDataRaw = HexFileHelper.BlankArray((int)device.Memory.EepromSize, Constants.EEPROM_BLANK_VALUE);
                                        device.Memory.ClearEEPROM(eepromEmptyDataRaw);
                                        if (device.Memory.WriteEEPROM(eepromDataRaw))
                                        {
                                            result = true;
                                        }
                                        else
                                        {
                                            ControllerManager.ShowMessage(Resources.ErrorCantWriteEEPROM, true, isCommandPrompt);
                                        }
                                        busyLed.SetState(false);
                                    }
                                    else
                                    {
                                        ControllerManager.ShowMessage(Resources.ErrorCantWriteEEPROM, true, isCommandPrompt);
                                    }
                                    device.Memory.SwitchEEPROM(false);
                                    device.Memory.ProgressChanged -= new Zensys.ZWave.Events.ProgressChangedEventHandler(OnDeviceMemoryProgressChanged);
                                }
                                else
                                {
                                    ControllerManager.ShowMessage(Resources.ErrorCantInitEEPROM, true, isCommandPrompt);
                                }
                            }
                            else
                            {
                                ControllerManager.ShowMessage(Resources.ErrorHexFileReadingFailed, true, isCommandPrompt);
                            }
                        }
                        else if (String.IsNullOrEmpty(hexFileName))
                        {
                            ControllerManager.ShowMessage(Resources.ErrorEEPROMHexFileNotSpecified, true, isCommandPrompt);
                        }
                        else
                        {
                            ControllerManager.ShowMessage(String.Format("{0} does not exists.", hexFileName), true, isCommandPrompt);
                        }
                    }
                    else
                    {
                        ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
                    }
                }
                else
                {
                    ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
                }
            }
            catch
            {
                device.Memory.SwitchEEPROM(false);
                ControllerManager.ShowMessage(Resources.ErrorCantWriteEEPROM, true, isCommandPrompt);
            }
            finally
            {
                if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
                    device.SetProgrammingMode(false);
                ControllerManager.DeviceClose();
            }
            if (result)
            {
                ControllerManager.ShowMessage(NVMChipTypeToString(nvm_type) + Resources.MsgNVMWriteFinished, false, isCommandPrompt, !isCommandPrompt);
            }
        }

        /// <summary>
        /// Called when EEPROM Compare click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnEEPROMCompareClick(object sender, EventArgs e)
        {
            bool compareResult = false;
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                NVMTypes nvm_type = NVMTypes.NVMInvalid;
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    IDevice device = ControllerManager.DeviceOpen();
                    try
                    {

                        if (device.SetProgrammingMode(true))
                        {
                            device.ReadSignatureBits();
                            if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
                            {
                                string hexFileName = ControllerManager.DocumentModel.GetEepromHexFilePath((ChipTypes)device.ChipType);
                                if (!String.IsNullOrEmpty(hexFileName) && File.Exists(hexFileName))
                                {
                                    byte[] eepromDataRaw = HexFileHelper.BlankArray((int)device.Memory.EepromSize, Constants.EEPROM_BLANK_VALUE);
                                    if (HexFileHelper.ReadIntelHexFile(hexFileName, eepromDataRaw, (int)device.Memory.EepromSize) == HexFileHelper.FILE_READ)
                                    {
                                        if ((nvm_type = device.Memory.SwitchEEPROM(true)) != NVMTypes.NVMInvalid)
                                        {
                                            IDeviceLed busyLed = device.Leds.GetByType(LedTypes.Busy);
                                            if (busyLed != null)
                                            {
                                                busyLed.SetState(true);
                                                device.Memory.ReadEEPROM();
                                                compareResult = device.Memory.Compare(eepromDataRaw);
                                                busyLed.SetState(false);
                                            }
                                            else
                                            {
                                                ControllerManager.ShowMessage(Resources.ErrorCantCompareEEPROM, true);
                                            }
                                            device.Memory.SwitchEEPROM(false);
                                        }
                                        else
                                        {
                                            ControllerManager.ShowMessage(Resources.ErrorCantInitEEPROM, true);
                                        }
                                    }
                                    else
                                    {
                                        ControllerManager.ShowMessage(Resources.ErrorHexFileReadingFailed, true);
                                    }
                                }
                                else if (String.IsNullOrEmpty(hexFileName))
                                {
                                    ControllerManager.ShowMessage(Resources.ErrorEEPROMHexFileNotSpecified, true);
                                }
                                else
                                {
                                    ControllerManager.ShowMessage(String.Format("{0} does not exists.", hexFileName), true);
                                }
                            }
                            else
                            {
                                ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true);
                            }
                        }
                        else
                        {
                            ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true);
                        }
                    }
                    catch
                    {
                        device.Memory.SwitchEEPROM(false);
                        ControllerManager.ShowMessage(Resources.ErrorCantCompareEEPROM, true);
                    }
                    finally
                    {
                        if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
                            device.SetProgrammingMode(false);
                        ControllerManager.DeviceClose();
                    }
                }), Resources.MsgReadEEPROMProgress, false, 0x00);

                string nvm_type_str = NVMChipTypeToString(nvm_type);
                if (nvm_type_str.Length != 0)
                {
                    nvm_type_str = "\n" + nvm_type_str;
                }
                if (compareResult)
                {
                    ControllerManager.ShowMessage(Resources.MsgCompareEepromSuccesful + nvm_type_str, false);
                }
                else
                {
                    ControllerManager.ShowMessage(Resources.ErrorCompareEepromFailed + nvm_type_str, true);
                }
            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }
        }

        /// <summary>
        /// Called when Read Home Id click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnReadHomeIdClick(object sender, EventArgs e)
        {
            byte[] homeId = null;
            ChipTypes deviceChipType = ChipTypes.UNKNOWN;
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                NVMTypes nvm_type = NVMTypes.NVMInvalid;
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    IDevice device = ControllerManager.DeviceOpen();
                    try
                    {
                        if (device.SetProgrammingMode(true))
                        {
                            device.ReadSignatureBits();
                            deviceChipType = (ChipTypes)device.ChipType;
                            if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
                            {
                                if ((nvm_type = device.Memory.SwitchEEPROM(true)) != NVMTypes.NVMInvalid)
                                {
                                    IDeviceLed busyLed = device.Leds.GetByType(LedTypes.Busy);
                                    if (busyLed != null)
                                    {
                                        busyLed.SetState(true);

                                        homeId = device.Memory.ReadHomeId();
                                        busyLed.SetState(false);
                                    }
                                    else
                                    {
                                        ControllerManager.ShowMessage(Resources.ErrorCantReadHomeId, true);
                                    }
                                    device.Memory.SwitchEEPROM(false);
                                }
                                else
                                {
                                    ControllerManager.ShowMessage(Resources.ErrorCantInitEEPROM, true);
                                }
                            }
                            else
                            {
                                ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true);
                            }
                        }
                        else
                        {
                            ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true);
                        }
                    }
                    catch
                    {
                        device.Memory.SwitchEEPROM(false);
                        ControllerManager.ShowMessage(Resources.ErrorCantReadHomeId, true);
                    }
                    finally
                    {
                        if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
                            device.SetProgrammingMode(false);
                        ControllerManager.DeviceClose();
                    }
                }), Resources.MsgReadHomeIdProgress, false, 0x00);
                if (homeId != null)
                {
                    UpdateHomeId(homeId, deviceChipType);
                    ControllerManager.ShowMessage(NVMChipTypeToString(nvm_type) + Resources.MsgFinished, false, false, true);
                }
            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }
        }

        /// <summary>
        /// Called when Write Home Id click.
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        public void OnWriteHomeIdClick(object sender, EventArgs e)
        {

            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    WriteHomeId(false, null);
                }), Resources.MsgWriteHomeIdProgress, false, 0x00);

            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }
        }
        /// <summary>
        /// Writes the Home Id.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c> [is command prompt].</param>
        /// <param name="homeIdString">The home id string.</param>
        public void WriteHomeId(bool isCommandPrompt, string homeIdString)
        {
            if (isCommandPrompt)
            {
                ControllerManager.ShowMessage(Resources.MsgWriteHomeIdProgress, false, isCommandPrompt);
            }
            byte[] homeId = null;
            ChipTypes deviceChipType = ChipTypes.UNKNOWN;
            IDevice device = ControllerManager.DeviceOpen();
            try
            {
                if (device.SetProgrammingMode(true))
                {
                    device.ReadSignatureBits();
                    deviceChipType = (ChipTypes)device.ChipType;
                    if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
                    {
                        NVMTypes nvm_type = NVMTypes.NVMInvalid;
                        if ((nvm_type = device.Memory.SwitchEEPROM(true)) != NVMTypes.NVMInvalid)
                        {
                            IDeviceLed busyLed = device.Leds.GetByType(LedTypes.Busy);
                            if (busyLed != null)
                            {
                                busyLed.SetState(true);
                                if (isCommandPrompt)
                                {
                                    homeId = new byte[HomeIdTextBox.MAX_ARRAY_LENGTH];
                                    if ((homeIdString.Length % 8) > 0)
                                    {
                                        homeIdString = homeIdString.PadLeft(homeIdString.Length + (homeIdString.Length % 8), '0');
                                    }
                                    try
                                    {
                                        byte[] tmpArr = Tools.FromHexString(homeIdString);
                                        if (tmpArr.Length < HomeIdTextBox.MAX_ARRAY_LENGTH)
                                        {
                                            for (int i = 0; i < tmpArr.Length; i++)
                                            {
                                                homeId[HomeIdTextBox.MAX_ARRAY_LENGTH - (i + 1)] = tmpArr[i];
                                            }
                                        }
                                        else if (tmpArr.Length >= HomeIdTextBox.MAX_ARRAY_LENGTH)
                                        {
                                            for (int i = 0; i < HomeIdTextBox.MAX_ARRAY_LENGTH; i++)
                                            {
                                                homeId[i] = tmpArr[i];
                                            }
                                        }
                                    }
                                    catch
                                    {
                                        homeId = new byte[HomeIdTextBox.MAX_ARRAY_LENGTH];
                                    }
                                }
                                else
                                {
                                    homeId = GetHomeId(deviceChipType);
                                }
                                if (homeId != null)
                                {
                                    device.Memory.WriteHomeId(homeId);
                                }
                                busyLed.SetState(false);

                                if (homeId != null)
                                {
                                    if (!isCommandPrompt)
                                    {
                                        UpdateHomeId(homeId, deviceChipType);
                                    }
                                    ControllerManager.ShowMessage(NVMChipTypeToString(nvm_type) + Resources.MsgFinished, false, isCommandPrompt, !isCommandPrompt);
                                }
                            }
                            else
                            {
                                ControllerManager.ShowMessage(Resources.ErrorCantWriteHomeId, true, isCommandPrompt);
                            }
                            device.Memory.SwitchEEPROM(false);
                        }
                        else
                        {
                            ControllerManager.ShowMessage(Resources.ErrorCantInitEEPROM, true, isCommandPrompt);
                        }
                    }
                    else
                    {
                        ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
                    }
                }
                else
                {
                    ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
                }
            }
            catch
            {
                device.Memory.SwitchEEPROM(false);
                ControllerManager.ShowMessage(Resources.ErrorCantWriteHomeId, true, isCommandPrompt);
            }
            finally
            {
                if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
                    device.SetProgrammingMode(false);
                ControllerManager.DeviceClose();
            }
        }

        private byte[] GetHomeId(ChipTypes chipType)
        {
            byte[] result = null;
            if (ControllerManager.ActiveView != null)
            {
                if (chipType == ChipTypes.ZW010x && ControllerManager.ActiveView is ZW010xForm)
                {
                    ZW010xForm form = ControllerManager.ActiveView as ZW010xForm;
                    form.Invoke(new EventHandler(delegate
                    {
                        if (form.AutoIncrementHomeIdCheckBox.Checked)
                        {
                            string lastHomeIdSettings = Settings.Default.ZW010xAutoIncrementLastHomeId;
                            string startHomeIdSettings = "";
                            string endHomeIdSettings = "";

                            result = ProcessAutoIncremetHomeIdLogic(
                                form.StartHomeIdTextBox.Text,
                                form.EndHomeIdTextBox.Text,
                                ref lastHomeIdSettings,
                                ref startHomeIdSettings,
                                ref endHomeIdSettings);

                            Settings.Default.ZW010xAutoIncrementLastHomeId = lastHomeIdSettings;
                            Settings.Default.ZW010xAutoIncrementStartHomeId = startHomeIdSettings;
                            Settings.Default.ZW010xAutoIncrementEndHomeId = endHomeIdSettings;
                            Settings.Default.Save();
                            form.StartHomeIdTextBox.Text = startHomeIdSettings;
                            form.EndHomeIdTextBox.Text = endHomeIdSettings;
                        }
                        else
                        {
                            result = Tools.FromHexString(form.CurrentHomeIdTextBox.Text);
                        }
                    }));
                }
                else if (chipType == ChipTypes.ZW020x && ControllerManager.ActiveView is ZW020xForm)
                {
                    ZW020xForm form = ControllerManager.ActiveView as ZW020xForm;
                    form.Invoke(new EventHandler(delegate
                    {
                        if (form.AutoIncrementHomeIdCheckBox.Checked)
                        {
                            string lastHomeIdSettings = Settings.Default.ZW020xAutoIncrementLastHomeId;
                            string startHomeIdSettings = "";
                            string endHomeIdSettings = "";

                            result = ProcessAutoIncremetHomeIdLogic(
                                form.StartHomeIdTextBox.Text,
                                form.EndHomeIdTextBox.Text,
                                ref lastHomeIdSettings,
                                ref startHomeIdSettings,
                                ref endHomeIdSettings);

                            Settings.Default.ZW020xAutoIncrementLastHomeId = lastHomeIdSettings;
                            Settings.Default.ZW020xAutoIncrementStartHomeId = startHomeIdSettings;
                            Settings.Default.ZW020xAutoIncrementEndHomeId = endHomeIdSettings;
                            Settings.Default.Save();
                            form.StartHomeIdTextBox.Text = startHomeIdSettings;
                            form.EndHomeIdTextBox.Text = endHomeIdSettings;
                        }
                        else
                        {
                            result = Tools.FromHexString(form.CurrentHomeIdTextBox.Text);
                        }
                    }));
                }
                else if (chipType == ChipTypes.ZW030x && ControllerManager.ActiveView is ZW030xForm)
                {
                    ZW030xForm form = ControllerManager.ActiveView as ZW030xForm;
                    form.Invoke(new EventHandler(delegate
                    {
                        if (form.AutoIncrementHomeIdCheckBox.Checked)
                        {
                            string lastHomeIdSettings = Settings.Default.ZW030xAutoIncrementLastHomeId;
                            string startHomeIdSettings = "";
                            string endHomeIdSettings = "";

                            result = ProcessAutoIncremetHomeIdLogic(
                                form.StartHomeIdTextBox.Text,
                                form.EndHomeIdTextBox.Text,
                                ref lastHomeIdSettings,
                                ref startHomeIdSettings,
                                ref endHomeIdSettings);

                            Settings.Default.ZW030xAutoIncrementLastHomeId = lastHomeIdSettings;
                            Settings.Default.ZW030xAutoIncrementStartHomeId = startHomeIdSettings;
                            Settings.Default.ZW030xAutoIncrementEndHomeId = endHomeIdSettings;
                            Settings.Default.Save();
                            form.StartHomeIdTextBox.Text = startHomeIdSettings;
                            form.EndHomeIdTextBox.Text = endHomeIdSettings;
                        }
                        else
                        {
                            result = Tools.FromHexString(form.CurrentHomeIdTextBox.Text);
                        }
                    }));
                }
                else if (chipType == ChipTypes.ZW040x && ControllerManager.ActiveView is ZW040xForm)
                {
                    ZW040xForm form = ControllerManager.ActiveView as ZW040xForm;
                    form.Invoke(new EventHandler(delegate
                    {
                        if (form.AutoIncrementHomeIdCheckBox.Checked)
                        {
                            string lastHomeIdSettings = Settings.Default.ZW040xAutoIncrementLastHomeId;
                            string startHomeIdSettings = "";
                            string endHomeIdSettings = "";

                            result = ProcessAutoIncremetHomeIdLogic(
                                form.StartHomeIdTextBox.Text,
                                form.EndHomeIdTextBox.Text,
                                ref lastHomeIdSettings,
                                ref startHomeIdSettings,
                                ref endHomeIdSettings);

                            Settings.Default.ZW040xAutoIncrementLastHomeId = lastHomeIdSettings;
                            Settings.Default.ZW040xAutoIncrementStartHomeId = startHomeIdSettings;
                            Settings.Default.ZW040xAutoIncrementEndHomeId = endHomeIdSettings;
                            Settings.Default.Save();
                            form.StartHomeIdTextBox.Text = startHomeIdSettings;
                            form.EndHomeIdTextBox.Text = endHomeIdSettings;
                        }
                        else
                        {
                            result = Tools.FromHexString(form.CurrentHomeIdTextBox.Text);
                        }
                    }));
                }
            }
            return result;
        }
        private byte[] ProcessAutoIncremetHomeIdLogic(string startHomeId,
            string endHomeId,
            ref string lastHomeIdSettings,
            ref string startHomeIdSettings,
            ref string endHomeIdSettings)
        {
            byte[] result = null;
            if (String.IsNullOrEmpty(startHomeId) || String.IsNullOrEmpty(endHomeId))
            {
                ControllerManager.ShowMessage(Resources.ErrorStartEndHomeIdEmpty, true);
            }
            else
            {
                if (UInt32.Parse(startHomeId, NumberStyles.AllowHexSpecifier) <
                    UInt32.Parse(endHomeId, NumberStyles.AllowHexSpecifier))
                {
                    if (!String.IsNullOrEmpty(lastHomeIdSettings))
                    {
                        uint lastHomeId = UInt32.Parse(lastHomeIdSettings, NumberStyles.AllowHexSpecifier);
                        uint _endHomeId = UInt32.Parse(endHomeId, NumberStyles.AllowHexSpecifier);
                        if (lastHomeId >= _endHomeId)
                        {
                            if (ControllerManager.ShowConfirmMessage(Resources.MsgAutoIncrementHomeIdFromStart, false) != System.Windows.Forms.DialogResult.Yes)
                            {
                                return result;
                            }
                            lastHomeId = UInt32.Parse(startHomeId, NumberStyles.AllowHexSpecifier);
                        }
                        else
                        {
                            lastHomeId++;
                        }
                        if (lastHomeId < UInt32.Parse(startHomeId, NumberStyles.AllowHexSpecifier))
                        {
                            lastHomeId = UInt32.Parse(startHomeId, NumberStyles.AllowHexSpecifier);
                        }
                        lastHomeIdSettings = Tools.ToHexString(lastHomeId);
                    }
                    else
                    {
                        uint lastHomeId = UInt32.Parse(startHomeId, NumberStyles.AllowHexSpecifier);
                        lastHomeIdSettings = Tools.ToHexString(lastHomeId);
                    }
                    startHomeIdSettings = startHomeId.PadLeft(8, '0');
                    endHomeIdSettings = endHomeId.PadLeft(8, '0');
                    result = Tools.FromHexString(lastHomeIdSettings);
                }
                else
                {
                    ControllerManager.ShowMessage(Resources.ErrorInvalidEndHomeId, true);
                }
            }
            return result;
        }

        private void UpdateHomeId(byte[] homeId, ChipTypes chipType)
        {
            if (ControllerManager.ActiveView != null)
            {
                if (chipType == ChipTypes.ZW010x && ControllerManager.ActiveView is ZW010xForm)
                {
                    ZW010xForm form = ControllerManager.ActiveView as ZW010xForm;
                    form.Invoke(new EventHandler(delegate
                    {
                        form.CurrentHomeIdTextBox.Text = Tools.ToHexString(homeId).PadLeft(8, '0');
                        form.ReadHomeIdTextBox.Text = Tools.ToHexString(homeId).PadLeft(8, '0');
                    }));

                }
                else if (chipType == ChipTypes.ZW020x && ControllerManager.ActiveView is ZW020xForm)
                {
                    ZW020xForm form = ControllerManager.ActiveView as ZW020xForm;
                    form.Invoke(new EventHandler(delegate
                    {
                        form.CurrentHomeIdTextBox.Text = Tools.ToHexString(homeId).PadLeft(8, '0');
                        form.ReadHomeIdTextBox.Text = Tools.ToHexString(homeId).PadLeft(8, '0');
                    }));
                }
                else if (chipType == ChipTypes.ZW030x && ControllerManager.ActiveView is ZW030xForm)
                {
                    ZW030xForm form = ControllerManager.ActiveView as ZW030xForm;
                    form.Invoke(new EventHandler(delegate
                    {
                        form.CurrentHomeIdTextBox.Text = Tools.ToHexString(homeId).PadLeft(8, '0');
                        form.ReadHomeIdTextBox.Text = Tools.ToHexString(homeId).PadLeft(8, '0');
                    }));
                }
                else if (chipType == ChipTypes.ZW040x && ControllerManager.ActiveView is ZW040xForm)
                {
                    ZW040xForm form = ControllerManager.ActiveView as ZW040xForm;
                    form.Invoke(new EventHandler(delegate
                    {
                        form.CurrentHomeIdTextBox.Text = Tools.ToHexString(homeId).PadLeft(8, '0');
                        form.ReadHomeIdTextBox.Text = Tools.ToHexString(homeId).PadLeft(8, '0');
                    }));
                }
            }
        }
        /// <summary>
        /// Detects the device.
        /// </summary>
        /// <returns></returns>
        public bool DetectDevice()
        {
            bool result = false;

            ControllerManager.DoAction(new EventHandler(delegate
                {
                    result = DetectDevice(false);
                    string status = Resources.MsgStatusDetectedDevice;
                    switch ((ChipTypes)ControllerManager.DocumentModel.SelectedChipType)
                    {
                        case ChipTypes.UNKNOWN:
                            status += "N/A";
                            break;
                        case ChipTypes.ZW010x:
                            status += "ZW010x";
                            break;
                        case ChipTypes.ZW020x:
                            status += "ZW020x";
                            break;
                        case ChipTypes.ZW030x:
                            status += "ZW030x";
                            break;
                        case ChipTypes.ZW040x:
                            status += "ZW040x";
                            break;
                        default:
                            status += String.Format("{0}", ControllerManager.DocumentModel.SelectedChipType);
                            break;
                    }
                    ControllerManager.ShowMessage(status, false, false, true);

                    if (result)
                    {
                        if (ControllerManager.DocumentModel.SelectedChipType != (byte)ChipTypes.UNKNOWN)
                        {
                            ControllerManager.SetActiveViewByChipType((ChipTypes)ControllerManager.DocumentModel.SelectedChipType);
                        }
                    }
                }), Resources.MsgDetectDeviceProgress, false, 0x00);

            return result;
        }

        /// <summary>
        /// Detects the device.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        /// <returns></returns>
        public bool DetectDevice(bool isCommandPrompt)
        {
            bool result = false;
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                try
                {
                    IDevice device = ControllerManager.DeviceOpen();
                    if (device.SetProgrammingMode(true))
                    {
                        device.ReadSignatureBits();
                        ControllerManager.DocumentModel.SelectedChipType = device.ChipType;
                        Settings.Default.SelectedChipType = device.ChipType;
                        if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
                            device.SetProgrammingMode(false);
                        Settings.Default.Save();
                        if (device.ChipType != (byte)ChipTypes.UNKNOWN)
                        {
                            result = true;
                        }
                        else
                        {
                            ControllerManager.ShowMessage(Resources.ErrorCantDectDeviceSignature, true, isCommandPrompt);
                            result = false;
                        }
                    }
                }
                catch (Exception ex)
                {
                    System.Diagnostics.Debug.WriteLine("!!!" + ex.Message);
                    ControllerManager.ShowMessage(Resources.ErrorCantDetectDevice, true, isCommandPrompt);
                }
                finally
                {
                    System.Diagnostics.Debug.WriteLine("DetectDevice Close Device");
                    ControllerManager.DeviceClose();
                }
            }
            return result;
        }
        /// <summary>
        /// Upgrade the Programmers firmware.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        /// <param name="fileName">Name of the file.</param>
        public bool UpgradeFirmware(bool isCommandPrompt, string fileName)
        {
            bool result = false;
            if (isCommandPrompt)
            {
                ControllerManager.ShowMessage(Resources.MsgUpgradeFirmwareProgress, false, isCommandPrompt);
            }

            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                try
                {
                    IDevice device = ControllerManager.DeviceOpen();
                    if (device.SetBootLoaderMode(true))
                    {
                        System.Threading.Thread.Sleep(10);

                        IDeviceLed busyLed = device.Leds.GetByType(LedTypes.Busy);
                        if (busyLed != null)
                        {
                            busyLed.SetState(true);
                            if (device.Flash.Erase(false))
                            {
                                if (!String.IsNullOrEmpty(fileName) && File.Exists(fileName))
                                {
                                    byte[] flashDataRaw = HexFileHelper.BlankArray((int)Constants.FIRMWARE_SIZE, Constants.BLANK_VALUE);
                                    if (HexFileHelper.ReadIntelHexFile(fileName, flashDataRaw, (int)Constants.FIRMWARE_SIZE) == HexFileHelper.FILE_READ)
                                    {
                                        if (device.Flash.Write(false, flashDataRaw, null, true))
                                            result = true;
                                    }
                                    else
                                    {
                                        ControllerManager.ShowMessage(Resources.ErrorFlashHexFileNotSpecified, true, isCommandPrompt);
                                    }
                                }
                                else if (String.IsNullOrEmpty(fileName))
                                {
                                    ControllerManager.ShowMessage(Resources.ErrorFlashHexFileNotSpecified, true, isCommandPrompt);
                                }
                                else
                                {
                                    ControllerManager.ShowMessage(String.Format("{0} does not exists.", fileName), true, isCommandPrompt);
                                }
                            }
                            busyLed.SetState(false);
                        }
                        else
                        {
                            ControllerManager.ShowMessage(Resources.ErrorCantUpgradeFirmware, true, isCommandPrompt);
                        }
                        device.SetBootLoaderMode(false);
                        ControllerManager.ShowMessage(Resources.MsgFinished, true, isCommandPrompt, !isCommandPrompt);
                    }
                    else
                    {
                        ControllerManager.ShowMessage(Resources.ErrorCantSetBootLoaderMode, true, isCommandPrompt);
                    }
                }
                catch
                {
                    ControllerManager.ShowMessage(Resources.ErrorCantUpgradeFirmware, true, isCommandPrompt);
                }
                finally
                {
                    ControllerManager.DeviceClose();
                }
            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true, isCommandPrompt);
            }
            return result;
        }

        /// <summary>
        /// Upgrade the Programmers firmware.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        /// <returns>0 - firmware updated ok, 1 - if update is not needed, -1 - if firware is out of date and not updated.</returns>
        public int UpgradeLatestFirmware(bool isCommandPrompt)
        {
            int result = -1;
            int versionGood = 0;
            int revisionGood = 0;
            string versionID = "ZWAVE_PROGRAMMER_VERSION_VVI_4523@7740=";
            byte[] versionData = HexFileHelper.FromIntelHexData(Resources.ZWaveProgrammer_FW, Constants.BLANK_VALUE, 0, Constants.FIRMWARE_SIZE);
            if (versionData != null)
            {
                int i, j;
                for (i = 0; i < versionData.Length; i++)
                {
                    for (j = 0; j < versionID.Length; j++)
                    {
                        if ((char)versionData[i + j] != versionID[j])
                        {
                            break;
                        }
                    }
                    if (j == versionID.Length)
                    {
                        break;
                    }
                }
                if (i != versionData.Length)
                {
                    versionGood = (int)versionData[i + versionID.Length + 0];
                    revisionGood = (int)versionData[i + versionID.Length + 1];

                    int version = 0;
                    int revision = 0;
                    if (ControllerManager.Actions.ProgrammerActions.GetCurrentFirmwareVersion(isCommandPrompt, out version, out revision))
                    {
                        if (version < versionGood || (version == versionGood && revision < revisionGood))
                        {
                            try
                            {
                                if (!isCommandPrompt)
                                {
                                    if (ControllerManager.ShowConfirmMessage(Resources.MsgFirmwareUpdateConfirmation, false) == DialogResult.Yes)
                                    {
                                        String fwGoodHexFileName = Path.GetTempFileName();
                                        StreamWriter sw = File.AppendText(fwGoodHexFileName);
                                        sw.Write(Resources.ZWaveProgrammer_FW);
                                        sw.Close();
                                        if (ControllerManager.Actions.ProgrammerActions.UpgradeFirmware(false, fwGoodHexFileName))
                                        {
                                            if (ControllerManager.Actions.ProgrammerActions.GetCurrentFirmwareVersion(isCommandPrompt, out version, out revision))
                                            {
                                                ControllerManager.ShowMessage(Resources.MsgFirmwareUpgradedSuccesful, false, isCommandPrompt);
                                                result = 0;
                                            }
                                        }
                                        File.Delete(fwGoodHexFileName);
                                    }
                                }
                                else
                                {
                                    ControllerManager.ShowMessage(Resources.ErrorFirmwareIsOutOfDate, true, isCommandPrompt);
                                }
                            }
                            catch
                            {
                                ControllerManager.ShowMessage(Resources.ErrorCantUpgradeFirmware, true, isCommandPrompt);
                            }
                        }
                        else
                        {
                            result = 1;
                        }
                    }

                }
            }
            return result;
        }

        /// <summary>
        /// Get the version of current Programmers firmware.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        /// <param name="version">output of version nubmer</param>
        /// <param name="revision">output of revision nubmer</param>
        public bool GetCurrentFirmwareVersion(bool isCommandPrompt, out int version, out int revision)
        {
            bool result = false;
            version = 0;
            revision = 0;
            if (isCommandPrompt)
            {
                ControllerManager.ShowMessage(Resources.MsgGetFirmwareVersionProgress, false, isCommandPrompt);
            }

            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                try
                {
                    IDevice device = ControllerManager.DeviceOpen();
                    IDeviceLed busyLed = device.Leds.GetByType(LedTypes.Busy);
                    if (busyLed != null)
                    {
                        busyLed.SetState(true);
                        if (device.GetCurrentFirmwareVersion(out version, out revision))
                            result = true;
                        busyLed.SetState(false);
                    }
                    ControllerManager.ShowMessage(Resources.MsgFinished, false, isCommandPrompt, !isCommandPrompt);
                }
                catch
                {
                    ControllerManager.ShowMessage(Resources.ErrorCantGetFirmwareVersion, true, isCommandPrompt);
                }
                finally
                {
                    System.Diagnostics.Debug.WriteLine("GetCurrentFirmwareVersion Close Device");
                    ControllerManager.DeviceClose();
                }
            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true, isCommandPrompt);
            }
            return result;
        }

        /// <summary>
        /// Resets the Z-Wave module.
        /// </summary>
        public void ResetZWaveModule(bool isCommandPrompt)
        {
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                try
                {
                    IDevice device = ControllerManager.DeviceOpen();
                    ControllerManager.DoAction(new EventHandler(delegate
                    {
                        device.SetProgrammingMode(true);
                        device.SetProgrammingMode(false);

                    }), Resources.MsgResetZWaveModuleProgress, false, 0x00);

                    ControllerManager.ShowMessage(Resources.MsgFinished, false, isCommandPrompt, !isCommandPrompt);
                }
                catch
                {
                    ControllerManager.ShowMessage(Resources.ErrorCantResetZWaveModule, true, isCommandPrompt);
                }
                finally
                {
                    ControllerManager.DeviceClose();
                }
            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true, isCommandPrompt);
            }
        }

        /// <summary>
        /// Called to Write Lock Bits.
        /// </summary>
        /// <param name="disableFlashRead"><c>true</c>, if reading the flash should be disabled.</param>
        /// <param name="lockBootBlock"><c>true</c>, if writing the boot block should be disabled.</param>
        /// <param name="bootSectorSize">index of Boot Sector Size</param>
        public void OnLockBitsSet(bool disableFlashRead, bool lockBootBlock, int bootSectorSize)
        {
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    byte lockBits = MakeLockBitsByte(disableFlashRead, lockBootBlock, bootSectorSize);
                    LockBitsSet(lockBits, false);

                }), Resources.MsgLockBitsSetProgress, false, 0x00);
            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }

        }

        /// <summary>
        /// Make Lock Bits Byte.
        /// </summary>
        /// <param name="disableFlashRead"><c>true</c>, if reading the flash should be disabled.</param>
        /// <param name="lockBootBlock"><c>true</c>, if writing the boot block should be disabled.</param>
        /// <param name="bootSectorSize">index of Boot Sector Size</param>
        /// <returns>Lock Bits Byte for Set Lock Bits Command</returns>
        public byte MakeLockBitsByte(bool disableFlashRead, bool lockBootBlock, int bootSectorSize)
        {
            byte lockBits = 0xff;
            if (disableFlashRead)
            {
                lockBits = (byte)(lockBits & ~0x01);
            }
            if (lockBootBlock)
            {
                lockBits = (byte)(lockBits & ~0x10);
            }
            if (bootSectorSize != -1)
            {
                lockBits = (byte)((lockBits & ~0x0e) | ((bootSectorSize << 1) & 0x0e));
            }
            lockBits = (byte)(lockBits & ~0xe0);
            return lockBits;
        }

        /// <summary>
        /// Called to Write Lock Bits and Modem Bits for ZW040x.
        /// </summary>
        /// <param name="disableReadBack"><c>true</c>, if reading the OTP memory should be disabled.</param>
        /// <param name="disableDevMode"><c>true</c>, if Development mode of operation should be disabled.</param>
        /// <param name="setModem"><c>true</c>, for set modem mode.</param>
        public void OnLockBitsSetZW040x(bool disableReadBack, bool disableDevMode, bool setModem)
        {
            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    byte lockBits = MakeLockBitsByteZW040x(disableReadBack, disableDevMode);
                    LockBitsSet(lockBits, false);
                    // Test Set Modem Bits functionality (should be disabled)
                    //if (setModem)
                    //{
                    //    try
                    //    {
                    //        IDevice device = ControllerManager.DeviceOpen();
                    //        device.SetProgrammingMode(true);
                    //        device.ReadSignatureBits();
                    //        device.SetModemBits();
                    //        if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
                    //            device.SetProgrammingMode(false);
                    //        ControllerManager.DeviceClose();
                    //    }
                    //    catch
                    //    {
                    //    }
                    //}
                }), Resources.MsgLockBitsSetProgress, false, 0x00);
            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }

        }

        /// <summary>
        /// Make Lock Bits Byte for ZW040x.
        /// </summary>
        /// <param name="disableReadBack"><c>true</c>, if reading the OTP memory should be disabled.</param>
        /// <param name="disableDevMode"><c>true</c>, if Development mode of operation should be disabled.</param>
        /// <returns>Lock Bits Byte for Set Lock Bits Command for ZW040x</returns>
        public byte MakeLockBitsByteZW040x(bool disableReadBack, bool disableDevMode)
        {
            byte lockBits = 0x00;
            if (disableReadBack)
            {
                lockBits = (byte)(lockBits | 0x01);
            }
            if (disableDevMode)
            {
                lockBits = (byte)(lockBits | 0x02);
            }
            return lockBits;
        }

        /// <summary>
        /// Set the lock bits.
        /// </summary>
        /// <param name="lockBits">Lock Bits to set</param>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        public void LockBitsSet(byte lockBits, bool isCommandPrompt)
        {
            if (isCommandPrompt)
            {
                ControllerManager.ShowMessage(Resources.MsgLockBitsSetProgress, false, isCommandPrompt);
            }
            IDevice device = ControllerManager.DeviceOpen();
            try
            {
                if (device.SetProgrammingMode(true))
                {
                    device.ReadSignatureBits();
                    if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
                    {
                        IDeviceLed busyLed = device.Leds.GetByType(LedTypes.Busy);
                        if (busyLed != null)
                        {
                            busyLed.SetState(true);
                            bool write_cycle_time_result = true;
                            if (device.ChipType == (byte)ChipTypes.ZW010x
                                || device.ChipType == (byte)ChipTypes.ZW020x
                                || device.ChipType == (byte)ChipTypes.ZW030x)
                            {
                                write_cycle_time_result = device.SetWriteCycleTime();
                            }
                            if (write_cycle_time_result)
                            {
                                if (device.WriteLockBits(lockBits))
                                {
                                    ControllerManager.ShowMessage(Resources.MsgFinished, false, isCommandPrompt, !isCommandPrompt);
                                }
                                else
                                {
                                    ControllerManager.ShowMessage(Resources.ErrorCantLockBitsSet, true, isCommandPrompt);
                                }
                            }
                            else
                            {
                                ControllerManager.ShowMessage(Resources.ErrorCantLockBitsSet, true, isCommandPrompt);
                            }
                            busyLed.SetState(false);
                        }
                        if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
                            device.SetProgrammingMode(false);
                    }
                    else
                    {
                        ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
                    }
                }
                else
                {
                    ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
                }
            }
            catch
            {
                ControllerManager.ShowMessage(Resources.ErrorCantLockBitsSet, true, isCommandPrompt);
            }
            finally
            {
                ControllerManager.DeviceClose();
            }
        }

        /// <summary>
        /// Called when Read Lock Bits click.
        /// </summary>
        /// <param name="disableFlashRead"><c>true</c>, if reading the flash should be disabled.</param>
        /// <param name="lockBootBlock"><c>true</c>, if writing the boot block should be disabled.</param>
        /// <param name="bootSectorSize">index of Boot Sector Size</param>
        public void OnLockBitsRead(out bool disableFlashRead, out bool lockBootBlock, out int bootSectorSize)
        {
            bool loc_disableFlashRead = false;
            bool loc_lockBootBlock = false;
            int loc_bootSectorSize = -1;

            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    byte lockBits = LockBitsRead(false);

                    loc_disableFlashRead = (lockBits & 0x01) == 0;
                    loc_lockBootBlock = (lockBits & 0x10) == 0;
                    loc_bootSectorSize = (lockBits & 0x0e) >> 1;

                }), Resources.MsgLockBitsReadProgress, false, 0x00);
            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, true);
            }

            disableFlashRead = loc_disableFlashRead;
            lockBootBlock = loc_lockBootBlock;
            bootSectorSize = loc_bootSectorSize;
        }
        /// <summary>
        /// Called when Read Lock Bits and Modem Bits for ZW040x click.
        /// </summary>
        /// <param name="disableReadBack"><c>true</c>, if reading the OTP memory should be disabled.</param>
        /// <param name="disableDevMode"><c>true</c>, if Development mode of operation should be disabled.</param>
        /// <param name="setModem"><c>true</c>, for set modem mode.</param>
        public void OnLockBitsReadZW040x(out bool disableReadBack, out bool disableDevMode, out bool setModem)
        {
            bool loc_disableReadBack = false;
            bool loc_disableDevMode = false;
            bool loc_setModem = false;

            if (ControllerManager.DocumentModel.PortInfo != null)
            {
                ControllerManager.DoAction(new EventHandler(delegate
                {
                    ParseLockBitsByteZW040x(LockBitsRead(false),
                        out loc_disableReadBack,
                        out loc_disableDevMode,
                        out loc_setModem);

                }), Resources.MsgLockBitsReadProgress, false, 0x00);
            }
            else
            {
                ControllerManager.ShowMessage(Resources.MsgInterfaceNotSelected, false);
            }

            disableReadBack = loc_disableReadBack;
            disableDevMode = loc_disableDevMode;
            setModem = loc_setModem;
        }
        /// <summary>
        /// Parse the Lock Bits byte and Modem Bits for ZW040x.
        /// </summary>
        /// <param name="lockBits">Lock Bits byte.</param>
        /// <param name="disableReadBack"><c>true</c>, if reading the OTP memory should be disabled.</param>
        /// <param name="disableDevMode"><c>true</c>, if Development mode of operation should be disabled.</param>
        /// <param name="setModem"><c>true</c>, for set modem mode.</param>
        public void ParseLockBitsByteZW040x(byte lockBits, out bool disableReadBack, out bool disableDevMode, out bool setModem)
        {
            disableReadBack = (lockBits & 0x01) != 0;
            disableDevMode = (lockBits & 0x02) != 0;
            setModem = false;
            // Test Set Modem Bits functionality (should be disabled)
            // Undocumented feature of ZW040x: it's seems, that bit 2 internally used as Set Modem Bit. It can't be written by Set Lock Bits command, but it read as '1' by Read Lock Bits command if it is set by Set Modem Bits command.
            //setModem = (lockBits & 0x04) != 0;
        }
        /// <summary>
        /// Read the Lock Bits.
        /// </summary>
        /// <param name="isCommandPrompt">if set to <c>true</c>, application is in command prompt mode.</param>
        /// <returns>Returns the readed Lock Bits.</returns>
        public byte LockBitsRead(bool isCommandPrompt)
        {
            byte lockBits = 0xff;

            if (isCommandPrompt)
            {
                ControllerManager.ShowMessage(Resources.MsgLockBitsReadProgress, false, isCommandPrompt);
            }
            IDevice device = ControllerManager.DeviceOpen();
            try
            {
                if (device.SetProgrammingMode(true))
                {
                    device.ReadSignatureBits();
                    if (ControllerManager.Actions.CommonActions.CheckDeviceChipTypeCompatibility(device))
                    {
                        IDeviceLed busyLed = device.Leds.GetByType(LedTypes.Busy);
                        if (busyLed != null)
                        {
                            busyLed.SetState(true);

                            lockBits = device.ReadLockBits();

                            busyLed.SetState(false);
                            ControllerManager.ShowMessage(Resources.MsgFinished, false, isCommandPrompt, !isCommandPrompt);
                        }
                    }
                    else
                    {
                        ControllerManager.ShowMessage(Resources.MsgIncorrectChipType, true, isCommandPrompt);
                    }
                    if (!ControllerManager.DocumentModel.Settings.HoldDeviceInReset)
                        device.SetProgrammingMode(false);
                }
                else
                {
                    ControllerManager.ShowMessage(Resources.ErrorCantSetProgrammingMode, true, isCommandPrompt);
                }
            }
            catch
            {
                ControllerManager.ShowMessage(Resources.ErrorCantLockBitsRead, true, isCommandPrompt);
            }
            finally
            {
                ControllerManager.DeviceClose();
            }

            return lockBits;
        }

        private bool IsEmpty(IDeviceFlash deviceFlash, int pagesCount, out byte calibrationValue)
        {
            calibrationValue = 0;
            bool res = true;

            if (deviceFlash.Device.ChipType == (byte)ChipTypes.ZW040x)
            {	// if disable readback lock bit is set all pages readed as empty in any case,
                // so, assume that OTP is already progammed
                bool loc_disableReadBack = false;
                bool loc_disableDevMode = false;
                bool loc_setModem = false;

                ParseLockBitsByteZW040x(deviceFlash.Device.ReadLockBits(),
                        out loc_disableReadBack,
                        out loc_disableDevMode,
                        out loc_setModem);

                if (loc_disableReadBack)
                    return false;
            }

            if (pagesCount > Constants.PAGES_IN_FLASH || pagesCount < 1)
            {
                pagesCount = Constants.PAGES_IN_FLASH;
            }
            int page;
            for (page = 0; page < pagesCount; page++)
            {
                IFlashPage pageData = deviceFlash.ReadPage(true, page);
                if (pageData == null)
                {
                    break;
                }
                for (int i = 0; i < pageData.Buffer.Length; i++)
                {
                    if (page == 0 && i == Constants.CALIBRATION_VALUE_OTP_POS)
                    {
                        calibrationValue = pageData.Buffer[i];
                        continue;
                    }
                    if (pageData.Buffer[i] != Constants.BLANK_VALUE)
                    {
                        res = false;
                        break;
                    }
                }
            }
            if (page != pagesCount)
                res = false;
            return res;
        }


        internal void CalibrateDevice()
        {
            ControllerManager.DoAction(new EventHandler(delegate
                {
                    IDevice device = ControllerManager.DeviceOpen();
                    device.SetProgrammingMode(true);
                    device.ReadSignatureBits();
                    device.SetProgrammingMode(false);
                    sbyte calibrationValue = PerformCalibration(false, 0, device);
                    if (calibrationValue != 0 && calibrationValue != -128)
                    {
                        device.SetProgrammingMode(true);
                        IFlashPage page = device.Flash.ReadPage(ControllerManager.DocumentModel.IsAsic, 0);
                        if (page.Buffer.Length > Constants.CALIBRATION_VALUE_OTP_POS)
                        {
                            if (page.Buffer[Constants.CALIBRATION_VALUE_OTP_POS] == 0)
                            {
                                page.Buffer = new byte[page.Buffer.Length];
                                page.Buffer[Constants.CALIBRATION_VALUE_OTP_POS] = (byte)calibrationValue;
                                device.Flash.WritePage(page, ControllerManager.DocumentModel.IsAsic, true);
                            }
                        }

                        device.SetProgrammingMode(false);
                    }
                    else
                    {

                    }
                }), Resources.MsgCalibrationProgress, false, 0x00);
        }
    }
}
