/******************************* one_button.c *******************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2001
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 *
 * Description: Implements functions that detects if a button has
 *              been pressed shortly or is beeing held
 *
 *              The module has 2 functions that can be used by an
 *              application:
 *
 *              OneButtonInit()  Initializes the 10ms timer that polls the
 *                               button state
 *
 *              OneButtonLastAction() This function returns the last action
 *                                    performed with the button.
 *
 *              The definitions of the timers used to determine when a
 *              button is pressed or held is in the one_button.h file and
 *              they are defined in 10ms counts.
 *
 * Author:   Peter Shorty
 *
 * Last Changed By:  $Author: psh $
 * Revision:         $Revision: 22652 $
 * Last Changed:     $Date: 2012-05-02 17:07:42 +0200 (Wed, 02 May 2012) $
 *
 ****************************************************************************/
#ifdef PATCH_ENABLE
/****************************************************************************/
/* Include assembly MACRO definitions for patch insertions.                 */
/*                                                                          */
/* Define $SET (MAKE_PATCHABLE_CODE) for making patchable code destinned    */
/* for OTP or ROM memory.                                                   */
/* Undefine $RESET (MAKE_PATCHABLE_CODE) for making code containing patch   */
/* code destinned for RAM or FLASH memory.                                  */
/****************************************************************************/
#if defined(WORK_PATCH) || defined(STARTER_PATCH)
/* Making code containing patch code destinned for development RAM memory.  */
#pragma asm
$RESET (MAKE_PATCHABLE_CODE)
$INCLUDE (ZW_patch.inc)
#pragma endasm
/* Rename CODE class to CODE_PATCH */
#pragma userclass (code = PATCH)
/* Rename CONST class to CONST_PATCH */
#pragma userclass (const = PATCH)
/* Rename XDATA class to XDATA_PATCH */
#pragma userclass (xdata = PATCH)
#else
/* Making patchable code destinned for OTP or ROM memory.                   */
#pragma asm
$SET (MAKE_PATCHABLE_CODE)
$INCLUDE (ZW_patch.inc)
#pragma endasm
#endif /* elsif defined(WORK_PATCH) || defined(STARTER_PATCH) */
#endif /* PATCH_ENABLE */

/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
#include <ZW_patch.h>
#include <ZW_basis_api.h>
#include <ZW_timer_api.h>
#include <ZW_pindefs.h>
#include <ZW_evaldefs.h>
#include <one_button.h>
#ifdef ZW_BUTTON_UART_CONTROL
#include <ZW_uart_api.h>
#endif
/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/
PATCH_VARIABLE BYTE  buttonAction;
PATCH_VARIABLE BYTE  buttonCount;

PATCH_VARIABLE BYTE bTriplePress;
PATCH_VARIABLE BYTE bTriplePressHandle;

#define TIME_TRIPLE_PRESS     100 /* Triple press timeout is set to 1.5sec */

/****************************************************************************/
/*                              EXPORTED DATA                               */
/****************************************************************************/


/****************************************************************************/
/*                            PRIVATE FUNCTIONS                             */
/****************************************************************************/
extern void OneButtonTriplePressTimeout(void)
#ifdef PATCH_ENABLE
reentrant
#endif
;


/*================================   OneButtonPoll   =========================
**    Poll function that polls the button every 10ms
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
static void             /*RET  Nothing                  */
OneButtonPoll( void )  /*IN  Nothing                   */
{
  /* Check button state */
#ifdef ZW_BUTTON_UART_CONTROL
  if(ZW_UART_REC_STATUS)
  {
    BYTE recData = ZW_UART_REC_BYTE;
    switch (recData)
    {
      case 'B':
        {
          buttonAction = BUTTON_WAS_PRESSED;
          ZW_UART_SEND_BYTE('B');
        }
        break;
      case 'D':
        {
          buttonAction = BUTTON_IS_HELD;
          ZW_UART_SEND_BYTE('D');
        }
        break;
      case 'U':
        {
          buttonAction = BUTTON_WAS_RELEASED;
          ZW_UART_SEND_BYTE('U');
        }
        break;
      case 'R':
        {
          /*Reset the module*/
          ZW_UART_SEND_BYTE('R');
          while(ZW_UART_SEND_STATUS);
          ZW_WATCHDOG_ENABLE; /*reset asic*/
          while(1);
        }
      break;
      default:
        {
          ZW_UART_SEND_BYTE('!');
        }
      break;
    }
  }
#endif
  if (BUTTON_PRESSED())
  {
    /* Check if button is pushed long */
    if (buttonCount<=BUTTON_HELD_COUNT)
      buttonCount++;
    else
      buttonAction = BUTTON_IS_HELD;
  }
  else
  {
    if (buttonCount)  /* Button has been pressed and is now released */
    {
      if  (buttonCount>DEBOUNCE_COUNT)
      {
        if ((buttonCount<SHORT_PRESS_COUNT))
        {
          buttonAction = BUTTON_WAS_PRESSED;
          buttonCount = 0;

          /* Handle tripple press */
          bTriplePress++;

          if (bTriplePress == 1)
          {
            /* First press, start timer */
            bTriplePressHandle = TimerStart(OneButtonTriplePressTimeout, TIME_TRIPLE_PRESS, 1);
            if (bTriplePressHandle == 0xFF)
              bTriplePressHandle = 0;
          }
          else if (bTriplePress == 3)
          {
            /* Triple press detected */
            if (bTriplePressHandle)
            {
              buttonAction = BUTTON_TRIPLE_PRESS;
              TimerCancel(bTriplePressHandle);
              bTriplePressHandle = 0;
            }
            bTriplePress = 0;
          }
        }
        else if (buttonAction == BUTTON_IS_HELD)
        {
          buttonAction = BUTTON_WAS_RELEASED;
          buttonCount = 0;
        }
      }
    }
  }
}

/*=========================   OneButtonTriplePressTimeout   =================
**    Timeout function for the tripple press detection
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(OneButtonTriplePressTimeout)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(OneButtonTriplePressTimeout)
#pragma endasm
#endif
  bTriplePress = 0;
  bTriplePressHandle = 0;
}


/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/

/*===============================   OneButtonInit   ========================
**    This function initializes the one button polling
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
BOOL
PATCH_FUNCTION_NAME(OneButtonInit)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
  register BYTE bButtonPollHandler;

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(OneButtonInit)
#pragma endasm
#endif
/****************************************************************************/
/*                 Initialize PRIVATE TYPES and DEFINITIONS                 */
/****************************************************************************/
  buttonAction = 0;
  buttonCount = 0;
  bTriplePress = 0;
  bTriplePressHandle = 0;

  bButtonPollHandler = TimerStart(OneButtonPoll, 1, TIMER_FOREVER);

  if (bButtonPollHandler == 0xFF)
    return FALSE;
#ifdef ZW_BUTTON_UART_CONTROL
  ZW_UART_INIT(1152);
  ZW_UART_SEND_BYTE('s');
#endif
  return TRUE;
}


/*==============================   LastButtonAction   =======================
**    This function returns the last button action detected.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
BYTE
PATCH_FUNCTION_NAME(OneButtonLastAction)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
  register bTemp;

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(OneButtonLastAction)
#pragma endasm
#endif
  bTemp = buttonAction;
  if (buttonAction != BUTTON_IS_HELD)
    buttonAction = 0;
  return bTemp;
}
