/******************************* ctrl_learn.c *******************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2001
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 *
 * Description: This file contains a sample of how learn mode could be implemented
 *              on ZW0201 standard controller.
 *              The module works for both battery operated and always listening
 *              devices.
 *
 * Author:   Henrik Holm
 *
 * Last Changed By:  $Author: psh $
 * Revision:         $Revision: 22652 $
 * Last Changed:     $Date: 2012-05-02 17:07:42 +0200 (Wed, 02 May 2012) $
 *
 ****************************************************************************/
#ifdef PATCH_ENABLE
/****************************************************************************/
/* Include assembly MACRO definitions for patch insertions.                 */
/*                                                                          */
/* Define $SET (MAKE_PATCHABLE_CODE) for making patchable code destinned    */
/* for OTP or ROM memory.                                                   */
/* Undefine $RESET (MAKE_PATCHABLE_CODE) for making code containing patch   */
/* code destinned for RAM or FLASH memory.                                  */
/****************************************************************************/
#if defined(WORK_PATCH) || defined(STARTER_PATCH)
/* Making code containing patch code destinned for development RAM memory.  */
#pragma asm
$RESET (MAKE_PATCHABLE_CODE)
$INCLUDE (ZW_patch.inc)
#pragma endasm
/* Rename CODE class to CODE_PATCH */
#pragma userclass (code = PATCH)
/* Rename CONST class to CONST_PATCH */
#pragma userclass (const = PATCH)
/* Rename XDATA class to XDATA_PATCH */
#pragma userclass (xdata = PATCH)
#else
/* Making patchable code destinned for OTP or ROM memory.                   */
#pragma asm
$SET (MAKE_PATCHABLE_CODE)
$INCLUDE (ZW_patch.inc)
#pragma endasm
#endif /* elsif defined(WORK_PATCH) || defined(STARTER_PATCH) */
#endif /* PATCH_ENABLE */

/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
#include <ZW_patch.h>
#include <ZW_controller_api.h>
#include <ZW_uart_api.h>
#include <ctrl_learn.h>

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/
#define LEARN_MODE_CLASSIC_TIMEOUT        2   /* Timeout count for classic innlusion */
#ifdef AV_REMOTE
#define LEARN_MODE_NWI_TIMEOUT          72 /* Timeout count for network wide innlusion is 3 minutes i a simple_AV_remote*/
#else
#define LEARN_MODE_NWI_TIMEOUT          720 /* Timeout count for network wide innlusion */
#endif
#define NWI_BASE_TIMEOUT                 50 /* Base delay for sending out inclusion req. */

#define MAX_NWI_REQUEST_TIMEOUT          27 /* Max number of increments in inclusiob request timeout */

#define ZW_LEARN_NODE_STATE_TIMEOUT 250     /* Learn mode base timeout  */

PATCH_VARIABLE BYTE learnStateHandle
#ifndef WORK_PATCH
 = 0xFF;
#endif
PATCH_VARIABLE BYTE bRequestTimeoutHandle
#ifndef WORK_PATCH
 = 0xFF;
#endif

PATCH_VARIABLE WORD wInclusionTimeoutCount;
PATCH_VARIABLE BYTE bRequestNWITimeoutCount;
PATCH_VARIABLE BYTE bSavetRequestNWITimeout;

PATCH_VARIABLE BYTE bLearnStarted
#ifndef WORK_PATCH
 = FALSE;
#endif
PATCH_VARIABLE BYTE bLastLearnMode;

/****************************************************************************/
/*                              EXPORTED DATA                               */
/****************************************************************************/
PATCH_VARIABLE BOOL learnInProgress  /* Application can use this flag to check if learn mode is active */
#ifndef WORK_PATCH
 = FALSE;
#endif

void StartLearnInternal(BYTE bMode)
#ifdef PATCH_ENABLE
reentrant
#endif
;

/****************************************************************************/
/*                               PROTOTYPES                                 */
/****************************************************************************/
void StopLearnInternal()
#ifdef PATCH_ENABLE
reentrant
#endif
;

/****************************************************************************/
/*                            PRIVATE FUNCTIONS                             */
/****************************************************************************/


/*============================   LearnModeCompleted   ========================
**    Function description
**      Callback which is called on learnmode completes
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void									/*RET	Nothing */
PATCH_FUNCTION_NAME(LearnModeCompleted)(
  LEARN_INFO *glearnNodeInfo)					/* IN resulting nodeID */
#ifdef PATCH_ENABLE
reentrant
#endif
{
  register BYTE bStatus;

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(LearnModeCompleted)
#pragma endasm
#endif
  /* No node info transmit during neighbor discovery */
  ZW_DEBUG_SEND_BYTE('c');
  ZW_DEBUG_SEND_NUM(glearnNodeInfo->bStatus);

  bStatus = glearnNodeInfo->bStatus;

  /* Stop sending inclusion requests */
  if (bRequestTimeoutHandle != 0xff)
  {
    ZW_DEBUG_SEND_BYTE('t');
      ZW_DEBUG_SEND_NUM(bRequestTimeoutHandle);
  	TimerCancel(bRequestTimeoutHandle);
  	bRequestTimeoutHandle = 0xff;
  }

  if (bStatus == LEARN_MODE_STARTED)
  {
    learnInProgress = TRUE;
  }
  else if ((bStatus == LEARN_MODE_DONE) || (bStatus == LEARN_MODE_FAILED))
  {
    /* Assignment was complete. Tell application */
  	if (learnInProgress == TRUE)
  	{
  	  learnInProgress = FALSE;
  	  StopLearnInternal();

  	  if (bStatus == LEARN_MODE_DONE)
  	  {
        LearnCompleted(glearnNodeInfo->bSource, TRUE);
      }
      else
      {
        /* Restart learn mode */
        StartLearnInternal(bLastLearnMode);
      }
    }
  }
}


/*============================   EndLearnNodeState   ========================
**    Function description
**      Timeout function that disables learnmode.
**      Should not be called directly.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(EndLearnNodeState)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(EndLearnNodeState)
#pragma endasm
#endif
  if (!(--wInclusionTimeoutCount))
  {
    ZW_DEBUG_SEND_BYTE('E');

    if (!learnInProgress)
    {
      StopLearnInternal();
      LearnCompleted(0, FALSE);
    }
    return;
  }
}


/*============================   SendExplorerRequest   ========================
**    Function description
**      Timeout function that sends out a explorer inclusion reuest
**      Should not be called directly.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(SendExplorerRequest)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(SendExplorerRequest)
#pragma endasm
#endif
  if (!(--bRequestNWITimeoutCount))
  {
    ZW_DEBUG_SEND_BYTE('R');

    ZW_ExploreRequestInclusion();

    /* Increase timeout if we havent reached max */
    if (bSavetRequestNWITimeout < MAX_NWI_REQUEST_TIMEOUT)
      bSavetRequestNWITimeout++;

    bRequestNWITimeoutCount = bSavetRequestNWITimeout;
  }
}


/*============================   StopLearnInternal   ========================
**    Function description
**      - Disables learn mode
**      - Stop timer
**      - Disable network wide inclusion
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(StopLearnInternal)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StopLearnInternal)
#pragma endasm
#endif
  ZW_DEBUG_SEND_BYTE('l');

  ZW_SetLearnMode(FALSE, NULL);

  if (learnStateHandle != 0xff)
  {
    ZW_DEBUG_SEND_BYTE('w');
      ZW_DEBUG_SEND_NUM(learnStateHandle);
  	TimerCancel(learnStateHandle);
  	learnStateHandle = 0xff;
  }
  if (bRequestTimeoutHandle != 0xff)
  {
    ZW_DEBUG_SEND_BYTE('t');
      ZW_DEBUG_SEND_NUM(bRequestTimeoutHandle);
  	TimerCancel(bRequestTimeoutHandle);
  	bRequestTimeoutHandle = 0xff;
  }

  bLearnStarted = FALSE;
}


/*============================   StartLearnInternal   ======================
**    Function description
**      Call this function from the application whenever learnmode
**      should be enabled.
**      This function do the following:
**        - Set the Slave in Learnmode
**        - Starts a one second timeout after which learn mode is disabled
**        - Broadcast the NODEINFORMATION frame once when called.
**      LearnCompleted will be called if a controller performs an assignID.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(StartLearnInternal)(
  BYTE bMode)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StartLearnInternal)
#pragma endasm
#endif
  ZW_DEBUG_SEND_BYTE('L');
  ZW_DEBUG_SEND_NUM(bMode);

  bLearnStarted = TRUE;
  ZW_SetLearnMode(bMode, LearnModeCompleted);

  if (learnStateHandle == 0xFF)
  {
    bLastLearnMode = bMode;

    if (bMode == ZW_SET_LEARN_MODE_CLASSIC)
    {
      /*Disable Learn mode after 1 sec.*/
      wInclusionTimeoutCount = LEARN_MODE_CLASSIC_TIMEOUT;
    }
    else
    {
      /*Disable Learn mode after 240 sec.*/
      wInclusionTimeoutCount = LEARN_MODE_NWI_TIMEOUT;

      /* Start timer sending  out a explore inclusion request */
      bSavetRequestNWITimeout = 1;
      bRequestNWITimeoutCount = bSavetRequestNWITimeout;
      bRequestTimeoutHandle = TimerStart(SendExplorerRequest,
                                         NWI_BASE_TIMEOUT + (ZW_Random() & 0x3F), /* base + random(0..63) */
                                         TIMER_FOREVER);
      ZW_DEBUG_SEND_BYTE('T');
      ZW_DEBUG_SEND_NUM(bRequestTimeoutHandle);
    }

    learnStateHandle = TimerStart(EndLearnNodeState, ZW_LEARN_NODE_STATE_TIMEOUT, TIMER_FOREVER);

    ZW_DEBUG_SEND_BYTE('W');
    ZW_DEBUG_SEND_NUM(learnStateHandle);
  }
}


/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/
/*============================   StartLearnModeNow   ======================
**    Function description
**      Call this function from the application whenever learnmode
**      should be enabled.
**      This function do the following:
**        - Set the Controller in Learnmode
**        - Starts a one second timeout after which learn mode is disabled
**        - Broadcast the NODEINFORMATION frame once when called.
**      LearnCompleted will be called if a controller performs an assignID.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(StartLearnModeNow)(
  BYTE bMode)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StartLearnModeNow)
#pragma endasm
#endif
  /* If learn is in progress then just exit */
  if (learnInProgress)
    return;

  if (bLearnStarted) /* Learn mode is started, stop it */
  {
    StopLearnInternal();
  }

  /* Start Learn mode */

  if (bMode)
    StartLearnInternal(bMode);
}


/*============================   StopLearnModeNow   ======================
**    Function description
**      Call this function from the application whenever learnmode
**      should be disabled.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
BYTE
PATCH_FUNCTION_NAME(StopLearnModeNow)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StopLearnModeNow)
#pragma endasm
#endif
  if (bLearnStarted && (!learnInProgress))
  {
    StopLearnInternal();
    return TRUE;
  }

  return FALSE;
}


/*==========================   ReArmLearnModeTimeout   =======================
**    Function description
**      Rearms the LearnMode timout handler and thereby extending the time
**      that the controller are in LearnMode/Receive
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
ReArmLearnModeTimeout()
{
  if (learnStateHandle != 0xFF)
  {
    TimerRestart(learnStateHandle);
  }
}
