/******************************* association.c *******************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2001
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 *
 * Description: Implements functions that make is easy to support
 *              Association Command Class
 *
 * Author:   Henrik Holm
 *
 * Last Changed By:  $Author: psh $
 * Revision:         $Revision: 22652 $
 * Last Changed:     $Date: 2012-05-02 17:07:42 +0200 (Wed, 02 May 2012) $
 *
 ****************************************************************************/
#ifdef PATCH_ENABLE
/****************************************************************************/
/* Include assembly MACRO definitions for patch insertions.                 */
/*                                                                          */
/* Define $SET (MAKE_PATCHABLE_CODE) for making patchable code destinned    */
/* for OTP or ROM memory.                                                   */
/* Undefine $RESET (MAKE_PATCHABLE_CODE) for making code containing patch   */
/* code destinned for RAM or FLASH memory.                                  */
/****************************************************************************/
#if defined(WORK_PATCH) || defined(STARTER_PATCH)
/* Making code containing patch code destinned for development RAM memory.  */
#pragma asm
$RESET (MAKE_PATCHABLE_CODE)
$INCLUDE (ZW_patch.inc)
#pragma endasm
/* Rename CODE class to CODE_PATCH */
#pragma userclass (code = PATCH)
/* Rename CONST class to CONST_PATCH */
#pragma userclass (const = PATCH)
/* Rename XDATA class to XDATA_PATCH */
#pragma userclass (xdata = PATCH)
#else
/* Making patchable code destinned for OTP or ROM memory.                   */
#pragma asm
$SET (MAKE_PATCHABLE_CODE)
$INCLUDE (ZW_patch.inc)
#pragma endasm
#endif /* elsif defined(WORK_PATCH) || defined(STARTER_PATCH) */
#endif /* PATCH_ENABLE */

/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
#include <ZW_patch.h>
#include <ZW_basis_api.h>
#include <ZW_uart_api.h>
#include <association.h>


#include <ZW_TransportLayer.h>
/* txBuf should be implemented elsewhere. It will be used to send */
/* ASSOCIATION commandclass frames */
extern ZW_APPLICATION_TX_BUFFER txBuf;

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/

PATCH_VARIABLE ASSOCIATION_GROUP groups[MAX_ASSOCIATION_GROUPS];

PATCH_VARIABLE BYTE groupReportSourceNode;

PATCH_VARIABLE /*I*/BYTE indx;

PATCH_VARIABLE BOOL associationReportStarted
#ifndef WORK_PATCH
 = FALSE
#endif
;

void
AssociationSendComplete(
  BYTE bStatus)
#ifdef PATCH_ENABLE
reentrant
#endif
;

//BOOL
//AssociationSendBasicSet(
//  BOOL doSend);

/****************************************************************************/
/*                              EXPORTED DATA                               */
/****************************************************************************/

/****************************************************************************/
/*                            PRIVATE FUNCTIONS                             */
/****************************************************************************/

/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/

/*============================   AssociationSendReport   ======================
**    Function description
**      Sends the ASSOCIATION_REPORT for the supplied Group to the node specified.
**      Transmit options are grapped from BYTE txOption
**    Side effects:
**
**--------------------------------------------------------------------------*/
BOOL                    /*RET TRUE if started FALSE if not */
PATCH_FUNCTION_NAME(AssociationSendReport)(
  BYTE groupNo,         /* IN Group Number to send report for */
  BYTE destNode,        /* IN Destination to send to */
  BYTE txOption         /* IN Send Data transtmit options */
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
  BYTE nodeCount;
  BYTE bGroupIndex;

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(AssociationSendReport)
#pragma endasm
#endif
  if (!associationReportStarted)
  {
    associationReportStarted = TRUE;
    nodeCount = 0;
    groupReportSourceNode = destNode;
#if MAX_ASSOCIATION_GROUPS > 1
    /*If the node supports several groups. Set the correct group index*/
    if(groupNo!=0)
    {
      bGroupIndex = groupNo-1;
    }
    if(groupNo> MAX_ASSOCIATION_GROUPS)
    {
      bGroupIndex = 0;
    }
#else
  /*If node only support one group. Ignore the group ID requested (As defined in Device class specification*/
    bGroupIndex = 0;
#endif
    txBuf.ZW_AssociationReport1byteFrame.cmdClass           = COMMAND_CLASS_ASSOCIATION;
    txBuf.ZW_AssociationReport1byteFrame.cmd                = ASSOCIATION_REPORT;
    txBuf.ZW_AssociationReport1byteFrame.groupingIdentifier = bGroupIndex+1;
    txBuf.ZW_AssociationReport1byteFrame.maxNodesSupported  = MAX_ASSOCIATION_IN_GROUP;
    txBuf.ZW_AssociationReport1byteFrame.reportsToFollow = 0; //Nodes fit in one report
    for (indx = 0; indx < MAX_ASSOCIATION_IN_GROUP; indx++)
    {
      if (groups[bGroupIndex].nodeID[indx])
      {
        *(&txBuf.ZW_AssociationReport1byteFrame.nodeid1 + nodeCount) = groups[bGroupIndex].nodeID[indx];
        nodeCount++;
      }
    }
	// TO#02827  	Not all steps of association done on secure binary sensor is carried out securely
//    if(!ZW_SEND_DATA(groupReportSourceNode,
//                 (BYTE*)&txBuf,
//                 sizeof(ZW_ASSOCIATION_REPORT_1BYTE_FRAME) - 1 + nodeCount,
//                 txOption,
//                 AssociationSendComplete
//				 ))

    if(!Transport_SendRequest(groupReportSourceNode,
                 (BYTE*)&txBuf,
                 sizeof(ZW_ASSOCIATION_REPORT_1BYTE_FRAME) - 1 + nodeCount,
                 txOption,
                 AssociationSendComplete
				 , FALSE
				 ))

    {
      associationReportStarted = FALSE;
    }
    return associationReportStarted;
  }
  else
  {
    return FALSE;
  }
}

/*========================   AssociationSendGroupings   ======================
**    Function description
**      Sends the number of groupings supported.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
BOOL
PATCH_FUNCTION_NAME(AssociationSendGroupings)(  /*RET TRUE If started, FALSE if not*/
  BYTE destNode,                                /* IN destination to send to*/
  BYTE txOption                                 /* IN Send data transmit options*/
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(AssociationSendGroupings)
#pragma endasm
#endif
  if (!associationReportStarted)
  {
    associationReportStarted = TRUE;
    txBuf.ZW_AssociationGroupingsReportFrame.cmdClass = COMMAND_CLASS_ASSOCIATION;
    txBuf.ZW_AssociationGroupingsReportFrame.cmd = ASSOCIATION_GROUPINGS_REPORT;
    txBuf.ZW_AssociationGroupingsReportFrame.supportedGroupings = MAX_ASSOCIATION_GROUPS;
    groupReportSourceNode = destNode;

	// TO#02827  	Not all steps of association done on secure binary sensor is carried out securely
//    if(!ZW_SEND_DATA(groupReportSourceNode, (BYTE*)&txBuf, sizeof(txBuf.ZW_AssociationGroupingsReportFrame), txOption, AssociationSendComplete))
    if(!Transport_SendRequest(groupReportSourceNode, (BYTE*)&txBuf, sizeof(txBuf.ZW_AssociationGroupingsReportFrame), txOption, AssociationSendComplete,FALSE))

    {
      associationReportStarted = FALSE;
    }
  }
  else
  {
    return FALSE;
  }
  return associationReportStarted;
}


void
PATCH_FUNCTION_NAME(AssociationSendComplete)(
  BYTE bStatus
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(AssociationSendComplete)
#pragma endasm
#endif
  associationReportStarted = FALSE; /* Done */
}


/*============================   AssociationAdd   ======================
**    Function description
**      Adds the nodes in nodeIdP to the group
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(AssociationAdd)(
  BYTE group,     /*IN group number to add nodes to*/
  BYTE_P nodeIdP, /*IN pointer to list of nodes*/
  BYTE noOfNodes  /*IN number of nodes in List*/
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
  BYTE i;
  BYTE tempID;

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(AssociationAdd)
#pragma endasm
#endif
  if (group)
  {
    group--;
  }
  ZW_DEBUG_SEND_BYTE('r');
  ZW_DEBUG_SEND_NUM(noOfNodes);

  for (i = 0; i < noOfNodes; i++)
  {
    BYTE vacant = 0xff;
    tempID = *(nodeIdP + i);
    for (indx = 0; indx < MAX_ASSOCIATION_IN_GROUP; indx++)
    {
      if (groups[group].nodeID[indx])
      {
        if (groups[group].nodeID[indx] == tempID)
        {
          break;  /* Allready in */
        }
      }
      else
      {
        if (vacant == 0xff)
        {
          vacant = indx;
        }
      }
    }
    if (vacant != 0xff)
    {
      groups[group].nodeID[vacant] = tempID;
      AssociationStoreAll();
    }
  }
}


/*============================   AssociationRemove   ======================
**    Function description
**      Removes association members from specified group
**      If noOfNodes = 0 group is removed
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(AssociationRemove)(
  BYTE group,      /*IN group number to remove nodes from*/
  BYTE_P nodeIdP,  /*IN pointer to array of nodes to remove*/
  BYTE noOfNodes   /*IN number of nodes in list*/
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
  BYTE i;

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(AssociationRemove)
#pragma endasm
#endif
  if (group)
  {
    group--;
  }
  ZW_DEBUG_SEND_BYTE('a');
  ZW_DEBUG_SEND_NUM(noOfNodes);
  ZW_DEBUG_SEND_NUM(group);
  if (noOfNodes)
  {
    for (i = 0; i < noOfNodes; i++)
    {
      for (indx = 0; indx < MAX_ASSOCIATION_IN_GROUP; indx++)
      {
        if (groups[group].nodeID[indx] == *(nodeIdP+i))
        {
          groups[group].nodeID[indx] = 0;
          break;  /* Found */
        }
      }
    }
  }
  else
  {
    /* If no supplied nodeIDs all nodes should be removed from group */
    indx = MAX_ASSOCIATION_IN_GROUP;
    do
    {
      groups[group].nodeID[indx - 1] = 0;
    } while (--indx);
  }
  if (indx < MAX_ASSOCIATION_IN_GROUP)
  {
    AssociationStoreAll();
  }
}
