/************************** ZW_TransportZIP.c ***************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2009
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 *
 * Description: Implements functions for transporting frames over the
 *               secure Z-Wave Network
 *
 * Author:   Valeriy Vyshnyak
 *
 * Last Changed By:  $Author: vvi $
 * Revision:         $Revision: 13417 $
 * Last Changed:     $Date: 2009-03-10 16:17:52 +0200 (Вв, 10 Бер 2009) $
 *
 ****************************************************************************/

/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
#ifdef ZW_CONTROLLER
#include <ZW_controller_api.h>
#else
#include <ZW_slave_api.h>
#endif
#include <ZW_TransportLayer.h>
#include <ZW_TransportZIP.h>

#include <ZW_uart_api.h>

#include "ZWZip6lowPanIphc.h"

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/
#define LOWPAN_PACKET_SEND_SIZE_MAX      40


#define OFFSET_CLASSCMD                       0x00
#define OFFSET_CMD                            0x01
#define OFFSET_PARAM_1                        0x02
#define OFFSET_PARAM_2                        0x03
#define OFFSET_PARAM_3                        0x04
#define OFFSET_PARAM_4                        0x05

BYTE	     lpDataSendTag = 0x0;
BOOL	     isLowPanPacket	= FALSE;
BOOL	     isDataCompressed	= FALSE;
BYTE	     lpDataTag = 0x0;
BYTE	     ipPacketSize;
WORD 		lpOutSize	= 0x0;
WORD 		ipPacketAllSize = 0;
/* new eeprom variables add only before this magic byte variable (and don't forget to change offset of magic byte!!!) */

#define ZIP_FIRST_SEGMENT_DATA_SIZE_MAX (META_DATA_MAX_DATA_SIZE - \
  (sizeof(ZW_COMMAND_ZIP_IP_DATAGRAM_SEGMENT_1BYTE_FRAME) - 1))
#define ZIP_NEXT_SEGMENT_DATA_SIZE_MAX (ZIP_FIRST_SEGMENT_DATA_SIZE_MAX + 1)

#define IP_PACKET_SIZE_MAX      160

/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/
TRANSPORT_STATUS_CALLBACK ReportTransportStatus = NULL;

#define TZIP_CMD_HANDLING_DEFAULT   0
#define TZIP_CMD_HANDLING_ZIP       1
#define TZIP_CMD_HANDLING_NATIVE    2
BYTE zipCmdHandling = TZIP_CMD_HANDLING_DEFAULT;  /* type of calling of ApplicationCommandHandler*/

BOOL nodeIncludedZIP = FALSE;                     /* TRUE if node is included as Z/IP node */

BOOL ipPacketReceivingInProgress = FALSE;         /* set by receiver of IP packet to TRUE during receiving the segments of IP packet */
BYTE ipPacket[IP_PACKET_SIZE_MAX];               /* original received IP packet */

BYTE lpComprPacket[IP_PACKET_SIZE_MAX];


WORD ipPacketSrcPort = TRANSPORT_ZIP_SOURCE_PORT_MIN; /* current source port of this node */
WORD ipPacketReceivedSize = 0;                    /* size of the received packet (used only during assembling the received IP packet) */
BYTE ipPacketSourceNodeID = 0;                    /* Source NodeID, from which we get IP packet. For example Z/IP Ack or ICMP echo reply will be sent to this node. */
BYTE ipPacketSourceTxOptions = 0;                 /* Transmit option flags for reports on received IP packet. For ex. Z/IP Ack or ICMP echo reply will be sent with this transmit options. */
ZW_COMMAND_ZIP_PACKET_1BYTE_FRAME *ipPacketZIPPacket = NULL; /* stored for ParseZIP address of Z/IP packet (properties1,2 members are not valid) */
BYTE ipPacketZIPPacketSize = 0;                         /* stored for ParseZIP Z/IP packet size */
BYTE ipPacketZIPPacketProperties2 = 0;            /* stored for ParseZIP properties2 member of received Z/IP packet */

ZW_APPLICATION_META_TX_BUFFER ipPacketZIPBuf;
ZW_APPLICATION_META_TX_BUFFER *ipPacketZIPBuffer = &ipPacketZIPBuf; /* buffer for transmitting Z-Wave Z/IP command */

BOOL ipPacketSendingInProgress = FALSE;           /* set by SendGatewayDataIP to TRUE during sending the segments of IP packet */
BYTE *ipPacketUnsentPart = NULL;                  /* Pointer to the part of IP packet, which must be send */
WORD ipPacketUnsentSize = 0;                      /* Size of the part of IP packet, which must be send */
BYTE ipZIPPacketSeqNo = 0;                        /* Sequence number of the Z/IP packet in IP packet */
VOID_CALLBACKFUNC_BYTE ipZIPPacketCompletedFunc = NULL;  /*  Transmit completed callback function, args: BYTE txStatus - IN Transmit status */
BYTE ipSegmentsSeqNo = 0;                         /* Sequence number of segment of IP packet */
BYTE ipSegmentsDestNodeID = 0;                    /* Destination node ID in Z-Wave Network (Z/IP Router Node ID) */
BYTE ipSegmentsTxOpt = 0;                         /* Transmit option flags for native ZW_SendData */

#define COMMAND_ZIP_PACKET_HEADER_LEN       5     /*header is only: cmdClass, cmd, properties1, 2, seqNo*/

#define COMMAND_ZIP_LOW_PAN_PACKET_HEADER_LEN       4
/****************************************************************************/
/*                              EXPORTED DATA                               */
/****************************************************************************/
extern	BYTE bMyNodeID;
extern	BYTE myHomeID[4];

/****************************************************************************/
/*                            PRIVATE FUNCTIONS                             */
/****************************************************************************/

/*===========================================================================
**    Function description
**      Calculate checksum for IP/ICMP packets
**    Side effects:
**
**--------------------------------------------------------------------------*/
unsigned short                            /* RET IP header checksum */
SendDataIP6_csum(
  void *buf,                              /* IN  Pointer to IP header */
  int nbytes)                             /* IN  Number of bytes in IP header or ICMP packet */
{
  unsigned long sum = 0x3A;

  while (nbytes > 1)
  {
    sum += 256 * (*((unsigned char *) buf)++);
    sum += *((unsigned char *) buf)++;
    nbytes -= 2;
  }
  if (nbytes)
    sum += 256 * (*((unsigned char *) buf));
  while (sum >> 16)
    sum = (sum >> 16) + (sum & 0xffff);

  return (unsigned short) (~sum);
}

unsigned int IntToHostOrder(unsigned int convert)
{
	return ((convert>>8)&0x00ff) | ((convert<<8)&0xff00);
}

/*===========================================================================
**    Function description
**      Calculate UDP checksum
**    Side effects:
**
**--------------------------------------------------------------------------*/
unsigned short                            /* RET UDP checksum */
SendDataIP_udp_csumIPv6(
  iphdr6* iph6)                             /* IN  Pointer to the IP header with UDP header, UDP payload, IP source and destination address in IP header */
{
  udphdr* udph;
  unsigned long sum = 0;
  unsigned int i;
  udphdr *buf = (udphdr*) (((unsigned char*) iph6) + IPV6_HEADER_SIZE_MIN);
  int nbytes = iph6->ip6_ctlun.ip6_un1.ip6_un1_plen;
  udph = ((BYTE*) iph6) + IPV6_HEADER_SIZE_MIN;

  // Add sum of UDP header length field and protocol
  sum += iph6->ip6_ctlun.ip6_un1.ip6_un1_nxt;
  sum += 256 * ((unsigned char*) &buf->len)[0];
  sum += ((unsigned char*) &buf->len)[1];

  // Sum of UDP header and data
  while (nbytes > 1)
  {
    sum += 256 * (*((unsigned char*) buf)++);
    sum += *((unsigned char*) buf)++;
    if (sum & 0x80000000)
      sum = (sum & 0xFFFF) + (sum >> 16);
    nbytes -= 2;
  }
  if (nbytes)
    sum += 256 * (*((unsigned char*) buf));

  // Add sum of IP source and destination
  for (i = 0; i < 16;)
  {
    sum += 256 * ((unsigned char*) &iph6->ip6_src)[i++];
    sum += ((unsigned char*) &iph6->ip6_src)[i++];
  }

  for (i = 0; i < 16;)
  {
    sum += 256 * ((unsigned char*) &iph6->ip6_dst)[i++];
    sum += ((unsigned char*) &iph6->ip6_dst)[i++];
  }

  while (sum >> 16)
    sum = (sum >> 16) + (sum & 0xffff);

  return (unsigned short) (~sum);
}

short ip_checksum(unsigned char * buf, int len)
{
unsigned short word16;
int sum=0;
int i;

	sum = 0x3A;
	for (i=0;i<len;i=i+2)
	{
		word16 =((buf[i]<<8)&0xFF00)+(buf[i+1]&0xFF);
		sum = (int)sum + (int)word16;
	}

	while (sum > 0xFFFF)
	{
	  sum = (sum & 0xFFFF)+(sum >> 16);
	}
	// one's complement the result
	return (short) ~sum;
}

/*===========================================================================
**    Function description
**      Release IP Packet buffer for next receiving or sending of IP packet
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                      /* RET nothing */
SendDataIP_ReleaseIPPacketBuf(
  void)
{

  __u16 udpPort;
  iphdr* iph = ((iphdr*)ipPacket);
  iphdr6* iph6 = ((iphdr6*)ipPacket);
  udphdr* udph;

  if(!ipPacketReceivingInProgress)
  {
    /* exchange back to original the destination and source in IP packet */
    if(((iph6->ip6_ctlun.ip6_un2_vfc & 0xf0)>>4) == IPV6_VERSION)
    {
      in6_addr ipAddr6;

	  memcpy(&ipAddr6,		&iph6->ip6_src,	IPV6_ADDRESS_SIZE);
	  memcpy(&iph6->ip6_src,&iph6->ip6_dst,	IPV6_ADDRESS_SIZE);
	  memcpy(&iph6->ip6_dst,&ipAddr6,		IPV6_ADDRESS_SIZE);

      switch (iph6->ip6_ctlun.ip6_un1.ip6_un1_nxt)
      {
	      case IPPROTO_ICMPV6:
	      {
	        break;
	      }
	      case IPPROTO_UDP:
	      {
	        udph = ((BYTE*) iph6) + IPV6_HEADER_SIZE_MIN;
	        udpPort = udph->source;
	        udph->source = udph->dest;
	        udph->dest = udpPort;
	        break;
	      }
      }
    }
  }
  ipPacketSendingInProgress = FALSE;
  ipPacketUnsentPart = NULL;
  ipPacketUnsentSize = 0;
  ipSegmentsSeqNo = 0;
  ipSegmentsDestNodeID = 0;
  ipSegmentsTxOpt = 0;

  ZW_DEBUG_SEND_BYTE('1');
  ZW_DEBUG_SEND_BYTE('1');
}

/*===========================================================================
**    Function description
**      Transmit next Z/IP segment of IP packet, if needed.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                                /* RET - nothing */
SendDataIP_NextZIPSegment(
  BYTE txStatus)                    /* IN Transmit status */
{
  udphdr* udph;
  ZW_COMMAND_ZIP_PACKET_1BYTE_FRAME* ziph;
  BYTE* zwcmd;
  BYTE len,i;

  if (ipPacketUnsentSize != 0 && txStatus == TRANSMIT_COMPLETE_OK)
  {
    ZW_DEBUG_SEND_BYTE('j');


  if(isLowPanPacket)
  {
	 ZW_DEBUG_SEND_NUM(ipPacketAllSize);
	 ZW_DEBUG_SEND_NUM(ipPacketUnsentSize);
	 ipPacketZIPBuffer->ZW_LowpanSubsequentFragment1byteFrame.cmdClass = 		  COMMAND_CLASS_ZIP_6LOWPAN;
	 ipPacketZIPBuffer->ZW_LowpanSubsequentFragment1byteFrame.cmd_Properties1 = LOWPAN_SUBSEQUENT_FRAGMENT|((ipPacketAllSize & 0x0700)>>8);
	 ipPacketZIPBuffer->ZW_LowpanSubsequentFragment1byteFrame.datagramSize2 =   (ipPacketAllSize & 0x0ff);
	 ipPacketZIPBuffer->ZW_LowpanSubsequentFragment1byteFrame.datagramTag =     lpDataSendTag;
	 ipPacketZIPBuffer->ZW_LowpanSubsequentFragment1byteFrame.datagramOffset =  (ipPacketAllSize-ipPacketUnsentSize)/8;
	 len = LOWPAN_PACKET_SEND_SIZE_MAX;
	 if (ipPacketUnsentSize <= (WORD)len)
	 {
	   len = ipPacketUnsentSize;
	 }
	 memcpy(&ipPacketZIPBuffer->ZW_LowpanSubsequentFragment1byteFrame.payload1, ipPacketUnsentPart, len);
	 ipPacketUnsentPart += len;
	 ipPacketUnsentSize -= len;
	 len += sizeof(ipPacketZIPBuffer->ZW_LowpanSubsequentFragment1byteFrame) - 1;
  }
  else
  {
    ipPacketZIPBuffer->ZW_CommandZipIpDatagramSegment1byteFrame.cmdClass = COMMAND_CLASS_ZIP_TUN_SERVICES;
    ipPacketZIPBuffer->ZW_CommandZipIpDatagramSegment1byteFrame.cmd = COMMAND_ZIP_IP_DATAGRAM_SEGMENT;
    ipSegmentsSeqNo++;
    ipPacketZIPBuffer->ZW_CommandZipIpDatagramSegment1byteFrame.properties1 =
      COMMAND_ZIP_IP_DATAGRAM_SEGMENT_PROPERTIES1_LAST_DATAGRAM_BIT_MASK
      | (ipSegmentsSeqNo & COMMAND_ZIP_IP_DATAGRAM_SEGMENT_PROPERTIES1_SEQUENCE_COUNT_MASK);

    len = ZIP_NEXT_SEGMENT_DATA_SIZE_MAX;
    if (ipPacketUnsentSize <= (WORD)len)
    {
      ipPacketZIPBuffer->ZW_CommandZipIpDatagramSegment1byteFrame.properties1 |= COMMAND_ZIP_IP_DATAGRAM_SEGMENT_PROPERTIES1_LAST_SEGMENT_BIT_MASK;
      len = ipPacketUnsentSize;
    }
    memcpy(&ipPacketZIPBuffer->ZW_CommandZipIpDatagramSegment1byteFrame.ipHeaderCompressionControl1, ipPacketUnsentPart, len);
    ipPacketUnsentPart += len;
    ipPacketUnsentSize -= len;
    len += sizeof(ipPacketZIPBuffer->ZW_CommandZipIpDatagramSegment1byteFrame) - 2;  // because without ipHeaderCompressionControl
    ipPacketSendingInProgress = TRUE;

    }
    if(ZW_SEND_DATA(ipSegmentsDestNodeID, (BYTE*)&ipPacketZIPBuffer->ZW_CommandZipIpDatagramSegment1byteFrame, len,
      ipSegmentsTxOpt, SendDataIP_NextZIPSegment))
    {
      return;
    }
    txStatus = TRANSMIT_COMPLETE_FAIL;
  }

  SendDataIP_ReleaseIPPacketBuf();

  if (ipZIPPacketCompletedFunc != NULL)
  {
    ZW_DEBUG_SEND_BYTE('k');
    ipZIPPacketCompletedFunc(txStatus);
    ipZIPPacketCompletedFunc = NULL;
  }
}

/*===========================================================================
**    Function description
**        Prepare and send first segment of IP packet, then next segmets
**        will be sended in copleted callback if needed.
**    Side effects:
**
**--------------------------------------------------------------------------*/
BOOL                          /* RET TRUE if sending started successfully */
SendDataIP_FirstZIPSegment(
  BYTE  nodeID,               /* IN  Destination node ID */
  BYTE  txOptions,            /* IN  Transmit option flags */
  VOID_CALLBACKFUNC_BYTE completedFunc) /* IN  Transmit completed callback function, args: BYTE txStatus - IN Transmit status */
{
  BYTE len,i;
  	iphdr* iph = ((iphdr*)ipPacket);
    iphdr6* iph6 = ((iphdr6*)ipPacket);
	if (((( iph6->ip6_ctlun.ip6_un2_vfc )>>4)&0x0f) == IPV6_VERSION)
	{
	   ipPacketUnsentSize = iph6->ip6_ctlun.ip6_un1.ip6_un1_plen + IPV6_HEADER_SIZE_MIN;
	}
  ipPacketUnsentPart = ipPacket;

  if(isLowPanPacket)
  {
  	 if(!isDataCompressed)
	 {
	    memcpy(lpComprPacket,ipPacket,IP_PACKET_SIZE_MAX);
		ipPacket[0] = 0x41;
	    memcpy(ipPacket+1,lpComprPacket,IP_PACKET_SIZE_MAX-1);
		ipPacketUnsentSize+=1;
	 }
	 else
	 {
		memcpy(lpComprPacket,ipPacket,IP_PACKET_SIZE_MAX);
		lpOutSize = IP_PACKET_SIZE_MAX;
		LowpanIpv6ToIphc(lpComprPacket,ipPacketSize,ipPacket,&lpOutSize,myHomeID,bMyNodeID,ipPacketSourceNodeID);
		ipPacketUnsentSize = lpOutSize;
	 }

	 ipPacketAllSize = ipPacketUnsentSize;

	 lpDataSendTag++;
	 ipPacketZIPBuffer->ZW_LowpanFirstFragment1byteFrame.cmdClass = 		  COMMAND_CLASS_ZIP_6LOWPAN;
	 ipPacketZIPBuffer->ZW_LowpanFirstFragment1byteFrame.cmd_Properties1 = LOWPAN_FIRST_FRAGMENT|((ipPacketAllSize & 0x0700)>>8);
	 ipPacketZIPBuffer->ZW_LowpanFirstFragment1byteFrame.datagramSize2 =   (ipPacketAllSize & 0x0ff);
	 ipPacketZIPBuffer->ZW_LowpanFirstFragment1byteFrame.datagramTag =     lpDataSendTag;

	 len = LOWPAN_PACKET_SEND_SIZE_MAX;
	 if (ipPacketUnsentSize <= (WORD)len)
	 {
	   len = ipPacketUnsentSize;
	 }
	  memcpy(&ipPacketZIPBuffer->ZW_LowpanFirstFragment1byteFrame.payload1, ipPacketUnsentPart, len);
	  ipPacketUnsentPart += len;
	  ipPacketUnsentSize -= len;
	  len += sizeof(ipPacketZIPBuffer->ZW_LowpanFirstFragment1byteFrame) - 1;
  }
  else
  {
	  ipPacketZIPBuffer->ZW_CommandZipIpDatagramSegment1byteFrame.cmdClass = COMMAND_CLASS_ZIP_TUN_SERVICES;
	  ipPacketZIPBuffer->ZW_CommandZipIpDatagramSegment1byteFrame.cmd = COMMAND_ZIP_IP_DATAGRAM_SEGMENT;
	  ipSegmentsSeqNo = 0;
	  ipPacketZIPBuffer->ZW_CommandZipIpDatagramSegment1byteFrame.properties1 =
	    COMMAND_ZIP_IP_DATAGRAM_SEGMENT_PROPERTIES1_LAST_DATAGRAM_BIT_MASK
	    | COMMAND_ZIP_IP_DATAGRAM_SEGMENT_PROPERTIES1_FIRST_SEGMENT_BIT_MASK
	    | (ipSegmentsSeqNo & COMMAND_ZIP_IP_DATAGRAM_SEGMENT_PROPERTIES1_SEQUENCE_COUNT_MASK);
	  ipPacketZIPBuffer->ZW_CommandZipIpDatagramSegment1byteFrame.ipHeaderCompressionControl1 = 0x00;
	  len = ZIP_FIRST_SEGMENT_DATA_SIZE_MAX;
	  if (ipPacketUnsentSize <= (WORD)len)
	  {
	    ipPacketZIPBuffer->ZW_CommandZipIpDatagramSegment1byteFrame.properties1 |= COMMAND_ZIP_IP_DATAGRAM_SEGMENT_PROPERTIES1_LAST_SEGMENT_BIT_MASK;
	    len = ipPacketUnsentSize;
	  }
	  memcpy(&ipPacketZIPBuffer->ZW_CommandZipIpDatagramSegment1byteFrame.payload1, ipPacketUnsentPart, len);
	  ipPacketUnsentPart += len;
	  ipPacketUnsentSize -= len;
	  len += sizeof(ipPacketZIPBuffer->ZW_CommandZipIpDatagramSegment1byteFrame) - 1;
  }

  ipSegmentsTxOpt = txOptions;
  ipZIPPacketCompletedFunc = completedFunc;
  ipSegmentsDestNodeID = nodeID;
  ipPacketSendingInProgress = TRUE;
  if(ZW_SEND_DATA(ipSegmentsDestNodeID,
    (BYTE*)&ipPacketZIPBuffer->ZW_CommandZipIpDatagramSegment1byteFrame, len,
    ipSegmentsTxOpt, SendDataIP_NextZIPSegment))
  {
    ZW_DEBUG_SEND_BYTE('O');
    return TRUE;
  }
  SendDataIP_ReleaseIPPacketBuf();
  return FALSE;
}

/*=============================   SendDataIP   =====++++======================
**    Transmit data buffer to a IP.
**
**    txOptions:
**          TRANSMIT_OPTION_LOW_POWER   transmit at low output power level (1/3 of
**                                      normal RF range).
**--------------------------------------------------------------------------*/
BYTE                                    /* RET  FALSE if transmitter queue overflow */
SendDataIP(
  BYTE gatewayNodeID,                   /* IN  Gateway node ID */
  BYTE ipAddressesType,
  BYTE *ipDestinationAddress,
  WORD ipDestinationPort,               /* IN  if 0 - will be autodetected */
  BYTE *ipSourceAddress,
  WORD ipSourcePort,                    /* IN  if 0 - will be autodetected */
  BYTE *pBufData,
  BYTE dataLength,
  BYTE txOptions,
  VOID_CALLBACKFUNC_BYTE completedFunc)
{
  BYTE len;
  udphdr* udph;
  iphdr6* iph6 = ((iphdr6*)ipPacket);

  ZW_COMMAND_ZIP_PACKET_1BYTE_FRAME* ziph;

  ZW_DEBUG_SEND_BYTE('I');
  ZW_DEBUG_SEND_NUM(ipPacketReceivingInProgress);
  ZW_DEBUG_SEND_NUM(ipPacketSendingInProgress);
  ZW_DEBUG_SEND_NUM(dataLength);
  ZW_DEBUG_SEND_NUM(pBufData[0]);

  if(!ipPacketReceivingInProgress    /* if receiving is in progress - we can't send IP packets, because we use same buffer for sending and receiving the IP packets.*/
      && !ipPacketSendingInProgress) /* we can't start new sending while old in progress */
  {
    switch (ipAddressesType)
    {
	    case IP_ADDR_TYPE_IPV6:
	    {
	      iph6->ip6_ctlun.ip6_un1.ip6_un1_flow = 0x00;
	      iph6->ip6_ctlun.ip6_un2_vfc  = 0x06 << 4;
	      iph6->ip6_ctlun.ip6_un1.ip6_un1_nxt   = IPPROTO_UDP;
	      iph6->ip6_ctlun.ip6_un1.ip6_un1_hlim	= HOP_LIMIT;

	      memcpy(&iph6->ip6_src, ipSourceAddress,      sizeof(iph6->ip6_src));
	      memcpy(&iph6->ip6_dst, ipDestinationAddress, sizeof(iph6->ip6_dst));

	      udph = ((BYTE*) iph6) + IPV6_HEADER_SIZE_MIN;
	      ziph = ((BYTE*) udph) + sizeof(udphdr);

	      udph->source = ipSourcePort;
	      if (udph->source == 0)
	      {
	        udph->source = ipPacketSrcPort;
	        if (++ipPacketSrcPort > TRANSPORT_ZIP_SOURCE_PORT_MAX)
	        {
	          ipPacketSrcPort = TRANSPORT_ZIP_SOURCE_PORT_MIN;
	        }
	      }
	      udph->dest = ipDestinationPort;
	      if (udph->dest == 0)
	      {
	        udph->dest = IPPORT_ZWAVE;
	      }
	      //encapsulate new data in the IP packet.
	      ziph->cmdClass = COMMAND_CLASS_ZIP_TUN_SERVICES;
	      ziph->cmd = COMMAND_ZIP_PACKET;
	      ziph->properties1 = 0;
	      ziph->properties2 = COMMAND_ZIP_PACKET_PROPERTIES2_Z_WAVE_CMD_INCLUDED_BIT_MASK;
	      ziph->seqNo = ++ipZIPPacketSeqNo;

	      /* calculate free number of bytes in buffer of IP packet for Z-Wave data: */
	      len = IP_PACKET_SIZE_MAX - ((((BYTE*)ziph) + COMMAND_ZIP_PACKET_HEADER_LEN) - iph6);

	      /* copy Z-Wave data, which fit in the IP packet buffer: */
	      if(len > dataLength)
	      {
	        len = dataLength;
	      }
	      if(pBufData != NULL && len != 0)
	      {
	        memcpy(((BYTE*)ziph) + COMMAND_ZIP_PACKET_HEADER_LEN, pBufData, len);
	      }

	      /* calculate the new length fields in IP packet: */
	      udph->len = sizeof(udphdr) + COMMAND_ZIP_PACKET_HEADER_LEN + len;
	      iph6->ip6_ctlun.ip6_un1.ip6_un1_plen  = udph->len;

	      /* calculate new checksum's of IP packet: */
	      udph->check = 0;
	      udph->check = SendDataIP_udp_csumIPv6(iph6);

	      /* Prepare and send first segment of IP packet,
	       *  then next segments will be sent in completed callback if needed.*/
	      SendDataIP_FirstZIPSegment(gatewayNodeID, txOptions, completedFunc);
	      break;
	    }
    }
  }
  ZW_DEBUG_SEND_BYTE('E');
  return FALSE;
}

/*=============================   SendGatewayDataIP   ===========================
**    Transmit data buffer to a single Z/IP node.
**     The buffer should contain the reply on previously received Z-Wave command
**     encapculated in the ipPacket. This function should be called only once for
**     each previously received Z-Wave command encapculated in the ipPacket.
**
**    txOptions:
**          TRANSMIT_OPTION_LOW_POWER   transmit at low output power level (1/3 of
**                                      normal RF range).
**--------------------------------------------------------------------------*/
BYTE                          /* RET  FALSE if transmitter queue overflow */
SendGatewayDataIP(
  BYTE  nodeID,               /* IN  Destination node ID */
  BYTE *pData,                /* IN  Data buffer pointer */
  BYTE  dataLength,           /* IN  Data buffer length */
  BYTE  txOptions,            /* IN  Transmit option flags */
  VOID_CALLBACKFUNC_BYTE completedFunc) /* IN  Transmit completed callback function, args: BYTE txStatus - IN Transmit status */
{
  BYTE len;

  __u16 udpPort;

  udphdr* udph;
  ZW_COMMAND_ZIP_PACKET_1BYTE_FRAME* ziph;
   iphdr6* iph6;

	 if(!isDataCompressed)
	 {
		iph6 = ((iphdr6*)(ipPacket+1));
	 }
	 else
	 {
	    memcpy(lpComprPacket,ipPacket,ipPacketSize);
	    lpOutSize = IP_PACKET_SIZE_MAX;
	    LowpanIphcToIpv6(lpComprPacket,ipPacketSize,ipPacket,&lpOutSize,myHomeID,ipPacketSourceNodeID,bMyNodeID);
	    ipPacketSize = lpOutSize;
	    iph6 = ((iphdr6*)ipPacket);
	 }

  if(!ipPacketReceivingInProgress    /* if receiving is in progress - we can't send IP packets, because we use same buffer for sending and receiving the IP packets.*/
      && !ipPacketSendingInProgress) /* we can't start new sending while old in progress */
  {

	ZW_DEBUG_SEND_BYTE(' ');
	ZW_DEBUG_SEND_BYTE('K');
    if(((iph6->ip6_ctlun.ip6_un2_vfc & 0xf0)>>4) == IPV6_VERSION)
    {
      ZW_DEBUG_SEND_BYTE('V');
      udph = ((BYTE*) iph6) + IPV6_HEADER_SIZE_MIN;
      ziph = ((BYTE*) udph) + sizeof(udphdr);

      if(iph6->ip6_ctlun.ip6_un1.ip6_un1_nxt == IPPROTO_UDP)
      {
        ZW_DEBUG_SEND_BYTE('U');
        if(udph->dest == IPPORT_ZWAVE)
        {
          in6_addr ipAddr6;
          ZW_DEBUG_SEND_BYTE('Z');

          /* exchange the destination and source in IP packet */
          memcpy(&ipAddr6,		&iph6->ip6_src,	IPV6_ADDRESS_SIZE);
          memcpy(&iph6->ip6_src,&iph6->ip6_dst,	IPV6_ADDRESS_SIZE);
          memcpy(&iph6->ip6_dst,&ipAddr6,		IPV6_ADDRESS_SIZE);

          udpPort = udph->source;
          udph->source = udph->dest;
          udph->dest = udpPort;
        }

          SendDataIP(nodeID, IP_ADDR_TYPE_IPV6,
            &iph6->ip6_dst, udph->dest,
            &iph6->ip6_src, udph->source,
            pData, dataLength, txOptions, completedFunc);


      }
    }
  }
  ZW_DEBUG_SEND_BYTE('E');
  return FALSE;
}

/*===========================================================================
**    Function description
**      Parse the ICMP IP packet and send the response on it. Only Echo reply
**      type is supported.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void											/* RET nothing */
ParseICMP6(
  void)                   /* IN  nothing */
{
  in6_addr ipAddr6;
  iphdr6 *iph6 = (iphdr6*)ipPacket;
  icmphdr6 *icmph6 = (icmphdr6*)(((BYTE*)iph6) + IPV6_HEADER_SIZE_MIN);

  if(iph6->ip6_ctlun.ip6_un1.ip6_un1_nxt == IPPROTO_ICMPV6)
  {
    switch (icmph6->icmp6_type)
    {
    case ICMP6_ECHO_REQUEST:
    {
      if(icmph6->icmp6_code == 0)
      {
		ZW_DEBUG_SEND_BYTE('Q');

	  	memcpy(&ipAddr6,		&iph6->ip6_src,	IPV6_ADDRESS_SIZE);
	  	memcpy(&iph6->ip6_src,	&iph6->ip6_dst,	IPV6_ADDRESS_SIZE);
	  	memcpy(&iph6->ip6_dst,	&ipAddr6,		IPV6_ADDRESS_SIZE);

        icmph6->icmp6_type = ICMP6_ECHO_REPLY;
        icmph6->icmp6_code = 0;

		icmph6->icmp6_cksum = (((icmph6->icmp6_cksum>>8)&0xff) | ((icmph6->icmp6_cksum<<8)&0xff00)) - 1;
		icmph6->icmp6_cksum = ((icmph6->icmp6_cksum>>8)&0xff) | ((icmph6->icmp6_cksum<<8)&0xff00);
/////////////////////////////////////////////////
//        icmph6->icmp6_cksum = SendDataIP6_csum(iph6+8, iph6->ip6_ctlun.ip6_un1.ip6_un1_plen+IPV6_HEADER_SIZE_MIN-8);
/////////////////////////////////////////////////
        /* Send echo reply IP packet: */
        SendDataIP_FirstZIPSegment(ipPacketSourceNodeID, ipPacketSourceTxOptions, NULL);
      }
      break;
    }
    }
  }
}

/*===========================================================================
**    Function description
**      Parse (de-encaplulate) Z-Wave command from ZIP packet and run Z-Wave
**       command handler.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void											/* RET nothing */
ParseZIP(
  BYTE txStatus)          /* IN Transmit status, must be set to TRANSMIT_COMPLETE_OK by default */
{
  ZW_DEBUG_SEND_BYTE('Z');
  if (ipPacketZIPPacket != NULL)
  {
    /*TODO: implement also detecting of position of included command by checking the other bits in properies2. Currently we assume, that only command is included and no other fields is included. */
    if (ipPacketZIPPacketProperties2 == COMMAND_ZIP_PACKET_PROPERTIES2_Z_WAVE_CMD_INCLUDED_BIT_MASK)
    {
      ZW_DEBUG_SEND_BYTE(' ');
	  ZW_DEBUG_SEND_BYTE('L');
	  ZW_DEBUG_SEND_BYTE('P');
	  ZW_DEBUG_SEND_BYTE('1');
      //Z-Wave command processing:
	  if(isLowPanPacket)
	  {
	    if (!isDataCompressed)
		{
			Transport_ApplicationCommandHandler(RECEIVE_STATUS_TYPE_SINGLE, ipPacketSourceNodeID,
        	((BYTE*)ipPacketZIPPacket) + COMMAND_ZIP_LOW_PAN_PACKET_HEADER_LEN+2, ipPacketZIPPacketSize - COMMAND_ZIP_LOW_PAN_PACKET_HEADER_LEN);
		}
		else
		{
			Transport_ApplicationCommandHandler(RECEIVE_STATUS_TYPE_SINGLE, ipPacketSourceNodeID,
        	((BYTE*)ipPacketZIPPacket) + COMMAND_ZIP_LOW_PAN_PACKET_HEADER_LEN+1, ipPacketZIPPacketSize - COMMAND_ZIP_LOW_PAN_PACKET_HEADER_LEN);
		}
	  }
	  else
	  {
      	Transport_ApplicationCommandHandler(RECEIVE_STATUS_TYPE_SINGLE, ipPacketSourceNodeID,
        ((BYTE*)ipPacketZIPPacket) + COMMAND_ZIP_PACKET_HEADER_LEN, ipPacketZIPPacketSize - COMMAND_ZIP_PACKET_HEADER_LEN);
	  }
    }
  }
}

/*===========================================================================
**    Function description
**      Parse the UDP IP packet and send the ACK and/or response on it.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                      /* RET nothing */
ParseUDP6(
  void)                   /* IN  nothing */
{
  ZW_COMMAND_ZIP_PACKET_1BYTE_FRAME* ziph;
  in6_addr ipAddr6;
  __u16 udpPort;
  udphdr* udph;
  iphdr6* iph6 = ((iphdr6*)ipPacket);

  udph = ((BYTE*) iph6) + IPV6_HEADER_SIZE_MIN;
  ziph = (ZW_COMMAND_ZIP_PACKET_1BYTE_FRAME*)(((BYTE*) udph) + sizeof(udphdr));
  ipPacketZIPPacket = NULL;

  ZW_DEBUG_SEND_BYTE('I');
  ZW_DEBUG_SEND_BYTE('6');

  if (iph6->ip6_ctlun.ip6_un1.ip6_un1_nxt == IPPROTO_UDP)
  {
    if (udph->dest == IPPORT_ZWAVE)
    {
      nodeIncludedZIP = TRUE;     //TODO: this is temporary. Some detection should be made or just not use this variable.
      ipPacketZIPPacketSize = udph->len - sizeof(udphdr);
      if(ipPacketZIPPacketSize >= COMMAND_ZIP_PACKET_HEADER_LEN)
      {
        if (ziph->properties1 & ~((BYTE)(
          COMMAND_ZIP_PACKET_PROPERTIES1_ACK_REQUEST_BIT_MASK             /*this flag is used to determine whether sent or not Z/IP ACK*/
          | COMMAND_ZIP_PACKET_PROPERTIES1_USE_NO_Z_WAVE_ACK_BIT_MASK)))  /*this flag is just ignored*/
        { /* we support only mentioned above bits of properties1 so,
           * if some other bits are set - ignore entire Z/IP packet: */
          ZW_DEBUG_SEND_BYTE('E');
        }
        else
        {
          ZW_DEBUG_SEND_BYTE('F');

          /* Store some Z/IP header members for later Parsing of Z/IP packet, because
           *  values of this members can be discarded during Z/IP Ack sending. */
          ipPacketZIPPacketProperties2 = ziph->properties2;
          ipPacketZIPPacket = ziph;

          if (ziph->properties1 & COMMAND_ZIP_PACKET_PROPERTIES1_ACK_REQUEST_BIT_MASK)
          { /* Sendig the Z/IP Ack if requested */
            /* fill the Z/IP Packet with Z/IP Ack: */

            ziph->cmdClass = COMMAND_CLASS_ZIP_TUN_SERVICES;
            ziph->cmd = COMMAND_ZIP_PACKET;
            ziph->properties1 = COMMAND_ZIP_PACKET_PROPERTIES1_ACK_RESPONSE_BIT_MASK;
            ziph->properties2 = 0;
            ziph->seqNo = ziph->seqNo;

            /* calculates the new length of IP/UPD packet: */
            udph->len = sizeof(udphdr) + COMMAND_ZIP_PACKET_HEADER_LEN;
            iph6->ip6_ctlun.ip6_un1.ip6_un1_plen  = udph->len;
            iph6->ip6_ctlun.ip6_un1.ip6_un1_hlim  = HOP_LIMIT;

            /* exchanging the source and destination: */
            memcpy(&ipAddr6,		&iph6->ip6_src,	IPV6_ADDRESS_SIZE);
            memcpy(&iph6->ip6_src,	&iph6->ip6_dst,	IPV6_ADDRESS_SIZE);
            memcpy(&iph6->ip6_dst,	&ipAddr6,		IPV6_ADDRESS_SIZE);

            udpPort = udph->source;
            udph->source = udph->dest;
            udph->dest = udpPort;

            /* calculate the new ckecksum's in IP and UDP headers */
            udph->check = 0;
      	    udph->check = SendDataIP_udp_csumIPv6(iph6);

            /* Send Z/IP Ack IP packet:
             * Z/IP packet will be parsed, when Z/IP Ack will be sent. */
            if(!SendDataIP_FirstZIPSegment(ipPacketSourceNodeID, ipPacketSourceTxOptions, ParseZIP))
            { /* fail to send Z/IP Ack. Ignore this and parse the Z/IP packet: */
              ParseZIP(TRANSMIT_COMPLETE_OK);
            }
          }
          else
          { /* Parse the Z/IP packet: */
            ParseZIP(TRANSMIT_COMPLETE_OK);
          }
        }
      }
    }
  }
}

void
Handler6LowPAN(
  void)
{
  int i;
  iphdr* iph;
  iphdr6* iph6;

  if(ipPacket[0]==0x41)
  {
	isDataCompressed	= FALSE;
	memcpy(lpComprPacket,ipPacket,ipPacketSize);
	ipPacketSize-=1;
	memcpy(ipPacket,(lpComprPacket+1),ipPacketSize);
  }
  else
  {
  	isDataCompressed	= TRUE;
	memcpy(lpComprPacket,ipPacket,ipPacketSize);
	lpOutSize = IP_PACKET_SIZE_MAX;
	LowpanIphcToIpv6(lpComprPacket,ipPacketSize,ipPacket,&lpOutSize,myHomeID,ipPacketSourceNodeID,bMyNodeID);
	ipPacketSize = lpOutSize;
  }

    switch (((iphdr*)ipPacket)->version)
    {
    case IPV6_VERSION:
    {
      iph6 = (iphdr6*)ipPacket;
        switch (iph6->ip6_ctlun.ip6_un1.ip6_un1_nxt)
        {
        case IPPROTO_ICMPV6:
          ParseICMP6();
          break;

        case IPPROTO_UDP:
          ParseUDP6();
          break;
        }
      break;
    }
    }
}


/*========================   ApplicationCommandHandler   ====================
**    Handling of a received application commands and requests
**
*
**--------------------------------------------------------------------------*/
void                              /*RET Nothing                  */
ApplicationCommandHandler(
  BYTE  rxStatus,                 /* IN Frame header info */
  BYTE  sourceNode,               /* IN Command sender Node ID */
  ZW_APPLICATION_TX_BUFFER *pCmd, /* IN Payload from the received frame, the union */
                                  /*    should be used to access the fields */
  BYTE   cmdLength)               /* IN Number of command bytes including the command */
{
  int i;
  BYTE bytesCount;

  BYTE param1 = *((BYTE*)pCmd + OFFSET_PARAM_1);
  BYTE param2 = *((BYTE*)pCmd + OFFSET_PARAM_2);
  BYTE param3 = *((BYTE*)pCmd + OFFSET_PARAM_3);

  BYTE txOption = ((rxStatus & RECEIVE_STATUS_LOW_POWER) ? TRANSMIT_OPTION_LOW_POWER : 0) |
                  TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_RETURN_ROUTE /*| TRANSMIT_OPTION_EXPLORE*/;
  iphdr* iph;
  iphdr6* iph6;

  if (cmdLength < 2)
    return;

  switch (pCmd->ZW_Common.cmdClass)
  {

	  case COMMAND_CLASS_ZIP_6LOWPAN:
	  {
		nodeIncludedZIP = TRUE;
		isLowPanPacket  = TRUE;

	  	if((pCmd->ZW_Common.cmd & LOWPAN_FIRST_FRAGMENT_MASK) == LOWPAN_FIRST_FRAGMENT)
		{
		   ipPacketReceivingInProgress = TRUE;
		   ipPacketReceivedSize = (pCmd->ZW_Common.cmd & LOWPAN_FIRST_FRAGMENT_PROPERTIES1_DATAGRAM_SIZE_1_MASK)<<8;
		   ipPacketReceivedSize |= param1;
		   if (lpDataTag != param2)
		   {
		   	  lpDataTag = param2;
			  memset(ipPacket,0x0,IP_PACKET_SIZE_MAX);
			  ipPacketSize = 0x0;
		   }
		   memcpy(ipPacket,(BYTE*)pCmd+OFFSET_PARAM_3,cmdLength-OFFSET_PARAM_3);
		   ipPacketSize += (cmdLength-OFFSET_PARAM_3);
		   if(ipPacketReceivedSize<=ipPacketSize)
		   {
		      ipPacketSourceNodeID = sourceNode;
	          ipPacketSourceTxOptions = txOption;
			  ipPacketReceivingInProgress = FALSE;
		      Handler6LowPAN();
		   }
		}
		else if((pCmd->ZW_Common.cmd & LOWPAN_FIRST_FRAGMENT_MASK) == LOWPAN_SUBSEQUENT_FRAGMENT)
		{
		   ipPacketReceivingInProgress = TRUE;
		   if (lpDataTag != param2)
		   {
		   	  lpDataTag = param2;
			  memset(ipPacket,0x0,IP_PACKET_SIZE_MAX);
			  ipPacketReceivedSize = -1;
			  ipPacketSize = 0x0;
		   }
		   memcpy(ipPacket+param3*8,(BYTE*)pCmd+OFFSET_PARAM_4,cmdLength-OFFSET_PARAM_4);
		   ipPacketSize += (cmdLength-OFFSET_PARAM_4);
		   if((ipPacketReceivedSize!=-1)&(ipPacketReceivedSize<=ipPacketSize))
		   {
		      ipPacketSourceNodeID = sourceNode;
	          ipPacketSourceTxOptions = txOption;
			  ipPacketReceivingInProgress = FALSE;
			  Handler6LowPAN();
		   }
		}
	  	break;
	  }

	  default:
	  {
	    /* All other non ZIP commands - pass to the application layer: */
	    zipCmdHandling = TZIP_CMD_HANDLING_NATIVE;       //message is not encapsulated, so reports also must be sent non-encapsulated.
	    Transport_ApplicationCommandHandler(rxStatus, sourceNode, pCmd, cmdLength);
	    zipCmdHandling = TZIP_CMD_HANDLING_DEFAULT;
	    break;
	  }
  }
}

/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/

/*========================   Transport_SendDataIP  ===========================
**      Send data in Z/IP packet to the specifed IP
**    Side effects :
**
**--------------------------------------------------------------------------*/
BYTE
Transport_SendDataIP(
  BYTE gatewayNodeID,
  IP_ADDR *ipDestinationAddress,
  IP_ADDR *ipSourceAddress,
  BYTE *pBufData,
  BYTE dataLength,
  BYTE txOptions,
  VOID_CALLBACKFUNC_BYTE completedFunc)
{
  return SendDataIP(gatewayNodeID, ipDestinationAddress->type,
    &ipDestinationAddress->u_IP_ADDR.ipv4, 0,
    &ipSourceAddress->u_IP_ADDR.ipv4, 0,
    pBufData, dataLength, txOptions, completedFunc);
}

/*========================   Transport_SendRequest   ============================
**      Send request command over Z/IP network
**    Side effects :
**
**--------------------------------------------------------------------------*/
BYTE
Transport_SendRequest(
  BYTE nodeID,
  BYTE *pBufData,
  BYTE dataLength,
  BYTE txOptions,
  VOID_CALLBACKFUNC_BYTE completedFunc,
  BYTE isForceNative)                     /* TRUE if data should be sent native, i.e. without using this transport.*/
{
  if(zipCmdHandling == TZIP_CMD_HANDLING_NATIVE)
  {
    isForceNative = TRUE;
  }
  if(!nodeIncludedZIP || isForceNative)
  {
    ZW_DEBUG_SEND_BYTE('N');
    return ZW_SEND_DATA(nodeID, pBufData, dataLength, txOptions, completedFunc);
  }
  else
  {
    ZW_DEBUG_SEND_BYTE('Z');
    return SendGatewayDataIP(nodeID, pBufData, dataLength, txOptions, completedFunc);
  }
}

/*========================   Transport_SendReport   ============================
**      This function must be called instead of Transport_SendRequest, if report
**      frame is sent.
**    Side effects :
**
**--------------------------------------------------------------------------*/
BYTE
Transport_SendReport(
  BYTE nodeID,
  BYTE *pBufData,
  BYTE dataLength,
  BYTE txOptions,
  VOID_CALLBACKFUNC_BYTE completedFunc,
  BYTE isForceNative)                     /* TRUE if data should be sent native, i.e. without using this transport.*/
{
  return Transport_SendRequest(nodeID, pBufData, dataLength, txOptions, completedFunc, isForceNative);
}

/*==============   Transport_OnApplicationInitHW   ============================
**      This function must be called in ApplicationInitHW
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
BYTE                                      /* return TRUE on success */
Transport_OnApplicationInitHW(
  BYTE bStatus)                          /* bStatus */
{
  //Transport_wakeUpReason = bStatus;
  return TRUE;
}

/*==============   Transport_OnApplicationInitSW   ============================
**      This function must be called in ApplicationInitSW
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
BYTE                                      /* return TRUE on success */
Transport_OnApplicationInitSW(
  BYTE *commandClassesList,               /* List of supported command classes, when node communicate by this transport */
  BYTE commandClassesListCount,           /* Count of elements in supported command classes list */

  BYTE eepromInit,                        /* TRUE if contents of eeprom must forced to be initialized to default values */
  BYTE eepromBufOffsSettings,             /* buffer offset in eeprom for storing transport settings*/
  BYTE eepromBufSizeSettings,             /* buffer size in eeprom for storing transport settings*/
  TRANSPORT_STATUS_CALLBACK statusCallbackFunc) /* callback function, which called by transport to inform application layer of its status. can be NULL, if status not needed. */
{
  ReportTransportStatus = statusCallbackFunc;
  //initialization:
  zipCmdHandling = TZIP_CMD_HANDLING_DEFAULT;
  return TRUE;
}

/*==============   Transport_OnLearnCompleted   ============================
**      This function must be called in LearnCompleted application function
**       callback
**    Side effects :
**
**--------------------------------------------------------------------------*/
BYTE                                      /* return TRUE on success */
Transport_OnLearnCompleted(
  BYTE nodeID)                            /* IN resulting nodeID */
{
  if (nodeID == 0)    /* Node was excluded: */
  {
    nodeIncludedZIP = FALSE;
  }
  else                /* Node was included: */
  {
  }
  return TRUE;
}




