/*****************************  ZW_serialapi.c  *****************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2007
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 *
 * Description: See below.
 *
 * Author:   Oleg Zadorozhnyy
 *
 * 2007-08-11   OZA     Integrated with Z-Wave Used with iccAVR
 *
 * Last Changed By:  $Author: oza $
 * Revision:         $Revision: 1.1 $
 * Last Changed:     $Date: 2007/10/02 14:39:20 $
 * Ported to iccAVR
 ****************************************************************************/

/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
#include "TYPES.H"      // Standard types
#include "io.h"
#include <string.h>
#include "ZW_basis.h"
#include "ZW_SerialAPI.h"
#include "ZW_CONTROLLER.H"
#include "p_button.h"
#include "RS232_IO.H"		// Serial communications
#ifdef __GNUC__
#include <avr/io.h>
#include <avr/interrupt.h>
#define __disable_interrupt() cli();
#define __enable_interrupt() sei();
#else
#include <iom128.h>
#include        <inavr.h>
#endif
////#include <ZW_transport_api.h>

#define NEW_NODEINFO

#define INVALID_TIMER_HANDLE 255
#define BUF_SIZE  74


/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/
typedef struct __IOFLAGS
{
	unsigned rx:1;
	unsigned tx:1;
	unsigned err:1;

}IOFLAGS;


extern void TimerAction( void );
/****************************************************************************/
/*                              EXPORTED DATA                               */
/****************************************************************************/
extern IOFLAGS	ioflags;
extern  BYTE rxCount;
extern  volatile PBYTE timerCount;
extern  volatile BYTE rxInPtr;
extern  volatile BYTE rxOutPtr;
///BYTE rxCount = 0;
/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/
void ioPoll( void );
void Dispatch( BYTE *pData );
void WaitForResponse( BYTE byFunc, BYTE *buffer, BYTE *byLen );
int ReceiveData( BYTE *buffer );




LEARN_INFO learnNodeInfo;

BYTE  timerReceiveTimer;
/**
  * \ingroup SerialAPI
  * Buffer for incoming frames.
  */
BYTE RecvBuffer[ BUF_SIZE ];

/**Warning[Pe161]: unrecognized #pragma C:\Zensys Projects\LED_DimmerAVR\ZW_serialapi.c 48

  * \ingroup SerialAPI
  * Buffer for outgoing frames.
  * Is primarily used to remember outgoing telegrams in case a retransmission is needed.
  */

BYTE TransmitBuffer[ BUF_SIZE ];

/**
  * \ingroup SerialAPI
  * Holds the length of the frame currently in the \ref TransmitBuffer.
  */
volatile static BYTE TransmitBufferLength = 0;

BYTE buffer[ BUF_SIZE ];
BYTE pCmd[ BUF_SIZE ];

static BYTE idx;
static BYTE byLen;
static BYTE byCompletedFunc;

/**
  * \ingroup SerialAPI
  * Max number of retransmissions/retries.
  */
#define MAX_SERIAL_RETRY 3
#define TIMEOUT_TIME 250

/**
  * \ingroup SerialAPI
  * \defgroup SAUSC UART Status Codes
  * @{
  */
#define STATUS_RXTIMEOUT 1
#define STATUS_FRAMESENT 2
#define STATUS_FRAMEERROR 3
#define STATUS_FRAMERECEIVED 4

/** @} */
/**
  * \ingroup SerialAPI
  * Used for storing the result of \ref ReceiveData.
  * \see SAUSC
  */
static BYTE result = 0;

/**
  * \ingroup SerialAPI
  * Indicate timeout occured during \ref ReceiveData.
  */
static BOOL receiveDataTimeOut;
BOOL  receiveAskTimeOut;
/**
  * \ingroup SerialAPI
  * Indicate timeout occured during \ref WaitForResponse.
  */
static BOOL waitForResponseTimeOut;

/**
  * \ingroup SerialAPI
  * Indicate whether acknowledgment is expected or not.
  */
static BOOL waitForAck;

/**
  * \ingroup SerialAPI
  * \defgroup SACB Callbacks
  * @{ZW_APPLICATION_TX_BUFFER
  */
static void ( *funcApplicationCommandHandler ) ( BYTE, BYTE, ZW_APPLICATION_TX_BUFFER*, BYTE );
static void ( *funcApplicationControllerUpdate ) ( BYTE,  BYTE, BYTE*, BYTE );
static void ( *cbFuncZWSendData ) ( BYTE txStatus );
static void ( *cbFuncZWSendDataMulti ) ( BYTE txStatus );
static void ( *cbFuncZWSendNodeInformation ) ( BYTE txStatus );
static void ( *cbFuncMemoryPutBuffer ) ( void );
static void ( *cbFuncZWSetDefault ) ( void );
static void ( *cbFuncZWNewController ) ( LEARN_INFO* );
static void ( *cbFuncZWReplicationSendData ) ( BYTE txStatus );
static void ( *cbFuncZWAssignReturnRoute ) ( BYTE bStatus );
static void ( *cbFuncZWAssignSUCReturnRoute ) ( BYTE bStatus );
static void ( *cbFuncZWDeleteSUCReturnRoute ) ( BYTE bStatus );
static void ( *cbFuncZWDeleteReturnRoute ) ( BYTE bStatus );
static void ( *cbFuncZWSetLearnMode ) (LEARN_INFO*/*BYTE bStatus, BYTE nodeID*/);
static void ( *cbFuncZWRequestNodeNodeNeighborUpdate ) ( BYTE bStatus );
static void ( *cbFuncZWSetSUCNodeID ) ( BYTE bSstatus );
static void ( *cbFuncZWRequestNetworkUpdate ) ( BYTE txStatus );
/** @} */

/****************************    SerialAPI_Poll    **************************/
/**
  * \ingroup SerialAPI
  * Evaluates \ref result and calls \ref ReceiveData.
  * Must be called regularly.
  */
/****************************************************************************/
void SerialAPI_Poll( void )
{
    switch ( result )
    {

        case STATUS_FRAMERECEIVED:
            Dispatch( RecvBuffer);
            break;

        case STATUS_FRAMESENT:
            break;

        case STATUS_FRAMEERROR:
            break;

        default:
            break;
    }

    result = ReceiveData(RecvBuffer);
}


/**
  * \ingroup SerialAPI
  * Calculate checksum for given buffer.
  * NOTE: Should only be called locally in the Serial API.
  * \param[in] pData    Pointer to data frame (without SOF)
  * \param[in] nLength  Length of data frame
  * \return Checksum
  */
BYTE CalculateChecksum( BYTE *pData, int nLength )
{
  BYTE byChecksum;
  byChecksum = 0xff;
  for ( ; nLength; nLength--)
  {
    byChecksum ^= *pData++;
  }
  return byChecksum;
}

/**
  * \ingroup SerialAPI
  * Initialize callbacks and transmission variables.
  * \see SACB waitForAck waitForResponseTimeOut receiveDataTimeOut
  * \param[in] funcApplCmdHandler   Pointer to callback for ApplicationCommandHandler
  * \param[in] funcAppConUpdate     Pointer to callback for ApplicationControllerUpdate
  * \return = TRUE if initialization was succesful
  */
BOOL SerialAPI_InitSW(
  void ( *funcApplCmdHandler ) (BYTE, BYTE, ZW_APPLICATION_TX_BUFFER *, BYTE),
  void ( *funcAppConUpdate ) (BYTE, BYTE, BYTE*, BYTE )
)
{
    cbFuncZWSendData = NULL;
    cbFuncZWSendDataMulti = NULL;
    cbFuncZWSendNodeInformation = NULL;
    cbFuncMemoryPutBuffer = NULL;
    cbFuncZWSetDefault = NULL;
    cbFuncZWNewController = NULL;
    cbFuncZWReplicationSendData = NULL;
    cbFuncZWAssignReturnRoute = NULL;
        cbFuncZWAssignSUCReturnRoute = NULL;
        cbFuncZWDeleteSUCReturnRoute = NULL;
        cbFuncZWDeleteReturnRoute = NULL;
    cbFuncZWSetLearnMode = NULL;
        cbFuncZWRequestNodeNodeNeighborUpdate = NULL;
    cbFuncZWSetSUCNodeID = NULL;
    cbFuncZWRequestNetworkUpdate = NULL;

    waitForAck = FALSE;
    receiveDataTimeOut = FALSE;
    waitForResponseTimeOut = FALSE;

    funcApplicationCommandHandler = funcApplCmdHandler;
    funcApplicationControllerUpdate = funcAppConUpdate;

    return TRUE;
}

/**
  * \ingroup SerialAPI
  * Callback for \ref ReceiveData.
  * \see receiveDataTimeOut
  */
void ReceiveDataIndicateTimeOut( void )
{
    receiveDataTimeOut = TRUE;
}

static BOOL bReceiveDataRunning = FALSE;

/**
  * \ingroup SerialAPI
  * Collects acknowledgments and frames from the serial hardware.
  * NOTE: Must only be called locally in the Serial API.
  * \param[in] buffer   Buffer in which to store received frame
  * \return \see SAUSC
  */
int ReceiveData( BYTE *buffer )
{
    int i;
    BYTE ch;
    BYTE bLen;

    i = 0;
    ch = 0;
    bLen = 255;
    if ( bReceiveDataRunning == TRUE )
    {
//        DEBUG_COMM_PUTS( " Warning: ReceiveData called multiple times! " );
    }

    bReceiveDataRunning = TRUE;
    result = 0;
    receiveDataTimeOut = FALSE;

    for ( ;ch != SOF; )
    {
        ioPoll();

        if ( bySerialCheck() == 0 )
        {
            bReceiveDataRunning = FALSE;
            return STATUS_RXTIMEOUT;
        }

        ch = bySerialGetByte();

        if ( waitForAck == TRUE )
        {
            waitForAck = FALSE;

            if ( ch == ACK )
            {
                bReceiveDataRunning = FALSE;
                return STATUS_FRAMESENT;
            }

            if ( ch == NAK )
            {
                bReceiveDataRunning = FALSE;
                return STATUS_FRAMEERROR;
            }

            bReceiveDataRunning = FALSE;
            return STATUS_FRAMEERROR;
        }

        if ( ch == SOF )
            break;

        bReceiveDataRunning = FALSE;

        return STATUS_FRAMEERROR;
    }

    for (i = 0; i < (WORD)bLen + 1; i++)
    {
        timerReceiveTimer = 100;
        for ( ;!bySerialCheck(); )
        {
            ioPoll();

            if ( receiveDataTimeOut == TRUE )
            {
                bReceiveDataRunning = FALSE;
                return STATUS_RXTIMEOUT;
            }
        }
        if ( i == 0 )
        {
            bLen = buffer[ i ] = bySerialGetByte(); // bLen is length of frame WITHOUT SOF and Checksum
        }
        else
        {
            buffer[ i ] = bySerialGetByte();
        }
    }
    bReceiveDataRunning = FALSE;

    if (0 != CalculateChecksum(buffer, i))
    {
        vSerialPutByte( NAK );	// Tell the sender something went wrong
        bReceiveDataRunning = FALSE;
        return STATUS_FRAMEERROR;
    }
    buffer[ i -1 ] = 0;
    vSerialPutByte( ACK );	// Tell the world we are happy...
    bReceiveDataRunning = FALSE;

    return STATUS_FRAMERECEIVED;
}

#define UDRE 5
#define DATA_REGISTER_EMPTY (1<<UDRE)

void ReceiveAckIndicateTimeOut( void )
{
    receiveAskTimeOut = TRUE;
}


BOOL WaitForAck()
{
  static BYTE ReceiveTimerHandle;
  receiveAskTimeOut = FALSE;
  ReceiveTimerHandle = TimerStart( ReceiveAckIndicateTimeOut, 200, TIMER_FOREVER );
  while(!bySerialCheck())
  {
    ioPoll();
    if ( receiveAskTimeOut == TRUE )
    {
      TimerCancel(ReceiveTimerHandle);
      ReceiveTimerHandle=INVALID_TIMER_HANDLE;
      return STATUS_RXTIMEOUT;
    }
  }
  TimerCancel(ReceiveTimerHandle);
  ReceiveTimerHandle=INVALID_TIMER_HANDLE;
  return  ReceiveData(RecvBuffer);
}

BYTE BufTX;

void TransmitData(BYTE *buffer, BYTE length)
{
  BYTE byChk;
  ioflags.tx = 1;
  for (byChk = 1; byChk < (length+1); byChk++)
  {
    while ((UCSR1A & DATA_REGISTER_EMPTY)==0) ioPoll();
    mIOPutByte(buffer[byChk]);
  }

  __disable_interrupt();
  ioflags.tx = 0;
  rxCount = 0;
  rxInPtr = 0;
  rxOutPtr = 0;
  __enable_interrupt();

}

BOOL getIoflagsTx()
{
  return ioflags.tx;
}

/**
  * \ingroup SerialAPI
  * Send request to remote side.
  * NOTE: Must only be called locally in the Serial API.
  * \param[in] buffer   Pointer to data frame.
  * \param[in] length   Length of data.
  */
void SendData(BYTE *buffer, BYTE length)
{
  BYTE byChk;

  if(ioflags.tx) return;

  if (buffer != &TransmitBuffer[1])
  {
    waitForAck = TRUE;
    TransmitBuffer[1] = SOF;
    for (byChk = 0; byChk < length; byChk++)
    {
      TransmitBuffer[2 + byChk] = buffer[byChk];
    }
    TransmitBuffer[2 + length] = CalculateChecksum(buffer, length);
    TransmitBufferLength = length;
    TransmitBuffer[0] = length + 2;               // SOH + data + chksum
  }

  for (byChk = 0; byChk < 3; byChk++)
  {
    TransmitData(TransmitBuffer,TransmitBuffer[0]);
    if(WaitForAck()==STATUS_FRAMESENT) {BufTX = 0xAA; return;}
  }
}

static BOOL bWaitForResponseRunning = FALSE;


/**
  * \ingroup SerialAPI
  * Send request to remote side and wait for response.
  * NOTE: Must only be called locally in the Serial API.
  * \param[in] buffer   Pointer to data frame.
  * \param[in] length   Length of data.
  * \param[in] byFunc   Function number expected in response.
  * \param[out] out     Buffer to store response in.
  * \param[out] byLen   Length of response.
  */
void SendDataAndWaitForResponse( BYTE *buffer, BYTE length, BYTE byFunc, BYTE *out, BYTE *byLen )
{
    if ( bWaitForResponseRunning == TRUE )
    {
      ;
    }

    waitForAck = TRUE;
    SendData( buffer, length );
    WaitForResponse( byFunc, out, byLen );
}

/**
  * \ingroup SerialAPI
  * Callback for \ref WaitForResponse.
  */
void WaitForResponseIndicateTimeOut( void )
{
    waitForResponseTimeOut = TRUE;
}


/**
  * \ingroup SerialAPI
  * Wait for response.
  * NOTE: Must only be called locally in the Serial API.
  * \see SendDataAndWaitForResponse
  * \param[in] byFunc   Function number expected in response.
  * \param[out] out     Buffer to store response in.
  * \param[out] byLen   Length of response.
  */
void WaitForResponse( BYTE byFunc, BYTE *buffer, BYTE *byLen )
{
    BYTE bFound;
    BYTE retry;
    static BYTE byTimerHandle;

    bFound = FALSE;
    bWaitForResponseRunning = TRUE;

    waitForResponseTimeOut = FALSE;
    retry = MAX_SERIAL_RETRY - 1;

    byTimerHandle = TimerStart( WaitForResponseIndicateTimeOut, TIMEOUT_TIME, TIMER_FOREVER );

    while ( byTimerHandle != INVALID_TIMER_HANDLE )
    {
        ioPoll();

        result = ReceiveData(RecvBuffer );

        if ( result == STATUS_FRAMERECEIVED )
        {
            if ( waitForResponseTimeOut == FALSE && ( RecvBuffer[ 1 ] == RESPONSE ) && ( RecvBuffer[ 2 ] == byFunc ) )
            {
                bFound = TRUE;
                TimerCancel(byTimerHandle);
                byTimerHandle = INVALID_TIMER_HANDLE;
            }
            else
            {
                waitForResponseTimeOut = TRUE;
            }
        }
        else
        {
            if ( waitForResponseTimeOut == TRUE )
            {
                waitForResponseTimeOut = FALSE;
                if ( retry != 0 )
                {
                  retry--;
                  if(retry>MAX_SERIAL_RETRY)
                  {
                    bFound = TRUE;
                    TimerCancel(byTimerHandle);
                    byTimerHandle = INVALID_TIMER_HANDLE;
                  }
                  else
                    SendData( /*( void* )*/ & TransmitBuffer[ 1 ], TransmitBufferLength );
                }
                else
                {
                    TimerCancel(byTimerHandle);
                    byTimerHandle = INVALID_TIMER_HANDLE;
                }
            }

          if(retry>MAX_SERIAL_RETRY)
          {
            bFound = TRUE;
            TimerCancel(byTimerHandle);
            byTimerHandle = INVALID_TIMER_HANDLE;
          }



        }
    }

    TimerCancel(byTimerHandle);
    byTimerHandle = INVALID_TIMER_HANDLE;

    if ( !bFound )
    {
        if ( byLen != NULL )
            * byLen = 0;

    }
    else
    {
        if ( buffer != NULL )
        {
            memcpy( buffer, RecvBuffer, RecvBuffer[ 0 ] );
        }

        if ( byLen != NULL )
        {
            *byLen = RecvBuffer[ 0 ];
        }
    }

    bWaitForResponseRunning = FALSE;

}



/**
  * \ingroup SerialAPI
  * Execute a callback based on the received frame.
  * NOTE: Must only be called locally in the Serial API.
  * \param[in] pData    Pointer to data frame (without SOF)
  * \param[in] byLen    Length of data frame
  */
void Dispatch( BYTE *pData )
{
#define IDX_CMD   2
#define IDX_DATA  3

  int i;

  i=0;
  if (pData[1]==1) return;

  switch ( pData[ IDX_CMD ] )
  {

    case FUNC_ID_ZW_APPLICATION_CONTROLLER_UPDATE:
      {
        if (pData[ IDX_DATA + 2 ]!=0x0)
        {
          for ( i = 0;i < pData[ IDX_DATA + 2 ];i++ )
              pCmd[ i ] = pData[ IDX_DATA + 3 + i ];

          if ( funcApplicationControllerUpdate )
          {
              learnNodeInfo.bStatus = pData[ IDX_DATA ];
              learnNodeInfo.bSource = pData[ IDX_DATA + 1 ];
              learnNodeInfo.pCmd = pCmd;
              learnNodeInfo.bLen = pData[ IDX_DATA + 2 ];
              funcApplicationControllerUpdate(learnNodeInfo.bStatus,
                                              learnNodeInfo.bSource,
                                              learnNodeInfo.pCmd,
                                              learnNodeInfo.bLen);
          }

        }

      }
      break;

    case FUNC_ID_APPLICATION_COMMAND_HANDLER:
      {
          for ( i = 0;i < pData[ IDX_DATA + 2 ];i++ )
              pCmd[ i ] = pData[ IDX_DATA + 3 + i ];
          funcApplicationCommandHandler( pData[ IDX_DATA ], pData[ IDX_DATA + 1 ], (ZW_APPLICATION_TX_BUFFER*)pCmd, pData[ IDX_DATA + 2 ] );
      }
      break;

          // The rest are callback functions

    case FUNC_ID_ZW_SEND_NODE_INFORMATION:
      if ( cbFuncZWSendNodeInformation != NULL )
      {
          cbFuncZWSendNodeInformation( pData[ IDX_DATA + 1 ] );
      }
      break;

    case FUNC_ID_ZW_SEND_DATA:
      {
          if ( cbFuncZWSendData != NULL )
              cbFuncZWSendData( pData[ IDX_DATA + 1 ] );
      }
      break;

    case FUNC_ID_ZW_SEND_DATA_MULTI:
      if ( cbFuncZWSendDataMulti != NULL )
      {
          cbFuncZWSendDataMulti( pData[ IDX_DATA + 1 ] );
      }
      break;

    case FUNC_ID_MEMORY_PUT_BUFFER:
      if ( cbFuncMemoryPutBuffer != NULL )
      {
          cbFuncMemoryPutBuffer();
      }
      break;

    case FUNC_ID_ZW_SET_DEFAULT:
      if ( cbFuncZWSetDefault != NULL )
      {
          cbFuncZWSetDefault();
      }
      break;

    case FUNC_ID_ZW_CONTROLLER_CHANGE:

    case FUNC_ID_ZW_CREATE_NEW_PRIMARY:

    case FUNC_ID_ZW_REMOVE_NODE_FROM_NETWORK:

    case FUNC_ID_ZW_ADD_NODE_TO_NETWORK:
      if ( cbFuncZWNewController != NULL )
      {

        if ((pData[ IDX_DATA + 1 ]==0x03) || (pData[ IDX_DATA + 1 ]==0x04))
        {

          for ( i = 0;i < pData[ IDX_DATA + 3 ];i++ )
          {
              pCmd[ i ] = pData[ IDX_DATA + 4 + i ];
          }
          learnNodeInfo.bSource = pData[ IDX_DATA + 2 ];
          learnNodeInfo.pCmd = pCmd;
          learnNodeInfo.bLen = pData[ IDX_DATA + 3 ];
        }
          learnNodeInfo.bStatus = pData[ IDX_DATA + 1 ];
   /*
          if (learnNodeInfo.bStatus==0x6)
            cbFuncZWNewController( &learnNodeInfo );
     */

          cbFuncZWNewController( &learnNodeInfo );
      }

      break;

    case FUNC_ID_ZW_REPLICATION_SEND_DATA:
      if ( cbFuncZWReplicationSendData != NULL )
      {
          cbFuncZWReplicationSendData( pData[ IDX_DATA + 1 ] );
      }
      break;

    case FUNC_ID_ZW_ASSIGN_RETURN_ROUTE:
      if ( cbFuncZWAssignReturnRoute != NULL )
      {
          cbFuncZWAssignReturnRoute( pData[ IDX_DATA + 1 ] );
      }
      break;

    case FUNC_ID_ZW_DELETE_RETURN_ROUTE:
      if ( cbFuncZWDeleteReturnRoute != NULL )
      {
          cbFuncZWDeleteReturnRoute( pData[ IDX_DATA + 1 ] );
      }
      break;

    case FUNC_ID_ZW_ASSIGN_SUC_RETURN_ROUTE:
      if ( cbFuncZWAssignSUCReturnRoute != NULL )
      {
          cbFuncZWAssignSUCReturnRoute( pData[ IDX_DATA + 1 ] );
      }
      break;

    case FUNC_ID_ZW_DELETE_SUC_RETURN_ROUTE:
      if ( cbFuncZWDeleteSUCReturnRoute != NULL )
      {
          cbFuncZWDeleteSUCReturnRoute( pData[ IDX_DATA + 1 ] );
      }
      break;

    case FUNC_ID_ZW_SET_LEARN_MODE:
      if ( cbFuncZWSetLearnMode != NULL )
      {
          learnNodeInfo.bStatus = pData[ IDX_DATA + 1 ];
          learnNodeInfo.bSource = pData[ IDX_DATA + 2 ];

          for ( i = 0;i < pData[ IDX_DATA + 3 ];i++ )
          {
              pCmd[ i ] = pData[ IDX_DATA + 4 + i ];
          }
//          learnNodeInfo.bSource = pData[ IDX_DATA + 2 ];
          learnNodeInfo.pCmd = pCmd;
          learnNodeInfo.bLen = pData[ IDX_DATA + 3 ];

//          learnNodeInfo.bStatus = pData[ IDX_DATA + 1 ];

          cbFuncZWSetLearnMode( &learnNodeInfo /*learnNodeInfo.bStatus, learnNodeInfo.bSource */);
      }
      break;

    case FUNC_ID_ZW_SET_SUC_NODE_ID:
      if ( cbFuncZWSetSUCNodeID != NULL )
      {
          cbFuncZWSetSUCNodeID( pData[ IDX_DATA + 1 ] );
      }
      break;

    case FUNC_ID_ZW_REQUEST_NODE_NEIGHBOR_UPDATE:
      if ( cbFuncZWRequestNodeNodeNeighborUpdate != NULL )
      {
          cbFuncZWRequestNodeNodeNeighborUpdate( pData[ IDX_DATA + 1 ] );
      }
      break;

    case FUNC_ID_ZW_REQUEST_NETWORK_UPDATE:
      if ( cbFuncZWRequestNetworkUpdate != NULL )
      {
          cbFuncZWRequestNetworkUpdate( pData[ IDX_DATA + 1 ] );
      }
      break;
    }
}

/**
  * \ingroup SerialAPI
  * \defgroup ZWCMD Z-Wave Commands
  */

/**
  * \ingroup ZWCMD
  * Initialize the Z-Wave RF chip.
  * \param[in] mode
  * = TRUE : Set the RF chip in receive mode and starts the data sampling. \n
  * = FALSE : Set the RF chip in power down mode. \n
  */
BYTE ZW_SetRFReceiveMode( BYTE mode )
{
    idx = 1;
    byLen = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_SET_RF_RECEIVE_MODE;
    buffer[ idx++ ] = mode;
    buffer[ 0 ] = idx;	// length
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_ZW_SET_RF_RECEIVE_MODE, buffer, &byLen );

    return buffer[ IDX_DATA ];
}

/*==========================   ZW_RFPowerLevelSet  ==========================
**    Set the powerlevel used in RF transmitting.
**    Valid powerlevel values are :
**
**       normalPower : Max power possible
**       minus2dBm    - normalPower - 2dBm
**       minus4dBm    - normalPower - 4dBm
**       minus6dBm    - normalPower - 6dBm
**       minus8dBm    - normalPower - 8dBm
**       minus10dBm   - normalPower - 10dBm
**       minus12dBm   - normalPower - 12dBm
**       minus14dBm   - normalPower - 14dBm
**       minus16dBm   - normalPower - 16dBm
**       minus18dBm   - normalPower - 18dBm
**
**--------------------------------------------------------------------------*/
BYTE                    /*RET The powerlevel set */
ZW_RFPowerLevelSet(
  BYTE powerLevel)      /* IN Powerlevel to set */
{
  idx = 1;
  byLen = 0;
  buffer[idx++] = REQUEST;
  buffer[idx++] = FUNC_ID_ZW_RF_POWER_LEVEL_SET;
  buffer[idx++] = powerLevel;
  buffer[0] = idx;      // length
  SendDataAndWaitForResponse(buffer, idx, FUNC_ID_ZW_RF_POWER_LEVEL_SET, buffer, &byLen);
  return buffer[ IDX_DATA ];
}

/*===========================   ZW_SendNodeInformation   ====================
**    Create and transmit a node information broadcast frame
**    RET  FALSE if transmitter queue overflow
**--------------------------------------------------------------------------*/
BYTE                            /*RET  FALSE if transmitter queue overflow   */
ZW_SendNodeInformation(
  BYTE destNode,                /*IN  Destination Node ID  */
  BYTE txOptions,               /*IN  Transmit option flags         */
  VOID_CALLBACKFUNC(completedFunc)(/*uto*/ BYTE))  /*IN  Transmit completed call back function  */
{
  idx = 0;
  byCompletedFunc = (completedFunc == NULL ? 0 : 0x03);
  buffer[idx++] = 0;
  buffer[idx++] = REQUEST;
  buffer[idx++] = FUNC_ID_ZW_SEND_NODE_INFORMATION;
  buffer[idx++] = destNode;
  buffer[idx++] = txOptions;
  buffer[idx++] = byCompletedFunc;      // Func id for CompletedFunc
  buffer[0] = idx;                      // length
  cbFuncZWSendNodeInformation = completedFunc;
  SendData(buffer, idx);
  return  0;
}

/*===============================   ZW_SendData   ===========================
**    Transmit data buffer to a single ZW-node or all ZW-nodes (broadcast).
**
**
**    txOptions:
**          TRANSMIT_OPTION_LOW_POWER     transmit at low output power level
**                                        (1/3 of normal RF range).
**          TRANSMIT_OPTION_ACK           the multicast frame will be followed
**                                        by a  singlecast frame to each of
**                                        the destination nodes
**                                        and request acknowledge from each
**                                        destination node.
**          TRANSMIT_OPTION_AUTO_ROUTE    request retransmission via repeater
**                                        nodes at normal output power level).
**
** extern BYTE            RET  FALSE if transmitter queue overflow
** ZW_SendData(
** BYTE  nodeID,          IN  Destination node ID (0xFF == broadcast)
** BYTE *pData,           IN  Data buffer pointer
** BYTE  dataLength,      IN  Data buffer length
** BYTE  txOptions,       IN  Transmit option flags
** void (*completedFunc)( IN  Transmit completed call back function
**        BYTE txStatus));    IN Transmit status
**--------------------------------------------------------------------------*/
BYTE                          /*RET  FALSE if transmitter busy      */
ZW_SendData(
  BYTE  nodeID,               /*IN  Destination node ID (0xFF == broadcast) */
  BYTE *pData,                /*IN  Data buffer pointer           */
  BYTE  dataLength,           /*IN  Data buffer length            */
  BYTE  txOptions,            /*IN  Transmit option flags         */
  VOID_CALLBACKFUNC(completedFunc)(/*auto*/ BYTE)) /*IN  Transmit completed call back function  */
{
  int i;

  idx = 0;
  byLen = 0;
  byCompletedFunc = (completedFunc == NULL ? 0 : 0x03);
  idx++;
  buffer[idx++] = REQUEST;
  buffer[idx++] = FUNC_ID_ZW_SEND_DATA;
  buffer[idx++] = nodeID;
  buffer[idx++] = dataLength;
  for (i = 0; i < dataLength; i++)
  {
    buffer[idx++] = pData[i];
  }
  buffer[idx++] = txOptions;
  buffer[idx++] = byCompletedFunc;      // Func id for CompletedFunc
  buffer[0] = idx;                      // length
  cbFuncZWSendData = completedFunc;
  SendDataAndWaitForResponse(buffer, idx, FUNC_ID_ZW_SEND_DATA, buffer, &byLen);
  return buffer[IDX_DATA];
}

/**
  * \ingroup ZWCMD
  * Transmit data buffer to a list of Z-Wave Nodes (multicast frame).
  * \param[in] pNodeIDList    List of destination node ID's
  * \param[in] numberNodes    Number of Nodes
  * \param[in] pData          Data buffer pointer
  * \param[in] dataLength     Data buffer length
  * \param[in] txOptions      Transmit option flags
  * \param[in] completedFunc  Transmit completed call back function
  * \return FALSE if transmitter queue overflow
  */
/*
BYTE ZW_SendDataMulti( BYTE *pNodeIDList, BYTE numberNodes, BYTE *pData, BYTE dataLength, BYTE txOptions, void ( *completedFunc ) ( BYTE txStatus ) )
{
    int i;
    idx = 0;
    BYTE byCompletedFunc = ( completedFunc == NULL ? 0 : 0x03 );

    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_SEND_DATA_MULTI;
    numberNodes = ( numberNodes <= MAX_NODES_IN_MULTICAST ) ? numberNodes : MAX_NODES_IN_MULTICAST;
    buffer[ idx++ ] = numberNodes;

    for ( i = 0; i < numberNodes; i++ )
    {
        buffer[ idx++ ] = pNodeIDList[ i ];
    }

    buffer[ idx++ ] = dataLength;

    for ( i = 0; i < dataLength; i++ )
    {
        buffer[ idx++ ] = pData[ i ];
    }

    buffer[ idx++ ] = txOptions;
    buffer[ idx++ ] = byCompletedFunc;	// Func id for CompletedFunc
    buffer[ 0 ] = idx;	// length
    byLen = 0;
    cbFuncZWSendDataMulti = completedFunc;
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_ZW_SEND_DATA_MULTI, buffer, &byLen );

    return buffer[ IDX_DATA ];
}
*/

/**
  * \ingroup ZWCMD
  * Copy the Home-ID and Node-ID to the specified RAM addresses.
  * \param[out] homeID  Home-ID pointer
  * \param[out] nodeID  Node-ID pointer
  */
void MemoryGetID( BYTE *pHomeID, BYTE *pNodeID )
{
    idx = 0;
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_MEMORY_GET_ID;
    buffer[ 0 ] = idx;	// length
    byLen = 0;
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_MEMORY_GET_ID, buffer, &byLen );

    pHomeID[ 0 ] = buffer[ IDX_DATA ];
    pHomeID[ 1 ] = buffer[ IDX_DATA + 1 ];
    pHomeID[ 2 ] = buffer[ IDX_DATA + 2 ];
    pHomeID[ 3 ] = buffer[ IDX_DATA + 3 ];
    pNodeID[ 0 ] = buffer[ IDX_DATA + 4 ];

}

/*============================   MemoryGetByte   =============================
**  Read one byte from the EEPROM
**
**  Side effects:
**--------------------------------------------------------------------------*/
BYTE                    /*RET Data          */
MemoryGetByte(
  WORD offset)          /*IN   Application area offset            */
{
  idx = 0;
  buffer[idx++] = 0;
  buffer[idx++] = REQUEST;
  buffer[idx++] = FUNC_ID_MEMORY_GET_BYTE;
  buffer[idx++] = (offset) >> 8;
  buffer[idx++] = (offset) & 0xFF;
  buffer[0] = idx;      // length
  byLen = 0;
  SendDataAndWaitForResponse(buffer, idx, FUNC_ID_MEMORY_GET_BYTE, buffer, &byLen);

  return buffer[IDX_DATA];
}

/*============================   MemoryPutByte   =============================
**  Add one byte to the EEPROM write queue
**
**  Side effects:
**
**--------------------------------------------------------------------------*/
BYTE                    /*RET FALSE if write buffer full    */
MemoryPutByte(
  WORD  offset,         /*IN   Application area offset   */
  BYTE  bData )         /*IN   Data to store             */
{
  idx = 0;
  buffer[idx++] = 0;
  buffer[idx++] = REQUEST;
  buffer[idx++] = FUNC_ID_MEMORY_PUT_BYTE;
  buffer[idx++] = (offset) >> 8;
  buffer[idx++] = (offset) & 0xFF;
  buffer[idx++] = bData;
  buffer[0] = idx;	    // length
  byLen = 0;
  SendDataAndWaitForResponse(buffer, idx, FUNC_ID_MEMORY_PUT_BYTE, buffer, &byLen);

  return buffer[IDX_DATA];
}

/**
  * \ingroup ZWCMD
  * Read number of bytes from the EEPROM to a RAM buffer.
  * \param[in] offset   Application area offset
  * \param[in] buffer   Buffer pointer
  * \param[in] length   Number of bytes to read
  */
void MemoryGetBuffer( WORD offset, BYTE *buf, BYTE length )
{
    int i;

    idx = 0;
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_MEMORY_GET_BUFFER;
    buffer[ idx++ ] = ( offset ) >> 8;
    buffer[ idx++ ] = ( offset ) & 0xFF;
    buffer[ idx++ ] = (BYTE)length;		// Number of bytes to read
    
//    if ( length > BUF_SIZE - 8 )
//        length = BUF_SIZE - 8;
//    buffer[ idx++ ] = ( length ) >> 8;
//    buffer[ idx++ ] = ( length ) & 0xFF;
    
    buffer[ 0 ] = idx;	// length
    byLen = 0;
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_MEMORY_GET_BUFFER, buffer, &byLen );

    for ( i = 0;i < length;i++ )
    {
        buf[ i ] = buffer[ IDX_DATA + i ];
    }
}

/**
  * \ingroup ZWCMD
  * Copy number of bytes from a RAM buffer to the EEPROM.
  * \param[in] offset   Application area offset
  * \param[in] buffer   Buffer pointer
  * \param[in] length   Number of bytes to write
  * \param[in] func     Write completed function pointer
  * \return FALSE if the buffer put queue is full
  */
BYTE MemoryPutBuffer( WORD offset, BYTE *buf, WORD length, void ( *func ) ( void ) )
{
    int i;

    idx = 0;
    byCompletedFunc = ( func == NULL ? 0 : 0x03 );
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_MEMORY_PUT_BUFFER;
    buffer[ idx++ ] = ( offset ) >> 8;
    buffer[ idx++ ] = ( offset ) & 0xFF;

    if ( length > BUF_SIZE - 8 )
        length = BUF_SIZE - 8;

    buffer[ idx++ ] = ( length ) >> 8;

    buffer[ idx++ ] = ( length ) & 0xFF;

    for ( i = 0; i < length; i++ )
    {
        buffer[ idx++ ] = buf[ i ];
    }

    buffer[ idx++ ] = byCompletedFunc;
    buffer[ 0 ] = idx;	// length
    cbFuncMemoryPutBuffer = func;
    byLen = 0;
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_MEMORY_PUT_BUFFER, buffer, &byLen );

    return buffer[ IDX_DATA ];
}






/**
  * \ingroup ZWCMD
  * Copy number of bytes from a RAM buffer to the EEPROM.
  * \param[in] offset   Application area offset
  * \param[in] buffer   Buffer pointer
  * \param[in] length   Number of bytes to write
  * \param[in] func     Write completed function pointer
  * \return FALSE if the buffer put queue is full
  */
BYTE ZW_MemoryPutBuffer( WORD offset, BYTE *buf, WORD length)
{
    int i;

    idx = 0;

    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_MEMORY_PUT_BUFFER;
    buffer[ idx++ ] = ( offset ) >> 8;
    buffer[ idx++ ] = ( offset ) & 0xFF;

    if ( length > BUF_SIZE - 8 )
        length = BUF_SIZE - 8;

    buffer[ idx++ ] = ( length ) >> 8;

    buffer[ idx++ ] = ( length ) & 0xFF;

    for ( i = 0; i < length; i++ )
    {
        buffer[ idx++ ] = buf[ i ];
    }
    buffer[ idx++ ] = byCompletedFunc;
    buffer[ 0 ] = idx;	// length

    byLen = 0;
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_MEMORY_PUT_BUFFER, buffer, &byLen );

    return buffer[ IDX_DATA ];
}







/**
  * \ingroup ZWCMD
  * Lock a response route for a specific node.
  * \param[in] bNodeID
  */
void ZW_LockRoute( BYTE bNodeID )
{
    idx = 0;
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_LOCK_ROUTE_RESPONSE;
    buffer[ idx++ ] = bNodeID;
    buffer[ 0 ] = idx;	// length
    SendData( buffer, idx );
}

#ifdef ZW_INSTALLER
/**
  * \ingroup ZWCMD
  * Read out neighbor information.
  * \param[in] bNodeID        The NodeID to get routing information from
  * \param[in] buf            Buffer pointer to where the data is placed
  * \param[in] bRemoveBad     If TRUE remove bad repeaters
  * \param[in] bRemoveNonReps If TRUE remove non repeating devices
  */
void ZW_GetRoutingInfo( BYTE bNodeID, BYTE *buf, BYTE bRemoveBad, BYTE bRemoveNonReps )
{
    int i;
    idx = 0;
    /*  bLine | bRemoveBad | bRemoveNonReps | funcID */
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_GET_ROUTING_TABLE_LINE;
    buffer[ idx++ ] = bNodeID;
    buffer[ idx++ ] = bRemoveBad;
    buffer[ idx++ ] = bRemoveNonReps;
    buffer[ 0 ] = idx;
    byLen = 0;
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_GET_ROUTING_TABLE_LINE, buffer, &byLen );

    for ( i = 0;i < 29;i++ )
    {
        buf[ i ] = buffer[ IDX_DATA + i ];
    }
}

/**
  * \ingroup ZWCMD
  * Reset TX Counter.
  */
void ZW_ResetTXCounter( void )
{
    idx = 0;
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_RESET_TX_COUNTER;
    buffer[ 0 ] = idx;	// length
    SendData( buffer, idx );
}

/**
  * \ingroup ZWCMD
  * Get TX Counter.
  * \return The number of frames transmitted since the TX Counter last was reset
  */
BYTE ZW_GetTXCounter( void )
{
    idx = 0;
    byLen = 0;
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_GET_TX_COUNTER;
    buffer[ 0 ] = idx;	// length
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_GET_TX_COUNTER, buffer, &byLen );
    return buffer[ IDX_DATA ];
}

#endif

/*=====================   ZW_RequestNodeNeighborUpdate  ======================
**
**    Start neighbor discovery for bNodeID, if any other nodes present.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
BYTE                                /*RET TRUE neighbor discovery started */
ZW_RequestNodeNeighborUpdate(
  BYTE bNodeID,                     /* IN Node id */
  VOID_CALLBACKFUNC(completedFunc)( /* IN Function to be called when the done */
    auto BYTE))
{
    idx = 0;
    byCompletedFunc = ( completedFunc == NULL ? 0 : 0x03 );
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_REQUEST_NODE_NEIGHBOR_UPDATE;
    buffer[ idx++ ] = bNodeID;
    buffer[ idx++ ] = byCompletedFunc;
    buffer[ 0 ] = idx;	// length
    cbFuncZWRequestNodeNodeNeighborUpdate = completedFunc;
    SendData( buffer, idx );
    return  0;
}

/**
  * \ingroup ZWCMD
  * Copy the Node's current protocol information from the non-volatile memory.
  * \param[in] nodeID      Node ID
  * \param[out] nodeInfo   Node info buffer
  */

void ZW_GetNodeProtocolInfo( BYTE bNodeID, NODEINFO *nodeInfo )
{
    idx = 0;
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_GET_NODE_PROTOCOL_INFO;
    buffer[ idx++ ] = bNodeID;
    buffer[ 0 ] = idx;	// length
    byLen = 0;
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_ZW_GET_NODE_PROTOCOL_INFO, buffer, &byLen );

    nodeInfo->capability = buffer[ IDX_DATA ];
    nodeInfo->security = buffer[ IDX_DATA + 1 ];
    nodeInfo->reserved = buffer[ IDX_DATA + 2 ];
#ifdef NEW_NODEINFO

    nodeInfo->nodeType.basic = buffer[ IDX_DATA + 3 ];
    nodeInfo->nodeType.generic = buffer[ IDX_DATA + 4 ];
    nodeInfo->nodeType.specific = buffer[ IDX_DATA + 5 ];
#else

    nodeInfo->nodeType = buffer[ IDX_DATA + 3 ];
#endif
}


/*===========================   ZW_SetDefault   =============================
**    Remove all Nodes and timers from the EEPROM memory.
**
**--------------------------------------------------------------------------*/
void                                /*RET Nothing */
ZW_SetDefault(
  VOID_CALLBACKFUNC(completedFunc)( /* IN Command completed call back function */
    void))
{
    idx = 0;
    byCompletedFunc = ( completedFunc == NULL ? 0 : 0x03 );
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_SET_DEFAULT;
    buffer[ idx++ ] = byCompletedFunc;
    buffer[ 0 ] = idx;	// length
    cbFuncZWSetDefault = completedFunc;
    SendData( buffer, idx );
}

/*========================   ZW_ControllerChange   ======================
**
**    Transfer the role as primary controller to another controller
**
**    The modes are:
**
**    CONTROLLER_CHANGE_START          Start the creation of a new primary
**    CONTROLLER_CHANGE_STOP           Stop the creation of a new primary
**    CONTROLLER_CHANGE_STOP_FAILED    Report that the replication failed
**
**    ADD_NODE_OPTION_HIGH_POWER       Set this flag in bMode for High Power exchange.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
ZW_ControllerChange(BYTE bMode,
                        VOID_CALLBACKFUNC(completedFunc)(auto LEARN_INFO*))
{
    idx = 0;
    byCompletedFunc = ( completedFunc == NULL ? 0 : 0x03 );
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_CONTROLLER_CHANGE;
    buffer[ idx++ ] = bMode;
    buffer[ idx++ ] = byCompletedFunc;
    buffer[ 0 ] = idx;	// length
    cbFuncZWNewController = completedFunc;
    SendData( buffer, idx );
}

/*==========================   ZW_AddNodeToNetwork   ========================
**
**    Add any type of node to the network
**
**    The modes are:
**
**    ADD_NODE_ANY            Add any node to the network
**    ADD_NODE_CONTROLLER     Add a controller to the network
**    ADD_NODE_SLAVE          Add a slaev node to the network
**    ADD_NODE_STOP           Stop learn mode without reporting an error.
**    ADD_NODE_STOP_FAILED    Stop learn mode and report an error to the
**                            new controller.
**
**    ADD_NODE_OPTION_HIGH_POWER    Set this flag in bMode for High Power inclusion.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
ZW_AddNodeToNetwork(BYTE bMode,
                    VOID_CALLBACKFUNC(completedFunc)(auto LEARN_INFO*))
{
    idx = 0;
    byCompletedFunc = ( completedFunc == NULL ? 0 : 0x03 );
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_ADD_NODE_TO_NETWORK;
    buffer[ idx++ ] = bMode;
    buffer[ idx++ ] = byCompletedFunc;
    buffer[ 0 ] = idx;	// length
    cbFuncZWNewController = completedFunc;
    SendData( buffer, idx );
}

/*==========================   ZW_RemoveNodeFromNetwork   ========================
**
**    Remove any type of node from the network
**
**    The modes are:
**
**    REMOVE_NODE_ANY            Remove any node from the network
**    REMOVE_NODE_CONTROLLER     Remove a controller from the network
**    REMOVE_NODE_SLAVE          Remove a slaev node from the network
**
**    REMOVE_NODE_STOP           Stop learn mode without reporting an error.
**
**    ADD_NODE_OPTION_HIGH_POWER    Set this flag in bMode for High Power exclusion.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
ZW_RemoveNodeFromNetwork(BYTE bMode,
                    VOID_CALLBACKFUNC(completedFunc)(auto LEARN_INFO*))
{
    idx = 0;
    byCompletedFunc = ( completedFunc == NULL ? 0 : 0x03 );
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_REMOVE_NODE_FROM_NETWORK;
    buffer[ idx++ ] = bMode;
    buffer[ idx++ ] = byCompletedFunc;
    buffer[ 0 ] = idx;	// length
    cbFuncZWNewController = completedFunc;
    SendData( buffer, idx );
}

/**
  * \ingroup ZWCMD
  * Sends command completed to master remote.
  * Called in replication mode when a command from the sender has been processed.
  */
/*
void ZW_ReplicationCommandComplete( void )
{
    idx = 0;
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_REPLICATION_COMMAND_COMPLETE;
    buffer[ 0 ] = idx;	// length
    SendData( buffer, idx );
}
*/

/**
  * \ingroup ZWCMD
  * Used when the controller is replication mode.
  * It sends the payload and expects the receiver to respond with a command complete message.
  * \param[in] nodeID         Destination node ID
  * \param[in] pData          Data buffer pointer
  * \param[in] dataLength     Data buffer length
  * \param[in] txOptions      Transmit option flags
  * \param[in] completedFunc  Transmit completed call back function
  * \param[in] txStatus       Transmit status
  * \return FALSE if transmitter queue overflow
  */
/*
BYTE ZW_ReplicationSendData( BYTE nodeID, BYTE *pData, BYTE dataLength, BYTE txOptions, void ( *completedFunc ) ( BYTE ) )
{
    int i;
    idx = 0;
    BYTE byCompletedFunc = ( completedFunc == NULL ? 0 : 0x03 );
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_REPLICATION_SEND_DATA;
    buffer[ idx++ ] = nodeID;
    buffer[ idx++ ] = dataLength;

    for ( i = 0;i < dataLength;i++ )
        buffer[ idx++ ] = pData[ i ];

    buffer[ idx++ ] = txOptions;
    buffer[ idx++ ] = byCompletedFunc;	// Func id for CompletedFunc
    buffer[ 0 ] = idx;	// length
    byLen = 0;
    cbFuncZWReplicationSendData = completedFunc;
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_ZW_REPLICATION_SEND_DATA, buffer, &byLen );

    return buffer[ IDX_DATA ];
}
*/

/*========================   ZW_AssignReturnRoute   =========================
**
**    Assign static return routes within a Routing Slave node.
**    Calculate the shortest transport routes from the Routing Slave node
**    to the route destination node and
**    transmit the return routes to the Routing Slave node.
**
**--------------------------------------------------------------------------*/
BOOL                                /*RET TRUE if assign was initiated. FALSE if not */
ZW_AssignReturnRoute(
  BYTE  bSrcNodeID,                 /* IN Routing Slave Node ID */
  BYTE  bDstNodeID,                 /* IN Route destination Node ID */
  VOID_CALLBACKFUNC(completedFunc)( /* IN Callback function called when done */
  BYTE bStatus))
{
    idx = 0;
    byCompletedFunc = ( completedFunc == NULL ? 0 : 0x03 );
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_ASSIGN_RETURN_ROUTE;
    buffer[ idx++ ] = bSrcNodeID;
    buffer[ idx++ ] = bDstNodeID;
    buffer[ idx++ ] = byCompletedFunc;	// Func id for CompletedFunc
    buffer[ 0 ] = idx;	// length
    cbFuncZWAssignReturnRoute = completedFunc;

    byLen = 0;
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_ZW_ASSIGN_RETURN_ROUTE, buffer, &byLen );
    return buffer[ IDX_DATA ];

///    SendData( buffer, idx );
///    return  0;
}

/**
  * \ingroup ZWCMD
  * Delete static return routes within a Routing Slave node.
  * Transmit "NULL" routes to the Routing Slave node.
  * \param[in] nodeID          Routing Slave
  * \param[in] completedFunc   Completion handler
  * \param[in] bStatus        Transmit complete status
  */
/*
void ZW_DeleteReturnRoute( BYTE nodeID, void ( *completedFunc ) ( BYTE ) )
{
    BYTE byCompletedFunc;

    idx = 0;
    byCompletedFunc = ( completedFunc == NULL ? 0 : 0x03 );
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_DELETE_RETURN_ROUTE;
    buffer[ idx++ ] = nodeID;
    buffer[ idx++ ] = byCompletedFunc;	// Func id for CompletedFunc
    buffer[ 0 ] = idx;	// length
    cbFuncZWDeleteReturnRoute = completedFunc;
    SendData( buffer, idx );
}
*/
/**
  * \ingroup ZWCMD
  * Assign static return routes within a Routing Slave node.
  * Calculate the shortest transport routes to a Routing Slave node
  * from the Static Update Controller (SUC) Node and
  * transmit the return routes to the Routing Slave node.
  * \param[in] bSrcNodeID     Routing Slave Node ID
  * \param[in] bDstNodeID     SUC node ID
  * \param[in] completedFunc  Completion handler
  * \param[in] bStatus        Transmit complete status
  */
/*
void ZW_AssignSUCReturnRoute( BYTE bSrcNodeID, BYTE bDstNodeID, void ( *completedFunc ) ( BYTE ) )
{
    BYTE byCompletedFunc;

    idx = 0;
    byCompletedFunc = ( completedFunc == NULL ? 0 : 0x03 );
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_ASSIGN_SUC_RETURN_ROUTE;
    buffer[ idx++ ] = bSrcNodeID;
    buffer[ idx++ ] = bDstNodeID;
    buffer[ idx++ ] = byCompletedFunc;	// Func id for CompletedFunc
    buffer[ 0 ] = idx;	// length
    cbFuncZWAssignSUCReturnRoute = completedFunc;
    SendData( buffer, idx );
}
*/

/**
  * \ingroup ZWCMD
  * Delete the ( Static Update Controller -SUC-) static return routes
  * within a Routing Slave node.
  * Transmit "NULL" routes to the Routing Slave node.
  * \param[in] nodeID         Routing Slave
  * \param[in] completedFunc  Completion handler
  * \param[in] bStatus        Transmit complete status
  */
/*
void ZW_DeleteSUCReturnRoute( BYTE nodeID, void ( *completedFunc ) ( BYTE ) )
{
    BYTE byCompletedFunc;

    idx = 0;
    byCompletedFunc = ( completedFunc == NULL ? 0 : 0x03 );
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_DELETE_SUC_RETURN_ROUTE;
    buffer[ idx++ ] = nodeID;
    buffer[ idx++ ] = byCompletedFunc;	// Func id for CompletedFunc
    buffer[ 0 ] = idx;	// length
    cbFuncZWDeleteSUCReturnRoute = completedFunc;
    SendData( buffer, idx );
}
*/

/**
  * \ingroup ZWCMD
  * Get the currently registered SUC node ID.
  * \return SUC node ID, ZERO if no SUC available
  */
BYTE ZW_GetSUCNodeID( void )
{
    idx = 0;
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_GET_SUC_NODE_ID;
    buffer[ 0 ] = idx;	// length
    byLen = 0;
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_ZW_GET_SUC_NODE_ID, buffer, &byLen );
    return buffer[ IDX_DATA ];
}

/*============================   ZW_SetSUCNodeID  ===========================
**    Function description
**    This function enable /disable a specified static controller
**    of functioning as the Static Update Controller
**
**--------------------------------------------------------------------------*/
BYTE                 /*RET TRUE target is a static controller*/
                     /*    FALSE if the target is not a static controller,  */
                     /*    the source is not primary or the SUC functinality is not enabled.*/
ZW_SetSUCNodeID(
  BYTE nodeID,       /* IN the node ID of the static controller to be a SUC */
  BYTE SUCState,     /* IN TRUE enable SUC, FALSE disable */
  BYTE bTxOption,    /* IN TRUE if to use low poer transmition, FALSE for normal Tx power */
  BYTE bCapabilities,             /* The capabilities of the new SUC */
  VOID_CALLBACKFUNC(completedFunc)(auto BYTE txStatus)) /* IN a call back function */
{
    idx = 0;
    byCompletedFunc = ( completedFunc == NULL ? 0 : 0x03 );
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_SET_SUC_NODE_ID;
    buffer[ idx++ ] = nodeID;
    buffer[ idx++ ] = SUCState;  /* Do we want to enable or disable?? */
    buffer[ idx++ ] = bTxOption;
    buffer[ idx++ ] = bCapabilities;
    buffer[ idx++ ] = byCompletedFunc;	// Func id for CompletedFunc

    buffer[ 0 ] = idx;	// length
    cbFuncZWSetSUCNodeID = completedFunc;
    byLen = 0;
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_ZW_SET_SUC_NODE_ID, buffer, &byLen );
    return buffer[ IDX_DATA ];
}

/*===========================   ZW_SetLearnMode   ===========================
**    Enable/Disable home/node ID learn mode.
**    When learn mode is enabled, received "Assign ID's Command" are handled:
**    If the current stored ID's are zero, the received ID's will be stored.
**    If the received ID's are zero the stored ID's will be set to zero.
**
**    The learnFunc is called when the received assign command has been handled.
**
**--------------------------------------------------------------------------*/
void                                    /*RET  Nothing        */
ZW_SetLearnMode(
  BYTE mode,                            /*IN  TRUE: Enable; FALSE: Disable   */
  VOID_CALLBACKFUNC(completedFunc)(LEARN_INFO*)/*VOID_CALLBACKFUNC(learnFunc)( BYTE bStatus,  BYTE nodeID)*/)  /*IN  Node learn call back function. */
{
  idx = 0;
  byCompletedFunc = (completedFunc == NULL ? 0 : 0x03);
  buffer[idx++] = 0;
  buffer[idx++] = REQUEST;
  buffer[idx++] = FUNC_ID_ZW_SET_LEARN_MODE;
  buffer[idx++] = mode;
  buffer[idx++] = byCompletedFunc;      // Func id for CompletedFunc
  buffer[0] = idx;                      // length
  cbFuncZWSetLearnMode = completedFunc;
  SendData(buffer, idx);
}

/**
  * \ingroup ZWCMD
  * Get the Z-Wave library basis version.
  * \param[out]   pBuf
  *   Pointer to buffer where version string will be
  *   copied to. Buffer must be at least 14 bytes long.
  */
BYTE ZW_Version( BYTE *pBuf )
{
    BYTE retVal;

    idx = 0;
    byLen = 0;
    retVal = 0;
    memset( buffer, 0, sizeof( buffer ) );
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_GET_VERSION;
    buffer[ 0 ] = idx;	// length
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_ZW_GET_VERSION, buffer, &byLen );

    if ( memcmp( (void *)&buffer[ IDX_DATA ], (void *)"Z-Wave", 6 ) == 0 )
    {
        retVal = buffer[ IDX_DATA + 12 ];
        memcpy( pBuf, (void *)&buffer[ IDX_DATA ], 12 );
        pBuf[ 12 ] = 0;
    }

    return retVal;
}

/**
  * \ingroup ZWCMD
  * Set ApplicationNodeInformation data to be used in subsequent
  * calls to \ref ZW_SendNodeInformation.
  * \param[in] listening   = TRUE : if this node is always on air
  * \param[in] nodeType    Device type
  * \param[in] nodeParm    Device parameter buffer
  * \param[in] parmLength  Number of Device parameter bytes
  */
#ifdef NEW_NODEINFO
void SerialAPI_ApplicationNodeInformation( BYTE listening, APPL_NODE_TYPE nodeType, BYTE *nodeParm, BYTE parmLength )
#else
void SerialAPI_ApplicationNodeInformation( BYTE listening, BYTE nodeType, BYTE *nodeParm, BYTE parmLength )
#endif
{
    int i;
    idx = 0;
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_SERIAL_API_APPL_NODE_INFORMATION;
    buffer[ idx++ ] = listening;
#ifdef NEW_NODEINFO

    buffer[ idx++ ] = nodeType.generic;
    buffer[ idx++ ] = nodeType.specific;
#else

    buffer[ idx++ ] = nodeType;
#endif

    buffer[ idx++ ] = parmLength;

    for ( i = 0; i != parmLength; i++ )
    {
        buffer[ idx++ ] = nodeParm[ i ];
    }

    buffer[ 0 ] = idx;			// length
    SendData( buffer, idx );
}

/**
  * \ingroup ZWCMD
  * Get Serial API initialization data from remote side (Enhanced Z-Wave module).
  * \param[out]   ver            Remote sides Serial API version
  * \param[out]   capabilities   Capabilities flag (GET_INIT_DATA_FLAG_xxx)
  *   Capabilities flag: \n
  *      Bit 0: 0 = Controller API; 1 = Slave API \n
  *      Bit 1: 0 = Timer functions not supported; 1 = Timer functions supported. \n
  *      Bit 2: 0 = Primary Controller; 1 = Secondary Controller \n
  *      Bit 3: 0 = Not SUC; 1 = This node is SUC (static controller only) \n
  *      Bit 3-7: Reserved \n
  *   Timer functions are: TimerStart, TimerRestart and TimerCancel.
  * \param[out]   len            Number of bytes in nodesList
  * \param[out]   nodesList      Bitmask list with nodes known by Z-Wave module
  */
BYTE SerialAPI_GetInitData( BYTE *ver, BYTE *capabilities, BYTE *len, BYTE *nodesList )
{
    int i;
    idx = 0;
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_SERIAL_API_GET_INIT_DATA;
    buffer[ 0 ] = idx;			// length
    byLen = 0;
    *ver = 0;
    *capabilities = 0;
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_SERIAL_API_GET_INIT_DATA, buffer, &byLen );
    *ver = buffer[ IDX_DATA ];

    //controller api eller slave api
    *capabilities = buffer[ IDX_DATA + 1 ];
    *len = buffer[ IDX_DATA + 2 ];

    for ( i = 0; i < buffer[ IDX_DATA + 2 ]; i++ )
        nodesList[ i ] = buffer[ IDX_DATA + 3 + i ];

    // Bit 2 tells if it is Primary Controller (FALSE) or Secondary Controller (TRUE).
    if ( ( *capabilities ) & GET_INIT_DATA_FLAG_SECONDARY_CTRL )
        return ( TRUE );
    else
        return ( FALSE );
}

/**
  * \ingroup ZWCMD
  * Enable the SUC functionality in a controller.
  * \param[in] state
  *   = TRUE : Enable SUC \n
  *   = FALSE : Disable SUC \n
  * \param[in] capabilities
  *   = ZW_SUC_FUNC_BASIC_SUC : Only enables the basic SUC functionality \n
  *   = ZW_SUC_FUNC_NODEID_SERVER : Enable the node ID server functionality to become a SIS. \n
  * \return
  *   = TRUE : The SUC functionality was enabled/disabled \n
  *   = FALSE : Attempting to disable a running SUC, not allowed \n
  */
BOOL ZW_EnableSUC( BYTE state, BYTE capabilities )
{
    idx = 0;
    idx++;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_ENABLE_SUC;
    buffer[ idx++ ] = state;
    buffer[ idx++ ] = capabilities;
    buffer[ 0 ] = idx;	// length
    byLen = 0;
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_ZW_ENABLE_SUC, buffer, &byLen );

    return buffer[ IDX_DATA ];
}

/*============================   ZW_RequestNodeInfo   ======================
**    Function description.
**     Request a node to send it's node information.
**     Function return TRUE if the request is send, else it return FALSE.
**     FUNC is a callback function, which is called with the status of the
**     Request nodeinformation frame transmission.
**     If a node sends its node info, ApplicationControllerUpdate will be called
**     with UPDATE_STATE_NODE_INFO_RECEIVED as status together with the received
**     nodeinformation.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
BOOL                      /*RET FALSE if transmitter busy */
ZW_RequestNodeInfo(
  BYTE nodeID,                     /*IN: node id of the node to request node info from it.*/
  VOID_CALLBACKFUNC(completedFunc)(auto BYTE)) /* IN Callback function */
{
    idx = 0;
    byLen = 0;
    idx++;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_REQUEST_NODE_INFO;
    buffer[ idx++ ] = nodeID;
    buffer[ 0 ] = idx;	// length
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_ZW_REQUEST_NODE_INFO, buffer, &byLen );

    return buffer[ IDX_DATA ];
}

/**
  * \ingroup ZWCMD
  * Used when the controller is replication mode.
  * It sends the payload and expects the receiver to respond with a command complete message.
  * \param[in] nodeID         Destination node ID
  * \param[in] pData          Data buffer pointer
  * \param[in] dataLength     Data buffer length
  * \param[in] txOptions      Transmit option flags
  * \param[in] completedFunc  Transmit completed call back function
  * \param[in] txStatus       Transmit status
  * \return FALSE if transmitter queue overflow
  */

BYTE ZW_ReplicationSend( BYTE nodeID, BYTE *pData, BYTE dataLength, BYTE txOptions, VOID_CALLBACKFUNC(completedFunc ) (auto  BYTE txStatus) )
{
    int i;
     byCompletedFunc = ( completedFunc == NULL ? 0 : 0x03 );
    idx = 0;

    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_REPLICATION_SEND_DATA;
    buffer[ idx++ ] = nodeID;
    buffer[ idx++ ] = dataLength;

    for ( i = 0;i < dataLength;i++ )
        buffer[ idx++ ] = pData[ i ];

    buffer[ idx++ ] = txOptions;
    buffer[ idx++ ] = byCompletedFunc;	// Func id for CompletedFunc
    buffer[ 0 ] = idx;	// length
    byLen = 0;
    cbFuncZWReplicationSendData = completedFunc;
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_ZW_REPLICATION_SEND_DATA, buffer, &byLen );

    return buffer[ IDX_DATA ];
}

/**
  * \ingroup ZWCMD
  * Get capabilities of a controller.
  * \return
  *   = CONTROLLER_IS_SECONDARY :
  *      If bit is set then the controller is a secondary controller \n
  *   = CONTROLLER_ON_OTHER_NETWORK :
  *      If this bit is set then this controller is not using its build-in home ID \n
  *   = CONTROLLER_IS_SUC :
  *      If this bit is set then this controller is a SUC \n
  *   = CONTROLLER_NODEID_SERVER_PRESENT :
  *      If this bit is set then there is a SUC ID server (SIS)
  *      in the network and this controller can therefore
  *      include/exclude nodes in the network (inclusion controller). \n
  */
BYTE ZW_GetControllerCapabilities( void )
{
    idx = 0;
    byLen = 0;
    idx++;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_GET_CONTROLLER_CAPABILITIES;
    buffer[ 0 ] = idx;	// length
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_ZW_GET_CONTROLLER_CAPABILITIES, buffer, &byLen );

    return buffer[ IDX_DATA ];
}

BOOL ZW_RequestNetworkUpdate( void ( *complFunc ) ( auto BYTE ) )
{
    idx = 0;
    byCompletedFunc = ( complFunc == NULL ? 0 : 0x03 );
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_REQUEST_NETWORK_UPDATE;
    buffer[ idx++ ] = byCompletedFunc;
    buffer[ 0 ] = idx;	// length
    cbFuncZWRequestNetworkUpdate = complFunc;
    byLen = 0;
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_ZW_REQUEST_NETWORK_UPDATE, buffer, &byLen );
    return buffer[ IDX_DATA ];
}

/**
  * \ingroup ZWCMD
  * Get the Z Wave library type.
  * \return
  *   = ZW_LIB_CONTROLLER_STATIC	Static controller library
  *   = ZW_LIB_CONTROLLER_BRIDGE	Bridge controller library
  *   = ZW_LIB_CONTROLLER	        Portable controller library
  *   = ZW_LIB_SLAVE_ENHANCED	    Enhanced slave library
  *   = ZW_LIB_SLAVE_ROUTING	    Routing slave library
  *   = ZW_LIB_SLAVE	            Slave library
  *   = ZW_LIB_INSTALLER	        Installer library
  */
BYTE ZW_Type_Library( void )
{
    idx = 0;
    byLen = 0;
    idx++;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_TYPE_LIBRARY;
    buffer[ 0 ] = idx;	// length
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_ZW_TYPE_LIBRARY, buffer, &byLen );

    return buffer[ IDX_DATA ];
}

/**
  * \ingroup ZWCMD
  * Lock a response route for a specific node.
  * \param[in] bNodeID
  */
void ZW_ReplicationReceiveComplete()
{
    idx = 0;
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_REPLICATION_COMMAND_COMPLETE;
    buffer[ 0 ] = idx;	// length
    SendData( buffer, idx );
}

/**
  * \ingroup ZWCMD
  * Lock a response route for a specific node.
  * \param[in] bNodeID
  */
void ZW_SoftReset()
{
    idx = 0;
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_SERIAL_API_SOFT_RESET;
    buffer[ 0 ] = idx;	// length
    SendData( buffer, idx );
}

/*==========================   ZW_RFPowerLevelGet  ==========================
**    Get the current powerlevel used in RF transmitting.
**    Possible powerlevel return values are :
**
**       normalPower : Max power possible
**       minus2dBm    - normalPower - 2dBm
**       minus4dBm    - normalPower - 4dBm
**       minus6dBm    - normalPower - 6dBm
**       minus8dBm    - normalPower - 8dBm
**       minus10dBm   - normalPower - 10dBm
**       minus12dBm   - normalPower - 12dBm
**       minus14dBm   - normalPower - 14dBm
**       minus16dBm   - normalPower - 16dBm
**       minus18dBm   - normalPower - 18dBm
**
**--------------------------------------------------------------------------*/
BYTE                /*RET The current powerlevel */
ZW_RFPowerLevelGet(
  void)            /* IN Nothing */
{
  ;
  return  0;
}

/**
  * \ingroup ZWCMD
  * Transmit data buffer to a list of Z-Wave Nodes (multicast frame).
  * \param[in] pNodeIDList    List of destination node ID's
  * \param[in] numberNodes    Number of Nodes
  * \param[in] pData          Data buffer pointer
  * \param[in] dataLength     Data buffer length
  * \param[in] txOptions      Transmit option flags
  * \param[in] completedFunc  Transmit completed call back function
  * \return FALSE if transmitter queue overflow
  */
BYTE ZW_SendDataMulti( BYTE *pNodeIDList, BYTE numberNodes, BYTE *pData, BYTE dataLength, BYTE txOptions, VOID_CALLBACKFUNC ( completedFunc ) ( auto BYTE txStatus ) )
{
    int i;
	BYTE byCompletedFunc;
    byCompletedFunc = ( completedFunc == NULL ? 0 : 0x03 );
	idx = 0;

    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = FUNC_ID_ZW_SEND_DATA_MULTI;
    numberNodes = ( numberNodes <= ZW_MAX_NODES ) ? numberNodes : ZW_MAX_NODES;
    buffer[ idx++ ] = numberNodes;

    for ( i = 0; i < numberNodes; i++ )
    {
        buffer[ idx++ ] = pNodeIDList[ i ];
    }

    buffer[ idx++ ] = dataLength;

    for ( i = 0; i < dataLength; i++ )
    {
        buffer[ idx++ ] = pData[ i ];
    }

    buffer[ idx++ ] = txOptions;
    buffer[ idx++ ] = byCompletedFunc;	// Func id for CompletedFunc
    buffer[ 0 ] = idx;	// length
    byLen = 0;
    cbFuncZWSendDataMulti = completedFunc;
    SendDataAndWaitForResponse( buffer, idx, FUNC_ID_ZW_SEND_DATA_MULTI, buffer, &byLen );

    return buffer[ IDX_DATA ];
}

void ioPoll( void )
{
  if (timerCount)
  {
    TimerCheckButtons();
    if (timerReceiveTimer)
    {
      timerReceiveTimer--;
      if (!timerReceiveTimer) ReceiveDataIndicateTimeOut();
    }
  }
  TimerAction();
}





BOOL ZW_GetRandomWord( BYTE *randomWord, BOOL bResetRadio)
{
    int i;
    idx = 0;
    /*  bLine | bRemoveBad | bRemoveNonReps | funcID */
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = 0x1c;
    buffer[ idx++ ] = 0x8;
    buffer[ 0 ] = idx;
    byLen = 0;
    SendDataAndWaitForResponse( buffer, idx, 0x1c, buffer, &byLen );

    for ( i = 0;i < 0x8;i++ )
    {
        randomWord[ i ] = buffer[ IDX_DATA + i +2];
    }
    return 0;
}

void ZW_GetBasic( BYTE *pData)
{
    int i;
    idx = 0;
    /*  bLine | bRemoveBad | bRemoveNonReps | funcID */
    buffer[ idx++ ] = 0;
    buffer[ idx++ ] = REQUEST;
    buffer[ idx++ ] = BASIC_GET;
    buffer[ 0 ] = idx;
    byLen = 0;
    SendDataAndWaitForResponse( buffer, idx, BASIC_GET, buffer, &byLen );

    for ( i = 0;i < 29;i++ )
    {
        pData[ i ] = buffer[ IDX_DATA + i +3];
    }
}

