/******************************* eeprom.c  *******************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2001
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 *
 * Description: This file contains functions for the external eeprom handling
 *
 * Author:   Henrik Holm
 *
 * Last Changed By:  $Author: psh $
 * Revision:         $Revision: 22659 $
 * Last Changed:     $Date: 2012-05-02 17:09:49 +0200 (Wed, 02 May 2012) $
 *
 ****************************************************************************/
#ifdef PATCH_ENABLE
/****************************************************************************/
/* Include assembly MACRO definitions for patch insertions.                 */
/*                                                                          */
/* Define $SET (MAKE_PATCHABLE_CODE) for making patchable code destinned    */
/* for OTP or ROM memory.                                                   */
/* Undefine $RESET (MAKE_PATCHABLE_CODE) for making code containing patch   */
/* code destinned for RAM or FLASH memory.                                  */
/****************************************************************************/
#if defined(WORK_PATCH) || defined(STARTER_PATCH)
/* Making code containing patch code destinned for development RAM memory.  */
#pragma asm
$RESET (MAKE_PATCHABLE_CODE)
$INCLUDE (ZW_patch.inc)
#pragma endasm
/* Rename CODE class to CODE_PATCH */
#pragma userclass (code = PATCH)
/* Rename CONST class to CONST_PATCH */
#pragma userclass (const = PATCH)
/* Rename XDATA class to XDATA_PATCH */
#pragma userclass (xdata = PATCH)
#else
/* Making patchable code destinned for OTP or ROM memory.                   */
#pragma asm
$SET (MAKE_PATCHABLE_CODE)
$INCLUDE (ZW_patch.inc)
#pragma endasm
#endif /* elsif defined(WORK_PATCH) || defined(STARTER_PATCH) */
#endif /* PATCH_ENABLE */

/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
#include <ZW_patch.h>
#include <ZW_controller_api.h>
#include <ZW_pindefs.h>
#include <ZW_uart_api.h>
#include <ZW_classcmd.h>
#include <ZW_nvm_addr.h>
#include <eeprom.h>

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/

/*Nodemask define*/
typedef struct _NODEMASK_S_
{
  BYTE  ID[MAX_NODES/8];
} NODEMASK;

typedef   NODEMASK  * NODEMASK_P;

/*Group table layout*/
typedef struct _GROUPTABLE_
{
  BYTE        currentDimLevel;      /* Last set Dim Level */
  NODEMASK    nodeMask;             /*Nodemask bit[n] = 1 indicates node n+1 is in the group*/
} GROUPTABLE;

typedef GROUPTABLE  * GROUPTABLE_P;

/*Define of EEPROM layout*/
typedef struct _EEPROM_
{
  GROUPTABLE    groupTable;
} EEPROM;

/*if any Application data is generated automaticly this offset can be changed to match the
  start of the manually defined EEPROM table*/
#define AUTOGENERATED_OFFSET  0

/* address of field q in a struct s of type m */
#define addressof(s,m,q)  (AUTOGENERATED_OFFSET + (WORD) offsetof(EEPROM,s) + offsetof(m,q))

/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/
/*RAM buffer for the nodemask array*/
PATCH_VARIABLE BYTE groupMask[MAX_NODES/8];

/****************************************************************************/
/*                              EXPORTED DATA                               */
/****************************************************************************/

/* Global used when getting nodeinformation from protocol */
extern NODEINFO nodeInfo;

/****************************************************************************/
/*                              PRIVATE FUNCTIONS                           */
/****************************************************************************/

/*============================   IsNodeInGroup   ======================
**    Function description
**      Check if a node is in the group
**    Side effects:
**
**--------------------------------------------------------------------------*/
BOOL
PATCH_FUNCTION_NAME(IsNodeInGroup)(
  BYTE bNodeID)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(IsNodeInGroup)
#pragma endasm
#endif
  return ZW_NODE_MASK_NODE_IN(&groupMask, bNodeID);
}

/*============================   AddNodeToGroup   ======================
**    Function description
**      Stores a node in the group.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(StoreNodeBitInGroup)(
  BYTE bNodeID, BYTE value)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StoreNodeBitInGroup)
#pragma endasm
#endif
  if(bNodeID)
  {
    if(value)
    {
      ZW_NODE_MASK_SET_BIT(&groupMask, bNodeID);
    }
    else
    {
      ZW_NODE_MASK_CLEAR_BIT(&groupMask, bNodeID);
    }
    /*We only save the relevant byte to extern eeprom*/
    ZW_MEM_PUT_BYTE(addressof(groupTable, GROUPTABLE, nodeMask)+((bNodeID-1) >> 3),groupMask[(bNodeID-1) >> 3]);
  }
}


/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/

/*============================   StoreGroupLevel   ======================
**    Function description
**      Stores the current dim level in External EEPROM
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(StoreGroupLevel)(
  BYTE level)           /*IN Current group dim level*/
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StoreGroupLevel)
#pragma endasm
#endif
  ZW_MEM_PUT_BYTE(addressof(groupTable, GROUPTABLE, currentDimLevel),level);
}

/*============================   GetGroupLevel   ======================
**    Function description
**      Gets the current dim level in External EEPROM
**    Side effects:
**
**--------------------------------------------------------------------------*/
BYTE
PATCH_FUNCTION_NAME(GetGroupLevel)(void)       /*RET Current group dim level*/
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(GetGroupLevel)
#pragma endasm
#endif
  return ZW_MEM_GET_BYTE(addressof(groupTable, GROUPTABLE, currentDimLevel));
}

/*============================   RemoveNodeFromGroup   ======================
**    Function description
**      Removes a node from a group.
**    Side effects:
**      updates the groupArray
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(RemoveNodeFromGroup)(
  BYTE bNodeID)             /*IN NodeID to remove*/
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(RemoveNodeFromGroup)
#pragma endasm
#endif
  StoreNodeBitInGroup(bNodeID, 0);
}

/*============================   AddNodeToGroup   ======================
**    Function description
**      Adds a node to a group.
**    Side effects:
**      updates the groupArray
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(AddNodeToGroup)(
  BYTE bNodeID)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(AddNodeToGroup)
#pragma endasm
#endif
  StoreNodeBitInGroup(bNodeID, 1);
}

/*============================   ClearGroupTable   ======================
**    Function description
**      Clears the group table and calls call back when done.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(ClearGroupTable)(
  void (*func)(void))
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(ClearGroupTable)
#pragma endasm
#endif
  ZW_NODE_MASK_CLEAR(&groupMask, sizeof(groupMask));
  ZW_MEM_PUT_BUFFER(addressof(groupTable, GROUPTABLE, nodeMask),&groupMask, sizeof(groupMask), func);
}

/*============================   UpdateRAMGroupList   ======================
**    Function description
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
BYTE                      /*RETURN number of nodes in the list*/
PATCH_FUNCTION_NAME(UpdateRAMGroupList)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(UpdateRAMGroupList)
#pragma endasm
#endif
  /*Get a fresh copy of the groupnodemask*/
  ZW_MEM_GET_BUFFER(addressof(groupTable, GROUPTABLE, nodeMask),&groupMask, sizeof(groupMask));
  return (ZW_NODE_MASK_BITS_IN(groupMask, sizeof(groupMask)));
}


/*============================   GetGroupListDimmers   ======================
**    Build a nodemask of nodes that has their generic type set to SWITCH_MULTILEVEL
**
**    Returns: Number of multilevel nodes found
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
BYTE                      /*RETURN number of nodes in the list*/
PATCH_FUNCTION_NAME(GetGroupListDimmers)(
  BYTE *pbList)
#ifdef PATCH_ENABLE
reentrant
#endif
{
  register BYTE i, bCount;

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(GetGroupListDimmers)
#pragma endasm
#endif
  bCount = 0;
  for (i = 1; i <= MAX_NODES; i++)
  {
    if (IsNodeInGroup(i))
    {
      ZW_GetNodeProtocolInfo(i,&nodeInfo);
      if (nodeInfo.nodeType.generic == GENERIC_TYPE_SWITCH_MULTILEVEL)
      {
        bCount++;
        continue;
      }
    }
    ZW_NodeMaskClearBit(pbList, i);
  }
  return bCount;
}
