/******************************* dev_ctrl.c  *******************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2001
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 *
 * Description: Main routines of the simple Development controller...
 *
 * Author:   Henrik Holm
 *
 * Last Changed By:  $Author: psh $
 * Revision:         $Revision: 22660 $
 * Last Changed:     $Date: 2012-05-02 17:21:51 +0200 (Wed, 02 May 2012) $
 *
 ****************************************************************************/

#if defined(WORK_PATCH) || defined(STARTER_PATCH)
#define ZW_DEBUG
#endif

#ifdef PATCH_ENABLE
/****************************************************************************/
/* Include assembly MACRO definitions for patch insertions.                 */
/*                                                                          */
/* Define $SET (MAKE_PATCHABLE_CODE) for making patchable code destinned    */
/* for OTP or ROM memory.                                                   */
/* Undefine $RESET (MAKE_PATCHABLE_CODE) for making code containing patch   */
/* code destinned for RAM or FLASH memory.                                  */
/****************************************************************************/
#if defined(WORK_PATCH) || defined(STARTER_PATCH)
/* Making code containing patch code destinned for development RAM memory.  */
#pragma asm
$RESET (MAKE_PATCHABLE_CODE)
$INCLUDE (ZW_patch.inc)
#pragma endasm
/* Rename CODE class to CODE_PATCH */
#pragma userclass (code = PATCH)
/* Rename CONST class to CONST_PATCH */
#pragma userclass (const = PATCH)
/* Rename XDATA class to XDATA_PATCH */
#pragma userclass (xdata = PATCH)
#else
/* Making patchable code destinned for OTP or ROM memory.                   */
#pragma asm
$SET (MAKE_PATCHABLE_CODE)
$INCLUDE (ZW_patch.inc)
#pragma endasm
#endif /* elsif defined(WORK_PATCH) || defined(STARTER_PATCH) */
#endif /* PATCH_ENABLE */

/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/

#include <config_app.h>
#include <ZW_patch.h>
#include <p_button.h>
#include <ZW_controller_api.h>
#include <ZW_uart_api.h>
#include <ZW_classcmd.h>
#include <eeprom.h>
#include <ZW_debug_api.h>
#include <ctrl_learn.h>
#include <dev_ctrl.h>

#ifdef DEBUG_HW
#include <INTRINS.H>
#endif
#include <ZW_TransportLayer.h>

#ifdef ZW_ISD51_DEBUG
#include "ISD51.h"
#endif

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/

/* mainState states */
#define GROUP_CONTROL_STATE       0x00
#define INCLUDE_NODE_STATE        0x01
#define REMOVE_NODE_STATE         0x02
#define NEW_CONTROLLER_STATE      0x03
#define RESET_CONTROLLER_STATE    0x04
#define ASSIGN_ROUTE_STATE        0x05
#define CHECK_BUTTON_STATE        0x06
#define EXECUTE_STATE             0x07
#define DIMMING_STATE             0x08
#define LEARN_STATE               0x09
#define PWR_UP_STATE              0x0A
#define ERROR_STATE               0x0B
#define WAIT_FOR_PB_RELEASE_STATE 0x0C
#define WAIT_FOR_PB_DOWN_STATE    0x0D
#define WAIT_FOR_PB_PRESS_STATE   0x0E
#define WAIT_FOR_PB_DEPRESS_STATE 0x0F
#define RETURN_TO_IDLE_STATE      0x10
/* timeouts */
#define BUSY_BLINK_RATE   20    /* (10ms ticks) Blinking rate during busy states */
#define ADD_TO_GROUP_RATE 10    /* (10ms ticks) Blinking rate during dimming and node/s added to group */
#define DIM_END_TIMEOUT   200   /* (10ms ticks) Wait before assuming that dim set is done */
#define RESET_TIMEOUT     200   /* (10ms ticks) Wait before resetting the controller */
#define LED0_ONTIME       50    /* (10ms ticks) Keep LED0 on for this time after command completed */
#define REPL_TIMEOUT      200		/* (10ms ticks) Replication timeout - wait for this time period before aborting replication */

/* Macros for LED access */
/* Note that LED1 connection on ZW020x/ZW030x requires a strap from RXD to LED1 pin */
/* on development controller board */
#define LED0_OFF PIN_ON(LED0)
#define LED1_OFF PIN_ON(LED1)
#define LED0_ON  PIN_OFF(LED0)
#define LED1_ON  PIN_OFF(LED1)

#define LED2_OFF PIN_ON(LED2)
#define LED2_ON  PIN_OFF(LED2)

/* Defines and macros used to update the groupState variable */
#define GROUP_ON_BIT        0x80
#define GROUP_DIR_BIT       0x40

/* Commands for the ControlGroup function */
#define TOGGLE    0x01
#define DIM_START 0x02
#define DIM_STOP  0x03
#define DIM_SET   0x04

#define GROUP_CTRL_ACTIVATION   IS_DOWN_PB0(glKey)
#define ADD_NODE_ACTIVATION     IS_DOWN_PB1(glKey)
#define RESET_NODE_ACTIVATION   IS_DOWN_PB2(glKey)

#define ASSIGN_ROUTE_ACTIVATION IS_DOWN_PB3(glKey)
#define ADD_CTRL_ACTIVATION     IS_DOWN_PB4(glKey)
#define RESET_ACTIVATION        IS_DOWN_PB0(glKey)&&IS_DOWN_PB4(glKey)

#define UPDATE_TIMEOUT 100      /* 100 * 10 ms */
#define MIN_DELAY  (BYTE)(60)   /* minimum delay for the route update to be started */
#define MAX_DELAY  (BYTE)(4*60)

/* Binary Switch definitions (They were defined in ZW_classcmd.h earlier) */
#define SWITCHED_ON                                     0xFF
#define SWITCHED_OFF                                    0x00

#ifdef APPL_PROD_TEST
/* Production test pin definition */
/* ADC3/ZEROX/GP_PWM/KP_OUT8 P3_7 pin now input */
#define SET_PRODUCTIONTEST_PIN  PIN_IN(ADC3, 0)
#define IN_PRODUCTIONTEST       (!PIN_GET(ADC3))
#endif /* APPL_PROD_TEST */


#define BUTTON_DEBOUNCE_COUNT   100
#ifdef JP
#define TEST_COUNT  6
#else
#define TEST_COUNT  4
#endif
/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/
PATCH_VARIABLE BYTE mainState;                /*State variable for the main loop*/
PATCH_VARIABLE BYTE glKey;                    /*PushButton variable*/

PATCH_VARIABLE BYTE lastLearnedNodeType       /*Store the last learned node type*/
#ifndef WORK_PATCH
 = 0
#endif
;
PATCH_VARIABLE BYTE lastLearnedNodeID
#ifndef WORK_PATCH
 = 0
#endif
;
/* Separate handles for all timers.. This number can be cut down if RAM */
/* is getting low */
PATCH_VARIABLE BYTE led0TimerHandle
#ifndef WORK_PATCH
 = 0
#endif
;
PATCH_VARIABLE BYTE led1TimerHandle
#ifndef WORK_PATCH
 = 0
#endif
;
PATCH_VARIABLE BYTE dimEndTimerHandle
#ifndef WORK_PATCH
 = 0
#endif
;
PATCH_VARIABLE BYTE resetTimerHandle
#ifndef WORK_PATCH
 = 0
#endif
;
PATCH_VARIABLE BYTE replicationTimerHandle
#ifndef WORK_PATCH
 = 0
#endif
;

/*Keeps track of the state of the group (ON/OFF and DIM direction)*/
PATCH_VARIABLE BYTE groupState;

/*Last set DIM level*/
PATCH_VARIABLE BYTE groupLevel;

/*Number of nodes in the group. 0 if no nodes*/
PATCH_VARIABLE BYTE nodesInGroup;

/*Index used to run through the groupNodeList*/
PATCH_VARIABLE BYTE nextNode
#ifndef WORK_PATCH
 = 0
#endif
;

/*Flag used to check if were waiting for a dim level report.*/
PATCH_VARIABLE BYTE receivedReport
#ifndef WORK_PATCH
 = TRUE
#endif
;

/*Flag that keeps track of the LED0 Blinker*/
PATCH_VARIABLE BYTE goON;

/*Contains the node IDs and node types used when assigning a route*/
PATCH_VARIABLE BYTE routingNode, routeToNode;

/*Buffer used to hold commands*/
PATCH_VARIABLE ZW_APPLICATION_TX_BUFFER txBuffer;

/*The command classes that this device supports besides the ones specified by
 its generic type*/
PATCH_VARIABLE BYTE supportedCmdClasses[2]
#ifndef WORK_PATCH
 = {COMMAND_CLASS_CONTROLLER_REPLICATION,COMMAND_CLASS_VERSION}
#endif
;

PATCH_VARIABLE BYTE  staticCtrlNodeID           /* hold the node id of the static update controller*/
#ifndef WORK_PATCH
 = 0x00
#endif
;
PATCH_VARIABLE BYTE timerHandle
#ifndef WORK_PATCH
 = 0xFF
#endif
;
PATCH_VARIABLE BYTE bTimerCount
#ifndef WORK_PATCH
 = 0
#endif
;
PATCH_VARIABLE BOOL receivingRemote
#ifndef WORK_PATCH
 = FALSE
#endif
;
PATCH_VARIABLE BOOL bStartUpdateTimer
#ifndef WORK_PATCH
 = FALSE
#endif
;

PATCH_VARIABLE BOOL txInProgress
#ifndef WORK_PATCH
 = FALSE
#endif
;

/* Group replication */
PATCH_VARIABLE BYTE bOtherController
#ifndef WORK_PATCH
 = 0
#endif
;

PATCH_VARIABLE BOOL productionTestFirstRun
#ifndef WORK_PATCH
 = TRUE
#endif
;
PATCH_VARIABLE BYTE bNWIIsStarted
#ifndef WORK_PATCH
 = FALSE
#endif
;
PATCH_VARIABLE BYTE bNWIStartup
#ifndef WORK_PATCH
 = FALSE
#endif
;

/*Global used when getting nodeinformation from protocol*/
PATCH_VARIABLE_STARTER NODEINFO nodeInfo;


PATCH_VARIABLE BYTE debounceCount
#ifndef WORK_PATCH
 = 0
#endif
;
PATCH_VARIABLE BOOL btPressed
#ifndef WORK_PATCH
 = FALSE
#endif
;
PATCH_VARIABLE BOOL btReleased
#ifndef WORK_PATCH
 = FALSE
#endif
;
PATCH_VARIABLE BYTE test_nr
#ifndef WORK_PATCH
 = 0
#endif
;
/****************************************************************************/
/*                              PRIVATE FUNCTIONS                           */
/****************************************************************************/

static void SetSUCDone(BYTE txStaus);
static void RequestUpdateTimeOut();

/*============================   GroupSend  =================================
**    Function description
**      Sends out multicast message to the group. If no nodes are in the group
**      it will call the callback func with TRANSMIT_COMPLETE_NO_ACK.
**    Side effects:
**      updates RAMGroupList
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(GroupSend)(
  BYTE *pData,                /*IN  Data buffer pointer           */
  BYTE dataLength,            /*IN  Data buffer length            */
  BYTE txOptions,             /*IN  Transmit option flags         */
  BYTE bCommand,
  void (code *func)(BYTE))    /*IN  Transmit completed call back function  */
#ifdef PATCH_ENABLE
reentrant
#endif
{
  /* TO#01645 fix  Make a group doesn't work correct. */
static BYTE TxGroupMask[MAX_NODES/8];

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(GroupSend)
#pragma endasm
#endif
  /* TO#1687 fix */
  if (bCommand == TOGGLE)
  {
    memcpy(TxGroupMask, groupMask, MAX_NODES/8);
  }
  else
  {
    memcpy(TxGroupMask, groupMask, MAX_NODES/8);
    nodesInGroup = GetGroupListDimmers(TxGroupMask);  /* Strip binary switches from TxGroupMask */
  }
  /* TO#1687 */

  if (nodesInGroup)
  {
    if (!ZW_SEND_DATA_MULTI(TxGroupMask, pData, dataLength, txOptions, func))
    {
      func(TRANSMIT_COMPLETE_NO_ACK);
    }
  }
  else
  {
    func(TRANSMIT_COMPLETE_NO_ACK);
  }
}


/*============================   LED0Blinker   ==============================
**    Function description
**      Handles the blinking of LED0
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(LED0Blinker)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(LED0Blinker)
#pragma endasm
#endif
  if (goON = !goON)
  {
    LED0_OFF;
  }
  else
  {
    LED0_ON;
  }
}


/*============================   StopLED0Blink   =============================
**    Function description
**      Stops the LED0Blinker timer and turns of LED0
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(StopLED0Blink)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StopLED0Blink)
#pragma endasm
#endif
  LED0_OFF;
  if (led0TimerHandle)
  {
    ZW_TIMER_CANCEL(led0TimerHandle);
  }
  led0TimerHandle = 0;
}


/*============================   StartLED0Blink   ===========================
**    Function description
**      (Re)starts the timer that runs LED0 blinking..
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(StartLED0Blink)(
  BYTE blinkRate)       /*IN blink rate in 10ms*/
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StartLED0Blink)
#pragma endasm
#endif
  StopLED0Blink();
  goON = TRUE;    /*Allways start with LED on*/
  led0TimerHandle = ZW_TIMER_START(LED0Blinker, blinkRate, TIMER_FOREVER);
}


/*============================   LED0_OFFTimeout   ======================
**    Function description
**      Timeout function that can be used to make a timed off on LED0
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(LED0_OFFTimeout)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(LED0_OFFTimeout)
#pragma endasm
#endif
  LED0_OFF;
}


/*============================   SetGroupDimLevel   ==========================
**    Function description
**      Sets the dim level of the nodes in the group to the last groupDimLevel
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(SetGroupDimLevel)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(SetGroupDimLevel)
#pragma endasm
#endif
  StopDimEndTimer();
  ControlGroup(DIM_SET,TxCompleteIgnoreAck);
}


/*============================   StopDimEndTimer   ======================
**    Function description
**      Stops the timeout function that is run when Dimming is ended..
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(StopDimEndTimer)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StopDimEndTimer)
#pragma endasm
#endif
  if (dimEndTimerHandle)
  {
    ZW_TIMER_CANCEL(dimEndTimerHandle);
  }
  dimEndTimerHandle = 0;
}


/*============================   StartDimEndTimer   ======================
**    Function description
**      Starts the timeout that is running when dim is ended
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(StartDimEndTimer)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StartDimEndTimer)
#pragma endasm
#endif
  StopDimEndTimer();
  dimEndTimerHandle = ZW_TIMER_START(SetGroupDimLevel, DIM_END_TIMEOUT, TIMER_ONE_TIME);
}


/*=========================   RemoveNodeCompleted   ========================
**    Function description
**        Callback function, used when removing switches from the network.
**          bStatus BYTE,     IN  Status of learn process
**          bSource BYTE,     IN  Node id of the node that send node info
**          BYTE* pCmd,       IN  Pointer to Node information
**          BYTE bLen));      IN  Node info length
**
**    Side effects: Shuts down RF receive mode when completed and removes the
**                  node from the group.
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(RemoveNodeCompleted)(
  LEARN_INFO *learnNodeInfo)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(RemoveNodeCompleted)
#pragma endasm
#endif
  ZW_DEBUG_SEND_BYTE('r');
  ZW_DEBUG_SEND_NUM(learnNodeInfo->bStatus);
  ZW_DEBUG_SEND_BYTE('.');
  ZW_DEBUG_SEND_NUM((*learnNodeInfo).bSource);

  if ((*learnNodeInfo).bStatus == REMOVE_NODE_STATUS_NODE_FOUND)
  {
    ZW_DEBUG_SEND_BYTE('P');
    StartLED0Blink(BUSY_BLINK_RATE);
    mainState = EXECUTE_STATE;
  }
  else if (((*learnNodeInfo).bStatus == REMOVE_NODE_STATUS_REMOVING_SLAVE)||
           ((*learnNodeInfo).bStatus == REMOVE_NODE_STATUS_REMOVING_CONTROLLER))
  {
    RemoveNodeFromGroup((*learnNodeInfo).bSource);
  }
  else if ((learnNodeInfo->bStatus == REMOVE_NODE_STATUS_DONE) ||
           (learnNodeInfo->bStatus == REMOVE_NODE_STATUS_FAILED) )
  {
    ZW_DEBUG_SEND_BYTE('D');

    ZW_REMOVE_NODE_FROM_NETWORK(REMOVE_NODE_STOP, NULL);

    mainState = WAIT_FOR_PB_RELEASE_STATE;
    ZW_TIMER_START(StopLED0Blink,LED0_ONTIME,TIMER_ONE_TIME);
    if (((*learnNodeInfo).bStatus == ADD_NODE_STATUS_FAILED))
    {
      LED1_ON;
    }
  }
}

/*============================   ConfigureSwitchComplete   ======================
**    Callback for when configuration has been sent to binary or
**		multilevel switch.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  <completion code>       */
PATCH_FUNCTION_NAME(ConfigureSwitchComplete)(
  BYTE bStatus)           /* Transmit status           */
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(ConfigureSwitchComplete)
#pragma endasm
#endif
	if (bStatus != TRANSMIT_COMPLETE_OK)
	{
		mainState = ERROR_STATE;
	}
	else
	{
		mainState = RETURN_TO_IDLE_STATE;
	}
}


/*=======================   AssociationAddToGroupComplete   ==================
**    Function description
**      Called when Association add to group has completed
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(AssociationAddToGroupComplete)(
  BYTE bStatus)             /*Transmit status*/
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(AssociationAddToGroupComplete)
#pragma endasm
#endif
  ZW_TIMER_START(StopLED0Blink, LED0_ONTIME, TIMER_ONE_TIME);
  ZW_DEBUG_SEND_BYTE('c');
  LED1_OFF;
  if (bStatus != TRANSMIT_COMPLETE_OK)
  {
    mainState = ERROR_STATE;
  }
  else
  {
    mainState = RETURN_TO_IDLE_STATE;
  }
}


/*============================   AssociationAddToGroup   =====================
**    Function description
**    Callback function called when Assign Return Route has finished. This
**    function sends an Association Set frame to the routing slave node
**    routingNode, which inserts the routeToNode node into association group 1.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                    /*RET Nothing */
PATCH_FUNCTION_NAME(AssociationAddToGroup)(
  BYTE bStatus)         /* IN Status of assign return route */
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(AssociationAddToGroup)
#pragma endasm
#endif
  if (bStatus == TRANSMIT_COMPLETE_OK)
  {
    ZW_DEBUG_SEND_BYTE('9');
    txBuffer.ZW_AssociationSet1byteFrame.cmdClass = COMMAND_CLASS_ASSOCIATION;
    txBuffer.ZW_AssociationSet1byteFrame.cmd = ASSOCIATION_SET;
    txBuffer.ZW_AssociationSet1byteFrame.groupingIdentifier = 1;
    txBuffer.ZW_AssociationSet1byteFrame.nodeId1 = routeToNode;
    if (!ZW_SEND_DATA(routingNode, (BYTE *)&txBuffer,
                      sizeof(ZW_ASSOCIATION_SET_1BYTE_FRAME),
                      TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE, AssociationAddToGroupComplete))
    {
      AssociationAddToGroupComplete(TRANSMIT_COMPLETE_NO_ACK);
    }
  }
  else
  {
    /* Assign Return Route failed somehow - skip Association Set and report */
    AssociationAddToGroupComplete(TRANSMIT_COMPLETE_NO_ACK);
  }
}


/*========================   AssignRouteNodeCompleted   ======================
**    Function description
**    Callback function for Assign route. This function fetches the node ids.
**
**          bStatus BYTE,     IN  Status of learn process
**          bSource BYTE,     IN  Node id of the node that send node info
**          BYTE* pCmd,       IN  Pointer to Node information
**          BYTE bLen));      IN  Node info length
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(AssignRouteNodeCompleted)(
  LEARN_INFO *learnNodeInfo)  /*IN Status*/
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(AssignRouteNodeCompleted)
#pragma endasm
#endif
  if((*learnNodeInfo).bLen)
  {
    /*Only store ids and info when bLen is not Zero*/
    lastLearnedNodeType = ((NODE_TYPE *)(*learnNodeInfo).pCmd)->generic;
    lastLearnedNodeID = (*learnNodeInfo).bSource;
  }
  if ((*learnNodeInfo).bStatus == ADD_NODE_STATUS_LEARN_READY)
  {
    /*If Button is pressed exit learnmode*/
    mainState = WAIT_FOR_PB_DOWN_STATE;
  }
  else if ((*learnNodeInfo).bStatus == ADD_NODE_STATUS_ADDING_SLAVE)
  {
    mainState = EXECUTE_STATE;
    if(!routeToNode)
    {
      /* This node ID is the device that calculated routes should be to */
      routeToNode = lastLearnedNodeID;
    }
    else if(!routingNode)
    {
      /* This node ID is the device that routes should be sent to*/
      routingNode = lastLearnedNodeID;
    }
    StartLED0Blink(BUSY_BLINK_RATE);
  }
  else if(((*learnNodeInfo).bStatus == ADD_NODE_STATUS_PROTOCOL_DONE))
  {
    /*Stop learnmode. Handle completed in Done*/
    ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP,AssignRouteNodeCompleted);
  }
  else if(((*learnNodeInfo).bStatus == ADD_NODE_STATUS_DONE))
  {
    /*If either is zero go back and look for new*/
    if (!routingNode)
    {
      StopLED0Blink();
      LED1_ON;
      ZW_ADD_NODE_TO_NETWORK(ADD_NODE_EXISTING,AssignRouteNodeCompleted);
    }
    else
    {
      /* This node is the node that should receive the routes */
      StartLED0Blink(BUSY_BLINK_RATE);
      ZW_DEBUG_SEND_BYTE('7');
      ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, NULL);

      if(ZW_ASSIGN_RETURN_ROUTE(routingNode, routeToNode, AssociationAddToGroup))
      {
        mainState = EXECUTE_STATE;
      }
      else
      {
        StopLED0Blink();
        mainState = ERROR_STATE;
      }
    }
  }
  else if((*learnNodeInfo).bStatus == ADD_NODE_STATUS_FAILED)
  {
    StopLED0Blink();
    mainState = ERROR_STATE;
  }
}

/*============================   ControlGroup   ======================
**    Function description
**      Executes a group command. This can be one of the following:
**        TOGGLE
**        DIM_START
**        DIM_STOP
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(ControlGroup)(
  BYTE bCommand,                  /* IN What to do*/
  void code *func(BYTE bStatus))  /* IN Callback function.*/
#ifdef PATCH_ENABLE
reentrant
#endif
{
  static BYTE bRouting, length;

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(ControlGroup)
#pragma endasm
#endif

  length = sizeof(ZW_BASIC_SET_FRAME);
  nodesInGroup = UpdateRAMGroupList();

  nextNode = 0;
  switch(bCommand)
  {
    case TOGGLE:
      /* We use Explore frame Route Resolution if all alse fails */
      bRouting = TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE;
      txBuffer.ZW_BasicSetFrame.cmdClass = COMMAND_CLASS_BASIC;
      txBuffer.ZW_BasicSetFrame.cmd = BASIC_SET;
      groupState ^= GROUP_ON_BIT;  /* Toggle */
      if (groupState & GROUP_ON_BIT)
      {
        txBuffer.ZW_BasicSetFrame.value = SWITCHED_ON;   /* Switch ON */
      }
      else
      {
        txBuffer.ZW_BasicSetFrame.value = SWITCHED_OFF;  /* Switch OFF */
      }
      length = sizeof(ZW_BASIC_SET_FRAME);
      break; /*TOGGLE*/

    case DIM_START:
      bRouting = TRANSMIT_OPTION_ACK;
      txBuffer.ZW_SwitchMultilevelStartLevelChangeFrame.cmdClass = COMMAND_CLASS_SWITCH_MULTILEVEL;
      txBuffer.ZW_SwitchMultilevelStartLevelChangeFrame.cmd = SWITCH_MULTILEVEL_START_LEVEL_CHANGE;
      /* Set dim direction in command and toggle direction bit in state */
      txBuffer.ZW_SwitchMultilevelStartLevelChangeFrame.level = groupState & GROUP_DIR_BIT;
      groupState ^= GROUP_DIR_BIT;
      txBuffer.ZW_SwitchMultilevelStartLevelChangeFrame.startLevel = groupLevel;
      length = sizeof(ZW_SWITCH_MULTILEVEL_START_LEVEL_CHANGE_FRAME);
    break; /*DIM_START*/

    case DIM_STOP:
      receivedReport = FALSE;
      bRouting = TRANSMIT_OPTION_ACK;
      txBuffer.ZW_SwitchMultilevelStopLevelChangeFrame.cmdClass = COMMAND_CLASS_SWITCH_MULTILEVEL;
      txBuffer.ZW_SwitchMultilevelStopLevelChangeFrame.cmd = SWITCH_MULTILEVEL_STOP_LEVEL_CHANGE;
      length = sizeof(ZW_SWITCH_MULTILEVEL_STOP_LEVEL_CHANGE_FRAME);
    break; /*DIM_STOP*/

    case DIM_SET:
      /* Do routing because we want all lamps to turn on/off */
      bRouting = TRANSMIT_OPTION_AUTO_ROUTE | TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE;
      txBuffer.ZW_SwitchMultilevelSetFrame.cmdClass = COMMAND_CLASS_SWITCH_MULTILEVEL;
      txBuffer.ZW_SwitchMultilevelSetFrame.cmd = SWITCH_MULTILEVEL_SET;
      txBuffer.ZW_SwitchMultilevelSetFrame.value = groupLevel;
      length = sizeof(ZW_SWITCH_MULTILEVEL_SET_FRAME);
    break;

    default:
    break;
  }
  GroupSend((BYTE *)&txBuffer, length, bRouting, bCommand, func);
}


/*============================   DimStopped   ===============================
**    Function description
**      Dim stop has been received. This function fetches a DIM level
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(DimStopped)(
  BYTE bStatus)
#ifdef PATCH_ENABLE
reentrant
#endif
{
  BYTE temp;

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(DimStopped)
#pragma endasm
#endif
  /*We do not care if the dim stop was succesfull or not. We want a level from a switch in the group*/
  temp = nextNode;
  ZW_DEBUG_SEND_BYTE(',');
  ZW_DEBUG_SEND_NUM(nodesInGroup);
  /* Any nodes at all? */
  if (!nodesInGroup)
  {
    mainState = ERROR_STATE;
    return;
  }

  /* Locate multilevel switches only */
  do
	{
	  temp = ZW_NodeMaskGetNextNode(temp, groupMask);
	  if (temp > 0)
	  {
      ZW_GET_NODE_STATE(temp,&nodeInfo);
    }
  } while ((temp > 0) && (nodeInfo.nodeType.generic != GENERIC_TYPE_SWITCH_MULTILEVEL));

  if (temp > 0)   /* If we have located next multilevel switch node */
  {
    txBuffer.ZW_SwitchMultilevelGetFrame.cmdClass = COMMAND_CLASS_SWITCH_MULTILEVEL;
    txBuffer.ZW_SwitchMultilevelGetFrame.cmd = SWITCH_MULTILEVEL_GET;
    if ((!nextNode) ||  /* first node In list */
        ((bStatus != TRANSMIT_COMPLETE_OK)))
    {
      nextNode = temp;
      ZW_DEBUG_SEND_NUM(nextNode);
      ZW_SEND_DATA(nextNode, (BYTE *)&txBuffer,
                   sizeof(ZW_SWITCH_MULTILEVEL_GET_FRAME), TRANSMIT_OPTION_ACK, DimStopped);
    }
    else
    {
      /* A node acked, or we timed out.. Wait for a response. */
      StartDimEndTimer();
    }
  }
  else
  {
    /* No node acked, or we timed out.. Wait for a response. */
    StartDimEndTimer();
  }
}


/*============================   TxCompleteCheckAck   ======================
**    Function description
**      This function checks transmit status and indicates if any
**      error occoured
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(TxCompleteCheckAck)(
  BYTE bStatus)             /*IN Transmit status*/
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(TxCompleteCheckAck)
#pragma endasm
#endif
  txInProgress = FALSE;

  ZW_TIMER_START(StopLED0Blink,LED0_ONTIME,TIMER_ONE_TIME);
  if(bStatus != TRANSMIT_COMPLETE_OK)
    mainState = ERROR_STATE;
  else
    mainState = RETURN_TO_IDLE_STATE;
}


/*============================   TxCompleteIgnoreAck   ======================
**    Function description
**      Callback function that ignores Transmit status and returns to idle state
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(TxCompleteIgnoreAck)(
  BYTE bStatus)             /*IN Transmit status*/
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(TxCompleteIgnoreAck)
#pragma endasm
#endif
  ZW_TIMER_START(StopLED0Blink,LED0_ONTIME,TIMER_ONE_TIME);
  mainState = RETURN_TO_IDLE_STATE;
}


/*============================   ToggleGroup   ==============================
**    Function description
**      Toggles the group on or off
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(ToggleGroup)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(ToggleGroup)
#pragma endasm
#endif
  /*Toggle state of the switches*/
  StartLED0Blink(BUSY_BLINK_RATE);
  ControlGroup(TOGGLE,TxCompleteCheckAck);
}


/*============================   DimStarted   ===============================
**    Function description
**        Call back called when Dim has been started. Ignores transmit status
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(DimStarted)(
  BYTE bStatus)       /*IN Transmit status*/
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(DimStarted)
#pragma endasm
#endif
  ZW_DEBUG_SEND_BYTE('s');
  mainState = DIMMING_STATE;
}


/*============================   ZW_ResetDone   ======================
**    Function description
**      Restarts from PWR_UP_STATE
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(ZW_ResetDone)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(ZW_ResetDone)
#pragma endasm
#endif
  mainState = PWR_UP_STATE;
  StopUpdateTimer();
  ZW_TIMER_START(StopLED0Blink,LED0_ONTIME,TIMER_ONE_TIME);
}


/*============================   ApplicationResetDone   ======================
**    Function description
**        Starts Protocol reset
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(ApplicationResetDone)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(ApplicationResetDone)
#pragma endasm
#endif
  ZW_SET_DEFAULT(ZW_ResetDone);
}


/*==================================   DoReset   ============================
**    Function description
**      Starts controller reset..
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(DoReset)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(DoReset)
#pragma endasm
#endif
    StartLED0Blink(BUSY_BLINK_RATE);
    LED1_ON;
    /*Make sure to exit any learnstate before reseting*/

    ZW_DEBUG_SEND_BYTE('8');
    ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, NULL);
    // ZW_SetLearnMode(FALSE, NULL);

    staticCtrlNodeID = DEFAULT_SUC_NODE_ID;
    ClearGroupTable(ApplicationResetDone);
    mainState = EXECUTE_STATE;
}


/*============================   StopResetTimer  ===========================
**    Function description
**      Stops the reset activate timeout
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(StopResetTimer)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StopResetTimer)
#pragma endasm
#endif
  if(resetTimerHandle)
    ZW_TIMER_CANCEL(resetTimerHandle);
}

/*============================   StartResetTimer   ==========================
**    Function description
**      Starts a timeout. When it runs out the controller will be reset.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(StartResetTimer)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StartResetTimer)
#pragma endasm
#endif
  StopResetTimer();
  resetTimerHandle = ZW_TIMER_START(DoReset,RESET_TIMEOUT,TIMER_ONE_TIME);
}


/*===========================   GetRandomNodeTime   ===========================
 *
 *  Function description
 *    Get a pseudo-random time
 *
 *  Side effects:
 *
 *--------------------------------------------------------------------------*/
static BYTE                    /*RET A pseudo random nodeID */
GetRandomTime( void ) /* IN Nothing */
{
  BYTE  delay;
  delay = ZW_RANDOM();
  if((delay > MAX_DELAY)||(delay < MIN_DELAY))
    delay = MAX_DELAY;
  return delay;
}

/*============================   StopUpdateTimer   ======================
**    Function description
**      Used to stop the request update timer.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(StopUpdateTimer)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StopUpdateTimer)
#pragma endasm
#endif
  if(timerHandle!=0xff)
  {
    ZW_TIMER_CANCEL(timerHandle);
    timerHandle = 0xff;
  }
}


/*============================   StartUpdateTimer   =========================
**    Function description
**      Starts a random timer that transmits update requests to the SUC
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(StartUpdateTimer)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StartUpdateTimer)
#pragma endasm
#endif
  StopUpdateTimer();
  timerHandle = ZW_TIMER_START(RequestUpdateTimeOut,UPDATE_TIMEOUT, TIMER_FOREVER);
  if (timerHandle != 0xFF)
  {
    bTimerCount = GetRandomTime();
    bStartUpdateTimer = FALSE;
  }
  else
  {
    bStartUpdateTimer = TRUE;
  }
}


/*============================   SetSucNodeDelay   =========================
**    Function description
**      Timer call back used to delay the setSUCNodeID function to give both
**       controllers (Tx and RX) to analyze the routing table.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(SetSucNodeDelay)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(SetSucNodeDelay)
#pragma endasm
#endif
  /* TO#1825 fix */
  if (!ZW_SET_SUC_NODEID(staticCtrlNodeID, TRUE, FALSE, ZW_SUC_FUNC_NODEID_SERVER, SetSUCDone))
  {
    SetSUCDone(ZW_SUC_SET_FAILED);
  }
}


/*============================   NewControllerAllDone   ======================
**    Function description
**        Called when replication is all done
**        BYTE bStatus,   IN  Status of learn process
**        BYTE bSource,  IN  Node id of the node that send node info
**        BYTE* pCmd,    IN  Pointer to Node information
**        BYTE bLen.     IN  Node info length

**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(NewControllerAllDone)(
  LEARN_INFO *learnNodeInfo)    /*IN Learn node info*/
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(NewControllerAllDone)
#pragma endasm
#endif
  ZW_DEBUG_SEND_BYTE('a');
  ZW_GET_NODE_STATE((*learnNodeInfo).bSource, &nodeInfo);
  staticCtrlNodeID = ZW_GET_SUC_NODEID();
/*#if 0*/  /* TO#01646 fix  Development can't set SUC/SIS. */
  if (nodeInfo.nodeType.basic == BASIC_TYPE_STATIC_CONTROLLER)
  /*If learnNodeInfo is not zero and equal the static ctrlID then assign it as SUC*/
  {
    if ((!staticCtrlNodeID) || ( staticCtrlNodeID == (*learnNodeInfo).bSource))
    {
      staticCtrlNodeID = (*learnNodeInfo).bSource;
      mainState = EXECUTE_STATE; /*Do not allow user to interrupt now*/
      if (ZW_TIMER_START(SetSucNodeDelay, 100, TIMER_ONE_TIME) == 0xFF)
      {
        SetSUCDone(ZW_SUC_SET_FAILED);
      }
      return;
    }
  }
/*#endif*/  /* TO#01646 fix  Development can't set SUC/SIS. */
  ZW_TIMER_START(StopLED0Blink, LED0_ONTIME, TIMER_ONE_TIME);
  mainState = WAIT_FOR_PB_RELEASE_STATE;
}



/*============================   NewControllerSendStarted   ==================
**    Function description
**        Called by the protocol as the replication progresses.
**        When NEW_CONTROLLER_LEARNED has been received the user can not abort
**        BYTE bStatus,   IN  Status of learn process
**        BYTE bSource,  IN  Node id of the node that send node info
**        BYTE* pCmd,    IN  Pointer to Node information
**        BYTE bLen.     IN  Node info length

**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(NewControllerSendStarted)(
  LEARN_INFO *learnNodeInfo)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(NewControllerSendStarted)
#pragma endasm
#endif
  ZW_DEBUG_SEND_BYTE('@');
  ZW_DEBUG_SEND_NUM((*learnNodeInfo).bStatus);
  ZW_DEBUG_SEND_BYTE('-');
  ZW_DEBUG_SEND_NUM((*learnNodeInfo).bSource);
  ZW_DEBUG_SEND_BYTE('-');
  ZW_DEBUG_SEND_NUM((*learnNodeInfo).bLen);
  if (learnNodeInfo->bStatus == ADD_NODE_STATUS_LEARN_READY)
  {
    /*We are ready to add a node*/
    StopLED0Blink();
    LED0_ON;
    mainState = LEARN_STATE;
    /* Used for group replication */
    bOtherController = 0;
  }
  else if ((learnNodeInfo->bStatus == ADD_NODE_STATUS_NODE_FOUND)||
           (learnNodeInfo->bStatus == ADD_NODE_STATUS_ADDING_CONTROLLER))
  {
    /* Store controller node id for group replication */
    if (learnNodeInfo->bStatus == ADD_NODE_STATUS_ADDING_CONTROLLER)
      bOtherController = learnNodeInfo->bSource;

    StartLED0Blink(BUSY_BLINK_RATE);
    mainState = EXECUTE_STATE;
  }
  else if (learnNodeInfo->bStatus == ADD_NODE_STATUS_PROTOCOL_DONE)
  {
    ZW_DEBUG_SEND_BYTE('1');


    /* If it was a controller we added then start group replication */
    if (bOtherController)
    {
      nextNode = 1;
      ReplicationSendGroup(TRANSMIT_COMPLETE_OK);
    }
    else  /* If not a controller then just stop */
      ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, NewControllerAllDone);

  }
  else if (learnNodeInfo->bStatus == ADD_NODE_STATUS_FAILED)
  {
    ZW_DEBUG_SEND_BYTE('2');
    /*We have added a new device to the controller*/
    ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, NULL);
    mainState = ERROR_STATE;
  }
  else if (learnNodeInfo->bStatus == ADD_NODE_STATUS_DONE)
  {
    NewControllerAllDone(learnNodeInfo);
  }
}



/*==========================   ApplicationControllerUpdate ==================
**    Function description
**      This function is called when a Static Update Controller
**      command have been received by the protocol.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME_STARTER(ApplicationControllerUpdate)(
  BYTE bStatus,             /*IN  Status of learn process*/
  BYTE bSource,             /*IN  Node id of the node that send node info*/
  BYTE* pCmd,               /*IN  Pointer to application node information*/
  BYTE bLen)                /*IN  Node info length*/
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationControllerUpdate)
#pragma endasm
#endif
  ZW_DEBUG_SEND_BYTE('?');
  ZW_DEBUG_SEND_NUM(bStatus);
  if (bLen != 0)
  {
    lastLearnedNodeType = ((NODE_TYPE *)pCmd)->generic;
    ZW_DEBUG_SEND_NUM(lastLearnedNodeType);
    ZW_DEBUG_SEND_NUM(bSource);
  }
  if (bStatus == UPDATE_STATE_NEW_ID_ASSIGNED)
  {
    /* TO#2009 Fix */
#if 0
    if ((lastLearnedNodeType == GENERIC_TYPE_SWITCH_BINARY) ||
        (lastLearnedNodeType == GENERIC_TYPE_SWITCH_MULTILEVEL))
    {
    /* For demonstration purpuses every switch learned from the SUC are
       added to the hotkey group*/
      AddNodeToGroup(bSource);
    }
    /*In any case we want to turn off learn mode..*/
    ZW_DEBUG_SEND_BYTE('n');
#endif
  }
  else if (bStatus == UPDATE_STATE_DELETE_DONE)
  {
    RemoveNodeFromGroup(bSource);
  }
  else if (bStatus == UPDATE_STATE_SUC_ID)
  {
    /* OK we got a SUC */
  }
  else if (bStatus == UPDATE_STATE_NODE_INFO_RECEIVED)
  {
    /* Only add to group if we are dimming */
    if (mainState == DIMMING_STATE)
    {
      /* TO#2136 fix - No reason for only including some specific nodetypes */
      /* into group - we differentiate at group activation */
#if 0
      /* Only store it in the group if it was a switch type we learned */
      if ((lastLearnedNodeType == GENERIC_TYPE_SWITCH_BINARY) ||
          (lastLearnedNodeType == GENERIC_TYPE_SWITCH_MULTILEVEL))
#endif
      {
        /* If it is already in the group then remove it */
        if (IsNodeInGroup(bSource))
        {
          RemoveNodeFromGroup(bSource);
        }
        else
        {
          AddNodeToGroup(bSource);
          StartLED0Blink(ADD_TO_GROUP_RATE);
        }
      }
    }
  }
}


/*============================   RxOffDelay   ================================
**    Function description
**      RX off is delayed a bit to give the other end a chance to retransmit if
**     it missed the ack on its frame to this controller.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(RxOffDelay)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(RxOffDelay)
#pragma endasm
#endif
  LED0_OFF;
  /*If we still have a SUC start the update requests*/
  if (staticCtrlNodeID)
  {
    bStartUpdateTimer = TRUE;
  }

  /*RX is switched off by CHECK_BUTTON_STATE*/
  if (mainState == EXECUTE_STATE)
  {
    LED1_OFF;
    mainState = RETURN_TO_IDLE_STATE;
  }
}


/*============================   SucUpdateDone   ======================
**    Function description
**    Called when a SUC update have completed.
**
**    The call back function parameter values can be:
**        ZW_SUC_UPDATE_DONE        The update process ended successfully
**        ZW_SUC_UPDATE_ABORT       The update process was aborted
**        ZW_SUC_UPDATE_WAIT        The SUC node is busy, try again later
**        ZW_SUC_UPDATE_DISABLED    The SUC functionality have been disabled
**        ZW_SUC_UPDATE_OVERFLOW    Too many changes to handle by automatic update
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(SucUpdateDone)(
  BYTE txStatus) /*IN callback status from protocol*/
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(SucUpdateDone)
#pragma endasm
#endif
  /*Regardless of status we get the SUC node ID again, it might have been changed by
  the update..*/
  staticCtrlNodeID = ZW_GET_SUC_NODEID();
  if((txStatus == ZW_SUC_UPDATE_OVERFLOW) ||
     (txStatus == ZW_SUC_UPDATE_DISABLED))
  {
    /*If state was disabled or overflow enter ERROR state so user know that a manual replication is needed*/
    mainState = ERROR_STATE;
  }

  if(ZW_TIMER_START(RxOffDelay,50,TIMER_ONE_TIME)==0xFF);
  {
    RxOffDelay();
  }
}


/*============================   RequestUpdateTimeOut   ======================
**    Function description
**      Timer based function which will request updates from a Static Update
**      controller if one is available
**    Side effects:
**      Disables user input while ongoing
**--------------------------------------------------------------------------*/
static void
RequestUpdateTimeOut(void)
{
  if (--bTimerCount == 0)
  {
    bTimerCount = GetRandomTime();
    if (mainState == CHECK_BUTTON_STATE)
    {
      StopUpdateTimer();
      ZW_SET_RX_MODE(TRUE);
      LED0_ON;
      mainState = EXECUTE_STATE;
      if (ZW_REQUEST_NETWORK_UPDATE(SucUpdateDone) == 0)
      {
        SucUpdateDone(ZW_SUC_UPDATE_DONE);
      }
    }
  }
}


/*================================   SetSUCDone  ============================
**    Function description
**      Called when a SUC NodeID have been set.
**    Side effects:
**
**--------------------------------------------------------------------------*/
static void
SetSUCDone(
  BYTE txStaus)
{
  /*txStaus == ZW_SUC_SET_SUCCEEDED if successful, but we handle failed the same way*/
  ZW_TIMER_START(StopLED0Blink,LED0_ONTIME, TIMER_ONE_TIME);
  mainState = RETURN_TO_IDLE_STATE;
  if (staticCtrlNodeID = ZW_GET_SUC_NODEID())
  {
    bStartUpdateTimer = TRUE;
  }
}


/*=====================   LearnCompleted  ======================
**    Function description
**        Called by the protocol as the learn process proceedes
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(LearnCompleted)(
  BYTE nodeID,                  /*IN The nodeID assigned*/
  BYTE bStatus)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(LearnCompleted)
#pragma endasm
#endif
  ZW_DEBUG_SEND_BYTE('&');
  ZW_DEBUG_SEND_NUM(bStatus);

  bNWIIsStarted = FALSE;

  if (bStatus == FALSE)
  {
    receivingRemote = FALSE;
    mainState = ERROR_STATE;
  }
  else if (bStatus == TRUE)
  {
    ZW_DEBUG_SEND_BYTE('k');
    /*All protocol data have been transmitted.
      Any application data should be sent before setting NEW_CTRL_STATE_STOP_OK*/
    receivingRemote = FALSE;
    staticCtrlNodeID = ZW_GET_SUC_NODEID();
    if (staticCtrlNodeID )
    {
      bStartUpdateTimer = TRUE;
    }
    ZW_TIMER_START(StopLED0Blink,LED0_ONTIME, TIMER_ONE_TIME);
    /* Update group list in case a group replication was done */
    mainState = WAIT_FOR_PB_RELEASE_STATE;
  }
}

/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/

/*==============================   ApplicationInitHW  =======================
**    Non Z-Wave hardware initialization
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/

BYTE                   /*RET  TRUE       */
PATCH_FUNCTION_NAME_STARTER(ApplicationInitHW)(
  BYTE bWakeupReason)  /*IN Nothing */
#ifdef PATCH_ENABLE
reentrant
#endif
{
  /* hardware initialization
     If hardware platform changes these needs to be looked over again.
     This can be solved by use the macros created for pin access
     (PIN_OUT, PIN_IN etc.) and defines from ZW_GRCdefs.h.*/
  BYTE x;

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationInitHW)
#pragma endasm
#endif

#ifdef ZW_ISD51_DEBUG
  ZW_UART1_init(1152, TRUE, TRUE);
#endif
  /*Setup the relevant IO settings.*/
  /*Set up push buttons*/
  PIN_IN(PB0, 1);
  PIN_OFF(PB0);
  PIN_IN(PB1, 1);
  PIN_OFF(PB1);
  PIN_IN(PB2, 1);
  PIN_OFF(PB2);
  PIN_IN(PB3, 1);
  PIN_OFF(PB3);
#ifdef ZW040x
  PIN_IN(PB4, 1);
  PIN_OFF(PB4);
#else
#ifndef ZW_DEBUG_PROTOCOL
#ifndef ZW_DEBUG
  PIN_IN(PB4, 1);
  PIN_OFF(PB4);
#endif
#endif
#endif /*ZW040x*/
  /*Set up LEDs*/
  PIN_OUT(LED0);
  PIN_OUT(LED1);
  LED0_OFF;
  LED1_OFF;

#ifdef APPL_PROD_TEST
  SET_PRODUCTIONTEST_PIN;
  for (x = 0; x < 10; x++) ;  /* Short delay in case pin is floating... */
  if (IN_PRODUCTIONTEST) /* Anyone forcing it into productiontest mode ? */
  {
    return(FALSE);  /* Enter production test mode */
  }
#endif /* APPL_PROD_TEST */
  /* If production test is wanted, check for conditions and return FALSE from here */
  return(TRUE);
}


/*===========================   ApplicationInitSW   =========================
**    Initialization of the Application Software
**
**
**--------------------------------------------------------------------------*/
BYTE                   /*RET  TRUE       */
PATCH_FUNCTION_NAME_STARTER(ApplicationInitSW)(void)  /*IN Nothing */
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationInitSW)
#pragma endasm
#endif
 /*Start keypad timer. No need to save return value. We wont stop it ever again*/
  ZW_TIMER_START(TimerCheckButtons, 1, TIMER_FOREVER);
  mainState = PWR_UP_STATE;
/* Do not reinitialize the UART if already initialized for ISD51 in ApplicationInitHW() */
#ifndef ZW_ISD51_DEBUG
#ifndef ZW_DEBUG_PROTOCOL
  ZW_DEBUG_INIT(1152);
#endif
#endif

  staticCtrlNodeID = ZW_GET_SUC_NODEID();
  ZW_DEBUG_SEND_NL();
  ZW_DEBUG_SEND_BYTE('D');
  ZW_DEBUG_SEND_BYTE('e');
  ZW_DEBUG_SEND_BYTE('v');
  ZW_DEBUG_SEND_BYTE(' ');
  if (staticCtrlNodeID)
  {
    bStartUpdateTimer = TRUE;
  }
  ZW_SET_RX_MODE(TRUE);
#if 0
  /* TO#2071 fix - Do not set RoutingMAX to 2 - must be library default */
  /* Set MAX number of route tries before resorting to Explore Frame Route Resolution */
  ZW_SetRoutingMAX(2);
#endif
  /* Set Nwtwork wide inclusion active if we dont have aa node ID */

  if (ZW_IsPrimaryCtrl() && (ZW_GetControllerCapabilities() & NO_NODES_INCUDED))
    bNWIStartup = TRUE;
  else
    bNWIStartup = FALSE;

  return(TRUE);
}

/*============================   ApplicationTestPoll   ======================
**    Function description
**      This function is called when the controller enters test mode.
**      It can be extended with the functionalities needed to verify the hardware
**      In this application it is kept very simple.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME_STARTER(ApplicationTestPoll)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationTestPoll)
#pragma endasm
#endif

#ifdef APPL_PROD_TEST

  /* Initialize RF enabling frame receive */
  if(productionTestFirstRun)
  {
    PIN_OUT(LED0);
    PIN_OUT(LED1);
    PIN_OUT(LED2);
    LED0_OFF;
    LED1_OFF;
    LED2_OFF;
    ZW_SET_RX_MODE(TRUE);
    productionTestFirstRun = FALSE;
  }
  /* If button is pressed enter constant carrier indefinitely */
  if (!btPressed)
  {
    if (PIN_GET(Button) == 0)
    {
      if (++debounceCount >= BUTTON_DEBOUNCE_COUNT)
      {
        btPressed = TRUE;
        debounceCount = 0;
      }
    }
    else
    {
      debounceCount = 0;
    }
  }
  else
  {
    if (PIN_GET(Button))
    {
      if (++debounceCount >= BUTTON_DEBOUNCE_COUNT)
      {
        btReleased = TRUE;
        debounceCount = 0;
      }
    }
    else
    {
      debounceCount = 0;
    }
  }
  if (btReleased)
  {
    btReleased = FALSE;
    btPressed = FALSE;
    switch(test_nr)
    {
      case 0:
        ZW_SendConst(1, 0, ZW_RF_TEST_SIGNAL_CARRIER);
        LED0_ON;
        LED1_OFF
        LED2_OFF;
      break;
      case 1:
        ZW_SendConst(1, 0, ZW_RF_TEST_SIGNAL_CARRIER_MODULATED);
        LED0_ON;
        LED1_OFF
        LED2_OFF;
      break;
      case 2:
        ZW_SendConst(1, 1, ZW_RF_TEST_SIGNAL_CARRIER);
        LED0_OFF;
        LED1_ON
        LED2_OFF;
      break;
      case 3:
        ZW_SendConst(1, 1, ZW_RF_TEST_SIGNAL_CARRIER_MODULATED);
        LED0_OFF;
        LED1_ON
        LED2_OFF;
      break;
      case 4:
        ZW_SendConst(1, 2, ZW_RF_TEST_SIGNAL_CARRIER);
        LED0_OFF;
        LED1_OFF;
        LED2_ON;
      break;
      case 5:
        ZW_SendConst(1, 2, ZW_RF_TEST_SIGNAL_CARRIER_MODULATED);
        LED0_OFF;
        LED1_OFF;
        LED2_ON;
      break;

    }
    if (++test_nr >= TEST_COUNT )
      test_nr = 0;
  }
#endif /* APPL_PROD_TEST */
}


/*===========================  ApplicationPoll ===============================
**    Function description
**      Main poll routine. This routine should be registered as the application
**      poll routine.
**      This routine handles menu navigation and executes any functions
**      registered as internalfunctions.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME_STARTER(ApplicationPoll)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationPoll)
#pragma endasm
#endif

#ifdef ZW_ISD51_DEBUG         /* init ISD51 only when the uVision2 Debugger tries to connect */
  ISDcheck();                 /* initialize uVision2 Debugger and continue program run */
#endif
  switch (mainState)
  {
    case PWR_UP_STATE:
      nodesInGroup = UpdateRAMGroupList();
      groupLevel = GetGroupLevel();
      if (!ButtonPressed())              /*Stay here until no key is pressed*/
      {
        mainState = RETURN_TO_IDLE_STATE;
      }
      LED0_OFF;
      LED1_OFF;

      if (bNWIStartup)
      {
        ZW_SET_RX_MODE(TRUE);
        bNWIIsStarted = TRUE;
        StartLearnModeNow(ZW_SET_LEARN_MODE_NWI);
        LED0_ON;
      }

      break;

    case RETURN_TO_IDLE_STATE:
      // ZW_SET_RX_MODE(FALSE);
      mainState = CHECK_BUTTON_STATE;
      /*Drop trough to next state*/

    case CHECK_BUTTON_STATE:

      /* No RF needed when were in this state. */

      if (bNWIStartup)
      {
        if (learnInProgress)
        {
          bNWIStartup = FALSE;
          StartLED0Blink(BUSY_BLINK_RATE);
          mainState = EXECUTE_STATE; /*The user should no longer be able to exit learn mode*/
        }
      }
      if (glKey = ButtonPressed())
      {

        if (bNWIIsStarted)
        {
          StopLearnModeNow();
          bNWIIsStarted = FALSE;
        }
        else
        {
          LED1_OFF;
        }

        /* No RF needed when were in this state. */
        // ZW_SET_RX_MODE(TRUE);                 /*Power up RF.*/
        if (GROUP_CTRL_ACTIVATION)
        {
          mainState = GROUP_CONTROL_STATE;    /*group Toggle, dim and add node to group*/
        }
        if (ADD_NODE_ACTIVATION)
        {
          mainState = INCLUDE_NODE_STATE;     /*Add a new node to network*/
        }
        if (RESET_NODE_ACTIVATION)
        {
          mainState = REMOVE_NODE_STATE;      /*Remove Node from network (and group)*/
        }
        if (ASSIGN_ROUTE_ACTIVATION)
        {
          mainState = ASSIGN_ROUTE_STATE;     /*Assign route from node to node*/
        }
        if (ADD_CTRL_ACTIVATION)
        {
          mainState = NEW_CONTROLLER_STATE;   /*Add a slave controller to the network*/
        }
        if (RESET_ACTIVATION)
        {
          StartResetTimer();
          mainState = RESET_CONTROLLER_STATE; /*Reset this controller*/
        }
      }
      else
      {
        if (bStartUpdateTimer)
        {
          if (!ZW_PRIMARYCTRL() ||
              ((ZW_GET_CONTROLLER_CAPABILITIES() & CONTROLLER_NODEID_SERVER_PRESENT) != 0))
          {
            StartUpdateTimer();
          }
          else
          {
            bStartUpdateTimer = FALSE;
          }
        }
      }
      break;

    case GROUP_CONTROL_STATE:
      /*User hit the Group control button, handle it*/
      glKey = ButtonPressed();
      if (RESET_ACTIVATION)
      {
        StartResetTimer();
        mainState = RESET_CONTROLLER_STATE; /*PB4 is down too! We are in the wrong state*/
        break;
      }
      if (!glKey)
      {
        /*Key was released before we entered dimming state*/
        LED0_ON;
        mainState = EXECUTE_STATE;
        ToggleGroup();
        txInProgress = TRUE;
      }
      else if(IS_PB_HELD(glKey))
      {
        /*If the key is being held. Start the dimming routine*/
        mainState = EXECUTE_STATE;
        LED0_ON;
        ControlGroup(DIM_START, DimStarted);
      }
      break; /*GROUP_CONTROL_STATE*/

    case INCLUDE_NODE_STATE:
      /* User hit the Include control button, handle it */
      glKey = ButtonPressed();
      if (!glKey)
      {
        mainState = EXECUTE_STATE;
        LED0_ON;
        /* User released before we entered Include state - send nodeinformation */
        ZW_SEND_NODE_INFO(NODE_BROADCAST, 0, TxCompleteIgnoreAck);
      }
      else if (IS_PB_HELD(glKey))
      {
        /* User wants to include a new slave node to the network. Handle it */
        StartLED0Blink(BUSY_BLINK_RATE);
        mainState = LEARN_STATE;
        /* Add nodes with Normal Power */
        /* Accept networkwide inclusion requests */
        ZW_ADD_NODE_TO_NETWORK((ADD_NODE_ANY | ADD_NODE_OPTION_HIGH_POWER | ADD_NODE_OPTION_NETWORK_WIDE), NewControllerSendStarted);
        ZW_DEBUG_SEND_BYTE('*');
      }
      break; /*INCLUDE_NODE_STATE*/

    case LEARN_STATE:
      /* Internal state. Handling of learn state */
      if (!(ButtonPressed() & PB_MASK))
      {
        ZW_TIMER_START(StopLED0Blink,LED0_ONTIME,TIMER_ONE_TIME);
        mainState = WAIT_FOR_PB_RELEASE_STATE;
      }
      break; /*LEARN_STATE*/

    case REMOVE_NODE_STATE:
      LED0_ON;
      /* Remove nodes with Normal Power */
      /* Accept networkwide exclusion requests */
      ZW_REMOVE_NODE_FROM_NETWORK((REMOVE_NODE_ANY | ADD_NODE_OPTION_HIGH_POWER | ADD_NODE_OPTION_NETWORK_WIDE), RemoveNodeCompleted);
      //ZW_REMOVE_NODE_FROM_NETWORK(REMOVE_NODE_ANY, RemoveNodeCompleted);
      ZW_DEBUG_SEND_BYTE('_');
      mainState = LEARN_STATE;
      break; /*REMOVE_NODE_STATE*/

    case ASSIGN_ROUTE_STATE:
      /*Get nodeID of the device the routing device should receive a route to*/
      if (!ButtonPressed())
      {
        /*When Button released do it*/
        ZW_SET_RX_MODE(TRUE);
        LED0_ON;
        mainState = EXECUTE_STATE;
        routeToNode = 0;
        routingNode = 0;

        ZW_ADD_NODE_TO_NETWORK(ADD_NODE_EXISTING, AssignRouteNodeCompleted);
        mainState = WAIT_FOR_PB_DOWN_STATE;
      }
      break; /*ASSIGN_ROUTE_STATE*/

    case NEW_CONTROLLER_STATE:
      glKey = ButtonPressed();
      if (!glKey)
      {
        /*Key was released, then start replication transmit*/
        if (ZW_PRIMARYCTRL())
        {
          mainState = ERROR_STATE;
          break;
          mainState = WAIT_FOR_PB_PRESS_STATE;
        }
        else
        {
          mainState = ERROR_STATE;
        }
      }
      else if(IS_PB_HELD(glKey))
      {
        /* If the key is being held. Start the New controller receive */
        LED0_ON;
        StopUpdateTimer();
        receivingRemote = TRUE;
        ClearGroupTable(NULL);
        StartLearnModeNow(ZW_SET_LEARN_MODE_CLASSIC);
        //ZW_SetLearnMode(TRUE, NewControllerReceiveStarted);
        mainState = WAIT_FOR_PB_RELEASE_STATE;
      }
      break; /*NEW_CONTROLLER_STATE*/

    case RESET_CONTROLLER_STATE:
      if (!ButtonPressed())
      {
        /*If button is released do not reset!!*/
        StopResetTimer();
        mainState = RETURN_TO_IDLE_STATE;
      }

      break; /*RESET_CONTROLLER_STATE*/

    /*****************************************************************/
    /*Sub states. These are not executed directly from a button press*/
    /*****************************************************************/
    case DIMMING_STATE: /* Wait for key to be released*/
      if (!ButtonPressed())
      {
        ZW_DEBUG_SEND_BYTE('d');
        StartLED0Blink(BUSY_BLINK_RATE);
        mainState = EXECUTE_STATE;

        ControlGroup(DIM_STOP,DimStopped);
      }
      if (RESET_ACTIVATION)
      {
        StartResetTimer();
        mainState = RESET_CONTROLLER_STATE; /*Reset this controller*/
      }
      break; /*DIMMING_STATE*/

    case ERROR_STATE: /*Last operation failed..*/
      LED1_ON;
      StopLED0Blink();
      mainState = WAIT_FOR_PB_RELEASE_STATE;
      break;

    case WAIT_FOR_PB_DOWN_STATE:
      if (ButtonPressed())
      {
        ZW_DEBUG_SEND_BYTE('s');
        mainState = RETURN_TO_IDLE_STATE;
        StopLED0Blink();
        LED1_OFF;
      }
      break; /*WAIT_FOR_PB_DOWN_STATE*/

    case WAIT_FOR_PB_RELEASE_STATE:
      /*Wait until no key is pressed, then return to idle.*/
      if (!ButtonPressed())
      {
        LED0_OFF;
        ZW_DEBUG_SEND_BYTE('3');
        ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, NULL);
        //ZW_SetLearnMode(FALSE, NULL);
        mainState = RETURN_TO_IDLE_STATE;
        ZW_DEBUG_SEND_BYTE('q');
      }
      if (RESET_ACTIVATION)
      {
        StartResetTimer();
        mainState = RESET_CONTROLLER_STATE; /*Reset this controller*/
      }

      if (receivingRemote)
      {
        /* We are still trying to include/update this controller */
        ReArmLearnModeTimeout();
      }

      /* Start busy blink if learn is in progress */
      if (learnInProgress)
      {
        StartLED0Blink(BUSY_BLINK_RATE);
        mainState = EXECUTE_STATE; /*The user should no longer be able to exit learn mode*/
      }


      break;

    case WAIT_FOR_PB_PRESS_STATE:
      if (ButtonPressed())
      {
        ZW_DEBUG_SEND_BYTE('4');
        ZW_ADD_NODE_TO_NETWORK(ADD_NODE_STOP, NULL);
        //ZW_SetLearnMode(FALSE, NULL);

        mainState = WAIT_FOR_PB_DEPRESS_STATE;
      }
      break;

    case WAIT_FOR_PB_DEPRESS_STATE:
      if (!ButtonPressed())
      {
        LED0_OFF;
        mainState = RETURN_TO_IDLE_STATE;
      }
      break;

    case EXECUTE_STATE: /* We are executing a command. State changes should be done by callbacks*/
      if (txInProgress)
      {
        glKey = ButtonPressed();

        if (GROUP_CTRL_ACTIVATION)
        {
          ZW_SendDataAbort();
          txInProgress = FALSE;
        }
      }
      break;

    default: /*Something went horribly wrong.. Return to idle*/
      mainState = RETURN_TO_IDLE_STATE;
      break;
  }
}


/*========================  ApplicationCommandHandler  ======================
**    Function description
**        This callback function is called by the Z-Wave system when a
**        RF message that should be handled by the application is received
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME_STARTER(ApplicationCommandHandler)(
  BYTE  rxStatus,                 /* IN Frame header info */
#if defined(ZW_CONTROLLER) && !defined(ZW_CONTROLLER_STATIC) && !defined(ZW_CONTROLLER_BRIDGE)
  /* TO#1692 */
  BYTE  destNode,                 /* IN  Frame destination ID, only valid when frame is not Multicast */
#endif
  BYTE  sourceNode,               /* IN Command sender Node ID */
  ZW_APPLICATION_TX_BUFFER *pCmd, /* IN Payload from the received frame,   */
                                  /*    the command is the very first byte */
  BYTE cmdLength)                 /* IN Number of command bytes including the command */
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationCommandHandler)
#pragma endasm
#endif
  switch(pCmd->ZW_Common.cmdClass)
  {
    case COMMAND_CLASS_SWITCH_MULTILEVEL:
      if (pCmd->ZW_Common.cmd == SWITCH_MULTILEVEL_REPORT)
      {
        groupLevel = pCmd->ZW_SwitchMultilevelReportFrame.value;
        StoreGroupLevel(groupLevel);
        StopDimEndTimer();
        if (!receivedReport)
        {
          SetGroupDimLevel();
        }
        receivedReport = TRUE;
      }
      break;

    case COMMAND_CLASS_SWITCH_BINARY:
      break;

    case COMMAND_CLASS_BASIC:
      break;

    case COMMAND_CLASS_CONTROLLER_REPLICATION:
      ZW_DEBUG_SEND_BYTE('R');
      /* Handle replication frames */
      if (receivingRemote)
      {
        /* We are only interested in group replication here */
        /* but all other replication can be handled in the same way. */
        /* In this application we ignore everything else and just */
        /* send the command complete reply */
        if (pCmd->ZW_Common.cmd == CTRL_REPLICATION_TRANSFER_GROUP)
        {
          /* We are only interested in group 1 */
          if (pCmd->ZW_CtrlReplicationTransferGroupFrame.groupId == 1)
          {
            /* Add the node ID to the group */
            AddNodeToGroup(pCmd->ZW_CtrlReplicationTransferGroupFrame.nodeId);
          }
        }
        /* Send command complete to other controller */
        /* NOTE: This call should be done when ready to receive */
        /* the next replication frame */
        ZW_REPLICATION_COMMAND_COMPLETE();
      }
      break;

    case COMMAND_CLASS_VERSION:
      if (pCmd->ZW_Common.cmd == VERSION_GET)
      {
        txBuffer.ZW_VersionReportFrame.cmdClass = COMMAND_CLASS_VERSION;
        txBuffer.ZW_VersionReportFrame.cmd = VERSION_REPORT;
        txBuffer.ZW_VersionReportFrame.zWaveLibraryType = ZW_TYPE_LIBRARY();
        txBuffer.ZW_VersionReportFrame.zWaveProtocolVersion = ZW_VERSION_MAJOR;
        txBuffer.ZW_VersionReportFrame.zWaveProtocolSubVersion = ZW_VERSION_MINOR;
        txBuffer.ZW_VersionReportFrame.applicationVersion = APP_VERSION;
        txBuffer.ZW_VersionReportFrame.applicationSubVersion = APP_REVISION;
        ZW_SEND_DATA(sourceNode, (BYTE *)&txBuffer, sizeof(txBuffer.ZW_VersionReportFrame),
          TRANSMIT_OPTION_AUTO_ROUTE|TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE, NULL);
      }
      break;

    default:
      /*Ignore any unknown commands*/
      break;
  }
}


/*======================ApplicationNodeInformation   =========================
**    Request Node information and current status
**    Called by the the Z-Wave application layer before transmitting a
**    "Node Information" frame.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void                        /*RET Nothing */
PATCH_FUNCTION_NAME_STARTER(ApplicationNodeInformation)(
  BYTE   *deviceOptionsMask,         /*OUT Bitmask with application options     */
  APPL_NODE_TYPE *nodeType, /*OUT  Device type                        */
  BYTE  **nodeParm,         /*OUT  Device parameter buffer pointer    */
  BYTE  *parmLength )       /*OUT  Number of Device parameter bytes   */
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationNodeInformation)
#pragma endasm
#endif
  /* this is a NON listening node and it supports optional CommandClasses*/
  *deviceOptionsMask = APPLICATION_NODEINFO_NOT_LISTENING|APPLICATION_NODEINFO_OPTIONAL_FUNCTIONALITY;
  nodeType->generic = GENERIC_TYPE_GENERIC_CONTROLLER; /* Generic Device Type */
  nodeType->specific = SPECIFIC_TYPE_PORTABLE_REMOTE_CONTROLLER;
  *nodeParm = supportedCmdClasses;
  *parmLength = sizeof(supportedCmdClasses);
}


/*============================   StopReplicationTimer   ======================
**    This function stops the replication timer
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  <completion code>       */
PATCH_FUNCTION_NAME(StopReplicationTimer)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StopReplicationTimer)
#pragma endasm
#endif
	if (replicationTimerHandle)
		ZW_TIMER_CANCEL(replicationTimerHandle);
}


/*============================   AbortReplication   ======================
**    This callback function is used to abort replication when the
**		replication timer times out. Replication is aborted if no
**		TRANSMISSION_COMPLETE is received
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  <completion code>       */
PATCH_FUNCTION_NAME(AbortReplication)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(AbortReplication)
#pragma endasm
#endif
	StopReplicationTimer();
	ReplicationSendGroup(TRANSMIT_COMPLETE_FAIL);
}


/*============================   StartReplicationTimer   ======================
**    This function starts the replication timer used when
**		detecting replication attempts that fails.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  <completion code>       */
PATCH_FUNCTION_NAME(StartReplicationTimer)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StartReplicationTimer)
#pragma endasm
#endif
	replicationTimerHandle = ZW_TIMER_START(AbortReplication,REPL_TIMEOUT,TIMER_ONE_TIME);
	if (replicationTimerHandle == 0xFF)
	{
		// Should do something about this
	}
}


/*======================== ReplicationSendGroup ===========================
**    This functions handles sending group replication information as part of the
**    Controller replication process
**
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(ReplicationSendGroup)(
  BYTE bStatus)
#ifdef PATCH_ENABLE
reentrant
#endif
{
  BYTE bReplicationSuccess;
  register BYTE bTempVar;

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(ReplicationSendGroup)
#pragma endasm
#endif
  bReplicationSuccess = ADD_NODE_STOP;
	StopReplicationTimer();
  /* Transmit failed stop replication */
  if (bStatus != TRANSMIT_COMPLETE_OK)
  {
    nextNode = MAX_NODES;
    bReplicationSuccess = ADD_NODE_STOP_FAILED;
  }

  /* Find the next node that belongs to the group */
  for (; nextNode < MAX_NODES; nextNode++)
  {
    if (IsNodeInGroup(nextNode))
    {
    	ZW_DEBUG_SEND_BYTE('4');
      break;
    }
  }

  /* Check if we are finished with transferring the group */
  if (nextNode == MAX_NODES)
  {
    /* Here we could transfer a group name, start the next group or start
       transferring scenes, but we dont have any of that in this controller so
       we will just stop */
    ZW_ADD_NODE_TO_NETWORK(bReplicationSuccess, NewControllerAllDone);
    /* TO#1986 fixed, bReplicationSuccess would always be FAIL after the first failure */
    bReplicationSuccess = ADD_NODE_STOP;
    return;
  }
  /* Send next node ID to other controller */
  txBuffer.ZW_CtrlReplicationTransferGroupFrame.cmdClass = COMMAND_CLASS_CONTROLLER_REPLICATION;
  txBuffer.ZW_CtrlReplicationTransferGroupFrame.cmd      = CTRL_REPLICATION_TRANSFER_GROUP;
  txBuffer.ZW_CtrlReplicationTransferGroupFrame.groupId  = 1; /* We only have one group in this controller */
  txBuffer.ZW_CtrlReplicationTransferGroupFrame.nodeId   = nextNode;

  /* Go to next node ID */
  nextNode++;

  /* Send the node id to the other controller */
  bTempVar = ZW_REPLICATION_SEND_DATA(bOtherController, (BYTE*)&txBuffer, sizeof(ZW_CTRL_REPLICATION_TRANSFER_GROUP_FRAME), TRANSMIT_OPTION_ACK, ReplicationSendGroup);
	StartReplicationTimer();

  /* Check if we could send replication frame */
  if (bTempVar == FALSE)
  {
    ReplicationSendGroup(TRANSMIT_COMPLETE_NO_ACK); /* Fail replication */
  }
}

/* Do not use absolut register adressing */
#pragma NOAREGS
/*==============================   ApplicationRfNotify   ======================
**    Function used when using an external PA
**
**--------------------------------------------------------------------------*/
void                   /*RET TRUE - If normal operation, FALSE - if production test*/
PATCH_FUNCTION_NAME_STARTER(ApplicationRfNotify)(
  BYTE bStatus)        /* IN RF status */
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationRfNotify)
#pragma endasm
#endif
}

