/****************************************************************************
 *
 * Copyright (c) 2001-2011
 * Sigma Designs, Inc.
 * All Rights Reserved
 *
 *---------------------------------------------------------------------------
 *
 * Description:       Functions for controlling application board via the UART
 *
 * Last Changed By:  $Author: efh $
 * Revision:         $Revision: 22708 $
 * Last Changed:     $Date: 2012-05-03 16:41:51 +0200 (Thu, 03 May 2012) $
 *
 ****************************************************************************/
#ifdef PATCH_ENABLE
/****************************************************************************/
/* Include assembly MACRO definitions for patch insertions.                 */
/*                                                                          */
/* Define $SET (MAKE_PATCHABLE_CODE) for making patchable code destinned    */
/* for OTP or ROM memory.                                                   */
/* Undefine $RESET (MAKE_PATCHABLE_CODE) for making code containing patch   */
/* code destinned for RAM or FLASH memory.                                  */
/****************************************************************************/
#if defined(WORK_PATCH) || defined(STARTER_PATCH)
/* Making code containing patch code destinned for development RAM memory.  */
#pragma asm
$RESET (MAKE_PATCHABLE_CODE)
$INCLUDE (ZW_patch.inc)
#pragma endasm
/* Rename CODE class to CODE_PATCH */
#pragma userclass (code = PATCH)
/* Rename CONST class to CONST_PATCH */
#pragma userclass (const = PATCH)
/* Rename XDATA class to XDATA_PATCH */
#pragma userclass (xdata = PATCH)
#else
/* Making patchable code destinned for OTP or ROM memory.                   */
#pragma asm
$SET (MAKE_PATCHABLE_CODE)
$INCLUDE (ZW_patch.inc)
#pragma endasm
#endif /* elsif defined(WORK_PATCH) || defined(STARTER_PATCH) */
#endif /* PATCH_ENABLE */

/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
#include <ZW_patch.h>
#include <ZW_basis_api.h>
#include <ZW_uart_api.h>
#include <ZW_SerialAPI.h>
#include <conhandle.h>
#include <ZW_conbufio.h>

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/
/* serial Protocol handler states */
enum
{
  stateSOFHunt = 0,
  stateLen = 1,
  stateType = 2,
  stateCmd = 3,
  stateData = 4,
  stateChecksum = 5
};


/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/
PATCH_VARIABLE BYTE serBuf[SERBUF_MAX];
PATCH_VARIABLE IBYTE serBufLen, con_state;
PATCH_VARIABLE IBYTE bChecksum_RX;
PATCH_VARIABLE STATIC IWORD timeOutRX_ACKStart;
PATCH_VARIABLE STATIC IWORD timeOutRX_BYTEStart;
PATCH_VARIABLE BOOL rxActive
#ifndef WORK_PATCH
 = FALSE
#endif
;
PATCH_VARIABLE BOOL AckNakNeeded
#ifndef WORK_PATCH
 = FALSE
#endif
;


/****************************************************************************/
/*                              EXPORTED DATA                               */
/****************************************************************************/
PATCH_VARIABLE IBYTE timeOutRX_ACK
#ifndef WORK_PATCH
 = RX_ACK_TIMEOUT_DEFAULT
#endif
;
PATCH_VARIABLE IBYTE timeOutRX_BYTE
#ifndef WORK_PATCH
 = RX_BYTE_TIMEOUT_DEFAULT
#endif
;



/****************************************************************************/
/*                            PRIVATE FUNCTIONS                             */
/****************************************************************************/

/*===============================   TimerReset   ============================
**    Reset startTick to current timer tick
**
**--------------------------------------------------------------------------*/
static void                          /*RET Nothing */
TimerReset(WORD *pwTimerStartTick)   /*IN  Nothing */
{
  *pwTimerStartTick = getTickTime();
}


/*=============================   TimerGetElapsedTime   ======================
**    Get elapsed time
**
**--------------------------------------------------------------------------*/
static WORD                                /* RET  Elapsed Time in 10 ms ticks */
TimerGetElapsedTime(WORD wTimerStartTick)  /* IN   Nothing */
{
  return (getTickTime() - wTimerStartTick);
}


/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/

/*===============================   ConTxFrame   =============================
**
**   Transmit frame via serial port by adding SOF, Len, Type, cmd and Chksum.
**   Frame  : SOF-Len-Type-Cmd-DATA-Chksum
**    where SOF is Start of frame byte
**          Len is length of bytes between and including Len to last DATA byte
**          Type is Response or Request
**          Cmd Serial application command describing what DATA is
**          DATA as it says
**          Chksum is a XOR checksum taken on all BYTEs from Len to last DATA byte
**
**          NOTE: If Buf is NULL then the previously used cmd, type, Buf and len
**          is used again (Retransmission)
**
**--------------------------------------------------------------------------*/
void          /*RET Nothing */
PATCH_FUNCTION_NAME(ConTxFrame)(
  BYTE cmd,   /* IN Command */
  BYTE type,  /* IN frame Type to send (Response or Request) */
  XBYTE *Buf, /* IN pointer to BYTE buffer containing DATA to send */
  BYTE len)   /* IN the length of DATA to transmit */
#ifdef PATCH_ENABLE
reentrant
#endif
{
  static BYTE wCmd, wType, wLen, *wBuf;
  BYTE i, bChecksum;

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(ConTxFrame)
#pragma endasm
#endif

  if (Buf != NULL)
  {
    wBuf = Buf;
    wLen = len;
    wCmd = cmd;
    wType = type;
  }
  bChecksum = 0xFF; /* Initialize checksum */
  ZW_SerialPutByte(SOF);
  ZW_SerialPutByte(wLen + 3);  // Remember the 'len', 'type' and 'cmd' bytes
  bChecksum ^= wLen + 3;
  ZW_SerialPutByte(wType);
  bChecksum ^= wType;
  ZW_SerialPutByte(wCmd);
  bChecksum ^= wCmd;
  for (i = 0; (wBuf != NULL) && (i < wLen); i++)
  {
    ZW_SerialPutByte(wBuf[i]);
    bChecksum ^= wBuf[i];
  }
  ZW_SerialPutByte(bChecksum);       // XOR checksum of
  ZW_SerialFlush();                //Start sending frame to the host.
  AckNakNeeded = 1;  // Now we need an ACK...
  TimerReset(&timeOutRX_ACKStart); /* Reset ACK timeout */
}


/*==============================   ConUpdate   =============================
**
**   Parses serial data sent from external controller module (PC-controller).
**   Should be frequently called by main loop.
**
**   Return: conIdle          if nothing has happened
**           conFrameReceived if valid frame was received
**           conFrameSent     if transmitted frame was ACKed by other end
**           conFrameErr      if received frame has error in Checksum
**           conRxTimeout     if a Rx timeout happened
**           conTxTimeout     if a Tx timeout waiting for ACK happened
**
**--------------------------------------------------------------------------*/
enum T_CON_TYPE     /*RET conState - See above */
PATCH_FUNCTION_NAME(ConUpdate)(
  BYTE acknowledge) /* IN do we send acknowledge and handle frame if received correctly */
#ifdef PATCH_ENABLE
reentrant
#endif
{
  BYTE c;
  enum T_CON_TYPE retVal = conIdle;

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(ConUpdate)
#pragma endasm
#endif

  if (ZW_SerialCheck())
  {
    do
    {
      c = ZW_SerialGetByte();

      switch (con_state)
      {
        case stateSOFHunt :
          if (c == SOF)
          {
            bChecksum_RX = 0xff;
            serBufLen = 0;
            rxActive = 1;  // now we're receiving - check for timeout
            con_state++;
            TimerReset(&timeOutRX_BYTEStart); /* Reset Inframe Byte timeout */
          }
          else
          {
            if (AckNakNeeded)
            {
              if (c == ACK)
              {
                retVal = conFrameSent;
                AckNakNeeded = 0;  // Done
              }
              else if (c == NAK)
              {
                retVal = conTxTimeout;
                AckNakNeeded = 0;
              }
              else
              {
                // Bogus character received...
              }
            }
          }
          break;

        case stateLen :
          // Check for length to be inside valid range
          if ((c < FRAME_LENGTH_MIN) || (c > FRAME_LENGTH_MAX))
          {
            con_state = stateSOFHunt; // Restart looking for SOF
            rxActive = 0;  // Not really active now...
            break;
          }
          // Drop through...

        case stateType :
          if (serBufLen && (c > RESPONSE))
          {
            con_state = stateSOFHunt; // Restart looking for SOF
            rxActive = 0;  // Not really active now...
            break;
          }
          // Drop through...

        case stateCmd :
          con_state++;
          // Drop through...

        case stateData :
          if (serBufLen < SERBUF_MAX)
          {
            serBuf[serBufLen] = c;
            serBufLen++;
            bChecksum_RX ^= c;
            if (serBufLen >= serFrameLen)
            {
              con_state++;
            }
          }
          else
          {
            con_state++;
          }
          TimerReset(&timeOutRX_BYTEStart); /* Reset Inframe Byte timeout */
          break;

        case stateChecksum :
          /* Do we send ACK/NAK according to checksum... */
          /* if not then the received frame is dropped! */
          if (acknowledge)
          {
            if (c == bChecksum_RX)
            {
              ZW_SerialPutByte(ACK);
              ZW_SerialFlush();                //Start sending frame to the host.
              retVal = conFrameReceived;  // Tell THE world that we got a packet
            }
            else
            {
              ZW_SerialPutByte(NAK);       // Tell them something is wrong...
              ZW_SerialFlush();                //Start sending frame to the host.
              retVal = conFrameErr;
            }
          }
          else
          {
            // We are in the process of looking for an acknowledge to a callback request
            // Drop the new frame we received - we don't have time to handle it.
            // Send a CAN to indicate what is happening...
            ZW_SerialPutByte(CAN);
            ZW_SerialFlush();                //Start sending frame to the host.
          }
          // Drop through

        default :
          con_state = stateSOFHunt; // Restart looking for SOF
          rxActive = 0;  // Not really active now...
          break;
      }
    } while ((retVal == conIdle) && ZW_SerialCheck());
  }
  /* Check for timeouts - if no other events detected */
  if (retVal == conIdle)
  {
    /* Are in the middle of collecting a frame and have we timed out? */
    if (rxActive && (TimerGetElapsedTime(timeOutRX_BYTEStart) >= timeOutRX_BYTE))
    {
      /* Reset to SOF hunting */
      con_state = stateSOFHunt;
      rxActive = 0; /* Not inframe anymore */
      retVal = conRxTimeout;
    }
    /* Are we waiting for ACK and have we timed out? */
    if (AckNakNeeded && (TimerGetElapsedTime(timeOutRX_ACKStart) >= timeOutRX_ACK))
    {
      /* Not waiting for ACK anymore */
      AckNakNeeded = 0;
      /* Tell upper layer we could not get the frame through */
      retVal = conTxTimeout;
    }
  }
  return retVal;
}


/*==============================   ConInit   =============================
**
**   Initialize the module.
**
**--------------------------------------------------------------------------*/
void                /*RET  Nothing */
PATCH_FUNCTION_NAME(ConInit)(
  void)             /*IN   Nothing */
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(ConInit)
#pragma endasm
#endif

  ZW_InitSerialIf();

  serBufLen = 0;
  con_state = stateSOFHunt;
}
