/********************************  DoorLock.c  ********************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2009
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 *
 * Description: Slave application for evaluation kit Door Lock
 *
 * Author:   Oleg Zadorozhnyy, Valeriy Vyshnyak
 *
 * Last Changed By:  $Author: psh $
 * Revision:         $Revision: 22667 $
 * Last Changed:     $Date: 2012-05-02 17:33:28 +0200 (Wed, 02 May 2012) $
 *
 ****************************************************************************/

#if defined(WORK_PATCH) || defined(STARTER_PATCH)
#define ZW_DEBUG
#endif


#define ZW_ADVANCED_DOOR_LOCK

#ifdef PATCH_ENABLE
/****************************************************************************/
/* Include assembly MACRO definitions for patch insertions.                 */
/*                                                                          */
/* Define $SET (MAKE_PATCHABLE_CODE) for making patchable code destinned    */
/* for OTP or ROM memory.                                                   */
/* Undefine $RESET (MAKE_PATCHABLE_CODE) for making code containing patch   */
/* code destinned for RAM or FLASH memory.                                  */
/****************************************************************************/
#if defined(WORK_PATCH) || defined(STARTER_PATCH)
/* Making code containing patch code destinned for development RAM memory.  */
#pragma asm
$RESET (MAKE_PATCHABLE_CODE)
$INCLUDE (ZW_patch.inc)
#pragma endasm
/* Rename CODE class to CODE_PATCH */
#pragma userclass (code = PATCH)
/* Rename CONST class to CONST_PATCH */
#pragma userclass (const = PATCH)
/* Rename XDATA class to XDATA_PATCH */
#pragma userclass (xdata = PATCH)
#else
/* Making patchable code destinned for OTP or ROM memory.                   */
#pragma asm
$SET (MAKE_PATCHABLE_CODE)
$INCLUDE (ZW_patch.inc)
#pragma endasm
#endif /* elsif defined(WORK_PATCH) || defined(STARTER_PATCH) */
#endif /* PATCH_ENABLE */

/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/
#include <config_app.h>
#include <ZW_patch.h>
#ifdef ZW_CONTROLLER
#include <ZW_controller_api.h>
#else
#include <ZW_slave_api.h>
#endif
#include <ZW_pindefs.h>
#include <ZW_evaldefs.h>
#include <ZW_uart_api.h>

#include <slave_learn.h>
#include <one_button.h>
#include <ZW_TransportLayer.h>

#include "DoorLock.h"
#include "eeprom.h"

#ifdef FLIRS
#include <ZW_FLiRS.h>
#endif

#ifdef ZW_ISD51_DEBUG
#include "ISD51.h"
#endif

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/

#ifndef TRANSPORT_WORK_END
#define TRANSPORT_WORK_END			0x00
#define TRANSPORT_WORK_START		0x01
#define TRANSPORT_WORK_ERROR		0x02
#endif

#ifdef APPL_PROD_TEST
/* Productiontest pin definition */

#ifdef ZW020x
/* SSNJ pin now input */
#define SET_PRODUCTIONTEST_PIN  PIN_IN(SSN, 0)
#define IN_PRODUCTIONTEST       (!PIN_GET(SSN))
#endif  /* ZW020x */

#ifdef ZW030x
/* SSNJ pin now input */
#define SET_PRODUCTIONTEST_PIN  PIN_IN(SSN, 0)
#define IN_PRODUCTIONTEST       (!PIN_GET(SSN))
#endif  /* ZW030x */

#endif /* APPL_PROD_TEST */


/* Timer repeat define */
#define FOREVER             TIMER_FOREVER
#define ONE_SHOT            TIMER_ONE_TIME
/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/

/* A list of the known command classes. Except the basic class which allways */
/* should be supported. Used when node info is send */
static code BYTE nodeInfo[] = {
#ifdef ZW_ADVANCED_DOOR_LOCK
					COMMAND_CLASS_DOOR_LOCK,
#else
                    COMMAND_CLASS_LOCK,
#endif
					COMMAND_CLASS_BASIC,
                    COMMAND_CLASS_MANUFACTURER_SPECIFIC,
                    COMMAND_CLASS_VERSION,
                    COMMAND_CLASS_POWERLEVEL,
#ifdef SECURITY
                    COMMAND_CLASS_SECURITY,
#endif
                    };

#ifdef SECURITY
extern	BYTE nodeSecure;
extern  BYTE secureCmdHandling;
#define TSEC_CMD_HANDLING_SECURE    1

static code BYTE nodeInfoAfterIncluded[] = {
                    COMMAND_CLASS_VERSION,
                    COMMAND_CLASS_MANUFACTURER_SPECIFIC,
                    COMMAND_CLASS_SECURITY
					};



static code BYTE nodeInfoForTransport[] = {
#ifdef ZW_ADVANCED_DOOR_LOCK
					COMMAND_CLASS_DOOR_LOCK,
#else
                    COMMAND_CLASS_LOCK,
#endif
					COMMAND_CLASS_BASIC,
                    COMMAND_CLASS_MANUFACTURER_SPECIFIC,
                    COMMAND_CLASS_VERSION,
                    COMMAND_CLASS_POWERLEVEL,
                    };
#else
#define nodeInfoForTransport nodeInfo
#endif

#define DOOR_LOCK_OPERATION_SET_TIMEOUT_NOT_SUPPORTED 0xFE

/* node status */
PATCH_VARIABLE volatile BYTE lockNodeStatus
#ifndef WORK_PATCH
 = DEFAULT_LOCK_STATUS
#endif
;
PATCH_VARIABLE BYTE lockUpdateHandle
#ifndef WORK_PATCH
 = 0xff
#endif
;

PATCH_VARIABLE BYTE pollState
#ifndef WORK_PATCH
 = POLLIDLE
#endif
;
PATCH_VARIABLE_STARTER ZW_APPLICATION_TX_BUFFER txBuf;

PATCH_VARIABLE BYTE txOption;

PATCH_VARIABLE BYTE timerPowerLevel
#ifndef WORK_PATCH
 = 0
#endif
;
PATCH_VARIABLE BYTE timerPowerLevelHandle
#ifndef WORK_PATCH
 = 0
#endif
;
PATCH_VARIABLE BYTE timerPowerLevelSec
#ifndef WORK_PATCH
 = 0
#endif
;
PATCH_VARIABLE BYTE testNodeID
#ifndef WORK_PATCH
 = ZW_TEST_NOT_A_NODEID
#endif
;

PATCH_VARIABLE BYTE DoorCondition
#ifndef WORK_PATCH
 = 0
#endif
;
PATCH_VARIABLE BYTE DoorLockMode
#ifndef WORK_PATCH
 = 0
#endif
;

PATCH_VARIABLE BYTE testSourceNodeID;
PATCH_VARIABLE BYTE testPowerLevel;
PATCH_VARIABLE WORD testFrameCount;
PATCH_VARIABLE WORD testFrameSuccessCount;
PATCH_VARIABLE BYTE testState
#ifndef WORK_PATCH
 = 0
#endif
;
PATCH_VARIABLE BYTE currentPower
#ifndef WORK_PATCH
 = normalPower
#endif
;

#ifdef DEBUG_STRESS
#define DEBUG_STRESS_TIMEOUT_START 250
#define DEBUG_STRESS_TIMEOUT       30
PATCH_VARIABLE BOOL debugStress
#ifndef WORK_PATCH
 = 0
#endif
;
PATCH_VARIABLE BYTE debugStressHandle
#ifndef WORK_PATCH
 = 0
#endif
;
PATCH_VARIABLE BYTE currentStressValue
#ifndef WORK_PATCH
 = 0
#endif
;
#endif

PATCH_VARIABLE BYTE bNWIStartup;
PATCH_VARIABLE BYTE bMyNodeID;

PATCH_VARIABLE BYTE wakeUpReason;

PATCH_VARIABLE BYTE myNodeID
#ifndef WORK_PATCH
 = 0
#endif
;
PATCH_VARIABLE BYTE myHomeID[4];
//extern void ( *cbFuncZWSecure ) ( BYTE txStatus );

/****************************************************************************/
/*                              EXPORTED DATA                               */
/****************************************************************************/

/****************************************************************************/
/*                            PRIVATE FUNCTIONS                             */
/****************************************************************************/

/*===============================   SaveStatus   ============================
**    Function description
**      Stores the current status of the lock on/off
**      in the application part of the EEPROM.
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
static void SaveStatus(void)
{
  ZW_MEM_PUT_BYTE(EEOFFSET_LOCK_STATUS, lockNodeStatus);
  ZW_MEM_PUT_BYTE(EEOFFSET_MAGIC, MAGIC_VALUE);
}

/*===============================   LoadStatus   ===========================
**    Function description
**      Fetches status of the lock on/off from the
**      application part of the EEPROM and uses it as the current status.
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
static void LoadStatus(void)
{
  lockNodeStatus = ZW_MEM_GET_BYTE(EEOFFSET_LOCK_STATUS);

	timerPowerLevel = ZW_MEM_GET_BYTE(EEOFFSET_POWER_LEVEL);
	timerPowerLevelSec = ZW_MEM_GET_BYTE(EEOFFSET_TIMER_POWER_LEVEL);

	testNodeID = ZW_MEM_GET_BYTE(EEOFFSET_TEST_NODE_ID);
	testPowerLevel = ZW_MEM_GET_BYTE(EEOFFSET_TEST_POWER_LEVEL);
	testFrameCount = ZW_MEM_GET_BYTE(EEOFFSET_TEST_FRAME_COUNT);
	testSourceNodeID = ZW_MEM_GET_BYTE(EEOFFSET_TEST_SOURCE_NODE_ID);

}

#ifdef DEBUG_STRESS
void
PATCH_FUNCTION_NAME(SendBasicSet)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(SendBasicSet)
#pragma endasm
#endif
  if (!debugStressHandle)
  {
    debugStressHandle = ZW_TIMER_START(SendBasicSet, DEBUG_STRESS_TIMEOUT, ONE_SHOT);
  }
  else
  {
    debugStressHandle = 0;
    txBuf.ZW_BasicSetFrame.cmdClass = COMMAND_CLASS_BASIC;
    txBuf.ZW_BasicSetFrame.cmd      = BASIC_SET;
    txBuf.ZW_BasicSetFrame.value    = currentStressValue;
    if (currentStressValue == 10)
    {
      currentStressValue = 50;
    }
    else
    {
      if (currentStressValue == 50)
      {
        currentStressValue = 0;
      }
      else
      {
        currentStressValue = 10;
      }
    }
    ZW_SEND_DATA(2, (BYTE *)&txBuf, sizeof(txBuf.ZW_BasicSetFrame), (TRANSMIT_OPTION_ACK), SendBasicSet);
  }
}
#endif

/*===========================   UpdateLock   ================================
**    Function description
**      Update lock according to current state (lockNodeStatus)
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
static void           /*RET  Nothing                  */
UpdateLock( void )  /* IN  Nothing                  */
{
  lockUpdateHandle = 0xff; /* Now done */
  if (lockNodeStatus & M_LOCK_ON_OFF)
  {
#ifdef LOW_FOR_ON
    PIN_OFF(LED3);
#else
    PIN_ON(LED3);
#endif
  }
  else
  {
#ifdef LOW_FOR_ON
    PIN_ON(LED3);
#else
    PIN_OFF(LED3);
#endif
  }
  ZW_DEBUG_SEND_BYTE('U');
  ZW_DEBUG_SEND_NUM(lockNodeStatus & M_LOCK_ON_OFF);

#ifdef FLIRS
  FLiRS_SleepTimeoutStart(FALSE);
#endif

}

/*===============================   SetLock   ================================
**    Function description
**      Set the door lock to:
**
**      CURRENT_STATE - The current lock state stored in eeprom
**      TOGGLE        - Toggle the lock on/off
**      ON            - ON
**      OFF           - OFF
**
**    Side effects :
**
**--------------------------------------------------------------------------*/
static void     /*RET Nothing */
SetLock(
  BYTE bAction) /* IN Action to be done */
{
  switch (bAction)
  {
    case TOGGLE:
      lockNodeStatus ^= M_LOCK_ON_OFF;
      break;

    case CURRENT_STATE:
      break;

    case ON:
      lockNodeStatus |= M_LOCK_ON_OFF;
      break;

    case OFF:
      lockNodeStatus &= ~(M_LOCK_ON_OFF);
      break;
  }

  ZW_DEBUG_SEND_BYTE('L');
  ZW_DEBUG_SEND_NUM(bAction);
  ZW_DEBUG_SEND_NUM(lockUpdateHandle);

  /* If this function called second time, before the UpdateLock is executed, i prefer to delete */
  /*  the old timer and start a new one. Because, just ignoring to starting new timer, in most cases */
  /*  lead to situation, when UpdateLosk timer handler is never called. */
  if (lockUpdateHandle != 0xff)
  {
    ZW_TIMER_CANCEL(lockUpdateHandle);
  }
  /* Postpone the lock relay update a bit */
  lockUpdateHandle = ZW_TIMER_START(UpdateLock, LOCK_UPDATETIME, TIMER_ONE_TIME); /* in 100ms we update lock */

  ZW_DEBUG_SEND_NUM(lockUpdateHandle);

  /* Save new status in EEPROM */
  SaveStatus();
}

/*=============================  ApplicationPoll   =========================
**    Application poll function for LED dimmer
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                    /*RET  Nothing                  */
PATCH_FUNCTION_NAME_STARTER(ApplicationPoll)(void) /* IN  Nothing                  */
#ifdef PATCH_ENABLE
reentrant
#endif
{
  BYTE lastAction;

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationPoll)
#pragma endasm
#endif
#ifdef ZW_ISD51_DEBUG         /* init ISD51 only when the uVision2 Debugger tries to connect */
  ISDcheck();                 /* initialize uVision2 Debugger and continue program run */
#endif
  lastAction = OneButtonLastAction();
//  ZW_DEBUG_CMD_POLL();

  if (bNWIStartup)
  {
#ifdef FLIRS
		FLiRS_SleepTimeoutStop();
#endif

    StartLearnModeNow(ZW_SET_LEARN_MODE_NWI);
    bNWIStartup = FALSE;
    pollState = POLLLEARNMODE;
    ZW_DEBUG_SEND_BYTE('N');
    ZW_DEBUG_SEND_BYTE('B');
  }

  switch (pollState)
  {
    case POLLIDLE:
      if (lastAction==BUTTON_WAS_PRESSED)
      {
#ifdef FLIRS
		FLiRS_SleepTimeoutStop();
#endif
        SetLock(TOGGLE);
        ZW_DEBUG_SEND_BYTE('B');
        ZW_DEBUG_SEND_BYTE('T');
      }
      else if (lastAction==BUTTON_IS_HELD)
      {
        /* do nothing */
        ZW_DEBUG_SEND_BYTE('B');
        ZW_DEBUG_SEND_BYTE('H');
      }
      else if (lastAction==BUTTON_TRIPLE_PRESS)
      {
        StartLearnModeNow(ZW_SET_LEARN_MODE_CLASSIC);
        pollState = POLLLEARNMODE;
        ZW_DEBUG_SEND_BYTE('N');
        ZW_DEBUG_SEND_BYTE('C');
      }
      break;

    case POLLLEARNMODE:
      /* Just wait... */
      if (lastAction==BUTTON_WAS_PRESSED)
      {
        /* Stop learn mode */
        StartLearnModeNow(FALSE);
        pollState = POLLIDLE;
        ZW_MEM_PUT_BYTE(EEOFFSET_LERN_MODE,TRUE);
        ZW_DEBUG_SEND_BYTE('N');
        ZW_DEBUG_SEND_BYTE('E');
      }
      break;
  }
}

#if defined(ZW020x) || defined(ZW030x) || defined(ZW040x)
/*===========================   SendTestReport   ============================
**    Send current Powerlevel test results
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
//void
//SendTestReport( void )
//{
void
PATCH_FUNCTION_NAME(SendTestReport)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(SendTestReport)
#pragma endasm
#endif

  txBuf.ZW_PowerlevelTestNodeReportFrame.cmdClass = COMMAND_CLASS_POWERLEVEL;
  txBuf.ZW_PowerlevelTestNodeReportFrame.cmd = POWERLEVEL_TEST_NODE_REPORT;
  txBuf.ZW_PowerlevelTestNodeReportFrame.testNodeid = testNodeID;
  txBuf.ZW_PowerlevelTestNodeReportFrame.statusOfOperation = testState;
  txBuf.ZW_PowerlevelTestNodeReportFrame.testFrameCount1 = (BYTE)(*((BYTE*)&testFrameSuccessCount));
  txBuf.ZW_PowerlevelTestNodeReportFrame.testFrameCount2 = (BYTE)*(&testFrameSuccessCount);
  /* Send Report - we do not care if it gets there or not - if needed report can be requested again */
  Transport_SendReport(testSourceNodeID, (BYTE *)&txBuf, sizeof(txBuf.ZW_PowerlevelTestNodeReportFrame), txOption, NULL, FALSE);
}

/*=============================   SendTestDone   ============================
**    Test frame has been transmitted to DUT and the result is noted for
**    later reporting. If not finished then another Test frame is transmitted.
**    If all test frames has been transmitted then the test is stopped
**    and the final result is reported to the PowerlevelTest initiator.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
//void
//SendTestDone(
//  BYTE bStatus)
//{

void
PATCH_FUNCTION_NAME(SendTestDone)(
  BYTE bStatus
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(SendTestDone)
#pragma endasm
#endif

  if (bStatus == TRANSMIT_COMPLETE_OK)
  {
    testFrameSuccessCount++;
  }
  if (testFrameCount && (--testFrameCount))
  {
    ZW_SEND_TEST_FRAME(testNodeID, testPowerLevel, SendTestDone);
  }
  else
  {
    if (testFrameSuccessCount)
    {
      testState = ZW_TEST_SUCCES;
    }
    else
    {
      testState = ZW_TEST_FAILED;
    }
    ZW_RF_POWERLEVEL_SET(currentPower);  /* Back to previous setting */
    SendTestReport();
  }
}

/*=======================   PowerLevelTimerCancel   ==========================
**    Cancels PowerLevel timer
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
//void
//PowerLevelTimerCancel( void )
//{
void
PATCH_FUNCTION_NAME(PowerLevelTimerCancel)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(PowerLevelTimerCancel)
#pragma endasm
#endif
  ZW_TIMER_CANCEL(timerPowerLevelHandle);
  timerPowerLevelHandle = 0;
}

/*===============================   StartTest   ==============================
**    Timer callback which maked sure that the RF transmit powerlevel is set
**    back to normalPower after the designated time period.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
//void
//PowerLevelTimeout( void )
//{
void
PATCH_FUNCTION_NAME(PowerLevelTimeout)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(PowerLevelTimeout)
#pragma endasm
#endif
  if (!--timerPowerLevelSec)
  {
    ZW_RF_POWERLEVEL_SET(normalPower);  /* Reset powerlevel to normalPower */
    PowerLevelTimerCancel();
  }
}

/*===============================   StartTest   ==============================
**    Start the powerlevel test run
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
//void
//StartTest(void)
//{
void
PATCH_FUNCTION_NAME(StartTest)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StartTest)
#pragma endasm
#endif
  testState = ZW_TEST_INPROGRESS;
  currentPower = ZW_RF_POWERLEVEL_GET();  /* Get current (normalPower) */
  ZW_RF_POWERLEVEL_SET(testPowerLevel);
  ZW_SEND_TEST_FRAME(testNodeID, testPowerLevel, SendTestDone);
}

#endif





/*========================   ApplicationCommandHandler   ====================
**    Handling of a received application commands and requests
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void                              /*RET Nothing                  */
PATCH_FUNCTION_NAME_STARTER(Transport_ApplicationCommandHandler)(
  BYTE  rxStatus,                 /* IN Frame header info */
  BYTE  sourceNode,               /* IN Command sender Node ID */
  ZW_APPLICATION_TX_BUFFER *pCmd, /* IN Payload from the received frame, the union */
                                  /*    should be used to access the fields */
  BYTE   cmdLength                /* IN Number of command bytes including the command */
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
  BYTE tmp;
  BYTE classcmd;
  BYTE cmd;
  BYTE param1;
  BYTE param2;
  BYTE param3;
  BYTE param4;
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(Transport_ApplicationCommandHandler)
#pragma endasm
#endif
  ZW_DEBUG_SEND_BYTE('H');
  ZW_DEBUG_SEND_NUM(pCmd->ZW_Common.cmdClass);
  ZW_DEBUG_SEND_NUM(pCmd->ZW_Common.cmd);

  classcmd = pCmd->ZW_Common.cmdClass;
  cmd = pCmd->ZW_Common.cmd;
  param1 = *((BYTE*)pCmd + OFFSET_PARAM_1);
  param2 = *((BYTE*)pCmd + OFFSET_PARAM_2);
  param3 = *((BYTE*)pCmd + OFFSET_PARAM_3);
  param4 = *((BYTE*)pCmd + OFFSET_PARAM_4);

 	txOption = ((rxStatus & RECEIVE_STATUS_LOW_POWER) ?
 	            TRANSMIT_OPTION_LOW_POWER : 0) | TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE;


 	switch(pCmd->ZW_Common.cmdClass)
 	{
	 	case COMMAND_CLASS_VERSION:
	  	{
		    switch (pCmd->ZW_VersionGetFrame.cmd)
		    {
			    case VERSION_GET:
			    {
			      txBuf.ZW_VersionReportFrame.cmdClass = COMMAND_CLASS_VERSION;
			      txBuf.ZW_VersionReportFrame.cmd = VERSION_REPORT;
			      txBuf.ZW_VersionReportFrame.zWaveLibraryType = ZW_TYPE_LIBRARY();
#ifdef ZW_VERSION_BETA
			      txBuf.ZW_VersionReportFrame.zWaveProtocolVersion = ZW_VERSION_MAJOR+(ZW_VERSION_BETA-0x60)*10;
#else
			      txBuf.ZW_VersionReportFrame.zWaveProtocolVersion = ZW_VERSION_MAJOR;
#endif
			      txBuf.ZW_VersionReportFrame.zWaveProtocolSubVersion = ZW_VERSION_MINOR;
			      txBuf.ZW_VersionReportFrame.applicationVersion = APP_VERSION;
			      txBuf.ZW_VersionReportFrame.applicationSubVersion = APP_REVISION;
			      Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_VersionReportFrame), txOption, NULL, FALSE);
			      break;
			    }

			    case VERSION_COMMAND_CLASS_GET:
			    {
			      txBuf.ZW_VersionCommandClassReportFrame.cmdClass = COMMAND_CLASS_VERSION;
			      txBuf.ZW_VersionCommandClassReportFrame.cmd = VERSION_COMMAND_CLASS_REPORT;
			      txBuf.ZW_VersionCommandClassReportFrame.requestedCommandClass = pCmd->ZW_VersionCommandClassGetFrame.requestedCommandClass;
			      switch (txBuf.ZW_VersionCommandClassReportFrame.requestedCommandClass)
			      {
				      case COMMAND_CLASS_BASIC:
				        txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = BASIC_VERSION;
				        break;

				      case COMMAND_CLASS_LOCK:
				        txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = LOCK_VERSION;
				        break;

				      case COMMAND_CLASS_MANUFACTURER_SPECIFIC:
				        txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = MANUFACTURER_SPECIFIC_VERSION;
				        break;

				      case COMMAND_CLASS_VERSION:
				        txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = VERSION_VERSION;
				        break;

				      case COMMAND_CLASS_POWERLEVEL:
				        txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = POWERLEVEL_VERSION;
				        break;
#ifdef SECURITY
				      case COMMAND_CLASS_SECURITY:
				        txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = SECURITY_VERSION;
				        break;
#endif
				      default:
				        txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = UNKNOWN_VERSION;
				        break;
			      }
			      Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_VersionCommandClassReportFrame), txOption, NULL, FALSE);
			      break;
			    }
		    }
	    	break;
	  	}

		case COMMAND_CLASS_MANUFACTURER_SPECIFIC:
		{
		    switch (pCmd->ZW_ManufacturerSpecificGetFrame.cmd)
		    {
			    case MANUFACTURER_SPECIFIC_GET:
			    {
			      txBuf.ZW_ManufacturerSpecificReportFrame.cmdClass = COMMAND_CLASS_MANUFACTURER_SPECIFIC;
			      txBuf.ZW_ManufacturerSpecificReportFrame.cmd = MANUFACTURER_SPECIFIC_REPORT;
			      txBuf.ZW_ManufacturerSpecificReportFrame.manufacturerId1 = 0x00; // Zensys manufacturer ID
			      txBuf.ZW_ManufacturerSpecificReportFrame.manufacturerId2 = 0x00;
			      txBuf.ZW_ManufacturerSpecificReportFrame.productTypeId1 = 0x00; // Assigned by manufacturer
			      txBuf.ZW_ManufacturerSpecificReportFrame.productTypeId2 = 0x00;
			      txBuf.ZW_ManufacturerSpecificReportFrame.productId1 = 0x00; // Assigned by manufacturer
			      txBuf.ZW_ManufacturerSpecificReportFrame.productId2 = 0x00;
			      Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_ManufacturerSpecificReportFrame), txOption, NULL, FALSE);
			      break;
			    }
		    }
		    break;
		}
	}


#ifdef SECURITY
  if(nodeSecure)
  if (secureCmdHandling != TSEC_CMD_HANDLING_SECURE)
  {
//	  if(currentState == STATE_COMMAND_HANDLER)
//	  {
//	    currentState = STATE_APPL_IDLE;
//	  }
  	  return;
  }
#endif

	ZW_DEBUG_SEND_BYTE(' ');
	ZW_DEBUG_SEND_BYTE('V');
	ZW_DEBUG_SEND_BYTE(' ');

 	switch(pCmd->ZW_Common.cmdClass)
 	{


#ifdef ZW_ADVANCED_DOOR_LOCK
 	case COMMAND_CLASS_DOOR_LOCK:
 	{
 	  switch(pCmd->ZW_Common.cmd)
 	  {
	 	  case DOOR_LOCK_OPERATION_SET:
		  {
			DoorLockMode = pCmd->ZW_DoorLockOperationSetFrame.doorLockMode;
			if (pCmd->ZW_DoorLockOperationSetFrame.doorLockMode == DOOR_LOCK_OPERATION_REPORT_DOOR_UNSECURED)
			{
			  SetLock(OFF);
			}
			else
			{
			  SetLock(ON);
			}
			break;
		  }
	 	  case DOOR_LOCK_OPERATION_GET:
	 	  {
	 	    txBuf.ZW_DoorLockOperationReportFrame.cmdClass 				=	COMMAND_CLASS_DOOR_LOCK;
	 	    txBuf.ZW_DoorLockOperationReportFrame.cmd 					=	DOOR_LOCK_OPERATION_REPORT;
			txBuf.ZW_DoorLockOperationReportFrame.doorLockMode			=	(lockNodeStatus & M_LOCK_ON_OFF) ?
	 	                                DOOR_LOCK_OPERATION_REPORT_DOOR_SECURED : DOOR_LOCK_OPERATION_REPORT_DOOR_UNSECURED;//DoorLockMode;
	 	    txBuf.ZW_DoorLockOperationReportFrame.properties1			=	ZW_MEM_GET_BYTE(EEOFFSET_HANDLES_STATE);
			txBuf.ZW_DoorLockOperationReportFrame.doorCondition			=	0x3; //the hardware status of the door lock device     // DoorCondition;
			txBuf.ZW_DoorLockOperationReportFrame.lockTimeoutMinutes	=	DOOR_LOCK_OPERATION_SET_TIMEOUT_NOT_SUPPORTED; //ZW_MEM_GET_BYTE(EEOFFSET_LOCK_MINUTES);
			txBuf.ZW_DoorLockOperationReportFrame.lockTimeoutSeconds	=	DOOR_LOCK_OPERATION_SET_TIMEOUT_NOT_SUPPORTED; //ZW_MEM_GET_BYTE(EEOFFSET_LOCK_SECONDS);

			Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_DoorLockOperationReportFrame), txOption | TRANSMIT_OPTION_EXPLORE, NULL, FALSE);
	 	    break;
	 	  }
	 	  case DOOR_LOCK_CONFIGURATION_SET:
	 	  {
			ZW_MEM_PUT_BYTE(EEOFFSET_OPERATION_TYPE,pCmd->ZW_DoorLockConfigurationSetFrame.operationType);
			ZW_MEM_PUT_BYTE(EEOFFSET_HANDLES_STATE,pCmd->ZW_DoorLockConfigurationSetFrame.properties1);
			ZW_MEM_PUT_BYTE(EEOFFSET_LOCK_MINUTES,pCmd->ZW_DoorLockConfigurationSetFrame.lockTimeoutMinutes);
			ZW_MEM_PUT_BYTE(EEOFFSET_LOCK_SECONDS,pCmd->ZW_DoorLockConfigurationSetFrame.lockTimeoutSeconds);
	 	    break;
	 	  }
	 	  case DOOR_LOCK_CONFIGURATION_GET:
	 	  {
	 	    txBuf.ZW_DoorLockConfigurationReportFrame.cmdClass 			=	COMMAND_CLASS_DOOR_LOCK;
	 	    txBuf.ZW_DoorLockConfigurationReportFrame.cmd 				=	DOOR_LOCK_CONFIGURATION_REPORT;
	 	    txBuf.ZW_DoorLockConfigurationReportFrame.operationType		=	0x1; // Constant operation  //ZW_MEM_GET_BYTE(EEOFFSET_OPERATION_TYPE);
			txBuf.ZW_DoorLockConfigurationReportFrame.properties1		=	ZW_MEM_GET_BYTE(EEOFFSET_HANDLES_STATE);
			txBuf.ZW_DoorLockConfigurationReportFrame.lockTimeoutMinutes=	DOOR_LOCK_OPERATION_SET_TIMEOUT_NOT_SUPPORTED; //ZW_MEM_GET_BYTE(EEOFFSET_LOCK_MINUTES);
			txBuf.ZW_DoorLockConfigurationReportFrame.lockTimeoutSeconds=	DOOR_LOCK_OPERATION_SET_TIMEOUT_NOT_SUPPORTED; //ZW_MEM_GET_BYTE(EEOFFSET_LOCK_SECONDS);

			Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_DoorLockConfigurationReportFrame), txOption | TRANSMIT_OPTION_EXPLORE, NULL, FALSE);
	 	    break;
	 	  }
 	  }
 	  break;
	}

 	case COMMAND_CLASS_BASIC:
 	{
 	  switch(pCmd->ZW_Common.cmd)
 	  {
		  //Must be ignored to avoid unintentional operation. Cannot be mapped to another command class.
	 	  case BASIC_SET:
			break;
	 	  case BASIC_GET:
	 	  {
	      /* Answer with the called class */
	 	    txBuf.ZW_LockReportFrame.cmdClass = pCmd->ZW_LockGetFrame.cmdClass;
	 	    txBuf.ZW_LockReportFrame.cmd = BASIC_REPORT;
	 	    txBuf.ZW_LockReportFrame.lockState = (lockNodeStatus & M_LOCK_ON_OFF) ?
	 	      DOOR_LOCK_OPERATION_REPORT_DOOR_SECURED : DOOR_LOCK_OPERATION_REPORT_DOOR_UNSECURED;
	      	Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_LockReportFrame), txOption | TRANSMIT_OPTION_EXPLORE, NULL, FALSE);
	 	    break;
	 	  }
 	  }
 	  break;
 	}


#else
 	case COMMAND_CLASS_BASIC:
 	case COMMAND_CLASS_LOCK:
 	{
 	  switch(pCmd->ZW_Common.cmd)
 	  {
 	  /*case BASIC_SET:*/
 	  case LOCK_SET:
 	  {
 	    if(cmdLength >= sizeof(pCmd->ZW_LockSetFrame))
 	    {
        ZW_DEBUG_SEND_BYTE('S');
        ZW_DEBUG_SEND_NUM(pCmd->ZW_LockSetFrame.lockState);
//        if(pCmd->ZW_LockSetFrame.cmdClass == COMMAND_CLASS_LOCK)
        {
          if (pCmd->ZW_LockSetFrame.lockState == DOOR_LOCK_OPERATION_REPORT_DOOR_UNSECURED)
          {
            SetLock(OFF);
          }
          else
          {
            SetLock(ON);
          }
        }
 	    }
 	    break;
 	  }
 	  /*case BASIC_GET:*/
 	  case LOCK_GET:
 	  {
      /* Answer with the called class */
 	    txBuf.ZW_LockReportFrame.cmdClass = pCmd->ZW_LockGetFrame.cmdClass;
 	    txBuf.ZW_LockReportFrame.cmd = LOCK_REPORT;
 	    txBuf.ZW_LockReportFrame.lockState = (lockNodeStatus & M_LOCK_ON_OFF) ?
 	      DOOR_LOCK_OPERATION_REPORT_DOOR_SECURED : DOOR_LOCK_OPERATION_REPORT_DOOR_UNSECURED;
      Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_LockReportFrame), txOption | TRANSMIT_OPTION_EXPLORE, NULL, FALSE);
 	    break;
 	  }
 	  }
 	  break;
 	}
#endif









#ifdef ZW_CONTROLLER
 	case COMMAND_CLASS_CONTROLLER_REPLICATION:
 	{
    /* Handle replication frames */
    /* We are only interested in group replication here */
    /* but all other replication can be handled in the same way. */
    /* In this application we ignore everything else and just */
    /* send the command complete reply */
#if 0
    if (pCmd->ZW_Common.cmd == CTRL_REPLICATION_TRANSFER_GROUP)
    {
      /* We are only interested in group 1 */
      if (pCmd->ZW_TransferGroupFrame.groupID == 1)
      {
        /* Add the node ID to the group */
        AddNodeToGroup(pCmd->ZW_TransferGroupFrame.nodeID);
      }
    }
#endif
 	  break;
 	}
#endif

#if defined(ZW020x) || defined(ZW030x) || defined(ZW040x)


  case COMMAND_CLASS_POWERLEVEL:
  {
    switch (cmd)
    {
      case POWERLEVEL_SET:
		if(( param1==normalPower)||(param1==minus1dBm)||(param1==minus2dBm)||
			(param1==minus3dBm)||(param1==minus4dBm)||(param1==minus5dBm)||
			(param1==minus6dBm)||(param1==minus7dBm)||(param1==minus8dBm)||(param1==minus9dBm))
		{
        if (param2 || (param1 == normalPower))  /* Only if timeout defined or powerlevel is normalPower */
        {
          param1 = ZW_RF_POWERLEVEL_SET(param1);  /* Set powerlevel according to command */
          if (timerPowerLevelHandle)
          {
            PowerLevelTimerCancel();
            timerPowerLevelSec = 0;
          }
          if (param1 != normalPower)  /* If normalPower then no timer is started */
          {
            timerPowerLevelSec = param2;
            timerPowerLevelHandle = ZW_TIMER_START(PowerLevelTimeout, TIMER_ONE_SECOND, FOREVER);
          }
        }
		}
        break;

      case POWERLEVEL_GET:
        txBuf.ZW_PowerlevelReportFrame.cmdClass = COMMAND_CLASS_POWERLEVEL;
        txBuf.ZW_PowerlevelReportFrame.cmd = POWERLEVEL_REPORT;
        txBuf.ZW_PowerlevelReportFrame.powerLevel = ZW_RF_POWERLEVEL_GET();
        txBuf.ZW_PowerlevelReportFrame.timeout = timerPowerLevelSec;
        Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_PowerlevelReportFrame), txOption, NULL, FALSE);
        break;

      case POWERLEVEL_TEST_NODE_SET:
        if (testState != ZW_TEST_INPROGRESS)
        {
          testSourceNodeID = sourceNode;
          testNodeID = param1;
          testPowerLevel = param2;
          testFrameCount = (WORD)((WORD)param3 << 8) + param4;
          testFrameSuccessCount = 0;
          if (testFrameCount)
          {
            StartTest();
          }
          else
          {
            testState = ZW_TEST_FAILED;
            SendTestReport();
          }
        }
        break;

      case POWERLEVEL_TEST_NODE_GET:
        SendTestReport();
        break;
        default:
        break;

    } /* switch */
  } /* else if == COMMAND_CLASS_POWERLEVEL */



// 	case COMMAND_CLASS_POWERLEVEL:
//  {
//    switch (pCmd->ZW_Common.cmd)
//    {
//    case POWERLEVEL_SET:
//    {
//      if (pCmd->ZW_PowerlevelSetFrame.timeout || (pCmd->ZW_PowerlevelSetFrame.powerLevel == normalPower))  /* Only if timeout defined or powerlevel is normalPower */
//      {
//        tmp = ZW_RF_POWERLEVEL_SET(pCmd->ZW_PowerlevelSetFrame.powerLevel);  /* Set powerlevel according to command */
//        if (timerPowerLevelHandle)
//        {
//          PowerLevelTimerCancel();
//          timerPowerLevelSec = 0;
//        }
//        if (tmp != normalPower)  /* If normalPower then no timer is started */
//        {
//          timerPowerLevelSec = pCmd->ZW_PowerlevelSetFrame.timeout;
//          timerPowerLevelHandle = ZW_TIMER_START(PowerLevelTimeout, TIMER_ONE_SECOND, TIMER_FOREVER);
//		  timerPowerLevel = pCmd->ZW_PowerlevelSetFrame.powerLevel;
//        }
//      }
//	  ZW_MEM_PUT_BYTE(EEOFFSET_POWER_LEVEL,timerPowerLevel);
//	  ZW_MEM_PUT_BYTE(EEOFFSET_TIMER_POWER_LEVEL,timerPowerLevelSec);
//      break;
//    }
//
//    case POWERLEVEL_GET:
//    {
//      txBuf.ZW_PowerlevelReportFrame.cmdClass = COMMAND_CLASS_POWERLEVEL;
//      txBuf.ZW_PowerlevelReportFrame.cmd = POWERLEVEL_REPORT;
//      txBuf.ZW_PowerlevelReportFrame.powerLevel = ZW_RF_POWERLEVEL_GET();
//      txBuf.ZW_PowerlevelReportFrame.timeout = timerPowerLevelSec;
//      Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_PowerlevelReportFrame), txOption, NULL, FALSE);
//      break;
//    }
//
//    case POWERLEVEL_TEST_NODE_SET:
//    {
//      if (testState != ZW_TEST_INPROGRESS)
//      {
//        testSourceNodeID = sourceNode;
//        testNodeID = pCmd->ZW_PowerlevelTestNodeSetFrame.testNodeid;
//        testPowerLevel = pCmd->ZW_PowerlevelTestNodeSetFrame.powerLevel;
//        testFrameCount = (WORD)((WORD)pCmd->ZW_PowerlevelTestNodeSetFrame.testFrameCount1 << 8) + pCmd->ZW_PowerlevelTestNodeSetFrame.testFrameCount1;
//        testFrameSuccessCount = 0;
//		ZW_MEM_PUT_BYTE(EEOFFSET_TEST_NODE_ID,testNodeID);
//		ZW_MEM_PUT_BYTE(EEOFFSET_TEST_POWER_LEVEL,testPowerLevel);
//		ZW_MEM_PUT_BYTE(EEOFFSET_TEST_FRAME_COUNT,testFrameCount);
//		ZW_MEM_PUT_BYTE(EEOFFSET_TEST_SOURCE_NODE_ID,testSourceNodeID);
//
//        if (testFrameCount)
//        {
//          StartTest();
//        }
//        else
//        {
//          testState = ZW_TEST_FAILED;
//          SendTestReport();
//        }
//      }
//      break;
//    }
//
//    case POWERLEVEL_TEST_NODE_GET:
//    {
//      SendTestReport();
//      break;
//    }
//    }
//    break;
//  }
#endif
  }
}

/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/

/*============================   ApplicationInitHW   ========================
**    Non Z-Wave hardware initialization
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/

BYTE                       /*RET  TRUE        */
PATCH_FUNCTION_NAME_STARTER(ApplicationInitHW)(
  BYTE bStatus
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef APPL_PROD_TEST
  BYTE i;
#endif

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationInitHW)
#pragma endasm
#endif
#ifdef ZW_ISD51_DEBUG
  ZW_UART1_init(1152, TRUE, TRUE);
#endif
#ifdef APPL_PROD_TEST
  SET_PRODUCTIONTEST_PIN;
  for (i = 0; i < 10; i++) ;  /* Short delay... */
  if (IN_PRODUCTIONTEST) /* Anyone forcing it into productiontest mode ? */
  {
    return(FALSE);  /*Enter production test mode*/
  }
#endif /* APPL_PROD_TEST */


#ifdef FLIRS
   EX1 = 0;
#endif


  /* hardware initialization */
  PIN_IN(Button, 1);
  wakeUpReason = bStatus;
  /*triac pin init*/
#ifdef LOW_FOR_ON
	PIN_ON(LED3);	/* HIGH for Relay OFF */
#endif
  PIN_OUT(LED3); /* Relay output */

  Transport_OnApplicationInitHW(bStatus);

  return(TRUE);
}

static void FuncZWSecure(BYTE txStatus)
{
	if (txStatus == TRANSPORT_WORK_END)
	{
#ifdef FLIRS
     FLiRS_SleepTimeoutStart(FALSE);
#endif
	}
	else if (txStatus == TRANSPORT_WORK_START)
	{
#ifdef FLIRS
		FLiRS_SleepTimeoutStop();
#endif
	}
	else if (txStatus == TRANSPORT_WORK_ERROR)
	{
#ifdef FLIRS
     FLiRS_SleepTimeoutStart(FALSE);
#endif
	}
}

/*===========================   ApplicationInitSW   =========================
**    Initialization of the Application Software
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
BYTE                      /*RET  TRUE       */
PATCH_FUNCTION_NAME_STARTER(ApplicationInitSW)(void) /* IN   Nothing   */
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationInitSW)
#pragma endasm
#endif
  /* get stored values */
  if (ZW_MEM_GET_BYTE(EEOFFSET_MAGIC) == MAGIC_VALUE)
  {
    LoadStatus();
  }
  else
  { /* Its alive */
    lockNodeStatus = DOOR_LOCK_OPERATION_REPORT_DOOR_UNSECURED;
    SaveStatus(); /* Now EEPROM should be OK */
	timerPowerLevel = 0;
	timerPowerLevelSec = 0;
	testNodeID = 0;
	testPowerLevel = 0;
	testFrameCount = 0;
	testSourceNodeID = 0;
	ZW_MEM_PUT_BYTE(EEOFFSET_POWER_LEVEL,timerPowerLevel);
	ZW_MEM_PUT_BYTE(EEOFFSET_TIMER_POWER_LEVEL,timerPowerLevelSec);
	ZW_MEM_PUT_BYTE(EEOFFSET_TEST_NODE_ID,testNodeID);
	ZW_MEM_PUT_BYTE(EEOFFSET_TEST_POWER_LEVEL,testPowerLevel);
	ZW_MEM_PUT_BYTE(EEOFFSET_TEST_FRAME_COUNT,testFrameCount);
	ZW_MEM_PUT_BYTE(EEOFFSET_TEST_SOURCE_NODE_ID,testSourceNodeID);

  }

/* Do not reinitialize the UART if already initialized for ISD51 in ApplicationInitHW() */
#ifndef ZW_ISD51_DEBUG
//  ZW_DEBUG_CMD_INIT(1152);
  ZW_DEBUG_INIT(1152);
  ZW_DEBUG_SEND_BYTE('V');
#endif

  OneButtonInit();
#ifdef PATCH_ENABLE
  SlaveLearnInit();
#endif

  /* Check if we have a node id */
  MemoryGetID(NULL, &bMyNodeID);

  /* Set Network wide inclusion active if we don't have a node ID */
  if (bMyNodeID)
    bNWIStartup = FALSE;
  else if (wakeUpReason == ZW_WAKEUP_RESET)
  {
    	bNWIStartup = TRUE;
  }
  else if (wakeUpReason == ZW_WAKEUP_POR)
  {
	 bNWIStartup = TRUE;
  }


  if (!ZW_MEM_GET_BYTE(EEOFFSET_LERN_MODE))
      bNWIStartup = TRUE;

  ZW_DEBUG_SEND_NUM(ZW_MEM_GET_BYTE(EEOFFSET_LERN_MODE));

  UpdateLock();

  Transport_OnApplicationInitSW(nodeInfoForTransport, sizeof(nodeInfoForTransport),
    FALSE, EEOFFSET_TRANSPORT_SETTINGS_START, EEOFFSET_TRANSPORT_SETTINGS_SIZE,
    FuncZWSecure);

  return(TRUE);
}

/*============================   ApplicationTestPoll   ======================
**    Function description
**      This function is called when the dimmer enters test mode.
**    Side effects:
**       Code will not exit until it is reset
**--------------------------------------------------------------------------*/
void PATCH_FUNCTION_NAME_STARTER(ApplicationTestPoll)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationTestPoll)
#pragma endasm
#endif
#ifdef APPL_PROD_TEST
  /*Send constant out*/
  ZW_SEND_CONST();
  while(1);
#endif
}

/*======================   ApplicationNodeInformation   =====================
**    Request Node information and current status
**    Called by the the Z-Wave application layer before transmitting a
**    "Node Information" frame.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
extern void                  /*RET  Nothing */
PATCH_FUNCTION_NAME_STARTER(ApplicationNodeInformation)(
  BYTE   *deviceOptionsMask,      /*OUT Bitmask with application options     */
  APPL_NODE_TYPE  *nodeType,  /*OUT  Device type Generic and Specific   */
  BYTE       **nodeParm,      /*OUT  Device parameter buffer pointer    */
  BYTE       *parmLength      /*OUT  Number of Device parameter bytes   */
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationNodeInformation)
#pragma endasm
#endif
  /* this is a listening node and it supports optional CommandClasses */
  *deviceOptionsMask = APPLICATION_NODEINFO_LISTENING | APPLICATION_NODEINFO_OPTIONAL_FUNCTIONALITY;
#ifdef FLIRS
  FLIRS_DEVICE_OPTIONS_MASK_MODIFY(*deviceOptionsMask);
#endif

  nodeType->generic = GENERIC_TYPE_ENTRY_CONTROL; /* Generic device type */

#ifdef ZW_ADVANCED_DOOR_LOCK
  nodeType->specific = SPECIFIC_TYPE_ADVANCED_DOOR_LOCK; //SPECIFIC_TYPE_DOOR_LOCK;   /* Specific class */
#else
  nodeType->specific = SPECIFIC_TYPE_DOOR_LOCK;   /* Specific class */
#endif

  *nodeParm = nodeInfo;                           /* Send list of known command classes. */
  *parmLength = sizeof(nodeInfo);                 /* Set length*/

#ifdef SECURITY
  MemoryGetID(myHomeID, &myNodeID);

#ifndef FLIRS
  nodeSecure = GetNodeSecure(EEOFFSET_TRANSPORT_SETTINGS_START);
#endif

  if (myNodeID)
	if (nodeSecure)
	{
  		*nodeParm = nodeInfoAfterIncluded;        /* Send list of known command classes. */
  		*parmLength = sizeof(nodeInfoAfterIncluded);       /* Set length*/
	}
#endif
}

#ifdef ZW_SLAVE
/*==========================   ApplictionSlaveUpdate   =======================
**   Inform a slave application that a node information is received.
**   Called from the slave command handler when a node information frame
**   is received and the Z-Wave protocol is not in a state where it is needed.
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME_STARTER(ApplicationSlaveUpdate)(
  BYTE bStatus,     /*IN  Status event */
  BYTE bNodeID,     /*IN  Node id of the node that send node info */
  BYTE* pCmd,       /*IN  Pointer to Application Node information */
  BYTE bLen       /*IN  Node info length                        */
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationSlaveUpdate)
#pragma endasm
#endif
}
#endif


#ifdef ZW_CONTROLLER
/*=====================   ApplicationControllerUpdate   =====================
**   Inform the controller of node information update done through
**   the network managment.
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME_STARTER(ApplicationControllerUpdate)(
  BYTE bStatus,     /*IN  Status event */
  BYTE bNodeID,     /*IN  Node id of the node that send node info */
  BYTE* pCmd,       /*IN  Pointer to Application Node information */
  BYTE bLen         /*IN  Node info length                        */
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationControllerUpdate)
#pragma endasm
#endif
}
#endif


/*============================   LearnCompleted   ========================
**    Function description
**      Called from learnmode.c when learnmode completed
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void									/*RET	Nothing */
PATCH_FUNCTION_NAME(LearnCompleted)(
  BYTE nodeID					/* IN resulting nodeID */
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(LearnCompleted)
#pragma endasm
#endif
  if (nodeID == 0)		/* Was it reset? */
  {
	 ZW_MEM_PUT_BYTE(EEOFFSET_LERN_MODE,FALSE);

	 ZW_DEBUG_SEND_BYTE('x');
	 ZW_DEBUG_SEND_NUM(ZW_MEM_GET_BYTE(EEOFFSET_LERN_MODE));
  }
#ifdef DEBUG_STRESS
  if (!debugStress)
  {
    debugStress = TRUE;
    debugStressHandle = ZW_TIMER_START(SendBasicSet, DEBUG_STRESS_TIMEOUT_START, ONE_SHOT);
  }
#endif


#ifdef FLIRS
  if (nodeID != 0)		/* Was it reset? */
  {
#ifdef SECURITY

      FLiRS_SleepTimeoutStop();
	  ZW_DEBUG_SEND_BYTE('z');
	  ZW_DEBUG_SEND_BYTE('z');

#else
     FLiRS_SleepTimeoutStart(FALSE);
#endif
  }
#endif


  Transport_OnLearnCompleted(nodeID);

 	  ZW_DEBUG_SEND_BYTE(' ');
 	  ZW_DEBUG_SEND_BYTE('E');
	  ZW_DEBUG_SEND_BYTE(' ');

}


/* Do not use absolut register adressing */
#pragma NOAREGS
/*==============================   ApplicationRfNotify   ======================
**    Function used when using an external PA
**
**--------------------------------------------------------------------------*/
void                   /*RET TRUE - If normal operation, FALSE - if production test*/
PATCH_FUNCTION_NAME_STARTER(ApplicationRfNotify)(
  BYTE bStatus)        /* IN RF status */
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationRfNotify)
#pragma endasm
#endif
}

