/*******************************  bell.C  ********************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2001
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 * Copyright Zensys A/S, 2001
 *
 * Description: Slave application for doorbell test application
 *
 * Author:   Peter Shorty
 *
 * Last Changed By:  $Author: psh $
 * Revision:         $Revision: 22663 $
 * Last Changed:     $Date: 2012-05-02 17:22:55 +0200 (Wed, 02 May 2012) $
 *
 ****************************************************************************/

#if defined(WORK_PATCH) || defined(STARTER_PATCH)
#define ZW_DEBUG
#endif

#ifdef PATCH_ENABLE
/****************************************************************************/
/* Include assembly MACRO definitions for patch insertions.                 */
/*                                                                          */
/* Define $SET (MAKE_PATCHABLE_CODE) for making patchable code destinned    */
/* for OTP or ROM memory.                                                   */
/* Undefine $RESET (MAKE_PATCHABLE_CODE) for making code containing patch   */
/* code destinned for RAM or FLASH memory.                                  */
/****************************************************************************/
#if defined(WORK_PATCH) || defined(STARTER_PATCH)
/* Making code containing patch code destinned for development RAM memory.  */
#pragma asm
$RESET (MAKE_PATCHABLE_CODE)
$INCLUDE (ZW_patch.inc)
#pragma endasm
/* Rename CODE class to CODE_PATCH */
#pragma userclass (code = PATCH)
/* Rename CONST class to CONST_PATCH */
#pragma userclass (const = PATCH)
/* Rename XDATA class to XDATA_PATCH */
#pragma userclass (xdata = PATCH)
#else
/* Making patchable code destinned for OTP or ROM memory.                   */
#pragma asm
$SET (MAKE_PATCHABLE_CODE)
$INCLUDE (ZW_patch.inc)
#pragma endasm
#endif /* elsif defined(WORK_PATCH) || defined(STARTER_PATCH) */
#endif /* PATCH_ENABLE */

/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/

#include <config_app.h>
#include <ZW_patch.h>

#define ZW_BEAM_RX_WAKEUP

#include <ZW_slave_api.h>
#include <ZW_pindefs.h>
#include <ZW_evaldefs.h>
#include <bell.h>
#include <eeprom_slave.h>
#include <slave_learn.h>
#include <ZW_uart_api.h>
#include <ZW_power_api.h>
#include <ZW_transport_api.h>
#include <one_button.h>

#include <ZW_TransportLayer.h>

#ifdef ZW_ISD51_DEBUG
#include "ISD51.h"
#endif

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/

#define LONG_SLEEP_TIMEOUT  200

/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/


/* A list of the known command classes. Except the basic class which allways */
/* should be supported. Used when node info is send */
static code t_nodeInfo nodeInfo = {COMMAND_CLASS_SWITCH_BINARY,
                       COMMAND_CLASS_VERSION};

PATCH_VARIABLE_STARTER ZW_APPLICATION_TX_BUFFER txBuf;
PATCH_VARIABLE BYTE txOption;

PATCH_VARIABLE BYTE  bMyNodeID;

#define STATE_IDLE                  0
#define STATE_WAITING_FOR_ADD       1
#define STATE_ADDING_DONE           2
#define STATE_POWER_UP              4
#define STATE_WAITING_FOR_TIMEOUT   5


PATCH_VARIABLE BYTE bMainState
#ifndef WORK_PATCH
 = STATE_IDLE
#endif
;

/* Buffer used to hold commands */
PATCH_VARIABLE ZW_APPLICATION_TX_BUFFER txBuffer;

PATCH_VARIABLE BYTE bSleepTimeout; /* Timer handler for sleep timer */
PATCH_VARIABLE BYTE bSleepTimeoutCount;

PATCH_VARIABLE BYTE bWakeupReason;  /* Reason for waking up */

PATCH_VARIABLE BYTE bBellStatus;

PATCH_VARIABLE BYTE bNWIStartup;

#ifdef TIMING_DEBUG
#define TXDEBUGPin      6
#define TXDEBUGPinPort  P1
#define TXDEBUGPinDDR   P1DIR
#endif

/****************************************************************************/
/*                              EXPORTED DATA                               */
/****************************************************************************/


/****************************************************************************/
/*                            PRIVATE FUNCTIONS                             */
/****************************************************************************/

/*=============================  SleepTimeout   =========================
**    Timeout function for going back to sleep
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(SleepTimeout)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(SleepTimeout)
#pragma endasm
#endif
  ZW_DEBUG_SEND_BYTE('T');
  StopSleepTimeout();
  GotoSleep();
}

/*=============================  StopSleepTimeout   =========================
**    Stop the timeout timer for going back to sleep
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(StopSleepTimeout)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StopSleepTimeout)
#pragma endasm
#endif
  if (bSleepTimeout)
  {
    TimerCancel(bSleepTimeout);
  }
  bSleepTimeout = 0;
}



/*=============================  StartSleepTimeout   =========================
**    Start the timeout timer for going back to sleep
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(StartSleepTimeout)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StartSleepTimeout)
#pragma endasm
#endif
  ZW_DEBUG_SEND_BYTE('S');

  if (!bSleepTimeout)
  {
    bSleepTimeout = TimerStart(SleepTimeout, LONG_SLEEP_TIMEOUT, 1);
  }
  else
  {
    TimerRestart(bSleepTimeout);
  }
}


/*=============================  ApplicationPoll   =========================
**    Application poll function for LED dimmer
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                    /*RET  Nothing                  */
PATCH_FUNCTION_NAME_STARTER(ApplicationPoll)(void) /* IN  Nothing                  */
#ifdef PATCH_ENABLE
reentrant
#endif
{
  register BYTE bStatus;

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationPoll)
#pragma endasm
#endif
#ifdef ZW_ISD51_DEBUG         /* init ISD51 only when the uVision2 Debugger tries to connect */
    ISDcheck();               /* initialize uVision2 Debugger and continue program run */
#endif
  bStatus = OneButtonLastAction();
  if (bNWIStartup)
  {
    StartLearnModeNow(ZW_SET_LEARN_MODE_NWI);
    bNWIStartup = FALSE;
    bMainState = STATE_WAITING_FOR_ADD;
  }
  switch (bMainState)
  {
    case STATE_POWER_UP:
      bBellStatus = 0;
      bSleepTimeout = 0;
      bMainState = STATE_IDLE;
      /* Turn on the receiver */
      ZW_SetRFReceiveMode(TRUE);
#ifdef TIMING_DEBUG
      PIN_ON(TXDEBUGPin);

      ZW_DEBUG_SEND_BYTE('W');
      ZW_DEBUG_SEND_NUM(bWakeupReason);
#endif
      /* Start sleep timeout */
      StartSleepTimeout();
      if (bWakeupReason == ZW_WAKEUP_RESET)
      {
        LED_ON(1);
      }
      else
      {
        LED_ON(2);
      }
      break;

    case STATE_IDLE:   /* Idle state check the button state */

      /* Check for button state */
      if (bStatus == BUTTON_WAS_PRESSED)
      {
        ZW_DEBUG_SEND_BYTE('p');
        // ZW_SendNodeInformation(NODE_BROADCAST, TRANSMIT_OPTION_LOW_POWER, NodeInfoSendComplete);
        bMainState = STATE_WAITING_FOR_TIMEOUT;
        ZW_MEM_PUT_BYTE(EEOFFSET_LERN_MODE, TRUE);
      }
      else if (bStatus == BUTTON_TRIPLE_PRESS)
      {
        LED_ON(3);
        ZW_DEBUG_SEND_BYTE('h');
        StartLearnModeNow(ZW_SET_LEARN_MODE_CLASSIC);
        bMainState = STATE_WAITING_FOR_ADD;
        StartSleepTimeout();
      }
      break;

    case STATE_WAITING_FOR_ADD:
      if (learnState == FALSE)
      {
        bMainState = STATE_ADDING_DONE;
      }
      else /* Stop sleep timeout while in learn mode */
      {
        StopSleepTimeout();

        if (bStatus == BUTTON_WAS_PRESSED)
        {
          /* Stop learn mode */
          StartLearnModeNow(FALSE);
          bMainState = STATE_ADDING_DONE;
        }
      }
      break;

    case STATE_ADDING_DONE:
      StartSleepTimeout();
      bMainState = STATE_IDLE;
      break;

    case STATE_WAITING_FOR_TIMEOUT:
      StartSleepTimeout();
      bMainState = STATE_IDLE;
      break;
  }
}


/*========================   ApplicationCommandHandler   ====================
**    Handling of a received application commands and requests
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void                              /*RET Nothing                  */
PATCH_FUNCTION_NAME_STARTER(ApplicationCommandHandler)(
  BYTE  rxStatus,                 /* IN Frame header info */
  BYTE  sourceNode,               /* IN Command sender Node ID */
  ZW_APPLICATION_TX_BUFFER *pCmd, /* IN Payload from the received frame, the union */
                                  /*    should be used to access the fields */
  BYTE   cmdLength                /* IN Number of command bytes including the command */
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
  register BYTE bTmpCount;
  BYTE classcmd;
  BYTE cmd;

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationCommandHandler)
#pragma endasm
#endif
  classcmd = pCmd->ZW_Common.cmdClass;
  cmd = pCmd->ZW_Common.cmd;
 	txOption = ((rxStatus & RECEIVE_STATUS_LOW_POWER) ?
 	            TRANSMIT_OPTION_LOW_POWER : 0) | TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE;
  if ((classcmd == COMMAND_CLASS_SWITCH_BINARY) ||
      (classcmd == COMMAND_CLASS_BASIC))
  {
    /* Single switch command */
    switch(cmd)
    {

      /* Controller wants the level */
      case SWITCH_BINARY_GET:
        StopSleepTimeout();
	  bBellStatus = ZW_MEM_GET_BYTE(EEOFFSET_LEVEL);
        SendReport(sourceNode, classcmd, bBellStatus);
        break;

      /* Jump to the specified level param1 */
      case SWITCH_BINARY_SET:
	  {
		// TO# 03114  Door Bell does not ignore invalid values in Switch Binary Set
		BYTE val = pCmd->ZW_SwitchBinarySetFrame.switchValue;
		if ((val > 0x63)&(val<0xff)) break;

		// TO# 03009  If Door Bell device receives a Binary Switch Set Command with values from 1 to 99 (0x01  0x63), those values shall mapped to 0xFF upon receipt of the Command in the device.
		bBellStatus = pCmd->ZW_SwitchBinarySetFrame.switchValue;
		if (bBellStatus)
        	bBellStatus = 0xff;/*pCmd->ZW_SwitchBinarySetFrame.switchValue; */
		else
			bBellStatus = 0x00;
		if (bBellStatus)
		{
	        LED_ON(1);
	        LED_ON(2);
	        LED_ON(3);
		}
		else
		{
	        LED_OFF(1);
	        LED_OFF(2);
	        LED_OFF(3);
		}
	  ZW_MEM_PUT_BYTE(EEOFFSET_LEVEL, bBellStatus);
	  }
        break;
    } /* switch */
  } /* if */
   /* else if == COMMAND_CLASS_SWITCH_ALL */

  else if (classcmd == COMMAND_CLASS_VERSION)
  {
    switch (cmd)
    {
      case VERSION_GET:
        txBuf.ZW_VersionReportFrame.cmdClass = COMMAND_CLASS_VERSION;
        txBuf.ZW_VersionReportFrame.cmd = VERSION_REPORT;
        txBuf.ZW_VersionReportFrame.zWaveLibraryType = ZW_TYPE_LIBRARY();

        txBuf.ZW_VersionReportFrame.zWaveProtocolVersion = ZW_VERSION_MAJOR;

        txBuf.ZW_VersionReportFrame.zWaveProtocolSubVersion = ZW_VERSION_MINOR;
        txBuf.ZW_VersionReportFrame.applicationVersion = APP_VERSION;
        txBuf.ZW_VersionReportFrame.applicationSubVersion = APP_REVISION;
        ZW_SEND_DATA(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_VersionReportFrame), txOption, NULL);
        break;

      case VERSION_COMMAND_CLASS_GET:
        txBuf.ZW_VersionCommandClassReportFrame.cmdClass = COMMAND_CLASS_VERSION;
        txBuf.ZW_VersionCommandClassReportFrame.cmd = VERSION_COMMAND_CLASS_REPORT;
        txBuf.ZW_VersionCommandClassReportFrame.requestedCommandClass = pCmd->ZW_VersionCommandClassGetFrame.requestedCommandClass;
        switch (pCmd->ZW_VersionCommandClassGetFrame.requestedCommandClass)
        {
          case COMMAND_CLASS_BASIC:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = BASIC_VERSION;
            break;

          case COMMAND_CLASS_SWITCH_BINARY:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = SWITCH_BINARY_VERSION;
            break;

          case COMMAND_CLASS_VERSION:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = VERSION_VERSION;
            break;

          default:
            txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = UNKNOWN_VERSION;
            break;
        }
        ZW_SendData(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_VersionCommandClassReportFrame), txOption, NULL);
        break;
    }
  }
}

/*==============================   TxDone   ==============================
**
**  Function:  Transmit of a frame done, start the powerdown timer
**
**    Sends a report frame to source node.
**
**  Side effects: None
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(TxDone)(
  BYTE bStatus
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(TxDone)
#pragma endasm
#endif
  StartSleepTimeout();
}

/*==============================   SendReport   ==============================
**
**  Function:  SendReport
**
**    Sends a report frame to source node.
**
**  Side effects: None
**
**--------------------------------------------------------------------------*/
void                /*RET Nothing */
PATCH_FUNCTION_NAME(SendReport)(
  BYTE sourceNode,  /* IN Source node - to whom report is to be send */
  BYTE classcmd,    /* IN Command class, which the report belongs to */
  BYTE reportparam  /* IN paramter to include in report frame */
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(SendReport)
#pragma endasm
#endif
  txBuf.ZW_BasicReportFrame.cmdClass = classcmd;
/* Report command definition - BASIC/ALL/PROTECTION/MULTI report use the same value ;-) */
  txBuf.ZW_BasicReportFrame.cmd      = BASIC_REPORT;
  txBuf.ZW_BasicReportFrame.value    = reportparam;
/* Size of Report (BASIC/ALL/PROTECTION/MULTI GET response) frame are equal for BASIC/ALL/PROTECTION/MULTI */
  ZW_SendData(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_BasicReportFrame), txOption, TxDone);
}


/*===========================   NodeInfoSendComplete   =======================
**    Start retrieving meta data
**
**
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(NodeInfoSendComplete)(
  BYTE bStatus
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(NodeInfoSendComplete)
#pragma endasm
#endif

}

/*================================   GotoSleep   ============================
**    Power down the ZW0201 chip
**
**
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(GotoSleep)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(GotoSleep)
#pragma endasm
#endif
  ZW_DEBUG_SEND_BYTE('G');

  LED_OFF(1);
  LED_OFF(2);
  LED_OFF(3);

  PIN_IN(LED1, 0);
  PIN_IN(LED2, 0);  /* Now tristate */
  PIN_IN(LED3, 0);  /* Now tristate */

  /* TO#1996 - Dont enable external int before powerdown */
  // EX1 = 1;  /* enable int1 (button) before power down */

  /* Goto sleep */
#ifdef TIMING_DEBUG
  PIN_OFF(TXDEBUGPin);
#endif

  /* ZW_SetSleepMode can fail now, handle it so application doesnt
     get confused */
  if (!ZW_SetSleepMode(ZW_FREQUENTLY_LISTENING_MODE, ZW_INT_MASK_EXT1, 0))
  {
    StartSleepTimeout();
  }
}


/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/

/*============================   ApplicationInitHW   ========================
**    Non Z-Wave hardware initialization
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
BYTE                       /*RET  TRUE        */
PATCH_FUNCTION_NAME_STARTER(ApplicationInitHW)(
  BYTE bReason             /* IN  Nothing     */
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationInitHW)
#pragma endasm
#endif
#ifdef ZW_ISD51_DEBUG
  ZW_UART1_init(1152, TRUE, TRUE);
#endif
  /* Disable external int 1 */
  EX1 = 0;

  /* hardware initialization */
  PIN_IN(Button, 1);

  /* Setup specifics for current dim module */
  PIN_OUT(LED1);
  PIN_OUT(LED2);
  PIN_OUT(LED3);

#ifdef  TIMING_DEBUG
  PIN_OFF(TXDEBUGPin);
  PIN_OUT(TXDEBUGPin);
#endif

  bWakeupReason = bReason;

  return(TRUE);
}


/*===========================   ApplicationInitSW   =========================
**    Initialization of the Application Software
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
BYTE                      /*RET  TRUE       */
PATCH_FUNCTION_NAME_STARTER(ApplicationInitSW)(void) /* IN   Nothing   */
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationInitSW)
#pragma endasm
#endif
/* Do not reinitialize the UART if already initialized for ISD51 in ApplicationInitHW() */
#ifndef ZW_ISD51_DEBUG
  ZW_DEBUG_INIT(1152);

  ZW_DEBUG_SEND_NL();
  ZW_DEBUG_SEND_BYTE('B');
  ZW_DEBUG_SEND_BYTE('e');
  ZW_DEBUG_SEND_BYTE('l');
  ZW_DEBUG_SEND_BYTE('l');
  ZW_DEBUG_SEND_BYTE(' ');
#endif

#ifdef PATCH_ENABLE
  SlaveLearnInit();
#endif

  /* Initialize button detection */
  OneButtonInit();

  /* Turn off all LED's */
  LED_OFF(1);
  LED_OFF(2);
  LED_OFF(3);

  /* Initialize flash if necessary */
  // if (MemoryGetByte(NVM_ASSOCIATION) == 0xFF)
  //{
  //  ZW_DEBUG_SEND_BYTE('X');
  //  MemoryPutBuffer(NVM_ASSOCIATION, NULL, sizeof(groups), NULL);
  //}
  //else
  //  AssociationGetAll();
  /* Get our node ID */
  MemoryGetID(NULL, &bMyNodeID);
  /* Set Network wide inclusion active if we dont have a node ID */
  if (bMyNodeID)
  {
    /* We allready have a nodeID. */
    bNWIStartup = FALSE;
  }
  else if (bWakeupReason == ZW_WAKEUP_RESET)
  {
    bNWIStartup = TRUE;
  }
  else if (bWakeupReason == ZW_WAKEUP_POR)
  {
    bNWIStartup = TRUE;
  }

  if (!ZW_MEM_GET_BYTE(EEOFFSET_LERN_MODE))
  {
    bNWIStartup = TRUE;
  }
  bMainState = STATE_POWER_UP;
  return(TRUE);
}


/*============================   ApplicationTestPoll   ======================
**    Function description
**      This function is called when the dimmer enters test mode.
**    Side effects:
**       Code will not exit until it is reset
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME_STARTER(ApplicationTestPoll)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationTestPoll)
#pragma endasm
#endif

}


/*======================   ApplicationNodeInformation   =====================
**    Request Node information and current status
**    Called by the the Z-Wave application layer before transmitting a
**    "Node Information" frame.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
void                  /*RET  Nothing */
PATCH_FUNCTION_NAME_STARTER(ApplicationNodeInformation)(
  BYTE       *listening,      /*OUT  TRUE if this node is always on air */
  APPL_NODE_TYPE  *nodeType,  /*OUT  Device type Generic and Specific   */
  BYTE       **nodeParm,      /*OUT  Device parameter buffer pointer    */
  BYTE       *parmLength      /*OUT  Number of Device parameter bytes   */
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationNodeInformation)
#pragma endasm
#endif
#ifdef ZW_3CH_SYSTEM
  *listening = APPLICATION_NODEINFO_NOT_LISTENING | APPLICATION_FREQ_LISTENING_MODE_1000ms;
#else
  *listening = APPLICATION_NODEINFO_NOT_LISTENING | APPLICATION_FREQ_LISTENING_MODE_250ms;
#endif
  nodeType->generic = GENERIC_TYPE_SWITCH_BINARY; /* Generic device type */
  nodeType->specific = SPECIFIC_TYPE_NOT_USED; /* Specific class */
  *nodeParm = (BYTE *)&nodeInfo;        /* Send list of known command classes. */
  *parmLength = sizeof(nodeInfo);       /* Set length*/
}


/*==========================   ApplicationSlaveUpdate   ======================
**   Inform a slave application that a node information is received.
**   Called from the slave command handler when a node information frame
**   is received and the Z-Wave protocol is not in a state where it is needed.
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME_STARTER(ApplicationSlaveUpdate)(
  BYTE bStatus,     /*IN  Status event */
  BYTE bNodeID,     /*IN  Node id of the node that send node info */
  BYTE* pCmd,       /*IN  Pointer to Application Node information */
  BYTE bLen         /*IN  Node info length                        */
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationSlaveUpdate)
#pragma endasm
#endif
}

/*============================   LearnCompleted   ========================
**    Function description
**      Called from learnmode.c when learnmode completed
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void									/*RET	Nothing */
PATCH_FUNCTION_NAME(LearnCompleted)(
  BYTE nodeID 					/* IN resulting nodeID */
)
#ifdef PATCH_ENABLE
 reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(LearnCompleted)
#pragma endasm
#endif
  ZW_DEBUG_SEND_BYTE('C');
  bMainState = STATE_ADDING_DONE;
  bMyNodeID = nodeID;
  if (nodeID == 0)    /* Was it reset? */
  {
    ZW_MEM_PUT_BYTE(EEOFFSET_LERN_MODE, FALSE);
  }
}

/* Do not use absolut register adressing */
#pragma NOAREGS
/*==============================   ApplicationRfNotify   ======================
**    Function used when using an external PA
**
**--------------------------------------------------------------------------*/
void                   /*RET TRUE - If normal operation, FALSE - if production test*/
PATCH_FUNCTION_NAME_STARTER(ApplicationRfNotify)(
  BYTE bStatus)        /* IN RF status */
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationRfNotify)
#pragma endasm
#endif
}

