/******************************* Bin_Sensor.c *******************************
 *           #######
 *           ##  ##
 *           #  ##    ####   #####    #####  ##  ##   #####
 *             ##    ##  ##  ##  ##  ##      ##  ##  ##
 *            ##  #  ######  ##  ##   ####   ##  ##   ####
 *           ##  ##  ##      ##  ##      ##   #####      ##
 *          #######   ####   ##  ##  #####       ##  #####
 *                                           #####
 *          Z-Wave, the wireless language.
 *
 *              Copyright (c) 2006
 *              Zensys A/S
 *              Denmark
 *
 *              All Rights Reserved
 *
 *    This source file is subject to the terms and conditions of the
 *    Zensys Software License Agreement which restricts the manner
 *    in which it may be used.
 *
 *---------------------------------------------------------------------------
 *
 * Description:
 *
 * The purpose of this sample code is to demonstrate how a battery powered
 * sensor could be programmed.
 *
 * This sensor supports the following features at the present time:
 *  - Support for battery operation:
 *   - WUT / RTC Wakeup (200 / 100 series)
 *   - WakeUpNotification at wakeup to master node if set
 *  - KeepAlive Capability if button is held down
 *  - Supports ASSOCIATION command class
 *  - Supports Self Healing (Lost/Rediscovery)
 *
 *
 * Author:   Thomas Kristensen
 *
 * Last Changed By:  $Author: vvi $
 * Revision:         $Revision: 10290 $
 * Last Changed:     $Date: 2008-05-12 17:51:24 +0200 (Mon, 12 May 2008) $
 *
 ****************************************************************************/

#if defined(WORK_PATCH) || defined(STARTER_PATCH)
#define ZW_DEBUG
#endif

#ifdef PATCH_ENABLE
/****************************************************************************/
/* Include assembly MACRO definitions for patch insertions.                 */
/*                                                                          */
/* Define $SET (MAKE_PATCHABLE_CODE) for making patchable code destinned    */
/* for OTP or ROM memory.                                                   */
/* Undefine $RESET (MAKE_PATCHABLE_CODE) for making code containing patch   */
/* code destinned for RAM or FLASH memory.                                  */
/****************************************************************************/
#if defined(WORK_PATCH) || defined(STARTER_PATCH)
/* Making code containing patch code destinned for development RAM memory.  */
#pragma asm
$RESET (MAKE_PATCHABLE_CODE)
$INCLUDE (ZW_patch.inc)
#pragma endasm
/* Rename CODE class to CODE_PATCH */
#pragma userclass (code = PATCH)
/* Rename CONST class to CONST_PATCH */
#pragma userclass (const = PATCH)
/* Rename XDATA class to XDATA_PATCH */
#pragma userclass (xdata = PATCH)
#else
/* Making patchable code destinned for OTP or ROM memory.                   */
#pragma asm
$SET (MAKE_PATCHABLE_CODE)
$INCLUDE (ZW_patch.inc)
#pragma endasm
#endif /* elsif defined(WORK_PATCH) || defined(STARTER_PATCH) */
#endif /* PATCH_ENABLE */

/****************************************************************************/
/*                              INCLUDE FILES                               */
/****************************************************************************/

#include <config_app.h>
#include <ZW_patch.h>

/* Enhanced Slave - needed for battery operation (RTC timer) on 100 series */
/* 200 Series have WUT */
#ifdef ZW_SLAVE_32
  #include <ZW_slave_32_api.h>
#else
  #ifdef  ZW_SLAVE
    #include <ZW_slave_api.h>
  #endif
#endif

#ifdef ZW_SELF_HEAL
  /* Lost functionality */
  #include <self_heal.h>
#endif

/* BATTERY defines if the node is a Battery Binary Sensor or just a Binary Sensor */
#ifdef BATTERY
  /* ASIC power management functionality */
  #include <ZW_power_api.h>

  /* Support functions for battery operation */
  #include <battery.h>
#endif

/* Z-Wave libraries */
#include <ZW_sysdefs.h>
#include <ZW_pindefs.h>
#include <ZW_evaldefs.h>
#include <ZW_classcmd.h>
#include <ZW_uart_api.h>



/* Allows data storage of application data even after reset */
#include <ZW_non_zero.h>


#include <Bin_Sensor.h>
#include <eeprom.h>
#include <slave_learn.h>

/* Support functions for association */
#include <association.h>
/* Support functions for button press detection */
#include <one_button.h>

#include <ZW_TransportLayer.h>

#ifdef ZW_ISD51_DEBUG
#include "ISD51.h"
#endif

/****************************************************************************/
/*                      PRIVATE TYPES and DEFINITIONS                       */
/****************************************************************************/

#define BUTTON_KEEPALIVECOUNT     0xFE  /* x 10msec 2seconds for KeepAlive */

PATCH_VARIABLE BYTE currentGroupIndex;
PATCH_VARIABLE BYTE currentGroupNodeIndex;
PATCH_VARIABLE BYTE frameSize;



PATCH_VARIABLE BYTE stateWakeUPTimerHandle
#ifndef WORK_PATCH
 = 0xFF
#endif
;


PATCH_VARIABLE BYTE stateWatchdogTimerHandle
#ifndef WORK_PATCH
 = 0xFF
#endif
;
PATCH_VARIABLE BYTE stateWatchdogTimeout
#ifndef WORK_PATCH
 = WATCHDOG_TIMEOUT
#endif
;

PATCH_VARIABLE BYTE myNodeID
#ifndef WORK_PATCH
 = 0
#endif
;
PATCH_VARIABLE BYTE myHomeID[4];
PATCH_VARIABLE BYTE currentState
#ifndef WORK_PATCH
 = STATE_APPL_IDLE
#endif
;
PATCH_VARIABLE BYTE nextState
#ifndef WORK_PATCH
 = STATE_APPL_IDLE
#endif
;
PATCH_VARIABLE BYTE watchState
#ifndef WORK_PATCH
 = STATE_APPL_IDLE
#endif
;

PATCH_VARIABLE BOOL keepAliveActive
#ifndef WORK_PATCH
 = FALSE
#endif
;

/* Next value to send in the BasicSet */
PATCH_VARIABLE BYTE toggleBasicSet;

/* MasterNodeID specifies the node the sensor will try to contact on wakeup */
/* and when lost. Defined extern, currenly used by "battery" and "self-heal"*/
PATCH_VARIABLE BYTE masterNodeID
#ifndef WORK_PATCH
 = 0xFF
#endif
;

PATCH_VARIABLE BYTE bNWIStartup
#ifndef WORK_PATCH
 = FALSE
#endif
;

PATCH_VARIABLE BYTE bTimerBasicSet;

/****************************************************************************/
/*                              PRIVATE DATA                                */
/****************************************************************************/

/* Node Status */
PATCH_VARIABLE volatile t_nodeStatus sensorNodeStatus;

/* A list of the known command classes. Except the basic class which allways */
/* should be supported. Used when node info is send */
/* TO#1938 fix - COMMAND_CLASS_BASIC need never be mentioned here, as it is mandatory */
static code t_nodeInfo nodeInfo = {
  COMMAND_CLASS_SENSOR_BINARY,
  COMMAND_CLASS_MANUFACTURER_SPECIFIC,
  COMMAND_CLASS_VERSION,
  COMMAND_CLASS_ASSOCIATION,
#ifdef BATTERY
  COMMAND_CLASS_WAKE_UP,
#endif
#ifdef SECURITY
  COMMAND_CLASS_SECURITY,
#endif
};

#ifdef SECURITY
extern BYTE nodeSecure;
extern BYTE secureCmdHandling;

#define TSEC_CMD_HANDLING_SECURE    1

static code BYTE nodeInfoAfterIncluded[] = {
  COMMAND_CLASS_VERSION,
  COMMAND_CLASS_MANUFACTURER_SPECIFIC,
  COMMAND_CLASS_SECURITY
};

static code BYTE nodeInfoForTransport[] = {
  COMMAND_CLASS_SENSOR_BINARY,
  COMMAND_CLASS_MANUFACTURER_SPECIFIC,
  COMMAND_CLASS_VERSION,
  COMMAND_CLASS_ASSOCIATION,
#ifdef BATTERY
  COMMAND_CLASS_WAKE_UP,
#endif
};
#else
#define nodeInfoForTransport nodeInfo
#define nodeInfoAfterIncluded nodeInfo
#endif

/****************************************************************************/
/*                              EXPORTED DATA                               */
/****************************************************************************/

PATCH_VARIABLE_STARTER ZW_APPLICATION_TX_BUFFER txBuf;

/****************************************************************************/
/*                            PRIVATE FUNCTIONS                             */
/****************************************************************************/

extern /*PATCH_VARIABLE*/ BYTE powerDownTimeout;

/*============================   WatchdogCallback  ==============================
**    Check to see if the current state have run longer than WATCHDOG_TIMEOUT
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
PATCH_FUNCTION_NAME(WakeUPCallbackStart)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(WakeUPCallbackStart)
#pragma endasm
#endif
		ZW_DEBUG_SEND_BYTE('W');
  		ZW_DEBUG_SEND_BYTE('U');
  		ZW_DEBUG_SEND_BYTE('C');
  		ZW_DEBUG_SEND_BYTE(' ');
#ifdef BATTERY
    powerDownTimeout = DEFAULT_POWERDOWNTIMEOUT;

	currentState = STATE_BASIC_CMD_WAIT;
	bTimerBasicSet = 0xFF;

	StartPowerDownTimer();
#endif
}

/*============================   StartWatchdog  ==============================
**    WatchdogTimer, make sure no state runs forever
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
PATCH_FUNCTION_NAME(StartWakeUP)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StartWakeUP)
#pragma endasm
#endif
  if(stateWakeUPTimerHandle != 0xFF)
  {
    ZW_TIMER_CANCEL(stateWakeUPTimerHandle);
  }
  stateWakeUPTimerHandle = ZW_TIMER_START(WakeUPCallbackStart,
                                            TIMER_ONE_SECOND/2,
                                            TIMER_ONE_TIME);
  bTimerBasicSet = 0xFF;
}

/*============================   WatchdogCallback  ==============================
**    Check to see if the current state have run longer than WATCHDOG_TIMEOUT
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
PATCH_FUNCTION_NAME(WatchdogCallback)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(WatchdogCallback)
#pragma endasm
#endif
  stateWatchdogTimeout--;
  if(stateWatchdogTimeout <= 0)
  {
    stateWatchdogTimeout = WATCHDOG_TIMEOUT;
    // State not changed
    if(watchState == currentState)
    {
		ZW_DEBUG_SEND_BYTE('W');
  		ZW_DEBUG_SEND_BYTE('f');
      // Force into IDLE state
      currentState = STATE_APPL_IDLE;
    }
 }
}

/*============================   StartWatchdog  ==============================
**    WatchdogTimer, make sure no state runs forever
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
PATCH_FUNCTION_NAME(StartWatchdog)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StartWatchdog)
#pragma endasm
#endif
  watchState = currentState;
  if(stateWatchdogTimerHandle != 0xFF)
  {
    ZW_TIMER_CANCEL(stateWatchdogTimerHandle);
  }
  ZW_DEBUG_SEND_BYTE('W');
  ZW_DEBUG_SEND_BYTE('s');
  stateWatchdogTimerHandle = ZW_TIMER_START(WatchdogCallback,
                                            TIMER_ONE_SECOND,
                                            TIMER_FOREVER);
  ZW_DEBUG_SEND_NUM(stateWatchdogTimerHandle);
}

/*============================   StopWatchdog  ==============================
**    WatchdogTimer, stop the timer
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
PATCH_FUNCTION_NAME(StopWatchdog)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(StopWatchdog)
#pragma endasm
#endif
  watchState = STATE_APPL_IDLE;
  if(stateWatchdogTimerHandle != 0xFF)
  {
    ZW_TIMER_CANCEL(stateWatchdogTimerHandle);
    stateWatchdogTimerHandle = 0xFF;
  }
}

/*============================   SaveConfiguration   ======================
**    This function saves the current configuration to EEPROM
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
PATCH_FUNCTION_NAME(SaveConfiguration)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(SaveConfiguration)
#pragma endasm
#endif
#ifdef BATTERY
  SaveBatteryConfiguration();
#endif

  ZW_DEBUG_SEND_BYTE('C');
  ZW_DEBUG_SEND_BYTE('s');

  ZW_MEM_PUT_BYTE(EEOFFSET_LOST_COUNTER, 0);

  /* Mark stored configuration as OK */
  ZW_MEM_PUT_BYTE(EEOFFSET_MAGIC, MAGIC_VALUE);
}

/*============================   SetDefaultConfiguration   ======================
**    Function resets configuration to default values.
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /*RET  Nothing       */
PATCH_FUNCTION_NAME(SetDefaultConfiguration)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(SetDefaultConfiguration)
#pragma endasm
#endif
#ifdef BATTERY
  SetDefaultBatteryConfiguration();
#endif
#ifdef ZW_SELF_HEAL
  SetDefaultNetworkUpdateConfiguration();
#endif

  ZW_DEBUG_SEND_BYTE('C');
  ZW_DEBUG_SEND_BYTE('d');
}


/*============================   LoadConfiguration   ======================
**    This function loads the application settings from EEPROM.
**    If no settings are found, default values are used and saved.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                   /* RET  Nothing      */
PATCH_FUNCTION_NAME(LoadConfiguration)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(LoadConfiguration)
#pragma endasm
#endif
  currentState = STATE_EXECUTE;
  /* Get this sensors identification on the network */
  MemoryGetID(myHomeID, &myNodeID);

  /* Check to see, if any valid configuration is stored in the EEPROM */
  if (ZW_MEM_GET_BYTE(EEOFFSET_MAGIC) == MAGIC_VALUE)
  {
    /* There is a configuration stored, so load it */
#ifdef BATTERY
    LoadBatteryConfiguration();
#endif

    ZW_DEBUG_SEND_BYTE('C');
    ZW_DEBUG_SEND_BYTE('l');
  }
  else
  {
    /* Apparently there is no valid configuration in EEPROM, so load */
    /* default values and save them to EEPROM. */
    SetDefaultConfiguration();
    SaveConfiguration();
  }
  currentState = STATE_APPL_IDLE;
}

/*=====================   HasAssociatedNode   ====================
**    Checks if any nodes have been associated with this node.
**
**    Side effects:
**
**---------------------------------------------------------------*/
BOOL                   /*RET TRUE if any nodes have been associated */
PATCH_FUNCTION_NAME(HasAssociatedNode)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
  BYTE nodeIndex;

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(HasAssociatedNode)
#pragma endasm
#endif
  for (nodeIndex = 0; nodeIndex < MAX_ASSOCIATION_IN_GROUP; nodeIndex++)
  {
    if (groups[currentGroupIndex].nodeID[nodeIndex])
    {
      return TRUE;
    }
  }
  return FALSE;
}

/*=========================   BasicSetWaitEnd   ====================
**    Timeoutfunction used to delay sending basic set to avoid
**    blocking learn mode with explorer frames
**
**    Side effects:
**
**---------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(BasicSetWaitEnd)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(BasicSetWaitEnd)
#pragma endasm
#endif
  if (currentState == STATE_BASIC_CMD_WAIT)
    currentState = STATE_BASIC_CMD_START;
  bTimerBasicSet = 0xFF;

}

/*=========================   PrepareBasicSet   ====================
**    Prepares the transmit buffer with data for a Basic Set frame.
**
**    Side effects:
**
**---------------------------------------------------------------*/
void                    /*RET  Nothing       */
PATCH_FUNCTION_NAME(PrepareBasicSet)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(PrepareBasicSet)
#pragma endasm
#endif
  ZW_DEBUG_SEND_BYTE('A');
  ZW_DEBUG_SEND_BYTE('3');

  txBuf.ZW_BasicSetFrame.cmdClass = COMMAND_CLASS_BASIC;
  txBuf.ZW_BasicSetFrame.cmd = BASIC_SET;
  txBuf.ZW_BasicSetFrame.value =  toggleBasicSet;
  toggleBasicSet = (toggleBasicSet ? 0 : 1 );
  ZW_MEM_PUT_BYTE(EEOFFSET_SENSOR_LEVEL, toggleBasicSet);

  frameSize = sizeof(ZW_BASIC_SET_FRAME);
}

/*=======================   PrepareBasicReport   ===================
**    Prepares the transmit buffer with data for a Basic Report
**    frame.
**
**    Side effects:
**
**---------------------------------------------------------------*/
void                    /*RET  Nothing       */
PATCH_FUNCTION_NAME(PrepareBasicReport)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(PrepareBasicReport)
#pragma endasm
#endif
    txBuf.ZW_BasicSetFrame.cmdClass = COMMAND_CLASS_BASIC;
    txBuf.ZW_BasicSetFrame.cmd = BASIC_REPORT;
    txBuf.ZW_BasicSetFrame.value = toggleBasicSet;
    frameSize = sizeof(ZW_BASIC_REPORT_FRAME);
}

/*============================   BasicCmd   ======================
**    Send a Basic Set command to associated nodes if any,
**    otherwise broadcast a Basic Report (broadcasting
**    Basic Set is inadvisable).
**
**    Side effects:
**
**---------------------------------------------------------------*/
void                   /*RET  Nothing       */
PATCH_FUNCTION_NAME(BasicCmd)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(BasicCmd)
#pragma endasm
#endif
  currentState = STATE_BASIC_CMD;
  /* A basic set is only sent when the node is member of a network */
  if (myNodeID != 0)
  {
    if (HasAssociatedNode())
    {
      PrepareBasicSet();
      AssociationSendToGroup();
    }
    else
    {
//      PrepareBasicReport();
//      ZW_SendData(NODE_BROADCAST, (BYTE*)&txBuf, frameSize,
//        (TRANSMIT_OPTION_RETURN_ROUTE | TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE), SendCompleted);
      currentState = STATE_APPL_IDLE;
    }
/////    toggleBasicSet = (toggleBasicSet ? FALSE : TRUE );
  }
  else
  {
   currentState = STATE_APPL_IDLE;
  }
}



/*============================   AssociationStoreAll   ======================
**    Function description
**      Stores all groups in external Nonvolatile memory. Used by association.c
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(AssociationStoreAll)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(AssociationStoreAll)
#pragma endasm
#endif
  ZW_DEBUG_SEND_BYTE('D');
  ZW_DEBUG_SEND_BYTE('2');
  MemoryPutBuffer(EEOFFSET_ASSOCIATION_START, (BYTE *)&groups[0], ASSOCIATION_SIZE, NULL);
}

/*============================   AssociationClearAll   ======================
**    Function description
**      Clears the Association area in Nonvolatile memory. Used by association.c
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(AssociationClearAll)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(AssociationClearAll)
#pragma endasm
#endif
  ZW_DEBUG_SEND_BYTE('D');
  ZW_DEBUG_SEND_BYTE('1');
  MemoryPutBuffer(EEOFFSET_ASSOCIATION_START, NULL, ASSOCIATION_SIZE, NULL);
  ZW_MEM_PUT_BYTE(EEOFFSET_ASSOCIATION_MAGIC, MAGIC_VALUE); /* Now ASSOCIATION should be OK */
}

/*============================   AssociationInit   ==========================
**    Function description
**      Reads the groups stored in the Nonvolatile memory
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(AssociationInit)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(AssociationInit)
#pragma endasm
#endif
  ZW_DEBUG_SEND_BYTE('D');
  ZW_DEBUG_SEND_BYTE('0');
  if (ZW_MEM_GET_BYTE(EEOFFSET_ASSOCIATION_MAGIC) != MAGIC_VALUE)
  {
    /* Clear it */
    AssociationClearAll();
  }
  MemoryGetBuffer(EEOFFSET_ASSOCIATION_START, (BYTE *)&groups[0], ASSOCIATION_SIZE);
}

/*============================   AssociationSendNext   ======================
**    Function description
**      Callback used when sending to association command
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(AssociationSendNext)(
  BYTE bStatus
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(AssociationSendNext)
#pragma endasm
#endif
  VerifyAssociatedTransmit(bStatus, groups[currentGroupIndex].nodeID[currentGroupNodeIndex]);
  ZW_DEBUG_SEND_BYTE('D');
  ZW_DEBUG_SEND_BYTE('5');
  currentGroupNodeIndex++;
  if (!AssociationSendBasicSet(FALSE))
  {
    SendCompleted(bStatus);  /* Now done... */
  }
}


/*============================   AssociationSendBasicSet   ======================
**    Function description
**      Sends Association frame to the node currentGroupNodeIndex (if exist)
**      in the current currentGroupIndex group.
**    Side effects:
**
**--------------------------------------------------------------------------*/
BOOL            /*RET TRUE if there may be more associated nodes to send to */
PATCH_FUNCTION_NAME(AssociationSendBasicSet)(
  BYTE broadcast            /*if TRUE - Broadcast is sent if no nodes are found*/
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(AssociationSendBasicSet)
#pragma endasm
#endif
  ZW_DEBUG_SEND_BYTE('D');
  ZW_DEBUG_SEND_BYTE('4');
  for (; currentGroupNodeIndex < MAX_ASSOCIATION_IN_GROUP; currentGroupNodeIndex++)
  {
    if (groups[currentGroupIndex].nodeID[currentGroupNodeIndex])
    {
      ZW_DEBUG_SEND_NUM(currentGroupNodeIndex);
      Transport_SendRequest(groups[currentGroupIndex].nodeID[currentGroupNodeIndex], (BYTE*)&txBuf, frameSize,
        (TRANSMIT_OPTION_RETURN_ROUTE | TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE), AssociationSendNext, FALSE);
      return TRUE;
    }
  }
  if (broadcast)
  {
    ZW_DEBUG_SEND_BYTE('E');
#ifdef ZW030x
    Transport_SendRequest(NODE_BROADCAST, (BYTE*)&txBuf, frameSize, (TRANSMIT_OPTION_RETURN_ROUTE | TRANSMIT_OPTION_ACK), SendCompleted, FALSE);
#else
    Transport_SendRequest(NODE_BROADCAST, (BYTE*)&txBuf, frameSize, 0, SendCompleted, FALSE);
#endif
    return TRUE;
  }
  return FALSE;
}

/*============================   SendCompleted   ======================
**    Callback for association sends...
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                    /*RET  Nothing       */
PATCH_FUNCTION_NAME(SendCompleted)(
  BYTE txStatus         /*IN   Transmission result           */
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(SendCompleted)
#pragma endasm
#endif
  ZW_DEBUG_SEND_BYTE('D');
  ZW_DEBUG_SEND_BYTE('6');
  currentState = STATE_APPL_IDLE;
}

/*============================   AssociationSendToGroup   ====================
**    Function description
**      Initiate transmission of a preprepared Association frame to nodes in
**      first group.
**    Side effects:
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(AssociationSendToGroup)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(AssociationSendToGroup)
#pragma endasm
#endif
  ZW_DEBUG_SEND_BYTE('D');
  ZW_DEBUG_SEND_BYTE('3');
  /* Setup to start transmit to first group (for now we have only one group!) */
  currentGroupIndex = 0;
  /* Setup to start with first node */
  currentGroupNodeIndex = 0;
  if (!AssociationSendBasicSet(TRUE))
  {
    SendCompleted(TRANSMIT_COMPLETE_OK);  /* Now done... */
  }
}

/*============================   LearnCompleted   ========================
**    Callback which is called on learnmode completed
**  Application specific handling of LearnModeCompleted - called from
**  slave_learn.c
**--------------------------------------------------------------------------*/
void									/*RET	Nothing */
PATCH_FUNCTION_NAME(LearnCompleted)(
  BYTE bNodeID					/* IN resulting nodeID */
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(LearnCompleted)
#pragma endasm
#endif
  ZW_DEBUG_SEND_BYTE('A');
  ZW_DEBUG_SEND_BYTE('2');

  myNodeID = bNodeID;
  if (myNodeID == 0)
  {
#ifdef ZW_SELF_HEAL
  	UpdateNetworkUpdateCount(TRUE); // Reset networkupdatecount
#endif
    SetDefaultConfiguration();
    SaveConfiguration();
    AssociationClearAll();
    AssociationInit();
  }
  currentState = STATE_APPL_IDLE;

  Transport_OnLearnCompleted(bNodeID);
}

/*============================   CheckButtonPress   ==========================
**    Check and handle button press from ApplicationPoll
**
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME(CheckButtonPress)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
  register BYTE bStatus;

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(CheckButtonPress)
#pragma endasm
#endif
  bStatus = OneButtonLastAction();

  if(bStatus == BUTTON_WAS_PRESSED)
  {
#ifdef ZW_SELF_HEAL
    currentHealMode = HEAL_NONE;
#endif

#ifdef BATTERY
    currentState = STATE_WAKEUP_NOTIFICATION_START;
#else
    currentState = STATE_APPL_IDLE;
#endif

    if(lostCount != 0 && currentHealMode != HEAL_NONE)
    {
      HealComplete(FALSE);
      currentHealMode = HEAL_NONE;
#ifdef BATTERY
      powerDownTicks = BIN_POWERDOWNTIMEOUT*2;
      CancelRediscoveryTimer();
#endif
    }

    /* TO#2043 Fix */
    // nextState = STATE_BASIC_CMD_START;
    nextState = STATE_BASIC_CMD_WAIT;
  }
  else if (bStatus == BUTTON_TRIPLE_PRESS)
  {
    currentState = STATE_LEARN_MODE_START;
#ifdef BATTERY
    nextState = STATE_KEEP_ALIVE;
    if(!keepAliveActive) /* Time to activate keepalive */
    {
      ZW_DEBUG_SEND_BYTE('K');
      ZW_DEBUG_SEND_BYTE('a');
      keepAliveActive = TRUE;
      LED_ON(3);
      powerDownTicks = BIN_KEEPALIVETIMEOUT;
    }
#endif
    // No button press
    if(currentState == STATE_CHECK_INPUT)
    {
      currentState = STATE_APPL_IDLE;
    }
  }

}

/****************************************************************************/
/*                           EXPORTED FUNCTIONS                             */
/****************************************************************************/


/*============================   ApplicationInitHW   ========================
**    Initialization of non Z-Wave module hardware
**
**    Side effects:
**       Returning FALSE from this function will force the API into
**       production test mode.
**--------------------------------------------------------------------------*/

BYTE                       /* RET TRUE        */
PATCH_FUNCTION_NAME_STARTER(ApplicationInitHW)(
  BYTE bWakeupReason
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
//  IBYTE i;
  BYTE i;

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationInitHW)
#pragma endasm
#endif
#ifdef ZW_ISD51_DEBUG
  ZW_UART1_init(1152, TRUE, TRUE);
#endif
#ifdef APPL_PROD_TEST
#ifdef __C51__
  SET_PRODUCTIONTEST_PIN;
  for (i = 0; i < 10; i++) ;  /* Short delay... */
  if (IN_PRODUCTIONTEST) /* Anyone forcing it into productiontest mode ? */
  {
    /* Return FALSE to enter production test mode */
    return(FALSE);
  }
#endif /* __C51__ */

#endif /* APPL_PROD_TEST */

  /* hardware initialization */
  PIN_IN(Button, 1);


#ifdef ZW030x
  PIN_IN(INT1pin, 1);   /* Sensor input pin */
#endif /* ZW030x */

#ifdef ZW040x
  PIN_IN(INT1, 1);   /* Sensor input pin */
#endif /* ZW040x */

  /* Setup specifics for current dim module */
  PIN_OUT(LED1);
  PIN_OUT(LED2);
  PIN_OUT(LED3);

#ifdef ZW040x
  PIN_OUT(LED4);
#endif /* ZW040x */

#ifndef __C51__
  PIN_OUT(LED5);
  PIN_OUT(IO10);
  PIN_ON(IO10);
#endif /* __C51__ */

  LED_OFF(1);
  LED_OFF(2);
  LED_OFF(3);

#ifdef ZW040x
  LED_ON(4);
#endif /* ZW040x */

#ifndef __C51__
  LED_OFF(5);
#endif /* __C51__ */


#ifdef ZW030x
  IT1 = 0; /*level triggered*/
#endif /* ZW030x */



#ifdef BATTERY
wakeUpReason = bWakeupReason;

#endif

  Transport_OnApplicationInitHW(bWakeupReason);

  /* Production test mode */
  /* return(FALSE); */

  /* Return true to continue normal operating mode */
  return(TRUE);
}

#ifdef SECURITY
void
PATCH_FUNCTION_NAME(FuncZWSecure)(
  BYTE txStatus
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY(FuncZWSecure)
#pragma endasm
#endif

	if (txStatus == TRANSPORT_WORK_END)
	{
#ifdef BATTERY
	   StartPowerDownTimer();
#endif
	}
	else if (txStatus == TRANSPORT_WORK_START)
	{
#ifdef BATTERY
	   StopPowerDownTimer();
#endif
	}
	else if (txStatus == TRANSPORT_WORK_ERROR)
	{
#ifdef BATTERY
	   StartPowerDownTimer();
#endif
	}
}
#endif


/*===========================   ApplicationInitSW   =========================
**    Initialization of the Application Software variables and states
**    100 series: Not called on RTC wakeup
**    200 series: Called on WUT wakeup
**
**--------------------------------------------------------------------------*/
BYTE                      /*RET  TRUE       */
PATCH_FUNCTION_NAME_STARTER(ApplicationInitSW)(void) /* IN   Nothing   */
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationInitSW)
#pragma endasm
#endif

/****************************************************************************/
/*                    Initialization of Utility modules                     */
/****************************************************************************/

#ifdef PATCH_ENABLE
  SlaveLearnInit();
#endif

/* Do not reinitialize the UART if already initialized for ISD51 in ApplicationInitHW() */
#ifndef ZW_ISD51_DEBUG
  ZW_DEBUG_INIT(1152);

  ZW_DEBUG_SEND_NL();
	ZW_DEBUG_SEND_BYTE('A');
  ZW_DEBUG_SEND_BYTE('0');
#endif

  /* Signal that the sensor is awake */
  LED_ON(1);
  LoadConfiguration();

//  slaveLearnInit();

  AssociationInit();

  /* Initialize button detection */
  OneButtonInit();

  Transport_OnApplicationInitSW(nodeInfoForTransport, sizeof(nodeInfoForTransport),
    FALSE, EEOFFSET_TRANSPORT_SETTINGS_START, EEOFFSET_TRANSPORT_SETTINGS_SIZE, NULL);

  bTimerBasicSet = 0xFF;

  /* Set Nwtwork wide inclusion active if we dont have aa node ID */
  if (myNodeID)
    bNWIStartup = FALSE;
#ifdef BATTERY
  /* TO#1948 */
  else if ((wakeUpReason == ZW_WAKEUP_RESET)||(wakeUpReason == ZW_WAKEUP_POR))
    bNWIStartup = TRUE;
  ZW_DEBUG_SEND_BYTE('W');
  ZW_DEBUG_SEND_NUM(wakeUpReason);
#ifdef ZW040x
  /*for now wake up by external interrupt is the same as hw reset*/
  if ((wakeUpReason == ZW_WAKEUP_EXT_INT) ||(wakeUpReason == ZW_WAKEUP_POR)||(wakeUpReason == ZW_WAKEUP_WATCHDOG) )
    wakeUpReason = 0;
#endif
#else
  else
    bNWIStartup = TRUE;
#endif /* ZW_BATTERY */

  if (ZW_MEM_GET_BYTE(EEOFFSET_MAGIC) == MAGIC_VALUE)
  {
    toggleBasicSet = ZW_MEM_GET_BYTE(EEOFFSET_SENSOR_LEVEL);
  }
  else
  {
    toggleBasicSet = 0;
    ZW_MEM_PUT_BYTE(EEOFFSET_SENSOR_LEVEL, toggleBasicSet);
  }

#ifdef BATTERY

  /* Update wakeup count if the WUT was what woke the sensor */
  if (wakeUpReason == ZW_WAKEUP_WUT) /* 0 - Reset, 1 - WUT, 2 - Wakeup Beam */
  {
//TO3500    LoadBatteryConfiguration();
    UpdateNetworkUpdateCount(FALSE);
    UpdateWakeupCount();

  } else if ( wakeUpReason == ZW_WAKEUP_RESET && masterNodeID == 0xFF)
  { /* Power Cycle, Set Initial Battery Information */
   	SetDefaultBatteryConfiguration();
  }

	powerDownTicks = BIN_POWERDOWNTIMEOUT;
	StartPowerDownTimer();
#endif /* ZW_BATTERY */

#ifndef BATTERY
#ifdef ZW_SELF_HEAL
 if (networkUpdateTimerHandle == 0xFF)
  {
    networkUpdateTimerHandle = ZW_TIMER_START(UpdateNetworkUpdateCountOneMinute,
                                                TIMER_ONE_SECOND,
                                                TIMER_FOREVER);
    ZW_DEBUG_SEND_BYTE('T');
    ZW_DEBUG_SEND_BYTE('s');
  }
#endif
#endif


#ifdef BATTERY
	if(currentState != STATE_WAKEUP_NOTIFICATION_START)
 	{
#endif
		currentState = STATE_APPL_IDLE;
#ifdef BATTERY
	}
#endif

  return(TRUE);
}

/*============================   ApplicationTestPoll   ======================
**    Function description
**      This function is called when the slave enters test mode.
**
**    Side effects:
**       Code will not exit until it is reset
**--------------------------------------------------------------------------*/
void PATCH_FUNCTION_NAME_STARTER(ApplicationTestPoll)(void)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationTestPoll)
#pragma endasm
#endif
}

/*=============================  ApplicationPoll   =========================
**    Application poll function for the Battery operated Bin Sensor
**
**    Side effects:
**
**--------------------------------------------------------------------------*/
void                    /*RET  Nothing                  */
PATCH_FUNCTION_NAME_STARTER(ApplicationPoll)(void) /* IN  Nothing                  */
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationPoll)
#pragma endasm
#endif

#ifdef ZW_ISD51_DEBUG         /* init ISD51 only when the uVision2 Debugger tries to connect */
    ISDcheck();               /* initialize uVision2 Debugger and continue program run */
#endif
  if (bNWIStartup)
  {
    StartLearnModeNow(ZW_SET_LEARN_MODE_NWI);
    bNWIStartup = FALSE;
    currentState = STATE_LEARN_MODE;
  }

  switch (currentState)
  {

    case STATE_APPL_IDLE:
    {
      /* Stop the watchdog and set the watchState to APPL_IDLE */
      StopWatchdog();
      /* If something else is in queue do that */
      if(nextState != STATE_APPL_IDLE)
      {
        currentState = nextState;
        nextState = STATE_APPL_IDLE;
      } else { /* When theres nothing else to do, check button and sensor */
        currentState = STATE_CHECK_INPUT;
      }
      break;
    }
    case STATE_COMMAND_HANDLER:
    {
      /* Do nothing  */
      break;
    }
    case STATE_HEAL_LOST:
    {
     // Do nothing, we are currently searching for a SUC node. - Accept button input
     currentState = STATE_CHECK_INPUT;
     nextState = STATE_HEAL_LOST;
     break;
    }
#ifdef BATTERY
    case STATE_WAKEUP_NOTIFICATION_START:
    {
      currentState = STATE_WAKEUP_NOTIFICATION;
      ZW_DEBUG_SEND_BYTE('S');
      ZW_DEBUG_SEND_BYTE('b');
      ZW_DEBUG_SEND_NUM(currentState);

      WakeupNotification();
      ZW_DEBUG_SEND_BYTE('a');
      ZW_DEBUG_SEND_NUM(currentState);
      StartWatchdog();
      break;
    }
    case STATE_WAKEUP_NOTIFICATION:
    {
      // Do nothing, wait for the controller to finish talking to us
////		StartWakeUP();
      break;
    }
#endif
    /* TO#2043 Fix */
    case STATE_BASIC_CMD_WAIT:
    {
      /* Wait to see if the user makes a triple press before sending
         basic set */
      if (bTimerBasicSet == 0xFF)
      {
        bTimerBasicSet = TimerStart(BasicSetWaitEnd, TIME_DELAY_BASIC_SET, TIMER_ONE_TIME);
      if (bTimerBasicSet == 0xFF)
        BasicSetWaitEnd();
      }
      CheckButtonPress();
      break;
    }
    case STATE_BASIC_CMD_START:
    {

      BasicCmd();
#ifdef BATTERY
      StartPowerDownTimer();
#endif
      /* Make sure we don't get stuck */
      StartWatchdog();
      break;
    }
    case STATE_BASIC_CMD:
    {
      // Do nothing but wait for basic set/report to complete
      break;
    }
    case STATE_LEARN_MODE:
      if (OneButtonLastAction()==BUTTON_WAS_PRESSED)
    {
        /* Stop learn mode */
        StartLearnModeNow(FALSE);
        currentState = STATE_APPL_IDLE;
    }
      break;

    case STATE_LEARN_MODE_START:
    {
      currentState = STATE_LEARN_MODE;

      ZW_DEBUG_SEND_BYTE('N');
	  ZW_DEBUG_SEND_BYTE('I');
      ZW_DEBUG_SEND_NUM(currentState);

      StartWatchdog();
      StartLearnModeNow(ZW_SET_LEARN_MODE_CLASSIC);

#ifdef BATTERY
		StartWakeUP();
#endif


      break;
    }
    case STATE_CHECK_INPUT:
    {
      CheckButtonPress();
      break;
    }
    case STATE_EXECUTE:
    {
      /* Do nothing but check button press - we are waiting for an operation to finish */
      CheckButtonPress();
      break;
    }
    default:
    {
      /* BTW: Something is terribly wrong if 'default' is ever executed */
      ZW_DEBUG_SEND_BYTE('X');
      ZW_DEBUG_SEND_NUM(currentState);
      currentState = STATE_APPL_IDLE;
      break;
    }

  }
}

/*========================   ApplicationCommandHandler   ====================
**    Handling of a received application commands and requests
**
**
**--------------------------------------------------------------------------*/
void                              /*RET Nothing                  */
PATCH_FUNCTION_NAME_STARTER(Transport_ApplicationCommandHandler)(
  BYTE  rxStatus,                 /* IN Frame header info */
  BYTE  sourceNode,               /* IN Command sender Node ID */
  ZW_APPLICATION_TX_BUFFER *pCmd, /* IN Payload from the received frame, the union */
                                  /*    should be used to access the fields */
  BYTE   cmdLength                /* IN Number of command bytes including the command */
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
  BYTE param1;
  BYTE param2;
  BYTE param3;
  BYTE param4;
  BYTE param5;
  BYTE param6;
  BYTE txOption;

#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(Transport_ApplicationCommandHandler)
#pragma endasm
#endif
  param1 = ((BYTE_P)pCmd)[OFFSET_PARAM_1];
  param2 = ((BYTE_P)pCmd)[OFFSET_PARAM_2];
  param3 = ((BYTE_P)pCmd)[OFFSET_PARAM_3];
  param4 = ((BYTE_P)pCmd)[OFFSET_PARAM_4];
  param5 = ((BYTE_P)pCmd)[OFFSET_PARAM_5];
  param6 = ((BYTE_P)pCmd)[OFFSET_PARAM_6];
  txOption = ((rxStatus & RECEIVE_STATUS_LOW_POWER) ?
              TRANSMIT_OPTION_LOW_POWER : 0) | TRANSMIT_OPTION_ACK;

  /* We were about to sleep, prevent it */
  if(currentState == STATE_APPL_IDLE)
  {
    currentState = STATE_COMMAND_HANDLER;
    StartWatchdog();
  }

#ifdef BATTERY
  powerDownTicks = BIN_POWERDOWNTIMEOUT;
#endif

  ZW_DEBUG_SEND_BYTE('A');
  ZW_DEBUG_SEND_BYTE('8');

  switch(pCmd->ZW_Common.cmdClass)
  {
#if 0 /* ??? removed in 4.5x Bin_Sensor->Bin_Sensor_Sec */
    case COMMAND_CLASS_SENSOR_BINARY:
    case COMMAND_CLASS_BASIC:
    {
      if (pCmd->ZW_Common.cmd == BASIC_GET){
          ZW_DEBUG_SEND_BYTE('B');
          ZW_DEBUG_SEND_BYTE('g');

          /* Controller wants the sensor level */
          txBuf.ZW_BasicReportFrame.cmdClass = pCmd->ZW_Common.cmdClass;
          txBuf.ZW_BasicReportFrame.cmd = BASIC_REPORT;
          txBuf.ZW_BasicReportFrame.value =  SENSOR_ACTIVATED();

#ifdef BATTERY
          ZW_SEND_DATA(sourceNode, (BYTE *)&txBuf, sizeof(ZW_BASIC_REPORT_FRAME),
            ((rxStatus & RECEIVE_STATUS_LOW_POWER) ? TRANSMIT_OPTION_LOW_POWER : 0) | TRANSMIT_OPTION_ACK, StartPowerDownTimer);
#else
          ZW_SEND_DATA(sourceNode, (BYTE *)&txBuf, sizeof(ZW_BASIC_REPORT_FRAME),
            ((rxStatus & RECEIVE_STATUS_LOW_POWER) ? TRANSMIT_OPTION_LOW_POWER : 0) | TRANSMIT_OPTION_ACK, NULL);
#endif

        }
    }break;
    case COMMAND_CLASS_WAKE_UP:
    {
#ifdef BATTERY
      HandleWakeupFrame(pCmd, txOption, sourceNode);
#endif
    }break;
#endif /* ??? removed in 4.5x Bin_Sensor->Bin_Sensor_Sec */
    case COMMAND_CLASS_VERSION:
    {
      if (pCmd->ZW_Common.cmd == VERSION_GET)
      {
        ZW_DEBUG_SEND_BYTE('V');
        ZW_DEBUG_SEND_BYTE('0');
        txBuf.ZW_VersionReportFrame.cmdClass = COMMAND_CLASS_VERSION;
        txBuf.ZW_VersionReportFrame.cmd = VERSION_REPORT;
        txBuf.ZW_VersionReportFrame.zWaveLibraryType = ZW_TYPE_LIBRARY();
        txBuf.ZW_VersionReportFrame.zWaveProtocolVersion = ZW_VERSION_MAJOR;
        txBuf.ZW_VersionReportFrame.zWaveProtocolSubVersion = ZW_VERSION_MINOR;
        txBuf.ZW_VersionReportFrame.applicationVersion = APP_VERSION;
        txBuf.ZW_VersionReportFrame.applicationSubVersion = APP_REVISION;
        Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_VersionReportFrame),
                   ((rxStatus & RECEIVE_STATUS_LOW_POWER) ? TRANSMIT_OPTION_LOW_POWER : 0)
                   | TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE, NULL, FALSE);
      }
      else if (pCmd->ZW_Common.cmd == VERSION_COMMAND_CLASS_GET)
      {
        ZW_DEBUG_SEND_BYTE('V');
        ZW_DEBUG_SEND_BYTE('1');
        txBuf.ZW_VersionCommandClassReportFrame.cmdClass = COMMAND_CLASS_VERSION;
        txBuf.ZW_VersionCommandClassReportFrame.cmd = VERSION_COMMAND_CLASS_REPORT;
        txBuf.ZW_VersionCommandClassReportFrame.requestedCommandClass = *((BYTE*)pCmd + OFFSET_PARAM_1);
        ZW_DEBUG_SEND_BYTE('V');
        if (*((BYTE*)pCmd + OFFSET_PARAM_1) == COMMAND_CLASS_BASIC)
        {
          ZW_DEBUG_SEND_BYTE('2');
          txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = BASIC_VERSION;
        }
        else if (*((BYTE*)pCmd + OFFSET_PARAM_1) == COMMAND_CLASS_SENSOR_BINARY)
        {
          ZW_DEBUG_SEND_BYTE('3');
          txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = SWITCH_BINARY_VERSION;
        }
        else if (*((BYTE*)pCmd + OFFSET_PARAM_1) == COMMAND_CLASS_CONFIGURATION)
        {
          ZW_DEBUG_SEND_BYTE('4');
          txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = CONFIGURATION_VERSION;
        }
        else if (*((BYTE*)pCmd + OFFSET_PARAM_1) == COMMAND_CLASS_WAKE_UP)
        {
          ZW_DEBUG_SEND_BYTE('5');
          txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = WAKE_UP_VERSION;
        }
        else if (*((BYTE*)pCmd + OFFSET_PARAM_1) == COMMAND_CLASS_MANUFACTURER_SPECIFIC)
        {
          ZW_DEBUG_SEND_BYTE('6');
          txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = MANUFACTURER_SPECIFIC_VERSION;
        }
        else if (*((BYTE*)pCmd + OFFSET_PARAM_1) == COMMAND_CLASS_VERSION)
        {
          ZW_DEBUG_SEND_BYTE('7');
          txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = VERSION_VERSION;
        }
        else
        {
          ZW_DEBUG_SEND_BYTE('8');
          txBuf.ZW_VersionCommandClassReportFrame.commandClassVersion = UNKNOWN_VERSION;
        }
        Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_VersionCommandClassReportFrame),
                    ((rxStatus & RECEIVE_STATUS_LOW_POWER) ? TRANSMIT_OPTION_LOW_POWER : 0)
                      | TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE, NULL, FALSE);
      }
    }break;
#if 0 /* ??? removed in 4.5x Bin_Sensor->Bin_Sensor_Sec */
    case COMMAND_CLASS_ASSOCIATION:
    {
        switch (pCmd->ZW_Common.cmd)
        {
          case ASSOCIATION_GET:
            AssociationSendReport(param1, sourceNode, txOption);
            break;

          case ASSOCIATION_SET:
#if MAX_ASSOCIATION_GROUPS > 1
            AssociationAdd(param1, (BYTE*)pCmd + OFFSET_PARAM_2, cmdLength - OFFSET_PARAM_2);
#else
            /* If node only support one group. Ignore the group ID requested */
            /* (As defined in Device class specification */
            AssociationAdd(1, (BYTE*)pCmd + OFFSET_PARAM_2, cmdLength - OFFSET_PARAM_2);
#endif
            break;

          case ASSOCIATION_REMOVE:
#if MAX_ASSOCIATION_GROUPS > 1
            AssociationRemove(param1, (BYTE*)pCmd + OFFSET_PARAM_2, cmdLength - OFFSET_PARAM_2);
#else
            /*If node only support one group. Ignore the group ID requested (As defined in Device class specification*/
            AssociationRemove(1, (BYTE*)pCmd + OFFSET_PARAM_2, cmdLength - OFFSET_PARAM_2);
#endif
            break;

          case ASSOCIATION_GROUPINGS_GET:
            AssociationSendGroupings(sourceNode, txOption);
            break;
        }
    }break;
#endif /* ??? removed in 4.5x Bin_Sensor->Bin_Sensor_Sec */
    case COMMAND_CLASS_MANUFACTURER_SPECIFIC:
    {
      if (pCmd->ZW_Common.cmd == MANUFACTURER_SPECIFIC_GET)
      {
        ZW_DEBUG_SEND_BYTE('M');
        ZW_DEBUG_SEND_BYTE('0');
        txBuf.ZW_ManufacturerSpecificReportFrame.cmdClass = COMMAND_CLASS_MANUFACTURER_SPECIFIC;
        txBuf.ZW_ManufacturerSpecificReportFrame.cmd = MANUFACTURER_SPECIFIC_REPORT;
        txBuf.ZW_ManufacturerSpecificReportFrame.manufacturerId1 = 0x00; /* Zensys manufacturer ID */
        txBuf.ZW_ManufacturerSpecificReportFrame.manufacturerId2 = 0x00;
        txBuf.ZW_ManufacturerSpecificReportFrame.productTypeId1 = 0x00; /* Assigned by manufacturer */
        txBuf.ZW_ManufacturerSpecificReportFrame.productTypeId2 = 0x00;
        txBuf.ZW_ManufacturerSpecificReportFrame.productId1 = 0x00; /* Assigned by manufacturer */
        txBuf.ZW_ManufacturerSpecificReportFrame.productId2 = 0x00;
        Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(txBuf.ZW_ManufacturerSpecificReportFrame),
                    ((rxStatus & RECEIVE_STATUS_LOW_POWER) ? TRANSMIT_OPTION_LOW_POWER : 0)
                    | TRANSMIT_OPTION_ACK | TRANSMIT_OPTION_EXPLORE, NULL, FALSE);
      }
      else
      {
        ZW_DEBUG_SEND_BYTE('M');
        ZW_DEBUG_SEND_BYTE('1');
      }
    }break;
  }



#ifdef SECURITY
  if(nodeSecure)
  if (secureCmdHandling != TSEC_CMD_HANDLING_SECURE)
  {
	  if(currentState == STATE_COMMAND_HANDLER)
	  {
	    currentState = STATE_APPL_IDLE;
	  }
  	  return;
  }
#endif


  switch(pCmd->ZW_Common.cmdClass)
  {
    case COMMAND_CLASS_SENSOR_BINARY:
    case COMMAND_CLASS_BASIC:
    {
      if (pCmd->ZW_Common.cmd == BASIC_GET){
          ZW_DEBUG_SEND_BYTE('B');
          ZW_DEBUG_SEND_BYTE('g');

          /* Controller wants the sensor level */
          txBuf.ZW_BasicReportFrame.cmdClass = pCmd->ZW_Common.cmdClass;
          txBuf.ZW_BasicReportFrame.cmd = BASIC_REPORT;
          txBuf.ZW_BasicReportFrame.value =  SENSOR_ACTIVATED();

#ifdef BATTERY
          Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(ZW_BASIC_REPORT_FRAME),
            ((rxStatus & RECEIVE_STATUS_LOW_POWER) ? TRANSMIT_OPTION_LOW_POWER : 0) | TRANSMIT_OPTION_ACK, StartPowerDownTimer, FALSE);
#else
          Transport_SendReport(sourceNode, (BYTE *)&txBuf, sizeof(ZW_BASIC_REPORT_FRAME),
            ((rxStatus & RECEIVE_STATUS_LOW_POWER) ? TRANSMIT_OPTION_LOW_POWER : 0) | TRANSMIT_OPTION_ACK, NULL, FALSE);
#endif

        }
    }break;
    case COMMAND_CLASS_WAKE_UP:
    {
#ifdef BATTERY
      HandleWakeupFrame(pCmd, txOption, sourceNode);
#endif
    }break;

    case COMMAND_CLASS_ASSOCIATION:
    {
		ZW_DEBUG_SEND_BYTE('>');
  		ZW_DEBUG_SEND_BYTE('A');
        switch (pCmd->ZW_Common.cmd)
        {
          case ASSOCIATION_GET:
            AssociationSendReport(param1, sourceNode, txOption);
            break;

          case ASSOCIATION_SET:
		    ZW_DEBUG_SEND_BYTE('A');
  			ZW_DEBUG_SEND_BYTE('9');
#if MAX_ASSOCIATION_GROUPS > 1
            AssociationAdd(param1, (BYTE*)pCmd + OFFSET_PARAM_2, cmdLength - OFFSET_PARAM_2);
#else
            /* If node only support one group. Ignore the group ID requested */
            /* (As defined in Device class specification */
            AssociationAdd(1, (BYTE*)pCmd + OFFSET_PARAM_2, cmdLength - OFFSET_PARAM_2);
#endif
			AssociationStoreAll();
            break;

          case ASSOCIATION_REMOVE:
#if MAX_ASSOCIATION_GROUPS > 1
            AssociationRemove(param1, (BYTE*)pCmd + OFFSET_PARAM_2, cmdLength - OFFSET_PARAM_2);
#else
            /*If node only support one group. Ignore the group ID requested (As defined in Device class specification*/
            AssociationRemove(1, (BYTE*)pCmd + OFFSET_PARAM_2, cmdLength - OFFSET_PARAM_2);
#endif
            break;

          case ASSOCIATION_GROUPINGS_GET:
            AssociationSendGroupings(sourceNode, txOption);
            break;
        }



#ifdef BATTERY
    powerDownTimeout = 0x02;
	currentState = STATE_APPL_IDLE;
	StartPowerDownTimer();
#endif

    }break;
  }

  if(currentState == STATE_COMMAND_HANDLER)
  {
    currentState = STATE_APPL_IDLE;
  }
}




/*==========================   ApplicationSlaveUpdate   =======================
**   Inform a slave application that a node information is received.
**   Called from the slave command  when a node information frame
**   is received and the Z-Wave protocol is not in a state where it is needed.
**
**--------------------------------------------------------------------------*/
void
PATCH_FUNCTION_NAME_STARTER(ApplicationSlaveUpdate)(
  BYTE bStatus,     /*IN  Status event */
  BYTE bNodeID,     /*IN  Node id of the node that send node info */
  BYTE* pCmd,       /*IN  Pointer to Application Node information */
  BYTE bLen         /*IN  Node info length                        */
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationSlaveUpdate)
#pragma endasm
#endif
}


/*======================   ApplicationNodeInformation   =====================
**    Request Node information and current status
**    Called by the the Z-Wave application layer before transmitting a
**    "Node Information" frame.
**
**    This is an application function example
**
**--------------------------------------------------------------------------*/
extern  void                  /*RET  Nothing */
PATCH_FUNCTION_NAME_STARTER(ApplicationNodeInformation)(
  BYTE   *deviceOptionsMask,      /*OUT Bitmask with application options     */
  APPL_NODE_TYPE  *nodeType,  /*OUT  Device type Generic and Specific   */
  BYTE       **nodeParm,      /*OUT  Device parameter buffer pointer    */
  BYTE       *parmLength     /*OUT  Number of Device parameter bytes   */
)
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationNodeInformation)
#pragma endasm
#endif
  /* this is a NON listening node and it supports optional CommandClasses*/
#ifdef BATTERY
  *deviceOptionsMask = APPLICATION_NODEINFO_NOT_LISTENING|APPLICATION_NODEINFO_OPTIONAL_FUNCTIONALITY;
#else
  *deviceOptionsMask = APPLICATION_NODEINFO_LISTENING|APPLICATION_NODEINFO_OPTIONAL_FUNCTIONALITY;
#endif
  nodeType->generic = GENERIC_TYPE_SENSOR_BINARY; /* Generic Device Type */
  nodeType->specific = SPECIFIC_TYPE_ROUTING_SENSOR_BINARY; /* Specific Device Type */
  *nodeParm = (BYTE *)&nodeInfo;        /* Send list of known command classes. */
  *parmLength = sizeof(nodeInfo);       /* Set length*/

#ifdef SECURITY
  MemoryGetID(myHomeID, &myNodeID);
  nodeSecure = GetNodeSecure(EEOFFSET_TRANSPORT_SETTINGS_START);
  if (myNodeID)
	if (nodeSecure)
	{
  		*nodeParm = nodeInfoAfterIncluded;        /* Send list of known command classes. */
  		*parmLength = sizeof(nodeInfoAfterIncluded);       /* Set length*/
	}
#endif

}

/* Do not use absolut register adressing */
#pragma NOAREGS
/*==============================   ApplicationRfNotify   ======================
**    Function used when using an external PA
**
**--------------------------------------------------------------------------*/
void                   /*RET TRUE - If normal operation, FALSE - if production test*/
PATCH_FUNCTION_NAME_STARTER(ApplicationRfNotify)(
  BYTE bStatus)        /* IN RF status */
#ifdef PATCH_ENABLE
reentrant
#endif
{
#ifdef PATCH_ENABLE
#pragma asm
PATCH_TABLE_ENTRY_STARTER(ApplicationRfNotify)
#pragma endasm
#endif
}

