using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave;
using Zensys.PCController.Interfaces;
using Zensys.PCController.Classes;
using Zensys.PCController.UI;
using Zensys.PCController.Dispatchers;
using System.Windows.Forms;
using Zensys.PCController.Properties;
using Zensys.ZWave.Devices;
using System.Collections;
using Zensys.ZWave.Enums;
using Zensys.PCController.Models;
using System.Threading;
using Zensys.Framework;
using Zensys.Framework.Application;
using Zensys.ZWave.ZWaveHAL;
using Zensys.ZWave.Exceptions;
using Zensys.ZWave.ZWaveHAL.Actions;
using System.IO;
using Zensys.ZWave.Application;

namespace Zensys.PCController.Controllers
{
    public class ControllerManager : IDisposable
    {
        private string mXmlZWaveDefinitionFile = @"ZWave_custom_cmd_classes.xml";
        private XmlDataManager mXmlDataManager;
        public XmlDataManager XmlDataManager
        {
            get { return mXmlDataManager; }
            set { mXmlDataManager = value; }
        }

        public void ReloadXml()
        {
            DefinitionConverter dconv = new DefinitionConverter(mXmlZWaveDefinitionFile, null);
            dconv.UpgradeConvert(false);
            XmlDataManager.ZWaveDefinition = dconv.ZWaveDefinition;
        }

        private ControllerManager()
        {
        }
        AutoResetEvent queueGo = new AutoResetEvent(false);
        public void Initialize(ApplicationMode applicationMode)
        {
            mApplicationMode = applicationMode;
            //Initialization of the Document Model
            switch (applicationMode)
            {
                case ApplicationMode.Console:
                    mDocumentModel = new DocumentModel(null);
                    break;
                case ApplicationMode.GUI:
                    mDocumentModel = new DocumentModel(MainForm.BindingContext);
                    break;
                default:
                    break;
            }
            Tools.StartHighPrecisionTimer();
            //Initialization of the Xml Data Manager
            XmlDataManager = new XmlDataManager();
            if (!string.IsNullOrEmpty(mXmlZWaveDefinitionFile) && File.Exists(mXmlZWaveDefinitionFile))
            {
                DefinitionConverter dconv = new DefinitionConverter(mXmlZWaveDefinitionFile, null);
                dconv.UpgradeConvert(false);
                XmlDataManager.ZWaveDefinition = dconv.ZWaveDefinition;
            }

            //Initialization of the ZWave Dll Manager
            mZWaveManager = new ZWaveManager();
            mZWaveManager.Init(LibraryModes.PcController, true);
            LogManager = new LogManager(WriteMessage, WriteOperationFailure);
            ExceptionManager = new ExceptionManager(WriteException);

            //several nodes can awake during processing command queue.
            Thread queueThread = new Thread(new ThreadStart(ProcessQueuedCommand));
            queueThread.IsBackground = true;
            queueThread.Start();
        }

        private object awakeNodesListLocker = new object();
        private SortedList<byte, bool> awakeNodesList = new SortedList<byte, bool>();
        public void ExecuteQueueCommand(byte sourceNodeId, bool isWakeUpNoMoreInformationFrameNeeded)
        {
            lock (awakeNodesListLocker)
            {
                if (!awakeNodesList.Keys.Contains(sourceNodeId))
                    awakeNodesList.Add(sourceNodeId, isWakeUpNoMoreInformationFrameNeeded);
                else if (isWakeUpNoMoreInformationFrameNeeded)
                {
                    awakeNodesList[sourceNodeId] = isWakeUpNoMoreInformationFrameNeeded;
                }
            }
            queueGo.Set();
        }

        private void ProcessQueuedCommand()
        {
            Queue<QueueAction> queue;
            while (true)
            {
                queueGo.WaitOne();
                bool someoneAwake = true;
                while (someoneAwake)
                {
                    someoneAwake = false;
                    Queue<QueueAction> innerqueue = new Queue<QueueAction>();
                    lock (awakeNodesListLocker)
                    {
                        for (int i = awakeNodesList.Count - 1; i >= 0; i--)
                        {
                            queue = null;
                            if (FramesToSend.ContainsKey(awakeNodesList.Keys[i]))
                            {
                                queue = FramesToSend[awakeNodesList.Keys[i]];
                                if (queue.Count > 0)
                                {
                                    innerqueue.Enqueue(queue.Dequeue());
                                    if (queue.Count > 0)
                                    {
                                        someoneAwake = true;
                                    }
                                }
                            }
                            if (queue == null || queue.Count == 0)
                            {
                                if (awakeNodesList.Values[i])
                                {
                                    innerqueue.Enqueue(CreateWakeUpNoMoreQueueAction(awakeNodesList.Keys[i]));
                                }
                                awakeNodesList.RemoveAt(i);
                            }
                        }
                    }
                    while (innerqueue.Count > 0)
                    {
                        if (!CustomLoader.GetEntryPoint().IsBusy)
                        {
                            QueueAction action = innerqueue.Dequeue();
                            if (action != null && action.ActionHandler != null)
                            {
                                this.DoAction(action);
                                if (action.WaitResponseTimeout > 0)
                                {
                                    Thread.Sleep(action.WaitResponseTimeout);
                                }
                            }
                            this.MainForm.LabelCommandQueue = innerqueue.Count.ToString();
                        }
                    }
                }
                Thread.Sleep(10);
            }
        }

        private QueueAction CreateWakeUpNoMoreQueueAction(byte sourceNodeId)
        {
            QueueAction ret = null;
            IDeviceInfo device = null;
            foreach (IDeviceInfo var in DocumentModel.Devices)
            {
                if (var.Id == sourceNodeId)
                {
                    device = var;
                    break;
                }
            }
            if (device != null)
            {
                ret = new QueueAction(
                    new ProcedureCaller<CommandClass,
                        Command,
                        byte,
                        IController,
                        List<byte>>(Actions.CommandClassFormActions.CommandClassHAL.SendCommand),
                    new object[] 
                    {
                        XmlDataManager.FindCommandClass("COMMAND_CLASS_WAKE_UP",1),
                        XmlDataManager.FindCommand("COMMAND_CLASS_WAKE_UP",1,"WAKE_UP_NO_MORE_INFORMATION"),
                        device.Id,
                        DocumentModel.Controller,
                        new List<byte>()
                    },
                Resources.MsgSendCommand, false, (byte)CommandTypes.CmdZWaveSendData, "Wake Up No More", 0);
            }
            return ret;
        }

        private void WriteMessage(string message)
        {
            Actions.LogFormActions.WriteMessage(message);
        }

        private void WriteOperationFailure(string message)
        {
            Actions.LogFormActions.WriteMessage(string.Format("{0}: {1}", Tools.CurrentDateTime.ToString("hh:mm:ss.fff"), "Request failed: " + message));
        }

        public void WriteException(Exception exception)
        {
            if (exception is RequestTimeoutException)
            {
                Actions.LogFormActions.WriteMessage(string.Format("{0}: {1}", Tools.CurrentDateTime.ToString("hh:mm:ss.fff"), "Request time-out period exceeded"));
            }
            else
            {
                Actions.LogFormActions.WriteMessage(string.Format("{0}: {1}", Tools.CurrentDateTime.ToString("hh:mm:ss.fff"), "Exception: " + exception.Message));
            }
        }

        //private SettingsWrapper mApplicationSettings;
        //public SettingsWrapper ApplicationSettings
        //{
        //    get { return mApplicationSettings; }
        //}

        private ApplicationMode mApplicationMode;
        public ApplicationMode ApplicationMode
        {
            get { return mApplicationMode; }
        }

        private DocumentModel mDocumentModel;
        public DocumentModel DocumentModel
        {
            get { return mDocumentModel; }
        }

        private LogManager mLogManager = new LogManager(null, null);
        public LogManager LogManager
        {
            get { return mLogManager; }
            set { mLogManager = value; }
        }

        private ExceptionManager mExceptionManager = new ExceptionManager(null);
        public ExceptionManager ExceptionManager
        {
            get { return mExceptionManager; }
            set { mExceptionManager = value; }
        }



        private static ControllerManager mInstance;
        /// <summary>
        /// Gets the instance.
        /// </summary>
        /// <value>The instance.</value>
        public static ControllerManager Instance
        {
            get
            {
                if (mInstance == null)
                {
                    mInstance = new ControllerManager();
                }
                return mInstance;
            }
        }
        private ActionCollection mActions;
        /// <summary>
        /// Gets the actions. <see cref="ActionCollection"/>
        /// </summary>
        /// <value>The actions.</value>
        public ActionCollection Actions
        {
            get
            {
                if (mActions == null)
                {
                    mActions = new ActionCollection(this);
                }
                return mActions;
            }
        }
        private MainForm mMainForm;
        /// <summary>
        /// Gets or sets the main form.
        /// </summary>
        /// <value>The main form.</value>
        public MainForm MainForm
        {
            get
            {
                return mMainForm;
            }
            set
            {
                Instance.UnregisterView(mMainForm, new MainFormDispatcher());
                mMainForm = value;
                if (value != null)
                {
                    Instance.RegisterView(value, new MainFormDispatcher());
                }

            }
        }

        private AssociationsForm mAssociationsForm;
        /// <summary>
        /// Gets or sets the associations form.
        /// </summary>
        /// <value>The associations form.</value>
        public AssociationsForm AssociationsForm
        {
            get { return mAssociationsForm; }
            set
            {
                Instance.UnregisterView(mAssociationsForm, new AssociationsFormDispatcher());
                mAssociationsForm = value;
                if (value != null)
                {
                    Instance.RegisterView(value, new AssociationsFormDispatcher());
                    value.DockStateChanged += new EventHandler(delegate
                    {
                        if (value.FloatPane != null && value.FloatPane.Parent != null)
                            value.FloatPane.Parent.MinimumSize = value.MinimumSize;
                    });
                }
            }
        }
        private CommandClassForm mCommandClassForm;
        /// <summary>
        /// Gets or sets the command class form.
        /// </summary>
        /// <value>The command class form.</value>
        public CommandClassForm CommandClassForm
        {
            get { return mCommandClassForm; }
            set
            {
                Instance.UnregisterView(mCommandClassForm, new CommandClassFormDispatcher());
                mCommandClassForm = value;
                if (value != null)
                {
                    Instance.RegisterView(value, new CommandClassFormDispatcher());
                    value.DockStateChanged += new EventHandler(delegate
                    {
                        if (value.FloatPane != null && value.FloatPane.Parent != null)
                            value.FloatPane.Parent.MinimumSize = value.MinimumSize;
                    });
                }
            }
        }

        private QueueActionsForm mQueueActionsForm;
        /// <summary>
        /// Gets or sets the command class form.
        /// </summary>
        /// <value>The command class form.</value>
        public QueueActionsForm QueueActionsForm
        {
            get { return mQueueActionsForm; }
            set
            {
                Instance.UnregisterView(mQueueActionsForm, new QueueActionsFormDispatcher());
                mQueueActionsForm = value;
                if (value != null)
                {
                    Instance.RegisterView(value, new QueueActionsFormDispatcher());
                }
            }
        }
        private RoutingTableForm mRoutingTableForm;
        public RoutingTableForm RoutingTableForm
        {
            get { return mRoutingTableForm; }
            set
            {
                Instance.UnregisterView(mRoutingTableForm, new RoutingTableFormDispatcher());
                mRoutingTableForm = value;
                if (value != null)
                {
                    Instance.RegisterView(value, new RoutingTableFormDispatcher());
                }
            }
        }
        private ControllerForm mControllerForm;
        /// <summary>
        /// Gets or sets the controller form.
        /// </summary>
        /// <value>The controller form.</value>
        public ControllerForm ControllerForm
        {
            get { return mControllerForm; }
            set
            {
                Instance.UnregisterView(mControllerForm, new ControllerFormDispatcher());
                mControllerForm = value;
                if (value != null)
                {
                    Instance.RegisterView(value, new ControllerFormDispatcher());
                    value.DockStateChanged += new EventHandler(delegate
                    {
                        if (value.FloatPane != null && value.FloatPane.Parent != null)
                            value.FloatPane.Parent.MinimumSize = value.MinimumSize;
                    });
                }
            }
        }
        private LogForm mLogForm;
        /// <summary>
        /// Gets or sets the log form.
        /// </summary>
        /// <value>The log form.</value>
        public LogForm LogForm
        {
            get { return mLogForm; }
            set
            {
                Instance.UnregisterView(mLogForm, new LogFormDispatcher());
                mLogForm = value;
                if (value != null)
                {
                    Instance.RegisterView(value, new LogFormDispatcher());
                    value.DockStateChanged += new EventHandler(delegate
                    {
                        if (value.FloatPane != null && value.FloatPane.Parent != null)
                            value.FloatPane.Parent.MinimumSize = value.MinimumSize;
                    });
                }
            }
        }
        private NodeForm mNodeForm;
        /// <summary>
        /// Gets or sets the node form.
        /// </summary>
        /// <value>The node form.</value>
        public NodeForm NodeForm
        {
            get { return mNodeForm; }
            set
            {
                Instance.UnregisterView(mNodeForm, new NodeFormDispatcher());
                mNodeForm = value;
                if (value != null)
                {
                    Instance.RegisterView(value, new NodeFormDispatcher());
                    value.DockStateChanged += new EventHandler(delegate
                    {
                        if (value.FloatPane != null && value.FloatPane.Parent != null)
                            value.FloatPane.Parent.MinimumSize = value.MinimumSize;
                    });
                }
            }
        }

        private ErttForm mErttForm;

        /// <summary>
        /// Gets or sets the ERTT form.
        /// </summary>
        /// <value>The ERTT form.</value>
        public ErttForm ErttForm
        {
            get { return mErttForm; }
            set
            {
                Instance.UnregisterView(mErttForm, new ErttFormDispatcher());
                mErttForm = value;
                if (value != null)
                {
                    Instance.RegisterView(value, new ErttFormDispatcher());
                    value.DockStateChanged += new EventHandler(delegate
                    {
                        if (value.FloatPane != null && value.FloatPane.Parent != null)
                            value.FloatPane.Parent.MinimumSize = value.MinimumSize;
                    });
                }
            }
        }

        private SetupRouteForm mSetupRouteForm;
        /// <summary>
        /// Gets or sets the setup route form.
        /// </summary>
        /// <value>The setup route form.</value>
        public SetupRouteForm SetupRouteForm
        {
            get { return mSetupRouteForm; }
            set
            {
                Instance.UnregisterView(mSetupRouteForm, new SetupRouteFormDispatcher());
                mSetupRouteForm = value;
                if (value != null)
                {
                    Instance.RegisterView(value, new SetupRouteFormDispatcher());
                    value.DockStateChanged += new EventHandler(delegate
                    {
                        if (value.FloatPane != null && value.FloatPane.Parent != null)
                            value.FloatPane.Parent.MinimumSize = value.MinimumSize;
                    });
                }
            }
        }

        private ZWaveManager mZWaveManager;
        /// <summary>
        /// Gets or sets the ZWave manager.
        /// </summary>
        /// <value>The Z wave manager.</value>
        public ZWaveManager ZWaveManager
        {
            get { return mZWaveManager; }
            set { mZWaveManager = value; }
        }
        /// <summary>
        /// Registers the view.
        /// </summary>
        /// <param name="view">The view.</param>
        /// <param name="dispatcher">The dispatcher.</param>
        public void RegisterView(IView view, IViewDispatcher dispatcher)
        {
            dispatcher.Bind(view, Actions, DocumentModel);
        }
        /// <summary>
        /// Unregisters the view.
        /// </summary>
        /// <param name="view">The view.</param>
        /// <param name="dispatcher">The dispatcher.</param>
        public void UnregisterView(IView view, IViewDispatcher dispatcher)
        {
            if (view != null)
                dispatcher.Drop(view, Actions, DocumentModel);
        }

        /// <summary>
        /// Starts the application.
        /// </summary>
        /// <param name="args">The args.</param>
        public static void Start(string[] args)
        {
            Application.EnableVisualStyles();
            Application.SetCompatibleTextRenderingDefault(false);
            AppDomain.CurrentDomain.ProcessExit += new EventHandler(OnProcessExit);
            AppDomain.CurrentDomain.UnhandledException += new UnhandledExceptionEventHandler(OnUnhandledException);
            Instance.MainForm = new MainForm();
            Instance.Initialize(ApplicationMode.GUI);
            Application.Run(Instance.MainForm);
        }

        /// <summary>
        /// Called when [process exit].
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.EventArgs"/> instance containing the event data.</param>
        static void OnProcessExit(object sender, EventArgs e)
        {
            if (mInstance != null)
                mInstance.Dispose();
        }
        /// <summary>
        /// Called when [unhandled exception].
        /// </summary>
        /// <param name="sender">The sender.</param>
        /// <param name="e">The <see cref="System.UnhandledExceptionEventArgs"/> instance containing the event data.</param>
        static void OnUnhandledException(object sender, UnhandledExceptionEventArgs e)
        {
            //LogManager.Instance.WriteLog(string.Format("Exception {0} occured", e.ExceptionObject.GetType().FullName), LogDetailLevel.Short);
            //NotificationForm nf = new NotificationForm(e.ExceptionObject.GetType().FullName, e.ExceptionObject.ToString());
            //nf.Text = "Error";
            ////((Exception)e.ExceptionObject).TargetSite.GetMethodBody()
            //switch (nf.ShowDialog())
            //{
            //    case DialogResult.Abort:
            //        Application.Exit();
            //        break;
            //    case DialogResult.Ignore:
            //        return;
            //    case DialogResult.OK:
            //        break;
            //    default:
            //        break;
            //}
        }
        #region IDisposable Members

        public void Dispose()
        {
            UnregisterView(this.MainForm, new MainFormDispatcher());
        }

        #endregion

        private bool isBusy = false;
        private ProgressForm progressForm;
        internal object DoAction(QueueAction action)
        {
            return DoAction(action.ActionHandler, action.ActionHandlerParameters, action.ProgressInfo, action.CanCancel, action.CommandType);
        }

        internal object DoAction(Delegate eventHandler, object[] parameters, string progressInfo, bool canCancel, byte commandType)
        {
            if (isBusy) return null;
            isBusy = true;
            try
            {
                Application.DoEvents();
            }
            catch (Exception)
            {
            }
            if (progressForm == null)
            {
                progressForm = new ProgressForm();
                progressForm.OnCancel += new EventHandler(ProgressFormOnCancel);
            }
            progressForm.ProgressDelegate = eventHandler;
            progressForm.ProgressDelegateParameters = parameters;
            progressForm.ProgressInfo = progressInfo;
            progressForm.CanCancel = canCancel;
            progressForm.CommandType = commandType;
            progressForm.ShowDialog();
            if (progressForm.ProgressException != null)
            {
                if (progressForm.ProgressException is RequestTimeoutException)
                {
                    this.ShowMessage(string.Format("{0} request timeout exceeded", eventHandler.Method.Name), true);
                }
                else
                {
                    this.ShowMessage(progressForm.ProgressException.Message, true);
                }
            }
            isBusy = false;
            return progressForm.Result;
        }

        private object lockerNWI = new object();
        private bool mNetworkWideInclusionCanceled;
        public bool NetworkWideInclusionCanceled
        {
            get
            {
                lock (lockerNWI)
                {
                    return mNetworkWideInclusionCanceled;
                }
            }
            set
            {
                lock (lockerNWI)
                {
                    mNetworkWideInclusionCanceled = value;
                }
            }
        }

        void ProgressFormOnCancel(object sender, EventArgs e)
        {
            ProgressForm form = (ProgressForm)sender;
            if (DocumentModel.Controller != null)
            {
                IController device = DocumentModel.Controller;
                try
                {
                    NetworkWideInclusionCanceled = true;
                    Instance.Actions.NodeFormActions.OnAbortAction(form.CommandType, device);
                }
                catch (Exception ex)
                {
                    throw ex;
                }
            }
            isBusy = false;
        }

        public void ShowView(Type type, bool show)
        {
            if (type == typeof(AssociationsForm))
            {
                if (AssociationsForm == null || AssociationsForm.IsDisposed)
                {
                    AssociationsForm = new AssociationsForm();
                }
                if (show)
                {
                    AssociationsForm.Show(MainForm.DockPanelMain);
                }
                else
                {
                    AssociationsForm.Hide();
                }
            }
            else if (type == typeof(CommandClassForm))
            {
                if (CommandClassForm == null || CommandClassForm.IsDisposed)
                {
                    CommandClassForm = new CommandClassForm();
                    CommandClassForm.FloatPane.MinimumSize = new System.Drawing.Size(1000, 1000);
                }
                if (show)
                {
                    CommandClassForm.Show(MainForm.DockPanelMain);
                }
                else
                {
                    CommandClassForm.Hide();
                }
            }
            else if (type == typeof(ControllerForm))
            {
                if (ControllerForm == null || ControllerForm.IsDisposed)
                {
                    ControllerForm = new ControllerForm();
                }
                if (show)
                {
                    ControllerForm.Show(MainForm.DockPanelMain);
                }
                else
                {
                    ControllerForm.Hide();
                }

            }
            else if (type == typeof(NodeForm))
            {
                if (NodeForm == null || NodeForm.IsDisposed)
                {
                    NodeForm = new NodeForm();
                }
                if (show)
                {
                    NodeForm.Show(MainForm.DockPanelMain);
                }
                else
                {
                    NodeForm.Hide();
                }
            }
            else if (type == typeof(SetupRouteForm))
            {
                if (SetupRouteForm == null || SetupRouteForm.IsDisposed)
                {
                    SetupRouteForm = new SetupRouteForm();
                }
                if (show)
                {
                    SetupRouteForm.Show(MainForm.DockPanelMain);
                }
                else
                {
                    SetupRouteForm.Hide();
                }
            }
            else if (type == typeof(LogForm))
            {
                if (LogForm == null || LogForm.IsDisposed)
                {
                    LogForm = new LogForm();
                }
                if (show)
                {
                    LogForm.Show(MainForm.DockPanelMain);
                }
                else
                {
                    LogForm.Hide();
                }
            }
            else if (type == typeof(ErttForm))
            {
                if (ErttForm == null || ErttForm.IsDisposed)
                {
                    ErttForm = new ErttForm();
                }
                if (show)
                {
                    ErttForm.Show(MainForm.DockPanelMain);
                }
                else
                {
                    ErttForm.Close();
                }
            }
            else if (type == typeof(RoutingTableForm))
            {
                if (RoutingTableForm == null || RoutingTableForm.IsDisposed)
                {
                    RoutingTableForm = new RoutingTableForm();
                }
                if (show)
                {
                    RoutingTableForm.Show(MainForm.DockPanelMain);
                }
                else
                {
                    RoutingTableForm.Hide();
                }
            }
        }


        internal void ShowMessage(string message, bool isError)
        {
            this.MainForm.BeginInvoke(new EventHandler(delegate
            {
                MessageBox.Show(this.MainForm, message, AboutBox.AssemblyProduct, MessageBoxButtons.OK, (isError == true) ? MessageBoxIcon.Exclamation : MessageBoxIcon.Information);
            }));
        }
        private Dictionary<byte, Queue<QueueAction>> mFramesToSend = new Dictionary<byte, Queue<QueueAction>>();
        /// <summary>
        /// The frames queue
        /// </summary>
        /// <value>The frames to send.</value>
        public Dictionary<byte, Queue<QueueAction>> FramesToSend
        {
            get { return mFramesToSend; }
        }

        public void AddQueueCommand(byte nodeId, QueueAction action)
        {
            Queue<QueueAction> queue = null;
            if (FramesToSend.ContainsKey(nodeId))
            {
                queue = FramesToSend[nodeId];
            }
            else
            {
                queue = new Queue<QueueAction>();
            }
            queue.Enqueue(action);
            FramesToSend[nodeId] = queue;
            this.MainForm.LabelCommandQueue = queue.Count.ToString();
            //System.Diagnostics.Debug.WriteLine("Queue changed.");
        }
        public void RemoveQueueCommand(byte nodeId)
        {
            if (FramesToSend.ContainsKey(nodeId))
            {
                Queue<QueueAction> queue = FramesToSend[nodeId];
                queue.Dequeue();
                this.MainForm.LabelCommandQueue = queue.Count.ToString();
            }
        }

        public void ClearQueueCommands()
        {
            FramesToSend.Clear();
        }
    }
}
