using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.InstallerTool.Classes;
using Zensys.ZWave.InstallerTool.Controllers;
using Zensys.ZWave.Devices;
using Zensys.ZWave.InstallerTool.Properties;
using Zensys.ZWave.Enums;
using Zensys.ZWave;
using Zensys.Framework;
using Zensys.ZWave.SerialPortApplication.Devices;
using System.Windows.Forms;
using Zensys.ZWave.ZWaveHAL.Actions;
using Zensys.ZWave.ZWaveHAL;
using Zensys.ZWave.Events;

namespace Zensys.ZWave.InstallerTool.Actions
{
    public class ControllerFormActions : BaseAction
    {
        /// <summary>
        /// SUC capabilities used in ZW_EnableSUC - Enable basic SUC functionality
        /// </summary>
        private const int ZW_SUC_FUNC_BASIC_SUC = 0x00;      //    ZW_SUC_FUNC_BASIC_SUC
        /// <summary>
        /// SUC capabilities used in ZW_EnableSUC - Enable basis SUC and NodeID
        /// Server functionalities
        /// </summary>
        private const int ZW_SUC_FUNC_NODEID_SERVER = 0x01;  // ZW_SUC_FUNC_NODEID_SERVER
        ControllerViewActions ControllerHAL;
        public ControllerFormActions(ControllerManager controller)
            : base(controller)
        {
            ControllerHAL = CustomLoader.CreateControllerViewActionsInstance();
            ControllerHAL.Initialize(controller.LogManager, controller.ExceptionManager, controller.XmlDataManager);
        }

        #region Form
        public void OnFormLoad(object sender, EventArgs e)
        {
            ControllerManager.ControllerForm.vcCreateNewPrimaryToolStripButton.Visible = false;
            ControllerManager.ControllerForm.vcRequestUpdateToolStripButton.Visible = false;
            ControllerManager.ControllerForm.vcControllerInfoTreeView.ImageList = ControllerManager.ControllerForm.controllerViewControl.Images;
            OnDocumentModelStateChanged();
        }
        public void OnFormClosing(object sender, FormClosingEventArgs e)
        {
            ControllerManager.MainForm.ControllerToolStripMenuItem.Checked = false;
            ControllerManager.ControllerForm.Hide();
            e.Cancel = true;
        }
        #endregion

        #region DocumentModel
        public void OnDocumentModelStateChanged(object sender, EventArgs e)
        {
            OnDocumentModelStateChanged();
        }

        private delegate void OnDocumentModelStateChangedDelegate();
        private void OnDocumentModelStateChanged()
        {
            if (ControllerManager.ControllerForm != null && !ControllerManager.ControllerForm.IsDisposed)
            {

                if (ControllerManager.ControllerForm.InvokeRequired)
                {
                    ControllerManager.ControllerForm.Invoke(new OnDocumentModelStateChangedDelegate(OnDocumentModelStateChanged));
                }
                else
                {
                    if (ControllerManager.DocumentModel.Controller != null)
                    {
                        ControllerManager.DocumentModel.Controller.ControllerUpdated -= new Zensys.ZWave.Events.ControllerUpdatedEventHandler(ControllerUpdated);
                        ControllerManager.DocumentModel.Controller.ControllerUpdated += new Zensys.ZWave.Events.ControllerUpdatedEventHandler(ControllerUpdated);
                        ControllerRoles role = ControllerManager.DocumentModel.Controller.NetworkRole;

                        FillTreeInfo();

                        ControllerManager.ControllerForm.controllerViewControl.vcSisRadioButton.Checked =
                            (role & ControllerRoles.SIS) != 0 || enabledRole == ControllerRoles.SIS;

                        ControllerManager.ControllerForm.controllerViewControl.vcSucRadioButton.Checked =
                            ((role & ControllerRoles.SIS) == 0 && (role & ControllerRoles.SUC) != 0) || enabledRole == ControllerRoles.SUC;

                        ControllerManager.ControllerForm.controllerViewControl.vcNoneRadioButton.Checked =
                            (role & ControllerRoles.SUC) == 0 && enabledRole == ControllerRoles.None;

                        ControllerManager.ControllerForm.vcControllerInfoTreeView.Enabled = true;

                        ControllerManager.ControllerForm.vcCmdAssign.Enabled =
                            ControllerManager.DocumentModel.Controller.Version.Library == Libraries.NoLib ||
                            ControllerManager.DocumentModel.Devices.Count == 1;

                        ControllerManager.ControllerForm.vcNoneRadioButton.Enabled =
                            (ControllerManager.DocumentModel.Controller.Version.Library == Libraries.NoLib ||
                            ControllerManager.DocumentModel.Devices.Count == 1) &&
                            (role & ControllerRoles.SUC) == 0;

                        ControllerManager.ControllerForm.vcSucRadioButton.Enabled =
                            ControllerManager.DocumentModel.Controller.Version.Library == Libraries.NoLib ||
                            ControllerManager.DocumentModel.Devices.Count == 1;

                        ControllerManager.ControllerForm.vcSisRadioButton.Enabled =
                            ControllerManager.DocumentModel.Controller.Version.Library == Libraries.NoLib ||
                            ControllerManager.DocumentModel.Devices.Count == 1;

                        ControllerManager.ControllerForm.vcShiftToolStripButton.Enabled =
                            (role & ControllerRoles.SUC) == 0 &&
                            (role & ControllerRoles.RealPrimary) != 0;

                        ControllerManager.ControllerForm.vcStartLearnModeToolStripButton.Enabled =
                            (((role & ControllerRoles.SUC) != 0 &&
                            (role & ControllerRoles.SIS) != 0 &&
                            ControllerManager.DocumentModel.Devices.Count == 1) ||
                            (ControllerManager.DocumentModel.Devices.Count == 1 ||
                            (role & ControllerRoles.RealPrimary) == 0)) &&
                            (!((role & ControllerRoles.SUC) != 0 && (role & ControllerRoles.SIS) == 0 && (role & ControllerRoles.RealPrimary) != 0));

                        ControllerManager.ControllerForm.vcCreateNewPrimaryToolStripButton.Enabled =
                            (role & ControllerRoles.SUC) != 0 &&
                            (role & ControllerRoles.SIS) == 0;

                        ControllerManager.ControllerForm.vcRequestUpdateToolStripButton.Enabled =
                              (role & ControllerRoles.SUC) == 0 &&
                            ControllerManager.DocumentModel.Controller.SucNodeId != 0x00 &&
                            (role & ControllerRoles.Inclusion) != 0;

                        ControllerManager.ControllerForm.vcResetToolStripButton.Enabled = true;
                    }
                    else
                    {
                        ControllerManager.ControllerForm.vcControllerInfoTreeView.BeginUpdate();
                        ControllerManager.ControllerForm.vcControllerInfoTreeView.Nodes.Clear();
                        ControllerManager.ControllerForm.vcControllerInfoTreeView.EndUpdate();

                        ControllerManager.ControllerForm.vcSisRadioButton.Checked = false;
                        ControllerManager.ControllerForm.vcSucRadioButton.Checked = false;
                        ControllerManager.ControllerForm.vcNoneRadioButton.Checked = false;

                        ControllerManager.ControllerForm.vcCmdAssign.Enabled = false;
                        ControllerManager.ControllerForm.vcControllerInfoTreeView.Enabled = false;
                        ControllerManager.ControllerForm.vcCreateNewPrimaryToolStripButton.Enabled = false;
                        ControllerManager.ControllerForm.vcNoneRadioButton.Enabled = false;
                        ControllerManager.ControllerForm.vcRequestUpdateToolStripButton.Enabled = false;
                        ControllerManager.ControllerForm.vcResetToolStripButton.Enabled = false;
                        ControllerManager.ControllerForm.vcShiftToolStripButton.Enabled = false;
                        ControllerManager.ControllerForm.vcSisRadioButton.Enabled = false;
                        ControllerManager.ControllerForm.vcStartLearnModeToolStripButton.Enabled = false;
                        ControllerManager.ControllerForm.vcSucRadioButton.Enabled = false;
                    }
                }
            }
        }

        private void FillTreeInfo()
        {
            ControllerManager.ControllerForm.vcControllerInfoTreeView.BeginUpdate();
            ControllerManager.ControllerForm.vcControllerInfoTreeView.Nodes.Clear();
            TreeNode root = new TreeNode(string.Format("Id: {0}", Tools.ToHexString(ControllerManager.DocumentModel.Controller.Id)), 0, 0);
            root.Nodes.Add("Home Id", string.Format("Home Id: {0}", Tools.ToHexString(ControllerManager.DocumentModel.Controller.HomeId)), 1, 1);
            root.Nodes.Add("Network Role", string.Format("Network Role: {0}", ControllerManager.DocumentModel.Controller.NetworkRole.ToString()), 1, 1);
            root.Nodes.Add("Serial Port", string.Format("Serial Port: {0}", ControllerManager.DocumentModel.Controller.SerialPort), 1, 1);
            root.Nodes.Add("Z-Wave library", string.Format("Z-Wave library: {0}", ControllerManager.DocumentModel.Controller.Version.Library.ToString()), 1, 1);
            root.Nodes.Add("Z-Wave device chip", string.Format("Z-Wave device chip: ZW{0}{1}", ControllerManager.DocumentModel.Controller.ChipType.ToString("00"), ControllerManager.DocumentModel.Controller.ChipRevision.ToString("00")), 1, 1);
            root.Nodes.Add("Z-Wave device firmware", string.Format("Z-Wave device firmware: {0}", ControllerManager.DocumentModel.Controller.Version.Version), 1, 1);
            TreeNode libNode = new TreeNode("Libraries", 1, 1);
            libNode.Nodes.Add("HAL", string.Format("High Application Layer library: {0}", GetAssemblyVersionString(ControllerHAL)), 2, 2);
            libNode.Nodes.Add("HAL", string.Format("Application Layer library: {0}", GetAssemblyVersionString(ControllerManager.ZWaveManager.ApplicationLayer)), 2, 2);
            libNode.Nodes.Add("HAL", string.Format("Session Layer library: {0}", GetAssemblyVersionString(ControllerManager.ZWaveManager.SessionLayer)), 2, 2);
            libNode.Nodes.Add("HAL", string.Format("Frame Layer library: {0}", GetAssemblyVersionString(ControllerManager.ZWaveManager.FrameLayer)), 2, 2);
            libNode.Nodes.Add("HAL", string.Format("Transport Layer library: {0}", GetAssemblyVersionString(ControllerManager.ZWaveManager.TransportLayer)), 2, 2);
            root.Nodes.Add(libNode);
            ControllerManager.ControllerForm.vcControllerInfoTreeView.Nodes.Add(root);
            root.Expand();
            ControllerManager.ControllerForm.vcControllerInfoTreeView.EndUpdate();
        }

        private string GetAssemblyVersionString(object Object)
        {
            Type type = Object.GetType();
            return string.Format("ver. {0}", type.Assembly.GetName().Version.ToString());
        }
        #endregion

        #region Other
        public void OnShiftControllerClick(object sender, EventArgs e)
        {
            ControllerManager.DoAction(
               new ProcedureCaller<IController>(ControllerHAL.ShiftControllerAction),
               new object[] { ControllerManager.DocumentModel.Controller },
               Resources.MsgShiftController, true, (byte)CommandTypes.None);
            ControllerManager.Actions.CommonActions.SetUpController();
        }

        public void OnResetControllerClick(object sender, EventArgs e)
        {
            ControllerManager.DoAction(
               new ProcedureCaller<IController>(ControllerHAL.ResetControllerAction),
               new object[] { ControllerManager.DocumentModel.Controller },
               Resources.MsgResetController, true, (byte)CommandTypes.None);
            //EnableSucSis(ControllerRoles.SIS);
            ControllerManager.Actions.CommonActions.SetUpController();
        }

        public void OnStartLearnModeClick(object sender, EventArgs e)
        {
            LearnMode lmode = LearnMode.Unknown;
            ControllerManager.DoAction(
               new ProcedureCaller<IController>(ControllerHAL.InitLearnModeAction),
               new object[] { ControllerManager.DocumentModel.Controller },
               Resources.MsgStartLearnMode, true, (byte)CommandTypes.CmdZWaveSetLearnMode);

            lmode = (LearnMode)ControllerManager.DoAction(
               new FunctionCaller<LearnMode, IController>(ControllerHAL.SetLearnModeAction),
               new object[] { ControllerManager.DocumentModel.Controller },
               Resources.MsgStartLearnMode, true, (byte)CommandTypes.CmdZWaveSetLearnMode);

            if (lmode != LearnMode.Done)
            {
                ControllerManager.ShowMessage(Resources.ErrLearnMode, true);
            }
            ControllerManager.Actions.CommonActions.SetUpController();
        }

        public void OnRequestNetworkUpdateClick(object sender, EventArgs e)
        {
            ControllerManager.DoAction(
              new ProcedureCaller<IController>(ControllerHAL.RequestNetworkUpdateAction),
              new object[] { ControllerManager.DocumentModel.Controller },
              Resources.MsgRequestNetworkUpdate, true, (byte)CommandTypes.None);
            ControllerManager.Actions.CommonActions.SetUpController();
        }

        public void OnCreateNewPrimaryClick(object sender, EventArgs e)
        {
            ControllerManager.DoAction(
                new ProcedureCaller<IController>(ControllerHAL.CreateNewPrimaryAction),
                new object[] { ControllerManager.DocumentModel.Controller },
                Resources.MsgCreateNewPrimary, true, (byte)CommandTypes.None);
            ControllerManager.Actions.CommonActions.SetUpController();
        }
        public void OnLoadCClassesFromDeviceMemory(object sender, EventArgs e)
        {
            ControllerManager.DoAction(
                new ProcedureCaller<IController>(ControllerHAL.LoadCClassesFromDeviceMemory),
                new object[] { ControllerManager.DocumentModel.Controller },
                Resources.MsgControllerUpdate, true, (byte)CommandTypes.CmdMemoryGetBuffer);
            foreach (IDeviceInfo device in ControllerManager.DocumentModel.Devices)
            {
                device.SupportedCommandClasses = ControllerManager.DocumentModel.Controller.CommandClassesStore.CmdClasses(device.Id);
            }
        }

        public void OnSaveCClassesToDeviceMemory(object sender, EventArgs e)
        {
            ControllerManager.DoAction(
               new ProcedureCaller<IController>(ControllerHAL.SaveCClassesToDeviceMemory),
               new object[] { ControllerManager.DocumentModel.Controller },
               Resources.MsgControllerUpdate, true, (byte)CommandTypes.CmdMemoryPutBuffer);
        }
        public void OnSISNetworkRoleOptionClick(object sender, EventArgs e)
        {
            EnableSucSis(ControllerRoles.SIS);
        }

        public void OnSUCNetworkRoleOptionClick(object sender, EventArgs e)
        {
            EnableSucSis(ControllerRoles.SUC);
        }

        public void OnNoneNetworkRoleOptionClick(object sender, EventArgs e)
        {
            EnableSucSis(ControllerRoles.None);
        }

        public void OnAssignNetworkRoleOptionClick(object sender, EventArgs e)
        {
            bool sucState = false;
            byte select_SUC_or_SIS = 0;
            if (ControllerManager.ControllerForm.controllerViewControl.vcSucRadioButton.Checked)
            {
                select_SUC_or_SIS = ZW_SUC_FUNC_BASIC_SUC;
                sucState = true;
            }
            else if (ControllerManager.ControllerForm.controllerViewControl.vcSisRadioButton.Checked)
            {
                select_SUC_or_SIS = ZW_SUC_FUNC_NODEID_SERVER;
                sucState = true;
            }

            ControllerManager.DoAction(
            new ProcedureCaller<IController, bool, byte>(ControllerHAL.AssignNetworkRoleOptionAction),
            new object[] { ControllerManager.DocumentModel.Controller, sucState, select_SUC_or_SIS },
            Resources.MsgAssignNetworkOption, true, (byte)CommandTypes.None);
            ControllerManager.Actions.CommonActions.SetUpController();
        }

        private ControllerRoles enabledRole = ControllerRoles.None;
        public void EnableSucSis(ControllerRoles controllerRole)
        {
            if (ControllerManager.DocumentModel.Controller != null)
            {
                ControllerManager.DoAction(
                    new ProcedureCaller<IController, ControllerRoles>(ControllerHAL.EnableSucSisAction),
                    new object[] { ControllerManager.DocumentModel.Controller, controllerRole },
                    Resources.MsgEnableSucSis, true, (byte)CommandTypes.None);
            }
            enabledRole = controllerRole;
        }
        #endregion

        public void ControllerUpdated(Zensys.ZWave.Events.ControllerUpdatedEventArgs args)
        {
            if (args.ChangedNode != null)
            {
                UpdateControllerDataDelegate updateControllerDataDelegate = new UpdateControllerDataDelegate(UpdateControllerData);
                updateControllerDataDelegate.BeginInvoke(args.ChangedNode, args.ChangedStatus, UpdateControllerDataCallback, updateControllerDataDelegate);
            }
        }

        private bool isNeededCustomAfterLearnModeActions = false;
        public void LearnModeStatusChanged(LearnModeStatusChangedEventArgs args)
        {
            if (args.Status == LearnMode.Started)
            {
                isNeededCustomAfterLearnModeActions = true;
            }
            else if (args.Status == LearnMode.Done)
            {
                ProcedureCaller caller = new ProcedureCaller(ControllerManager.Actions.CommonActions.SetUpController);
                caller.BeginInvoke(null, null);
            }
        }

        private delegate void UpdateControllerDataDelegate(IDeviceInfo node, ApplicationControllerUpdateStatuses changedStatus);
        /// <summary>
        /// Updates the controller data.
        /// </summary>
        /// <param name="node">The node.</param>
        /// <param name="changedStatus">The changed status.</param>
        private void UpdateControllerData(IDeviceInfo node, ApplicationControllerUpdateStatuses changedStatus)
        {
            if (node.Id != ControllerManager.DocumentModel.Controller.Id)
            {
                if (changedStatus == ApplicationControllerUpdateStatuses.ADD_DONE)
                {
                    if (ControllerManager.DocumentModel.Controller.ConnectionStatus == Zensys.ZWave.Enums.ConnectionStatuses.Closed)
                    {
                        ControllerManager.DocumentModel.Controller.Open(ControllerManager.DocumentModel.Controller.SerialPort);
                    }
                    IDeviceInfo _addedNode = ControllerManager.DocumentModel.Controller.GetProtocolInfo(node.Id, false);
                    node.Capability = _addedNode.Capability;
                    node.Security = _addedNode.Security;
                    node.Reserved = _addedNode.Reserved;
                    if (node.SupportedCommandClasses != null)
                    {
                        foreach (byte b in node.SupportedCommandClasses)
                        {
                            ControllerManager.DocumentModel.Controller.CommandClassesStore.Store(node.Id, b);
                        }
                    }
                    ControllerManager.Actions.NodeFormActions.AddControllerNode(node);
                }
                else if (changedStatus == ApplicationControllerUpdateStatuses.DELETE_DONE)
                {
                    ControllerManager.Actions.NodeFormActions.RemoveControllerNode(node.Id);
                }
                else if (changedStatus == ApplicationControllerUpdateStatuses.SUC_ID)
                {
                    //DoActions cannot be called from here! SetUpController calling after learn mode done.

                    //ControllerManager.Actions.CommonActions.SetUpController();//replication from another controller called
                    //OnDocumentModelStateChanged();
                }
            }
            else if (changedStatus == ApplicationControllerUpdateStatuses.SUC_ID)
            {
                ControllerManager.DocumentModel.Controller.GetControllerCapabilities();
                ControllerManager.DocumentModel.SetDocumentModelStateChanged();
            }
        }
        private void UpdateControllerDataCallback(IAsyncResult asyncResult)
        {
            UpdateControllerDataDelegate caller = (UpdateControllerDataDelegate)asyncResult.AsyncState;
            caller.EndInvoke(asyncResult);
        }
    }
}