using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.ComponentModel;

namespace Zensys.Framework.UI.Controls
{
    public class TreeView2D : TreeView
    {
        public delegate void BuildTreeDelegate();
        public delegate PrefetchedObject CreatePrefetchedObjectDelegate(object currentObject);
        public delegate IBindingList GetInnerDataSourceDelegate(object currentObject);
        public delegate CurrencyManager GetInnerCurrencyManagerDelegate(object currentObject);
        
        private IBindingList mDataSource;
        private CurrencyManager mCurrencyManager = null;

        private CreatePrefetchedObjectDelegate mCreatePrefetchedObject;
        public CreatePrefetchedObjectDelegate CreatePrefetchedObject
        {
            set { mCreatePrefetchedObject = value; }
        }

        private GetInnerDataSourceDelegate mGetInnerDataSource;
        public GetInnerDataSourceDelegate GetInnerDataSource
        {
            set { mGetInnerDataSource = value; }
        }

        private GetInnerCurrencyManagerDelegate mGetInnerCurrencyManager;
        public GetInnerCurrencyManagerDelegate GetInnerCurrencyManager
        {
            set { mGetInnerCurrencyManager = value; }
        }

        [Category("Data")]
        public IBindingList DataSource
        {
            get
            {
                return mDataSource;
            }
            set
            {
                if (value != null)
                {
                    mDataSource = value;
                    mDataSource.ListChanged += new ListChangedEventHandler(mDataSource_ListChanged);
                    mCurrencyManager = (CurrencyManager)this.BindingContext[value];
                    BuildTree();
                }
                else
                {
                    this.mCurrencyManager = null;
                    this.Nodes.Clear();
                }
            }
        }

        void mDataSource_ListChanged(object sender, ListChangedEventArgs e)
        {
            BuildTree();
        }

        private SortedDictionary<string, IBindingList> mInnerDataSources = new SortedDictionary<string, IBindingList>();
        private SortedDictionary<string, CurrencyManager> mInnerCurrencyManagers = new SortedDictionary<string, CurrencyManager>();

        public void BuildTree()
        {
            if (this.InvokeRequired)
            {
                this.Invoke(new BuildTreeDelegate(BuildTree));
            }
            else
            {
                if (mCreatePrefetchedObject == null || mGetInnerDataSource == null || mGetInnerCurrencyManager == null)
                {
                    throw new InvalidOperationException("You must specify CreatePrefetchedObject, GetInnerDataSource and GetInnerCurrencyManager callbacks");
                }
                BeginUpdate();
                if (this.Nodes != null && this.Nodes.Count > 0)
                {
                    GetNodesVisibleState(this.Nodes[0].Nodes);
                }
                this.Nodes.Clear();
                mInnerDataSources.Clear();
                mInnerCurrencyManagers.Clear();
                foreach (object dataObj in mDataSource)
                {
                    PrefetchedObject prefetchedObject = mCreatePrefetchedObject(dataObj);
                    TreeNode rootNode = new TreeNode(prefetchedObject.Text);
                    rootNode.Tag = dataObj;
                    rootNode.ImageIndex = prefetchedObject.ImageIndex;
                    rootNode.SelectedImageIndex = prefetchedObject.ImageIndex;
                    Nodes.Add(rootNode);
                    IBindingList innerDS = mGetInnerDataSource(dataObj);
                    innerDS.ListChanged += new ListChangedEventHandler(innerDS_ListChanged);
                    mInnerDataSources.Add(rootNode.Text, innerDS);
                    mInnerCurrencyManagers.Add(rootNode.Text, mGetInnerCurrencyManager(dataObj));
                    AddInnerNodes(rootNode);
                }
                if (this.Nodes != null && this.Nodes.Count > 0)
                {
                    SetNodesVisibleState(this.Nodes[0].Nodes);
                    this.Nodes[0].Expand();
                }
                EndUpdate();
            }
        }

        void innerDS_ListChanged(object sender, ListChangedEventArgs e)
        {
            BuildTree();
        }

        private List<bool> nodesVisibilityVector = new List<bool>();

        private void SetNodesVisibleState(TreeNodeCollection treeNodeCollection)
        {
            int position = 0;
            if (nodesVisibilityVector.Count > 0)
            {
                foreach (TreeNode node in treeNodeCollection)
                {
                    if (nodesVisibilityVector.Count > position && nodesVisibilityVector[position])
                    {
                        node.Expand();
                    }
                    else
                    {
                        node.Collapse();
                    }
                    position++;
                }
            }
        }

        private void GetNodesVisibleState(TreeNodeCollection treeNodeCollection)
        {
            nodesVisibilityVector.Clear();
            if (treeNodeCollection != null)
            {
                foreach (TreeNode node in treeNodeCollection)
                {
                    nodesVisibilityVector.Add(node.IsExpanded);
                }
            }
        }

        private void AddInnerNodes(TreeNode rootNode)
        {
            foreach (object innerDataObj in mInnerDataSources[rootNode.Text])
            {
                PrefetchedObject prefetchedObject = mCreatePrefetchedObject(innerDataObj);
                TreeNode node = new TreeNode(prefetchedObject.Text);
                node.ImageIndex = prefetchedObject.ImageIndex;
                node.SelectedImageIndex = prefetchedObject.ImageIndex;
                node.Tag = innerDataObj;
                rootNode.Nodes.Add(node);
            }
        }

        private void InitializeComponent()
        {
        }

        public event EventHandler SelectedChanged;

        protected override void OnAfterSelect(TreeViewEventArgs e)
        {
            base.OnAfterSelect(e);
            int position = -1;
            if (e.Node.Parent != null)
            {
                position = -1;
                foreach (object ro in mDataSource)
                {
                    position++;
                    if (ro == e.Node.Parent.Tag)
                    {
                        break;
                    }
                }
                mCurrencyManager.Position = position;
                position = -1;
                foreach (object ro in mInnerDataSources[e.Node.Parent.Text])
                {
                    position++;
                    if (ro.Equals(e.Node.Tag))
                    {
                        break;
                    }
                }
                mInnerCurrencyManagers[e.Node.Parent.Text].Position = position;
            }
            else
            {
                position = -1;
                foreach (object ro in mDataSource)
                {
                    position++;
                    if (ro == e.Node.Tag)
                    {
                        break;
                    }
                }
                mCurrencyManager.Position = position;
            }

            if (SelectedChanged != null)
            {
                SelectedChanged(this, EventArgs.Empty);
            }

        }
    }
}
