using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using System.Runtime.InteropServices;

namespace Zensys.Framework.UI.Controls
{
    public class ToolStripNumericTextBox : ToolStripTextBox
    {
        [DllImport("user32", CharSet = CharSet.Auto, SetLastError = true)]
        internal static extern bool MessageBeep(int type);

        #region "Variables"
        private int ii_ScaleOnLostFocus = 0;
        private Decimal idec_InternalValue = 0;
        private Decimal idec_NumericValue = 0;
        private int ii_ScaleOnFocus = 0;
        private int ii_Precision = 1;
        private bool ib_AllowNegative = true;
        private bool ib_NoChangeEvent = false;
        private bool ib_ZeroNotValid = false;
        #endregion

        #region "Properties"

        /// <summary>
        /// Indicates if the value zero (0) valid.
        /// </summary>
        [System.ComponentModel.Category("Numeric settings")]
        public bool ZeroIsValid
        {
            get { return ib_ZeroNotValid; }
            set { ib_ZeroNotValid = value; }
        }

        /// <summary>
        /// Maximum allowed precision
        /// </summary>
        [System.ComponentModel.Category("Numeric settings")]
        public int NumericPrecision
        {
            get { return ii_Precision; }
            set
            {
                //Precision cannot be negative
                if (value < 0)
                {
                    return;
                }

                if (value < this.NumericScaleOnFocus)
                {
                    this.NumericScaleOnFocus = value;
                }

                ii_Precision = value;
            }
        }

        /// <summary>
        /// The maximum scale allowed
        /// </summary>
        [System.ComponentModel.RefreshProperties(System.ComponentModel.RefreshProperties.All),
        System.ComponentModel.Category("Numeric settings")]
        public int NumericScaleOnFocus
        {
            get { return ii_ScaleOnFocus; }
            set
            {
                //Scale cannot be negative
                if (value < 0)
                {
                    return;
                }

                //Scale cannot be larger than precision
                if (value >= this.NumericPrecision)
                {
                    return;
                }

                ii_ScaleOnFocus = value;

                if (ii_ScaleOnFocus > 0)
                {
                    this.Text = "0" + DecimalSeperator + new string(Convert.ToChar("0"), ii_ScaleOnFocus);
                }
                else
                {
                    this.Text = "0";
                }
            }
        }


        /// <summary>
        /// Scale displayed when the textbox has no focus 
        /// </summary>
        [System.ComponentModel.RefreshProperties(System.ComponentModel.RefreshProperties.All),
        System.ComponentModel.Category("Numeric settings")]
        public int NumericScaleOnLostFocus
        {
            get { return ii_ScaleOnLostFocus; }
            set
            {
                //Scale cannot be negative
                if (value < 0)
                {
                    return;
                }

                //Scale cannot be larger than precision
                if (value >= this.NumericPrecision)
                {
                    return;
                }

                ii_ScaleOnLostFocus = value;
            }
        }

        private string DecimalSeperator
        {
            get
            {
                return System.Globalization.NumberFormatInfo.CurrentInfo.NumberDecimalSeparator;
            }
        }

        private string GroupSeperator
        {
            get
            {
                return System.Globalization.NumberFormatInfo.CurrentInfo.NumberGroupSeparator;
            }
        }


        /// <summary>
        /// Indicates if negative numbers are allowed?
        /// </summary>
        [System.ComponentModel.Category("Numeric settings")]
        public bool AllowNegative
        {
            get { return ib_AllowNegative; }
            set { ib_AllowNegative = value; }
        }

        /// <summary>
        /// The current numeric value displayed in the textbox
        /// </summary>
        [System.ComponentModel.Bindable(true),
        System.ComponentModel.Category("Numeric settings")]
        public object NumericValue
        {
            get { return idec_NumericValue; }
            set
            {
                if (value.Equals(DBNull.Value))
                {
                    if (value.Equals(0))
                    {
                        this.Text = Convert.ToString(0);
                        idec_NumericValue = Convert.ToDecimal(0);
                        OnNumericValueChanged(new System.EventArgs());
                        return;
                    }
                }

                if (!value.Equals(idec_NumericValue))
                {
                    this.Text = Convert.ToString(value);
                    idec_NumericValue = Convert.ToDecimal(value);
                    OnNumericValueChanged(new System.EventArgs());
                }
            }
        }
        #endregion

        /// <summary>
        /// Initializes a new instance of the <see cref="ToolStripNumericTextBox"/> class.
        /// </summary>
        public ToolStripNumericTextBox()
        {
            this.TextAlign = System.Drawing.ContentAlignment.TopRight;
            this.Text = "0";
        }

        /// <summary>
        /// Occurs when numeric value changed.
        /// </summary>
        public event EventHandler NumericValueChanged;
        /// <summary>
        /// Raises the NumericValueChanged event
        /// </summary>
        /// <param name="e">The eventargs</param>
        protected void OnNumericValueChanged(System.EventArgs e)
        {
            if (NumericValueChanged != null)
            {
                NumericValueChanged(this, e);
            }
        }

        /// <summary>
        /// Formats a the text inf the textbox (which represents a number) according to
        /// the scale,precision and the enviroment settings.
        /// </summary>
        protected string FormatNumber()
        {
            StringBuilder lsb_Format = new StringBuilder();
            int li_Counter = 1;
            long ll_Remainder = 0;

            if (this.Focused)
            {
                while (li_Counter <= ii_Precision - ii_ScaleOnFocus)
                {
                    if (li_Counter == 1)
                    {
                        lsb_Format.Insert(0, '0');
                    }
                    else
                    {
                        lsb_Format.Insert(0, '#');
                    }

                    System.Math.DivRem(li_Counter, 3, out ll_Remainder);
                    if ((ll_Remainder == 0) && (li_Counter + 1 <= ii_Precision - ii_ScaleOnFocus))
                    {
                        //lsb_Format.Insert(0,',');
                    }

                    li_Counter++;
                }

                li_Counter = 1;

                if (ii_ScaleOnFocus > 0)
                {
                    lsb_Format.Append(".");

                    while (li_Counter <= ii_ScaleOnFocus)
                    {
                        lsb_Format.Append('0');
                        li_Counter++;
                    }
                }

            }
            else
            {
                while (li_Counter <= ii_Precision - ii_ScaleOnLostFocus)
                {
                    if (li_Counter == 1)
                    {
                        lsb_Format.Insert(0, '0');
                    }
                    else
                    {
                        lsb_Format.Insert(0, '#');
                    }
                    System.Math.DivRem(li_Counter, 3, out ll_Remainder);
                    if ((ll_Remainder == 0) && (li_Counter + 1 <= ii_Precision - ii_ScaleOnLostFocus))
                    {
                        //lsb_Format.Insert(0,',');
                    }
                    li_Counter++;
                }

                li_Counter = 1;

                if (ii_ScaleOnLostFocus > 0)
                {
                    lsb_Format.Append(".");

                    while (li_Counter <= ii_ScaleOnLostFocus)
                    {
                        lsb_Format.Append('0');
                        li_Counter++;
                    }
                }
            }
            return Convert.ToDecimal(this.Text).ToString(lsb_Format.ToString());

        }

        /// <summary>
        /// Raises the <see cref="E:System.Windows.Forms.ToolStripItem.TextChanged"></see> event.
        /// </summary>
        /// <param name="e">An <see cref="T:System.EventArgs"></see> that contains the event data.</param>
        protected override void OnTextChanged(EventArgs e)
        {
            int li_SelStart = 0;
            bool lb_PositionCursorBeforeComma = false;

            //Indicates that no change event should happen
            //Prevent event from firing on changing the text in the change
            //event
            if (ib_NoChangeEvent || (this.SelectionStart == -1))
            {
                return;
            }

            //No Change event
            ib_NoChangeEvent = true;

            if (string.Empty.Equals(this.Text.Trim()))
            {
                this.Text = "0";
            }

            if (this.Text.Substring(0, 1) == GroupSeperator)
            {
                this.Text = this.Text.Substring(1);
            }

            //if (! ii_Precision = 0 ) {
            if (!(ii_ScaleOnFocus == 0))
            {
                //if ( the current position is just before the comma
                if (this.SelectionStart == (this.Text.IndexOf(DecimalSeperator)))
                {
                    lb_PositionCursorBeforeComma = true;
                }
                else
                {
                    li_SelStart = this.SelectionStart;
                }
            }
            else
            {
                li_SelStart = this.SelectionStart;
            }
            try
            {
                idec_InternalValue = Convert.ToDecimal(this.Text);
                this.NumericValue = Convert.ToDecimal(this.Text);
            }
            catch (FormatException)
            {
                this.Text = "0";
                idec_InternalValue = 0;
                this.NumericValue = 0;
                ib_NoChangeEvent = false;
                return;
            }



            if (this.Focused)
            {
                //set { the text to the new format
                //if (! ii_Precision = 0 ) {
                if (!(ii_ScaleOnFocus == 0))
                {
                    //this.Text = Strings.FormatNumber(this.Text, this.NumericScaleOnFocus, Microsoft.VisualBasic.TriState.True, Microsoft.VisualBasic.TriState.False, Microsoft.VisualBasic.TriState.True);
                    this.Text = this.FormatNumber();
                }
                else
                {
                    if (this.Text.IndexOf('-') < 0)
                    {
                        //this.Text = Strings.FormatNumber(this.Text, this.NumericScaleOnFocus, Microsoft.VisualBasic.TriState.True, Microsoft.VisualBasic.TriState.False, Microsoft.VisualBasic.TriState.True);
                        this.Text = this.FormatNumber();
                    }
                    else
                    {
                        if (this.Text.Equals('-'))
                        {
                            this.Text = "";
                        }
                        else
                        {
                            //this.Text = Convert.ToString(System.Math.Abs(Convert.ToDouble(this.Text)));
                            //this.Text = Strings.FormatNumber(this.Text, this.NumericScaleOnFocus, Microsoft.VisualBasic.TriState.True, Microsoft.VisualBasic.TriState.False, Microsoft.VisualBasic.TriState.True);
                            this.Text = this.FormatNumber();
                        }
                    }
                }
            }
            else
            {
                //set { the text to the new format
                //if (! ii_Precision = 0 ) {
                if (!(ii_ScaleOnLostFocus == 0))
                {
                    //this.Text = Strings.FormatNumber(this.Text, this.NumericScaleOnLostFocus, Microsoft.VisualBasic.TriState.True, Microsoft.VisualBasic.TriState.False, Microsoft.VisualBasic.TriState.True);
                    this.Text = this.FormatNumber();
                }
                else
                {
                    if (this.Text.IndexOf('-') < 0)
                    {
                        //this.Text = Strings.FormatNumber(this.Text, this.NumericScaleOnLostFocus, Microsoft.VisualBasic.TriState.True, Microsoft.VisualBasic.TriState.False, Microsoft.VisualBasic.TriState.True);
                        this.Text = this.FormatNumber();
                    }
                    else
                    {
                        if (this.Text.Equals('-'))
                        {
                            this.Text = "";
                        }
                        else
                        {
                            //this.Text = Convert.ToString(System.Math.Abs(Convert.ToDouble(this.Text)));
                            //this.Text = Strings.FormatNumber(this.Text, this.NumericScaleOnLostFocus, Microsoft.VisualBasic.TriState.True, Microsoft.VisualBasic.TriState.False, Microsoft.VisualBasic.TriState.True);
                            this.Text = this.FormatNumber();
                        }
                    }
                }

            }

            //if ( the position was before the comma
            //then put again before the comma
            if (!(ii_ScaleOnFocus == 0))
            {
                if (lb_PositionCursorBeforeComma)
                {
                    this.SelectionStart = (this.Text.IndexOf(DecimalSeperator));
                }
                else
                {
                    this.SelectionStart = li_SelStart;
                }
            }
            else
            {
                this.SelectionStart = li_SelStart;
            }

            //Change event may fire
            ib_NoChangeEvent = false;

        }

        /// <summary>
        /// Raises the <see cref="E:System.Windows.Forms.ToolStripControlHost.KeyPress"></see> event.
        /// </summary>
        /// <param name="e">A <see cref="T:System.Windows.Forms.KeyPressEventArgs"></see> that contains the event data.</param>
        protected override void OnKeyPress(System.Windows.Forms.KeyPressEventArgs e)
        {
            base.OnKeyPress(e);
            if (!char.IsDigit(e.KeyChar) && e.KeyChar != '\x0008' && e.KeyChar != '\x007f')
            {
                if ((Control.ModifierKeys & (Keys.Alt | Keys.Control)) == Keys.None)
                {
                    e.Handled = true;

                    MessageBeep(0 /*MB_OK*/);
                }
            }
        }
    }
}
