using System;
using System.Collections.Generic;
using System.Text;
using Zensys.ZWave.Enums;
using Zensys.ZWave.Devices;
using Zensys.ZWave.Events;
using System.Threading;
using Zensys.Framework;
using Zensys.ZWave.Logging;
using Zensys.ZWave.Exceptions;

namespace Zensys.ZWave.SerialPortApplication.Devices
{
    public class Device : IDevice
    {
        //protected const int TIMEOUT = 180000;
        public event StatusChangedEventHandler ConnectionStatusChanged;
        public event DeviceAppCommandHandlerEventHandler ApplicationCommandHandlerEvent;
        public event DeviceAppCommandHandler_BridgeEventHandler ApplicationCommandHandler_BridgeEvent;

        private ISessionLayer mSessionLayer;
        public ISessionLayer SessionLayer
        {
            get
            {
                return mSessionLayer;
            }
        }

        protected Device()
        {
        }

        public Device(ISessionLayer sessionLayer)
        {
            mSessionLayer = sessionLayer;
            if (mSessionLayer != null)
            {
                //mSessionLayer.ResponseReceived += new ResponseReceivedEventHandler(OnSessionLayerResponseReceived);
                mSessionLayer.ResponseReceived = OnSessionLayerResponseReceived;
            }
            mFlash = new DeviceFlash(sessionLayer);
            mFlash.Device = this;
            mLeds = new DeviceLedCollection();
            mMemory = new DeviceMemory(sessionLayer);
            mMemory.Device = this;
            mTimer = new DeviceTimer();
        }

        protected void LogTopSession()
        {
            //SessionLayer.LogDataSource.AddTopSession(new LogSession(string.Format("{0} {1} {2}", Tools.CurrentDateTime.ToString("hh:mm:ss.ffff"), SessionLayer.SequenceNumber, Tools.GetMethodName(1))));
        }

        private void OnSessionLayerResponseReceived(ResponseReceivedEventArgs args)
        {
            if (args.CommandType == (byte)CommandTypes.CmdApplicationCommandHandler)
            {
                if (ApplicationCommandHandlerEvent != null)
                {
                    byte commandClass = 0;
                    byte commandKey = 0;
                    byte sourceNodeId = 0;
                    bool isBroadcast = args.Data[0] == 4;
                    byte[] cmdData = ProcessResponse(args.FrameBuffer, (CommandTypes)args.CommandType, ref sourceNodeId, ref commandClass, ref commandKey);
                    ApplicationCommandHandlerEvent(new DeviceAppCommandHandlerEventArgs(cmdData, sourceNodeId, commandClass, commandKey, isBroadcast));
                }
            }
            else if (args.CommandType == (byte)CommandTypes.CmdApplicationCommandHandler_Bridge ||
                args.CommandType == (byte)CommandTypes.CmdApplicationSlaveCommandHandler) //see 5.03 DevKit
            {
                if (ApplicationCommandHandler_BridgeEvent != null)
                {
                    byte commandClass = 0;
                    byte commandKey = 0;
                    byte sourceNodeId = 0;
                    byte[] destinationNodeIds = null;
                    byte[] cmdData = ProcessResponse_Bridge(args.FrameBuffer, (CommandTypes)args.CommandType, ref sourceNodeId, ref destinationNodeIds, ref commandClass, ref commandKey);
                    ApplicationCommandHandler_BridgeEvent(new DeviceAppCommandHandler_BridgeEventArgs(cmdData, sourceNodeId, destinationNodeIds, commandClass, commandKey));
                }
            }
            else if (args.CommandType == (byte)CommandTypes.CmdZWaveSendData ||
                args.CommandType == (byte)CommandTypes.CmdZWaveSendDataMulti)
            {
                //sessionLayerResponses.Add(args.Data);
                //if (sessionLayerResponses.Count >= sessionLayerResponseCount)
                //{
                //    Tools._writeDebugDiagnosticMessage("11sessionLayerResponses:" + sessionLayerResponses.Count.ToString() + "/" + sessionLayerResponseCount, true, true, 1);
                //    sessionLayerResponseCount = 0;
                //    SetSignalResponseReceived();
                //}
            }
            OnSessionLayerResponseReceivedCustomActions(args);
        }

        protected virtual void OnSessionLayerResponseReceivedCustomActions(ResponseReceivedEventArgs args)
        {
            //throw new NotImplementedException("The method or operation should be ovverided in inherited class.");
            if (args.CommandType == (byte)ProgrammerCommandTypes.FUNC_ID_BUTTON_PRESSED)
            {
                if (ApplicationCommandHandlerEvent != null)
                {
                    ApplicationCommandHandlerEvent(new DeviceAppCommandHandlerEventArgs(args.FrameBuffer, 0, 0, args.CommandType, false));
                }
            }
        }

        private byte[] ProcessResponse(
            byte[] response,
            CommandTypes requestCommand,
            ref byte sourceNodeId,
            ref byte commandClass,
            ref byte commandKey)
        {
            byte[] ret = null;
            if (response != null && response.Length > 7 && (CommandTypes)response[3] == requestCommand)
            {
                sourceNodeId = response[5];
                byte cmdLength = response[6];
                if (response.Length > cmdLength + 7)
                {
                    ret = new byte[cmdLength - 2];
                    for (int i = 0; i < cmdLength - 2; i++)
                    {
                        ret[i] = response[i + 7 + 2];
                    }
                    commandClass = response[7];
                    commandKey = response[8];
                }
            }
            return ret;
        }

        private byte[] ProcessSlaveResponse(byte[] response, CommandTypes requestCommand, ref byte sourceNodeId, ref byte destinationNodeId, ref byte commandClass, ref byte commandKey)
        {
            byte[] ret = null;
            if (response != null && response.Length > 8 && (CommandTypes)response[3] == requestCommand)
            {
                destinationNodeId = response[5];
                sourceNodeId = response[6];
                byte cmdLength = response[7];
                if (response.Length > cmdLength + 8)
                {
                    ret = new byte[cmdLength - 2];
                    for (int i = 0; i < cmdLength - 2; i++)
                    {
                        ret[i] = response[i + 8 + 2];
                    }
                    commandClass = response[8];
                    commandKey = response[9];
                }
            }
            return ret;
        }

        private byte[] ProcessResponse_Bridge(byte[] response, CommandTypes requestCommand, ref byte sourceNodeId, ref byte[] destinationNodeIds, ref byte commandClass, ref byte commandKey)
        {
            byte[] ret = null;
            if (response != null && response.Length > 8 && (CommandTypes)response[3] == requestCommand)
            {
                byte rxStatus = response[4];
                rxStatus = (byte)((rxStatus & 0x0C) >> 2);
                if (rxStatus == 0) //singlecast
                {
                    destinationNodeIds = new byte[] { response[5] };
                }
                else if (rxStatus == 2) //multicast
                {
                    //TODO parse destination nodes
                    destinationNodeIds = new byte[] { 0x00 };
                }
                else //broadcast
                {
                    destinationNodeIds = new byte[] { 0xFF };
                }

                sourceNodeId = response[6];
                byte cmdLength = response[7];
                if (response.Length > cmdLength + 8)
                {
                    ret = new byte[cmdLength - 2];
                    for (int i = 0; i < cmdLength - 2; i++)
                    {
                        ret[i] = response[i + 8 + 2];
                    }
                    commandClass = response[8];
                    commandKey = response[9];
                }
            }
            return ret;
        }

        #region IDevice Members
        #region Properties
        private byte mId;
        public byte Id
        {
            get
            {
                return mId;
            }
            set
            {
                mId = value;
            }
        }

        private byte[] mHomeId;
        public byte[] HomeId
        {
            get
            {
                return mHomeId;
            }
            set
            {
                mHomeId = value;
            }
        }

        private byte mChipType = 0x00;
        public byte ChipType
        {
            get
            {
                return mChipType;
            }
            set
            {
                mChipType = value;
            }
        }

        private byte mChipRevision;
        public byte ChipRevision
        {
            get
            {
                return mChipRevision;
            }
            set
            {
                mChipRevision = value;
            }
        }

        private byte mSerialApiVersion;
        public byte SerialApiVersion
        {
            get
            {
                return mSerialApiVersion;
            }
            set
            {
                mSerialApiVersion = value;
            }
        }

        private string mSerialPort;
        public string SerialPort
        {
            get
            {
                return mSerialPort;
            }
            set
            {
                mSerialPort = value;
            }
        }

        private IDeviceFlash mFlash;
        public IDeviceFlash Flash
        {
            get
            {
                return mFlash;
            }
            set
            {
                mFlash = value;
            }
        }

        private IDeviceTimer mTimer;
        public IDeviceTimer Timer
        {
            get
            {
                return mTimer;
            }
            set
            {
                mTimer = value;
            }
        }

        private IDeviceMemory mMemory;
        public IDeviceMemory Memory
        {
            get
            {
                return mMemory;
            }
            set
            {
                mMemory = value;
            }
        }

        private DeviceLedCollection mLeds;
        public DeviceLedCollection Leds
        {
            get { return mLeds; }
        }

        private ConnectionStatuses mConnectionStatus;
        public ConnectionStatuses ConnectionStatus
        {
            get
            {
                return mConnectionStatus;
            }
            set
            {
                StatusChangedArgs args = new StatusChangedArgs((int)mConnectionStatus, (int)value);
                mConnectionStatus = value;
                if (ConnectionStatusChanged != null)
                {
                    ConnectionStatusChanged(args);
                }
            }
        }

        private VersionInfo mVersion = new VersionInfo();
        public VersionInfo Version
        {
            get
            {
                return mVersion;
            }
            set
            {
                mVersion = value;
            }
        }

        private NodeBitmask mNodeBitmask = new NodeBitmask();
        public NodeBitmask CapabilityBitmask
        {
            get
            {
                return mNodeBitmask;
            }
            set
            {
                mNodeBitmask = value;
            }
        }

        private bool mIsSlaveApi;
        public bool IsSlaveApi
        {
            get
            {
                return mIsSlaveApi;
            }
            set
            {
                mIsSlaveApi = value;
            }
        }

        private bool mIsVirtual;
        public bool IsVirtual
        {
            get
            {
                return mIsVirtual;
            }
            set
            {
                mIsVirtual = value;
            }
        }

        private bool mIsFailed;
        public bool IsFailed
        {
            get
            {
                return mIsFailed;
            }
            set
            {
                mIsFailed = value;
            }
        }

        private byte mCapability;
        public byte Capability
        {
            get { return mCapability; }
            set { mCapability = value; }
        }

        private byte mSecurity;
        public byte Security
        {
            get { return mSecurity; }
            set { mSecurity = value; }
        }

        private byte mReserved;
        public byte Reserved
        {
            get { return mReserved; }
            set { mReserved = value; }
        }

        private byte mBasic;
        public byte Basic
        {
            get { return mBasic; }
            set { mBasic = value; }
        }

        private byte mGeneric;
        public byte Generic
        {
            get { return mGeneric; }
            set { mGeneric = value; }
        }

        private byte mSpecific;
        public byte Specific
        {
            get { return mSpecific; }
            set { mSpecific = value; }
        }
        public bool IsListening
        {
            get
            {
                return (this.Capability & 0x80) != 0;
            }
        }

        private byte[] mSupportedCommandClasses;
        public byte[] SupportedCommandClasses
        {
            get { return mSupportedCommandClasses; }
            set { mSupportedCommandClasses = value; }
        }

        private byte mSyncCount;
        public byte SyncCount
        {
            get
            {
                return mSyncCount;
            }
        }
        #endregion
        public void Open(string portName)
        {
            mSessionLayer.Open(portName);
            this.SerialPort = portName;
            ConnectionStatus = ConnectionStatuses.Opened;
        }

        public void Close()
        {
            mSessionLayer.Close();
            ConnectionStatus = ConnectionStatuses.Closed;
        }

        public void Reset()
        {
            throw new Exception("The method or operation is not implemented.");
        }

        public void RequestNodeInfo()
        {
            throw new Exception("The method or operation is not implemented.");
        }

        private ManualResetEvent asyncResponseReceived = new ManualResetEvent(false);

        /// <summary>
        /// Resets the signal response received.
        /// </summary>
        protected void ResetSignalResponseReceived()
        {
            Tools._writeDebugDiagnosticMessage("ResetSignalResponseReceived", true, true, 2);
            asyncResponseReceived.Reset();
            SessionLayer.IsAutoIncrementSequenceNumber = false;
        }

        /// <summary>
        /// Sets the signal response received.
        /// </summary>
        protected virtual void SetSignalResponseReceived()
        {
            Tools._writeDebugDiagnosticMessage("SetSignalResponseReceived", true, true, 2);
            asyncResponseReceived.Set();
            SessionLayer.IsAutoIncrementSequenceNumber = true;
        }

        /// <summary>
        /// Waits the one signal response received.
        /// </summary>
        protected void WaitOneSignalResponseReceived()
        {
            Tools._writeDebugDiagnosticMessage("WaitOneSignalResponseReceived", true, true, 2);
            asyncResponseReceived.WaitOne();
        }

        /// <summary>
        /// Waits the one signal response received.
        /// </summary>
        /// <param name="milisecondsTimeout">The miliseconds timeout.</param>
        /// <param name="exitContext">if set to <c>true</c> [exit context].</param>
        protected bool WaitOneSignalResponseReceived(int milisecondsTimeout, bool exitContext)
        {
            Tools._writeDebugDiagnosticMessage("WaitOneSignalResponseReceived", true, true, 2);
            return asyncResponseReceived.WaitOne(milisecondsTimeout, exitContext);
        }

        protected int SEND_DATA_TIMEOUT = 1000 * 30; // 30 min
        private TransmitStatuses SendDataInternal(byte nodeId, byte[] data, TransmitOptions txOptions, CommandTypes command)
        {
            IsSupportedSerialApiCommand((byte)command, true);

            List<byte> request = new List<byte>();
            request.Add(nodeId);
            request.Add((byte)data.Length);
            for (int i = 0; i < data.Length; i++)
            {
                request.Add(data[i]);
            }
            request.Add((byte)txOptions);
            request.Add(mSessionLayer.SequenceNumber);
            List<byte[]> responses = SessionLayer.ExecuteRequest(
                (byte)command,
                2,
                SEND_DATA_TIMEOUT,
                request.ToArray());
            if (responses.Count == 2 && responses[1].Length > 1)
            {
                return (TransmitStatuses)responses[1][1];
            }
            else
            {
                return TransmitStatuses.ResMissing;
            }
        }

        public virtual TransmitStatuses SendData(byte nodeId, byte[] data, TransmitOptions txOptions)
        {
            return SendDataInternal(nodeId, data, txOptions, CommandTypes.CmdZWaveSendData);
        }

        public virtual TransmitStatuses SendDataMeta(byte nodeId, byte[] data, TransmitOptions txOptions)
        {
            return SendDataInternal(nodeId, data, txOptions, CommandTypes.CmdZWaveSendDataMeta);
        }

        public virtual TransmitStatuses SendDataMulti(List<byte> nodeIdList, byte[] data, TransmitOptions txOptions)
        {
            LogTopSession();
            IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveSendDataMulti, true);
            byte[] request = null;
            byte nodeCount = 0;
            int index = 1;
            nodeCount = (byte)nodeIdList.Count;
            request = new byte[4 + nodeCount + data.Length];
            request[0] = nodeCount;
            foreach (byte nodeId in nodeIdList)
            {
                request[index] = nodeId;
                index++;
            }
            request[index++] = (byte)data.Length;
            for (int i = 0; i < data.Length; i++)
            {
                request[index] = data[i];
                index++;
            }
            request[2 + nodeCount + data.Length] = (byte)txOptions;
            request[3 + nodeCount + data.Length] = SessionLayer.SequenceNumber;
            List<byte[]> responses = mSessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveSendDataMulti, 2, SEND_DATA_TIMEOUT, request);
            if (responses.Count == 2 && responses[1].Length > 1)
            {
                return (TransmitStatuses)responses[1][1];
            }
            else
            {
                return TransmitStatuses.ResMissing;
            }
        }

        public TransmitStatuses SendZWaveTest(byte testCmd, ushort testDelay, byte testPayloadLength, int testCount, TransmitOptions txOptions, byte maxLength, byte[] testNodeMask)
        {
            LogTopSession();
            IsSupportedSerialApiCommand((byte)CommandTypes.CmdSerialApiTest, true);

            if (testNodeMask == null)
            {
                throw new ArgumentNullException("testNodeMask");
            }

            byte[] request = new byte[8 + testNodeMask.Length];
            request[0] = testCmd;
            if (testCmd > 0)
            {
                request[1] = (byte)((testDelay >> 8) & 0xff);
                request[2] = (byte)(testDelay & 0xff);
                request[3] = testPayloadLength;
                request[4] = (byte)((testCount >> 8) & 0xff);
                request[5] = (byte)(testCount & 0xff);
                request[6] = (byte)txOptions;
                request[7] = (byte)testNodeMask.Length;
                for (int i = 8; i < testNodeMask.Length + 8; i++)
                {
                    request[i] = testNodeMask[i - 8];
                }
            }
            List<byte[]> responses = mSessionLayer.ExecuteRequest((byte)CommandTypes.CmdSerialApiTest, 2, (int)SessionLayer.RequestTimeout.TotalMilliseconds, request);
            if (responses.Count == 2 && responses[1].Length > 1)
            {
                return (TransmitStatuses)responses[1][1];
            }
            else
            {
                return TransmitStatuses.ResMissing;
            }
        }

        public TransmitStatuses SendDataMR(byte nodeId, byte[] data, TransmitOptions txOptions, byte[] route, int timeout)
        {
            LogTopSession();
            TransmitStatuses result;
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveSendDataMR, true))
            {
                byte[] finalRoute = new byte[4];
                for (int i = 0; i < finalRoute.Length; i++)
                {
                    if (i < route.Length)
                    {
                        finalRoute[i] = route[i];
                    }
                    else
                    {
                        finalRoute[i] = 0;
                    }
                }
                List<byte> request = new List<byte>();
                request.Add(nodeId);
                request.Add((byte)data.Length);
                request.AddRange(data);
                request.Add((byte)txOptions);
                request.AddRange(finalRoute);
                request.Add(SessionLayer.SequenceNumber);
                List<byte[]> responses = SessionLayer.ExecuteRequest(
                    (byte)CommandTypes.CmdZWaveSendDataMR,
                    2,
                    timeout,
                    request.ToArray());
                if (responses.Count == 2 && responses[1] != null && responses[1].Length > 1)
                {
                    result = (TransmitStatuses)responses[1][1];
                }
                else
                {
                    throw new InvalidResponseSerialApiException();
                }
            }
            else
            {
                throw new NotSupportedCommandSerialApiException();
            }
            return result;
        }

        public bool SetProgrammingMode(bool enable)
        {
            LogTopSession();
            if (enable)
            {
                byte[] request = mSessionLayer.ExecuteRequest((byte)ProgrammerCommandTypes.FUNC_ID_ZW0x0x_PROG_ENABLE, new byte[] { mSessionLayer.SequenceNumber });
                if (request.Length > 0)
                {
                    mSyncCount = request[0];
                }
            }
            else
            {

                byte[] request = mSessionLayer.ExecuteRequest((byte)ProgrammerCommandTypes.FUNC_ID_ZW0x0x_PROG_RELEASE, new byte[] { this.ChipType, mSessionLayer.SequenceNumber });
                //if (request.Length > 0)
                //{
                //    mSyncCount = request[0];
                //}
            }
            return true;
        }

        public void EraseChip(bool isAsic)
        {
            this.Flash.Erase(isAsic);
        }

        public void ReadSignatureBits()
        {
            LogTopSession();
            byte[] signature = mSessionLayer.ExecuteRequest((byte)ProgrammerCommandTypes.FUNC_ID_ZW0x0x_READ_SIG_BYTE, new byte[] { 0x00, mSessionLayer.SequenceNumber });
            if (signature.Length == 8)
            {
                if (signature[0] == 0x7f && signature[1] == 0x7f && signature[2] == 0x7f && signature[3] == 0x7f
                    && signature[4] == 0x1f && (signature[5] == 0x02 || signature[5] == 0x03))
                {
                    Constants.BLANK_VALUE = 0x00;
                    Constants.FLASH_SIZE = 65536;
                    Constants.PAGES_IN_FLASH = 256;
                    this.ChipType = (byte)ChipTypes.ZW040x;
                    this.Flash.Pages = new List<IFlashPage>(new IFlashPage[Constants.PAGES_IN_FLASH]);
                }
                else if (signature[3] == 0x7F && signature[4] == 0x1F)
                {
                    Constants.BLANK_VALUE = 0xFF;
                    Constants.FLASH_SIZE = 32768;
                    Constants.PAGES_IN_FLASH = 128;
                    this.Flash.Pages = new List<IFlashPage>(new IFlashPage[Constants.PAGES_IN_FLASH]);

                    if (signature[6] >= 0x06)
                    {
                        this.ChipType = (byte)ChipTypes.ZW030x;
                    }
                    else
                    {
                        this.ChipType = (byte)ChipTypes.ZW020x;
                    }
                }
                else if (signature[3] == 0x9E && signature[4] == 0x95)
                {
                    Constants.BLANK_VALUE = 0xFF;
                    Constants.FLASH_SIZE = 32768;
                    Constants.PAGES_IN_FLASH = 256;
                    this.Flash.Pages = new List<IFlashPage>(new IFlashPage[Constants.PAGES_IN_FLASH]);
                    this.ChipType = (byte)ChipTypes.ZW010x;
                }
                else
                {
                    this.ChipType = (byte)ChipTypes.UNKNOWN;
                }
                Constants.BYTES_IN_PAGE = (Constants.FLASH_SIZE / Constants.PAGES_IN_FLASH);
            }

        }

        public byte ReadLockBits()
        {
            LogTopSession();
            byte lockBits = 0xff;
            byte seqNum = mSessionLayer.SequenceNumber;
            byte[] request = mSessionLayer.ExecuteRequest(
                (byte)ProgrammerCommandTypes.FUNC_ID_ZW0x0x_READ_LOCKBITS, new byte[] { this.ChipType, seqNum });
            if (request.Length == 2)
            {
                if (request[1] == seqNum)
                    lockBits = request[0];
            }
            return lockBits;
        }

        public bool WriteLockBits(byte lockBits)
        {
            LogTopSession();
            bool res = false;
            byte seqNum = mSessionLayer.SequenceNumber;
            byte[] request = mSessionLayer.ExecuteRequest(
                (byte)ProgrammerCommandTypes.FUNC_ID_ZW0x0x_WRITE_LOCKBITS, new byte[] { this.ChipType, lockBits, seqNum });
            if (request.Length == 2)
            {
                res = ((request[0] == (byte)FlashProgrammingStatuses.DONE ||
                        request[0] == (byte)FlashProgrammingStatuses.SUCCESS) && request[1] == seqNum);
            }
            return res;
        }

        public bool SetWriteCycleTime()
        {
            return SetWriteCycleTime(-1.0F);
        }

        public bool SetWriteCycleTime(float xtalFrequencyMHz)
        {
            LogTopSession();
            bool res = false;
            byte seqNum = mSessionLayer.SequenceNumber;
            byte[] request = mSessionLayer.ExecuteRequest(
                (byte)ProgrammerCommandTypes.FUNC_ID_ZW0x0x_SET_WRITE_CYCLE,
                new byte[] { CalculateWriteCycle(xtalFrequencyMHz), seqNum });
            if (request.Length == 2)
            {
                res = ((request[0] == (byte)FlashProgrammingStatuses.DONE ||
                        request[0] == (byte)FlashProgrammingStatuses.SUCCESS) && request[1] == seqNum);
            }
            return res;
        }

        public bool SetMode(WorkingModes mode)
        {
            LogTopSession();
            bool res = false;
            if (mode == WorkingModes.Development)
            {
                //set development mode
                byte seqNum = mSessionLayer.SequenceNumber;
                byte[] request = mSessionLayer.ExecuteRequest(
                    (byte)ProgrammerCommandTypes.FUNC_ID_ZW0x0x_DEV_MODE_ENABLE, new byte[] { this.ChipType, seqNum });
                if (request.Length == 2)
                {
                    res = ((request[0] == (byte)FlashProgrammingStatuses.DONE ||
                            request[0] == (byte)FlashProgrammingStatuses.SUCCESS) && request[1] == seqNum);
                }
            }
            else if (mode == WorkingModes.ExecuteOutOfSram)
            {
                //set execute out of sram mode
                byte seqNum = mSessionLayer.SequenceNumber;
                byte[] request = mSessionLayer.ExecuteRequest(
                    (byte)ProgrammerCommandTypes.FUNC_ID_ZW0x0x_SRAM_EXECUTE, new byte[] { this.ChipType, seqNum });
                if (request.Length == 2)
                {
                    res = ((request[0] == (byte)FlashProgrammingStatuses.DONE ||
                            request[0] == (byte)FlashProgrammingStatuses.SUCCESS) && request[1] == seqNum);
                }
            }
            return res;
        }

        public bool SetModemBits()
        {
            LogTopSession();
            bool res = false;
            //Set the bit that identifies the Chip as a Modem device
            byte seqNum = mSessionLayer.SequenceNumber;
            byte[] request = mSessionLayer.ExecuteRequest(
                (byte)ProgrammerCommandTypes.FUNC_ID_ZW0x0x_MODEM_BIT_WRITE, new byte[] { this.ChipType, seqNum });
            if (request.Length == 2)
            {
                res = ((request[0] == (byte)FlashProgrammingStatuses.DONE ||
                        request[0] == (byte)FlashProgrammingStatuses.SUCCESS) && request[1] == seqNum);
            }
            return res;
        }

        public byte ReadState()
        {
            //read status from the ZW040x chip.
            LogTopSession();
            byte state = 0;
            byte seqNum = mSessionLayer.SequenceNumber;
            byte[] request = mSessionLayer.ExecuteRequest(
                (byte)ProgrammerCommandTypes.FUNC_ID_ZW0x0x_STATUS_READ, new byte[] { this.ChipType, seqNum });
            if (request.Length == 2)
            {
                if (request[1] == seqNum)
                    state = request[0];
            }
            return state;
        }

        public short ReadWriteOtpStats()
        {
            //read the write OTP stats from the ZW040x chip.
            LogTopSession();
            short num = 0;
            byte seqNum = mSessionLayer.SequenceNumber;
            byte[] request = mSessionLayer.ExecuteRequest(
                (byte)ProgrammerCommandTypes.FUNC_ID_ZW0x0x_WRITE_OTP_STATS_READ, new byte[] { this.ChipType, seqNum });
            if (request.Length == 3)
            {
                if (request[2] == seqNum)
                {
                    num = request[1];
                    num |= (short)(((short)request[0]) << 8);
                }
            }
            return num;
        }

        public VersionInfo GetVersion()
        {
            LogTopSession();
            byte[] response = mSessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveGetVersion, null);
            if (response != null)
            {
                System.Text.UTF7Encoding utf = new System.Text.UTF7Encoding();
                if (response.Length > 6 + 6)
                {
                    this.Version.Version = utf.GetString(response, 6, 6);

                    if (this.Version.Version.EndsWith("\0"))
                    {
                        this.Version.Version = this.Version.Version.Remove(this.Version.Version.Length - 1, 1);
                        string verString = this.Version.Version.Replace("Z-Wave", "").Trim();
                        string[] arr = verString.Split('.');
                        byte protocolVersion = 0;
                        byte protocolSubVersion = 0;
                        if (arr.Length == 2 && byte.TryParse(arr[0], out protocolVersion) && byte.TryParse(arr[1], out protocolSubVersion))
                        {
                            this.Version.ZWaveProtocolVersion = protocolVersion;
                            this.Version.ZWaveProtocolSubVersion = protocolSubVersion;
                        }
                    }
                    this.Version.Library = (Libraries)response[12];
                }
            }
            return this.Version;
        }

        public void GetCapabilities()
        {
            LogTopSession();
            byte[] capabilities = mSessionLayer.ExecuteRequest((byte)CommandTypes.CmdSerialApiGetCapabilities, null);
            if (capabilities.Length > 8)
            {
                this.Version.ZWaveApplicationVersion = capabilities[0];
                this.Version.ZWaveApplicationSubVersion = capabilities[1];
                byte[] temp = new byte[capabilities.Length - 8];
                for (int n = 0; n < temp.Length; n++)
                {
                    temp[n] = capabilities[n + 8];
                }
                this.CapabilityBitmask.Store(temp);
            }
        }

        public bool IsSupportedSerialApiCommand(byte commandId)
        {
            return IsSupportedSerialApiCommand(commandId, false);
        }

        public bool IsSupportedSerialApiCommand(byte commandId, bool throwException)
        {
            if (!throwException)
            {
                return this.CapabilityBitmask.ZWaveNodeMaskNodeIn(commandId);
            }
            else
            {
                if (this.CapabilityBitmask.Length > 0)
                {
                    if (!this.CapabilityBitmask.ZWaveNodeMaskNodeIn(commandId))
                    {
                        throw new Exception(((CommandTypes)commandId).ToString() + " is not implemented in the Z-Wave Device");
                    }
                }
                return true;
            }
        }

        public byte[] SerialApiInitData()
        {
            LogTopSession();
            byte[] initDataResponse = null;
            if (this.IsSupportedSerialApiCommand((byte)CommandTypes.CmdSerialApiGetInitData, true))
            {
                initDataResponse = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdSerialApiGetInitData, null);


                this.SerialApiVersion = initDataResponse[0];
                this.IsSlaveApi = (initDataResponse[1] & Constants.GET_INIT_DATA_FLAG_SLAVE_API) != 0;
                byte length = initDataResponse[2];
                if (initDataResponse.Length > length)
                {
                    this.ChipType = initDataResponse[3 + length];
                    this.ChipRevision = initDataResponse[4 + length];
                }

            }
            return initDataResponse;
        }

        public bool SetBootLoaderMode(bool enable)
        {
            LogTopSession();
            bool result = false;
            if (enable)
            {
                byte seqNum = mSessionLayer.SequenceNumber;
                byte[] request = mSessionLayer.ExecuteRequest((byte)ProgrammerCommandTypes.FUNC_ID_M128_ENTER_PROG_MODE, new byte[] { seqNum });
                if (request.Length == 2)
                {
                    if (request[0] == (byte)FlashProgrammingStatuses.DONE &&
                        request[1] == seqNum)
                    {
                        Thread.Sleep(250);
                        byte checksum = 0xff;
                        byte[] sigBytes = { 0xa5, 0xfe, 0x1a, 0x56 };// random values
                        for (int i = 0; i < sigBytes.Length; i++)
                        {
                            checksum ^= sigBytes[i];
                        }
                        seqNum = mSessionLayer.SequenceNumber;
                        //send enter bootloader command again, but with bootloader signature and its sckecksum.
                        request = mSessionLayer.ExecuteRequest((byte)ProgrammerCommandTypes.FUNC_ID_M128_ENTER_PROG_MODE,
                            new byte[] { sigBytes[0], sigBytes[1], sigBytes[2], sigBytes[3], checksum, seqNum });
                        if (request.Length == 2)
                        {
                            if (request[0] == (byte)FlashProgrammingStatuses.DONE &&
                                request[1] == seqNum)
                            {
                                result = true;
                            }
                        }
                    }
                }
            }
            else
            {
                byte seqNum = mSessionLayer.SequenceNumber;
                byte[] request = mSessionLayer.ExecuteRequest((byte)ProgrammerCommandTypes.FUNC_ID_EXIT_PROG_MODE, new byte[] { seqNum });
                if (request.Length == 2)
                {
                    if (request[0] == (byte)FlashProgrammingStatuses.DONE &&
                        request[1] == seqNum)
                    {
                        result = true;
                    }
                }
            }
            return result;
        }

        public bool GetCurrentFirmwareVersion(out int version, out int revision)
        {
            LogTopSession();
            bool result = false;
            version = 0;
            revision = 0;
            byte seqNum = mSessionLayer.SequenceNumber;
            byte[] request = mSessionLayer.ExecuteRequest((byte)ProgrammerCommandTypes.FUNC_ID_M128_GET_SW_VER, new byte[] { seqNum });
            if (request.Length == 3)
            {
                if (request[2] == seqNum)
                {
                    version = request[0];
                    revision = request[1];
                    result = true;
                }
            }
            return result;
        }

        public byte CalculateWriteCycle()
        {
            return CalculateWriteCycle(-1.0F);
        }

        public byte CalculateWriteCycle(float xtalFrequencyMHz)
        {
            byte c = 0;
            if (this.ChipType == (byte)ChipTypes.ZW010x)
            {
                if (xtalFrequencyMHz == -1.0F)
                    xtalFrequencyMHz = 7.3F;	//3.6F;
                c = (byte)(30.0 * xtalFrequencyMHz / (16.0 * 1.0)); // 30 uS / (16*clock_periode)

            }
            else if (this.ChipType == (byte)ChipTypes.ZW020x ||
                      this.ChipType == (byte)ChipTypes.ZW030x)
            {
                if (xtalFrequencyMHz == -1.0F)
                    xtalFrequencyMHz = 32.0F;
                c = (byte)(30.0 * xtalFrequencyMHz / (16.0 * 4.0)); // 30 uS / (16*4*Tclock_periode)
            }
            return c;
        }

        public byte[] GetRandomNumber(int randomBytes)
        {
            LogTopSession();
            byte[] result = null;
            if (randomBytes > 32)
            {
                throw new ApplicationLayerException("You can request a maximum of 32 bytes of random data per call");
            }
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveGetRandom, true))
            {
                result = mSessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveGetRandom, new byte[] { (byte)randomBytes });
            }
            return result;
        }

        public TransmitStatuses SendNodeInformation(byte destination, TransmitOptions txOptions)
        {
            LogTopSession();
            byte[] result = null;
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveSendNodeInformation, true))
            {
                result = mSessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveSendNodeInformation, new byte[] { destination, (byte)txOptions, mSessionLayer.SequenceNumber });
            }
            if (result != null && result.Length > 1)
            {
                return (TransmitStatuses)result[1];
            }
            else
            {
                return TransmitStatuses.ResMissing;
            }
        }

        virtual public LearnMode SetLearnMode(bool learnMode)
        {
            return LearnMode.Unknown;
        }

        virtual public void SetDefault()
        {
        }

        public void SetNodeInformation(bool isListening, byte generic, byte specific, byte[] nodeParameter)
        {
            LogTopSession();
            if (nodeParameter == null)
            {
                throw new ArgumentNullException("nodeParm");
            }
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdSerialApiApplNodeInformation, true))
            {
                byte[] request = new byte[nodeParameter.Length + 4];
                if (isListening)
                {
                    request[0] = 0x01;
                }
                else
                {
                    request[0] = 0x00;
                }
                request[1] = generic;
                request[2] = specific;
                request[3] = (byte)nodeParameter.Length;
                for (int i = 0; i < nodeParameter.Length; i++)
                {
                    request[i + 4] = nodeParameter[i];
                }
                SessionLayer.ExecuteNonRequest((byte)CommandTypes.CmdSerialApiApplNodeInformation, request);
                SupportedCommandClasses = nodeParameter;
                if (SupportedCommandClasses != null && this is IController)
                {
                    foreach (byte b in SupportedCommandClasses)
                    {
                        ((IController)this).CommandClassesStore.Store(Id, b);
                    }
                }
            }
        }
        public bool SetRFReceiveMode(byte mode)
        {
            LogTopSession();
            byte[] result = null;
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveSetRFReceiveMode, true))
            {
                result = mSessionLayer.ExecuteRequest((byte)CommandTypes.CmdZWaveSetRFReceiveMode, new byte[] { mode });
            }
            if (result != null && result.Length > 0)
            {
                return result[0] == 0 ? false : true;
            }
            else
            {
                return false;
            }
        }
        public TransmitStatuses SendTestFrame(byte nodeId, PowerLevels powerLevel)
        {
            LogTopSession();
            TransmitStatuses result = TransmitStatuses.ResMissing;
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveSendTestFrame, true))
            {
                List<byte[]> responses = mSessionLayer.ExecuteRequest(
                    (byte)CommandTypes.CmdZWaveSendTestFrame,
                    2,
                    (int)SessionLayer.RequestTimeout.TotalMilliseconds,
                    new byte[] { (byte)powerLevel, mSessionLayer.SequenceNumber });
                if (responses.Count == 2 && responses[1] != null && responses[1].Length > 1)
                {
                    result = (TransmitStatuses)responses[1][1];
                }
            }
            return result;
        }
        public void SetSleepMode(SleepMode mode, byte intEnable)
        {
            LogTopSession();
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveSetSleepMode, true))
            {
                SessionLayer.ExecuteNonRequest((byte)CommandTypes.CmdZWaveSetSleepMode, new byte[] { (byte)mode, intEnable });
            }
        }

        public byte[] SerialApiSetTimeouts(byte acknowledgeTimeout, byte timeout)
        {
            LogTopSession();
            byte[] ret = null;
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdSerialApiSetTimeouts, true))
            {
                ret = SessionLayer.ExecuteRequest((byte)CommandTypes.CmdSerialApiSetTimeouts, new byte[] { acknowledgeTimeout, timeout });
            }
            return ret;
        }

        public RediscoveryNeededReturnValues RediscoveryNeeded(byte nodeId)
        {
            LogTopSession();
            RediscoveryNeededReturnValues result;
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveRediscoveryNeeded, true))
            {
                List<byte[]> responses = SessionLayer.ExecuteRequest(
                        (byte)CommandTypes.CmdZWaveRediscoveryNeeded,
                        3,
                        60000,
                        new byte[] { SessionLayer.SequenceNumber });
                if (responses.Count > 1 && responses[1] != null && responses[1].Length > 0)
                {
                    Tools._writeDebugDiagnosticMessage(Tools.ToHexString(responses[1]), true, true);
                    result = (RediscoveryNeededReturnValues)responses[1][0];
                    if (responses.Count > 2 && responses[2] != null && responses[2].Length > 0 &&
                        result == RediscoveryNeededReturnValues.LostAccepted)
                    {
                        result = (RediscoveryNeededReturnValues)responses[2][0];
                    }
                }
                else
                {
                    Tools._writeDebugDiagnosticMessage("response count = " + responses.Count, true, true);
                    throw new InvalidResponseSerialApiException();
                }
            }
            else
            {
                throw new NotSupportedCommandSerialApiException();
            }
            return result;
        }

        public RequestNetworkUpdateStatuses RequestNetworkUpdate()
        {
            LogTopSession();
            RequestNetworkUpdateStatuses result;
            if (IsSupportedSerialApiCommand((byte)CommandTypes.CmdZWaveRequestNetworkUpdate, true))
            {
                List<byte[]> responses = SessionLayer.ExecuteRequest(
                    (byte)CommandTypes.CmdZWaveRequestNetworkUpdate,
                    2,
                    (int)SessionLayer.RequestTimeout.TotalMilliseconds,
                    new byte[] { SessionLayer.SequenceNumber });
                if (responses.Count == 2 && responses[1] != null && responses[1].Length > 1)
                {
                    Tools._writeDebugDiagnosticMessage(Tools.ToHexString(responses[1]), true, true);
                    result = (RequestNetworkUpdateStatuses)responses[1][1];
                }
                else
                {
                    Tools._writeDebugDiagnosticMessage("response count = " + responses.Count, true, true);
                    throw new InvalidResponseSerialApiException();
                }
            }
            else
            {
                throw new NotSupportedCommandSerialApiException();
            }
            return result;
        }
        public bool Calibration(bool enable)
        {
            LogTopSession();
            if (enable)
            {
                byte[] request = mSessionLayer.ExecuteRequest((byte)ProgrammerCommandTypes.FUNC_ID_CALIBRATION_START, new byte[] { this.ChipType, mSessionLayer.SequenceNumber });

            }
            else
            {

                mSessionLayer.ExecuteNonRequest((byte)ProgrammerCommandTypes.FUNC_ID_CALIBRATION_STOP, new byte[] { this.ChipType, mSessionLayer.SequenceNumber });

            }
            return true;
        }


        #endregion

        #region IDisposable Members

        public void Dispose()
        {
            Dispose(true);
            GC.SuppressFinalize(this);

        }

        protected virtual void Dispose(bool disposing)
        {
            if (disposing)
            {
                //SessionLayer.ResponseReceived -= new ResponseReceivedEventHandler(OnSessionLayerResponseReceived);
                SessionLayer.ResponseReceived = null;
            }
        }


        #endregion

        #region IComparable<IDevice> Members

        public int CompareTo(IDevice other)
        {
            if (Id < other.Id)
            {
                return -1;
            }
            else if (Id > other.Id)
            {
                return 1;
            }
            else
            {
                return 0;
            }
        }

        #endregion

        #region IEquatable<IDevice> Members

        public bool Equals(IDevice other)
        {
            if (Id == other.Id)
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        #endregion

    }
}
