
/*! stargate-webui - v05.01.19-689 - 2015-02-05 02-02-18 */

swc.constructors.ApplicationCollection = Backbone.Collection.extend({

    applications: [
        {
            "name" : "Storage and Media",
            "description" : "Save and access documents, photos, music and videos centrally on your %ProductName%.",
            "type" : "central_storage",
            "classes" : "",
            "route" : "#storage/data"
        },
        {
            "name" : "Central Storage Remote Access",
            "description" : "Manage the data sharing and access from your devices to the %ProductName%.",
            "type" : "storage_remote",
            "classes" : "",
            "route" : "#storage/settings/access-on-move"
        },
        {
            "name" : "Speedtest",
            "description" : "Measure your access and your home network speed.",
            "type" : "speedtest",
            "classes" : "",
            "route" : "#applications/speedtest"
        },
        {
             "name" : "Swisscom DynDNS",
             "description" : "Save and access documents, photos, music and videos centrally on your Internet-Box.",
             "type" : "swisscom_dyndns",
             "classes": "expert-mode-only disabled-for-superadmin hide-AP5",
             "route" : "#applications/dyndns"
        }
    ],

    sync: function() {
        var self = this,
            applications = self.applications,
            vpnTileConfig = {
                "name" : "VPN Server",
                "description" : "VPN lets you access your homenet-work from anywhere in the world.",
                "type" : "vpn_server",
                "classes" : "expert-mode-only hide-AP5",
                "route" : "#applications/vpn"
            },
        deferred = new $.Deferred();

        this.reset();

        _.each(applications, function(model) {
            self.add({
                'name' : model.name,
                'description' : model.description,
                'type' : model.type,
                'route' : model.route,
                'classes' : model.classes
            });
        });

        $.when(
            swc.models.Applications.isVpnEnable()
        ).done(function(vpnGlobalEnable) {
            if (vpnGlobalEnable) {
                self.add(vpnTileConfig);
            }
            deferred.resolve();
        });

        return deferred.promise();
    }
});;swc.constructors.VPNCollection = swc.BaseCollection.extend({

    model: 'VPNClient',

    url: {
        read: '/ws'
    },

    device: swc.Utils.DeviceType,

    ajaxParameters: JSON.stringify({
        service: "com.swisscom.stargate/ws/vpn.com.swisscom.stargate.vpn",
        method: "getDevices",
        parameters: {}
    }),

    ajaxSettings: {
        requestType: 'POST',
        requestDataType: 'json',
        contentType: 'application/x-sah-ws-4-call+json',
        idle: true
    },

    sync: function() {
        if(this.device === 'default') {
           this.sync = (new swc.BaseCollection()).sync;
        }
    },

    isFetching: true,

    apiToJSON: function(response) {
        var clientsParsed = [],
            responseParsed = {};

        this.error = false;

        try {
            responseParsed = JSON.parse(response.status);

            if (!_.isEmpty(responseParsed)) {
                var vpnClients = responseParsed;

                _.each(vpnClients, function(client) {

                    var tmpType = client.connectionTime.format[0];
                    var resultType,
                        countInt = parseInt(client.connectionTime.count, 10),
                        count;

                    if(tmpType === 's') {
                        resultType = 'm1';
                        count = "<1";
                    }
                    else if(tmpType === 'm' && countInt === 1) {
                        resultType = 'm1';
                        count = client.connectionTime.count;
                    }
                    else if(tmpType === 'm' && countInt > 1) {
                        resultType = 'm';
                        count = client.connectionTime.count;
                    }
                    else if(tmpType === 'h' && countInt === 1) {
                        resultType = 'h1';
                        count = client.connectionTime.count;
                    }
                    else if(tmpType === 'h' && countInt > 1) {
                        resultType = 'h';
                        count = client.connectionTime.count;
                    }
                    else if(tmpType === 'd' && countInt === 1) {
                        resultType = 'd1';
                        count = client.connectionTime.count;
                    }
                    else if(tmpType === 'd' && countInt > 1) {
                        resultType = 'd';
                        count = client.connectionTime.count;
                    }

                    clientsParsed.push({
                        id: client.ipAddress,
                        ipAddress: client.ipAddress,
                        connectedForType: resultType,
                        connectedForCount: count
                    });
                });
            }
        } catch (e) {
            // do nothing because everything has default values already
        }

        return clientsParsed;
    }
});
;swc.constructors.USBHostDeviceCollection = swc.BaseCollection.extend({

    url: "/sysbus/USBHosts:getDevices",

    apiToJSON: function(response) {
        var parsed = [];

        if (response && response.status) {
            _.each(response.status, function(data) {
                parsed.push(data);
            });
        }

        return parsed;
    },

    /**
     * Get the data of the first element of collection
     * (see @classdesc)
     *
     * @returns {Object}
     */
    getHWInfo: function() {
        var hwInfo = {};

        if (this.length) {
            hwInfo = this.toJSON()[0];
        }

        return hwInfo;
    }

});;swc.constructors.DHCPLeases = Backbone.Collection.extend({
    
    defaultModel: { name: '', statusClass: 'disconnected', interfaceType: 'none', icon: 'unrecognized' },

    autoSync: true,

    sync: function(fromListener){
        var deferred = new $.Deferred(),
            self = this;

        if(self.autoSync===false){
         return deferred.resolve();
        }

        swc.models.Rest.sendRequest({
            url: '/sysbus/DHCPv4/Server/Pool/default:getStaticLeases',
            fromListener: fromListener,

            data: {
                "parameters":{ }
            },

            success: function(response) {
                self.build(response);
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    build: function(response){
        var self = this;

        this.reset();

        if (!response || !response['status']) {
            return;
        }

        _.each(response['status'], function(data){
            var model = new Backbone.Model();
            if (data.MACAddress) {
                data.MACAddress = data.MACAddress.toUpperCase();
            }
            model.set({'id': data.MACAddress});
            model.set(data);
            self.add(model);
        });
    },

    getActiveLeases: function(){
        var data = [],
            res = {},
            me = this;

        data = _.map(this.toJSON(), function(model){
            var mac = model.MACAddress.toUpperCase(),
                device = swc.models.NetworkDevices.where({mac: mac});

            if (device.length) {
                model['device'] = device[0].toJSON();
            } else {
                model['device'] = me.defaultModel;
            }

            return model;
        });

        _.each(data, function(item){
            res[item.MACAddress] = item;
        });

        return res;
    },

    saveDHCPLeasesEdit: function(data){
        var self = this,
            ipaddr, macaddr,
            deferred = new $.Deferred(),
            editedId;

        $.each(data, function(key, value) {
            if (value.parameterName === "DHCPLeaseIPAddress") {
                ipaddr = value.parameterValue;
            }
            if (value.parameterName === "DHCPLeaseEditedMacAddress") {
                macaddr = value.parameterValue.toUpperCase();
            }
            editedId = value.parameterData.editedLease;
        });

        $.when(this.deleteDHCPLease(editedId))
            .done(function() {
                if (data.deleted) {
                    deferred.resolve();
                } else {
                    $.when(self.saveDHCPLeaseRequest(ipaddr, macaddr))
                        .done(function() {
                            deferred.resolve();
                        });
                }
            });

        return deferred.promise();
    },

    saveDHCPLeases: function(data){
        var self = this,
            pool = [],
            deferred = new $.Deferred();

        $.each(data, function(key, value) {
            pool.push(self.saveDHCPLease(value));
        });

        $.when(pool).done(function () {
            deferred.resolve();
        }).fail(function () {
            deferred.reject();
        });

        return deferred.promise();
    },

    saveDHCPLease: function(data){
        var ipaddr, macaddr, edited = false;

        // Get neccessary params from array of page data:
        $.each(data, function(key, value) {
            if (value.parameterName === "DHCPLeaseIPAddress") {
                ipaddr = value.parameterValue;
            }
            if (value.parameterName === "DHCPLeaseMacAddress") {
                macaddr = value.parameterValue.toUpperCase();
            }
            if (value.parameterName === "DHCPLeaseEditedMacAddress") {
                edited = true;
            }
        });

        if (edited) {
            return this.saveDHCPLeasesEdit(data);
        }

        return this.saveDHCPLeaseRequest(ipaddr, macaddr);
    },

    saveDHCPLeaseRequest: function(ip, mac){
        var deferred = new $.Deferred();

        if(_.isEmpty(ip) && _.isEmpty(mac)) {
            deferred.resolve();
        } else {
            swc.models.Rest.sendRequest({
                url: '/sysbus/DHCPv4/Server/Pool/default:addStaticLease',
                data: {
                    parameters: {
                        IPAddress: ip,
                        MACAddress: mac.toUpperCase()
                    }
                },
    
                success: function() {
                    deferred.resolve();
                },
    
                error: function() {
                    deferred.reject();
                }
            });
        }

        return deferred.promise();
    },

    deleteDHCPLease: function(mac){
        var deferred = new $.Deferred(),
            self = this;

        swc.models.Rest.sendRequest({
            url: '/sysbus/DHCPv4/Server/Pool/default:deleteStaticLease',
            data: {
                parameters: {
                    MACAddress: mac.toUpperCase()
                }
            },

            success: function() {
                self.remove(mac);
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    validation: {
        validateNewDevice: function(data){
            var device, isEdited = false, ipAddress,
                validationStatus = {
                    'status': true,
                    'messages': []
                };

            // Get neccessary params from array of page data:
            $.each(data, function(key, value) {
                if (value.parameterName === "DHCPLeaseNewDevice" || value.parameterName === "DHCPLeaseNewDevice") {
                    device = value.parameterValue;
                }

                if (value.parameterName === "DHCPLeaseEditFlag" && value.parameterValue) {
                    isEdited = true;
                }

                if (value.parameterName === "DHCPLeaseIPAddress") {
                    ipAddress = value.parameterValue;
                }
            });

            if (!isEdited && (_.isEmpty(device)) && (ipAddress && ipAddress !== "")){
                validationStatus.status = false;
                validationStatus.messages.push("global");
            }

            return validationStatus;
        },

        validateIP: function(data){
            var netValidation = swc.models.Network.validation,
                validationStatus = {
                    'status': true,
                    'messages': []
                };

            var Address = netValidation.getIPField(data, 'Address'),
                APAddress = netValidation.getIPField(data, 'APAddress'),
                Netmask = netValidation.getIPField(data, 'Netmask'),
                DHCPMin = netValidation.getIPField(data, 'DHCPRange', 'range-from'),
                DHCPMax = netValidation.getIPField(data, 'DHCPRange', 'range-to'),
                LeaseIPAddress = netValidation.getIPField(data, 'DHCPLeaseIPAddress'),
                LeaseIPAddressStr = netValidation.decimalToIpString(LeaseIPAddress),
                DHCPLease = (_.findWhere(data, {parameterName: 'DHCPLeaseMacAddress'}) || _.findWhere(data, {parameterName: 'DHCPLeaseEditedMacAddress'})),
                DHCPLeaseMacAddress = DHCPLease ? DHCPLease.parameterValue : false;

            // NOTE: copy-pasted from network.js file
            var Subnet = (Address & Netmask) >>> 0;
            var Broadcast = (Address | ~Netmask) >>> 0;

            if (Address === false || Netmask === false || DHCPMin === false || DHCPMax === false) {
                return validationStatus;
            }

            var editedModelId = _.findWhere(data, { parameterName: "DHCPLeaseEditFlag" });

            if (editedModelId && editedModelId.parameterValue) {
                editedModelId = editedModelId.parameterValue;
            } else {
                editedModelId = false;
            }

            // Check if format is valid
            if (LeaseIPAddress === false) {
                validationStatus.status = false;
                validationStatus.messages.push('Invalid IPv4 Address Format Custom');
                return validationStatus;
            }

            // Check if address is within right subnet
            if (!netValidation.isAddressWithinSubnet(LeaseIPAddress, Address, Netmask)) {
                validationStatus.status = false;
                validationStatus.messages.push('IPv4 Address is outside of allowed host addresses range');
                return validationStatus;
            }

            // NOTE: copy-pasted from network.js file [BEGIN]
            if (LeaseIPAddress === Broadcast) {
                validationStatus.status = false;
                validationStatus.messages.push("Not match broadcast");
                return validationStatus;
            }

            if (LeaseIPAddress === Subnet) {
                validationStatus.status = false;
                validationStatus.messages.push("Not match subnet");
                return validationStatus;
            }
            // [END]

            // if (DHCPMin <= LeaseIPAddress && LeaseIPAddress <= DHCPMax) {
            //     validationStatus.status = false;
            //     validationStatus.messages.push('IPv4 Address Within DHCP Range');
            //     return validationStatus;
            // }

            // Validate that address is unique
            if (LeaseIPAddress === Address) {
                validationStatus.status = false;
                validationStatus.messages.push('IPv4 Address Already in use');
                return validationStatus;
            }

            if (LeaseIPAddress === APAddress) {
                validationStatus.status = false;
                validationStatus.messages.push('IPv4 Address Already in use');
                return validationStatus;
            }

            swc.models.NetworkDevices.each(function(model){
                var modelAddress = model.get('address'),
                    modelMac = model.get('mac');

                if (modelAddress.IPv4 === LeaseIPAddressStr && DHCPLeaseMacAddress && modelMac !== DHCPLeaseMacAddress) {
                    validationStatus.status = false;
                    validationStatus.messages.push('IPv4 Address Already in use');
                }
            });

            swc.models.DHCPLeases.each(function(model){
                if ((model.get("IPAddress") === LeaseIPAddressStr) && (model.get("id") !== editedModelId)) {
                    validationStatus.status = false;
                    validationStatus.messages.push('IPv4 Address Already in use');
                }
            });

            return validationStatus;
        }
    }

});
;swc.constructors.DynDNSProviderCollection = swc.BaseCollection.extend({

    url: {
        read: '/sysbus/DynDNS:getHosts'
    },

    model: 'DynDNSProvider',

    defaults: {
        'id': '',
        'service': '',
        'hostname': '',
        'username': '',
        'password': '',
        'last_update': null,
        'status': null,
        'enable': false
    },

    // Parses host list
    apiToJSON: function (response) {
        var result = [];

        _.each(response.status, function(providerData) {
            // Manually create ID. From NP point of view it is a "hostname"
            providerData['id'] = providerData['hostname'];
            result.push(providerData);
        });

        return result;
    }

});
;swc.constructors.DynDNSService = Backbone.Model.extend({
    defaults: {
        id: '',
        name: ''
    }
});

swc.constructors.DynDNSServiceCollection = swc.BaseCollection.extend({

    url: {
        'read': '/sysbus/DynDNS:getServices'
    },

    model: 'DynDNSService',

    apiToJSON: function (response) {
        var result = [];

        _.each(response.status, function (providerId) {
            result.push({
                id: providerId,
                name: providerId
            });
        });

        return result;
    },

    /**
     * Prepare options for the Service provider dropdown list
     * @returns {Array}
     */
    formatProvidersOptions: function() {
        var options = [];

        this.each(function(provider) {
            options.push({
                name:  provider.get('name').capitalize(),
                value: provider.get('name')
            });
        });

        return options;
    }

});
;swc.constructors.FirewallRules = Backbone.Collection.extend({

    /**
     * List of predefined rules:
     * @param predefined {Array}
     */
    predefined: [
        { "name": "Internet Key Exchange", "port": [ "500" ], "protocol": "17" },
        { "name": "Kerberos", "port": [ "88" ], "protocol": "6,17" },
        { "name": "SUN RPC",  "port": [ "111" ], "protocol": "6" },
        { "name": "Microsoft RPC", "port": [ "135" ], "protocol": "6" },
        { "name": "NETBIOS-SSN", "port": [ "139" ], "protocol": "6" },
        { "name": "Microsoft-DS", "port": [ "445" ], "protocol": "6" },
        { "name": "Remote Login", "port": [ "513" ], "protocol": "6" },
        { "name": "Remote Shell", "port": [ "514" ], "protocol": "6" },
        { "name": "AFP", "port": [ "548" ], "protocol": "6" },
        { "name": "Internet Printing protocol", "port": [ "631" ], "protocol": "6" },
        { "name": "SSDP (Port 9000)", "port": [ "1900" ], "protocol": "17" },
        { "name": "SSDP (Port 2869)", "port": [ "2869" ], "protocol": "6" },
        { "name": "UPnP Discovery", "port": [ "3702" ], "protocol": "17" },
        { "name": "Multicast DNS", "port": [ "5353" ], "protocol": "17" },
        { "name": "LLMNR", "port": [ "5355" ], "protocol": "17" },
        { "name": "SSH", "port": [ "22" ], "protocol": "6" },
        { "name": "Telnet", "port": [ "23" ], "protocol": "6" },
        { "name": "Http", "port": [ "80" ], "protocol": "6" },
        { "name": "Remote Desktop protocol", "port": [ "3389" ], "protocol": "6" },
        { "name": "VNC", "port": [ "5900" ], "protocol": "6" }
    ],

    /**
     * List of validation methods:
     */
    validation: {

        RuleName: function(data) {
            var validationData = swc.Utils.getDataToValidate({
                    mode: { elementName: 'service-mode' },
                    policy: { elementName: 'policy' },
                    name: { elementName: 'RuleName' },
                    id: { elementName: 'RuleKey' }
                }, data),
                validationMessages = [],
                searchModel;

            // Skip validation for predefined mode
            if (validationData.mode !== "predefined") {

                if (!$.trim(validationData.name)) {
                    validationMessages.push('must not be empty');
                }

                if (validationData.name.length < 1 || validationData.name.length > 40) {
                    validationMessages.push('must be from 1 to 40 characters');
                }

                // All printable characters are allowed
                if (!/^[ -~]+$/.test(validationData.name)) {
                    validationMessages.push('allowed characters ASCII non-printable');
                }

                // Search for same rules only within one chain target (inbound / outbound)
                if (validationData.policy.split('_')[1] === 'inbound') {
                    searchModel = swc.models.FirewallRules.
                        findWhere({ name: validationData.name, chain: 'Custom_V6In' });
                } else {
                    searchModel = swc.models.FirewallRules.
                        findWhere({ name: validationData.name, chain: 'Custom_V6Out' });
                }

                // Check for custom rules:
                // Skip check if model is editing
                if (!_.isUndefined(searchModel) && searchModel.get('id') !== validationData.id) {
                    validationMessages.push('already existed rule names are not allowed');
                }
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        PredefinedRuleName: function(data) {
            var validationData = swc.Utils.getDataToValidate({
                    mode: { elementName: 'service-mode' },
                    name: { elementName: 'PredefinedRuleName' },
                    policy: { elementName: 'policy' }
                }, data),
                validationMessages = [],
                searchModel;

            // Skip validation for predefined mode
            if (validationData.mode !== "custom") {

                // Search for same rules only within one chain target (inbound / outbound)
                if (validationData.policy.split('_')[1] === 'inbound') {
                    searchModel = swc.models.FirewallRules.
                        findWhere({name: validationData.name, chain: 'Custom_V6In'});
                } else {
                    searchModel = swc.models.FirewallRules.
                        findWhere({name: validationData.name, chain: 'Custom_V6Out'});
                }

                // Check if rule matched:
                if (!_.isUndefined(searchModel)) {
                    validationMessages.push('already existed rule names are not allowed');
                }
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        RulePort: function(data) {
            var validationData = swc.Utils.getDataToValidate({
                    mode: { elementName: "ports-mode" },
                    from: { elementName: "RulePort", elementClass: "from" },
                    to: { elementName: "RulePort", elementClass: "to" },
                    id: { elementName: 'RuleKey' },
                    policy: { elementName: 'policy' }
                }, data),
                validationMessages = [],
                searchModels = [],
                portsRange = [];

            if (!$.trim(validationData.from)) {
                validationMessages.push('rule port must not be empty');
            }

            if (/\D/.test(validationData.from)) {
                validationMessages.push('allowed characters 0-9');
            }

            // Convert to integer values:
            validationData.from = +validationData.from;
            validationData.to = +validationData.to;

            if (validationData.from < 1 || validationData.from > 65535) {
                validationMessages.push('allowed range 1 to 65535');
            }

            if (validationData.mode === "range") {

                if (!$.trim(validationData.to)) {
                    validationMessages.push('rule port must not be empty');
                }

                if (/\D/.test(validationData.to)) {
                    validationMessages.push('allowed characters 0-9');
                }

                if (validationData.to < 1 || validationData.to > 65535) {
                    validationMessages.push('allowed range 1 to 65535');
                }

                if (validationData.to <= validationData.from) {
                    validationMessages.push('range option first value MUST be less then second value');
                }

                portsRange = [ validationData.from, validationData.to ];
            } else {
                portsRange = [ validationData.from ];
            }

            // Search for same rules only within one chain target (inbound / outbound)
            if (validationData.policy.split('_')[1] === 'inbound') {
                searchModels = swc.models.FirewallRules.where({ chain: 'Custom_V6In' });
            } else {
                searchModels = swc.models.FirewallRules.where({ chain: 'Custom_V6Out' });
            }

            // Check ports crossing conflict for custom rules:
            _.each(searchModels, function(rule) {
                var rulePorts = rule.get('port'),
                    rulePortsNotCrossing = swc.models.FirewallRules.validation.helpers.
                        portsNotInRange(portsRange, rulePorts);

                if (!rulePortsNotCrossing && (rule.get('id') !== validationData.id)) {
                    validationMessages.push('already used ports are not allowed');
                }
            });

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        helpers: {
            /**
             * Check for ports not conflicting
             *
             * @param ports [1-2]
             * @param range [1-2]
             *
             * @returns {boolean}
             */
            portsNotInRange: function(ports, range) {
                // Case 1 [] && []
                if (ports.length === 1 && range.length === 1) {
                    return parseInt(ports[0], 10) !== parseInt(range[0], 10);
                }

                // Case 2 [][] && []
                if (ports.length === 2 && range.length === 1) {
                    return parseInt(range[0], 10) < parseInt(ports[0], 10) ||
                        parseInt(range[0], 10) > parseInt(ports[1], 10);
                }

                // Case 3 [] && [][]
                if (ports.length === 1 && range.length === 2) {
                    return parseInt(ports[0], 10) < parseInt(range[0], 10) ||
                        parseInt(ports[0], 10) > parseInt(range[1], 10);
                }

                // Case 4 [][] && [][]
                if (ports.length === 2 && range.length === 2) {
                    return (parseInt(ports[0], 10) < parseInt(range[0], 10) &&
                        parseInt(ports[1], 10) < parseInt(range[0], 10)) ||
                        (parseInt(range[0], 10) < parseInt(ports[0], 10) &&
                            parseInt(range[1], 10) < parseInt(ports[0], 10));
                }
            }
        }
    },

    /**
     * All rules sorted by port value ASC.
     *
     * @param rule1 {Object -> Backbone.Model}
     * @param rule2 {Object -> Backbone.Model}
     *
     * @returns {number}
     */
    comparator: function(rule1, rule2) {
        return parseInt(rule1.get("port"), 10) - parseInt(rule2.get("port"), 10);
    },

    /**
     * Get all rules from the server and compare them to the collection
     * @param fromListener {Boolean}
     * @returns {*}
     */
    sync: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        $.when(this.fetchInbound(fromListener), this.fetchOutbound(fromListener))
            .done(function(inbound, outbound) {
                self.parseRules(inbound.status, outbound.status);

                deferred.resolve();
            });

        return deferred.promise();
    },

    /**
     * Request inbound rules:
     * @param fromListener {Boolean}
     * @returns {*}
     */
    fetchInbound: function(fromListener) {
        return this.requestRules({ parameters: { chain: "Custom_V6In" }}, fromListener);
    },

    /**
     * Request outbound rules:
     * @param fromListener {Boolean}
     * @returns {*}
     */
    fetchOutbound: function(fromListener) {
        return this.requestRules({ parameters: { chain: "Custom_V6Out" }}, fromListener);
    },

    /**
     * Send request to the server to get list of rules
     * @param params {Object}
     * @param fromListener {Boolean}
     * @returns {*}
     */
    requestRules: function(params, fromListener) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Firewall:getCustomRule',
            data: params,
            fromListener: fromListener,

            success: function(response) {
                deferred.resolve(response);
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    /**
     * Create normalized JSON basing on server response
     * @param inbound {Array of Objects}
     * @param outbound {Array of Objects}
     */
    parseRules: function(inbound, outbound) {
        var parsedRules = [];

        // Parse inbound rules:
        _.each(inbound, function(rule, key) {
            parsedRules.push({
                id: key,
                status: rule.Enable,
                name: rule.Description,
                port: rule.DestinationPort.split('-'),
                policy: rule.Target + '_inbound',
                protocol: rule.Protocol,
                chain: 'Custom_V6In'
            });
        });

        // Parse outbound rules:
        _.each(outbound, function(rule, key) {
            parsedRules.push({
                id: key,
                status: rule.Enable,
                name: rule.Description,
                port: rule.DestinationPort.split('-'),
                policy: rule.Target + '_outbound',
                protocol: rule.Protocol,
                chain: 'Custom_V6Out'
            });
        });

        // Update collection data:
        this._reset();
        this.set(parsedRules);
    },

    /**
     * Save changed rules (edit / delete) to the server
     * @param rulesToChangeState {Array of objects}
     * @param rulesToDelete {Array}
     * @returns {*}
     */
    updateRulesList: function(rulesToChangeState, rulesToDelete) {
        var self = this,
            deferred = new $.Deferred(),
            deferredActions = [];

        // Prepare list of actions for changed rules:
        _.each(rulesToChangeState, function(rule, key) {
            var ruleModel = self.findWhere({ id: key });

            if (!_.isUndefined(ruleModel)) {
                deferredActions.push(
                    self.changeRule(ruleModel.toJSON(), _.extend(ruleModel.toJSON(), { status: rule.value }))
                );
            }

        });

        // Prepare list of actions for deleted rules:
        _.each(rulesToDelete, function(rule) {
            var ruleModel = self.findWhere({ id: rule });

            if (!_.isUndefined(ruleModel)) {
                deferredActions.push(self.removeRule(ruleModel.toJSON()));
            }
        });

        $.when.apply(null, deferredActions)
            .done(function() {
                deferred.resolve();
            })
            .fail(function() {
                deferred.reject();
            });

        return deferred.promise();
    },

    /**
     * Send request to create rule on the server:
     * @param ruleData {Object}
     * @returns {*}
     */
    createRule: function(ruleData) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Firewall:setCustomRule',
            data: {
                parameters: {
                    id: swc.Utils.generateId(),
                    description: ruleData.name,
                    enable: ruleData.status,
                    protocol: ruleData.protocol,
                    ipversion: 6,
                    destinationPort: ruleData.port.join('-'),
                    persistent: true,
                    action: ruleData.policy.split('_')[0],
                    chain: ruleData.chain
                }
            },

            success: function() {
                swc.models.Rest.sendRequest({
                    url: '/sysbus/Firewall:commit',
                    data: {
                        parameters: {}
                    },

                    success: function() {
                        deferred.resolve();
                    }
                });
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    /**
     * Change rule on the server (1. delete, 2. create new)
     * @param previousRule {Object}
     * @param newRule {Object}
     * @returns {*}
     */
    changeRule: function(previousRule, newRule) {
        var self = this,
            deferred = new $.Deferred();

        $.when(this.removeRule(previousRule)).done(function() {
            $.when(self.createRule(newRule)).done(function() {
                deferred.resolve();
            });
        });

        return deferred.promise();
    },

    /**
     * Delete rule from the server:
     * @param ruleData {Object}
     * @returns {*}
     */
    removeRule: function(ruleData) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Firewall:deleteCustomRule',
            data: {
                parameters: {
                    id: ruleData.id,
                    chain: ruleData.chain
                }
            },

            success: function() {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    }

});
;swc.constructors.Network = Backbone.Collection.extend({

    autoSync: true,
    validation: {
         /** Used to ommit validation in some fields ACL alike
         * 0 - validate all
         * 1 - dont validate (DHCPMax - DHCPMin > 1) AND dont validate ((DHCPMin & Netmask) >>> 0 !== Subnet || (DHCPMax & Netmask) >>> 0 !== Subnet )
         *
         */
        validationLevel: 0,

        /** Provides valid interface for view.pageValidation */
        validateIPAddress: function(data) {
            var Address = this.getIPField(data, 'Address'),
                Netmask = this.getIPField(data, 'Netmask'),
                DHCPMin = this.getIPField(data, 'DHCPRange', 'range-from'),
                DHCPMax = this.getIPField(data, 'DHCPRange', 'range-to');

            var validationStatus = {
                    status: true,
                    messages: []
                };

            if (Netmask === false) {
                validationStatus.status = false;
                validationStatus.messages.push("Bad netmask");
                return validationStatus;
            }
    
            var netmaskString = Netmask.toString(2);
            if (netmaskString.match(/01/) || !netmaskString.match(/[0]{2,}$/ig)) { //Has raising sequence 01 or less than 2 zeroes at end
                validationStatus.status = false;
                validationStatus.messages.push("Bad netmask");
                return validationStatus;
            }

            return this.validateIPAddressDecimal(Address, Netmask, DHCPMin, DHCPMax);
        },
        
        validateAPIPAddress: function(data) {
            var Address = this.getIPField(data, 'Address'),
                APAddress = this.getIPField(data, 'APAddress'),
                Netmask = this.getIPField(data, 'Netmask'),
                DHCPMin = this.getIPField(data, 'DHCPRange', 'range-from'),
                DHCPMax = this.getIPField(data, 'DHCPRange', 'range-to'),
                address,
                me = this;

            var validationStatus = {
                    status: true,
                    messages: []
                };

            if (Netmask === false) {
                validationStatus.status = false;
                validationStatus.messages.push("Bad netmask");
                return validationStatus;
            }
    
            var netmaskString = Netmask.toString(2);
            if (netmaskString.match(/01/) || !netmaskString.match(/[0]{2,}$/ig)) { //Has raising sequence 01 or less than 2 zeroes at end
                validationStatus.status = false;
                validationStatus.messages.push("Bad netmask");
                return validationStatus;
            }
            
            var result = ( swc.Utils.DeviceType === 'default') ? this.validateIPAddressDecimal(APAddress, Netmask, DHCPMin, DHCPMax, undefined, true) : {status: true};
            
            var netmaskValidation = this.validateNetmaskSubnetDecimal(APAddress, Netmask, Address);
            if (netmaskValidation.status === false) {
                return netmaskValidation;
            }
            // check if it not np address
            if(result.status) {
                if(APAddress === Address) {
                    result.status = false;
                    result.messages.push("Address already used");
                }
            }
            
            // lookup for ip in all devices
            if(result.status) {
                var devices = swc.models.NetworkDevices.models;
                _.each(devices, function(device) {
                    if (device.attributes.status) {
                        address = me.ipStringToDecimal(device.attributes.address.IPv4);
                        if(address === APAddress) {
                            result.status = false;
                            result.messages.push("Address already used");
                        }
                    }
                });
            }
            
            // lookup for ip in static leases
            if(result.status) {
                var leases = swc.models.DHCPLeases.models;
                _.each(leases, function(lease) {
                    address = me.ipStringToDecimal(lease.get('IPAddress'));
                    if(address === APAddress) {
                        result.status = false;
                        result.messages.push("Address already used");
                    }
                });
            }
            
            return result;
        },
        
        validateNetmask: function(data) {
            var Address = this.getIPField(data, 'Address'),
                Netmask = this.getIPField(data, 'Netmask'),
                DHCPMin = this.getIPField(data, 'DHCPRange', 'range-from'),
                DHCPMax = this.getIPField(data, 'DHCPRange', 'range-to');

            return this.validateNetmaskDecimal(Address, Netmask, DHCPMin, DHCPMax, true);
        },

        validateDHCPSettings: function(data) {
            var Address = this.getIPField(data, 'Address'),
                Netmask = this.getIPField(data, 'Netmask'),
                DHCPMin = this.getIPField(data, 'DHCPRange', 'range-from'),
                DHCPMax = this.getIPField(data, 'DHCPRange', 'range-to');

            var validationStatus = {
                    status: true,
                    messages: []
                };

            if (Netmask === false) {
                validationStatus.status = false;
                validationStatus.messages.push("Bad netmask");
                return validationStatus;
            }
    
            var netmaskString = Netmask.toString(2);
            if (netmaskString.match(/01/) || !netmaskString.match(/[0]{2,}$/ig)) { //Has raising sequence 01 or less than 2 zeroes at end
                validationStatus.status = false;
                validationStatus.messages.push("Bad netmask");
                return validationStatus;
            }

            return this.validateDHCPSettingsDecimal(Address, Netmask, DHCPMin, DHCPMax);
        },

        validateDMZ: function(data) {
            var dmzAddress = _.findWhere(data, {parameterName: 'DMZIPAddress'}),
                validationStatus = {
                    status: true,
                    messages: []
                };

            if (dmzAddress === false) {
                return validationStatus;
            }

            if (($.inArray('disabled', dmzAddress.parameterClasses) ===-1) && !dmzAddress.parameterValue) {
                validationStatus.status = false;
                validationStatus.messages.push("global");
            }

            return validationStatus;
        },
        
        validateCustomDMZ: function(data) {
            var dmzAddress = _.findWhere(data, {parameterName: 'DMZCustomIpAddress'}),
                dmzCustomAddress = _.findWhere(data, {parameterName: 'DMZIPAddress'}).parameterValue,
                boxIPAddress = this.getIPField(data, 'Address', 'ip-address'),
                storageIPAddress = this.getIPField(data, 'APAddress', 'ip-address'),
                isdmzEnabled = _.findWhere(data, {parameterName: 'DMZEnable'}).parameterValue,
                IPFormat = this.getIPField(data, 'DMZCustomIpAddress', 'ip-address'),
                NetMask = this.getIPField(data, 'Netmask', 'ip-address'),
                noValidationErrors = {
                    status: true,
                    messages: []
                },
                self = this,
                validationResult = null;

            //don`t validate if dmz is disabled showing custom ip address
            if (dmzCustomAddress === "custom" && isdmzEnabled === false ) {
                return noValidationErrors;
            }
            //don`t show errors if field is hidden
            if (dmzCustomAddress !== "custom" || dmzCustomAddress === undefined){
                return noValidationErrors;
            }

            if (dmzAddress === false || (dmzCustomAddress === "custom" && isdmzEnabled === false) ) {
                 return noValidationErrors;
            }

            if (($.inArray('disabled', dmzAddress.parameterClasses) ===-1) && !dmzAddress.parameterValue) {
                return {
                    status: false,
                    messages: ["Invalid IPv4 Address Format"]
                };
            }
            
            var ipValue = dmzAddress.parameterValue;

            if(!ipValue.match(/^(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$/)){
                return {
                    status: false,
                    messages: ["Invalid IPv4 Address Format Custom"]
                };
            }

            if(boxIPAddress === IPFormat) {
                return {
                    status: false,
                    messages: ["DMZ custom ip address can not be the same as Internet-Box ip address"]
                };
            }

            if(storageIPAddress === IPFormat && swc.Utils.DeviceType === 'default') {
                return {
                    status: false,
                    messages: ["DMZ custom ip address can not be the same as central storage ip address"]
                };
            }

            validationResult = self.validateNetmaskSubnetDecimal(IPFormat, NetMask);
            if (validationResult && !validationResult.status) {
                return validationResult;
            }

            if ((IPFormat & NetMask) >>> 0 !== (boxIPAddress & NetMask) >>> 0) {
                return {
                    status: false,
                    messages: ["Selected DMZ IP address must belong to the subnet"]
                };
            }

            if(!(IPFormat > self.getMinAllowedHostAsDecimal(boxIPAddress, NetMask) &&
                IPFormat <= self.getMaxAllowedHostAsDecimal(boxIPAddress, NetMask))) {
                return {
                    status: false,
                    messages: ["IPv4 Address Within DHCP Range"]
                };
            }
            return noValidationErrors;
        },

        // Validates decimal values of IP Address and other fields
        validateIPAddressDecimal: function(Address, Netmask, DHCPMin, DHCPMax, ignoreNetwork, ignoreDHCP) {
            var validationStatus = {
                status: true,
                messages: []
            };

            if (Address === false) {
                validationStatus.status = false;
                validationStatus.messages.push("Invalid IPv4 Address Format");
                return validationStatus;
            }

            if (!(Address > 3232235520 && Address < 3232301055)/* 192.168.0.0 - 192.168.255.255*/ &&
                !(Address > 167772160 && Address < 184549375)/* 10.0.0.0 - 10.255.255.255 */ &&
                !(Address > 2886729728 && Address < 2887778303)/* 172.16.0.0 - 172.32.255.255 */) {
                validationStatus.status = false;
                validationStatus.messages.push("Non-private subnet");
                //'You have choosen an invalid network. Only private Networks according RFC 1918 are allowed'
                return validationStatus;
            }
            
            if(ignoreNetwork) {
                return validationStatus;
            }

            //We need Netmask to proceed validation
            if (Netmask === false) {
                return validationStatus;
            }

            var netmaskValidation = this.validateNetmaskSubnetDecimal(Address, Netmask);
            if (netmaskValidation.status === false) {
                return netmaskValidation;
            }

            //We need DHCP to validate further
            if (DHCPMin === false || DHCPMax === false) {
                return validationStatus;
            }
            if(ignoreDHCP !== true) {
    this.validationLevel=1;
                var validDHCPSubnet = this.validateDHCPSubnetDecimal(DHCPMin, DHCPMax, Address, Netmask);
    this.validationLevel=0;
                if (validDHCPSubnet.status === false) {
                    return validDHCPSubnet;
                }
            }

            return validationStatus;
        },

        // Validate decimal value of Netmask and other fields if provided
        validateNetmaskDecimal: function(Address, Netmask, DHCPMin, DHCPMax, disableValidationIP) {
            if(this.validateIPAddressDecimal(Address, Netmask, DHCPMin, DHCPMax, true).status === true) {
                var validationStatus = {
                    status: true,
                    messages: []
                };
    
                if (Netmask === false) {
                    validationStatus.status = false;
                    validationStatus.messages.push("Invalid Netmask format");
                    return validationStatus;
                }
    
                var netmaskString = Netmask.toString(2);
                if (netmaskString.match(/01/) || !netmaskString.match(/[0]{2,}$/ig)) { //Has raising sequence 01 or less than 2 zeroes at end
                    validationStatus.status = false;
                    validationStatus.messages.push("Invalid Netmask format");
                    return validationStatus;
                }

                var netmaskValidation = this.validateNetmaskSubnetDecimal(Address, Netmask, undefined, disableValidationIP);

                if (netmaskValidation.status === false) {
                    return netmaskValidation;
                }

                //We need DHCP to validate further
                if (DHCPMin === false || DHCPMax === false) {
                    return validationStatus;
                }
                if(!_.isUndefined(disableValidationIP) && disableValidationIP === true) {
                    this.validationLevel=1;
                                    var validDHCPSubnet = this.validateDHCPSubnetDecimal(DHCPMin, DHCPMax, Address, Netmask, disableValidationIP);
                    this.validationLevel=0;
                                    if (validDHCPSubnet.status === false) {
                                        return validDHCPSubnet;
                                    }
                }

    
                return validationStatus;
            } else {
                return {
                    status: true,
                    messages: []
                };
            }
        },

        // Validates that DHCP settings fit subnet and are correct
        validateDHCPSettingsDecimal: function(Address, Netmask, DHCPMin, DHCPMax) {
            var validationStatus = {
                status: true,
                messages: []
            };

            if (DHCPMin === false || DHCPMax === false) {
                validationStatus.status = false;
                validationStatus.messages.push('Invalid IPv4 Address Format');
                return validationStatus;
            }

            // We can proceed only if Address and Netmask are specified
            if (Address === false || Netmask === false) {
                return validationStatus;
            }

            var validDHCPSubnet = this.validateDHCPSubnetDecimal(DHCPMin, DHCPMax, Address, Netmask);
            if (validDHCPSubnet.status === false) {
                return validDHCPSubnet;
            }

            // Check if DHCP Settings range is correct
            if (DHCPMax - DHCPMin < 1) {
                validationStatus.status = false;
                validationStatus.messages.push("Incorrect DHCP range");
            }

            return validationStatus;
        },

        // Check if netmask and address combination is ok for private subnet
        // return bool validation status with message
        validateNetmaskSubnetDecimal: function(Address, Netmask, CheckIP, disableValidationIP) {
            var validationStatus = {
                    status: true,
                    messages: []
                },
                validNets  = ['0xC0A80000', /*r192*/   '0xAC100000', /*r172*/ '0x0A000000' /*r10*/ ],
                validMasks = ['0xFFFF0000', /*mask16*/ '0xFFF00000', /*mask20*/ '0xFF000000' /*mask24*/],
                usedNetGuestWifi = this.ipStringToDecimal('10.128.0.0'),
                usedNetAPNP = this.ipStringToDecimal('10.129.0.0'),
                usedNetNetmask = this.ipStringToDecimal('255.255.0.0');

            /**
             * NOTE:
             * We use shift right for 0 positions to move from signed to unsigned number.
             * Because it may happen that binary AND of Gateway IP Address and Netmask results in a negative value.
             */
            var Subnet = (Address & Netmask) >>> 0;
            var Broadcast = (Address | ~Netmask) >>> 0;

            validNets = _.map(validNets, function(val) {
                return parseInt(val, 16);
            });
            validMasks = _.map(validMasks, function(val) {
                return parseInt(val, 16);
            });

            var netValid = false;

            $.each(validNets, function(i) {
                if ((Address & validMasks[i]) >>> 0 === validNets[i]) {
                    netValid = true;

                    // The subnet mask can also be more specific than the default one
                    if ((Netmask & validMasks[i]) >>> 0  !== validMasks[i]) {
                        validationStatus.status = false;
                        validationStatus.messages.push("The subnetmask does not fit to the ip range");
                    }
                }
            });

            if (!netValid) {
                validationStatus.status = false;
                validationStatus.messages.push("Non-private subnet");
            }

            if (Address === Broadcast) {
                validationStatus.status = false;
                validationStatus.messages.push("Not match broadcast");
                return validationStatus;
            }

            if (_.isUndefined(disableValidationIP) && Address === Subnet) {
                validationStatus.status = false;
                validationStatus.messages.push("Not match subnet");
                return validationStatus;
            }

            // Validate that network doesn't match guest wifi or AP-NP network
            var maxNetmask = (Netmask & usedNetNetmask) >>> 0;

            if ((Address & maxNetmask) >>> 0 === usedNetGuestWifi) {
                validationStatus.status = false;
                validationStatus.messages.push("Subnet 10.128.0.0/16 is used for guest wlan");
                return validationStatus;
            }

            if ((Address & maxNetmask) >>> 0 === usedNetAPNP) {
                validationStatus.status = false;
                validationStatus.messages.push("Subnet 10.129.0.0/16 is used for internal purposes");
                return validationStatus;
            }

            if(!_.isUndefined(CheckIP)) {
                if(this.isAddressWithinSubnet(CheckIP, Address, Netmask) === false) {
                    validationStatus.status = false;
                    validationStatus.messages.push("IP addresses do not match");
                }
            }

            return validationStatus;
        },


        // We need to check if DHCP subnet is ok, if one of [Address, Netmask, DHCP] have changed
        validateDHCPSubnetDecimal: function(DHCPMin, DHCPMax, Address, Netmask) {
            var validationStatus = {
                status: true,
                messages: []
            };

            var Subnet = (Address & Netmask) >>> 0;
            var Broadcast = (Address | ~Netmask) >>> 0;

            // Check if DHCP Subnet match to IP Subnet
            if (this.validationLevel < 1 &&  ((DHCPMin & Netmask) >>> 0 !== Subnet || (DHCPMax & Netmask) >>> 0 !== Subnet )) {
                validationStatus.status = false;
                validationStatus.messages.push("DHCP Subnet doesnt match");
                return validationStatus;
            }

            // Validate if IP Address matches part is out of range:
            if (DHCPMax >= Address && Address >= DHCPMin) {
                validationStatus.status = false;
                validationStatus.messages.push("IPv4 Address Within DHCP Range");

                return validationStatus;
            }

            // Check if DHCP Settings range is correct
            if (this.validationLevel < 1 && (DHCPMax === Broadcast || DHCPMin === Subnet)) {
                validationStatus.status = false;
                validationStatus.messages.push("Incorrect DHCP range");
                return validationStatus;
            }

            return validationStatus;
        },

        // data is [{parameterName: key, parameterValue: value, parameterClasses: []}] array
        // returns false if text inserted into field is not a valid IP address
        // returns decimal value of IP address if everything is ok
        getIPField: function(data, key, className) {
            var val = _.where(data, {parameterName: key});

            if (className) {
                val = _.filter(val, function(element) {
                    return $.inArray(className, element.parameterClasses) !== -1;
                });
            }

            if (val.length > 0) {
                if ($.inArray('disabled', val[0].parameterClasses) !== -1) {
                    return false;
                }

                var ipString = val[0].parameterValue;
                return this.ipStringToDecimal(ipString);
            }

            return false;
        },

        /** Replaces subnet address part:
        *
        *   1) It splits previousAddress into Network (192.168.1.) and Host(.25) parts using Netmask.
        *   2) Replaces Network part with one from Address (10.0.0.). Address Host part is gained using Netmask.
        *   3) Result is constructed from Address network part, and previousAddress Host part
        *
        *   This is used to auto-replace some Static IP / DHCP Range to new subnet, as result of Address/Netmask field change.
        *
        *   Example:  self.replaceSubnet(192.168.1.25, 10.0.0.1, 255.255.255.0)
        *   Result: 10.0.0.25
        */
        replaceSubnet: function(previousAddressStr, AddressStr, NetmaskStr) {
            var Address = this.ipStringToDecimal(AddressStr),
                Netmask = this.ipStringToDecimal(NetmaskStr),
                previousAddress = this.ipStringToDecimal(previousAddressStr);

            if (Address && Netmask && previousAddress) {
                previousAddress = ((previousAddress & ~Netmask) + (Address & Netmask)) >>>0;
                return this.decimalToIpString(previousAddress);
            } else {
                return false;
            }
        },

        // Converts 192.168.1.1 to decimal representation
        ipStringToDecimal: function(ipString) {
            if (!_.isString(ipString) || !ipString.match(/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/)) { //Not valid format
                return false;
            }

            var ip = ipString.split("."),
                ipNumber = 0,
                invalid = false;

            $.each(ip, function(i, v) {
                var octet = parseInt(v, 10);

                if (octet > 255 || octet < 0) { // Check if octet number is not exceeding valid value
                    invalid = true;
                }
                ipNumber += octet * Math.pow(2, 8*(3-i));
            });

            if (invalid) {
                return false;
            }

            return ipNumber;
        },

        // Converts decimal to string (192.168.1.1) representation
        decimalToIpString: function(ipDecimal) {
            var result = "",
                octet = 0,
                left = 0,
                pow = Math.pow(2,8);

            left = ipDecimal;

            for (var i=0; i<4; i++) {
                octet = left % pow;
                left = Math.floor(left / pow);
                result = "." + octet.toString(10) + result;
            }

            return result.substr(1);
        },

        // Checks if address is withing required subnet
        isAddressWithinSubnet: function(CheckIP, Address, Netmask) {
            if (Address===false || Netmask===false) {
                return false;
            }
            return (Address & Netmask >>> 0) === (CheckIP & Netmask >>> 0);
        },

        /** Calculates allowed host range string for subnet
        *   @param  int   123451231      Ip address in decimal form that is contained in subnet
        *   @param  int   3423525325     Ip netmask in decimal form to identify subnet
        *   @return string               Range string '192.168.1.1 - 192.168.1.126' if Address/Netmask are valid, bool false if not
        */
        getAllowedHostRange: function(Address, Netmask) {
            var MinAddr, MaxAddr;

            if (Address!==false && Netmask!==false) {
                MinAddr = ((Address & Netmask) >>> 0) + 1;
                MaxAddr = ((Address & Netmask >>> 0) - 1 + ~Netmask) >>> 0;

                return this.decimalToIpString(MinAddr) + " - " +this.decimalToIpString(MaxAddr);
            }

            return false;
        },
        /** Calculates allowed host range min Decimal for subnet
        *   @param  int   123451231      Ip address in decimal form that is contained in subnet
        *   @param  int   3423525325     Ip netmask in decimal form to identify subnet
        *   @return string               Range string '192.168.1.1 - 192.168.1.126' if Address/Netmask are valid, bool false if not
        */
        getMinAllowedHostAsDecimal: function(Address, Netmask) {
            var MinAddr;

            if (Address!==false && Netmask!==false) {
                MinAddr = ((Address & Netmask) >>> 0) + 1;
                return MinAddr;
            }
            return false;
        },
        /** Calculates allowed host range max Decimal for subnet
        *   @param  int   123451231      Ip address in decimal form that is contained in subnet
        *   @param  int   3423525325     Ip netmask in decimal form to identify subnet
        *   @return string               Range string '192.168.1.1 - 192.168.1.126' if Address/Netmask are valid, bool false if not
        */
        getMaxAllowedHostAsDecimal: function(Address, Netmask) {
            var  MaxAddr;

            if (Address!==false && Netmask!==false) {
                MaxAddr = ((Address & Netmask >>> 0) - 1 + ~Netmask) >>> 0;
                return MaxAddr;
            }
            return false;
        }
    },

    sync: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();
        if(self.autoSync===false){
         return deferred.promise();
        }
        this.reset();

        $.when(this.getIPSettings(fromListener),
                this.getNetworkStatus(fromListener),
                this.loadDMZSettings(fromListener))
            .done(function() {
                self.trigger('change');
                deferred.resolve();
            });

        return deferred.promise();
    },

    getIPSettings: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC:getLANIP',
            fromListener: fromListener,

            data: {
                "parameters": {}
            },

            success: function(response) {
                var model = new swc.constructors.SimpleModel();

                if (response.data) {
                    var data = response.data;

                    // Set model id
                    model.set('id', 'ip-settings');

                    // Set model data
                    model.set('LocalIPAddress', data.Address);
                    model.set('Netmask', data.Netmask);
                    model.set('DHCPStatus', data.DHCPEnable);
                    model.set('DHCPStart', data.DHCPMinAddress);
                    model.set('DHCPFinish', data.DHCPMaxAddress);

                    // Add model to current collection
                    self.add(model);

                    deferred.resolve();
                } else {
                    deferred.resolve();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    /**
     * Get gateway internet connection status based on data passed
     *
     * @param {Object} data Set of data returned by NP side
     *
     * @return {boolean}
     */
    hasInternetConnection: function(data) {
        var state = data.ConnectionState,
            isBound = (state === "Bound"),
            isRenewing = (state === "Renewing" && data.Protocol === "dhcp"),
            isConnected = (data.ConnectionState === "Connected");

        var result = ((isBound || isRenewing || isConnected) && (data.LinkState === "up") && (data.IPAddress !== "0.0.0.0"));

        return result;
    },

    getConnectionMode: function (data) {
        var result = null;
        if (data.LinkState === "down") {
            result = "NOT_CONNECTED";
        } else {
            if (data.ConnectionState === "Bound" || data.ConnectionState === "Connected") {
                result = "CONNECTED";
            } else {
                result = "CONNECTION_IN_PROGRESS";
            }
        }
        return result;
    },

    getNetworkStatus: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC:getWANStatus',
            fromListener: fromListener,

            data: {
                "parameters": {}
            },

            success: function(response) {
                var model = new swc.constructors.SimpleModel(),
                    moreInfo = [];

                if (response.data) {
                    var data = response.data;

                    // Set model id
                    model.set('id', 'status');

                    // Set model data
                    model.set('ConnectionType', data.LinkType);
                    model.set('ConnectionState', data.ConnectionState);
                    model.set('ConnectionStatus', self.hasInternetConnection(data));
                    model.set('ConnectionMode', self.getConnectionMode(data));
                    model.set('DeviceName', 'Internet Box');
                    model.set('Protocol', data.Protocol);
                    model.set('LastConnection', 'none');
                    model.set('RemoteIPAddress', data.IPAddress);
                    model.set('DNSServers', data.DNSServers ? data.DNSServers.split(',') : []);
                    model.set('IPv6Address', data.IPv6Address);
                    model.set('StorageIPAddress', 'unknown');

                    // Add model to current collection
                    self.add(model);

                    moreInfo.push(self.getStorageIP(model, fromListener));
                    if (!model.get('ConnectionStatus')) {
                        moreInfo.push(self.getLastConnection(model, fromListener));
                    }
                    moreInfo.push(self.getMacAddress(model, fromListener));
                    moreInfo.push(self.getIPv6Prefix(model, fromListener));
                    $.when.apply(null, moreInfo).then(function () {
                        deferred.resolve();
                    });

                } else {
                    deferred.resolve();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    getIPv6Prefix: function(model, fromListener) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NeMo/Intf/data:getMIBs',
            fromListener: fromListener,

            data: {
                "parameters": {
                    "mibs": "6rd"
                }
            },

            success: function(response) {
                if (response.status) {
                    if(response.status['6rd']['6rd']) {
                        model.set('IPv6Prefix', response.status['6rd']['6rd'].DelegatedPrefix);
                    }

                    deferred.resolve();
                } else {
                    deferred.reject();
                }
            },

            error: function() {
                deferred.resolve();
            }
        });

        return deferred.promise();
    },
    
    getMacAddress: function(model, fromListener) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NeMo/Intf/lan:getFirstParameter',
            fromListener: fromListener,

            data: {
                "parameters": {
                    "name": "LLAddress",
                    "flag": "",
                    "traverse": "down"
                }
            },

            success: function(response) {
                if (response.status) {
                    if (response.status) {

                        model.set('MacAddress', response.status);

                        deferred.resolve();
                    } else {
                        deferred.reject();
                    }
                }
                deferred.resolve();
            },

            error: function() {
                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    getLastConnection: function (model, fromListener) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NeMo/Intf/data:getMIBs',
            fromListener: fromListener,

            data: {
                "parameters": {}
            },

            success: function(response) {
                if (response.status) {
                    var data = response.status,
                        last;
                    if (model.get('ConnectionType') === 'ethernet' ) {
                        last = data.eth;
                    }
                    else {
                        last = data.dsl;
                    }
                    last = _.values(last);
                    if (last.length > 0 && last[0].LastChange) {
                        last = (new Date()).getTime() - last[0].LastChange * 1000;
                        model.set('LastConnection', last);

                    }
                }
                deferred.resolve();
            },

            error: function() {
                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    getStorageIP: function (model, fromListener) {
        var deferred = new $.Deferred();

        if(swc.Utils.DeviceType !== 'default') {
            model.set('StorageIPAddress', undefined);
            deferred.resolve();
            return deferred.promise();
        }

        swc.models.Rest.sendRequest({
            url: '/sysbus/APController/LAN:get',
            data: {
                parameters: null
            },
            fromListener: fromListener,

            success: function(response) {
                if (response.status) {
                    var data = response.status;
                    _.each(data, function (val, name) {
                        if (name === 'IP') {
                            model.set('StorageIPAddress', val);
                        }
                    });
                }
                deferred.resolve();
            },

            error: function() {
                deferred.resolve();
            }
        });
        return deferred.promise();
    },
    
    saveStorageIP: function(address) {
        var deferred = new $.Deferred();
        
        swc.models.Rest.sendRequest({
            url: '/sysbus/APController/LAN:setIPAddress',
            timeout: 2000,
            data: {
                parameters: {
                    ipaddress: address
                }
            },
            
            success: function(response) {
                if (response.status) {
                    deferred.resolve();
                } else {
                    deferred.reject();
                }
            },

            error: function() {
                deferred.reject();
            }
        });
        return deferred.promise();
    },

    saveIPSettings: function(data) {
        var self = this,
            address, netmask, dhcpStatus, dhcpMin, dhcpMax,
            refreshUrl = swc.settings.application.get('url') + '#network/settings/ip-settings',
            addressOld = this.get('ip-settings').get('LocalIPAddress'),
            deferred = new $.Deferred();

        // Get neccessary params from array of page data:
        $.each(data, function(key, value) {
            if (value.parameterName === "Address") {
                address = value.parameterValue;
            }

            if (value.parameterName === "Netmask") {
                netmask = value.parameterValue;
            }

            if (value.parameterName === "DHCPEnable") {
                dhcpStatus = value.parameterValue;
            }

            if (value.parameterName === "DHCPRange") {
                if ($.inArray('range-from', value.parameterClasses) !== -1) {
                    dhcpMin = value.parameterValue;
                } else {
                    dhcpMax = value.parameterValue;
                }
            }
        });

        if (!dhcpMin) { //NP requires DHCPMin and DHCPMax even if DHCPStatus=false
            dhcpMin = self.get('ip-settings').get('DHCPStart');
        }

        if (!dhcpMax) {
            dhcpMax = self.get('ip-settings').get('DHCPFinish');
        }

        
        // append timout to auto resolve if we do not have answer
        var timeout = setTimeout(function() {
            deferred.resolve();
        }, 3000);

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC:setLANIP',
            timeout: 2000,
            data: {
                parameters: {
                    Address: address,
                    Netmask: netmask,
                    DHCPEnable: dhcpStatus,
                    DHCPMinAddress: dhcpMin,
                    DHCPMaxAddress: dhcpMax
                }
            },

            success: function(response) {
                clearTimeout(timeout);
                
                if (response.errors && response.errors.length > 0) {
                    deferred.reject(response.errors[0].description);
                } else {
                    // in case of success the return "{status: null}"
                    deferred.resolve();
                }
            },

            error: function(xhr, ajaxOptions) {
                clearTimeout(timeout);
                
                if (ajaxOptions === "timeout") {
                    if (addressOld !== address) {
                        document.location.href = refreshUrl;
                    } else {
                        deferred.reject();
                    }
                } else {
                    deferred.reject();
                }
            }
        });
        
        return deferred.promise();
    },

    loadDMZSettings: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Firewall:getDMZ',
            fromListener: fromListener,
            data: {
                parameters: {}
            },

            success: function(response) {
                var model = new swc.constructors.SimpleModel();

                model.set('id', 'dmz');

                // Check if DMS is empty
                if (response.status === null || !response.status.webui) {
                    model.set('deviceIP', '');
                } else {
                    // the returned value is not a valid json string / not parsable, tryed JSON.parse and $.parseJSON()
                    model.set('deviceIP',
                        response.status.webui.DestinationIPAddress // Bullshit
                    );
                }

                self.add(model);

                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    saveDMZSettings: function(data) {

        var deferred = new $.Deferred(),
            enabled = false,
            assotiatedDeviceMAC,
            assotiatedDeviceIP;

        // Get neccessary params from array of page data:
        $.each(data, function(key, value) {
            if (value.parameterName === "DMZEnable") {
                enabled = value.parameterValue;
            }

            if (value.parameterName === "DMZIPAddress") {
                assotiatedDeviceMAC = value.parameterValue;
            }
        });

        // Check for critical situation:
        if (enabled && !assotiatedDeviceMAC) {
            return deferred.resolve();
        }

        //check the if mac address is not set to custom if so get the custom ip
        if(assotiatedDeviceMAC === "custom"){
            assotiatedDeviceIP  = _.findWhere(data, {parameterName: 'DMZCustomIpAddress'});
        }
        else {
           // Get device ip address:
           assotiatedDeviceIP = swc.models.NetworkDevices.getDeviceIP(assotiatedDeviceMAC);
        }
        if (!enabled) {
            $.when(this.disableDMZ()).done(function (){
                deferred.resolve();
            });
        } else {
            $.when(this.enableDMZ(assotiatedDeviceIP)).done(function (){
                deferred.resolve();
            });
        }

        return deferred.promise();
    },

    enableDMZ: function(deviceIP) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Firewall:setDMZ',
            data: {
                parameters: {
                    destinationIPAddress: deviceIP,
                    enable: true,
                    id: "webui",
                    sourceInterface: "data"
                }
            },

            success: function() {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    disableDMZ: function() {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Firewall:deleteDMZ',
            data: {
                parameters: {
                    id: "webui"
                }
            },

            success: function() {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    getParameter: function(parameter, storage) {
        var search = this.where({ id: storage });

        if (search.length) {
            return search[0].get(parameter);
        } else {
            return false;
        }
    },

    doesParameterExist: function(parameter, storage) {
        var search = this.where({ id: storage });

        if (search.length) {
            return true;
        } else {
            return false;
        }
    }

});
;swc.constructors.NetworkDevices = Backbone.Collection.extend({

    autoSync: true,
    /**
     * Sort ASC by IPv4 address.
     * Disconnected devices should be displayed last
     */
    comparator: function(device1, device2) {
        // If no IP address - show device as last
         if (device1.get('address').IPv4 && !device2.get('address').IPv4) {
            return -1;
        }

        // All devices with IP address provided should be sorted ASC
        if (device1.get('address') && device2.get('address')) {
            if (!_.isEmpty(device1.get('address').IPv4) && !_.isEmpty(device2.get('address').IPv4)) {
                var device1_ipComponents = device1.get('address').IPv4.split("."),
                    device2_ipComponents = device2.get('address').IPv4.split(".");
                return _.reduce(device1_ipComponents, function (memo, nextVal) {
                    return memo * 255 + parseInt(nextVal, 10);
                }) < _.reduce(device2_ipComponents, function (memo, nextVal) {
                    return memo * 255 + parseInt(nextVal, 10);
                }) ? -1 : 1;
            } else {
                return 1;
            }
        }

        return 0;
    },

    sync: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        if(self.autoSync===false){
         return deferred.resolve();
        }
        // Check if required models exists:
        swc.models.Wireless = !_.isUndefined(swc.models.Wireless) ? swc.models.Wireless : new swc.constructors.Wireless();

        // Fetch wifi model before fetching devices list:
        swc.models.Wireless.sync(fromListener).then(function() {
            $.when(self.fetchDevices(fromListener))
                .done(function() {
                    self.trigger('devices-loaded', true); // Trigger event to notify - devices update finished
                    deferred.resolve();
                })
                .fail(function() {
                    deferred.reject();
                });
        });

        return deferred.promise();
    },

    /**
     * Get list of devices from gMap EP
     *
     * @note:
     *
     * skip device with name "internetbox-nas"
     *
     * @param fromListener {Boolean}
     *
     * @returns {*}
     */
    fetchDevices: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        function error() {
            self.parseDevices([]);
            deferred.resolve();
        }

        var testingJSONMalformed = swc.settings.application.get('testing').testingJSONMalformed === 'true' ? true : false;

        swc.models.Rest.sendRequest({
            url: '/ws',
            method: 'POST',
            fromListener: fromListener,
            testingJSONMalformed: testingJSONMalformed,

            data: {
                service: 'Devices',
                method: 'get',
                parameters: {
                    expression: 'lan and not self'
                }
            },

            parseProblem: function() {
                self.parseDevices([]);
                deferred.resolve();
            },

            success: function(response) {
                if(response.error) {
                    error();
                } else {
                    var devices = [],
                        currentDateStamp = new Date(swc.Utils.getCurrentDate()).getTime(),
                        deviceLastConnectionStamp;

                    // Check if response is correct:
                    if (!_.isUndefined(response.status) && _.isArray(response.status)) {
                        _.each(response.status, function(device) {

                            deviceLastConnectionStamp = new Date(device.LastConnection).getTime();

                            // Skip internetbox-nas devices for devices list and devices older than 30 days
                            if (device.Name !== 'internetbox-nas' && (currentDateStamp < deviceLastConnectionStamp + swc.Utils.maxDeviceLastConnectionStamp || device.Active)) {
                                devices.push(device);
                            }
                        });
                    }
                    
                    self.parseDevices(devices);
                    deferred.resolve();
                }
            },

            fail: function() {
                error();
            }
        });

        return deferred.promise();
    },

    /**
     * Parse response from gMap devices end point
     *
     * @description:
     *
     * Method creates models for devices based on response from EP
     *
     * @param devices {Collection}
     */
    parseDevices: function(devices) {
        var self = this;

        // Check if response has devices:
        if (!_.isArray(devices) || !devices.length) {
            return;
        }

        // Reset collection before setting new devices:
        this.reset();

        // Go through each device and create a model for it:
        _.each(devices, function(device) {
            // Add model to the collection
            self.add(self.prepareDeviceModel(device));
        });
    },
    
    findDeviceType: function(device) {
        var type = device.DeviceType;
        
        if(_.isEmpty(type)) {
            type = device.DeviceTypes[0].Type;
        }
        
        type = type.toLowerCase();
        
        return type;
    },

    /**
     * Set type of device from data param and check if this type
     * should be mapped according to map "devicesMapping" in the
     * application.json file
     *
     * This map was added according the following tickets:
     *      - SA-2134
     *      - SA-2514
     *
     * @param data {Object} Response from request "/sysbus/Hosts:getDevices"
     * @returns {String}
     */
    getDeviceType: function(type) {
        var devicesMapping = swc.settings.application.get('devicesMapping'),
            internalType = type.toLowerCase();
        
        if (!_.isUndefined(devicesMapping[internalType])) {
            type = devicesMapping[internalType];
        }

        return type;
    },

    /**
     * Get icon of device by deviceType param from hashes
     * "supportedWiredDevices" and "supportedWirelessDevices"
     * in the application.json file
     *
     * @param interfaceType {String} wired | wireless
     * @param deviceType {String} Type of device as is from the field "deviceType" in the response
     */
    getDeviceIcon: function(interfaceType, deviceType) {
        var supportedWired = swc.settings.application.get('supportedWiredDevices'),
            supportedWireless = swc.settings.application.get('supportedWirelessDevices'),
            icon = '';

        if (interfaceType === 'wired') {
            if (!supportedWired[deviceType] || _.isEmpty(deviceType)) {
                icon = supportedWired["unrecognized"].deviceIcon;
            } else {
                icon = supportedWired[deviceType].deviceIcon;
            }
        }

        if (interfaceType === 'wireless') {
            if (!supportedWireless[deviceType] || _.isEmpty(deviceType)) {
                icon = supportedWireless["unrecognized"].deviceIcon;
            } else {
                icon = supportedWireless[deviceType].deviceIcon;
            }
        }

        return icon;
    },

    /**
     * Get interface of access point based on the
     * field "layer2Interface" in the response
     * of the request "/sysbus/Hosts:getDevices"
     *
     * @param wifiApLayer {String}
     */
    getInterfaceAP: function(wifiApLayer) {
        var map = { 'wl0': '2.4GHz',
                    'wl1': '5GHz',
                    'wl0.1': 'Guest-2.4GHz',
                    'wl1.1': 'Guest-5GHz'};
                    // 'wl0.1': getTranslationStrings('Guest-2.4GHz'),
                    // 'wl1.1': getTranslationStrings('Guest-5GHz')};

        return !_.isUndefined(map[wifiApLayer]) ? map[wifiApLayer] : '';
    },

    /**
     * Parse data in the response of the request "/sysbus/Hosts:getDevices"
     * and set appropriate fields to new device model
     *
     * @param data {Object} Response data
     * @return {Backbone.Model} Device model
     */
    createDevice: function(data) {
        var model = new swc.constructors.SimpleModel(),
            status = false,
            statusClass='disconnected',
            addressIPv4 = '', addressIPv6 = [],
            addressObjIPv4 = {}, addressObjIPv6 = [],
            interfaceType = '', interfaceSSID = '',
            wifiOperatingChannelBandwidth = getTranslationStrings('Auto'),
            wifiOperatingStandards = '',
            gatewayPort = '', icon = '',
            type, wifiApLayer, interfaceAP;

        // Check if device is valid
        if (!data) {
            return model;
        }

        var deviceType = this.findDeviceType(data);
        type = this.getDeviceType(deviceType);
        wifiApLayer = data.layer2Interface;

        // Define interface type
        if (data.interfaceType === "Ethernet") {
            interfaceType = "wired";
            icon = this.getDeviceIcon('wired', type);
        } else {
            interfaceType = "wireless";
            icon = this.getDeviceIcon('wireless', type);
            interfaceAP = this.getInterfaceAP(wifiApLayer);
            interfaceSSID = swc.models.Wireless.getParameter("name", interfaceAP);
            wifiOperatingChannelBandwidth = swc.models.Wireless.getParameter("OperatingChannelBandwidth", interfaceAP);
            wifiOperatingStandards = this.preparedWiFiStandard(swc.models.Wireless.getParameter("OperatingStandards", interfaceAP));
        }

        // Define IP Address:
        $.each(data.addresses, function(key, ipAddrItem){
            if ( ipAddrItem.family === 'IPv4') {
                addressIPv4 = ipAddrItem.ipAddress;
                addressObjIPv4 = ipAddrItem;
            } else {
                addressIPv6.push(ipAddrItem.ipAddress);
                addressObjIPv6.push(ipAddrItem);
            }
        });

        // Define device status:
        if ((addressIPv4.length || addressIPv6.length) && data.active) {
            status = true;
        }

        if (status) {
            statusClass = 'connected';
        }

        if (data.interfaceType === "Ethernet" && /eth\d*/.test(wifiApLayer)){
            gatewayPort = parseInt(wifiApLayer.replace('eth',''), 10) + 1;
        }

        // Base data:
        model.set('mac', data.physAddress.toUpperCase());
        model.set('name', data.hostName);
        model.set('dnsName', data.dnsName);
        model.set('type', type);
        model.set('icon', icon);
        model.set('status', status);
        model.set('statusClass', statusClass);

        // IP Address:
        model.set('address', {
            'IPv4': addressIPv4,
            'IPv6': addressIPv6
        });
        model.set('addressObjects', {
            'IPv4': addressObjIPv4,
            'IPv6': addressObjIPv6
        });

        // Interface type:
        model.set('interfaceType', interfaceType);
        model.set('interfaceSSID', interfaceSSID);
        model.set('interfaceAP', interfaceAP);
        model.set('gatewayPort', gatewayPort);

        // Some information about WiFi connection
        model.set('wifiOperatingChannelBandwidth', wifiOperatingChannelBandwidth);
        model.set('wifiOperatingStandards', wifiOperatingStandards);

        // Last connection and change:
        model.set('lastChange', data.lastChange);
        model.set('lastConnection', data.lastConnection);
        
        // Hidden flag
        model.set('isHidden', data.isHidden);

        return model;
    },

    // According to SA-3217:
    preparedWiFiStandard: function (standard) {
        if (standard === 'bgn') {
            return 'b/g/n';
        }

        if (standard === 'ac') {
            return 'a/n/ac';
        }

        return standard;
    },

    getDevice: function(MacAddress) {
        var search = this.where({ mac: MacAddress });

        if (search.length) {
            return search[0].attributes;
        } else {
            return false;
        }
    },

    getDeviceIP: function(MacAddress) {
        var device = this.getDevice(MacAddress),
            ip = '';

        if (!device) {
            return ip;
        }

        if (device.address['IPv4']) {
            ip = device.address['IPv4'];
        } else {
            if (device.address['IPv6'].length) {
                ip = device.address['IPv6'][0];
            }
        }

        return ip;
    },

    getDeviceMac: function(IPAddress) {
        var devicesAll = this.getConnectedDevices(),
            devices = devicesAll['wired'].concat(devicesAll['wireless']),
            mac = '';

        $.each(devices, function(key, device) {
            if (device.address['IPv4'] === IPAddress) {
                mac = device.mac;
            } else {
                if (device.address['IPv6'].length) {
                    $.each(device.address['IPv6'], function(ipKey, ipValue) {
                        if (ipValue === IPAddress) {
                            mac = device.mac;
                        }
                    });
                }
            }
        });

        return mac;
    },

    getDeviceByIP: function(IPAddress) {
        var device;

        _.each(this.models, function(model) {
            if (!_.isEmpty(model.get('address')['IPv4']) && model.get('address')['IPv4'] === IPAddress) {
                device = model;
            } else {
                if (!_.isEmpty(model.get('address')['IPv6'])) {
                    _.each(model.get('address')['IPv6'], function(IPv6) {
                        if (IPv6 === IPAddress) {
                            device = model;
                        }
                    });
                }
            }
        });

        return device;
    },

    getDevicesList: function() {
        var devices = [],
            preconfigured = [],
            connected = [],
            disconnected = [],
            schedulerDeviceCollection,
            thirtyDays = 30*24*3600000;

        if (!_.isUndefined(swc.models.SchedulerDeviceCollection)) {
            schedulerDeviceCollection = swc.models.SchedulerDeviceCollection;
        }

        // Sort devices to connected and disconnected
        $.each(this.models, function(key, model) {
            var device = model.attributes,
                lastConnection = moment(device.lastConnection).unix() * 1000,
                now = new Date().getTime(),
                isPreconfigured = device.address['static'],
                scheduler = schedulerDeviceCollection.get(device.mac);


            if(!device.isHidden) {
                if (!device.status) {
                    if(!isPreconfigured && scheduler) {
                        if(scheduler.get('enable') === true && swc.mixins.Scheduler.isEmpty(scheduler.get('schedule')) === false) {
                            preconfigured.push(device);
                            return;
                        }
                    }

                    if (isPreconfigured) {
                        preconfigured.push(device);
                    } else if (disconnected.length < 10 && ((device.lastConnection === '0001-01-01T00:00:00Z') || ((now - lastConnection) < thirtyDays))) {
                        disconnected.push(device);
                    }
                } else {
                    connected.push(device);
                }
            }
        });

        function sortDevicesByIP(deviceA, deviceB) {
            if(deviceA.address.IPv4 === undefined) {
                return -1;
            }
            else if(deviceB.address.IPv4 === undefined) {
                return 1;
            }
            else {
                return deviceA.address.IPv4.localeCompare(deviceB.address.IPv4);
            }
        }

        function sortDevicesByName(deviceA, deviceB) {
            return deviceA.name.localeCompare(deviceB.name);
        }

        connected.sort(sortDevicesByIP);
        preconfigured.sort(sortDevicesByIP);
        disconnected.sort(sortDevicesByName);

        return devices.concat(connected).concat(preconfigured).concat(disconnected);
    },

    getConnectedDevices: function() {
        var devices = {
            'wired': [],
            'wireless': []
        };

        // Sort devices to connected and disconnected
        $.each(this.models, function(key, model) {
            var device = model.attributes;

            if (device.status) {
                devices[device.interfaceType].push(device);
            }
        });

        return devices;
    },

    countWiredConnectedDevices: function() {
        return this.where({ status: true, interfaceType: 'wired' }).length;
    },

    removeDevice: function(deviceMac) {
        var device = this.findWhere({ mac: deviceMac }),
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                "service": "Devices.Device." + device.get('key'),
                "method": "setTag",
                "parameters": {
                    "tag": "scs_hidden"
                }
            },

            success: function() {
                deferred.resolve();
            },

            fail: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    customizeDevice: function(data) {
        var device = this.findWhere({ mac: data.deviceMac }),
            deferred = new $.Deferred();

        function sendRequest(parametersObject, successCallback, deferred, delay) {
            setTimeout(function () {
                swc.models.Rest.sendRequest({
                    url: '/ws',
                    data: {
                        service: 'Devices.Device.' + device.get('key'),
                        method: 'set',
                        parameters: {
                            parameters: parametersObject
                        }
                    },
                    success: successCallback,
                    fail: function() {
                        deferred.reject();
                    }
                });
            }, delay);
        }

        sendRequest(
            {
                Name: data.deviceName
            }, function () {
                sendRequest(
                    {
                        DeviceType: data.deviceType
                    }, function () {
                        deferred.resolve();
                    }, deferred, 1500
                );
            }, deferred, 0
        );

        return deferred.promise();
    },
/**
 * params.filter        - status
 *                      - DHCPLeases
 * params.WlanGuest     - true //made only for status check, allows returning non guest devices
 *                      - false 
 */
    getDeviceDropdownOptions: function(params) {
        var devices = swc.models.NetworkDevices.filter(function(model) {
                    if (!params || !params.filter) {
                        return true;
                    } else if (params.filter === 'status') {
                        if(params.WlanGuest === undefined ){
                            return !!model.get('status');
                        } else {
                            if(params.WlanGuest === false && "Guest-2.4GHz" === model.get("interfaceAP")){
                                return false;
                            } else {
                                return !!model.get('status');
                            }
                        }
                    } else if (params.filter === 'DHCPLeases') {
                        if(params.WlanGuest === false && "Guest-2.4GHz" === model.get("interfaceAP")){
                                return false;
                        }
                        if((_.isUndefined(model.get('addressObjects').IPv4) || _.isEmpty(model.get('addressObjects').IPv4)) && (model.get('addressObjects').IPv6 !== undefined && !_.isEmpty(model.get('addressObjects').IPv6))){
                            return false;
                        }
                        return !swc.models.DHCPLeases.get(model.get("mac"));
                    }
                }),
            options = _.map(devices, function(model) {
                var ipAddress = model.get('status') ? model.get('address').IPv4 : "";

                if (ipAddress && ipAddress.length > 15) { // Truncate IPv6 last 3 groups
                    ipAddress = ipAddress.match(/.*((?::[a-z0-9]*){3})/)[1];
                }

                var option = {
                    "name": model.get("name"),
                    "value": model.get("mac"),
                    "additionalLabel": ipAddress
                };

                return option;
            });

        return options;
    },
    /*
    Method for getting IP conflicts pairs
    */
    getConflicts: function() {

        var devices = swc.models.NetworkDevices.models,
            length  = devices.length,
            storageIP = swc.models.Network.getParameter('StorageIPAddress', 'status') || '',
            ret = [];

            //clear conflicts
            for(var i = 0; i < length; i++) {
                devices[i].attributes.hasConflict = false;
            }

            for(i = 0; i < length; i++) {
                for(var j = i; j < length; j++) {
                    if(devices[i].attributes.status && devices[j].attributes.status && devices[i].attributes.key !== devices[j].attributes.key && devices[i].attributes.address.IPv4 === devices[j].attributes.address.IPv4) {
                        devices[i].attributes.hasConflict = true;
                        devices[j].attributes.hasConflict = true;
                        ret.push([devices[i], devices[j]]);
                    }
                }
            }

            if(swc.Utils.DeviceType !== "default") {
                for(i = 0; i < length; i++) {
                    if(devices[i].attributes.status && devices[i].attributes.address.IPv4 === storageIP) {
                        devices[i].attributes.hasConflict = true;
                        ret.push([swc.models.Network.get('status'), devices[i]]);
                    }
                }
            }
            

            return ret;
    },

    getConflictStatus: function(IP) {

        var devices = swc.models.NetworkDevices.models,
            status = false,
            adr = IP || false;

        _.each(devices, function(device){
             if(adr && device.attributes.address.IPv4 === adr &&  device.attributes.status) {
                status = true;
            }
            else if(!adr && device.attributes.name !== "internetbox-nas" && device.attributes.hasConflict && device.attributes.status) {
                status = true;
            }
        });

        return status;
    },
    
    getDeviceDropdownOptionsOnlyIpv4: function(params) {
        var ipAddress = null;
        var devices = swc.models.NetworkDevices.filter(function(model) {
                    if (!params || !params.filter) {
                        return true;
                    } else if (params.filter === 'status') {
                        ipAddress = model.get('status') ? model.get('address').IPv6 : "";
                        if (model.get('addressObjects').IPv6 !== undefined && !_.isEmpty(model.get('addressObjects').IPv6)){ //check if ip is ipv6
                            //in case device uses ipv4 and ipv6
                            if(model.get('addressObjects').IPv4 !== undefined && !_.isEmpty(model.get('addressObjects').IPv4)){
                                if(params.WlanGuest === false && "Guest-2.4GHz" === model.get("interfaceAP")){
                                    return false;
                                }
                                return !!model.get('status');
                            }else {
                                return false;
                            }
                        }
                        if(params.WlanGuest === undefined){
                            if (ipAddress && ipAddress.length > 15){
                            }
                        } else {
                            if(params.WlanGuest === false && "Guest-2.4GHz" === model.get("interfaceAP")){
                                return false;
                            } else {
                                return !!model.get('status');
                            }
                        }
                    } else if (params.filter === 'DHCPLeases') {
                        return !swc.models.DHCPLeases.get(model.get("mac"));
                    }
                }),
            
            options = _.map(devices, function(model) {
                var ipAddress = model.get('status') ? model.get('address').IPv4 : "";
                
                var option = {
                    "name": model.get("name"),
                    "value": model.get("mac"),
                    "additionalLabel": ipAddress
                };
                
                return option;
                
            });

        return options;
    },
    /**
     * method for converting device data to device model
     * return Backbon.Model
     * 
     * @param device - json object with device info
     */
    prepareDeviceModel:function(device){
        var model = new Backbone.Model(),
                settings = {},
                self= this,
                wlanModel = swc.models.Wireless;

            // Define interface type
            if (_.indexOf(device.Tags.split(' '), 'wifi') !== -1) {
                settings.interfaceType = "wireless";

                // Define Wlan access point and it's name
                settings.interfaceAP = self.getInterfaceAP(device.Layer2Interface);
                settings.interfaceAPName = getTranslationStrings(settings.interfaceAP);
                settings.interfaceSSID = wlanModel.getParameter("name", settings.interfaceAP);

                // Get additional wlan settings:
                settings.wlanStandart = self.preparedWiFiStandard(wlanModel.getParameter("OperatingStandards", settings.interfaceAP));
            } else {
                settings.interfaceType = "wired";
                if(!_.isUndefined(device.Layer2Interface)) {
                    settings.gatewayPort = parseInt(device.Layer2Interface.replace('eth',''), 10) + 1;
                }
            }

            // Get device icon and type:
            var deviceType = self.findDeviceType(device);
            settings.type = self.getDeviceType(deviceType);
            settings.icon = self.getDeviceIcon(settings.interfaceType, settings.type);

            // Define device ip addresses:
            settings.address = {
                IPv4: device.IPAddress,
                IPv6: _.pluck(device.IPv6Address, 'Address')
            };

            // Define device ip addresses objects:
            settings.addressObjects = {
                IPv4: device.IPv4Address,
                IPv6: device.IPv6Address
            };

            // Define device status:
            if ((!_.isEmpty(device.IPv4Address) || !_.isEmpty(device.IPv6Address)) && device.Active) {
                settings.status = true;
            } else {
                settings.status = false;
            }
            
            // Define if device is hidden
            if (_.indexOf(device.Tags.split(' '), 'scs_hidden') !== -1) {
                settings.isHidden = true;
            } else {
                settings.isHidden = false;
            }

            // Define device status class:
            settings.statusClass = settings.status === true ? 'connected' : 'disconnected';

            // Set model attributes:
            model.set(_.extend(settings, {
                key: device.Key,
                mac: device.PhysAddress,
                name: device.Name,
                dnsName: device.Name,
                lastChange: device.LastChanged,
                lastConnection: device.LastConnection
            }));
        return model;
    },
    updateDeviceList: function(devices){
        var self = this;
        
        // Go through each device and create a model for it:
        _.each(devices, function(device) {
            //remove old model depending on mac address
            for(var i=0;i<self.models.length;i++){
                if(self.models[i].attributes.key===devices[0].Key){
                    self.models[i].destroy();
                }
            }
            // {at: index, merge:true} - splice didnt work in backbone
            self.add(self.prepareDeviceModel(device));
        });
    }
});
;swc.constructors.PortForwarding = swc.BaseCollection.extend({

    model: 'PortForwardingRule',

    url: {
        read: "/sysbus/Firewall:getPortForwarding"
    },

    comparator: function(item) {
        return parseInt(item.get('ports').entry, 10);
    },

    // TODO :: when normal validation mechanism will be done -> refactor this
    validation: {

        deviceIP: function(data) {
            var validationData = swc.Utils.getDataToValidate({
                    deviceIP: { elementName: 'deviceIP' }
                }, data),
                validationMessages = [];

            if (validationData.deviceIP === "0.0.0.0") {
                validationMessages.push('device must be selected');
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },
        
        validateIPAddress: function(data) {
            // validate only if is custom ip
            var validationData = swc.Utils.getDataToValidate({
                    deviceIP: { elementName: 'deviceIP' },
                    ipAddressAP: { elementName: 'ipAddressAP' }
                }, data),
                netValidation = swc.models.Network.validation,
                validationStatus = {
                    status: true,
                    messages: []
                };
                
            if(netValidation.decimalToIpString(netValidation.getIPField(data, 'address')) === validationData.ipAddressAP) {
                    validationStatus.status = false;
                    validationStatus.messages.push("Cant set rule to AP address");
                    return validationStatus;
            }

            if(validationData.deviceIP === 'custom') {
                var checkIP = netValidation.getIPField(data, 'address'),
                    address = netValidation.ipStringToDecimal(swc.models.Network.get('ip-settings').get('LocalIPAddress')),
                    netmask = netValidation.ipStringToDecimal(swc.models.Network.get('ip-settings').get('Netmask')),
                    subnet = (address & netmask) >>> 0,
                    broadcast = (address | ~netmask) >>> 0;
                
                    // Check if format is valid
                if (checkIP === false) {
                    validationStatus.status = false;
                    validationStatus.messages.push('Invalid IPv4 Address Format Custom');
                    return validationStatus;
                }
    
                // Check if address is within right subnet
                if (!netValidation.isAddressWithinSubnet(checkIP, address, netmask)) {
                    validationStatus.status = false;
                    validationStatus.messages.push('IPv4 Address is outside of allowed host addresses range');
                    return validationStatus;
                }
    
                if (checkIP === broadcast) {
                    validationStatus.status = false;
                    validationStatus.messages.push("Not match broadcast");
                    return validationStatus;
                }
    
                if (checkIP === subnet) {
                    validationStatus.status = false;
                    validationStatus.messages.push("Not match subnet");
                    return validationStatus;
                }
            }
            
            return validationStatus;
        },

        serviceName: function(data) {
            var validationData = swc.Utils.getDataToValidate({
                    mode: { elementName: 'service-mode' },
                    name: { elementName: 'serviceName' },
                    id: { elementName: 'serviceID' }
                }, data),
                validationMessages = [];

            // Skip validation for predefined mode
            if (validationData.mode !== "predefined") {
                var searchModel = swc.models.PortForwarding.findWhere({ name: validationData.name });

                if (!$.trim(validationData.name)) {
                    validationMessages.push('must not be empty');
                }

                if (validationData.name.length < 1 || validationData.name.length > 40) {
                    validationMessages.push('must be from 1 to 40 characters');
                }

                // All printable characters are allowed
                if (!/^[ -~]+$/.test(validationData.name)) {
                    validationMessages.push('allowed characters ASCI non-printable');
                }

                // Skip check if model is editing
                if (!_.isUndefined(searchModel) && searchModel.get('id') !== validationData.id) {
                    validationMessages.push('already existed rule names are not allowed');
                }
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        predefinedServiceName: function(data) {
            var validationData = swc.Utils.getDataToValidate({
                    mode: { elementName: 'service-mode' },
                    name: { elementName: 'predefinedServiceName' }
                }, data),
                validationMessages = [];

            // Skip validation for predefined mode
            if (validationData.mode !== "custom") {
                var searchModel = swc.models.PortForwarding.findWhere({ name: validationData.name });

                if (!_.isUndefined(searchModel)) {
                    validationMessages.push('already existed rule names are not allowed');
                }
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        entryPort: function(data) {
            var validationData = swc.Utils.getDataToValidate({
                    mode: { elementName: "entry-ports-mode" },
                    from: { elementName: "entryPort", elementClass: "from" },
                    to: { elementName: "entryPort", elementClass: "to" },
                    id: { elementName: 'serviceID' }
                }, data),
                validationMessages = [],
                portsRange = [];
                
            // Convert to integer values:
            if (validationData.mode === "range") {
                portsRange = [ validationData.from, validationData.to ];
            } else {
                portsRange = [ validationData.from ];
            }
            
            var array = validationData.from.split(',');
            if(array.length > 1) {
                // check both ports
                var values = [$.trim(array[0])];
                if(portsRange.length > 1) {
                    values.push(portsRange[1]);
                }
                
                validationMessages = validationMessages.concat(
                    this.helpers.validateEntryPorts(validationData.id, values, validationData.mode));
                    
                values = [$.trim(array[1])];
                if(portsRange.length > 1) {
                    values.push(portsRange[1]);
                }
                
                validationMessages = validationMessages.concat(
                    this.helpers.validateEntryPorts(validationData.id, values, validationData.mode));
            } else {
                validationMessages = validationMessages.concat(
                    this.helpers.validateEntryPorts(validationData.id, portsRange, validationData.mode));
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        destinationPort: function(data) {
            var validationData = swc.Utils.getDataToValidate({
                    mode: { elementName: "destination-ports-mode" },
                    from: { elementName: "destinationPort", elementClass: "from" },
                    to: { elementName: "destinationPort", elementClass: "to" },
                    deviceIP: { elementName: 'deviceIP' },
                    id: { elementName: 'serviceID' }
                }, data),
                validationMessages = [],
                portsRange = [];

            // Convert to integer values:
            if (validationData.mode === "range") {
                portsRange = [ validationData.from, validationData.to ];
            } else {
                portsRange = [ validationData.from ];
            }
            
            var array = validationData.from.split(',');
            if(array.length > 1) {
                // check two ports
                var portsRanges = [[ $.trim(array[0]) ], [ $.trim(array[1]) ]];
                if(portsRange.length > 1) {
                    portsRanges[0].push(portsRange[1]);
                    portsRanges[1].push(portsRange[1]);
                }
                
                validationMessages = validationMessages.concat(this.helpers.validateDestinationPorts(validationData.id, validationData.deviceIP, portsRanges[0]));
                validationMessages = validationMessages.concat(this.helpers.validateDestinationPorts(validationData.id, validationData.deviceIP, portsRanges[1]));
            } else {
                validationMessages = validationMessages.concat(this.helpers.validateDestinationPorts(validationData.id, validationData.deviceIP, portsRange));
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        helpers: {

            /**
             * Check if ports are not crossing between each other
             *
             * @description
             *
             *  [1, 5] and [3, 6] are conflicting, because they are in same range
             *
             *  [1, 5] and [6, 7] are not conflicting
             *
             * @param ports {Array} length (1 || 2) Port to validate for
             * @param range {Array} length (1 || 2) Ports to validate against
             *
             * @returns {boolean}
             */
            portsNotInRange: function(ports, range){
                // Case 1 [] && []
                if (ports.length === 1 && range.length === 1) {
                    return +ports[0] !== +range[0];
                }

                // Case 2 [][] && []
                if (ports.length === 2 && range.length === 1) {
                    return +range[0] < +ports[0] ||
                        +range[0] > +ports[1];
                }

                // Case 3 [] && [][]
                if (ports.length === 1 && range.length === 2) {
                    return +ports[0] < +range[0] ||
                        +ports[0] > +range[1];
                }

                // Case 4 [][] && [][]
                if (ports.length === 2 && range.length === 2) {
                    return (+ports[0] < +range[0] &&
                        +ports[1] < +range[0]) ||
                        (+range[0] < +ports[0] &&
                            +range[1] < +ports[0]);
                }
            },
            
            validateEntryPorts: function(id, portsRange, mode) {
                var validationMessages = [],
                    from = +portsRange[0],
                    to = +portsRange[1];
    
                if (from < 1 || from > 65535) {
                    validationMessages.push('allowed range 1 to 65535');
                }
    
                if (/\D+/.test(from)) {
                    validationMessages.push('allowed characters 0-9');
                }
    
                // Skip validation for predefined mode
                if (mode === "range") {
    
                    if (!$.trim(to)) {
                        validationMessages.push('must not be empty');
                    }
    
                    if (/\D+/.test(to)) {
                        validationMessages.push('allowed characters 0-9');
                    }
    
                    if (to < 1 || to > 65535) {
                        validationMessages.push('allowed range 1 to 65535');
                    }
    
                    if (to <= from) {
                        validationMessages.push('range option first value MUST be less then second value');
                    }
                }

                var vpnModel1 = swc.models.PortForwarding.findWhere({ name: 'strongSwanIKE' }),
                vpnModel2 = swc.models.PortForwarding.findWhere({ name: 'strongSwanIPSec' });

                if(swc.models.PortForwarding.vpnGlobalEnable === true) {
                    if(swc.models.VPNServer.get('enable') === true) {
                        if (!swc.models.PortForwarding.validation.helpers.portsNotInRange(portsRange, [7547]) &&
                            (!swc.models.PortForwarding.validation.helpers.portsNotInRange(portsRange, [500]) ||
                            !swc.models.PortForwarding.validation.helpers.portsNotInRange(portsRange, [4500]))) {
                            validationMessages.push('ports 500 and 4500 are not allowed because these ports reserved for voip and port forwarding rules must not include the port 7547');
                        }
                    }
                } else if((!_.isUndefined(vpnModel1) && vpnModel1.get('origin') === 'internal') ||
                    (!_.isUndefined(vpnModel2) && vpnModel2.get('origin') === 'internal')) {
                    if (!swc.models.PortForwarding.validation.helpers.portsNotInRange(portsRange, [7547]) &&
                            (!swc.models.PortForwarding.validation.helpers.portsNotInRange(portsRange, [500]) ||
                            !swc.models.PortForwarding.validation.helpers.portsNotInRange(portsRange, [4500]))) {
                            validationMessages.push('ports 500 and 4500 are not allowed because these ports reserved for voip and port forwarding rules must not include the port 7547');
                    }
                }

                // For 7547 CWMP port
                if (!swc.models.PortForwarding.validation.helpers.portsNotInRange(portsRange, [7547])) {
                    validationMessages.push('ports 1024-1048 are not allowed because this ports reserved for system use');
                }

                if(swc.models.PortForwarding.vpnGlobalEnable === true) {
                    if(swc.models.VPNServer.get('enable') === true) {
                        if ((!swc.models.PortForwarding.validation.helpers.portsNotInRange(portsRange, [500]) ||
                            !swc.models.PortForwarding.validation.helpers.portsNotInRange(portsRange, [4500]))) {
                            validationMessages.push('ports 500 and 4500 are not allowed because these ports reserved for voip');
                        }
                    }
                } else if((!_.isUndefined(vpnModel1) && vpnModel1.get('origin') === 'internal') ||
                    (!_.isUndefined(vpnModel2) && vpnModel2.get('origin') === 'internal')) {
                    if ((!swc.models.PortForwarding.validation.helpers.portsNotInRange(portsRange, [500]) ||
                            !swc.models.PortForwarding.validation.helpers.portsNotInRange(portsRange, [4500]))) {
                            validationMessages.push('ports 500 and 4500 are not allowed because these ports reserved for voip');
                    }
                }

                
                // Check ports crossing conflict:
                _.each(swc.models.PortForwarding.models, function(model) {
                    if (!swc.models.PortForwarding.validation.helpers.
                        portsNotInRange(portsRange, model.get('ports').entry)) {
    
                        // Skip check when editing model
                        if (model.get('id') !== id) {
                            validationMessages.push('already used ports are not allowed');
                        }
                    }
                });
                
                return validationMessages;
            },
            
            validateDestinationPorts: function(id, deviceIP, portsRange) {
                var validationMessages = [],
                    from = +portsRange[0];
    
                if (/\D+/.test(from)) {
                    validationMessages.push('allowed characters 0-9');
                }
    
                if (from < 1 || from > 65535) {
                    validationMessages.push('allowed range 1 to 65535');
                }
    
                // Check ports crossing conflict for current device:
                _.each(swc.models.PortForwarding.where({ 'deviceIP': deviceIP}), function(model) {
                    if (!swc.models.PortForwarding.validation.helpers.
                        portsNotInRange(portsRange, model.get('ports').destination)) {
    
                        // Skip check when editing model
                        if (model.get('id') !== id) {
                            validationMessages.push('already used ports for same device are not allowed');
                        }
                    }
                });
                
                return validationMessages;
            }
        }
    },

    /**
     * List of predefined service which is used to allow user easily add rules using this templates
     *
     * @param predefinedServices {Map}
     */
    predefinedServices: {
        "FTP":    { port: "21",   protocol: "6"  },
        "SSH":    { port: "22",   protocol: "6"  },
        "Telnet": { port: "23",   protocol: "6"  },
        "SMTP":   { port: "25",   protocol: "6"  },
        "TFTP":   { port: "69",   protocol: "17" },
        "HTTP":   { port: "80",   protocol: "6"  },
        "HTTPS":  { port: "443",  protocol: "6"  },
        "POP3":   { port: "110",  protocol: "6"  },
        "SNMP":   { port: "161",  protocol: "17" },
        "PPTP":   { port: "1723", protocol: "6"  },
        "IPSec":  { port: "500, 4500",  protocol: "17" }
    },

    apiToJSON: function(response) {
        var rules = response.status,
            rulesSourcesSkip = [ "cwmp" ], // Disabled rules sources (acs.. etc.)
            rulesParsed = [];

        _.each(rules, function(rule, key) {
            if (_.indexOf(rulesSourcesSkip, rule.Origin) === -1) {
                rulesParsed.push({
                    id: rule.Id,
                    isUPnP: key.indexOf('upnp') !== -1,
                    status: rule.Enable,
                    name: rule.Description,
                    deviceIP: rule.DestinationIPAddress,
                    protocol: rule.Protocol,
                    origin: rule.Origin,
                    sourceInterface: rule.SourceInterface,
                    ports: {
                        entry: rule.ExternalPort.split('-'),
                        destination: rule.InternalPort.split('-')
                    }
                });
            }
        });

        return rulesParsed;
    }

});
;swc.constructors.CentralStorage = Backbone.Collection.extend({

    autoSync: true,

    sync: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        if(self.autoSync === false || swc.Utils.DeviceType !== 'default'){
			deferred.resolve();
			return deferred.promise();
        }

        // Delete deprecated models from collection
        this.reset();

        // Load new models to the collection
        $.when(self.getGlobalStatus(fromListener))
            .done(function() {
                $.when(self.getMediaServerStatus(fromListener), self.getUSBDevices(fromListener))
                    .done(function() {
                        self.trigger('central-storage-update');
                        deferred.resolve();
                    });
            });

        return deferred.promise();
    },

    getGlobalStatus: function(fromListener) {
        var self = this,
            deferred = new $.Deferred(),
            model = new swc.constructors.SimpleModel();

        if(swc.Utils.DeviceType !== 'default') {

            model.set('id', 'status');
            model.set('status', false);

			deferred.resolve();
			return deferred.promise();
        }

        swc.models.Rest.sendRequest({
            url: '/sysbus/APController/PowerManagement:getMode',
            fromListener: fromListener,
            data: {
                "parameters": {}
            },

            success: function(response) {

                // Create new model:
                model.set('id', 'status');

                // Check if data exists:
                if (!response.data) {
                    model.set('status', false);
                } else {
                    model.set('status', response.data.mode === "ON");
                }

                // Add new model to the collection:
                self.add(model);

                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    setGlobalStatus: function(state) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/APController/PowerManagement:setMode',
            data: {
                "parameters": {
                    "mode": state ? "ON" : "OFF"
                }
            },

            success: function() {
                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    getMediaServerStatus: function(fromListener) {
        var model = this.where({ id: 'status' })[0],
            deferred = new $.Deferred();
            
        function error() {
            // Update status model with media streaming server value:
            model.set('mediaServerStatus', false);

            deferred.resolve();
        }

        swc.models.Rest.sendRequest({
            url: '/ws',
            fromListener: fromListener,

            data: {
                "service": "com.swisscom.stargate/ws/dlna.com.swisscom.stargate.dlna",
                "method": "getMediaStreamingState",
                "parameters":{}
            },

            success: function(response) {
                if(response.error) {
                    error();
                } else {
                    var data;
    
                    // Check if AP has successfully started
                    if (!_.isNull(response) && !_.isNull(response.status) && response.status.indexOf('status') !== -1) {
                        data = JSON.parse(response.status).data;
                    } else {
                        data = false;
                    }
    
                    // Update status model with media streaming server value:
                    model.set('mediaServerStatus', data["mediaStreamingEnabled"]);
    
                    deferred.resolve();
                }
            },

            error: function() {
                error();
            }
        });

        return deferred.promise();
    },

    /**
     * Get device icon by its type
     *
     * @param deviceType {String} usb-disk | sd-card
     * @param supported {Boolean} If this device is supported by InternetBox
     * @return {String}
     */
    getIcon: function(deviceType, supported) {
        var icon;

        // if the storage device is not SD-card(mouse, keyboard etc)
        // then it's considered as usb-disk
        if (deviceType === 'sd-card') {
            icon = 'sd-card';
        } else {
            icon = 'usb-disk';
        }

        if (supported !== true) {
            icon += '-undefined';
        }

        return icon;
    },

    /**
     * Set cloud backup state
     *
     * @param responseSyncState
     * @returns {String}
     */
    getSyncState: function(responseSyncState) {
        var map = {
            'IN_PROGRESS': 'progress',
            'UPLOAD_IN_PROGRESS': 'progress',
            'DELETE_IN_PROGRESS': 'progress',
            'COMPLETE': 'complete',
            'ERROR': 'error'
        };

        if (_.isUndefined(map[responseSyncState])) {
            return 'not_sync';
        }

        return map[responseSyncState];
    },

    /**
     * Comparator for collection with USB devices
     * Sorts devices by 'label'
     *
     * @param {Backbone Model}
     * @returns {string}
     */
    comparatorUSBDevices: function(device) {
        return device.get('label').toUpperCase();
    },
    /**
     * Converts json data to usb device model
     * 
     * @param device data
     * @returns {Backbone.Model}
     */
    prepareUSBDeviceModel: function(device){
        var self=this,
            model = new swc.constructors.SimpleModel();

            model.set({
                id: device.id,
                capacity: swc.Utils.getBytesWithUnit(device.capacity * 1024),
                freeSpace: swc.Utils.getBytesWithUnit(device.freeSpace * 1024),
                fsType: device.fsType,
                isSync: device.isSync || false,
                label: device.label,
                syncErrorCode: device.syncErrorCode,
                syncState: self.getSyncState(device.syncState),
                type: self.getIcon(device.type, device.supported)
            });

        return model;

    },
    /**
     * Get list of USB devices
     *
     * @param fromListener {Boolean}
     * @returns {Promise}
     */
    getUSBDevices: function(fromListener) {
        var self = this,
            deferred = new $.Deferred(),
            usbModel = new swc.constructors.SimpleModel(),
            usbDevices = new Backbone.Collection();


        function error() {
            // Add new devices collection to data storage:
            usbModel.set("devices", []);
            usbModel.set('device-loaded', 'none');

            // Add new devices collection to data storage:
            self.add(usbModel);

            deferred.resolve();
        }

        if(swc.Utils.DeviceType !== 'default') {
			error();
			return deferred.promise();
        }
        // set method of sorting for device list
        usbDevices.comparator = this.comparatorUSBDevices;

        // Set model id:
        usbModel.set('id', 'usb-devices');
        usbModel.set('device-loaded', 'in-progress');
        

        swc.models.Rest.sendRequest({
            url: '/ws',
            fromListener: fromListener,

            data: {
                "service": "com.swisscom.stargate/ws/nas/devices.com.swisscom.stargate.nas.devices",
                "method": "list",
                "parameters": {}
            },

            success: function(response) {
                if(response.error) {
                    error();
                } else {
                    var devicesData;
    
                    // Check if AP has successfully started
                    if (response.status === null || response.status.indexOf('status') === -1) {
                        devicesData = { data: [] };
                    } else {
                        try {
                            devicesData = JSON.parse(response.status);
                        } catch(e) {
                            devicesData = { data: [] };
                        }
                    }
    
                    if (devicesData.data.length) {
                        $.each(devicesData.data, function(key, device) {
                            // Add current device to usb devices collection:
                            usbDevices.add(self.prepareUSBDeviceModel(device));
                        });
                    }
    
                    // Add new devices collection to data storage:
                    usbModel.set("devices", usbDevices.toJSON());
                    usbModel.set('device-loaded', 'completed');
    
                    // Add new usb model to the collection:
                    self.add(usbModel);
    
                    deferred.resolve();
                }
            },

            error: function() {
                error();
            }
        });

        return deferred.promise();
    },
    updateDeviceList: function(devices){
        var self = this;
        // Go through each device and create a model for it:
        _.each(devices, function(device) {
            //remove old model 
            for(var i=0;i<self.models.length;i++){
                if(self.models[i].attributes.key===devices[0].Key){
                    self.models[i].destroy();
                }
            }
            // {at: index, merge:true} - splice didnt work in backbone
            self.add(self.prepareUSBDeviceModel(device));
        });
    },
    // FIXME: Create corresponding method in Storage model and call it here
    removeUSBDevice: function(id) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                "service": "com.swisscom.stargate/ws/nas/devices.com.swisscom.stargate.nas.devices",
                "method": "unmount",
                "parameters": {
                    "id": id
                }
            },

            success: function() {
                deferred.resolve();
            },

            error: function () {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    /**
     * Get necessary parameter from one of the storage models:
     * @param parameter {String}
     * @param component {String}
     */
    getParameter: function(parameter, component) {
        var search = this.where({ id: component });

        if (search.length) {
            return search[0].get(parameter);
        } else {
            return '';
        }
    },

    /**
    * Returns true if the Central Storage is ON, false otherwise
    */
    isEnabled: function () {
        var centralStorageInfo = this.get('status');

        return centralStorageInfo ? centralStorageInfo.get('status') : false;
    },

    /**
     * Returns the list of all supported usb devices and sd-cards
     * in dropdown-ready format.
     */
    getDeviceDropdownOptions: function () {
        var usbDevices = this.get('usb-devices'),
            devices = _.isEmpty(usbDevices) ? [] : usbDevices.get('devices'),
            supportedUsbDevices = _.keys(swc.settings.application.get('supportedUsbDevices')),
            options = [];

        // Get all usb disks to the options list
        _.each(devices, function(device) {

            // Don't show "undefined-storage" - unsupported device type
            if ($.inArray(device.type.toLowerCase(), supportedUsbDevices) !== -1) {
                options.push({
                    name: device.label,
                    value: device.id
                });
            }
        });

        if (_.isEmpty(options)) {
            options = [
                {
                    name: getTranslationStrings("No Connected Devices"),
                    value: "no-devices-connected",
                    isDefault: true
                }
            ];
        } else {
            options = _.sortBy(options, function (option) {
                return option.name.toLowerCase();
            });

            options.unshift({
                name: getTranslationStrings("Choose a device"),
                value: "choose-device",
                isDefault: true
            });
        }

        return options;
    },

    /**
     * Helper that returns oldValue if there is  an options element with value===oldValue
     * and returns the id of the first element otherwise.
     * Is used to process getDeviceDropdownOptions method results.
     *
     * @param options {Object}
     * @param dropdownDataValue {String} - value saved in dropdown `data` attribute
     */
    getDefaultDeviceOption: function (options, dropdownDataValue) {
        var defaultOptions = [ "no-devices-connected", "choose-device" ],
            defaultDevice = defaultOptions[0];

        // Set device to previously selected, or show "choose a device" option
        if (options.length >= 2) {
            if (!_.isEmpty(dropdownDataValue) && _.indexOf(dropdownDataValue, defaultOptions) === -1) {
                _.each(options, function (obj) {
                    if (obj.value === dropdownDataValue) {
                        defaultDevice = obj.value;
                    }
                });
            } else {
                defaultDevice = defaultOptions[1];
            }
        }

        return defaultDevice;
    }
});
;swc.constructors.SchedulerDeviceCollection = swc.BaseCollection.extend({

    model: 'SchedulerDevice',

    url: {
        read: '/sysbus/Scheduler:getCompleteSchedules'
    },

    ajaxParameters: {
        parameters: {
            type: 'ToD'
        }
    },

    initialize: function() {
        this.on('add', function(model) {
            model.addPixelScheduleInfo();
        });

        swc.BaseCollection.prototype.initialize.apply(this, arguments);
    },

    /**
     * Comparator to sort schedulers according to sort requirements
     *
     * <pre>
     * Sort requirements:
     *      1. Currently connected devices, even if scheduler is not enabled
     *      2. Disconnected devices with scheduler enabled
     *      3. Disconnected devices with scheduler disabled
     * </pre>
     *
     * @param {Scheduler} model - Instance of scheduler model
     * @return {Number}
     */
    comparator: function(model) {
        return -( (model.get('active') ? 1 : 0) * 2 + (model.get('enable') ? 1 : 0) );
    },

    /**
     * Parse response from the server
     *
     * @param {Object} response
     * @returns {Array}
     *      [{
     *          id: Int          - unique ID for this schedule
     *          override: String - override value for the schedule, if empty no override is present
     *          schedule [
     *              {
     *                  begin: Int,
     *                  end: Int,
     *                  state: Disable
     *              },
     *              ...
     *          ],
     *          enable: Bool    - indicates if this schedule is enabled
     *          active: Bool    - NOT native field, indicates if according device is connected,
     *                            on this stage this value is hardcoded(false). It can be set
     *                            to actual value by SchedulerDeviceManager.applySchedulesToDevices method
     *       }
     *       ...
     *      ]
     */
    apiToJSON: function(response) {
        var parsed = [];

        if (response.status && response.data && response.data.scheduleInfo) {
            _.each(response.data.scheduleInfo, function(scheduleInfo) {
                parsed.push({
                    override: '',
                    schedule: scheduleInfo.schedule || [],
                    enable: scheduleInfo.enable,
                    id: scheduleInfo.ID,
                    active: false
                });
            });
        }

        return parsed;
    },

    /**
     * Save all schedules in collection
     *
     * @returns {*}
     */
    save: function() {
        var deferred = new $.Deferred(),
            scheduleModels = [];

        this.each(function(model) {
            scheduleModels.push(model.sync('update'));
        });

        $.when.apply(this, scheduleModels)
            .done(function() {
                deferred.resolve();
            })
            .fail(function() {
                deferred.reject();
            });

        return deferred.promise();
    }

});;// cloud services list
swc.constructors.CloudAccountCollection = swc.BaseCollection.extend({

    model: 'CloudAccount',

    url: {
        read: "/ws"
    },

    ajaxSettings: {
        requestDataType: 'json',
        contentType: 'application/x-sah-ws-4-call+json',
        requestType: {
            read: 'post'
        }
    },

    ajaxParameters:  JSON.stringify({
        method: 'getCloudSyncInfo',
        service: 'com.swisscom.stargate/ws/cloud/backup.com.swisscom.stargate.cloud.backup',
        parameters: {}
    }),

    sync: function(method, options) {
		if(swc.Utils.DeviceType === 'default') {
			this.sync = (new swc.BaseCollection()).sync;
			return this.sync(method, options);
		}
    },

    validation: {

        usbUID: function(data) {
            var validationData = swc.Utils.getDataToValidate({
                    usbUID: { elementName: 'usbUID' }
                }, data),
                validationMessages = [];

            if (validationData.usbUID === "no-devices-connected") {
                validationMessages.push('no connected devices');
            }

            if (validationData.usbUID === "choose-device") {
                validationMessages.push('no device selected');
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        folderName: function(data) {
            var validationData = swc.Utils.getDataToValidate({
                    usbUID: { elementName: 'usbUID' },
                    folderName: { elementName: 'folderName' }
                }, data),
                validationMessages = [],
                deferred;

            // Don't show error about folder name if device was not selected yet
            if (validationData.usbUID === "choose-device") {
                return {
                    status: true,
                    messages: validationMessages
                };
            }

            validationData.folderName = $.trim(validationData.folderName);

            if (!validationData.folderName.length) {
                validationMessages.push('can not be empty');

                return {
                    status: false,
                    messages: validationMessages
                };
            }

            deferred = new $.Deferred();

            $.when(swc.models.CloudServices.validateFolder(validationData.usbUID, validationData.folderName))
                .done(function () {
                    deferred.resolve();
                })
                .fail(function (status) {
                    if (!_.isUndefined(status)) {
                        switch (status.failure) {
                            case "SYNC_FOLDER_ALREADY_EXISTS":
                                validationMessages.push('already exists');
                                break;

                            case "SYNC_FOLDER_INVALID":
                            case "BAD_REQUEST":
                            case "INTERNAL_ERROR":
                                validationMessages.push('unable to create folder');
                                break;

                            default:
                                validationMessages.push('unable to create folder');
                                break;
                        }
                    } else {
                        validationMessages.push('unable to create folder');
                    }

                    deferred.reject({
                            status: false,
                            messages: validationMessages
                        });
                });

            return deferred.promise();
        }
    },

    /**
     * Get login url to cloud service from <default> cloud service
     * @param cloudServiceID {String}
     * @returns {*}
     */
    getLoginURL: function(cloudServiceID) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',

            data: {
                service: "com.swisscom.stargate/ws/cloud/auth.com.swisscom.stargate.cloud.authorization",
                method: "getServiceAuthUrl",
                parameters: {
                    data: JSON.stringify({
                            serviceId: cloudServiceID,
                            domain: location.host
                        })
                }
            },

            success: function(response) {
                var parsedResponse = JSON.parse(response.status),
                    url;
                if (parsedResponse.data && parsedResponse.data.authUrl) {
                    url = parsedResponse.data.authUrl;
                    deferred.resolve(url);
                } else {
                    deferred.reject();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    /**
     * Create folder on NAS devices, while connecting to cloud account
     * @param partitionUID {String} - Usb device id
     * @param folderName {String} - user defined folder
     */
    createFolder: function(partitionUID, folderName) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',

            // There is some specific in AP backend for this call
            data: {
                service: "com.swisscom.stargate/ws/cloud/backup.com.swisscom.stargate.cloud.backup",
                method: "syncFSResources",
                parameters: {
                    resources: [
                        JSON.stringify({
                            deviceId: partitionUID,
                            content: [
                                JSON.stringify({
                                    path: folderName,
                                    action: 'CREATE'
                                })
                            ]
                        })
                    ]
                }
            },

            success: function(response) {
                var responseParsed = JSON.parse(response.status),
                    status = responseParsed.status,
                    error = responseParsed.error || "";

                switch (status) {
                    case 409:
                        error = "already exists";
                        break;
                    case 422:
                        error = "invalid name";
                        break;
                    default:
                        error = "";
                        break;
                }

                if (_.isEmpty(error)) {
                    deferred.resolve();
                } else {
                    deferred.reject({ message: error });
                }
            }
        });

        return deferred.promise();

    },

    /**
     * Movee folder on NAS devices, while connecting to cloud account
     * @param partitionUID {String} - Usb device id
     * @param folderName {String} - user defined folder
     */
    moveFolder: function(model, partitionUID, folderName) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',

            // There is some specific in AP backend for this call
            data: {
                service: "com.swisscom.stargate/ws/cloud/backup.com.swisscom.stargate.cloud.backup",
                method: "moveSyncFolder",
                parameters: {
                    data: JSON.stringify({
                        oldFolder: {
                            partitionUUID: model.get('deviceId'),
                            folderName: model.get('folderName')
                        },
                        newFolder: {
                            partitionUUID: partitionUID,
                            folderName: folderName
                        }
                    })
                }
            },

            success: function(response) {
                var responseParsed = JSON.parse(response.status),
                    status = responseParsed.status,
                    error = responseParsed.error || "";

                switch (status) {
                    case 409:
                        error = "already exists";
                        break;
                    case 422:
                        error = "invalid name";
                        break;
                    default:
                        error = "";
                        break;
                }

                if (_.isEmpty(error)) {
                    deferred.resolve();
                } else {
                    deferred.reject({ message: error });
                }
            }
        });

        return deferred.promise();

    },

    apiToJSON: function(response) {
        var respStatus = {},
            accounts = [],
            result = [];

        try {
            // try JSON.parse
            respStatus = JSON.parse(response.status);
            if (respStatus && respStatus.data) {
                accounts = respStatus.data.syncServices; // TODO: old API is used temporarily
            }
        } catch (e) {
            // Do nothing since accounts already initialized
        }

        _.each(accounts, function(cloudAccount) {
            var deviceId = '',
                deviceLabel = '[Unknown Device]';

            if (cloudAccount.authStatusCode !== 503) {
                if (cloudAccount.partitionUUID) {
                    deviceLabel = cloudAccount.partitionLabel;
                    deviceId = cloudAccount.partitionUUID;
                }

                result.push({
                    id: cloudAccount.id,
                    cloudServiceName: getTranslationStrings(cloudAccount.id === 'dropbox' ? 'Dropbox' : 'Google drive'),
                    deviceLabel: deviceLabel,
                    deviceId: deviceId,
                    accountName: cloudAccount.accountName,
                    status: cloudAccount.syncState ? cloudAccount.syncState : 'DISABLED',
                    error: cloudAccount.error || '',
                    inaccessibleFiles: cloudAccount.inaccessibleFiles,
                    // they return space in megabytes, but we operate with Kilobytes, thus have to multiply by 1024
                    totalSpace: cloudAccount.totalSpace * 1024,
                    usedSpace: cloudAccount.usedSpace * 1024,
                    lastSyncDate: cloudAccount.lastSyncDate,
                    folderName: cloudAccount.localFolderPath
                });
            }
        });

        return result;
    },

    /**
     * Handle oauth callback.
     * This method called when AP requested OAuthCallback handler after first phase of cloud login procedure
     *
     * @param oauthCallbackParams {Object|Map} list of parameters send to WebUI by AP during first step of sign-in phase
     */
    handleOauthCallback: function (oauthCallbackParams) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',

            timeout: 60000,

            data: {
                service: "com.swisscom.stargate/ws/cloud/auth.com.swisscom.stargate.cloud.authorization",
                method: "handleAuthCallback",
                parameters: { authParams: oauthCallbackParams }
            },

            success: function (response) {
                var parsedResponse = JSON.parse(response.status);
                if (parsedResponse.status !== 200) {
                    deferred.reject();
                } else {
                    deferred.resolve(parsedResponse);
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    }
});
;swc.constructors.RemoteNASAccounts = swc.BaseCollection.extend({

    model: 'RemoteNASAccount',

    url: {
        read: '/ws'
    },

    ajaxParameters: JSON.stringify({
        service: 'com.vestiacom.rnas/com/vestiacom/rnas.com.vestiacom.rnas',
        method: 'GetPairing',
        parameters: {}
    }),
    
    isFetching: false,
    ajaxSettings: {
        requestType: 'POST',
        requestDataType: 'json',
        contentType: 'application/x-sah-ws-4-call+json',
        idle:true
    },
    /**
     * @example Success Response:
     *
     * { "status": "SUCCESS", "accounts": [{ "name": "n1", "email": "e1" }, { "name": "n2", "email": "e2" }] }
     *
     * @example Fail Response:
     *
     * {"status":"ERROR"}
     */
    apiToJSON: function(response) {
        var accountsParsed = [],
            responseParsed = {};

        this.error = false;

        try {
            responseParsed = JSON.parse(response.status);

            if (!_.isEmpty(responseParsed) && responseParsed.status === "SUCCESS") {
                var rnasAccounts = responseParsed.accounts;

                _.each(rnasAccounts, function(account) {
                    accountsParsed.push({
                        id: account.email,
                        name: account.name,
                        email: account.email
                    });
                });
            }

            if (responseParsed.status !== "SUCCESS") {
                this.error = true;
            }
        } catch (e) {
            // do nothing because everything has default values already
        }

        return accountsParsed;
    },

    isError: function () {
        return this.error === true;
    }
    /* in case we get error while long list is refreshing
    hasChanged: function() {
        if(this.isFetching) {
            console.log('fetching');
            return false;
        } else {
            return swc.constructors.RemoteNASAccounts.__super__.hasChanged.apply(this);
        }
    }*/

});
;swc.constructors.StorageDevices = Backbone.Collection.extend({

    autoSync: true,
    initialize: function() {
        this.temporaryDevices = [];
        this.isRefreshable = true;
    },

    sync: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        if(self.autoSync===false){
         return deferred.resolve();
        }

        // Reset collection before sync process:
        this.reset();

        $.when(self.getDevices(fromListener))
            .done(function () {
                deferred.resolve();
            })
            .fail(function () {
                deferred.reject();
            });

        return deferred.promise();
    },

    /**
     * Comparator for collection with USB devices
     * Sorts devices by 'label'
     *
     * @param {Backbone Model}
     * @returns {string}
     */
    comparator: function(device) {
        return device.get('label').toUpperCase();
    },

    /**
     * Get connected storage devices and internal storage data
     *
     * Devices response:
     * [
     *    {
     *       id : "4C532000021016111590", // Static ID of storage device: uniquely identifies the device.
     *       type: "USB"                  // Either "SD" or "USB"
     *       label: "Transcend 32Gb",     // User-friendly name of the device
     *       fsType: "NTFS",              // Detected file system type e.g. NTFS, FAT32, etc.
     *       capacity: 4194304,           // Capacity in KiB
     *       freeSpace: 2046122,          // Free space in KiB
     *       isSync : true,               // Is synchronized - either true or false.
     *       syncState : "COMPLETE",      // One of: "COMPLETE", "ERROR", "IN_PROGRESS", null
     *       syncError : 7                // Error ID, that can be translated to the meaningful message (either number, or null).
     *    },
     *    ...
     * ]
     *
     * @localParam storage - List of devices with the following attributes
     *
     *  {
     *      device-loaded : 'completed'
     *      devices : Array of devices params
     *  }
     *
     * @localParam fileManager - List of files for each device
     * @returns {deferred.then()}
     */
    getDevices: function(fromListener) {
        var self = this,
            deferred = new $.Deferred(),
            storage = new swc.constructors.CentralStorage();

        $.when(storage.getUSBDevices(fromListener))
            .done(function() {
                var deviceList = storage.models[0].get('devices'),
                    devicesCount = deviceList.length, // Number of devices
                    devicesFilled = 0;                // number of devices with successful files request

                // If no devices in storage collection, it's not possible to retrieve files for them
                if (deviceList.length === 0) {
                    return deferred.resolve();
                }

                _.each(deviceList, function(deviceParams) {
                    var device = new Backbone.Model();

                    // Set device params
                    device.set(deviceParams);

                    // request for files for every separate device.
                    //fileManager = new swc.constructors.FileManager([], { deviceID: device.get('id') });

              //      $.when(fileManager.sync('read', {isListener: fromListener}))
               //         .done(function() {
                            // Add files params
                            //device.set('files', fileManager);

                            // Check if all files on the device are synced:
                            // If device have not any content, we take it's sync status from syncState
                            // progress - sync is turned off
                            // complete - sync is turned on
                            //if (fileManager.length === 0) {
                            //   device.set('isSync', device.get('syncState') === 'complete');
                            //} else {
                            //   device.set('isSync', !fileManager.findWhere({ isSync: false }));
                            //}

                            self.temporaryDevices.push(device);

                            devicesFilled++;

                            if (devicesFilled === devicesCount) {
                                if (self.isRefreshable) {
                                    self.reset(self.temporaryDevices);
                                }

                                self.temporaryDevices = [];
                                deferred.resolve();
                            }
                    //    });
                });
            })
            .fail(function() {
                deferred.reject();
            });

        return deferred.promise();
    },

    /**
     * Sends changed sync statuses of files and devices to server
     *
     * @returns deferred
     */
    setSyncStatus: function() {
        var sendArray = [],
            deferred = new $.Deferred();

        /**
         * make the right json structure for sending - all arrays should contain stringified objects
         */
        this.each(function(model){
            var deviceObj = {},
                supportedUsbDevices = _.keys(swc.settings.application.get('supportedUsbDevices'));

            deviceObj.deviceId = model.get('id');
            deviceObj.sync = model.get('isSync');

            // Fill the files
            deviceObj.content = [];
            model.get('files').each(function(model){
                var fileObj = {};
                fileObj.path = '/'+model.get('name');
                fileObj.sync = model.get('isSync');
                fileObj.virtual = model.get('virtual');
                deviceObj.content.push(JSON.stringify(fileObj));
            });

            if ($.inArray(model.get('type').toLowerCase(), supportedUsbDevices) !== -1) {
                sendArray.push(JSON.stringify(deviceObj));
            }
        });

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                "service":"com.swisscom.stargate/ws/cloud/backup.com.swisscom.stargate.cloud.backup",
                "method":"syncFSResources",
                "parameters":{
                    "resources": sendArray
                }
            },

            success: function() {
                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    unmountDevice: function(deviceId) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                "service":"com.swisscom.stargate/ws/nas/devices.com.swisscom.stargate.nas.devices",
                "method":"unmount",
                "parameters":{
                    "id": deviceId
                }
            },

            success: function() {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    renameDevice: function(deviceId) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                "service":"com.swisscom.stargate/ws/nas/devices.com.swisscom.stargate.nas.devices",
                "method":"rename",
                "parameters":{
                    "id": deviceId,
                    "label": this.get(deviceId).get('label')
                }
            },

            success: function() {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    renameDevices: function() {
        var deferred = new $.Deferred(),
            self = this,
            requestsArray= [];

        self.each(function(model){
            requestsArray.push(self.renameDevice(model.id));
        });

        $.when.apply(this, requestsArray).done(function(){
            deferred.resolve();
        });

        return deferred.promise();
    }

});
;swc.constructors.DeviceMeasurements = Backbone.Collection.extend({

    autoSync: true,

    getRequest: function(params, fromListener) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NeMo/Intf/data:getMIBs',
            data: {
                "parameters": params
            },
            fromListener: fromListener,

            success: function(response) {
                deferred.resolve(response);
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    sync: function(fromListener) {
        var deferred = new $.Deferred(),
            self = this;

        if(self.autoSync===false){
         return deferred.resolve();
        }

        $.when(this.getRequest({mibs: "dsl"}, fromListener), this.getRequest({mibs: "sfp"}, fromListener))
            .done(function(dsl, sfp){
                self.build(dsl, sfp);
                deferred.resolve();
            });

        return deferred.promise();
    },

    build: function(dsl, sfp) {
        var dslObj,
            sfpObj,
            self = this;

        if (dsl && dsl['status'] && dsl['status']['dsl']){
            _.each(dsl['status']['dsl'], function(obj, key){
                if (_.isEmpty(obj)) {
                    return;
                }
                dslObj = obj;
                dslObj.__key__ = key;
                dslObj.id = "dsl";
                self.add(dslObj);
            });
        }

        if (sfp && sfp['status'] && sfp['status']['sfp']){
            _.each(sfp['status']['sfp'], function(obj, key){
                if (_.isEmpty(obj)) {
                    return;
                }

                sfpObj = obj;
                sfpObj.__key__ = key;
                sfpObj.id = "sfp";
                self.add(sfpObj);
            });
        }
    },

    getParams: function() {
        var dsl = this.get("dsl"),
            sfp = this.get("sfp");

        if (dsl){
            return {dsl: dsl.toJSON()};
        } else if (sfp){
            return {sfp: sfp.toJSON()};
        }

        return {};
    }
});
;swc.constructors.CallCollection = swc.BaseCollection.extend({

    model: 'Call',

    url: {
        read: "/sysbus/VoiceService/VoiceApplication:getCallList"
    },

    apiToJSON: function(response) {
        var parsed = [],
            restClient = new swc.constructors.Rest(),
            phonebook = [],
            self = this;

        restClient.sendRequest({
            url: '/sysbus/Phonebook:getAllContacts',
            method: 'POST',
            contentType: 'application/x-sah-ws-4-call+json',
            async: false,
            data: {
                "parameters": {}
            },
            success: function(data) {
                phonebook = data;
            }
        });

        _.each(response.status, function(contact) {

            _.each(phonebook.status, function(item){
               if(!_.isUndefined(_.findWhere(item.telephoneNumbers, {name: contact.remoteNumber}))) {
                contact.name = item.formattedName;
               }
            });

            parsed.push({
                'id': contact.callId,
                'name': contact.name || contact.remoteName || 'Unknown',
                'number': contact.remoteNumber,
                'type': self.mapCallType(contact),
                'direction': contact.callOrigin,
                'duration': contact.duration,
                'startTime': contact.startTime,
                'viewed': contact.viewed,
                'inContactList': !!contact.name,
                'deleted': false
            });
        });

        return parsed;
    },

    mapCallType: function(item) {
        if(item.callType === 'missed')  { return 'missed'; }
        if(item.callOrigin === 'local') { return 'outgoing'; }
        if(item.callOrigin !== 'local') { return 'incoming'; }
    },

    removeCalls: function(id) {
        var params = id? {callId: id} : {},
            deferred = new $.Deferred();


        swc.models.Rest.sendRequest({
            url: '/sysbus/VoiceService/VoiceApplication:clearCallList',
            method: 'POST',
            contentType: 'application/x-sah-ws-4-call+json',
            data: {
                "parameters": params
            },
            success: function() {
                deferred.resolve();
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    }
});;swc.constructors.Phonebook = swc.BaseCollection.extend({

    model: 'PhonebookContact',

    url: {
        read: "/sysbus/Phonebook:getAllContacts"
    },

    /**
     * Name of request parameter ID which is stored in the localStorage
     */
    requestParamNameID: 'telephony:phonebook:edit-contactID',

    validation: {
        checkAllowableName: function(data, validateKey) {
            var validationMessages = [],
                map = {},
                validationData;

            map[validateKey] = { elementName: validateKey };
            validationData = swc.Utils.getDataToValidate(map, data);

            // Ascii is defined as the characters in the range of 000-177 (octal), therefore non-printing characters \000-\037
            if (/[\000-\037]+/.test(validationData[validateKey])) {
                validationMessages.push('allowed characters ASCI non-printable');
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        checkEmpty: function(data, validateKey) {
            var validationMessages = [],
                map = {},
                validationData;

            map[validateKey] = { elementName: validateKey };
            validationData = swc.Utils.getDataToValidate(map, data);

            if (_.isEmpty(validationData[validateKey])) {
                validationMessages.push('must not be empty');
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        checkAllowableNumber: function(data, validateKey) {
            var validationMessages = [],
                map = {},
                validationData;

            map[validateKey] = { elementName: validateKey };
            validationData = swc.Utils.getDataToValidate(map, data);

            if (!/^[\+]?[\d\b#*]*$/.test(validationData[validateKey])) {
                validationMessages.push('phone number contain not allowed characters');
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        firstName:  function(data) {
            return this.checkAllowableName(data, 'firstName');
        },
        lastName:   function(data) {
            return this.checkAllowableName(data, 'lastName');
        },
        homeNumber: function(data) {
            return this.checkAllowableNumber(data, 'homeNumber');
        },
        cellNumber: function(data) {
            return this.checkAllowableNumber(data, 'cellNumber');
        },
        workNumber: function(data) {
            return this.checkAllowableNumber(data, 'workNumber');
        }
    },

    // comparator: function(model) {
        // return $.trim(model.get('formattedName')).toLowerCase();
    // },

    /**
     * Get phonebook
     * @returns json
     * {
     *  "status":[
     *      {
     *          "key":"5",
     *          "lastRevision":"1970-01-01T01:54:48Z",
     *          "uniqueID":"urn:uuid:153a6f06-bd93-4b2b-97e9-9b2da9d27745",
     *          "name":"N:Gogo;Pogo;",
     *          "formattedName":"Gogo Pogo",
     *          "birthDay":"0001-01-01T00:00:00Z",
     *          "title":"",
     *          "organisation":"",
     *          "url":"",
     *          "ringtone":"",
     *          "tags":"",
     *          "addresses":[],
     *          "labels":[],
     *          "telephoneNumbers":[
     *              {
     *                  "key":"1",
     *                  "name":"67890",
     *                  "type":"WORK",
     *                  "preferred":false
     *              },{
     *                  "key":"2",
     *                  "name":"12345",
     *                  "type":"CELL",
     *                  "preferred":false
     *              },{
     *                  "key":"3",
     *                  "name":"12345",
     *                  "type":"HOME",
     *                  "preferred":false
     *              }
     *          ],
     *          "emails":[],
     *          "photos":[]
     *      }
     *  ]
     *}
     */
    apiToJSON: function(response) {
        var telephoneNumbers = [],
            names = [],
            parsed = [],
            numberOfSlashes = 0,
            firstName = "";

        _.each(response.status, function(contact) {
            // sort telephone numbers
            telephoneNumbers = [
                _.where(contact.telephoneNumbers, {'type': 'HOME'})[0],
                _.where(contact.telephoneNumbers, {'type': 'CELL'})[0],
                _.where(contact.telephoneNumbers, {'type': 'WORK'})[0]
            ];

            names = [];

            var nameRaw = contact.name.slice(2);

            for(var i = 0; i < nameRaw.length; i++) {
                if(nameRaw[i] === ';' && numberOfSlashes % 2 === 0) {
                    names.push(nameRaw.slice(0,i).replace(/\\\\/g, '\\').replace(/\\;/g, ';'));
                    // names.push(nameRaw.slice(i+1).replace(/\\\\/g, '\\').replace(/\\;/g, ';'));
                    firstName = nameRaw.slice(i+1);
                    break;
                }

                if(nameRaw[i] === '\\') {
                    numberOfSlashes++;
                } else {
                    numberOfSlashes = 0;
                }
            }

            numberOfSlashes = 0;

            for(i = 0; i < firstName.length; i++) {
                if(firstName[i] === ';' && numberOfSlashes % 2 === 0) {
                    names.push(firstName.slice(0,firstName.length-1).replace(/\\\\/g, '\\').replace(/\\;/g, ';'));
                    break;
                }

                if(i === firstName.length - 1) {
                    names.push(firstName.replace(/\\\\/g, '\\').replace(/\\;/g, ';'));
                }

                if(firstName[i] === '\\') {
                    numberOfSlashes++;
                } else {
                    numberOfSlashes = 0;
                }
            }

            telephoneNumbers = _.compact(telephoneNumbers);

            parsed.push({
                'formattedName': contact.formattedName,
                'name': contact.name,
                'firstName': names[1] || '',
                'lastName': names[0] || '',
                'telephoneNumbers': telephoneNumbers,
                'id': contact.uniqueID,
                'modified': false,
                'deleted': false
            });
        });

        return parsed;
    },

    deleteAllContacts: function() {
        var deletedContacts = this.where({deleted: true}),
            deferred = new $.Deferred(),
            modificationArray = [];

        /**
         * TODO: create deletion in general way via changedModels({ onlyDeleted: true  })
         * and model.sync('delete') methods (see save() method in the
         * swc.constructors.StorageSettingsDatasharingView as example)
         */
        _.each(deletedContacts, function(model){
            modificationArray.push(model.destroy());
        });

        $.when.apply(this, modificationArray)
            .done(function() {
                deferred.resolve();
            }).
            fail(function() {
                deferred.reject();
            });

        return deferred.promise();
    }

});
;swc.constructors.Wireless = Backbone.Collection.extend({

    // SA-3612
    // We have to enter string values due to invalid comparison algorithm in js/modules/core/template.html:60 - selectedOption.value === key
    // if we try to comparison a string and a digit without conversion or using a trick +"" we are not able to comparison two values of different type.
    static5GChannelList: ["36","40","44","48","52","56","60","64","100","104","108","112"],
    autoSync: false,
    validation: {

        ssid: function(data) {
            var ssid,
                validationStatus = true,
                validationMessages = [];

            // Get neccessary params from array of page data:
            $.each(data, function(key, value) {
                if (value.parameterName === "ssid") {
                    ssid = value.parameterValue;
                }
            });

            if (!ssid.length) {
                validationStatus = false;
                validationMessages.push('name must not be empty');
            }

            if (ssid.length < 3 || ssid.length > 32) {
                validationStatus = false;
                validationMessages.push('name must be from 3 to 32 symbols');
            }

            if (!/^[ -~]+$/.test(ssid)) {
                validationStatus = false;
                validationMessages.push('name must not contain disabled characters');
            }

            if (/[\"\$\+\?\[\\\]]+/.test(ssid)) {
                validationStatus = false;
                validationMessages.push('name must not contain disabled characters');
            }

            if (ssid.replace(/\s/g, '').length === 0) {
                validationStatus = false;
                validationMessages.push('name must not contain only spaces');
            }

            if (validationStatus === true) {
                $(".password-original").trigger("change");
            }

            return {
                status: validationStatus,
                messages: validationMessages
            };
        },

        password: function(data) {
            var password = '',
                encryption = '',
                credentials = swc.models.Wireless.getParameter('credentials', "2.4GHz"),
                userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin(),
                validationStatus = true,
                validationMessages = [];

            $.each(data, function(key, value) {
                if (value.parameterName === "password") {
                    password = value.parameterValue;
                }

                if (value.parameterName === "encryption") {
                    encryption = value.parameterValue;
                }
            });

            // Skip check for superadmin and if encryption is none
            if (encryption === "None" || userIsSuperAdmin) {
                return {
                    status: validationStatus,
                    messages: validationMessages
                };
            }

            if ( !/^.{8,63}$/.test(password) ) {
                validationStatus = false;
                validationMessages.push('Password WPA length error');
            }

            /**
             * FIXME: We should use swc.Utils.validatePassword() method, no need to be too specific in error message
             */
            if ( !/^[ -~]+$/.test(password) ) {
                validationStatus = false;
                validationMessages.push('password contain not allowed characters');
            }

            if (typeof(credentials['2_4']) !== "undefined" && typeof(credentials['5_0']) !== "undefined" && ((
                    this.validationContext === '2_4' && $(".ssid").val() === credentials['5_0'].ssid && password !== credentials['5_0'].password) || (
                    this.validationContext === '5_0' && credentials['2_4'].ssid === $(".ssid").val() && password !== credentials['2_4'].password
                        ))) {
                validationStatus = false;
                validationMessages.push('password for WLAN 2.4GHz and WLAN 5GHz must be the same if ssids are the same');
            }

            return {
                status: validationStatus,
                messages: validationMessages
            };
        },
        setValidationContext: function(validationContext) {
            this.validationContext = validationContext;
        }
    },
    sync: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        // Delete deprecated models from collection
        this.reset();

        // Load new models to the collection
        $.when(
                self.getSystemTime(fromListener),
                self.getGlobalStatus(fromListener),
                self.getGuestStatus(fromListener),
                self.getGuestActivationTimer(fromListener),
                self.loadSettings(fromListener),
                self.loadSettingsGuest(fromListener)
            ).done(function() {
                self.trigger('change');
                deferred.resolve();
            });

        return deferred.promise();
    },

    getSystemTime: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Time:getTime',
            fromListener: fromListener,

            data: {
                "parameters": {}
            },

            success: function(response) {
                var model = new swc.constructors.SimpleModel(),
                    date= new Date();

                model.set('accessPoint', 'systemTime');
                if(!_.isEmpty(response.data)){
                    model.set('time', response.data.time.split('GMT')[0]); // DATE is coming from server with TimeZone.
                }
                else{
                    model.set('time', date.toUTCString().split('GMT')[0]);
                }
                self.add(model);

                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },
    // set 2.4GHz attributes from event
    set24State: function(event){
        var interface24=this.get("2.4GHz");

        if(event.Active===false){
            this.remove(interface24);
        }
        else{
            if(!_.isEmpty(interface24)){
                interface24.name=event.SSID;
            }
        }
        this.trigger('change');
    },
    // set 5GHz attributes from event
    set5State: function(event){
        var interface5=this.get("5GHz");

        if(event.Active===false){
            this.remove(interface5);
        }
        else{
            if(!_.isEmpty(interface5)){
                interface5.name=event.SSID;
            }
        }
        this.trigger('change');
    },
    // set Guest attributes from event 
    setGuestState: function(event){
        var interface5=this.get("Guest-2.4GHz");
        if(event.Active===false){
            this.remove(interface5);
        }
        else{
            if(!_.isEmpty(interface5)){
                interface5.name=event.SSID;
            }
        }
        this.trigger('change');
    },
    getGlobalStatus: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC/Wifi:get',
            fromListener: fromListener,

            data: {
                "parameters": {}
            },

            success: function(response) {
                var model = new swc.constructors.SimpleModel(),
                    status,
                    mode;

                model.set('accessPoint', 'status');

                // Define responce existence
                if (!response.data) {
                    return deferred.resolve();
                }

                // Define configurationg mode existence
                mode = response.data.ConfigurationMode;

                // Global status - wifi is on:
                if (response.data.Enable && response.data.Status) {
                    status = "on";
                }

                // Global status - wifi is off:
                if (!response.data.Enable && !response.data.Status) {
                    status = "off";
                }

                // Global status - wifi is off by scheduler:
                if (response.data.Enable && !response.data.Status) {
                    status = "off-sheduler";
                }

                // Global status - wifi is on till reboot of device:
                if (!response.data.Enable && response.data.Status) {
                    status = "on";
                }

                model.set('status', status);
                model.set('configurationMode', mode);
                model.set('WPSEnable', response.data.WPSEnable);

                self.add(model);

                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    getGuestStatus: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC/Guest:get',
            fromListener: fromListener,

            data: {
                "parameters": {}
            },

            success: function(response) {
                var model = new swc.constructors.SimpleModel(),
                    status = false;

                if (response.data) {
                    status = response.data.Status === "Enabled";
                }

                model.set('accessPoint', 'guestStatus');
                model.set('status', status);

                self.add(model);

                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    getGuestActivationTimer: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC/WlanTimer:getActivationTimer',
            fromListener: fromListener,

            data: {
                "parameters": {
                    "InterfaceName": "guest"
                }
            },

            success: function(response) {
                var model = new swc.constructors.SimpleModel();

                model.set('accessPoint', 'guestTimer');
                model.set('timeout', response.status);

                // Add model to the collection
                self.add(model);

                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    loadSettings: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NeMo/Intf/lan:getMIBs',
            fromListener: fromListener,

            data: {
                "parameters": {
                    "mibs": "wlanradio || wlanvap || penable",
                    "flag": "",
                    "traverse": "down"
                }
            },

            success: function(response) {
                if (response.status) {
                    if (response.status.wlanradio && response.status.wlanvap && response.status.penable) {

                        // Create models from response
                        self.createModels(response);

                        deferred.resolve();
                    } else {
                        deferred.reject();
                    }
                } else {
                    deferred.reject();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    loadSettingsGuest: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NeMo/Intf/guest:getMIBs',
            fromListener: fromListener,

            data: {
                "parameters": {
                    "mibs": "wlanradio || wlanvap || penable",
                    "flag": "",
                    "traverse": "down"
                }
            },

            success: function(response) {
                if (response.status) {
                    if (response.status.wlanradio && response.status.wlanvap && response.status.penable) {

                        // Create models from response
                        self.createModels(response, true);

                        deferred.resolve();
                    } else {
                        deferred.reject();
                    }
                } else {
                    deferred.reject();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    setGlobalStatus: function(status) {
        var deferred = new $.Deferred(),
            configurationMode = swc.models.Wireless.getParameter("configurationMode", "status");

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC/Wifi:set',
            data: {
                "parameters": {
                    "Enable": status,
                    "ConfigurationMode": configurationMode,
                    "Status": status
                }
            },

            success: function() {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    setGuestStatus: function(status) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC/Guest:set',
            data: {
                "parameters": {
                    "Enable": status
                }
            },

            success: function() {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    setGlobalConfigurationMode: function(configurationMode) {
        var deferred = new $.Deferred(),
            status = swc.models.Wireless.getParameter("status", "status");

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC/Wifi:set',
            data: {
                "parameters": {
                    "Enable": status === 'on',
                    "ConfigurationMode": configurationMode,
                    "Status": status === 'on'
                }
            },

            success: function() {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    setGuestActivationTimer: function(timeout) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC/WlanTimer:setActivationTimer',
            data: {
                "parameters": {
                    "Timeout": timeout,
                    "InterfaceName": "guest"
                }
            },

            success: function() {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    setWPSState: function(state) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC/Wifi:setWPSEnable',
            data: {
                "parameters": {
                    "enable": state
                }
            },

            success: function() {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();

    },

    setVAPStatus: function(accessPoint, data) {
        var deferred = new $.Deferred(),
            status = '',
            wlDriver = this.getParameter("driver", accessPoint),
            saveData = {
                "parameters": {
                    "mibs": {
                        "wlanradio":{}
                    }
                }
            };

        // Get neccessary params from array of page data:
        $.each(data, function(key, value) {
            if (value.parameterName === "status") {
                status = value.parameterValue;
            }
        });

        // SSID Availability selection
        saveData.parameters.mibs.wlanradio[wlDriver] = {
            "PersistentEnable": status
        };

        swc.models.Rest.sendRequest({
            url: '/sysbus/NeMo/Intf/lan:setWLANConfig',
            data: saveData,

            success: function() {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    startWPSPairing: function() {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC/Wifi:startPairing',
            data: {
                "parameters": {}
            },

            success: function() {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    stopWPSPairing: function() {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC/Wifi:stopPairing',
            data: {
                "parameters": {}
            },

            success: function() {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    saveSettings: function(accessPoint, data) {
        var ssid = '', password = '', encryption = '',
            visibility = '', channel = '', channelBandwidth = '',
            operating = this.getParameter('operating', accessPoint),
            wlID = this.getParameter("accessPointID", accessPoint),
            wlDriver = this.getParameter("driver", accessPoint),
            saveData = {
                "parameters": {
                    "mibs": {
                        "wlanvap": {},
                        "wlanradio": {},
                        "penable": {}
                    }
                }
            },
            deferred = new $.Deferred();

        // Get neccessary params from array of page data:
        $.each(data, function(key, value) {
            if (value.parameterName === "ssid") {
                ssid = value.parameterValue;
            }

            if (value.parameterName === "password") {
                password = value.parameterValue;
            }

            if (value.parameterName === "encryption") {
                encryption = value.parameterValue;
            }

            if (value.parameterName === "channel") {
                channel = value.parameterValue;
            }

            if (value.parameterName === "visibility") {
                visibility = value.parameterValue;
            }

            if (value.parameterName === "channel-bandwidth") {
                channelBandwidth = value.parameterValue;
            }
        });

        var userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin();
        var security = {
                "KeyPassPhrase": password,
                "WEPKey": this.getParameter("wepkey", accessPoint),
                "ModeEnabled": encryption
        };

        if(userIsSuperAdmin === true) {
            security = {
                "WEPKey": this.getParameter("wepkey", accessPoint),
                "ModeEnabled": encryption
            };
        }

        // Main data selection
        saveData.parameters.mibs.wlanvap[wlID] = {
            "SSID": ssid,
            "SSIDAdvertisementEnabled": visibility === "visible" ? true : false,
            "Security": security
        };

        // Hack for SAH not saving password if security set to No Password
        if (encryption === "None") {
            saveData.parameters.mibs.wlanvap[wlID].Security.WEPKey = this.getParameter('wepkey', accessPoint);
            saveData.parameters.mibs.wlanvap[wlID].Security.KeyPassPhrase = this.getParameter('password', accessPoint);
        }

        // Channel and Operating mode selections options
        if (channel === "auto") {
            saveData.parameters.mibs.wlanradio[wlDriver] = {
                "AutoChannelEnable": true,
                "OperatingStandards": operating,
                "OperatingChannelBandwidth": channelBandwidth
            };
        } else {
            saveData.parameters.mibs.wlanradio[wlDriver] = {
                "Channel": channel,
                "AutoChannelEnable": false,
                "OperatingStandards": operating,
                "OperatingChannelBandwidth": channelBandwidth
            };
        }

        // Send saving request:
        swc.models.Rest.sendRequest({
            url: '/sysbus/NeMo/Intf/' + wlID + ':setWLANConfig',
            data: saveData,

            success: function(response) {
                if (response.status === null) {
                    deferred.resolve();
                } else {
                    deferred.reject();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    saveSettingsCombined: function(accessPoint, data) {
        var ssid = '', password = '', encryption = '', visibility = '',
            wlID = this.getParameter("accessPointID", accessPoint),
            saveData = {
                "parameters": {
                    "mibs": {
                        "wlanvap": {}
                    }
                }
            },
            deferred = new $.Deferred();

        // Get neccessary params from array of page data:
        $.each(data, function(key, value) {
            if (value.parameterName === "ssid") {
                ssid = value.parameterValue;
            }

            if (value.parameterName === "password") {
                password = value.parameterValue;
            }

            if (value.parameterName === "encryption") {
                encryption = value.parameterValue;
            }

            if (value.parameterName === "visibility") {
                visibility = value.parameterValue;
            }
        });

        var userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin();
        var security = {
                "KeyPassPhrase": password,
                "WEPKey": this.getParameter("wepkey", accessPoint),
                "ModeEnabled": encryption
        };

        if(userIsSuperAdmin === true) {
            security = {
                "WEPKey": this.getParameter("wepkey", accessPoint),
                "ModeEnabled": encryption
            };
        }

        // Main data selection
        saveData.parameters.mibs.wlanvap[wlID] = {
            "SSID": ssid,
            "SSIDAdvertisementEnabled": visibility === "visible" ? true : false,
            "Security": security
        };

        // Hack for SAH not saving password if security set to No Password
        if (encryption === "None") {
            saveData.parameters.mibs.wlanvap[wlID].Security.WEPKey = this.getParameter('wepkey', accessPoint);
            saveData.parameters.mibs.wlanvap[wlID].Security.KeyPassPhrase = this.getParameter('password', accessPoint);
        }

        // Send saving request:
        swc.models.Rest.sendRequest({
            url: '/sysbus/NeMo/Intf/' + wlID + ':setWLANConfig',
            data: saveData,

            success: function(response) {
                if (response.status === null) {
                    deferred.resolve();
                } else {
                    deferred.reject();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    switchToCombinedMode: function(accessPoint) {
        var ssid = this.getParameter('name', '5GHz'),
            password = this.getParameter('password', '5GHz'),
            encryption = this.getParameter('encryption', '5GHz'),
            visibility = this.getParameter('visibility', '5GHz'),
            wlID = this.getParameter("accessPointID", accessPoint),
            saveData = {
                "parameters": {
                    "mibs": {
                        "wlanvap": {}
                    }
                }
            },
            deferred = new $.Deferred();

        var userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin();
        var security = {
                "KeyPassPhrase": password,
                "WEPKey": this.getParameter("wepkey", accessPoint),
                "ModeEnabled": encryption
        };

        if(userIsSuperAdmin === true) {
            security = {
                "WEPKey": this.getParameter("wepkey", accessPoint),
                "ModeEnabled": encryption
            };
        }

        // Main data selection
        saveData.parameters.mibs.wlanvap[wlID] = {
            "SSID": ssid,
            "SSIDAdvertisementEnabled": visibility,
            "Security": security
        };

        // Send saving request:
        swc.models.Rest.sendRequest({
            url: '/sysbus/NeMo/Intf/' + wlID + ':setWLANConfig',
            data: saveData,

            success: function(response) {
                if (response.status === null) {
                    deferred.resolve();
                } else {
                    deferred.reject();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    saveSettingsGuest: function(accessPoint, data) {
        var ssid = '',
            password = '',
            encryption = '',
            wlID = this.getParameter("accessPointID", "Guest-" + accessPoint),
            saveData = {
                "parameters": {
                    "mibs": {
                        "wlanvap": {}
                    }
                }
            },
            deferred = new $.Deferred();

        // Get neccessary params from array of page data:
        $.each(data, function(key, value) {
            if (value.parameterName === "ssid") {
                ssid = value.parameterValue;
            }

            if (value.parameterName === "password") {
                password = value.parameterValue;
            }

            if (value.parameterName === "encryption") {
                encryption = value.parameterValue;
            }
        });

        var userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin();
        var security = {
                "KeyPassPhrase": password,
                "WEPKey": this.getParameter("wepkey", "Guest-" + accessPoint),
                "ModeEnabled": encryption
        };

        if(userIsSuperAdmin === true) {
            security = {
                "WEPKey": this.getParameter("wepkey", "Guest-" + accessPoint),
                "ModeEnabled": encryption
            };
        }

        // Prepare data to be saved:
        saveData.parameters.mibs.wlanvap[wlID] = {
            "SSID": ssid,
            "Security": security
        };

        // Hack for SAH not saving password if security set to No Password
        if (encryption === "None") {
            saveData.parameters.mibs.wlanvap[wlID].Security.WEPKey = this.getParameter('wepkey', "Guest-" + accessPoint);
            saveData.parameters.mibs.wlanvap[wlID].Security.KeyPassPhrase = this.getParameter('password', "Guest-" + accessPoint);
        }

        // Send saving request:
        swc.models.Rest.sendRequest({
            url: '/sysbus/NeMo/Intf/' + wlID + ':setWLANConfig',
            data: saveData,

            success: function(response) {
                if (response.status === null) {
                    deferred.resolve();
                } else {
                    deferred.reject();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    createModels: function(response, isGuest) {

        var self = this,
            accessPoints = response.status.wlanradio,
            accessPointData = response.status.wlanvap,
            accessPEnableData = response.status.penable;

        // Create models:
        $.each(accessPoints, function(accessPoint, data) {

            var model = new swc.constructors.SimpleModel(),
                accessPointName = data.OperatingFrequencyBand,
                accessPointID,
                baseData;

            // Check if guest Access Point
            if (isGuest) {
                baseData = accessPointName === '2.4GHz' ? accessPointData.wlguest2 : accessPointData.wlguest5;
                accessPointID =  accessPointName === '2.4GHz' ? 'wlguest2' : 'wlguest5';
                accessPointName = 'Guest-' + data.OperatingFrequencyBand;
            } else {
                baseData = accessPointName === '2.4GHz' ? accessPointData.wl0 : accessPointData.wl1;
                accessPointID =  accessPointName === '2.4GHz' ? 'wl0' : 'wl1';
                model.set('credentials', {
                    "2_4": {
                        "ssid": accessPointData.wl0.SSID,
                        "password": accessPointData.wl0.Security.KeyPassPhrase
                    },
                    "5_0": {
                        "ssid": accessPointData.wl1.SSID,
                        "password": accessPointData.wl1.Security.KeyPassPhrase
                    }
                });
            }

            // Set main options
            model.set('id', accessPointName);
            model.set('accessPoint', accessPointName);
            model.set('driver', accessPoint);
            model.set('accessPointID', accessPointID);

            // Main data:
            model.set('name', baseData.SSID);
            // FIX SA-1319: Incorrect data used to get Activation status of radio access points
            model.set('status', accessPEnableData[accessPoint].PersistentEnable);
            model.set('visibility', baseData.SSIDAdvertisementEnabled);

            if(!_.isUndefined(baseData.Security.KeyPassPhrase)) {
                model.set('password', baseData.Security.KeyPassPhrase);
            } else {
                model.set('password', '');
            }

            // Security data:
            // model.set('password', baseData.Security.KeyPassPhrase);
            model.set('wepkey', baseData.Security.WEPKey);
            model.set('encryption', baseData.Security.ModeEnabled);
            model.set('encryptionsSupported', baseData.Security.ModesSupported.split(','));

            // Set driver data:
            model.set('channel', data.Channel);
            model.set('channelAutoEnable', data.AutoChannelEnable);
            model.set('channelsSupported', data.PossibleChannels.split(','));
            model.set('operating', data.OperatingStandards);
            model.set('operatingsSupported', data.SupportedStandards.split(','));
            model.set('OperatingStandards', data.OperatingStandards);
            model.set('OperatingChannelBandwidth', (data.OperatingChannelBandwidth === 'Auto') ? '20MHz' : data.OperatingChannelBandwidth);

            // Add model to the collection:
            self.add(model);
        });
    },

    getParameter: function(parameter, accessPoint) {
        var search = this.where({ accessPoint: accessPoint });

        if (search.length) {
            return search[0].get(parameter);
        } else {
            return '';
        }
    },

    isAccessPointsEquals: function(interfaceOne, interfaceTwo, keys) {
        var self = this,
            isEquals = true;

        if (keys.length) {
            $.each(keys, function(key, value) {
                if (self.getParameter(value, interfaceOne) !== self.getParameter(value, interfaceTwo)) {
                    isEquals = false;
                }
            });
        }

        return isEquals;
    },

    formatModeOptions: function() {
        return {
            "combined": {
                name: getTranslationStrings("Combined"),
                value: "combined"
            },
            "separate": {
                name: getTranslationStrings("Separate"),
                value: "separate"
            }
        };
    },

    formatEncryptionOptions: function() {
        var options = [],
            enabled = {
                "WPA2-Personal": getTranslationStrings("WPA2"),
                "WPA-WPA2-Personal": getTranslationStrings("WPA/WPA2"),
                "None": getTranslationStrings("No encryption")
            };

        $.each(enabled, function(encryption, value) {
            options.push({
                name:  value,
                value: encryption
            });
        });

        return options;
    },

    formatOperatingOptions: function(accessPoint) {
        var accessPointOptions = this.getParameter('operatingsSupported', accessPoint),
            options = {},
            enabled = {
                "n": getTranslationStrings("802.11 n"),
                "ac": getTranslationStrings("802.11 ac"),
                "an": getTranslationStrings("Auto (802.11 a/n/ac)"),
                "bgn": getTranslationStrings("Auto (802.11 b/g/n)")
            };

        // Walk throuh all supported and enabled options
        $.each(accessPointOptions, function(key, value) {
            if (enabled[value]) {
                options[value] = {
                    name:  enabled[value],
                    value: value
                };
            }
        });

        return options;
    },

    formatChannelsOptions: function(accessPoint) {
        var search = this.where({ accessPoint: accessPoint }),
            channelsSupported,
            model,
            options = {
                "auto": {
                    name:  getTranslationStrings("Auto channel"),
                    value: "auto"
                }
            };

        if (search.length) {
            model = search[0];
            channelsSupported = [];
            
            if(accessPoint === '5GHz') {
                channelsSupported = this.static5GChannelList;
            } else {
                channelsSupported = model.get("channelsSupported");
            }

            if (channelsSupported.length) {
                $.each(channelsSupported, function(key, channel) {
                    options[channel] = {
                        name:  channel,
                        value: channel
                    };
                });
            } else {
                return options;
            }
        } else {
            return options;
        }

        return options;
    },

    formatChannelBandwidthOptions: function () {
        return [
            {
                name: "20MHz",
                value: "20MHz"
            },{
                name: "20/40MHz",
                value: "40MHz"
            }
        ];
    },

    formatDeactivationOptions: function() {
        return [
            { name: getTranslationStrings("6 hours"), value: "21600", isDefault: true },
            { name: getTranslationStrings("24 hours"), value: "86400" },
            { name: getTranslationStrings("No auto disabling"), value: "0" }
        ];
    }
});
;swc.constructors.Applications = Backbone.Model.extend({

    isVpnEnable: function() {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                service: "com.swisscom.stargate/ws/vpn.com.swisscom.stargate.vpn",
                method: "getGlobalStatus",
                parameters: {}
            },
            headers: {
                'X-Context': $.cookie(swc.Utils.getDeviceID() + '/context'),
                'Authorization': 'X-Sah ' + $.cookie(swc.Utils.getDeviceID() + '/context'),
                'X-Sah-Request-Type': "idle",
                'idle': true
            },

            success: function(response) {
                var vpnGlobalEnable = response.status;
                deferred.resolve(vpnGlobalEnable);
            },
            error: function() {
                deferred.reject(false);
            }
        });

        return deferred.promise();
    },

    sync: function() {}

});

;swc.constructors.SwisscomDyndns = swc.BaseModel.extend({

    validation: {
        name: function(data) {
            var status = true, messages = [], prev = swc.models.SwisscomDyndns.get('previous_hostname');
            data = _.findWhere(data,{parameterName: 'host-name'}).parameterValue;

            if(!/^[a-zA-Z0-9][a-zA-Z0-9-]{1,62}$/.test(data) || (data.length > 0 && data[data.length - 1] === '-')) {
                messages.push("hostname domain address invalid");
                status = false;
            }

            if(data === '' || data === prev) {
                status = true;
            }

            return {
                status: status,
                messages: messages
            };
        }
    },

	//com.swisscom.stargate/dyndns.com.swisscom.stargate.dyndns
    sync: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        $.when(this.getStatus(fromListener), this.getName(fromListener))
            .done(function(status, name) {
                // if(status.status !== 'OK' || name.status !== 'OK') {
                //     return deferred.reject();
                // }
                self.set({
                    status: status.result,
                    hostname: (name.result === 'Null response' || name.result === '' || name.result === null) ? '' : name.result
                });
                deferred.resolve();
            }).fail(function(){
                deferred.resolve();
            });

        return deferred.promise();
    },

    getStatus: function(fromListener) {
		var deferred = new $.Deferred(),
			self = this;

        swc.models.Rest.sendRequest({
            url: '/ws',
            fromListener: fromListener,
            timeout: 7000,

            data: {
				service: 'com.swisscom.stargate/dyndns.com.swisscom.stargate.dyndns',
				method: 'IsEnabled',
                parameters: {

                }
            },

            success: function(response) {
                try {
                    response = JSON.parse(response.status);
                    self.enableView();
                    deferred.resolve(response);
                } catch(e) {
                    deferred.reject();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    getName: function(fromListener) {
		var deferred = new $.Deferred(),
			self = this;

        swc.models.Rest.sendRequest({
            url: '/ws',
            fromListener: fromListener,
            timeout: 7000,

            data: {
				service: 'com.swisscom.stargate/dyndns.com.swisscom.stargate.dyndns',
				method: 'GetName',
                parameters: {

                }
            },

            success: function(response) {
                try {
                    response = JSON.parse(response.status);
                    self.enableView();
                    deferred.resolve(response);
                } catch(e) {
                    deferred.reject();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    updateConfig: function(params) {
        var deferred = new $.Deferred(),
            self = this;

        swc.models.Rest.sendRequest({
            url: '/ws',
            fromListener: null,
            timeout: 7000,

            data: {
                service: 'com.swisscom.stargate/dyndns.com.swisscom.stargate.dyndns',
                method: 'Configure',
                parameters: params
            },

            success: function(response) {
                response = JSON.parse(response.status);
                self.sync();
				self.enableView();
                deferred.resolve(response);
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    isNameTaken: function(name) {
        var deferred = new $.Deferred(),
            self = this;

        swc.models.Rest.sendRequest({
            url: '/ws',
            fromListener: null,
            timeout: 7000,

            data: {
                service: 'com.swisscom.stargate/dyndns.com.swisscom.stargate.dyndns',
                method: 'IsNameTaken',
                parameters: {
                    name: name
                }
            },

            success: function(response) {
                response = JSON.parse(response.status);
                self.sync();
                if(name === '') {
                    response.result = false;
                }
				self.enableView();
                deferred.resolve(response);
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();

    },

    enableView: function() {
		if(swc.views.applicationsDyndnsView !== undefined) {
			//swc.views.applicationsDyndnsView.hideServerError();
		}
    }

});;swc.constructors.VPNClient = swc.BaseModel.extend({

    url: {
        'delete': '/sysbus/Phonebook:removeContactByUniqueID'
    },

    defaults: function() {
        return {
            "ipAddress": '',
            "connectedFor": ''
        };
    },

    toJSON: function() {
        var json;

        json = {
            parameters: {
                uniqueID: this.get('id'),
                contact: {
                    ipAddress: this.get('formattedName')
                }
            }
        };

        return JSON.stringify(json);
    }

});
;swc.constructors.VPNServer = Backbone.Model.extend({

    validation: {
        username: function(data) {
            var username = "",
                password = "",
                enable,
                validationStatus = true,
                validationMessages = [];

            var userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin();

            if(userIsSuperAdmin === true) {
                return {
                    status: validationStatus,
                    messages: validationMessages
                };
            }

            $.each(data, function(key, value) {
                if (value.parameterName === "vpn_username") {
                    username = value.parameterValue;
                }
                if (value.parameterName === "server_state") {
                    enable = value.parameterValue;
                }
                if (value.parameterName === "vpn_password") {
                    password = value.parameterValue;
                }
            });

            if(enable === false) {
                return {
                    status: validationStatus,
                    messages: validationMessages
                };
            }

            if(!username.length) {
                validationStatus = false;
                validationMessages.push('username must not be empty');
            } else
            if(username === password) {
                validationStatus = false;
                validationMessages.push('password must be different than username');
            } else
            if(username.length < 3 || username.length > 32) {
                validationStatus = false;
                validationMessages.push('username must be from 3 to 32 symbols');
            } else
            if(!/^[\x20-\x7E]{3,}$/gm.test(username)) {
                validationStatus = false;
                validationMessages.push('invalid characters');
            }
            return {
                status: validationStatus,
                messages: validationMessages
            };
        },

        password: function(data) {
            var username = "",
                password = "",
                shared_secret = "",
                enable,
                validationStatus = true,
                validationMessages = [];

            var userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin();

            if(userIsSuperAdmin === true) {
                return {
                    status: validationStatus,
                    messages: validationMessages
                };
            }

            $.each(data, function(key, value) {
                if (value.parameterName === "vpn_username") {
                    username = value.parameterValue;
                }
                if (value.parameterName === "vpn_password") {
                    password = value.parameterValue;
                }
                if (value.parameterName === "server_state") {
                    enable = value.parameterValue;
                }
                if (value.parameterName === "vpn_shared_secret") {
                    shared_secret = value.parameterValue;
                }
            });

            if(enable === false) {
                return {
                    status: validationStatus,
                    messages: validationMessages
                };
            }

            if(password.length < 8 || password.length > 64) {
                validationStatus = false;
                validationMessages.push('password must be from 8 to 64 symbols');
            } else
            if(username === password) {
                validationStatus = false;
                validationMessages.push('password must be different than username');
            } else
            if(shared_secret === password) {
                validationStatus = false;
                validationMessages.push('shared secret must be different than password');
            } else
            if(!/^[\x20-\x7E]{8,}$/gm.test(password)) {
                validationStatus = false;
                validationMessages.push('invalid characters');
            } else
            if(!/\d{1,}|[!@#$%^&*()_+\-=\[\]{};':"\\|,.<>\/?]{1,}/.test(password)) {
                validationStatus = false;
                validationMessages.push('password must contain at least one digit');
            } else
            if(!/[a-zA-Z]{1,}/.test(password)) {
                validationStatus = false;
                validationMessages.push('password must contain at least one letter');
            }

            return {
                status: validationStatus,
                messages: validationMessages
            };
        },

        vpn_shared_secret: function(data) {
            var enable,
                password = "",
                username = "",
                shared_secret = "",
                validationStatus = true,
                validationMessages = [];

            var userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin();

            if(userIsSuperAdmin === true) {
                return {
                    status: validationStatus,
                    messages: validationMessages
                };
            }

            $.each(data, function(key, value) {
                if (value.parameterName === "vpn_username") {
                    username = value.parameterValue;
                }
                if (value.parameterName === "vpn_password") {
                    password = value.parameterValue;
                }
                if (value.parameterName === "server_state") {
                    enable = value.parameterValue;
                }
                if (value.parameterName === "vpn_shared_secret") {
                    shared_secret = value.parameterValue;
                }
            });

            if(enable === false) {
                return {
                    status: validationStatus,
                    messages: validationMessages
                };
            }

            if(shared_secret.length < 8 || shared_secret.length > 64) {
                validationStatus = false;
                validationMessages.push('password must be from 8 to 64 symbols');
            } else
            if(username === shared_secret) {
                validationStatus = false;
                validationMessages.push('shared secret must be different than username');
            } else
            if(shared_secret === password) {
                validationStatus = false;
                validationMessages.push('shared secret must be different than password');
            } else
            if(!/^[\x20-\x7E]{8,}$/gm.test(shared_secret)) {
                validationStatus = false;
                validationMessages.push('invalid characters');
            } else
            if(!/\d{1,}|[!@#$%^&*()_+\-=\[\]{};':"\\|,.<>\/?]{1,}/.test(shared_secret)) {
                validationStatus = false;
                validationMessages.push('shared key must contain at least one digit');
            } else
            if(!/[a-zA-Z]{1,}/.test(shared_secret)) {
                validationStatus = false;
                validationMessages.push('shared key must contain at least one letter');
            }

            console.log('validationMessages', validationMessages);

            return {
                status: validationStatus,
                messages: validationMessages
            };
        }
    },

    sync: function(){
        var self = this,
            deferred = new $.Deferred();

        $.when(
                self.getConfig()
            ).done(function(){
                self.trigger('change');
                deferred.resolve();
            });

        return deferred.promise();
    },

    getConfig: function(){
        var self = this,
            deferred = new $.Deferred(),
            method ="getConfig";

        var userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin();

        if(userIsSuperAdmin === true) {
            method = "getConfigRestricted";
        }

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                service: "com.swisscom.stargate/ws/vpn.com.swisscom.stargate.vpn",
                method: method,
                parameters: {}
            },
            headers: {
                'X-Context': $.cookie(swc.Utils.getDeviceID() + '/context'),
                'Authorization': 'X-Sah ' + $.cookie(swc.Utils.getDeviceID() + '/context'),
                'X-Sah-Request-Type': "idle",
                'idle': true
            },

            success: function(response) {
                var status = $.parseJSON(response.status);
                self.set('enable', status.enabled);

                if(!_.isUndefined(status.username)) {
                    self.set('username', status.username);
                } else {
                    self.set('username', '');
                }

                if(!_.isUndefined(status.password)) {
                    self.set('password', status.password);
                } else {
                    self.set('password', '');
                }

                if(!_.isUndefined(status.presharedkey)) {
                    self.set('shared_secret', status.presharedkey);
                } else {
                    self.set('shared_secret', '');
                }

                self.set('portForwardingStatus', status.portForwardingStatus);
                deferred.resolve();
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    updateConfig: function(params){
        var deferred = new $.Deferred(),
        username = params.vpn_username,
        password = params.vpn_password,
        presharedkey = params.vpn_shared_secret,
        method = "updateConfig";

        var userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin();

        if(params.server_state === false) {
            username = this.get('username');
            password = this.get('password');
            presharedkey = this.get('shared_secret');
        }

        var parameters = {
            enabled: params.server_state,
            username: username,
            password: password,
            presharedkey: presharedkey
        };

        if(userIsSuperAdmin === true) {
            method = "updateConfigRestricted";
            parameters = {
                enabled: params.server_state
            };
        }

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                service: "com.swisscom.stargate/ws/vpn.com.swisscom.stargate.vpn",
                method: method,
                parameters: parameters
            },
            success: function() {
                deferred.resolve();
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    }
});

;swc.constructors.FileManager = swc.BaseCollection.extend({

    url: {
        read: '/ws'
    },

    ajaxParameters: {
        service: "com.swisscom.stargate/ws/nas/filemanager.com.swisscom.stargate.nas.filemanager",
        method: "listFiles",
        parameters: {
            id: 0,
            path: "/"
        }
    },

    initialize: function() {
        var args = Array.prototype.slice.call(arguments),
            errorMessage = 'For retrieving list of files the device ID should be set';

        if (args.length === 0 || _.isUndefined(args[1])) {
            throw new Error(errorMessage);
        }

        if (_.isUndefined(args[1].deviceID)) {
            throw new Error(errorMessage);
        }

        this.ajaxParameters.parameters.id = args[1].deviceID;
        swc.BaseCollection.prototype.initialize.apply(this, arguments);
    },

    /**
     * Comparator sorts elements, so folders will go first
     *
     * @return {String}
     */
    comparator: function(model) {
        var ftype = model.get('type');

        ftype = ftype.toLowerCase();
        ftype = ftype.split('');
        ftype = _.map(ftype, function(letter) {
            return String.fromCharCode(-(letter.charCodeAt(0)));
        });

        return ftype;
    },

    apiToJSON: function(response) {
        var self = this,
            parsedResponse = {},
            parsedData = [];

        try {
            parsedResponse = JSON.parse(response.status);

            if (parsedResponse.status === 200) {
                _.each(parsedResponse.data, function(file) {
                    parsedData.push({
                        isSync: file.syncEnabled,
                        id: file.name,
                        name: file.name,
                        size: file.size * 1024,
                        syncState: self.getSyncState(file.syncState),
                        type: file.type,
                        virtual: file.virtual
                    });
                });
            }
        } catch (e) {
            // we already have default values defined
        }

        return parsedData;
    },

    /**
     * Set cloud backup state
     *
     * @param responseSyncState
     * @returns {*}
     */
    getSyncState: function(responseSyncState) {
        var map = {
            'IN_PROGRESS': 'progress',
            'UPLOAD_IN_PROGRESS': 'progress',
            'DELETE_IN_PROGRESS': 'progress',
            'COMPLETE': 'complete',
            'ERROR': 'error'
        };

        if (_.isUndefined(map[responseSyncState])) {
            return 'not_sync';
        }

        return map[responseSyncState];
    }

});
;swc.constructors.InternetBackupStick = swc.BaseModel.extend({

    url: "/sysbus/NeMo/Intf/wwan:getMIBs",

    checkForUndefinedResponse: true,

    apiToJSON: function(json) {
        var data = (json && json.status && json.status.wwan) ? json.status.wwan.wwan : {};

        var signalVal = 0;

        if(_.isUndefined(data.SignalStrength) || data.SignalStrength === 0) {
            signalVal = 0;
        } else if(data.SignalStrength < 13) {
            signalVal = 1;
        } else if(data.SignalStrength < 19) {
            signalVal = 2;
        } else if(data.SignalStrength < 23) {
            signalVal = 3;
        } else if(data.SignalStrength < 39) {
            signalVal = 4;
        } else {
            signalVal = 5;
        }

        // round to 20 - this is signal precision
        var signalLevel = (signalVal * 20);

        return {
            isConnected: data.ConnectionStatus === "Connected",
            isPresent: (data.ConnectionStatus && (data.ConnectionStatus !== "NotPresent")),
            status: data.ConnectionStatus,
            provider: data.APN,
            technology: data.Technology,
            signalLevel: signalLevel,
            manufacturer: data.Manufacturer,
            model: data.Model
        };
    }

});;swc.constructors.DynDNS = swc.BaseModel.extend({

    url: {
        'read': '/sysbus/DynDNS:getGlobalEnable',
        'update': '/sysbus/DynDNS:setGlobalEnable'
    },

    toJSON: function() {
        return {
            parameters: {
                'enable': this.get('enable')
            }
        };
    },

    apiToJSON: function (response) {
        return {'enable': response.status};
    }

});
;swc.constructors.DynDNSProvider = swc.BaseModel.extend({

    url: {
        'create': '/sysbus/DynDNS:addHost',
        'update': '',
        'delete': '/sysbus/DynDNS:delHost'
    },

    toJSON: function () {
        return {
            parameters: this.attributes
        };
    },

    validation: {
        service: function (data) {
            var validationMessages = [],
                validationData = swc.Utils.getDataToValidate({
                    service: { elementName: 'Service' },
                    globalStatus: { elementName: 'DynDNSEnable' }
                }, data);

            if (validationData.globalStatus === false) {
                return {
                    status: true,
                    messages: []
                };
            }

            if (!$.trim(validationData['service'])) {
                validationMessages.push("should not be empty");
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        hostname: function (data) {
            var validationMessages = [],
                validationData = swc.Utils.getDataToValidate({
                    "hostname": { elementName: 'hostname' },
                    "username": { elementName: 'username' },
                    "password": { elementName: 'password' },
                    globalStatus: { elementName: 'DynDNSEnable' }
                }, data),
                hostname = validationData['hostname'],
                username = validationData['username'],
                password = validationData['password'];

            var userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin();

            // if all are empty - validate success
            if ((validationData.globalStatus === false) || userIsSuperAdmin === true || (_.isEmpty(hostname) && _.isEmpty(username) && _.isEmpty(password))) {
                return {
                    status: true,
                    messages: []
                };
            }

            if (!$.trim(hostname)) {
                validationMessages.push("should not be empty");
            }
            
            if(!/^[A-Za-z]([A-Za-z0-9]|[A-Za-z0-9-]+[A-Za-z0-9])?(\.[A-Za-z]([A-Za-z0-9]|[A-Za-z0-9-]+[A-Za-z0-9]))*(\.[A-Za-z]{2,})$/.test(hostname)) {
                validationMessages.push("hostname domain address invalid");
            }
            
            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        username: function (data) {
            var validationMessages = [],
                validationData = swc.Utils.getDataToValidate({
                    "hostname": { elementName: 'hostname' },
                    "username": { elementName: 'username' },
                    "password": { elementName: 'password' },
                    globalStatus: { elementName: 'DynDNSEnable' }
                }, data),
                hostname = validationData['hostname'],
                username = validationData['username'],
                password = validationData['password'];

            var userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin();

            // if all are empty - validate success
            if ((validationData.globalStatus === false) || userIsSuperAdmin === true || (_.isEmpty(hostname) && _.isEmpty(username) && _.isEmpty(password))) {
                return {
                    status: true,
                    messages: []
                };
            }

            if (!$.trim(username)) {
                validationMessages.push("should not be empty");
            }

            if ( !/^[ -~]+$/.test(username) ) {
                validationMessages.push('username contain not allowed characters');
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        password: function (data) {
            var validationMessages = [],
                validationData = swc.Utils.getDataToValidate({
                    "hostname": { elementName: 'hostname' },
                    "username": { elementName: 'username' },
                    "password": { elementName: 'password' },
                    globalStatus: { elementName: 'DynDNSEnable' }
                }, data),
                hostname = validationData['hostname'],
                username = validationData['username'],
                password = validationData['password'];

            var userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin();

            // if all are empty - validate success
            if ((validationData.globalStatus === false) || userIsSuperAdmin === true || (_.isEmpty(hostname) && _.isEmpty(username) && _.isEmpty(password))) {
                return {
                    status: true,
                    messages: []
                };
            }

            if (!$.trim(password)) {
                validationMessages.push("should not be empty");
            }

            if ( !/^[ -~]+$/.test(password) ) {
                validationMessages.push('password contain not allowed characters');
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        }
    }

});
;swc.constructors.FirewallPolicy = swc.base.Model.extend({

    actionGet: "/sysbus/Firewall:getChainPolicy",
    actionSet: "/sysbus/Firewall:setChainPolicy",

    sync: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        $.when(
                this.makeRequest({chain: "Custom_V6In"}, this.actionGet, fromListener),
                this.makeRequest({chain: "Custom_V6Out"}, this.actionGet, fromListener)
            ).done(function(inbound, outbound){
                self.build(inbound, outbound);
                deferred.resolve();
            });

        return deferred.promise();
    },

    makeRequest: function(parameters, endpoint, fromListener){
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: endpoint,
            fromListener: fromListener,
            data: {
                "parameters": parameters
            },

            success: function(response) {
                deferred.resolve(response);
            }
        });

        return deferred.promise();
    },

    build: function(inbound, outbound) {
        if (!inbound['status'] || !outbound['status'] || inbound['status'] === "None" || outbound['status'] === "None") {
            this.set({
                inbound: 'Accept',
                outbound: 'Accept'
            });
        } else {
            this.set({
                inbound: inbound['status'],
                outbound: outbound['status']
            });
        }
    },

    getValue: function(){
        var inbound = this.get('inbound'),
            outbound = this.get('outbound');

        if (!inbound || !outbound) {
            return;
        }

        return inbound + '_In-' + outbound + '_Out';
    },

    setValue: function(data) {
        var deferred = new $.Deferred(),
            self = this,
            actions,
            policy,
            toDo = [];

        // Get neccessary params from array of page data:
        $.each(data, function(key, value) {
            if (value.parameterName === "policy") {
                policy = value.parameterValue;
            }
        });

        // Parse actions to two
        actions = this.parseValue(policy);

        // Get list of request to be executed
        _.each(actions, function(action) {
            toDo.push(self.makeRequest(action, self.actionSet));
        });

        $.when.apply(null, toDo).done(function() {
            deferred.resolve();
        });

        return deferred.promise();
    },

    parseValue: function(value){
        var targets = value.split('-'),
            targetHash = {};

        _.each(targets, function(val){
            targetHash[val.split('_')[1]] = val.split('_')[0];
        });

        return [{chain: "Custom_V6In", policy: targetHash["In"]},
            {chain: "Custom_V6Out", policy: targetHash["Out"]}];

    },

    formatPolicyOptions: function() {
        var options = {},
            enabled = {
                "Accept_In-Accept_Out": getTranslationStrings("Allow inbound and outbound"),
                "Drop_In-Drop_Out": getTranslationStrings("Block inbound and outbound"),
                "Accept_In-Drop_Out": getTranslationStrings("Allow inbound/Block outbound"),
                "Drop_In-Accept_Out": getTranslationStrings("Block inbound/Allow outbound")
            };

        // Walk throuh all supported and enabled options
        $.each(enabled, function(key, value) {
            options[key] = {
                name:  value,
                value: key
            };
        });

        return options;
    }
});
;swc.constructors.FirewallStatus = swc.base.Model.extend({

    json: {parameters:{}},

    autoSync: true,

    saveJSON: {parameters:{}},

    devices: ['default', 'starlite'],

    possibleStatus: {
        "off": "Low",
        "balanced": "Medium",
        "strict": "High",
        "custom": "Custom"
    },

    getFirewallState: function(fromListener){
        var deferred = new $.Deferred(),
            self= this;

        if(self.autoSync === false || swc.Utils.DeviceType === 'starlink'){
         return deferred.resolve();
        }

        swc.models.Rest.sendRequest({
            url: '/sysbus/NetMaster/IPv6/IPv6rd:get',
            fromListener: fromListener,

            data: {
                "parameters": {}
            },

            success: function(response) {
                deferred.resolve(response);
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();

    },

    getSecurityLevel: function(fromListener){
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Firewall:getFirewallIPv6Level',
            fromListener: fromListener,

            data: {
                "parameters": {}
            },

            success: function(response) {
                deferred.resolve(response);
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();

    },

    /*
     * Redefined sync method, because sync in base model fall down if there is param 'status' in response
     * */
    sync: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        if(self.autoSync === false || swc.Utils.DeviceType === 'starlink'){
         return deferred.resolve();
        }

        $.when(this.getSecurityLevel(fromListener), this.getFirewallState(fromListener))
            .done(function(level, state) {
                self.set({
                    status: level.status,
                    state: state.status
                });
                deferred.resolve();
            });

        return deferred.promise();
    },

    saveSettedLevel: function(data) {
        var self = this,
            deferred = new $.Deferred(),
            newStatus;

        // Get neccessary params from array of page data:
        $.each(data, function(key, value) {
            if (value.parameterName === "security-level") {
                newStatus = value.parameterValue;
            }
        });

        swc.models.Rest.sendRequest({
            url: '/sysbus/Firewall:setFirewallIPv6Level',
            data: {
                "parameters": {
                    level: self.possibleStatus[newStatus]
                }
            },

            success: function(response) {
                deferred.resolve(response);
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    saveFirewallState: function(data) {
        var newState,
            deferred = new $.Deferred();

        // Get neccessary params from array of page data:
        $.each(data, function(key, value) {
            if (value.parameterName === "IPv6Enable") {
                newState = value.parameterValue;
            }
        });

        swc.models.Rest.sendRequest({
            url: '/sysbus/NetMaster/IPv6/IPv6rd:set',
            data: {
                "parameters": {
                    "Enable": newState
                }
            },

            success: function(response) {
                deferred.resolve(response);
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    }

});
;swc.constructors.PortForwardingRule = swc.BaseModel.extend({

    url: {
        // keys quoted uniformly, but the reason is 'delete'
        // because IE thinks it's a key word
        'create': '/sysbus/Firewall:setPortForwarding',
        'update': '/sysbus/Firewall:setPortForwarding',
        'delete': '/sysbus/Firewall:deletePortForwarding',
        'commit': '/sysbus/Firewall:commit'
    },

    apiToJSON: function(response) {
        return response;
    },

    toJSON: function() {
        var json = {
            parameters: {
                id: this.get('id'),
                description: this.get('name'),
                persistent: true,
                origin: 'webui',
                sourceInterface: 'data', // Scope of PF rules
                sourcePrefix: '',        // Has to be empty for PF, used for IPv6
                enable: this.get('status'),
                protocol: this.get('protocol'),
                destinationIPAddress: this.get('deviceIP'),
                internalPort: this.get('ports').destination[0],
                externalPort: this.get('ports').entry[0]
            }
        };

        if (this.get('ports').destination.length === 2) {
            json.parameters.internalPort = this.get('ports').destination.join('-');
        }

        if (this.get('ports').entry.length === 2) {
            json.parameters.externalPort = this.get('ports').entry.join('-');
        }

        // IF rule is UPnP it must be empty
        if (this.get('isUPnP')) {
            json.parameters.origin = 'upnp';
            json.parameters.sourceInterface = '';
            json.parameters.sourcePrefix = '';
        }

        return JSON.stringify(json);
    }

});
;swc.constructors.UPnP = swc.BaseModel.extend({

    url: {
        read: "/sysbus/UPnP-IGD:get",
        update: "/sysbus/UPnP-IGD:set"
    },

    apiToJSON: function(json) {
        var data = json.status;

        return {
            status: data ? data.Enable : false
        };
    },

    toJSON: function() {
        return JSON.stringify({
            "parameters": {
                "Enable": this.get('status')
            }
        });
    }
});
;swc.constructors.ApplicationPart = Backbone.Model.extend({

    sync: function(fromListener){
        var self = this, deferred = new $.Deferred();
        $.when(self.getServiceState(fromListener))
            .done(function(){
                deferred.resolve();
            });

        return deferred.promise();
    },

    setServiceState: function(){
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/APController/PowerManagement:setMode',
            method: 'POST',
            contentType: 'application/x-sah-ws-4-call+json',
            data: {
                "parameters":{
                    "mode": self.get('serviceState')?"ON":"OFF"
                }
            },

            success: function() {
                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    getServiceState: function(fromListener){
        var self = this,
            deferred = new $.Deferred();

        if(swc.Utils.DeviceType !== 'default') {
			deferred.resolve();
			return deferred.promise();
        }

        swc.models.Rest.sendRequest({
            url: '/sysbus/APController/PowerManagement:getMode',
            method: 'POST',
            contentType: 'application/x-sah-ws-4-call+json',
            fromListener: fromListener,

            data: { "parameters":{} },

            success: function(response) {
                if(response.data && response.data.mode){
                    self.set('serviceState', response.data.mode === "ON" ? true : false);
                }
                deferred.resolve();
            }
        });

        return deferred.promise();
    }
});
;swc.constructors.SchedulerAp = swc.BaseModel.extend({

    defaults: {
        override: '',
        enable: false,
        schedule: []
    },

    url: {
        'read': '/sysbus/Scheduler:getCompleteSchedules',
        'update': '/sysbus/Scheduler:addSchedule'
    },

    ajaxParameters: {
        parameters: {
            type: 'AP'
        }
    },

    /**
     * Initialize method makes the following actions:
     * 1) extends current model with schedule mixin
     * 2) adds listener between schedule data and its widget
     */
    initialize: function() {
        _.extend(this, swc.mixins.Scheduler);
        this.addWidgetListener();
        swc.BaseModel.prototype.initialize.apply(this, arguments);
    },

    /**
     * Apply data from server on model
     *
     * @param {Object} response Response from server
     * @returns {Array}
     *          {
     *              override
     *              enable: true | false,       if the schedule is enabled
     *              schedule: [                 schedule data in seconds
     *                  {
     *                      begin: int,         start of disabled period in seconds
     *                      end: int ,          end of disabled period in seconds
     *                      state: "Disable",   constant value
     *                  },
     *                  ...
     *              ],
     *              pixelSchedule: [            schedule data in pixels
     *                 {
     *                      begin: 108,         start of enabled period in pixels
     *                      end: 414            end of enabled period in pixels
     *                 },
     *                  ...
     *              ]
     *          }
     */
    apiToJSON: function(response) {
        var parsed = [],
            scheduleInfo;

        if (response.status && response.data && response.data.scheduleInfo) {
            scheduleInfo = _.findWhere(response.data.scheduleInfo, { "ID": "APSchedule" });

            if (!_.isUndefined(scheduleInfo)) {
                parsed = {
                    override: '',
                    enable: scheduleInfo.enable,
                    schedule: scheduleInfo.schedule || []
                };
            } else {
                parsed = this.defaults;
            }

            // if scheduler is fresh, the default scheduler is used
            if (_.isEmpty(parsed.schedule) || this.isEmpty(parsed.schedule)) {
                parsed['pixelSchedule'] =  _.clone(this.getDefaultScheduleInPixels(), true);
                parsed['schedule'] = this.convertToSeconds(parsed['pixelSchedule']);
            } else {
                if (!_.isUndefined(scheduleInfo)) {
                    parsed['pixelSchedule'] =  this.convertToPixels(this.splitByDays(scheduleInfo.schedule));
                }
            }
        }

        return parsed;
    },

    toJSON: function() {
        var json;

        json = {
            parameters: {
                type: 'AP',
                info: {
                    ID: 'APSchedule',
                    enable: this.isEmpty() ? false : this.get('enable'),
                    base: 'Weekly',
                    def: 'Enable',
                    override: '',
                    schedule: this.get('schedule')
                }
            }
        };

        return JSON.stringify(json);
    }

});
;swc.constructors.SchedulerDevice = swc.BaseModel.extend({

    url: {
        'update': '/sysbus/Scheduler:addSchedule'
    },

    defaults: {
        override: '',
        enable: false,
        schedule: []
    },

    /**
     * Initialize method makes the following actions:
     * 1) extends current model with schedule mixin
     * 2) adds listener between schedule data and its widget
     */
    initialize: function() {
        _.extend(this, swc.mixins.Scheduler);
        this.addWidgetListener();
        swc.BaseModel.prototype.initialize.apply(this, arguments);
    },

    /**
     * Add schedule data to device model:
     *  - if the schedule property is empty then init this property with default values
     *  - add converted to pixels schedule
     */
    addPixelScheduleInfo: function() {
        // if scheduler is fresh, the default scheduler is used
        if (_.isEmpty(this.get('schedule'))) {
            this.set('pixelSchedule', _.clone(this.getDefaultScheduleInPixels(), true));
            this.set('schedule', this.convertToSeconds(this.get('pixelSchedule')));
        } else {
            this.set('pixelSchedule', this.convertToPixels(this.splitByDays(this.get('schedule'))));
        }
    },

    toJSON: function() {
        var json;

        json = {
            parameters: {
                type: 'ToD',
                info: {
                    ID: this.get('id'),
                    enable: this.get('enable'),
                    base: 'Weekly',
                    def: 'Enable',
                    override: '',
                    schedule: this.get('schedule')
                }
            }
        };

        return JSON.stringify(json);
    }
});;swc.constructors.SchedulerDeviceManager = Backbone.Model.extend({

    /**
     * Should be rewritten to prevent default behaviour
     *
     * @returns {Bool} Since calling method expect promised value
     */
    sync: function() {
        return true;
    },

    /**
     * The collection of devices and collection of device schedulers are received from
     * server independently and list of devices can differ from list of related schedulers.
     * This method:
     * 1. adds default scheduler to SchedulerDevicesCollection for devices which
     * don't have scheduler in the SchedulerDevicesCollection currently;
     * 2. removes from SchedulerDevicesCollection instances that don't match any network device
     * 3. updates SchedulerDevicesCollection with actual parameter 'active' which matches
     *    current device status
     * 4. resort SchedulerDevicesCollection according updated parameter 'active'
     *
     * @param {swc.models.NetworkDevices} devices
     * @param {swc.models.SchedulerDevicesCollection} schedules
     */
    applySchedulesToDevices: function(devices, schedules, dontSort) {
        var scheduleModel = null;

        devices.each(function(device) {
            if(!device.get('isHidden')) {
                if (_.isUndefined(schedules.findWhere({ id: device.get('mac') }))) {
                    scheduleModel = new swc.constructors.SchedulerDevice();
    
                    scheduleModel.set({
                        id: device.get('mac'),
                        active: device.get('status')
                    });
    
                    scheduleModel.addPixelScheduleInfo();
    
                    schedules.push(scheduleModel);
                } else {
                    // add device info to schedule model
                    var schedule = schedules.findWhere({ id: device.get('mac') });
                    schedule.set({
                        active: device.get('status')
                    });
                }
            }
        });

        schedules.each(function(schedule) {
            if (_.isUndefined(devices.where({ mac: schedule.get('id') }))) {
                schedules.remove(schedule, { silent: true });
            }
        });

        if(!dontSort) {
            schedules.sort();
        }
    }
});;swc.mixins.Scheduler = {

    /**
     * Enumeration of week days
     */
    weekDays: ['monday', 'tuesday', 'wednesday', 'thursday', 'friday', 'saturday', 'sunday'],

    /**
     * width of the scheduler widget
     * on the screen in pixels
     */
    widgetWidthInPixels: 432,

    /**
     * Amount of seconds in week
     */
    amountOfSecondsInWeek: 604800,

    /**
     * One minimal section is 30 min that is equivalent to 1800 sec
     */
    minimalSectionInSeconds: 1800,

    /**
     * Amount of minimal sections in the day range, i.e.
     * Days are splitted on 24 hours and one minimal section is 30 min,
     * so one day contains 48 sections
     */
    quantityOfSections: 48,
    
    getCurrentDay: function(now) {
        var day = now.getDay() - 1;
        if(day < 0) {
            day = 6;
        }
        
        return this.weekDays[day];
    },

    /**
     * Get default schedule in pixels
     *
     * @param {Object} schedule begin and end points are in pixels
     *        each day allows values from 0 to widgetWidthInPixels
     *        This array represents ON-states of scheduler
     *        {
     *            'monday' : [ {begin: 108, end: 414} ],
     *            'tuesday' : [ {begin: 108, end: 414} ],
     *            ...
     *        }
     */
    getDefaultScheduleInPixels: function() {
        var schedule = {};

        _.each(this.weekDays, function(day) {
            schedule[day] = [
                {
                    'begin': 108,
                    'end': 414
                }
            ];
            //schedule[day] = [];
        });

        return schedule;
    },

    /**
     * Add binding between schedule model and schedule widget
     */
    addWidgetListener: function() {
        var self = this;

        this.on('change:pixelSchedule', function() {
            self.set({'schedule': self.convertToSeconds(self.get('pixelSchedule'))}, { silent:true });
        });
    },

    /**
     * Invert weekly scheduler - on the WebUI we need 'enabled' ranges, but NP needs 'disabled' ranges
     * 
     * @param {Array} schedule
     */
    invertSchedule: function (schedule) {
        var self = this,
            startPoint = 0, hops = 0, invertedSchedule = [],
            sortedSchedule = _.sortBy(schedule, function(obj) {
                return obj.begin;
            });

        _.each(sortedSchedule, function(obj) {
            var begin = startPoint,
                end = obj.begin;

            startPoint = obj.end;
            hops++;
            
            if ((end-begin)>60) {
                invertedSchedule.push({begin: begin, end: end});
            }
            
            if (hops === sortedSchedule.length) {
                begin = startPoint;
                end = self.amountOfSecondsInWeek;
                if ((end - begin) > 60) {
                    invertedSchedule.push({begin: begin, end: end});
                }
            }
        });

        if (schedule.length === 0) {
            invertedSchedule = [{ 'begin': 0, 'end': self.amountOfSecondsInWeek }];
        }

        return invertedSchedule;
    },

    /**
     * separate scheduler range to 7 parts by days
     *
     * @param {Array} schedule Array of begin and end points(in seconds)
     *        from Mon 00:00 till Sun 24:00 without separating
     *        on days.
     *        This array represents ON-states of scheduler
     *        [
     *            { begin: 0, end: 3600},
     *            { begin: 77400, end: 86400},
     *            { begin: 115200, end: 138600},
     *            ...
     *        ]
     * @returns {Object} separatedSchedule begin and end points are in second.
     *        Each day allows values from 0 to 86400(24h)
     *        This array represents ON-states of scheduler
     *        {
     *            'monday' : [ {begin: 0, end: 3600}, {begin: 77400, end: 86400} ],
     *            'tuesday' : [ {begin: 28800, end: 52200} ],
     *            ...
     *        }
     */
    splitByDays: function (schedule) {
        var self = this,
            dayList = _.clone(this.weekDays, true),
            hops = 0,
            separatedSchedule = {
                'monday': [],
                'tuesday': [],
                'wednesday': [],
                'thursday': [],
                'friday': [],
                'saturday': [],
                'sunday': []
            },
            invertedSchedule = this.invertSchedule(schedule),
            sortedSchedule = _.sortBy(invertedSchedule, function(obj) {
                return obj.begin;
            });

        $.each(separatedSchedule, function(currentDay, dayData) {
            _.each(sortedSchedule, function(obj) {
                var begin, end,
                    secondsInDayBegin = swc.Utils.getDaySeconds(obj.begin),
                    secondsInDayEnd = swc.Utils.getDaySeconds(obj.end);

                if (dayList[secondsInDayBegin.hops] === currentDay) {
                    begin = secondsInDayBegin.seconds;
                    if (dayList[secondsInDayEnd.hops] !== currentDay) {
                        end = 86400;
                    }
                }
                if (dayList[secondsInDayEnd.hops] === currentDay) {
                    end = secondsInDayEnd.seconds;
                    if (dayList[secondsInDayBegin.hops] !== currentDay) {
                        begin = 0;
                    }
                }
                if (secondsInDayBegin.hops < hops && secondsInDayEnd.hops > hops) {
                    begin = 0;
                    end = 86400;
                }

                if (end && end > begin && Math.abs(begin - end) >= self.minimalSectionInSeconds) {
                    dayData.push({ begin: begin, end: end });
                }

            });

            hops++;
        });

        return separatedSchedule;
    },

    /**
     * Check if the schedule is empty, i.e. no one range
     * on the whole schedule.
     *
     * If the param scheduleArray is present then its values will be checked
     * else the schedule field of the current model will be checked
     *
     * @param {Array} scheduleArray Optional, array with schedule
     * @returns {*|boolean|boolean}
     */
    isEmpty: function(scheduleArray) {
        var schedule = scheduleArray ? scheduleArray : this.get('schedule');

        return (schedule && schedule[0] && schedule[0].begin === 0 && schedule[0].end === this.amountOfSecondsInWeek);
    },

    /**
     * Convert schedule data from seconds to pixels
     *
     * @description
     *
     *      the method divides begin and end values in seconds
     *      on convertCoefficient to get begin and end values
     *      in pixels
     *
     * @param {Object} separatedSchedule begin and end points are in second.
     *        Each day allows values from 0 to 86400(24h)
     *        This array represents ON-states of scheduler
     *        {
     *            'monday' : [ {begin: 0, end: 3600}, {begin: 77400, end: 86400} ],
     *            'tuesday' : [ {begin: 28800, end: 52200} ],
     *            ...
     *        }
     * @returns {Object} separatedSchedule begin and end points are in pixels.
     *        Each day allows values from 0 to this.widgetWidthInPixels
     *        This array represents ON-states of scheduler
     *        {
     *            'monday' : [ {begin: 0, end: 18}, {begin: 387, end: 432} ],
     *            'tuesday' : [ {begin: 144, end: 261} ],
     *            ...
     *        }
     */
    convertToPixels: function(schedule) {
        var convertCoefficient = 86400 / this.widgetWidthInPixels,
            convertedSchedule = _.clone(schedule, true);

        _.each(convertedSchedule, function(dayData) {
            _.each(dayData, function(data) {
                data.begin = Math.round(data.begin / convertCoefficient);
                data.end = Math.round(data.end / convertCoefficient);
            });
        });

        return convertedSchedule;
    },

    /**
     * Convert schedule from object into array for sending to server
     *
     * @param {Object} schedule begin and end points are in pixels
     *        each day allows values from 0 to widgetWidthInPixels
     *        This array represents ON-states of scheduler
     *        {
     *            'monday' : [ {begin: 0, end: 3600}, {begin: 77400, end: 86400} ],
     *            'tuesday' : [ {begin: 28800, end: 52200} ],
     *            ...
     *        }
     * @return {Array} convertedSchedule begin and end points are in seconds
     *        This array represents OFF-states of scheduler
     *        [
     *            { begin: 0, end: 3600, state: "Disable" },
     *            { begin: 77400, end: 86400, state: "Disable" },
     *            { begin: 115200, end: 138600, state: "Disable" },
     *            ...
     *        ]
     */
    convertToSeconds: function (schedule) {
        var convertedSchedule = [],
            hops = 0,
            convertVar = 86400 / this.widgetWidthInPixels;

        $.each(schedule, function(day, array) {
            _.each(array, function(obj) {
                var range = {},
                    begin = obj.begin * convertVar,
                    end = obj.end * convertVar;

                range.begin = hops * 86400 + begin;
                range.end = hops * 86400 + end;

                if (range.end) {
                    convertedSchedule.push(range);
                }
            });
            hops++;
        });

        convertedSchedule.sort(function(obj1, obj2) {
            return obj1.begin - obj2.begin;
        });

        _.each(convertedSchedule, function(obj, index) {
            if (convertedSchedule[index + 1]) {
                if (obj.end === convertedSchedule[index + 1].begin) {
                    var range = {
                        begin: obj.begin,
                        end: convertedSchedule[index + 1].end
                    };
                    convertedSchedule[index] = range;
                    convertedSchedule.splice(index + 1, 1);
                }
            }
        });

        convertedSchedule = this.invertSchedule(convertedSchedule);

        _.each(convertedSchedule, function(obj) {
            obj.state = "Disable";
        });

        return convertedSchedule;
    },

    /**
     * Set default schedule data to model
     *
     * @param id
     */
    makeDefault: function() {
        this.set({
            'enable': true,
            'schedule': this.convertToSeconds(this.get('pixelSchedule')),
            'pixelSchedule': _.clone(this.getDefaultScheduleInPixels(), true)
        }, {silent:true} );
    },
    
    enable: function(value) {
        var me = this/*,
            schedule = {}*/;
        
        me.set('enable', value);
        /*if(value && me.isEmpty()) {
            _.each(me.weekDays, function(day) {
                schedule[day] = [
                    {
                        'begin': 108,
                        'end': 414
                    }
                ];
            });
            me.set('pixelSchedule', schedule);
            me.set('schedule', me.convertToSeconds(schedule));
        }*/
    }
};;swc.constructors.SchedulerPowerSavingManager = Backbone.Model.extend({

    /**
     * Should be rewritten to prevent default behaviour.
     *
     * @returns {Bool} Since calling method expect promised value
     */
    sync: function() {
        return true;
    },

    /**
     * Check if all necessary models are loaded
     *
     * @param {Object} options
     *      saveWlan {Boolean}, optional, if omitted then false,
     *                          determine if WLAN schedule should be saved
     *      saveAp   {Boolean}, optional, if omitted then false
     *                          determine if AP schedule should be saved
     *      saveDect {Boolean}, optional, if omitted then false
     *                          determine if DECT ECO mode should be saved
     * @returns {boolean}
     * @private
     */
    _necessaryModelsAccessible: function(options) {
        var saveWlan = options && options.saveWlan ? options.saveWlan : false,
            saveAp = options && options.saveAp ? options.saveAp : false,
            saveDect = options && options.saveDect ? options.saveDect : false;

        // check that necessary models are loaded
        if ((saveWlan && !swc.models.SchedulerWlan) ||
            (saveAp && !swc.models.SchedulerAp) ||
            (saveDect && !swc.models.Telephony)) {
            return false;
        }

        // check dependencies for WLAN scheduler
        if (saveWlan && swc.models.SchedulerWlan.isEmpty() && !swc.models.Wireless) {
            return false;
        }

        // check dependencies for AP scheduler
        if (saveAp && swc.models.SchedulerAp.isEmpty() && !swc.models.apServiceState) {
            return false;
        }

        return true;
    },

    /**
     * Do operations in chain:
     *    - save the scheduler,
     *    - if WLAN scheduler is empty disable WiFi globally,
     *    - if AP scheduler is empty disable AP globally,
     *
     * The save actions should be processed in chain because NP
     * enables WiFi if scheduler is updated
     *
     * @param {Object} options Can contain the following parameters
     *      saveWlan {Boolean}, optional, if omitted then false,
     *                          determine if WLAN schedule should be saved
     *      saveAp   {Boolean}, optional, if omitted then false
     *                          determine if AP schedule should be saved
     *      saveDect {Boolean}, optional, if omitted then false
     *                          determine if DECT ECO mode should be saved
     */
    save: function(options) {
        var deferred = new $.Deferred(),
            deferredArray = [],
            saveWlan = options && options.saveWlan ? options.saveWlan : false,
            saveAp = options && options.saveAp ? options.saveAp : false,
            saveDect = options && options.saveDect ? options.saveDect : false;

        if (!this._necessaryModelsAccessible(options)) {
            return deferred.reject();
        }

        // the scheduler shouldn't be saved if it doesn't have changes
        saveWlan = saveWlan && swc.models.SchedulerWlan.hasChanged();
        saveAp = saveAp && swc.models.SchedulerAp.hasChanged();
        saveDect = saveDect && swc.models.Telephony.hasChanged();

        if (saveWlan) {
            deferredArray.push(swc.models.SchedulerWlan.sync('update'));
        }

        if (saveAp && swc.models.Application.get('GlobalEnable')) {
            deferredArray.push(swc.models.SchedulerAp.sync('update'));
        }

        if (saveDect) {
            deferredArray.push(swc.models.Telephony.setDectEco());
        }

        $.when.apply(null, deferredArray)
            .done(function() {
                var processedWlanIsEmpty = saveWlan && swc.models.SchedulerWlan.isEmpty(),
                    processedApIsEmpty = saveAp && swc.models.SchedulerAp.isEmpty(),
                    deferredArrayOfActionsOnEmpty = [];

                // Some delay is needed because scheduler is not applied immediately
                setTimeout(function() {
                    if (processedWlanIsEmpty) {
                        deferredArrayOfActionsOnEmpty.push(swc.models.Wireless.setGlobalStatus(false));
                    }

                    if (processedApIsEmpty) {
                        swc.models.apServiceState.set('status', false);
                        deferredArrayOfActionsOnEmpty.push(swc.models.apServiceState.sync('update'));
                    }

                    $.when.apply(null, deferredArrayOfActionsOnEmpty)
                        .done(function() {
                            deferred.resolve();
                        })
                        .fail(function() {
                            deferred.reject();
                        });
                }, 1000);
            })
            .fail(function() {
                deferred.reject();
            });

        return deferred.promise();
    },

    /**
     * Proxy method to save schedules for WLAN, AP, DECT
     *
     * @returns {*}
     */
    saveAll: function() {
        return this.save({ saveWlan: true, saveAp: true, saveDect: true });
    },

    /**
     * Proxy method to save schedules for WLAN
     *
     * @returns {*}
     */
    saveWlan: function() {
        return this.save({ saveWlan: true });
    },

    /**
     * Proxy method to save schedules for AP
     *
     * @returns {*}
     */
    saveAp: function() {
        return this.save({ saveAp: true });
    },

    /**
     * Proxy method to save schedules for DECT
     *
     * @returns {*}
     */
    saveDect: function() {
        return this.save({ saveDect: true });
    }

});;swc.constructors.SchedulerWlan = swc.BaseModel.extend({

    defaults: {
        override: '',
        enable: false,
        schedule: []
    },

    url: {
        'read': '/sysbus/Scheduler:getCompleteSchedules',
        'update': '/sysbus/Scheduler:addSchedule'
    },

    ajaxParameters: {
        parameters: {
            type: 'WLAN'
        }
    },

    /**
     * Initialize method makes the following actions:
     * 1) extends current model with schedule mixin
     * 2) adds listener between schedule data and its widget
     */
    initialize: function() {
        _.extend(this, swc.mixins.Scheduler);
        this.addWidgetListener();
        swc.BaseModel.prototype.initialize.apply(this, arguments);
    },

    /**
     * Apply data from server on model
     *
     * @param {Object} response Response from server
     * @returns {Array}
     *          {
     *              override
     *              enable: true | false,       if the schedule is enabled
     *              schedule: [                 schedule data in seconds
     *                  {
     *                      begin: int,         start of disabled period in seconds
     *                      end: int ,          end of disabled period in seconds
     *                      state: "Disable",   constant value
     *                  },
     *                  ...
     *              ],
     *              pixelSchedule: [            schedule data in pixels
     *                 {
     *                      begin: 108,         start of enabled period in pixels
     *                      end: 414            end of enabled period in pixels
     *                 },
     *                  ...
     *              ]
     *          }
     */
    apiToJSON: function(response) {
        var parsed = [],
            scheduleInfo;

        if (response.status && response.data && response.data.scheduleInfo) {
            scheduleInfo = _.findWhere(response.data.scheduleInfo, { "ID": "wl0" });

            if (!_.isUndefined(scheduleInfo)) {
                parsed = {
                    override: '',
                    enable: scheduleInfo.enable,
                    schedule: scheduleInfo.schedule || []
                };
            } else {
                parsed = this.defaults;
            }

            // if scheduler is fresh, the default scheduler is used
            if (_.isEmpty(parsed.schedule) || this.isEmpty(parsed.schedule)) {
                parsed['pixelSchedule'] =  _.clone(this.getDefaultScheduleInPixels(), true);
                parsed['schedule'] = this.convertToSeconds(parsed['pixelSchedule']);
            } else {
                if (!_.isUndefined(scheduleInfo)) {
                    parsed['pixelSchedule'] =  this.convertToPixels(this.splitByDays(scheduleInfo.schedule));
                }
            }
        }

        return parsed;
    },

    toJSON: function() {
        var json;

        json = {
            parameters: {
                type: 'WLAN',
                info: {
                    ID: 'wl0',
                    enable: this.isEmpty() ? false : this.get('enable'),
                    base: 'Weekly',
                    def: 'Enable',
                    override: '',
                    schedule: this.get('schedule')
                }
            }
        };

        return JSON.stringify(json);
    }

});
;swc.constructors.AfpServices = swc.BaseModel.extend({

    url: '/ws',

    ajaxParameters: JSON.stringify({
        service: 'com.swisscom.stargate/ws/nas/fileservices.com.swisscom.stargate.nas.fileservices',
        method: 'list',
        parameters: JSON.stringify({})
    }),

    /**
     * Using try {} catch {}, because if AP is disabled, it will return not valid json
     * @param response {Object}
     * @returns {Object}
     */
    apiToJSON: function(response) {
        var result = {
                status: false
            },
            parsedResponse;

        try {
            parsedResponse = JSON.parse(response.status);

            if (parsedResponse.success) {
                result.status = _.findWhere(parsedResponse.fileServices, { service: 'AFP' }).enabled;
            }
        } catch (e) {
            // we have default values already
        }

        return result;
    },

    toJSON: function() {
        return JSON.stringify({
            service: "com.swisscom.stargate/ws/nas/fileservices.com.swisscom.stargate.nas.fileservices",
            method: "setState",
            parameters: {
                data: JSON.stringify({
                    service: 'AFP',
                    enabled: this.get('status')
                })
            }
        });
    }

});
;swc.constructors.apServiceLoadingState = swc.BaseModel.extend({

    url: '/ws',

    ajaxParameters: JSON.stringify({
        service: 'com.swisscom.stargate/ws/core/info.com.swisscom.stargate.app',
        method: 'getApplicationStatus',
        parameters: JSON.stringify({})
    }),

    sync: function(method, options) {
		if(swc.Utils.DeviceType === 'default') {
			this.sync = (new swc.BaseModel()).sync;
			return this.sync(method, options);
		}
    },

    isForceSync: true,

    /**
     * Using try {} catch {}, because if AP is disabled, it will return not valid json
     * @param response {Object}
     * @returns {Object}
     */
    apiToJSON: function(response) {
        var result = {
                isLoading: true
            },
            parsedResponse;

        try {
            parsedResponse = JSON.parse(response.status);
            result = parsedResponse;

            if (!_.isUndefined(parsedResponse.status) && parsedResponse.status === "STARTED") {
                result.isLoading = false;
            }
        } catch (e) {
            // we have default values already
        }

        return result;
    },

    /**
     * Helper method to check if AP is completely started
     *
     * @returns {boolean}
     */
    isApStarted: function () {
        return this.get("status") === "STARTED";
    }

});
;swc.constructors.apServiceState = swc.BaseModel.extend({

    url: {
        read: '/sysbus/APController/PowerManagement:getMode',
        update: '/sysbus/APController/PowerManagement:setMode'
    },

    isForceSync: true,

    apiToJSON: function(response) {
        var result = {
                status: false
            };

        if (!_.isUndefined(response.data) && !_.isUndefined(response.data.mode)) {
            result.status = response.data.mode === 'ON';
        }

        return result;
    },

    sync: function(method, options) {
		if(swc.Utils.DeviceType === 'default') {
			this.sync = (new swc.BaseModel()).sync;
			return this.sync(method, options);
		}
    },

    toJSON: function() {
        return JSON.stringify({
            parameters: {
                mode: this.get('status') === true ? 'ON': 'OFF'
            }
        });
    },

    /**
     * Helper method to check if AP is enabled
     *
     * @returns {Boolean}
     */
    isEnabled: function () {
        return this.get('status');
    }

});
;// cloud services item
swc.constructors.CloudAccount = swc.BaseModel.extend({

    fields: {
        id: '',
        cloudServiceName: '',
        deviceLabel: '',
        deviceId: '',
        accountName: '',
        status: '',
        error: '',
        totalSpace: '',
        usedSpace: '',
        lastSyncDate: '',
        folderName: ''
    },

    getExtendedJSON: function () {
        var info = this.toJSON(),
            status = this.get('status'),
            usedSpace = this.get('usedSpace'),
            totalSpace = this.get('totalSpace'),
            lastSyncDate = this.get('lastSyncDate'),
            // get server's timezone for date formatting:
            timezone = swc.settings.serverTimeZone || '';

        moment.lang(swc.models.Locale.locale);

        info.isActivated = status !== 'DISABLED';
        info.lastSync = lastSyncDate ? moment(lastSyncDate).zone(timezone).format("D MMM YYYY, HH:mm") : "";
        info.lastLogin = moment(new Date()).zone(timezone).format("D MMM YYYY, HH:mm");
        info.usedSpace = swc.Utils.getBytesWithUnit(usedSpace);
        info.freeSpace = (totalSpace < usedSpace) ? 0 : swc.Utils.getBytesWithUnit(totalSpace - usedSpace);
        info.totalSpace = swc.Utils.getBytesWithUnit(totalSpace);
        info.freePercents = (totalSpace < usedSpace) ? 0 : Math.floor(parseFloat(totalSpace - usedSpace) /
                parseFloat(totalSpace ? totalSpace : 1) * 100);

        return info;
    },

    // TODO: migrate to new API when ready
    setBackupState: function(state) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                service: "com.swisscom.stargate/ws/cloud/backup.com.swisscom.stargate.cloud.backup",
                method: "setFolderSyncState",
                parameters: {
                    resources: JSON.stringify({
                        partitionUUID: this.get('deviceId'),
                        folderName: this.get('folderName'),
                        isEnabled: state
                    })
                }
            },

            success: function() {
                self.set('status', state ? 'IN_PROGRESS' : 'DISABLED');
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    // TODO: include model.id or  something when backend is done
    logout: function (keepData) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                service: "com.swisscom.stargate/ws/cloud/backup.com.swisscom.stargate.cloud.backup",
                method: "syncFSResources",
                parameters: {
                    resources: [
                        JSON.stringify({
                            deviceId: this.get('deviceId'),
                            content: [
                                JSON.stringify({
                                    path: this.get('folderName'),
                                    action: keepData ? 'REMOVE' : 'REMOVE_AND_DELETE_LOCAL_DATA'
                                })
                            ]
                        })
                    ]
                }
            },

            success: function(response) {
                // Sometimes error may happen when one try remove folder. See ticket [SA-2780].
                if (response && response.status) {
                    var result = JSON.parse(response.status);

                    // some iternal server error happen, show error message
                    if (result.status === 500) {
                        return deferred.reject(result); // stop further execution
                    }
                }
                swc.models.CloudAccountCollection.sync('read');
                deferred.resolve();
            },

            error: function(){
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    forceSyncDevice: function(){
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                "service": "com.swisscom.stargate/ws/cloud/backup.com.swisscom.stargate.cloud.backup",
                "method": "forceFailedTasksForPartition",
                "parameters": {
                    "partitionUUID": this.get("deviceId")
                }
            },

            success: function() {
                // Timeout is used because the board responds immediately
                // but actually needs some time to change the sync state.
                // sad but true :(
                setTimeout(function(){
                    deferred.resolve();
                }, 3000);
            }
        });

        return deferred.promise();
    }
});
;swc.constructors.CloudServices = Backbone.Model.extend({

    autoSync: true,

    sync: function(fromListener){
        var self = this, deferred = new $.Deferred();

        if(self.autoSync===false){
         return deferred.resolve();
        }

        $.when(self.getServices(fromListener)).done(function() {
            deferred.resolve();
        });

        return deferred.promise();
    },

    /**
     * activates/deactivates cloud backup feature
     *
     * @param state - boolean
     */
    setBackupState: function(state){
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                service: "com.swisscom.stargate/ws/cloud/backup.com.swisscom.stargate.cloud.backup",
                method: "setCloudSyncState",
                parameters: {
                    isEnabled: state
                }
            },

            success: function() {
                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    /**
     * gets list of cloud services and state of backup feature
     * model.get('services') returns collection of backup services
     * @returns {*}
     *
     {"status": "200"
      "data": {"syncActivated":"true",
               "syncServices": [
                  {"id":"dropbox",
                   "displayName": "Drop Box",
                   "authStatusCode": "200",
                   "default":"true",
                   "totalSpace" : "200000", //in KB
                   "usedSpace" : "1500", //in KB
                   "lastSyncDate" : "10.12.2012, 15:45"
                  },
                  {"id":"googledrive",
                   "displayName": "Google Drive",
                   "authStatusCode": "503",
                   "default": "false",
                   "totalSpace" : "", //in KB
                   "usedSpace" : "", //in KB
                   "lastSyncDate" : ""
                   }
               ]
      }
     }
     * status codes:
     * 200 - authorized
     * 403 - invalid login/password - connection error
     * 503 - deauthorized
     */
    getServices: function(fromListener) {
        var self = this,
            deferred = new $.Deferred(),
            defaultData = {data: {
                syncActivated: false,
                syncServices: [
                    {
                        "id": "dropbox",
                        "displayName": "Drop Box",
                        "authStatusCode": "503",
                        "default": "true",
                        "totalSpace": "200000000",
                        "usedSpace": "1500000",
                        "usedPercents": 1,
                        "lastSyncDate": "10.12.2012, 15:45"
                    }
                ]
            }};

        if(swc.Utils.DeviceType !== 'default') {
			deferred.resolve();
			return deferred.promise();
        }

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                service: "com.swisscom.stargate/ws/cloud/backup.com.swisscom.stargate.cloud.backup",
                method: "getCloudSyncInfo",
                parameters: {}
            },
            fromListener: fromListener,

            success: function(response) {
                var services =  new Backbone.Collection(),
                    resp;

                try {
                    resp = JSON.parse(response.status);
                } catch (e) {
                    resp = defaultData;
                }

                self.set('services', services);

                if (!_.isUndefined(resp) && !_.isNull(resp) && !_.isUndefined(resp.data)) {
                    self.set('active', resp.data.syncActivated);

                    _.each(resp.data.syncServices, function(obj) {
                        var serviceModel = new Backbone.Model(obj);

                        serviceModel.set({
                            usedSpace: obj.usedSpace * 1024,
                            totalSpace : obj.totalSpace * 1024,
                            usedPercents: Math.floor(parseFloat(obj.usedSpace) / parseFloat(obj.totalSpace ? obj.totalSpace : 1) * 100)
                        });

                        self.get('services').add(serviceModel);
                    });
                } else {
                    self.set('active', false);
                }

                deferred.resolve();
            },

            error: function() {
                var services =  new Backbone.Collection(),
                    resp = defaultData;

                self.set('services', services);

                self.set('active', resp.data.syncActivated);

                _.each(resp.data.syncServices, function(obj) {
                    var serviceModel = new Backbone.Model(obj);
                    self.get('services').add(serviceModel);
                });

                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    /**
     * Get ID of service in the list which was synced most recently. Although at this moment there is only one service.
     * If collection of services is empty use ID of default service - "dropbox"
     *
     * @return {String} ID of default service, e.g. 'dropbox', 'googledrive'
     */
    getServiceID: function() {
        var serviceID = 'dropbox';

        // Sort services by LastSyncDate parameter Descending
        var sortedServices = _.extend({},
            new Backbone.Collection({comparator: function (service1, service2) {
                // lsd = Last Sync Date :)
                // "10.12.2012, 15:45" <-> "DD.MM.YYYY, H:m"
                var dateFormat = "DD.MM.YYYY, H:m";

                var lsd1 = +moment(service1.get("lastSyncDate"), dateFormat).valueOf(),
                    lsd2 = +moment(service2.get("lastSyncDate"), dateFormat).valueOf();

                return lsd2 - lsd1; // we have to have recently synced at the top, thus such sequence of operands
            }}),
            this.get('services')
        );

        sortedServices.each(function(model) {
            serviceID = model.get('id');
        });

        return serviceID;
    },

    /**
     * Authorizes backup service that have 'default' value as true.
     * Sends user to authorization page
     */
    authorize: function() {
        var self = this;

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                service: "com.swisscom.stargate/ws/cloud/auth.com.swisscom.stargate.cloud.authorization",
                method: "getServiceAuthUrl",
                parameters: {
                    data: JSON.stringify({
                        serviceId: self.getServiceID(),
                        domain: window.location.hostname
                    })
                }
            },

            success: function (response) {
                response = JSON.parse(response.status);
                if (response.status !== 200) {
                    var alertBox = $('<div/>', {
                        'class': 'modal cloud-error'
                    });
                    alertBox.html(response.error);
                    alertBox.modal('show');
                    alertBox.on('hidden', function() {
                        alertBox.remove();
                    });
                } else {
                    document.location.href = response.data.authUrl;
                }
            }
        });
    },

    /**
     * Check oauth credentials on access to cloud service
     */
    checkCredentials: function(){
        var self = this;

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                service: "com.swisscom.stargate/ws/cloud/auth.com.swisscom.stargate.cloud.authorization",
                method: "validateDefaultServiceCredentials",
                parameters: {}
            },

            success: function(response) {
                response = JSON.parse(response.status);
                if (response.status === 200) {
                    window.location.reload();
                } else {
                    self.authorize();
                }
            }
        });
    },

    validateFolder: function (partitionUUID, folderName) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                service: "com.swisscom.stargate/ws/cloud/backup.com.swisscom.stargate.cloud.backup",
                method: "validateSyncFolder",
                parameters: {
                    data: JSON.stringify({
                        partitionUUID: partitionUUID,
                        folderName: folderName
                    })
                }
            },

            success: function(response) {
                var status;

                // Check if AP has successfully started
                if (response && response.status) {
                    try {
                        status = JSON.parse(response.status);
                        if (status && status.success === true) {
                            deferred.resolve(status);
                            return;
                        }
                    } catch (e) {
                        // nothing to do
                    }
                }
                deferred.reject(status);
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    }
});
;swc.constructors.СloudServicesStatus = swc.BaseModel.extend({

    url: '/ws',

    ajaxParameters: JSON.stringify({
        service: 'com.swisscom.stargate/ws/cloud/backup.com.swisscom.stargate.cloud.backup',
        method: 'cloudSyncEnabled',
        parameters: JSON.stringify({})
    }),

    /**
     * Using try {} catch {}, because if AP is disabled, it will return not valid json
     * @param response {Object}
     * @example { "status": "{\"data\":{\"isEnabled\":true},\"status\":200}" }
     * @returns {Object}
     */
    apiToJSON: function(response) {
        var result = {
                isEnabled: false
            },
            parsedResponse;

        try {
            parsedResponse = JSON.parse(response.status);

            if (parsedResponse.status === 200 && !_.isUndefined(parsedResponse.data)) {
                result.isEnabled = !_.isUndefined(parsedResponse.data.isEnabled) ? parsedResponse.data.isEnabled : false;
            }
        } catch (e) {
            // we have default values already
        }

        return result;
    },

    sync: function(method, options) {
		if(swc.Utils.DeviceType === 'default') {
			this.sync = (new swc.BaseModel()).sync;
			return this.sync(method, options);
		}
    }

});
;swc.constructors.MediaServices = swc.BaseModel.extend({

    url: '/ws',

    ajaxParameters: JSON.stringify({
        service: 'com.swisscom.stargate/ws/dlna.com.swisscom.stargate.dlna',
        method: 'getMediaStreamingState',
        parameters: {}
    }),

    /**
     * Using try {} catch {}, because if AP is disabled, it will return not valid json
     * @param response {Object}
     * @returns {Object}
     */
    apiToJSON: function(response) {
        var result = {
                status: false
            },
            parsedResponse;

        try {
            parsedResponse = JSON.parse(response.status);

            if (parsedResponse.status === 200) {
                result.status = parsedResponse.data.mediaStreamingEnabled;
            }
        } catch (e) {
            // we have default values already
        }

        return result;
    },

    toJSON: function() {
        return JSON.stringify({
            service: "com.swisscom.stargate/ws/dlna.com.swisscom.stargate.dlna",
            method: "setMediaStreamingState",
            parameters: {
                isEnabled: this.get('status')
            }
        });
    }

});
;swc.constructors.StorageProtection = Backbone.Model.extend({

    url: '/ws',

    ajaxParameters: JSON.stringify({
        service: 'com.swisscom.stargate/ws/nas/fileservices.com.swisscom.stargate.nas.fileservices',
        method: 'getCredentialsForUserShares',
        parameters: JSON.stringify({})
    }),

    /**
     * Using try {} catch {}, because if AP is disabled, it will return not valid json
     * @param response {Object}
     * @returns {Object}
     */
    apiToJSON: function(response) {
        var result = {
                status: false
            },
            parsedResponse;

        try {
            parsedResponse = JSON.parse(response.status);

            if (parsedResponse.success) {
                result.status = _.findWhere(parsedResponse.fileServices, { service: 'AFP' }).enabled;
            }
        } catch (e) {
            // we have default values already
        }

        return result;
    },

    toJSON: function() {
        return JSON.stringify({
            service: "com.swisscom.stargate/ws/nas/fileservices.com.swisscom.stargate.nas.fileservices",
            method: "updateCredentialsForUserShares",
            parameters: {
                data: JSON.stringify({
                    service: 'AFP',
                    enabled: this.get('status')
                })
            }
        });
    },

    sync: function(){
        var self = this,
            deferred = new $.Deferred();

        $.when(
            self.getConfig()
        ).done(function(){
            self.trigger('change');
            deferred.resolve();
        });

        return deferred.promise();
    },
    getConfig: function() {
        var self = this,
            deferred = new $.Deferred(),
            method = "getCredentialsForUserShares";

        var userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin();

        if(userIsSuperAdmin === true) {
            method = "getRestrictedCredentialsForUserShares";
        }

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                service: "com.swisscom.stargate/ws/nas/fileservices.com.swisscom.stargate.nas.fileservices",
                method: method,
                parameters: {}
            },
            success: function(response) {
                var responseObj = $.parseJSON($.parseJSON(response.status));
                // if(userIsSuperAdmin !== true) {
                self.set('protection', responseObj);
                // }
                self.set('enabled', responseObj.enabled);
                deferred.resolve();
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },
    updateConfig: function(params) {
        var deferred = new $.Deferred();
        var userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin();
        var method = "updateCredentialsForUserShares";
        var parameters = {
                    enabled: params.enabled,
                    username: params.username,
                    password: params.password
        };

        if(userIsSuperAdmin === true) {
            method = "changeStateCredentialsForUserShares";
            parameters = {
                enabled: params.enabled
            };
        }

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                service: "com.swisscom.stargate/ws/nas/fileservices.com.swisscom.stargate.nas.fileservices",
                method: method,
                parameters: parameters
            },
            success: function() {
                deferred.resolve();
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    validation: {
        username: function(data) {
            var username = "",
                result = {
                    status: true,
                    messages: []
                };

            var userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin();

            if(userIsSuperAdmin === true) {
                return result;
            }

            $.each(data, function(key, value) {
                if (value.parameterName === "storage_protection_username") {
                    username = value.parameterValue;
                }
            });

            if (swc.models.StorageProtection.get('enabled')) {
                result = swc.Utils.DefaultValidator.username(username);
            }
            return result;
        },

        password: function(data) {
            var username = "",
                password = "",
                result = {
                    status: true,
                    messages: []
                };

            var userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin();

            if(userIsSuperAdmin === true) {
                return result;
            }

            $.each(data, function(key, value) {
                if (value.parameterName === "storage_protection_username") {
                    username = value.parameterValue;
                }
                if (value.parameterName === "storage_protection_password") {
                    password = value.parameterValue;
                }
            });

            if (swc.models.StorageProtection.get('enabled')) {
                return swc.Utils.DefaultValidator.password(username, password);
            }
            return result;
        }
    }
});
;swc.constructors.RemoteNASAccess = swc.BaseModel.extend({

    url: '/ws',

    ajaxParameters: JSON.stringify({
        service: 'com.vestiacom.rnas/com/vestiacom/rnas.com.vestiacom.rnas',
        method: 'GetEnabled',
        parameters: {}
    }),

    /**
     * Using try {} catch {}, because if AP is disabled, it will return not valid json
     *
     * @example
     * <code>
     *     // in case of error:
     *     response = {"status": "{\"status\":\"ERROR\"}"}
     *     // or in case of success:
     *     {"status": true }
     * </code>
     *
     * @param response {Object}
     *
     * @returns {Object}
     */
    apiToJSON: function(response) {
        var result = {
                "status": false
            },
            responseParsed;

        if(response.error) {
            try {
                responseParsed = JSON.parse(response.status);
                if (_.isObject(responseParsed)) {
                    result = responseParsed;
                }
            } catch (e) {
                // do nothing
            }
        } else {
            result = response;
        }

        return result;
    },

    toJSON: function() {
        return JSON.stringify({
            service: "com.vestiacom.rnas/com/vestiacom/rnas.com.vestiacom.rnas",
            method: "SetEnabled",
            parameters: {
                enable: this.get('status')
            }
        });
    },

    isEnabled: function() {
        return this.get('status') === true;
    }
});
;swc.constructors.RemoteNASAccount = swc.BaseModel.extend({

    url: {
        'delete': '/ws'
    },

    toJSON: function() {
        return JSON.stringify({
            service: 'com.vestiacom.rnas/com/vestiacom/rnas.com.vestiacom.rnas',
            method: 'CancelPairing',
            parameters: {
                'email': this.get('email')
            }
        });
    }

});
;swc.constructors.UsbStatus = swc.BaseModel.extend({

    url: '/ws',

    ajaxParameters: JSON.stringify({
        service: 'com.swisscom.stargate/ws/nas/devices.com.swisscom.stargate.nas.devices',
        method: 'getUsbMode',
        parameters: JSON.stringify({})
    }),

    /**
     * Using try {} catch {}, because if AP is disabled, it will return not valid json
     *
     * @description
     *
     *  AP returns following response: { "status": { "data": { "mode" : 3 || 2 }, "status": 200 }}, where mode is current
     *  USB mode. Can be 2 or 3.
     *
     * @param response {Object}
     * @returns {Object}
     */
    apiToJSON: function(response) {
        var result = {
                status: false
            },
            parsedResponse;

        try {
            parsedResponse = JSON.parse(response.status);

            if (parsedResponse.data.mode) {
                result.status = parsedResponse.data.mode === 3;
            }
        } catch (e) {
            // we have default values already
        }

        return result;
    },

    /**
     * @description:
     *
     *  To enable usb 3.0 or 2.0, it's needed to send 3 or 2 as current mode parameter:
     *
     * @returns {*}
     */
    toJSON: function() {
        return JSON.stringify({
            service: "com.swisscom.stargate/ws/nas/devices.com.swisscom.stargate.nas.devices",
            method: "setUsbMode",
            parameters: {
                mode: this.get('status') === true ? 3 : 2
            }
        });
    }

});
;swc.constructors.Screen = Backbone.Model.extend({

    responseHandles : [],
    sync: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        $.when(self.getParameters(fromListener))
            .done(function(){
                deferred.resolve();
            });

        return deferred.promise();
    },

    getLanguage: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Screen:getLanguage',
            fromListener: fromListener,

            data: {
                "parameters": {}
            },

            success: function(response) {
                self.set('lang', response.status || 'EN');

                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    getBrightTime: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Screen:getBrightTime',
            fromListener: fromListener,

            data: {
                "parameters": {}
            },

            success: function(response) {
                self.set('brightTime', response.status || 60);

                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    getShowPassword: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Screen:getShowWifiPassword',
            fromListener: fromListener,

            data: {
                "parameters": {}
            },

            success: function(response) {
                self.set('showPass', response.status || false);

                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    getShowGuestPassword: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Screen:getShowGuestPassword',
            fromListener: fromListener,

            data: {
                "parameters": {}
            },

            success: function(response) {
                self.set('showGuestPass', response.status || false);

                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    setLanguage: function(){
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Screen:setLanguage',
            data: {
                "parameters":{
                    "language" : self.get('lang')
                }
            },

            success: function(data) {
                self.responseHandles.push(data);
                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    setBrightTime: function(){
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Screen:setBrightTime',
            data: {
                "parameters":{
                    "time" : self.get('brightTime')
                }
            },

            success: function(data) {
                self.responseHandles.push(data);
                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    setShowPassword: function(){
        var self = this, deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Screen:setShowWifiPassword',
            data: {
                "parameters":{
                    "enable" : self.get('showPass')
                }
            },

            success: function(data) {
                self.responseHandles.push(data);
                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    setShowGuestPassword: function(){
        var self = this, deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Screen:setShowGuestPassword',
            data: {
                "parameters":{
                    "enable" : self.get('showGuestPass')
                }
            },

            success: function(data) {
                self.responseHandles.push(data);
                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    getParameters: function(fromListener){
        var self = this, deferred = new $.Deferred();

        $.when(self.getBrightTime(fromListener), self.getLanguage(fromListener), self.getShowPassword(fromListener), self.getShowGuestPassword(fromListener))
            .done(function(){
                deferred.resolve();
            });

        return deferred.promise();
    },

    setParameters: function(){
        var self = this,
            deferred = new $.Deferred();

        self.responseHandles=[]; //clear responses

        $.when(self.setBrightTime(), self.setLanguage(), self.setShowPassword(), self.setShowGuestPassword())
            .done(function(){
                var connectioErrorFlag=false;
                $(self.responseHandles).each(function(){
                    if(this.error === 196618){
                       connectioErrorFlag=true;
                    }
                });

                if(connectioErrorFlag){
                    deferred.reject("connection");
                } else {
                    deferred.resolve();
                }
            });

        return deferred.promise();
    }
});

;swc.constructors.Call = swc.BaseModel.extend({

    url: {
        'delete': '/sysbus/Phonebook:removeContactByUniqueID'
    },

    defaults: function() {
        return {
            "formattedName": ''
        };
    },

    toJSON: function() {
        var json;

        json = {
            parameters: {
                uniqueID: this.get('id'),
                contact: {
                    formattedName: this.get('formattedName')
                }
            }
        };

        return JSON.stringify(json);
    }

});;swc.constructors.PhonebookContact = swc.BaseModel.extend({

    url: {
        // keys quoted uniformly, but the reason is 'delete'
        // because IE thinks it's a key word
        'create': '/sysbus/Phonebook:addContactAndGenUUID',
        'update': '/sysbus/Phonebook:modifyContactByUniqueID',
        'delete': '/sysbus/Phonebook:removeContactByUniqueID'
    },

    phoneSet: ['HOME', 'CELL', 'WORK'],

    defaults: function() {
        var telephoneNumbers = [];

        _.each(this.phoneSet, function(phoneType) {
            telephoneNumbers.push({
                "type": phoneType,
                "preferred": false,
                "name": ''
            });
        });

        return {
            "firstName": '',
            "lastName": '',
            "formattedName": '',
            "telephoneNumbers": telephoneNumbers
        };
    },

    toJSON: function() {
        var json;

        var rawNumbers = this.get("telephoneNumbers"),
            telephoneNumbers = [
                _.where(rawNumbers, {'type': 'HOME'})[0],
                _.where(rawNumbers, {'type': 'CELL'})[0],
                _.where(rawNumbers, {'type': 'WORK'})[0]
            ];

        telephoneNumbers = _.compact(telephoneNumbers);

        _.each(telephoneNumbers, function(number){
            delete number.key;
        });

        json = {
            parameters: {
                uniqueID: this.get('id'),
                contact: {
                    name: this.getSerializedName(),
                    formattedName: this.getFormattedName(),
                    key: this.get('key'),
                    telephoneNumbers: telephoneNumbers
                }
            }
        };

        return JSON.stringify(json);
    },

    /**
     * formattedName - name that you see on phonebook.
     *
     * @example
     *      "formattedName": "Teste Mr",
     */
    getFormattedName: function() {
        return this.get('lastName').trim() + ' ' + this.get('firstName').trim();
    },

    /**
     * name - contact name in special format "N:"+ last name + ";" + first name.
     *
     * @example
     *      "name": "N:Mr;Teste;",
     */
    getSerializedName: function() {
        var firstName = this.get('firstName').trim().replace(/\\/g, '\\\\').replace(/;/g, '\\;');
        var lastName = this.get('lastName').trim().replace(/\\/g, '\\\\').replace(/;/g, '\\;');
        return 'N:' + lastName + ';' + firstName;
    }

});
;swc.constructors.Telephony = Backbone.Model.extend({

    dectInternalNumbers: [ 203, 204, 205, 206, 207, 208, 209 ],
    sipInternalNumbers: [ 210, 211, 212, 213, 214, 215, 216, 217, 218, 219 ],

    autoSync: true,

    // lastFailOverLoop: undefined,

    sync: function(fromListener, failOverLoop){
        var self = this,
            deferred = new $.Deferred(),
            timeout = 5000;

        if(self.autoSync===false && _.isUndefined(failOverLoop)){
            return deferred.resolve();
        }

        if(!_.isUndefined(failOverLoop)) {
            timeout = 10000;
        }

        $.when(
                self.getPin(fromListener, timeout),
                self.getPhones(fromListener, timeout),
                self.getDectStatus(fromListener, timeout),
                self.getDectEco(fromListener, timeout)
            ).done(function() {
                if(_.isUndefined(failOverLoop)) {
                    self.set('errorStatus', false);
                    deferred.resolve();
                } else {
                    deferred.resolve(true);
                }
            }).fail(function() {
                if(_.isUndefined(failOverLoop)) {
                    self.set('errorStatus', true);
                    deferred.resolve();
                } else {
                    deferred.resolve(false);
                }
            });

        return deferred.promise();
    },

    initialize: function(){
        var self = this,
            PhonesCollectionConstructor = Backbone.Collection.extend({
                comparator: function(model){
                    return model.id;
                }
            }),
            phonesCollection = new PhonesCollectionConstructor(),
            groupsCollection = new Backbone.Collection();

        this.set('groups', groupsCollection);
        this.set('phones', phonesCollection);
        this.get('phones').on('change:externalNumber', function(model, values, options){
            if(!options || (options && !options.notProceed)){
                var newNumber = model.get('externalNumber');
                var numberFound = false;
                self.get('voip').each(function(voipModel){
                    if(voipModel.get('directoryNumber') === newNumber){
                        numberFound = true;
                        model.set({'outgoingTrunkLine': voipModel.id}, {notProceed: true});
                    }
                });
                if(!numberFound){
                    model.set({'externalNumber': self.get('voip').get(model.get('outgoingTrunkLine')).get('directoryNumber')}, {notProceed: true});
                }
            }

        });

        this.get('phones').on('change:outgoingTrunkLine', function (model, values, options) {
            if(!options || (options && !options.notProceed)){
                model.set({'externalNumber': self.get('voip').get(model.get('outgoingTrunkLine')).get('directoryNumber')}, {notProceed: true});
            }
        });

        this.get('phones').on('change:incomingLines', function (model, values) {
            var activeLines = self.get('voip').where({'enable': 'Enabled'});

            _.each(activeLines, function(line){
                var lineGroup = self.get('groups').get(line.get('groupId')),
                    devices = lineGroup.get('ep_names'),
                    deviceOptions = _.where(values, {'line': line.id})[0],
                    changed = false;

                if(deviceOptions.value){
                    if(!_.contains(devices, model.id)){
                        devices.push(model.id);
                        changed = true;
                    }
                } else {
                    if(_.contains(devices, model.id)){
                        devices = _.without(devices, model.id);
                        changed = true;
                    }
                }

                if(changed){
                    lineGroup.set({'ep_names': devices, 'changed': true});
                }
            });
        });
    },

    validation: {

        dectPin: function(data) {
            var validationMessages = [],
                validationData = swc.Utils.getDataToValidate({
                    'dectPin' : { elementName: 'pin', elementClass: 'pin-original' }
                }, data);

            if ($.trim(validationData['dectPin']).length < 4) {
                validationMessages.push('must contain 4 symbols');
            }

            if (!/^[\d]*$/.test(validationData['dectPin'])) {
                validationMessages.push('must contain 4 digits from 0 to 9');
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        dectName: function(data) {
            var validationMessages = [],
                validationData = swc.Utils.getDataToValidate({
                    'dectName' : { elementName: 'dect-name-editable' }
                }, data);

            if (_.isEmpty($.trim(validationData['dectName']))) {
                validationMessages.push('must not be empty');
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        sipName: function(data) {
            var validationMessages = [],
                validationData = swc.Utils.getDataToValidate({
                    'sipName' : { elementName: 'sip-name' }
                }, data);

            if (_.isEmpty($.trim(validationData['sipName']))) {
                validationMessages.push('must not be empty');
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        internalNumber: function(data) {
            var validationMessages = [],
                validationData = swc.Utils.getDataToValidate({
                    'internalNumber' : { elementName: 'internal-number-editable' }
                }, data);

            if (_.isUndefined(swc.models.Telephony)) {
                return {
                    status: _.isEmpty(validationMessages),
                    messages: validationMessages
                };
            }

            var phones = swc.models.Telephony.get('phones'),
            // get integer array of used phone numbers
                usedInternalNumbers = _.map(phones.pluck('directoryNumber'), function(v) { return +v; }),
            // get integer array of similar phone numbers
                similarValues = _.filter(usedInternalNumbers, function(v) { return v === validationData['internalNumber']; });

            // if the length of similar phone numbers > 1 then error should be shown
            if (similarValues.length > 1) {
                validationMessages.push('number already is used');
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        }
    },
    /**
     * Update device list from event
                     * directoryNumber "203"
                     * enable "Enabled"
                     * endpointType "DECT CAT-iq 2.0"
                     * line "HS0171A0B3C9"
                     * name "DECT203"
                     * outgoingTrunkLine "LINE1"
     */
    updateDeviceList: function(devices){
        var self = this;

        if(!_.isUndefined(devices.length) && devices.length > 0) {
            swc.models.Telephony.set('dectStatus', true);
        }

        _.each(devices, function(device) {
            device=device;

            if(device.Active===true){
                //we have to alter the device object becouse of the change in NP 6.0.0.2d vield names are now camelcase starting letter changed to uppercase
                device.directoryNumber= device.DirectoryNumber;
                device.enable= "Enabled";
                device.endpointType=device.EndpointType;
                device.line=device.Key;
                device.name=device.Name;
                device.outgoingTrunkLine=device.OutgoingTrunkLine;
                self.attributes.phones.add(self.prepareTelephonyDeviceModel(device));
            }
            else{
              _.each(self.attributes.phones.models,function(who){
                  if(who.attributes.id===devices[0].Key){
                      self.attributes.phones.remove(who);
                  }
              });
              
            }
        });
    },
    /**
     * Get voip status
     * @returns json
     * {
     *  "status":[
     *      {
     *          "name":"SIP-Trunk",
     *          "signalingProtocol":"SIP",
     *          "enable":"Disabled",
     *          "trunk_lines":[
     *              {
     *                  "name":"LINE1",
     *                  "groupId":"Group1",
     *                  "enable":"Disabled",
     *                  "status":"Disabled",
     *                  "statusInfo":"",
     *                  "directoryNumber":"",
     *                  "uri":"",
     *                  "authUserName":"",
     *                  "authPassword":"",
     *                  "event_subscribe_lines":[
     *                      {
     *                          "eventSubscribeEvent":"message-summary",
     *                          "eventSubscribeAuthUserName":"",
     *                          "eventSubscribeAuthPassword":""
     *                      }
     *                  ]
     *              },
     *              {
     *                  "name":"LINE2",
     *                  "groupId":"Group2",
     *                  "enable":"Disabled",
     *                  "status":"Disabled",
     *                  "statusInfo":"",
     *                  "directoryNumber":"",
     *                  "uri":"",
     *                  "authUserName":"",
     *                  "authPassword":"",
     *                  "event_subscribe_lines":[
     *                      {
     *                          "eventSubscribeEvent":"message-summary",
     *                          "eventSubscribeAuthUserName":"",
     *                          "eventSubscribeAuthPassword":""
     *                      }
     *                  ]
     *              }
     *          ],
     *          "sip":{
     *              "proxyServer":"imst.swisscom.ch",
     *              "proxyServerPort":5060,
     *              "registrarServer":"imst.swisscom.ch",
     *              "registrarServerPort":5060,
     *              "outboundProxyServer":"",
     *              "outboundProxyServerPort":5060,
     *              "userAgentDomain":"imst.swisscom.ch",
     *              "userAgentPort":5060,
     *              "subscriptionInfo":[
     *                  {
     *                      "event":"message-summary",
     *                      "notifyServer":"",
     *                      "notifyServerPort":5060
     *                  }
     *              ]
     *          },
     *          "h323":{}
     *      },
     *      {
     *          "name":"H323-Trunk",
     *          "signalingProtocol":"H.323",
     *          "enable":"Disabled",
     *          "trunk_lines":[
     *              {
     *                  "name":"LINE3",
     *                  "groupId":"Group3",
     *                  "enable":"Disabled",
     *                  "status":"Disabled",
     *                  "statusInfo":"",
     *                  "directoryNumber":"1011",
     *                  "uri":"",
     *                  "authUserName":"",
     *                  "authPassword":"",
     *                  "event_subscribe_lines":[]
     *              }
     *          ],
     *          "sip":{},
     *          "h323":{
     *              "gatekeeper":"172.16.1.1",
     *              "gatekeeperPort":1719
     *          }
     *      }
     *  ]}
     */
    getVoipStatus: function(fromListener){
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/VoiceService/VoiceApplication:listTrunks',
            fromListener: fromListener,
            data: {
                "parameters": {}
            },

            success: function(response) {
                var swisscomSip = _.where(response.status, {signalingProtocol: "SIP"})[0],
                    Lines = Backbone.Collection,
                    lines = new Lines(),
                    voipActivated;

                // NOTE: we do not take global SIP status - swisscomSip.enable. We take it from the lines status

                if (swisscomSip) {
                    _.each(swisscomSip.trunk_lines, function(line){
                        var sipLine = new Backbone.Model();
                        sipLine.set({
                            id: line.name,
                            directoryNumber: line.directoryNumber,
                            enable: line.enable,
                            groupId: line.groupId,
                            name: line.name,
                            status: line.status,
                            statusInfo: line.statusInfo,
                            uri: line.uri
                        });

                        if(sipLine.get('enable') === 'Enabled'){
                            lines.add(sipLine);
                        }
                    });
                }

                // Check if lines collection is not empty
                voipActivated = (lines.length > 0);

                // voipStatus means that at least one line is activated
                self.set({
                    'voip': lines,
                    "voipStatus": voipActivated
                });

                deferred.resolve();
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    getSipExtensionsStatus: function(fromListener){
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/VoiceService/VoiceApplication:getSipExtensionsStatus',
            fromListener: fromListener,
            data: {
                "parameters": {}
            },

            success: function(response) {
                self.set({'sipExtensionsStatus': response.status});
                deferred.resolve();
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    /**
     * Get groups.
     * Used for incoming numbers - Calls to related lines(LINE1 is related to Group1) will redirect to devices listed in "ep_names" array
     * @returns {*}
     *
     * @example
     * {"status":[
     *      {
     *          "group_id":"Group1",           – Group name, related to line name. Group1 = LINE1 etc...
     *          "ep_names":["FXS1","FXS2"]     – List of devices in group
     *      },{
     *          "group_id":"Group2",
     *          "ep_names":["FXS2"]
     *      }
     * ]}
     */
    getGroups: function(fromListener){
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/VoiceService/VoiceApplication:listGroups',
            fromListener: fromListener,
            data: {
                "parameters": {}
            },

            success: function(response) {
                var groupCol = [];
                _.each(response.status, function(group){
                    var model = {
                        id: group.group_id,
                        ep_names: group.ep_names,
                        changed: false
                    };
                    groupCol.push(model);
                });
                self.get('groups').reset(groupCol);
                deferred.resolve();
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    setGroups: function(){
        var self = this,
            deferred = new $.Deferred(),
            groupArray = [];

        self.get('groups').each(function(model){
            if(model.get('changed')){
                var obj = {
                    "group_id": model.id,
                    "ep_names": model.get('ep_names')
                };
                groupArray.push(obj);
            }
        });

        swc.models.Rest.sendRequest({
            url: '/sysbus/VoiceService/VoiceApplication:setGroups',
            data: {
                "parameters": {
                    "groups": groupArray

                }
            },

            success: function() {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    /**
     * Get voip lines
     * @returns json
     * {
     *      "status":[
     *          {
     *              "line":"FXS1",
     *              "name":"FXS1",
     *              "enable":"Enabled",
     *              "directoryNumber":"201",
     *              "endpointType":"FXS",
     *              "outgoingTrunkLine":"LINE1"
     *          },
     *          {
     *              "line":"HS0106EA54BB",
     *              "name":"HS0106EA54BB",
     *              "enable":"Enabled",
     *              "directoryNumber":"204",
     *              "endpointType":"DECT GAP",
     *              "outgoingTrunkLine":"LINE1"
     *          },
     *          {
     *              "line":"HS024270E3C1",
     *              "name":"HS024270E3C1",
     *              "enable":"Enabled",
     *              "directoryNumber":"205",
     *              "endpointType":"DECT CAT-iq 1.0",
     *              "outgoingTrunkLine":"LINE1"
     *          },
     *          {
     *              "line":"Account311",
     *              "name":"Account311",
     *              "enable":"Disabled",
     *              "directoryNumber":"311",
     *              "endpointType":"SIP",
     *              "outgoingTrunkLine":"LINE1"
     *          }
     *      ]
     *  }
     */
    getPhones: function(fromListener, timeout){
        var self = this,
            deferred = new $.Deferred();

        function getLines(){
            var deferred = new $.Deferred();

            swc.models.Rest.sendRequest({
                url: '/sysbus/VoiceService/VoiceApplication:listHandsets',
                fromListener: fromListener,
                timeout: timeout,
                data: {
                    "parameters": {}
                },

                success: function(response) {
                    var lines = [];

                    _.each(response.status, function(obj){
                        if(obj.endpointType.indexOf('DECT')>=0 || obj.endpointType === "FXS" || obj.endpointType === 'SIP'){
                            //////////
                            lines.push(self.prepareTelephonyDeviceModel(obj));
                        }
                    });

                    self.get('phones').reset(lines);
                    deferred.resolve();
                },
                error: function() {
                    deferred.reject();
                }
            });

            return deferred.promise();
        }

        $.when(self.getVoipStatus(fromListener), self.getGroups(fromListener), self.getSipExtensionsStatus(fromListener)).done(function(){
            $.when(getLines(fromListener)).done(function(){
                deferred.resolve();
            });
        });

        return deferred.promise();
    },
    prepareTelephonyDeviceModel: function(obj){
        var self=this, line = new Backbone.Model(), deviceId = obj.line, incomingLines = [], deviceType = 'dect',
            activeLines = self.get('voip');
        activeLines.each(function(line){
            var key = line.id, value = false;
            var lineGroup = self.get('groups').get(line.get('groupId'));
            if(lineGroup){
                if(_.contains(lineGroup.get('ep_names'), deviceId)){
                    value = true;
                }
                incomingLines.push({'line': key, 'number': line.get('directoryNumber'), 'value': value});
            } else {
                /*
                 If groupId in lines and groupId in groups are not the same, the gateway setup is wrong. So we will show this modal window
                 */
                if (!$.cookie('noGroups')) {
                    $.cookie('noGroups', true);
                    (new swc.constructors.NoGroupsModal()).render();
                }
            }

        });

        if(obj.endpointType === "FXS"){
            deviceType = 'wired';
        } else if(obj.endpointType === 'SIP') {
            deviceType = 'sip';
        }

        line.set({
            id: deviceId,
            line: deviceId,
            deviceType: deviceType,
            handsetModel: obj.handsetModel || obj.HandsetModel,
            name: obj.name,
            enable: (obj.enable === "Enabled"),
            directoryNumber: obj.directoryNumber,
            externalNumber: self.get('voip').get(obj.outgoingTrunkLine) ? self.get('voip').get(obj.outgoingTrunkLine).get('directoryNumber') : "",
            incomingLines: incomingLines,
            type: 'DECT',
            outgoingTrunkLine: obj.outgoingTrunkLine,
            is_changed: false
        });


    return line;

    },
    ringPhone: function(phone){
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                "service": "Devices.Device." + phone,
                "method": 'ring',
                "parameters": {
                  
                }
            },

            success: function(response) {
                // TODO: put this error handler in the common place
                // in this case response can be successful and
                // contain error message. That is why it should be
                //  checked on it here
                if (_.isUndefined(response.errors)) {
                    setTimeout(function() {
                        deferred.resolve();
                    }, 1000);
                } else {
                    deferred.reject();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();

    },

    ringSip: function(phone){
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/VoiceService/VoiceApplication:ring',
            data: {
                "parameters": {
                    "line": phone,
                    "duration": 14000
                }
            },

            success: function() {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();

    },


    startPairing: function(){
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/DECT:startPairing',

            data: {
                "parameters": {
                }
            },

            success: function(response) {
                deferred.resolve(response);
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    stopPairing: function() {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/DECT:stopPairing',

            data: {
                "parameters": {
                }
            },

            success: function(response) {
                deferred.resolve(response);
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    unpairDect: function(id){
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/VoiceService/VoiceApplication:deleteHandset',
            data: {
                "parameters": {
                    "line": id
                }
            },

            success: function(response) {
                deferred.resolve(response);
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    unpairSip: function(id) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/VoiceService/VoiceApplication:setHandset',
            data: {
                "parameters": {
                    "line" : {
                        "line": id,
                        "enable": "Disabled"
                    }
                }
            },

            success: function(response) {
                deferred.resolve(response);
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();

    },

    /**
     * Get PIN for device pairing
     * @returns json
     * {"status":"0000"}
     */

    getPin: function(fromListener, timeout){
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/DECT:getPIN',
            fromListener: fromListener,
            timeout: timeout,
            data: {
                "parameters": {
                }
            },

            success: function(response) {
                self.set('dectPIN', response.status);
                deferred.resolve();
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    setPin: function(){
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/DECT:setPIN',
            data: {
                "parameters": {
                    "pin": self.get('dectPIN')
                }
            },

            success: function() {
                setTimeout(function(){
                    deferred.resolve();
                }, 1000);
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    phoneEdit: function(){

        var self = this, deferred = new $.Deferred(),
            changedModels = this.get('phones').filter(function(model){
                return model.get('is_changed');
            });

        var editsArray = [];

        if(changedModels.length === 0){
            deferred.resolve();
        }

        _.each(changedModels, function(model) {
            var editRequest = function() {
                var deferred = new $.Deferred(),
                    extNumber = "" + model.get('externalNumber'),
                    incomingLines = model.get('incomingLines'),
                    lines = _.where(incomingLines, {number: extNumber}),
                    outgoingTrunkLine = !_.isEmpty(lines) ? lines[0].line : model.get('outgoingTrunkLine');

                swc.models.Rest.sendRequest({
                    url: '/sysbus/VoiceService/VoiceApplication:setHandset',
                    data: {
                        "parameters": {
                            "line": {
                                "line": model.get('line'),
                                "directoryNumber": model.get('directoryNumber'),
                                "outgoingTrunkLine": outgoingTrunkLine,
                                "name": model.get('name')
                            }
                        }
                    },

                    success: function() {
                        deferred.resolve();
                    },
                    error: function() {
                        deferred.reject();
                    }
                });

                return deferred.promise();
            };

            editsArray.push(editRequest());
        });

        $.when.apply(this, editsArray)
            .done(function () {
                $.when(self.setGroups())
                    .done(function () {
                        setTimeout(function () {
                            deferred.resolve();
                        }, 2000);
                    });
                });

        return deferred.promise();
    },

    dectReset: function(){
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/DECT:reset',
            data: {
                "parameters": {
                }
            },

            success: function() {
                deferred.resolve();
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    dectRemoveDects: function(){
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/DECT:removeAllHandsets',
            data: {
                "parameters": {
                }
            },

            success: function() {
                deferred.resolve();
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    setDectStatus: function(){
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/DECT:setBaseState',
            data: {
                "parameters": {
                    "state": self.get('dectStatus')
                }
            },

            success: function() {
                // Give DECT a little time to turn on
                setTimeout(function(){
                    deferred.resolve();
                }, self.get('dectStatus')?2000:1);
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();

    },

    getDectStatus: function(fromListener, timeout){
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/DECT:getBaseState',
            fromListener: fromListener,
            timeout: timeout,
            data: {
                "parameters": {}
            },

            success: function(response) {
                self.set('dectStatus', response.status);
                deferred.resolve();
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    setDectEco: function(){
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/DECT:setNEMoState',
            data: {
                "parameters": {
                    "state": self.get('dectEco')
                }
            },

            success: function() {
                setTimeout(function(){
                    deferred.resolve();
                }, 1500);
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();

    },

    getDectEco: function(fromListener, timeout){
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/DECT:getNEMoState',
            fromListener: fromListener,
            timeout: timeout,
            data: {
                "parameters": {}
            },

            success: function(response) {
                self.set('dectEco', response.status);
                deferred.resolve();
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();

    },

    getPhonesList: function() {
        var dectPhonesArray = [],
            wiredPhonesArray = [],
            sipPhonesArray = [],
            wiredKey = 1;

        this.get('phones').each(function(model, index) {
            var obj = {
                key: (model.get('deviceType') === "wired")? wiredKey++ : index+1,
                id: model.id,
                isDisabled: !model.get('enable'),
                name: model.get('name'),
                handsetModel: model.get('handsetModel'),
                internalNumber: model.get('directoryNumber'),
                externalNumber: model.get('externalNumber'),
                incomingLines: model.get('incomingLines')
            };

            if (model.get('deviceType') === "dect") {
                dectPhonesArray.push(obj);
            } else if(model.get('deviceType') ==='sip') {
                sipPhonesArray.push(obj);
            } else if(model.get('deviceType') === 'wired') {
                wiredPhonesArray.push(obj);
            }

            dectPhonesArray.sort(function (a, b) {
                if (b.internalNumber < a.internalNumber) {
                    return 1;
                }

                return -1;
            });
        });

        return {
            dect: dectPhonesArray,
            cord: wiredPhonesArray,
            sip: sipPhonesArray
        };
    }
});

;swc.constructors.ApplicationsOverviewView = swc.base.PageView.extend({

    className: 'applications-ap',

    models: [
        'Applications',
        'ApplicationCollection',
        'SchedulerAp',
        'apServiceState',
        'apServiceLoadingState',
        'System'
    ],

    disallowedDevices: ['starlink', 'starlite'],

    events: {
        'swc-switcher:change .set-ap-state': 'setApState'
    },

    listenerInterval: 10,

    listenerEnabled: true,

    listenerPages: [ 'applications' ],

    setTemplateData: function(){
        var self = this;
        this.templateData = {
            applications: self.filterApplications(),
            apServiceState: swc.models.apServiceState.get('status')
        };
    },

    filterApplications: function() {
        var collection = [];

        collection = swc.models.ApplicationCollection.filter(function(model) {
            var response = false;

            if (model.get('deleted')) {
                response = false;
            } else {
                response = true;
            }

            return response;
        });

        return collection;
    },

    renderComplete: function() {
        this.updatePageUI();
    },

    onListenerComplete: function() {
        this.stopPageLoading();
        this.updatePageUI();
    },

    updatePageUI: function() {
        var pageState = true,
            apState = swc.models.apServiceState.get('status'),
            apServiceLoadingState = swc.models.apServiceLoadingState.get('isLoading'),
            apStateSwitcher = this.$el.find('.set-ap-state');

        // Check if AP is loading or OFF now, and disable tabs and page content
        if (apState !== true || apServiceLoadingState === true) {
            pageState = false;
        }

        // If AP has started -> fully reload page: (because of child view is not updating)
        if (pageState === true && this.isPageEnabled === false) {
            this.stopListener();
            this.render();
        } else {
            // Disable / Enable page and tabs based on page status:
            this.setPageState(pageState);
            this.setTabsState(pageState);

            // Check if AP is OFF now and show disabled message:
            if (apState === false) {
                this.showDisabledMessage({
                    container: '.apStateOffMessage',
                    className: 'above'
                });
            }

            // Check if AP is loading now and show loading message:
            if (apServiceLoadingState === true && apState === true) {
                this.showDisabledMessage({
                    container: '.apStateLoadingMessage',
                    className: 'above'
                });
            }

            // Set correct switcher position:
            apStateSwitcher.trigger('swc-switcher:swc-change', apState);

            // Start listener if page is disabled or loading:
            this.listenerEnabled = true;
            this.startListener();
        }
    },

    setApState: function(e, value) {
        var self = this;

        this.stopListener();
        this.listenerEnabled = false;

        this.showPageLoading("Updating Central Storage state");

        // Update parameter in the model:
        swc.models.apServiceState.set('status', value);

        // Start listener again when data saved:
        $.when(swc.models.apServiceState.sync('update')).done(function() {
            swc.models.apServiceState.fetch();
            self.listenerEnabled = true;
            self.startListener();
        });
    }
});
;swc.constructors.ApplicationsDyndnsView = swc.base.PageView.extend({

    className: 'application-dyndns',

    pageTemplateID: 'applications:dyndns',

    allowedMods: ['expert'],

    disallowedDevices: ['starlink', 'starlite'],

    models: ['Network', 'SwisscomDyndns'],

    events: {
		'click .add-account': 'showAddAccount',
		'swc-checkbox:change .swc-checkbox.DynDNSState': 'setDynDNSState',
		'click .do-delete': 'removeAccount',
		'validation-fail input.host-name': 'validationFail',
        'validation-success input.host-name': 'validationSuccess',
        'keyup .host-name': 'change'
    },

    validation: {
		'host-name': 'SwisscomDyndns:name'
    },

    page: null,
    validationError: null,
    previousName: null,
    deletedAccount: false,
    serverError: false,
    block: false,
    remainEnabled: false,

    validationSuccess: function() {
		this.buttonsVisibility();
		this.hideValidationError();
    },

    validationFail: function() {
		//this.deactivateButtons();
		this.showValidationError();
    },

    registerValidationListener: function($element) {
        if (!$element.size()) {
            return;
        }

        $element.on('validation-success', function() {
            $element.toggleClass('valid', !_.isEmpty($element.val()));
        });

        $element.on('validation-fail', function() {
            $element.removeClass('valid');
        });
    },


    setTemplateData: function() {
		var status = swc.models.Network.get('status');
		this.templateData = {
			externalIP: swc.models.Network.getParameter('RemoteIPAddress', 'status'),
			dyndns: swc.models.SwisscomDyndns.attributes,
			validationError: this.validationError,
			serverError: this.serverError,
			networkStatus: _.isUndefined(status) ? 'NOT_CONNECTED' : status.get('ConnectionMode')
		};
    },

    revertState: function(){
		var self = this;

		if(!this.block) {
			$('.dyndns-block').toggleClass('disabled');
			$('.DynDNSState').toggleClass('checked');
		}

		this.block = true;

		setTimeout(function() {
			self.block = false;
		}, 2000);
    },

    enableState: function() {
		this.remainEnabled = true;
		$('.DynDNSState').addClass('checked');
		$('.dyndns-block').removeClass('disabled');
    },

    setDynDNSState: function(e, value) {
		var $state = $('.DynDNSState');

        if (value && !this.remainEnabled) {
			$state.data('value', true);
            $('.dyndns-block').removeClass('disabled');
            $state.addClass('checked');
        } else {
			$state.data('value', false);
            $('.dyndns-block').addClass('disabled');
            $state.removeClass('checked');
            this.remainEnabled = false;
        }

        this.setElementsState();
    },

    showAddAccount: function(e) {
		$(e.currentTarget).hide();
		$('.dyndns-add-account').show();
		this.page = 'add-account';
    },

    showNameTakenError: function() {
		this.enableState();
		$('.wrong-name-info').hide();
		$('.taken-info').show();
		$('.taken').show();
    },

    hideNameTakenError: function(){
		$('.taken-info').hide();
		$('.taken').hide();
    },

    showValidationError: function() {
		this.hideNameTakenError();
		this.enableState();
		$('.wrong-name-info').show();
		$('.wrong-name').show();
    },

    hideValidationError: function() {
		$('.wrong-name-info').hide();
		$('.wrong-name').hide();
    },

    showServerError: function() {
		this.revertState();
		$('.save-connection-error').show();
		this.serverError = true;
    },

    hideServerError: function() {
		$('.save-connection-error').hide();
		this.serverError = false;
    },

    hideAllErrors: function(){
		this.hideValidationError();
		this.hideNameTakenError();
		this.hideServerError();
    },


    removeAccount: function() {
		var self = this;

		self.activateButtons();
		this.page = 'add-account';

		$('.save-success').hide();
		$('.dyndns-add-account').show();
		$('.expandable-dyndns').hide();
		$('.host-name').val('');
		swc.models.SwisscomDyndns.set({previous_hostname: swc.models.SwisscomDyndns.get('hostname')});
		swc.models.SwisscomDyndns.set({hostname: ''});

		self.deletedAccount = true;
    },

    renderComplete: function() {
		var state = $('.DynDNSState').hasClass('checked') ? true : false,
			hostname = swc.models.SwisscomDyndns.get('hostname');

		var hostname_input = this.$el.find("input.host-name");

		this.registerValidationListener(hostname_input);

		if(hostname === '' || hostname === null) {
			this.page = 'add-account';
		} else {
			this.page = 'index';
		}

		if(state !== true || this.page === 'index') {
			$('taken-info').hide();
		}

		if(this.serverError) {
			this.showServerError();
		}

		this.deletedAccount = false;
		this.serverError = false;
		this.remainEnabled = false;
    },

    change: function(e) {
		var hostName = $(e.currentTarget).val();
		this.activateButtons();
		if (hostName.length === 0) {
			this.deactivateButtons();
		}
    },


    save: function() {
		var deferred = new $.Deferred(),
		name = $('.host-name').val() || swc.models.SwisscomDyndns.get('hostname'),
		state = $('.DynDNSState').hasClass('checked')? true : false,
		previous = swc.models.SwisscomDyndns.get('previous_hostname'),
		self = this;

		this.previousName = $('.host-name').val();
		this.hideAllErrors();

		if(name === '') {
			state = false;
			previous = '';
			swc.models.SwisscomDyndns.set({previous_hostname: ''});
		}

		if(this.page === 'add-account') {
			$.when(swc.models.SwisscomDyndns.isNameTaken(name)).done(function(data) {
				if(data.result === false || name === previous) {
					self.validationError = false;
					$.when(swc.models.SwisscomDyndns.updateConfig({enable: state, name: name})).done(function(data){
						if(data.status === 'OK' && data.result === true) {
							self.page = 'index';
							return deferred.resolve();
						} else {
							self.showServerError();
							return deferred.reject();
						}
					}).fail(function(){
						self.showServerError();
						return deferred.reject();
					});
				}
				else if(data.status !== 'OK') {
					self.validationError = true;
					self.stopPageLoading();
					self.showServerError();
					return deferred.resolve();
				}
				else {
					self.validationError = true;
					self.stopPageLoading();
					self.$el.find("input.host-name").trigger('validation-fail');
					self.showNameTakenError();
				}

				}).fail(function() {
					self.showServerError();
					return deferred.reject();
			}).fail(function(){
				self.showServerError();
				return deferred.reject();
			});
		} else {
				$.when(swc.models.SwisscomDyndns.updateConfig({enable: state, name: name})).done(function(resp){
					if(resp.status === 'OK'){
						deferred.resolve();
						self.page = 'index';
					} else {
						self.showServerError();
						deferred.reject();
						self.page = 'index';
					}
				}).fail(function(){
					self.showServerError();
					return deferred.reject();
				});
		}

		return deferred.promise();
	},

	isAbleToNavigate: function() {
        var userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin();

        if(userIsSuperAdmin === true) {
            return false;
        }
        return true;
    },

    changeOnEditPage: function(){
		return (!$('.DynDNSState').hasClass('checked') && this.page === 'add-account');
    },

    checkModification: function() {
		var previous = swc.models.SwisscomDyndns.get('previous_hostname');
        return (!this.pageCheckDefaultValues() || this.previousName === $('.host-name').val() || this.changeOnEditPage() || (this.page === 'add-account' && !_.isUndefined(previous)));
    },

    hasChanged: function() {
		return !$('.save-changes').hasClass('disabled');
		//return (!this.pageCheckDefaultValues() && $('.host-name').val() === '');
    },

    checkboxChanged: function(){
		var def = swc.models.SwisscomDyndns.get('status'),
			checked = $('.DynDNSState').hasClass('checked');

		return def === checked;
    },

    setElementsState: function() {
        // if there are some modification(edit, delete of some amount of contacts)
        // then buttons save/cancel should be disabled
        if (this.checkModification()) {
			this.buttonsVisibility();
		} else {
			this.deactivateButtons();
		}
    },

	buttonsVisibility: function () {
		var hostName = $('.host-name').val() ? $('.host-name').val() : "",
			previous = swc.models.SwisscomDyndns.get('previous_hostname');

        if ((hostName.length !== 0 && this.page === 'add-account' && (previous === '' || previous === undefined)) || (this.page === 'add-account' && (!_.isUndefined(previous) && previous !== '')) || this.page === 'index') {
            this.activateButtons();
        } else {
            this.deactivateButtons();
        }
    },

	activateButtons: function(){
        this.$('.save-changes').removeClass('disabled');
        this.$('.cancel-changes').removeClass('disabled');
    },

    deactivateButtons: function(){
        this.$('.save-changes').addClass('disabled');
        this.$('.cancel-changes').addClass('disabled');
    }


});;swc.constructors.ApplicationsSpeedtestView =  swc.base.PageView.extend(swc.base.SpeedcheckProtoView).extend({

	pageTemplateID: "app:speedtest",

	disallowedDevices: ['starlite', 'starlink'],

	outsidePanel: false,

	preRender: function() {
        this.tab = $.template("pageContent", swc.Templates.get("applications:speedtest").get('content'));
	},

	speedcheckRenderComplete: function() {
		var content = $('.tabs-content');

		content.html($.tmpl(this.tab, {
			localeStrings: swc.models.Locale.getLocaleStrings("system/diagnostics/speedcheck"),
            localeString: getTranslationStringsjQuery
		}));
	},

	renderComplete: function() {
        swc.base.SpeedcheckProtoView.renderComplete.apply(this, arguments);

        $('.page-overview.submenu').find('a').removeClass('active');
    }
});;swc.constructors.ApplicationsVpnView = swc.base.PageView.extend({

    className: 'application-vpn',

    models: [ 'VPNServer', 'VPNCollection', 'Network', 'DynDNS', 'DynDNSProviderCollection', 'EventDispatcher', 'Applications', 'SwisscomDyndns'],

    disallowedDevices: ['starlite', 'starlink'],

    eventActions: [
        {
            event: 'changed',
            action: 'vpnClientUpdated'
        }
    ],

    events: {
        'swc-checkbox:change .swc-checkbox.vpnServerEnable': 'setGlobalState',
        'click .do-delete': 'tryDeleteClient',
        'keyup input[name*=vpn_shared_secret]': 'copySecret',
        'change input[name*=vpn_shared_secret]': 'copySecret',
        'keyup input[name*=vpn_password]': 'copyPassword',
        'change input[name*=vpn_password]': 'copyPassword',
        'swc-checkbox:change .swc-checkbox.show-secret': 'setPasswordState'
    },

    validation: {
        'vpn_username': 'VPNServer:username',
        'vpn_shared_secret': 'VPNServer:vpn_shared_secret',
        'vpn_password': 'VPNServer:password'
    },

    allowedMods: ['expert'],

    listenerInterval: 10,

    listenerEnabled: false,

    blockEvent: false,

    listenerPages: [ 'vpn' ],

    deletedClients: false,

    showGlobalError: false,

    passEvent: function(event, eventObject) {
        swc.constructors.EventController(this.eventActions)
                      .pass.call(this, event, eventObject);
    },

    preRender: function() {
        $.when(
            swc.models.Applications.isVpnEnable()
        ).done(function(vpnGlobalEnable) {
            if (!vpnGlobalEnable) {
                swc.router.navigate('applications/overview');
            }
        });
        this.eventDispatcher=swc.models.EventDispatcher;
        this.eventDispatcher.observ("vpn_server",this);
    },

    vpnClientUpdated: function() {
        this.onListenerComplete();
    },

    setTemplateData: function(){
        var network = swc.models.Network.get('status').toJSON(),
            swisscomDyndns = swc.models.SwisscomDyndns,
            dyndns = swc.models.DynDNS,
            dyndnsProvider = swc.models.DynDNSProviderCollection.models[0];

        function portForwardingStatus() {
            var rawData = swc.models.VPNServer.get('portForwardingStatus'),
                portForwardingStatusResult = {
                    blockedPorts: 0,
                    singlePort: false,
                    notFreePorts: []
                };

            for (var port in rawData) {
                if (rawData[port] === "PORT_IN_USE") {
                    portForwardingStatusResult.notFreePorts.push(port);
                    portForwardingStatusResult.blockedPorts++;
                }
            }
            if (portForwardingStatusResult.blockedPorts > 1) {
                portForwardingStatusResult.singlePort = false;
            }

            portForwardingStatusResult.notFreePorts = _.sortBy(portForwardingStatusResult.notFreePorts, function(num){ return parseInt(num, 10); });

            return portForwardingStatusResult;
        }

        this.templateData = {
            username: swc.models.VPNServer.get('username'),
            password: swc.models.VPNServer.get('password'),
            shared_secret: swc.models.VPNServer.get('shared_secret'),
            clients: this.filterClients(),
            VPNServerState: swc.models.VPNServer.get('enable'),
            portForwardingStatus: portForwardingStatus(),
            netStatus: network,
            swisscomDyndns: swisscomDyndns,
            dyndns: dyndns,
            dyndnsProvider: dyndnsProvider,
            superAdmin: swc.models.Login.checkUserSuperAdmin()
        };
    },

    renderComplete: function() {
        var usernameBox = this.$el.find('input.username'),
            secretOrigBox = this.$el.find('input.secret-original'),
            secretCopyBox = this.$el.find('input.secret-copy'),
            passwordOrigBox = this.$el.find('input.password-original'),
            passwordCopyBox = this.$el.find('input.password-copy');

        this.registerValidationListener(usernameBox);
        this.registerValidationListener(secretOrigBox);
        this.registerValidationListener(secretCopyBox);
        this.registerValidationListener(passwordOrigBox);
        this.registerValidationListener(passwordCopyBox);

        $('body').css('background-color', '#ffffff');

        //fix for weird faded backdrop 
        var state = swc.models.VPNServer.get('enable');
        var self = this;
        _.delay(function() {
            self.showState(state);
        },100);

        this.setPasswordState(null, false);

        $('.page-overview.submenu').find('a').removeClass('active');
    },

    registerValidationListener: function($element) {
        // Check if element exists on the page:
        if (!$element.size()) {
            return;
        }

        // Add green mark when success validation happens:
        $element.on('validation-success', function() {
            $element.toggleClass('valid', !_.isEmpty($element.val()));
        });

        // Remove green mark when validation fails:
        $element.on('validation-fail', function() {
            $element.removeClass('valid');
        });
    },

    activateButtons: function(){
        this.$('.save-changes').removeClass('disabled');
        this.$('.cancel-changes').removeClass('disabled');
    },

    /**
     * Disable save/cancel buttons
     */
    deactivateButtons: function(){
        this.$('.save-changes').addClass('disabled');
        this.$('.cancel-changes').addClass('disabled');
    },

    copyPassword: function(e) {
        var element = $(e.target),
            visibleField,
            hiddenField;

        if (element.hasClass('password-original')) {
            visibleField = this.$('input.password-original');
            hiddenField = this.$('input.password-copy');
        } else {
            visibleField = this.$('input.password-copy');
            hiddenField = this.$('input.password-original');
        }

        hiddenField.val(visibleField.val());
    },

    copySecret: function(e) {
        var element = $(e.target),
            visibleField,
            hiddenField;

        if (element.hasClass('secret-original')) {
            visibleField = this.$('input.secret-original');
            hiddenField = this.$('input.secret-copy');
        } else {
            visibleField = this.$('input.secret-copy');
            hiddenField = this.$('input.secret-original');
        }

        hiddenField.val(visibleField.val());
    },

    setPasswordState: function(e, value) {
        var secretOrigBox = $('input.secret-original'),
            secretCopyBox = $('input.secret-copy'),
            passwordOrigBox = $('input.password-original'),
            passwordCopyBox = $('input.password-copy');

        if (value) {
            passwordOrigBox.hide();
            passwordCopyBox.show();
        } else {
            passwordCopyBox.hide();
            passwordOrigBox.show();
        }

        if (value) {
            secretOrigBox.hide();
            secretCopyBox.show();
        } else {
            secretCopyBox.hide();
            secretOrigBox.show();
        }
    },

    onListenerComplete: function() {
        // FIXME: Remove this once correct implementation of router done
        // This protects from rendering device list on a wrong page
        if (Backbone.history.fragment !== swc.settings.application.get("navigation")['vpn']) {
            //this.stopListener();
            return;
        }
        this.updateVPNClients();
        this.stopPageLoading();
    },

    setElementsState: function() {
        if (this.checkModification()) {
            this.activateButtons();
        } else {
            this.deactivateButtons();
        }
    },

    updateVPNClients: function() {
        var vpnClients = $('.vpnClients'), self = this;
        $.when(swc.models.VPNCollection.fetch()).done(function() {
            var clientsTemplate = $.template("clientsForm",
                swc.Templates.get("applications:vpn:clients").get('content')),
            clients = self.filterClients(),
            tmpl = $.tmpl(clientsTemplate, {
                    localeString: getTranslationStringsjQuery,
                    localeStrings: swc.models.Locale.getLocaleStrings(),
                    clients: clients
                }),
            settingsFormContainer = self.$(".vpnClients");
            vpnClients.html('');
            settingsFormContainer.html(tmpl);
        }).fail(function() {
        });
    },

    showState: function(forceState) {
        var state = swc.models.VPNServer.get('enable');

        if(!_.isUndefined(forceState)) {
            state = forceState;
        }

        this.setPageState(state, {disabledBlockClass: "disableInsideApplication"});

        if (state === false) {
            this.showDisabledMessage({
                container: '.vpnServerOffMessage',
                className: 'above'
            });
            this.$('input').attr('disabled', 'disabled');
        } else {
            this.hideDisabledMessage();
            this.$('input').removeAttr('disabled');
        }
    },

    setGlobalState: function (e, value) {
        var self = this;
        swc.models.VPNServer.set('enable', value);
        self.showState();
    },

    filterClients: function() {
        var self = this,
            collection = [];

        collection = swc.models.VPNCollection.filter(function(model) {
            var response = false;

            if (model.get('deleted')) {
                self.deletedClients = true;
                response = false;
            } else {
                response = true;
            }

            return response;
        });

        return collection;
    },

    checkModification: function() {
        return (!this.pageCheckDefaultValues() || this.deletedClients);
    },
    
    hasChanged: function() {
        return (!this.pageCheckDefaultValues() || this.deletedClients);
    },

    deleteClient: function(e) {
        var item = $(e.target).closest('.expandable-list-item'), itemId = item.attr('data-key'),
        client = swc.models.VPNCollection.get(itemId),
        noVpnClients = $('div.no-vpn-clients');
        client.set({'deleted': true});

        item.fadeOut(function(){
            item.remove();
        });

        if(swc.models.VPNCollection.models.length === 0 ||
            _.isUndefined(swc.models.VPNCollection.find(function (model) { return model.get('deleted') === false }))
        ) {
            _.delay(function() {noVpnClients.removeClass('hidden')}, 700);
        }
    },

    tryDeleteClient: function(e) {
        var self = this, ev = e;
        SWCElements.modalWindow.show({
            templateID: 'applications:vpn_server:settings:delete-client',
            templateData: {
            },
            className: 'removeclient-modal',

            onApply: function() {
                SWCElements.modalWindow.hide();
                self.deleteClient(ev);
            },

            onCancel: function() {
                SWCElements.modalWindow.hide();
            }
        });
    },

    save: function() {
        var deferred = new $.Deferred(),
            pageData = [], params = {};

        $.each(this.getElements(), function(key, element) {
            var param = getParameter($(element));
            params[param.parameterName] = param.parameterValue;
            pageData.push(getParameter($(element)));
        });

        $.when(swc.models.VPNServer.updateConfig(params)).done(function() {
            deferred.resolve();
        }).fail(function() {
            return deferred.reject();
        });

        return deferred.promise();
    }
});
;swc.constructors.StorageSchedulerView = swc.base.PageView.extend({

    className: 'storage-scheduler',

    listenerInterval: 10,

    listenerEnabled: true,

    listenerPages: [ 'storage-schedulers' ],

    models: [
        'SchedulerAp',
        'SchedulerPowerSavingManager',
        'apServiceState',
        'apServiceLoadingState'
    ],

    events: {
        'swc-checkbox:change .enable-scheduler':'enableScheduler',
        'swc-switcher:change .set-ap-state': 'setApState'
    },

    changedScheduler: false,

    setTemplateData: function() {
        this.changedScheduler = false;

        this.templateData = {
            schedulerState: swc.models.SchedulerAp.get('enable')
        };
    },

    showSaveSuccess: function() {
        this.changedScheduler = false;
        swc.base.PageView.prototype.showSaveSuccess.apply(this, arguments);
    },

    renderComplete: function() {
        var self = this,
            scheduler = swc.models.SchedulerAp,
            apSchedulerState = scheduler ? scheduler.get('enable') : false,
            apState = swc.models.apServiceState.get('status'),
            apServiceLoadingState = swc.models.apServiceLoadingState.get('isLoading'),
            apStateSwitcher = this.$el.find('.set-ap-state');

        this.$('.enable-scheduler').trigger('swc-checkbox:swc-change', apSchedulerState);
        
        swc.constructors.dispatcher.off('scheduler:change');
        swc.constructors.dispatcher.on('scheduler:change', function() {
            self.changedScheduler = true;
            self.setButtonsState();
        });

        self.schedulerDiapazon = new swc.constructors.SchedulerLiner({
            isDisabled: !apSchedulerState,
            scaleWidth: scheduler.widgetWidthInPixels,
            diapazones: scheduler.get('pixelSchedule'),
            model: scheduler,
            schedulerType: 'storage'
        });

        this.$('.scheduler-container').html(self.schedulerDiapazon.el);
        apStateSwitcher.trigger('swc-switcher:swc-change', apState);

        if (apServiceLoadingState === true && apState === true) {
            this.updateUI();
        }
    },

    updateUI: function() {
        var apState = swc.models.apServiceState.get('status'),
        apServiceLoadingState = swc.models.apServiceLoadingState.get('isLoading');

        if (apServiceLoadingState === false) {
            this.stopListener();
            // this.render();
            this.setPageState(true);
            this.setTabsState(true);
            this.hideDisabledMessage();
        } else {
            if (apServiceLoadingState === true && apState === true) {
                this.setPageState(false);
                this.setTabsState(false);
                this.showDisabledMessage({
                    container: '.apStateLoadingMessage',
                    className: 'above'
                });
            }

            this.listenerEnabled = true;
            this.startListener();
        }
    },

    onListenerComplete: function() {
        this.renderComplete();
        this.updateUI();
    },

    showModalEmptyAp: function() {
        var self = this,
            deferred = new $.Deferred();

        SWCElements.modalWindow.show({
            templateID: 'power:ap:modal',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings('power'),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            },
            className: 'storage-schedule',
            onApply: function() {
                SWCElements.modalWindow.hide();
                self.showPageLoading('Saving page data..');

                $.when(swc.models.SchedulerPowerSavingManager.saveAp())
                    .done(function() {
                        deferred.resolve();
                    })
                    .fail(function() {
                        deferred.reject();
                    });
            },
            onCancel: function() {
                SWCElements.modalWindow.hide();
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    save: function() {
        var self = this,
            deferred = new $.Deferred();

        if (swc.models.SchedulerAp.isEmpty() && swc.models.SchedulerAp.get('enable')) {
            this.stopPageLoading();
            return self.showModalEmptyAp();
        } else {
            $.when(swc.models.SchedulerPowerSavingManager.saveAp())
                .done(function() {
                    deferred.resolve();
                })
                .fail(function() {
                    deferred.reject();
                });
        }

        return deferred.promise();
    },
    
    onCancel: function() {
        this.changedScheduler = false;
    },

    enableScheduler: function(e, value) {
        swc.models.SchedulerAp.enable(!!value);
        this.renderComplete();
    },
    /**
     * Update application part status:
     * @param e {Object}
     * @param value {String}
     */
    setApState: function(e, value) {
        var self = this,
            pageRoutes = swc.settings.application.get('navigation'),
            page = Backbone.history.fragment;

        this.stopListener();
        this.listenerEnabled = false;

        this.showPageLoading("Updating Central Storage state");

        // Update parameter in the model:
        swc.models.apServiceState.set('status', value);

        // Start listener again when data saved:
        $.when(swc.models.apServiceState.sync('update')).done(function() {
            if (pageRoutes['data-sharing'] === page) {
                swc.models.apServiceState.fetch();
                self.listenerEnabled = true;
                self.startListener();
            } else {
                swc.router.navigate(pageRoutes['data-sharing'], { skipUnsavedChanges: true });
            }
        });
    },

    hasChanged: function() {
        return (!this.pageCheckDefaultValues() || this.changedScheduler);
    },

    setButtonsState: function() {
        $('.save-changes, .cancel-changes').toggleClass('disabled', !this.hasChanged());
    }

});
;swc.constructors.StorageSettingsAccessonmoveView = swc.base.TabView.extend({

    className: 'data-sharing',

    models: [ 'RemoteNASAccess', 'RemoteNASAccounts' ],
    
    fetchAccounts: true,
    
    listenerInterval: 10,

    listenerPages: [ 'data-sharing' ],

    events: {
        'swc-checkbox:change .swc-checkbox.set-usb-3-status': 'onUsbStatusChange',
        'swc-checkbox:change .swc-checkbox.set-remote-access-status': 'onRemoteAccessStatusChange',

        'click .icons.process-delete': 'removeNASAccount'
    },
    
    onListenerComplete: function() {
        this.refreshModel();
    },
    
    initialize: function() {
        var me = this;
        
        // create model by hand
        var model = new swc.constructors['RemoteNASAccounts']();
        swc.models['RemoteNASAccounts'] = model;
        
        // load model async
        model.isFetching = true;
        
        model.bind('change', function() {
            model.isFetching = false;

            // me.fetchAccounts = false;
            // save message state
            var container = $('.buttons-container-message'),
                saveSuccessHidden = container.find('.save-success').is(":hidden"),
                saveErrorHidden = container.find('.save-error').is(":hidden"),
                saveButtonDisabled = container.find('.save-changes').hasClass("disabled"),
                cancelButtonDisabled = container.find('.cancel-changes').hasClass("disabled"),
                saveValidationErrorsHidden = container.find('.save-validation-errors').is(":hidden");
                
            me.setTemplateData();
            me.getTemplateContent();

            container = $('.buttons-container-message');
            
            if(saveSuccessHidden) {
                container.find('.save-success').hide();
            } else {
                container.find('.save-success').show();
            }
            
            if(saveErrorHidden) {
                container.find('.save-error').hide();
            } else {
                container.find('.save-error').show();
            }
            
            if(saveValidationErrorsHidden) {
                container.find('.save-validation-errors').hide();
            } else {
                container.find('.save-validation-errors').show();
            }
            //sets the same state of buttons, after refresh
            if(!saveButtonDisabled){
                $('.buttons-container-message').find('.save-changes').removeClass("disabled");
            }
            if(!cancelButtonDisabled){
                $('.buttons-container-message').find('.cancel-changes').removeClass("disabled");
            }
        }, me);
        
        me.listenerEnabled = true;
        me.startListener();

        swc.constructors.StorageSettingsAccessonmoveView.__super__.initialize.apply(this);
    },
    
    refreshModel: function() {
        // default - load model
        if(this.fetchAccounts) {
            var model = swc.models.RemoteNASAccounts;
            model.isFetching = true;
            $.when(model.fetch()).done(function() {
            });
        } else {
            // when we are refreshing after get accounts just put flag back
            this.fetchAccounts = true;
        }
    },
    
    renderComplete: function() {
        var self = this;
        this.refreshModel();
        
        // Update buttons state to see, that changes were done on the page:
        this.setButtonsState();
        $('.page-overview.submenu').find('a').removeClass('active');

        _.delay(function() {
            self.refreshModel();
        },2000);
    },

    cancelChanges: function() {
        swc.router.navigate('storage/settings/access-on-move', { skipUnsavedChanges: true });
    },
    
    /**
     * Serialize models and collections and pass values to the template
     */
    setTemplateData: function() {
        this.templateData = {
            usbStatus: false,//swc.models.UsbStatus.get('status'),
            rnasError: swc.models.RemoteNASAccounts.isError(),
            remoteNasStatus: swc.models.RemoteNASAccess.isEnabled(),
            remoteNasAccounts: swc.models.RemoteNASAccounts.toJSON(),
            remoteNasFetching: swc.models.RemoteNASAccounts.isFetching
        };
    },

    /**
     * Handle USB 3.0 Checkbox value change, show confirmation wizard / change model
     *
     * @description:
     *
     *  When user switches on `usb 3.0 standard` - he must be shown confirmation window with alert message. If user click
     *  `apply`, nothing happens, if user clicks `cancel` -> checkbox must be set to false
     *
     * @param e {Object} -> Event
     * @param value {Boolean}
     */
    onUsbStatusChange: function(e, value) {
        var self = this;

        if (value === true) {
            SWCElements.modalWindow.show({
                templateID: 'storage:settings:modal-windows:enable-usb-3',
                templateData: {},
                className: 'enable-usb-3',

                onCancel: function() {
                    self.$('.swc-checkbox.set-usb-3-status').trigger('swc-checkbox:swc-change', false);
                },

                onApply: function() {
                    self.setUsbModelStatus(value);
                    SWCElements.modalWindow.hide();
                }
            });
        } else {
            this.setUsbModelStatus(value);
        }
    },

    /**
     * Update direct USB 3.0 Model status, and set buttons state:
     * @param value {Boolean}
     */
    setUsbModelStatus: function(/*value*/) {
        //swc.models.UsbStatus.set('status', value);

        // Update buttons state to see, that changes were done on the page:
        this.setButtonsState();
    },

    /**
     * Update remote access status in the model, after user clicked on linked checkbox
     * @param e {Object}
     * @param value {Boolean}
     */
    onRemoteAccessStatusChange: function(e, value) {
        swc.models.RemoteNASAccess.set('status', value);

        // Update buttons state to see, that changes were done on the page:
        this.setButtonsState();
    },

    /**
     * Remove account from the collection of Remote Access Accounts
     * @param e {Object}
     */
    removeNASAccount: function(e) {
        var button = $(e.target).closest('.process-delete'),
            accountEmail = button.data('account'),
            accountItem = this.$el.find('.expandable-list-item[data-key="' + accountEmail + '"]'),
            accountModel = swc.models.RemoteNASAccounts.findWhere({ email: accountEmail });

        // Update model in the collection:
        swc.models.RemoteNASAccounts.remove(accountModel);

        // Remove item from the list
        accountItem.remove();

        // Update buttons state to see, that changes were done on the page:
        this.setButtonsState();
    },

    /**
     * FIXME :: when all view will be moved to new architecture remove this
     * @note page implemented using 2.0 core components
     * @override
     */
    pageCheckDefaultValues: function() {
        // NOTE: We only check other models if AP is ON and completely ready
        var isModelsChanged = function() {
            var status = swc.models.apServiceState.get('status'),
                ra = !swc.models.RemoteNASAccess.hasChanged(),
                rn = !swc.models.RemoteNASAccounts.hasChanged();
             /*!swc.models.UsbStatus.hasChanged()*/
             
             return status && ra && rn;
        };

        this.parentView.pageCheckDefaultValues = isModelsChanged;

        return isModelsChanged();
    },

    hasChanged: function() {
        // avoiding modal messages when AP start is not finished:
        if (swc.models.apServiceLoadingState.get('isLoading') && swc.models.apServiceState.get('status')) {
            return false;
        }

        return swc.models.apServiceState.get('status') && !this.pageCheckDefaultValues();
    },

    save: function() {
        var //me = this,
            deferred = new $.Deferred(),
            toDo = [
                //swc.models.UsbStatus.sync('update'),
                swc.models.RemoteNASAccess.sync('update')
            ],
            deletedAccounts = swc.models.RemoteNASAccounts.changedModels({ onlyDeleted: true  });

        _.each(deletedAccounts, function(model) {
            toDo.push(
                model.sync('delete')
            );
        });

        $.when.apply(this, toDo)
            .done(function() {
                deferred.resolve();
            })
            .fail(function() {
                deferred.reject();
            })
            .always(function() {
                //me.refreshModel();
            });

        return deferred.promise();
    }

});
;swc.constructors.StorageSettingsCloudbackupView  = swc.base.TabView.extend({

    className: 'cloud-backup',

    listenerEnabled: true,

    listenerPages: [ 'cloud-backup' ],

    models: [
        'СloudServicesStatus',
        'CloudServices',
        'CentralStorage',
        'Network'
    ],

    validation: {
        usbUID: 'CloudAccountCollection:usbUID',
        folderName: 'CloudAccountCollection:folderName'
    },

    events: {
        "swc-radio-buttons:change .select-cloud-service": "onChangeCloudService",
        "swc-dropdown:change .device-selection": "onChangeDevice"
    },

    setTemplateData: function() {
        // TODO :: change <cloudService> and <cloudServicesList> calls to the collection:
        this.templateData = {
            centralStorageState: swc.models.CentralStorage.isEnabled(),
            cloudService: sessionStorage.getItem('cloudService') || "dropbox", // Default service - 'Dropbox'
            cloudServicesList: [
                { id: "dropbox", name: "Dropbox" },
                { id: "googledrive", name: "Google Drive" }
            ]
        };
    },

    preRender: function () {
        var deferred = new $.Deferred(),
            self = this,
            oauthCallbackParams,
            oauthCallbackParamsString = sessionStorage.getItem("oauthCallbackParams"),
            syncFolderDataString = sessionStorage.getItem('syncFolderData');

        sessionStorage.removeItem("oauthCallbackParams");
        sessionStorage.removeItem("syncFolderData");

        if (syncFolderDataString) {
            try {
                this.syncFolderData = JSON.parse(syncFolderDataString);
            } catch (e) {
                return;
            }
        }

        this.errorMessages = [];
        this.folderValidationMessages = [];

        if (!swc.models.CloudAccountCollection) {
            swc.models.CloudAccountCollection = new swc.constructors.CloudAccountCollection();
        }

        if (oauthCallbackParamsString) {
            // authentication is in progress (redirected from oauthHandler):
            this.stopPageLoading();
            this.showPageLoading("Processing login to Cloud Service");

            try {
                oauthCallbackParams = JSON.parse(oauthCallbackParamsString);
                if (!_.isUndefined(oauthCallbackParams.error) && oauthCallbackParams.error === "access_denied") {
                    throw 'access denied';
                }
                if (oauthCallbackParams.not_approved) {
                    throw 'not approved';
                }

                $.when(swc.models.CloudAccountCollection.handleOauthCallback(oauthCallbackParams))
                    .done(function(){
                        var folderPromise = self.syncFolderData ? self.folderNameValidation(self.syncFolderData) : true;
                        // Create folder used saved data from sessionStorage
                        $.when(folderPromise)
                            .done(function() {
                                self.syncFolderData = null; // remove folder data before navigate
                                self.navigateToList(); // switch to another view
                                deferred.reject(); // stop rendering this view
                            })
                            .fail(function(validationMessages){
                                // proceed rendering with validation messages:
                                self.folderValidationMessages = validationMessages;
                                deferred.resolve();
                            });
                    })
                    .fail(function(){
                        // proceed rendering with validation messages:
                        self.errorMessages.push('could-not-connect');
                        deferred.resolve();
                    });
            } catch(e) {
                // oauth failed
                // proceed rendering with validation messages
                // (currently the same error for all cases):
                this.errorMessages.push('could-not-connect');
                deferred.resolve();
            }
        } else {
            // authentication is not in progress:

            if (!swc.models.CentralStorage.isEnabled()) {
                // proceed rendering:
                deferred.resolve();
            } else {
                $.when(swc.models.CloudAccountCollection.sync('read'))
                    .done(function () {
                        if (swc.models.CloudAccountCollection.models.length > 0) {
                            self.navigateToList(); // switch to another view
                            deferred.reject(); // stop rendering this view
                        } else {
                            // proceed rendering:
                            deferred.resolve();
                        }
                    }).fail(function () {
                        // can't load collection but we don't really need it to render the page, so
                        // proceed rendering:
                        deferred.resolve();
                    });
            }
        }

        return deferred.promise();
    },

    onListenerComplete: function() {
        var apServiceLoadingState = swc.models.apServiceLoadingState.get('isLoading');

        // Check if AP turned off or DBUS crashed
        if (apServiceLoadingState === true) {
            swc.router.navigate('storage/settings/data-sharing', { trigger: false, skipUnsavedChanges: true });
        } else {
            if (!swc.models.СloudServicesStatus.get('isEnabled')) {
                this.onCloudServicesDisable();
            }
        }
    },

    /**
     * Listen to cloud services `disable` event
     *
     * @description:
     *
     * <СloudServicesStatus> model is listening to TR-069 parameter, which has a flag if cloud services are enabled to
     * the user. This parameter can be set to false during user work with cloud services. This method will prevent all
     * user actions and show modal window with redirection to other page
     */
    onCloudServicesDisable: function() {
        this.stopListener();

        SWCElements.modalWindow.show({
            templateID: 'storage:settings:modal-windows:disable-cloud-services',
            templateData: {},
            className: 'disable-cloud-services',

            onApply: function() {
                SWCElements.modalWindow.hide();
                swc.router.navigate('storage/settings/data-sharing', { trigger: false, skipUnsavedChanges: true });
            }
        });
    },

    navigateToList: function () {
        swc.router.navigate('storage/settings/cloud-backup/list', { trigger: false, skipUnsavedChanges: true});
    },

    setDefaults: function () {
        // Set default value for service
        this.$(".swc-radio-buttons.select-cloud-service").data("default-value", this.templateData.cloudService);

        if (this.syncFolderData) {
            // for storage
            this.$(".swc-dropdown.device-selection").data("default-value", this.syncFolderData.partitionUID);
            // for folder name
            this.$("input[name=folderName]").data("default-value", this.syncFolderData.folderName);
        } else {
            this.$(".swc-dropdown.device-selection").data("default-value", "null");
        }
    },

    /**
     * Set predefined values after page has been rendered:
     */
    renderComplete: function() {
        var self = this;

        // Check if Central Storage is Enabled, else -> disable page
        if (!swc.models.CentralStorage.isEnabled()) {
            return;
        }

        if (!swc.models.СloudServicesStatus.get('isEnabled')) {
            this.onCloudServicesDisable();
        }

        // Call default methods to (re)set values on the page:
        this.setDefaults();
        this.onChangeCloudService(null, this.templateData.cloudService, true);
        this.setDevicesDropdownValues(true);

        // Populate folder data
        if (this.syncFolderData) {
            this.$el.find(".device-selection").trigger("swc-dropdown:swc-change", this.syncFolderData.partitionUID);
            this.syncFolderData = null;
        }

        if (this.showSaveError) {
            this.showSaveError = false;
            this.errorMessages.push('can-not-receive-url');
        }

        if (!_.isEmpty(this.folderValidationMessages)) {
            this.setFolderNameValidationErrors(this.folderValidationMessages);
            this.folderValidationMessages = [];
        }

        if (!_.isEmpty(this.errorMessages)) {
            this.showSaveSuccess('error', this.errorMessages);
        }

        if (!this.devicesDropdownInterval) {
            this.devicesDropdownInterval = setInterval(function () {
                if (self.$el.parents('body').length > 0) {
                    $.when(swc.models.CentralStorage.sync('read', {silent: true}))
                        .done(function () {
                            self.setDevicesDropdownValues();
                        });
                } else {
                    // if the current view is not in the DOM
                    // we don't need the interval anymore:
                    clearInterval(self.devicesDropdownInterval);
                }
            }, 10000);
        }

        this.stopPageLoading();
    },

    /**
     * Change cloud service by user / application. Update correct model in cloud services collection:
     * @param e {Object} Event
     * @param cloudServiceID {String}
     */
    onChangeCloudService: function(e, cloudServiceID) {
        var serviceData = _.findWhere(this.templateData.cloudServicesList, { id: cloudServiceID }),
            serviceName = serviceData ? serviceData.name : '',
            input = this.$("input[name=folderName]"),
            isCalledFromRenderComplete = _.isNull(e);

        // TODO :: change service in collection

        if (this.syncFolderData && this.syncFolderData.folderName) {
            input.val(this.syncFolderData.folderName);
        } else if (input.data('default-value') === input.val() || isCalledFromRenderComplete) {
            input.val(serviceName);
        }

        input.data('default-value', serviceName);

        if (!isCalledFromRenderComplete) {
            input.trigger('validate');
        }
    },

    onChangeDevice: function (e) {
        var input = this.$("input[name=folderName]");

        if (e) {
            input.trigger('validate');
        }
    },

    /**
     * Update usb devices dropdown with list of usb devices
     * @param isSkipValidation {Boolean} skip / not skip validation for folder name
     */
    setDevicesDropdownValues: function(isSkipValidation) {
        var dropdown = this.$el.find('.swc-dropdown.device-selection'),
            oldValue = dropdown.data('value'),
            input = this.$("input[name=folderName]"),
            options = swc.models.CentralStorage.getDeviceDropdownOptions(),
            defaultDevice = swc.models.CentralStorage.getDefaultDeviceOption(options, oldValue);

        // Set default value to the dropdown if not set:
        dropdown.data('default-value', defaultDevice);

        // Set dropdown data:
        dropdown.data('options', options);
        dropdown.data('hide-default', true);
        dropdown.trigger('swc-dropdown:swc-change', defaultDevice);

        // Validate dropdown to show message to the user
        if (defaultDevice === "no-devices-connected") {
            this.pageValidation(dropdown, false);
        } else {
            this.clearValidationMessages(dropdown);

            // Validate folder name only when user did some changes on the page:
            if (!isSkipValidation && !this.pageCheckDefaultValues()) {
                input.trigger('validate');
            }
        }
    },

    folderNameValidation: function(syncFolderData) {
        var deferred = new $.Deferred();

        // Try to create a folder before login to CS
        $.when(swc.models.CloudAccountCollection.createFolder(syncFolderData.partitionUID, syncFolderData.folderName))
            .done(function() {
                deferred.resolve();
            })
            .fail(function(error) {
                var validationMessages = [];

                switch (error.message) {
                    case 'invalid name':
                        validationMessages.push("unable to create folder");
                    break;

                    case 'already exists':
                        validationMessages.push("already exists");
                    break;

                    default:
                        validationMessages.push("unable to create folder");
                }

                deferred.reject(validationMessages);
            });

        return deferred.promise();
    },

    /**
     * Show validation errors of folder name in UI, after validation was not completed on AP:
     * @param messsages
     */
    setFolderNameValidationErrors: function(messsages) {
        var folderNameInput = this.$el.find('input[name="folderName"]'),
            folderNameValidationContainer = this.$el.find('.validation-message[data-parameter-name="folderName"]');

        folderNameValidationContainer.find('.error-message').hide();

        if (!_.isEmpty(messsages)) {
            // Add validation error to the field:
            folderNameInput.addClass('validation-error');
            folderNameInput.data('validation-error', true);

            // Show validation error in UI:
            folderNameValidationContainer.show();
            folderNameValidationContainer.find('.error-message[data-error="' + messsages[0] + '"]').show();
        } else {
            folderNameInput.removeClass('validation-error');
            folderNameInput.data('validation-error', false);
            folderNameValidationContainer.hide();
        }
    },

    /**
     * Save method will be called when user will press "Log In" button;
     * @override
     */
    save: function() {
        var self = this,
            cloudService = this.$el.find(".select-cloud-service").data('value'),
            syncFolderData = {
                partitionUID: this.$el.find(".device-selection").data('value'),
                folderName: this.$el.find("input[name='folderName']").val()
            },
            deferred = new $.Deferred(),
            validationContainer = this.$('.validation-message[data-parameter-name="global"]');

        // Show user login message (not "saving changes")
        this.stopPageLoading();
        this.showPageLoading("Processing login to Cloud Service");

        validationContainer.hide();
        validationContainer.find('.error-message').hide();

        $.when(swc.models.Network.sync())
            .done(function() {
                // Check if device connected to the internet TODO :: remove after implementation on AP side
                if (!swc.models.Network.getParameter('ConnectionStatus', 'status')) {
                    self.stopPageLoading();
                    self.showSaveError = true;

                    return deferred.reject("can-not-receive-url");
                }
                $.when(swc.models.CloudServices.validateFolder(syncFolderData.partitionUID, syncFolderData.folderName))
                    .done(function() {
                        // Steps required to login:
                        // 1. getAuthURL()
                        // 2. handleOAtuhCallback()
                        // 3. syncFSResources() == create a folder
                        $.when(swc.models.CloudAccountCollection.getLoginURL(cloudService))
                            .done(function(url) {
                                // save folder name into local storage (or SessionStorage)
                                sessionStorage.setItem('syncFolderData', JSON.stringify(syncFolderData));
                                sessionStorage.setItem('cloudService', cloudService);
                                document.location.href = url;
                            })
                            .fail(function() {
                                self.showSaveError = true;
                                self.stopPageLoading();
                                deferred.reject();
                            });
                    })
                    .fail(function() {
                        self.stopPageLoading();
                        deferred.reject();
                    });
            });

        return deferred.promise();
    }
});
;swc.constructors.cloudAccountItem = Backbone.View.extend({

    pageTemplateID: 'storage:cloud:account-item',

    events: {
        'swc-checkbox:change .activate-cloud': 'toggleActivate',
        'click .do-custom-edit': 'edit',
        'click .do-disconnect': 'disconnect',
        'click .login-cloud': 'login',
        'click .sync-cloud': 'sync',
        'click .show-unsynced-files': 'showUnsyncedFiles'
    },

    render: function () {
        this.$el.empty();
        this.setTemplateData();
        this.$el.append($.tmpl(swc.Templates.get(this.pageTemplateID).get('content'), {
            data: this.templateData,
            localeStrings: swc.models.Locale.getLocaleStrings('storage'),
            localeString: getTranslationStringsjQuery,
            formatDate: swc.models.Locale.formatDate,
            GlobalEnable: swc.models.Application.get('GlobalEnable')
        }));

        return this;
    },

    showUnsyncedFiles: function() {
        var obj = this.model.getExtendedJSON(),
            data = _.extend(obj, {
                syncAction: false,
                loginAction: false
            });

        if (_.isEmpty(data.inaccessibleFiles)) {
            return;
        }

        data.inaccessibleFiles = _.map(data.inaccessibleFiles, function(file) {
            var list = file.split('/');

            list.splice(-1, 1, '<strong>' + list.slice(-1) + '</strong>');

            return list.join('/');
        });

        SWCElements.modalWindow.show({
            templateID: 'storage:cloud-backup:unsynced-files:modal-window',
            templateData: {
                syncDetails: data
            },
            onApply: function() {
                SWCElements.modalWindow.hide();
            }
        });
    },

    setTemplateData: function(){
        var obj = this.model.getExtendedJSON(),
            data;

        data = _.extend(obj, {
                syncAction: false,
                loginAction: false
            });

        switch (data.status) {

            case "IN_PROGRESS":
                data.tooltip = "Sync in progress";
                break;

            case "DISABLED":
                data.tooltip = "Sync stopped";
                break;

            case "COMPLETE":
                if (_.isEmpty(data.inaccessibleFiles)) {
                    data.tooltip = "Synced";
                } else {
                    data.status = 'INCOMPLETE';
                    data.tooltip = "Some files could not be synced";
                }
                break;

            case "ERROR":

                switch (data.error) {

                    case "NOT_ENOUGH_LOCAL_SPACE":
                        data.tooltip = "There is not enough space on your Central Storage to sync data. " +
                            "Please free some space.";
                        data.status = "WARNING";
                        break;

                    case "NOT_ENOUGH_REMOTE_SPACE":
                        data.tooltip = "There is not enough space on your Cloud Account %cloudService% " +
                            "(%accountName%) to sync data.";
                        data.status = "WARNING";
                        break;

                    case "INVALID_CREDENTIALS":
                        data.tooltip = "Could not connect to %cloudService%. Last login attempt on %lastLogin% has " +
                            "failed. Please try to login again later.";
                        data.loginAction = true;
                        break;

                    case "COMMUNICATION_ERROR":
                        data.tooltip = "There are troubles syncing some content. Try again by " +
                            "clicking \"Sync\" button or wait for 1 min for next sync attempt.";
                        data.status = "WARNING";
                        data.syncAction = true;
                        break;

                    default:
                        data.tooltip = "Unknown";
                }

                break;
            case "STORAGE_DISCONNECTED":
                data.tooltip = "Unable to connect to Cloud Account folder. Please check" +
                                " if your Central Storage device is connected to %ProductName%";
                data.deviceDisconnected = true;
                data.status = "ERROR";
                break;
            default:
                data.tooltip = "Unknown";
                break;
        }
        data.status = data.status.toLowerCase();

        this.templateData = data;
    },

    toggleActivate: function (e) {
        var value = $(e.currentTarget).data('value') ? 'COMPLETE' : 'DISABLED';
        this.model.set({status: value}, {silent: true});
    },

    edit: function () {
        var self = this;

        SWCElements.modalWindow.show({
            templateID: 'storage:cloud:cloud-accounts:edit-modal',
            templateData: {
                cloudServiceName: self.model.get('cloudServiceName'),
                accountName: self.model.get('accountName'),
                location: self.model.get('deviceName'),
                folderName: self.model.get('folderName')
            },
            className: 'cloud-edit-modal',

            onCancel: function() {
                SWCElements.modalWindow.hide();
                // Hide dropdown overlay if user presses "Cancel" when DropDown list is opened
                SWCElements.dropDown.closeAll();
                // Stop populating deviceDropDownValues
                swc.models.CentralStorage.off("central-storage-update");
            },

            onApply: function() {
                var deviceId = $('.modalWindow .device-selection').data('value'),
                    folderName = $('.modalWindow input[name=folderName]').val();
                $.when(self.folderNameValidation(deviceId, folderName))
                    .done(function () {
                        SWCElements.modalWindow.hide();
                        // Hide dropdown overlay if user presses "Apply" when DropDown list is opened
                        SWCElements.dropDown.closeAll();
                        swc.models.CloudAccountCollection.sync('read');
                        // Stop populating deviceDropDownValues
                        swc.models.CentralStorage.off("central-storage-update");
                    });
            },

            onShow: function () {
                // Set existing values to elements
                $('.modalWindow .device-selection').data("value", self.model.get('deviceId'));
                $('.modalWindow input[name=folderName]').val(self.model.get('folderName'));
                self.setDevicesDropdownValues();
                swc.models.CentralStorage.on("central-storage-update", self.setDevicesDropdownValues);
            }
        });
    },

    /**
     * Update usb devices dropdown with list of usb devices
     */
    setDevicesDropdownValues: function() {
        var dropdown = $('.modalWindow .swc-dropdown.device-selection'),
            validationMessages = $('.modalWindow .validation-message[data-parameter-name="usbUID"]'),
            validationMessage = validationMessages.find('.error-message[data-error="no connected devices"]'),
            oldValue = dropdown.data('value'),
            options = swc.models.CentralStorage.getDeviceDropdownOptions(),
            defaultDevice = swc.models.CentralStorage.getDefaultDeviceOption(options, oldValue);

        // Set dropdown data:
        dropdown.data('options', options);
        dropdown.data('hide-default', true);
        dropdown.trigger('swc-dropdown:swc-change', defaultDevice);

        // Validate dropdown to show message to the user
        if (defaultDevice === "no-devices-connected") {
            validationMessages.show();
            validationMessage.show();
            dropdown.addClass('validation-error');
        } else {
            validationMessages.hide();
            validationMessage.hide();
            dropdown.removeClass('validation-error');
        }
    },

    folderNameValidation: function(partitionUID, folderName) {
        var self = this,
            deferred = new $.Deferred();

        if (partitionUID === this.model.get('deviceId') && folderName === this.model.get('folderName')) {
            var validationMessages = [];
            validationMessages.push("already exists");
            this.setFolderNameValidationErrors(validationMessages);
            return deferred.reject();
        }

        // Try to create a folder
        $.when(swc.models.CloudServices.validateFolder(partitionUID, folderName))
            .done(function () {
                $.when(swc.models.CloudAccountCollection.moveFolder(self.model, partitionUID, folderName))
                    .done(function() {
                        var folderNameValidationContainer = $('.modalWindow .validation-message[data-parameter-name="folderName"]');

                        folderNameValidationContainer.find('.error-message').hide();
                        deferred.resolve();
                    })
                    .fail(function(error) {
                        var validationMessages = [];

                        switch (error.message) {
                            case 'invalid name':
                                validationMessages.push("unable to create folder");
                            break;

                            case 'already exists':
                                validationMessages.push("already exists");
                            break;

                            default:
                                validationMessages.push("unable to create folder");
                        }

                        self.setFolderNameValidationErrors(validationMessages);
                        deferred.reject();
                    });
            })
            .fail(function (status) {
                var validationMessages = [];
                switch (status.failure) {
                    case 'SYNC_FOLDER_ALREADY_EXISTS':
                        validationMessages.push("already exists");
                    break;

                    default:
                        validationMessages.push("unable to create folder");
                }
                self.setFolderNameValidationErrors(validationMessages);
                deferred.reject();
            });

        return deferred.promise();
    },

    setFolderNameValidationErrors: function(messsages) {
        var folderNameInput = $('.modalWindow input[name="folderName"]'),
            folderNameValidationContainer = $('.modalWindow .validation-message[data-parameter-name="folderName"]');

        // Add validation error to the field:
        folderNameInput.addClass('validation-error');
        folderNameInput.data('validation-error', true);

        // Show validation error in UI:
        folderNameValidationContainer.show();
        folderNameValidationContainer.find('.error-message').hide();
        folderNameValidationContainer.find('.error-message[data-error="' + messsages[0] + '"]').show();
    },

    disconnect: function () {
        var self = this;

        swc.views.Application.showPageLoading('Loading page data..');
        $.when(swc.models.CentralStorage.sync('read'))
            .done(function () {
                var devices = swc.models.CentralStorage.get('usb-devices').get('devices'),
                    device = _.find(devices, function (obj){
                        return obj.id === self.model.get('deviceId');
                    });
                if (device) {
                    self.disconnectModal();
                } else {
                    self.doDisconnect(true);
                }
            }).always(function () {
                swc.views.Application.stopPageLoading();
            });
    },

    disconnectModal: function () {
        var self = this;

        SWCElements.modalWindow.show({
            templateID: 'storage:cloud:cloud-accounts:disconnect-modal',
            templateData: {
                cloudServiceName: getTranslationStrings(self.model.get('cloudServiceName'), true),
                accountName: self.model.get('accountName')
            },
            className: 'cloud-disconnect-modal',

            onCancel: function() {
                SWCElements.modalWindow.hide();
            },

            onApply: function() {
                var keepData = getParameter($('.cloud-disconnect-modal .keep-data-selection'));
                self.doDisconnect(keepData.parameterValue === 'keep');
            }
        });
    },

    doDisconnect: function (keepData) {
        var self = this,
            validationMessageContainer = $(".modalWindow .save-validation-errors");

        $.when(this.model.logout(keepData))
            .done(function () {
                self.$el.remove();
                validationMessageContainer.hide();
                SWCElements.modalWindow.hide();

                // Remove cloud service id on logout process
                sessionStorage.removeItem("cloudService");

                // If no accounts left - redirect user to Cloud Login page
                if (swc.models.CloudAccountCollection.models.length === 0) {
                    swc.router.navigate('storage/settings/cloud-backup', { trigger: false, skipUnsavedChanges: true});
                    return;
                }
            })
            .fail(function () {
                validationMessageContainer.show();
                validationMessageContainer.find("div[data-error='can not remove folder']").show();
            });
    },

    login: function () {
        swc.models.CloudServices.checkCredentials();
    },

    forceSyncDevice: function(deviceId){
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                "service": "com.swisscom.stargate/ws/cloud/backup.com.swisscom.stargate.cloud.backup",
                "method": "forceFailedTasksForPartition",
                "parameters": {
                    "partitionUUID": deviceId
                }
            },

            success: function() {
                // Timeout is used because the board responds immediately
                // but actually needs some time to change the sync state.
                // sad but true :(
                setTimeout(function(){
                    deferred.resolve();
                }, 3000);
            }
        });

        return deferred.promise();
    },

    /**
     * Starts sync for the service
     * @deprecated
     */
    sync: function () {
        var self = this;

        self.showPageLoading('Saving page data..');
        $.when(this.forceSyncDevice(this.model.get('deviceId')))
            .done(function(){
                swc.models.CloudAccountCollection.sync('read')
                    .done(function(){
                        self.stopPageLoading();
                    });
            });
    }
});
;swc.constructors.StorageSettingsCloudbackupListView = swc.base.TabView.extend({

    className: 'cloud-accounts',

    pageTemplateID: 'storage:settings:cloud-accounts',

    listenerEnabled: true,

    listenerPages: ['cloud-backup-list'],

    events: {
        'swc-checkbox:change .activate-cloud': 'memorizeCheckboxesState',
        'click .save-changes': 'save'
    },

    message: "",

    models: [ 'apServiceState', 'apServiceLoadingState',  'СloudServicesStatus', 'CloudServices', 'CentralStorage' ],

    checkboxes: {},

    // Dirty crutch.
    // Should be fixed in base classes to support pageTemplateID property
    // without need of initialize method overriding
    initialize: function() {
        this.template = $.template("pageContent", swc.Templates.get(this.pageTemplateID).get('content'));
        this.events = _.extend({}, swc.base.PageView.prototype.events, this.events);
    },

    preRender: function () {
        var self = this,
            deferred = new $.Deferred();

        if (!swc.models.CloudAccountCollection) {
            swc.models.CloudAccountCollection = new swc.constructors.CloudAccountCollection();
        }

        if (!swc.models.CentralStorage.isEnabled()) {
            deferred.resolve();
        } else {
            $.when(swc.models.CloudAccountCollection.sync('read'))
                .done(function () {
                    if (_.indexOf(self.models, 'CloudAccountCollection') === -1) {
                        // we need it to refresh automatically but can not add it to models from the
                        // beginning because it is unnecessary when the Central Storage is down
                        self.models.push('CloudAccountCollection');
                    }

                    if (swc.models.CloudAccountCollection.models.length > 0) {
                        var deferredForceSyncActions = [];
                        // force sync for every account
                        // FIXME: Only sync those devices which has some trouble status
                        swc.models.CloudAccountCollection.each(function (accountModel) {
                            deferredForceSyncActions.push(accountModel.forceSyncDevice());
                        });
                        $.when.apply(this, deferredForceSyncActions)
                            // never mind, just resolve and see what we get when background listener runs sync
                            .always(function () {
                                // now re-read the whole collection
                                swc.models.CloudAccountCollection.sync('read')
                                    .always(function(){
                                        deferred.resolve();
                                    });
                            });
                    } else {
                        self.switchToCloudLoginView();
                        deferred.reject();
                    }
                })
                .fail(function () {
                    deferred.resolve();
                });
        }

        return deferred.promise();
    },

    renderComplete: function(){
        var self = this;

        var listView = new swc.base.listView({
            model: swc.models.CloudAccountCollection,
            itemView: 'cloudAccountItem',

            renderComplete: function () {
                self.restoreCheckboxes();
            }
        });

        if (!swc.models.СloudServicesStatus.get('isEnabled')) {
            this.onCloudServicesDisable();
        }

        this.$el.find('.account-list-container').replaceWith(listView.render().el);

        if (sessionStorage['clouds-saved'] === 'ok') {
            this.$('.buttons-container-message .save-success').show();
            sessionStorage.removeItem('clouds-saved');

            setTimeout(function () {
                self.$('.buttons-container-message .save-success').hide();
            }, 3000);
        }

        if (!swc.models.CentralStorage.isEnabled()) {
            this.setPageState(false);
        }
    },

    onListenerComplete: function() {
        var apServiceLoadingState = swc.models.apServiceLoadingState.get('isLoading');

        // Check if AP turned off or DBUS crashed
        if (apServiceLoadingState === true) {
            swc.router.navigate('storage/settings/data-sharing', { trigger: false, skipUnsavedChanges: true });
        } else {
            if (!swc.models.СloudServicesStatus.get('isEnabled')) {
                this.onCloudServicesDisable();
            }
        }
    },

    /**
     * Listen to cloud services `disable` event
     *
     * @description:
     *
     * <СloudServicesStatus> model is listening to TR-069 parameter, which has a flag if cloud services are enabled to
     * the user. This parameter can be set to false during user work with cloud services. This method will prevent all
     * user actions and show modal window with redirection to other page
     */
    onCloudServicesDisable: function() {
        this.stopListener();

        SWCElements.modalWindow.show({
            templateID: 'storage:settings:modal-windows:disable-cloud-services',
            templateData: {},
            className: 'disable-cloud-services',

            onApply: function() {
                SWCElements.modalWindow.hide();
                swc.router.navigate('storage/settings/data-sharing', { trigger: false, skipUnsavedChanges: true });
            }
        });
    },

    switchToCloudLoginView: function () {
        swc.router.navigate('storage/settings/cloud-backup', { trigger: false, skipUnsavedChanges: true});
    },

    memorizeCheckboxesState: function () {
        var self = this;
        this.checkboxes = {};

        if (!this.pageCheckDefaultValues()) {
            this.$('.swc-checkbox.activate-cloud').each(function () {
                var this$ = $(this),
                    id = this$.closest('.expandable-list-item').data('key');
                if (this$.data('value') !== this$.data('default-value')) {
                    self.checkboxes[id] = this$.data('value');
                }
            });
        }
    },

    // restoring checkboxes states that have been memorized in memorizeCheckboxesState()
    restoreCheckboxes: function () {
        var self = this;

        this.$('.expandable-list-item').each(function () {
            var this$ = $(this),
                id = this$.data('key'),
                checkbox = this$.find('.swc-checkbox.activate-cloud');
            if (_.has(self.checkboxes, id)) {
                checkbox.trigger('swc-checkbox:swc-change', self.checkboxes[id]);
            }
        });
    },

    save: function () {
        var self =  this,
            promises = [];

        _.each(swc.models.CloudAccountCollection.models, function (model) {
            // we use the memorized checkboxes states to avoid sync problem:
            if (_.has(self.checkboxes, model.id)) {
                promises.push(model.setBackupState(self.checkboxes[model.id]));
            }
        });

        $.when(promises)
            .done(function () {
                sessionStorage['clouds-saved'] = 'ok';
                swc.models.CloudAccountCollection.sync('read');
                self.checkboxes = {};
            });
    },

    onCancel: function () {
        this.checkboxes = {};
    }
});
;swc.constructors.StorageDataView =  swc.base.PageView.extend({

    className: 'storage-devices-list',

    listenerInterval: 5,

    listenerEnabled: false,

    blockEvent: false,

    disableStorageProtection: false,

    listenerPages: [ 'data' ],

    models: [ 'MediaServices', 'AfpServices', 'apServiceState', 'apServiceLoadingState', 'StorageDevices', 'CentralStorage', 'SchedulerAp', 'System', 'EventDispatcher', 'StorageProtection' ],

    /**
     * Defines which item is expanded at the current moment
     */
    openedItem: null,

    events: {
        'swc-switcher:change .set-ap-state': 'setApState',
        'click .remove-storage': 'unmountDevice',
        'expandable:open .expandable-list': 'onExpandableOpen',
        'expandable:close .expandable-list': 'onExpandableClose',
        'click .hoverable': 'onExpandTrigger',
        'hover .hoverable': 'onHover',

        'swc-checkbox:change .swc-checkbox.set-media-server-status': 'onMediaServiceStatusChange',
        'swc-checkbox:change .swc-checkbox.set-afp-status': 'onAFPStatusChange',

        'swc-checkbox:change .swc-checkbox.show-secret': 'setProtectionPasswordState',
        'swc-checkbox:change .swc-checkbox.protectionEnable': 'setProtectionEnable',
        'keyup input[name*=storage_protection_password]': 'copyPassword',
        'change input[name*=storage_protection_password]': 'copyPassword'

    },

    validation: {
         'storage_protection_username': 'StorageProtection:username',
         'storage_protection_password': 'StorageProtection:password'
    },

    eventActions: [
        {
            event: 'self_device_updated',
            action: 'updateDeviceList'

        },
        {
            event: 'add',
            action: 'updateDeviceList'
        },
        {
            event: 'usb_notification_device_deleted',
            action: 'updateDeviceList'
        }
    ],

    passEvent: function(event, eventObject) {
      swc.constructors.EventController(this.eventActions)
                      .pass.call(this, event, eventObject);
    },

    setTemplateData: function() {
        var storageDevicesArray = [],
            apScheduler = swc.models.SchedulerAp,
            apSchedulerState = !_.isUndefined(apScheduler) ? apScheduler.get('enable') : false;

        var APVersion = swc.models.System.attributes.APVersion || '0.0.0';
        APVersion = parseInt(APVersion.split('.')[0], 10);
        if(APVersion < 6) {
            this.models = _.without(this.models, 'StorageProtection');
            this.disableStorageProtection = true;
        }

        swc.models.StorageDevices.each(function(model) {
            var isSupported=true;
            //chceck if device is supported
            if(model.get('type') === "usb-disk-undefined" || model.get('type') === "sd-card-undefined"){
                isSupported= false;
            }
            var deviceObj = {
                id: model.id,
                label: model.get('label'),
                type: model.get('type'),
                capacity: model.get('capacity'),
                freeSpace: model.get('freeSpace'),
                deviceIsSupported: isSupported
            };

            storageDevicesArray.push(deviceObj);
        });

        var protection;

        if(this.disableStorageProtection === false) {
            protection = swc.models.StorageProtection.get('protection');
        }

        this.templateData = {
            apServiceState: swc.models.apServiceState.get('status'),
            schedulerState: apSchedulerState,
            devices: storageDevicesArray,
            mediaServerStatus: swc.models.MediaServices.get('status'),
            afpStatus: swc.models.AfpServices.get('status'),
            protection: protection
        };
    },

    /**
     * Update media service status in the model, after user clicked on linked checkbox
     * @param e {Object}
     * @param value {Boolean}
     */
    onMediaServiceStatusChange: function(e, value) {
        swc.models.MediaServices.set('status', value);

        // Update buttons state to see, that changes were done on the page:
        this.setButtonsState();
    },

    /**
     * Update afp status in the model, after user clicked on linked checkbox
     * @param e {Object}
     * @param value {Boolean}
     */
    onAFPStatusChange: function(e, value) {
        swc.models.AfpServices.set('status', value);

        // Update buttons state to see, that changes were done on the page:
        this.setButtonsState();
    },

    preRender: function() {
        /*
         * this call resets expanded item when the page
         * "Storage devices" is opened
         */
        this.resetExpandableItem();
    },


    registerValidationListener: function($element) {
        if (!$element.size()) {
            return;
        }

        $element.on('validation-success', function() {
            $element.toggleClass('valid', !_.isEmpty($element.val()));
        });

        $element.on('validation-fail', function() {
            $element.removeClass('valid');
        });
    },

    renderComplete: function() {
        var username  =  this.$el.find("input.username"),
            pass      =  this.$el.find("input.password-original"),
            pass_copy =  this.$el.find("input.password-copy");

        this.registerValidationListener(username);
        this.registerValidationListener(pass);
        this.registerValidationListener(pass_copy);
        
        this.eventDispatcher = swc.models.EventDispatcher;
        this.eventDispatcher.observ("storage_devices",this);

        swc.models.StorageDevices.autoSync = false;
        swc.models.CentralStorage.autoSync = false;

        this.updatePageUI();
        this.openExpandableItem();

        if(this.disableStorageProtection === false) {
            this.setProtectionPasswordState(null, false);
            swc.models.StorageProtection.autoSync = true;
        }
        // $(".protection-form").hide();


        // $.when(swc.models.StorageProtection.sync())
        //      .done(function () {
        //         swc.models.StorageProtection.autoSync = false;
        //          if (swc.models.StorageProtection.get("protection").enabled) {
        //              $(".protection-form").show();
        //          }
        //      });

        $('.page-overview.submenu').find('a').removeClass('active');
    },

    onListenerComplete: function() {
        // FIXME: Remove this once correct implementation of router done
        // This protects from rendering device list on a wrong page
        if (Backbone.history.fragment !== swc.settings.application.get("navigation")['data']) {
            //this.stopListener();
            return;
        }

        this.setTemplateData();
        this.getTemplateContent();
        this.displayPage();

        this.updatePageUI();
        this.openExpandableItem();
        this.stopPageLoading();
    },

    updatePageUI: function() {
        var pageState = true,
            apState = swc.models.apServiceState.get('status'),
            apServiceLoadingState = swc.models.apServiceLoadingState.get('isLoading'),
            apStateSwitcher = this.$el.find('.set-ap-state');

        // Check if AP is loading or OFF now, and disable tabs and page content
        if (apState !== true || apServiceLoadingState === true) {
            pageState = false;
        }

        // If AP has started -> fully reload page: (because of child view is not updating)
        if (pageState === true && this.isPageEnabled === false) {
            //this.stopListener();
            this.render();
        } else {
            // Disable / Enable page and tabs based on page status:
            this.setPageState(pageState);
            this.setTabsState(pageState);

            // Check if AP is OFF now and show disabled message:
            if (apState === false) {
                this.showDisabledMessage({
                    container: '.apStateOffMessage',
                    className: 'above'
                });
            }

            // Check if AP is loading now and show loading message:
            if (apServiceLoadingState === true && apState === true) {
                this.showDisabledMessage({
                    container: '.apStateLoadingMessage',
                    className: 'above'
                });
            }

            // Set correct switcher position:
            apStateSwitcher.trigger('swc-switcher:swc-change', apState);

            // Start listener if page is disabled or loading:
            //this.listenerEnabled = true;
            //this.startListener();
        }
    },

    save: function() {
        function readCurrentStorageProtectionSettings() {
             var enabled = swc.models.StorageProtection.get('enabled');
             return {
                 "enabled": enabled,
                 "username": $("input[name='storage_protection_username']").val(),
                 "password": $("input[name='storage_protection_password']").val()
             };
        }

        var deferred = new $.Deferred(),
            toDo = [
                swc.models.MediaServices.sync('update'),
                swc.models.AfpServices.sync('update')
            ];

        if(this.disableStorageProtection === false) {
            toDo.push(swc.models.StorageProtection.updateConfig(readCurrentStorageProtectionSettings()));
        }

        $.when.apply(this, toDo)
            .done(function() {
                deferred.resolve();
            })
            .fail(function() {
                deferred.reject();
            })
            .always(function() {
            });

        return deferred.promise();
    },

    /**
     * Update application part status:
     * @param e {Object}
     * @param value {String}
     */
    setApState: function(e, value) {
        //var self = this;

        //this.stopListener();
       // this.listenerEnabled = true;

        this.showPageLoading("Updating Central Storage state");

        // Update parameter in the model:
        swc.models.apServiceState.set('status', value);

        // Start listener again when data saved:
        $.when(swc.models.apServiceState.sync('update')).done(function() {
            //self.startListener();
        });
    },

    unmountDevice: function(e) {
         var self = this,
            element = $(e.target),
            deviceId = element.closest('.expandable-list-item').data('key'),
            deviceName = element.closest('.expandable-list-item').find(".name.state-default").text();

        self.blockEvent = true;

        // as unmount request themselves doesn't check session state
        // the checkSessionState() method should wrap it
        swc.models.System.checkSessionState(function() {
            SWCElements.modalWindow.show({
                className: 'unmount-device',
                templateID: 'storage:settings:modal-windows:unmount-device-process',
                templateData: {
                    localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                    localeString: getTranslationStringsjQuery,
                    formatDate: swc.models.Locale.formatDate,
                    deviceName: deviceName
                },
                onShow: function(){
                    $.when(swc.models.StorageDevices.unmountDevice(deviceId)).done(function(){
                        self.unmountConfirm(deviceName);
                    });
                }
            });
        });
    },

    unmountConfirm: function(deviceName) {
        var self = this;

        SWCElements.modalWindow.hide();
        SWCElements.modalWindow.show({
            templateID: 'storage:settings:modal-windows:unmount-device-complete',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate,
                deviceName: deviceName
            },
            className: 'unmount-device',
            onApply: function(){
                self.blockEvent = false;
                SWCElements.modalWindow.hide();
                self.resetExpandableItem();
                self.render();
            }
        });
    },

    /**
     * Reset current opened item value
     */
    resetExpandableItem: function() {
        this.openedItem = null;
    },

    /**
     * Trigger expandable list item according to
     * current opened item value
     */
    openExpandableItem: function() {
        if (!_.isNull(this.openedItem)) {
            this.$el.find('.expandable-list').trigger('expandable:swc-open', this.openedItem);
        }
    },

    /**
     * Event handler: is fired on open expandable list item
     *
     * @param e
     * @param value
     */
    onExpandableOpen: function(e, value) {
            this.openedItem = value;
            $(e.currentTarget).find(".device,  .name").removeClass("selected");
            $(e.currentTarget).find(".expanded .device,.expanded .name").addClass("selected");
    },

    /**
     * Event handler: is fired on close expandable list item
     *
     * @param e
     * @param value
     */
    onExpandableClose: function(e, value) {
        if (this.openedItem === value) {
            this.resetExpandableItem();
        }
        //remove hovers on all devices
        this.$el.find(".name, .device").removeClass('selected');
    },
    /**
     * Event handler: is fired on hovering over list item
     *
     * @param e
     */
    onHover: function (e){
	//adding class is more  flexible than triggering hover effects in css, adding new icons is easier
	if(e.type ==="mouseenter") {
            $(e.currentTarget).find(".name, .device").addClass('selected');
	}
	else {
             if(!$(e.currentTarget).parent().parent().parent().hasClass("expanded")) {
                $(e.currentTarget).find(".name, .device").removeClass('selected');
             }
	}
    },
    /**
     * Event handler: is fired on clicking over name and icon
     *
     * @param e
     */
    onExpandTrigger: function(e){
	//getting parent container for state "expanded" and buttons to trigger click events
	var showBtn = $(e.currentTarget).parent().parent().find(".show-hidden.default"),
            hideBtn = $(e.currentTarget).parent().parent().find(".show-hidden.expanded"),
            parent = $(e.currentTarget).parent().parent().parent();
	
	if($(parent).hasClass("expanded")) {
            $(hideBtn).trigger("click");
	}
	else {
           $(showBtn).trigger("click");
	}
	//clears all selections on page
	this.$el.find(".name, .device").removeClass('selected');
	
	//after click hovers lose focus allows buttons to be highlited
	$(e.currentTarget).find(".name, .device").addClass('selected');
	
    },

    updateDeviceList: function() {
        var self = this;
        setTimeout(function() {
            var storageDevicesArray = [];
            if(!self.blockEvent) {
                var dataDevices = $('.dataStorageDevicesForList');
                $.when(self.loadModels(false)).done(function() {
                    swc.models.StorageDevices.each(function(model) {
                        var isSupported=true;
                        //chceck if device is supported
                        if(model.get('type') === "usb-disk-undefined" || model.get('type') === "sd-card-undefined"){
                            isSupported= false;
                        }
                        var deviceObj = {
                            id: model.id,
                            label: model.get('label'),
                            type: model.get('type'),
                            capacity: model.get('capacity'),
                            freeSpace: model.get('freeSpace'),
                            deviceIsSupported: isSupported
                        };

                        storageDevicesArray.push(deviceObj);
                    });

                    var devicesTemplate = $.template("devicesForm",
                        swc.Templates.get("storage:data:devices-for-list").get('content')),
                    tmpl = $.tmpl(devicesTemplate, {
                            localeString: getTranslationStringsjQuery,
                            localeStrings: swc.models.Locale.getLocaleStrings(),
                            devices: storageDevicesArray,
                            apServiceState: swc.models.apServiceState.get('status')
                        }),
                    settingsFormContainer = self.$(".dataStorageDevicesForList");
                    dataDevices.html('');
                    settingsFormContainer.html(tmpl);
                }).fail(function() {
                });
            }
        }, 1000);
    },

    setProtectionPasswordState: function(e, value) {
        var secretOrigBox = $('input.secret-original'),
            secretCopyBox = $('input.secret-copy'),
            passwordOrigBox = $('input.password-original'),
            passwordCopyBox = $('input.password-copy');

        if (value) {
            passwordOrigBox.hide();
            passwordCopyBox.show();
            secretOrigBox.hide();
            secretCopyBox.show();
        } else {
            passwordCopyBox.hide();
            passwordOrigBox.show();
            secretCopyBox.hide();
            secretOrigBox.show();
        }
    },
    copyPassword: function(e) {
        var element = $(e.target),
            visibleField,
            hiddenField;

        if (element.hasClass('secret-original')) {
            visibleField = this.$('input.secret-original');
            hiddenField = this.$('input.secret-copy');
        } else {
            visibleField = this.$('input.secret-copy');
            hiddenField = this.$('input.secret-original');
        }

        if (element.hasClass('password-original')) {
            visibleField = this.$('input.password-original');
            hiddenField = this.$('input.password-copy');
        } else {
            visibleField = this.$('input.password-copy');
            hiddenField = this.$('input.password-original');
        }

        hiddenField.val(visibleField.val());
    },
    setProtectionEnable: function (e, value) {
        var protectionForm = $('.protection-form');

        if (value) {
            protectionForm.show();
        } else {
            protectionForm.hide();
        }

        swc.models.StorageProtection.set('enabled', value);
    }


});
;swc.constructors.StorageSettingsView = swc.base.PageView.extend({

    className: 'storage-settings',

    sharingState: true,

    events: {
        'swc-switcher:change .set-ap-state': 'setApState'
    },

    listenerInterval: 10,

    eventDispatcher:  null,

    listenerEnabled: true,

    listenerPages: [ 'data-sharing' ],

    models: [ 'apServiceState', 'apServiceLoadingState', 'СloudServicesStatus', 'SchedulerAp' ],

    setTemplateData: function() {
        var apScheduler = swc.models.SchedulerAp,
            apSchedulerState = !_.isUndefined(apScheduler) ? apScheduler.get('enable') : false;

        this.templateData = {
            apServiceState: swc.models.apServiceState.get('status'),
            cloudServicesState: swc.models.СloudServicesStatus.get('isEnabled'),
            schedulerState: apSchedulerState
        };
    },

    /**
     * Check if application part is starting or disabled and disable / enable page
     *
     * <apState> can be :: `on`, `off`, `loading`
     */
    renderComplete: function() {
        this.updatePageUI();
    },

    onListenerComplete: function() {
        this.stopPageLoading();
        this.updatePageUI();
        
    },

    updatePageUI: function() {
        var pageState = true,
            apState = swc.models.apServiceState.get('status'),
            apServiceLoadingState = swc.models.apServiceLoadingState.get('isLoading'),
            apStateSwitcher = this.$el.find('.set-ap-state');

        // Check if AP is loading or OFF now, and disable tabs and page content
        if (apState !== true || apServiceLoadingState === true) {
            pageState = false;
        }

        // If AP has started -> fully reload page: (because of child view is not updating)
        if (pageState === true && this.isPageEnabled === false) {
            this.stopListener();
            this.render();
        } else {
            // Disable / Enable page and tabs based on page status:
            this.setPageState(pageState);
            this.setTabsState(pageState);

            // Check if AP is OFF now and show disabled message:
            if (apState === false) {
                this.showDisabledMessage({
                    container: '.apStateOffMessage',
                    className: 'above'
                });
            }

            // Check if AP is loading now and show loading message:
            if (apServiceLoadingState === true && apState === true) {
                this.showDisabledMessage({
                    container: '.apStateLoadingMessage',
                    className: 'above'
                });
            }

            // Set correct switcher position:
            apStateSwitcher.trigger('swc-switcher:swc-change', apState);

            // Start listener if page is disabled or loading:
            this.listenerEnabled = true;
            this.startListener();
        }
    },

    /**
     * Update application part status:
     * @param e {Object}
     * @param value {String}
     */
    setApState: function(e, value) {
        var self = this,
            pageRoutes = swc.settings.application.get('navigation'),
            page = Backbone.history.fragment;

        this.stopListener();
        this.listenerEnabled = false;

        // Update parameter in the model:
        swc.models.apServiceState.set('status', value);

        // Start listener again when data saved:
        $.when(swc.models.apServiceState.sync('update')).done(function() {
            if (pageRoutes['data-sharing'] === page) {
                swc.models.apServiceState.fetch();
                self.listenerEnabled = true;
                self.startListener();
            } else {
                swc.router.navigate(pageRoutes['data-sharing'], { skipUnsavedChanges: true });
            }
        });
    }
});
;swc.constructors.DevelopersView = swc.base.PageView.extend({

    className: 'developers',

    setTemplateData: function() {
        this.templateData = {
            webui: swc.settings.application.get('build') || {},
            jenkins: swc.settings.application.get('jenkins') || {}
        };
    }
});;swc.constructors.NetworkDevicesView = swc.base.PageView.extend({

    className: 'network-device-list',

    events: {
        'expandable:open .expandable-list': 'onExpandableOpen',
        'expandable:close .expandable-list': 'onExpandableClose',
        'expandable:delete .expandable-list': 'removeDevice',

        'click .wps-pairing:not(.disabled)': 'startWpsPairing',

        'click .customize-device': 'editDevice',

        'hover .expandable-list-item:not(.expanded) .block.device-name': 'highlightIcon'
    },

    models: [
        'NetworkDevices',
        'DHCPLeases',
        'PortForwarding',
        'FirewallStatus',
        'SchedulerDeviceCollection',
        'SchedulerDeviceManager',
        'EventDispatcher'
    ],

    eventActions: [
        {
            event: 'wifi_device_updated',
            action: 'networkDevicesUpdated'
        },
        {
            event: 'eth_device_updated',
            action: 'networkDevicesUpdated'
        }
    ],

    listenerEnabled: false,

    listenerPages: [
        'network-devices'
    ],

    /**
     * Defines which item is expanded at the current moment
     * It can have the following values
     * 'default'     - default value which is assigned automatically,
     *                 can be changed to <MAC address> by script when user opens the page
     * 'closed'      - this value is assigned when the user closes the item manually,
     *                 can NOT be changed by script
     * <MAC address> - this value is assigned when the user opens the item manually,
     *                 can be changed to 'default' by script when user removes the opened item
     */
    openedItem: 'default',

    highlightIcon: function(event) {
        var $icon = $(event.target).closest('.device-name').find('.device.small');

        if (event.type === 'mouseenter') {
            $icon.addClass('selected');
        } else {
            $icon.removeClass('selected');
        }
    },

    networkDevicesUpdated: function(event) {
      swc.Utils.updateDevicesByModelName("NetworkDevices", event);
      swc.models.NetworkDevices.getConflicts();
    },

    setTemplateData: function() {
        var self = this,
            devices = swc.models.NetworkDevices.getDevicesList(),
            DHCPLeases = _.pluck(swc.models.DHCPLeases.getActiveLeases(), 'MACAddress');

        /*
          Make matching between device collection and schedule collection
          After invoking of this method the swc.models.SchedulerDeviceCollection is adjusted
        */
        swc.models.SchedulerDeviceManager.applySchedulesToDevices(
            swc.models.NetworkDevices,
            swc.models.SchedulerDeviceCollection
        );

        var updatedDevices = _.map(devices, function (originalDevice) {
            var aDevice = _.clone(originalDevice, true);

            if( _.contains(DHCPLeases, aDevice.mac)) {
                aDevice.address['static'] = true;
            }

            if (aDevice.addressObjects.IPv6 && aDevice.addressObjects.IPv6.length > 0) {
                $.each(aDevice.addressObjects.IPv6, function(addressKey, value) {
                    aDevice.address.IPv6[addressKey] = swc.Utils.formatIPv6Address(value.Address);
                });
            }
            aDevice['portForwarding'] = self.getPortForwardingInfo(aDevice.address.IPv4);
            aDevice['parentalControlStatus'] = swc.models.SchedulerDeviceCollection.get(aDevice.mac).get('enable');
            
            return aDevice;
        });

        this.templateData = {
            wifiState: swc.models.Wireless.getParameter("status", "status"),
            WPSStatus: swc.models.Wireless.getParameter('WPSEnable', 'status'),
            devices: updatedDevices,
            IPv6Enabled: swc.models.FirewallStatus.get('state') ? swc.models.FirewallStatus.get('state').Enable : false
        };
    },
    passEvent: function(event, eventObject) {
      swc.constructors.EventController(this.eventActions)
                      .pass.call(this, event, eventObject);
    },
    /**
     * According to implemented listener mechanism
     * this method is invoked only once.
     * In this method should be performed some init
     * actions
     *          and
     * the global object variable
     * this.openedItem set to NULL !!!
     */
    preRender: function() {
        var self = this;
         /*
         * this call resets expanded item when the page
         * "Device list" is opened
         */
        this.resetExpandableItem();
        //add event listener
        this.eventDispatcher=swc.models.EventDispatcher;
        this.eventDispatcher.observ("network_device_list",this);
        swc.models.NetworkDevices.autoSync=false;
        swc.models.DHCPLeases.autoSync=false;
        swc.models.PortForwarding.autoSync=false;
        swc.models.FirewallStatus.autoSync=false;
        swc.models.SchedulerDeviceCollection.autoSync=false;
        swc.models.SchedulerDeviceManager.autoSync=false;
        swc.models.NetworkDevices.on('devices-loaded', function() {
            self.onListenerComplete();
        });
    },

    /**
     * Hook renderComplete: is fired after rendering
     * of the page and before listener's start
     */
    renderComplete: function() {
        /*
         * After render the page the appropriate
         * expandable item should be opened
         */
        this.openExpandableItem();
        swc.router.isLoadingModule = false;
    },

    /**
     * Hook onListenerComplete: is fired as final part
     * of listener and performs only necessary render
     * actions instead of whole render process
     */
    onListenerComplete: function() {
        if(!_.isUndefined(swc.router) && swc.router.isLoadingModule !== true) {
            this.setTemplateData();
            this.getTemplateContent();
            this.openExpandableItem();
        }
    },

    getPortForwardingInfo: function(ip) {
        var objects = swc.models.PortForwarding.where({"deviceIP": ip}),
            jsonData = _.map(objects, function(item) {
                return item.attributes;
            });

        return jsonData;
    },

    removeDevice: function(e, mac){
        var ruleItem = $('.expandable-list-item[data-key="' + mac + '"]');

        ruleItem.fadeOut(function(){
            ruleItem.remove();
        });

        swc.models.NetworkDevices.removeDevice(mac);
        this.closeExpandableItem();
        this.openExpandableItem();
    },

    editDevice: function(e) {
        var self = this,
            element = $(e.target),
            type = element.data('type'),
            macAddress = element.data('macaddress'),
            device = swc.models.NetworkDevices.getDevice(macAddress);

        this.customizeDevice({
            device: device,
            type: type,

            onApply: function(options) {
                self.showPageLoading("Updating device information");

                $.when(swc.models.NetworkDevices.customizeDevice(options)).done(function() {
                    self.render();
                });
            }
        });
    },

    /**
     * Event handler: is fired on the pushing button "Start WPS"
     */
    startWpsPairing: function() {
        // lazy mixin:
        _.extend(this, swc.constructors.WpsPairingMixin);
        this.startWpsPairing();
    },

    /**
     * Set current opened item as "default"
     */
    resetExpandableItem: function() {
        this.openedItem = 'default';
    },

    /**
     * Set current opened item as "close"
     */
    closeExpandableItem: function() {
        this.openedItem = 'closed';
    },

    /**
     * Open expandable list item according to
     * current opened item value
     */
    openExpandableItem: function() {
        var devices = swc.models.NetworkDevices;

        /*
         * if selected(opened) item was disconnected
         * then current opened item should be reset
         */
        if (this.openedItem === 'closed' && devices.length && _.isEmpty(devices.where({ mac: this.openedItem }))) {
            this.resetExpandableItem();
        }

        /*
         * if no current opened item
         * then first item should be opened
         */
        if (this.openedItem === 'default' && swc.models.NetworkDevices.length) {
            // find first connected
            var device = _.find(devices.models, function(item) {
                return item.get('status');
            });
            if(device) {
                this.openedItem = device.get('mac');
            }
        }

        if (this.openedItem !== 'default' && this.openedItem !== 'closed') {
            this.$el.find('.expandable-list').trigger('expandable:swc-open', this.openedItem);
        }
    },

    /**
     * Event handler: is fired on open expandable list item
     *
     * @param e
     * @param value
     */
    onExpandableOpen: function(e, value) {
        this.openedItem = value;
    },

    /**
     * Event handler: is fired on close expandable list item
     *
     * @param e
     * @param value
     */
    onExpandableClose: function(e, value) {
        if (this.openedItem.toString() === value.toString()) {
            this.closeExpandableItem();
        }
    }

});
;swc.constructors.NetworkSettingsDyndnsView = swc.base.TabView.extend({

    className: 'dyndns',

    models: [
        'DynDNS',
        'DynDNSProviderCollection'
    ],

    allowedMods: ['expert'],

    disallowedDevices: ['starlink'],

    settingsFormTemplate: '',

    events: {
        'keyup input.password-original': 'updatePasswordsChange',
        'keyup input.password-copy': 'updatePasswordsChange',
        'change input.password-original': 'updatePasswordsChange',
        'change input.password-copy': 'updatePasswordsChange',
        'swc-checkbox:change .swc-checkbox.show-password': 'setPasswordState',
        'swc-dropdown:change .swc-dropdown.provider-selection': 'changeService',
        'swc-checkbox:change .swc-checkbox.DynDNSEnable': 'setGlobalState',
        'validation-success input.hostname': 'validationSuccess',
        'validation-fail input.hostname': 'validationFail',
        'validation-success input.username': 'validationSuccess',
        'validation-fail input.username': 'validationFail',
        'validation-success input.password': 'validationSuccess',
        'validation-fail input.password': 'validationFail'
    },

    validation: {
        'Service': 'DynDNSProvider:service',
        'hostname': 'DynDNSProvider:hostname',
        'username': 'DynDNSProvider:username',
        'password': 'DynDNSProvider:password'
    },
    
    isAbleToNavigate: function() {
        var userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin();

        if(userIsSuperAdmin === true) {
            return false;
        }
        return true;
    },
    
    validationSuccess: function(event) {
        var self = this,
            $element = $(event.currentTarget);
        
        $element.toggleClass('valid', !_.isEmpty($element.val()));

        self.pageCheckDefaultValues();
        
        // if all fields are empty, reset validation
        if(self.isAllEmpty()) {
            self.resetValidationState();
        }
    },
    
    isAllEmpty: function() {
        var hostname = this.$('input.hostname'),
            username = this.$('input.username'),
            password = this.$('input.password');

        return (_.isEmpty(hostname.val()) && _.isEmpty(username.val()) && _.isEmpty(password.val()));
    },

    validationFail: function(event) {
        var $element = $(event.currentTarget);
        
        // Remove green mark when validation fails:
        $element.removeClass('valid');
    },

    setTemplateData: function() {
        this.templateData = {
            dyndnsStatus: swc.models.DynDNS.get('enable')
        };
    },

    initialize: function () {
        swc.base.TabView.prototype.initialize.apply(this, arguments);

        // Prepare template for the settings form:
        this.settingsFormTemplate = $.template("settingsForm",
            swc.Templates.get("network:settings:dyndns:service-settings-form").get('content'));
    },

    renderComplete: function() {
        var self = this;

        if (!swc.models.DynDNSServiceCollection) {
            swc.models.DynDNSServiceCollection = new swc.constructors.DynDNSServiceCollection();
        }

        $.when(swc.models.DynDNSServiceCollection.fetch())
            .done(function() {
                // Find currently active provider and set it as selected
                var provider = swc.models.DynDNSProviderCollection.findWhere({enable: true}),
                    defaultProvider = swc.models.DynDNSServiceCollection.at(0),
                    providerId = provider ? provider.get('service') : '';

                // According to requirements, when no providers configured yet -
                // first element in option list should be selected
                if (!providerId) {
                    providerId = defaultProvider ? defaultProvider.get('id') : '';
                }

                self.changeService(null, providerId);
                self.setProvidersData(providerId);
                self.showState();
            });
    },
    
    /**
     * Handler for validation event triggered on the field:
     *
     * @description:
     *
     * When the page was reset to default values all validation messages should be cleared from the page
     */
    resetValidationState: function() {
        var self = this,
            $elements = [
                this.$('input.hostname'),
                this.$('input.username'),
                this.$('input.password')
            ];

        // Go through each element and clear it's validation state:
        _.each($elements, function($element) {
            if ($element.size()) {
                var $errorMessagesSection = self.$('.validation-message[data-parameter-name="' + $element.attr('name') + '"]');

                // Remove valitation errors from the element:
                $element.removeClass('valid validation-error');

                // Hide error messages section:
                $errorMessagesSection.hide();
                $errorMessagesSection.find('span').hide();
            }
        });
    },

    showState: function() {
        var state = swc.models.DynDNS.get('enable');

        this.$('.swc-dropdown, .swc-checkbox.show-password').toggleClass('disabled', !state);

        this.setPageState(state, {disabledBlockClass: "faded"});

        if (state === false) {
            this.showDisabledMessage({
                container: '.dynDnsOffMessage',
                className: 'above'
            });
            this.$('input').attr('disabled', 'disabled');
        } else {
            this.hideDisabledMessage();
            this.$('input').removeAttr('disabled');
        }
    },

    /**
     * Set active provider as selected in dropdown list
     *
     * @param {String} providerId
     */
    setProvidersData: function(providerId) {
        var options = swc.models.DynDNSServiceCollection.formatProvidersOptions(),
            dropdown = this.$('.provider-selection');

        dropdown.data('options', options);
        dropdown.data('default-value', providerId);
        dropdown.trigger('swc-dropdown:swc-change', providerId);
    },

    /**
     * Show form with auth data for the selected service
     * @param e
     * @param {String} value Provider to select
     */
    changeService: function(e, value) {
        moment.lang(swc.models.Locale.locale);

        var provider = swc.models.DynDNSProviderCollection.findWhere({service: value}),
            dnsProviderData;

        if (!_.isEmpty(provider)) {
            var lastUpdate = provider.get('last_update');

            if (lastUpdate && lastUpdate !== "0001-01-01T00:00:00Z") {
                moment.lang(swc.models.Locale.locale);
                lastUpdate = moment(lastUpdate).zone(lastUpdate).format("DD. MMM YYYY, HH:mm");
            } else {
                lastUpdate = '';
            }

            dnsProviderData = {
                service: provider.get('service').capitalize(),
                status: provider.get('status'),
                lastUpdate: lastUpdate
            };
        }

        // Render template with data
        var tmpl = $.tmpl(this.settingsFormTemplate, {
                service: provider ? provider.attributes : (new swc.constructors.DynDNSProvider()).attributes,
                localeString: getTranslationStringsjQuery,
                localeStrings: swc.models.Locale.getLocaleStrings(),
                dyndnsStatus: swc.models.DynDNS.get('enable'),
                dnsProviderData: dnsProviderData
            }),
            settingsFormContainer = this.$(".table.provider-data");

        console.log(tmpl);
        // Render template
        settingsFormContainer.html(tmpl);
    },

    /**
     * Show textual representation of a password.
     * Uses addition hidden field to store clone of value been input into password field.
     *
     * @param e
     * @param value
     */
    setPasswordState: function(e, value) {
        var providerSettingsFormContainer = this.$('.table.provider-data'),
            passwordOrigBox = providerSettingsFormContainer.find('input.password-original'),
            passwordCopyBox = providerSettingsFormContainer.find('input.password-copy');

        if (value) {
            passwordOrigBox.hide();
            passwordCopyBox.show();
        } else {
            passwordCopyBox.hide();
            passwordOrigBox.show();
        }
    },

    /**
     * Copies value from password field into hiddent text field char-by-char.
     */
    updatePasswordsChange: function() {
        var providerSettingsFormContainer = this.$('.table.provider-data'),
            passwordOrigBox = providerSettingsFormContainer.find('input.password-original'),
            passwordCopyBox = providerSettingsFormContainer.find('input.password-copy'),
            flag;

        // Copy values from one to another and vice-versa
        if (passwordOrigBox.is(":visible")) {
            passwordCopyBox.val(passwordOrigBox.val());
        } else {
            passwordOrigBox.val(passwordCopyBox.val());
        }

        flag = swc.Utils.validatePassword(passwordCopyBox.val());
        passwordOrigBox.toggleClass('valid', flag);
        passwordCopyBox.toggleClass('valid', flag);
    },

    setGlobalState: function (e, value) {
        swc.models.DynDNS.set('enable', value);
        this.showState();
    },

    save: function() {
        var self = this,
            deferred = new $.Deferred(),
            pageData = {},
            dynDnsStatus = swc.models.DynDNS.get('enable');

        // Get elements name => value map
        $.each(this.getElements(), function(key, element) {
            var parameter = getParameter($(element));
            pageData[parameter['parameterName']] = parameter['parameterValue'];
        });

        console.log(pageData);

        var userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin();
        var dynDNSProviderData = {
            'service': pageData['Service'],
            'hostname': pageData['hostname'],
            'username': pageData['username'],
            'password': pageData['password'],
            'enable': true
        };

        // Fill in data for selected service provider
        var provider = new swc.constructors.DynDNSProvider(dynDNSProviderData);

        /**
         * Add newly created provider
         *
         * @private
         *
         * @return Deferre.Promise
         */
        var addNewProvider = function() {
            // Add newly created provider
            $.when(provider.sync("create"))
                .done(function (response) {
                    if (response.status === true) {
                        // Re-read collection of added hosts
                        swc.models.DynDNSProviderCollection.fetch();
                        deferred.resolve();
                    } else {
                        deferred.reject("Fill in all fields");
                    }
                })
                .fail(function (xhr, error, status) {
                    deferred.reject(status);
                });
        };

        // Update global DynDNS status & save new DynDNS provider marked as 'enabled'
        $.when(swc.models.DynDNS.sync("update"))
            .done(function () {
                // Do not re-add services if we deactivated DynDNS globally
                if (dynDnsStatus && userIsSuperAdmin !== true) {
                    // we actually always will have only one item in collection
                    var todo = [];

                    swc.models.DynDNSProviderCollection.forEach(function (aProvider) {
                        todo.push(aProvider.destroy());
                    });

                    $.when.apply(this, todo)
                        .done(function () {
                            if(self.isAllEmpty()) {
                                deferred.resolve();
                            } else {
                                addNewProvider();
                            }
                        })
                        .fail(function () {
                            deferred.reject();
                        });
                } else {
                    deferred.resolve();
                }
            })
            .fail(function() {
                deferred.reject();
            });

        return deferred.promise();
    }
});
;swc.constructors.NetworkSettingsIpv6firewallAddView = swc.base.TabView.extend({

    className: 'ipv6-firewall-rules-add',

    allowedMods: [ 'expert' ],

    disallowedDevices: ['starlink'],

    events: {
        'swc-radio-buttons:change .swc-radio-buttons.service-mode': 'onRulesModeChange',
        'swc-radio-buttons:change .swc-radio-buttons.ports-mode': 'onPortsModeChange',
        'swc-radio-buttons:swc-change .swc-radio-buttons.ports-mode': 'onPortsModeChange',

        'swc-dropdown:change .swc-dropdown.policy-selection': 'onPolicyChange',
        'swc-dropdown:change .swc-dropdown.service-name-selection': 'onPredefinedRuleSelection',

        'input input[name="RulePort"].from': 'limitSetOfCharacters',
        'input input[name="RulePort"].to': 'limitSetOfCharacters',

        'keydown input[name="RulePort"].from': 'limitSetOfCharactersDown',
        'keydown input[name="RulePort"].to': 'limitSetOfCharactersDown'
    },

    models:[
        'FirewallPolicy',
        'FirewallRules'
    ],

    validation: {
        'RuleName': 'FirewallRules:RuleName',
        'RulePort': 'FirewallRules:RulePort',
        'PredefinedRuleName': 'FirewallRules:PredefinedRuleName'
    },

    setTemplateData: function() {
        this.templateData = {
            isAdding: true,
            predefinedRulesList: swc.models.FirewallRules.predefined,
            ruleType: "custom",
            ruleName: "",
            ruleProtocol: "6,17",
            rulePortsMode: "single",
            rulePorts: [ "" ],
            rulePolicy: "Drop_inbound"
        };
    },

    initialize: function() {
        this.template = $.template("pageContent", swc.Templates.get('network:settings:ipv6-firewall:rules-add').get('content'));
        this.events = _.extend({}, swc.base.PageView.prototype.events, this.events);
    },

    hasChanged: function(){
        return !this.pageCheckDefaultValues();
    },

    renderComplete: function() {
        this.setPredefinedRulesDropdown();
        this.resetFormValues();
    },

    /**
     * Reset rule adding / editing form to its default values
     *
     * @description:
     *
     * If values are passed:
     *
     *  Page will be filled in with values which are passed as arguments to the function.
     *
     * If values are not passed:
     *
     *  Page will be filled in with values which are set in <this.templateData>
     *
     * @param values {Object} same structure as <this.templateData>
     */
    resetFormValues: function(values) {
        var formValues = !_.isUndefined(values) ? values : this.templateData,
            $ruleNameInput = this.$('input[name="RuleName"]'),
            $rulePortFromInput = this.$('input[name="RulePort"].from'),
            $rulePortToInput = this.$('input[name="RulePort"].to'),
//            $ruleTypeRadioGroup = this.$('.swc-radio-buttons.service-mode'),
            $rulePortRadioGroup = this.$('.swc-radio-buttons.ports-mode'),
//            $ruleNameDropdown = this.$('.swc-dropdown.service-name-selection'),
            $ruleProtocolDropdown = this.$('.swc-dropdown.protocol-selection'),
            $rulePolicyDropdown = this.$('.swc-dropdown.policy-selection');

        // Update page dropdown values:
        this.setProtocolDropdown(formValues.ruleProtocol);
        this.setPolicyDropdown(formValues.rulePolicy);

        // Remove all previous validation messages:
        this.clearValidationMessages();

        // Reset rule name:
        $ruleNameInput.val(formValues.ruleName);

        // Reset rule protocol value:
        $ruleProtocolDropdown.trigger('swc-dropdown:swc-change', formValues.ruleProtocol);

        // Reset rule ports and rule ports mode:
        $rulePortFromInput.val(formValues.rulePorts[0]);
        $rulePortRadioGroup.trigger('swc-radio-buttons:swc-change', formValues.rulePortsMode);

        if (formValues.rulePortsMode === "range") {
            $rulePortToInput.val(formValues.rulePorts[1]);
        } else {
            $rulePortToInput.val("");
        }

        // Reset rule policy value:
        $rulePolicyDropdown.trigger('swc-dropdown:swc-change', formValues.rulePolicy);
    },

    /**
     * Handler for changing rule adding type:
     *
     * @param e {Object}
     * @param value {String}
     */
    onRulesModeChange: function(e, value) {
        var type = !_.isUndefined(value) ? value : this.templateData.ruleType;

        if (type === "predefined") {
            this.onPredefinedRuleSelection();
        } else {
            this.resetFormValues();
        }

        this.$('.rule-name-section').toggleClass("type-predefined", value === "predefined");

        this.setButtonsState();
    },

    /**
     * Handler for changing rule ports mode (single / range):
     *
     * @param e {Object}
     * @param value {String}
     */
    onPortsModeChange: function(e, value) {
        if (value !== "range") {
            this.$('input[name="RulePort"].to').val("");
        }

        this.$('.ports-values-section').toggleClass("range-type", value === "range");
    },

    limitSetOfCharactersDown: function(e) {
        return swc.Utils.limitSetOfCharactersDown(e, /[^\d]/g);
    },

    limitSetOfCharacters: function(e) {
        return swc.Utils.limitSetOfCharactersInput(e, /[^\d]/g);
    },

    /**
     * Handler for changing predefined rules dropdown:
     *
     * @param e {Object}
     * @param value {Value}
     */
    onPredefinedRuleSelection: function(e, value) {
        var dropdown = this.$('.swc-dropdown.service-name-selection'),
            ruleKey = dropdown.data('value'),
            ruleData = {};

        // Override dropdown value with new value:
        if (!_.isUndefined(value)) {
            ruleKey = value;
        }

        // Get rule data from list of prdefined rules:
        ruleData = _.findWhere(this.templateData.predefinedRulesList, { name: ruleKey });

        // Reset form values with predefined rule:
        this.resetFormValues({
            ruleType: "predefined",
            ruleName: ruleData.name,
            ruleProtocol: ruleData.protocol,
            rulePortsMode: ruleData.port.length === 1 ? "single" : "range",
            rulePorts: ruleData.port,
            rulePolicy: 'Drop_inbound'
        });
    },

    /**
     * Handler for changing policy mode dropdown:
     */
    onPolicyChange: function() {
        this.pageValidation(true);
    },

    /**
     * Add options to protocol dropdown
     *
     * @param value {String} (optional)
     */
    setProtocolDropdown: function(value) {
        var dropdown = this.$('.swc-dropdown.protocol-selection'),
            dropdownValue = !_.isUndefined(value) ? value : this.templateData.ruleProtocol,
            options = [
                { name: getTranslationStrings("TCP/UDP"), value: "6,17" },
                { name: getTranslationStrings("TCP"),     value: "6"    },
                { name: getTranslationStrings("UDP"),     value: "17"   }
            ];

        // Set dropdown data:
        dropdown.data('options', options);
        dropdown.trigger('swc-dropdown:swc-change', dropdownValue);
    },

    /**
     * Add options to policy dropdown
     *
     * @param value {String} (optional)
     */
    setPolicyDropdown: function(value) {
        var dropdown = this.$('.swc-dropdown.policy-selection'),
            dropdownValue = !_.isUndefined(value) ? value : this.templateData.rulePolicy,
            options = [
                { value: "Drop_inbound", name: getTranslationStrings("Block inbound") },
                { value: "Drop_outbound", name: getTranslationStrings("Block outbound") },
                { value: "Accept_inbound", name: getTranslationStrings("Allow inbound") },
                { value: "Accept_outbound", name: getTranslationStrings("Allow outbound") }
            ];

        // Set dropdown data:
        dropdown.data('options', options);
        dropdown.trigger('swc-dropdown:swc-change', dropdownValue);
    },

    /**
     * Add options to predefined rules dropdown
     */
    setPredefinedRulesDropdown: function() {
        var rules = this.templateData.predefinedRulesList,
            dropdownValue = rules[0].name,
            dropdown = this.$('.swc-dropdown.service-name-selection'),
            options = [];

        // Create dropdown options object:
        _.each(rules, function (rule) {
            options.push({
                name: rule.name,
                value: rule.name
            });
        });

        // Set dropdown data:
        dropdown.data('options', options);
        dropdown.trigger('swc-dropdown:swc-change', dropdownValue);
    },

    /**
     * Serialize all page data to JSON object
     *
     * @returns {Object}
     */
    serializePageData: function() {
        var jsonData = {},
            $ruleKeyInput = this.$('input[name="RuleKey"]'),
            $ruleNameInput = this.$('input[name="RuleName"]'),
            $rulePortFromInput = this.$('input[name="RulePort"].from'),
            $rulePortToInput = this.$('input[name="RulePort"].to'),
            $ruleTypeRadioGroup = this.$('.swc-radio-buttons.service-mode'),
            $rulePortRadioGroup = this.$('.swc-radio-buttons.ports-mode'),
            $ruleNameDropdown = this.$('.swc-dropdown.service-name-selection'),
            $ruleProtocolDropdown = this.$('.swc-dropdown.protocol-selection'),
            $rulePolicyDropdown = this.$('.swc-dropdown.policy-selection');

        // Handle edit / add mode
        if ($ruleTypeRadioGroup.data('value') === 'custom') {
            if ($ruleKeyInput.size()) {
                jsonData.id = $ruleKeyInput.val();
            } else {
                jsonData.id = $ruleNameInput.val();
            }

            jsonData.name = $ruleNameInput.val();
        } else {
            jsonData.id = $ruleNameDropdown.data('value');
            jsonData.name = $ruleNameDropdown.data('value');
        }

        // Protocol dropdownd handling:
        jsonData.protocol = $ruleProtocolDropdown.data('value');

        // Policy dropdownd handling:
        jsonData.policy = $rulePolicyDropdown.data('value').split('_')[0];
        jsonData.policyTarget = $rulePolicyDropdown.data('value').split('_')[1];

        // Add hash to rule id in adding mode:
        if (!_.isUndefined(this.templateData.isAdding)) {
            if (jsonData.policyTarget === 'inbound') {
                jsonData.id = swc.Utils.generateId() + '_inbound';
            } else {
                jsonData.id = swc.Utils.generateId() + '_outbound';
            }
        }

        // Add chain target to rule:
        if (jsonData.policyTarget === 'inbound') {
            jsonData.chain = 'Custom_V6In';
        } else {
            jsonData.chain = 'Custom_V6Out';
        }

        // Handle edit / add mode
        jsonData.status = !_.isUndefined(this.templateData.ruleState) ? this.templateData.ruleState : true;

        // Single / range port mode handling:
        if ($rulePortRadioGroup.data('value') === "single") {
            jsonData.port = [ $rulePortFromInput.val() ];
        } else {
            jsonData.port = [ $rulePortFromInput.val(), $rulePortToInput.val() ];
        }

        return jsonData;
    },

    /**
     * @override
     *
     * On this page it's needed to handle only "save" button state:
     *
     * @param e {Object}
     * @param toDisable {Boolean}
     */
    setButtonsState: function(e, toDisable) {
        var container = this.$('.buttons-container-message'),
            buttonSave = container.find('.button.save-changes'),
            allValuesDefault = this.pageCheckDefaultValues();

        buttonSave.toggleClass('disabled', toDisable || allValuesDefault);
    },

    /**
     * @override
     *
     * On this page cancel changes button should redirect user to other page:
     */
    cancelChanges: function() {
        swc.router.navigate('network/settings/ipv6-firewall/customize', { skipUnsavedChanges: true });
    },

    /**
     * @override
     *
     * On this page save changes button should redirect user to other page after saving complete:
     */
    save: function() {
        var ruleData = this.serializePageData(),
            deferred = new $.Deferred();

        $.when(swc.models.FirewallRules.createRule(ruleData))
            .done(function() {
                localStorage.setItem('firewall:change-rule', 'done');
                swc.router.navigate('network/settings/ipv6-firewall/customize', { skipUnsavedChanges: true });
            })
            .fail(function() {
                localStorage.setItem('firewall:change-rule', 'fail');
                swc.router.navigate('network/settings/ipv6-firewall/customize', { skipUnsavedChanges: true });
            });

        return deferred.promise();
    }
});
;swc.constructors.NetworkSettingsIpv6firewallEditView = swc.constructors.NetworkSettingsIpv6firewallAddView.extend({

    disallowedDevices: ['starlink'],

    initialize: function() {
        this.template = $.template("pageContent", swc.Templates.get('network:settings:ipv6-firewall:rules-edit').get('content'));
        this.events = _.extend({}, swc.base.PageView.prototype.events, this.events);
    },

    setTemplateData: function() {
        var ruleID = localStorage.getItem('firewal:edit-ruleID'),
            rule = swc.models.FirewallRules.get(ruleID);

        this.rule = ruleID;

        if (!ruleID || !rule) {
            swc.router.navigate('network/settings/ipv6-firewall/customize', { skipUnsavedChanges: true });
        }

        this.templateData = {
            predefinedRulesList: swc.models.FirewallRules.predefined,
            ruleType: "custom",
            ruleKey: rule.get('id'),
            ruleState: rule.get('status'),
            ruleName: rule.get('name'),
            ruleProtocol: rule.get('protocol'),
            rulePortsMode: rule.get('port').length === 1 ? "single" : "range",
            rulePorts: rule.get('port'),
            rulePolicy: rule.get('policy'),
            ruleChain: rule.get('chain')
        };
    },

    /**
     * @override
     *
     * On this page save changes button should redirect user to other page after saving complete:
     */
    save: function() {
        var rulePrevious = swc.models.FirewallRules.findWhere({ id: this.rule }).toJSON(),
            ruleNew = this.serializePageData(),
            deferred = new $.Deferred();

        $.when(swc.models.FirewallRules.changeRule(rulePrevious, ruleNew))
            .done(function() {
                localStorage.setItem('firewall:change-rule', 'done');
                swc.router.navigate('network/settings/ipv6-firewall/customize', { skipUnsavedChanges: true });
            })
            .fail(function() {
                localStorage.setItem('firewall:change-rule', 'fail');
                swc.router.navigate('network/settings/ipv6-firewall/customize', { skipUnsavedChanges: true });
            });

        return deferred.promise();
    }
});
;swc.constructors.NetworkSettingsIpv6firewallCustomizeView = swc.base.TabView.extend({

    className: 'ipv6-firewall-rules-list',

    allowedMods: ['expert'],

    disallowedDevices: ['starlink'],

    events: {
        'click .expandable-list .trash.process-delete:not(.disabled)': 'deleteRule',
        'click .expandable-list .pencil.process-edit:not(.disabled)': 'editRule',

        'swc-checkbox:change .swc-checkbox.enable-service': 'setRuleState'
    },

    models:[
        'FirewallPolicy',
        'FirewallRules'
    ],

    rulesToDelete: [],

    rulesToChangeState: {},

    setTemplateData: function() {
        var firewallRules = swc.models.FirewallRules.toJSON();

        this.templateData = {
            rules: firewallRules,
            rulesArrayLength: getObjectLength(firewallRules)
        };
    },

    initialize: function() {
        // Prepare template for current page:
        this.template = $.template("pageContent", swc.Templates.get('network:settings:ipv6-firewall:rules-list').get('content'));

        // Extend global events:
        this.events = _.extend({}, swc.base.PageView.prototype.events, this.events);
    },

    renderComplete: function() {
        var policyOptions = swc.models.FirewallPolicy.formatPolicyOptions(),
            policyValue = swc.models.FirewallPolicy.getValue(),
            dropdownPolicy = this.$el.find('.swc-dropdown.policy-selection'),
            isRuleActionSuccess = localStorage.getItem('firewall:change-rule');

        // Display rule saving success or fail message:
        if (!_.isNull(isRuleActionSuccess)) {
            this.showSaveSuccess(isRuleActionSuccess === "done" ? 'success' : 'error', true);
            localStorage.removeItem('firewall:change-rule');
        }

        this.fillDropdown(dropdownPolicy, policyOptions, policyValue);
    },

    fillDropdown: function(dropdown, data, selected) {
        for (var defaultAction in data) {
            break;
        }

        dropdown.data('options', data);

        if (selected) {
            dropdown.data('default-value', selected);
            dropdown.trigger('swc-dropdown:swc-change', selected);
        } else {
            dropdown.data('default-value', defaultAction);
            dropdown.trigger('swc-dropdown:swc-change', defaultAction);
        }
    },

    deleteRule: function(e) {
        var parameter = getParameter($(e.target)),
            ruleID = parameter.parameterData.rule.toString(),
            ruleItem = this.$el.find('.expandable-list-item[data-key="' + ruleID + '"]'),
            storage = this.$el.find('input[name="rules-to-delete"]');

        this.rulesToDelete.push(ruleID);

        // The only way for now to handle deletion change on the page
        storage.val('action-happened');

        // Remove this from list of changed state rules (anyway it will be deleted)
        if (this.rulesToChangeState[ruleID]) {
            delete this.rulesToChangeState[ruleID];
        }

        // Remove item from the list
        ruleItem.remove();

        // Update buttons state to see, that rule have been deleted
        this.setButtonsState();
    },

    setRuleState: function(e, value) {
        var parameter = getParameter($(e.target)),
            ruleID = parameter.parameterData.rule.toString();

        // The only way for now to handle editing of custom change on the page
        if (value !== parameter.parameterData.defaultValue) {
            this.rulesToChangeState[ruleID] = {
                value: value
            };
        } else {
            if (this.rulesToChangeState[ruleID]) {
                delete this.rulesToChangeState[ruleID];
            }
        }
    },

    editRule: function(e) {
        var parameter = getParameter($(e.target));

        localStorage.setItem('firewal:edit-ruleID', parameter.parameterData.rule);
        localStorage.setItem('firewall:editing', 'true');

        swc.router.navigate('network/settings/ipv6-firewall/edit', { trigger: true });
    },

    /**
     * @override
     * @description:
     *
     *  Display message from editing / adding rules:
     */
    showSaveSuccess: function(type, isFromRulePage) {
        var messageNew = this.$('.buttons-container-message .save-' + (type || 'success'));

        // Hide previous messages:
        this.$('.buttons-container-message .save-error').hide();
        this.$('.buttons-container-message .save-success').hide();

        // Defined what message to show:
        if (!_.isUndefined(isFromRulePage)) {
            messageNew = messageNew.filter('.from-rule');
        } else {
            messageNew = messageNew.filter(':not(.from-rule)');
        }

        messageNew.show();
    },

    save: function() {
        var self = this,
            deferred = new $.Deferred(),
            deferredActions,
            pageData = [];

        // Get elements name => value map
        $.each(this.getElements(), function(key, element) {
            pageData.push(getParameter($(element)));
        });

        // Create list of actions to be done:
        deferredActions = [
            swc.models.FirewallRules.updateRulesList(self.rulesToChangeState, self.rulesToDelete),
            swc.models.FirewallPolicy.setValue(pageData)
        ];

        $.when.apply(null, deferredActions)
            .done(function() {
                deferred.resolve();
            })
            .fail(function() {
                deferred.reject();
            });

        return deferred.promise();
    }
});
;swc.constructors.NetworkSettingsIpv6firewallView = swc.base.TabView.extend({

    className: 'ipv6-firewall',

    allowedMods: ['expert'],

    disallowedDevices: ['starlink'],

    events: {
        "click .button-customize:not(.disabled)": "customizeFirewall",
        "swc-radio-buttons:change .security-selection": "setCustomizeButtonState",
        "swc-checkbox:change .IPv6Enable" : "changeFirewallState"
    },

    models:[
        'FirewallStatus'
    ],

    validation:{
        "RuleName":"FirewallCustomRules:RuleName",
        "DestinationPortsRange": "FirewallCustomRules:DestinationPortsRange"
    },

    hasChanged: function() {
        return !this.pageCheckDefaultValues();
    },

    setTemplateData: function() {
        var firewallModel = swc.models.FirewallStatus,
            posibleLevels =  _.invert(firewallModel.possibleStatus);

        this.templateData = {
            IPv6Status: firewallModel.get("state").Enable,
            securityLevel: posibleLevels[firewallModel.get('status')]
        };
    },

    customizeFirewall: function() {
        if (!this.hasChanged()) {
            swc.router.navigate('network/settings/ipv6-firewall/customize', { skipUnsavedChanges: true });
        } else {
            $.when(this.save())
                .done(function() {
                    swc.router.navigate('network/settings/ipv6-firewall/customize', { skipUnsavedChanges: true });
                });
        }
    },

    setCustomizeButtonState: function(e, policyState) {
        this.$('.button-customize').toggleClass('disabled', policyState !== "custom");
    },

    changeFirewallState: function(e, value) {
        var selectedLevelRadio = this.$(".swc-radio-buttons .mode.active");

        swc.models.FirewallStatus.set('state', {"Enable": value});
        this.$('.security-selection').toggleClass('disabled', !value);

        if (!value || (value && selectedLevelRadio.data("value") !== "custom")) {
            this.$('.button-customize').addClass('disabled');
        } else if (value && selectedLevelRadio.data("value") === "custom"){
            this.$('.button-customize').removeClass('disabled');
        }
    },

    save: function() {
        var deferred = new $.Deferred(),
            pageData = [],
            actionsToDo;

        // Get elements name => value map
        $.each(this.getElements(), function(key, element) {
            pageData.push(getParameter($(element)));
        });

        actionsToDo = [
            swc.models.FirewallStatus.saveSettedLevel(pageData),
            swc.models.FirewallStatus.saveFirewallState(pageData)
        ];

        // Process save request:
        $.when.apply(null, actionsToDo)
            .done(function() {
                return deferred.resolve();
            })
            .fail(function() {
                return deferred.reject();
            });

        return deferred.promise();
    }
});
;swc.constructors.NetworkSettingsIpsettingsView = swc.base.TabView.extend({

    className: 'ip-settings',

    models: [
        'NetworkDevices',
        'DHCPLeases',
        'PortForwarding'
    ],

    validation: {
        'Address': 'Network:validateIPAddress',
        'APAddress': 'Network:validateAPIPAddress',
        'Netmask': 'Network:validateNetmask',
        'DHCPRange': 'Network:validateDHCPSettings',
        'DHCPLeaseIPAddress': 'DHCPLeases:validateIP',
        'DHCPLeaseNewDevice': 'DHCPLeases:validateNewDevice',
        'DMZIPAddress': 'Network:validateDMZ',
        'DMZCustomIpAddress': 'Network:validateCustomDMZ'
    },

    events: {
        'swc-checkbox:change .swc-checkbox.DHCPEnable': 'setDHCPState',
        'swc-checkbox:change .swc-checkbox.enable-dmz': 'setDMZState',
        'swc-dropdown:change .swc-dropdown.dmz-device': 'dmzDeviceChanged',
        'swc-dropdown:change .swc-dropdown.dhcplease-device-selection': 'selectDHCPLeaseDevice',
        'change input[name="Address"],input[name="DHCPRange"],input[name="Netmask"]': 'revalidateForm',
        'input input[name="Address"],input[name="DHCPRange"],input[name="Netmask"],input[name="APAddress"]': 'mobileValidateFix',
        'keyup input[name="Address"],input[name="Netmask"]': 'updateSubnetInfo',

        'expandable:edit .expandable-list.dhcp-leases-list': 'setEditLease',
        'expandable:delete .expandable-list.dhcp-leases-list': 'deleteDHCPLease',
        'input input[name="DMZCustomIpAddress"]' : 'limitSetOfCharacters',
        'keydown input[name="DMZCustomIpAddress"]' : 'limitSetOfCharactersDown',
        'input input[name="DHCPLeaseIPAddress"]' : 'limitSetOfCharacters',
        'keydown input[name="DHCPLeaseIPAddress"]' : 'limitSetOfCharactersDown'
    },

    isUserSuperAdmin: false,
    ableToChangeApAddress: true,

    limitSetOfCharactersDown: function(e) {
        return swc.Utils.limitSetOfCharactersDown(e, /[^\.\d]/g, [190]);
    },

    limitSetOfCharacters: function(e) {
        return swc.Utils.limitSetOfCharactersInput(e, /[^\.\d]/g);
    },

    setTemplateData: function() {
        var validation = swc.models.Network.validation,
            Address = swc.models.Network.get('ip-settings').get('LocalIPAddress'),
            Netmask = swc.models.Network.get('ip-settings').get('Netmask'),
            APAddress = swc.models.Network.get('status').get('StorageIPAddress');

        this.isUserSuperAdmin = swc.models.Login.checkUserSuperAdmin();

        this.templateData = {
            'Address': Address,
            'Netmask': Netmask,
            'DHCPMin': swc.models.Network.get('ip-settings').get('DHCPStart'),
            'DHCPMax': swc.models.Network.get('ip-settings').get('DHCPFinish'),
            'DHCPState': swc.models.Network.get('ip-settings').get('DHCPStatus'),
            'DHCPLeases': swc.models.DHCPLeases.getActiveLeases(),
            'AddressRange' : validation.getAllowedHostRange(validation.ipStringToDecimal(Address), validation.ipStringToDecimal(Netmask)),
            'APAddress': APAddress
        };
    },

    revalidateForm: function(e) {
        var elem = $(e.currentTarget);

        if (!elem.hasClass('validation-error')) {
            this.pageValidation();
        }
    },

    mobileValidateFix: function(e) {
        var position = $(e.target).getCursorPosition(),
            replacement = $(e.target).val();
        replacement = replacement.replace(/[^0-9\.]/g,'');
        $(e.currentTarget).val(replacement);
        $(e.currentTarget).focus();
        $(e.currentTarget)[0].setSelectionRange(position, position);
    },

    renderComplete: function() {
        var self = this,
            ipAddressBox = this.$el.find('input[name="Address"]'),
            apAddressBox = this.$el.find('input[name="APAddress"]'),
            netmaskAddressBox = this.$el.find('input[name="Netmask"]'),
            DHCPMinBox = this.$el.find('input[name="DHCPRange"].range-from'),
            DHCPMaxBox = this.$el.find('input[name="DHCPRange"].range-to'),
            dmz = this.$el.find('.dmz-setting');

        // Listener for IP Address field change
        ipAddressBox.on('validation-success', function() {
            self.updateDHCPSettings();
            self.pageValidation(DHCPMinBox, false);
            self.pageValidation(DHCPMaxBox, false);
        });

        netmaskAddressBox.on('validation-success', function() {
            self.updateDHCPSettings();
            self.pageValidation(DHCPMinBox, false);
            self.pageValidation(DHCPMaxBox, false);
            self.pageValidation(ipAddressBox, false);
            self.pageValidation(apAddressBox, false);
        });

        this.setDevicesDropdown();
        this.setDMZData();
        this.showHideCustomIp();
        $('input, textarea').placeholder();

        dmz.css('margin-left', $('.enable-dmz').width()+10);
        this.ableToChangeApAddress = true;
    },

    pageCheckDefaultValues: function() {
        var validation = swc.base.PageView.prototype.pageCheckDefaultValues.apply(this, arguments);

        if (this.$('.expandable-list-item.deleted').length > 0) {
            return false;
        }

        return validation;
    },

    setDevicesDropdown: function() {
        var dropdown = this.$('.dhcplease-device-selection'),
            options = swc.models.NetworkDevices.getDeviceDropdownOptions({'filter': 'DHCPLeases', 'WlanGuest': false}),
            defaultDevice = options[0];

        dropdown.data('options', options);
        dropdown.trigger('swc-dropdown:swc-change', defaultDevice);
    },

    /**
     * Event handler: is fired on changing device in the dropdown list
     *
     * @param e
     * @param value
     */
    selectDHCPLeaseDevice: function(e, value) {
        var $select = $(e.currentTarget),
            $item = $select.parents('.expandable-list-item'),
            $mac = $item.find('.lease-mac-address input'),
            $ip = $item.find('.lease-ip-address input'),
            device = swc.models.NetworkDevices.getDevice(value);

        $mac.val(value);

        if (device && device.address && device.status) {
            $ip.val(device.address.IPv4);
        } else {
            $ip.val('');
        }

        // Set editing flag to let view know that page changed:
        this.$('.edit-form-flag').val(value);

        // if it's an item for new lease then trigger its edit event
        if ($item.is('.new-dhcplease-form')) {
            this.$('.expandable-static-ip .expandable-list').trigger('expandable:edit', 'newItem');
        }

        this.setButtonsState();
    },

    /**
     * Close sibling items in the expandable list
     *
     * @param $item {jQuery element}
     */
    closeSiblingItems: function($item) {
        var self = this;

        $item.siblings('.expandable-list-item.mode-editing').each(function() {
            var dataKey = $(this).data('key');
            self.$('.expandable-list').trigger('expandable:swc-no-edit', dataKey);
        });
    },

    /**
     * Switch validation of elements in the sibling items
     * by adding/removing class 'skip-validation' to them
     *
     * if on === true then add validation (remove class 'skip-validation')
     * if on === false then remove validation (add class 'skip-validation')
     *
     * @param $item {jQuery element}
     * @param on {Boolean}
     */
    setValidationOnSiblingItems: function($item, on) {
        $item.siblings('.expandable-list-item').each(function() {
            $(this).find('input, .swc-dropdown').toggleClass('skip-validation', !on);
        });
    },

    /**
     * Switch validation of elements in the currently edit item
     * by adding/removing class 'skip-validation' on them
     *
     * if on === true then add validation (remove class 'skip-validation')
     * if on === false then remove validation (add class 'skip-validation')
     *
     * @param $item {jQuery element}
     * @param on {Boolean}
     */
    setValidationOnItem: function($item, on) {
        $item.find('input, .swc-dropdown').toggleClass('skip-validation', !on);
    },

    /**
     * Hide list item for adding new lease
     * and switch off validation on its elements
     */
    hideNewLeaseItem: function() {
        var $item = this.$('.new-dhcplease-form');

        $item.addClass('hidden');
        this.setValidationOnItem($item, false);
    },

    /**
     * Set enabled/disabled mode of sibling items by removing/adding
     * class 'disabled' to buttons with classes 'do-edit', 'do-delete'
     *
     * if on === true then enable sibling items (remove class 'disabled')
     * if on === false then disable sibling items (add class 'disabled')
     *
     * @param $item {jQuery element}
     * @param show {Boolean}
     */
    setEnabledModeForSiblingItems: function($item, on) {
        $item.siblings('.expandable-list-item').each(function() {
            $(this).find('.do-edit, .do-delete').toggleClass('disabled', !on);
        });
    },

    /**
     * Event handler: is fired on start edit static IP-address item
     *
     * @param e {Event}
     * @param value {String}
     */
    setEditLease: function(e, value) {
        var $item = $(e.currentTarget).find('[data-key="' + value + '"]');

        this.closeSiblingItems($item);
        this.setEnabledModeForSiblingItems($item, false);
        this.setValidationOnSiblingItems($item, false);
        this.setValidationOnItem($item, true);

        if (value !== 'newItem') {
            this.hideNewLeaseItem();
        }

        $item.find('.do-edit').addClass('disabled');

        // Set editing flag to let view know that page changed:
        this.$('.edit-form-flag').val(value);

        this.setButtonsState();
    },

    /**
     * Enable table with static IP addresses:
     *  - remove opacity from table
     *  - remove hover popup with info message above table
     */
    enableDHCPLeaseTable: function() {
        var dhcpTable = $(".dhcp-settings"),
            inputs = dhcpTable.find(".swc-input");

        _.each(inputs, function(input) {
            $(input).removeClass("skip-validation").removeClass("disabled").removeAttr("disabled", "disabled");
        });

        this.$('.dhcp-leases-list').removeClass('disabled');
    },

    /**
     * Disable table with static IP addresses:
     *  - add opacity to table
     *  - add hover popup with info message above table
     */
    disableDHCPLeaseTable: function() {
        var dhcpTable = $(".dhcp-settings"),
            inputs = dhcpTable.find(".swc-input");

        _.each(inputs, function(input) {
            $(input).addClass("skip-validation").addClass("disabled").attr("disabled", "disabled");
        });

        this.$('.dhcp-leases-list').addClass('disabled');
    },

    /**
     * Event handler: is fired on changing state of DHCP
     *
     * @param e {Event}
     * @param value {String}
     */
    setDHCPState: function(e, value) {
        if (value) {
            this.enableDHCPLeaseTable();
        } else {
            this.disableDHCPLeaseTable();
        }

        this.pageValidation(); //Revalidate DHCP and so on
    },

    /**
     * Event handler: is fired on delete static IP-address item
     *
     * @param e {Event}
     * @param value {String}
     */
    deleteDHCPLease: function(e, value){
        var $item = $(e.currentTarget).find('[data-key="' + value + '"]');

        this.setValidationOnItem($item, false);
        this.setEnabledModeForSiblingItems($item, true);

        $item.addClass('deleted').hide();

        this.setButtonsState();
    },
    //triggered if checkbox clicked "Activate DMZ..."
    setDMZState: function(e, value) {
	
        var DMZIPAddress = this.$('[data-name="DMZIPAddress"]');
        var DMZCustomIPAddress = this.$('[data-name="DMZCustomIpAddress"]');

        $(".dmz-device").toggleClass("disabled", !value);
        $(".dmz-settings .ip-address").toggleClass("disabled", !value);

        if (!value) {
            this.pageValidation(DMZIPAddress, false);
            this.pageValidation(DMZCustomIPAddress, false);
        }
        this.showHideCustomIp(value);
        this.pageValidation(DMZCustomIPAddress);
    },

    setDMZData: function() {
        var options = swc.models.NetworkDevices.getDeviceDropdownOptionsOnlyIpv4({'filter': 'status','WlanGuest': false}),
            deviceIP = swc.models.Network.getParameter('deviceIP', 'dmz'),
            selected = swc.models.NetworkDevices.getDeviceMac(deviceIP),
            dmzCheckbox = this.$el.find('.swc-checkbox.enable-dmz'),
            dmzDropDown = this.$el.find('.dmz-device'),
            devIP = deviceIP.match(/(parameterValue:)([0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3}\.[0-9]{1,3})/);
        
        // for empty device - create at least ip address
        if(!_.isEmpty(deviceIP) && _.isEmpty(selected) && _.isEmpty(devIP)) {
            selected = '00:00:00:00:00:00';
            options.push({ additionalLabel: deviceIP,
                name: '',
                value: selected
            });
        }
        options.push({ additionalLabel: "",
            name: getTranslationStrings("Enter Device by IP"),
            value: "custom"
        });

        dmzDropDown.data('options', options);
        dmzDropDown.toggleClass("disabled", !deviceIP.length);

        if (!deviceIP.length) {
            dmzCheckbox.trigger('swc-checkbox:swc-change', false);
            dmzCheckbox.data("default-value", false);
            dmzDropDown.data("default-value", "");
        } else {
            
            if(!_.isEmpty(devIP)){
                this.$('.dmz-settings .swc-input.ip-address').val(devIP[2]);
                this.$('.dmz-settings .swc-input.ip-address').attr("default-value",devIP[2]);
                selected="custom";
            }
            
            dmzDropDown.data("default-value", selected);
            dmzCheckbox.data("default-value", true);
            dmzCheckbox.trigger('swc-checkbox:swc-change', true);
            dmzDropDown.trigger('swc-dropdown:swc-change', selected);
        }
        
        if(dmzCheckbox.data("default-value") === false) {
            this.$('.dmz-settings .swc-input.ip-address').attr("disabled", "disabled").hide();
        }
    },
    // listener for dmz device change
    dmzDeviceChanged: function(){
        this.showHideCustomIp();
    },
    /**
     * Shows or hides custom ip input field for dmz device
     *
     */
    showHideCustomIp: function(isDisabled) {
        var deviceSelection = this.$('.swc-dropdown.dmz-device'),
        value = deviceSelection.data('value'),
        address = this.$('.dmz-settings .swc-input.ip-address');
        if(value === 'custom') {
            address.show();
        } else {
            address.hide();
            //address.val('');
        }
        // checking if we have to enable or disable the custom ip dmz input field
         if(isDisabled === undefined) {
            if(this.$('.dmz-settings .swc-checkbox.enable-dmz').hasClass("checked") && value === 'custom') {
                address.attr("disabled", false);
            }
            else {
                address.attr("disabled", "disabled");
            }
         }
         else {
             if(isDisabled && value === 'custom'){
                address.attr("disabled", false);
             }
             else {
                address.attr("disabled", "disabled");
             }
         }

    },

    saveChanges: function() {

        this.ableToChangeApAddress = false;

        swc.base.PageView.prototype.saveChanges.apply(this, arguments);
    },

    //Change DHCP range subnet
    updateDHCPSettings: function() {
        var validation = swc.models.Network.validation,
            Address = this.$('input[name="Address"]').val(),
            APAddress = this.$('input[name="APAddress"]'),
            Netmask = this.$('input[name="Netmask"]').val(),
            DHCPMin = this.$('input[name="DHCPRange"].range-from'),
            DHCPMax = this.$('input[name="DHCPRange"].range-to'),

            AddressDecimal = validation.ipStringToDecimal(Address),
            NetmaskDecimal = validation.ipStringToDecimal(Netmask);

        if (validation.validateNetmaskDecimal(AddressDecimal, NetmaskDecimal, false, false).status !== false) {

            var minVal = validation.replaceSubnet(DHCPMin.val(), Address, Netmask);
            var maxVal = validation.replaceSubnet(DHCPMax.val(), Address, Netmask);
            //Check that DHCPMin is less then DCHPMax
            var valsAreValid = (minVal !== false) && (maxVal !== false) &&
             (validation.ipStringToDecimal(minVal) < validation.ipStringToDecimal(maxVal));

            if (minVal && (!maxVal || valsAreValid)) {
                DHCPMin.val(minVal);
            }
            if (maxVal && (!minVal || valsAreValid)) {
                DHCPMax.val(maxVal);
            }

            if(this.ableToChangeApAddress === true) {
                var apVal = validation.replaceSubnet(APAddress.val(), Address, Netmask);
                if(apVal !== false) {
                    APAddress.val(apVal);
                }
            }
        }
    },

    //In case if subnet have changed, update, DHCP range, and hint for network range
    updateSubnetInfo: function() {
        var validation = swc.models.Network.validation,
            Address = validation.ipStringToDecimal(this.$('input[name="Address"]').val()),
            Netmask = validation.ipStringToDecimal(this.$('input[name="Netmask"]').val());

        if (validation.validateNetmaskDecimal(Address, Netmask, false, false).status !== false &&
            validation.validateIPAddressDecimal(Address, Netmask, false, false).status !== false) {

            var popoverData = this.$('.netmask-settings-info').data('popover-template-data');
            popoverData.subnetMask = validation.decimalToIpString(Netmask);
            popoverData.subnetRange = validation.getAllowedHostRange(Address, Netmask);
            this.$('.netmask-settings-info').data('popover-template-data', popoverData);
        }

        this.updateDHCPSettings();
    },

    getDHCPLeasesData: function () {
        var possibleElements = [
                '.swc-input', '.swc-checkbox', '.swc-extended', '.swc-radio-buttons', '.swc-dropdown'
            ],
            selector = possibleElements.join(','),
            pool = [];

        this.$('.expandable-static-ip .expandable-list-item')
            .filter('.mode-editing:not(.skip-validation), .edited:not(.skip-validation), .deleted, .new-dhcplease-form:visible:not(.skip-validation)').each(function () {
                var self = $(this),
                    item = self.find(selector).map(function () {
                        return getParameter($(this));
                    });
                if (self.is('.deleted')) {
                    item.deleted = true;
                }
                pool.push(item);
            });

        return pool;
    },

    save: function() {
        var self = this,
            deferred = new $.Deferred(),
            deferred2 = new $.Deferred(),
            pageData = [],
            warningMessages = [],
            actionsToDo,
            elements = this.getElements(),
            DHCPLeasesData = this.getDHCPLeasesData();

        // Get elements name => value map
        $.each(elements, function(key, element) {
            pageData.push(getParameter($(element)));
        });

        // warningMessages can be filled when IP setting are saved correctly already
        warningMessages = self.checkNetworkChangeWarnings(pageData);

        /*
         * if there are some warning messages then hide page loading modal window
         * and show information modal window
         */
        if (!_.isEmpty(warningMessages)) {
            self.stopPageLoading();
        }
        
        // check if we are setting dmz for non static device
        var addStaticLease = false,
            address = this.getParamValue(pageData, "DMZIPAddress"),
            customAddress = this.getParamValue(pageData, "DMZCustomIpAddress"),
            oldCustomAddress = this.$el.find("[name='DMZCustomIpAddress']").attr('default-value'),
            oldAddress = this.$el.find('.dmz-device').data('default-value'),
            oldIsEnabled = this.$el.find('.enable-dmz').data('default-value'),
            isEnabled = this.getParamValue(pageData, "DMZEnable"),
            isNewAddress = (address !== oldAddress),
            isNewEnabled = (isEnabled !== oldIsEnabled),
            isNewCustomDMZ = (oldCustomAddress !== customAddress);

        if((isNewAddress || isNewEnabled) && address && isEnabled) {
            addStaticLease = !this.checkIfDeviceHasStaticAddress(address, DHCPLeasesData);
        }

        function saveData() {
            if (!_.isEmpty(warningMessages)) {
                self.showPageLoading('Saving changes...');
                if(warningMessages[0]==="1_ip"){
                    self.showWarningModal(deferred, false, warningMessages[0]);
                }
            }
            
            var apAddress = self.getApAddress(pageData);
            
            // Define what has to be done on save
            actionsToDo = [];
            if(isNewAddress || isNewEnabled || isNewCustomDMZ) {
                actionsToDo.push(swc.models.Network.saveDMZSettings(pageData));
            }
            actionsToDo.push(swc.models.DHCPLeases.saveDHCPLeases(DHCPLeasesData));
            
            if(apAddress !== undefined) {
                // remove port forwarding silently
                var items = swc.models.PortForwarding.where({ deviceIP: apAddress });
                _.each(items, function(model) {
                    model.sync('delete');
                    model.sync('commit');
                });
            }
            $.when.apply(null, actionsToDo)
                .done(function() {
                    actionsToDo = [swc.models.Network.saveIPSettings(pageData)];
                    if(apAddress !== undefined) {
                        actionsToDo.push(swc.models.Network.saveStorageIP(apAddress));
                    }

                    // IP settings are saved the last because we can't save anything if IP is changed:
                    $.when.apply(null, (actionsToDo))
                        .done(function() {
                            self.ableToChangeApAddress = true;
                            deferred.resolve();
                        })
                        .fail(function() {
                            self.ableToChangeApAddress = true;
                            deferred.reject();
                        });
                }).fail(function() {
                    self.ableToChangeApAddress = true;
                    deferred.reject();
                });
        }
        
        if(isNewAddress && addStaticLease && address !== "custom") {
            var device = swc.models.NetworkDevices.getDevice(address);
            
            DHCPLeasesData.push([{
                parameterName : "DHCPLeaseNewDevice",
                parameterValue : address
            }, {
                parameterName : "DHCPLeaseMacAddress",
                parameterValue : address
            }, {
                parameterName : "DHCPLeaseIPAddress",
                parameterValue : device.address.IPv4
            }]);

            self.stopPageLoading();
            $.when(self.addStaticLease(device.address.IPv4)).done(function() {
                saveData();
            });
        } else {
            //if we choose custom DMZ we have to rewrite the ip from the custom field
        if(address === "custom") {
           address= this.$('.dmz-settings .swc-input.ip-address').val();
         }
            saveData();
        }

        function afterSave(rejected) {
            if (!_.isEmpty(warningMessages) && !rejected) {
                self.showWarningModal(deferred2, rejected, warningMessages[0]);
            } else {
                if (rejected) {
                    if(!_.isEmpty(warningMessages)){
                        self.showWarningModal(deferred2, rejected, warningMessages[0]);
                    }
                    deferred2.reject();
                } else {
                    deferred2.resolve();
                }
            }
        }

        function afterFail() {
            afterSave(true);
        }

        deferred
            .done(afterSave)
            .fail(afterFail);

        return deferred2.promise();
    },
    
    getApAddress: function(data) {
        var result;
        
        $.each(data, function(key, value) {
            if (value.parameterName === "APAddress") {
                if(value.parameterData.defaultValue !== value.parameterValue) {
                    result = value.parameterValue;
                }
            }
        });
        
        return result;
    },
    
    addStaticLease: function (deviceIP) {
        var deferred = new $.Deferred();

        SWCElements.modalWindow.show({
            templateID: 'network:settings:ip-settings:addstaticlease-modal',
            templateData: {
                deviceIP: deviceIP
            },
            className: 'addstaticlease-modal',

            onApply: function() {
                SWCElements.modalWindow.hide();

                deferred.resolve();
            }
        });
        
        return deferred;
    },
    
    getParamValue: function(pageData, param) {
        var result = null,
            item = null;
        
        for(var i = 0; ((i < pageData.length) && (result === null)); i++) {
            item = pageData[i];
            if(item.parameterName === param) {
                result = item.parameterValue;
            }
        }

        return result;
    },
    
    checkIfDeviceHasStaticAddress: function(address, data) {
        var macaddr = null,
            item, value,
            result = false;

        // check on list if it is not for delete
        for(var i = 0; ((i < data.length) && !result); i++) {
            item = data[i];
            
            for(var j = 0; ((j < item.length) && !result); j++) {
                value = item[j];
                if (value.parameterName === "DHCPLeaseMacAddress" || value.parameterName === "DHCPLeaseEditedMacAddress") {
                    macaddr = value.parameterValue.toUpperCase();
                    result = (macaddr === address);
                }
            }
            
            if(result && item.deleted) {
                return false;
            }
        }
        
        // can be saved and not modified
        if(!result) {
            if(swc.models.DHCPLeases.get(address)) {
                result = true;
            }
        }

        return result;
    },

    /**
     * Get an array of warning messages if the net is changed/reduced so that some devices may need reconnection.
     * The only first message will be shown to user, that is why all messages are sorted by priority.
     * E.g. in case of editing "InternetBox IP address" and "Static IP address" only message about changing
     * "InternetBox IP address" should be shown
     *
     * @params data {Array} Array of form values
     * @return {Array} Warning messages
     */
    checkNetworkChangeWarnings: function(data) {
        var oldMask,
            newMask,
            warningMessages = [];

        function ipToDecimal(ip) {
            return _.reduce(ip.split('.'), function (m, n) {
                    return m * 256 + parseInt(n, 10);
                }) >>> 0;
        }

        _.each(data, function(element) {
            if (element.parameterName === 'Address' && element.parameterValue !== element.parameterData.defaultValue) {
                warningMessages.push('1_ip');
            }

            if (element.parameterName === 'Netmask') {
                oldMask = ipToDecimal(element.parameterData.defaultValue);
                newMask = ipToDecimal(element.parameterValue);

                if (oldMask < newMask) {
                    warningMessages.push('2_mask');
                }
            }

            if (element.parameterName === 'DHCPLeaseIPAddress' &&
                element.parameterValue !== element.parameterData.defaultValue) {
                warningMessages.push('3_lease_ip');
            }

        });

        /*
         * only first message will be shown to user sort messages by priority
         * and all messages should sorted by priority
         */
        warningMessages.sort();

        return warningMessages;
    },

    showWarningModal: function (deferred, rejected, messageID) {
        var self = this,
        className = 'network-warning-modal';
        self.stopPageLoading();

        if(messageID === '1_ip') {
            className = 'network-warning-modal-ip';
        }

        SWCElements.modalWindow.show({
            templateID: 'network:settings:ip-settings:warning-modal',
            templateData: {
                messageID: messageID
            },
            className: className,

            onApply: function() {
                SWCElements.modalWindow.hide();

                if(!self.isUserSuperAdmin && messageID === '1_ip') {
                    _.delay(function() {
                        if(window.location.href.indexOf('internetbox') === -1) {
                            document.location.href = 'http://' + self.$('input[name=Address]').val() + '/#' + Backbone.history.fragment;
                        } else {
                            window.location.reload();
                        }
                    },100);
                } else {
                    deferred.resolve();
                }
            }
        });

        function sendTimeRequest() {
            $.ajax({
                url: 'http://' + self.$('input[name=Address]').val() + '/sysbus/Time:getTime',
                type: 'POST',
                timeout: 100,
                async: true,
                success: function() {
                },

                error: function(jqXHR, textStatus) {
                    if(textStatus === 'timeout') {
                        sendTimeRequest();
                    } else {
                        $('.network-warning-modal-ip .apply-changes').show();
                        $('.network-warning-modal-ip .ip-adress-changing-label').hide();
                    }
                }
            });
        }

        if(messageID === '1_ip') {
            sendTimeRequest();
        }

        return deferred.promise();
    }
});
;swc.constructors.NetworkSettingsView = swc.base.PageView.extend({

    className: 'network-settings',

    models: [
        'Network',
        'Wireless'
    ]
});;swc.constructors.NetworkSettingsPortforwardingAddView = swc.base.TabView.extend({

    className: 'port-forwarding-add',

    allowedMods: [
        'expert'
    ],

    disallowedDevices: ['starlink'],

    models: [
        'PortForwarding',
        'NetworkDevices',
        'DHCPLeases',
        'Applications'
    ],

    validation: {
        'deviceIP': 'PortForwarding:deviceIP',
        'serviceName': 'PortForwarding:serviceName',
        'predefinedServiceName': 'PortForwarding:predefinedServiceName',
        'entryPort': 'PortForwarding:entryPort',
        'destinationPort': 'PortForwarding:destinationPort',
        'address': 'PortForwarding:validateIPAddress'
    },

    events: {
        'swc-radio-buttons:change .swc-radio-buttons.service-mode': 'setServiceAddingMode',
        'swc-radio-buttons:change .swc-radio-buttons.entry-mode': 'setEntryPortMode',
        'swc-radio-buttons:change .swc-radio-buttons.destination-mode': 'setDestinationPortMode',

        'keyup input[name="entryPort"].from': 'updateDestinationPorts',
        'keyup input[name="entryPort"].to': 'updateDestinationPorts',
        'change input[name="entryPort"].from': 'updateDestinationPorts',
        'change input[name="entryPort"].to': 'updateDestinationPorts',

        'input input[name="entryPort"].from': 'limitSetOfCharacters',
        'input input[name="entryPort"].to': 'limitSetOfCharacters',

        'keydown input[name="entryPort"].from': 'limitSetOfCharactersDown',
        'keydown input[name="entryPort"].to': 'limitSetOfCharactersDown',

        'input input[name="destinationPort"].from': 'limitSetOfCharacters',
        'input input[name="destinationPort"].to': 'limitSetOfCharacters',

        'keydown input[name="destinationPort"].from': 'limitSetOfCharactersDown',
        'keydown input[name="destinationPort"].to': 'limitSetOfCharactersDown',

        'swc-dropdown:change .swc-dropdown.service-name-selection': 'setPredefinedValues',
        'swc-dropdown:change .swc-dropdown.device-selection': 'deviceSelectChanged',
        'input input[name="address"]' : 'limitSetOfCharactersAddress',
        'keydown input[name="address"]' : 'limitSetOfCharactersAddressDown'
    },

    //page has just rendered? Prevents from displaying 'Do you want so save changes?' after page rendering
    justRendered: true,

    setTemplateData: function() {
        this.templateData = {
            ipAddressAP: swc.models.Network.getParameter('StorageIPAddress', 'status'),
            serviceMode: "custom", // Default option 'Create your own'
            servicePred: "FTP",    // Default option 'Predefined rule: FTP'
            name: "",
            port: {
                entry: [],
                destination: []
            },
            portMode: {
                entry: "single", // Default option 'Single'
                destination: "single"  // Default option 'Single'
            },
            device: "0.0.0.0",      // Default option 'Select a device'
            protocol: "6,17"        // Default option 'TCP/UDP'
        };
    },

    limitSetOfCharactersAddressDown: function(e) {
        return swc.Utils.limitSetOfCharactersDown(e, /[^\.\d]/g, [190]);
    },

    limitSetOfCharactersAddress: function(e) {
        return swc.Utils.limitSetOfCharactersInput(e, /[^\.\d]/g);
    },

    preRender: function() {
        var deferred = new $.Deferred();
        if (!swc.models.VPNServer) {
            swc.models.VPNServer = new swc.constructors.VPNServer();
        }

        $.when(
            swc.models.Applications.isVpnEnable()
        ).done(function(vpnGlobalEnable) {
            swc.models.PortForwarding.vpnGlobalEnable = vpnGlobalEnable;
            if(vpnGlobalEnable === true) {
                $.when(swc.models.VPNServer.sync()).done(function() {
                    deferred.resolve();
                });
            } else {
                deferred.resolve();
            }
        }).fail(function() {
            swc.models.PortForwarding.vpnGlobalEnable = false;
            deferred.resolve();
        });
        return deferred.promise();
    },

    /**
     * @override on {'swc.BaseView.initialize()'} method
     *
     * Need to set new template in order to support /1/2/3/4/.../n/ lvls of views including.
     *
     * @returns {boolean}
     */
    initialize: function() {
        // Prepare template for current page:
        this.template = $.template("pageContent", swc.Templates.get('network:settings:port-forwarding:rules-add').get('content'));

        // Extend global events:
        this.events = _.extend({}, swc.base.PageView.prototype.events, this.events);
    },

    /**
     * @override on {'swc.BaseView.pageCheckDefaultValues()'} method
     *
     * Need to return always true, becuase this page must not check for editing
     *
     * @returns {boolean}
     */
    // pageCheckDefaultValues: function() {
        // return true;
    // },

    /**
     * Override from base class - never mark this page as changed.
     * FIXME: After implementation of 2-way binding we have to fix this method and check if any of model changed
     *
     * Possible workarounds: remove and global hasChanged() will handle case,
     * but for that we need to FIXME: implement disabling/enabling of "Save" button.
     * @returns {boolean}
     */
    hasChanged: function(){
        if(this.justRendered || localStorage['port-forwarding:editing'] === 'true') {
            this.justRendered = false;
            localStorage['port-forwarding:editing'] = 'false';
            this.pageCheckDefaultValues();
            return false;
        }
        return !this.pageCheckDefaultValues();
    },

    /**
     * @override
     *
     * @description
     *
     * Need to keep buttons enabled all time
     */
    setButtonsState: function() {
        var buttonCancel = $('.buttons-container-message .button.cancel-changes');

        swc.constructors.NetworkSettingsPortforwardingAddView.__super__.setButtonsState.apply(this);
        buttonCancel.removeClass('disabled');
    },

    /**
     * @override on {'swc.BaseView.renderComplete()'} method
     *
     * Need to setup some values when page is ready to be rendered
     *
     * @returns {boolean}
     */
    renderComplete: function() {
        this.setDefaultValues();
    },

    /**
     * Get values from Ports inputs. Convert undefined values to 0
     *
     * @param type {String}
     * @returns {Array of Strings}
     */
    getPortsValuesFromView: function(type) {
        var inputEntryFrom = this.$('input[name="entryPort"].from'),
            inputEntryTo = this.$('input[name="entryPort"].to'),
            inputDestinationPortFrom = this.$('input[name="destinationPort"].from'),
            inputDestinationPortTo = this.$('input[name="destinationPort"].to'),
            portValues = {
                entry: [
                    !$.trim(inputEntryFrom.val()) ? "" : +inputEntryFrom.val(),
                    !$.trim(inputEntryTo.val()) ? "" : +inputEntryTo.val()
                ],
                destination: [
                    !$.trim(inputDestinationPortFrom.val()) ? "" : +inputDestinationPortFrom.val(),
                    !$.trim(inputDestinationPortTo.val()) ? "" : +inputDestinationPortTo.val()
                ]
            };

        return !_.isUndefined(portValues[type]) ? portValues[type] : [];
    },

    /**
     * Reset form elements to default values.
     *
     * @description
     *
     *      Default value for fields when adding rule are following:
     *
     *          Service - Create Your Own
     *          Entry Port - Single
     *          Protocol - TCP/UDP
     *          To Device - Choose a device
     *          Destination Port - Single
     *
     *      All input fields have to be empty
     */
    setDefaultValues: function() {
        this.setPredefinedRulesOptions();
        this.setAssignedDeviceOptions();
        this.setProtocolOptions();
        this.showHideCustomIp();

        // Set service name field to default value:
        var serviceName = this.$('input[name="serviceName"]');
        serviceName.val(this.templateData.name);
        serviceName.data('default-value', serviceName.val());

        // Set entry ports fields to default values:
        var entryPortFrom = this.$('input[name="entryPort"].from');
        if (!_.isUndefined(this.templateData.port.entry[0])) {
            entryPortFrom.val(this.templateData.port.entry[0]);
        } else {
            entryPortFrom.val("");
        }
        entryPortFrom.data('default-value', entryPortFrom.val());

        var entryPortTo = this.$('input[name="entryPort"].to');
        if (!_.isUndefined(this.templateData.port.entry[1])) {
            this.templateData.portMode.entry = "range";

            entryPortTo.val(this.templateData.port.entry[1]);
            this.$('.entry-port-values-section').addClass("range-type");
        } else {
            entryPortTo.val("");
            this.$('.entry-port-values-section').removeClass("range-type");
            this.$('.swc-radio-buttons.destination-mode .mode[data-value="range"]').addClass('disabled');
        }
        entryPortTo.data('default-value', entryPortTo.val());

        // Set destination ports fields to default values:
        var destinationPortFrom = this.$('input[name="destinationPort"].from');
        if (!_.isUndefined(this.templateData.port.destination[0])) {
            destinationPortFrom.val(this.templateData.port.destination[0]);
            this.$('.text-block-destination-from').text(this.templateData.port.destination[0]);
        } else {
            destinationPortFrom.val("");
            this.$('.text-block-destination-from').text("");
        }
        destinationPortFrom.data('default-value', destinationPortFrom.val());

        var destinationPortTo = this.$('input[name="destinationPort"].to');
        if (!_.isUndefined(this.templateData.port.destination[1])) {
            this.templateData.portMode.destination = "range";

            destinationPortTo.val(this.templateData.port.destination[1]);
            this.$('.text-block-destination-to').text(this.templateData.port.destination[1]);
            this.$('.destination-port-values-section').addClass("range-type");
        } else {
            destinationPortTo.val("");
            this.$('.text-block-destination-to').text("");
            this.$('.destination-port-values-section').removeClass("range-type");
        }
        destinationPortTo.data('default-value', destinationPortTo.val());

        // Reset radio buttons state to default
        var entryMode = this.$('.swc-radio-buttons.entry-mode');
        entryMode.trigger('swc-radio-buttons:swc-change', this.templateData.portMode.entry);
        entryMode.data('default-value', this.templateData.portMode.entry);

        var destinationMode = this.$('.swc-radio-buttons.destination-mode');
        destinationMode.trigger('swc-radio-buttons:swc-change', this.templateData.portMode.destination);
        destinationMode.data('default-value', this.templateData.portMode.destination);

        // Reset dropdowns state to default
        var protocolSelection = this.$('.swc-dropdown.protocol-selection');
        protocolSelection.trigger('swc-dropdown:swc-change', this.templateData.protocol);
        protocolSelection.data('default-value', protocolSelection.data('value'));

        var deviceSelection = this.$('.swc-dropdown.device-selection');
        deviceSelection.trigger('swc-dropdown:swc-change', this.templateData.device);
        deviceSelection.data('default-value', deviceSelection.data('value'));
        
        var serviceNameSelecetion = $('.service-name-selection');
        serviceNameSelecetion.data('default-value', serviceNameSelecetion.data('value'));
        
        var serviceMode = $('.service-mode');
        serviceMode.data('default-value', serviceMode.data('value'));

        // Remove all previous validation messages:
        this.clearValidationMessages();
        
        this.setButtonsState();
    },

    /**
     * Handler for resetting fields to predefined rule values:
     *
     * @description
     *
     *      After selecting predefined options, all the fields and inputs in the page MUST be filled with port's values
     *
     * @param e {Object} Event
     * @param value {String}
     */
    setPredefinedValues: function(e, value) {
        var serviceName = !_.isUndefined(value) ? value : this.templateData.servicePred,
            service = swc.models.PortForwarding.predefinedServices[serviceName],
            servicePort = service.port,
            entryRadioButtons = this.$('.swc-radio-buttons.entry-mode'),
            entryRadioButtonsRangeMode = this.$('.swc-radio-buttons.entry-mode .mode[data-value="range"]'),
            destinationRadioButtons = this.$('.swc-radio-buttons.destination-mode'),
            protocolSelection = this.$('.swc-dropdown.protocol-selection');

        // Update inputs:
        this.$('input[name="entryPort"].from').val(servicePort);
        this.$('input[name="destinationPort"].from').val(servicePort);
        this.$('.text-block-destination-from').text(servicePort);
        this.$('.text-block-entry').text(servicePort);
        this.$('.text-block-destination').text(servicePort);

        // Set correct radio buttons position for entry ports:
        entryRadioButtons.trigger('swc-radio-buttons:swc-change', "single");
        destinationRadioButtons.trigger('swc-radio-buttons:swc-change', "single");

        this.setEntryPortMode(null, "single");
        this.setDestinationPortMode(null, "single");
        
        entryRadioButtonsRangeMode.addClass('disabled');
        protocolSelection.addClass('disabled');

        // Update protocol:
        this.$('.swc-dropdown.protocol-selection').trigger('swc-dropdown:swc-change', service.protocol);
        
        // change inputs on labels
        this.$('.entry-port-values-section').removeClass("range-type");
        this.$('.entry-port-values-section').addClass("value-type");
        this.$('.destination-port-values-section').removeClass("range-type");
        this.$('.destination-port-values-section').addClass("value-type");

        // Remove all previous validation messages:
        this.clearValidationMessages();
    },

    /**
     * Handler for switching between Create your own / Predefined modes
     *
     * @descriptions
     *
     *      IF on add rule page the User will chose "Create Your Own" option THEN all field MUST have either default value
     *      (if any) or be empty, user MUST be able to change any field on the page
     *
     *      IF on add rule page the User will chose "Predefned Rule" option THEN in Service dropdown menu he MUST be able
     *      to select it from the list (see {'swc.collections.PortForwarding.defaultServices'}).
     *
     * @param e {Object} Event
     * @param value {String}
     */
    setServiceAddingMode: function(e, value) {
        var type = !_.isUndefined(value) ? value : this.templateData.serviceMode;

        if (type === "predefined") {
            this.setDefaultValues();
            this.setPredefinedValues();
        } else {
            this.setDefaultValues();
                        
            // reset reange entry port
            this.$('.swc-radio-buttons.entry-mode .mode[data-value="range"]').removeClass('disabled');
            this.$('.swc-dropdown.protocol-selection').removeClass('disabled');

            // change inputs on labels
            this.$('.entry-port-values-section').removeClass("value-type");
            this.$('.destination-port-values-section').removeClass("value-type");
        }

        this.$('.rule-name-section').toggleClass("type-predefined", value === "predefined");
    },

    /**
     * Handler for switching between Single / Range modes for entry ports:
     *
     * @description
     *
     *      IF in Entry Port user will choose 'Single' option THEN User MUST be able to select ONLY Single option in
     *      Destination Port, Range option MUST be disabled, the User MUST be able to edit port number
     *
     *      IF in Entry Port user will choose Range option THEN user MUST be able to select BOTH Single and Range option
     *      in Destination Port
     *
     *      IF in Entry Port user will choose Range option AND choose Single option in Destination Port THEN ports number
     *      from Entry Ports MUST be copied to Destination Port, the User MUST be able to edit this port number
     *
     *      IF in Entry Port user will choose Range option AND choose Range option in Destination Port THEN ports range
     *      from Entry Ports MUST be copied to Destination Port range, the User MUST NOT be able to edit this ports numbers
     *
     *      IF the User will enter port number in Entry port field with Single option chosen and after will switch to Range
     *      option THEN second field MUST be empty in order the User will be able to enter port number
     *
     * @param e {Object} Event
     * @param value {String}
     */
    setEntryPortMode: function(e, value) {
        var entryPorts = this.getPortsValuesFromView('entry'),
            inputEntryPortFrom = this.$('input[name="entryPort"].from'),
            inputEntryPortTo = this.$('input[name="entryPort"].to'),
            inputDestinationPortFrom = this.$('input[name="destinationPort"].from'),
            inputDestinationPortTo = this.$('input[name="destinationPort"].to'),
            spanDestinationFrom = this.$('.text-block-destination-from'),
            spanDestinationTo = this.$('.text-block-destination-to'),
            destinationRadioButtons = this.$('.swc-radio-buttons.destination-mode'),
            destinationValuesSection = this.$('.destination-port-values-section'),
            destinationRadioButtonsRangeMode = this.$('.swc-radio-buttons.destination-mode .mode[data-value="range"]');

        if (value === "range") {
            // Enable if was disabled range mode for destination ports:
            destinationRadioButtonsRangeMode.removeClass('disabled');
            destinationRadioButtons.trigger('swc-radio-buttons:swc-change', "range");
            destinationValuesSection.addClass("range-type");

            // When changed by user
            if (e) {
                // Copy entry port 'from' and 'to' to destination ports:
                inputDestinationPortFrom.val(entryPorts[0]);
                inputDestinationPortTo.val(entryPorts[1]);

                // Copy values to spans:
                spanDestinationFrom.text(entryPorts[0]);
                spanDestinationTo.text(entryPorts[1]);
            }

            if (!_.isEmpty(inputEntryPortFrom.val())) {
                this.pageValidation(inputEntryPortFrom, false);
                this.pageValidation(inputEntryPortTo, false);
                this.pageValidation(inputDestinationPortFrom, false);
            }

        } else {
            // Set range mode to single for destination ports:
            destinationRadioButtons.trigger('swc-radio-buttons:swc-change', "single");
            destinationRadioButtonsRangeMode.addClass('disabled');
            destinationValuesSection.removeClass("range-type");

            // Copy entry port 'from' to destination port (when changed by user):
            if (e) {
                inputDestinationPortFrom.val(entryPorts[0]);

                // Blink field on update
                this.notifyPortsUpdate(inputDestinationPortFrom);
            }

            // Re-validate field to clear message from range ports:
            if (!_.isEmpty(inputEntryPortFrom.val())) {
                this.pageValidation(inputEntryPortFrom, false);
                this.pageValidation(inputEntryPortTo, false);
                this.pageValidation(inputDestinationPortFrom, false);
            }
        }

        this.$('.entry-port-values-section').toggleClass("range-type", value === "range");
    },

    /**
     * Handler for switching between Single / Range modes for destination ports:
     *
     * @description
     *
     *      There are no strict conditions for this handler, if user will select single - he will be able only to edit
     *      {'from'} destination port value. Either if he will select Range - he will be able to edit both.
     *
     * @param e {Object} Event
     * @param value {String}
     */
    setDestinationPortMode: function(e, value) {
        var entryPorts = this.getPortsValuesFromView('entry'),
            inputDestinationPortFrom = this.$('input[name="destinationPort"].from'),
            inputDestinationPortTo = this.$('input[name="destinationPort"].to'),
            spanDestinationFrom = this.$('.text-block-destination-from'),
            spanDestinationTo = this.$('.text-block-destination-to');

        if (value === "range") {

            // when changed by user
            if (e) {
                // Copy entry port 'from' and 'to' to destination ports:
                inputDestinationPortFrom.val(entryPorts[0]);
                inputDestinationPortTo.val(entryPorts[1]);

                // Copy values to spans:
                spanDestinationFrom.text(entryPorts[0]);
                spanDestinationTo.text(entryPorts[1]);
            }
        } else {

            // when changed by user
            if (e) {
                // Copy entry port 'from' to destination ports:
                inputDestinationPortFrom.val(entryPorts[0]);

                // Blink field on update
                this.notifyPortsUpdate(inputDestinationPortFrom);
            }

            // Re-validate field to clear message from range ports:
            this.pageValidation(inputDestinationPortFrom, false);
        }

        this.$('.destination-port-values-section').toggleClass("range-type", value === "range");
    },

    /**
     * Handler for settings values for {'Select device'} dropdown
     *
     * @description
     *
     *      Default value: Select Device
     *
     *      MUST contain the list of currently connected and recently connected devices
     */
    setAssignedDeviceOptions: function() {
        var dropdown = this.$('.swc-dropdown.device-selection'),
            options = {
                "0.0.0.0": {
                    name: getTranslationStrings("Choose a device"),
                    value: "0.0.0.0",
                    isDefault: true
                }
            };

        // Define which devices to show in dropdown:
        _.each(swc.models.NetworkDevices.models, function(model) {
            var ipAddress = model.get('address')['IPv4'];
            var addNewModel = true;

            // In this list can be devices which have IPv4 Address:
            if (!_.isEmpty(ipAddress) && !_.isUndefined(ipAddress)) {
                if("Guest-2.4GHz" === model.get("interfaceAP")) {
                    addNewModel = false;
                }
                if((_.isUndefined(model.get('addressObjects').IPv4) || _.isEmpty(model.get('addressObjects').IPv4)) && (model.get('addressObjects').IPv6 !== undefined && !_.isEmpty(model.get('addressObjects').IPv6))){
                    addNewModel = false;
                }

                if(addNewModel === true) {
                    options[ipAddress] = {
                        name: model.get('name'),
                        value: ipAddress,
                        additionalLabel: ipAddress
                    };
                }
            }
        });

        // append enter ip by user
        options["custom"] = {
            name: getTranslationStrings("Enter Device by IP"),
            value: "custom"
        };

        // Set dropdown data:
        dropdown.data('options', options);
        dropdown.data('hide-default', true);
        dropdown.trigger('swc-dropdown:swc-change', this.templateData.device);
    },

    /**
     * Handler for settings values for {'Protocol'} dropdown
     *
     * @description
     *
     *      Default value: TCP/UDP
     *
     *      Values of Protocol dropdown menu
     *          - TCP/UDP
     *          - TCP
     *          - UDP
     */
    setProtocolOptions: function() {
        var dropdown = this.$('.swc-dropdown.protocol-selection'),
            options = [
                { name: getTranslationStrings("TCP/UDP"), value: "6,17" },
                { name: getTranslationStrings("TCP"),     value: "6"    },
                { name: getTranslationStrings("UDP"),     value: "17"   }
            ];

        // Set dropdown data:
        dropdown.data('options', options);
        dropdown.trigger('swc-dropdown:swc-change', this.templateData.protocol);
    },

    /**
     * Handler for settings values for {'Predefined Rules'} dropdown
     *
     * @description
     *
     *      Default value: FTP
     *
     *      See list of rules in
     */
    setPredefinedRulesOptions: function() {
        var dropdown = this.$('.swc-dropdown.service-name-selection'),
            services = swc.models.PortForwarding.predefinedServices,
            options = [];

        // Go through each service and create options for dropdown
        _.each(services, function(service, key) {
            options.push({name: key, value: key});
        });

        // Set dropdown data:
        dropdown.data('options', options);
        dropdown.trigger('swc-dropdown:swc-change', this.templateData.servicePred);
    },

    /**
     * Handle for updating Destination ports, when Entry ports inputs are changing:
     *
     * @param e {Object}
     * @param value {String}
     */
    updateDestinationPorts: function(e) {
        var entryInput = $(e.target),
            entryPorts = this.getPortsValuesFromView('entry'),
            inputDestinationPortFrom = this.$('input[name="destinationPort"].from'),
            inputDestinationPortTo = this.$('input[name="destinationPort"].to'),
            spanDestinationFrom = this.$('.text-block-destination-from'),
            spanDestinationTo = this.$('.text-block-destination-to');

        // Blink field on update
        this.notifyPortsUpdate(inputDestinationPortFrom);

        // Update correct parts of inputs and text blocks
        if (entryInput.hasClass("from")) {
            inputDestinationPortFrom.val(entryPorts[0]);
            spanDestinationFrom.text(entryPorts[0]);
        } else {
            inputDestinationPortTo.val(entryPorts[1]);
            spanDestinationTo.text(entryPorts[1]);
        }

        this.pageValidation(inputDestinationPortFrom, false);
    },

    limitSetOfCharactersDown: function(e) {
        return swc.Utils.limitSetOfCharactersDown(e, /[^\d]/g);
    },

    limitSetOfCharacters: function(e) {
        return swc.Utils.limitSetOfCharactersInput(e, /[^\d]/g);
    },
    
    deviceSelectChanged: function() {
        var inputEntryPortFrom = this.$('input[name="entryPort"].from'),
            inputEntryPortTo = this.$('input[name="entryPort"].to'),
            inputDestinationPortFrom = this.$('input[name="destinationPort"].from');

        this.clearValidationMessages();
        this.postDeviceSelectionValidation();
        this.showHideCustomIp();

        if (!_.isEmpty(inputEntryPortFrom.val())) {
            this.pageValidation(inputEntryPortFrom, false);
            this.pageValidation(inputEntryPortTo, false);
            this.pageValidation(inputDestinationPortFrom, false);
        }
    },
    
    showHideCustomIp: function() {
        var deviceSelection = this.$('.swc-dropdown.device-selection'),
            value = deviceSelection.data('value'),
            address = this.$('.swc-input.ip-address');
            
        if(value === 'custom') {
            address.show();
        } else {
            address.hide();
            address.val('');
            //this.pageValidation(address, false);
        }
    },

    // Validates decimal values of IP Address and other fields
    validateIPAddressDecimal: function(Address, Netmask, DHCPMin, DHCPMax) {
        var validationStatus = {
            status: true,
            messages: []
        };
        
        if (Address === false) {
            validationStatus.status = false;
            validationStatus.messages.push("Invalid IPv4 Address Format");
            return validationStatus;
        }

        if (!(Address > 3232235520 && Address < 3232301055)/* 192.168.0.0 - 192.168.255.255*/ &&
            !(Address > 167772160 && Address < 184549375)/* 10.0.0.0 - 10.255.255.255 */ &&
            !(Address > 2886729728 && Address < 2887778303)/* 172.16.0.0 - 172.32.255.255 */) {
            validationStatus.status = false;
            validationStatus.messages.push("Non-private subnet");
            //'You have choosen an invalid network. Only private Networks according RFC 1918 are allowed'
            return validationStatus;
        }

        //We need Netmask to proceed validation
        if (Netmask === false) {
            return validationStatus;
        }
        
        var netmaskValidation = this.validateNetmaskSubnetDecimal(Address, Netmask);
        if (netmaskValidation.status === false) {
            return netmaskValidation;
        }

        //We need DHCP to validate further
        if (DHCPMin === false || DHCPMax === false) {
            return validationStatus;
        }
        var validDHCPSubnet = this.validateDHCPSubnetDecimal(DHCPMin, DHCPMax, Address, Netmask);
        if (validDHCPSubnet.status === false) {
            return validDHCPSubnet;
        }

        return validationStatus;
    },

    /**
     * Validate destionation ports after device has changed, because they are assigned to correct device
     */
    postDeviceSelectionValidation: function() {
        var inputDestinationPortFrom = this.$('input[name="destinationPort"].from'),
            inputDestinationPortTo = this.$('input[name="destinationPort"].to');

        if (!_.isEmpty(inputDestinationPortFrom.val())) {
            this.pageValidation(inputDestinationPortFrom, false);
            this.pageValidation(inputDestinationPortTo, false);
        }
    },

    /**
     * Notify that one of ports fields will be updated on the page
     * @param element
     */
    notifyPortsUpdate: function(element) {
        if (!_.isUndefined(element) && element.size()) {
            element.addClass('input-attention');

            setTimeout(function() {
                element.removeClass('input-attention');
            }, 250);
        }
    },

    /**
     * Update model which stores information about current rule before synching it with server
     * @returns {swc.constructors.PortForwardingRule}
     */
    updateRuleModel: function() {
        var ruleModel = _.isUndefined(this.model) ? new swc.constructors.PortForwardingRule() : this.model,
            data = {
                name: { elementName: "serviceName" },
                entryPortTo: { elementName: "entryPort", elementClass: "to" },
                entryPortFrom: { elementName: "entryPort", elementClass: "from" },
                protocol: { elementName: "protocol" },
                assignedDeviceIP: { elementName: "deviceIP" },
                customDeviceIP: { elementName: "address" },
                desinationPortTo: { elementName: "destinationPort", elementClass: "to" },
                desinationPortFrom: { elementName: "destinationPort", elementClass: "from" },
                predefinedMode: { elementName: "service-mode" },
                predefinedName: { elementName: "predefinedServiceName" }
            },
            elementsMap = [],
            ports = {
                entry: [],
                destination: []
            }, id, name, status,
            portsEntryMode = $(".entry-mode").data('value'),
            portsDestinationMode = $(".destination-mode").data('value'),
            result = null;

        // Collect elements map:
        _.each(this.getElements(), function(element) {
            elementsMap.push(getParameter($(element)));
        });

        // Get data values
        data = swc.Utils.getDataToValidate(data, elementsMap);

        // Update rule model values:
        if (data.predefinedMode === "custom") {
            id = swc.Utils.generateId();
            name = data.name;
            status = true;
        } else {
            // This is a case when rule is editing (some elements are absent)
            id = _.isString(data.predefinedMode) ? swc.Utils.generateId() : ruleModel.get('id');
            name = _.isString(data.predefinedMode) ? data.predefinedName : data.name;
            status = _.isString(data.predefinedMode) ? true : ruleModel.get('status');
        }

        ruleModel.set('id', $.trim(id));
        ruleModel.set('name', $.trim(name));
        ruleModel.set('status', status);

        ruleModel.set('protocol', data.protocol);
        
        if(data.assignedDeviceIP === 'custom') {
            ruleModel.set('deviceIP', data.customDeviceIP);
        } else {
            ruleModel.set('deviceIP', data.assignedDeviceIP);
        }

        // destination port -> port on stargate device
        if (portsEntryMode === "range") {
            ports.entry = [ data.entryPortFrom, data.entryPortTo ];
        } else {
            ports.entry = [ data.entryPortFrom ];
        }

        // destination port -> port on assigned device
        if (portsDestinationMode === "range") {
            ports.destination = [ data.desinationPortFrom, data.desinationPortTo ];
        } else {
            ports.destination = [ data.desinationPortFrom ];
        }
        
        ruleModel.set('ports', ports);
        result = [ruleModel];
        
        // for IPSec we need to double model!
        if($.trim(name) === 'IPSec') {
            var array = ports.entry[0].split(',');
            if(array.length > 1) {
                ruleModel.set('ports', {
                    entry: [$.trim(array[0])],
                    destination: [$.trim(array[0])]
                });
            
                var ruleModel2 = new swc.constructors.PortForwardingRule();
                ruleModel2.set('id', swc.Utils.generateId());
                ruleModel2.set('name', ruleModel.get('name'));
                ruleModel2.set('status', ruleModel.get('status'));
                ruleModel2.set('protocol', ruleModel.get('protocol'));
                ruleModel2.set('deviceIP', ruleModel.get('deviceIP'));
                ruleModel2.set('ports', {
                    entry: [$.trim(array[1])],
                    destination: [$.trim(array[1])]
                });
                
                result.push(ruleModel2);
            }
        }

        return result;
    },
    
    checkIfDeviceHasStaticAddress: function(address) {
        var result = false;

        if(swc.models.DHCPLeases.get(address)) {
            result = true;
        }

        return result;
    },
    
    addStaticLease: function (deviceIP) {
        var deferred = new $.Deferred();

        SWCElements.modalWindow.show({
            templateID: 'network:settings:ip-settings:addstaticlease-modal',
            templateData: {
                deviceIP: deviceIP
            },
            className: 'addstaticlease-modal',

            onApply: function() {
                SWCElements.modalWindow.hide();

                deferred.resolve();
            }
        });
        
        return deferred;
    },

    /**
     * Handler for saving page. Will trigger, when user will press "Save" button
     *
     * @description
     *
     *      After user will press "Save" button, page validation should trigger. If validation is success, rule has to
     *      be added to the NP, collection of rules has to be resynced and user has to be navigated to the Port Forwarding
     *      rules page, in order to see his rule.
     */
    save: function() {
        var me = this,
            updatedModel = this.updateRuleModel(),
            deferred = new $.Deferred(),
            toDo = [];
            
        // check if we are setting dmz for non static device
        var addStaticLease = false,
            address = updatedModel[0].get('deviceIP'), // second always will be same :)
            device = swc.models.NetworkDevices.getDeviceByIP(address),
            macaddr = swc.models.NetworkDevices.getDeviceMac(address);
        if(!_.isEmpty(macaddr) && (device.get('addressObjects').IPv4.length > 0) && address) {
            addStaticLease = !this.checkIfDeviceHasStaticAddress(macaddr);
        }
        
        if(updatedModel.length > 1) {
            // in first place, ignore redirect!
            toDo.push(me.saveModel(updatedModel[0]));
            toDo.push(me.saveModel(updatedModel[1]));
        } else {
            toDo.push(me.saveModel(updatedModel[0]));
        }
        
        function redirect(success, desc) {
            localStorage.setItem('port-forwarding:change-rule', success ? 'success' : me.getError(desc));
            swc.router.navigate('network/settings/port-forwarding', { skipUnsavedChanges: true });
        }
        
        $.when.apply(this, toDo).done(function() {
                if(addStaticLease) {
                    var DHCPLeasesData = [];
            
                    DHCPLeasesData.push([{
                        parameterName : "DHCPLeaseNewDevice",
                        parameterValue : macaddr
                    }, {
                        parameterName : "DHCPLeaseMacAddress",
                        parameterValue : macaddr
                    }, {
                        parameterName : "DHCPLeaseIPAddress",
                        parameterValue : address
                    }]);
        
                    $.when(swc.models.DHCPLeases.saveDHCPLeases(DHCPLeasesData)).done(function() {
                        me.stopPageLoading();
                        $.when(me.addStaticLease(address)).done(function () {
                            redirect(true);
                        });
                    }).fail(function(desc) {
                        redirect(true, desc);
                    });
                } else {
                    redirect(true);
                }
            }).fail(function(desc) {
                redirect(false, desc);
            });
        
        return deferred.promise();
    },
    
    getError: function(desc) {
        var result;
        
        switch(desc) {
            case 'overlapping':
                result = 'error-overlapping';
                break;
            default:
                result = 'error-default';
                break;
        }
        
        return result;
    },
    
    saveModel: function(model) {
        var deferred = new $.Deferred();
        
        // NP requires commit action after any changes done to rules:
        $.when(model.sync('create'), model.sync('commit'))
            .done(function(response) {
                var overlapping = false;
                if(response) {
                    if(_.isString(response)) {
                        try {
                            response = jQuery.parseJSON(response);
                        } catch(e) {
                            
                        }
                    }
                    if(!_.isEmpty(response.errors)) {
                        _.each(response.errors, function(error){
                            var desc = error["description"];
                            if (desc === "Overlapping rule") {
                                overlapping = true;
                            }
                        });
                    }
                }

                if(!overlapping) {
                    deferred.resolve();
                } else {
                    deferred.reject("overlapping");
                }
            })
            .fail(function() {
                deferred.reject();
            }
        );
        
        return deferred.promise();
    },

    /**
     * @override on {'swc.BaseView.cancelChanges()'} method
     *
     * Need to navigate user to Rules List page when clicking cancel
     *
     * @returns {boolean}
     */
    cancelChanges: function() {
        swc.router.navigate('network/settings/port-forwarding', { skipUnsavedChanges: true });
    }
});
;swc.constructors.NetworkSettingsPortforwardingEditView = swc.constructors.NetworkSettingsPortforwardingAddView.extend({

    className: 'port-forwarding-edit',

    disallowedDevices: ['starlink'],

    events: {
        'swc-radio-buttons:change .swc-radio-buttons.entry-mode': 'setEntryPortMode',
        'swc-radio-buttons:change .swc-radio-buttons.destination-mode': 'setDestinationPortMode',

        'change input[name="entryPort"].from': 'updateDestinationPorts',
        'change input[name="entryPort"].to': 'updateDestinationPorts',
        'input input[name="address"]' : 'limitSetOfCharactersAddress',
        'keydown input[name="address"]' : 'limitSetOfCharactersAddressDown'
    },

    setTemplateData: function() {
        var rule = swc.models.PortForwarding.findWhere({ id: localStorage.getItem('port-forwarding:edit-ruleID') }),
            device, isUndefined;

        // Rule is not existing. Impossible situation:
        if (_.isUndefined(rule)) {
            swc.router.navigate('network/settings/port-forwarding', { skipUnsavedChanges: true });
            return;
        }

        // Catch current model for future work with it:
        this.model = rule;

        // Check if device with current IP exists in NP db
        device = swc.models.NetworkDevices.getDeviceByIP(rule.get('deviceIP'));
        isUndefined = _.isUndefined(device);

        this.templateData = {
            ipAddressAP: swc.models.Network.getParameter('StorageIPAddress', 'status'),
            name: rule.get('name'),
            id: rule.get('id'),
            port: {
                entry: rule.get('ports').entry,
                destination: rule.get('ports').destination
            },
            portMode: {
                entry: rule.get('ports').entry.length === 2 ? "range" : "single",
                destination: rule.get('ports').destination.length === 2 ? "range" : "single"
            },
            device: !isUndefined ? rule.get('deviceIP') : "custom",
            protocol: rule.get('protocol'),
            address: isUndefined ? rule.get('deviceIP') : ""
        };
    },

    limitSetOfCharactersAddressDown: function(e) {
        return swc.Utils.limitSetOfCharactersDown(e, /[^\.\d]/g, [190]);
    },

    limitSetOfCharactersAddress: function(e) {
        return swc.Utils.limitSetOfCharactersInput(e, /[^\.\d]/g);
    },

    /**
     * @override on {'swc.BaseView.initialize()'} method
     *
     * Need to set new template in order to support /1/2/3/4/.../n/ lvls of views including.
     *
     * @returns {boolean}
     */
    initialize: function() {
        // Prepare template for current page:
        this.template = $.template("pageContent", swc.Templates.get('network:settings:port-forwarding:rules-edit').get('content'));

        // Extend global events:
        this.events = _.extend(
            swc.constructors.NetworkSettingsPortforwardingAddView.prototype.events,
            swc.base.PageView.prototype.events,
            this.events
        );
    },

    save: function() {
        var deferred = new $.Deferred(),
            updatedModel = this.updateRuleModel();

        // NP requires commit action after any changes done to rules:
        $.when(updatedModel[0].sync('update'))
            .done(function(response) {
                var overlapping = false;
                if(response) {
                    if(!_.isEmpty(response.errors)) {
                        _.each(response.errors, function(error){
                            var desc = error["description"];
                            if (desc === "Overlapping rule") {
                                overlapping = true;
                            }
                        });
                    }
                }
                
                if(!overlapping) {
                    localStorage.setItem('port-forwarding:change-rule', 'success');
                    swc.router.navigate('network/settings/port-forwarding', { skipUnsavedChanges: true });
                } else {
                    localStorage.setItem('port-forwarding:change-rule', 'error-overlapping');
                    swc.router.navigate('network/settings/port-forwarding', { skipUnsavedChanges: true });
                }
            })
            .fail(function() {
                localStorage.setItem('port-forwarding:change-rule', 'error-default');
                swc.router.navigate('network/settings/port-forwarding', { skipUnsavedChanges: true });
            });

        return deferred.promise();
    }
});
;swc.constructors.NetworkSettingsPortforwardingView = swc.base.TabView.extend({

    className: 'port-forwarding',

    allowedMods: [
        'expert'
    ],

    disallowedDevices: ['starlink'],

    events: {
        'click .expandable-list .trash.process-delete:not(.disabled)': 'deleteRule',
        'click .expandable-list .pencil.process-edit:not(.disabled)': 'editRule',

        'swc-checkbox:change .swc-checkbox.upnp': 'setUPnPStatus',
        'swc-checkbox:change .swc-checkbox.enable-rule': 'setRuleStatus'
    },

    models: [
        'UPnP',
        'PortForwarding',
        'NetworkDevices'
    ],

    pageUrl: 'network/settings/port-forwarding',

    setTemplateData: function() {
        var rules = swc.models.PortForwarding.toJSON(),
            ipAddressAP = swc.models.Network.getParameter('StorageIPAddress', 'status'),
            webRules = [],
            upnpRules = [],
            isAnyRule = false;

        // Get device name by ip from rule:
        _.each(rules, function(rule) {
            isAnyRule = true;
            
            var deviceIP = rule.deviceIP,
                origin = rule.origin;
            
            // Filter rules - hide when dest ip is AP address or Origin is 'cwmp' (TR-069)
            if(origin !== 'cwmp' && deviceIP !== ipAddressAP) {
                var device = swc.models.NetworkDevices.getDeviceByIP(deviceIP);
    
                /**
                 * NP returns IP address of Assigned Device. And there can be a situation that device IP was changed, or
                 * device was not connected to the NP during last 10 days. So assigned IP for rule will refer to nowhere,
                 * because such device will no longer exist in NP base.
                 */
                if (!_.isUndefined(device)) {
                    rule.deviceName = device.get('name');
                } else {
                    rule.deviceName = deviceIP;
                }
                
                switch(origin) {
                    case 'webui':
                        webRules.push(rule);
                        break;
                    case 'upnp':
                        upnpRules.push(rule);
                        break;
                }
            }
        });

        this.templateData = {
            upnpStatus: swc.models.UPnP.get('status'),
            rules: [
                { name: null, list: webRules },
                { name: getTranslationStrings('Established by UPnP IGD'), list: upnpRules }
            ],
            isAnyRule: isAnyRule
        };
    },

    setUPnPStatus: function(e, upnpStatus) {
        swc.models.UPnP.set('status', upnpStatus);

        // Update buttons state to see, that upnp has been changed
        this.setButtonsState();
    },

    deleteRule: function(e) {
        var parameter = getParameter($(e.target)),
            ruleID = parameter.parameterData.rule.toString(),
            ruleItem = this.$('.expandable-list-item[data-key="' + ruleID + '"]'),
            model = swc.models.PortForwarding.findWhere({ id: ruleID });

        // Update model in the collection:
        swc.models.PortForwarding.remove(model);

        // Remove item from the list
        ruleItem.remove();

        // Update buttons state to see, that rule have been deleted
        this.setButtonsState();
    },

    setRuleStatus: function(e, value) {
        var parameter = getParameter($(e.target)),
            ruleID = parameter.parameterData.rule.toString(),
            model = swc.models.PortForwarding.findWhere({ id: ruleID });

        // Update model in the collection:
        model.set('status', value);

        // Update buttons state to see, that rule have been deleted
        this.setButtonsState();
    },

    pageCheckDefaultValues: function() {

        // BAD BAD BAD FIXME :: fix this when we will have same handling in all views
        this.parentView.pageCheckDefaultValues = function() {
            return !swc.models.UPnP.hasChanged() && !swc.models.PortForwarding.hasChanged();
        };

        return !swc.models.UPnP.hasChanged() && !swc.models.PortForwarding.hasChanged();
    },

    editRule: function(e) {
        var parameter = getParameter($(e.target));

        // This is done to get Rule on edit page
        localStorage.setItem('port-forwarding:edit-ruleID', parameter.parameterData.rule);
        localStorage.setItem('port-forwarding:editing', 'true');

        swc.router.navigate(this.pageUrl + '/edit', { trigger: true });
    },

    save: function() {
        var deferred = new $.Deferred(),
            toDo = [
                swc.models.UPnP.sync('update')
            ],
            deletedRules = swc.models.PortForwarding.changedModels({ onlyDeleted: true  }),
            changedRules = swc.models.PortForwarding.changedModels({ onlyEdited: true  });

        _.each(deletedRules, function(model) {
            toDo.push(
                model.sync('delete'),
                model.sync('commit')
            );
        });

        _.each(changedRules, function(model) {
            toDo.push(
                model.sync('update'),
                model.sync('commit')
            );
        });

        $.when.apply(this, toDo)
            .done(function() {
                deferred.resolve();
            })
            .fail(function() {
                deferred.reject();
            });

        return deferred.promise();
    },

    renderComplete: function() {
        var ruleActionSuccess = localStorage.getItem('port-forwarding:change-rule');

        // Display rule saving success or fail message:
        if (!_.isNull(ruleActionSuccess)) {
            localStorage.removeItem('port-forwarding:change-rule');
            this.showSaveSuccess(ruleActionSuccess);
        }
    },
    
    showSaveSuccess: function(type) {
        var messageNew = this.$('.buttons-container-message .save-' + type);

        // Hide previous messages:
        this.$('.buttons-container-message .save-error').hide();
        this.$('.buttons-container-message .save-success').hide();

        messageNew.show();
    }

});
;swc.constructors.NocookiesView = swc.base.PageView.extend({

    className: 'nocookies',

    pageTemplateID: 'nocookies',

    preRender: function () {
        // Login View has special layout
        if (swc.views.Application.contentArea && swc.views.Application.contentArea.size()) {
            swc.views.Application.contentArea.remove();
            swc.views.Application.prepareLayout('login-layout');
        }
    },

    setTemplateData: function() {
        this.templateData = {
            localeStrings: swc.models.Locale.getLocaleStrings(),
            localeString: getTranslationStringsjQuery,
            formatDate: swc.models.Locale.formatDate
        };
    },
    
    checkPermissions: function() {
        // skip this, nocookies is visible for everyone
    }
});
;swc.constructors.OverviewView = swc.base.PageView.extend({

    className: 'overview',

    events: {
        'click .cloud-login-button': 'loginToCloudService',

        'click .customize-device': 'editDevice',
        'click .customize-dect': 'editDect',
        'click .usb-secure-remove': 'secureRemoveUSB',
        'click .speed-check': 'redirectToSpeedCheck',

        'click .js-go-to-cloud': 'navigateToCloudManagment',

        'swc-switcher:change .set-wifi-state': 'setWiFiState',
        'swc-switcher:change .set-dect-state': 'setDECTState',
        'swc-switcher:change .set-central-storage-state': 'setCentralStorageState'
    },
    //hook for event dispatcher, we need it to stop the event listener
    eventDispatcher:  null,

    listenerInterval: 10,

    listenerEnabled: true,

    listenerPages: [
        'overview'
    ],
    refreshIntervalDelay: 2000,

    usbRefreshIntervalDelay: 10000,
    
    usbRefreshIntervalCalls:0,
    
    usbRefreshInterval:null,
    
    apRefreshIntervalCalls:0,
    
    apRefreshInterval:null,
    
    trunk8settings: {
       lines: 2
    },

    failOverLoopCount: 0,
    failOverLoopName: 'overview',

    models: [
        'SchedulerWlan', 'CentralStorage', 'СloudServicesStatus', 'CloudAccountCollection', 'CloudServices',
        'Telephony', 'Network', 'NetworkDevices', 'DHCPLeases',
        'FirewallStatus', 'StorageDevices',
        'InternetBackupStick', 'USBHostDeviceCollection',
        'VPNCollection',
        'EventDispatcher'
    ],

     //'event' is an event name and 'action' is a function name in a current scope
    eventActions: [
      {
        event: 'NeMo.Intf.wwan',
        action: 'updateStick'
      },
      {
        event: 'wifi_device_updated',
        action: 'networkDevicesUpdated'
      },
      {
        event: 'eth_device_updated',
        action: 'networkDevicesUpdated'
      },
      {
        event: 'voice_device_updated',
        action: 'telephoneyDevicesUpdated'
      },
      {
        event: 'self_device_updated',
        action: 'selfDevicesUpdated'
      },
      {
        event: 'changed',
        action: 'vpnClientUpdated'
      },
      {
        event: 'usb_device_added',
        action: 'usbDevicesUpdated'
      },
      {
        event: 'add',
        action: 'updateCentralStorageDelayed'
      },
      {
        event: 'usb_notification_device_deleted',
        action: 'updateCentralStorageDelayed'
      },
      {
        event: 'device_deleted',
        action: 'usbDevicesUpdated'
      }
    ],

    navigateToCloudManagment: function () {
        swc.router.navigate('storage/settings/cloud-backup/list', { trigger: false, skipUnsavedChanges: true});
    },

    passEvent: function(event, eventObject) {
        swc.constructors.EventController(this.eventActions)
                      .pass.call(this, event, eventObject);
    },

    updateStick: function() {
        swc.Utils.reSync("InternetBackupStick", "read");
        swc.models.NetworkDevices.trigger('devices-loaded', true);
    },
    updateCentralStorageDelayed: function() {
        var self = this;

        setTimeout(function() {
            self.updateCentralStorage();
        }, 800);
    },

    updateCentralStorage: function() {
        $.when(swc.Utils.reSync("StorageDevices", "read"), swc.Utils.reSync("CentralStorage", "read"))
            .done(function(){
                swc.models.CentralStorage.trigger('central-storage-update');
            });
    },

    networkDevicesUpdated: function(event) {
      swc.Utils.updateDevicesByModelName("NetworkDevices", event);
    },

    telephoneyDevicesUpdated: function(event) {
      swc.Utils.updateDevicesByModelName("Telephony", event);
    },

    selfDevicesUpdated: function(event) {
      swc.Utils.updateDevicesByModelName("NetworkDevices", event, this.switchSelfUpdated);
    },

    usbDevicesUpdated: function() {
        $.when(swc.Utils.reSync("USBHostDeviceCollection", "read"))
            .done(function(){
                swc.models.USBHostDeviceCollection.trigger('devices-loaded', true);
            });
    },
    switchSelfUpdated: function(who, self, arr) {
      switch(who.Name){
       case "wl0":
          swc.models.Wireless.set24State(who);
       break;
       case "wlguest2":
          swc.models.Wireless.setGuestState(who);
       break;
       case "wl1":
          swc.models.Wireless.set5State(who);
       break;
       case "AP":
       case 'internetbox-nas':
            swc.Utils.reSync('CentralStorage');
       break;
       case "lan":
       break;
       case "eth0":
       break;
       case "eth1":
       break;
       case "eth2":
       break;
       case "eth3":
       break;
       case "eth5": //fiber
       break;
       case "internetbox":
          swc.Utils.reSync("Network");
       break;
       default:
          arr.push(who);
         //console.debug(who);
       break;
    }
  },
    setTemplateData: function() {
        var fibreState = swc.models.Network.getParameter('ConnectionStatus', 'status'),
            connectionMode = swc.models.Network.getParameter('ConnectionMode', 'status'),
            modemState = swc.models.InternetBackupStick.get('isConnected'),
            internetState = ((fibreState !== false) || (modemState !== false)),
            phones = swc.models.Telephony.getPhonesList(),
            cloudAccounts = swc.models.CloudAccountCollection.models,
            cloud = !_.isEmpty(cloudAccounts) ? cloudAccounts[0].getExtendedJSON() : '';

        if(swc.models.Network.doesParameterExist('ConnectionStatus', 'status') === false) {
            fibreState = true;
            internetState = ((fibreState !== false) || (modemState !== false));
        }

        this.templateData = {
            'dectStatus': swc.models.Telephony.get('dectStatus'),
            'device': swc.Utils.DeviceType,
            'voipStatus': swc.models.Telephony.get('voipStatus'),
            'wiredPhones': !_.isEmpty(phones.cord) ? phones.cord : [],
            'internetState': internetState,
            'connectionMode': connectionMode,
            'wiredState': true, // This value is hardcoded because there is no way to check real status
            'dectState': swc.models.Telephony.get('dectStatus'),
            'centralStorageState': swc.models.CentralStorage.getParameter("status", "status"),
            'IPv6Address': swc.models.Network.getParameter('IPv6Address', 'status'),
            'GlobalEnable': swc.models.Application.get('GlobalEnable'),
            'wifiState': swc.models.Wireless.getParameter("status", "status"),
            'wifiSchedulerState': swc.models.SchedulerWlan.get('enable') || false,
            'cloudService': cloud,
            'conflict': swc.models.NetworkDevices.getConflictStatus()
        };
    },

    filterVpnClients: function() {
        var collection = [];

        collection = swc.models.VPNCollection.filter(function(model) {
            var response = false;

            if (model.get('deleted')) {
                response = false;
            } else {
                response = true;
            }

            return response;
        });

        return collection;
    },

    isSpeedTestAvailable: function() {
        var speedCheck = $('.speed-check-module');

        if(swc.Utils.DeviceType === 'default') {
            swc.models.Rest.sendRequest({
                url: '/ws',
                data: {
                    "service":"APController",
                    "method":"getSoftWareVersion",
                    "parameters": {}
                },

                success: function(response) {
                    if(response.status){
                        var APVersion = response.data.version || '0.0.0';
                
                    // find major version
                        APVersion = parseInt(APVersion.split('.')[0], 10);
                        if(APVersion < 5) {
                            speedCheck.hide();
                        } else {
                            speedCheck.show();
                        }
                    } else {
                        speedCheck.hide();
                    }
                },

                error: function() {
                    speedCheck.hide();
                }
            });
        }
    },

    renderComplete: function() {
        var self = this;
        self.eventDispatcher = swc.models.EventDispatcher;
        self.eventDispatcher.observ("overview",self);

        swc.models.NetworkDevices.getConflicts();
        //block autosync on models that can handle events
        swc.models.Wireless.autoSync = false;
        swc.models.NetworkDevices.autoSync = false;
        swc.models.Network.autoSync = false;
        swc.models.Telephony.autoSync = false;

        //blocks autosync on the rest of models and helper models
        //contains Filemanager constructor 
        swc.models.StorageDevices.autoSync = false;
        swc.models.SchedulerWlan.autoSync = false;
        swc.models.CentralStorage.autoSync = false;
        swc.models.СloudServicesStatus.autoSync = false;
        swc.models.CloudAccountCollection.autoSync = false;
        swc.models.CloudServices.autoSync = false;
        swc.models.DHCPLeases.autoSync = false;
        swc.models.FirewallStatus.autoSync = false;
        //Backup stick connection state should be added
        swc.models.InternetBackupStick.autoSync = false;
        swc.models.USBHostDeviceCollection.autoSync = false;
        
        // Set listener on Network State change:
        swc.models.Network.on('change', function() {
            self.updateInternetState();
            self.updateGatewayState();
        });

        // Set listener on Backup stick change:
        swc.models.InternetBackupStick.on('change', function() {
            self.updateInternetState();
            self.updateGatewayState();
            self.renderUSBDevices();
        });

        // Set Listener on Central Storage manager change:
        swc.models.CentralStorage.on('central-storage-update', function() {
            self.updateCentralStorageState();
            self.renderUSBDevices();
        });

        swc.models.CloudAccountCollection.on('change', function() {
            self.updateCloudState();
        });

        // Set Listener on WiFi manager change:
        swc.models.Wireless.on('change', function() {
            self.updateWirelessState();
        });

        // Set Listener on WAN Device manager change:
        swc.models.NetworkDevices.on('devices-loaded', function() {
            self.updateNetworkState();
            swc.models.NetworkDevices.getConflicts();
            self.renderNetworkDevices();
            self.updateGatewayState();
        });

        swc.models.USBHostDeviceCollection.on('devices-loaded', function(){
            self.renderUSBDevices();
        });

        // Set default states:
        self.updateInternetState();
        self.updateGatewayState();
        self.updateNetworkState();
        self.updateWirelessState();
        self.updatePhonesState();
        self.updateCentralStorageState();
        self.updateCloudState();
        
        // Render devices:
        self.renderNetworkDevices();
        self.renderUSBDevices();
        self.renderDECTDevices();
        
        self.isSpeedTestAvailable();

        self.failOverLoopCount = 0;

        function failOverDect(loopCount) {
            $.when(
                swc.models.Telephony.sync(false, self.failOverLoopName)
            ).done(function(rep) {
                var pagesMatch = false,
                pagesRoutes = swc.settings.application.get('navigation'),
                page = Backbone.history.fragment;

                $.each(self.listenerPages, function(key, value) {
                    if (pagesRoutes[value] === page) {
                        pagesMatch = true;
                    }
                });

                if (pagesMatch) {
                    if(rep === true) {
                        self.updatePhonesState();
                        self.renderDECTDevices();
                    } else if(loopCount === self.failOverLoopCount && loopCount < 100) {
                        self.failOverLoopCount = loopCount + 1;
                        failOverDect(loopCount + 1);
                    }
                }
            });
        }

        if(swc.models.Telephony.get('errorStatus') === true) {
            swc.models.Telephony.set('errorStatus', undefined);
            failOverDect(0);
        }
    },

    vpnClientUpdated: function() {
        this.updateVPNClients();
    },

    updateVPNClients: function() {
        var self = this;
        $.when(swc.models.VPNCollection.fetch()).done(function() {
            self.renderNetworkDevices();
        }).fail(function() {
        });
    },

    updateInternetState: function() {
        if(swc.models.Network.doesParameterExist('ConnectionStatus', 'status') === false) {
            return;
        }

        var connection = this.$el.find('.connection.internet'),
            connectionState = connection.hasClass('disconnected') ? false : true,
            speedCheck = this.$el.find('.speed-check'),
            connectionType = this.getConnectionType(),
            connectionData = {
                "status": swc.models.Network.getParameter('ConnectionStatus', 'status'),
                "LTEmodemStatus": swc.models.InternetBackupStick.get('isConnected'),
                "localIPAddress" : swc.models.Network.getParameter('LocalIPAddress',  'ip-settings'),
                "remoteIPAddress": swc.models.Network.getParameter('RemoteIPAddress', 'status'),
                "connectionType" : connectionType,
                "lastConnection" : swc.models.Network.getParameter('LastConnection',  'status'),
                "deviceName"     : this.getInternetBackupStickName()
            },
            elementData = connection.data('popover-template-data'),
            elementDataChanged = this.checkElementDataChanged(elementData, connectionData);

        // Set visual connection updates

        connection.toggleClass('disconnected', !connectionData.status);
        connection.toggleClass('LTEmodemConnected', connectionData.LTEmodemStatus);
        speedCheck.toggleClass('disconnected', !connectionData.status);

        // Update popover information if something was changed to current element:
        if (elementDataChanged || connectionState !== connectionData.status) {
            this.updateConnectionsData(connection, connectionData);
            this.updateConnectionsPopover(connection);
        }
    },

    getInternetBackupStickName: function() {
        return swc.models.InternetBackupStick.get('manufacturer') + ' ' + swc.models.InternetBackupStick.get('model');
    },
    
    getConnectionType: function() {
        var result = '';
        
        if(swc.models.InternetBackupStick.get('isConnected')) {
            result = 'GSM';
        } else {
            switch(swc.models.Network.getParameter('ConnectionType',  'status')) {
                case 'ethernet':
                    result = 'Ethernet';
                    break;
                case 'vdsl':
                    result = 'VDSL';
                    break;
                case 'dsl':
                    result = 'ADSL';
                    break;
            }
        }
        
        return result;
    },

    updateGatewayState: function() {
        var connection = this.$el.find('.connection.gateway'),
            connectionState = connection.hasClass('disconnected') ? false : true,
            IPv6Address = swc.models.Network.getParameter('IPv6Address',  'status'),
            storageIP = swc.models.Network.getParameter('StorageIPAddress', 'status'),
            conflict = swc.models.NetworkDevices.getConflictStatus(),
            connectionData = {
                "status"          : swc.models.Network.getParameter('ConnectionStatus', 'status'),
                "device"          : swc.Utils.DeviceType,
                "connectionMode"  : swc.models.Network.getParameter('ConnectionMode', 'status'),
                "localIPAddress"  : swc.models.Network.getParameter('LocalIPAddress',  'ip-settings'),
                "remoteIPAddress" : swc.models.Network.getParameter('RemoteIPAddress', 'status'),
                "storageIPAddress": storageIP,
                "connectionType"  : swc.models.Network.getParameter('ConnectionType',  'status'),
                "lastConnection"  : swc.models.Network.getParameter('LastConnection',  'status'),
                'IPv6Address'     : IPv6Address,
                "deviceName"      : swc.models.Network.getParameter('DeviceName',      'status'),
                "GlobalEnable"    : swc.models.Application.get('GlobalEnable'),
                "conflict"        : swc.models.NetworkDevices.getConflictStatus(),
                "APconflict"      : (swc.Utils.DeviceType === "default") ? swc.models.NetworkDevices.getConflictStatus(storageIP) : false
            },
            elementData = connection.data('popover-template-data'),
            elementDataChanged = this.checkElementDataChanged(elementData, connectionData);

        // Set visual connection updates
        connection.toggleClass('disconnected', !connectionData.status);
        connection.toggleClass('conflict', conflict);

        // Update popover information if something was changed to current element:
        if ((elementDataChanged || connectionState !== connectionData.status) && (connectionData.localIPAddress !== false && connectionData.remoteIPAddress !== false && connectionData.storageIPAddress !== false)) {
            this.updateConnectionsData(connection, connectionData);
            this.updateConnectionsPopover(connection);
        }
    },

    updateNetworkState: function() {
        var connection = this.$el.find('.connection.wired'),
            connectionState = connection.hasClass('disconnected') ? false : true,
            connectionData = {
                "status": true,
                "wiredDevicesCount": swc.models.NetworkDevices.countWiredConnectedDevices()
            },
            elementData = connection.data('popover-template-data'),
            elementDataChanged = this.checkElementDataChanged(elementData, connectionData);

        // Set visual connection updates
        connection.toggleClass('disconnected', !connectionData.status);

        // Update popover information if something was changed to current element:
        if (elementDataChanged || connectionState !== connectionData.status) {
            this.updateConnectionsData(connection, connectionData);
            this.updateConnectionsPopover(connection);
        }
    },

    updateCloudState: function() {
        var connection = this.$el.find('.connection.cloud'),
            connectionState = connection.hasClass('disconnected') ? false : true,
            elementData = connection.data('popover-template-data'),
            models = swc.models.CloudAccountCollection.models,
            cloud = !_.isEmpty(models) ? models[0].getExtendedJSON() : '',
            connectionData = {
                'status': !!cloud,
                "centralStorageStatus": swc.models.CentralStorage.getParameter("status", "status"),
                'internetStatus': swc.models.Network.getParameter('ConnectionStatus', 'status'),
                'cloudService': cloud
            },
            elementDataChanged = this.checkElementDataChanged(elementData, connectionData),
            status = connectionData.cloudService.status,
            error = connectionData.cloudService.error,
            credentialsProblem = status === 'ERROR' && error === 'INVALID_CREDENTIALS';

        // Set visual connection updates
        if (!connectionData.cloudService.status) {
            connection.addClass('disconnected');
        } else {
            connection.removeClass('disconnected');
            connection.toggleClass('credentialsProblem', credentialsProblem);
            connection.toggleClass('complete', status === 'COMPLETE');
            connection.toggleClass('synching', status === 'IN_PROGRESS');
            connection.toggleClass('warning', status === 'ERROR');
            connection.toggleClass('error', status === 'STORAGE_DISCONNECTED');

            // Set current service to visible state:
            connection.find('.cloud-service').addClass('hidden');
            connection.find('.' + connectionData.cloudService.id + '-account').removeClass('hidden');
        }

        // Update popover information if something was changed to current element:
        if (elementDataChanged || connectionState !== connectionData.status) {
            this.updateConnectionsData(connection, connectionData);
            this.updateConnectionsPopover(connection);
        }
    },

    updatePhonesState: function() {
        var phones = swc.models.Telephony.getPhonesList(),
            dectStatus = swc.models.Telephony.get('dectStatus'),
            voipStatus = swc.models.Telephony.get('voipStatus'),
            connection = this.$el.find('.connection.phones'),
            connectionPortOne = this.$el.find('.list.phones .static-phone-1'),
            connectionPortTwo = this.$el.find('.list.phones .static-phone-2'),
            connectionState = connection.hasClass('disconnected') ? false : true,
            connectionData = {
                "dectStatus": dectStatus,
                "voipStatus": voipStatus,
                "port-1": {
                    "status": voipStatus && (!_.isEmpty(phones.cord[0]) && _.isBoolean(phones.cord[0].isDisabled) && !phones.cord[0].isDisabled),
                    "number": 1,
                    "numbers":  {
                        "internal": !_.isEmpty(phones.cord[0]) ? phones.cord[0].internalNumber : '',
                        "external": !_.isEmpty(phones.cord[0]) ? phones.cord[0].externalNumber : ''
                    }
                },
                "port-2": {
                    "status": voipStatus && (!_.isEmpty(phones.cord[1]) && _.isBoolean(phones.cord[1].isDisabled) && !phones.cord[1].isDisabled),
                    "number": 2,
                    "numbers":  {
                        "internal": !_.isEmpty(phones.cord[1]) ? phones.cord[1].internalNumber : '',
                        "external": !_.isEmpty(phones.cord[1]) ? phones.cord[1].externalNumber : ''
                    }
                }
            },
            elementData = connection.data('popover-template-data'),
            elementDataChanged = this.checkElementDataChanged(elementData, connectionData);

        // Set visual connection updates
        connection.toggleClass('disconnected', !connectionData.dectStatus);

        // Add class to mark connection as partially connected
        connection.toggleClass('partially', connectionData.voipStatus);

        // Set phone ports connection state:
        connectionPortOne.toggleClass('disconnected', !connectionData["port-1"].status);
        connectionPortTwo.toggleClass('disconnected', !connectionData["port-2"].status);

        // Update popover information if something was changed to current element:
        if (elementDataChanged || connectionState !== connectionData.status) {
            this.updateConnectionsData(connection, connectionData);
            this.updateConnectionsData(connectionPortOne, connectionData["port-1"]);
            this.updateConnectionsData(connectionPortTwo, connectionData["port-2"]);

            this.updateConnectionsDataName(connectionPortOne, connectionData["port-1"]);
            this.updateConnectionsDataName(connectionPortTwo, connectionData["port-2"]);

            this.updateConnectionsPopover(connection);
            this.updateConnectionsPopover(connectionPortOne);
            this.updateConnectionsPopover(connectionPortTwo);
        }
    },

    updateCentralStorageState: function() {
        // swc.models.StorageDevices.getDevices();
        var connection = this.$el.find('.connection.usb'),
            connectionState = connection.hasClass('disconnected') ? false : true,
            connectionData = {
                "status": (swc.Utils.DeviceType === 'default') ? (swc.models.CentralStorage.getParameter("status", "status") && swc.models.Application.get('GlobalEnable')) : true,
                "usbDevicesCount": swc.models.CentralStorage.getParameter("devices", "usb-devices").length,
                "mediaServerStatus": swc.models.CentralStorage.getParameter("mediaServerStatus", "status"),
                "GlobalEnable": swc.models.Application.get("GlobalEnable")
            },
            elementData = connection.data('popover-template-data'),
            elementDataChanged = this.checkElementDataChanged(elementData, connectionData);

        // Set visual connection updates
        connection.toggleClass('disconnected', !connectionData.status);

        // Update popover information if something was changed to current element:
        if (elementDataChanged || connectionState !== connectionData.status) {
            this.updateConnectionsData(connection, connectionData);
            this.updateConnectionsPopover(connection);
        }
    },

    updateWirelessState: function() {
        var connection = this.$el.find('.connection.wifi'),
            connectionState = connection.hasClass('disconnected') ? false : true,
            connectionData = {
                "status": swc.models.Wireless.getParameter("status", "status") === "on",
                "wifiSchedulerState": swc.models.SchedulerWlan.get('enable'),
                "wifiCombinedMode": swc.models.Wireless.getParameter("configurationMode", "status"),
                "wifi24GHzName":   swc.models.Wireless.getParameter("name", "2.4GHz"),
                "wifi24GHzStatus": swc.models.Wireless.getParameter("status", "2.4GHz"),
                "wifi5GHzName":    swc.models.Wireless.getParameter("name", "5GHz"),
                "wifi5GHzStatus":  swc.models.Wireless.getParameter("status", "5GHz"),
                "wifiGuestName":   swc.models.Wireless.getParameter("name", "Guest-2.4GHz"),
                "wifiGuestStatus": swc.models.Wireless.getParameter("status", "guestStatus")
            },
            elementData = connection.data('popover-template-data'),
            elementDataChanged = this.checkElementDataChanged(elementData, connectionData);

        // Set visual connection updates
        connection.toggleClass('disconnected', !connectionData.status);

        // Update popover information if something was changed to current element:
        if (elementDataChanged || connectionState !== connectionData.status) {
            this.updateConnectionsData(connection, connectionData);
            this.updateConnectionsPopover(connection);
        }
    },

    updateConnectionsPopover: function(connection) {
        if (connection.hasClass('selected')) {
            SWCElements.popovers.open(connection.find('.icon'));
        }
    },

    updateConnectionsData: function(element, connectionData) {
        var prefix = element.hasClass('disconnected') ? 'disconnected' : 'connected';

        element.data({
            "popover-template-id": "overview:popovers:" + element.data('type') + "-" + prefix,
            "popover-element": ".icon",
            "popover-placement": "left",
            "popover-holder": ".connection." + element.data("type"),
            "popover-container": ".current-page .overview",
            "popover-template-data": connectionData
        });
    },

    updateConnectionsDataName: function(element, connectionData) {
        var prefix = element.hasClass('disconnected') ? 'disconnected' : 'connected';

        element.find('.has-popover').data({
            "popover-template-id": "overview:popovers:" + element.data('type') + "-" + prefix,
            "popover-element": ".name",
            "popover-placement": "left",
            "popover-holder": ".connection." + element.data("type"),
            "popover-container": ".current-page .overview",
            "popover-template-data": connectionData
        });
    },

    checkElementDataChanged: function(elementData, newData) {
        var elementDataChanged = false;

        if (!elementData) {
            elementDataChanged = true;
        } else {
            $.each(newData, function(key, value) {
                if (!typeof(elementData[key])) {
                    elementDataChanged = true;
                } else {
                    if (elementData[key] !== value) {
                        elementDataChanged = true;
                    }
                }
            });
        }

        return elementDataChanged;
    },

    renderNetworkDevices: function() {
        var NetworkDevices = swc.models.NetworkDevices.getConnectedDevices(),
            DHCPLeases = _.pluck(swc.models.DHCPLeases.getActiveLeases(), 'MACAddress'),
            template = swc.Templates.get("components:device").get('content'),
            disconnected = { "wired": 0, "wireless": 0 },
            activeDevicePhysAddress,
            self = this;

        var clients = this.filterVpnClients();
        var client = null, vpnClients = [];

        $.each(clients, function(index, tmpClient) {
            var ipAddress = tmpClient.get('ipAddress');
            client = {};
            client.status = "connected";
            client.name = "VPN Remote";
            client.icon = "vpn";
            client.IPv6Enabled = false;
            client.address = {};
            client.address.IPv4 = ipAddress;
            client.dnsName = ipAddress;
            client.hasConflict = false;
            client.interfaceType = "wired";
            client.isHidden = false;
            client.nameShort = ipAddress;
            client.isVPNDevice = true;
            client.type = "vpn";
            client.address.IPv6 = [];
            client.address.IPv6.push(ipAddress);
            client.addressObjects = {};
            client.addressObjects.IPv4 = [];
            client.addressObjects.IPv6 = [];
            client.mac = ipAddress;
            client.key = ipAddress;
            vpnClients.push(client);
        });

        if(vpnClients.length > 0) {
            NetworkDevices['wired'] = NetworkDevices['wired'].concat(vpnClients);
            // NetworkDevices['wired'].push(clients);
        }

        // Fill in WAN Devices List:
        $.each(NetworkDevices, function(connectionType, devicesArray) {
            var list = self.$el.find('.devices-list .list.' + connectionType),
                selectedDevice = list.find('.device.has-popover.selected');

            // Check if any device has popover on it
            if (selectedDevice.size()) {
                activeDevicePhysAddress = selectedDevice.data('macaddress');
                SWCElements.popovers.closeAll();
            }

            // Remove old devices:
            list.find('.device').remove();

            if (!NetworkDevices[connectionType].length) {
                list.find('.no-devices').show();
            } else {
                list.find('.no-devices').hide();
            }

            //Looking for IP conflicts
            swc.models.NetworkDevices.getConflicts();

            // Render devices
            $.each(devicesArray, function(key, device) {
                var templatePrefix = device.status ? 'connected' : 'disconnected';

                // Only 10 not connected devices must be shown to the user:
                if (!device.status && disconnected[device.interfaceType] >= 10 || device.name === 'internetbox-nas' ) {
                    return;
                }
                if( _.contains(DHCPLeases, device.mac)) {
                   device.address['static'] = true;
                }
                // Set correct length of device name:
                device.nameShort = device.name;
                device.IPv6Enabled = swc.models.FirewallStatus.get('state').Enable;

                // All IPv6 Addresses have to be displayed in short form:
                // Display only global IPv6 addresses on overview page if they are present.
                // If only link scope address present, display link scope address
                if (!_.isUndefined(device.addressObjects) && device.addressObjects.IPv6.length > 0) {
                    device.address.IPv6 = [];
                    $.each(device.addressObjects.IPv6, function(key, address) {
                        // Show only Global IPv6 address on Overview page
                        if (address.Scope === "global") {
                            device.address.IPv6.push(swc.Utils.formatIPv6Address(address.Address));
                        }
                    });
                }

                var classNames = (device.status ? 'connected' : 'disconnected') +
                                 (device.hasConflict ? ' conflict' : '');

                if(device.type === "vpn") {
                    templatePrefix = "vpn";
                }

                // Append device to the list:
                list.append(
                    $.tmpl(template, {
                        device: device,

                        className: classNames,

                        popoverEnable: true,
                        popoverTemplateID: 'overview:popovers:device-' + templatePrefix,
                        popoverTemplateData: JSON.stringify(device),
                        popoverContainer: '.current-page .overview',
                        popoverPlacement: connectionType === 'wired' ? 'right' : 'left',
                        localeStrings: swc.models.Locale.getLocaleStrings(),
                        localeString: getTranslationStringsjQuery,

                        // Need to make a moving to new line in template
                        lastInRow: (key + 1) % 3 === 0 ? "last-in-row" : ""
                    })
                );
            });

            // Reopen popover for selected device:
            if (activeDevicePhysAddress && activeDevicePhysAddress.length) {
                SWCElements.popovers.open(
                    $('.device.has-popover[data-macaddress="' + activeDevicePhysAddress + '"]').find('.icon')
                );
            }
        });

        $('.device .name').trunk8(self.trunk8settings);
    },

    renderUSBDevices: function() {
        var usbDevices = !swc.models.Application.get('GlobalEnable') ? [] : swc.models.CentralStorage.getParameter("devices", "usb-devices"),
            template = swc.Templates.get("components:device-storage").get('content'),
            list = this.$el.find('.devices-list .list.usb'),
            selectedDevice = list.find('.device.has-popover.selected'),
            activeDeviceID = selectedDevice.size() ? selectedDevice.data('usbid') : 0,
            emergencyStick = swc.models.InternetBackupStick,
            supportedUsbDevices = _.keys(swc.settings.application.get('supportedUsbDevices'));

        // Check if any device has popover on it
        if (selectedDevice.size()) {
            SWCElements.popovers.closeAll();
        }

        // Remove old devices:
        list.find('.device').remove();

        // Check if devices exists:
        if (!usbDevices.length && !emergencyStick.get('isPresent') && swc.models.USBHostDeviceCollection.size() === 0) {
            list.find('.no-devices').show();
        } else {
            list.find('.no-devices').hide();
        }

        // Check if emergency stick is active:
        if (emergencyStick.get('isPresent')) {
            //show nothing when technology is unknown
            var tech=emergencyStick.get('technology');
            if(tech==="none"){
               tech="";
            }
            var device = _.extend(emergencyStick.toJSON(), {
                status: '3G-' + emergencyStick.get('status'),
                isConnected: emergencyStick.get('isConnected'),
                isStandBy: (emergencyStick.get('status') === 'Disconnected'),
                cloudServicesState: swc.models.СloudServicesStatus.get('isEnabled'),
                deviceName: this.getInternetBackupStickName(),
                technology: tech,
                id: 'internet-backup-stick'
            });

            list.append(
                $.tmpl(swc.Templates.get("components:device-emergency").get('content'), {
                    device: device,

                    popoverEnable: true,
                    popoverTemplateID: 'overview:popovers:device-emergency',
                    popoverTemplateData: JSON.stringify(_.extend(device, { cloudServicesState: false })),
                    popoverContainer: '.current-page .overview',
                    localeStrings: swc.models.Locale.getLocaleStrings(),
                    localeString: getTranslationStringsjQuery,
                    formatDate: swc.models.Locale.formatDate
                })
            );
        } else {
            if(swc.models.USBHostDeviceCollection.size() > 0) {
                var dev = {type: 'usb-disk-undefined'};
                list.append(
                    $.tmpl(template, {
                        device: dev,
                        deviceIsSupported: false,
                        popoverEnable: true,
                        popoverTemplateID: 'overview:popovers:device-storage-usb-device-unsupported',
                        popoverTemplateData: JSON.stringify(dev),
                        popoverContainer: '.current-page .overview',
                        localeStrings: swc.models.Locale.getLocaleStrings(),
                        localeString: getTranslationStringsjQuery,
                        formatDate: swc.models.Locale.formatDate
                    })
                );
            }
        }

        // Append devices to the list:
        $.each(usbDevices, function(key, device) {
            var suffix = '',
                deviceIsSupported = ($.inArray(device.type.toLowerCase(), supportedUsbDevices) !== -1);

                
            switch(device.type) {
                case 'usb-disk-undefined':
                    if(device.fsType === null) {
                        suffix = "-usb-device-unsupported";
                    } else {
                        suffix = "-usb-disc-unsupported";
                    }
                    break;
                case 'sd-card-undefined':
                    suffix = "-sd-card-unsupported";
                    break;
            }

            // Extend device object with new properties:
            device = _.extend(device, {
                cloudServicesState: swc.models.СloudServicesStatus.get('isEnabled')
            });

            list.append(
                $.tmpl(template, {
                    device: device,
                    deviceIsSupported: deviceIsSupported,
                    popoverEnable: true,
                    popoverTemplateID: 'overview:popovers:device-storage' + suffix,
                    popoverTemplateData: JSON.stringify(device),
                    popoverContainer: '.current-page .overview',
                    localeStrings: swc.models.Locale.getLocaleStrings(),
                    localeString: getTranslationStringsjQuery,
                    formatDate: swc.models.Locale.formatDate
                })
            );
        });

        // Reopen popover for selected device:
        if (activeDeviceID) {
            SWCElements.popovers.open(
                $('.device.has-popover[data-usbid="' + activeDeviceID + '"]').find('.icon')
            );
        }
        
        $('.device .name').trunk8(this.trunk8settings);
    },

    renderDECTDevices: function() {
        var dectDevices = swc.models.Telephony.get('phones').where({deviceType:'dect'}),
            template = swc.Templates.get("components:device-dect").get('content'),
            list = this.$el.find('.devices-list .dect-devices'),
            selectedDevice = list.find('.device.has-popover.selected'),
            activeDeviceID = selectedDevice.size() ? selectedDevice.data('dectid') : 0;

        // Check if any device has popover on it
        if (selectedDevice.size()) {
            SWCElements.popovers.closeAll();
        }

        // Remove old devices:
        list.find('.device').remove();

        // Check if devices exists and DECT is On:
        if (!dectDevices.length || !swc.models.Telephony.get('dectStatus')) {
            list.find('.no-devices').show();
            return;
        } else {
            list.find('.no-devices').hide();
        }

        // Append devices to the list:
        $.each(dectDevices, function(key, device) {
            var data = {
                id: device.get('id'),
                icon: 'dect',
                name: device.get('name'),
                line: device.get('line'),
                nameDisplay: device.get('name'), // discussed with Marcel, no need to make lowercase
                phoneNumber: device.get('externalNumber'),
                internalNumber: device.get('directoryNumber'),
                outgoingTrunkLine: device.get('outgoingTrunkLine')
            };

            list.append(
                $.tmpl(template, {
                    device: data,
                    popoverEnable: true,
                    popoverTemplateID: 'overview:popovers:device-dect',
                    popoverTemplateData: JSON.stringify(data),
                    popoverContainer: '.current-page .overview',
                    localeStrings: swc.models.Locale.getLocaleStrings(),
                    localeString: getTranslationStringsjQuery,
                    formatDate: swc.models.Locale.formatDate
                })
            );
        });

        // Reopen popover for selected device:
        if (activeDeviceID) {
            SWCElements.popovers.open(
                $('.device.has-popover[data-dectid="' + activeDeviceID + '"]').find('.icon')
            );
        }
        
        $('.device .name').trunk8(this.trunk8settings);
    },

    setWiFiState: function(e) {
        var self = this,
            switcher = $(e.target).hasClass('.swc-switcher') ? $(e.target) : $(e.target).closest('.swc-switcher'),
            state = switcher.hasClass('enable') ? true : false;

        // Set small delay on switcher change for more user-friendly behaviour
        setTimeout(function() {

            // Show saving page state:
            self.showPageLoading("Updating WiFi state");

            // Send requests to update wifi status on the device:
            $.when(
                    swc.models.Wireless.setGlobalStatus(state)
                ).done(function() {
                    self.render();
                });
        }, 200);
    },

    setCentralStorageState: function(e) {
        var self = this,
            switcher = $(e.target).hasClass('.swc-switcher') ? $(e.target) : $(e.target).closest('.swc-switcher'),
            state = switcher.hasClass('enable') ? true : false;

        // Set small delay on switcher change for more user-friendly behaviour
        setTimeout(function() {

            // Show saving page state:
            self.showPageLoading("Updating Central Storage state");

            // Send requests to update Central Storage status on the device:
            $.when(
                    swc.models.CentralStorage.setGlobalStatus(state)
                ).done(function() {
                    self.render();
                });
        }, 200);
    },

    setDECTState: function(e) {
        var self = this,
            switcher = $(e.target).hasClass('.swc-switcher') ? $(e.target) : $(e.target).closest('.swc-switcher'),
            state = switcher.hasClass('enable') ? true : false;

        // Set small delay on switcher change for more user-friendly behaviour
        setTimeout(function() {

            // Show saving page state:
            self.showPageLoading("Updating Phones and Dect state");

            // Update Telephony model Dect status:
            swc.models.Telephony.set('dectStatus', state);

            // Send requests to update Telephony status on the device:
            $.when(
                    swc.models.Telephony.setDectStatus()
                ).done(function() {
                    self.render();
                });
        }, 200);
    },

    editDevice: function(e) {
        var self = this,
            element = $(e.target),
            type = element.data('type'),
            macAddress = element.data('macaddress'),
            device = swc.models.NetworkDevices.getDevice(macAddress);

        this.customizeDevice({
            device: device,
            type: type,

            onApply: function(options) {
                self.showPageLoading("Updating device information");

                $.when(swc.models.NetworkDevices.customizeDevice(options)).done(function() {
                    $.when(swc.models.NetworkDevices.sync()).done(function() {
                        self.updateNetworkState();
                        self.renderNetworkDevices();
                        self.stopPageLoading();
                    });

                    SWCElements.popovers.open(
                        $('.device[data-macaddress="' + options.deviceMac + '"]').find('.icon')
                    );
                });
            }
        });
    },

    editDect: function(e) {
        var self = this,
            element = $(e.target),
            name = element.data('name'),
            line = element.data('line'),
            device = {
                type: 'dect',
                name: name,
                line: line
            };

        this.customizeDevice({
            device: device,
            type: 'dect',
            template: 'overview:modal-windows:customize-dect',

            onApply: function(options) {

                self.showPageLoading("Updating device information");

                swc.models.Telephony.get('phones').get(line).set({
                    'name': options.name,
                    'is_changed': true
                });

                $.when(swc.models.Telephony.phoneEdit()).done(function(){
                    $.when(swc.models.Telephony.getPhones()).done(function() {
                        self.renderDECTDevices();
                        self.stopPageLoading();
                    });

                    SWCElements.popovers.open(
                        $('.device.has-popover[data-dectid="' + line + '"]').find('.icon')
                    );
                });
            }
        });
    },

    secureRemoveUSB: function(e) {
        var self = this,
            button = $(e.target).hasClass('.button') ? $(e.target) : $(e.target).closest('.button'),
            id = button.data('id'),
            deviceName = swc.models.StorageDevices.get(id).get('label');

        // Set small delay on switcher change for more user-friendly behaviour
        setTimeout(function() {

            // Show saving page state:
            self.showPageLoading("Unmounting usb device");

            // Send requests to update Telephony status on the device:
            $.when(
                    swc.models.CentralStorage.removeUSBDevice(id)
                ).done(function() {
                    self.stopPageLoading();
                    self.unmountConfirm(deviceName);
                });
        }, 200);
    },
    
    unmountConfirm: function(deviceName) {
        var self = this;

        SWCElements.modalWindow.hide();
        SWCElements.modalWindow.show({
            templateID: 'storage:settings:modal-windows:unmount-device-complete',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate,
                deviceName: deviceName
            },
            className: 'unmount-device',
            onApply: function(){
                SWCElements.modalWindow.hide();
                self.render();
            }
        });
    },

    loginToCloudService: function() {
        // Login to selected cloud service:
        swc.models.CloudServices.checkCredentials();
    },
    
    redirectToSpeedCheck: function() {
        swc.router.navigate('speedcheck', { trigger: false, skipUnsavedChanges: true});
    }
});
;swc.constructors.ParentalView = swc.base.PageView.extend({

    className: 'parental-control',

    models: [
        'NetworkDevices',
        'SchedulerDeviceCollection',
        'SchedulerDeviceManager',
        'EventDispatcher'
    ],

    disallowedDevices: ['starlink'],

    events: {
        'swc-switcher:change .parental-activate': 'switchParental',
        'expandable:open .expandable-list': 'expandableOpen',
        'expandable:close .expandable-list': 'expandableClose',
        'hover .expandable-list-item:not(.expanded) .device-name': 'highlightIcon'
    },

    eventActions: [
      {
        event: 'wifi_device_updated',
        action: 'networkDevicesUpdated'
      },
      {
        event: 'eth_device_updated',
        action: 'networkDevicesUpdated'
      }
    ],

    listenerInterval: 10,

    listenerEnabled: false,

    listenerPages: [
        'parental'
    ],
    refreshInterval: null,
    refreshIntervalCalls: 0,
    refreshIntervalDelay: 2000,

    changedScheduler: false,

    setTemplateData: function() {
        var self = this,
            key = 1,
            thirtyDays = 30 * 24 * 3600000,
            devices = [];

        self.changedScheduler = false;

        /*
          Make matching between device collection and schedule collection
          After invoking of this method the swc.models.SchedulerDeviceCollection is adjusted
        */
        swc.models.SchedulerDeviceManager.applySchedulesToDevices(
            swc.models.NetworkDevices,
            swc.models.SchedulerDeviceCollection,
            true
        );

        swc.models.SchedulerDeviceCollection.each(function(schedule) {
            var device = swc.models.NetworkDevices.findWhere({ 'mac': schedule.get('id') });

            if (key < 200 && device) {
                var deviceName = device.get('name'),
                    isActive = device.get('status'),
                    isHidden = device.get('isHidden'),
                    connectionType = device.get('interfaceType'),
                    icon = device.get('icon'),
                    mac = device.get('mac'),
                    isEnabled = schedule.get('enable') ? true : false,
                    status = "disabled",

                    // we use moment.js here for date conversion because IE sometimes cannot parse
                    // the format that is coming from the serverside
                    lastConnection = moment(device.get('lastConnection')).unix() * 1000,
                    now = new Date().getTime();

                if ((isEnabled || (now - lastConnection < thirtyDays) ||
                    // protection against factory reset w/o internet connection
                    device.get('lastConnection').indexOf("0001-01-01") !== -1) && isHidden===false ) {

                    if(isActive){
                      status = "";
                    } else {
                      status = "disconnected";
                    }

                    devices.push({
                        mac: mac,
                        deviceName: deviceName,
                        icon: icon,
                        conType: connectionType,
                        key: key,
                        isConnected: isActive,
                        isEnabled: isEnabled,
                        statusClass: status
                    });
                    key++;
                }
            }
        });
    
        devices = _.sortBy(devices, function(device){ return device.isConnected === true ? 0 : 1; });

        self.templateData = {
            devices: devices
        };
    },
    networkDevicesUpdated: function(event) {
      swc.Utils.updateDevicesByModelName("NetworkDevices", event);
    },

    activateButtons: function(){
        this.$('.save-changes').removeClass('disabled');
        this.$('.cancel-changes').removeClass('disabled');
    },

    deactivateButtons: function(){
        this.$('.save-changes').addClass('disabled');
        this.$('.cancel-changes').addClass('disabled');
    },

    checkModification: function() {
        return this.changedScheduler;
    },

    highlightIcon: function(event){
        var $icon = $(event.target).closest('.device-name').find('.device.small');

        if (event.type==='mouseenter') {
            $icon.addClass('selected');
        } else {
            $icon.removeClass('selected');
        }
    },
    
    /**
     * Hook onListenerComplete: is fired as final part
     * of listener and performs only necessary render
     * actions instead of whole render process
     */
    onListenerComplete: function() {
        var self = this;
        swc.models.SchedulerDeviceManager.applySchedulesToDevices(
            swc.models.NetworkDevices,
            swc.models.SchedulerDeviceCollection,
            true
        );

        self.refreshView();
    },
    
    refreshView: function() {
        var self = this;

        self.refreshList();
    },

    openExpandableItem: function() {
        /*
         * if no current opened item
         * then first item should be opened
         */
        if (!this.openedItem) {
            this.openedItem = swc.models.SchedulerDeviceCollection.at(0).get('id');
        }

        if (this.openedItem !== 'default' && this.openedItem !== 'closed') {
            this.$el.find('.expandable-list').trigger('expandable:swc-open', this.openedItem);
        }
    },

    expandableOpen: function(e, value) {
        this.openedItem = value;
        this.refreshList();
    },

    expandableClose: function(e, value) {
        if (this.openedItem === value) {
            this.openedItem = "";
        }
    },

    save: function() {
        var self = this,
            deferred =  new $.Deferred();

        $.when(swc.models.SchedulerDeviceCollection.save())
            .done(function() {
                self.changedScheduler = false;
                deferred.resolve();
            }).fail(function() {
                deferred.reject();
            });

        return deferred.promise();
    },

    switchParental: function(e, value) {
        var element = $(e.target),
            self = this,
            deviceId = element.closest('.expandable-list-item').attr('data-key'),
            model = swc.models.SchedulerDeviceCollection.get(deviceId);

        model.enable(!!value);

        //this.showPageLoading("Saving page data..");
        
        setTimeout(function() {
            $.when(model.sync('update'))
                .done(function() {
                    self.changedScheduler = false;
                    self.refreshView();
                });
        }, 100);
    },

    passEvent: function(event, eventObject) {
      swc.constructors.EventController(this.eventActions)
                      .pass.call(this, event, eventObject);
    },
    renderComplete: function() {
        var self = this;

        if(!_.isUndefined(swc.router)) {
            swc.router.isLoadingModule = false;
        }

        self.eventDispatcher = swc.models.EventDispatcher;
        self.eventDispatcher.observ("parental_controll",self);
        //block autosync on models that can handle events
        swc.models.Wireless.autoSync=false;
        swc.models.NetworkDevices.autoSync=false;

        swc.models.NetworkDevices.on("devices-loaded",function(){
            self.refreshIntervalCalls++;
            if(self.refreshInterval === null){
               self.refreshInterval = setInterval(function(){
                   var modificationDone = false;
                   if(self.changedScheduler === true) {
                    modificationDone = true;
                   } else {
                    modificationDone = false;
                   }

                   if(!_.isUndefined(swc.router) && swc.router.isLoadingModule === true) {
                     return;
                   }
                   
                   self.setTemplateData();
                   self.getTemplateContent();
                   
                   self.refreshIntervalCalls = 0;
                   self.refreshList();
                   self.openExpandableItem();
                   //clear refresh interval
                   clearInterval(self.refreshInterval);
                   self.refreshInterval=null;
                   if (modificationDone === true) {
                    self.activateButtons();
                   } else {
                    self.deactivateButtons();
                   }
               }, self.refreshIntervalDelay);
            }
        });

        self.refreshList();
        self.openExpandableItem();
    },
    
    refreshList: function() {
        var self = this;

        swc.constructors.dispatcher.off('scheduler:change');
        swc.constructors.dispatcher.on('scheduler:change', function() {
            self.changedScheduler = true;
            self.$('.save-changes, .cancel-changes').removeClass('disabled');
        });

        this.$('.expandable-devices .scheduler-container').each(function () {

            var container = $(this),
                component = container.closest('.expandable-list-item'),
                deviceId = component.attr('data-key'),
                deviceModel = swc.models.SchedulerDeviceCollection.findWhere({ 'id': deviceId });

            if((typeof self.openedItem === 'undefined') || (deviceId === self.openedItem)) {
                if (!_.isUndefined(deviceModel)) {
                    // turn on automatically
                    var componentActive = component.find('.parental-activate'),
                        pixelSchedule = deviceModel.get('pixelSchedule'),
                        enabled = deviceModel.get('enable');
                        
                    if(componentActive.data('default-value') !== enabled) {
                        componentActive.trigger('swc-switcher:swc-change', enabled);
                    }
    
                    var template = new swc.constructors.SchedulerLiner({
                        'isDisabled': !enabled,
                        'scaleWidth' : deviceModel.widgetWidthInPixels,
                        'diapazones' : pixelSchedule,
                        'model': deviceModel,
                        'schedulerType': 'device'
                    });
                    container.html(template.el);
                }
            }
        });
    }
});
;swc.constructors.PowerView = swc.base.PageView.extend({

    className: 'power-settings',

    models: [
        'Wireless',
        'SchedulerWlan',
        'SchedulerAp',
        'Telephony',
        'apServiceState',
        'SchedulerPowerSavingManager'
    ],

    disallowedDevices: ['starlink'],

    events: {
        'swc-checkbox:change .wifi .enable-service-schedule':'changeWifiScheduler',
        'swc-checkbox:change .storage .enable-service-schedule':'changeStorageScheduler',
        'swc-checkbox:change .enable-eco-mode':'enableEcoMode',

        'expandable:open .expandable-list': 'expandableOpen',

        'swc-switcher:change .swc-switcher.enable-wifi-service': 'changeWifiState',
        'swc-switcher:change .swc-switcher.enable-storage-service': 'changeAPState',
        'swc-switcher:change .swc-switcher.enable-dect-service': 'changeDectState'
    },
    
    // every 10 minutes
    listenerInterval: 10 * 60,

    listenerEnabled: true,

    listenerPages: [
        'power'
    ],

    changedScheduler: false,

    changeWifiScheduler: function(e, value) {
        swc.models.SchedulerWlan.enable(value);
        this.renderComplete();
        this.setButtonsState();
    },

    changeStorageScheduler: function(e, value) {
        swc.models.SchedulerAp.enable(value);
        this.renderComplete();
        this.setButtonsState();
    },

    enableEcoMode: function(e, value) {
        swc.models.Telephony.set('dectEco', value);
        this.renderComplete();
        this.setButtonsState();
    },

    changeWifiState: function(e, value){
        var self = this;
        self.changedScheduler = false;
        // Show saving page state:
        this.showPageLoading(getTranslationStrings("Saving page data.."));

        // Send requests to update wifi status on the device:
        $.when(swc.models.Wireless.setGlobalStatus(value))
            .done(function() {
                $.when(swc.models.Wireless.sync()).done(function() {
                    self.render();
                });
            });
    },

    changeAPState: function(e, value){
        var self = this;
        self.changedScheduler = false;
        // Show saving page state:
        this.showPageLoading(getTranslationStrings("Saving page data.."));

        swc.models.apServiceState.set('status', value);

        $.when(swc.models.apServiceState.sync('update')).done(function() {
            self.render();
        });
    },

    changeDectState: function(e, value){
        var self = this;
        self.changedScheduler = false;
        // Show saving page state:
        this.showPageLoading(getTranslationStrings("Saving page data.."));

        swc.models.Telephony.set('dectStatus', value);

        $.when(swc.models.Telephony.setDectStatus()).done(function() {
            self.render();
        });

    },

    expandableOpen: function(e, value){
        this.openedItem = value;
    },

    onCancel: function(){
        this.changedScheduler = false;
        this.render();
    },

    setButtonsState: function(){
        if (this.changedScheduler || !this.pageCheckDefaultValues()) {
            this.$('.save-success').hide();
            this.$('.buttons-container-message a').removeClass('disabled');
        } else {
            this.$('.buttons-container-message a').addClass('disabled');
        }
    },

    showSaveSuccess: function() {
        this.changedScheduler = false;
        swc.base.PageView.prototype.showSaveSuccess.apply(this, arguments);
    },

    save: function() {
        var self = this,
            deferred = new $.Deferred(),
            wlanSchedulerIsEmpty = swc.models.SchedulerWlan.isEmpty(),
            apSchedulerIsEmpty = swc.models.SchedulerAp.isEmpty();

        self.changedScheduler = false;

        if ((wlanSchedulerIsEmpty && swc.models.SchedulerWlan.get('enable')) || (apSchedulerIsEmpty && swc.models.SchedulerAp.get('enable'))) {
            this.stopPageLoading();
            if (wlanSchedulerIsEmpty && apSchedulerIsEmpty) {
                return self.showModalEmptyWlanAndAP();
            } else if (wlanSchedulerIsEmpty) {
                return self.showModalEmptyWlan();
            } else if (apSchedulerIsEmpty) {
                return self.showModalEmptyAp();
            }
        } else {
            $.when(swc.models.SchedulerPowerSavingManager.saveAll())
                .done(function() {
                    deferred.resolve();
                })
                .fail(function() {
                    deferred.reject();
                });
        }

        return deferred.promise();
    },

    showModalEmptyWlan: function(){
        var self = this,
            deferred = new $.Deferred();

        SWCElements.modalWindow.show({
            templateID: 'power:wifi:modal',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            },
            className: 'storage-schedule',
            onCancel: function() {
                SWCElements.modalWindow.hide();
                deferred.reject();
            },
            onApply: function() {
                SWCElements.modalWindow.hide();
                self.showPageLoading(getTranslationStrings('Saving page data..'));

                $.when(swc.models.SchedulerPowerSavingManager.saveAll())
                    .done(function() {
                        deferred.resolve();
                    })
                    .fail(function() {
                        deferred.reject();
                    });
            }
        });

        return deferred.promise();
    },

    showModalEmptyAp: function(){
        var self = this,
            deferred = new $.Deferred();

        SWCElements.modalWindow.show({
            templateID: 'power:ap:modal',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            },
            className: 'storage-schedule',
            onCancel: function() {
                SWCElements.modalWindow.hide();
                deferred.reject();
            },
            onApply: function() {
                SWCElements.modalWindow.hide();
                self.showPageLoading(getTranslationStrings('Saving page data..'));

                $.when(swc.models.SchedulerPowerSavingManager.saveAll())
                    .done(function() {
                        deferred.resolve();
                    })
                    .fail(function() {
                        deferred.reject();
                    });
            }
        });

        return deferred.promise();
    },

    showModalEmptyWlanAndAP: function(){
        var self = this,
            deferred = new $.Deferred();

        SWCElements.modalWindow.show({
            templateID: 'power:both:modal',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            },
            className: 'storage-schedule',
            onCancel: function() {
                SWCElements.modalWindow.hide();
                deferred.reject();
            },
            onApply: function() {
                SWCElements.modalWindow.hide();
                self.showPageLoading(getTranslationStrings('Saving page data..'));

                $.when(swc.models.SchedulerPowerSavingManager.saveAll())
                    .done(function() {
                        deferred.resolve();
                    })
                    .fail(function() {
                        deferred.reject();
                    });
            }
        });

        return deferred.promise();
    },

    setTemplateData: function(){
        var wifiState,
            apState,
            dectState,
            dectEcoState;

        wifiState = (swc.models.Wireless.getParameter("status", "status") === 'on');
        apState = swc.models.apServiceState.isEnabled();

        dectState = swc.models.Telephony.get('dectStatus');
        dectEcoState = swc.models.Telephony.get('dectEco');

        this.templateData = {
            wifiState: wifiState,
            wifiSchedule: swc.models.SchedulerWlan.get('enable'),
            apState: apState,
            apSchedule: swc.models.SchedulerAp.get('enable'),
            dectActivated: dectState,
            dectEco: dectEcoState,
            GlobalEnable: swc.models.Application.get('GlobalEnable')
        };
    },

    /**
     * Set text for "Power saving status" next to the leafs for schedules
     */
    setScheduleStateText: function() {
        var wifiState = (swc.models.Wireless.getParameter("status", "status") === 'on'),
            apState = swc.models.apServiceState.isEnabled(),
    
            dectState = swc.models.Telephony.get('dectStatus'),
            dectEcoState = swc.models.Telephony.get('dectEco'),
    
            isEnabledWifiScheduler = swc.models.SchedulerWlan.get('enable'),
            isEnableApScheduler = swc.models.SchedulerAp.get('enable'),
    
            setTextInBlock = function(enabled, state, block) {
                if (enabled) {
                    if (state) {
                        block.find('.text .activated-schedule').show();
                    } else {
                        block.find('.text .activated-disable').show();
                    }
                } else {
                    block.find('.leaf').addClass('disabled');
                    block.find('.text .not-activated').show();
                }
            };

        $('.wifi-statuses, .dect-statuses, .ap-statuses').find('.text span').hide();
        $('.leaf').removeClass('disabled');

        setTextInBlock(isEnabledWifiScheduler, wifiState, this.$('.wifi-statuses'));
        setTextInBlock(isEnableApScheduler, apState, this.$('.ap-statuses'));
        setTextInBlock(dectEcoState, dectState, this.$('.dect-statuses'));
    },

    renderComplete: function(){
        var self = this,
            wifiSchedulerModel = swc.models.SchedulerWlan,
            apSchedulerModel = swc.models.SchedulerAp;

        this.setTemplateData();

        this.setScheduleStateText();

        swc.constructors.dispatcher.off('scheduler:change');
        swc.constructors.dispatcher.on('scheduler:change', function(){
            self.changedScheduler = true;
            self.setButtonsState();
        });

        var wifiBlock = this.$('.expandable-list-item.wifi'),
            storageBlock = this.$('.expandable-list-item.storage');

        var wifiScheduler = new swc.constructors.SchedulerLiner({
            isDisabled: !self.templateData.wifiSchedule,
            scaleWidth: swc.models.SchedulerWlan.widgetWidthInPixels,
            diapazones: wifiSchedulerModel.get('pixelSchedule'),
            model: wifiSchedulerModel,
            schedulerType: 'wifi'
        });

        var storageScheduler = new swc.constructors.SchedulerLiner({
            isDisabled: !self.templateData.apSchedule,
            scaleWidth: swc.models.SchedulerWlan.widgetWidthInPixels,
            diapazones: apSchedulerModel.get('pixelSchedule'),
            model: apSchedulerModel,
            schedulerType: 'storage'
        });

        wifiBlock.find('.scheduler-container').empty().append(wifiScheduler.el);
        storageBlock.find('.scheduler-container').empty().append(storageScheduler.el);

        if (!this.openedItem) {
            this.openedItem = this.$('.expandable-list-item').eq(0).attr('data-key');
        }

        this.$(".expandable-list").trigger('expandable:swc-open', this.openedItem);

        this.checkPermissions();

        if (this.showSuccessMessage) {
            this.showSuccessMessage = false;
            this.$('.save-success').show();
        }
    }

});
;swc.constructors.SystemDiagnosticsView = swc.base.PageView.extend({

    className: 'system-diagnostics',

    models: ['Network', 'System', 'SwisscomDyndns'],
    
    renderComplete: function() {
        var APVersion = swc.models.System.attributes.APVersion || '0.0.0',
            gatewayTab = $('.tab-gateway'),
            speedTab = $('.tab-speedcheck');
            
        // find major version
        APVersion = parseInt(APVersion.split('.')[0], 10);
        if(APVersion < 5) {
            gatewayTab.addClass('last');
            speedTab.hide();
        } else {
            // gatewayTab.removeClass('last');
            // speedTab.show();
        }
    }
});
;swc.constructors.SystemDiagnosticsGatewayView = swc.base.TabView.extend({

    className: 'gateway',

    models: ['System', 'Network', 'FirewallStatus'],

    setTemplateData: function() {
        var systemData = swc.models.System.attributes,
            wanData = swc.models.Network.get('status').attributes;

        this.templateData = {
            'build': swc.settings.application.get('build'),
            'manufacturer': systemData.manufacturer,
            'serial': systemData.serialNumber,
            'model': systemData.model,
            'localIP': swc.models.Network.get('ip-settings').get('LocalIPAddress'),
            'localIPv6': swc.Utils.formatIPv6Address(wanData.IPv6Address),
            'storageIP': wanData.StorageIPAddress,
            'mac': wanData.MacAddress,
            'NPFirmware': systemData.NPVersion || '0.0.0',
            'APFirmware': systemData.APVersion || '0.0.0',
            'IPv6Status': swc.models.FirewallStatus.get('state') ? swc.models.FirewallStatus.get('state').Enable : false,
            'GlobalEnable': swc.models.Application.get('GlobalEnable')
        };
    },

    renderComplete: function(){
        var element = $(this.el),
            systemData = swc.models.System.attributes;

            //
            //
            //
            //
        /**
         * This is needed for handling translation of plural and singular words for Day(s), Hour(s), Minute(s)
         *
         * Generates object with data about days hours and minutes
         * @example
         * <code>
         * {
         *      d: { translation: "d0", value: 0 },
         *      h: { translation: "h1", value: 23 },
         *      m: { translation: "m1", value: 1 }
         * }
         * </code>
         *
         * @type {Object}
         */
        var upTime = _.object(_.map(systemData.upTime, function(value, key) {
            var item, translation;

            if (value === 0) {
                translation = key + '0';
            } else if (value === 1) {
                translation = key + '1';
            } else {
                translation = key;
            }

            item = [
                key, {
                    value: value,
                    translation: translation
                }
            ];

            return item;
        }));

        element.find('.uptime').html($.tmpl(swc.Templates.get('system:diagnostics:upTimeTmpl').get('content'), {
            localeStrings: swc.models.Locale.getLocaleStrings("system"),
            localeString: getTranslationStringsjQuery,
            formatDate: swc.models.Locale.formatDate,
            upTime: upTime
        }));
    }
});
;swc.constructors.GatewayPasswordModalView = Backbone.View.extend({
    className: 'modal gateway-password',

    tagName: 'div',

    hidden: false,

    events: {
        'click .logout': 'startLogout'
    },

    startLogout: function(){
        $(this.el).modal('hide');
        this.remove();
        swc.models.Login.processLogout({ action: 'user-logout' });
    },

    initialize: function(){
        this.template = swc.Templates.get('system:settings:gateway:modal');
        this.render();
    },

    render: function(){
        var self = this, element = $(this.el);
        element.html($.tmpl(self.template.get('content'), {
            localeStrings: swc.models.Locale.getLocaleStrings("system"),
            localeString: getTranslationStringsjQuery,
            formatDate: swc.models.Locale.formatDate
        }));
    }

});;swc.constructors.SystemDiagnosticsOverviewView = swc.base.TabView.extend({

    className: 'overview',

    events: {
        'click .export-log': 'exportLogs'
    },

    models: ['Network', 'System', 'DeviceMeasurements', 'DynDNS', 'DynDNSProviderCollection', 'SwisscomDyndns', 'InternetBackupStick'],

    exportLogs: function(){
        swc.models.System.getLogs();
    },
    
    getConnectionType: function() {
        var result = '';
        
        if(swc.models.InternetBackupStick.get('isConnected')) {
            result = 'GSM';
        } else {
            switch(swc.models.Network.getParameter('ConnectionType',  'status')) {
                case 'ethernet':
                    result = 'Ethernet';
                    break;
                case 'vdsl':
                    result = 'VDSL';
                    break;
                case 'dsl':
                    result = 'ADSL';
                    break;
            }
        }
        
        return result;
    },

    setTemplateData: function(){
        var dnsProvider = swc.models.DynDNSProviderCollection.at(0),
            dnsProviderData,
            connectionType = this.getConnectionType(),
            network = swc.models.Network.get('status').toJSON(),
            swisscomDyndns = swc.models.SwisscomDyndns;

        if (!_.isEmpty(dnsProvider)) {
            var lastUpdate = dnsProvider.get('last_update');

            if (lastUpdate && lastUpdate !== "0001-01-01T00:00:00Z") {
                moment.lang(swc.models.Locale.locale);
                lastUpdate = moment(lastUpdate).zone(lastUpdate).format("DD. MMM YYYY, HH:mm");
            } else {
                lastUpdate = '';
            }

            dnsProviderData = {
                service: dnsProvider.get('service').capitalize(),
                status: dnsProvider.get('status'),
                lastUpdate: lastUpdate
            };
        }
        
        this.templateData = {
            'netStatus': network,
            'GlobalEnable': swc.models.Application.get('GlobalEnable'),
            'dyndnsStatus': swc.models.DynDNS.get('enable'),
            'dnsProviderData': dnsProviderData,
            'connectionType' : connectionType,
            'swisscomDyndns' : swisscomDyndns
        };
        _.extend(
            this.templateData,
            swc.models.DeviceMeasurements.getParams()
        );
    }
});
;swc.constructors.SystemDisplayView = swc.base.PageView.extend({

    className: 'system-display',

    models: ['Screen'],

    events: {
        'swc-dropdown:change .time-selection': 'changeSettings',
        'swc-dropdown:change .language-selection': 'changeSettings',
        'swc-checkbox:change .show-wifi-password': 'changeSettings',
        'swc-checkbox:change .show-wifi-guest-password': 'changeSettings'
    },

    setTemplateData: function() {
        this.templateData = {
            time: swc.models.Screen.get('brightTime'),
            lang: swc.models.Screen.get('lang'),
            show: swc.models.Screen.get('showPass'),
            showGuest: swc.models.Screen.get('showGuestPass')
        };
    },

    activateButtons: function() {
        if (this.pageCheckDefaultValues()) {
            this.$('.buttons-container-message a').addClass('disabled');
        } else {
            this.$('.buttons-container-message a').removeClass('disabled');
        }
    },

    changeSettings: function() {
        swc.models.Screen.set({
            'brightTime': this.$('.swc-dropdown.time-selection').data('value'),
            'lang': this.$('.swc-dropdown.language-selection').data('value'),
            'showPass': this.$('.swc-checkbox.show-wifi-password').data('value'),
            'showGuestPass': this.$('.swc-checkbox.show-wifi-guest-password').data('value')
        });

        this.activateButtons();
    },

    preRender: function() {
        if(swc.Utils.DeviceType !== 'default') {
            window.history.back();
        }
    },

    save: function() {
        var deferred = new $.Deferred();

        $.when(swc.models.Screen.setParameters())
            .done(function() {
                deferred.resolve();
            })
            .fail(function (message) {
                deferred.reject(message);
            });

        return deferred.promise();
    },

    showSaveSuccess: function() {
        swc.base.PageView.prototype.showSaveSuccess.apply(this, arguments);
        // if(type === 'error' && message === 'connection') {
        //     this.$el.find('.system-display-error').show();
        //     this.$el.find('.system-display-error .text').show();
        //     this.$el.find('.system-display-error .connection-message').show();
        // }
    },

    renderComplete: function() {
        var timeEl = this.$('.swc-dropdown.time-selection'),
            langEl = this.$('.swc-dropdown.language-selection'),
            showEl = this.$('.swc-checkbox.show-wifi-password'),
            showGuestEl = this.$('.swc-checkbox.show-wifi-guest-password');

        var localeSec = swc.models.Locale.getLocaleStrings("system").sec;

        timeEl.data('options', {
            '30': {
                'value': '30',
                'name': '30 '+localeSec
            },
            '60': {
                'value': '60',
                'name': '60 '+localeSec
            },
            '120': {
                'value': '120',
                'name': '120 '+localeSec
            }
        });

        langEl.data('options', {
            'DE': {
                'value': 'DE',
                'name': 'Deutsch'
            },
            'FR': {
                'value': 'FR',
                'name': 'Français'
            },
            'IT': {
                'value': 'IT',
                'name': 'Italiano'
            },
            'EN': {
                'value': 'EN',
                'name': 'English'
            }
        });

        timeEl.trigger('swc-dropdown:swc-change', swc.models.Screen.get('brightTime'));
        langEl.trigger('swc-dropdown:swc-change', swc.models.Screen.get('lang'));
        showEl.trigger('swc-checkbox:swc-change', swc.models.Screen.get('showPass'));
        showGuestEl.trigger('swc-checkbox:swc-change', swc.models.Screen.get('showGuestPass'));
    }

});
;swc.constructors.SystemSettingsResetView = swc.base.TabView.extend({

    className: 'reset',

    models: ['System', 'Telephony', 'apServiceState', 'apServiceLoadingState'],

    events: {
        'click .factory-reset':  'onFactoryReset',
        'click .save-config':    'onSaveConfig',
        'click .restore-config': 'onRestoreConfig'
    },

    listenerInterval: 5,

    listenerPages: [ 'reset' ],
    
    backupPlace: null,

    preRender: function() {
        // We have to load model `apServiceLoadingState` only if AP is enabled
        if (swc.models.apServiceState.isEnabled()) {
            swc.models.apServiceLoadingState = new swc.constructors.apServiceLoadingState();
        }
    },

    doBackup: function() {
                
        if (this.backupPlace === "local") {
            this.doBackupLocal();
        } else {
            this.doBackupAcs();
        }
        this.backupPlace = null;
    
    },
    

    /**
     * Handler when user clicks on backup button:
     *
     * @description:
     *
     *  When user clicks on backup button, he must be shown modal window, where he will choose backup option. If user will
     *  choose `backup to acs network` -> new RPC call will be invoked, else he will go through default procedure of backup
     */
    onSaveConfig: function(e) {
        var self = this;

        e.preventDefault();
        
        // before show modal window the session state should be checked
        swc.models.System.checkSessionState(function() {
            self.showModalSaveConfig();
        });
    },

    /**
     * Show save config modal window
     *
     * @description: show modal window with switcher:
     *  - save configuration in the Swisscom Customer Center (backupPlace = acs)
     *  - Save configuration in the file locally on your computer (backupPlace = local)
     *
     * After pushing button "Apply" the save process starts
     *
     * @localParam backupPlace {String} Place where the configuration
     *             settings will be saved (local | acs)
     */
    showModalSaveConfig: function() {
        var self = this;

        SWCElements.modalWindow.show({
            templateID: 'system:settings:modal-windows:backup-settings',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery,
                status: swc.models.apServiceState.get('status')
            },
            className: 'backup-settings',

            onApply: function() {
                var backupPlace = $('.modalWindow.backup-settings').find('.backup-options').data('value');

                SWCElements.modalWindow.hide();
                self.processSaveConfig(backupPlace);
            }
        });
    },

    /**
     * Start process of data backup according to selected
     * place for saving configuration
     *
     * @param backupPlace {String} local | acs
     */
    processSaveConfig: function(backupPlace) {
        this.backupPlace = backupPlace;
        this.doBackup();
    },

    /**
     * Handler when backup finishes
     *
     * @param status {Boolean} -> status of backup process (true - succes, false - fail)
     */
    onBackupComplete: function(status) {
        var messagesContainer = this.$('.buttons-container-message.save-config-message');

        // Remove loading window:
        swc.views.Application.stopPageLoading();

        // Display message about success / fail saving:
        if (status === true) {
            messagesContainer.find('.save-error').hide();
            messagesContainer.find('.save-success').show();
        } else {
            messagesContainer.find('.save-success').hide();
            messagesContainer.find('.save-error').show();
        }
    },

    /**
     * Create local backup file
     *
     * @returns {*}
     */
    doBackupLocal: function() {
        var self = this,
            deferred = new $.Deferred();

        // Create backup iframe and hide it by default:
        this.backupIframe = $.tmpl(swc.Templates.get('system:settings:backup:iframe').get('content'),
            { context: $.cookie(swc.Utils.getDeviceID() + '/context') }
        );

        // Append it to body:
        $('body').append(this.backupIframe);

        // Display loading message:
        swc.views.Application.showPageLoading('Downloading Backup File', 200000);

        // Call RPC endpoint to create a backup file:
        $.when(swc.models.System.configBackup())
            .done(function() {
                self.onBackupComplete(true);
            })
            .fail(function() {
                self.onBackupComplete(false);
            });

        return deferred.promise();
    },

    /**
     * Save device settings to swisscom ACS network:
     */
    doBackupAcs: function() {
        var self = this;

        // Display loading message:
        swc.views.Application.showPageLoading('Downloading Backup File', 200000);

        // Call RPC endpoint to save settings to Swisscom ACS network:
        $.when(swc.models.System.configBackupACS())
            .done(function() {
                self.onBackupComplete(true);
            })
            .fail(function() {
                self.onBackupComplete(false);
            });
    },

    wrongFileModal: function() {
        var self = this;
        SWCElements.modalWindow.show({
            templateID: 'system:settings:restore:modal:restore-wrongfile',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            },
            className: 'system-restore',
            onCancel: function() {
                self.removeForm();
            }
        });
    },

    confirmModal: function() {
        var self = this;

        SWCElements.modalWindow.show({
            templateID: 'system:settings:restore:modal:restore-confirm',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate,
                filename: self.fileName
            },
            onShow: function(){
                var modal = $('.modalWindow');
                modal.find('.apply-changes').on('click', function () {
                    SWCElements.modalWindow.hide();
                    self.processRestoreConfig();
                    self.form.submit();
                });

            },
            className: 'system-restore',
            onCancel: function() {
                self.removeForm();
            }
        });
    },

    processRestoreConfig: function() {
        var self = this;

        SWCElements.modalWindow.show({
            templateID: 'system:settings:restore:modal:restore-process',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            },
            className: 'system-restore',
            onShow: function(){
                var element = $('.modalWindow'),
                    isSuccess = false,
                    restoreTime = swc.models.System.minExpectedRebootTime,
                    resetRuler;

                element.find('.ruler-block').html($.tmpl(swc.Templates.get('ruler').get('content'), {
                    'className': 'restore-ruler'
                }));

                // NOTE:
                // Don't try to move it into `var` initialization section,
                // because element will be present only if previous LOC executed
                resetRuler = element.find('.restore-ruler');

                // NP throws custom event to document if upload was successful.
                // If we haven't received event for 2 minutes, or event was not successful, throw error
                $(document).off('sah:models:System:state');
                $(document).on('sah:models:System:state', function(e, options){
                    if(options === 'restore-success'){
                        isSuccess = true;
                    } else {
                        setTimeout(function(){
                            resetRuler.trigger('swc-ruler:finish');
                        }, 50);
                    }
                });

                var postRestoreCallback = function() {
                    self.removeForm();
                    // clean local storage with saved language
                    localStorage.removeItem('locale');
                    swc.models.System.doLogout();
                };

                resetRuler.on('swc-ruler:finish', function(){
                    if(isSuccess){
                        if (swc.models.apServiceState.isEnabled()) {
                            // swc.models.System.whenApUp(postRestoreCallback);
                            postRestoreCallback();
                        } else {
                            swc.models.System.whenDeviceUp(postRestoreCallback);
                        }
                    } else {
                        SWCElements.modalWindow.hide();
                        self.errorModal();
                    }
                });

                setTimeout(function(){
                    resetRuler.trigger('swc-ruler:start', {time: restoreTime});
                }, 0);
            }
        });
    },

    errorModal: function() {
        var self = this;
        SWCElements.modalWindow.show({
            templateID: 'system:settings:restore:modal:restore-error',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            },
            className: 'system-restore',
            onCancel: function() {
                self.removeForm();
            }
        });
    },

    removeForm: function(){
        if(this.restoreIframe){
            this.restoreIframe.remove();
        }
        if(this.saveIframe){
            this.saveIframe.remove();
        }
    },

    onRestoreConfig: function() {
        this.restoreConfig();
    },

    restoreConfig: function() {
        var self = this;

        this.restoreIframe = $.tmpl(swc.Templates.get('system:settings:restore:iframe').get('content'), {});
        this.form = $('#restore-form');

        this.restoreIframe.hide();
        $('body').append(this.restoreIframe);

        this.form.find('input[name="context"]').val($.cookie(swc.Utils.getDeviceID() + '/context'));

        this.form.find('#config-restore').on('change', function(){
            var filePath = self.form.find('input[type=file]').val(), fileExtension;

            if(filePath.indexOf('\\') < 0){
                self.fileName = filePath;
            } else {
                self.fileName = filePath.slice(filePath.lastIndexOf("\\")+1, filePath.length);
            }

            fileExtension = self.fileName.slice(self.fileName.lastIndexOf(".")+1, self.fileName.length);

            if(self.fileName && fileExtension === 'cfg') {
                // before show modal window the session state should be checked
                swc.models.System.checkSessionState(function() {
                    self.confirmModal();
                });
            } else {
                self.wrongFileModal();
            }
        });
    },

    /**
     * Event handler: is fired on click button 'Reset to factory settings'
     *
     * @param e
     */
    onFactoryReset: function(e) {
        var self = this;

        e.preventDefault();

        // before show modal window the session state should be checked
        swc.models.System.checkSessionState(function() {
            self.showModalFactoryReset();
        });
    },

    /**
     * Show factory reset modal window
     *
     * @description: show modal window with switcher
     *  - reset with saving DECT pairing
     *  - complete reset
     *
     *  After pushing button "Apply" this modal window closes and
     *  the modal window with progress bar appears. The reset process
     *  is started in background
     */
    showModalFactoryReset: function() {
        var self = this;

        SWCElements.modalWindow.show({
            templateID: 'system:settings:reset:modal',
            tempalteData: {
                localeStrings: swc.models.Locale.getLocaleStrings("system"),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            },
            className: 'reset-gateway',

            onApply: function() {
                var resetType = $('.modalWindow.reset-gateway').find('.reset-type').data('value'),
                    resetDect = resetType === 'with';

                SWCElements.modalWindow.hide();
                self.processFactoryReset(resetDect);
            }
        });
    },

    /**
     * Show modal window with progress bar and start reset device process.
     * When the device is up then modal window disappears
     *
     * @param resetDect
     */
    processFactoryReset: function(resetDect) {
        SWCElements.modalWindow.show({
            templateID: 'system:settings:reset-process:modal',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings("system"),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            },
            className: 'system-restore',
            onShow: function(){
                var element = $('.modalWindow'),
                    resetTime = swc.models.System.minExpectedRebootTime,
                    resetRuler;

                element.find('.ruler-block').html($.tmpl(swc.Templates.get('ruler').get('content'), {
                    'className': 'reset-ruler'
                }));

                resetRuler = element.find('.reset-ruler');

                // hide modal window only when device is up
                swc.models.System.on('device:up', function () {
                    SWCElements.modalWindow.hide();
                });

                setTimeout(function(){
                    resetRuler.trigger('swc-ruler:start', {time: resetTime});
                }, 10);

                // if resetDect is true, we reset dect at first. Otherwise, start reset gateway immediately
                if(resetDect){
                    $.when(swc.models.Telephony.dectReset(), swc.models.Telephony.dectRemoveDects()).done(function(){
                        swc.models.System.resetGateway(resetTime);
                    });
                } else {
                    swc.models.System.resetGateway(resetTime);
                }
            }
        });
    }

});
;swc.constructors.SystemSettingsGatewayView = swc.constructors.SetPasswordView.extend({

    models: [ 'System', 'PasswordRecovery', 'UserManagement' ],

    validation: {
        'current-password':     'UserManagement:currentPassword',
        'new-password':         'UserManagement:newPassword',
        'new-password-confirm':  'UserManagement:newPasswordConfirm'
    },

    events: {
        'input input.validatable': 'limitSetOfCharacters',
        'keydown input.validatable': 'limitSetOfCharactersDown',
        'blur input.validatable': 'pageValidation'
    },

    initialize: function() {
        this.events = _.extend({}, swc.constructors.SetPasswordView.prototype.events, this.events);
        swc.base.TabView.prototype.initialize.apply(this, arguments);
    },

    setButtonsState: function() {
        var container = this.$('.buttons-container-message'),
            buttonSave = container.find('.button.save-changes'),
            buttonCancel = container.find('.button.cancel-changes'),
            currentPasswordValid = this.$('input[name=current-password]').hasClass('valid'), // FIXME: chenge to real validation
            newPasswordValue = this.$('input[name=new-password]').val(),
            newPasswordConfirmValue = this.$('input[name=new-password-confirm]').val(),
            isValid = (swc.Utils.validatePassword(newPasswordValue) && (newPasswordValue === newPasswordConfirmValue) && currentPasswordValid),
            globalEnableChanged = this.isGlobaEnableChanged();

        if(isValid || globalEnableChanged) {
            buttonSave.removeClass('disabled');
            buttonCancel.removeClass('disabled');
        } else {
            buttonSave.addClass('disabled');
            buttonCancel.addClass('disabled');
        }
    },

    limitSetOfCharactersDown: function(e) {
        return swc.Utils.limitSetOfCharactersDown(e, swc.Utils.nonPrintableRegexGlobal());
    },

    limitSetOfCharacters: function(e) {
        return swc.Utils.limitSetOfCharactersInput(e, swc.Utils.nonPrintableRegexGlobal());
    }
});
;swc.constructors.RebootModalView = Backbone.View.extend({
    className: 'modal reboot-gateway',

    tagName: 'div',

    hidden: false,

    events: {
        'click .cancel-reboot': 'closeModal',
        'click .start-reboot': 'startReboot'
    },

    closeModal: function(){
        $(this.el).modal('hide');
    },

    startReboot: function(e){
        e.preventDefault();

        var rebootProcessModal = new swc.constructors.RebootProcessModalView();

        swc.models.Login.on('beforeLogout', rebootProcessModal.closeModal, rebootProcessModal);

        $(rebootProcessModal.el).on('hidden', function(){
            $(rebootProcessModal.el).remove();
        });

        this.closeModal();
        rebootProcessModal.$el.modal({backdrop: 'static',keyboard: false});
    },

    initialize: function(){
        this.template = swc.Templates.get('system:settings:reboot:modal');
        this.render();
    },

    render: function(){
        var self = this, element = $(this.el);
        element.html($.tmpl(self.template.get('content'), {
            localeStrings: swc.models.Locale.getLocaleStrings("system"),
            localeString: getTranslationStringsjQuery,
            formatDate: swc.models.Locale.formatDate
        }));
    }

});;swc.constructors.RebootProcessModalView = Backbone.View.extend({
    className: 'modal reboot-gateway',

    tagName: 'div',

    events: {
        'click .close-modal': 'closeModal'
    },

    initialize: function(){
        this.template = swc.Templates.get('system:settings:reboot-process:modal');
        this.rulerTemplate = swc.Templates.get('ruler');
        this.render();
    },

    closeModal: function(){
        $(this.el).modal('hide');
    },

    render: function(){
        var self = this, element = $(this.el);
        element.html($.tmpl(self.template.get('content'), {
            localeStrings: swc.models.Locale.getLocaleStrings("system"),
            localeString: getTranslationStringsjQuery,
            formatDate: swc.models.Locale.formatDate
        }));

        element.find('.ruler-block').html($.tmpl(self.rulerTemplate.get('content'), {
            'className': 'reboot-ruler'
        }));

        setTimeout(function(){
            element.find('.reboot-ruler').trigger('swc-ruler:start', {time: swc.models.System.minExpectedRebootTime});
        }, 0);

        swc.models.System.rebootGateway(swc.models.System.minExpectedRebootTime)
            .fail(function () {
                self.$('.progress-block').addClass('hidden');
                self.$('.error-message').removeClass('hidden');
            });
        swc.models.System.on('device:up', function () {
            self.$el.modal('hide');
        });

    }

});
;swc.constructors.SystemSettingsRebootView = swc.base.TabView.extend({

    className: 'reboot',

    models: ['System', 'apServiceState'],

    events: {
        'click .reboot-gateway': 'onProcessReboot'
    },

    preRender: function() {
        // We have to load model `apServiceLoadingState` only if AP is enabled
        if (swc.models.apServiceState.isEnabled()) {
            swc.models.apServiceLoadingState = new swc.constructors.apServiceLoadingState();
        }
    },

    onProcessReboot: function(e) {
        var self = this;

        e.preventDefault();

        // before show modal window the session state should be checked
        swc.models.System.checkSessionState(function() {
            self.processReboot();
        });
    },

    processReboot: function(){
        var rebootModal = new swc.constructors.RebootModalView();

        swc.models.Login.on('beforeLogout', rebootModal.closeModal, rebootModal);

        $(rebootModal.el).on('hidden', function(){
            $(rebootModal.el).remove();
        });

        rebootModal.$el.modal({backdrop: 'static',keyboard: false});

    }

});
;swc.constructors.SystemSettingsView = swc.base.PageView.extend({

    className: 'system-settings'

});;swc.constructors.SystemSettingsFirmwareView = swc.base.TabView.extend({

    className: 'firmware',

    models: ['System', 'apServiceLoadingState'],

    allowedMods: ['expert'],

    fileExtension: null,
    file: null,
    wasOff: false,

    events: {
        'click .upgrade-config:not(.disabled)': 'onUpgradeConfig',
        'click .upgrade-process:not(.disabled)': 'onUpgradeProcess',
        'mouseover .has-tooltip': 'showTooltip',
        'mouseleave .has-tooltip': 'hideTooltip'
    },

    setTemplateData: function(){
        var buildInfo = swc.settings.application.get('build');

        this.templateData = {
            GlobalEnable: swc.models.Application.get('GlobalEnable'),
            npVersion: swc.models.System.get('NPVersion') || '0.0.0',
            apVersion: swc.models.System.get('APVersion') || '0.0.0',
            uiVersion: buildInfo["build"] ? buildInfo["build"] : null
        };
    },

    wrongFileModal: function(){
        var self = this;
        SWCElements.modalWindow.show({
            templateID: 'system:settings:upgrade:modal:upgrade-wrongfile',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate,
                GlobalEnable: swc.models.Application.get('GlobalEnable')
            },
            className: 'system-restore',
            onCancel: function() {
                self.clearForm();
            }
        });
    },

    processModal: function(){
        var self = this;

        SWCElements.modalWindow.show({
            templateID: 'system:settings:upgrade:modal:upgrade-process',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            },
            className: 'system-upgrade',
            onShow: function(){
                var element = $('.modalWindow'),
                    // Upgrade time is not fixed value; we play with it from time to time
                    // in order to be consistent with progress bar in WebUI
                    // Current value is just temporary result of experiments
                    upgradeTime = swc.models.System.maxExpectedRebootTime + swc.models.System.minExpectedRebootTime,
                    resetRuler,
                    fileField,
                    fileType,
                    formData;
                element.find('.ruler-block').html($.tmpl(swc.Templates.get('ruler').get('content'), {
                    'className': 'reset-ruler'
                }));

                resetRuler = element.find('.reset-ruler');

                fileField = $('input[type=file]');
                fileType = fileField.attr('name');

                formData = new FormData();

                // append the fileobject to the formdata
                // !! the filetype should be used as the name for the entry
                formData.append(fileType, self.file);
                
                function doUpgrade() {
                    // send the POST request to /webuiUpgrade
                    $.when(swc.models.System.upgradeGateway(formData))
                        .done(function(){
                            // NP sends response immediately after file with firmware was uploaded,
                            // but after that actually upgrade starts, so let's wait for a while to not ping NP too early
                            setTimeout(function() {
                                var postUpgradeCallback = function() {
                                    function reset() {
                                        self.clearForm();
                                        swc.models.System.doReset();
                                    }
                                    
                                    if(self.wasOff) {
                                        self.turnOffAp(reset);
                                    } else {
                                        reset();
                                    }
                                };
    
                                // Which checker to use depends actually of what we are upgrading: AP or NP (+ web.rui)
                                // We upgrade NP part, this leads to Device follow this states: Down -> Initializing -> Up
                                // Only when device is Up it has sense to reboot the device
                                if (self.fileExtension === "rui") {
                                    swc.models.System.whenDeviceUp(postUpgradeCallback);
                                } else if (self.fileExtension === "acs") {
                                    // In case if AP upgraded, device never goes down, so device=Up returned always
                                    // and we should check if APController.get() returns "UP" for the Connection state
                                    // swc.models.System.whenApUp(postUpgradeCallback);
                                    postUpgradeCallback();
                                }
                            }, swc.models.System.minExpectedRebootTime);
                        })
                        .fail(function(){
                            resetRuler.trigger('swc-ruler:finish');
                            SWCElements.modalWindow.hide();
                            self.errorModal();
                        }
                    );
                }
                
                // check if we have ap upgrade and ap is on!
                if(self.fileExtension === "acs") {
                    upgradeTime += swc.models.System.apBootTime;
                    self.checkAPState(doUpgrade);
                } else {
                    doUpgrade();
                }
    
                setTimeout(function(){
                    resetRuler.trigger('swc-ruler:start', {time: upgradeTime});
                }, 0);
            }
        });
    },

    errorModal: function(){
        var self = this;
        SWCElements.modalWindow.show({
            templateID: 'system:settings:upgrade:modal:upgrade-error',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            },
            className: 'system-upgrade',
            onCancel: function() {
                self.clearForm();
            },
            onApply: function(){
                SWCElements.modalWindow.hide();
                self.confirmModal();
            }
        });
    },

    clearForm: function(){
        this.$el.find('form')[0].reset();
        this.$el.find('.file-name .no-file').show();
        this.$el.find('.file-name .file').hide();
    },

    handleFile: function(dropped) {
        var self = this,
            $fileInput = $('#config-upgrade');

        var filePath = (dropped === true) ? self.file.name : self.form.find('input[type=file]').val();

        if(filePath.indexOf('\\') < 0){
            self.fileName = filePath;
        } else {
            self.fileName = filePath.slice(filePath.lastIndexOf("\\")+1, filePath.length);
        }

            // We need to save this reference to know later which call to use to check if upgrade finished
            self.fileExtension = self.fileName.slice(self.fileName.lastIndexOf(".") + 1, self.fileName.length);
            if(dropped !== true) {
                self.file = $fileInput[0].files[0]; // file object
            }

            if(self.fileName){
                // file type can either be
                //   AP for application processor
                //   SOP for hgw SOP firmware
                if(self.fileExtension === 'rui'){
                    $fileInput.attr('name', 'SOP');
                } else if(self.fileExtension === 'acs' && swc.models.Application.get('GlobalEnable')){
                    $fileInput.attr('name', 'AP');
                } else {
                    self.wrongFileModal();
                    $fileInput.val('').attr('name', 'file');
                    self.file = null;
                }

                $('.file-name .no-file').hide();
                $('.file-name .file').html(self.fileName).show();
            } else {
                self.clearForm();
            }

            self.activateButton();
    },

    onUpgradeConfig: function(){

        this.form = $('#upgrade-form');

        this.form.find('#config-upgrade').off('change');
        this.form.find('#config-upgrade').on('change', $.proxy(this.handleFile, this));
    },

    onUpgradeProcess: function() {
        var self = this;

        // before show modal window the session state should be checked
        swc.models.System.checkSessionState(function() {
            self.processModal();
        });
    },

    enableDragAndDrop: function() {

        var self = this,
            $dropZone = $('.software-upgrade'),
            $fileButton = $('.upgrade-config');
            //$fileInput = $('#config-upgrade');

        $(document).on('dragover', function(e){
            e.preventDefault();
            e.stopPropagation();
            e.originalEvent.dataTransfer.dropEffect = 'copy';
            $dropZone.addClass('dashed');
            $fileButton.addClass('drag');
        });

        $(document).on('dragleave drop', function(e) {
            e.preventDefault();
            e.stopPropagation();
            $dropZone.removeClass('dashed');
            $fileButton.removeClass('drag');
        });

        $dropZone.on('dragover', function(e){
            e.stopPropagation();
            e.preventDefault();
            e.originalEvent.dataTransfer.dropEffect = 'move';

            $dropZone.addClass('dashed');
            $fileButton.removeClass('drag');
        });

        $dropZone.on('drop', function(e){
            e.stopPropagation();
            e.preventDefault();

            if(e.originalEvent.dataTransfer.files) {
                $dropZone.trigger('dragleave');
                self.file = e.originalEvent.dataTransfer.files[0];
                self.handleFile(true);
            }

        });

    },
    
    checkAPState: function(doUpgrade) {
        var self = this,
            apServiceLoadingState = swc.models.apServiceLoadingState.get('isLoading'),
            apState = swc.models.apServiceState.get('status');
            
        // if AP is on and is not loading just do backup
        if(apState === true && apServiceLoadingState !== true) {
            doUpgrade();
        } else {
            self.wasOff = true;
            self.turnOnAp(doUpgrade);
        }
    },
    
    processApTurnOn: function(doUpgrade) {
        var self = this,
            apServiceLoadingState = swc.models.apServiceLoadingState.get('isLoading');

        if(apServiceLoadingState === true) {
            setTimeout(function() {
                swc.models.apServiceLoadingState.fetch();
                self.processApTurnOn(doUpgrade);
            }, 5000);
        } else {
            swc.views.Application.stopPageLoading();
            
            doUpgrade();
        }
    },
    
    turnOnAp: function(doUpgrade) {
        var self = this;

        // Update parameter in the model:
        swc.models.apServiceState.set('status', true);

        // Start listener again when data saved:
        $.when(swc.models.apServiceState.sync('update')).done(function() {
            self.processApTurnOn(doUpgrade);
        });
    },
    
    turnOffAp: function(callback) {
        // Update parameter in the model:
        swc.models.apServiceState.set('status', false);

        // Start listener again when data saved:
        $.when(swc.models.apServiceState.sync('update')).done(function() {
            callback();
        });
    },

    activateButton: function(){
        if(this.file !== null){
            this.$el.find('.upgrade-process').removeClass('disabled');
        } else {
            this.$el.find('.upgrade-process').addClass('disabled');
        }
    },

    renderComplete: function(){
        if(swc.Utils.getIEVersion() > 1 && swc.Utils.getIEVersion() < 10){
            this.$el.find('.upgrade-config').addClass('disabled has-tooltip has-popover');
            this.$el.find('.upgrade-process').addClass('has-tooltip has-popover');
            $('#config-upgrade').remove();
        }

        //Check support for file apis
        if (window.File && window.FileReader && window.FileList) {
            this.enableDragAndDrop();
        }
    },
    
    showTooltip: function(e) {
        e.preventDefault();
        e.stopPropagation();

        var element = $(e.currentTarget);
        if(!element.hasClass('selected')) {
            element.trigger('popover:toggle');
        }
    },
    
    hideTooltip: function(e) {
        e.preventDefault();
        e.stopPropagation();

        var element = $(e.currentTarget);
        element.trigger('popover:close');
    }
});
;swc.constructors.TelephonyCallsAllView = swc.base.TabView.extend({

    className: 'all-calls',

    models: [ 'CallCollection' ],

    events: {
        'click .do-delete': 'deleteCall',
        'click .do-add-contactcall': 'tryAddContact',
        'click #delete-all': 'deleteAll',

        'mouseover .do-delete': 'showPopover',
        'mouseout .do-delete': 'hidePopover',

        'mouseover .do-add-contactcall': 'showPopover',
        'mouseout .do-add-contactcall': 'hidePopover',

        'mouseover .icon.incoming': 'showPopover',
        'mouseout .icon.incoming': 'hidePopover',

        'mouseover .icon.outgoing': 'showPopover',
        'mouseout .icon.outgoing': 'hidePopover',

        'mouseover .icon.missed': 'showPopover',
        'mouseout .icon.missed': 'hidePopover',

        'mouseover .caller-name': 'showPopover',
        'mouseout .caller-name': 'hidePopover',

        'click .cancel-changes': 'onCancel',

        'input .search-input-contact': 'searchContacts',
        'keyup .search-input-contact': 'searchContacts',
        'change .search-input-contact': 'searchContacts'
    },

    validation: {
    },

    listenerInterval: 10,

    listenerPages: [ ],

    deletedCalls: false,

    pageTemplateID: 'telephony:calls:all',

    filter: null,

    setTemplateData: function(){
        this.templateData = {
            calls: this.filterCalls(),
            filter: this.filter
        };
    },

    preRender: function() {
        if(swc.Utils.NPVersion === 'NP5') {
            window.location.href = "#telephony/device";
        }
        return (new $.Deferred()).resolve();
    },

    onCancel: function() {
        this.deletedCalls = false;
    },

    renderComplete: function() {
        $('body').css('background-color', '#ffffff');
        var missedTab = $('.tab-missed').find('span');

        var missedText = missedTab.text();
        if(missedText.indexOf('(') > -1) {
            missedText = missedText.substr(0, missedText.indexOf('('));
        }

        var numberOfMissed = this.getNumberOfMissed();
        missedTab.text(missedText + " (" + numberOfMissed + ")");
    },

    activateButtons: function(){
        this.$('.save-changes').removeClass('disabled');
        this.$('.cancel-changes').removeClass('disabled');
    },

    setElementsState: function() {
        // if there are some modification(edit, delete of some amount of contacts)
        // then buttons save/cancel should be disabled
        if (this.checkModification()) {
            this.activateButtons();
        } else {
            this.deactivateButtons();
        }
    },

    showPopover: function(e) {
        var item = $(e.target);
        SWCElements.popovers.open(
            item
        );
    },

    hidePopover: function() {
        SWCElements.popovers.closeAll();
    },

    filterCalls: function() {
        var self = this,
            collection = [];

        collection = swc.models.CallCollection.filter(function(model) {
            var response = false;

            if (model.get('deleted')) {
                self.deletedCalls = true;
                response = false;
            } else if (self.searchKeyword) {
                var searchKeywordLC = $.trim(self.searchKeyword.toLowerCase());
                if (model.get('name').toLowerCase().indexOf(searchKeywordLC) !== -1) {
                    response = true;
                }
            } else if (model.get('number') === '' && model.get('direction') === '') { //fix for NP bug with empty call entries. WTF?
                response = false;
            }
             else {
                response = true;
            }

            //Filter by type
            if(self.filter !== null) {
                if(model.get('type') !== self.filter) {
                    response = false;
                }
            }


            return response;
        });

        _.map(collection, function(item){
            item.set({
                duration: self.formatDuration(item.get('duration')),
                startTimeFormatted: self.formatStartDate(item.get('startTime'))
            });

            return item;
        });

        return collection.reverse();
    },

    searchContacts: function(e) {
        e.stopPropagation(); // We cannot allow events bubbling
        //update current carret position
        var searchInput = $(e.target);
        this.searchKeyword = $(searchInput).val();
        this.searchProcess = true;
        //swapped default render to a version that does not override the input field
        this.dataRender();
        this.searchProcess = false;
    },

    dataRender: function(){
        var self = this,
             templateCache= null;

        templateCache = $.tmpl(this.template, {
                localeStrings: swc.models.Locale.getLocaleStrings("telephony"),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate,
                data: self.templateData
        });

         // Set template data:
        this.setTemplateData();

        $(this.el).find(".expandable-calls").html($(templateCache).find(".expandable-list"));
        //$(this.el).html(templateCache);

        this.renderComplete();
    },

    formatDuration: function(duration) {
        var sec_num = parseInt(duration, 10),
            hours   = Math.floor(sec_num / 3600),
            minutes = Math.floor((sec_num - (hours * 3600)) / 60),
            seconds = sec_num - (hours * 3600) - (minutes * 60);

        if (hours   < 10) { hours   = "0"+hours; }
        if (minutes < 10) { minutes = "0"+minutes; }
        if (seconds < 10) { seconds = "0"+seconds; }

        var time    = hours+':'+minutes+':'+seconds;
        return time;
    },

    formatStartDate: function(date) {
       return moment(date).format("DD.MM.YY HH:mm");
    },

    postInitialize: function() {
        this.deletedCalls = false;
    },

    tryAddContact: function(e) {
        var self = this, ev = e;
        SWCElements.modalWindow.show({
            templateID: 'telephony:calls:add-callcontact',
            templateData: {
            },
            className: 'add-callcontact-modal',

            onApply: function() {
                SWCElements.modalWindow.hide();
                self.addContact(ev);
            },

            onCancel: function() {
                SWCElements.modalWindow.hide();
                sessionStorage.removeItem("addFromCallList");
            }
        });
    },

    checkModification: function() {
        return (!this.pageCheckDefaultValues() || this.deletedCalls);
    },

    hasChanged: function() {
        return (!this.pageCheckDefaultValues() || this.deletedCalls);
    },

    deleteCall: function(e) {
        if($(e.target).hasClass('disabled')) {
            return;
        }
        var item = $(e.target).closest('.expandable-list-item'), itemId = item.attr('data-key'),
        call = swc.models.CallCollection.get(itemId),
        noCalls = $('div.no-calls');
        call.set({'deleted': true});
        this.deletedCalls = true;
        this.setElementsState();

        // this.render();

        item.fadeOut(function(){
            item.remove();
        });

        if(swc.models.CallCollection.models.length === 0 ||
            _.isUndefined(swc.models.CallCollection.find(function (model) { return model.get('deleted') === false }))
        ) {
            _.delay(function() {noCalls.removeClass('hidden')}, 700);
        }
    },

    deleteAll: function(e) {
        if($(e.currentTarget).find('a').hasClass('disabled')) {
            return;
        }

        var noCalls = $('div.no-calls');

        $('.expandable-list-item').hide();

        swc.models.CallCollection.map(function(item) {
            item.set({'deleted': true});
            return item;
        });

        noCalls.removeClass('hidden');
        this.deletedCalls = true;
        this.setElementsState();

    },

    getDeleted: function() {
        var results = [];

        swc.models.CallCollection.each(function(val){
            if(val.get('deleted') === true) { results.push(val.get('id')); }
        });

        return results;
    },

    getNumberOfMissed: function() {
        return swc.models.CallCollection.where({type: "missed", viewed: false}).length;
    },

    addContact: function(e) {
        var item = $(e.target).closest('.expandable-list-item'), itemId = item.attr('data-key'),
            call = swc.models.CallCollection.get(itemId);

        sessionStorage.setItem("addFromCallList", call.get('number'));
        if(!call.get('inContactList')) {
            sessionStorage.setItem("addFromCallListName", call.get('name') !== 'Unknown' ? call.get('name') : null);
        } else {
            sessionStorage.removeItem("addFromCallListName");
        }

        swc.router.navigate('telephony/phonebook/edit');
    },

    save: function() {
        var deferred = new $.Deferred(),
            actionsToDo = [],
            deletedList = this.getDeleted();

        this.stopPageLoading();
        this.showPageLoading('Saving changes...');

        if(swc.models.CallCollection.length === deletedList.length) {
            swc.models.CallCollection.removeCalls().done(function() {
                 return deferred.resolve();
            });

            return deferred.promise();
        }

        _.each(deletedList, function(item) {
            actionsToDo.push(swc.models.CallCollection.removeCalls(item));
        });

        // Process save request:
        $.when.apply(null, actionsToDo)
        .done(function() {
            deferred.resolve();
        }).fail(function() {
            deferred.reject();
        });

        _.delay(function() {deferred.resolve(); }, 3000);

        return deferred.promise();
    },

    /**
     * Disable save/cancel buttons
     */
    deactivateButtons: function(){
        this.$('.save-changes').addClass('disabled');
        this.$('.cancel-changes').addClass('disabled');
    }

    // initialize: function () {
    //     swc.base.TabView.prototype.initialize.apply(this, arguments);
    //     this.template = $.template("pageContent", swc.Templates.get('telephony:calls:all').get('content'));
    // }
});
;swc.constructors.TelephonyCallsView = swc.base.PageView.extend({

    className: 'telephony-calls',

    models: [
    ],

    isAbleToNavigate: function() {
		var userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin();

		if(userIsSuperAdmin === true) {
			return false;
		}
		return true;
    }
});;swc.constructors.TelephonyCallsIncomingView = swc.constructors.TelephonyCallsAllView.extend({

    filter: "incoming"
});
;swc.constructors.TelephonyCallsMissedView = swc.constructors.TelephonyCallsAllView.extend({

    filter: "missed"


    // className: 'missed-calls',

    // models: [
    // ],

    // events: {
    // },

    // validation: {
    // },

    // setTemplateData: function() {
    //     this.templateData = {
    //     };
    // },

    // initialize: function () {
    //     swc.base.TabView.prototype.initialize.apply(this, arguments);

    //     // // Prepare template for the settings form:
    //     // this.settingsFormTemplate = $.template("settingsForm",
    //     //     swc.Templates.get("network:settings:dyndns:service-settings-form").get('content'));
    // }
});
;swc.constructors.TelephonyCallsOutgoingView = swc.constructors.TelephonyCallsAllView.extend({

    filter: "outgoing"


});
;swc.constructors.TelephonyDectView = swc.base.PageView.extend({

    className: 'dect',

    models: [ 'Telephony', 'System', 'EventDispatcher' ],

    validation: {
        'pin': 'Telephony:dectPin'
    },

    events: {
        'swc-switcher:change .enable-dect': 'dectStatus',

        'swc-checkbox:change .show-pin-password': 'setPinState',
        'swc-checkbox:change .eco-status': 'ecoStatus',

        'keyup input.pin-original': 'onPinChange',
        'keyup input.pin-copy': 'onPinChange',
        'change input.pin-original': 'onPinChange',
        'change input.pin-copy': 'onPinChange',

        'click .pair-dect': 'pairDect',

        'input input.pin-original': 'limitSetOfCharacters',
        'input input.pin-copy': 'limitSetOfCharacters',

        'keydown input.pin-original': 'limitSetOfCharactersDown',
        'keydown input.pin-copy': 'limitSetOfCharactersDown'
    },

    eventActions: [
      {
        event: 'pairingDone',
        action: 'paringDone'
      }
    ],

    listenerPages: [
        'dect'
    ],

    failOverLoopCount: 0,
    failOverLoopName: 'dect',

    passEvent: function(event, eventObject) {
      swc.constructors.EventController(this.eventActions)
                      .pass.call(this, event, eventObject);
    },

    paringDone: function(event){
        if(event.data.object.attributes.reason === "Canceled") {
            $('.close-modal').click();
        }
    },

    setTemplateData: function(){
        this.templateData = {
            dectStatus: swc.models.Telephony.get('dectStatus'),
            dectEco: swc.models.Telephony.get('dectEco'),
            pin: swc.models.Telephony.get('dectPIN')
        };
    },

    dectStatus: function(e, value){
        var self = this;

        this.showPageLoading('Updating DECT status');

        swc.models.Telephony.set('dectStatus', value);

        $.when(swc.models.Telephony.setDectStatus()).done(function(){
            self.render();
        });
    },

    ecoStatus: function(e, value){
        swc.models.Telephony.set('dectEco', value);
        this.setButtonsState();
    },

    setPinState: function(e, value) {
        var pinOrigBox = $('input.pin-original'),
            pinCopyBox = $('input.pin-copy');

        if (value) {
            pinOrigBox.hide();
            pinCopyBox.show();
        } else {
            pinCopyBox.hide();
            pinOrigBox.show();
        }
    },

    limitSetOfCharactersDown: function(e) {
        return swc.Utils.limitSetOfCharactersDown(e, /[^\d]/g);
    },

    limitSetOfCharacters: function(e) {
        return swc.Utils.limitSetOfCharactersInput(e, /[^\d]/g);
    },

    onPinChange: function (e) {
        this.limitSetOfCharacters(e);
        var parameters = getParameter($(e.target)),
            pinOrigBox = $('input.pin-original'),
            pinCopyBox = $('input.pin-copy');

        if ($.inArray('pin-original', parameters.parameterClasses) !== -1) {
            pinCopyBox.val(pinOrigBox.val());
        } else {
            pinOrigBox.val(pinCopyBox.val());
        }

        swc.models.Telephony.set('dectPIN', $(e.target).val());

        this.setButtonsState();
    },

    renderComplete: function() {
        var self = this;
        self.eventDispatcher = swc.models.EventDispatcher;
        self.eventDispatcher.observ("dectSettings",self);

        self.failOverLoopCount = 0;

        function failOverDect(loopCount) {
            $.when(
                swc.models.Telephony.sync(false, self.failOverLoopName),
                swc.models.System.sync(false, self.failOverLoopName)
            ).done(function(rep, rep2) {
                var pagesMatch = false,
                pagesRoutes = swc.settings.application.get('navigation'),
                page = Backbone.history.fragment;

                $.each(self.listenerPages, function(key, value) {
                    if (pagesRoutes[value] === page) {
                        pagesMatch = true;
                    }
                });

                if (pagesMatch) {
                    if(rep === true && rep2 === true) {
                        self.setTemplateData();
                        self.getTemplateContent();
                    } else if(loopCount === self.failOverLoopCount && loopCount < 100) {
                        self.failOverLoopCount = loopCount + 1;
                        failOverDect(loopCount + 1);
                    }
                }
            });
        }

        if(swc.models.Telephony.get('errorStatus') === true || swc.models.System.get('errorStatus') === true) {
            swc.models.Telephony.set('errorStatus', undefined);
            swc.models.System.set('errorStatus', undefined);
            failOverDect(0);
        }
    },

    /*
    * First we check if the server is logged in. Just any AJAX call will do.
    */
    pairDect: function() {
        var self = this;
        swc.models.System.checkSessionState(function () {
            self._pairDect();
        });
    },

    _pairDect: function() {
        var self = this,
            withoutSip = swc.models.Telephony.get('phones').filter(function(el){
                return el.get('deviceType') !== 'sip';
            });
 

        if (!swc.models.Login.checkUserLogin()) {
            return false;
        }

        if((withoutSip.length-2) < swc.models.Telephony.dectInternalNumbers.length){
            var pairingModal = new swc.constructors.DectPairingModalView();

            $(pairingModal.el).on('hidden', function(){
                $(pairingModal.el).remove();
                self.render();
            });

        } else {

            var modal = swc.Templates.get('telephony:device:pair:modal:manyDects').get('content'),
                div = $('<div/>', {'class':'modalWindow modal manyDects-modal'});

            div.html($.tmpl(modal, {
                localeStrings: swc.models.Locale.getLocaleStrings("telephony"),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            }));

            div.on('hidden', function(){
                div.remove();
            });

            div.find('.close-modal').on('click', function() {
                $('.manyDects-modal').modal('hide');
            });

            div.modal();
        }
    },

    save: function() {
        var deferred = new $.Deferred();

        $.when(swc.models.Telephony.setPin(), swc.models.Telephony.setDectEco())
            .done(function() {
                deferred.resolve();
            })
            .fail(function() {
                deferred.reject();
            });

        return deferred.promise();
    }
});
;swc.constructors.TelephonyPhonebookEditView = swc.base.PageView.extend({
    className: 'phonebook-new-contact',

    models: ['Phonebook'],

    validation: {
        'firstName':  'Phonebook:firstName',
        'lastName':   'Phonebook:lastName',
        'homeNumber': 'Phonebook:homeNumber',
        'cellNumber': 'Phonebook:cellNumber',
        'workNumber': 'Phonebook:workNumber'
    },

    events: {
        'input .contact-number': 'onChangeContactNumber',
        'input .contact-name': 'onChangeContact',
        'blur .contact-name': 'onContactNameBlur',
        'blur .contact-number': 'onContactBlur'
    },

    phoneSet: ['HOME', 'CELL', 'WORK'],

    model: null,

    initialize: function() {
        swc.base.PageView.prototype.initialize.apply(this, arguments);
        this.template = $.template("pageContent", swc.Templates.get('telephony:phonebook:new-number').get('content'));
    },

    /**
     * Event handler: is fired on changing value in the input field
     *
     * @description:
     *
     * this wrapper is used to defer validation check until all events will be triggered
     * this approach is similar to use setTimeout(func, 0)
     */
    onChangeContact: function() {
        _.defer(this.changeContact, this);
    },
    /**
     * Event handler: is fired on changing value in the input field of type number
     *
     * @description:
     *
     * this wrapper is used to defer validation check until all events will be triggered
     * this approach is similar to use setTimeout(func, 0)
     * additionally a regexp check can be commited to check if the data is a phone number matching "^[\d\b\+#*]*$"
     * this action happens on every field update
     */
    onChangeContactNumber: function(e) {
      var replacement=$(e.target).val();
      //filters paste or entered by key data before model validation occurs / live filter / native backbone validation causes errors in firefox.
      replacement=replacement.replace(/[^0-9#+*]/g,'');
      $(e.currentTarget).val(replacement.substring(0, 30));
      _.defer(this.changeContact, this);
    },

    /**
     * Handle data during user input in the edit form of items in the phonebook
     *
     * @param e {Object} -> Event
     */
    changeContact: function(self) {
        var model = self.getContactModel(),
            elements = self.getElements(),
            elementParams = [];

        _.each(elements, function(element) {
            elementParams.push(getParameter($(element)));
        });

        model.set({'modified': !self.pageCheckDefaultValues()});

        // fill contact model by first, last names
        model.set({
            firstName: $.trim(_.where(elementParams, { parameterName: 'firstName' })[0].parameterValue),
            lastName: $.trim(_.where(elementParams, { parameterName: 'lastName' })[0].parameterValue)
        });

        // fill contact model by phone numbers
        _.each(self.phoneSet, function(phoneType) {
            var numberObj = _.where(model.get('telephoneNumbers'), { type: phoneType })[0],
                numberValue = $.trim(_.where(elementParams, { parameterName: phoneType.toLowerCase() + 'Number'})[0].parameterValue);

            if (!numberObj) {
                model.get('telephoneNumbers').push({
                    name: numberValue,
                    preferred: false,
                    type: phoneType
                });
            } else {
                numberObj.name = numberValue;
            }
        });
    },
    onContactNameBlur: function(){
        var self = this,
            onSaveErrors = this.onSaveValidation();
        
        if (_.isEmpty(onSaveErrors)) {
            self.showSaveSuccess(self, 'error', onSaveErrors);
        } else {
            
            _.defer(function() {
                self.showSaveSuccess.call(self, 'error', onSaveErrors);
            });
        }
        this.showSaveSuccess();
    },
    onContactBlur: function(e) {
        var element = $(e.target),
            val = element.val(),
            self = this,
            onSaveErrors = this.onSaveValidation();
            
        if(val.length > 0) {
            if(element.val().charAt(0) === '+') {
                val = '00' + val.slice(1);
            }
        }
if (_.isEmpty(onSaveErrors)) {
            self.showSaveSuccess(self, 'error', onSaveErrors);
        } else {
            
            _.defer(function() {
                self.showSaveSuccess.call(self, 'error', onSaveErrors);
            });
        }
        element.val(val);
        _.defer(this.changeContact, this);
        
    },

    /**
     * Validation for form 'Add new contact' checks if the first or last name is empty
     *
     * @returns {boolean}
     */
    onSaveValidation: function() {
        var model = this.getContactModel(),
            fullName = model.get('firstName') + model.get('lastName'),
            emptyNumbers = _.where(model.get('telephoneNumbers'), { name: '' }),
            allNamesAreEmpty = _.isEmpty(fullName),
            allNumbersAreEmpty = (emptyNumbers.length === 3),
            validation = {};

        
        if (allNamesAreEmpty && allNumbersAreEmpty) {
            validation['allfields'] = {
                status: false,
                message: 'at least one field should be filled'
            };
            return validation;
        }

        var fullNameSize = swc.Utils.lengthInUtf8Bytes(fullName.trim().replace(/\\/g, '\\\\').replace(/;/g, '\\;')),
        sumCharacters = ';'.length + fullNameSize;

        if (sumCharacters > 57) {
            validation['names'] = {
                status: false,
                message: 'too many characters for firstname and lastname combined'
            };

            return validation;
        }

        if (allNamesAreEmpty) {
            validation['names'] = {
                status: false,
                message: 'at least one field should be filled'
            };
        }

        if (allNumbersAreEmpty) {
            validation['numbers'] = {
                status: false,
                message: 'at least one field should be filled'
            };
        }

        return validation;
    },

    /**
     * reset temporary data
     */
    resetContactModel: function() {
        this.model = null;
        sessionStorage.removeItem(swc.models.Phonebook.requestParamNameID);
    },

    onCancel: function() {
        this.resetContactModel();
        swc.router.navigate('telephony/phonebook', { trigger: false, skipUnsavedChanges: true } );
    },

    /**
     * @override: upon pressing "Save" button the common validation
     * for all fields should be invoked.
     */
    saveChanges: function() {
        var self = this,
            onSaveErrors = this.onSaveValidation();

        if (_.isEmpty(onSaveErrors)) {
            swc.base.PageView.prototype.saveChanges.apply(this, arguments);
        } else {
            /*
             validation for all fields should be triggered after
             individual validation to have opportunity to assign
             CSS classes correctly
             */
            _.defer(function() {
                self.showSaveSuccess.call(self, 'error', onSaveErrors);
            });
        }
    },

    save: function() {
        var self = this,
            deferred = new $.Deferred(),
            model = this.getContactModel();

        $.when(model.save())
            .done(function(data) {
                self.resetContactModel();
                deferred.resolve();
                if(data.status === false){
                    SWCElements.modalWindow.show({
                        templateID: 'telephony:phonebook:new-number:error-saving',
                        templateData: {},
                        className: 'phonebook-clearall',
                        onApply: function() {
                            swc.router.navigate('telephony/phonebook', { skipUnsavedChanges: true });
                            SWCElements.modalWindow.hide();
                        }
                    });
                }
                else {
                    swc.router.navigate('telephony/phonebook', { skipUnsavedChanges: true });
                }
                
                
            })
            .fail(function() {
                deferred.reject();
            });

        return deferred.promise();
    },

    setButtonsState: function() {
        this.$('.button.save-changes').toggleClass('disabled', this.pageCheckDefaultValues());
    },

    /**
     * Show unsuccessfull page save result
     */
    showSaveSuccess: function(type, onSaveErrors) {
        var self = this;

        // if no error do nothing
        if (_.isEmpty(onSaveErrors)) {
            return;
        }

        this.clearValidationMessages();

        _.each(onSaveErrors, function(error, errorName) {
            var container = this.$('.validation-message[data-parameter-name="' + errorName + '"]');

            container.show();
            container.find('.error-message').hide();
            container.find('.error-message[data-error="' + error.message + '"]').show();
            self.highlightField(errorName);
        });
    },

    highlightField: function(errorName) {
        if (errorName === 'allfields') {
            this.$('.contact-name.validatable').addClass('validation-error');
            this.$('.contact-number.validatable:first').addClass('validation-error');
            this.$('.validation-message[data-parameter-name="allfields"]').show();
        }
        if (errorName === 'names') {
            this.$('.contact-name.validatable').addClass('validation-error');
            this.$('.validation-message[data-parameter-name="names"]').show();
        }
        if (errorName === 'numbers') {
            this.$('.contact-number.validatable:first').addClass('validation-error');
            this.$('.validation-message[data-parameter-name="numbers"]').show();
        }
    },

    /**
     * Set template data for output phone's types in the cycle
     */
    setTemplateData: function() {
        var model = this.getContactModel(),
        numbers = null;
        //self = this;
        if(model === undefined){
            SWCElements.modalWindow.show({
                templateID: 'telephony:phonebook:new-number:no-data',
                templateData: {
                    component: 'NOCONTACT'
                },
                className: 'phonebook-clearall',
                onApply: function() {
                    swc.router.navigate('telephony/phonebook', { skipUnsavedChanges: true });
                    SWCElements.modalWindow.hide();
                }
            });
            return false;
        }
        else{
            numbers = model.get('telephoneNumbers');
        }
        // make object for editable number fields
        model.set('phoneNumbers', {
            'HOME': _.where(numbers, { type: 'HOME' })[0] ? _.where(numbers, { type: 'HOME' })[0].name : '',
            'CELL': _.where(numbers, { type: 'CELL' })[0] ? _.where(numbers, { type: 'CELL' })[0].name : '',
            'WORK': _.where(numbers, { type: 'WORK' })[0] ? _.where(numbers, { type: 'WORK' })[0].name : ''
        });


        var fromCallList = sessionStorage.getItem("addFromCallList"),
            fromCallListName = sessionStorage.getItem("addFromCallListName");

        if(!_.isUndefined(fromCallList) && fromCallList !== null && fromCallList !== '') {
            fromCallList = fromCallList.trim();
        }

        if(!_.isUndefined(fromCallListName) && fromCallListName !== null && fromCallListName !== '' && fromCallListName !== 'null') {
            this.addingFromCallListName = true;
        }

        this.templateData = {
            nameTypes: {
                'firstName': 'First Name',
                'lastName': 'Last Name'
            },
            phoneTypes: {
                'HOME': 'Home',
                'CELL': 'Mobile',
                'WORK': 'Office'
            },
            fromCallList: fromCallList,
            fromCallListName: fromCallListName,
            localeStrings: swc.models.Locale.getLocaleStrings("telephony"),
            localeString: getTranslationStringsjQuery,
            isNew: model.isNew(),
            contact: model
        };
        sessionStorage.removeItem("addFromCallList");
        sessionStorage.removeItem("addFromCallListName");
    },

    getContactModel: function() {
        var contactID;

        if (this.model === null) {
            contactID = sessionStorage.getItem(swc.models.Phonebook.requestParamNameID);

            if (contactID) {
                this.model = swc.models.Phonebook.get(contactID);
            } else {
                this.model = new swc.constructors.PhonebookContact();
            }
        }

        return this.model;
    },

    renderComplete: function() {
        var self = this,
        container = self.$('.buttons-container-message'),
        buttonSave = container.find('.button.save-changes');

        if (this.addingFromCallListName === true) {
            this.addingFromCallListName = undefined;
            buttonSave.toggleClass('disabled', false);
            self.onChangeContact();
        }

        // this event is used to reset model after leaving edit page
        swc.constructors.dispatcher.on('route:change', function() {
            self.resetContactModel();
            swc.constructors.dispatcher.off('route:change');
        });
    }

});
;swc.constructors.TelephonyPhonebookList = swc.base.PageView.extend({

    className: 'view-contacts',

    models: ['Phonebook'],

    events : {
        'input .search-input': 'searchContacts',
        'keyup .search-input': 'searchContacts',
        'change .search-input': 'searchContacts',

        'click .new-contact': 'redirectToAddContact',

        'click .clear-contacts:not(.disabled)': 'clearAllModal',
        'click .do-delete': 'deleteContact',

        'expandable:edit .expandable-list': 'listEditMode'
    },

    searchKeyword: "",

    searchProcess: false,

    showedSuccessMessage: false,

    // A flag showing if there are some contacts prepared for deletion
    deletedContacts: false,

    initialize: function() {
        swc.base.PageView.prototype.initialize.apply(this, arguments);
        this.template = $.template("pageContent", swc.Templates.get('telephony:phonebook:phonebook').get('content'));
    },

    /**
     * Event handler: is fired on start edit
     *
     * @param e
     * @param value
     */
    listEditMode: function(e, value) {
        var activeButton= $(e.currentTarget).find("div[data-key='"+value+"'] .block-edit .do-edit.pencil"),
        currentView = swc.views.currentView;
        sessionStorage.setItem(swc.models.Phonebook.requestParamNameID, value);

        $.when(swc.models.Phonebook.sync('read'))
        .done(function() {
            if(value) {
                var model = swc.models.Phonebook.get(value);

                if(model === null || _.isUndefined(model)){
                    SWCElements.modalWindow.show({
                        templateID: 'telephony:phonebook:new-number:no-data',
                        templateData: {
                            component: 'NOCONTACT'
                        },
                        className: 'phonebook-clearall',
                        onApply: function() {
                            SWCElements.modalWindow.hide();
                        }
                    });
                    return true;
                }
            }

            if(currentView && currentView.hasChanged()){

            SWCElements.modalWindow.show({
                templateID: 'application:modal:changes-confirm',
                templateData: {
                    formatDate: swc.models.Locale.formatDate
                },
                localeStrings: swc.models.Locale.getLocaleStrings('application'), // TODO :: no time to identify. Refactor this
                className: 'confirm-changes',

                onCancel: function() {
                    sessionStorage.setItem(swc.models.Phonebook.requestParamNameID, value);
                    swc.router.navigate('telephony/phonebook/edit',{skipUnsavedChanges: true});
                },

                onApply: function() {
                    //simulating click to regain default state
                    activeButton.trigger("click");
                    SWCElements.modalWindow.hide();
                }
            });
            } else{
                swc.router.navigate('telephony/phonebook/edit',{skipUnsavedChanges: true});
            }
        });
    },

    redirectToAddContact: function() {
        sessionStorage.removeItem(swc.models.Phonebook.requestParamNameID);
        swc.router.navigate('telephony/phonebook/edit');
    },

    /**
     * Get actual collection of contacts.
     * The following contacts are thrown away from collection:
     *    - which are marked as deleted
     *    - which are not matched to the search keyword(both name and number are checked)
     *
     * @returns {Array}
     */
    filterContactsBySearch: function() {
        var self = this,
            collection = [];

        collection = swc.models.Phonebook.filter(function(model) {
            var response = false,
                numbers = model.get('telephoneNumbers');

            // make object for editable number fields
            model.set('numberFields', {
                'HOME': _.where(numbers, {'type': 'HOME'})[0] ? _.where(numbers, {'type': 'HOME'})[0].name : '',
                'CELL': _.where(numbers, {'type': 'CELL'})[0] ? _.where(numbers, {'type': 'CELL'})[0].name : '',
                'WORK': _.where(numbers, {'type': 'WORK'})[0] ? _.where(numbers, {'type': 'WORK'})[0].name : ''
            });

            if (model.get('deleted')) {
                self.deletedContacts = true;
                response = false;
            } else if (self.searchKeyword) {
                var searchKeywordLC = $.trim(self.searchKeyword.toLowerCase());
                if (model.get('formattedName').toLowerCase().indexOf(searchKeywordLC) !== -1) {
                    response = true;
                }

                _.each(model.get('telephoneNumbers'), function(number) {
                    if (number.name.indexOf(searchKeywordLC) !== -1) {
                        response = true;
                    }
                });
            } else {
                response = true;
            }

            return response;
        });
        
        // sort
        collection = swc.Utils.sortByCollate(collection, function(model) {
            // default ignore case
            return $.trim(model.get('formattedName'));
        });

        return collection;
    },

    /**
     * Event handler: is fired during user input in the search field
     *
     * @param e
     */
    searchContacts: function() {
        //update current carret position
        var searchInput = this.$('.search-input')[0];
        this.searchKeyword = $(searchInput).val();
        this.searchProcess = true;
        //swapped default render to a version that does not override the input field
        this.dataRender();
        this.searchProcess = false;
    },

    /**
     * Save all contacts.
     * The following flags influences on the actual action during saving
     *    modified { Boolean }
     *    deleted { Boolean }
     */
    save: function() {
        var self = this;
        $.when(swc.models.Phonebook.deleteAllContacts())
            .done(function() {
                swc.views.telephonyPhonebookView.showSuccessMessage = true;
                self.showedSuccessMessage = true;
                swc.views.telephonyPhonebookView.render();
            });
    },

    /**
     * Event handler: render page in the default state
     */
    onCancel: function() {
        swc.views.telephonyPhonebookView.render();
    },

    /**
     * Event handler: delete picked contact from model
     * Actually contact is not deleted but marked as deleted
     *
     * @param e {Object}
     */
    deleteContact: function(e) {
        var itemId = $(e.target).closest('.expandable-list-item').attr('data-key');
        swc.models.Phonebook.get(itemId).set({'deleted': true});
        this.deletedContacts = true;
        this.setButtonsState();
        this.render();
    },

    /**
     * Delete all contacts
     * Is used as a handler of the event clearAllModal
     */
    deleteAllContacts: function() {
        swc.models.Phonebook.each(function(model) {
            model.set({'deleted': true});
        });
        this.saveChanges();
        this.setElementsState();
    },

    /**
     * Event handler: show modal window with prompt to delete all contacts from phone book
     */
    clearAllModal: function() {
        var self = this,
            templateData = {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            };

        SWCElements.modalWindow.show({
            templateID: 'telephony:modal-windows:phonebook:clear-all',
            templateData: templateData,
            className: 'phonebook-clearall',
            onApply: function() {
                self.deleteAllContacts();
                SWCElements.modalWindow.hide();
            }
        });
    },

    checkModification: function() {
        return (!this.pageCheckDefaultValues() || this.deletedContacts);
    },
    
    hasChanged: function() {
        return (!this.pageCheckDefaultValues() || this.deletedContacts);
    },

    /**
     * Enable save/cancel buttons
     */
    activateButtons: function(){
        this.$('.save-changes').removeClass('disabled');
        this.$('.cancel-changes').removeClass('disabled');
    },

    /**
     * Disable save/cancel buttons
     */
    deactivateButtons: function(){
        this.$('.save-changes').addClass('disabled');
        this.$('.cancel-changes').addClass('disabled');
    },

    /**
     * Enable search input
     */
    activateSearchElement: function(){
        this.$('.search-form div').removeClass('disabled');
        this.$('.search-form input').removeAttr('disabled');
    },

    /**
     * Disable search input
     */
    deactivateSearchElement: function() {
        this.$('.search-form div').addClass('disabled');
        this.$('.search-form input').attr('disabled', 'disabled');
    },

    /**
     * Set state(enable/disable) of buttons and search input
     */
    setElementsState: function() {
        // if there are some modification(edit, delete of some amount of contacts)
        // or phone book list is empty
        // then search input should be disabled
        if (this.checkModification() || !swc.models.Phonebook.length) {
            this.deactivateSearchElement();
        } else {
            this.activateSearchElement();
        }

        // if there are some modification(edit, delete of some amount of contacts)
        // then buttons save/cancel should be disabled
        if (this.checkModification()) {
            this.activateButtons();
        } else {
            this.deactivateButtons();
        }
    },

    setTemplateData: function() {
        var self = this;

        this.templateData = {
            phoneTypes: {
                'HOME': 'Home',
                'CELL': 'Mobile',
                'WORK': 'Office'
            },
            searchProcess: self.searchProcess,
            isModified: self.checkModification(),
            contacts: self.filterContactsBySearch()
        };
    },
    dataRender: function(){
         var self = this,
             templateCache= null;

         templateCache= $.tmpl(this.template, {
                localeStrings: swc.models.Locale.getLocaleStrings("telephony"),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate,
                data: self.templateData
            });
         
         // Set template data:
        this.setTemplateData();
        
        $(this.el).find(".expandable-contacts").html($(templateCache).find(".expandable-list"));
        //$(this.el).html(templateCache);
        
    this.renderComplete();
    
    },
    render: function() {
        var self = this;

        // Set template data:
        this.setTemplateData();

        $(this.el).html(
            $.tmpl(this.template, {
                localeStrings: swc.models.Locale.getLocaleStrings("telephony"),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate,
                data: self.templateData
            })
        );

        this.renderComplete();
    },

    renderComplete: function() {
        this.setElementsState();

        // due to overriding render method the
        // success message should be shown explicitly
        if (swc.views.telephonyPhonebookView.showSuccessMessage && this.showedSuccessMessage === false) {
            this.$('.buttons-container-message .save-success').show();
            swc.views.telephonyPhonebookView.showSuccessMessage = false;
        }
    }
});
;swc.constructors.TelephonyPhonebookView = swc.base.PageView.extend({

    className: 'phonebook',

    models: ['Phonebook'],

    /**
     * Due to overriding render method in the child view TelephonyPhonebookList
     * success message should be shown explicitly.
     *
     * This variable is used in the child view TelephonyPhonebookList
     * to determine if the success message should be shown after save.
     * This variable is declared here because the child view always has
     * new instance after rendering of this(parent) view
     *
     * @type {Boolean}
     */
    showSuccessMessage: false,

    renderComplete: function(){
        var view,
            element = $(this.el);

        swc.views.TelephonyPhonebookList = new swc.constructors.TelephonyPhonebookList();
        view = swc.views.TelephonyPhonebookList;

        element.find('.phonebook').html(view.el);
        view.render();
        view.delegateEvents();
    },

    isAbleToNavigate: function() {
        var userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin();

        if(userIsSuperAdmin === true) {
            return false;
        }
        return true;
    }

});;swc.constructors.NoGroupsModal = Backbone.View.extend({
    className: 'modal',

    tagName: 'div',

    initialize: function(){
        this.pageTemplateID = "telephony";
    },

    render: function(){
        var self = this;

        SWCElements.modalWindow.show({
            templateID: 'telephony:modal-windows:no-groups',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            },
            className: 'cloud-unchecked',

            onApply: function() {
                SWCElements.modalWindow.hide();
            }
        });

    }
});
;swc.constructors.DectPairingModalView = swc.base.PageView.extend({
    className: 'modal dect-pairing',

    tagName: 'div',

    hidden: false,

    events: {
        'click .close-modal': 'closeModal',
        'click .try-pairing': 'tryPairing',
        'hide': 'hideModal'
    },

    closeModal: function(){
        var self = this;
        if($(".dect-pairing .close-modal").hasClass("disabled") !== true){
        $.when(swc.models.Telephony.stopPairing())
            .always(function() {
                //swc.models.Login.off('beforeLogout', self.closeModal, self);
                self.hideModal();
                $(self.el).modal('hide');
            });
        }
    },

    tryPairing: function(){
        this.render();
    },

    hideModal: function(){
        this.hidden = true;
    },

    initialize: function(){
        this.pairingTemplate = swc.Templates.get('telephony:device:pair:modal:pairing');
        this.failTemplate = swc.Templates.get('telephony:device:pair:modal:failed');
        this.successTemplate = swc.Templates.get('telephony:device:pair:modal:complete');
        this.rulerTemplate = swc.Templates.get('ruler');
        this.render();
    },

    render: function(){
        var self = this, element = $(this.el);
        element.html($.tmpl(self.pairingTemplate.get('content'), {
            localeStrings: swc.models.Locale.getLocaleStrings("telephony"),
            localeString: getTranslationStringsjQuery,
            formatDate: swc.models.Locale.formatDate
        }));
        
        //lock close button until startPairing has ended
        element.find(".close-modal").addClass("disabled");
        
        this.$el.modal({keyboard: false, backdrop: 'static'});

        var countDectPhones = function(){
                return swc.models.Telephony.get('phones').length-2;
            },
            dectPhonesCount = countDectPhones();

        element.find('.ruler-block').html($.tmpl(self.rulerTemplate.get('content'), {
            'className': 'pairing-ruler'
        }));

        //swc.models.Login.on('beforeLogout', this.closeModal, this);

        $.when(swc.models.Telephony.startPairing())
            .done(function(){
                var timeEnded = false,
                    pairingTime = 120000,

                    throwSuccess = function(){
                        element.html(
                            $.tmpl(self.successTemplate.get('content'), {
                                localeStrings: swc.models.Locale.getLocaleStrings("telephony"),
                                localeString: getTranslationStringsjQuery,
                                formatDate: swc.models.Locale.formatDate
                            })
                            
                        );
                        // $(self.el).trigger('dectPaired');
                    },

                    throwError = function(){
                        element.html(
                            $.tmpl(self.failTemplate.get('content'), {
                                localeStrings: swc.models.Locale.getLocaleStrings("telephony"),
                                localeString: getTranslationStringsjQuery,
                                formatDate: swc.models.Locale.formatDate
                            })
                        );
                    },

                    checkPaired = function(){
                        $.when(swc.models.Telephony.getPhones()).done(function(){
                            if(countDectPhones() > dectPhonesCount){
                                throwSuccess();
                            } else {
                                if(!timeEnded && !self.hidden){
                                    setTimeout(function(){
                                        checkPaired();
                                    }, 1000);
                                } else {
                                    throwError();
                                }
                            }
                        });
                    };
                element.find('.ruler-container').trigger('swc-ruler:start', {time: pairingTime});
                //if start pairing ended successfully unlock cancel button
                element.find(".close-modal").removeClass("disabled");
                
                setTimeout(function(){
                    timeEnded = true;
                }, pairingTime);

                checkPaired();
            });
    }
});
;swc.constructors.TelephonyDeviceView = swc.base.PageView.extend({

    className: 'dect-and-phones',

    models: ['Telephony', 'System', 'EventDispatcher'],

    validation: {
        'dect-name-editable': 'Telephony:dectName',
        'sip-name': 'Telephony:sipName',
        'internal-number-editable': 'Telephony:internalNumber'
    },

    events: {
        'click .pair-dect': 'pairDect',
        'click .unpair-phone': 'unpairPhone',
        'click .unpair-sip': 'unpairSip',
        'click .ring-phone': 'ringPhone',
        'click .ring-sip': 'ringSip',

        'keyup .dect-name': 'renamePhone',
        'keyup .sip-name': 'renamePhone',

        'swc-checkbox:change .incoming-line': 'changeIncomingNumber',

        'swc-dropdown:change .swc-dropdown.internal-number': 'changeInternalNumber',
        'swc-dropdown:change .swc-dropdown.external-number': 'changeExternalNumber',

        'expandable:edit .expandable-list': 'listEditMode',
        'expandable:no-edit .expandable-list': 'listDefaultMode'
    },
    eventActions: [
      {
        event: 'pairingDone',
        action: 'paringDone'
      },
      {
        event: 'voice_device_updated',
        action: 'updateDectDevices'
      }
    ],

    listenerPages: [
        'telephony'
    ],

    /**
     * List of expandable blocks for different telephony types
     */
    phoneTypes: ['wired', 'dect'],

    /* Indicates validation status */
    isValid : true,
    oldErrorLineStatement : '',
    failOverLoopCount: 0,
    failOverLoopName: 'telephony',

    /**
     * Set data to be passed to template
     */
    setTemplateData: function() {
        var phones = swc.models.Telephony.getPhonesList(),
            phoneLines = swc.models.Telephony.get('voip').where({'enable': 'Enabled'});

        // var phonesSip = this.analyzePhonesSip(phones);

        this.templateData = {
            dectPhones: phones.dect,
            wiredPhones: phones.cord,
            sipPhones: _.where(phones.sip, {isDisabled: false}),
            manyVoips: phoneLines.length >= 2,
            dectStatus: swc.models.Telephony.get('dectStatus'),
            voipStatus: swc.models.Telephony.get('voipStatus'),
            lines: swc.models.Telephony.get('voip'),
            sipExtensions: swc.models.Telephony.get('sipExtensionsStatus')
        };

        this.wiredEditing = null;
        this.dectEditing = null;
    },

    sessionChecking: function(interval) {
        var self = this;

        if (swc.models.Login.checkUserLogin()) {
            $.when(swc.models.System.updateSessionStatePromise()).done(function() {
                var pagesMatch = false,
                pagesRoutes = swc.settings.application.get('navigation'),
                page = Backbone.history.fragment;

                $.each(self.listenerPages, function(key, value) {
                    if (pagesRoutes[value] === page) {
                        pagesMatch = true;
                    }
                });

                if (pagesMatch) {
                    _.delay(function() {self.sessionChecking(interval)}, interval * 1000);
                }
            }).fail(function() {
            });
        }
    },

    /**
     * Redefine render complete function()
     */
    renderComplete: function() {
        var self = this,
        interval = 20;
        self.eventDispatcher = swc.models.EventDispatcher;
        self.eventDispatcher.observ("telephony",self);

        this.setTemplateData();
        this.getTemplateContent();
        this.displayPage();
        this.hideSingleIncomingNumbers();

        _.defer(this.fillDropDowns, this);

        if (self.wiredEditing) {
            this.$('.expandable-wired-phones .expandable-list').trigger('expandable:swc-edit', self.wiredEditing);
        }

        if (self.dectEditing) {
            this.$('.expandable-dect-phones .expandable-list').trigger('expandable:swc-edit', self.dectEditing);
        }

        if (this.showSuccessMessage) {
            this.$('.buttons-container-message .save-success').show();
            this.showSuccessMessage = false;
        }

        this.validationCustom = function() {
            return self.checkIncomingLineValidationError();
        };

        this.oldErrorLineStatement = '';

        this.checkIncomingLineValidationError();

        self.failOverLoopCount = 0;

        function failOverDect(loopCount) {
            $.when(
                swc.models.Telephony.sync(false, self.failOverLoopName),
                swc.models.System.sync(false, self.failOverLoopName)
            ).done(function(rep, rep2) {
                var pagesMatch = false,
                pagesRoutes = swc.settings.application.get('navigation'),
                page = Backbone.history.fragment;

                $.each(self.listenerPages, function(key, value) {
                    if (pagesRoutes[value] === page) {
                        pagesMatch = true;
                    }
                });

                if (pagesMatch) {
                    if(rep === true && rep2 === true) {
                        self.setTemplateData();
                        self.getTemplateContent();
                        self.checkIncomingLineValidationError();
                    } else if(loopCount === self.failOverLoopCount && loopCount < 100) {
                        self.failOverLoopCount = loopCount + 1;
                        failOverDect(loopCount + 1);
                    }
                }
            });
        }

        if(swc.models.Telephony.get('errorStatus') === true || swc.models.System.get('errorStatus') === true) {
            swc.models.Telephony.set('errorStatus', undefined);
            swc.models.System.set('errorStatus', undefined);
            failOverDect(0);
        }

        _.delay(function() {self.sessionChecking(interval)}, interval * 1000);

        if(swc.Utils.getIEVersion() !== 0){
            $('.expandable-list-item .device-type .text').css('position', 'relative');
            $('.expandable-list-item .device-type .text').css('top', '50%');
            $('.expandable-list-item .device-type .text').css('transform', 'translateY(-50%)');
        }
    },

    /**
     * Reset view variables
     */
    emptyEditing: function() {
        this.dectEditing = "";
        this.wiredEditing = "";
    },

    checkIncomingLineValidationError: function() {
        var dectFXSModels = swc.models.Telephony.get('phones').filter(function(model){
                return (model.get('deviceType') === 'wired' || model.get('deviceType') === 'dect') && model.get('type') === 'DECT' && model.get('enable') === true;
            });

        var lineValidationError = false, number = 0;

        _.each(swc.models.Telephony.get('voip').where({'enable': 'Enabled'}), function(line) {
            var lineChecked = false;
            _.each(dectFXSModels, function(model) {
                _.each(model.get('incomingLines'), function(incoming) {
                    if(line.get('directoryNumber') === incoming.number && incoming.value === true) {
                        lineChecked = true;
                    }
                });
            });

            if(lineChecked === false) {
                lineValidationError = true;
                number = line.get('directoryNumber');
            }
        });

        if(lineValidationError === true) {
            var error = '';
            if(this.oldErrorLineStatement === '') {
                this.oldErrorLineStatement = $('.validation-message.incoming-number-min-device .error-message').html();
            }

            error = this.oldErrorLineStatement.replace('%number%', number);
            $('.validation-message.incoming-number-min-device .error-message').html(error);
            $('.validation-message.incoming-number-min-device').show();
            return false;
        } else {
            $('.validation-message.incoming-number-min-device').hide();
            return true;
        }
    },

    /**
     * Event handler: is fired on click on the checkbox next to the incoming number
     *
     * @param e
     * @param value
     */
    changeIncomingNumber: function(e, value){
        var element = $(e.target).closest('.swc-checkbox'),
            line = element.data('line'),
            device = element.closest('.expandable-list-item').data('key'),
            deviceModel = swc.models.Telephony.get('phones').get(device),
            deviceOptions = JSON.stringify(deviceModel.get('incomingLines')),
            deviceOptionsClone = JSON.parse(deviceOptions);

        _.each(deviceOptionsClone, function(obj) {
            if (obj.line === line) {
                obj.value = value;
            }
        });

        deviceModel.set('incomingLines', deviceOptionsClone);

        this.checkIncomingLineValidationError();

        this.setButtonsState();
    },

    /**
     * Show all DOM elements of telephone incoming/outgoing numbers
     * for edit mode of the expandable-list-item
     *
     * @param phoneID {Int} Phone model ID
     */
    showSingleIncomingNumbers: function(phoneID) {
        this.$('.expandable-list-item.' + phoneID + ' .outgoing-number .state-default').toggleClass('hidden', false);
        this.$('.expandable-list-item.' + phoneID + ' .incoming-number').toggleClass('hidden', false);
    },

    /**
     * Switch visibility for DOM elements of telephone incoming/outgoing numbers
     * for default mode of the expandable-list-item
     *
     * @description:
     * The only one number should be displaied
     * if outgoing number equals to incoming number
     */
    hideSingleIncomingNumbers: function() {
        var self = this,
            phonesList = swc.models.Telephony.getPhonesList(),
            phonesSip = _.where(phonesList.sip, {isDisabled: false}),
            phones = _.union(phonesList.dect, phonesList.cord, phonesSip);

        _.each(phones, function(phone) {
            var activeIncomingLines = _.where(phone.incomingLines, {'value': true}),
                activeIncoming = (activeIncomingLines.length > 0),
                singleNumber = (activeIncomingLines.length === 1) && (phone.externalNumber.toString() === activeIncomingLines[0].number.toString());

            self.$('.expandable-list-item.' + phone.id + ' .outgoing-number .left-part .state-default').toggleClass('hidden', singleNumber);
            self.$('.expandable-list-item.' + phone.id + ' .incoming-number .left-part .state-default').toggleClass('hidden', singleNumber );
            self.$('.expandable-list-item.' + phone.id + ' .incoming-number').toggleClass('hidden', !activeIncoming || singleNumber);

            /*
               if more than one incoming line are active
               then visibility for its DOM element values should
               be set according to its model values
            */
            if (activeIncoming && !singleNumber) {
                _.each(phone.incomingLines, function(phoneLine) {
                    self.$('.expandable-list-item.' + phone.id + ' .incoming-number .right-part .state-default.' + phoneLine.line).toggleClass('hidden', !phoneLine.value);
                });
            }
        });
    },

    /*
    * Start pairing process.
    * Before start the session should be checked(just any AJAX call will do)
    */
    pairDect: function() {
        var self = this;

        if($('.pair-dect').hasClass('disabled')) {
            return;
        }

        swc.models.System.checkSessionState(function () {
            self._pairDect();
        });
    },

    /**
     * Pairing process handler: if the amount of connected DECT phones less then allowable limit(4 devices)
     * then open modal window with progress bar and start the connection process themself
     * else open modal window with alert
     *
     * @private
     */
    _pairDect: function() {
        var self = this,
            withoutSip = swc.models.Telephony.get('phones').filter(function(el){
                return el.get('deviceType') !== 'sip';
            });


        if (((withoutSip.length - 2) < swc.models.Telephony.dectInternalNumbers.length) &&
            (swc.models.Telephony.get('phones').where({ deviceType:'dect' }).length < 4)) {

            var pairingModal = new swc.constructors.DectPairingModalView();

            $(pairingModal.el).on('hidden', function(){
                $(pairingModal.el).remove();
                self.render();
            });

            // $(pairingModal.el).on('dectPaired', function(){
            //     _.delay(function() {
            //         $('.close-modal').click();
            //     }, 3000);
            // });
        } else {
            var modal = swc.Templates.get('telephony:device:pair:modal:manyDects').get('content'),
                div = $('<div/>', {'class':'modalWindow modal manyDects-modal'});

            div.html($.tmpl(modal, {
                localeStrings: swc.models.Locale.getLocaleStrings("telephony"),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            }));

            div.on('hidden', function(){
                div.remove();
            });

            div.find('.close-modal').on('click', function() {
                $('.manyDects-modal').modal('hide');
            });

            div.modal();
        }
    },

    /**
     * Unpair selected DECT phone
     *
     * @param e
     */
    unpairPhone: function(e) {
        var self = this,
            dataKey = $(e.target).closest('.expandable-list-item').attr('data-key');

        $.when(swc.models.Telephony.unpairDect(dataKey)).done(function() {
            self.render();
        });
    },

    unpairSip: function(e) {
        var self = this,
            dataKey = $(e.target).closest('.expandable-list-item').attr('data-key');

        $.when(swc.models.Telephony.unpairSip(dataKey)).done(function() {
            self.render();
        });

    },

    /**
     * Make test call
     *
     * @param e
     */
    ringPhone: function(e) {
        var dataKey = $(e.target).closest('.expandable-list-item').attr('data-key');

        swc.models.Telephony.ringPhone(dataKey);
    },

    ringSip: function(e) {
        var dataKey = $(e.target).closest('.expandable-list-item').attr('data-key');

        swc.models.Telephony.ringSip(dataKey);
    },

    /**
     * Rename selected phone
     *
     * @param e
     */
    renamePhone: function(e) {
        var item = $(e.target).closest('.expandable-list-item'),
            dataKey = item.attr('data-key');

        // update the default element with current value
        item.find('.device-name .state-default .text').text(e.target.value);

        swc.models.Telephony.get('phones').get(dataKey).set({
            'name': e.target.value,
            'is_changed': true
        });
    },

    /**
     * Event handler: is fired on change of internal number
     *
     * @param e {Event}
     * @param value {String}
     */
    changeInternalNumber: function(e, value) {
        var element = $(e.target).closest('.expandable-list-item'),
            deviceId = element.attr('data-key');

        // update the default element with current value
        element.find('.default-internal').text(value);

        swc.models.Telephony.get('phones').get(deviceId).set({
            'directoryNumber': value,
            'is_changed': true
        }, { silent: true });

        // refresh phone numbers in the dropdown list
        _.defer(this.fillDropDowns, this);

        this.setButtonsState();
    },

    /**
     * Event handler: is fired on change of external number
     *
     * @param e {Event}
     * @param value {String}
     */
    changeExternalNumber: function(e, value) {
        var element = $(e.target).closest('.expandable-list-item'),
            deviceId = element.attr('data-key');

        // update the default element with current value
        element.find('.default-external').text(value);

        swc.models.Telephony.get('phones').get(deviceId).set({
            externalNumber: value,
            is_changed: true
        }, { silent: true });

        // refresh phone numbers in the dropdown list
        _.defer(this.fillDropDowns, this);

        this.setButtonsState();
    },

    /**
     * This method do the following actions:
     * - closes all sibling item in the current expandable list
     * (i.e. expandable list which contains editable item)
     * - closes all items in the sibling expandable lists
     *
     * @param item {jQquery element}
     * @param phoneType {String} dect | wired
     */
    closeSiblingItems: function(item, phoneType) {
        var self = this,
            restPhoneTypes = _.without(this.phoneTypes, phoneType);

        // Close edit mode of sibling tabs in the expandable list which contains editable item
        _.each(item.siblings('.expandable-list-item'), function(item) {
            var dataKey = $(item).data('key');
            self.$('.expandable-' + phoneType + '-phones .expandable-list').trigger('expandable:swc-no-edit', dataKey);
        });

        // Close edit mode of all tabs in the sibling expandable lists
        // (i.e. lists which don't contain editable item)
        _.each(restPhoneTypes, function(restPhoneType) {
            var expandableList = self.$('.expandable-' + restPhoneType + '-phones .expandable-list');

            _.each(expandableList.find('.expandable-list-item'), function(item) {
                var dataKey = $(item).data('key');
                expandableList.trigger('expandable:swc-no-edit', dataKey);
            });
        });
    },

    /**
     * Event handler: is fired on the end of edit of the list item
     *
     * @param e {Event}
     * @param value {String}
     */
    listDefaultMode: function(e, value) {
        var self = this,
            target = this.$('.expandable-list-item[class~="' + value + '"]'),
            validation = this.pageValidation(null, true);

        $.when(validation)
            .done(function() {
                self.revertItemFromEditMode($(target));
                self.hideSingleIncomingNumbers();
            })
            .fail(function(result) {
                if(result === 'incoming lines') {
                    self.revertItemFromEditMode($(target));
                    self.hideSingleIncomingNumbers();
                } else {
                    target.addClass('mode-editing');
                }
            });
    },

    /**
     * Event handler: is fired on the start of edit of the list item
     *
     * @param e {Event}
     * @param value {String}
     */
    listEditMode: function(e, value) {
        var self = this,
            target = this.$('.expandable-list-item[class~="' + value + '"]'),
            validation = this.pageValidation(null, true),
            targetList;

        // if some of the elements won't pass validation the opportunity
        // to change editable element should be interrupted
        $.when(validation)
            .done(function() {
                if (target.closest('.expandable-list').hasClass('dect')) {
                    targetList = 'dect';
                    self.dectEditing = value;
                } else {
                    targetList = 'wired';
                    self.wiredEditing = value;
                }

                self.setItemToEditMode(target);
                self.closeSiblingItems(target, targetList);
                self.showSingleIncomingNumbers(value);
            })
            .fail(function(result) {
                if(result === 'incoming lines') {
                    if (target.closest('.expandable-list').hasClass('dect')) {
                        targetList = 'dect';
                        self.dectEditing = value;
                    } else {
                        targetList = 'wired';
                        self.wiredEditing = value;
                    }

                    self.setItemToEditMode(target);
                    self.closeSiblingItems(target, targetList);
                    self.showSingleIncomingNumbers(value);
                } else {
                    target.removeClass('mode-editing');
                }
            });
    },

    /**
     * Add suffix '-editable' to elements and appropriate error blocks
     * which are located inside the jQuery element item
     *
     * This manipulation is required to have validation only for elements which
     * are edited at this moment
     *
     * @param item {jQuery element} e.g. $('.expandable-list-item[class~="' + value + '"]')
     */
    setItemToEditMode: function(item) {
        // add '-editable' to dect-name
        item.find('.dect-name').attr('name', 'dect-name-editable');
        item.find('.validation-message[data-parameter-name="dect-name"]').attr('data-parameter-name', 'dect-name-editable');
        // add '-editable' to internal-number
        item.find('.internal-number').attr('data-name', 'internal-number-editable');
        item.find('.internal-number').data('name', 'internal-number-editable');
        item.find('.validation-message[data-parameter-name="internal-number"]').attr('data-parameter-name', 'internal-number-editable');
    },

    /**
     * Remove suffix '-editable' from elements and appropriate error blocks
     * which are located inside the jQuery element item
     *
     * This manipulation is required to have validation only for elements which
     * are edited at this moment
     *
     * @param item {jQuery element} e.g. $('.expandable-list-item[class~="' + value + '"]')
     */
    revertItemFromEditMode: function(item) {
        // remove '-editable' from dect-name
        item.find('.dect-name').attr('name', 'dect-name');
        item.find('.validation-message[data-parameter-name="dect-name-editable"]').attr('data-parameter-name', 'dect-name');
        // remove '-editable' from internal-number
        item.find('.internal-number').attr('data-name', 'internal-number');
        item.find('.internal-number').data('name', 'internal-number');
        item.find('.validation-message[data-parameter-name="internal-number-editable"]').attr('data-parameter-name', 'internal-number');
    },

    /**
     * Fill dropdown menus with phone numbers
     */
    fillDropDowns: function(self) {
        var internalNumbersCopy,
            dectExternalNumbers = [],
            // Possible internal numbers for DECT and SIP phones
            dectInternalNumbers = _.clone(swc.models.Telephony.dectInternalNumbers),
            sipInternalNumbers = _.clone(swc.models.Telephony.sipInternalNumbers),
            wiredPhones = swc.models.Telephony.get('phones').where({deviceType: 'wired'}),
            dectPhones = swc.models.Telephony.get('phones').where({deviceType: 'dect'}),
            sipPhones = swc.models.Telephony.get('phones').where({deviceType: 'sip'}),
            phoneLines = swc.models.Telephony.get('voip').where({'enable': 'Enabled'}),
            usedDectInternalNumbers = _.map(_.pluck(_.pluck(dectPhones,'attributes'), 'directoryNumber'), function(v) {return +v; }),
            usedSipInternalNumbers = _.map(_.pluck(_.where(_.pluck(sipPhones,'attributes'), {enable: true}), 'directoryNumber'), function(v) {return +v; }),
            // internal numbers which are not occupied by any DECT and SIP phone
            freeDectInternalNumbers = _.difference(dectInternalNumbers, usedDectInternalNumbers),
            freeSipInternalNumbers = _.difference(sipInternalNumbers, usedSipInternalNumbers),

            numberDropdownConstructor = function(array) {
                var obj = {};
                _.each(array, function(number) {
                    obj[number] = {
                        'name': number,
                        'value': number.toString()
                    };
                });
                return obj;
            },

            internalNumberFactory = function(list) {
                return function(model) {
                    // Reset current list of internal numbers up to free numbers
                    if(!_.isUndefined(list)) {
                        internalNumbersCopy = _.clone(list);
                        // Add currently used number to array
                        internalNumbersCopy.push(parseInt(model.get('directoryNumber'), 10));
                        internalNumbersCopy.sort();

                        self.$('.swc-dropdown.internal-number[data-class="internal-number-' + model.id + '"]').data({
                            "options": numberDropdownConstructor(internalNumbersCopy)
                        })
                        .trigger('swc-dropdown:swc-change', model.get('directoryNumber'));
                    }

                    // if there is more than one external number
                    if (dectExternalNumbers[1]) {
                        self.$('.swc-dropdown.external-number[data-class="external-number-' + model.id + '"]').data({
                            "options": numberDropdownConstructor(dectExternalNumbers)
                        })
                        .trigger('swc-dropdown:swc-change', model.get('externalNumber'));
                    }
                };
            };


        _.each(phoneLines, function(model) {
            dectExternalNumbers.push(model.get('directoryNumber'));
        });

        _.each(wiredPhones, internalNumberFactory());
        _.each(dectPhones, internalNumberFactory(freeDectInternalNumbers));
        _.each(sipPhones, internalNumberFactory(freeSipInternalNumbers));
    },
    passEvent: function(event, eventObject) {
      swc.constructors.EventController(this.eventActions)
                      .pass.call(this, event, eventObject);
    },
    paringDone: function(event){
        if(event.data.object.attributes.reason === "Canceled") {
            $('.close-modal').click();
        }
    },

    updateDectDevices: function() {
        var self = this;
        $.when(swc.models.Telephony.sync())
                .done(function(){
                    self.render();
                });
    },

    onCancel: function() {
        this.showSuccessMessage = false;
        this.emptyEditing();
    },

    save: function() {
        var self = this,
            deferred = new $.Deferred();

        // Before doing heavy edit operation, which even has a callback delayed for 2 seconds (see phoneEdit() method),
        // let's check if session is still valid.
        swc.models.System.checkSessionState(function(){
            self.$('.expandable-list-item .validation-message').hide();
            self.$('.swc-input').removeClass('validation-error');

            self.emptyEditing();
            self.showPageLoading("Saving page data..");

            $.when(swc.models.Telephony.phoneEdit())
                .done(function () {
                    self.showSuccessMessage = true;
                    // deferred.resolve();
                    deferred.reject();
                })
                .fail(function () {
                    deferred.reject();
                });
        });

        return deferred.promise();
    }
});
;swc.constructors.TelephonyView = swc.base.PageView.extend({

    className: 'telephony'

});;swc.constructors.WifiGuestView = swc.base.PageView.extend({

    className: 'wifi-guest',

    models: [
        "Wireless"
    ],

    disallowedDevices: ['starlink'],

    validation: {
        'ssid': 'Wireless:ssid',
        'password': 'Wireless:password'
    },

    events: {
        'swc-switcher:change .set-wifi-status': 'setWiFiState',

        'swc-checkbox:change .swc-checkbox.show-password': 'setPasswordState',

        'swc-dropdown:change .swc-dropdown.encryption-selection': 'setEncryptionState',
        
        'keyup input[name*=password]': 'copyPassword',
        'change input[name*=password]': 'copyPassword'
    },
    
    timeout: 15 * 1000, // timeout for stabilize status

    setTemplateData: function() {
        var time = new Date(swc.models.Wireless.getParameter("time", "systemTime")).getTime(),
            status = this.getStatus(),
            timeout = this.getTimeout(status),
            date = {
                hours: new Date(time + timeout * 1000).getHours(),
                minutes: new Date(time + timeout * 1000).getMinutes()
            };

        // Set template data:
        this.templateData = {
            "state": status,
            "ssid": swc.models.Wireless.getParameter("name", "Guest-2.4GHz"),
            "password": swc.models.Wireless.getParameter("password", "Guest-2.4GHz"),
            "encryption": swc.models.Wireless.getParameter("encryption", "Guest-2.4GHz"),
            "encryptionOptions": swc.models.Wireless.formatEncryptionOptions("Guest-2.4GHz"),
            "deactivationOptions": swc.models.Wireless.formatDeactivationOptions(),
            "deactivationTimeout": timeout,
            "deactivationSelected": 0,
            "deactivationTime": ''
        };

        // Check hours value for consistensy
        if (date.hours < 10) {
            date.hours = "0" + date.hours;
        }

        // Check minutes value for consistensy
        if (date.minutes < 10) {
            date.minutes = "0" + date.minutes;
        }

        // Update deactivation time value:
        this.templateData["deactivationTime"] = date.hours + ':' + date.minutes;
    },
    

    copyPassword: function(e) {
        var element = $(e.target),
            value = element.val(),
            visibleField,
            hiddenField;

        /**
         * We copy values from currently visible element into currently hidden element
         */
        if (element.hasClass('password-original')) {
            visibleField = this.$('input.password-original');
            hiddenField = this.$('input.password-copy');
        } else {
            visibleField = this.$('input.password-copy');
            hiddenField = this.$('input.password-original');
        }
        
        // validate
        var validate = swc.models.Wireless.validation.password([
            {
                parameterName: 'password',
                parameterValue: value
            },
            {
                parameterName: 'encryption',
                parameterValue: $('.swc-dropdown.encryption-selection').data('value')
            }
        ]);
        $('input[name="password"]').toggleClass('valid', validate.status);

        hiddenField.val(visibleField.val());
    },
    
    getTimeout: function(status) {
        var timeout;
        
        if(status) {
            timeout = swc.models.Wireless.getParameter("timeout", "guestTimer");
        } else {
            timeout = _.findWhere(swc.models.Wireless.formatDeactivationOptions(), { isDefault: true }).value;
        }
        
        return timeout;
    },
    
    getStatus: function() {
        // check if we have record in localStorage
        var me = this,
            result = false,
            lastChange = localStorage.getItem('wlan-guest:lastChange'),
            now = new Date().getTime();
            
        // parse as json
        lastChange = lastChange ? JSON.parse(lastChange) : null;

        if((lastChange !== null) && ((lastChange.timestamp + me.timeout) > now)) {
            // load status from storage and wait for next tick
            me.listenerEnabled = true;
            me.startListener();
            
            result = lastChange.status;
        } else {
            me.stopListener();
            me.listenerEnabled = false;
            
            // remove and reload from Status
            localStorage.removeItem('wlan-guest:lastChange');
            
            result = swc.models.Wireless.getParameter("status", "guestStatus");
        }
        
        return result;
    },

    renderComplete: function() {
        var self = this,
            encryptionDropdown = this.$el.find('.swc-dropdown.encryption-selection'),
            ssidBox = this.$el.find('input.ssid'),
            passwordOrigBox = $('input.password-original'),
            passwordCopyBox = $('input.password-copy');
        
        // Register validation listeners:
        this.registerValidationListener(ssidBox);
        this.registerValidationListener(passwordOrigBox);
        this.registerValidationListener(passwordCopyBox);

        // Listener on Encryption dropdown change
        encryptionDropdown.on('validation-success', function() {

            // Validate visible password box:
            if (passwordOrigBox.css('display') !== 'none') {
                self.pageValidation(passwordOrigBox, false);
            } else {
                self.pageValidation(passwordCopyBox, false);
            }
        });

        // Set encryption data when page is loaded:
        this.setEncryptioData();

        // Set deactivation data when page is loaded:
        this.setDeactivationData();

        this.setShowPasswordState();
        
        //this.updatePageUI();
    },
    
    registerValidationListener: function($element) {
        var self = this;

        // Check if element exists on the page:
        if (!$element.size()) {
            return;
        }

        // Add green mark when success validation happens:
        $element.on('validation-success', function() {
            $element.toggleClass('valid', !_.isEmpty($element.val()));

            self.pageCheckDefaultValues();
            // Re-validate form when all elements set to default state:
            // if (self.pageCheckDefaultValues()) {
                // self.resetValidationState();
            // }
        });

        // Remove green mark when validation fails:
        $element.on('validation-fail', function() {
            $element.removeClass('valid');
        });
    },

    setShowPasswordState: function() {
        var encryption = this.templateData["encryption"],
            $showPasswordCheckbox = this.$('.swc-checkbox.show-password');

        if (encryption === 'None') {
           $showPasswordCheckbox.addClass('disabled').trigger('swc-checkbox:swc-change', false);
        } else {
            this.setPasswordState(null, true);
        }
    },

    setEncryptioData: function() {
        var encryptionDropdown = this.$('.swc-dropdown.encryption-selection');

        // Set dropdown data:
        encryptionDropdown.data('options', this.templateData["encryptionOptions"]);
        encryptionDropdown.trigger('swc-dropdown:swc-change', this.templateData["encryption"]);
        encryptionDropdown.data("default-value", this.templateData["encryption"]);

        // Update password boxes state:
        this.setEncryptionState(null, this.templateData["encryption"]);
    },

    setDeactivationData: function() {
        var dropDownDeactivation = this.$el.find('.swc-dropdown.deactivation-selection');

        // Calculate nearest value:
        if (this.templateData["deactivationTimeout"] > 0 && this.templateData["deactivationTimeout"] <= 21600) {
            this.templateData["deactivationSelected"] = 21600; // 6h
        }
        else if (this.templateData["deactivationTimeout"] > 43200 && this.templateData["deactivationTimeout"] <= 86400) {
            this.templateData["deactivationSelected"] = 86400; // 24h
        }
        else {
            this.templateData["deactivationSelected"] = 0; // No auto disable
        }

        // Set dropdown data:
        dropDownDeactivation.data('options', this.templateData["deactivationOptions"]);
        dropDownDeactivation.data("default-value", this.templateData["deactivationSelected"]);
        dropDownDeactivation.trigger('swc-dropdown:swc-change', this.templateData["deactivationSelected"]);
    },

    setWiFiState: function(/*e, value*/) {
        var self = this/*,
            toDo = []*/;

        // Show page saving message
        this.showPageLoading("Updating Guest WiFi state");
        
        $.when(self.save()).done(function() {
            self.render();
        });

        /*// Set activation timer on wifi activation:
        if (value) {
            toDo = [
                swc.models.Wireless.setGuestStatus(false),
                swc.models.Wireless.setGuestActivationTimer(6)
            ];
        } else {
            toDo.push(
                swc.models.Wireless.setGuestStatus(false)
            );
        }

        // Wait till all changes are completed
        $.when.apply(null, toDo).done(function() {
            $.when(swc.models.Wireless.sync()).done(function() {
                localStorage.setItem('wlan-guest:lastChange', JSON.stringify({ timestamp: new Date().getTime(), status: value }));
                self.render();
            });
        });*/
    },

    setPasswordState: function(e, value) {
        var passwordOrigBox = $('input.password-original'),
            passwordCopyBox = $('input.password-copy');

        if (value) {
            passwordOrigBox.hide();
            passwordCopyBox.show();
        } else {
            passwordCopyBox.hide();
            passwordOrigBox.show();
        }
    },

    setEncryptionState: function(e, value) {
        var passwordOrigBox = $('input.password-original'),
            passwordCopyBox = $('input.password-copy'),
            $showPasswordCheckbox = this.$('.swc-checkbox.show-password');

        if (value === "None") {
            passwordOrigBox.addClass("disabled").attr("disabled", "disabled").removeClass("validatable");
            passwordCopyBox.addClass("disabled").attr("disabled", "disabled").removeClass("validatable");
        } else {
            $showPasswordCheckbox.removeClass('disabled');
            passwordOrigBox.removeClass("disabled").removeAttr("disabled", "disabled").addClass("validatable");
            passwordCopyBox.removeClass("disabled").removeAttr("disabled", "disabled").addClass("validatable");
        }
    },

    save: function() {
        var self = this,
            deferred = new $.Deferred(),
            switcher = $('.set-wifi-status'),
            switcherValue = switcher.data('value'),
            switcherDefault = switcher.data('default-value'),
            timerDropdown = this.$el.find('.swc-dropdown.deactivation-selection'),
            timerDropdownValue = getParameter(timerDropdown),
            toDo = [],
            warningMessages = [],
            pageData = [];

        // Get elements name => value map
        $.each(this.getElements(), function(key, element) {
            pageData.push(getParameter($(element)));
        });
        
        // warningMessages can be filled when IP setting are saved correctly already
        warningMessages = self.checkChangeWarnings(pageData);
        
        /*
         * if there are some warning messages then hide page loading modal window
         * and show information modal window
         */
        if (!_.isEmpty(warningMessages)) {
            self.stopPageLoading();
            self.showPageLoading('Saving changes...');
        }
        
        toDo.push(
            swc.models.Wireless.setGuestStatus(false)
        );

        // Check if timer has changed:
        if ((switcherValue && (switcherValue !== switcherDefault)) || (parseInt(timerDropdown.data('value'), 10) !== parseInt(timerDropdown.data('default-value'), 10))) {
            toDo = [
                swc.models.Wireless.setGuestActivationTimer((timerDropdownValue.parameterValue / 60) / 60)
            ];

            if(switcherValue === false) {
                switcherValue = true;
                switcher.trigger('swc-switcher:swc-change', switcherValue);
            }
        }

        // Push saving state:
        toDo.push(swc.models.Wireless.saveSettingsGuest('2.4GHz', pageData));
        // toDo.push(swc.models.Wireless.saveSettingsGuest('5GHz', pageData)); // Removed due to SAH problems

        $.when.apply(null, toDo).done(function() {
            $.when(swc.models.Wireless.sync()).
            done(function() {
                if (!_.isEmpty(warningMessages)) {
                    self.showWarningModal(deferred, warningMessages[0]);
                } else {
                    localStorage.setItem('wlan-guest:lastChange', JSON.stringify({ timestamp: new Date().getTime(), status: switcherValue }));
                    return deferred.resolve();
                }
            }).fail(function() {
                return deferred.reject();
            });
        }).fail(function() {
                return deferred.reject();
            });

        return deferred.promise();
    },
    
    checkChangeWarnings: function(data) {
        var warningMessages = [];
        
        $.each(data, function(key, element) {
            if ((element.parameterName === "ssid") && (element.parameterValue !== element.parameterData.defaultValue)) {
                warningMessages.push('ssid_warning');
            } else if ((element.parameterName === "password") && (element.parameterValue !== element.parameterData.defaultValue)) {
                warningMessages.push('password_warning');
            } else if ((element.parameterName === "encryption") && (element.parameterValue !== element.parameterData.defaultValue)) {
                warningMessages.push('encryption_warning');
            }
        });

        return warningMessages;
    },

    showWarningModal: function (deferred, messageID) {
        this.stopPageLoading();
        
        SWCElements.modalWindow.show({
            templateID: 'wifi:guest:warning-modal',
            templateData: {
                messageID: messageID
            },
            className: 'network-warning-modal',

            onApply: function() {
                
                SWCElements.modalWindow.hide();

                deferred.resolve();
            }
        });
    }
});
;swc.constructors.WifiSchedulerView = swc.base.PageView.extend({

    className: 'wifi-scheduler',

    listenerInterval: 10 * 60,

    listenerEnabled: true,

    disallowedDevices: ['starlink'],
    
    listenerPages: [
        'wifi-scheduler'
    ],
    
    models: [
        'SchedulerWlan',
        'Wireless',
        'SchedulerPowerSavingManager'
    ],

    events: {
        'swc-checkbox:change .enable-scheduler':'enableScheduler',
        'mouseup': 'destroyPopover'
    },

    destroyPopover: function(e) {
        if((!_.isUndefined(e.srcElement) && e.srcElement.className === "icons has-popover info wifi-scheduler-info layer1 layered selected") || (!_.isUndefined(e.target) && e.target.className === "icons has-popover info wifi-scheduler-info layer1 layered selected")) {
        } else {
            SWCElements.popovers.closeAll(true);
        }
    },

    setTemplateData: function() {
        this.templateData = {
            schedulerState: swc.models.SchedulerWlan.get('enable')
        };
    },

    enableScheduler: function(e, value) {
        var schedulerState = value;

        swc.models.SchedulerWlan.enable(schedulerState);
        this.renderComplete();
    },

    showSaveSuccess: function() {
        this.changedScheduler = false;
        swc.base.PageView.prototype.showSaveSuccess.apply(this, arguments);
    },

    showModalEmptyWlan: function(){
        var deferred = new $.Deferred(),
            self = this;

        SWCElements.modalWindow.show({
            templateID: 'power:wifi:modal',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings('power'),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            },
            className: 'storage-schedule',
            onApply: function() {
                SWCElements.modalWindow.hide();
                self.showPageLoading('Saving page data..');

                $.when(swc.models.SchedulerPowerSavingManager.saveWlan())
                    .done(function () {
                        deferred.resolve();
                    })
                    .fail(function() {
                        deferred.reject();
                    });
            }
        });

        return deferred.promise();
    },

    save: function() {
        var deferred = new $.Deferred();

        if (swc.models.SchedulerWlan.isEmpty()) {
            this.stopPageLoading();
            return this.showModalEmptyWlan();
        } else {
            $.when(swc.models.SchedulerPowerSavingManager.saveWlan())
                .done(function () {
                    deferred.resolve();
                })
                .fail(function() {
                    deferred.reject();
                });
        }

        return deferred.promise();
    },

    onCancel: function() {
        this.changedScheduler = false;
        this.render();
    },

    setButtonsState: function() {
        var element = $(this.el);

        if (this.pageCheckDefaultValues() && !this.changedScheduler) {
            element.find('.buttons-container-message .button').addClass('disabled');
        } else {
            element.find('.buttons-container-message .button').removeClass('disabled');
        }
    },

    renderComplete: function() {
        var self = this,
            schedulerWlan = swc.models.SchedulerWlan;

        swc.constructors.dispatcher.off('scheduler:change');
        swc.constructors.dispatcher.on('scheduler:change', function() {
            self.changedScheduler = true;
            self.setButtonsState();
        });

        self.schedulerDiapazon = new swc.constructors.SchedulerLiner({
            isDisabled: !schedulerWlan.get('enable'),
            scaleWidth: schedulerWlan.widgetWidthInPixels,
            diapazones: schedulerWlan.get('pixelSchedule'),
            model: schedulerWlan,
            schedulerType: 'wifi'
        });

        $(self.el).find('.scheduler-container').html(self.schedulerDiapazon.el);

        self.delegateEvents();

        this.changedScheduler = false;

        this.setButtonsState();
    }
});
;;swc.constructors.WifiSettingsType24View = swc.base.TabView.extend({

    className: 'type-24',

    accessPointInterface: '2.4GHz',

    validation: {
        'ssid': 'Wireless:ssid',
        'password': 'Wireless:password'
    },

    events: {
        'swc-checkbox:change .swc-checkbox.WiFiEnable': 'showStatusChangeWarning',
        'swc-checkbox:change .swc-checkbox.show-password': 'setPasswordState',
        'swc-checkbox:change .swc-checkbox.EnableWPS': 'toggleWPS',
        'swc-radio-buttons:change': 'changeVisibility',

        'swc-dropdown:change .swc-dropdown.channel-selection': 'onChanelChange',
        'swc-dropdown:change .swc-dropdown.encryption-selection': 'setEncryptionState',
        'swc-dropdown:change .swc-dropdown.channel-bandwidth-selection': 'onChannelBandwidthSelection',

        'click .wps-pairing:not(.disabled)': 'startWpsPairing',
        'keyup input[name*=password]': 'copyPassword',
        'change input[name*=password]': 'copyPassword'
    },

    setTemplateData: function() {
        this.templateData = {
            wifiState: swc.models.Wireless.getParameter("status", "status"),
            wifiConfigurationMode: swc.models.Wireless.getParameter("configurationMode", "status"),
            status: swc.models.Wireless.getParameter("status", this.accessPointInterface),
            ssid: swc.models.Wireless.getParameter("name", this.accessPointInterface),
            password: swc.models.Wireless.getParameter("password", this.accessPointInterface),
            visibility: swc.models.Wireless.getParameter("visibility", this.accessPointInterface) ? "visible" : "hidden",
            encryption: swc.models.Wireless.getParameter("encryption", this.accessPointInterface),
            encryptionOptions: swc.models.Wireless.formatEncryptionOptions(this.accessPointInterface),
            channel: swc.models.Wireless.getParameter('channel', this.accessPointInterface),
            channelAutoEnable: swc.models.Wireless.getParameter('channelAutoEnable', this.accessPointInterface),
            channelOptions: swc.models.Wireless.formatChannelsOptions(this.accessPointInterface),
            channelBandwidth: swc.models.Wireless.getParameter('OperatingChannelBandwidth', this.accessPointInterface),
            channelBandwidthOptions: swc.models.Wireless.formatChannelBandwidthOptions(),
            operating: swc.models.Wireless.getParameter('operating', this.accessPointInterface),
            operatingOptions: swc.models.Wireless.formatOperatingOptions(this.accessPointInterface),
            WPSStatus: swc.models.Wireless.getParameter('WPSEnable', 'status'),
            deviceType: swc.Utils.DeviceType
        };

        // For 5 GHz 36,40,44,48 will report 42, so 40 will be set
        if (this.templateData.channel === 42) {
            this.templateData.channel = 40;
        }

        // For 5 GHz 52,56,60,64 will report 58, so 52 will be set
        if (this.templateData.channel === 58) {
            this.templateData.channel = 52;
        }

        // For 5 GHz 100,104,108,112 will report 106, so 100 will be set
        if (this.templateData.channel === 106) {
            this.templateData.channel = 100;
        }
    },

    renderComplete: function() {
        var self = this,
            encryptionDropdown = this.$el.find('.swc-dropdown.encryption-selection'),
            ssidBox = this.$el.find('input.ssid'),
            passwordOrigBox = this.$el.find('input.password-original'),
            passwordCopyBox = this.$el.find('input.password-copy');

        this.setEncryptionData();
        this.setChannelData();
        this.setChannelBandwidth();
        this.setOperatingMode();
        this.setShowPasswordState();
        
        // Register validation listeners:
        this.registerValidationListener(ssidBox);
        this.registerValidationListener(passwordOrigBox);
        this.registerValidationListener(passwordCopyBox);

        // Listener on Encryption dropdown change (Validate visible password box)
        encryptionDropdown.on('validation-success', function() {
            if (passwordOrigBox.css('display') !== 'none') {
                self.pageValidation(passwordOrigBox, false);
            } else {
                self.pageValidation(passwordCopyBox, false);
            }
        });

        var wpsButton = self.$el.find('.button.wps-pairing'),
        myAccessPointInterface = self.accessPointInterface,
        otherAccessPointInterface = self.accessPointInterface === '2.4GHz' ? '5GHz' : '2.4GHz',
        otherVisibility = swc.models.Wireless.getParameter("visibility", otherAccessPointInterface) ? "visible" : "hidden",
        myVisibility = swc.models.Wireless.getParameter("visibility", myAccessPointInterface) ? "visible" : "hidden",
        statusMy = swc.models.Wireless.getParameter("status", myAccessPointInterface),
        statusOther = swc.models.Wireless.getParameter("status", otherAccessPointInterface),
        status = swc.models.Wireless.getParameter("status", "status");

        self._changeVisibilityWPS(wpsButton, status, statusMy, statusOther, myVisibility, otherVisibility);

        localStorage.setItem('accessPointInterface', self.accessPointInterface);
    },
    
    registerValidationListener: function($element) {
        // Check if element exists on the page:
        if (!$element.size()) {
            return;
        }

        // Add green mark when success validation happens:
        $element.on('validation-success', function() {
            $element.toggleClass('valid', !_.isEmpty($element.val()));
        });

        // Remove green mark when validation fails:
        $element.on('validation-fail', function() {
            $element.removeClass('valid');
        });
    },

    setShowPasswordState: function() {
        var encryption = this.templateData["encryption"],
            $showPasswordCheckbox = this.$('.swc-checkbox.show-password');

        if (encryption === 'None') {
           $showPasswordCheckbox.addClass('disabled').trigger('swc-checkbox:swc-change', false);
        }
    },

    _changeVisibilityWPS: function(wpsButton, status, statusMy, statusOther, visibilityMy, visibilityOther) {
        if(status === 'off') {
            wpsButton.addClass('disabled');
        } else if(status === 'on' && (statusMy === false && statusOther === false)) {
            wpsButton.addClass('disabled');
        } else if(status === 'on' && ((statusMy === true && visibilityMy === 'hidden') && (statusOther === false))) {
            wpsButton.addClass('disabled');
        } else if(status === 'on' && ((statusMy === true && visibilityMy === 'hidden') && (statusOther === true && visibilityOther === 'hidden'))) {
            wpsButton.addClass('disabled');
        } else if(status === 'on' && ((statusMy === false) && (statusOther === true && visibilityOther === 'hidden'))) {
            wpsButton.addClass('disabled');
        } else {
            wpsButton.removeClass('disabled');
        }
    },

    changeVisibility: function() {
        var self = this,
        selectedVisibilityRadio = this.$(".swc-radio-buttons.visibility-selection"),
        wpsButton = self.$el.find('.button.wps-pairing'),
        myAccessPointInterface = self.accessPointInterface,
        otherAccessPointInterface = self.accessPointInterface === '2.4GHz' ? '5GHz' : '2.4GHz',
        otherVisibility = swc.models.Wireless.getParameter("visibility", otherAccessPointInterface) ? "visible" : "hidden",
        statusMy = swc.models.Wireless.getParameter("status", myAccessPointInterface),
        statusOther = swc.models.Wireless.getParameter("status", otherAccessPointInterface),
        status = swc.models.Wireless.getParameter("status", "status");

        if(swc.models.Wireless.getParameter("configurationMode", "status") === false) {
            self._changeVisibilityWPS(wpsButton, status, statusMy, statusOther, selectedVisibilityRadio.data("value"), otherVisibility);
        }
        else {
            if(status === 'on' && selectedVisibilityRadio.data("value") === 'visible') {
                wpsButton.removeClass('disabled');
            } else {
                wpsButton.addClass('disabled');
            }
        }
    },

    copyPassword: function(e) {
        var element = $(e.target),
            value = element.val(),
            visibleField,
            hiddenField;

        /**
         * We copy values from currently visible element into currently hidden element
         */
        if (element.hasClass('password-original')) {
            visibleField = this.$('input.password-original');
            hiddenField = this.$('input.password-copy');
        } else {
            visibleField = this.$('input.password-copy');
            hiddenField = this.$('input.password-original');
        }

        if (this.accessPointInterface === "2.4GHz") {
            swc.models.Wireless.validation.setValidationContext('2_4');
        } else {
            swc.models.Wireless.validation.setValidationContext('5_0');
        }
        
        // validate
        var validate = swc.models.Wireless.validation.password([
            {
                parameterName: 'password',
                parameterValue: value
            },
            {
                parameterName: 'encryption',
                parameterValue: $('.swc-dropdown.encryption-selection').data('value')
            }
        ]);
        $('input[name="password"]').toggleClass('valid', validate.status);

        hiddenField.val(visibleField.val());
    },


    /**
     * Handler for channel changes
     *
     * @description:
     *
     *  When user changes channel, and one of watched values has selected -> user must be shown a confirmation dialog:
     *
     *  if user presses "apply" value has to be applied to the dropdown
     *  if user presses "cancel" value has to be reverted to previous one
     *
     * @param e
     * @param value
     */
    onChanelChange: function(e, value) {
        var self = this,
            dropdown = $(e.target),
            previousValue = dropdown.data("previous-value"),
            watchedValues = [ 116, 120, 124, 128 ];

        if (_.indexOf(watchedValues, value) !== -1) {
            SWCElements.modalWindow.show({
                templateID: 'wifi:settings:modal-windows:on-change-chanel',
                templateData: {},
                className: 'wifi-on-change-chanel',

                onCancel: function() {
                    dropdown.trigger('swc-dropdown:swc-change', previousValue);
                    self.setButtonsState();
                },

                onApply: function() {
                    SWCElements.modalWindow.hide();
                    self.setButtonsState();
                }
            });
        }
    },

    setEncryptionData: function() {
        var encryptionDropdown = this.$el.find('.swc-dropdown.encryption-selection');

        // Set dropdown data:
        encryptionDropdown.data('options', this.templateData["encryptionOptions"]);
        encryptionDropdown.trigger('swc-dropdown:swc-change', this.templateData["encryption"]);
        encryptionDropdown.data("default-value", this.templateData["encryption"]);

        // Update password boxes state:
        this.setEncryptionState(null, this.templateData["encryption"]);
    },

    setChannelData: function() {
        var dropDownChannel = this.$el.find('.swc-dropdown.channel-selection');

        // Update dropdown data
        dropDownChannel.data('options', this.templateData["channelOptions"]);

        if (this.templateData["channelAutoEnable"]) {
            dropDownChannel.data("default-value", "auto");
            dropDownChannel.trigger('swc-dropdown:swc-change', "auto");
        } else {
            dropDownChannel.data("default-value", this.templateData["channel"]);
            dropDownChannel.trigger('swc-dropdown:swc-change', this.templateData["channel"]);
        }
    },

    setChannelBandwidth: function () {
        var dropDownChannel = this.$el.find('.swc-dropdown.channel-bandwidth-selection');

        // Update dropdown data
        dropDownChannel.data('options', this.templateData["channelBandwidthOptions"]);

        // set value and change trigger
        dropDownChannel.data("default-value", this.templateData["channelBandwidth"]);
        dropDownChannel.trigger('swc-dropdown:swc-change', this.templateData["channelBandwidth"]);
    },

    setOperatingMode: function() {
        var dropDownOperation = this.$el.find('.swc-dropdown.operating-mode-selection');

        // Update dropdown data
        dropDownOperation.data('options', this.templateData["operatingOptions"]);
        dropDownOperation.data("default-value", this.templateData["operating"]);
        dropDownOperation.trigger('swc-dropdown:swc-change', this.templateData["operating"]);
    },

    setPasswordState: function(e, value) {
        var passwordOrigBox = $('input.password-original'),
            passwordCopyBox = $('input.password-copy');

        if (value) {
            passwordOrigBox.hide();
            passwordCopyBox.show();
        } else {
            passwordCopyBox.hide();
            passwordOrigBox.show();
        }
    },

    setEncryptionState: function(e, value) {
        var passwordOrigBox = $('input.password-original'),
            passwordCopyBox = $('input.password-copy'),
            $showPasswordCheckbox = this.$('.swc-checkbox.show-password');

        if (value === "None") {
            passwordOrigBox.addClass("disabled").attr("disabled", "disabled").removeClass("validatable");
            passwordCopyBox.addClass("disabled").attr("disabled", "disabled").removeClass("validatable");
        } else {
            $showPasswordCheckbox.removeClass('disabled');
            passwordOrigBox.removeClass("disabled").removeAttr("disabled", "disabled").addClass("validatable");
            passwordCopyBox.removeClass("disabled").removeAttr("disabled", "disabled").addClass("validatable");
        }
    },

    onChannelBandwidthSelection: function () {
        var dropDownChannel = this.$el.find('.swc-dropdown.channel-selection');
        dropDownChannel.data('options', this.templateData["channelOptions"]);
        dropDownChannel.data("default-value", "auto");
        dropDownChannel.trigger('swc-dropdown:swc-change', "auto");
    },

    showStatusChangeWarning: function() {
        var self = this,
            checkBox = this.$el.find('.swc-checkbox.WiFiEnable'),
            parameter = getParameter(checkBox);

        if (!parameter.parameterValue) {
            SWCElements.modalWindow.show({
                templateID: 'wifi:settings:modal-windows:turn-' + self.accessPointInterface + '-off',
                templateData: {},
                className: 'wifi-ap-change-state-warning',
                onCancel: function() {
                    checkBox.trigger('swc-checkbox:swc-change', true);
                    self.setButtonsState();
                },

                onApply: function() {
                    var checkBox = self.$el.find('.swc-checkbox.WiFiEnable'),
                    wpsButton = self.$el.find('.button.wps-pairing'),
                    myAccessPointInterface = self.accessPointInterface,
                    otherAccessPointInterface = self.accessPointInterface === '2.4GHz' ? '5GHz' : '2.4GHz',
                    otherVisibility = swc.models.Wireless.getParameter("visibility", otherAccessPointInterface) ? "visible" : "hidden",
                    myVisibility = swc.models.Wireless.getParameter("visibility", myAccessPointInterface) ? "visible" : "hidden",
                    statusOther = swc.models.Wireless.getParameter("status", otherAccessPointInterface),
                    status = swc.models.Wireless.getParameter("status", "status");

                    SWCElements.modalWindow.hide();
                    self.setButtonsState();

                    self._changeVisibilityWPS(wpsButton, status, false, statusOther, myVisibility, otherVisibility);

                    if(status === true) {
                        checkBox.removeClass('disabled');
                    } else {
                        checkBox.addClass('disabled');
                    }
                }
            });
        }
    },

    startWpsPairing: function() {
        // lazy mixin:
        _.extend(this, swc.constructors.WpsPairingMixin);
        this.startWpsPairing();
    },

    toggleWPS: function(event) {
        var self = this,
            target = $(event.target),
            action = target.hasClass('checked') ? 'enable' : 'disable';

         SWCElements.modalWindow.show({
                templateID: 'wifi:settings:wps-'+action,
                templateData: {},
                className: 'wifi-ap-change-state-warning',
                onCancel: function() {
                    target.click();
                    self.setButtonsState();
                },

                onApply: function() {
                    SWCElements.modalWindow.hide();
                    self.setButtonsState();

                    if(action === 'disable') {
                        self.$el.find('.button.wps-pairing').hide();
                        self.$el.find('.inline-blocks.enable-wps-mode').hide();
                    } else {
                        self.$el.find('.button.wps-pairing').show();
                        self.$el.find('.inline-blocks.enable-wps-mode').show();
                    }
                }
            });
    },

    save: function() {
        var self = this,
            deferred = new $.Deferred(),
            pageData = [],
            warningMessages = [],
            actionsToDo = [];

        // Get elements name => value map
        $.each(this.getElements(), function(key, element) {
            pageData.push(getParameter($(element)));
        });

        // warningMessages can be filled when IP setting are saved correctly already
        warningMessages = self.checkChangeWarnings(pageData);
        
        /*
         * if there are some warning messages then hide page loading modal window
         * and show information modal window
         */
        if (!_.isEmpty(warningMessages)) {
            self.stopPageLoading();
            self.showPageLoading('Saving changes...');
        }

        // Define what settings has to be saved:
        if (swc.models.Wireless.getParameter("configurationMode", "status")) {
            actionsToDo = [
                swc.models.Wireless.saveSettingsCombined('2.4GHz', pageData),
                swc.models.Wireless.saveSettingsCombined('5GHz', pageData)
            ];
        } else {
            actionsToDo = [
                swc.models.Wireless.saveSettings(self.accessPointInterface, pageData),
                swc.models.Wireless.setVAPStatus(self.accessPointInterface, pageData)
            ];
        }

        //Add change WPS request
        var WPS = _.findWhere(pageData, {parameterName: "EnableWPS"});

        if(WPS.parameterData.defaultValue !== WPS.parameterData.value) {
            actionsToDo.push(swc.models.Wireless.setWPSState(WPS.parameterData.value));
        }

        // Process save request:
        $.when.apply(null, actionsToDo).done(function() {
            $.when(swc.models.Wireless.sync()).done(function() {
                if (!_.isEmpty(warningMessages)) {
                    self.showWarningModal(deferred, warningMessages[0]);
                } else {
                    return deferred.resolve();
                }
            }).fail(function() {
                    return deferred.reject();
                });
        }).fail(function() {
            return deferred.reject();
        });

        return deferred.promise();
    },
    
    checkChangeWarnings: function(data) {
        var warningMessages = [];
        
        $.each(data, function(key, element) {
            if ((element.parameterName === "ssid") && (element.parameterValue !== element.parameterData.defaultValue)) {
                warningMessages.push('ssid_warning');
            } else if ((element.parameterName === "password") && (element.parameterValue !== element.parameterData.defaultValue)) {
                warningMessages.push('password_warning');
            } else if ((element.parameterName === "encryption") && (element.parameterValue !== element.parameterData.defaultValue)) {
                warningMessages.push('encryption_warning');
            }
        });

        return warningMessages;
    },

    showWarningModal: function (deferred, messageID) {
        this.stopPageLoading();
        
        SWCElements.modalWindow.show({
            templateID: 'wifi:settings:warning-modal',
            templateData: {
                messageID: messageID
            },
            className: 'network-warning-modal',

            onApply: function() {
                
                SWCElements.modalWindow.hide();

                deferred.resolve();
            }
        });
    }
});
;swc.constructors.WifiSettingsType5View = swc.constructors.WifiSettingsType24View.extend({

    className: 'type-5',

    disallowedDevices: ['starlink'],

    accessPointInterface: '5GHz'
});
;swc.constructors.WifiSettingsView = swc.base.PageView.extend({

    className: 'wifi-settings',

    events: {
        'swc-switcher:change .enable-wifi-global': 'setWiFiGlobalState',
        'swc-dropdown:change .swc-dropdown.wifi-mode': 'updateWiFiGlobalMode',
        'click .swc-button-separate': 'switchToSeparateMode'
    },
    eventActions: [
      {
        event: '',
        action: 'setWiFiGlobalState'
      }
    ],

    models: [
        'SchedulerWlan',
        'NetworkDevices',
        'EventDispatcher',
        'Wireless'
    ],

    passEvent: function(event, eventObject) {
      swc.constructors.EventController(this.eventActions)
                      .pass.call(this, event, eventObject);
    },

    setTemplateData: function() {
        this.templateData = {
            wifiState: swc.models.Wireless.getParameter("status", "status"),
            wifiConfigurationMode: swc.models.Wireless.getParameter("configurationMode", "status"),
            wifiSchedulerState: swc.models.SchedulerWlan.get('enable')
        };
    },

    renderComplete: function() {
        this.setWiFiGlobalMode();
        this.eventDispatcher = swc.models.EventDispatcher;
        this.eventDispatcher.observ("wlan",this);
    },

    setWiFiGlobalState: function(e, value) {
        // Show saving page state:
        var self = this;
        this.showPageLoading("Updating WiFi state");

        // Send requests to update wifi status on the device:
        $.when(
                swc.models.Wireless.setGlobalStatus(value)
            ).done(function() {
                $.when(swc.models.Wireless.sync()).done(function() {
                    var allValuesDefault = self.pageCheckDefaultValues(),
                    myAccessPointInterface = localStorage.getItem('accessPointInterface'),
                    checkBox = self.$el.find('.swc-checkbox.WiFiEnable'),
                    wpsButton = self.$el.find('.button.wps-pairing'),
                    otherAccessPointInterface = myAccessPointInterface === '2.4GHz' ? '5GHz' : '2.4GHz',
                    otherVisibility = swc.models.Wireless.getParameter("visibility", otherAccessPointInterface) ? "visible" : "hidden",
                    myVisibility = self.$(".swc-radio-buttons.visibility-selection").data("value"),
                    statusMy = swc.models.Wireless.getParameter("status", myAccessPointInterface),
                    statusOther = swc.models.Wireless.getParameter("status", otherAccessPointInterface);

                    self.stopPageLoading();
                    if(allValuesDefault === false) {
                        swc.router.history.push(Backbone.history.fragment);
                        swc.router.navigate(Backbone.history.fragment);
                    }
                    localStorage.removeItem('wlan-guest:lastChange');

                    if(swc.models.Wireless.getParameter("configurationMode", "status") === false) {
                        if(value === true && ((statusMy === true && myVisibility === 'visible') || (statusOther === true && otherVisibility === 'visible'))) {
                            wpsButton.removeClass('disabled');
                        } else {
                            wpsButton.addClass('disabled');
                        }

                        if(value === true) {
                            checkBox.removeClass('disabled');
                            checkBox.trigger('swc-checkbox:swc-change', true);
                        } else {
                            checkBox.addClass('disabled');
                        }
                    } else {
                        if(value === true && myVisibility === 'visible') {
                            wpsButton.removeClass('disabled');
                        } else {
                            wpsButton.addClass('disabled');
                        }
                    }

                    self.setButtonsState();
                });
            });
    },

    setWiFiGlobalMode: function() {
        var dropDown = this.$el.find(".swc-dropdown.wifi-mode"),
            isEquals = swc.models.Wireless.isAccessPointsEquals('2.4GHz', '5GHz', ['name', 'password', 'encryption', 'visibility', 'status']),
            data = {
                configurationMode: swc.models.Wireless.getParameter("configurationMode", "status"),
                options: swc.models.Wireless.formatModeOptions()
            };

        // TR-069 Upgrade Regression fix (when mode is combined but APs not equals
        if (data.configurationMode && !isEquals) {
            this.resetWiFiGlobalMode('separate');
        } else {
            dropDown.data('options', data["options"]);
            dropDown.data('default-value', data["configurationMode"] ? 'combined' : 'separate');
            dropDown.trigger('swc-dropdown:swc-change', data["configurationMode"] ? 'combined' : 'separate');
        }
    },

    updateWiFiGlobalMode: function(e) {
        var parameter = getParameter($(e.target)),
            dropDown = this.$el.find(".swc-dropdown.wifi-mode"),
            isEquals = swc.models.Wireless.isAccessPointsEquals('2.4GHz', '5GHz', ['name', 'password', 'encryption']),
            encryption = swc.models.Wireless.getParameter("encryption", '5GHz'),
            self = this,
            data = {
                ssid: swc.models.Wireless.getParameter("name", '5GHz'),
                password: swc.models.Wireless.getParameter("password", '5GHz'),
                encryption: ""
            };

        // Pass encryption as localised string
        switch (encryption) {
            case 'None':
                data.encryption = getTranslationStrings("No encryption");
                break;
            case 'WPA2-Personal':
                data.encryption = getTranslationStrings("WPA2");
                break;
            case 'WPA-WPA2-Personal':
                data.encryption = getTranslationStrings("WPA/WPA2");
                break;
            default:
                data.encryption = getTranslationStrings("WPA/WPA2");
                break;
        }

        // Check if current switch selected to combined mode:
        if (parameter.parameterValue === "combined") {
            if (!isEquals) {
                SWCElements.modalWindow.show({
                    templateID: 'wifi:settings:modal-windows:set-combined-mode',
                    templateData: data,
                    className: 'wifi-global-change-mode',

                    onCancel: function() {
                        dropDown.trigger('swc-dropdown:swc-change', 'separate');
                    },

                    onApply: function() {
                        self.resetWiFiGlobalMode('combined');
                    }
                });
            } else {
                self.resetWiFiGlobalMode('combined');
            }
        } else {
            self.resetWiFiGlobalMode('separate');
        }
    },

    resetWiFiGlobalMode: function(mode) {
        var pages = Backbone.history.fragment.split('/'),
            toDo = [
                swc.models.Wireless.setGlobalConfigurationMode(mode === "combined")
            ];

        if(swc.models.Wireless.getParameter("configurationMode", "status") === true && mode === "combined") {
            return;
        }

        // Close current modal window:
        SWCElements.modalWindow.hide();

        // Show page loading message
        this.showPageLoading("Switching Wi-Fi to " + mode + " mode");

        // Switch to selected mode:
        if (mode === "combined") {
            toDo.push(swc.models.Wireless.switchToCombinedMode('2.4GHz'));
        }

        // Save changes to Configuration Mode
        $.when.apply(null, toDo).done(function() {
            swc.router.navigate(pages[0] + "/" + pages[1] + "/" + 'type-24', {skipUnsavedChanges: true});
        });
    },

    switchToSeparateMode: function() {
        var pages = Backbone.history.fragment.split('/'),
            toDo = [
                swc.models.Wireless.setGlobalConfigurationMode(false)
            ];

        // Close current modal window:
        SWCElements.modalWindow.hide();

        // Show page loading message
        this.showPageLoading("Switching Wi-Fi to separate mode");

        // Save changes to Configuration Mode
        $.when.apply(null, toDo).done(function() {
            swc.router.navigate(pages[0] + "/" + pages[1] + "/" + 'type-24', {skipUnsavedChanges: true});
        });
    }
});
;swc.constructors.WpsPairingMixin = {

    startWpsPairing: function() {
        var self = this;

        self.showPageLoading('Starting WPS Pairing...');

        // Request WPS pairing from NP:
        $.when(swc.models.Wireless.startWPSPairing())
            .done(function() {
                self.stopPageLoading();
                self.wpsPairingModal();
            })
            .fail(function() {
                self.wpsPairingFail();
            });
    },

    /**
     * Add listener for pairing Wifi events:
     *  - started
     *  - success
     *  - timeout
     *  - canceled
     *
     *  @description: if the "cancel" button was pushed on the device
     *  then progress bar on the WebUI should be closed
     *  ATTENTION! Upon pressing "cancel" button on the device
     *  modal window with progress bar disappears on the WebUI,
     *  but the message "Try again" is appeared on the device. In
     *  this case the listener doesn't continue to work and
     *  pressing "Select" button on the device won't trigger
     *  modal window on the WebUI
     */
    addEventManagerListener: function() {
        var self = this,
            eventListenerInterval = null,
            eventManager = new swc.constructors.EventManager({ events: [{"handler": "NMC.Wifi"}], eventsVersionToUse: 1 }),
            eventManagerListener = new swc.constructors.EventManagerListener({
                eventsVersionToUse: 1
            });

        $.when(eventManager.openChannel())
            .done(function() {
                eventManagerListener.setAjaxParameters(JSON.stringify({
                    "channelid": eventManager.get('channelid'),
                    "events": [{ "handler": "NMC.Wifi" }]
                }));

                eventManagerListener.on('change', function() {
                    var wifiEvent = eventManagerListener.pop(),
                        state,
                        resetEventState; // function with reset event actions is declared beyond

                    if (_.isEmpty(wifiEvent)) {
                        return;
                    }

                    resetEventState = function() {
                        eventManagerListener.off('change');
                        clearInterval(eventListenerInterval);
                        eventListenerInterval = null;
                    };

                    state = wifiEvent.get('attributes').reason;

                    /*
                     * possible combinations of the event manager response:
                     * reason: pairingReady|110 attributes.reason: Started
                     * reason: pairingDone|111  attributes.reason: Canceled
                     * reason: pairingDone|111  attributes.reason: Success
                     * reason: pairingDone|111  attributes.reason: Timeout
                     *
                     * In this switch is used only "attributes.reason" parameter
                     * because it's different for all "reason" parameters
                     *
                     * The default section is absent because we don't need any
                     * action for some other device responses
                     */
                    switch (state) {
                        case 'Success':
                            resetEventState();
                            self.wpsPairingSuccess();
                            self.trigger('wps-pairing:complete');
                            break;
                        case 'Timeout':
                            resetEventState();
                            self.wpsPairingFail();
                            self.trigger('wps-pairing:complete');
                            break;
                        case 'Canceled':
                            resetEventState();
                            if($('.modalWindow.wps-pairing').css('display') === 'block') {
                                self.trigger('wps-pairing:complete');
                            }
                            break;
                        /*default:
                            resetEventState();
                            self.wpsPairingFail();
                            self.trigger('wps-pairing:complete');
                            break;*/
                    }
                });

                /* for each new eventManagerListener
                   new eventListenerInterval should be created */
                eventListenerInterval = setInterval(function() {
                    eventManagerListener.listenChannel();
                }, 5000);
            })
            .always(function() {
                $('.cancel-changes.wps-pairing').removeClass('disabled');
            });
    },

    wpsPairingModal: function() {
        var self = this;

        SWCElements.modalWindow.show({
            className: 'wps-pairing',
            templateID: 'wifi:wps:modal:pairing',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery
            },

            onShow: function() {
                var element = $('.modalWindow'),
                    pairingTime = 120000,
                    wpsRuler;
                    
                // disable cancel until we do not get confirmation from sg
                $('.cancel-changes.wps-pairing').addClass('disabled');

                self.addEventManagerListener();

                // Insert progress ruller to the template:
                element.find('.ruler-block').html(
                    $.tmpl(swc.Templates.get('ruler').get('content'), {
                        'className': 'reset-ruler'
                    })
                );

                wpsRuler = element.find('.reset-ruler');

                self.once('wps-pairing:complete', function() {
                    wpsRuler.trigger('swc-ruler:finish', true);
                    SWCElements.modalWindow.hide();
                });

                setTimeout(function() {
                    wpsRuler.trigger('swc-ruler:start', { time: pairingTime });
                }, 10);
            },

            onCancel: function() {
                swc.models.Wireless.stopWPSPairing();
            }
        });
    },

    wpsPairingSuccess: function() {
        var self = this;

        SWCElements.modalWindow.show({
            className: 'wps-pairing',
            templateID: 'wifi:wps:modal:complete',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery
            },

            onApply: function() {
                SWCElements.modalWindow.hide();
                swc.router.navigate('network/devices', { trigger: true });
            }
        });
    },

    wpsPairingFail: function() {
        var self = this;

        SWCElements.modalWindow.show({
            className: 'wps-pairing',
            templateID: 'wifi:wps:modal:failed',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery
            },

            onApply: function() {
                SWCElements.modalWindow.hide();
                self.startWpsPairing();
            }
        });
    }
};
