// Some data to work with each test case
var pfRules, validation, helpers;

// It was copy-pasted from PortForwardingRules collection on 2014.01.20
var predefinedServices = {
    "FTP":    { port: "21",   protocol: "6"  },
    "SSH":    { port: "22",   protocol: "6"  },
    "Telnet": { port: "23",   protocol: "6"  },
    "SMTP":   { port: "25",   protocol: "6"  },
    "TFTP":   { port: "69",   protocol: "17" },
    "HTTP":   { port: "80",   protocol: "6"  },
    "HTTPS":  { port: "443",  protocol: "6"  },
    "POP3":   { port: "110",  protocol: "6"  },
    "SNMP":   { port: "161",  protocol: "17" },
    "PPTP":   { port: "1723", protocol: "6"  }
};

beforeEach(function () {
    // reset pfRules collection before every test
    pfRules = new swc.constructors.PortForwarding();
    validation = pfRules.validation;
    validationHelpers = validation.helpers;
    
    // for some tests we have to instantiate swc.models.PortForwarding model
    swc.models.PortForwarding = new swc.constructors.PortForwarding();
});

// Re-set PortForwarding collection with an empty instance
afterEach(function () {
    // for this test we have to instantiate swc.models.PortForwarding model
    swc.models.PortForwarding = new swc.constructors.PortForwarding();
});

describe("Port-Forwarding Collection", function () {
    it("should have correct Model name configured", function () {
        expect(pfRules.model).to.eql(swc.constructors['PortForwardingRule']);
    });
    
    it("should have predefined set of rules", function () {
        expect(pfRules.predefinedServices).to.eql(predefinedServices);
    });
    
    it("should sort Rules by entry port number ASC", function () {
        pfRules.reset([{'ports': {'entry': 10, 'destination': 1}},
                       {'ports': {'entry': 20, 'destination': 30}},
                       {'ports': {'entry': 3, 'destination': 40}}]);
        // here should comparator do it work
        pfRules.sort();
        expect(pfRules.toJSON()).to.eql([{'ports': {'entry': 3, 'destination': 40}},
                                         {'ports': {'entry': 10, 'destination': 1}},
                                         {'ports': {'entry': 20, 'destination': 30}}]);
    });

    //---------------------------------
    // VALIDATION SUBSECTION
    //---------------------------------    
    describe("Validation", function () {
        it("should not allow empty device (with IP address 0.0.0.0)", function () {            
            var data = [{ parameterName: 'deviceIP', parameterValue: '0.0.0.0' }];
            expect(validation.deviceIP(data).messages).to.include('device must be selected');
        });

        it('should not validate predefined rules', function() {
            var data = [
                { parameterName: 'deviceIP', parameterValue: '192.168.11.1' },
                { parameterName: 'service-mode', parameterValue: 'predefined' },
                { parameterName: 'serviceName', parameterValue: '' },
                { parameterName: 'serviceID', parameterValue: '1' }
            ];
            expect(validation.serviceName(data).messages).to.be.empty;
        });
        
        it('should not allow name with spaces only', function () {
            var data = [
                { parameterName: 'service-mode', parameterValue: 'custom' },
                { parameterName: 'serviceName', parameterValue: '          ' }
            ];
            expect(validation.serviceName(data).messages).to.include('must not be empty');
        });
        
        it('should not allow empty rule name or less than 1 char length', function () {
            var data = [
                { parameterName: 'deviceIP', parameterValue: '192.168.11.1' },
                { parameterName: 'service-mode', parameterValue: 'custom' },
                { parameterName: 'serviceName', parameterValue: '' },
                { parameterName: 'serviceID', parameterValue: '1' }
            ];
            
            expect(validation.serviceName(data).messages).to.have
                .members(['must not be empty', 'must be from 1 to 40 characters']);
        });
        
        it('should not allow too long (> 40 chars) name', function () {
            var data = [
                { parameterName: 'deviceIP', parameterValue: '192.168.11.1' },
                { parameterName: 'service-mode', parameterValue: 'custom' },
                { parameterName: 'serviceName', parameterValue: 'very-very-very-long-name-up-to-forty-symbols' },
                { parameterName: 'serviceID', parameterValue: '1' }
            ];
            
            expect(validation.serviceName(data).messages).to.include('must be from 1 to 40 characters');
        });
        
        it('should not allow ASCII non-printable characters as a rule name', function () {
            var data = [
                { parameterName: 'service-mode', parameterValue: 'custom' },                
                { parameterName: 'serviceName', parameterValue: String.fromCharCode(2,10,23,14,31) }
            ];
            
            expect(validation.serviceName(data).messages).to.include('allowed characters ASCI non-printable');
        });
        
        it('should not create 2 rules with the same name', function () {
            // simulate just like we already have rule with ID = 1
            // for this test we have to instantiate swc.models.PortForwarding model
            swc.models.PortForwarding = new swc.constructors.PortForwarding();
            swc.models.PortForwarding.add({'id': 1, 'name': 'Existing rule', 'ports': {'entry': 10}});
            
            // it is like a new rule with the same name
            var data = [
                { parameterName: 'deviceIP', parameterValue: '192.168.11.1' },
                { parameterName: 'service-mode', parameterValue: 'custom' },
                { parameterName: 'serviceName', parameterValue: 'Existing rule' },
                { parameterName: 'serviceID', parameterValue: '2' }
            ];
            
            expect(validation.serviceName(data).messages).to.include('already existed rule names are not allowed');
        });
        
        it('should not allow empty entry or 0 (zero) port', function () {
            var data = [
                { parameterName: 'entry-ports-mode', parameterValue: 'single' },
                { parameterName: 'entryPort', parameterValue: '', parameterClasses: ['from'] },
                { parameterName: 'entryPort', parameterValue: '16', parameterClasses: ['to'] },
                { parameterName: 'serviceID', parameterValue: '1' }
            ];
            
            expect(validation.entryPort(data).messages).to.include('allowed range 1 to 65535');
        });

        it('should not allow value > 65535 for entry (from) port', function () {
            var data = [
                { parameterName: 'entry-ports-mode', parameterValue: 'single' },
                { parameterName: 'entryPort', parameterValue: '65536', parameterClasses: ['from'] },
                { parameterName: 'entryPort', parameterValue: '16', parameterClasses: ['to'] },
                { parameterName: 'serviceID', parameterValue: '1' }
            ];

            expect(validation.entryPort(data).messages).to.include('allowed range 1 to 65535');
        });

        it('should not allow value > 65535 for entry (to) port', function () {
            var data = [
                { parameterName: 'entry-ports-mode', parameterValue: 'range' },
                { parameterName: 'entryPort', parameterValue: '65', parameterClasses: ['from'] },
                { parameterName: 'entryPort', parameterValue: '164758', parameterClasses: ['to'] },
                { parameterName: 'serviceID', parameterValue: '1' }
            ];

            expect(validation.entryPort(data).messages).to.include('allowed range 1 to 65535');
        });
        
        it('should not allow non-digit characters for entry (from) port', function () {
            var data = [
                { parameterName: 'entry-ports-mode', parameterValue: 'single' },
                { parameterName: 'entryPort', parameterValue: 'ab34-', parameterClasses: ['from'] },
                { parameterName: 'entryPort', parameterValue: '16', parameterClasses: ['to'] },
                { parameterName: 'serviceID', parameterValue: '1' }
            ];
            
            expect(validation.entryPort(data).messages).to.include('allowed characters 0-9');
        });

        it('should not allow non-digit characters for entry (to) port', function () {
            var data = [
                { parameterName: 'entry-ports-mode', parameterValue: 'range' },
                { parameterName: 'entryPort', parameterValue: '34', parameterClasses: ['from'] },
                { parameterName: 'entryPort', parameterValue: '16abs', parameterClasses: ['to'] },
                { parameterName: 'serviceID', parameterValue: '1' }
            ];

            expect(validation.entryPort(data).messages).to.include('allowed characters 0-9');
        });

        it('should not validate entry values if port mode is single', function () {
            var data = [
                { parameterName: 'entry-ports-mode', parameterValue: 'single' },
                { parameterName: 'entryPort', parameterValue: '34', parameterClasses: ['from'] },
                { parameterName: 'entryPort', parameterValue: '16abs', parameterClasses: ['to'] },
                { parameterName: 'serviceID', parameterValue: '1' }
            ];

            expect(validation.entryPort(data).messages).not.to.include('allowed characters 0-9');
        });

        it('should check if value of from-port < than value of to-port', function () {
            var data = [
                { parameterName: 'entry-ports-mode', parameterValue: 'range' },
                { parameterName: 'entryPort', parameterValue: '65', parameterClasses: ['from'] },
                { parameterName: 'entryPort', parameterValue: '30', parameterClasses: ['to'] },
                { parameterName: 'serviceID', parameterValue: '1' }
            ];

            expect(validation.entryPort(data).messages).
                to.include('range option first value MUST be less then second value');
        });
    });
    
    describe('Helpers', function () {        
        it('should not allow use occupied/crossed range values', function () {
            expect(validationHelpers.portsNotInRange([1, 5], [3, 6])).to.be.false;
        });
        
        it('should allow not conflicting values', function () {
            expect(validationHelpers.portsNotInRange([1, 5], [6, 7])).to.be.true;
        });
    });
    
    it('should parse response correct', function () {
        
    });
    
    it('should not have ACS provisioned rules in result', function () {
        var fakeResponse = {"status": {
            "webui_not_ups": {
                "Id": 1,
                "Enable": true,
                "Description": "Rule description",
                "DestinationIPAddress": "192.168.1.1",
                "Protocol": "tcp",
                "Origin": "webui",
                "SourceInterface": "webui",
                "ExternalPort": "1-2",
                "InternalPort": "3-4"
            },
            "webui_upnp": {
                "Id": 2,
                "Enable": true,
                "Description": "Second Rule description",
                "DestinationIPAddress": "192.168.1.1",
                "Protocol": "tcp/udp",
                "Origin": "webui",
                "SourceInterface": "webui",
                "ExternalPort": "5-6",
                "InternalPort": "7-8"
            },
            "cwmp_third_rule": {
                "Id": 3,
                "Enable": true,
                "Description": "Third Rule description",
                "DestinationIPAddress": "192.168.1.1",
                "Protocol": "udp",
                "Origin": "cwmp",
                "SourceInterface": "cwmp",
                "ExternalPort": "9-10",
                "InternalPort": "11-12"
            }
        }};
        
        var expectedResult = [
            {
                id: 1,
                isUPnP: false,
                status: true,
                name: "Rule description",
                deviceIP: "192.168.1.1",
                protocol: "tcp",
                origin: "webui",
                sourceInterface: "webui",
                ports: {
                    entry: ['1', '2'],
                    destination: ['3', '4']
                }
            },
            {
                id: 2,
                isUPnP: true,
                status: true,
                name: "Second Rule description",
                deviceIP: "192.168.1.1",
                protocol: "tcp/udp",
                origin: "webui",
                sourceInterface: "webui",
                ports: {
                    entry: ['5', '6'],
                    destination: ['7', '8']
                }
            }
        ];
        
        expect(pfRules.apiToJSON(fakeResponse)).to.eql(expectedResult);
    });
    
});
