describe("Network Collection", function() {


    it("Decimal to IPString and back", function() {
        var validation = (new swc.constructors.Network([])).validation;
            expect( validation.decimalToIpString(validation.ipStringToDecimal("192.168.1.1"))).to.equal("192.168.1.1");
    });

    it("Subnet replace", function() {
        var validation = (new swc.constructors.Network([])).validation;
        expect( validation.replaceSubnet('10.11.1.5', '192.168.1.1', '255.255.0.0')).to.equal("192.168.1.5");
    });

    it("Allowed host range", function() {
        var validation = (new swc.constructors.Network([])).validation;
        expect( 
            validation.getAllowedHostRange(
                validation.ipStringToDecimal('192.168.1.1'),
                validation.ipStringToDecimal('255.255.255.128')
            )
        ).to.equal("192.168.1.1 - 192.168.1.126");
    });

    it("Validation should correctly represent IP as number", function() {
        var validation = (new swc.constructors.Network([])).validation;

        var data = [
            { parameterName: 'Netmask', parameterValue: '255.255.255.0' },
            { parameterName: 'Address', parameterValue: '192.168.1.1' },
            { parameterName: 'Invalid', parameterValue: '192.168.1.a' },
            { parameterName: 'Invalid2', parameterValue: '192.168.1.256' },
            { parameterName: 'Invalid3', parameterValue: '192.168.1.1.2' },
            { parameterName: 'DHCPRange', parameterValue: '192.168.1.100', parameterClasses: ['range-from'] },
            { parameterName: 'DHCPRange', parameterValue: '192.168.1.160', parameterClasses: ['range-to'] }
        ];
        
        expect( validation.getIPField(data, 'Netmask').toString(2)   ).to.equal( "11111111111111111111111100000000");
        expect( validation.getIPField(data, 'Address').toString(2)   ).to.equal( "11000000101010000000000100000001");
        expect( validation.getIPField(data, 'Invalid') ).to.equal(false);
        expect( validation.getIPField(data, 'Invalid2') ).to.equal(false);
        expect( validation.getIPField(data, 'Invalid3') ).to.equal(false);
        expect( validation.getIPField(data, 'DHCPRange', 'range-from').toString(2) ).to.equal( "11000000101010000000000101100100");
        expect( validation.getIPField(data, 'DHCPRange', 'range-to').toString(2)   ).to.equal( "11000000101010000000000110100000");
    });

    it("Netmask validation fails", function() {
        var validation = (new swc.constructors.Network([])).validation;
        var data = [
            { parameterName: 'Netmask', parameterValue: '255.254.0.0' },
            { parameterName: 'Address', parameterValue: '192.168.1.1' }
        ];
        
        expect(validation.validateIPAddress(data).status).to.equal(false);
        expect(validation.validateNetmask(data).status).to.equal(false);
    });

    it("IP out of local subnetwork. RFC 1918", function() {
        var validation = (new swc.constructors.Network([])).validation;
        var data = [
            { parameterName: 'Netmask', parameterValue: '255.255.0.0' },
            { parameterName: 'Address', parameterValue: '11.1.0.0' }
        ];
        
        expect(validation.validateIPAddress(data).status).to.equal(false);
        expect(validation.validateNetmask(data).status).to.equal(false);
    });

    it("Only DHCP is not ok", function() {
        var validation = (new swc.constructors.Network([])).validation;
        var data = [
            { parameterName: 'Netmask', parameterValue: '255.255.255.128' },
            { parameterName: 'Address', parameterValue: '192.168.11.1' },
            { parameterName: 'DHCPRange', parameterValue: '192.168.11.100', parameterClasses: ['range-from'] },
            { parameterName: 'DHCPRange', parameterValue: '192.168.11.99', parameterClasses: ['range-to'] }
        ];
        
        expect(validation.validateIPAddress(data).status).to.equal(true);
        expect(validation.validateNetmask(data).status).to.equal(true);
        expect(validation.validateDHCPSettings(data).status).to.equal(false);
    });

    it("DHCP is out of net", function() {
        var validation = (new swc.constructors.Network([])).validation;
        var data = [
            { parameterName: 'Netmask', parameterValue: '255.255.255.128' },
            { parameterName: 'Address', parameterValue: '192.168.11.1' },
            { parameterName: 'DHCPRange', parameterValue: '192.168.11.100', parameterClasses: ['range-from'] },
            { parameterName: 'DHCPRange', parameterValue: '192.168.11.170', parameterClasses: ['range-to'] }
        ];
        
        expect(validation.validateIPAddress(data).status).to.equal(false);
        expect(validation.validateNetmask(data).status).to.equal(false);
        expect(validation.validateDHCPSettings(data).status).to.equal(false);
    });

    it("Net is not within GUEST wifi", function() {
        var validation = (new swc.constructors.Network([])).validation;
        var data = [
            { parameterName: 'Netmask', parameterValue: '255.255.255.0' },
            { parameterName: 'Address', parameterValue: '10.128.0.1' }
        ];

        expect(validation.validateIPAddress(data).messages).to.contain("Subnet 10.128.0.0/16 is used for guest wlan");
        expect(validation.validateNetmask(data).messages).to.contain("Subnet 10.128.0.0/16 is used for guest wlan");
    });

    it("Net is not within AP-NP network", function() {
        var validation = (new swc.constructors.Network([])).validation;
        var data = [
            { parameterName: 'Netmask', parameterValue: '255.255.0.0' },
            { parameterName: 'Address', parameterValue: '10.129.0.1' }
        ];

        expect(validation.validateIPAddress(data).messages).to.contain("Subnet 10.129.0.0/16 is used for internal purposes");
        expect(validation.validateNetmask(data).messages).to.contain("Subnet 10.129.0.0/16 is used for internal purposes");
    });
});
