//! moment.js
//! version : 2.4.0
//! authors : Tim Wood, Iskren Chernev, Moment.js contributors
//! license : MIT
//! momentjs.com

(function (undefined) {

    /************************************
        Constants
    ************************************/

    var moment,
        VERSION = "2.4.0",
        round = Math.round,
        i,

        YEAR = 0,
        MONTH = 1,
        DATE = 2,
        HOUR = 3,
        MINUTE = 4,
        SECOND = 5,
        MILLISECOND = 6,

        // internal storage for language config files
        languages = {},

        // check for nodeJS
        hasModule = (typeof module !== 'undefined' && module.exports),

        // ASP.NET json date format regex
        aspNetJsonRegex = /^\/?Date\((\-?\d+)/i,
        aspNetTimeSpanJsonRegex = /(\-)?(?:(\d*)\.)?(\d+)\:(\d+)(?:\:(\d+)\.?(\d{3})?)?/,

        // from http://docs.closure-library.googlecode.com/git/closure_goog_date_date.js.source.html
        // somewhat more in line with 4.4.3.2 2004 spec, but allows decimal anywhere
        isoDurationRegex = /^(-)?P(?:(?:([0-9,.]*)Y)?(?:([0-9,.]*)M)?(?:([0-9,.]*)D)?(?:T(?:([0-9,.]*)H)?(?:([0-9,.]*)M)?(?:([0-9,.]*)S)?)?|([0-9,.]*)W)$/,

        // format tokens
        formattingTokens = /(\[[^\[]*\])|(\\)?(Mo|MM?M?M?|Do|DDDo|DD?D?D?|ddd?d?|do?|w[o|w]?|W[o|W]?|YYYYY|YYYY|YY|gg(ggg?)?|GG(GGG?)?|e|E|a|A|hh?|HH?|mm?|ss?|S{1,4}|X|zz?|ZZ?|.)/g,
        localFormattingTokens = /(\[[^\[]*\])|(\\)?(LT|LL?L?L?|l{1,4})/g,

        // parsing token regexes
        parseTokenOneOrTwoDigits = /\d\d?/, // 0 - 99
        parseTokenOneToThreeDigits = /\d{1,3}/, // 0 - 999
        parseTokenThreeDigits = /\d{3}/, // 000 - 999
        parseTokenFourDigits = /\d{1,4}/, // 0 - 9999
        parseTokenSixDigits = /[+\-]?\d{1,6}/, // -999,999 - 999,999
        parseTokenDigits = /\d+/, // nonzero number of digits
        parseTokenWord = /[0-9]*['a-z\u00A0-\u05FF\u0700-\uD7FF\uF900-\uFDCF\uFDF0-\uFFEF]+|[\u0600-\u06FF\/]+(\s*?[\u0600-\u06FF]+){1,2}/i, // any word (or two) characters or numbers including two/three word month in arabic.
        parseTokenTimezone = /Z|[\+\-]\d\d:?\d\d/i, // +00:00 -00:00 +0000 -0000 or Z
        parseTokenT = /T/i, // T (ISO seperator)
        parseTokenTimestampMs = /[\+\-]?\d+(\.\d{1,3})?/, // 123456789 123456789.123

        // preliminary iso regex
        // 0000-00-00 0000-W00 or 0000-W00-0 + T + 00 or 00:00 or 00:00:00 or 00:00:00.000 + +00:00 or +0000)
        isoRegex = /^\s*\d{4}-(?:(\d\d-\d\d)|(W\d\d$)|(W\d\d-\d)|(\d\d\d))((T| )(\d\d(:\d\d(:\d\d(\.\d+)?)?)?)?([\+\-]\d\d:?\d\d|Z)?)?$/,

        isoFormat = 'YYYY-MM-DDTHH:mm:ssZ',

        isoDates = [
            'YYYY-MM-DD',
            'GGGG-[W]WW',
            'GGGG-[W]WW-E',
            'YYYY-DDD'
        ],

        // iso time formats and regexes
        isoTimes = [
            ['HH:mm:ss.SSSS', /(T| )\d\d:\d\d:\d\d\.\d{1,3}/],
            ['HH:mm:ss', /(T| )\d\d:\d\d:\d\d/],
            ['HH:mm', /(T| )\d\d:\d\d/],
            ['HH', /(T| )\d\d/]
        ],

        // timezone chunker "+10:00" > ["10", "00"] or "-1530" > ["-15", "30"]
        parseTimezoneChunker = /([\+\-]|\d\d)/gi,

        // getter and setter names
        proxyGettersAndSetters = 'Date|Hours|Minutes|Seconds|Milliseconds'.split('|'),
        unitMillisecondFactors = {
            'Milliseconds' : 1,
            'Seconds' : 1e3,
            'Minutes' : 6e4,
            'Hours' : 36e5,
            'Days' : 864e5,
            'Months' : 2592e6,
            'Years' : 31536e6
        },

        unitAliases = {
            ms : 'millisecond',
            s : 'second',
            m : 'minute',
            h : 'hour',
            d : 'day',
            D : 'date',
            w : 'week',
            W : 'isoWeek',
            M : 'month',
            y : 'year',
            DDD : 'dayOfYear',
            e : 'weekday',
            E : 'isoWeekday',
            gg: 'weekYear',
            GG: 'isoWeekYear'
        },

        camelFunctions = {
            dayofyear : 'dayOfYear',
            isoweekday : 'isoWeekday',
            isoweek : 'isoWeek',
            weekyear : 'weekYear',
            isoweekyear : 'isoWeekYear'
        },

        // format function strings
        formatFunctions = {},

        // tokens to ordinalize and pad
        ordinalizeTokens = 'DDD w W M D d'.split(' '),
        paddedTokens = 'M D H h m s w W'.split(' '),

        formatTokenFunctions = {
            M    : function () {
                return this.month() + 1;
            },
            MMM  : function (format) {
                return this.lang().monthsShort(this, format);
            },
            MMMM : function (format) {
                return this.lang().months(this, format);
            },
            D    : function () {
                return this.date();
            },
            DDD  : function () {
                return this.dayOfYear();
            },
            d    : function () {
                return this.day();
            },
            dd   : function (format) {
                return this.lang().weekdaysMin(this, format);
            },
            ddd  : function (format) {
                return this.lang().weekdaysShort(this, format);
            },
            dddd : function (format) {
                return this.lang().weekdays(this, format);
            },
            w    : function () {
                return this.week();
            },
            W    : function () {
                return this.isoWeek();
            },
            YY   : function () {
                return leftZeroFill(this.year() % 100, 2);
            },
            YYYY : function () {
                return leftZeroFill(this.year(), 4);
            },
            YYYYY : function () {
                return leftZeroFill(this.year(), 5);
            },
            gg   : function () {
                return leftZeroFill(this.weekYear() % 100, 2);
            },
            gggg : function () {
                return this.weekYear();
            },
            ggggg : function () {
                return leftZeroFill(this.weekYear(), 5);
            },
            GG   : function () {
                return leftZeroFill(this.isoWeekYear() % 100, 2);
            },
            GGGG : function () {
                return this.isoWeekYear();
            },
            GGGGG : function () {
                return leftZeroFill(this.isoWeekYear(), 5);
            },
            e : function () {
                return this.weekday();
            },
            E : function () {
                return this.isoWeekday();
            },
            a    : function () {
                return this.lang().meridiem(this.hours(), this.minutes(), true);
            },
            A    : function () {
                return this.lang().meridiem(this.hours(), this.minutes(), false);
            },
            H    : function () {
                return this.hours();
            },
            h    : function () {
                return this.hours() % 12 || 12;
            },
            m    : function () {
                return this.minutes();
            },
            s    : function () {
                return this.seconds();
            },
            S    : function () {
                return toInt(this.milliseconds() / 100);
            },
            SS   : function () {
                return leftZeroFill(toInt(this.milliseconds() / 10), 2);
            },
            SSS  : function () {
                return leftZeroFill(this.milliseconds(), 3);
            },
            SSSS : function () {
                return leftZeroFill(this.milliseconds(), 3);
            },
            Z    : function () {
                var a = -this.zone(),
                    b = "+";
                if (a < 0) {
                    a = -a;
                    b = "-";
                }
                return b + leftZeroFill(toInt(a / 60), 2) + ":" + leftZeroFill(toInt(a) % 60, 2);
            },
            ZZ   : function () {
                var a = -this.zone(),
                    b = "+";
                if (a < 0) {
                    a = -a;
                    b = "-";
                }
                return b + leftZeroFill(toInt(10 * a / 6), 4);
            },
            z : function () {
                return this.zoneAbbr();
            },
            zz : function () {
                return this.zoneName();
            },
            X    : function () {
                return this.unix();
            }
        },

        lists = ['months', 'monthsShort', 'weekdays', 'weekdaysShort', 'weekdaysMin'];

    function padToken(func, count) {
        return function (a) {
            return leftZeroFill(func.call(this, a), count);
        };
    }
    function ordinalizeToken(func, period) {
        return function (a) {
            return this.lang().ordinal(func.call(this, a), period);
        };
    }

    while (ordinalizeTokens.length) {
        i = ordinalizeTokens.pop();
        formatTokenFunctions[i + 'o'] = ordinalizeToken(formatTokenFunctions[i], i);
    }
    while (paddedTokens.length) {
        i = paddedTokens.pop();
        formatTokenFunctions[i + i] = padToken(formatTokenFunctions[i], 2);
    }
    formatTokenFunctions.DDDD = padToken(formatTokenFunctions.DDD, 3);


    /************************************
        Constructors
    ************************************/

    function Language() {

    }

    // Moment prototype object
    function Moment(config) {
        checkOverflow(config);
        extend(this, config);
    }

    // Duration Constructor
    function Duration(duration) {
        var normalizedInput = normalizeObjectUnits(duration),
            years = normalizedInput.year || 0,
            months = normalizedInput.month || 0,
            weeks = normalizedInput.week || 0,
            days = normalizedInput.day || 0,
            hours = normalizedInput.hour || 0,
            minutes = normalizedInput.minute || 0,
            seconds = normalizedInput.second || 0,
            milliseconds = normalizedInput.millisecond || 0;

        // store reference to input for deterministic cloning
        this._input = duration;

        // representation for dateAddRemove
        this._milliseconds = +milliseconds +
            seconds * 1e3 + // 1000
            minutes * 6e4 + // 1000 * 60
            hours * 36e5; // 1000 * 60 * 60
        // Because of dateAddRemove treats 24 hours as different from a
        // day when working around DST, we need to store them separately
        this._days = +days +
            weeks * 7;
        // It is impossible translate months into days without knowing
        // which months you are are talking about, so we have to store
        // it separately.
        this._months = +months +
            years * 12;

        this._data = {};

        this._bubble();
    }

    /************************************
        Helpers
    ************************************/


    function extend(a, b) {
        for (var i in b) {
            if (b.hasOwnProperty(i)) {
                a[i] = b[i];
            }
        }

        if (b.hasOwnProperty("toString")) {
            a.toString = b.toString;
        }

        if (b.hasOwnProperty("valueOf")) {
            a.valueOf = b.valueOf;
        }

        return a;
    }

    function absRound(number) {
        if (number < 0) {
            return Math.ceil(number);
        } else {
            return Math.floor(number);
        }
    }

    // left zero fill a number
    // see http://jsperf.com/left-zero-filling for performance comparison
    function leftZeroFill(number, targetLength) {
        var output = number + '';
        while (output.length < targetLength) {
            output = '0' + output;
        }
        return output;
    }

    // helper function for _.addTime and _.subtractTime
    function addOrSubtractDurationFromMoment(mom, duration, isAdding, ignoreUpdateOffset) {
        var milliseconds = duration._milliseconds,
            days = duration._days,
            months = duration._months,
            minutes,
            hours;

        if (milliseconds) {
            mom._d.setTime(+mom._d + milliseconds * isAdding);
        }
        // store the minutes and hours so we can restore them
        if (days || months) {
            minutes = mom.minute();
            hours = mom.hour();
        }
        if (days) {
            mom.date(mom.date() + days * isAdding);
        }
        if (months) {
            mom.month(mom.month() + months * isAdding);
        }
        if (milliseconds && !ignoreUpdateOffset) {
            moment.updateOffset(mom);
        }
        // restore the minutes and hours after possibly changing dst
        if (days || months) {
            mom.minute(minutes);
            mom.hour(hours);
        }
    }

    // check if is an array
    function isArray(input) {
        return Object.prototype.toString.call(input) === '[object Array]';
    }

    function isDate(input) {
        return  Object.prototype.toString.call(input) === '[object Date]' ||
                input instanceof Date;
    }

    // compare two arrays, return the number of differences
    function compareArrays(array1, array2, dontConvert) {
        var len = Math.min(array1.length, array2.length),
            lengthDiff = Math.abs(array1.length - array2.length),
            diffs = 0,
            i;
        for (i = 0; i < len; i++) {
            if ((dontConvert && array1[i] !== array2[i]) ||
                (!dontConvert && toInt(array1[i]) !== toInt(array2[i]))) {
                diffs++;
            }
        }
        return diffs + lengthDiff;
    }

    function normalizeUnits(units) {
        if (units) {
            var lowered = units.toLowerCase().replace(/(.)s$/, '$1');
            units = unitAliases[units] || camelFunctions[lowered] || lowered;
        }
        return units;
    }

    function normalizeObjectUnits(inputObject) {
        var normalizedInput = {},
            normalizedProp,
            prop,
            index;

        for (prop in inputObject) {
            if (inputObject.hasOwnProperty(prop)) {
                normalizedProp = normalizeUnits(prop);
                if (normalizedProp) {
                    normalizedInput[normalizedProp] = inputObject[prop];
                }
            }
        }

        return normalizedInput;
    }

    function makeList(field) {
        var count, setter;

        if (field.indexOf('week') === 0) {
            count = 7;
            setter = 'day';
        }
        else if (field.indexOf('month') === 0) {
            count = 12;
            setter = 'month';
        }
        else {
            return;
        }

        moment[field] = function (format, index) {
            var i, getter,
                method = moment.fn._lang[field],
                results = [];

            if (typeof format === 'number') {
                index = format;
                format = undefined;
            }

            getter = function (i) {
                var m = moment().utc().set(setter, i);
                return method.call(moment.fn._lang, m, format || '');
            };

            if (index != null) {
                return getter(index);
            }
            else {
                for (i = 0; i < count; i++) {
                    results.push(getter(i));
                }
                return results;
            }
        };
    }

    function toInt(argumentForCoercion) {
        var coercedNumber = +argumentForCoercion,
            value = 0;

        if (coercedNumber !== 0 && isFinite(coercedNumber)) {
            if (coercedNumber >= 0) {
                value = Math.floor(coercedNumber);
            } else {
                value = Math.ceil(coercedNumber);
            }
        }

        return value;
    }

    function daysInMonth(year, month) {
        return new Date(Date.UTC(year, month + 1, 0)).getUTCDate();
    }

    function daysInYear(year) {
        return isLeapYear(year) ? 366 : 365;
    }

    function isLeapYear(year) {
        return (year % 4 === 0 && year % 100 !== 0) || year % 400 === 0;
    }

    function checkOverflow(m) {
        var overflow;
        if (m._a && m._pf.overflow === -2) {
            overflow =
                m._a[MONTH] < 0 || m._a[MONTH] > 11 ? MONTH :
                m._a[DATE] < 1 || m._a[DATE] > daysInMonth(m._a[YEAR], m._a[MONTH]) ? DATE :
                m._a[HOUR] < 0 || m._a[HOUR] > 23 ? HOUR :
                m._a[MINUTE] < 0 || m._a[MINUTE] > 59 ? MINUTE :
                m._a[SECOND] < 0 || m._a[SECOND] > 59 ? SECOND :
                m._a[MILLISECOND] < 0 || m._a[MILLISECOND] > 999 ? MILLISECOND :
                -1;

            if (m._pf._overflowDayOfYear && (overflow < YEAR || overflow > DATE)) {
                overflow = DATE;
            }

            m._pf.overflow = overflow;
        }
    }

    function initializeParsingFlags(config) {
        config._pf = {
            empty : false,
            unusedTokens : [],
            unusedInput : [],
            overflow : -2,
            charsLeftOver : 0,
            nullInput : false,
            invalidMonth : null,
            invalidFormat : false,
            userInvalidated : false,
            iso: false
        };
    }

    function isValid(m) {
        if (m._isValid == null) {
            m._isValid = !isNaN(m._d.getTime()) &&
                m._pf.overflow < 0 &&
                !m._pf.empty &&
                !m._pf.invalidMonth &&
                !m._pf.nullInput &&
                !m._pf.invalidFormat &&
                !m._pf.userInvalidated;

            if (m._strict) {
                m._isValid = m._isValid &&
                    m._pf.charsLeftOver === 0 &&
                    m._pf.unusedTokens.length === 0;
            }
        }
        return m._isValid;
    }

    function normalizeLanguage(key) {
        return key ? key.toLowerCase().replace('_', '-') : key;
    }

    /************************************
        Languages
    ************************************/


    extend(Language.prototype, {

        set : function (config) {
            var prop, i;
            for (i in config) {
                prop = config[i];
                if (typeof prop === 'function') {
                    this[i] = prop;
                } else {
                    this['_' + i] = prop;
                }
            }
        },

        _months : "January_February_March_April_May_June_July_August_September_October_November_December".split("_"),
        months : function (m) {
            return this._months[m.month()];
        },

        _monthsShort : "Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec".split("_"),
        monthsShort : function (m) {
            return this._monthsShort[m.month()];
        },

        monthsParse : function (monthName) {
            var i, mom, regex;

            if (!this._monthsParse) {
                this._monthsParse = [];
            }

            for (i = 0; i < 12; i++) {
                // make the regex if we don't have it already
                if (!this._monthsParse[i]) {
                    mom = moment.utc([2000, i]);
                    regex = '^' + this.months(mom, '') + '|^' + this.monthsShort(mom, '');
                    this._monthsParse[i] = new RegExp(regex.replace('.', ''), 'i');
                }
                // test the regex
                if (this._monthsParse[i].test(monthName)) {
                    return i;
                }
            }
        },

        _weekdays : "Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday".split("_"),
        weekdays : function (m) {
            return this._weekdays[m.day()];
        },

        _weekdaysShort : "Sun_Mon_Tue_Wed_Thu_Fri_Sat".split("_"),
        weekdaysShort : function (m) {
            return this._weekdaysShort[m.day()];
        },

        _weekdaysMin : "Su_Mo_Tu_We_Th_Fr_Sa".split("_"),
        weekdaysMin : function (m) {
            return this._weekdaysMin[m.day()];
        },

        weekdaysParse : function (weekdayName) {
            var i, mom, regex;

            if (!this._weekdaysParse) {
                this._weekdaysParse = [];
            }

            for (i = 0; i < 7; i++) {
                // make the regex if we don't have it already
                if (!this._weekdaysParse[i]) {
                    mom = moment([2000, 1]).day(i);
                    regex = '^' + this.weekdays(mom, '') + '|^' + this.weekdaysShort(mom, '') + '|^' + this.weekdaysMin(mom, '');
                    this._weekdaysParse[i] = new RegExp(regex.replace('.', ''), 'i');
                }
                // test the regex
                if (this._weekdaysParse[i].test(weekdayName)) {
                    return i;
                }
            }
        },

        _longDateFormat : {
            LT : "h:mm A",
            L : "MM/DD/YYYY",
            LL : "MMMM D YYYY",
            LLL : "MMMM D YYYY LT",
            LLLL : "dddd, MMMM D YYYY LT"
        },
        longDateFormat : function (key) {
            var output = this._longDateFormat[key];
            if (!output && this._longDateFormat[key.toUpperCase()]) {
                output = this._longDateFormat[key.toUpperCase()].replace(/MMMM|MM|DD|dddd/g, function (val) {
                    return val.slice(1);
                });
                this._longDateFormat[key] = output;
            }
            return output;
        },

        isPM : function (input) {
            // IE8 Quirks Mode & IE7 Standards Mode do not allow accessing strings like arrays
            // Using charAt should be more compatible.
            return ((input + '').toLowerCase().charAt(0) === 'p');
        },

        _meridiemParse : /[ap]\.?m?\.?/i,
        meridiem : function (hours, minutes, isLower) {
            if (hours > 11) {
                return isLower ? 'pm' : 'PM';
            } else {
                return isLower ? 'am' : 'AM';
            }
        },

        _calendar : {
            sameDay : '[Today at] LT',
            nextDay : '[Tomorrow at] LT',
            nextWeek : 'dddd [at] LT',
            lastDay : '[Yesterday at] LT',
            lastWeek : '[Last] dddd [at] LT',
            sameElse : 'L'
        },
        calendar : function (key, mom) {
            var output = this._calendar[key];
            return typeof output === 'function' ? output.apply(mom) : output;
        },

        _relativeTime : {
            future : "in %s",
            past : "%s ago",
            s : "a few seconds",
            m : "a minute",
            mm : "%d minutes",
            h : "an hour",
            hh : "%d hours",
            d : "a day",
            dd : "%d days",
            M : "a month",
            MM : "%d months",
            y : "a year",
            yy : "%d years"
        },
        relativeTime : function (number, withoutSuffix, string, isFuture) {
            var output = this._relativeTime[string];
            return (typeof output === 'function') ?
                output(number, withoutSuffix, string, isFuture) :
                output.replace(/%d/i, number);
        },
        pastFuture : function (diff, output) {
            var format = this._relativeTime[diff > 0 ? 'future' : 'past'];
            return typeof format === 'function' ? format(output) : format.replace(/%s/i, output);
        },

        ordinal : function (number) {
            return this._ordinal.replace("%d", number);
        },
        _ordinal : "%d",

        preparse : function (string) {
            return string;
        },

        postformat : function (string) {
            return string;
        },

        week : function (mom) {
            return weekOfYear(mom, this._week.dow, this._week.doy).week;
        },

        _week : {
            dow : 0, // Sunday is the first day of the week.
            doy : 6  // The week that contains Jan 1st is the first week of the year.
        },

        _invalidDate: 'Invalid date',
        invalidDate: function () {
            return this._invalidDate;
        }
    });

    // Loads a language definition into the `languages` cache.  The function
    // takes a key and optionally values.  If not in the browser and no values
    // are provided, it will load the language file module.  As a convenience,
    // this function also returns the language values.
    function loadLang(key, values) {
        values.abbr = key;
        if (!languages[key]) {
            languages[key] = new Language();
        }
        languages[key].set(values);
        return languages[key];
    }

    // Remove a language from the `languages` cache. Mostly useful in tests.
    function unloadLang(key) {
        delete languages[key];
    }

    // Determines which language definition to use and returns it.
    //
    // With no parameters, it will return the global language.  If you
    // pass in a language key, such as 'en', it will return the
    // definition for 'en', so long as 'en' has already been loaded using
    // moment.lang.
    function getLangDefinition(key) {
        var i = 0, j, lang, next, split,
            get = function (k) {
                if (!languages[k] && hasModule) {
                    try {
                        require('./lang/' + k);
                    } catch (e) { }
                }
                return languages[k];
            };

        if (!key) {
            return moment.fn._lang;
        }

        if (!isArray(key)) {
            //short-circuit everything else
            lang = get(key);
            if (lang) {
                return lang;
            }
            key = [key];
        }

        //pick the language from the array
        //try ['en-au', 'en-gb'] as 'en-au', 'en-gb', 'en', as in move through the list trying each
        //substring from most specific to least, but move to the next array item if it's a more specific variant than the current root
        while (i < key.length) {
            split = normalizeLanguage(key[i]).split('-');
            j = split.length;
            next = normalizeLanguage(key[i + 1]);
            next = next ? next.split('-') : null;
            while (j > 0) {
                lang = get(split.slice(0, j).join('-'));
                if (lang) {
                    return lang;
                }
                if (next && next.length >= j && compareArrays(split, next, true) >= j - 1) {
                    //the next array item is better than a shallower substring of this one
                    break;
                }
                j--;
            }
            i++;
        }
        return moment.fn._lang;
    }

    /************************************
        Formatting
    ************************************/


    function removeFormattingTokens(input) {
        if (input.match(/\[[\s\S]/)) {
            return input.replace(/^\[|\]$/g, "");
        }
        return input.replace(/\\/g, "");
    }

    function makeFormatFunction(format) {
        var array = format.match(formattingTokens), i, length;

        for (i = 0, length = array.length; i < length; i++) {
            if (formatTokenFunctions[array[i]]) {
                array[i] = formatTokenFunctions[array[i]];
            } else {
                array[i] = removeFormattingTokens(array[i]);
            }
        }

        return function (mom) {
            var output = "";
            for (i = 0; i < length; i++) {
                output += array[i] instanceof Function ? array[i].call(mom, format) : array[i];
            }
            return output;
        };
    }

    // format date using native date object
    function formatMoment(m, format) {

        if (!m.isValid()) {
            return m.lang().invalidDate();
        }

        format = expandFormat(format, m.lang());

        if (!formatFunctions[format]) {
            formatFunctions[format] = makeFormatFunction(format);
        }

        return formatFunctions[format](m);
    }

    function expandFormat(format, lang) {
        var i = 5;

        function replaceLongDateFormatTokens(input) {
            return lang.longDateFormat(input) || input;
        }

        localFormattingTokens.lastIndex = 0;
        while (i >= 0 && localFormattingTokens.test(format)) {
            format = format.replace(localFormattingTokens, replaceLongDateFormatTokens);
            localFormattingTokens.lastIndex = 0;
            i -= 1;
        }

        return format;
    }


    /************************************
        Parsing
    ************************************/


    // get the regex to find the next token
    function getParseRegexForToken(token, config) {
        var a;
        switch (token) {
        case 'DDDD':
            return parseTokenThreeDigits;
        case 'YYYY':
        case 'GGGG':
        case 'gggg':
            return parseTokenFourDigits;
        case 'YYYYY':
        case 'GGGGG':
        case 'ggggg':
            return parseTokenSixDigits;
        case 'S':
        case 'SS':
        case 'SSS':
        case 'DDD':
            return parseTokenOneToThreeDigits;
        case 'MMM':
        case 'MMMM':
        case 'dd':
        case 'ddd':
        case 'dddd':
            return parseTokenWord;
        case 'a':
        case 'A':
            return getLangDefinition(config._l)._meridiemParse;
        case 'X':
            return parseTokenTimestampMs;
        case 'Z':
        case 'ZZ':
            return parseTokenTimezone;
        case 'T':
            return parseTokenT;
        case 'SSSS':
            return parseTokenDigits;
        case 'MM':
        case 'DD':
        case 'YY':
        case 'GG':
        case 'gg':
        case 'HH':
        case 'hh':
        case 'mm':
        case 'ss':
        case 'M':
        case 'D':
        case 'd':
        case 'H':
        case 'h':
        case 'm':
        case 's':
        case 'w':
        case 'ww':
        case 'W':
        case 'WW':
        case 'e':
        case 'E':
            return parseTokenOneOrTwoDigits;
        default :
            a = new RegExp(regexpEscape(unescapeFormat(token.replace('\\', '')), "i"));
            return a;
        }
    }

    function timezoneMinutesFromString(string) {
        var tzchunk = (parseTokenTimezone.exec(string) || [])[0],
            parts = (tzchunk + '').match(parseTimezoneChunker) || ['-', 0, 0],
            minutes = +(parts[1] * 60) + toInt(parts[2]);

        return parts[0] === '+' ? -minutes : minutes;
    }

    // function to convert string input to date
    function addTimeToArrayFromToken(token, input, config) {
        var a, datePartArray = config._a;

        switch (token) {
        // MONTH
        case 'M' : // fall through to MM
        case 'MM' :
            if (input != null) {
                datePartArray[MONTH] = toInt(input) - 1;
            }
            break;
        case 'MMM' : // fall through to MMMM
        case 'MMMM' :
            a = getLangDefinition(config._l).monthsParse(input);
            // if we didn't find a month name, mark the date as invalid.
            if (a != null) {
                datePartArray[MONTH] = a;
            } else {
                config._pf.invalidMonth = input;
            }
            break;
        // DAY OF MONTH
        case 'D' : // fall through to DD
        case 'DD' :
            if (input != null) {
                datePartArray[DATE] = toInt(input);
            }
            break;
        // DAY OF YEAR
        case 'DDD' : // fall through to DDDD
        case 'DDDD' :
            if (input != null) {
                config._dayOfYear = toInt(input);
            }

            break;
        // YEAR
        case 'YY' :
            datePartArray[YEAR] = toInt(input) + (toInt(input) > 68 ? 1900 : 2000);
            break;
        case 'YYYY' :
        case 'YYYYY' :
            datePartArray[YEAR] = toInt(input);
            break;
        // AM / PM
        case 'a' : // fall through to A
        case 'A' :
            config._isPm = getLangDefinition(config._l).isPM(input);
            break;
        // 24 HOUR
        case 'H' : // fall through to hh
        case 'HH' : // fall through to hh
        case 'h' : // fall through to hh
        case 'hh' :
            datePartArray[HOUR] = toInt(input);
            break;
        // MINUTE
        case 'm' : // fall through to mm
        case 'mm' :
            datePartArray[MINUTE] = toInt(input);
            break;
        // SECOND
        case 's' : // fall through to ss
        case 'ss' :
            datePartArray[SECOND] = toInt(input);
            break;
        // MILLISECOND
        case 'S' :
        case 'SS' :
        case 'SSS' :
        case 'SSSS' :
            datePartArray[MILLISECOND] = toInt(('0.' + input) * 1000);
            break;
        // UNIX TIMESTAMP WITH MS
        case 'X':
            config._d = new Date(parseFloat(input) * 1000);
            break;
        // TIMEZONE
        case 'Z' : // fall through to ZZ
        case 'ZZ' :
            config._useUTC = true;
            config._tzm = timezoneMinutesFromString(input);
            break;
        case 'w':
        case 'ww':
        case 'W':
        case 'WW':
        case 'd':
        case 'dd':
        case 'ddd':
        case 'dddd':
        case 'e':
        case 'E':
            token = token.substr(0, 1);
            /* falls through */
        case 'gg':
        case 'gggg':
        case 'GG':
        case 'GGGG':
        case 'GGGGG':
            token = token.substr(0, 2);
            if (input) {
                config._w = config._w || {};
                config._w[token] = input;
            }
            break;
        }
    }

    // convert an array to a date.
    // the array should mirror the parameters below
    // note: all values past the year are optional and will default to the lowest possible value.
    // [year, month, day , hour, minute, second, millisecond]
    function dateFromConfig(config) {
        var i, date, input = [], currentDate,
            yearToUse, fixYear, w, temp, lang, weekday, week;

        if (config._d) {
            return;
        }

        currentDate = currentDateArray(config);

        //compute day of the year from weeks and weekdays
        if (config._w && config._a[DATE] == null && config._a[MONTH] == null) {
            fixYear = function (val) {
                return val ?
                  (val.length < 3 ? (parseInt(val, 10) > 68 ? '19' + val : '20' + val) : val) :
                  (config._a[YEAR] == null ? moment().weekYear() : config._a[YEAR]);
            };

            w = config._w;
            if (w.GG != null || w.W != null || w.E != null) {
                temp = dayOfYearFromWeeks(fixYear(w.GG), w.W || 1, w.E, 4, 1);
            }
            else {
                lang = getLangDefinition(config._l);
                weekday = w.d != null ?  parseWeekday(w.d, lang) :
                  (w.e != null ?  parseInt(w.e, 10) + lang._week.dow : 0);

                week = parseInt(w.w, 10) || 1;

                //if we're parsing 'd', then the low day numbers may be next week
                if (w.d != null && weekday < lang._week.dow) {
                    week++;
                }

                temp = dayOfYearFromWeeks(fixYear(w.gg), week, weekday, lang._week.doy, lang._week.dow);
            }

            config._a[YEAR] = temp.year;
            config._dayOfYear = temp.dayOfYear;
        }

        //if the day of the year is set, figure out what it is
        if (config._dayOfYear) {
            yearToUse = config._a[YEAR] == null ? currentDate[YEAR] : config._a[YEAR];

            if (config._dayOfYear > daysInYear(yearToUse)) {
                config._pf._overflowDayOfYear = true;
            }

            date = makeUTCDate(yearToUse, 0, config._dayOfYear);
            config._a[MONTH] = date.getUTCMonth();
            config._a[DATE] = date.getUTCDate();
        }

        // Default to current date.
        // * if no year, month, day of month are given, default to today
        // * if day of month is given, default month and year
        // * if month is given, default only year
        // * if year is given, don't default anything
        for (i = 0; i < 3 && config._a[i] == null; ++i) {
            config._a[i] = input[i] = currentDate[i];
        }

        // Zero out whatever was not defaulted, including time
        for (; i < 7; i++) {
            config._a[i] = input[i] = (config._a[i] == null) ? (i === 2 ? 1 : 0) : config._a[i];
        }

        // add the offsets to the time to be parsed so that we can have a clean array for checking isValid
        input[HOUR] += toInt((config._tzm || 0) / 60);
        input[MINUTE] += toInt((config._tzm || 0) % 60);

        config._d = (config._useUTC ? makeUTCDate : makeDate).apply(null, input);
    }

    function dateFromObject(config) {
        var normalizedInput;

        if (config._d) {
            return;
        }

        normalizedInput = normalizeObjectUnits(config._i);
        config._a = [
            normalizedInput.year,
            normalizedInput.month,
            normalizedInput.day,
            normalizedInput.hour,
            normalizedInput.minute,
            normalizedInput.second,
            normalizedInput.millisecond
        ];

        dateFromConfig(config);
    }

    function currentDateArray(config) {
        var now = new Date();
        if (config._useUTC) {
            return [
                now.getUTCFullYear(),
                now.getUTCMonth(),
                now.getUTCDate()
            ];
        } else {
            return [now.getFullYear(), now.getMonth(), now.getDate()];
        }
    }

    // date from string and format string
    function makeDateFromStringAndFormat(config) {

        config._a = [];
        config._pf.empty = true;

        // This array is used to make a Date, either with `new Date` or `Date.UTC`
        var lang = getLangDefinition(config._l),
            string = '' + config._i,
            i, parsedInput, tokens, token, skipped,
            stringLength = string.length,
            totalParsedInputLength = 0;

        tokens = expandFormat(config._f, lang).match(formattingTokens) || [];

        for (i = 0; i < tokens.length; i++) {
            token = tokens[i];
            parsedInput = (getParseRegexForToken(token, config).exec(string) || [])[0];
            if (parsedInput) {
                skipped = string.substr(0, string.indexOf(parsedInput));
                if (skipped.length > 0) {
                    config._pf.unusedInput.push(skipped);
                }
                string = string.slice(string.indexOf(parsedInput) + parsedInput.length);
                totalParsedInputLength += parsedInput.length;
            }
            // don't parse if it's not a known token
            if (formatTokenFunctions[token]) {
                if (parsedInput) {
                    config._pf.empty = false;
                }
                else {
                    config._pf.unusedTokens.push(token);
                }
                addTimeToArrayFromToken(token, parsedInput, config);
            }
            else if (config._strict && !parsedInput) {
                config._pf.unusedTokens.push(token);
            }
        }

        // add remaining unparsed input length to the string
        config._pf.charsLeftOver = stringLength - totalParsedInputLength;
        if (string.length > 0) {
            config._pf.unusedInput.push(string);
        }

        // handle am pm
        if (config._isPm && config._a[HOUR] < 12) {
            config._a[HOUR] += 12;
        }
        // if is 12 am, change hours to 0
        if (config._isPm === false && config._a[HOUR] === 12) {
            config._a[HOUR] = 0;
        }

        dateFromConfig(config);
        checkOverflow(config);
    }

    function unescapeFormat(s) {
        return s.replace(/\\(\[)|\\(\])|\[([^\]\[]*)\]|\\(.)/g, function (matched, p1, p2, p3, p4) {
            return p1 || p2 || p3 || p4;
        });
    }

    // Code from http://stackoverflow.com/questions/3561493/is-there-a-regexp-escape-function-in-javascript
    function regexpEscape(s) {
        return s.replace(/[-\/\\^$*+?.()|[\]{}]/g, '\\$&');
    }

    // date from string and array of format strings
    function makeDateFromStringAndArray(config) {
        var tempConfig,
            bestMoment,

            scoreToBeat,
            i,
            currentScore;

        if (config._f.length === 0) {
            config._pf.invalidFormat = true;
            config._d = new Date(NaN);
            return;
        }

        for (i = 0; i < config._f.length; i++) {
            currentScore = 0;
            tempConfig = extend({}, config);
            initializeParsingFlags(tempConfig);
            tempConfig._f = config._f[i];
            makeDateFromStringAndFormat(tempConfig);

            if (!isValid(tempConfig)) {
                continue;
            }

            // if there is any input that was not parsed add a penalty for that format
            currentScore += tempConfig._pf.charsLeftOver;

            //or tokens
            currentScore += tempConfig._pf.unusedTokens.length * 10;

            tempConfig._pf.score = currentScore;

            if (scoreToBeat == null || currentScore < scoreToBeat) {
                scoreToBeat = currentScore;
                bestMoment = tempConfig;
            }
        }

        extend(config, bestMoment || tempConfig);
    }

    // date from iso format
    function makeDateFromString(config) {
        var i,
            string = config._i,
            match = isoRegex.exec(string);

        if (match) {
            config._pf.iso = true;
            for (i = 4; i > 0; i--) {
                if (match[i]) {
                    // match[5] should be "T" or undefined
                    config._f = isoDates[i - 1] + (match[6] || " ");
                    break;
                }
            }
            for (i = 0; i < 4; i++) {
                if (isoTimes[i][1].exec(string)) {
                    config._f += isoTimes[i][0];
                    break;
                }
            }
            if (parseTokenTimezone.exec(string)) {
                config._f += "Z";
            }
            makeDateFromStringAndFormat(config);
        }
        else {
            config._d = new Date(string);
        }
    }

    function makeDateFromInput(config) {
        var input = config._i,
            matched = aspNetJsonRegex.exec(input);

        if (input === undefined) {
            config._d = new Date();
        } else if (matched) {
            config._d = new Date(+matched[1]);
        } else if (typeof input === 'string') {
            makeDateFromString(config);
        } else if (isArray(input)) {
            config._a = input.slice(0);
            dateFromConfig(config);
        } else if (isDate(input)) {
            config._d = new Date(+input);
        } else if (typeof(input) === 'object') {
            dateFromObject(config);
        } else {
            config._d = new Date(input);
        }
    }

    function makeDate(y, m, d, h, M, s, ms) {
        //can't just apply() to create a date:
        //http://stackoverflow.com/questions/181348/instantiating-a-javascript-object-by-calling-prototype-constructor-apply
        var date = new Date(y, m, d, h, M, s, ms);

        //the date constructor doesn't accept years < 1970
        if (y < 1970) {
            date.setFullYear(y);
        }
        return date;
    }

    function makeUTCDate(y) {
        var date = new Date(Date.UTC.apply(null, arguments));
        if (y < 1970) {
            date.setUTCFullYear(y);
        }
        return date;
    }

    function parseWeekday(input, language) {
        if (typeof input === 'string') {
            if (!isNaN(input)) {
                input = parseInt(input, 10);
            }
            else {
                input = language.weekdaysParse(input);
                if (typeof input !== 'number') {
                    return null;
                }
            }
        }
        return input;
    }

    /************************************
        Relative Time
    ************************************/


    // helper function for moment.fn.from, moment.fn.fromNow, and moment.duration.fn.humanize
    function substituteTimeAgo(string, number, withoutSuffix, isFuture, lang) {
        return lang.relativeTime(number || 1, !!withoutSuffix, string, isFuture);
    }

    function relativeTime(milliseconds, withoutSuffix, lang) {
        var seconds = round(Math.abs(milliseconds) / 1000),
            minutes = round(seconds / 60),
            hours = round(minutes / 60),
            days = round(hours / 24),
            years = round(days / 365),
            args = seconds < 45 && ['s', seconds] ||
                minutes === 1 && ['m'] ||
                minutes < 45 && ['mm', minutes] ||
                hours === 1 && ['h'] ||
                hours < 22 && ['hh', hours] ||
                days === 1 && ['d'] ||
                days <= 25 && ['dd', days] ||
                days <= 45 && ['M'] ||
                days < 345 && ['MM', round(days / 30)] ||
                years === 1 && ['y'] || ['yy', years];
        args[2] = withoutSuffix;
        args[3] = milliseconds > 0;
        args[4] = lang;
        return substituteTimeAgo.apply({}, args);
    }


    /************************************
        Week of Year
    ************************************/


    // firstDayOfWeek       0 = sun, 6 = sat
    //                      the day of the week that starts the week
    //                      (usually sunday or monday)
    // firstDayOfWeekOfYear 0 = sun, 6 = sat
    //                      the first week is the week that contains the first
    //                      of this day of the week
    //                      (eg. ISO weeks use thursday (4))
    function weekOfYear(mom, firstDayOfWeek, firstDayOfWeekOfYear) {
        var end = firstDayOfWeekOfYear - firstDayOfWeek,
            daysToDayOfWeek = firstDayOfWeekOfYear - mom.day(),
            adjustedMoment;


        if (daysToDayOfWeek > end) {
            daysToDayOfWeek -= 7;
        }

        if (daysToDayOfWeek < end - 7) {
            daysToDayOfWeek += 7;
        }

        adjustedMoment = moment(mom).add('d', daysToDayOfWeek);
        return {
            week: Math.ceil(adjustedMoment.dayOfYear() / 7),
            year: adjustedMoment.year()
        };
    }

    //http://en.wikipedia.org/wiki/ISO_week_date#Calculating_a_date_given_the_year.2C_week_number_and_weekday
    function dayOfYearFromWeeks(year, week, weekday, firstDayOfWeekOfYear, firstDayOfWeek) {
        var d = new Date(Date.UTC(year, 0)).getUTCDay(),
            daysToAdd, dayOfYear;

        weekday = weekday != null ? weekday : firstDayOfWeek;
        daysToAdd = firstDayOfWeek - d + (d > firstDayOfWeekOfYear ? 7 : 0);
        dayOfYear = 7 * (week - 1) + (weekday - firstDayOfWeek) + daysToAdd + 1;

        return {
            year: dayOfYear > 0 ? year : year - 1,
            dayOfYear: dayOfYear > 0 ?  dayOfYear : daysInYear(year - 1) + dayOfYear
        };
    }

    /************************************
        Top Level Functions
    ************************************/

    function makeMoment(config) {
        var input = config._i,
            format = config._f;

        if (typeof config._pf === 'undefined') {
            initializeParsingFlags(config);
        }

        if (input === null) {
            return moment.invalid({nullInput: true});
        }

        if (typeof input === 'string') {
            config._i = input = getLangDefinition().preparse(input);
        }

        if (moment.isMoment(input)) {
            config = extend({}, input);

            config._d = new Date(+input._d);
        } else if (format) {
            if (isArray(format)) {
                makeDateFromStringAndArray(config);
            } else {
                makeDateFromStringAndFormat(config);
            }
        } else {
            makeDateFromInput(config);
        }

        return new Moment(config);
    }

    moment = function (input, format, lang, strict) {
        if (typeof(lang) === "boolean") {
            strict = lang;
            lang = undefined;
        }
        return makeMoment({
            _i : input,
            _f : format,
            _l : lang,
            _strict : strict,
            _isUTC : false
        });
    };

    // creating with utc
    moment.utc = function (input, format, lang, strict) {
        var m;

        if (typeof(lang) === "boolean") {
            strict = lang;
            lang = undefined;
        }
        m = makeMoment({
            _useUTC : true,
            _isUTC : true,
            _l : lang,
            _i : input,
            _f : format,
            _strict : strict
        }).utc();

        return m;
    };

    // creating with unix timestamp (in seconds)
    moment.unix = function (input) {
        return moment(input * 1000);
    };

    // duration
    moment.duration = function (input, key) {
        var isDuration = moment.isDuration(input),
            isNumber = (typeof input === 'number'),
            duration = (isDuration ? input._input : (isNumber ? {} : input)),
            // matching against regexp is expensive, do it on demand
            match = null,
            sign,
            ret,
            parseIso,
            timeEmpty,
            dateTimeEmpty;

        if (isNumber) {
            if (key) {
                duration[key] = input;
            } else {
                duration.milliseconds = input;
            }
        } else if (!!(match = aspNetTimeSpanJsonRegex.exec(input))) {
            sign = (match[1] === "-") ? -1 : 1;
            duration = {
                y: 0,
                d: toInt(match[DATE]) * sign,
                h: toInt(match[HOUR]) * sign,
                m: toInt(match[MINUTE]) * sign,
                s: toInt(match[SECOND]) * sign,
                ms: toInt(match[MILLISECOND]) * sign
            };
        } else if (!!(match = isoDurationRegex.exec(input))) {
            sign = (match[1] === "-") ? -1 : 1;
            parseIso = function (inp) {
                // We'd normally use ~~inp for this, but unfortunately it also
                // converts floats to ints.
                // inp may be undefined, so careful calling replace on it.
                var res = inp && parseFloat(inp.replace(',', '.'));
                // apply sign while we're at it
                return (isNaN(res) ? 0 : res) * sign;
            };
            duration = {
                y: parseIso(match[2]),
                M: parseIso(match[3]),
                d: parseIso(match[4]),
                h: parseIso(match[5]),
                m: parseIso(match[6]),
                s: parseIso(match[7]),
                w: parseIso(match[8])
            };
        }

        ret = new Duration(duration);

        if (isDuration && input.hasOwnProperty('_lang')) {
            ret._lang = input._lang;
        }

        return ret;
    };

    // version number
    moment.version = VERSION;

    // default format
    moment.defaultFormat = isoFormat;

    // This function will be called whenever a moment is mutated.
    // It is intended to keep the offset in sync with the timezone.
    moment.updateOffset = function () {};

    // This function will load languages and then set the global language.  If
    // no arguments are passed in, it will simply return the current global
    // language key.
    moment.lang = function (key, values) {
        var r;
        if (!key) {
            return moment.fn._lang._abbr;
        }
        if (values) {
            loadLang(normalizeLanguage(key), values);
        } else if (values === null) {
            unloadLang(key);
            key = 'en';
        } else if (!languages[key]) {
            getLangDefinition(key);
        }
        r = moment.duration.fn._lang = moment.fn._lang = getLangDefinition(key);
        return r._abbr;
    };

    // returns language data
    moment.langData = function (key) {
        if (key && key._lang && key._lang._abbr) {
            key = key._lang._abbr;
        }
        return getLangDefinition(key);
    };

    // compare moment object
    moment.isMoment = function (obj) {
        return obj instanceof Moment;
    };

    // for typechecking Duration objects
    moment.isDuration = function (obj) {
        return obj instanceof Duration;
    };

    for (i = lists.length - 1; i >= 0; --i) {
        makeList(lists[i]);
    }

    moment.normalizeUnits = function (units) {
        return normalizeUnits(units);
    };

    moment.invalid = function (flags) {
        var m = moment.utc(NaN);
        if (flags != null) {
            extend(m._pf, flags);
        }
        else {
            m._pf.userInvalidated = true;
        }

        return m;
    };

    moment.parseZone = function (input) {
        return moment(input).parseZone();
    };

    /************************************
        Moment Prototype
    ************************************/


    extend(moment.fn = Moment.prototype, {

        clone : function () {
            return moment(this);
        },

        valueOf : function () {
            return +this._d + ((this._offset || 0) * 60000);
        },

        unix : function () {
            return Math.floor(+this / 1000);
        },

        toString : function () {
            return this.clone().lang('en').format("ddd MMM DD YYYY HH:mm:ss [GMT]ZZ");
        },

        toDate : function () {
            return this._offset ? new Date(+this) : this._d;
        },

        toISOString : function () {
            return formatMoment(moment(this).utc(), 'YYYY-MM-DD[T]HH:mm:ss.SSS[Z]');
        },

        toArray : function () {
            var m = this;
            return [
                m.year(),
                m.month(),
                m.date(),
                m.hours(),
                m.minutes(),
                m.seconds(),
                m.milliseconds()
            ];
        },

        isValid : function () {
            return isValid(this);
        },

        isDSTShifted : function () {

            if (this._a) {
                return this.isValid() && compareArrays(this._a, (this._isUTC ? moment.utc(this._a) : moment(this._a)).toArray()) > 0;
            }

            return false;
        },

        parsingFlags : function () {
            return extend({}, this._pf);
        },

        invalidAt: function () {
            return this._pf.overflow;
        },

        utc : function () {
            return this.zone(0);
        },

        local : function () {
            this.zone(0);
            this._isUTC = false;
            return this;
        },

        format : function (inputString) {
            var output = formatMoment(this, inputString || moment.defaultFormat);
            return this.lang().postformat(output);
        },

        add : function (input, val) {
            var dur;
            // switch args to support add('s', 1) and add(1, 's')
            if (typeof input === 'string') {
                dur = moment.duration(+val, input);
            } else {
                dur = moment.duration(input, val);
            }
            addOrSubtractDurationFromMoment(this, dur, 1);
            return this;
        },

        subtract : function (input, val) {
            var dur;
            // switch args to support subtract('s', 1) and subtract(1, 's')
            if (typeof input === 'string') {
                dur = moment.duration(+val, input);
            } else {
                dur = moment.duration(input, val);
            }
            addOrSubtractDurationFromMoment(this, dur, -1);
            return this;
        },

        diff : function (input, units, asFloat) {
            var that = this._isUTC ? moment(input).zone(this._offset || 0) : moment(input).local(),
                zoneDiff = (this.zone() - that.zone()) * 6e4,
                diff, output;

            units = normalizeUnits(units);

            if (units === 'year' || units === 'month') {
                // average number of days in the months in the given dates
                diff = (this.daysInMonth() + that.daysInMonth()) * 432e5; // 24 * 60 * 60 * 1000 / 2
                // difference in months
                output = ((this.year() - that.year()) * 12) + (this.month() - that.month());
                // adjust by taking difference in days, average number of days
                // and dst in the given months.
                output += ((this - moment(this).startOf('month')) -
                        (that - moment(that).startOf('month'))) / diff;
                // same as above but with zones, to negate all dst
                output -= ((this.zone() - moment(this).startOf('month').zone()) -
                        (that.zone() - moment(that).startOf('month').zone())) * 6e4 / diff;
                if (units === 'year') {
                    output = output / 12;
                }
            } else {
                diff = (this - that);
                output = units === 'second' ? diff / 1e3 : // 1000
                    units === 'minute' ? diff / 6e4 : // 1000 * 60
                    units === 'hour' ? diff / 36e5 : // 1000 * 60 * 60
                    units === 'day' ? (diff - zoneDiff) / 864e5 : // 1000 * 60 * 60 * 24, negate dst
                    units === 'week' ? (diff - zoneDiff) / 6048e5 : // 1000 * 60 * 60 * 24 * 7, negate dst
                    diff;
            }
            return asFloat ? output : absRound(output);
        },

        from : function (time, withoutSuffix) {
            return moment.duration(this.diff(time)).lang(this.lang()._abbr).humanize(!withoutSuffix);
        },

        fromNow : function (withoutSuffix) {
            return this.from(moment(), withoutSuffix);
        },

        calendar : function () {
            var diff = this.diff(moment().zone(this.zone()).startOf('day'), 'days', true),
                format = diff < -6 ? 'sameElse' :
                diff < -1 ? 'lastWeek' :
                diff < 0 ? 'lastDay' :
                diff < 1 ? 'sameDay' :
                diff < 2 ? 'nextDay' :
                diff < 7 ? 'nextWeek' : 'sameElse';
            return this.format(this.lang().calendar(format, this));
        },

        isLeapYear : function () {
            return isLeapYear(this.year());
        },

        isDST : function () {
            return (this.zone() < this.clone().month(0).zone() ||
                this.zone() < this.clone().month(5).zone());
        },

        day : function (input) {
            var day = this._isUTC ? this._d.getUTCDay() : this._d.getDay();
            if (input != null) {
                input = parseWeekday(input, this.lang());
                return this.add({ d : input - day });
            } else {
                return day;
            }
        },

        month : function (input) {
            var utc = this._isUTC ? 'UTC' : '',
                dayOfMonth;

            if (input != null) {
                if (typeof input === 'string') {
                    input = this.lang().monthsParse(input);
                    if (typeof input !== 'number') {
                        return this;
                    }
                }

                dayOfMonth = this.date();
                this.date(1);
                this._d['set' + utc + 'Month'](input);
                this.date(Math.min(dayOfMonth, this.daysInMonth()));

                moment.updateOffset(this);
                return this;
            } else {
                return this._d['get' + utc + 'Month']();
            }
        },

        startOf: function (units) {
            units = normalizeUnits(units);
            // the following switch intentionally omits break keywords
            // to utilize falling through the cases.
            switch (units) {
            case 'year':
                this.month(0);
                /* falls through */
            case 'month':
                this.date(1);
                /* falls through */
            case 'week':
            case 'isoWeek':
            case 'day':
                this.hours(0);
                /* falls through */
            case 'hour':
                this.minutes(0);
                /* falls through */
            case 'minute':
                this.seconds(0);
                /* falls through */
            case 'second':
                this.milliseconds(0);
                /* falls through */
            }

            // weeks are a special case
            if (units === 'week') {
                this.weekday(0);
            } else if (units === 'isoWeek') {
                this.isoWeekday(1);
            }

            return this;
        },

        endOf: function (units) {
            units = normalizeUnits(units);
            return this.startOf(units).add((units === 'isoWeek' ? 'week' : units), 1).subtract('ms', 1);
        },

        isAfter: function (input, units) {
            units = typeof units !== 'undefined' ? units : 'millisecond';
            return +this.clone().startOf(units) > +moment(input).startOf(units);
        },

        isBefore: function (input, units) {
            units = typeof units !== 'undefined' ? units : 'millisecond';
            return +this.clone().startOf(units) < +moment(input).startOf(units);
        },

        isSame: function (input, units) {
            units = typeof units !== 'undefined' ? units : 'millisecond';
            return +this.clone().startOf(units) === +moment(input).startOf(units);
        },

        min: function (other) {
            other = moment.apply(null, arguments);
            return other < this ? this : other;
        },

        max: function (other) {
            other = moment.apply(null, arguments);
            return other > this ? this : other;
        },

        zone : function (input) {
            var offset = this._offset || 0;
            if (input != null) {
                if (typeof input === "string") {
                    input = timezoneMinutesFromString(input);
                }
                if (Math.abs(input) < 16) {
                    input = input * 60;
                }
                this._offset = input;
                this._isUTC = true;
                if (offset !== input) {
                    addOrSubtractDurationFromMoment(this, moment.duration(offset - input, 'm'), 1, true);
                }
            } else {
                return this._isUTC ? offset : this._d.getTimezoneOffset();
            }
            return this;
        },

        zoneAbbr : function () {
            return this._isUTC ? "UTC" : "";
        },

        zoneName : function () {
            return this._isUTC ? "Coordinated Universal Time" : "";
        },

        parseZone : function () {
            if (typeof this._i === 'string') {
                this.zone(this._i);
            }
            return this;
        },

        hasAlignedHourOffset : function (input) {
            if (!input) {
                input = 0;
            }
            else {
                input = moment(input).zone();
            }

            return (this.zone() - input) % 60 === 0;
        },

        daysInMonth : function () {
            return daysInMonth(this.year(), this.month());
        },

        dayOfYear : function (input) {
            var dayOfYear = round((moment(this).startOf('day') - moment(this).startOf('year')) / 864e5) + 1;
            return input == null ? dayOfYear : this.add("d", (input - dayOfYear));
        },

        weekYear : function (input) {
            var year = weekOfYear(this, this.lang()._week.dow, this.lang()._week.doy).year;
            return input == null ? year : this.add("y", (input - year));
        },

        isoWeekYear : function (input) {
            var year = weekOfYear(this, 1, 4).year;
            return input == null ? year : this.add("y", (input - year));
        },

        week : function (input) {
            var week = this.lang().week(this);
            return input == null ? week : this.add("d", (input - week) * 7);
        },

        isoWeek : function (input) {
            var week = weekOfYear(this, 1, 4).week;
            return input == null ? week : this.add("d", (input - week) * 7);
        },

        weekday : function (input) {
            var weekday = (this.day() + 7 - this.lang()._week.dow) % 7;
            return input == null ? weekday : this.add("d", input - weekday);
        },

        isoWeekday : function (input) {
            // behaves the same as moment#day except
            // as a getter, returns 7 instead of 0 (1-7 range instead of 0-6)
            // as a setter, sunday should belong to the previous week.
            return input == null ? this.day() || 7 : this.day(this.day() % 7 ? input : input - 7);
        },

        get : function (units) {
            units = normalizeUnits(units);
            return this[units]();
        },

        set : function (units, value) {
            units = normalizeUnits(units);
            if (typeof this[units] === 'function') {
                this[units](value);
            }
            return this;
        },

        // If passed a language key, it will set the language for this
        // instance.  Otherwise, it will return the language configuration
        // variables for this instance.
        lang : function (key) {
            if (key === undefined) {
                return this._lang;
            } else {
                this._lang = getLangDefinition(key);
                return this;
            }
        }
    });

    // helper for adding shortcuts
    function makeGetterAndSetter(name, key) {
        moment.fn[name] = moment.fn[name + 's'] = function (input) {
            var utc = this._isUTC ? 'UTC' : '';
            if (input != null) {
                this._d['set' + utc + key](input);
                moment.updateOffset(this);
                return this;
            } else {
                return this._d['get' + utc + key]();
            }
        };
    }

    // loop through and add shortcuts (Month, Date, Hours, Minutes, Seconds, Milliseconds)
    for (i = 0; i < proxyGettersAndSetters.length; i ++) {
        makeGetterAndSetter(proxyGettersAndSetters[i].toLowerCase().replace(/s$/, ''), proxyGettersAndSetters[i]);
    }

    // add shortcut for year (uses different syntax than the getter/setter 'year' == 'FullYear')
    makeGetterAndSetter('year', 'FullYear');

    // add plural methods
    moment.fn.days = moment.fn.day;
    moment.fn.months = moment.fn.month;
    moment.fn.weeks = moment.fn.week;
    moment.fn.isoWeeks = moment.fn.isoWeek;

    // add aliased format methods
    moment.fn.toJSON = moment.fn.toISOString;

    /************************************
        Duration Prototype
    ************************************/


    extend(moment.duration.fn = Duration.prototype, {

        _bubble : function () {
            var milliseconds = this._milliseconds,
                days = this._days,
                months = this._months,
                data = this._data,
                seconds, minutes, hours, years;

            // The following code bubbles up values, see the tests for
            // examples of what that means.
            data.milliseconds = milliseconds % 1000;

            seconds = absRound(milliseconds / 1000);
            data.seconds = seconds % 60;

            minutes = absRound(seconds / 60);
            data.minutes = minutes % 60;

            hours = absRound(minutes / 60);
            data.hours = hours % 24;

            days += absRound(hours / 24);
            data.days = days % 30;

            months += absRound(days / 30);
            data.months = months % 12;

            years = absRound(months / 12);
            data.years = years;
        },

        weeks : function () {
            return absRound(this.days() / 7);
        },

        valueOf : function () {
            return this._milliseconds +
              this._days * 864e5 +
              (this._months % 12) * 2592e6 +
              toInt(this._months / 12) * 31536e6;
        },

        humanize : function (withSuffix) {
            var difference = +this,
                output = relativeTime(difference, !withSuffix, this.lang());

            if (withSuffix) {
                output = this.lang().pastFuture(difference, output);
            }

            return this.lang().postformat(output);
        },

        add : function (input, val) {
            // supports only 2.0-style add(1, 's') or add(moment)
            var dur = moment.duration(input, val);

            this._milliseconds += dur._milliseconds;
            this._days += dur._days;
            this._months += dur._months;

            this._bubble();

            return this;
        },

        subtract : function (input, val) {
            var dur = moment.duration(input, val);

            this._milliseconds -= dur._milliseconds;
            this._days -= dur._days;
            this._months -= dur._months;

            this._bubble();

            return this;
        },

        get : function (units) {
            units = normalizeUnits(units);
            return this[units.toLowerCase() + 's']();
        },

        as : function (units) {
            units = normalizeUnits(units);
            return this['as' + units.charAt(0).toUpperCase() + units.slice(1) + 's']();
        },

        lang : moment.fn.lang,

        toIsoString : function () {
            // inspired by https://github.com/dordille/moment-isoduration/blob/master/moment.isoduration.js
            var years = Math.abs(this.years()),
                months = Math.abs(this.months()),
                days = Math.abs(this.days()),
                hours = Math.abs(this.hours()),
                minutes = Math.abs(this.minutes()),
                seconds = Math.abs(this.seconds() + this.milliseconds() / 1000);

            if (!this.asSeconds()) {
                // this is the same as C#'s (Noda) and python (isodate)...
                // but not other JS (goog.date)
                return 'P0D';
            }

            return (this.asSeconds() < 0 ? '-' : '') +
                'P' +
                (years ? years + 'Y' : '') +
                (months ? months + 'M' : '') +
                (days ? days + 'D' : '') +
                ((hours || minutes || seconds) ? 'T' : '') +
                (hours ? hours + 'H' : '') +
                (minutes ? minutes + 'M' : '') +
                (seconds ? seconds + 'S' : '');
        }
    });

    function makeDurationGetter(name) {
        moment.duration.fn[name] = function () {
            return this._data[name];
        };
    }

    function makeDurationAsGetter(name, factor) {
        moment.duration.fn['as' + name] = function () {
            return +this / factor;
        };
    }

    for (i in unitMillisecondFactors) {
        if (unitMillisecondFactors.hasOwnProperty(i)) {
            makeDurationAsGetter(i, unitMillisecondFactors[i]);
            makeDurationGetter(i.toLowerCase());
        }
    }

    makeDurationAsGetter('Weeks', 6048e5);
    moment.duration.fn.asMonths = function () {
        return (+this - this.years() * 31536e6) / 2592e6 + this.years() * 12;
    };


    /************************************
        Default Lang
    ************************************/


    // Set default language, other languages will inherit from English.
    moment.lang('en', {
        ordinal : function (number) {
            var b = number % 10,
                output = (toInt(number % 100 / 10) === 1) ? 'th' :
                (b === 1) ? 'st' :
                (b === 2) ? 'nd' :
                (b === 3) ? 'rd' : 'th';
            return number + output;
        }
    });

    // moment.js language configuration
// language : Moroccan Arabic (ar-ma)
// author : ElFadili Yassine : https://github.com/ElFadiliY
// author : Abdel Said : https://github.com/abdelsaid

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('ar-ma', {
        months : "ййиЇйиБ_йиЈиБиЇйиБ_йиЇиБиГ_иЃиЈиБйй_йиЇй_ййййй_йййййиВ_иКиДиЊ_иДиЊйиЈиБ_иЃйиЊйиЈиБ_йййиЈиБ_иЏиЌйиЈиБ".split("_"),
        monthsShort : "ййиЇйиБ_йиЈиБиЇйиБ_йиЇиБиГ_иЃиЈиБйй_йиЇй_ййййй_йййййиВ_иКиДиЊ_иДиЊйиЈиБ_иЃйиЊйиЈиБ_йййиЈиБ_иЏиЌйиЈиБ".split("_"),
        weekdays : "иЇйиЃи­иЏ_иЇйиЅиЊййй_иЇйиЋйиЇиЋиЇиЁ_иЇйиЃиБиЈиЙиЇиЁ_иЇйиЎййиГ_иЇйиЌйиЙиЉ_иЇйиГиЈиЊ".split("_"),
        weekdaysShort : "иЇи­иЏ_иЇиЊййй_иЋйиЇиЋиЇиЁ_иЇиБиЈиЙиЇиЁ_иЎййиГ_иЌйиЙиЉ_иГиЈиЊ".split("_"),
        weekdaysMin : "и­_й_иЋ_иБ_иЎ_иЌ_иГ".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd D MMMM YYYY LT"
        },
        calendar : {
            sameDay: "[иЇйййй иЙйй иЇйиГиЇиЙиЉ] LT",
            nextDay: '[иКиЏиЇ иЙйй иЇйиГиЇиЙиЉ] LT',
            nextWeek: 'dddd [иЙйй иЇйиГиЇиЙиЉ] LT',
            lastDay: '[иЃйиГ иЙйй иЇйиГиЇиЙиЉ] LT',
            lastWeek: 'dddd [иЙйй иЇйиГиЇиЙиЉ] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : "йй %s",
            past : "ййиА %s",
            s : "иЋйиЇй",
            m : "иЏйййиЉ",
            mm : "%d иЏйиЇиІй",
            h : "иГиЇиЙиЉ",
            hh : "%d иГиЇиЙиЇиЊ",
            d : "ййй",
            dd : "%d иЃйиЇй",
            M : "иДйиБ",
            MM : "%d иЃиДйиБ",
            y : "иГйиЉ",
            yy : "%d иГййиЇиЊ"
        },
        week : {
            dow : 6, // Saturday is the first day of the week.
            doy : 12  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : Arabic (ar)
// author : Abdel Said : https://github.com/abdelsaid
// changes in months, weekdays : Ahmed Elkhatib

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('ar', {
        months : "ййиЇйиБ/ йиЇййй иЇйиЋиЇйй_йиЈиБиЇйиБ/ иДиЈиЇиЗ_йиЇиБиГ/ иЂиАиЇиБ_иЃиЈиБйй/ ййиГиЇй_йиЇйй/ иЃйиЇиБ_ййййй/ и­иВйиБиЇй_ййййй/ иЊййиВ_иЃиКиГиЗиГ/ иЂиЈ_иГиЈиЊйиЈиБ/ иЃйййй_иЃйиЊйиЈиБ/ иЊиДиБйй иЇйиЃйй_ййййиЈиБ/ иЊиДиБйй иЇйиЋиЇйй_иЏйиГйиЈиБ/ йиЇййй иЇйиЃйй".split("_"),
        monthsShort : "ййиЇйиБ/ йиЇййй иЇйиЋиЇйй_йиЈиБиЇйиБ/ иДиЈиЇиЗ_йиЇиБиГ/ иЂиАиЇиБ_иЃиЈиБйй/ ййиГиЇй_йиЇйй/ иЃйиЇиБ_ййййй/ и­иВйиБиЇй_ййййй/ иЊййиВ_иЃиКиГиЗиГ/ иЂиЈ_иГиЈиЊйиЈиБ/ иЃйййй_иЃйиЊйиЈиБ/ иЊиДиБйй иЇйиЃйй_ййййиЈиБ/ иЊиДиБйй иЇйиЋиЇйй_иЏйиГйиЈиБ/ йиЇййй иЇйиЃйй".split("_"),
        weekdays : "иЇйиЃи­иЏ_иЇйиЅиЋййй_иЇйиЋйиЇиЋиЇиЁ_иЇйиЃиБиЈиЙиЇиЁ_иЇйиЎййиГ_иЇйиЌйиЙиЉ_иЇйиГиЈиЊ".split("_"),
        weekdaysShort : "иЇйиЃи­иЏ_иЇйиЅиЋййй_иЇйиЋйиЇиЋиЇиЁ_иЇйиЃиБиЈиЙиЇиЁ_иЇйиЎййиГ_иЇйиЌйиЙиЉ_иЇйиГиЈиЊ".split("_"),
        weekdaysMin : "и­_й_иЋ_иБ_иЎ_иЌ_иГ".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd D MMMM YYYY LT"
        },
        calendar : {
            sameDay: "[иЇйййй иЙйй иЇйиГиЇиЙиЉ] LT",
            nextDay: '[иКиЏиЇ иЙйй иЇйиГиЇиЙиЉ] LT',
            nextWeek: 'dddd [иЙйй иЇйиГиЇиЙиЉ] LT',
            lastDay: '[иЃйиГ иЙйй иЇйиГиЇиЙиЉ] LT',
            lastWeek: 'dddd [иЙйй иЇйиГиЇиЙиЉ] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : "йй %s",
            past : "ййиА %s",
            s : "иЋйиЇй",
            m : "иЏйййиЉ",
            mm : "%d иЏйиЇиІй",
            h : "иГиЇиЙиЉ",
            hh : "%d иГиЇиЙиЇиЊ",
            d : "ййй",
            dd : "%d иЃйиЇй",
            M : "иДйиБ",
            MM : "%d иЃиДйиБ",
            y : "иГйиЉ",
            yy : "%d иГййиЇиЊ"
        },
        week : {
            dow : 6, // Saturday is the first day of the week.
            doy : 12  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : bulgarian (bg)
// author : Krasen Borisov : https://github.com/kraz

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('bg', {
        months : "баНбаАбаИ_баЕаВббаАбаИ_аМаАбб_аАаПбаИаЛ_аМаАаЙ_баНаИ_баЛаИ_аАаВаГббб_баЕаПбаЕаМаВбаИ_аОаКбаОаМаВбаИ_аНаОаЕаМаВбаИ_аДаЕаКаЕаМаВбаИ".split("_"),
        monthsShort : "баНб_баЕаВ_аМаАб_аАаПб_аМаАаЙ_баНаИ_баЛаИ_аАаВаГ_баЕаП_аОаКб_аНаОаЕ_аДаЕаК".split("_"),
        weekdays : "аНаЕаДаЕаЛб_аПаОаНаЕаДаЕаЛаНаИаК_аВбаОбаНаИаК_бббаДаА_баЕбаВббббаК_аПаЕббаК_ббаБаОбаА".split("_"),
        weekdaysShort : "аНаЕаД_аПаОаН_аВбаО_ббб_баЕб_аПаЕб_ббаБ".split("_"),
        weekdaysMin : "аНаД_аПаН_аВб_бб_бб_аПб_баБ".split("_"),
        longDateFormat : {
            LT : "H:mm",
            L : "D.MM.YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd, D MMMM YYYY LT"
        },
        calendar : {
            sameDay : '[ааНаЕб аВ] LT',
            nextDay : '[аЃббаЕ аВ] LT',
            nextWeek : 'dddd [аВ] LT',
            lastDay : '[абаЕбаА аВ] LT',
            lastWeek : function () {
                switch (this.day()) {
                case 0:
                case 3:
                case 6:
                    return '[а аИаЗаМаИаНаАаЛаАбаА] dddd [аВ] LT';
                case 1:
                case 2:
                case 4:
                case 5:
                    return '[а аИаЗаМаИаНаАаЛаИб] dddd [аВ] LT';
                }
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : "баЛаЕаД %s",
            past : "аПбаЕаДаИ %s",
            s : "аНбаКаОаЛаКаО баЕаКбаНаДаИ",
            m : "аМаИаНббаА",
            mm : "%d аМаИаНббаИ",
            h : "баАб",
            hh : "%d баАбаА",
            d : "аДаЕаН",
            dd : "%d аДаНаИ",
            M : "аМаЕбаЕб",
            MM : "%d аМаЕбаЕбаА",
            y : "аГаОаДаИаНаА",
            yy : "%d аГаОаДаИаНаИ"
        },
        ordinal : function (number) {
            var lastDigit = number % 10,
                last2Digits = number % 100;
            if (number === 0) {
                return number + '-аЕаВ';
            } else if (last2Digits === 0) {
                return number + '-аЕаН';
            } else if (last2Digits > 10 && last2Digits < 20) {
                return number + '-баИ';
            } else if (lastDigit === 1) {
                return number + '-аВаИ';
            } else if (lastDigit === 2) {
                return number + '-баИ';
            } else if (lastDigit === 7 || lastDigit === 8) {
                return number + '-аМаИ';
            } else {
                return number + '-баИ';
            }
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : breton (br)
// author : Jean-Baptiste Le Duigou : https://github.com/jbleduigou

(function (factory) {
    factory(moment);
}(function (moment) {
    function relativeTimeWithMutation(number, withoutSuffix, key) {
        var format = {
            'mm': "munutenn",
            'MM': "miz",
            'dd': "devezh"
        };
        return number + ' ' + mutation(format[key], number);
    }

    function specialMutationForYears(number) {
        switch (lastNumber(number)) {
        case 1:
        case 3:
        case 4:
        case 5:
        case 9:
            return number + ' bloaz';
        default:
            return number + ' vloaz';
        }
    }

    function lastNumber(number) {
        if (number > 9) {
            return lastNumber(number % 10);
        }
        return number;
    }

    function mutation(text, number) {
        if (number === 2) {
            return softMutation(text);
        }
        return text;
    }

    function softMutation(text) {
        var mutationTable = {
            'm': 'v',
            'b': 'v',
            'd': 'z'
        };
        if (mutationTable[text.charAt(0)] === undefined) {
            return text;
        }
        return mutationTable[text.charAt(0)] + text.substring(1);
    }

    return moment.lang('br', {
        months : "Genver_C'hwevrer_Meurzh_Ebrel_Mae_Mezheven_Gouere_Eost_Gwengolo_Here_Du_Kerzu".split("_"),
        monthsShort : "Gen_C'hwe_Meu_Ebr_Mae_Eve_Gou_Eos_Gwe_Her_Du_Ker".split("_"),
        weekdays : "Sul_Lun_Meurzh_Merc'her_Yaou_Gwener_Sadorn".split("_"),
        weekdaysShort : "Sul_Lun_Meu_Mer_Yao_Gwe_Sad".split("_"),
        weekdaysMin : "Su_Lu_Me_Mer_Ya_Gw_Sa".split("_"),
        longDateFormat : {
            LT : "h[e]mm A",
            L : "DD/MM/YYYY",
            LL : "D [a viz] MMMM YYYY",
            LLL : "D [a viz] MMMM YYYY LT",
            LLLL : "dddd, D [a viz] MMMM YYYY LT"
        },
        calendar : {
            sameDay : '[Hiziv da] LT',
            nextDay : '[Warc\'hoazh da] LT',
            nextWeek : 'dddd [da] LT',
            lastDay : '[Dec\'h da] LT',
            lastWeek : 'dddd [paset da] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "a-benn %s",
            past : "%s 'zo",
            s : "un nebeud segondennoУЙ",
            m : "ur vunutenn",
            mm : relativeTimeWithMutation,
            h : "un eur",
            hh : "%d eur",
            d : "un devezh",
            dd : relativeTimeWithMutation,
            M : "ur miz",
            MM : relativeTimeWithMutation,
            y : "ur bloaz",
            yy : specialMutationForYears
        },
        ordinal : function (number) {
            var output = (number === 1) ? 'aУБ' : 'vet';
            return number + output;
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : bosnian (bs)
// author : Nedim Cholich : https://github.com/frontyard
// based on (hr) translation by Bojan MarkoviФ

(function (factory) {
    factory(moment);
}(function (moment) {

    function translate(number, withoutSuffix, key) {
        var result = number + " ";
        switch (key) {
        case 'm':
            return withoutSuffix ? 'jedna minuta' : 'jedne minute';
        case 'mm':
            if (number === 1) {
                result += 'minuta';
            } else if (number === 2 || number === 3 || number === 4) {
                result += 'minute';
            } else {
                result += 'minuta';
            }
            return result;
        case 'h':
            return withoutSuffix ? 'jedan sat' : 'jednog sata';
        case 'hh':
            if (number === 1) {
                result += 'sat';
            } else if (number === 2 || number === 3 || number === 4) {
                result += 'sata';
            } else {
                result += 'sati';
            }
            return result;
        case 'dd':
            if (number === 1) {
                result += 'dan';
            } else {
                result += 'dana';
            }
            return result;
        case 'MM':
            if (number === 1) {
                result += 'mjesec';
            } else if (number === 2 || number === 3 || number === 4) {
                result += 'mjeseca';
            } else {
                result += 'mjeseci';
            }
            return result;
        case 'yy':
            if (number === 1) {
                result += 'godina';
            } else if (number === 2 || number === 3 || number === 4) {
                result += 'godine';
            } else {
                result += 'godina';
            }
            return result;
        }
    }

    return moment.lang('bs', {
        months : "januar_februar_mart_april_maj_juni_juli_avgust_septembar_oktobar_novembar_decembar".split("_"),
        monthsShort : "jan._feb._mar._apr._maj._jun._jul._avg._sep._okt._nov._dec.".split("_"),
        weekdays : "nedjelja_ponedjeljak_utorak_srijeda_Фetvrtak_petak_subota".split("_"),
        weekdaysShort : "ned._pon._uto._sri._Фet._pet._sub.".split("_"),
        weekdaysMin : "ne_po_ut_sr_Фe_pe_su".split("_"),
        longDateFormat : {
            LT : "H:mm",
            L : "DD. MM. YYYY",
            LL : "D. MMMM YYYY",
            LLL : "D. MMMM YYYY LT",
            LLLL : "dddd, D. MMMM YYYY LT"
        },
        calendar : {
            sameDay  : '[danas u] LT',
            nextDay  : '[sutra u] LT',

            nextWeek : function () {
                switch (this.day()) {
                case 0:
                    return '[u] [nedjelju] [u] LT';
                case 3:
                    return '[u] [srijedu] [u] LT';
                case 6:
                    return '[u] [subotu] [u] LT';
                case 1:
                case 2:
                case 4:
                case 5:
                    return '[u] dddd [u] LT';
                }
            },
            lastDay  : '[juФer u] LT',
            lastWeek : function () {
                switch (this.day()) {
                case 0:
                case 3:
                    return '[proХЁlu] dddd [u] LT';
                case 6:
                    return '[proХЁle] [subote] [u] LT';
                case 1:
                case 2:
                case 4:
                case 5:
                    return '[proХЁli] dddd [u] LT';
                }
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : "za %s",
            past   : "prije %s",
            s      : "par sekundi",
            m      : translate,
            mm     : translate,
            h      : translate,
            hh     : translate,
            d      : "dan",
            dd     : translate,
            M      : "mjesec",
            MM     : translate,
            y      : "godinu",
            yy     : translate
        },
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : catalan (ca)
// author : Juan G. Hurtado : https://github.com/juanghurtado

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('ca', {
        months : "Gener_Febrer_MarУЇ_Abril_Maig_Juny_Juliol_Agost_Setembre_Octubre_Novembre_Desembre".split("_"),
        monthsShort : "Gen._Febr._Mar._Abr._Mai._Jun._Jul._Ag._Set._Oct._Nov._Des.".split("_"),
        weekdays : "Diumenge_Dilluns_Dimarts_Dimecres_Dijous_Divendres_Dissabte".split("_"),
        weekdaysShort : "Dg._Dl._Dt._Dc._Dj._Dv._Ds.".split("_"),
        weekdaysMin : "Dg_Dl_Dt_Dc_Dj_Dv_Ds".split("_"),
        longDateFormat : {
            LT : "H:mm",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd D MMMM YYYY LT"
        },
        calendar : {
            sameDay : function () {
                return '[avui a ' + ((this.hours() !== 1) ? 'les' : 'la') + '] LT';
            },
            nextDay : function () {
                return '[demУ  a ' + ((this.hours() !== 1) ? 'les' : 'la') + '] LT';
            },
            nextWeek : function () {
                return 'dddd [a ' + ((this.hours() !== 1) ? 'les' : 'la') + '] LT';
            },
            lastDay : function () {
                return '[ahir a ' + ((this.hours() !== 1) ? 'les' : 'la') + '] LT';
            },
            lastWeek : function () {
                return '[el] dddd [passat a ' + ((this.hours() !== 1) ? 'les' : 'la') + '] LT';
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : "en %s",
            past : "fa %s",
            s : "uns segons",
            m : "un minut",
            mm : "%d minuts",
            h : "una hora",
            hh : "%d hores",
            d : "un dia",
            dd : "%d dies",
            M : "un mes",
            MM : "%d mesos",
            y : "un any",
            yy : "%d anys"
        },
        ordinal : '%dТК',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : czech (cs)
// author : petrbela : https://github.com/petrbela

(function (factory) {
    factory(moment);
}(function (moment) {
    var months = "leden_УКnor_bХezen_duben_kvФten_Фerven_Фervenec_srpen_zУЁХУ­_ХУ­jen_listopad_prosinec".split("_"),
        monthsShort = "led_УКno_bХe_dub_kvФ_Фvn_Фvc_srp_zУЁХ_ХУ­j_lis_pro".split("_");

    function plural(n) {
        return (n > 1) && (n < 5) && (~~(n / 10) !== 1);
    }

    function translate(number, withoutSuffix, key, isFuture) {
        var result = number + " ";
        switch (key) {
        case 's':  // a few seconds / in a few seconds / a few seconds ago
            return (withoutSuffix || isFuture) ? 'pУЁr vteХin' : 'pУЁr vteХinami';
        case 'm':  // a minute / in a minute / a minute ago
            return withoutSuffix ? 'minuta' : (isFuture ? 'minutu' : 'minutou');
        case 'mm': // 9 minutes / in 9 minutes / 9 minutes ago
            if (withoutSuffix || isFuture) {
                return result + (plural(number) ? 'minuty' : 'minut');
            } else {
                return result + 'minutami';
            }
            break;
        case 'h':  // an hour / in an hour / an hour ago
            return withoutSuffix ? 'hodina' : (isFuture ? 'hodinu' : 'hodinou');
        case 'hh': // 9 hours / in 9 hours / 9 hours ago
            if (withoutSuffix || isFuture) {
                return result + (plural(number) ? 'hodiny' : 'hodin');
            } else {
                return result + 'hodinami';
            }
            break;
        case 'd':  // a day / in a day / a day ago
            return (withoutSuffix || isFuture) ? 'den' : 'dnem';
        case 'dd': // 9 days / in 9 days / 9 days ago
            if (withoutSuffix || isFuture) {
                return result + (plural(number) ? 'dny' : 'dnУ­');
            } else {
                return result + 'dny';
            }
            break;
        case 'M':  // a month / in a month / a month ago
            return (withoutSuffix || isFuture) ? 'mФsУ­c' : 'mФsУ­cem';
        case 'MM': // 9 months / in 9 months / 9 months ago
            if (withoutSuffix || isFuture) {
                return result + (plural(number) ? 'mФsУ­ce' : 'mФsУ­cХЏ');
            } else {
                return result + 'mФsУ­ci';
            }
            break;
        case 'y':  // a year / in a year / a year ago
            return (withoutSuffix || isFuture) ? 'rok' : 'rokem';
        case 'yy': // 9 years / in 9 years / 9 years ago
            if (withoutSuffix || isFuture) {
                return result + (plural(number) ? 'roky' : 'let');
            } else {
                return result + 'lety';
            }
            break;
        }
    }

    return moment.lang('cs', {
        months : months,
        monthsShort : monthsShort,
        monthsParse : (function (months, monthsShort) {
            var i, _monthsParse = [];
            for (i = 0; i < 12; i++) {
                // use custom parser to solve problem with July (Фervenec)
                _monthsParse[i] = new RegExp('^' + months[i] + '$|^' + monthsShort[i] + '$', 'i');
            }
            return _monthsParse;
        }(months, monthsShort)),
        weekdays : "nedФle_pondФlУ­_УКterУН_stХeda_Фtvrtek_pУЁtek_sobota".split("_"),
        weekdaysShort : "ne_po_УКt_st_Фt_pУЁ_so".split("_"),
        weekdaysMin : "ne_po_УКt_st_Фt_pУЁ_so".split("_"),
        longDateFormat : {
            LT: "H:mm",
            L : "DD.MM.YYYY",
            LL : "D. MMMM YYYY",
            LLL : "D. MMMM YYYY LT",
            LLLL : "dddd D. MMMM YYYY LT"
        },
        calendar : {
            sameDay: "[dnes v] LT",
            nextDay: '[zУ­tra v] LT',
            nextWeek: function () {
                switch (this.day()) {
                case 0:
                    return '[v nedФli v] LT';
                case 1:
                case 2:
                    return '[v] dddd [v] LT';
                case 3:
                    return '[ve stХedu v] LT';
                case 4:
                    return '[ve Фtvrtek v] LT';
                case 5:
                    return '[v pУЁtek v] LT';
                case 6:
                    return '[v sobotu v] LT';
                }
            },
            lastDay: '[vФera v] LT',
            lastWeek: function () {
                switch (this.day()) {
                case 0:
                    return '[minulou nedФli v] LT';
                case 1:
                case 2:
                    return '[minulУЉ] dddd [v] LT';
                case 3:
                    return '[minulou stХedu v] LT';
                case 4:
                case 5:
                    return '[minulУН] dddd [v] LT';
                case 6:
                    return '[minulou sobotu v] LT';
                }
            },
            sameElse: "L"
        },
        relativeTime : {
            future : "za %s",
            past : "pХed %s",
            s : translate,
            m : translate,
            mm : translate,
            h : translate,
            hh : translate,
            d : translate,
            dd : translate,
            M : translate,
            MM : translate,
            y : translate,
            yy : translate
        },
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : chuvash (cv)
// author : Anatoly Mironov : https://github.com/mirontoli

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('cv', {
        months : "аКФбаЛаАб_аНаАбФб_аПбб_аАаКаА_аМаАаЙ_УЇФббаМаЕ_ббФ_УЇббаЛаА_аАаВФаН_баПаА_бгГаК_баАббаАаВ".split("_"),
        monthsShort : "аКФб_аНаАб_аПбб_аАаКаА_аМаАаЙ_УЇФб_ббФ_УЇбб_аАаВ_баПаА_бгГаК_баАб".split("_"),
        weekdays : "аВбббаАбаНаИаКбаН_ббаНбаИаКбаН_ббаЛаАбаИаКбаН_баНаКбаН_аКФУЇаНаЕбаНаИаКбаН_ббаНаЕаКбаН_бФаМаАбаКбаН".split("_"),
        weekdaysShort : "аВбб_ббаН_ббаЛ_баН_аКФУЇ_ббаН_бФаМ".split("_"),
        weekdaysMin : "аВб_баН_бб_баН_аКУЇ_бб_баМ".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD-MM-YYYY",
            LL : "YYYY [УЇбаЛбаИ] MMMM [баЙФбФаН] D[-аМФбФ]",
            LLL : "YYYY [УЇбаЛбаИ] MMMM [баЙФбФаН] D[-аМФбФ], LT",
            LLLL : "dddd, YYYY [УЇбаЛбаИ] MMMM [баЙФбФаН] D[-аМФбФ], LT"
        },
        calendar : {
            sameDay: '[ааАбаН] LT [баЕбаЕббаЕ]',
            nextDay: '[аЋбаАаН] LT [баЕбаЕббаЕ]',
            lastDay: '[ФаНаЕб] LT [баЕбаЕббаЕ]',
            nextWeek: '[УаИбаЕб] dddd LT [баЕбаЕббаЕ]',
            lastWeek: '[аббаНФ] dddd LT [баЕбаЕббаЕ]',
            sameElse: 'L'
        },
        relativeTime : {
            future : function (output) {
                var affix = /баЕбаЕб$/i.exec(output) ? "баЕаН" : /УЇбаЛ$/i.exec(output) ? "баАаН" : "баАаН";
                return output + affix;
            },
            past : "%s аКаАбаЛаЛаА",
            s : "аПФб-аИаК УЇаЕаКаКбаНб",
            m : "аПФб аМаИаНбб",
            mm : "%d аМаИаНбб",
            h : "аПФб баЕбаЕб",
            hh : "%d баЕбаЕб",
            d : "аПФб аКбаН",
            dd : "%d аКбаН",
            M : "аПФб баЙФб",
            MM : "%d баЙФб",
            y : "аПФб УЇбаЛ",
            yy : "%d УЇбаЛ"
        },
        ordinal : '%d-аМФб',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : Welsh (cy)
// author : Robert Allen

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang("cy", {
        months: "Ionawr_Chwefror_Mawrth_Ebrill_Mai_Mehefin_Gorffennaf_Awst_Medi_Hydref_Tachwedd_Rhagfyr".split("_"),
        monthsShort: "Ion_Chwe_Maw_Ebr_Mai_Meh_Gor_Aws_Med_Hyd_Tach_Rhag".split("_"),
        weekdays: "Dydd Sul_Dydd Llun_Dydd Mawrth_Dydd Mercher_Dydd Iau_Dydd Gwener_Dydd Sadwrn".split("_"),
        weekdaysShort: "Sul_Llun_Maw_Mer_Iau_Gwe_Sad".split("_"),
        weekdaysMin: "Su_Ll_Ma_Me_Ia_Gw_Sa".split("_"),
        // time formats are the same as en-gb
        longDateFormat: {
            LT: "HH:mm",
            L: "DD/MM/YYYY",
            LL: "D MMMM YYYY",
            LLL: "D MMMM YYYY LT",
            LLLL: "dddd, D MMMM YYYY LT"
        },
        calendar: {
            sameDay: '[Heddiw am] LT',
            nextDay: '[Yfory am] LT',
            nextWeek: 'dddd [am] LT',
            lastDay: '[Ddoe am] LT',
            lastWeek: 'dddd [diwethaf am] LT',
            sameElse: 'L'
        },
        relativeTime: {
            future: "mewn %s",
            past: "%s yn &#244;l",
            s: "ychydig eiliadau",
            m: "munud",
            mm: "%d munud",
            h: "awr",
            hh: "%d awr",
            d: "diwrnod",
            dd: "%d diwrnod",
            M: "mis",
            MM: "%d mis",
            y: "blwyddyn",
            yy: "%d flynedd"
        },
        // traditional ordinal numbers above 31 are not commonly used in colloquial Welsh
        ordinal: function (number) {
            var b = number,
                output = '',
                lookup = [
                    '', 'af', 'il', 'ydd', 'ydd', 'ed', 'ed', 'ed', 'fed', 'fed', 'fed', // 1af to 10fed
                    'eg', 'fed', 'eg', 'eg', 'fed', 'eg', 'eg', 'fed', 'eg', 'fed' // 11eg to 20fed
                ];

            if (b > 20) {
                if (b === 40 || b === 50 || b === 60 || b === 80 || b === 100) {
                    output = 'fed'; // not 30ain, 70ain or 90ain
                } else {
                    output = 'ain';
                }
            } else if (b > 0) {
                output = lookup[b];
            }

            return number + output;
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : danish (da)
// author : Ulrik Nielsen : https://github.com/mrbase

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('da', {
        months : "januar_februar_marts_april_maj_juni_juli_august_september_oktober_november_december".split("_"),
        monthsShort : "jan_feb_mar_apr_maj_jun_jul_aug_sep_okt_nov_dec".split("_"),
        weekdays : "sУИndag_mandag_tirsdag_onsdag_torsdag_fredag_lУИrdag".split("_"),
        weekdaysShort : "sУИn_man_tir_ons_tor_fre_lУИr".split("_"),
        weekdaysMin : "sУИ_ma_ti_on_to_fr_lУИ".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd D. MMMM, YYYY LT"
        },
        calendar : {
            sameDay : '[I dag kl.] LT',
            nextDay : '[I morgen kl.] LT',
            nextWeek : 'dddd [kl.] LT',
            lastDay : '[I gУЅr kl.] LT',
            lastWeek : '[sidste] dddd [kl] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "om %s",
            past : "%s siden",
            s : "fУЅ sekunder",
            m : "et minut",
            mm : "%d minutter",
            h : "en time",
            hh : "%d timer",
            d : "en dag",
            dd : "%d dage",
            M : "en mУЅned",
            MM : "%d mУЅneder",
            y : "et УЅr",
            yy : "%d УЅr"
        },
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : german (de)
// author : lluchs : https://github.com/lluchs
// author: Menelion ElensУКle: https://github.com/Oire

(function (factory) {
    factory(moment);
}(function (moment) {
    function processRelativeTime(number, withoutSuffix, key, isFuture) {
        var format = {
            'm': ['eine Minute', 'einer Minute'],
            'h': ['eine Stunde', 'einer Stunde'],
            'd': ['ein Tag', 'einem Tag'],
            'dd': [number + ' Tage', number + ' Tagen'],
            'M': ['ein Monat', 'einem Monat'],
            'MM': [number + ' Monate', number + ' Monaten'],
            'y': ['ein Jahr', 'einem Jahr'],
            'yy': [number + ' Jahre', number + ' Jahren']
        };
        return withoutSuffix ? format[key][0] : format[key][1];
    }

    return moment.lang('de', {
        months : "Januar_Februar_MУЄrz_April_Mai_Juni_Juli_August_September_Oktober_November_Dezember".split("_"),
        monthsShort : "Jan._Febr._Mrz._Apr._Mai_Jun._Jul._Aug._Sept._Okt._Nov._Dez.".split("_"),
        weekdays : "Sonntag_Montag_Dienstag_Mittwoch_Donnerstag_Freitag_Samstag".split("_"),
        weekdaysShort : "So._Mo._Di._Mi._Do._Fr._Sa.".split("_"),
        weekdaysMin : "So_Mo_Di_Mi_Do_Fr_Sa".split("_"),
        longDateFormat : {
            LT: "H:mm [Uhr]",
            L : "DD.MM.YYYY",
            LL : "D. MMMM YYYY",
            LLL : "D. MMMM YYYY LT",
            LLLL : "dddd, D. MMMM YYYY LT"
        },
        calendar : {
            sameDay: "[Heute um] LT",
            sameElse: "L",
            nextDay: '[Morgen um] LT',
            nextWeek: 'dddd [um] LT',
            lastDay: '[Gestern um] LT',
            lastWeek: '[letzten] dddd [um] LT'
        },
        relativeTime : {
            future : "in %s",
            past : "vor %s",
            s : "ein paar Sekunden",
            m : processRelativeTime,
            mm : "%d Minuten",
            h : processRelativeTime,
            hh : "%d Stunden",
            d : processRelativeTime,
            dd : processRelativeTime,
            M : processRelativeTime,
            MM : processRelativeTime,
            y : processRelativeTime,
            yy : processRelativeTime
        },
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : modern greek (el)
// author : Aggelos Karalias : https://github.com/mehiel

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('el', {
        monthsNominativeEl : "ЮЮБЮНЮПЯЮЌЯЮЙЮПЯ_ЮІЮЕЮВЯЮПЯЮЌЯЮЙЮПЯ_ЮЮЌЯЯЮЙЮПЯ_ЮЯЯЮЏЮЛЮЙЮПЯ_ЮЮЌЮЙЮПЯ_ЮЮПЯЮНЮЙЮПЯ_ЮЮПЯЮЛЮЙЮПЯ_ЮЯЮГЮПЯЯЯЮПЯ_ЮЃЮЕЯЯЮ­ЮМЮВЯЮЙЮПЯ_ЮЮКЯЯЮВЯЮЙЮПЯ_ЮЮПЮ­ЮМЮВЯЮЙЮПЯ_ЮЮЕЮКЮ­ЮМЮВЯЮЙЮПЯ".split("_"),
        monthsGenitiveEl : "ЮЮБЮНЮПЯЮБЯЮЏЮПЯ_ЮІЮЕЮВЯЮПЯЮБЯЮЏЮПЯ_ЮЮБЯЯЮЏЮПЯ_ЮЯЯЮЙЮЛЮЏЮПЯ_ЮЮБЮЮПЯ_ЮЮПЯЮНЮЏЮПЯ_ЮЮПЯЮЛЮЏЮПЯ_ЮЯЮГЮПЯЯЯЮПЯ_ЮЃЮЕЯЯЮЕЮМЮВЯЮЏЮПЯ_ЮЮКЯЯЮВЯЮЏЮПЯ_ЮЮПЮЕЮМЮВЯЮЏЮПЯ_ЮЮЕЮКЮЕЮМЮВЯЮЏЮПЯ".split("_"),
        months : function (momentToFormat, format) {
            if (/D/.test(format.substring(0, format.indexOf("MMMM")))) { // if there is a day number before 'MMMM'
                return this._monthsGenitiveEl[momentToFormat.month()];
            } else {
                return this._monthsNominativeEl[momentToFormat.month()];
            }
        },
        monthsShort : "ЮЮБЮН_ЮІЮЕЮВ_ЮЮБЯ_ЮЯЯ_ЮЮБЯ_ЮЮПЯЮН_ЮЮПЯЮЛ_ЮЯЮГ_ЮЃЮЕЯ_ЮЮКЯ_ЮЮПЮЕ_ЮЮЕЮК".split("_"),
        weekdays : "ЮЯЯЮЙЮБЮКЮЎ_ЮЮЕЯЯЮ­ЯЮБ_ЮЄЯЮЏЯЮЗ_ЮЄЮЕЯЮЌЯЯЮЗ_Ю Ю­ЮМЯЯЮЗ_Ю ЮБЯЮБЯЮКЮЕЯЮЎ_ЮЃЮЌЮВЮВЮБЯЮП".split("_"),
        weekdaysShort : "ЮЯЯ_ЮЮЕЯ_ЮЄЯЮЙ_ЮЄЮЕЯ_Ю ЮЕЮМ_Ю ЮБЯ_ЮЃЮБЮВ".split("_"),
        weekdaysMin : "ЮЯ_ЮЮЕ_ЮЄЯ_ЮЄЮЕ_Ю ЮЕ_Ю ЮБ_ЮЃЮБ".split("_"),
        meridiem : function (hours, minutes, isLower) {
            if (hours > 11) {
                return isLower ? 'ЮМЮМ' : 'ЮЮ';
            } else {
                return isLower ? 'ЯЮМ' : 'Ю Ю';
            }
        },
        longDateFormat : {
            LT : "h:mm A",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd, D MMMM YYYY LT"
        },
        calendarEl : {
            sameDay : '[ЮЃЮЎЮМЮЕЯЮБ {}] LT',
            nextDay : '[ЮЯЯЮЙЮП {}] LT',
            nextWeek : 'dddd [{}] LT',
            lastDay : '[ЮЇЮИЮЕЯ {}] LT',
            lastWeek : '[ЯЮЗЮН ЯЯЮПЮЗЮГЮПЯЮМЮЕЮНЮЗ] dddd [{}] LT',
            sameElse : 'L'
        },
        calendar : function (key, mom) {
            var output = this._calendarEl[key],
                hours = mom && mom.hours();

            return output.replace("{}", (hours % 12 === 1 ? "ЯЯЮЗ" : "ЯЯЮЙЯ"));
        },
        relativeTime : {
            future : "ЯЮЕ %s",
            past : "%s ЯЯЮЙЮН",
            s : "ЮДЮЕЯЯЮЕЯЯЮЛЮЕЯЯЮБ",
            m : "Ю­ЮНЮБ ЮЛЮЕЯЯЯ",
            mm : "%d ЮЛЮЕЯЯЮЌ",
            h : "ЮМЮЏЮБ ЯЯЮБ",
            hh : "%d ЯЯЮЕЯ",
            d : "ЮМЮЏЮБ ЮМЮ­ЯЮБ",
            dd : "%d ЮМЮ­ЯЮЕЯ",
            M : "Ю­ЮНЮБЯ ЮМЮЎЮНЮБЯ",
            MM : "%d ЮМЮЎЮНЮЕЯ",
            y : "Ю­ЮНЮБЯ ЯЯЯЮНЮПЯ",
            yy : "%d ЯЯЯЮНЮЙЮБ"
        },
        ordinal : function (number) {
            return number + 'ЮЗ';
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : australian english (en-au)

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('en-au', {
        months : "January_February_March_April_May_June_July_August_September_October_November_December".split("_"),
        monthsShort : "Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec".split("_"),
        weekdays : "Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday".split("_"),
        weekdaysShort : "Sun_Mon_Tue_Wed_Thu_Fri_Sat".split("_"),
        weekdaysMin : "Su_Mo_Tu_We_Th_Fr_Sa".split("_"),
        longDateFormat : {
            LT : "h:mm A",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd, D MMMM YYYY LT"
        },
        calendar : {
            sameDay : '[Today at] LT',
            nextDay : '[Tomorrow at] LT',
            nextWeek : 'dddd [at] LT',
            lastDay : '[Yesterday at] LT',
            lastWeek : '[Last] dddd [at] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "in %s",
            past : "%s ago",
            s : "a few seconds",
            m : "a minute",
            mm : "%d minutes",
            h : "an hour",
            hh : "%d hours",
            d : "a day",
            dd : "%d days",
            M : "a month",
            MM : "%d months",
            y : "a year",
            yy : "%d years"
        },
        ordinal : function (number) {
            var b = number % 10,
                output = (~~ (number % 100 / 10) === 1) ? 'th' :
                (b === 1) ? 'st' :
                (b === 2) ? 'nd' :
                (b === 3) ? 'rd' : 'th';
            return number + output;
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : canadian english (en-ca)
// author : Jonathan Abourbih : https://github.com/jonbca

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('en-ca', {
        months : "January_February_March_April_May_June_July_August_September_October_November_December".split("_"),
        monthsShort : "Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec".split("_"),
        weekdays : "Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday".split("_"),
        weekdaysShort : "Sun_Mon_Tue_Wed_Thu_Fri_Sat".split("_"),
        weekdaysMin : "Su_Mo_Tu_We_Th_Fr_Sa".split("_"),
        longDateFormat : {
            LT : "h:mm A",
            L : "YYYY-MM-DD",
            LL : "D MMMM, YYYY",
            LLL : "D MMMM, YYYY LT",
            LLLL : "dddd, D MMMM, YYYY LT"
        },
        calendar : {
            sameDay : '[Today at] LT',
            nextDay : '[Tomorrow at] LT',
            nextWeek : 'dddd [at] LT',
            lastDay : '[Yesterday at] LT',
            lastWeek : '[Last] dddd [at] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "in %s",
            past : "%s ago",
            s : "a few seconds",
            m : "a minute",
            mm : "%d minutes",
            h : "an hour",
            hh : "%d hours",
            d : "a day",
            dd : "%d days",
            M : "a month",
            MM : "%d months",
            y : "a year",
            yy : "%d years"
        },
        ordinal : function (number) {
            var b = number % 10,
                output = (~~ (number % 100 / 10) === 1) ? 'th' :
                (b === 1) ? 'st' :
                (b === 2) ? 'nd' :
                (b === 3) ? 'rd' : 'th';
            return number + output;
        }
    });
}));
// moment.js language configuration
// language : great britain english (en-gb)
// author : Chris Gedrim : https://github.com/chrisgedrim

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('en-gb', {
        months : "January_February_March_April_May_June_July_August_September_October_November_December".split("_"),
        monthsShort : "Jan_Feb_Mar_Apr_May_Jun_Jul_Aug_Sep_Oct_Nov_Dec".split("_"),
        weekdays : "Sunday_Monday_Tuesday_Wednesday_Thursday_Friday_Saturday".split("_"),
        weekdaysShort : "Sun_Mon_Tue_Wed_Thu_Fri_Sat".split("_"),
        weekdaysMin : "Su_Mo_Tu_We_Th_Fr_Sa".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd, D MMMM YYYY LT"
        },
        calendar : {
            sameDay : '[Today at] LT',
            nextDay : '[Tomorrow at] LT',
            nextWeek : 'dddd [at] LT',
            lastDay : '[Yesterday at] LT',
            lastWeek : '[Last] dddd [at] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "in %s",
            past : "%s ago",
            s : "a few seconds",
            m : "a minute",
            mm : "%d minutes",
            h : "an hour",
            hh : "%d hours",
            d : "a day",
            dd : "%d days",
            M : "a month",
            MM : "%d months",
            y : "a year",
            yy : "%d years"
        },
        ordinal : function (number) {
            var b = number % 10,
                output = (~~ (number % 100 / 10) === 1) ? 'th' :
                (b === 1) ? 'st' :
                (b === 2) ? 'nd' :
                (b === 3) ? 'rd' : 'th';
            return number + output;
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : esperanto (eo)
// author : Colin Dean : https://github.com/colindean
// komento: Mi estas malcerta se mi korekte traktis akuzativojn en tiu traduko.
//          Se ne, bonvolu korekti kaj avizi min por ke mi povas lerni!

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('eo', {
        months : "januaro_februaro_marto_aprilo_majo_junio_julio_aХ­gusto_septembro_oktobro_novembro_decembro".split("_"),
        monthsShort : "jan_feb_mar_apr_maj_jun_jul_aХ­g_sep_okt_nov_dec".split("_"),
        weekdays : "DimanФo_Lundo_Mardo_Merkredo_ФДaХ­do_Vendredo_Sabato".split("_"),
        weekdaysShort : "Dim_Lun_Mard_Merk_ФДaХ­_Ven_Sab".split("_"),
        weekdaysMin : "Di_Lu_Ma_Me_ФДa_Ve_Sa".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "YYYY-MM-DD",
            LL : "D[-an de] MMMM, YYYY",
            LLL : "D[-an de] MMMM, YYYY LT",
            LLLL : "dddd, [la] D[-an de] MMMM, YYYY LT"
        },
        meridiem : function (hours, minutes, isLower) {
            if (hours > 11) {
                return isLower ? 'p.t.m.' : 'P.T.M.';
            } else {
                return isLower ? 'a.t.m.' : 'A.T.M.';
            }
        },
        calendar : {
            sameDay : '[HodiaХ­ je] LT',
            nextDay : '[MorgaХ­ je] LT',
            nextWeek : 'dddd [je] LT',
            lastDay : '[HieraХ­ je] LT',
            lastWeek : '[pasinta] dddd [je] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "je %s",
            past : "antaХ­ %s",
            s : "sekundoj",
            m : "minuto",
            mm : "%d minutoj",
            h : "horo",
            hh : "%d horoj",
            d : "tago",//ne 'diurno', Фar estas uzita por proksimumo
            dd : "%d tagoj",
            M : "monato",
            MM : "%d monatoj",
            y : "jaro",
            yy : "%d jaroj"
        },
        ordinal : "%da",
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : spanish (es)
// author : Julio NapurУ­ : https://github.com/julionc

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('es', {
        months : "enero_febrero_marzo_abril_mayo_junio_julio_agosto_septiembre_octubre_noviembre_diciembre".split("_"),
        monthsShort : "ene._feb._mar._abr._may._jun._jul._ago._sep._oct._nov._dic.".split("_"),
        weekdays : "domingo_lunes_martes_miУЉrcoles_jueves_viernes_sУЁbado".split("_"),
        weekdaysShort : "dom._lun._mar._miУЉ._jue._vie._sУЁb.".split("_"),
        weekdaysMin : "Do_Lu_Ma_Mi_Ju_Vi_SУЁ".split("_"),
        longDateFormat : {
            LT : "H:mm",
            L : "DD/MM/YYYY",
            LL : "D [de] MMMM [de] YYYY",
            LLL : "D [de] MMMM [de] YYYY LT",
            LLLL : "dddd, D [de] MMMM [de] YYYY LT"
        },
        calendar : {
            sameDay : function () {
                return '[hoy a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
            },
            nextDay : function () {
                return '[maУБana a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
            },
            nextWeek : function () {
                return 'dddd [a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
            },
            lastDay : function () {
                return '[ayer a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
            },
            lastWeek : function () {
                return '[el] dddd [pasado a la' + ((this.hours() !== 1) ? 's' : '') + '] LT';
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : "en %s",
            past : "hace %s",
            s : "unos segundos",
            m : "un minuto",
            mm : "%d minutos",
            h : "una hora",
            hh : "%d horas",
            d : "un dУ­a",
            dd : "%d dУ­as",
            M : "un mes",
            MM : "%d meses",
            y : "un aУБo",
            yy : "%d aУБos"
        },
        ordinal : '%dТК',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : estonian (et)
// author : Henry Kehlmann : https://github.com/madhenry

(function (factory) {
    factory(moment);
}(function (moment) {
    function translateSeconds(number, withoutSuffix, key, isFuture) {
        return (isFuture || withoutSuffix) ? 'paari sekundi' : 'paar sekundit';
    }

    return moment.lang('et', {
        months        : "jaanuar_veebruar_mУЄrts_aprill_mai_juuni_juuli_august_september_oktoober_november_detsember".split("_"),
        monthsShort   : "jaan_veebr_mУЄrts_apr_mai_juuni_juuli_aug_sept_okt_nov_dets".split("_"),
        weekdays      : "pУМhapУЄev_esmaspУЄev_teisipУЄev_kolmapУЄev_neljapУЄev_reede_laupУЄev".split("_"),
        weekdaysShort : "P_E_T_K_N_R_L".split("_"),
        weekdaysMin   : "P_E_T_K_N_R_L".split("_"),
        longDateFormat : {
            LT   : "H:mm",
            L    : "DD.MM.YYYY",
            LL   : "D. MMMM YYYY",
            LLL  : "D. MMMM YYYY LT",
            LLLL : "dddd, D. MMMM YYYY LT"
        },
        calendar : {
            sameDay  : '[TУЄna,] LT',
            nextDay  : '[Homme,] LT',
            nextWeek : '[JУЄrgmine] dddd LT',
            lastDay  : '[Eile,] LT',
            lastWeek : '[Eelmine] dddd LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "%s pУЄrast",
            past   : "%s tagasi",
            s      : translateSeconds,
            m      : "minut",
            mm     : "%d minutit",
            h      : "tund",
            hh     : "%d tundi",
            d      : "pУЄev",
            dd     : "%d pУЄeva",
            M      : "kuu",
            MM     : "%d kuud",
            y      : "aasta",
            yy     : "%d aastat"
        },
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : euskara (eu)
// author : Eneko Illarramendi : https://github.com/eillarra

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('eu', {
        months : "urtarrila_otsaila_martxoa_apirila_maiatza_ekaina_uztaila_abuztua_iraila_urria_azaroa_abendua".split("_"),
        monthsShort : "urt._ots._mar._api._mai._eka._uzt._abu._ira._urr._aza._abe.".split("_"),
        weekdays : "igandea_astelehena_asteartea_asteazkena_osteguna_ostirala_larunbata".split("_"),
        weekdaysShort : "ig._al._ar._az._og._ol._lr.".split("_"),
        weekdaysMin : "ig_al_ar_az_og_ol_lr".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "YYYY-MM-DD",
            LL : "YYYY[ko] MMMM[ren] D[a]",
            LLL : "YYYY[ko] MMMM[ren] D[a] LT",
            LLLL : "dddd, YYYY[ko] MMMM[ren] D[a] LT",
            l : "YYYY-M-D",
            ll : "YYYY[ko] MMM D[a]",
            lll : "YYYY[ko] MMM D[a] LT",
            llll : "ddd, YYYY[ko] MMM D[a] LT"
        },
        calendar : {
            sameDay : '[gaur] LT[etan]',
            nextDay : '[bihar] LT[etan]',
            nextWeek : 'dddd LT[etan]',
            lastDay : '[atzo] LT[etan]',
            lastWeek : '[aurreko] dddd LT[etan]',
            sameElse : 'L'
        },
        relativeTime : {
            future : "%s barru",
            past : "duela %s",
            s : "segundo batzuk",
            m : "minutu bat",
            mm : "%d minutu",
            h : "ordu bat",
            hh : "%d ordu",
            d : "egun bat",
            dd : "%d egun",
            M : "hilabete bat",
            MM : "%d hilabete",
            y : "urte bat",
            yy : "%d urte"
        },
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : Persian Language
// author : Ebrahim Byagowi : https://github.com/ebraminio

(function (factory) {
    factory(moment);
}(function (moment) {
    var symbolMap = {
        '1': 'лБ',
        '2': 'лВ',
        '3': 'лГ',
        '4': 'лД',
        '5': 'лЕ',
        '6': 'лЖ',
        '7': 'лЗ',
        '8': 'лИ',
        '9': 'лЙ',
        '0': 'лА'
    }, numberMap = {
        'лБ': '1',
        'лВ': '2',
        'лГ': '3',
        'лД': '4',
        'лЕ': '5',
        'лЖ': '6',
        'лЗ': '7',
        'лИ': '8',
        'лЙ': '9',
        'лА': '0'
    };

    return moment.lang('fa', {
        months : 'киЇййлй_ййиБлй_йиЇиБиГ_иЂйиБлй_йй_кйиІй_кйиІлй_иЇйиЊ_иГйОиЊиЇйиЈиБ_иЇкЉиЊиЈиБ_ййиЇйиЈиБ_иЏиГиЇйиЈиБ'.split('_'),
        monthsShort : 'киЇййлй_ййиБлй_йиЇиБиГ_иЂйиБлй_йй_кйиІй_кйиІлй_иЇйиЊ_иГйОиЊиЇйиЈиБ_иЇкЉиЊиЈиБ_ййиЇйиЈиБ_иЏиГиЇйиЈиБ'.split('_'),
        weekdays : 'лкЉ\u200cиДйиЈй_иЏйиДйиЈй_иГй\u200cиДйиЈй_кйиЇиБиДйиЈй_йОйиЌ\u200cиДйиЈй_иЌйиЙй_иДйиЈй'.split('_'),
        weekdaysShort : 'лкЉ\u200cиДйиЈй_иЏйиДйиЈй_иГй\u200cиДйиЈй_кйиЇиБиДйиЈй_йОйиЌ\u200cиДйиЈй_иЌйиЙй_иДйиЈй'.split('_'),
        weekdaysMin : 'л_иЏ_иГ_к_йО_иЌ_иД'.split('_'),
        longDateFormat : {
            LT : 'HH:mm',
            L : 'DD/MM/YYYY',
            LL : 'D MMMM YYYY',
            LLL : 'D MMMM YYYY LT',
            LLLL : 'dddd, D MMMM YYYY LT'
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 12) {
                return "йиЈй иЇиВ иИйиБ";
            } else {
                return "иЈиЙиЏ иЇиВ иИйиБ";
            }
        },
        calendar : {
            sameDay : '[иЇйиБйиВ иГиЇиЙиЊ] LT',
            nextDay : '[йиБиЏиЇ иГиЇиЙиЊ] LT',
            nextWeek : 'dddd [иГиЇиЙиЊ] LT',
            lastDay : '[иЏлиБйиВ иГиЇиЙиЊ] LT',
            lastWeek : 'dddd [йОлиД] [иГиЇиЙиЊ] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : 'иЏиБ %s',
            past : '%s йОлиД',
            s : 'кйиЏлй иЋиЇйлй',
            m : 'лкЉ иЏйлйй',
            mm : '%d иЏйлйй',
            h : 'лкЉ иГиЇиЙиЊ',
            hh : '%d иГиЇиЙиЊ',
            d : 'лкЉ иБйиВ',
            dd : '%d иБйиВ',
            M : 'лкЉ йиЇй',
            MM : '%d йиЇй',
            y : 'лкЉ иГиЇй',
            yy : '%d иГиЇй'
        },
        preparse: function (string) {
            return string.replace(/[лА-лЙ]/g, function (match) {
                return numberMap[match];
            }).replace(/и/g, ',');
        },
        postformat: function (string) {
            return string.replace(/\d/g, function (match) {
                return symbolMap[match];
            }).replace(/,/g, 'и');
        },
        ordinal : '%dй',
        week : {
            dow : 6, // Saturday is the first day of the week.
            doy : 12 // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : finnish (fi)
// author : Tarmo Aidantausta : https://github.com/bleadof

(function (factory) {
    factory(moment);
}(function (moment) {
    var numbers_past = 'nolla yksi kaksi kolme neljУЄ viisi kuusi seitsemУЄn kahdeksan yhdeksУЄn'.split(' '),
        numbers_future = ['nolla', 'yhden', 'kahden', 'kolmen', 'neljУЄn', 'viiden', 'kuuden',
                          numbers_past[7], numbers_past[8], numbers_past[9]];

    function translate(number, withoutSuffix, key, isFuture) {
        var result = "";
        switch (key) {
        case 's':
            return isFuture ? 'muutaman sekunnin' : 'muutama sekunti';
        case 'm':
            return isFuture ? 'minuutin' : 'minuutti';
        case 'mm':
            result = isFuture ? 'minuutin' : 'minuuttia';
            break;
        case 'h':
            return isFuture ? 'tunnin' : 'tunti';
        case 'hh':
            result = isFuture ? 'tunnin' : 'tuntia';
            break;
        case 'd':
            return isFuture ? 'pУЄivУЄn' : 'pУЄivУЄ';
        case 'dd':
            result = isFuture ? 'pУЄivУЄn' : 'pУЄivУЄУЄ';
            break;
        case 'M':
            return isFuture ? 'kuukauden' : 'kuukausi';
        case 'MM':
            result = isFuture ? 'kuukauden' : 'kuukautta';
            break;
        case 'y':
            return isFuture ? 'vuoden' : 'vuosi';
        case 'yy':
            result = isFuture ? 'vuoden' : 'vuotta';
            break;
        }
        result = verbal_number(number, isFuture) + " " + result;
        return result;
    }

    function verbal_number(number, isFuture) {
        return number < 10 ? (isFuture ? numbers_future[number] : numbers_past[number]) : number;
    }

    return moment.lang('fi', {
        months : "tammikuu_helmikuu_maaliskuu_huhtikuu_toukokuu_kesУЄkuu_heinУЄkuu_elokuu_syyskuu_lokakuu_marraskuu_joulukuu".split("_"),
        monthsShort : "tammi_helmi_maalis_huhti_touko_kesУЄ_heinУЄ_elo_syys_loka_marras_joulu".split("_"),
        weekdays : "sunnuntai_maanantai_tiistai_keskiviikko_torstai_perjantai_lauantai".split("_"),
        weekdaysShort : "su_ma_ti_ke_to_pe_la".split("_"),
        weekdaysMin : "su_ma_ti_ke_to_pe_la".split("_"),
        longDateFormat : {
            LT : "HH.mm",
            L : "DD.MM.YYYY",
            LL : "Do MMMM[ta] YYYY",
            LLL : "Do MMMM[ta] YYYY, [klo] LT",
            LLLL : "dddd, Do MMMM[ta] YYYY, [klo] LT",
            l : "D.M.YYYY",
            ll : "Do MMM YYYY",
            lll : "Do MMM YYYY, [klo] LT",
            llll : "ddd, Do MMM YYYY, [klo] LT"
        },
        calendar : {
            sameDay : '[tУЄnУЄУЄn] [klo] LT',
            nextDay : '[huomenna] [klo] LT',
            nextWeek : 'dddd [klo] LT',
            lastDay : '[eilen] [klo] LT',
            lastWeek : '[viime] dddd[na] [klo] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "%s pУЄУЄstУЄ",
            past : "%s sitten",
            s : translate,
            m : translate,
            mm : translate,
            h : translate,
            hh : translate,
            d : translate,
            dd : translate,
            M : translate,
            MM : translate,
            y : translate,
            yy : translate
        },
        ordinal : "%d.",
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : faroese (fo)
// author : Ragnar Johannesen : https://github.com/ragnar123

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('fo', {
        months : "januar_februar_mars_aprУ­l_mai_juni_juli_august_september_oktober_november_desember".split("_"),
        monthsShort : "jan_feb_mar_apr_mai_jun_jul_aug_sep_okt_nov_des".split("_"),
        weekdays : "sunnudagur_mУЁnadagur_tУНsdagur_mikudagur_hУГsdagur_frУ­ggjadagur_leygardagur".split("_"),
        weekdaysShort : "sun_mУЁn_tУНs_mik_hУГs_frУ­_ley".split("_"),
        weekdaysMin : "su_mУЁ_tУН_mi_hУГ_fr_le".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd D. MMMM, YYYY LT"
        },
        calendar : {
            sameDay : '[У dag kl.] LT',
            nextDay : '[У morgin kl.] LT',
            nextWeek : 'dddd [kl.] LT',
            lastDay : '[У gjУЁr kl.] LT',
            lastWeek : '[sУ­УАstu] dddd [kl] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "um %s",
            past : "%s sУ­УАani",
            s : "fУЁ sekund",
            m : "ein minutt",
            mm : "%d minuttir",
            h : "ein tУ­mi",
            hh : "%d tУ­mar",
            d : "ein dagur",
            dd : "%d dagar",
            M : "ein mУЁnaУАi",
            MM : "%d mУЁnaУАir",
            y : "eitt УЁr",
            yy : "%d УЁr"
        },
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : canadian french (fr-ca)
// author : Jonathan Abourbih : https://github.com/jonbca

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('fr-ca', {
        months : "janvier_fУЉvrier_mars_avril_mai_juin_juillet_aoУЛt_septembre_octobre_novembre_dУЉcembre".split("_"),
        monthsShort : "janv._fУЉvr._mars_avr._mai_juin_juil._aoУЛt_sept._oct._nov._dУЉc.".split("_"),
        weekdays : "dimanche_lundi_mardi_mercredi_jeudi_vendredi_samedi".split("_"),
        weekdaysShort : "dim._lun._mar._mer._jeu._ven._sam.".split("_"),
        weekdaysMin : "Di_Lu_Ma_Me_Je_Ve_Sa".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "YYYY-MM-DD",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd D MMMM YYYY LT"
        },
        calendar : {
            sameDay: "[Aujourd'hui У ] LT",
            nextDay: '[Demain У ] LT',
            nextWeek: 'dddd [У ] LT',
            lastDay: '[Hier У ] LT',
            lastWeek: 'dddd [dernier У ] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : "dans %s",
            past : "il y a %s",
            s : "quelques secondes",
            m : "une minute",
            mm : "%d minutes",
            h : "une heure",
            hh : "%d heures",
            d : "un jour",
            dd : "%d jours",
            M : "un mois",
            MM : "%d mois",
            y : "un an",
            yy : "%d ans"
        },
        ordinal : function (number) {
            return number + (number === 1 ? 'er' : '');
        }
    });
}));
// moment.js language configuration
// language : french (fr)
// author : John Fischer : https://github.com/jfroffice

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('fr', {
        months : "janvier_fУЉvrier_mars_avril_mai_juin_juillet_aoУЛt_septembre_octobre_novembre_dУЉcembre".split("_"),
        monthsShort : "janv._fУЉvr._mars_avr._mai_juin_juil._aoУЛt_sept._oct._nov._dУЉc.".split("_"),
        weekdays : "dimanche_lundi_mardi_mercredi_jeudi_vendredi_samedi".split("_"),
        weekdaysShort : "dim._lun._mar._mer._jeu._ven._sam.".split("_"),
        weekdaysMin : "Di_Lu_Ma_Me_Je_Ve_Sa".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd D MMMM YYYY LT"
        },
        calendar : {
            sameDay: "[Aujourd'hui У ] LT",
            nextDay: '[Demain У ] LT',
            nextWeek: 'dddd [У ] LT',
            lastDay: '[Hier У ] LT',
            lastWeek: 'dddd [dernier У ] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : "dans %s",
            past : "il y a %s",
            s : "quelques secondes",
            m : "une minute",
            mm : "%d minutes",
            h : "une heure",
            hh : "%d heures",
            d : "un jour",
            dd : "%d jours",
            M : "un mois",
            MM : "%d mois",
            y : "un an",
            yy : "%d ans"
        },
        ordinal : function (number) {
            return number + (number === 1 ? 'er' : '');
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : galician (gl)
// author : Juan G. Hurtado : https://github.com/juanghurtado

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('gl', {
        months : "Xaneiro_Febreiro_Marzo_Abril_Maio_XuУБo_Xullo_Agosto_Setembro_Outubro_Novembro_Decembro".split("_"),
        monthsShort : "Xan._Feb._Mar._Abr._Mai._XuУБ._Xul._Ago._Set._Out._Nov._Dec.".split("_"),
        weekdays : "Domingo_Luns_Martes_MУЉrcores_Xoves_Venres_SУЁbado".split("_"),
        weekdaysShort : "Dom._Lun._Mar._MУЉr._Xov._Ven._SУЁb.".split("_"),
        weekdaysMin : "Do_Lu_Ma_MУЉ_Xo_Ve_SУЁ".split("_"),
        longDateFormat : {
            LT : "H:mm",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd D MMMM YYYY LT"
        },
        calendar : {
            sameDay : function () {
                return '[hoxe ' + ((this.hours() !== 1) ? 'УЁs' : 'УЁ') + '] LT';
            },
            nextDay : function () {
                return '[maУБУЁ ' + ((this.hours() !== 1) ? 'УЁs' : 'УЁ') + '] LT';
            },
            nextWeek : function () {
                return 'dddd [' + ((this.hours() !== 1) ? 'УЁs' : 'a') + '] LT';
            },
            lastDay : function () {
                return '[onte ' + ((this.hours() !== 1) ? 'УЁ' : 'a') + '] LT';
            },
            lastWeek : function () {
                return '[o] dddd [pasado ' + ((this.hours() !== 1) ? 'УЁs' : 'a') + '] LT';
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : function (str) {
                if (str === "uns segundos") {
                    return "nuns segundos";
                }
                return "en " + str;
            },
            past : "hai %s",
            s : "uns segundos",
            m : "un minuto",
            mm : "%d minutos",
            h : "unha hora",
            hh : "%d horas",
            d : "un dУ­a",
            dd : "%d dУ­as",
            M : "un mes",
            MM : "%d meses",
            y : "un ano",
            yy : "%d anos"
        },
        ordinal : '%dТК',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : Hebrew (he)
// author : Tomer Cohen : https://github.com/tomer
// author : Moshe Simantov : https://github.com/DevelopmentIL
// author : Tal Ater : https://github.com/TalAter

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('he', {
        months : "зз зззЈ_зЄззЈзззЈ_ззЈзЅ_ззЄзЈзз_ззз_ззз з_зззз_зззззЁз_зЁзЄззззЈ_зззЇззззЈ_з зззззЈ_ззІзззЈ".split("_"),
        monthsShort : "зз ззГ_зЄззЈзГ_ззЈзЅ_ззЄзЈзГ_ззз_ззз з_зззз_ззззГ_зЁзЄззГ_зззЇзГ_з зззГ_ззІззГ".split("_"),
        weekdays : "зЈззЉзз_зЉз з_зЉзззЉз_зЈзззЂз_ззззЉз_зЉззЉз_зЉззЊ".split("_"),
        weekdaysShort : "ззГ_ззГ_ззГ_ззГ_ззГ_ззГ_зЉзГ".split("_"),
        weekdaysMin : "з_з_з_з_з_з_зЉ".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD/MM/YYYY",
            LL : "D [з]MMMM YYYY",
            LLL : "D [з]MMMM YYYY LT",
            LLLL : "dddd, D [з]MMMM YYYY LT",
            l : "D/M/YYYY",
            ll : "D MMM YYYY",
            lll : "D MMM YYYY LT",
            llll : "ddd, D MMM YYYY LT"
        },
        calendar : {
            sameDay : '[зззз зжО]LT',
            nextDay : '[зззЈ зжО]LT',
            nextWeek : 'dddd [ззЉзЂз] LT',
            lastDay : '[ззЊззз зжО]LT',
            lastWeek : '[зззз] dddd [ззззЈзз ззЉзЂз] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "ззЂзз %s",
            past : "ззЄз з %s",
            s : "ззЁзЄзЈ зЉз зззЊ",
            m : "ззЇз",
            mm : "%d ззЇззЊ",
            h : "зЉзЂз",
            hh : function (number) {
                if (number === 2) {
                    return "зЉзЂзЊззз";
                }
                return number + " зЉзЂззЊ";
            },
            d : "ззз",
            dd : function (number) {
                if (number === 2) {
                    return "зззззз";
                }
                return number + " зззз";
            },
            M : "ззззЉ",
            MM : function (number) {
                if (number === 2) {
                    return "ззззЉззз";
                }
                return number + " ззззЉзз";
            },
            y : "зЉз з",
            yy : function (number) {
                if (number === 2) {
                    return "зЉз зЊззз";
                }
                return number + " зЉз зз";
            }
        }
    });
}));
// moment.js language configuration
// language : hindi (hi)
// author : Mayank Singhal : https://github.com/mayanksinghal

(function (factory) {
    factory(moment);
}(function (moment) {
    var symbolMap = {
        '1': 'рЅЇ',
        '2': 'рЅЈ',
        '3': 'рЅЉ',
        '4': 'рЅЊ',
        '5': 'рЅЋ',
        '6': 'рЅЌ',
        '7': 'рЅ­',
        '8': 'рЅЎ',
        '9': 'рЅЏ',
        '0': 'рЅІ'
    },
    numberMap = {
        'рЅЇ': '1',
        'рЅЈ': '2',
        'рЅЉ': '3',
        'рЅЊ': '4',
        'рЅЋ': '5',
        'рЅЌ': '6',
        'рЅ­': '7',
        'рЅЎ': '8',
        'рЅЏ': '9',
        'рЅІ': '0'
    };

    return moment.lang('hi', {
        months : 'рЄрЄЈрЄЕрЄАрЅ_рЄЋрЄМрЄАрЄЕрЄАрЅ_рЄЎрЄОрЄАрЅрЄ_рЄрЄЊрЅрЄАрЅрЄВ_рЄЎрЄ_рЄрЅрЄЈ_рЄрЅрЄВрЄОрЄ_рЄрЄрЄИрЅрЄЄ_рЄИрЄПрЄЄрЄЎрЅрЄЌрЄА_рЄрЄрЅрЄрЅрЄЌрЄА_рЄЈрЄЕрЄЎрЅрЄЌрЄА_рЄІрЄПрЄИрЄЎрЅрЄЌрЄА'.split("_"),
        monthsShort : 'рЄрЄЈ._рЄЋрЄМрЄА._рЄЎрЄОрЄАрЅрЄ_рЄрЄЊрЅрЄАрЅ._рЄЎрЄ_рЄрЅрЄЈ_рЄрЅрЄВ._рЄрЄ._рЄИрЄПрЄЄ._рЄрЄрЅрЄрЅ._рЄЈрЄЕ._рЄІрЄПрЄИ.'.split("_"),
        weekdays : 'рЄАрЄЕрЄПрЄЕрЄОрЄА_рЄИрЅрЄЎрЄЕрЄОрЄА_рЄЎрЄрЄрЄВрЄЕрЄОрЄА_рЄЌрЅрЄЇрЄЕрЄОрЄА_рЄрЅрЄАрЅрЄЕрЄОрЄА_рЄЖрЅрЄрЅрЄАрЄЕрЄОрЄА_рЄЖрЄЈрЄПрЄЕрЄОрЄА'.split("_"),
        weekdaysShort : 'рЄАрЄЕрЄП_рЄИрЅрЄЎ_рЄЎрЄрЄрЄВ_рЄЌрЅрЄЇ_рЄрЅрЄАрЅ_рЄЖрЅрЄрЅрЄА_рЄЖрЄЈрЄП'.split("_"),
        weekdaysMin : 'рЄА_рЄИрЅ_рЄЎрЄ_рЄЌрЅ_рЄрЅ_рЄЖрЅ_рЄЖ'.split("_"),
        longDateFormat : {
            LT : "A h:mm рЄЌрЄрЅ",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY, LT",
            LLLL : "dddd, D MMMM YYYY, LT"
        },
        calendar : {
            sameDay : '[рЄрЄ] LT',
            nextDay : '[рЄрЄВ] LT',
            nextWeek : 'dddd, LT',
            lastDay : '[рЄрЄВ] LT',
            lastWeek : '[рЄЊрЄПрЄрЄВрЅ] dddd, LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "%s рЄЎрЅрЄ",
            past : "%s рЄЊрЄЙрЄВрЅ",
            s : "рЄрЅрЄ рЄЙрЅ рЄрЅрЄЗрЄЃ",
            m : "рЄрЄ рЄЎрЄПрЄЈрЄ",
            mm : "%d рЄЎрЄПрЄЈрЄ",
            h : "рЄрЄ рЄрЄрЄрЄО",
            hh : "%d рЄрЄрЄрЅ",
            d : "рЄрЄ рЄІрЄПрЄЈ",
            dd : "%d рЄІрЄПрЄЈ",
            M : "рЄрЄ рЄЎрЄЙрЅрЄЈрЅ",
            MM : "%d рЄЎрЄЙрЅрЄЈрЅ",
            y : "рЄрЄ рЄЕрЄАрЅрЄЗ",
            yy : "%d рЄЕрЄАрЅрЄЗ"
        },
        preparse: function (string) {
            return string.replace(/[рЅЇрЅЈрЅЉрЅЊрЅЋрЅЌрЅ­рЅЎрЅЏрЅІ]/g, function (match) {
                return numberMap[match];
            });
        },
        postformat: function (string) {
            return string.replace(/\d/g, function (match) {
                return symbolMap[match];
            });
        },
        // Hindi notation for meridiems are quite fuzzy in practice. While there exists
        // a rigid notion of a 'Pahar' it is not used as rigidly in modern Hindi.
        meridiem : function (hour, minute, isLower) {
            if (hour < 4) {
                return "рЄАрЄОрЄЄ";
            } else if (hour < 10) {
                return "рЄИрЅрЄЌрЄЙ";
            } else if (hour < 17) {
                return "рЄІрЅрЄЊрЄЙрЄА";
            } else if (hour < 20) {
                return "рЄЖрЄОрЄЎ";
            } else {
                return "рЄАрЄОрЄЄ";
            }
        },
        week : {
            dow : 0, // Sunday is the first day of the week.
            doy : 6  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : hrvatski (hr)
// author : Bojan MarkoviФ : https://github.com/bmarkovic

// based on (sl) translation by Robert SedovХЁek

(function (factory) {
    factory(moment);
}(function (moment) {

    function translate(number, withoutSuffix, key) {
        var result = number + " ";
        switch (key) {
        case 'm':
            return withoutSuffix ? 'jedna minuta' : 'jedne minute';
        case 'mm':
            if (number === 1) {
                result += 'minuta';
            } else if (number === 2 || number === 3 || number === 4) {
                result += 'minute';
            } else {
                result += 'minuta';
            }
            return result;
        case 'h':
            return withoutSuffix ? 'jedan sat' : 'jednog sata';
        case 'hh':
            if (number === 1) {
                result += 'sat';
            } else if (number === 2 || number === 3 || number === 4) {
                result += 'sata';
            } else {
                result += 'sati';
            }
            return result;
        case 'dd':
            if (number === 1) {
                result += 'dan';
            } else {
                result += 'dana';
            }
            return result;
        case 'MM':
            if (number === 1) {
                result += 'mjesec';
            } else if (number === 2 || number === 3 || number === 4) {
                result += 'mjeseca';
            } else {
                result += 'mjeseci';
            }
            return result;
        case 'yy':
            if (number === 1) {
                result += 'godina';
            } else if (number === 2 || number === 3 || number === 4) {
                result += 'godine';
            } else {
                result += 'godina';
            }
            return result;
        }
    }

    return moment.lang('hr', {
        months : "sjeФanj_veljaФa_oХОujak_travanj_svibanj_lipanj_srpanj_kolovoz_rujan_listopad_studeni_prosinac".split("_"),
        monthsShort : "sje._vel._oХОu._tra._svi._lip._srp._kol._ruj._lis._stu._pro.".split("_"),
        weekdays : "nedjelja_ponedjeljak_utorak_srijeda_Фetvrtak_petak_subota".split("_"),
        weekdaysShort : "ned._pon._uto._sri._Фet._pet._sub.".split("_"),
        weekdaysMin : "ne_po_ut_sr_Фe_pe_su".split("_"),
        longDateFormat : {
            LT : "H:mm",
            L : "DD. MM. YYYY",
            LL : "D. MMMM YYYY",
            LLL : "D. MMMM YYYY LT",
            LLLL : "dddd, D. MMMM YYYY LT"
        },
        calendar : {
            sameDay  : '[danas u] LT',
            nextDay  : '[sutra u] LT',

            nextWeek : function () {
                switch (this.day()) {
                case 0:
                    return '[u] [nedjelju] [u] LT';
                case 3:
                    return '[u] [srijedu] [u] LT';
                case 6:
                    return '[u] [subotu] [u] LT';
                case 1:
                case 2:
                case 4:
                case 5:
                    return '[u] dddd [u] LT';
                }
            },
            lastDay  : '[juФer u] LT',
            lastWeek : function () {
                switch (this.day()) {
                case 0:
                case 3:
                    return '[proХЁlu] dddd [u] LT';
                case 6:
                    return '[proХЁle] [subote] [u] LT';
                case 1:
                case 2:
                case 4:
                case 5:
                    return '[proХЁli] dddd [u] LT';
                }
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : "za %s",
            past   : "prije %s",
            s      : "par sekundi",
            m      : translate,
            mm     : translate,
            h      : translate,
            hh     : translate,
            d      : "dan",
            dd     : translate,
            M      : "mjesec",
            MM     : translate,
            y      : "godinu",
            yy     : translate
        },
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : hungarian (hu)
// author : Adam Brunner : https://github.com/adambrunner

(function (factory) {
    factory(moment);
}(function (moment) {
    var weekEndings = 'vasУЁrnap hУЉtfХn kedden szerdУЁn csУМtУЖrtУЖkУЖn pУЉnteken szombaton'.split(' ');

    function translate(number, withoutSuffix, key, isFuture) {
        var num = number,
            suffix;

        switch (key) {
        case 's':
            return (isFuture || withoutSuffix) ? 'nУЉhУЁny mУЁsodperc' : 'nУЉhУЁny mУЁsodperce';
        case 'm':
            return 'egy' + (isFuture || withoutSuffix ? ' perc' : ' perce');
        case 'mm':
            return num + (isFuture || withoutSuffix ? ' perc' : ' perce');
        case 'h':
            return 'egy' + (isFuture || withoutSuffix ? ' УГra' : ' УГrУЁja');
        case 'hh':
            return num + (isFuture || withoutSuffix ? ' УГra' : ' УГrУЁja');
        case 'd':
            return 'egy' + (isFuture || withoutSuffix ? ' nap' : ' napja');
        case 'dd':
            return num + (isFuture || withoutSuffix ? ' nap' : ' napja');
        case 'M':
            return 'egy' + (isFuture || withoutSuffix ? ' hУГnap' : ' hУГnapja');
        case 'MM':
            return num + (isFuture || withoutSuffix ? ' hУГnap' : ' hУГnapja');
        case 'y':
            return 'egy' + (isFuture || withoutSuffix ? ' УЉv' : ' УЉve');
        case 'yy':
            return num + (isFuture || withoutSuffix ? ' УЉv' : ' УЉve');
        }

        return '';
    }

    function week(isFuture) {
        return (isFuture ? '' : '[mУКlt] ') + '[' + weekEndings[this.day()] + '] LT[-kor]';
    }

    return moment.lang('hu', {
        months : "januУЁr_februУЁr_mУЁrcius_УЁprilis_mУЁjus_jУКnius_jУКlius_augusztus_szeptember_oktУГber_november_december".split("_"),
        monthsShort : "jan_feb_mУЁrc_УЁpr_mУЁj_jУКn_jУКl_aug_szept_okt_nov_dec".split("_"),
        weekdays : "vasУЁrnap_hУЉtfХ_kedd_szerda_csУМtУЖrtУЖk_pУЉntek_szombat".split("_"),
        weekdaysShort : "vas_hУЉt_kedd_sze_csУМt_pУЉn_szo".split("_"),
        weekdaysMin : "v_h_k_sze_cs_p_szo".split("_"),
        longDateFormat : {
            LT : "H:mm",
            L : "YYYY.MM.DD.",
            LL : "YYYY. MMMM D.",
            LLL : "YYYY. MMMM D., LT",
            LLLL : "YYYY. MMMM D., dddd LT"
        },
        calendar : {
            sameDay : '[ma] LT[-kor]',
            nextDay : '[holnap] LT[-kor]',
            nextWeek : function () {
                return week.call(this, true);
            },
            lastDay : '[tegnap] LT[-kor]',
            lastWeek : function () {
                return week.call(this, false);
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : "%s mУКlva",
            past : "%s",
            s : translate,
            m : translate,
            mm : translate,
            h : translate,
            hh : translate,
            d : translate,
            dd : translate,
            M : translate,
            MM : translate,
            y : translate,
            yy : translate
        },
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : Bahasa Indonesia (id)
// author : Mohammad Satrio Utomo : https://github.com/tyok
// reference: http://id.wikisource.org/wiki/Pedoman_Umum_Ejaan_Bahasa_Indonesia_yang_Disempurnakan

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('id', {
        months : "Januari_Februari_Maret_April_Mei_Juni_Juli_Agustus_September_Oktober_November_Desember".split("_"),
        monthsShort : "Jan_Feb_Mar_Apr_Mei_Jun_Jul_Ags_Sep_Okt_Nov_Des".split("_"),
        weekdays : "Minggu_Senin_Selasa_Rabu_Kamis_Jumat_Sabtu".split("_"),
        weekdaysShort : "Min_Sen_Sel_Rab_Kam_Jum_Sab".split("_"),
        weekdaysMin : "Mg_Sn_Sl_Rb_Km_Jm_Sb".split("_"),
        longDateFormat : {
            LT : "HH.mm",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY [pukul] LT",
            LLLL : "dddd, D MMMM YYYY [pukul] LT"
        },
        meridiem : function (hours, minutes, isLower) {
            if (hours < 11) {
                return 'pagi';
            } else if (hours < 15) {
                return 'siang';
            } else if (hours < 19) {
                return 'sore';
            } else {
                return 'malam';
            }
        },
        calendar : {
            sameDay : '[Hari ini pukul] LT',
            nextDay : '[Besok pukul] LT',
            nextWeek : 'dddd [pukul] LT',
            lastDay : '[Kemarin pukul] LT',
            lastWeek : 'dddd [lalu pukul] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "dalam %s",
            past : "%s yang lalu",
            s : "beberapa detik",
            m : "semenit",
            mm : "%d menit",
            h : "sejam",
            hh : "%d jam",
            d : "sehari",
            dd : "%d hari",
            M : "sebulan",
            MM : "%d bulan",
            y : "setahun",
            yy : "%d tahun"
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : icelandic (is)
// author : Hinrik Уrn SigurУАsson : https://github.com/hinrik

(function (factory) {
    factory(moment);
}(function (moment) {
    function plural(n) {
        if (n % 100 === 11) {
            return true;
        } else if (n % 10 === 1) {
            return false;
        }
        return true;
    }

    function translate(number, withoutSuffix, key, isFuture) {
        var result = number + " ";
        switch (key) {
        case 's':
            return withoutSuffix || isFuture ? 'nokkrar sekУКndur' : 'nokkrum sekУКndum';
        case 'm':
            return withoutSuffix ? 'mУ­nУКta' : 'mУ­nУКtu';
        case 'mm':
            if (plural(number)) {
                return result + (withoutSuffix || isFuture ? 'mУ­nУКtur' : 'mУ­nУКtum');
            } else if (withoutSuffix) {
                return result + 'mУ­nУКta';
            }
            return result + 'mУ­nУКtu';
        case 'hh':
            if (plural(number)) {
                return result + (withoutSuffix || isFuture ? 'klukkustundir' : 'klukkustundum');
            }
            return result + 'klukkustund';
        case 'd':
            if (withoutSuffix) {
                return 'dagur';
            }
            return isFuture ? 'dag' : 'degi';
        case 'dd':
            if (plural(number)) {
                if (withoutSuffix) {
                    return result + 'dagar';
                }
                return result + (isFuture ? 'daga' : 'dУЖgum');
            } else if (withoutSuffix) {
                return result + 'dagur';
            }
            return result + (isFuture ? 'dag' : 'degi');
        case 'M':
            if (withoutSuffix) {
                return 'mУЁnuУАur';
            }
            return isFuture ? 'mУЁnuУА' : 'mУЁnuУАi';
        case 'MM':
            if (plural(number)) {
                if (withoutSuffix) {
                    return result + 'mУЁnuУАir';
                }
                return result + (isFuture ? 'mУЁnuУАi' : 'mУЁnuУАum');
            } else if (withoutSuffix) {
                return result + 'mУЁnuУАur';
            }
            return result + (isFuture ? 'mУЁnuУА' : 'mУЁnuУАi');
        case 'y':
            return withoutSuffix || isFuture ? 'УЁr' : 'УЁri';
        case 'yy':
            if (plural(number)) {
                return result + (withoutSuffix || isFuture ? 'УЁr' : 'УЁrum');
            }
            return result + (withoutSuffix || isFuture ? 'УЁr' : 'УЁri');
        }
    }

    return moment.lang('is', {
        months : "janУКar_febrУКar_mars_aprУ­l_maУ­_jУКnУ­_jУКlУ­_УЁgУКst_september_oktУГber_nУГvember_desember".split("_"),
        monthsShort : "jan_feb_mar_apr_maУ­_jУКn_jУКl_УЁgУК_sep_okt_nУГv_des".split("_"),
        weekdays : "sunnudagur_mУЁnudagur_УОriУАjudagur_miУАvikudagur_fimmtudagur_fУЖstudagur_laugardagur".split("_"),
        weekdaysShort : "sun_mУЁn_УОri_miУА_fim_fУЖs_lau".split("_"),
        weekdaysMin : "Su_MУЁ_Уr_Mi_Fi_FУЖ_La".split("_"),
        longDateFormat : {
            LT : "H:mm",
            L : "DD/MM/YYYY",
            LL : "D. MMMM YYYY",
            LLL : "D. MMMM YYYY [kl.] LT",
            LLLL : "dddd, D. MMMM YYYY [kl.] LT"
        },
        calendar : {
            sameDay : '[У­ dag kl.] LT',
            nextDay : '[УЁ morgun kl.] LT',
            nextWeek : 'dddd [kl.] LT',
            lastDay : '[У­ gУІr kl.] LT',
            lastWeek : '[sУ­УАasta] dddd [kl.] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "eftir %s",
            past : "fyrir %s sУ­УАan",
            s : translate,
            m : translate,
            mm : translate,
            h : "klukkustund",
            hh : translate,
            d : translate,
            dd : translate,
            M : translate,
            MM : translate,
            y : translate,
            yy : translate
        },
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : italian (it)
// author : Lorenzo : https://github.com/aliem
// author: Mattia Larentis: https://github.com/nostalgiaz

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('it', {
        months : "Gennaio_Febbraio_Marzo_Aprile_Maggio_Giugno_Luglio_Agosto_Settembre_Ottobre_Novembre_Dicembre".split("_"),
        monthsShort : "Gen_Feb_Mar_Apr_Mag_Giu_Lug_Ago_Set_Ott_Nov_Dic".split("_"),
        weekdays : "Domenica_LunedУЌ_MartedУЌ_MercoledУЌ_GiovedУЌ_VenerdУЌ_Sabato".split("_"),
        weekdaysShort : "Dom_Lun_Mar_Mer_Gio_Ven_Sab".split("_"),
        weekdaysMin : "D_L_Ma_Me_G_V_S".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd, D MMMM YYYY LT"
        },
        calendar : {
            sameDay: '[Oggi alle] LT',
            nextDay: '[Domani alle] LT',
            nextWeek: 'dddd [alle] LT',
            lastDay: '[Ieri alle] LT',
            lastWeek: '[lo scorso] dddd [alle] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : function (s) {
                return ((/^[0-9].+$/).test(s) ? "tra" : "in") + " " + s;
            },
            past : "%s fa",
            s : "secondi",
            m : "un minuto",
            mm : "%d minuti",
            h : "un'ora",
            hh : "%d ore",
            d : "un giorno",
            dd : "%d giorni",
            M : "un mese",
            MM : "%d mesi",
            y : "un anno",
            yy : "%d anni"
        },
        ordinal: '%dТК',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : japanese (ja)
// author : LI Long : https://github.com/baryon

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('ja', {
        months : "1ц_2ц_3ц_4ц_5ц_6ц_7ц_8ц_9ц_10ц_11ц_12ц".split("_"),
        monthsShort : "1ц_2ц_3ц_4ц_5ц_6ц_7ц_8ц_9ц_10ц_11ц_12ц".split("_"),
        weekdays : "цЅццЅ_цццЅ_чЋццЅ_цАДццЅ_цЈццЅ_щццЅ_хццЅ".split("_"),
        weekdaysShort : "цЅ_ц_чЋ_цАД_цЈ_щ_х".split("_"),
        weekdaysMin : "цЅ_ц_чЋ_цАД_цЈ_щ_х".split("_"),
        longDateFormat : {
            LT : "Ahцmх",
            L : "YYYY/MM/DD",
            LL : "YYYYхЙДMцDцЅ",
            LLL : "YYYYхЙДMцDцЅLT",
            LLLL : "YYYYхЙДMцDцЅLT dddd"
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 12) {
                return "хх";
            } else {
                return "ххО";
            }
        },
        calendar : {
            sameDay : '[фЛцЅ] LT',
            nextDay : '[ццЅ] LT',
            nextWeek : '[цЅщБ]dddd LT',
            lastDay : '[цЈцЅ] LT',
            lastWeek : '[хщБ]dddd LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "%sхО",
            past : "%sх",
            s : "цАчЇ",
            m : "1х",
            mm : "%dх",
            h : "1цщ",
            hh : "%dцщ",
            d : "1цЅ",
            dd : "%dцЅ",
            M : "1уЖц",
            MM : "%dуЖц",
            y : "1хЙД",
            yy : "%dхЙД"
        }
    });
}));
// moment.js language configuration
// language : Georgian (ka)
// author : Irakli Janiashvili : https://github.com/irakli-janiashvili

(function (factory) {
    factory(moment);
}(function (moment) {

    function monthsCaseReplace(m, format) {
        var months = {
            'nominative': 'сссссс с_ссссс сссс_ссс сЂс_ссс ссс_ссссЁс_сссссЁс_сссссЁс_сссссЁсЂс_сЁссЅсЂссссс с_ссЅсЂссссс с_ссссссс с_сссссссс с'.split('_'),
            'accusative': 'сссссс сЁ_ссссс ссссЁ_ссс сЂсЁ_ссс ссссЁ_ссссЁсЁ_сссссЁсЁ_сссссЁсЁ_сссссЁсЂсЁ_сЁссЅсЂссссс сЁ_ссЅсЂссссс сЁ_ссссссс сЁ_сссссссс сЁ'.split('_')
        },

        nounCase = (/D[oD] *MMMM?/).test(format) ?
            'accusative' :
            'nominative';

        return months[nounCase][m.month()];
    }

    function weekdaysCaseReplace(m, format) {
        var weekdays = {
            'nominative': 'сссс с_сс сЈссссс_сЁсссЈссссс_сссЎсЈссссс_сЎсЃссЈссссс_ссс ссЁсссс_сЈссссс'.split('_'),
            'accusative': 'сссс ссЁ_сс сЈсссссЁ_сЁсссЈсссссЁ_сссЎсЈсссссЁ_сЎсЃссЈсссссЁ_ссс ссЁссссЁ_сЈсссссЁ'.split('_')
        },

        nounCase = (/(сЌссс|сЈссссс)/).test(format) ?
            'accusative' :
            'nominative';

        return weekdays[nounCase][m.day()];
    }

    return moment.lang('ka', {
        months : monthsCaseReplace,
        monthsShort : "ссс_ссс_ссс _ссс _ссс_ссс_ссс_ссс_сЁссЅ_ссЅсЂ_ссс_ссс".split("_"),
        weekdays : weekdaysCaseReplace,
        weekdaysShort : "ссс_сс сЈ_сЁсс_сссЎ_сЎсЃс_ссс _сЈсс".split("_"),
        weekdaysMin : "сс_сс _сЁс_сс_сЎсЃ_сс_сЈс".split("_"),
        longDateFormat : {
            LT : "h:mm A",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd, D MMMM YYYY LT"
        },
        calendar : {
            sameDay : '[ссІссЁ] LT[-сс]',
            nextDay : '[сЎссс] LT[-сс]',
            lastDay : '[ссЃсЈсс] LT[-сс]',
            nextWeek : '[сЈссссс] dddd LT[-сс]',
            lastWeek : '[сЌссс] dddd LT-сс',
            sameElse : 'L'
        },
        relativeTime : {
            future : function (s) {
                return (/(сЌссс|сЌсЃсс|сЁсссс|сЌссс)/).test(s) ?
                    s.replace(/с$/, "сЈс") :
                    s + "сЈс";
            },
            past : function (s) {
                if ((/(сЌссс|сЌсЃсс|сЁсссс|ссІс|ссс)/).test(s)) {
                    return s.replace(/(с|с)$/, "ссЁ сЌсс");
                }
                if ((/сЌссс/).test(s)) {
                    return s.replace(/сЌссс$/, "сЌсссЁ сЌсс");
                }
            },
            s : "с сссссссс сЌссс",
            m : "сЌсЃсс",
            mm : "%d сЌсЃсс",
            h : "сЁсссс",
            hh : "%d сЁсссс",
            d : "ссІс",
            dd : "%d ссІс",
            M : "ссс",
            MM : "%d ссс",
            y : "сЌссс",
            yy : "%d сЌссс"
        },
        ordinal : function (number) {
            if (number === 0) {
                return number;
            }

            if (number === 1) {
                return number + "-сс";
            }

            if ((number < 20) || (number <= 100 && (number % 20 === 0)) || (number % 100 === 0)) {
                return "сс-" + number;
            }

            return number + "-с";
        },
        week : {
            dow : 1,
            doy : 7
        }
    });
}));
// moment.js language configuration
// language : korean (ko)
// author : Kyungwook, Park : https://github.com/kyungw00k

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('ko', {
        months : "1ь_2ь_3ь_4ь_5ь_6ь_7ь_8ь_9ь_10ь_11ь_12ь".split("_"),
        monthsShort : "1ь_2ь_3ь_4ь_5ь_6ь_7ь_8ь_9ь_10ь_11ь_12ь".split("_"),
        weekdays : "ьМььМ_ьььМ_эььМ_ьььМ_ыЊЉььМ_ъИььМ_э ььМ".split("_"),
        weekdaysShort : "ьМ_ь_э_ь_ыЊЉ_ъИ_э ".split("_"),
        weekdaysMin : "ьМ_ь_э_ь_ыЊЉ_ъИ_э ".split("_"),
        longDateFormat : {
            LT : "A hь mmыЖ",
            L : "YYYY.MM.DD",
            LL : "YYYYы MMMM DьМ",
            LLL : "YYYYы MMMM DьМ LT",
            LLLL : "YYYYы MMMM DьМ dddd LT"
        },
        meridiem : function (hour, minute, isUpper) {
            return hour < 12 ? 'ьЄь ' : 'ьЄэ';
        },
        calendar : {
            sameDay : 'ьЄы LT',
            nextDay : 'ыДьМ LT',
            nextWeek : 'dddd LT',
            lastDay : 'ьДь  LT',
            lastWeek : 'ьЇыьЃМ dddd LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "%s э",
            past : "%s ь ",
            s : "ыЊьД",
            ss : "%dьД",
            m : "ьМыЖ",
            mm : "%dыЖ",
            h : "эьъА",
            hh : "%dьъА",
            d : "эыЃЈ",
            dd : "%dьМ",
            M : "эыЌ",
            MM : "%dыЌ",
            y : "ьМы",
            yy : "%dы"
        },
        ordinal : '%dьМ'
    });
}));
// moment.js language configuration
// language : Lithuanian (lt)
// author : Mindaugas MozХЋras : https://github.com/mmozuras

(function (factory) {
    factory(moment);
}(function (moment) {
    var units = {
        "m" : "minutФ_minutФs_minutФ",
        "mm": "minutФs_minuФiХГ_minutes",
        "h" : "valanda_valandos_valandФ",
        "hh": "valandos_valandХГ_valandas",
        "d" : "diena_dienos_dienФ",
        "dd": "dienos_dienХГ_dienas",
        "M" : "mФnuo_mФnesio_mФnesФЏ",
        "MM": "mФnesiai_mФnesiХГ_mФnesius",
        "y" : "metai_metХГ_metus",
        "yy": "metai_metХГ_metus"
    },
    weekDays = "pirmadienis_antradienis_treФiadienis_ketvirtadienis_penktadienis_ХЁeХЁtadienis_sekmadienis".split("_");

    function translateSeconds(number, withoutSuffix, key, isFuture) {
        if (withoutSuffix) {
            return "kelios sekundФs";
        } else {
            return isFuture ? "keliХГ sekundХОiХГ" : "kelias sekundes";
        }
    }

    function translateSingular(number, withoutSuffix, key, isFuture) {
        return withoutSuffix ? forms(key)[0] : (isFuture ? forms(key)[1] : forms(key)[2]);
    }

    function special(number) {
        return number % 10 === 0 || (number > 10 && number < 20);
    }

    function forms(key) {
        return units[key].split("_");
    }

    function translate(number, withoutSuffix, key, isFuture) {
        var result = number + " ";
        if (number === 1) {
            return result + translateSingular(number, withoutSuffix, key[0], isFuture);
        } else if (withoutSuffix) {
            return result + (special(number) ? forms(key)[1] : forms(key)[0]);
        } else {
            if (isFuture) {
                return result + forms(key)[1];
            } else {
                return result + (special(number) ? forms(key)[1] : forms(key)[2]);
            }
        }
    }

    function relativeWeekDay(moment, format) {
        var nominative = format.indexOf('dddd LT') === -1,
            weekDay = weekDays[moment.weekday()];

        return nominative ? weekDay : weekDay.substring(0, weekDay.length - 2) + "ФЏ";
    }

    return moment.lang("lt", {
        months : "sausio_vasario_kovo_balandХОio_geguХОФs_birХОФlio_liepos_rugpjХЋФio_rugsФjo_spalio_lapkriФio_gruodХОio".split("_"),
        monthsShort : "sau_vas_kov_bal_geg_bir_lie_rgp_rgs_spa_lap_grd".split("_"),
        weekdays : relativeWeekDay,
        weekdaysShort : "Sek_Pir_Ant_Tre_Ket_Pen_Х eХЁ".split("_"),
        weekdaysMin : "S_P_A_T_K_Pn_Х ".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "YYYY-MM-DD",
            LL : "YYYY [m.] MMMM D [d.]",
            LLL : "YYYY [m.] MMMM D [d.], LT [val.]",
            LLLL : "YYYY [m.] MMMM D [d.], dddd, LT [val.]",
            l : "YYYY-MM-DD",
            ll : "YYYY [m.] MMMM D [d.]",
            lll : "YYYY [m.] MMMM D [d.], LT [val.]",
            llll : "YYYY [m.] MMMM D [d.], ddd, LT [val.]"
        },
        calendar : {
            sameDay : "[Х iandien] LT",
            nextDay : "[Rytoj] LT",
            nextWeek : "dddd LT",
            lastDay : "[Vakar] LT",
            lastWeek : "[PraФjusФЏ] dddd LT",
            sameElse : "L"
        },
        relativeTime : {
            future : "po %s",
            past : "prieХЁ %s",
            s : translateSeconds,
            m : translateSingular,
            mm : translate,
            h : translateSingular,
            hh : translate,
            d : translateSingular,
            dd : translate,
            M : translateSingular,
            MM : translate,
            y : translateSingular,
            yy : translate
        },
        ordinal : function (number) {
            return number + '-oji';
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : latvian (lv)
// author : Kristaps Karlsons : https://github.com/skakri

(function (factory) {
    factory(moment);
}(function (moment) {
    var units = {
        'mm': 'minХЋti_minХЋtes_minХЋte_minХЋtes',
        'hh': 'stundu_stundas_stunda_stundas',
        'dd': 'dienu_dienas_diena_dienas',
        'MM': 'mФnesi_mФneХЁus_mФnesis_mФneХЁi',
        'yy': 'gadu_gadus_gads_gadi'
    };

    function format(word, number, withoutSuffix) {
        var forms = word.split('_');
        if (withoutSuffix) {
            return number % 10 === 1 && number !== 11 ? forms[2] : forms[3];
        } else {
            return number % 10 === 1 && number !== 11 ? forms[0] : forms[1];
        }
    }

    function relativeTimeWithPlural(number, withoutSuffix, key) {
        return number + ' ' + format(units[key], number, withoutSuffix);
    }

    return moment.lang('lv', {
        months : "janvФris_februФris_marts_aprФЋlis_maijs_jХЋnijs_jХЋlijs_augusts_septembris_oktobris_novembris_decembris".split("_"),
        monthsShort : "jan_feb_mar_apr_mai_jХЋn_jХЋl_aug_sep_okt_nov_dec".split("_"),
        weekdays : "svФtdiena_pirmdiena_otrdiena_treХЁdiena_ceturtdiena_piektdiena_sestdiena".split("_"),
        weekdaysShort : "Sv_P_O_T_C_Pk_S".split("_"),
        weekdaysMin : "Sv_P_O_T_C_Pk_S".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD.MM.YYYY",
            LL : "YYYY. [gada] D. MMMM",
            LLL : "YYYY. [gada] D. MMMM, LT",
            LLLL : "YYYY. [gada] D. MMMM, dddd, LT"
        },
        calendar : {
            sameDay : '[Х odien pulksten] LT',
            nextDay : '[RФЋt pulksten] LT',
            nextWeek : 'dddd [pulksten] LT',
            lastDay : '[Vakar pulksten] LT',
            lastWeek : '[PagФjuХЁФ] dddd [pulksten] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "%s vФlФk",
            past : "%s agrФk",
            s : "daХОas sekundes",
            m : "minХЋti",
            mm : relativeTimeWithPlural,
            h : "stundu",
            hh : relativeTimeWithPlural,
            d : "dienu",
            dd : relativeTimeWithPlural,
            M : "mФnesi",
            MM : relativeTimeWithPlural,
            y : "gadu",
            yy : relativeTimeWithPlural
        },
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : malayalam (ml)
// author : Floyd Pink : https://github.com/floydpink

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('ml', {
        months : 'рДрДЈрЕрДЕрДАрДП_рДЋрЕрДЌрЕрДАрЕрДЕрДАрДП_рДЎрДОрЕМрДрЕрДрЕ_рДрДЊрЕрДАрДПрЕН_рДЎрЕрДЏрЕ_рДрЕрЕК_рДрЕрДВрЕ_рДрДрДИрЕрДБрЕрДБрЕ_рДИрЕрДЊрЕрДБрЕрДБрДрДЌрЕМ_рДрДрЕрДрЕрДЌрЕМ_рДЈрДЕрДрДЌрЕМ_рДЁрДПрДИрДрДЌрЕМ'.split("_"),
        monthsShort : 'рДрДЈрЕ._рДЋрЕрДЌрЕрДАрЕ._рДЎрДОрЕМ._рДрДЊрЕрДАрДП._рДЎрЕрДЏрЕ_рДрЕрЕК_рДрЕрДВрЕ._рДрД._рДИрЕрДЊрЕрДБрЕрДБ._рДрДрЕрДрЕ._рДЈрДЕрД._рДЁрДПрДИрД.'.split("_"),
        weekdays : 'рДрДОрДЏрДБрДОрДДрЕрД_рДЄрДПрДрЕрДрДГрДОрДДрЕрД_рДрЕрДЕрЕрДЕрДОрДДрЕрД_рДЌрЕрДЇрДЈрДОрДДрЕрД_рДЕрЕрДЏрДОрДДрДОрДДрЕрД_рДЕрЕрДГрЕрДГрДПрДЏрДОрДДрЕрД_рДЖрДЈрДПрДЏрДОрДДрЕрД'.split("_"),
        weekdaysShort : 'рДрДОрДЏрЕМ_рДЄрДПрДрЕрДрЕО_рДрЕрДЕрЕрДЕ_рДЌрЕрДЇрЕЛ_рДЕрЕрДЏрДОрДДрД_рДЕрЕрДГрЕрДГрДП_рДЖрДЈрДП'.split("_"),
        weekdaysMin : 'рДрДО_рДЄрДП_рДрЕ_рДЌрЕ_рДЕрЕрДЏрДО_рДЕрЕ_рДЖ'.split("_"),
        longDateFormat : {
            LT : "A h:mm -рДЈрЕ",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY, LT",
            LLLL : "dddd, D MMMM YYYY, LT"
        },
        calendar : {
            sameDay : '[рДрДЈрЕрДЈрЕ] LT',
            nextDay : '[рДЈрДОрДГрЕ] LT',
            nextWeek : 'dddd, LT',
            lastDay : '[рДрДЈрЕрДЈрДВрЕ] LT',
            lastWeek : '[рДрДДрДПрДрЕрД] dddd, LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "%s рДрДДрДПрДрЕрДрЕ",
            past : "%s рДЎрЕрЕЛрДЊрЕ",
            s : "рДрЕНрДЊ рДЈрДПрДЎрДПрДЗрДрЕрДрЕО",
            m : "рДрДАрЕ рДЎрДПрДЈрДПрДБрЕрДБрЕ",
            mm : "%d рДЎрДПрДЈрДПрДБрЕрДБрЕ",
            h : "рДрДАрЕ рДЎрДЃрДПрДрЕрДрЕрЕМ",
            hh : "%d рДЎрДЃрДПрДрЕрДрЕрЕМ",
            d : "рДрДАрЕ рДІрДПрДЕрДИрД",
            dd : "%d рДІрДПрДЕрДИрД",
            M : "рДрДАрЕ рДЎрДОрДИрД",
            MM : "%d рДЎрДОрДИрД",
            y : "рДрДАрЕ рДЕрЕМрДЗрД",
            yy : "%d рДЕрЕМрДЗрД"
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 4) {
                return "рДАрДОрДЄрЕрДАрДП";
            } else if (hour < 12) {
                return "рДАрДОрДЕрДПрДВрЕ";
            } else if (hour < 17) {
                return "рДрДрЕрД рДрДДрДПрДрЕрДрЕ";
            } else if (hour < 20) {
                return "рДЕрЕрДрЕрДЈрЕрДЈрЕрДАрД";
            } else {
                return "рДАрДОрДЄрЕрДАрДП";
            }
        }
    });
}));
// moment.js language configuration
// language : Marathi (mr)
// author : Harshad Kale : https://github.com/kalehv

(function (factory) {
    factory(moment);
}(function (moment) {
    var symbolMap = {
        '1': 'рЅЇ',
        '2': 'рЅЈ',
        '3': 'рЅЉ',
        '4': 'рЅЊ',
        '5': 'рЅЋ',
        '6': 'рЅЌ',
        '7': 'рЅ­',
        '8': 'рЅЎ',
        '9': 'рЅЏ',
        '0': 'рЅІ'
    },
    numberMap = {
        'рЅЇ': '1',
        'рЅЈ': '2',
        'рЅЉ': '3',
        'рЅЊ': '4',
        'рЅЋ': '5',
        'рЅЌ': '6',
        'рЅ­': '7',
        'рЅЎ': '8',
        'рЅЏ': '9',
        'рЅІ': '0'
    };

    return moment.lang('mr', {
        months : 'рЄрЄОрЄЈрЅрЄЕрЄОрЄАрЅ_рЄЋрЅрЄЌрЅрЄАрЅрЄЕрЄОрЄАрЅ_рЄЎрЄОрЄАрЅрЄ_рЄрЄЊрЅрЄАрЄПрЄВ_рЄЎрЅ_рЄрЅрЄЈ_рЄрЅрЄВрЅ_рЄрЄрЄИрЅрЄ_рЄИрЄЊрЅрЄрЅрЄрЄЌрЄА_рЄрЄрЅрЄрЅрЄЌрЄА_рЄЈрЅрЄЕрЅрЄЙрЅрЄрЄЌрЄА_рЄЁрЄПрЄИрЅрЄрЄЌрЄА'.split("_"),
        monthsShort: 'рЄрЄОрЄЈрЅ._рЄЋрЅрЄЌрЅрЄАрЅ._рЄЎрЄОрЄАрЅрЄ._рЄрЄЊрЅрЄАрЄП._рЄЎрЅ._рЄрЅрЄЈ._рЄрЅрЄВрЅ._рЄрЄ._рЄИрЄЊрЅрЄрЅрЄ._рЄрЄрЅрЄрЅ._рЄЈрЅрЄЕрЅрЄЙрЅрЄ._рЄЁрЄПрЄИрЅрЄ.'.split("_"),
        weekdays : 'рЄАрЄЕрЄПрЄЕрЄОрЄА_рЄИрЅрЄЎрЄЕрЄОрЄА_рЄЎрЄрЄрЄГрЄЕрЄОрЄА_рЄЌрЅрЄЇрЄЕрЄОрЄА_рЄрЅрЄАрЅрЄЕрЄОрЄА_рЄЖрЅрЄрЅрЄАрЄЕрЄОрЄА_рЄЖрЄЈрЄПрЄЕрЄОрЄА'.split("_"),
        weekdaysShort : 'рЄАрЄЕрЄП_рЄИрЅрЄЎ_рЄЎрЄрЄрЄГ_рЄЌрЅрЄЇ_рЄрЅрЄАрЅ_рЄЖрЅрЄрЅрЄА_рЄЖрЄЈрЄП'.split("_"),
        weekdaysMin : 'рЄА_рЄИрЅ_рЄЎрЄ_рЄЌрЅ_рЄрЅ_рЄЖрЅ_рЄЖ'.split("_"),
        longDateFormat : {
            LT : "A h:mm рЄЕрЄОрЄрЄЄрЄО",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY, LT",
            LLLL : "dddd, D MMMM YYYY, LT"
        },
        calendar : {
            sameDay : '[рЄрЄ] LT',
            nextDay : '[рЄрЄІрЅрЄЏрЄО] LT',
            nextWeek : 'dddd, LT',
            lastDay : '[рЄрЄОрЄВ] LT',
            lastWeek: '[рЄЎрЄОрЄрЅрЄВ] dddd, LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "%s рЄЈрЄрЄЄрЄА",
            past : "%s рЄЊрЅрЄАрЅрЄЕрЅ",
            s : "рЄИрЅрЄрЄрЄІ",
            m: "рЄрЄ рЄЎрЄПрЄЈрЄПрЄ",
            mm: "%d рЄЎрЄПрЄЈрЄПрЄрЅ",
            h : "рЄрЄ рЄЄрЄОрЄИ",
            hh : "%d рЄЄрЄОрЄИ",
            d : "рЄрЄ рЄІрЄПрЄЕрЄИ",
            dd : "%d рЄІрЄПрЄЕрЄИ",
            M : "рЄрЄ рЄЎрЄЙрЄПрЄЈрЄО",
            MM : "%d рЄЎрЄЙрЄПрЄЈрЅ",
            y : "рЄрЄ рЄЕрЄАрЅрЄЗ",
            yy : "%d рЄЕрЄАрЅрЄЗрЅ"
        },
        preparse: function (string) {
            return string.replace(/[рЅЇрЅЈрЅЉрЅЊрЅЋрЅЌрЅ­рЅЎрЅЏрЅІ]/g, function (match) {
                return numberMap[match];
            });
        },
        postformat: function (string) {
            return string.replace(/\d/g, function (match) {
                return symbolMap[match];
            });
        },
        meridiem: function (hour, minute, isLower)
        {
            if (hour < 4) {
                return "рЄАрЄОрЄЄрЅрЄАрЅ";
            } else if (hour < 10) {
                return "рЄИрЄрЄОрЄГрЅ";
            } else if (hour < 17) {
                return "рЄІрЅрЄЊрЄОрЄАрЅ";
            } else if (hour < 20) {
                return "рЄИрЄОрЄЏрЄрЄрЄОрЄГрЅ";
            } else {
                return "рЄАрЄОрЄЄрЅрЄАрЅ";
            }
        },
        week : {
            dow : 0, // Sunday is the first day of the week.
            doy : 6  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : Bahasa Malaysia (ms-MY)
// author : Weldan Jamili : https://github.com/weldan

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('ms-my', {
        months : "Januari_Februari_Mac_April_Mei_Jun_Julai_Ogos_September_Oktober_November_Disember".split("_"),
        monthsShort : "Jan_Feb_Mac_Apr_Mei_Jun_Jul_Ogs_Sep_Okt_Nov_Dis".split("_"),
        weekdays : "Ahad_Isnin_Selasa_Rabu_Khamis_Jumaat_Sabtu".split("_"),
        weekdaysShort : "Ahd_Isn_Sel_Rab_Kha_Jum_Sab".split("_"),
        weekdaysMin : "Ah_Is_Sl_Rb_Km_Jm_Sb".split("_"),
        longDateFormat : {
            LT : "HH.mm",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY [pukul] LT",
            LLLL : "dddd, D MMMM YYYY [pukul] LT"
        },
        meridiem : function (hours, minutes, isLower) {
            if (hours < 11) {
                return 'pagi';
            } else if (hours < 15) {
                return 'tengahari';
            } else if (hours < 19) {
                return 'petang';
            } else {
                return 'malam';
            }
        },
        calendar : {
            sameDay : '[Hari ini pukul] LT',
            nextDay : '[Esok pukul] LT',
            nextWeek : 'dddd [pukul] LT',
            lastDay : '[Kelmarin pukul] LT',
            lastWeek : 'dddd [lepas pukul] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "dalam %s",
            past : "%s yang lepas",
            s : "beberapa saat",
            m : "seminit",
            mm : "%d minit",
            h : "sejam",
            hh : "%d jam",
            d : "sehari",
            dd : "%d hari",
            M : "sebulan",
            MM : "%d bulan",
            y : "setahun",
            yy : "%d tahun"
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : norwegian bokmУЅl (nb)
// authors : Espen Hovlandsdal : https://github.com/rexxars
//           Sigurd Gartmann : https://github.com/sigurdga

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('nb', {
        months : "januar_februar_mars_april_mai_juni_juli_august_september_oktober_november_desember".split("_"),
        monthsShort : "jan._feb._mars_april_mai_juni_juli_aug._sep._okt._nov._des.".split("_"),
        weekdays : "sУИndag_mandag_tirsdag_onsdag_torsdag_fredag_lУИrdag".split("_"),
        weekdaysShort : "sУИ._ma._ti._on._to._fr._lУИ.".split("_"),
        weekdaysMin : "sУИ_ma_ti_on_to_fr_lУИ".split("_"),
        longDateFormat : {
            LT : "H.mm",
            L : "DD.MM.YYYY",
            LL : "D. MMMM YYYY",
            LLL : "D. MMMM YYYY [kl.] LT",
            LLLL : "dddd D. MMMM YYYY [kl.] LT"
        },
        calendar : {
            sameDay: '[i dag kl.] LT',
            nextDay: '[i morgen kl.] LT',
            nextWeek: 'dddd [kl.] LT',
            lastDay: '[i gУЅr kl.] LT',
            lastWeek: '[forrige] dddd [kl.] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : "om %s",
            past : "for %s siden",
            s : "noen sekunder",
            m : "ett minutt",
            mm : "%d minutter",
            h : "en time",
            hh : "%d timer",
            d : "en dag",
            dd : "%d dager",
            M : "en mУЅned",
            MM : "%d mУЅneder",
            y : "ett УЅr",
            yy : "%d УЅr"
        },
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : nepali/nepalese
// author : suvash : https://github.com/suvash

(function (factory) {
    factory(moment);
}(function (moment) {
    var symbolMap = {
        '1': 'рЅЇ',
        '2': 'рЅЈ',
        '3': 'рЅЉ',
        '4': 'рЅЊ',
        '5': 'рЅЋ',
        '6': 'рЅЌ',
        '7': 'рЅ­',
        '8': 'рЅЎ',
        '9': 'рЅЏ',
        '0': 'рЅІ'
    },
    numberMap = {
        'рЅЇ': '1',
        'рЅЈ': '2',
        'рЅЉ': '3',
        'рЅЊ': '4',
        'рЅЋ': '5',
        'рЅЌ': '6',
        'рЅ­': '7',
        'рЅЎ': '8',
        'рЅЏ': '9',
        'рЅІ': '0'
    };

    return moment.lang('ne', {
        months : 'рЄрЄЈрЄЕрЄАрЅ_рЄЋрЅрЄЌрЅрЄАрЅрЄЕрЄАрЅ_рЄЎрЄОрЄАрЅрЄ_рЄрЄЊрЅрЄАрЄПрЄВ_рЄЎрЄ_рЄрЅрЄЈ_рЄрЅрЄВрЄОрЄ_рЄрЄрЄЗрЅрЄ_рЄИрЅрЄЊрЅрЄрЅрЄЎрЅрЄЌрЄА_рЄрЄрЅрЄрЅрЄЌрЄА_рЄЈрЅрЄ­рЅрЄЎрЅрЄЌрЄА_рЄЁрЄПрЄИрЅрЄЎрЅрЄЌрЄА'.split("_"),
        monthsShort : 'рЄрЄЈ._рЄЋрЅрЄЌрЅрЄАрЅ._рЄЎрЄОрЄАрЅрЄ_рЄрЄЊрЅрЄАрЄП._рЄЎрЄ_рЄрЅрЄЈ_рЄрЅрЄВрЄОрЄ._рЄрЄ._рЄИрЅрЄЊрЅрЄ._рЄрЄрЅрЄрЅ._рЄЈрЅрЄ­рЅ._рЄЁрЄПрЄИрЅ.'.split("_"),
        weekdays : 'рЄрЄрЄЄрЄЌрЄОрЄА_рЄИрЅрЄЎрЄЌрЄОрЄА_рЄЎрЄрЅрЄрЄВрЄЌрЄОрЄА_рЄЌрЅрЄЇрЄЌрЄОрЄА_рЄЌрЄПрЄЙрЄПрЄЌрЄОрЄА_рЄЖрЅрЄрЅрЄАрЄЌрЄОрЄА_рЄЖрЄЈрЄПрЄЌрЄОрЄА'.split("_"),
        weekdaysShort : 'рЄрЄрЄЄ._рЄИрЅрЄЎ._рЄЎрЄрЅрЄрЄВ._рЄЌрЅрЄЇ._рЄЌрЄПрЄЙрЄП._рЄЖрЅрЄрЅрЄА._рЄЖрЄЈрЄП.'.split("_"),
        weekdaysMin : 'рЄрЄ._рЄИрЅ._рЄЎрЄрЅ_рЄЌрЅ._рЄЌрЄП._рЄЖрЅ._рЄЖ.'.split("_"),
        longDateFormat : {
            LT : "AрЄрЅ h:mm рЄЌрЄрЅ",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY, LT",
            LLLL : "dddd, D MMMM YYYY, LT"
        },
        preparse: function (string) {
            return string.replace(/[рЅЇрЅЈрЅЉрЅЊрЅЋрЅЌрЅ­рЅЎрЅЏрЅІ]/g, function (match) {
                return numberMap[match];
            });
        },
        postformat: function (string) {
            return string.replace(/\d/g, function (match) {
                return symbolMap[match];
            });
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 3) {
                return "рЄАрЄОрЄЄрЅ";
            } else if (hour < 10) {
                return "рЄЌрЄПрЄЙрЄОрЄЈ";
            } else if (hour < 15) {
                return "рЄІрЄПрЄрЄрЄИрЅ";
            } else if (hour < 18) {
                return "рЄЌрЅрЄВрЅрЄрЄО";
            } else if (hour < 20) {
                return "рЄИрЄОрЄрЄ";
            } else {
                return "рЄАрЄОрЄЄрЅ";
            }
        },
        calendar : {
            sameDay : '[рЄрЄ] LT',
            nextDay : '[рЄ­рЅрЄВрЅ] LT',
            nextWeek : '[рЄрЄрЄрЄІрЅ] dddd[,] LT',
            lastDay : '[рЄЙрЄПрЄрЅ] LT',
            lastWeek : '[рЄрЄрЄрЅ] dddd[,] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "%sрЄЎрЄО",
            past : "%s рЄрЄрЄОрЄЁрЅ",
            s : "рЄрЅрЄЙрЅ рЄИрЄЎрЄЏ",
            m : "рЄрЄ рЄЎрЄПрЄЈрЅрЄ",
            mm : "%d рЄЎрЄПрЄЈрЅрЄ",
            h : "рЄрЄ рЄрЄЃрЅрЄрЄО",
            hh : "%d рЄрЄЃрЅрЄрЄО",
            d : "рЄрЄ рЄІрЄПрЄЈ",
            dd : "%d рЄІрЄПрЄЈ",
            M : "рЄрЄ рЄЎрЄЙрЄПрЄЈрЄО",
            MM : "%d рЄЎрЄЙрЄПрЄЈрЄО",
            y : "рЄрЄ рЄЌрЄАрЅрЄЗ",
            yy : "%d рЄЌрЄАрЅрЄЗ"
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : dutch (nl)
// author : Joris RУЖling : https://github.com/jjupiter

(function (factory) {
    factory(moment);
}(function (moment) {
    var monthsShortWithDots = "jan._feb._mrt._apr._mei_jun._jul._aug._sep._okt._nov._dec.".split("_"),
        monthsShortWithoutDots = "jan_feb_mrt_apr_mei_jun_jul_aug_sep_okt_nov_dec".split("_");

    return moment.lang('nl', {
        months : "januari_februari_maart_april_mei_juni_juli_augustus_september_oktober_november_december".split("_"),
        monthsShort : function (m, format) {
            if (/-MMM-/.test(format)) {
                return monthsShortWithoutDots[m.month()];
            } else {
                return monthsShortWithDots[m.month()];
            }
        },
        weekdays : "zondag_maandag_dinsdag_woensdag_donderdag_vrijdag_zaterdag".split("_"),
        weekdaysShort : "zo._ma._di._wo._do._vr._za.".split("_"),
        weekdaysMin : "Zo_Ma_Di_Wo_Do_Vr_Za".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD-MM-YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd D MMMM YYYY LT"
        },
        calendar : {
            sameDay: '[vandaag om] LT',
            nextDay: '[morgen om] LT',
            nextWeek: 'dddd [om] LT',
            lastDay: '[gisteren om] LT',
            lastWeek: '[afgelopen] dddd [om] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : "over %s",
            past : "%s geleden",
            s : "een paar seconden",
            m : "УЉУЉn minuut",
            mm : "%d minuten",
            h : "УЉУЉn uur",
            hh : "%d uur",
            d : "УЉУЉn dag",
            dd : "%d dagen",
            M : "УЉУЉn maand",
            MM : "%d maanden",
            y : "УЉУЉn jaar",
            yy : "%d jaar"
        },
        ordinal : function (number) {
            return number + ((number === 1 || number === 8 || number >= 20) ? 'ste' : 'de');
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : norwegian nynorsk (nn)
// author : https://github.com/mechuwind

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('nn', {
        months : "januar_februar_mars_april_mai_juni_juli_august_september_oktober_november_desember".split("_"),
        monthsShort : "jan_feb_mar_apr_mai_jun_jul_aug_sep_okt_nov_des".split("_"),
        weekdays : "sundag_mУЅndag_tysdag_onsdag_torsdag_fredag_laurdag".split("_"),
        weekdaysShort : "sun_mУЅn_tys_ons_tor_fre_lau".split("_"),
        weekdaysMin : "su_mУЅ_ty_on_to_fr_lУИ".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD.MM.YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd D MMMM YYYY LT"
        },
        calendar : {
            sameDay: '[I dag klokka] LT',
            nextDay: '[I morgon klokka] LT',
            nextWeek: 'dddd [klokka] LT',
            lastDay: '[I gУЅr klokka] LT',
            lastWeek: '[FУИregУЅende] dddd [klokka] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : "om %s",
            past : "for %s siden",
            s : "noen sekund",
            m : "ett minutt",
            mm : "%d minutt",
            h : "en time",
            hh : "%d timar",
            d : "en dag",
            dd : "%d dagar",
            M : "en mУЅnad",
            MM : "%d mУЅnader",
            y : "ett УЅr",
            yy : "%d УЅr"
        },
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : polish (pl)
// author : Rafal Hirsz : https://github.com/evoL

(function (factory) {
    factory(moment);
}(function (moment) {
    var monthsNominative = "styczeХ_luty_marzec_kwiecieХ_maj_czerwiec_lipiec_sierpieХ_wrzesieХ_paХКdziernik_listopad_grudzieХ".split("_"),
        monthsSubjective = "stycznia_lutego_marca_kwietnia_maja_czerwca_lipca_sierpnia_wrzeХnia_paХКdziernika_listopada_grudnia".split("_");

    function plural(n) {
        return (n % 10 < 5) && (n % 10 > 1) && (~~(n / 10) !== 1);
    }

    function translate(number, withoutSuffix, key) {
        var result = number + " ";
        switch (key) {
        case 'm':
            return withoutSuffix ? 'minuta' : 'minutФ';
        case 'mm':
            return result + (plural(number) ? 'minuty' : 'minut');
        case 'h':
            return withoutSuffix  ? 'godzina'  : 'godzinФ';
        case 'hh':
            return result + (plural(number) ? 'godziny' : 'godzin');
        case 'MM':
            return result + (plural(number) ? 'miesiФce' : 'miesiФcy');
        case 'yy':
            return result + (plural(number) ? 'lata' : 'lat');
        }
    }

    return moment.lang('pl', {
        months : function (momentToFormat, format) {
            if (/D MMMM/.test(format)) {
                return monthsSubjective[momentToFormat.month()];
            } else {
                return monthsNominative[momentToFormat.month()];
            }
        },
        monthsShort : "sty_lut_mar_kwi_maj_cze_lip_sie_wrz_paХК_lis_gru".split("_"),
        weekdays : "niedziela_poniedziaХek_wtorek_Хroda_czwartek_piФtek_sobota".split("_"),
        weekdaysShort : "nie_pon_wt_Хr_czw_pt_sb".split("_"),
        weekdaysMin : "N_Pn_Wt_Хr_Cz_Pt_So".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD.MM.YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd, D MMMM YYYY LT"
        },
        calendar : {
            sameDay: '[DziХ o] LT',
            nextDay: '[Jutro o] LT',
            nextWeek: '[W] dddd [o] LT',
            lastDay: '[Wczoraj o] LT',
            lastWeek: function () {
                switch (this.day()) {
                case 0:
                    return '[W zeszХФ niedzielФ o] LT';
                case 3:
                    return '[W zeszХФ ХrodФ o] LT';
                case 6:
                    return '[W zeszХФ sobotФ o] LT';
                default:
                    return '[W zeszХy] dddd [o] LT';
                }
            },
            sameElse: 'L'
        },
        relativeTime : {
            future : "za %s",
            past : "%s temu",
            s : "kilka sekund",
            m : translate,
            mm : translate,
            h : translate,
            hh : translate,
            d : "1 dzieХ",
            dd : '%d dni',
            M : "miesiФc",
            MM : translate,
            y : "rok",
            yy : translate
        },
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : brazilian portuguese (pt-br)
// author : Caio Ribeiro Pereira : https://github.com/caio-ribeiro-pereira

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('pt-br', {
        months : "Janeiro_Fevereiro_MarУЇo_Abril_Maio_Junho_Julho_Agosto_Setembro_Outubro_Novembro_Dezembro".split("_"),
        monthsShort : "Jan_Fev_Mar_Abr_Mai_Jun_Jul_Ago_Set_Out_Nov_Dez".split("_"),
        weekdays : "Domingo_Segunda-feira_TerУЇa-feira_Quarta-feira_Quinta-feira_Sexta-feira_SУЁbado".split("_"),
        weekdaysShort : "Dom_Seg_Ter_Qua_Qui_Sex_SУЁb".split("_"),
        weekdaysMin : "Dom_2ТЊ_3ТЊ_4ТЊ_5ТЊ_6ТЊ_SУЁb".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD/MM/YYYY",
            LL : "D [de] MMMM [de] YYYY",
            LLL : "D [de] MMMM [de] YYYY LT",
            LLLL : "dddd, D [de] MMMM [de] YYYY LT"
        },
        calendar : {
            sameDay: '[Hoje У s] LT',
            nextDay: '[AmanhУЃ У s] LT',
            nextWeek: 'dddd [У s] LT',
            lastDay: '[Ontem У s] LT',
            lastWeek: function () {
                return (this.day() === 0 || this.day() === 6) ?
                    '[Уltimo] dddd [У s] LT' : // Saturday + Sunday
                    '[Уltima] dddd [У s] LT'; // Monday - Friday
            },
            sameElse: 'L'
        },
        relativeTime : {
            future : "em %s",
            past : "%s atrУЁs",
            s : "segundos",
            m : "um minuto",
            mm : "%d minutos",
            h : "uma hora",
            hh : "%d horas",
            d : "um dia",
            dd : "%d dias",
            M : "um mУЊs",
            MM : "%d meses",
            y : "um ano",
            yy : "%d anos"
        },
        ordinal : '%dТК'
    });
}));
// moment.js language configuration
// language : portuguese (pt)
// author : Jefferson : https://github.com/jalex79

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('pt', {
        months : "Janeiro_Fevereiro_MarУЇo_Abril_Maio_Junho_Julho_Agosto_Setembro_Outubro_Novembro_Dezembro".split("_"),
        monthsShort : "Jan_Fev_Mar_Abr_Mai_Jun_Jul_Ago_Set_Out_Nov_Dez".split("_"),
        weekdays : "Domingo_Segunda-feira_TerУЇa-feira_Quarta-feira_Quinta-feira_Sexta-feira_SУЁbado".split("_"),
        weekdaysShort : "Dom_Seg_Ter_Qua_Qui_Sex_SУЁb".split("_"),
        weekdaysMin : "Dom_2ТЊ_3ТЊ_4ТЊ_5ТЊ_6ТЊ_SУЁb".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD/MM/YYYY",
            LL : "D [de] MMMM [de] YYYY",
            LLL : "D [de] MMMM [de] YYYY LT",
            LLLL : "dddd, D [de] MMMM [de] YYYY LT"
        },
        calendar : {
            sameDay: '[Hoje У s] LT',
            nextDay: '[AmanhУЃ У s] LT',
            nextWeek: 'dddd [У s] LT',
            lastDay: '[Ontem У s] LT',
            lastWeek: function () {
                return (this.day() === 0 || this.day() === 6) ?
                    '[Уltimo] dddd [У s] LT' : // Saturday + Sunday
                    '[Уltima] dddd [У s] LT'; // Monday - Friday
            },
            sameElse: 'L'
        },
        relativeTime : {
            future : "em %s",
            past : "%s atrУЁs",
            s : "segundos",
            m : "um minuto",
            mm : "%d minutos",
            h : "uma hora",
            hh : "%d horas",
            d : "um dia",
            dd : "%d dias",
            M : "um mУЊs",
            MM : "%d meses",
            y : "um ano",
            yy : "%d anos"
        },
        ordinal : '%dТК',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : romanian (ro)
// author : Vlad Gurdiga : https://github.com/gurdiga
// author : Valentin Agachi : https://github.com/avaly

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('ro', {
        months : "Ianuarie_Februarie_Martie_Aprilie_Mai_Iunie_Iulie_August_Septembrie_Octombrie_Noiembrie_Decembrie".split("_"),
        monthsShort : "Ian_Feb_Mar_Apr_Mai_Iun_Iul_Aug_Sep_Oct_Noi_Dec".split("_"),
        weekdays : "DuminicФ_Luni_MarХЃi_Miercuri_Joi_Vineri_SУЂmbФtФ".split("_"),
        weekdaysShort : "Dum_Lun_Mar_Mie_Joi_Vin_SУЂm".split("_"),
        weekdaysMin : "Du_Lu_Ma_Mi_Jo_Vi_SУЂ".split("_"),
        longDateFormat : {
            LT : "H:mm",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY H:mm",
            LLLL : "dddd, D MMMM YYYY H:mm"
        },
        calendar : {
            sameDay: "[azi la] LT",
            nextDay: '[mУЂine la] LT',
            nextWeek: 'dddd [la] LT',
            lastDay: '[ieri la] LT',
            lastWeek: '[fosta] dddd [la] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : "peste %s",
            past : "%s УЎn urmФ",
            s : "cУЂteva secunde",
            m : "un minut",
            mm : "%d minute",
            h : "o orФ",
            hh : "%d ore",
            d : "o zi",
            dd : "%d zile",
            M : "o lunФ",
            MM : "%d luni",
            y : "un an",
            yy : "%d ani"
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : russian (ru)
// author : Viktorminator : https://github.com/Viktorminator
// Author : Menelion ElensУКle : https://github.com/Oire

(function (factory) {
    factory(moment);
}(function (moment) {
    function plural(word, num) {
        var forms = word.split('_');
        return num % 10 === 1 && num % 100 !== 11 ? forms[0] : (num % 10 >= 2 && num % 10 <= 4 && (num % 100 < 10 || num % 100 >= 20) ? forms[1] : forms[2]);
    }

    function relativeTimeWithPlural(number, withoutSuffix, key) {
        var format = {
            'mm': 'аМаИаНббаА_аМаИаНббб_аМаИаНбб',
            'hh': 'баАб_баАбаА_баАбаОаВ',
            'dd': 'аДаЕаНб_аДаНб_аДаНаЕаЙ',
            'MM': 'аМаЕббб_аМаЕбббаА_аМаЕбббаЕаВ',
            'yy': 'аГаОаД_аГаОаДаА_аЛаЕб'
        };
        if (key === 'm') {
            return withoutSuffix ? 'аМаИаНббаА' : 'аМаИаНббб';
        }
        else {
            return number + ' ' + plural(format[key], +number);
        }
    }

    function monthsCaseReplace(m, format) {
        var months = {
            'nominative': 'баНаВаАбб_баЕаВбаАаЛб_аМаАбб_аАаПбаЕаЛб_аМаАаЙ_аИбаНб_аИбаЛб_аАаВаГббб_баЕаНббаБбб_аОаКббаБбб_аНаОбаБбб_аДаЕаКаАаБбб'.split('_'),
            'accusative': 'баНаВаАбб_баЕаВбаАаЛб_аМаАббаА_аАаПбаЕаЛб_аМаАб_аИбаНб_аИбаЛб_аАаВаГбббаА_баЕаНббаБбб_аОаКббаБбб_аНаОбаБбб_аДаЕаКаАаБбб'.split('_')
        },

        nounCase = (/D[oD]?(\[[^\[\]]*\]|\s+)+MMMM?/).test(format) ?
            'accusative' :
            'nominative';

        return months[nounCase][m.month()];
    }

    function monthsShortCaseReplace(m, format) {
        var monthsShort = {
            'nominative': 'баНаВ_баЕаВ_аМаАб_аАаПб_аМаАаЙ_аИбаНб_аИбаЛб_аАаВаГ_баЕаН_аОаКб_аНаОб_аДаЕаК'.split('_'),
            'accusative': 'баНаВ_баЕаВ_аМаАб_аАаПб_аМаАб_аИбаНб_аИбаЛб_аАаВаГ_баЕаН_аОаКб_аНаОб_аДаЕаК'.split('_')
        },

        nounCase = (/D[oD]?(\[[^\[\]]*\]|\s+)+MMMM?/).test(format) ?
            'accusative' :
            'nominative';

        return monthsShort[nounCase][m.month()];
    }

    function weekdaysCaseReplace(m, format) {
        var weekdays = {
            'nominative': 'аВаОбаКбаЕбаЕаНбаЕ_аПаОаНаЕаДаЕаЛбаНаИаК_аВбаОбаНаИаК_ббаЕаДаА_баЕбаВаЕбаГ_аПббаНаИбаА_ббаБаБаОбаА'.split('_'),
            'accusative': 'аВаОбаКбаЕбаЕаНбаЕ_аПаОаНаЕаДаЕаЛбаНаИаК_аВбаОбаНаИаК_ббаЕаДб_баЕбаВаЕбаГ_аПббаНаИбб_ббаБаБаОбб'.split('_')
        },

        nounCase = (/\[ ?[ааВ] ?(?:аПбаОбаЛбб|баЛаЕаДббббб)? ?\] ?dddd/).test(format) ?
            'accusative' :
            'nominative';

        return weekdays[nounCase][m.day()];
    }

    return moment.lang('ru', {
        months : monthsCaseReplace,
        monthsShort : monthsShortCaseReplace,
        weekdays : weekdaysCaseReplace,
        weekdaysShort : "аВб_аПаН_аВб_бб_бб_аПб_баБ".split("_"),
        weekdaysMin : "аВб_аПаН_аВб_бб_бб_аПб_баБ".split("_"),
        monthsParse : [/^баНаВ/i, /^баЕаВ/i, /^аМаАб/i, /^аАаПб/i, /^аМаА[аЙ|б]/i, /^аИбаН/i, /^аИбаЛ/i, /^аАаВаГ/i, /^баЕаН/i, /^аОаКб/i, /^аНаОб/i, /^аДаЕаК/i],
        longDateFormat : {
            LT : "HH:mm",
            L : "DD.MM.YYYY",
            LL : "D MMMM YYYY аГ.",
            LLL : "D MMMM YYYY аГ., LT",
            LLLL : "dddd, D MMMM YYYY аГ., LT"
        },
        calendar : {
            sameDay: '[аЁаЕаГаОаДаНб аВ] LT',
            nextDay: '[ааАаВббаА аВ] LT',
            lastDay: '[абаЕбаА аВ] LT',
            nextWeek: function () {
                return this.day() === 2 ? '[ааО] dddd [аВ] LT' : '[а] dddd [аВ] LT';
            },
            lastWeek: function () {
                switch (this.day()) {
                case 0:
                    return '[а аПбаОбаЛаОаЕ] dddd [аВ] LT';
                case 1:
                case 2:
                case 4:
                    return '[а аПбаОбаЛбаЙ] dddd [аВ] LT';
                case 3:
                case 5:
                case 6:
                    return '[а аПбаОбаЛбб] dddd [аВ] LT';
                }
            },
            sameElse: 'L'
        },
        relativeTime : {
            future : "баЕбаЕаЗ %s",
            past : "%s аНаАаЗаАаД",
            s : "аНаЕбаКаОаЛбаКаО баЕаКбаНаД",
            m : relativeTimeWithPlural,
            mm : relativeTimeWithPlural,
            h : "баАб",
            hh : relativeTimeWithPlural,
            d : "аДаЕаНб",
            dd : relativeTimeWithPlural,
            M : "аМаЕббб",
            MM : relativeTimeWithPlural,
            y : "аГаОаД",
            yy : relativeTimeWithPlural
        },

        // M. E.: those two are virtually unused but a user might want to implement them for his/her website for some reason

        meridiem : function (hour, minute, isLower) {
            if (hour < 4) {
                return "аНаОбаИ";
            } else if (hour < 12) {
                return "бббаА";
            } else if (hour < 17) {
                return "аДаНб";
            } else {
                return "аВаЕбаЕбаА";
            }
        },

        ordinal: function (number, period) {
            switch (period) {
            case 'M':
            case 'd':
            case 'DDD':
                return number + '-аЙ';
            case 'D':
                return number + '-аГаО';
            case 'w':
            case 'W':
                return number + '-б';
            default:
                return number;
            }
        },

        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : slovak (sk)
// author : Martin Minka : https://github.com/k2s
// based on work of petrbela : https://github.com/petrbela

(function (factory) {
    factory(moment);
}(function (moment) {
    var months = "januУЁr_februУЁr_marec_aprУ­l_mУЁj_jУКn_jУКl_august_september_oktУГber_november_december".split("_"),
        monthsShort = "jan_feb_mar_apr_mУЁj_jУКn_jУКl_aug_sep_okt_nov_dec".split("_");

    function plural(n) {
        return (n > 1) && (n < 5);
    }

    function translate(number, withoutSuffix, key, isFuture) {
        var result = number + " ";
        switch (key) {
        case 's':  // a few seconds / in a few seconds / a few seconds ago
            return (withoutSuffix || isFuture) ? 'pУЁr sekУКnd' : 'pУЁr sekundami';
        case 'm':  // a minute / in a minute / a minute ago
            return withoutSuffix ? 'minУКta' : (isFuture ? 'minУКtu' : 'minУКtou');
        case 'mm': // 9 minutes / in 9 minutes / 9 minutes ago
            if (withoutSuffix || isFuture) {
                return result + (plural(number) ? 'minУКty' : 'minУКt');
            } else {
                return result + 'minУКtami';
            }
            break;
        case 'h':  // an hour / in an hour / an hour ago
            return withoutSuffix ? 'hodina' : (isFuture ? 'hodinu' : 'hodinou');
        case 'hh': // 9 hours / in 9 hours / 9 hours ago
            if (withoutSuffix || isFuture) {
                return result + (plural(number) ? 'hodiny' : 'hodУ­n');
            } else {
                return result + 'hodinami';
            }
            break;
        case 'd':  // a day / in a day / a day ago
            return (withoutSuffix || isFuture) ? 'deХ' : 'dХom';
        case 'dd': // 9 days / in 9 days / 9 days ago
            if (withoutSuffix || isFuture) {
                return result + (plural(number) ? 'dni' : 'dnУ­');
            } else {
                return result + 'dХami';
            }
            break;
        case 'M':  // a month / in a month / a month ago
            return (withoutSuffix || isFuture) ? 'mesiac' : 'mesiacom';
        case 'MM': // 9 months / in 9 months / 9 months ago
            if (withoutSuffix || isFuture) {
                return result + (plural(number) ? 'mesiace' : 'mesiacov');
            } else {
                return result + 'mesiacmi';
            }
            break;
        case 'y':  // a year / in a year / a year ago
            return (withoutSuffix || isFuture) ? 'rok' : 'rokom';
        case 'yy': // 9 years / in 9 years / 9 years ago
            if (withoutSuffix || isFuture) {
                return result + (plural(number) ? 'roky' : 'rokov');
            } else {
                return result + 'rokmi';
            }
            break;
        }
    }

    return moment.lang('sk', {
        months : months,
        monthsShort : monthsShort,
        monthsParse : (function (months, monthsShort) {
            var i, _monthsParse = [];
            for (i = 0; i < 12; i++) {
                // use custom parser to solve problem with July (Фervenec)
                _monthsParse[i] = new RegExp('^' + months[i] + '$|^' + monthsShort[i] + '$', 'i');
            }
            return _monthsParse;
        }(months, monthsShort)),
        weekdays : "nedeФОa_pondelok_utorok_streda_ХЁtvrtok_piatok_sobota".split("_"),
        weekdaysShort : "ne_po_ut_st_ХЁt_pi_so".split("_"),
        weekdaysMin : "ne_po_ut_st_ХЁt_pi_so".split("_"),
        longDateFormat : {
            LT: "H:mm",
            L : "DD.MM.YYYY",
            LL : "D. MMMM YYYY",
            LLL : "D. MMMM YYYY LT",
            LLLL : "dddd D. MMMM YYYY LT"
        },
        calendar : {
            sameDay: "[dnes o] LT",
            nextDay: '[zajtra o] LT',
            nextWeek: function () {
                switch (this.day()) {
                case 0:
                    return '[v nedeФОu o] LT';
                case 1:
                case 2:
                    return '[v] dddd [o] LT';
                case 3:
                    return '[v stredu o] LT';
                case 4:
                    return '[vo ХЁtvrtok o] LT';
                case 5:
                    return '[v piatok o] LT';
                case 6:
                    return '[v sobotu o] LT';
                }
            },
            lastDay: '[vФera o] LT',
            lastWeek: function () {
                switch (this.day()) {
                case 0:
                    return '[minulУК nedeФОu o] LT';
                case 1:
                case 2:
                    return '[minulУН] dddd [o] LT';
                case 3:
                    return '[minulУК stredu o] LT';
                case 4:
                case 5:
                    return '[minulУН] dddd [o] LT';
                case 6:
                    return '[minulУК sobotu o] LT';
                }
            },
            sameElse: "L"
        },
        relativeTime : {
            future : "za %s",
            past : "pred %s",
            s : translate,
            m : translate,
            mm : translate,
            h : translate,
            hh : translate,
            d : translate,
            dd : translate,
            M : translate,
            MM : translate,
            y : translate,
            yy : translate
        },
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : slovenian (sl)
// author : Robert SedovХЁek : https://github.com/sedovsek

(function (factory) {
    factory(moment);
}(function (moment) {
    function translate(number, withoutSuffix, key) {
        var result = number + " ";
        switch (key) {
        case 'm':
            return withoutSuffix ? 'ena minuta' : 'eno minuto';
        case 'mm':
            if (number === 1) {
                result += 'minuta';
            } else if (number === 2) {
                result += 'minuti';
            } else if (number === 3 || number === 4) {
                result += 'minute';
            } else {
                result += 'minut';
            }
            return result;
        case 'h':
            return withoutSuffix ? 'ena ura' : 'eno uro';
        case 'hh':
            if (number === 1) {
                result += 'ura';
            } else if (number === 2) {
                result += 'uri';
            } else if (number === 3 || number === 4) {
                result += 'ure';
            } else {
                result += 'ur';
            }
            return result;
        case 'dd':
            if (number === 1) {
                result += 'dan';
            } else {
                result += 'dni';
            }
            return result;
        case 'MM':
            if (number === 1) {
                result += 'mesec';
            } else if (number === 2) {
                result += 'meseca';
            } else if (number === 3 || number === 4) {
                result += 'mesece';
            } else {
                result += 'mesecev';
            }
            return result;
        case 'yy':
            if (number === 1) {
                result += 'leto';
            } else if (number === 2) {
                result += 'leti';
            } else if (number === 3 || number === 4) {
                result += 'leta';
            } else {
                result += 'let';
            }
            return result;
        }
    }

    return moment.lang('sl', {
        months : "januar_februar_marec_april_maj_junij_julij_avgust_september_oktober_november_december".split("_"),
        monthsShort : "jan._feb._mar._apr._maj._jun._jul._avg._sep._okt._nov._dec.".split("_"),
        weekdays : "nedelja_ponedeljek_torek_sreda_Фetrtek_petek_sobota".split("_"),
        weekdaysShort : "ned._pon._tor._sre._Фet._pet._sob.".split("_"),
        weekdaysMin : "ne_po_to_sr_Фe_pe_so".split("_"),
        longDateFormat : {
            LT : "H:mm",
            L : "DD. MM. YYYY",
            LL : "D. MMMM YYYY",
            LLL : "D. MMMM YYYY LT",
            LLLL : "dddd, D. MMMM YYYY LT"
        },
        calendar : {
            sameDay  : '[danes ob] LT',
            nextDay  : '[jutri ob] LT',

            nextWeek : function () {
                switch (this.day()) {
                case 0:
                    return '[v] [nedeljo] [ob] LT';
                case 3:
                    return '[v] [sredo] [ob] LT';
                case 6:
                    return '[v] [soboto] [ob] LT';
                case 1:
                case 2:
                case 4:
                case 5:
                    return '[v] dddd [ob] LT';
                }
            },
            lastDay  : '[vФeraj ob] LT',
            lastWeek : function () {
                switch (this.day()) {
                case 0:
                case 3:
                case 6:
                    return '[prejХЁnja] dddd [ob] LT';
                case 1:
                case 2:
                case 4:
                case 5:
                    return '[prejХЁnji] dddd [ob] LT';
                }
            },
            sameElse : 'L'
        },
        relativeTime : {
            future : "Фez %s",
            past   : "%s nazaj",
            s      : "nekaj sekund",
            m      : translate,
            mm     : translate,
            h      : translate,
            hh     : translate,
            d      : "en dan",
            dd     : translate,
            M      : "en mesec",
            MM     : translate,
            y      : "eno leto",
            yy     : translate
        },
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : Albanian (sq)
// author : FlakУЋrim Ismani : https://github.com/flakerimi
// author: Menelion ElensУКle: https://github.com/Oire (tests)

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('sq', {
        months : "Janar_Shkurt_Mars_Prill_Maj_Qershor_Korrik_Gusht_Shtator_Tetor_NУЋntor_Dhjetor".split("_"),
        monthsShort : "Jan_Shk_Mar_Pri_Maj_Qer_Kor_Gus_Sht_Tet_NУЋn_Dhj".split("_"),
        weekdays : "E Diel_E HУЋnУЋ_E Marte_E MУЋrkure_E Enjte_E Premte_E ShtunУЋ".split("_"),
        weekdaysShort : "Die_HУЋn_Mar_MУЋr_Enj_Pre_Sht".split("_"),
        weekdaysMin : "D_H_Ma_MУЋ_E_P_Sh".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd, D MMMM YYYY LT"
        },
        calendar : {
            sameDay : '[Sot nУЋ] LT',
            nextDay : '[Neser nУЋ] LT',
            nextWeek : 'dddd [nУЋ] LT',
            lastDay : '[Dje nУЋ] LT',
            lastWeek : 'dddd [e kaluar nУЋ] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "nУЋ %s",
            past : "%s me parУЋ",
            s : "disa seconda",
            m : "njУЋ minut",
            mm : "%d minutea",
            h : "njУЋ orУЋ",
            hh : "%d orУЋ",
            d : "njУЋ ditУЋ",
            dd : "%d ditУЋ",
            M : "njУЋ muaj",
            MM : "%d muaj",
            y : "njУЋ vit",
            yy : "%d vite"
        },
        ordinal : '%d.',
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : swedish (sv)
// author : Jens Alm : https://github.com/ulmus

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('sv', {
        months : "januari_februari_mars_april_maj_juni_juli_augusti_september_oktober_november_december".split("_"),
        monthsShort : "jan_feb_mar_apr_maj_jun_jul_aug_sep_okt_nov_dec".split("_"),
        weekdays : "sУЖndag_mУЅndag_tisdag_onsdag_torsdag_fredag_lУЖrdag".split("_"),
        weekdaysShort : "sУЖn_mУЅn_tis_ons_tor_fre_lУЖr".split("_"),
        weekdaysMin : "sУЖ_mУЅ_ti_on_to_fr_lУЖ".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "YYYY-MM-DD",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd D MMMM YYYY LT"
        },
        calendar : {
            sameDay: '[Idag] LT',
            nextDay: '[Imorgon] LT',
            lastDay: '[IgУЅr] LT',
            nextWeek: 'dddd LT',
            lastWeek: '[FУЖrra] dddd[en] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : "om %s",
            past : "fУЖr %s sedan",
            s : "nУЅgra sekunder",
            m : "en minut",
            mm : "%d minuter",
            h : "en timme",
            hh : "%d timmar",
            d : "en dag",
            dd : "%d dagar",
            M : "en mУЅnad",
            MM : "%d mУЅnader",
            y : "ett УЅr",
            yy : "%d УЅr"
        },
        ordinal : function (number) {
            var b = number % 10,
                output = (~~ (number % 100 / 10) === 1) ? 'e' :
                (b === 1) ? 'a' :
                (b === 2) ? 'a' :
                (b === 3) ? 'e' : 'e';
            return number + output;
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : thai (th)
// author : Kridsada Thanabulpong : https://github.com/sirn

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('th', {
        months : "рИЁрИрИЃрИВрИрИЁ_рИрИИрИЁрИ рИВрИрИБрИрИрЙ_рИЁрИЕрИрИВрИрИЁ_рЙрИЁрИЉрИВрИЂрИ_рИрИЄрИЉрИ рИВрИрИЁ_рИЁрИДрИрИИрИрИВрИЂрИ_рИрИЃрИрИрИВрИрИЁ_рИЊрИДрИрИЋрИВрИрИЁ_рИрИБрИрИЂрИВрИЂрИ_рИрИИрИЅрИВрИрИЁ_рИрИЄрИЈрИрИДрИрИВрИЂрИ_рИрИБрИрИЇрИВрИрИЁ".split("_"),
        monthsShort : "рИЁрИрИЃрИВ_рИрИИрИЁрИ рИВ_рИЁрИЕрИрИВ_рЙрИЁрИЉрИВ_рИрИЄрИЉрИ рИВ_рИЁрИДрИрИИрИрИВ_рИрИЃрИрИрИВ_рИЊрИДрИрИЋрИВ_рИрИБрИрИЂрИВ_рИрИИрИЅрИВ_рИрИЄрИЈрИрИДрИрИВ_рИрИБрИрИЇрИВ".split("_"),
        weekdays : "рИ­рИВрИрИДрИрИЂрЙ_рИрИБрИрИрИЃрЙ_рИ­рИБрИрИрИВрИЃ_рИрИИрИ_рИрИЄрИЋрИБрИЊрИрИрИЕ_рИЈрИИрИрИЃрЙ_рЙрИЊрИВрИЃрЙ".split("_"),
        weekdaysShort : "рИ­рИВрИрИДрИрИЂрЙ_рИрИБрИрИрИЃрЙ_рИ­рИБрИрИрИВрИЃ_рИрИИрИ_рИрИЄрИЋрИБрИЊ_рИЈрИИрИрИЃрЙ_рЙрИЊрИВрИЃрЙ".split("_"), // yes, three characters difference
        weekdaysMin : "рИ­рИВ._рИ._рИ­._рИ._рИрИЄ._рИЈ._рИЊ.".split("_"),
        longDateFormat : {
            LT : "H рИрИВрИЌрИДрИрИВ m рИрИВрИрИЕ",
            L : "YYYY/MM/DD",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY рЙрИЇрИЅрИВ LT",
            LLLL : "рИЇрИБрИddddрИрИЕрЙ D MMMM YYYY рЙрИЇрИЅрИВ LT"
        },
        meridiem : function (hour, minute, isLower) {
            if (hour < 12) {
                return "рИрЙрИ­рИрЙрИрИЕрЙрИЂрИ";
            } else {
                return "рИЋрИЅрИБрИрЙрИрИЕрЙрИЂрИ";
            }
        },
        calendar : {
            sameDay : '[рИЇрИБрИрИрИЕрЙ рЙрИЇрИЅрИВ] LT',
            nextDay : '[рИрИЃрИИрЙрИрИрИЕрЙ рЙрИЇрИЅрИВ] LT',
            nextWeek : 'dddd[рИЋрИрЙрИВ рЙрИЇрИЅрИВ] LT',
            lastDay : '[рЙрИЁрИЗрЙрИ­рИЇрИВрИрИрИЕрЙ рЙрИЇрИЅрИВ] LT',
            lastWeek : '[рИЇрИБрИ]dddd[рИрИЕрЙрЙрИЅрЙрИЇ рЙрИЇрИЅрИВ] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "рИ­рИЕрИ %s",
            past : "%sрИрИЕрЙрЙрИЅрЙрИЇ",
            s : "рЙрИЁрЙрИрИЕрЙрИЇрИДрИрИВрИрИЕ",
            m : "1 рИрИВрИрИЕ",
            mm : "%d рИрИВрИрИЕ",
            h : "1 рИрИБрЙрИЇрЙрИЁрИ",
            hh : "%d рИрИБрЙрИЇрЙрИЁрИ",
            d : "1 рИЇрИБрИ",
            dd : "%d рИЇрИБрИ",
            M : "1 рЙрИрИЗрИ­рИ",
            MM : "%d рЙрИрИЗрИ­рИ",
            y : "1 рИрИЕ",
            yy : "%d рИрИЕ"
        }
    });
}));
// moment.js language configuration
// language : Tagalog/Filipino (tl-ph)
// author : Dan Hagman

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('tl-ph', {
        months : "Enero_Pebrero_Marso_Abril_Mayo_Hunyo_Hulyo_Agosto_Setyembre_Oktubre_Nobyembre_Disyembre".split("_"),
        monthsShort : "Ene_Peb_Mar_Abr_May_Hun_Hul_Ago_Set_Okt_Nob_Dis".split("_"),
        weekdays : "Linggo_Lunes_Martes_Miyerkules_Huwebes_Biyernes_Sabado".split("_"),
        weekdaysShort : "Lin_Lun_Mar_Miy_Huw_Biy_Sab".split("_"),
        weekdaysMin : "Li_Lu_Ma_Mi_Hu_Bi_Sab".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "MM/D/YYYY",
            LL : "MMMM D, YYYY",
            LLL : "MMMM D, YYYY LT",
            LLLL : "dddd, MMMM DD, YYYY LT"
        },
        calendar : {
            sameDay: "[Ngayon sa] LT",
            nextDay: '[Bukas sa] LT',
            nextWeek: 'dddd [sa] LT',
            lastDay: '[Kahapon sa] LT',
            lastWeek: 'dddd [huling linggo] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : "sa loob ng %s",
            past : "%s ang nakalipas",
            s : "ilang segundo",
            m : "isang minuto",
            mm : "%d minuto",
            h : "isang oras",
            hh : "%d oras",
            d : "isang araw",
            dd : "%d araw",
            M : "isang buwan",
            MM : "%d buwan",
            y : "isang taon",
            yy : "%d taon"
        },
        ordinal : function (number) {
            return number;
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : turkish (tr)
// authors : Erhan Gundogan : https://github.com/erhangundogan,
//           Burak YiФit Kaya: https://github.com/BYK

(function (factory) {
    factory(moment);
}(function (moment) {

    var suffixes = {
        1: "'inci",
        5: "'inci",
        8: "'inci",
        70: "'inci",
        80: "'inci",

        2: "'nci",
        7: "'nci",
        20: "'nci",
        50: "'nci",

        3: "'УМncУМ",
        4: "'УМncУМ",
        100: "'УМncУМ",

        6: "'ncФБ",

        9: "'uncu",
        10: "'uncu",
        30: "'uncu",

        60: "'ФБncФБ",
        90: "'ФБncФБ"
    };

    return moment.lang('tr', {
        months : "Ocak_Хubat_Mart_Nisan_MayФБs_Haziran_Temmuz_AФustos_EylУМl_Ekim_KasФБm_AralФБk".split("_"),
        monthsShort : "Oca_Хub_Mar_Nis_May_Haz_Tem_AФu_Eyl_Eki_Kas_Ara".split("_"),
        weekdays : "Pazar_Pazartesi_SalФБ_УarХamba_PerХembe_Cuma_Cumartesi".split("_"),
        weekdaysShort : "Paz_Pts_Sal_Уar_Per_Cum_Cts".split("_"),
        weekdaysMin : "Pz_Pt_Sa_Уa_Pe_Cu_Ct".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD.MM.YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd, D MMMM YYYY LT"
        },
        calendar : {
            sameDay : '[bugУМn saat] LT',
            nextDay : '[yarФБn saat] LT',
            nextWeek : '[haftaya] dddd [saat] LT',
            lastDay : '[dУМn] LT',
            lastWeek : '[geУЇen hafta] dddd [saat] LT',
            sameElse : 'L'
        },
        relativeTime : {
            future : "%s sonra",
            past : "%s УЖnce",
            s : "birkaУЇ saniye",
            m : "bir dakika",
            mm : "%d dakika",
            h : "bir saat",
            hh : "%d saat",
            d : "bir gУМn",
            dd : "%d gУМn",
            M : "bir ay",
            MM : "%d ay",
            y : "bir yФБl",
            yy : "%d yФБl"
        },
        ordinal : function (number) {
            if (number === 0) {  // special case for zero
                return number + "'ФБncФБ";
            }
            var a = number % 10,
                b = number % 100 - a,
                c = number >= 100 ? 100 : null;

            return number + (suffixes[a] || suffixes[b] || suffixes[c]);
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : Morocco Central Atlas TamaziЩЃt in Latin (tzm-la)
// author : Abdel Said : https://github.com/abdelsaid

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('tzm-la', {
        months : "innayr_brЫЄayrЫЄ_marЫЄsЫЄ_ibrir_mayyw_ywnyw_ywlywz_ЩЃwХЁt_ХЁwtanbir_ktЫЄwbrЫЄ_nwwanbir_dwjnbir".split("_"),
        monthsShort : "innayr_brЫЄayrЫЄ_marЫЄsЫЄ_ibrir_mayyw_ywnyw_ywlywz_ЩЃwХЁt_ХЁwtanbir_ktЫЄwbrЫЄ_nwwanbir_dwjnbir".split("_"),
        weekdays : "asamas_aynas_asinas_akras_akwas_asimwas_asiсИyas".split("_"),
        weekdaysShort : "asamas_aynas_asinas_akras_akwas_asimwas_asiсИyas".split("_"),
        weekdaysMin : "asamas_aynas_asinas_akras_akwas_asimwas_asiсИyas".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd D MMMM YYYY LT"
        },
        calendar : {
            sameDay: "[asdkh g] LT",
            nextDay: '[aska g] LT',
            nextWeek: 'dddd [g] LT',
            lastDay: '[assant g] LT',
            lastWeek: 'dddd [g] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : "dadkh s yan %s",
            past : "yan %s",
            s : "imik",
            m : "minuсИ",
            mm : "%d minuсИ",
            h : "saЩa",
            hh : "%d tassaЩin",
            d : "ass",
            dd : "%d ossan",
            M : "ayowr",
            MM : "%d iyyirn",
            y : "asgas",
            yy : "%d isgasn"
        },
        week : {
            dow : 6, // Saturday is the first day of the week.
            doy : 12  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : Morocco Central Atlas TamaziЩЃt (tzm)
// author : Abdel Said : https://github.com/abdelsaid

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('tzm', {
        months : "тЕтЕтЕтДАтЕЂтЕ_тДБтЕтДАтЕЂтЕ_тЕтДАтЕтЕ_тЕтДБтЕтЕтЕ_тЕтДАтЕЂтЕЂтЕ_тЕЂтЕтЕтЕЂтЕ_тЕЂтЕтЕтЕЂтЕтЕЃ_тЕтЕтЕтЕ_тЕтЕтЕтДАтЕтДБтЕтЕ_тДНтЕтЕтДБтЕ_тЕтЕтЕЁтДАтЕтДБтЕтЕ_тДЗтЕтЕтЕтДБтЕтЕ".split("_"),
        monthsShort : "тЕтЕтЕтДАтЕЂтЕ_тДБтЕтДАтЕЂтЕ_тЕтДАтЕтЕ_тЕтДБтЕтЕтЕ_тЕтДАтЕЂтЕЂтЕ_тЕЂтЕтЕтЕЂтЕ_тЕЂтЕтЕтЕЂтЕтЕЃ_тЕтЕтЕтЕ_тЕтЕтЕтДАтЕтДБтЕтЕ_тДНтЕтЕтДБтЕ_тЕтЕтЕЁтДАтЕтДБтЕтЕ_тДЗтЕтЕтЕтДБтЕтЕ".split("_"),
        weekdays : "тДАтЕтДАтЕтДАтЕ_тДАтЕЂтЕтДАтЕ_тДАтЕтЕтЕтДАтЕ_тДАтДНтЕтДАтЕ_тДАтДНтЕЁтДАтЕ_тДАтЕтЕтЕтЕЁтДАтЕ_тДАтЕтЕтДЙтЕЂтДАтЕ".split("_"),
        weekdaysShort : "тДАтЕтДАтЕтДАтЕ_тДАтЕЂтЕтДАтЕ_тДАтЕтЕтЕтДАтЕ_тДАтДНтЕтДАтЕ_тДАтДНтЕЁтДАтЕ_тДАтЕтЕтЕтЕЁтДАтЕ_тДАтЕтЕтДЙтЕЂтДАтЕ".split("_"),
        weekdaysMin : "тДАтЕтДАтЕтДАтЕ_тДАтЕЂтЕтДАтЕ_тДАтЕтЕтЕтДАтЕ_тДАтДНтЕтДАтЕ_тДАтДНтЕЁтДАтЕ_тДАтЕтЕтЕтЕЁтДАтЕ_тДАтЕтЕтДЙтЕЂтДАтЕ".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "dddd D MMMM YYYY LT"
        },
        calendar : {
            sameDay: "[тДАтЕтДЗтЕ тДД] LT",
            nextDay: '[тДАтЕтДНтДА тДД] LT',
            nextWeek: 'dddd [тДД] LT',
            lastDay: '[тДАтЕтДАтЕтЕ тДД] LT',
            lastWeek: 'dddd [тДД] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : "тДЗтДАтДЗтЕ тЕ тЕЂтДАтЕ %s",
            past : "тЕЂтДАтЕ %s",
            s : "тЕтЕтЕтДН",
            m : "тЕтЕтЕтЕтДК",
            mm : "%d тЕтЕтЕтЕтДК",
            h : "тЕтДАтЕтДА",
            hh : "%d тЕтДАтЕтЕтДАтЕтЕтЕ",
            d : "тДАтЕтЕ",
            dd : "%d oтЕтЕтДАтЕ",
            M : "тДАтЕЂoтЕтЕ",
            MM : "%d тЕтЕЂтЕЂтЕтЕтЕ",
            y : "тДАтЕтДГтДАтЕ",
            yy : "%d тЕтЕтДГтДАтЕтЕ"
        },
        week : {
            dow : 6, // Saturday is the first day of the week.
            doy : 12  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : ukrainian (uk)
// author : zemlanin : https://github.com/zemlanin
// Author : Menelion ElensУКle : https://github.com/Oire

(function (factory) {
    factory(moment);
}(function (moment) {
    function plural(word, num) {
        var forms = word.split('_');
        return num % 10 === 1 && num % 100 !== 11 ? forms[0] : (num % 10 >= 2 && num % 10 <= 4 && (num % 100 < 10 || num % 100 >= 20) ? forms[1] : forms[2]);
    }

    function relativeTimeWithPlural(number, withoutSuffix, key) {
        var format = {
            'mm': 'баВаИаЛаИаНаА_баВаИаЛаИаНаИ_баВаИаЛаИаН',
            'hh': 'аГаОаДаИаНаА_аГаОаДаИаНаИ_аГаОаДаИаН',
            'dd': 'аДаЕаНб_аДаНб_аДаНбаВ',
            'MM': 'аМббббб_аМббббб_аМбббббаВ',
            'yy': 'ббаК_баОаКаИ_баОаКбаВ'
        };
        if (key === 'm') {
            return withoutSuffix ? 'баВаИаЛаИаНаА' : 'баВаИаЛаИаНб';
        }
        else if (key === 'h') {
            return withoutSuffix ? 'аГаОаДаИаНаА' : 'аГаОаДаИаНб';
        }
        else {
            return number + ' ' + plural(format[key], +number);
        }
    }

    function monthsCaseReplace(m, format) {
        var months = {
            'nominative': 'бббаЕаНб_аЛббаИаЙ_аБаЕбаЕаЗаЕаНб_аКаВббаЕаНб_ббаАаВаЕаНб_баЕбаВаЕаНб_аЛаИаПаЕаНб_баЕбаПаЕаНб_аВаЕбаЕбаЕаНб_аЖаОаВбаЕаНб_аЛаИббаОаПаАаД_аГббаДаЕаНб'.split('_'),
            'accusative': 'бббаНб_аЛббаОаГаО_аБаЕбаЕаЗаНб_аКаВббаНб_ббаАаВаНб_баЕбаВаНб_аЛаИаПаНб_баЕбаПаНб_аВаЕбаЕбаНб_аЖаОаВбаНб_аЛаИббаОаПаАаДаА_аГббаДаНб'.split('_')
        },

        nounCase = (/D[oD]? *MMMM?/).test(format) ?
            'accusative' :
            'nominative';

        return months[nounCase][m.month()];
    }

    function weekdaysCaseReplace(m, format) {
        var weekdays = {
            'nominative': 'аНаЕаДбаЛб_аПаОаНаЕаДбаЛаОаК_аВбаВбаОбаОаК_баЕбаЕаДаА_баЕбаВаЕб_аПтббаНаИбб_ббаБаОбаА'.split('_'),
            'accusative': 'аНаЕаДбаЛб_аПаОаНаЕаДбаЛаОаК_аВбаВбаОбаОаК_баЕбаЕаДб_баЕбаВаЕб_аПтббаНаИбб_ббаБаОбб'.split('_'),
            'genitive': 'аНаЕаДбаЛб_аПаОаНаЕаДбаЛаКаА_аВбаВбаОбаКаА_баЕбаЕаДаИ_баЕбаВаЕбаГаА_аПтббаНаИбб_ббаБаОбаИ'.split('_')
        },

        nounCase = (/(\[[ааВаЃб]\]) ?dddd/).test(format) ?
            'accusative' :
            ((/\[?(?:аМаИаНбаЛаОб|аНаАбббаПаНаОб)? ?\] ?dddd/).test(format) ?
                'genitive' :
                'nominative');

        return weekdays[nounCase][m.day()];
    }

    function processHoursFunction(str) {
        return function () {
            return str + 'аО' + (this.hours() === 11 ? 'аБ' : '') + '] LT';
        };
    }

    return moment.lang('uk', {
        months : monthsCaseReplace,
        monthsShort : "ббб_аЛбб_аБаЕб_аКаВбб_ббаАаВ_баЕбаВ_аЛаИаП_баЕбаП_аВаЕб_аЖаОаВб_аЛаИбб_аГббаД".split("_"),
        weekdays : weekdaysCaseReplace,
        weekdaysShort : "аНаД_аПаН_аВб_бб_бб_аПб_баБ".split("_"),
        weekdaysMin : "аНаД_аПаН_аВб_бб_бб_аПб_баБ".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD.MM.YYYY",
            LL : "D MMMM YYYY б.",
            LLL : "D MMMM YYYY б., LT",
            LLLL : "dddd, D MMMM YYYY б., LT"
        },
        calendar : {
            sameDay: processHoursFunction('[аЁбаОаГаОаДаНб '),
            nextDay: processHoursFunction('[ааАаВббаА '),
            lastDay: processHoursFunction('[абаОбаА '),
            nextWeek: processHoursFunction('[аЃ] dddd ['),
            lastWeek: function () {
                switch (this.day()) {
                case 0:
                case 3:
                case 5:
                case 6:
                    return processHoursFunction('[ааИаНбаЛаОб] dddd [').call(this);
                case 1:
                case 2:
                case 4:
                    return processHoursFunction('[ааИаНбаЛаОаГаО] dddd [').call(this);
                }
            },
            sameElse: 'L'
        },
        relativeTime : {
            future : "аЗаА %s",
            past : "%s баОаМб",
            s : "аДаЕаКбаЛбаКаА баЕаКбаНаД",
            m : relativeTimeWithPlural,
            mm : relativeTimeWithPlural,
            h : "аГаОаДаИаНб",
            hh : relativeTimeWithPlural,
            d : "аДаЕаНб",
            dd : relativeTimeWithPlural,
            M : "аМббббб",
            MM : relativeTimeWithPlural,
            y : "ббаК",
            yy : relativeTimeWithPlural
        },

        // M. E.: those two are virtually unused but a user might want to implement them for his/her website for some reason

        meridiem : function (hour, minute, isLower) {
            if (hour < 4) {
                return "аНаОбб";
            } else if (hour < 12) {
                return "баАаНаКб";
            } else if (hour < 17) {
                return "аДаНб";
            } else {
                return "аВаЕбаОбаА";
            }
        },

        ordinal: function (number, period) {
            switch (period) {
            case 'M':
            case 'd':
            case 'DDD':
            case 'w':
            case 'W':
                return number + '-аЙ';
            case 'D':
                return number + '-аГаО';
            default:
                return number;
            }
        },

        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 1st is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : uzbek
// author : Sardor Muminov : https://github.com/muminoff

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('uz', {
        months : "баНаВаАбб_баЕаВбаАаЛб_аМаАбб_аАаПбаЕаЛб_аМаАаЙ_аИбаНб_аИбаЛб_аАаВаГббб_баЕаНббаБбб_аОаКббаБбб_аНаОбаБбб_аДаЕаКаАаБбб".split("_"),
        monthsShort : "баНаВ_баЕаВ_аМаАб_аАаПб_аМаАаЙ_аИбаН_аИбаЛ_аАаВаГ_баЕаН_аОаКб_аНаОб_аДаЕаК".split("_"),
        weekdays : "аЏаКбаАаНаБаА_аббаАаНаБаА_аЁаЕбаАаНаБаА_аЇаОббаАаНаБаА_ааАаЙбаАаНаБаА_абаМаА_аЈаАаНаБаА".split("_"),
        weekdaysShort : "аЏаКб_абб_аЁаЕб_аЇаОб_ааАаЙ_абаМ_аЈаАаН".split("_"),
        weekdaysMin : "аЏаК_аб_аЁаЕ_аЇаО_ааА_аб_аЈаА".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD/MM/YYYY",
            LL : "D MMMM YYYY",
            LLL : "D MMMM YYYY LT",
            LLLL : "D MMMM YYYY, dddd LT"
        },
        calendar : {
            sameDay : '[абаГбаН баОаАб] LT [аДаА]',
            nextDay : '[а­ббаАаГаА] LT [аДаА]',
            nextWeek : 'dddd [аКбаНаИ баОаАб] LT [аДаА]',
            lastDay : '[ааЕбаА баОаАб] LT [аДаА]',
            lastWeek : '[аЃбаГаАаН] dddd [аКбаНаИ баОаАб] LT [аДаА]',
            sameElse : 'L'
        },
        relativeTime : {
            future : "аЏаКаИаН %s аИбаИаДаА",
            past : "ааИб аНаЕбаА %s аОаЛаДаИаН",
            s : "ббббаАб",
            m : "аБаИб аДаАаКаИаКаА",
            mm : "%d аДаАаКаИаКаА",
            h : "аБаИб баОаАб",
            hh : "%d баОаАб",
            d : "аБаИб аКбаН",
            dd : "%d аКбаН",
            M : "аБаИб аОаЙ",
            MM : "%d аОаЙ",
            y : "аБаИб аЙаИаЛ",
            yy : "%d аЙаИаЛ"
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 7  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : vietnamese (vn)
// author : Bang Nguyen : https://github.com/bangnk

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('vn', {
        months : "thУЁng 1_thУЁng 2_thУЁng 3_thУЁng 4_thУЁng 5_thУЁng 6_thУЁng 7_thУЁng 8_thУЁng 9_thУЁng 10_thУЁng 11_thУЁng 12".split("_"),
        monthsShort : "Th01_Th02_Th03_Th04_Th05_Th06_Th07_Th08_Th09_Th10_Th11_Th12".split("_"),
        weekdays : "chсЛЇ nhсК­t_thсЛЉ hai_thсЛЉ ba_thсЛЉ tЦА_thсЛЉ nФm_thсЛЉ sУЁu_thсЛЉ bсКЃy".split("_"),
        weekdaysShort : "CN_T2_T3_T4_T5_T6_T7".split("_"),
        weekdaysMin : "CN_T2_T3_T4_T5_T6_T7".split("_"),
        longDateFormat : {
            LT : "HH:mm",
            L : "DD/MM/YYYY",
            LL : "D MMMM [nФm] YYYY",
            LLL : "D MMMM [nФm] YYYY LT",
            LLLL : "dddd, D MMMM [nФm] YYYY LT",
            l : "DD/M/YYYY",
            ll : "D MMM YYYY",
            lll : "D MMM YYYY LT",
            llll : "ddd, D MMM YYYY LT"
        },
        calendar : {
            sameDay: "[HУДm nay lУКc] LT",
            nextDay: '[NgУ y mai lУКc] LT',
            nextWeek: 'dddd [tuсКЇn tсЛi lУКc] LT',
            lastDay: '[HУДm qua lУКc] LT',
            lastWeek: 'dddd [tuсКЇn rсЛi lУКc] LT',
            sameElse: 'L'
        },
        relativeTime : {
            future : "%s tсЛi",
            past : "%s trЦАсЛc",
            s : "vУ i giУЂy",
            m : "mсЛt phУКt",
            mm : "%d phУКt",
            h : "mсЛt giсЛ",
            hh : "%d giсЛ",
            d : "mсЛt ngУ y",
            dd : "%d ngУ y",
            M : "mсЛt thУЁng",
            MM : "%d thУЁng",
            y : "mсЛt nФm",
            yy : "%d nФm"
        },
        ordinal : function (number) {
            return number;
        },
        week : {
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : chinese
// author : suupic : https://github.com/suupic
// author : Zeno Zeng : https://github.com/zenozeng

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('zh-cn', {
        months : "фИц_фКц_фИц_хц_фКц_х­ц_фИц_хЋц_фЙц_хц_хфИц_хфКц".split("_"),
        monthsShort : "1ц_2ц_3ц_4ц_5ц_6ц_7ц_8ц_9ц_10ц_11ц_12ц".split("_"),
        weekdays : "цццЅ_ццфИ_ццфК_ццфИ_ццх_ццфК_ццх­".split("_"),
        weekdaysShort : "хЈцЅ_хЈфИ_хЈфК_хЈфИ_хЈх_хЈфК_хЈх­".split("_"),
        weekdaysMin : "цЅ_фИ_фК_фИ_х_фК_х­".split("_"),
        longDateFormat : {
            LT : "AhчЙmm",
            L : "YYYYхЙДMMMDцЅ",
            LL : "YYYYхЙДMMMDцЅ",
            LLL : "YYYYхЙДMMMDцЅLT",
            LLLL : "YYYYхЙДMMMDцЅddddLT",
            l : "YYYYхЙДMMMDцЅ",
            ll : "YYYYхЙДMMMDцЅ",
            lll : "YYYYхЙДMMMDцЅLT",
            llll : "YYYYхЙДMMMDцЅddddLT"
        },
        meridiem : function (hour, minute, isLower) {
            var hm = hour * 100 + minute;
            if (hm < 600) {
                return "хцЈ";
            } else if (hm < 900) {
                return "цЉфИ";
            } else if (hm < 1130) {
                return "фИх";
            } else if (hm < 1230) {
                return "фИ­х";
            } else if (hm < 1800) {
                return "фИх";
            } else {
                return "цфИ";
            }
        },
        calendar : {
            sameDay : function () {
                return this.minutes() === 0 ? "[фЛхЄЉ]Ah[чЙцД]" : "[фЛхЄЉ]LT";
            },
            nextDay : function () {
                return this.minutes() === 0 ? "[цхЄЉ]Ah[чЙцД]" : "[цхЄЉ]LT";
            },
            lastDay : function () {
                return this.minutes() === 0 ? "[цЈхЄЉ]Ah[чЙцД]" : "[цЈхЄЉ]LT";
            },
            nextWeek : function () {
                var startOfWeek, prefix;
                startOfWeek = moment().startOf('week');
                prefix = this.unix() - startOfWeek.unix() >= 7 * 24 * 3600 ? '[фИ]' : '[цЌ]';
                return this.minutes() === 0 ? prefix + "dddAhчЙцД" : prefix + "dddAhчЙmm";
            },
            lastWeek : function () {
                var startOfWeek, prefix;
                startOfWeek = moment().startOf('week');
                prefix = this.unix() < startOfWeek.unix()  ? '[фИ]' : '[цЌ]';
                return this.minutes() === 0 ? prefix + "dddAhчЙцД" : prefix + "dddAhчЙmm";
            },
            sameElse : 'L'
        },
        ordinal : function (number, period) {
            switch (period) {
            case "d":
            case "D":
            case "DDD":
                return number + "цЅ";
            case "M":
                return number + "ц";
            case "w":
            case "W":
                return number + "хЈ";
            default:
                return number;
            }
        },
        relativeTime : {
            future : "%sх",
            past : "%sх",
            s : "х чЇ",
            m : "1хщ",
            mm : "%dхщ",
            h : "1хАцЖ",
            hh : "%dхАцЖ",
            d : "1хЄЉ",
            dd : "%dхЄЉ",
            M : "1фИЊц",
            MM : "%dфИЊц",
            y : "1хЙД",
            yy : "%dхЙД"
        },
        week : {
            // GB/T 7408-1994уцАцЎххфКЄцЂц МхМТЗфПЁцЏфКЄцЂТЗцЅцхцЖщДшЁЈчЄКцГуфИISO 8601:1988ч­ц
            dow : 1, // Monday is the first day of the week.
            doy : 4  // The week that contains Jan 4th is the first week of the year.
        }
    });
}));
// moment.js language configuration
// language : traditional chinese (zh-tw)
// author : Ben : https://github.com/ben-lin

(function (factory) {
    factory(moment);
}(function (moment) {
    return moment.lang('zh-tw', {
        months : "фИц_фКц_фИц_хц_фКц_х­ц_фИц_хЋц_фЙц_хц_хфИц_хфКц".split("_"),
        monthsShort : "1ц_2ц_3ц_4ц_5ц_6ц_7ц_8ц_9ц_10ц_11ц_12ц".split("_"),
        weekdays : "цццЅ_ццфИ_ццфК_ццфИ_ццх_ццфК_ццх­".split("_"),
        weekdaysShort : "щБцЅ_щБфИ_щБфК_щБфИ_щБх_щБфК_щБх­".split("_"),
        weekdaysMin : "цЅ_фИ_фК_фИ_х_фК_х­".split("_"),
        longDateFormat : {
            LT : "AhщЛmm",
            L : "YYYYхЙДMMMDцЅ",
            LL : "YYYYхЙДMMMDцЅ",
            LLL : "YYYYхЙДMMMDцЅLT",
            LLLL : "YYYYхЙДMMMDцЅddddLT",
            l : "YYYYхЙДMMMDцЅ",
            ll : "YYYYхЙДMMMDцЅ",
            lll : "YYYYхЙДMMMDцЅLT",
            llll : "YYYYхЙДMMMDцЅddddLT"
        },
        meridiem : function (hour, minute, isLower) {
            var hm = hour * 100 + minute;
            if (hm < 900) {
                return "цЉфИ";
            } else if (hm < 1130) {
                return "фИх";
            } else if (hm < 1230) {
                return "фИ­х";
            } else if (hm < 1800) {
                return "фИх";
            } else {
                return "цфИ";
            }
        },
        calendar : {
            sameDay : '[фЛхЄЉ]LT',
            nextDay : '[цхЄЉ]LT',
            nextWeek : '[фИ]ddddLT',
            lastDay : '[цЈхЄЉ]LT',
            lastWeek : '[фИ]ddddLT',
            sameElse : 'L'
        },
        ordinal : function (number, period) {
            switch (period) {
            case "d" :
            case "D" :
            case "DDD" :
                return number + "цЅ";
            case "M" :
                return number + "ц";
            case "w" :
            case "W" :
                return number + "щБ";
            default :
                return number;
            }
        },
        relativeTime : {
            future : "%sхЇ",
            past : "%sх",
            s : "хЙОчЇ",
            m : "фИхщ",
            mm : "%dхщ",
            h : "фИхАц",
            hh : "%dхАц",
            d : "фИхЄЉ",
            dd : "%dхЄЉ",
            M : "фИхц",
            MM : "%dхц",
            y : "фИхЙД",
            yy : "%dхЙД"
        }
    });
}));

    moment.lang('en');


    /************************************
        Exposing Moment
    ************************************/

    function makeGlobal(deprecate) {
        var warned = false, local_moment = moment;
        /*global ender:false */
        if (typeof ender !== 'undefined') {
            return;
        }
        // here, `this` means `window` in the browser, or `global` on the server
        // add `moment` as a global object via a string identifier,
        // for Closure Compiler "advanced" mode
        if (deprecate) {
            this.moment = function () {
                if (!warned && console && console.warn) {
                    warned = true;
                    console.warn(
                            "Accessing Moment through the global scope is " +
                            "deprecated, and will be removed in an upcoming " +
                            "release.");
                }
                return local_moment.apply(null, arguments);
            };
        } else {
            this['moment'] = moment;
        }
    }

    // CommonJS module is defined
    if (hasModule) {
        module.exports = moment;
        makeGlobal(true);
    } else if (typeof define === "function" && define.amd) {
        define("moment", function (require, exports, module) {
            if (module.config().noGlobal !== true) {
                // If user provided noGlobal, he is aware of global
                makeGlobal(module.config().noGlobal === undefined);
            }

            return moment;
        });
    } else {
        makeGlobal();
    }
}).call(this);
