var SWCElements = {

    /**
     * Custom swc checkboxes
     */
    checkbox:  {

        /**
         * Current element class name:
         *
         * @param className {String}
         *
         */
        className: 'swc-checkbox',

        /**
         * Current element active class name:
         *
         * @param activeClassName {String}
         *
         */
        activeClassName: 'checked',

        /**
         * Init all custom checkboxes on the page:
         */
        initAll: function() {
            var self = this;

            $('body')
                .on('click', '.' + this.className + ':not(.prevent-click, .disabled)', function(e) {
                    self.triggerChange($(e.target));
                })
                .on('swc-checkbox:swc-change', '.' + this.className, function(e, value) {
                    self.customChange($(e.target), value);
                });
        },

        /**
         * Trigger user changes on a checkbox:
         *
         * @param element {Object}
         * @param fireEvent {Boolean}
         *
         */
        triggerChange: function(element) {
            var checkBox = element.hasClass(this.className) ? element : element.closest('.' + this.className),
                state = checkBox.data('value');

            if (state) {
                checkBox.removeClass(this.activeClassName);
            } else {
                checkBox.addClass(this.activeClassName);
            }

            checkBox.data('value', !state);
            checkBox.trigger('swc-checkbox:change', !state);
        },

        /**
         * Trigger application changes on a checkbox:
         *
         * @param element {Object}
         *
         */
        customChange: function(element, value) {
            var checkBox = element.hasClass(this.className) ? element : element.closest('.' + this.className);

            if (value) {
                checkBox.addClass(this.activeClassName);
            } else {
                checkBox.removeClass(this.activeClassName);
            }

            checkBox.data('value', value);
        }
    },

    /**
     * Custom swc checkbox with one additional position
     */
    checkboxExtended:  {

        /**
         * Current element class name:
         *
         * @param className {String}
         *
         */
        className: 'swc-extended',

        /**
         * Current element active class name:
         *
         * @param activeClassName {String}
         *
         */
        activeClassName: 'checked',

        partlyClassName: 'partly',

        /**
         * Init all custom checkboxes on the page:
         */
        initAll: function() {
            var self = this;

            $('body')
                .on('click', '.' + this.className + ':not(.prevent-click)', function(e) {
                    self.triggerChange($(e.target));
                })
                .on('swc-extended:swc-change', '.' + this.className, function(e, value) {
                    self.customChange($(e.target), value);
                });
        },

        /**
         * Trigger user changes on a checkbox:
         *
         * @param element {Object}
         * @param fireEvent {Boolean}
         *
         */
        triggerChange: function(element) {
            var checkBox = element.hasClass(this.className) ? element : element.closest('.' + this.className),
                state = checkBox.data('value'),
                result = state,
                extend = checkBox.hasClass('partly');

            checkBox.removeClass(this.activeClassName);

            if (state || extend) {
                result = false;
            } else {
                checkBox.addClass(this.activeClassName);
                result = true;
            }


            checkBox.data('value', result);
            checkBox.trigger('swc-checkbox:change', result);
        },

        /**
         * Trigger application changes on a checkbox:
         *
         * @param element {Object}
         *
         */
        customChange: function(element, value) {
            var checkBox = element.hasClass(this.className) ? element : element.closest('.' + this.className);

            checkBox.removeClass(this.activeClassName).removeClass(this.partlyClassName);

            if (value === "partly") {
                checkBox.addClass(this.partlyClassName);
            } else if(value) {
                checkBox.addClass(this.activeClassName);
            }

            checkBox.data('value', value);
        }
    },

    /**
     * Custom ON-OFF switcher
     */
    switcher:  {

        /**
         * Current element class name:
         *
         * @param className {String}
         *
         */
        className: 'swc-switcher',

        /**
         * Current element active class name:
         *
         * @param activeClassName {String}
         *
         */
        activeClassName: 'checked',

        /**
         * Current element width:
         *
         * @param activeClassName {String}
         *
         */
        switcherWidth: 40,

        /**
         * Init all custom switchers on the page:
         */
        initAll: function() {
            var self = this, preventClick = false;

            $('body')
                .on('click', '.' + this.className + ' .switcher-container', function(e) {
                    if(!preventClick){
                        self.triggerChange($(e.target));
                    }
                })
                .on('swc-switcher:swc-change', '.' + this.className, function(e, value) {
                    self.customChange($(e.target), value);
                })
                .on('mousedown', '.' + this.className + ' .btn', function(e) {
                    var element = $(this);
                    element.data('switch', true);
                    element.data('mouseOffset', e.pageX);
                    element.data('isChecked', element.closest('.'+self.className).hasClass(self.activeClassName));
                })
                .on('mouseup', '.' + this.className + ' .btn', function(e) {
                    var element = $(this);
                    if(element.data('switch')){
                        element.data('switch', false);
                        stopModify(element.closest('.switcher-container').find('.container'));
                    }
                })
                .on('mouseleave', '.' + this.className + ' .btn', function(e) {
                    var element = $(this);
                    if(element.data('switch')){
                        element.data('switch', false);
                        stopModify(element.closest('.switcher-container').find('.container'));
                    }
                })
                .on('mousemove', '.' + this.className + ' .btn', function(e) {
                    var element = $(this);
                    if(element.data('switch')){
                        preventClick = true;
                        var alpha = +element.data('mouseOffset') - e.pageX;
                        var containerPosition = +element.closest('.switcher-container').find('.container').css('right').replace('px','');
                        if(containerPosition+alpha>=0 && containerPosition+alpha<=40){
                            element.closest('.switcher-container').find('.container').css({
                                'right': '+='+alpha+'px'
                            });
                            element.closest('.switcher-container').find('.btn').css({
                                'right': '+='+alpha+'px'
                            });
                            element.data('mouseOffset', e.pageX);
                        }
                    }
                });
            var stopModify = function(element){
                /**
                 * prevents 'click' event execution
                 */
                setTimeout(function(){preventClick = false;}, 10);
                var containerPosition = +element.css('right').replace('px',''),
                    switcher = element.closest('.'+self.className),
                    newState = true;

                if(containerPosition>19){
                    newState = false;
                }
                self.customChange(element, newState);
                var stateWasChanged = (newState !== element.closest('.'+self.className).find('.btn').data('isChecked'));
                if(stateWasChanged){
                    switcher.trigger('swc-switcher:change', newState);
                }
            };
        },

        /**
         * Trigger user changes on a switcher:
         *
         * @param element {Object}
         * @param fireEvent {Boolean}
         *
         */
        triggerChange: function(element) {
            var switcher = element.hasClass(this.className) ? element : element.closest("." + this.className),
                state = switcher.data('value');

            if (!state) {
                switcher.addClass(this.activeClassName);
                switcher.find('.container').animate({'right': '0px'}, 100);
                switcher.find('.btn').animate({'right': '0px'}, 100);
            } else {
                switcher.removeClass(this.activeClassName);
                switcher.find('.container').animate({'right': '38px'}, 100);
                switcher.find('.btn').animate({'right': '40px'}, 100);
            }

            switcher.data('value', !state);
            switcher.trigger('swc-switcher:change', !state);
        },

        /**
         * Trigger application changes on a switcher:
         *
         * @param element {Object}
         *
         */
        customChange: function(element, value) {
            var switcher = element.hasClass(this.className) ? element : element.closest("." + this.className);

            if (value) {
                switcher.addClass(this.activeClassName);
                switcher.find('.container').animate({'right': '0px'}, 100);
                switcher.find('.btn').animate({'right': '0px'}, 100);
            } else {
                switcher.removeClass(this.activeClassName);
                switcher.find('.container').animate({'right': '38px'}, 100);
                switcher.find('.btn').animate({'right': '40px'}, 100);
            }

            switcher.data('value', value);

        }
    },

    /**
     * Custom swc radio buttons
     */
    radioButton: {

        /**
         * Current element class name:
         *
         * @param className {String}
         *
         */
        className: 'swc-radio-buttons',

        /**
         * Current element active class name:
         *
         * @param activeClassName {String}
         *
         */
        activeClassName: 'active',

        /**
         * Init all custom radio buttons on the page:
         */
        initAll: function() {
            var self = this;

            $('body')
                .on('click', '.' + this.className + ':not(.disabled) .mode:not(.disabled)', function(e) {
                    self.triggerChange($(e.target));
                })
                .on('swc-radio-buttons:swc-change', '.' + this.className, function(e, value) {
                    self.customChange($(e.target), value);
                });
        },

        /**
         * Trigger user changes on a radio button:
         *
         * @param element {Object}
         * @param fireEvent {Boolean}
         *
         */
        triggerChange: function(element) {
            var mode = element.hasClass('.mode') ? element : element.closest('.mode'),
                radioGroup = mode.closest('.' + this.className);

            // Do nothing if clicking on active radio button
            if (mode.hasClass('active')) {
                return;
            }

            // Remove active class from previous mode
            radioGroup.find('.mode').removeClass('active');

            // Set active class to current button
            mode.addClass('active');

            // Set radioGroup group value
            radioGroup.data('value', mode.data('value'));
            radioGroup.trigger('swc-radio-buttons:change', mode.data('value'));
        },

        /**
         * Trigger application changes on a radio button:
         *
         * @param button {Object}
         *
         */
        customChange: function(element, value) {
            var radioGroup = element.hasClass(this.className) ? element : element.closest('.' + this.className);

            // Remove active class from previous mode
            radioGroup.find('.mode').removeClass('active');

            // Set active class to current button
            radioGroup.find('.mode[data-value="' + value + '"]').addClass('active');

            radioGroup.data('value', value);
        }

    },

    /**
     * Custom swc drop downs
     */
    dropDown: {

        /**
         * Current element class name:
         *
         * @param className {String}
         *
         */
        className: 'swc-dropdown',

        /**
         * Current element overlay class name:
         *
         * @param classNameOverlay {String}
         *
         */
        classNameOverlay: 'swc-dropdown-overlay',

        /**
         * Set element type to be handled correctly
         */
        type: 'absolute-position',

        /**
         * Current element tag name:
         *
         * @param tagName {String}
         *
         */
        tagName: 'div',

        /**
         * Current element hidden tag name:
         *
         * @param hiddenField {String}
         *
         */
        hiddenField: 'input',

        /**
         * Current element active class name:
         *
         * @param activeClassName {String}
         *
         */
        activeClassName: 'selected',

        /**
         * Init all custom elements on the page:
         */
        initAll: function() {
            var self = this;

            $('body')
                .on('click', '.' + this.className + ':not(.disabled)', function(e) {
                    SWCElements.global.closeAll(self.className);
                    self.open($(e.target));
                    e.stopPropagation();
                })
                .on('click', '.' + this.classNameOverlay + ' .option.selected', function(e) {
                    self.close($(e.target));
                    e.stopPropagation();
                })
                .on('click', '.' + this.classNameOverlay + ' .option:not(.selected)', function(e) {
                    self.change($(e.target));
                    e.stopPropagation();
                })
                .on(this.className + ':swc-change', '.' + this.className, function(e, value) {
                    self.customChange($(e.target), value);
                });
        },

        /**
         * Trigger drop down open:
         *
         * @param element {Object}
         *
         */
        open: function(element) {
            var dropDown = element.hasClass(this.className) ? element : element.closest("." + this.className),
                className = dropDown.data('class'),
                hideDefault = dropDown.data('hide-default'),
                options = dropDown.data('options'),
                selected = dropDown.data('selected'),
                selectedDefault = selected,
                position = dropDown.offset(),
                width = dropDown.innerWidth(),
                value = dropDown.data('value'),
                // This additional width needed to handle scrollbar which appears when option list is not empty
                additionalWidth = getObjectLength(options) > 5 ? 20 : 0,
                template = $.template("swc-dropdown", swc.Templates.get("swc-dropdown").get('content')),
                optionList = [], // support for option list as array
                selectedOption;

            // Remove existing dropdowns
            if ($('.' + this.classNameOverlay).size()) {
                this.closeAll();
            }

            // Hide native dropdown:
            dropDown.css('visibility', 'hidden');
            
            if (!selected) {
                selectedDefault = dropDown.find('.selected-option').text();
            }
            
            // Correct behaviour: options should be an array of objects,
            // so for backward compatibility we add this processor here
            if (_.isArray(options)) {
                optionList = options;
                selectedOption = _.findWhere(optionList, {value: selected + ""});
            } else {
                if (selected) {
                    selectedOption = {
                        value: selected,
                        name: options[selected].name,
                        additionalLabel: options[selected].additionalLabel
                    };
                }
            }
            
            // Append element to body
            $('body').append(
                $.tmpl(template, {
                        className: className,
                        options: options,
                        optionList: optionList.length > 0 ? optionList : null,
                        hideDefault: hideDefault,
                        selectedDefault: selectedDefault,
                        selectedOption: selectedOption
                    }
                )
            );
            
            // Set element position and width
            $('.' + this.classNameOverlay).css({
                'top': position.top + 'px',
                'left': position.left + 'px',
                'width': width + additionalWidth
            });

            $('body').trigger('swc-dropdown:opened', [{'$dropDown': $('.' + this.classNameOverlay)}]);
        },

        /**
         * Trigger dropdown change event:
         *
         * @param element
         *
         */
        change: function(element) {
            var dropDownOverlay = element.hasClass(this.classNameOverlay) ?
                    element :
                    element.closest('.' + this.classNameOverlay),
                className = dropDownOverlay.data('class'),
                dropDown = $('.swc-dropdown[data-class="' + className + '"]'),
                option = $(element).hasClass('option') ? $(element) : $(element).closest('.option'),
                value = option.data('value');
            
            this.renderSelectedOption(element, dropDown);

            // Trigger change event:
            dropDown.trigger('swc-dropdown:change', value);

            this.close(element);
        },

        /**
         * Trigger dropdown application change:
         *
         * @param element
         * @param value
         */
        customChange: function(element, key) {
            var dropDown = element.hasClass(this.className) ? element : element.closest("." + this.className);

            // Update data and Insert new value to visible part of dropdown
            this.renderSelectedOption(element, dropDown, key);
        },

        /**
         * Update data and Insert new value to visible part of dropdown
         * 
         * @param element {DOMElement} on which event was triggered
         * @param dropDown {swcElements.Dropdown} element
         * @param selectedValue {String} - optional, only used when triggered by custom change.
         */
        renderSelectedOption: function (element, dropDown, selectedValue) {
            var optionTemplate = $.template("swc-select-option", swc.Templates.get("swc-select-option").get('content')),
                options = dropDown.data('options'),
                option = $(element).hasClass('option') ? $(element) : $(element).closest('.option'),
                selectedOption;

            // Update data and Insert new value to visible part of dropdown
            if (options) {
                if (_.isArray(options)) {
                    // NOTE: `findWhere` can't compare String and Number correctly,
                    // but sometimes {selectedOption.value} is a digit, thus we appending an empty string to it
                    selectedOption = _.findWhere(options,
                        { value: (!_.isUndefined(selectedValue) ? selectedValue : option.data('value')) + "" }
                    );
                } else {
                    if (selectedValue && options[selectedValue]) {
                        selectedOption = {
                            "value": options[selectedValue].value,
                            "name": options[selectedValue].name,
                            "additionalLabel": options[selectedValue].additionalLabel
                        }
                    } else if(options[option.data('key')]) {
                        selectedOption = {
                            "value": options[option.data('key')].value,
                            "name": options[option.data('key')].name,
                            "additionalLabel": options[option.data('key')].additionalLabel
                        }
                    }
                }

                // Update dropdown element (a kind of "selectbox") with selected data/value
                if (selectedOption) {
                    var previousValue = dropDown.data('value');

                    if (!_.isUndefined(previousValue)) {
                        dropDown.data('previous-value', previousValue);
                    } else {
                        dropDown.data('previous-value', selectedOption.value);
                    }

                    dropDown.data('selected', selectedOption.value);
                    dropDown.data('value', selectedOption.value);

                    dropDown.find('.selected-option').html(
                        $.tmpl(optionTemplate, { selectedOption: selectedOption })
                    );
                }
            }
        },

        /**
         * Trigger dropdown overlay to be closed
         *
         * @param element
         */
        close: function(element) {
            var dropDownOverlay = element.hasClass(this.classNameOverlay) ?
                    element :
                    element.closest('.' + this.classNameOverlay),
                className = dropDownOverlay.data('class'),
                dropDown = $('.swc-dropdown[data-class="' + className + '"]');

            // Remove dropdown overlay:
            if (dropDownOverlay.size()) {
                dropDownOverlay.remove();
            }

            // Show dropdown if exists:
            if (dropDown.size()) {
                dropDown.css('visibility', 'visible');
            }
        },

        /**
         * Close all dropdowns on the page:
         */
        closeAll: function() {
            var self = this;

            $.each($('.' + this.classNameOverlay), function(key, element) {
                self.close($(element));
            });
        }
    },

    expandableList: {

        /**
         * Current element class name:
         *
         * @param className {String}
         *
         */
        className: 'expandable-list',

        /**
         * Current element item class name:
         *
         * @param itemClassName {String}
         *
         */
        itemClassName: 'expandable-list-item',

        /**
         * Current element item hidden class name:
         *
         * @param itemHiddenClassName {String}
         *
         */
        itemHiddenClassName: 'list-item-hidden',

        /**
         * Current element tag name:
         *
         * @param tagName {String}
         *
         */
        tagName: 'div',

        /**
         * Init all custom elements on the page:
         */
        initAll: function() {
            var self = this;

            $('body')
                .on('click', '.' + this.className + ' .show-hidden', function(e) {
                    self.setSelected($(e.target));
                })
                .on('click', '.' + this.className + ' .do-edit:not(.disabled)', function(e) {
                    self.setStateEdit($(e.target), {user: true, undo: false});
                })
                .on('click', '.' + this.className + ' .do-edit.active-edit', function(e) {
                    self.setStateEdit($(e.target), {user: true, undo: true});
                })
                .on('click', '.' + this.className + ' .do-delete:not(.disabled)', function(e) {
                    self.setStateDelete($(e.target));
                })
                .on('click', '.' + this.className + ' .do-custom-edit:not(.disabled)', function(e){
                    self.setStateCustomEdit($(e.target));
                })
                .on('expandable:swc-open', '.' + this.className, function(e, value) {
                    self.setSelected($('.' + self.itemClassName+"[data-key='"+value+"'] .show-hidden:visible"), true);
                })
                .on('expandable:swc-close', '.' + this.className, function(e, value) {
                    self.setSelected($('.' + self.itemClassName+"[data-key='"+value+"'] .show-hidden:visible"), false);
                })
                .on('expandable:swc-edit', '.' + this.className, function(e, value) {
                    self.setStateEdit($('.' + self.itemClassName+"[data-key='"+value+"'] .do-edit:visible"), {user: false, undo: false});
                })
                .on('expandable:swc-no-edit', '.' + this.className, function(e, value) {
                    self.setStateEdit($('.' + self.itemClassName+"[data-key='"+value+"'] .do-edit:visible"), {user: false, undo: true});
                })
                .on('expandable:swc-delete', '.' + this.className, function(e, value) {
                    self.setStateDelete($('.' + self.itemClassName+"[data-key='"+value+"'] .do-delete:visible"));
                });

        },

        setSelected: function(button, toOpen) {
            var fullList = button.closest('.' + this.className),
                itemList = button.closest('.' + this.itemClassName),
                state = button.closest(".show-hidden").hasClass('expanded') ? "expanded" : "closed";

            fullList.find('.' + this.itemClassName).removeClass("expanded");
            fullList.find('.' + this.itemClassName).find('.device').removeClass("selected");
            if (state === "closed" || toOpen) {
                itemList.addClass("expanded");
                itemList.find('.device').addClass("selected");
                fullList.trigger('expandable:open', itemList.attr('data-key'));
            } else {
                fullList.trigger('expandable:close', itemList.attr('data-key'));
            }
        },

        setStateEdit: function(button, options) {
            var fullList = button.closest('.' + this.className),
                itemList = button.closest('.' + this.itemClassName),
                state = itemList.hasClass('mode-editing') ? true : false;
            if(!state && !options.undo){
                itemList.addClass('mode-editing');
                if(options.user){
                    fullList.trigger('expandable:edit', itemList.attr('data-key'));
                }
            } else if(options.undo){
                itemList.removeClass('mode-editing');
                fullList.trigger('expandable:no-edit', itemList.attr('data-key'));
            }
        },

        setStateCustomEdit: function(button){
            var fullList = button.closest('.' + this.className),
                itemList = button.closest('.' + this.itemClassName);
            fullList.trigger('expandable:custom-edit', itemList.attr('data-key'));
        },

        setStateDelete: function(button) {
            var fullList = button.closest('.' + this.className),
                itemList = button.closest('.' + this.itemClassName);
            fullList.trigger('expandable:delete', itemList.attr('data-key'));
        }
    },

    /**
     * Popover elements
     */
    popovers: {

        /**
         * Set element type to be handled correctly
         */
        type: 'absolute-position',

        /**
         * Elements popover active toggle class:
         */
        className: 'has-popover',

        initAll: function() {
            var self = this;

            $('body')
                .on('click', '.' + this.className, function(e) {
                    var element = $(e.target).hasClass(self.className) ? $(e.target) : $(e.target).closest('.' + self.className);

                    if(!element.hasClass('no-click')){
                        if (element.hasClass('selected')) {
                            self.closeAll();
                        } else {
                            self.open($(e.target));
                        }

                        e.stopPropagation();
                    }
                })
                .on('popover:toggle', '.' + this.className, function(e) {
                    var element = $(e.target);

                    if (element.hasClass('selected')) {
                        self.closeAll();
                    } else {
                        self.open(element);
                    }

                    e.stopPropagation();

                })
                .on('popover:close', function(e) {
                    self.closeAll();
                    e.stopPropagation();
                });
        },

        /**
         * Open popover on selected element:
         */
        open: function(e) {
            var self = this,
                holder = e.hasClass(this.className) ? e : e.closest('.' + this.className),
                popover,
                popoverElement,
                template,
                popoverClassName = holder.data('popover-class-name'),
                options = {
                    template: holder.data('popover-template-id'),
                    templateData: holder.data('popover-template-data') ? holder.data('popover-template-data') : {},
                    placement: holder.data('popover-placement'),
                    holderClassName: holder.data('popover-holder')
                };

            if (!holder.data('popover-element')) {
                popoverElement = holder;
            } else {
                var searchClass = holder.data('popover-element');

                popoverElement = holder.find(holder.data('popover-element'));

                if (!e.hasClass(searchClass.replace('.', ''))) {
                    self.closeAll();
                    return;
                }
            }

            // Select template
            if (!swc.Templates.get(options.template)) {
                self.closeAll();
                return;
            }

            // Add values to template data:
            options.templateData.localeStrings = swc.models.Locale.getLocaleStrings();
            options.templateData.localeString = getTranslationStringsjQuery;
            options.templateData.formatDate = swc.models.Locale.formatDate;

            if (typeof(options.templateData) === 'string') {
                options.templateData = JSON.parse(options.templateData);
            }

            // Close all other elements:
            SWCElements.global.closeAll(self.className);

            // Prepare popover on element
            popoverElement.popover({
                html: true,
                content: $.tmpl(swc.Templates.get(options.template).get('content'), options.templateData),
                placement: options.placement,
                container: holder.data('popover-container') ? holder.data('popover-container') : "body"
            });

            // Display popover:
            popoverElement.popover('show');

            // Set holder elements selected
            holder.addClass('selected');
            popoverElement.addClass('selected');


            // Add data options to active popover:
            popover = $('body').find('.popover');

            popover.addClass(popoverClassName);

            // Set popover data
            popover.data('holderClassName', options.holderClassName);
        },

        /**
         * Close all popovers on the page:
         */
        closeAll: function() {
            $.each($('.has-popover.selected'), function(key, value) {
                var holder = $(value),
                    popoverElement;

                // Select base popover element:
                if (!holder.data('popover-element')) {
                    popoverElement = holder;
                } else {
                    popoverElement = holder.find(holder.data('popover-element'));
                }

                // Remove popover:
                holder.removeClass('selected');
                popoverElement.removeClass('selected');

                // Remove popover:
                popoverElement.popover("destroy");
            });
            $('body > .popover').remove();
        }
    },

    /**
     * Global body handler:
     */
    global: {

        // Close all absolute elements on body click
        initAll: function() {
            var self = this;

            $('body')
                .on('click', '.popover', function(e) {
                    e.stopPropagation();
                })
                .on('click', function(e) {
                    self.closeAll();
                });
        },

        /**
         * Close all elements:
         * @param activeElement
         */
        closeAll: function(activeElement) {
            $.each(SWCElements, function(key, value) {
                if (value.type && value.type === 'absolute-position') {
                    if ($.isFunction(value.closeAll)) {
                        value.closeAll();
                    }
                }
            });
        }
    },

    charactersFilter: {

        initAll: function() {
            var self = this;

            $('body')
                .on('keydown', '.characters-filter', function(e) {
                    self.checkCharacter(e);
                });
        },

        checkCharacter: function(e) {
            var character = getCharacterByKey(e.shiftKey, e.keyCode),
                regexp = $(e.target).data('characters-regexp'),
                // the following array contains codes of "C", "X", "V", "Z" keys
                // to have opportunity to check appropriate Ctrl+Key combinations
                serviceCtrlKeys = [67, 88, 86, 90],
                // the following array contains code of "Ins" key
                // to have opportunity to check appropriate Shift+Key combination
                serviceShiftKeys = [45];

            // Filter dots:
            if (e.keyCode === 190 || e.keyCode === 110) {
                character = ".";
            }

            // Enable Ctrl combinations
            if (e.ctrlKey && $.inArray(e.keyCode, serviceCtrlKeys) !== -1) {
                return;
            }

            // Enable Shift combinations
            if (e.shiftKey && $.inArray(e.keyCode, serviceShiftKeys) !== -1) {
                return;
            }

            // Enable global key codes:
            if (character && !character.length || !regexp) {
                return;
            } else {
                var regTemp = new RegExp(regexp, "ig");

                if (!regTemp.test(character)) {
                    e.preventDefault();
                }
            }
        }
    },

    /**
     * Modal windows functionality
     */
    modalWindow: {

        initAll: function() {
            var self = this;

            $('body')
                .on('click', '.modalWindow', function(e) {
                    e.stopPropagation();
                });
        },

        /**
         * Display modal windows
         *
         * @param options {Object}
         *
         * @description:
         *
         *  templateID: {String} - id of template to be shown in a modal window
         *  templateData: {Object} - data to be passed to template
         *
         *  className: {String} - will be added to the modal window
         *
         *  onApply: <function> which will be called when apply button is pressed
         *  onCancel: <function> which will be called when cancel button is pressed
         *  onShow: <function> which will be called when dialog window is shown
         *  onHide: <function> which will be called when dialog window is closed
         */
        show: function(options) {
            var modalShadow = $('<div class="modalWindowShadow"></div>'),
                modalWindow,
                modalWindowData = {
                    'width': 0,
                    'height': 0
                },
                modalWindowTemplate = $.template("modalWindow", swc.Templates.get('application:modal-window').get('content')),
                modalWindowTemplateContent = $("<span>undefined template</span>"),
                modalLocaleStrings = !_.isUndefined(options.localeStrings) ? options.localeStrings : swc.models.Locale.getLocaleStrings();

            // Hide previously opened window, when new window request (fixing windows duplicate):
            if ($('body').find('.modalWindow').size()) {
                SWCElements.modalWindow.hide();
            }

            // We need to add specific translations for the template:
            modalLocaleStrings = _.extend(modalLocaleStrings, swc.models.Locale.getLocaleStrings(options.templateID.replaceAll(":", "/")));

            // Define template for modal window content:
            if (options.templateID) {
                modalWindowTemplateContent = $.template("modalWindowContent", swc.Templates.get(options.templateID).get('content'));
            }

            // Append modal window shadow and modal window to application:
            $('body').append(modalShadow);
            $('body').append($.tmpl(modalWindowTemplate, {}));

            // Handle clicks on modal window shadow:
            modalShadow.on('click', function(e) {
                e.stopPropagation();
            });

            // Define modal window reference:
            modalWindow = $('.modalWindow');

            // Append modal window content
            modalWindow.html(
                $.tmpl(modalWindowTemplateContent, {
                        data: options.templateData ? options.templateData : {},
                        deviceIndex: function (item, map) {
                            var indexToIncriment = 0,
                                index = 0;

                            $.each(map, function(key, device) {
                                if (key === item) {
                                    index = indexToIncriment;
                                }

                                indexToIncriment++;
                            });

                            return index;
                        },
                        localeStrings: modalLocaleStrings,
                        localeString: getTranslationStringsjQuery,
                        formatDate: swc.models.Locale.formatDate
                    }
                )
            );

            // Add addition class name to modal window:
            if (options.className) {
                modalWindow.addClass(options.className);
            }

            // Set correct positioning of modal window:
            modalWindow.css({
                marginTop: -(modalWindow.outerHeight() / 2 + 75) + 'px',
                marginLeft: -(modalWindow.outerWidth() / 2) + 'px'
            });

            // Display the modal window itself
            modalWindow.show();

            // Trigger options onShow function
            if (options.onShow && $.isFunction(options.onShow)) {
                options.onShow();
            }

            // Cancel changes handler:
            modalWindow.find('.cancel-changes').on('mousedown', function(e) {
                var button = $(e.target).hasClass('button') ? $(e.target) : $(e.target).closest('.button');

                if (button.hasClass('disabled')) {
                    return;
                }

                if (options.onCancel && $.isFunction(options.onCancel)) {
                    options.onCancel();
                }

                e.stopPropagation();

                SWCElements.modalWindow.hide();
            });

            // Apply changes handler:
            modalWindow.find('.apply-changes').on('mousedown', function(e) {
                var button = $(e.target).hasClass('button') ? $(e.target) : $(e.target).closest('.button');

                if (button.hasClass('disabled')) {
                    return;
                }

                if (options.onApply && $.isFunction(options.onApply)) {
                    options.onApply();
                }

                e.stopPropagation();
            });
        },

        /**
         * Hide modal window
         */
        hide: function() {
            var modalShadow = $('.modalWindowShadow'),
                modalWindow = $('.modalWindow');

            // Hide modal window shadow
            modalShadow.remove();

            // Remove modal window content:
            if (modalWindow.size()) {
                modalWindow.remove();
            }
        }
    },

    progressRuler: {

        /**
         * Current element class name:
         *
         * @param className {String}
         *
         */
        className: 'ruler-container',

        /**
         * Init all custom rulers on the page:
         */
        initAll: function() {
            var self = this;

            $('body')
                .on('swc-ruler:start', '.' + this.className, function(e, params) {
                    e.stopPropagation();
                    self.customChange($(e.target), params);
                });
        },

        /**
         * Animates progress bar during `time` passed as an option
         *
         * @param DOMElement
         * @param options {Object} Example: {length: length, time:time }
         *                                 where length - width alpha in percents
         *                                 time - time of changing in milliseconds
         */
        customChange: function(element, options) {
            var progressBar = element.find('.fill');

            if (!options.time) {
                element.trigger('swc-ruler:finish');
            } else {
                progressBar.animate(
                    { width: '+=' + (options.length || element.width()) },
                    options.time,
                    'linear',
                    function() {
                        element.trigger('swc-ruler:finish');
                    }
                );
            }
        }


    }
};

$(document).ready(function() {
    _.each(SWCElements, function(value, key) {
        if (_.isFunction(SWCElements[key].initAll)) {
            SWCElements[key].initAll();
        }
    });
});
