/**
 * Constructor `{BaseModel}` extends Backbone.Model object literal argument to
 * extend is the prototype for the BaseModel constructor
 *
 * @type {Function|Constructor}
 */
swc.BaseModel = Backbone.Model.extend({

    /**
     * @override on default {'Backbone.model.defaults'} property
     *
     * Default values for a model
     *
     * @param defaults {Object}
     */
    defaults: {},

    /**
     * @override on default {'Backbone.model.url'} property
     *
     * URL for model to be synced / fetched. Can be either string or object formated to default backbone SYNC actions
     *
     * @descirption
     *
     *      url: '/books/' || url: {
     *          'read': '/books/get',
     *          'create': '/books/set',
     *          'update': '/books/edit',
     *          'delete': '/books/delete'
     *      }
     *
     *      Either you can add any method you want to URL object and it will work, if you will call sync() with correct
     *      method. You can extend default behaviour by your custom methods
     *
     *      Example:
     *
     *      ulr: {
     *          'read':     '/get/books',
     *          'replace':  '/set/books/id'
     *      }
     *
     *      Then you call sync('replace'), and it will use correct ulr (/set/books/id)
     *
     * @param url {String || Object}
     */
    url: '',

    /**
     * List of the attributes which will be first time to the model, to check if it will be changed
     *
     * @param originalAttributes {Object}
     *
     * @note Please do not use this Object for your purposes. This is an urgent variable for model.
     */
    originalAttributes: {},

    /**
     * Temporary value for application, to know that this models have to be handled by other way
     *
     * FIXME :: remove this when all models and collection will be moved to the new architecture
     *
     * @description
     *
     *      This will be removed when all models will be moved to new architecture
     *
     * @param isNewArchitecture {Boolean}
     */
    isNewArchitecture: true,

    /**
     * The flag if model will be synced even if it has been changed
     * @param isForceSync {Boolean}
     */
    isForceSync: false,

    /**
     * Parameters for sending '{READ}' request to the server
     *
     * FIXME :: move to separate object which will handle AJAX requests of model:
     *
     * @param ajaxParameters {Object}
     */
    ajaxParameters: {
        'parameters': {}
    },

    /**
     * Parameter for sending '{ANY}' type of request to the server, default ajax options for the model:
     *
     * FIXME :: move to separate object which will handle AJAX requests of model:
     *
     * @param ajaxSettings {Object}
     */
    ajaxSettings: {
        requestType: 'POST',
        requestDataType: 'json',
        contentType: 'application/x-sah-ws-4-call+json'
    },

    /**
     * @override on default {'Backbone.model.initialize()'} method
     *
     * @description
     *
     *      Was overrided because need to set {'this.originalAttributes'}. Why this is needed you can see in following
     *      methods:
     *
     * @memberOf hasChanged()
     * @memberOf changedAttributes()
     */
    initialize: function(attributes, options) {

        attributes = !_.isEmpty(attributes) ? attributes : this.defaults;

        if (_.isEmpty(this.originalAttributes)) {
            this.originalAttributes = attributes;
        }

        this.set(attributes, options);
    },

    /**
     * @override on default {'Backbone.model.sync()'} method
     *
     * Default {'Backbone.sync()'} doesn't support Application requirements.
     *
     * @description
     *
     *      This method was overrided becuase of specific Server (NP) API. It was impossible to use native {'.sync()'}
     *      method with current server API.
     *
     * @param method {String}
     * @param options {Object}
     */
    sync: function(method, options) {
        var self = this,
            url = this.url,
            deferred = new $.Deferred();

        // Define action which has to be processed:
        if (_.isObject(this.url) && !_.isEmpty(this.url[method])) {
            url = this.url[method];
        }

        // This check helps to avoid unnecessary AJAX calls when model is changed:
        if (!this.canChangeModel(method, options)) {
            return deferred.resolve(); // No need to sync model if it was changed by user
        }

        // Send request to the server and fill the model with data
        $.when(self.sendRequest(method, url, options))
            .done(function(response) {

                // This check is added to avoid problems when the model has been changed
                // between AJAX start and AJAX resolve.
                if (self.canChangeModel(method, options)) {
                    self.set(self.parse(response)); // No need to set if the model is changed
                }

                deferred.resolve(response);
            })
            .fail(function(xhr, ajaxOptions, thrownError) {

                if (self.canChangeModel(method, options)) {
                    self.set(self.parse(self.defaults));
                }

                deferred.reject(xhr, ajaxOptions, thrownError);
            });

        return deferred.promise();
    },

    canChangeModel: function(method, options) {
        if (method === "read" && !this.isForceSync) {
            if (!_.isUndefined(options) && options.isListener === true && this.hasChanged()) {
                return false;
            }
        }

        return true;
    },

    /**
     * Send ajax request to the server when model sync is processed:
     *
     * FIXME :: move to separate object which will handle AJAX requests of model:
     *
     * @param method {String}
     * @param url {String}
     * @param options {Object}
     *
     * @returns {*}
     */
    sendRequest: function(method, url, options) {
        var self = this,
            deferred = new $.Deferred(),
            data = method === "read" ? self.ajaxParameters : self.toJSON(options),
            requestType = this.ajaxSettings.requestType,
            headers = {
                'X-Context': $.cookie(getDeviceID() + '/context'),
                'Authorization': 'X-Sah ' + $.cookie(getDeviceID() + '/context')
            };

        // Check if something is missing from mandatory arguments:
        if (_.isEmpty(url) || _.isEmpty(method)) {
            return deferred.resolve();
        }

        // Check if current request is making in background:
        if (!_.isUndefined(options) && options.isListener === true) {
            headers.idle = true;
            headers['X-Sah-Request-Type'] = 'idle';
        }

        // Check if it's needed different request types for get / set requests
        if (_.isObject(this.ajaxSettings.requestType) && !_.isEmpty(this.ajaxSettings.requestType[method])) {
            requestType = this.ajaxSettings.requestType[method];
        }

        // Send request to the server:
        $.ajax({
            url: url,
            type: requestType,
            dataType: self.ajaxSettings.requestDataType,
            contentType: self.ajaxSettings.contentType,
            headers: headers,
            cache: false,
            
            data: $.isPlainObject(data) ? JSON.stringify(data) : data,

            success: function(response) {
                swc.models.Login.checkAccessToDevice(response);
                deferred.resolve(response);
            },

            error: function(xhr, ajaxOptions, thrownError) {
                swc.models.Login.checkAccessToDevice(xhr);
                deferred.reject(xhr, ajaxOptions, thrownError);
            },

            statusCode: {
                403: function() {
                    // TODO :: crete handling of this errors:
                },

                404: function() {
                    if (_.isFunction(this.on404)) {
                        this.on404();
                    }
                },

                500: function() {
                    // TODO :: crete handling of this errors:
                },

                502: function() {
                    if (_.isFunction(this.on502)) {
                        this.on502();
                    }
                }
            }
        });

        return deferred.promise();
    },

    /**
     * @override on default {'Backbone.model.parse()'} method
     *
     * This method is called whenever a model's data is returned by the server, in fetch, and save. The function is
     * passed the raw response object, and should return the attributes hash to be set on the model
     *
     * @param response {Object}
     * @param options {Object}
     *
     * @returns {Object}
     */
    parse: function(response, options) {
        var json = response;

        // Convert json to suitable format if method for this exists
        if (_.isFunction(this.apiToJSON)) {
            json = this.apiToJSON(json);
        }

        // Save original json in order to check for model changes
        this.originalAttributes = json;

        return json;
    },

    /**
     * @override on default {'Backbone.model.toJSON()'} method
     *
     * Transform JSON form model to valid server JSON, to be saved
     *
     * @param json {Object}
     *
     * @returns {Object}
     */
    toJSON: function() {
        return Backbone.Model.prototype.toJSON.call(this);
    },

    /**
     * @override on default {'Backbone.model.hasChanged()'} method
     *
     * Default method doesn't fit current application requirements
     *
     * @description
     *
     *      var model = new Backbone.model({ a: 10 });
     *
     *          model.set('a', 15);
     *          model.set('a', 10);
     *
     *          model.hasChanged() -> true
     *
     *      Current implementation does not allow us to see if model was really changed, so here is the override
     *
     *      var model = new swc.BaseModel({ a: 10 });
     *
     *          model.set('a', 15);
     *          model.set('a', 10);
     *
     *          model.hasChanged() -> false
     *
     * @param attributes {String || Array of Strings} (saving native Backbone.hasChanged(attrs) interface)
     *
     * @returns {Boolean}
     */
    hasChanged: function(attributes) {
        var changedAttributes = this.changedAttributes(),
            hasChanged = !_.isEmpty(changedAttributes);

        if (!_.isUndefined(attributes) && !_.isEmpty(attributes)) {
            hasChanged = false;

            attributes = !_.isArray(attributes) ? [ attributes ] : attributes;

            _.each(attributes, function(key) {
                if (!_.isUndefined(changedAttributes[key])) {
                    hasChanged = true;
                }
            });
        }
        
        return hasChanged;
    },

    /**
     * @override on default {'Backbone.model.changedAttributes()'} method
     *
     * Default method doesn't fit current application requirements
     *
     * @description
     *
     *      var model = new Backbone.model({ a: 10 });
     *
     *          model.set('a', 15);
     *          model.set('a', 10);
     *
     *          model.changedAttributes('a') -> true
     *
     *      Current implementation does not allow us to see if model was really changed, so here is the override
     *
     *      var model = new swc.BaseModel({ a: 10 });
     *
     *          model.set('a', 15);
     *          model.set('a', 10);
     *
     *          model.changedAttributes('a') || model.changedAttributes(['a'])  -> false
     *
     * @param attributes {String || Array of Strings} (saving native Backbone.changedAttributes(attrs) interface)
     *
     * @returns {Array of Objects}
     */
    changedAttributes: function(attributes) {
        var self = this,
            attrs = {},
            attrsDiff = _.keys(this.attributes);

        if (!_.isUndefined(attributes) && !_.isEmpty(attributes)) {
            attrsDiff = !_.isArray(attributes) ? [ attributes ] : attributes;
        }

        _.each(attrsDiff, function(attribute, key) {
            var mapsEqual = _.isEqual(
                _.pick(self.attributes, attribute),
                _.pick(self.originalAttributes, attribute)
            );

            if (!mapsEqual) {
                attrs[attribute] = self.get(attribute);
            }
        });

        return attrs;
    }
});
