/**
 * Constructor `{BaseCollection}` extends Backbone.Collection object literal argument to
 * extend is the prototype for the BaseCollection constructor
 *
 * @type {Object}
 */
swc.BaseCollection = Backbone.Collection.extend({

    /**
     * @override on default {'Backbone.Collection.defaults'} property
     *
     * Default values for collection
     *
     * @param defaults {Array of Objects}
     */
    defaults: [],

    /**
     * @override on default {'Backbone.Collection.url'} property
     *
     * URL for collection to be synced / fetched. Can be either string or object formated to default backbone SYNC actions
     *
     * @description
     *
     * url: '/books/' || url: {
     *      'read': '/books/get',
     *      'create': '/books/set',
     *      'update': '/books/edit',
     *      'delete': '/books/delete'
     * }
     *
     * @param url {String || Object}
     */
    url: '',

    /**
     * @override on default {'Backbone.Collection.model'} property
     *
     * Model id assigned to the collection. Each collection model will be created using this constructor
     *
     * @description
     *
     *      When {'.initialize()'} method will be called, collection will check if current model id is a valid
     *      constructor and exists in swc.models.* scope. If everything is OK -> this constructor will be used
     *      for any model assigned to this collection.
     *
     * @param model {String || Function -> Constructor}
     */
    model: '',

    /**
     * Temporary value for application, to know that this collection have to be handled by other way
     *
     * FIXME :: remove this when all models and collection will be moved to the new architecture
     *
     * @description
     *
     * This will be removed when all collections will be moved to new architecture
     *
     * @param isNewArchitecture {Boolean}
     */
    isNewArchitecture: true,

    /**
     * The flag if collection will be synced even if it has been changed
     * @param isForceSync {Boolean}
     */
    isForceSync: false,

    /**
     * Parameters for sending '{READ}' request to the server
     *
     * @param ajaxParameters {Object}
     */
    ajaxParameters: {
        'parameters': {}
    },

    /**
     * Parameter for sending '{ANY}' type of request to the server, default ajax options for the collection:
     *
     * @param ajaxSettings {Object}
     */
    ajaxSettings: {
        requestType: 'POST',
        requestDataType: 'json',
        contentType: 'application/x-sah-ws-4-call+json'
    },

    /**
     * @override on default {'Backbone.Collection.initialize()'} method
     *
     * @description
     *
     *      Initialize method sets models which are passed to the constructor. If this models are empty,
     *      collection will be filled in with {'this.defaults'}. If {'this.defaults'} will be empty, collection
     *      during initialization will also be empty.
     *
     *      Also in this method collection decides what model constructor will be used for it's models.
     *
     *      if {'this.model'} is passed as a function, consistancy of Backbone.Collection.model will be saved. If
     *                        Function is not a constructor, error will be thrown.
     *
     *      if {'this.model'} is passed as a string, initialize method will go through models constructors scope and look
     *                        for constructor with the same id. If constructor will be found, it will be assigned as a
     *                        collection constructor, else error will be thrown
     *
     *      if {'this.model'} is passed as it is, (is not overrided by extending collection) swc.BaseModel will be set as
     *                        collection models constructor
     *
     */
    initialize: function(models, options) {

        // Define what will be set as collection models:
        models = !_.isEmpty(models) ? models : this.defaults;

        // Saving posibility of native Backbone.Collection.model property to be a constructor itself
        if (!_.isUndefined(this.model) && !_.isEmpty(this.model)) {
            if (_.isString(this.model)) {

                // TODO :: when swc.constructors will be separated to swc.constructors.models -> change this

                if (!_.isFunction(swc.constructors[this.model])) {
                    throw new Error("swc.constructors." + this.model + " is undefined or not a constructor");
                } else {
                    this.model = swc.constructors[this.model];
                }
            } else {
                if (!_.isFunction(this.model)) {
                    throw new Error(this.model + " is undefined or not a constructor");
                }
            }
        } else {
            this.model = swc.BaseModel;
        }

        this.reset(models, _.extend({ silent: true }, options));
    },

    /**
     * @override on default {'Backbone.Collection.sync()'} method
     *
     * Syncs collection with a server
     *
     * @param method {String} -> one of backbone supported methods (Read, Update, Create, Delete)
     * @param options {Object} (optional)
     */
    sync: function(method, options) {
        var self = this,
            url = this.url,
            deferred = new $.Deferred();

        // Define action which has to be processed:
        if (_.isObject(this.url) && !_.isEmpty(this.url[method])) {
            url = this.url[method];
        }

        // Reset collection on read method if no changes were done and sync request is not from listener:
        if (!this.canChangeCollection()) {
            return deferred.resolve(); // No need to sync collection if it was changed by user
        } else {
            this.reset([], _.extend({ silent: true }, options));
        }

        // Send request to the server and fill the model with data
        $.when(self.sendRequest(method, url, options))
            .done(function(response) {

                if (self.canChangeCollection()) {
                    self.set(self.parse(response));
                    self.trigger('change');
                }

                deferred.resolve();
            })
            .fail(function(xhr, ajaxOptions, thrownError) {

                if (self.canChangeCollection()) {
                    self.set(self.defaults);
                }

                deferred.reject(xhr, ajaxOptions, thrownError);
            });

        return deferred.promise();
    },

    canChangeCollection: function(method, options) {
        if (method === "read" && !this.isForceSync) {
            if (!_.isUndefined(options) && options.isListener === true && this.hasChanged()) {
                return false;
            }
        }

        return true;
    },
    /**
     * Send ajax request to the server when collection sync is processed:
     *
     * @param method {String}
     * @param url {String}
     * @param options {Object}
     *
     * @returns {*}
     */
    sendRequest: function(method, url, options) {
        var self = this,
            deferred = new $.Deferred(),
            data = method === "read" ? self.ajaxParameters : self.toJSON(options),
            requestType = this.ajaxSettings.requestType,
            timeout = !_.isUndefined(options) && !_.isUndefined(options.timeout) ? options.timeout : 20000,
            headers = {
                'X-Context': $.cookie(getDeviceID() + '/context'),
                'Authorization': 'X-Sah ' + $.cookie(getDeviceID() + '/context')
            };

        // Check if something is missing from mandatory arguments:
        if (_.isEmpty(url) || _.isEmpty(method)) {
            return deferred.resolve();
        }

        // Check if current request is making in background:
        if (!_.isUndefined(options) && !_.isUndefined(options.isListener)) {
            headers.idle = true;
            headers['X-Sah-Request-Type'] = 'idle';
        }

        // Check if it's needed different request types for get / set requests
        if (_.isObject(this.ajaxSettings.requestType) && !_.isEmpty(this.ajaxSettings.requestType[method])) {
            requestType = this.ajaxSettings.requestType[method];
        }

        // Send request to the server:
        $.ajax({
            url: url,
            type: requestType,
            dataType: self.ajaxSettings.requestDataType,
            contentType: self.ajaxSettings.contentType,
            headers: headers,
            timeout: timeout,

            cache: false,

            data: $.isPlainObject(data) ? JSON.stringify(data) : data,

            success: function(response) {
                swc.models.Login.checkAccessToDevice(response);
                deferred.resolve(response);
            },

            error: function(xhr, ajaxOptions, thrownError) {
                swc.models.Login.checkAccessToDevice(xhr);
                deferred.reject(xhr, ajaxOptions, thrownError);
            },

            statusCode: {

                403: function() {
                    // TODO :: crete handling of this errors:
                },

                404: function() {
                    if (_.isFunction(this.on404)) {
                        this.on404();
                    }
                },

                500: function() {
                    // TODO :: crete handling of this errors:
                },

                502: function() {
                    if (_.isFunction(this.on502)) {
                        this.on502();
                    }
                }

            }
        });

        return deferred.promise();
    },

    /**
     * @override on default {'Backbone.Collection.parse()'} method
     *
     * This method is called whenever a collection's data is returned by the server, in fetch, and save. The function is
     * passed the raw response object, and should return the attributes hash to be set on the collection
     *
     * @param response {Object}
     * @param options {Object} (optional)
     *
     * @returns {Object}
     */
    parse: function(response, options) {
        var json = response;

        if (_.isFunction(this.apiToJSON)) {
            json = this.apiToJSON(json);
        }

        return json;
    },

    /**
     * @override on default {'Backbone.Collection.toJSON()'} method
     *
     * Transform collection to valid JSON. Goes through each model and call's it {'toJSON()'} method
     *
     * @param options {Object}
     *
     *      options = { showDeleted: true } -> includes deleted models to list of models
     *      options = {} -> includes only not deleted models
     *
     * @returns {Array of Objects}
     */
    toJSON: function(options) {
        var json = [];

        _.each(this.models, function(model, key) {
            if (_.isUndefined(model.get('isDeletedByCollection'))) {
                json.push(Backbone.Model.prototype.toJSON.apply(model));
            } else {
                if (!_.isUndefined(options) && options.showDeleted === true) {
                    json.push(Backbone.Model.prototype.toJSON.apply(model));
                }
            }
        });

        return json;
    },

    /**
     * @override on default {'Backbone.Collection.remove()'} method
     *
     * We need to know what models were removed from the collections, to sync them with server
     *
     * @param models {Array of Objects || Objects}
     * @param options {Object}
     *
     * @description
     *
     *      options = { permanent: true } will use native Backbone.Collection.remove();
     *
     *      When options are not passed, there works an override on default Backbone.Collection.Remove() method
     *      which set Model custom parameter {'isDeletedByCollection'} to true, which let's collection know that
     *      this model was deleted from the collection.
     *
     *      This case will be needed to get list of deleted models, when collection will be saving. Because NP doesn't
     *      support Full REST api, so it will be needed to go through all {isDeletedByCollection} models and sync them
     *      with server
     */
    remove: function(models, options) {
        var modelsDiff = [];

        if (!_.isUndefined(options)) {
            if (!_.isUndefined(options.permanent) && options.permanent === true) {
                Backbone.Collection.prototype.remove.apply(this, arguments);
            }
        } else {

            if (!_.isUndefined(models) && !_.isEmpty(models)) {
                modelsDiff = !_.isArray(models) ? [ models ] : models;
            }

            _.each(modelsDiff, function(model, key) {
                model.set('isDeletedByCollection', true);
            });
        }
    },

    /**
     * Creating similar to Backbone.Model.hasChanged() method, which will say if collection has changed:
     *
     * @description:
     *
     *      Goes through each model in the collections and call it's method {'.hasChanged()'}
     *      If one the models will return {'true'} that will mean that collection has changed
     *
     * // TODO :: improvement allow pass list of model as arguments.
     *
     * @returns {Boolean}
     */
    hasChanged: function() {
        var hasChanged = false;

        _.each(this.models, function(model) {
            if (model.hasChanged()) {
                hasChanged = true;
            }
        });
        
        return hasChanged;
    },

    /**
     * Get list of changed models. Including edited / deleted models:
     *
     * @param options {Object}
     *
     * @description
     *
     *      options = { onlyEdited: true } will return only changed models (excluding deleted);
     *      options = { onlyDeleted: true } will return only deleted models (excluding changed);
     *      options = {} || undefined will return all changed and deleted models
     *
     * @returns {Array of Objects}
     */
    changedModels: function(options) {
        var models = [];

        _.each(this.models, function(model, key) {

            // Selection for only edited or deleted models:
            if (!_.isUndefined(options) && !_.isEmpty(options)) {

                // Selection for only changed models:
                if (!_.isUndefined(options.onlyEdited) && options.onlyEdited === true) {
                    if (model.hasChanged() && _.isUndefined(model.get('isDeletedByCollection'))) {
                        models.push(model);
                    }
                }

                // Selection for only deleted models:
                if (!_.isUndefined(options.onlyDeleted) && options.onlyDeleted === true) {
                    if (!_.isUndefined(model.get('isDeletedByCollection')) &&
                        model.get('isDeletedByCollection') === true) {
                        models.push(model);
                    }
                }
            } else {
                if (model.hasChanged()) {
                    models.push(model);
                }
            }
        });

        return models;
    }

});

