/**
 * Base BackBone view for displaying pages
 *
 * @type {*}
 *
 */
swc.base.PageView = Backbone.View.extend({

    /**
     * Selected view tag name
     *
     * @param tagName {String}
     *
     */
    tagName: 'div',

    /**
     * Selected view class name
     *
     * @param className {String}
     *
     */
    className: '',

    /**
     * List of allowed modes for current page:
     *
     * @param allowedMods {Array}
     *
     */
    allowedMods: ["standard", "expert"],

    /**
     * List of models which have to be preloaded before page is rendered
     *
     * @param models {Object} :: object :: callback function
     *
     */
    models: [],

    /**
     * List of methods for each validateable component to be called:
     * @param validation {Object || Map}
     *
     * @description can be used in two ways - passing arguments as <string> : <string>, or
     *              passing arguments as <string> : <array>
     *
     * @example
     *              validation: {
     *                  'input[name="somename"]': 'IPModel.validateIPAddress'
     *                  'input[name="othername"]': [
     *                                                 'IPModel.validateDHCP',
     *                                                 'IPModel.validateDMZ'
     *                                             ]
     *              }
     */
    validation: [],

    /**
     * Template data storage which will be passed to the template when models are loaded:
     */
    templateData: {},

    /**
     * Default interval for listeners - 10 seconds
     */
    listenerInterval: 10,

    /**
     * Flag which enable / disable pages:
     */
    listenerEnabled: false,

    /**
     * List of pages where listeners will work for current view
     */
    listenerPages: [],

    /**
     * List of default events for each page - will be extended by each view
     * @param events {Object}
     */
    events: {
        // special event to be called from the code:
        'validate .validatable': 'pageValidation',

        // When user makes a change on the field and leaves focus from it:
        'change input.validatable': 'pageValidation',

        // When user makes a change on checkbox:
        'swc-checkbox:change .swc-checkbox.validatable': 'pageValidation',

        // When user makes a change on checkbox extended:
        'swc-checkbox-extened:change .swc-checkbox.validatable': 'pageValidation',

        // When user makes a change on dropdown:
        'swc-dropdown:change .swc-dropdown.validatable': 'pageValidation',

        // When user makes a change on radiobutton:
        'swc-radio-buttons:change .swc-radio-buttons.validatable': 'pageValidation',

        // When user makes a change in the field but focus is still in
        'keyup input.swc-input': 'setButtonsState',

        // Button cancel changes click handler
        'click .button.cancel-changes:not(.disabled)': 'cancelChanges',

        // Button save changes click handler
        'mousedown .button.save-changes:not(.disabled)': 'saveChanges'
    },

    /**
     * Flag to identify if page content is disabled by loading backdrop
     * @param isPageEnabled {Boolean}
     */
    isPageEnabled: true,

    /**
     * Flag to identify if page tabs are disabled by loading backdrop
     * @param isTabsEnabled {Boolean}
     */
    isTabsEnabled: true,

    /**
     * Initialise view
     */
    initialize: function() {
        var pagesArray = Backbone.history.fragment.split("/");

        // Define page template and locale ID:
        if (pagesArray.length === 1) {
            if (!pagesArray[0].length) {
                this.pageTemplateID = 'overview';
            } else {
                this.pageTemplateID = pagesArray[0];
            }
        } else {
            this.pageTemplateID = pagesArray[0] + ':' + pagesArray[1];
        }

        // Prepare template for current page:
        this.template = $.template("pageContent", swc.Templates.get(this.pageTemplateID).get('content'));

        // Extend global events:
        this.events = _.extend({}, swc.base.PageView.prototype.events, this.events);
    },

    /**
     * Check if current view has unsaved changes on it, e.g. has "dirty model".
     * So far it check if there us enabled "Save" button.
     * 
     * @return {boolean}
     */
    hasChanged: function() {
        return this.$el.find(".client-buttons a[class*='apply']:not(.disabled), " +
            ".client-buttons a[class*='save']:not(.disabled)").size() > 0;
    },

    /**
     * Load all models which are included to current page
     *
     * @param callback {Function}
     *
     */
    loadModels: function(fromListener) {
        var deferred = new $.Deferred(),
            toDo = [],
            modelsToLoad = !_.isUndefined(this.models) ? _.values(this.models) : [];

        if (this.hasTabs) {
            modelsToLoad = _.union(modelsToLoad, !_.isUndefined(this.tabView.models) ? _.values(this.tabView.models) : []);
        }

        if (_.isEmpty(modelsToLoad)) {
            return deferred.resolve();
        } else {
            _.each(modelsToLoad, function(model, key) {
                if (!swc.models[model]) {
                    swc.models[model] = new swc.constructors[model]();
                }

                if (swc.models[model].isNewArchitecture) {
                    toDo.push(
                        swc.models[model].sync('read', { isListener: fromListener === true ? true : false })
                    );
                } else {
                    toDo.push(
                        swc.models[model].sync(fromListener === true ? true : false)
                    );
                }
            });

            $.when.apply(this, toDo)
                .done(function() {
                    deferred.resolve();
                })
                .fail(function() {
                    // TODO :: navigate user to coresponding sticky page
                    deferred.resolve();
                });
        }

        return deferred.promise();
    },

    /**
     * To be overriden if necessary.
     * this method is called after the models are synced
     * @returns a promise object.
     * If the promise is resolved then the render method continues executing.
     * If the promise is rejected then the render method stops executing.
     * Useful when preconditions may require redirecting to another page.
     */
    preRender: function () {
        // default behavior is - to render on
        return (new $.Deferred()).resolve();
    },

    /**
     * Render Selected Page
     */
    render: function(action) {
        var self = this,
            promise;

        // Check if render was called from child view:
        if (self.parentView) {
            self.parentView.render();
            return;
        }

        // Check if current application mode allows user to visit current page
        self.checkPermissions();

        // Show loading window:
        self.showPageLoading('Loading page data..');

        // Check if current page has tab and init it:
        if (self.hasTabs) {
            self.initTabView();
        }

        // Reset values to its defaults
        this.isPageEnabled = true;
        this.isTabsEnabled = true;

        // Load page models:
        $.when(self.loadModels(false))
            .done(function() {

                // Check for preconditions (if any)
                // (actually, it should be called in initialize method, but in our case we have
                // models loading in render!!! method, so we have to make a way...)
                // If there are tabs then we have to call tabView method instead:
                if (self.hasTabs) {
                    promise = self.tabView.preRender();
                } else {
                    promise = self.preRender();
                }

                $.when(promise)
                    .done(function() {

                        // Set template data:
                        self.setTemplateData();

                        // Check if page has tabs:
                        if (self.hasTabs) {
                            // Set template data to tab view:
                            self.tabView.setTemplateData();

                            // Get parent templates content:
                            self.getTemplateContent();

                            // Get child template content:
                            self.tabView.getTemplateContent();

                            // Render parent template
                            self.templateContent.find('.tabs-content').html(self.tabView.templateContent);

                            // Display page basing on content:
                            self.displayPage();

                            // Render complete method trigger:
                            self.renderComplete();
                            self.tabView.renderComplete();
                            self.tabView.startListener();
                            self.tabView.trigger('render:finished');
                        } else {
                            self.getTemplateContent();
                            self.displayPage();
                            self.renderComplete();
                        }
                        self.trigger('render:finished');
                        self.startListener();
                    });
            });
        return this;
    },

    /**
     * This method will be overriden in each view. In this method can be set <templateData> variable
     */
    setTemplateData: function() {},

    /**
     * Get page template content:
     * @returns {*}
     */
    getTemplateContent: function() {
        var self = this;

        this.templateContent = this.$el.html($.tmpl(self.template, {
            data: self.templateData,
            localeStrings: swc.models.Locale.getLocaleStrings(),
            localeString: getTranslationStringsjQuery,
            staticPagesLinks: swc.models.Locale.getStaticPagesLinks(),
            formatDate: swc.models.Locale.formatDate
        }));
    },

    /**
     * Insert ready template to application content:
     */
    displayPage: function() {
        var self = this,
            area = $('#current-page');

        // Check if application is already rendered:
        if (!area.size()) {
            swc.views.Application.renderApplicationArea();
        }

        // Render template content to application area:
        swc.views.Application.contentArea.html(
            self.templateContent
        );

        // Set menu link state:
        swc.views.Application.setMenuActiveLink();

        // Delegate events:
        this.delegateEvents();

        // Delegate events on tab view:
        if (this.hasTabs) {
            this.tabView.delegateEvents();
        }

        // Hide loading window dialog:
        this.stopPageLoading();
    },

    /**
     * Method which will be called when main part of a page will be
     * rendered and all necessary data will be loaded
     *
     * @description Here you can place any observers on existing elements, fill in drop downs, add devices etc.
     */
    renderComplete: function() {},

    /**
     * Render Page Selected Tab:
     *
     * @description Tab constructor id is building in following way:
     *
     * (currentPageName.capitalise() + currentSubPageName.capitalise() + currentTabName.capitalise()).replace(/-/g,'') + 'View'
     *
     * For example for storage/settings/cloud-backup it will be "StorageSettingsCloudbackupView"
     *
     */
    initTabView: function(action) {
        var pagesArray = Backbone.history.fragment.split("/"),
            TabConstructorName = _.map(pagesArray, function(n) {
                return n.capitalize();
            }).concat("View").join("").replace(/-/g,''),
            TabViewName = TabConstructorName.deCapitalize();

        if (!swc.views[TabViewName]) {
            swc.views[TabViewName] = new swc.constructors[TabConstructorName]();
            swc.views[TabViewName].parentView = this;
        }

        this.tabView = swc.views[TabViewName];
        // Check if current application mode allows user to visit current page
        this.tabView.checkPermissions();
    },

    /**
     * Show page Saving / Loading dialog
     *
     * @description: when each view method render() will be called - the loading dialog will appear
     *               this method is extended by each view in application, so it can be called from any
     *               place of the application by invoking this.showPageLoading(argumet : string);
     *
     * @param action {String}
     */
    showPageLoading: function(message) {
        var self = this,
            template = $.template('loading', swc.Templates.get('application:loading-window').get('content')),
            dialog = $('.loading-window'),
            id = Math.random();

        // Close all popover
        SWCElements.popovers.closeAll();

        // Do nothing if loading window exists
        if (dialog.size()) {
            return;
        }

        // Append loading window to body:
        $('#components').append(
            $.tmpl(template, {
                id: id,
                message: getTranslationStrings(message, true),
                localeStrings: swc.models.Locale.getLocaleStrings('application'),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            })
        );

        setTimeout(function() {
            var loadingWindow = $('.loading-window[data-id="' + id + '"]');

            if (loadingWindow.size()) {
                self.showPageLoadingFail();
            }
        }, 30000);
    },

    /**
     * IF Loading of page data failed - show user a modal window:
     */
    showPageLoadingFail: function() {
        this.stopPageLoading();

        SWCElements.modalWindow.show({
            templateID: 'application:loading-window-fail',
            templateData: {},
            className: 'loading-window-fail',

            onApply: function() {
                window.location.reload();
            }
        });
    },

    /**
     * Removes Loading / Saving dialog
     */
    stopPageLoading: function() {
        var dialog = $('#components .loading-window');

        if (dialog.size()) {
            dialog.remove();
        }
    },

    /**
     * Disable / Enable tabs on current page
     * 
     * @param isEnabled {Boolean}
     * @param options {Object} -> { className: <string>, isTabOverlay: <boolean>, ... }
     * 
     * @return void
     */
    setTabsState: function(isEnabled, options) {
        var self = this,
            pageTabs = this.$el.find('li.tab');

        if (!_.isEmpty(pageTabs)) {
            if ((this.isTabsEnabled === true && isEnabled === false) || (this.isTabsEnabled === false && isEnabled === true)) {
                _.each(pageTabs, function(tabElement, key) {
                    var $tab = $(tabElement),
                        $tabLink = $tab.find('a');

                    if (isEnabled === false) {
                        self.createDisabledOverlay($tabLink, _.extend({
                            className: $tab.hasClass('active') ? '' : 'transparent',
                            isTabOverlay: true
                        }, options));
                    } else {
                        self.removeDisabledOverlay(_.extend({
                            removeTabsOverlay: true
                        }, options));
                    }

                    $tab.toggleClass("disabled", isEnabled === false);
                });
            }

            this.isTabsEnabled = isEnabled;
        }
    },

    /**
     * Covers current page's content with an semi-transparent overlay element.
     * Thus Disables/Enables the page.
     * 
     * @param isEnabled {Boolean} Defines either page should be enabled or disabled
     * @param options {Object} Possible options:
     *                         - isPageOverlay {boolean}:     Whether or not hide page under semi-transparent overlay
     *                         - disabledBlockClass {String}: Name of CSS class with used to mark a block which will be
     *                                                        covered with overlay, instead of covering the whole page.
     *                                                        
     * @retun void
     */
    setPageState: function(isEnabled, options) {
        var self = this,
            $blockToDisable = $('.tabs-content').size() ? $('.tabs-content') : $('.base-content'),
            $disabledOverlay = $('#current-page').find('.showDisabledBackDrop.page-overlay');
        
        // Sometimes we should not add overlay to the whole page, but rather to some region on the page
        // thus we can mark such region with a special css class name passed in 'disabledBlockClass' option
        if (options && !_.isUndefined(options['disabledBlockClass'])) {
            $blockToDisable = $('#current-page').find("." + options['disabledBlockClass']);
        }

        if (isEnabled === false) {
            if (this.isPageEnabled || !$disabledOverlay.size()) {
                self.createDisabledOverlay($blockToDisable, _.extend({
                    isPageOverlay: true
                }, options));
            }
        } else {
            self.removeDisabledOverlay(_.extend({
                removePageOverlay: true
            }, options));
        }

        this.isPageEnabled = isEnabled;
    },

    /**
     * Creates an overlay element with size based on size of element to be faded with the overlay.
     * 
     * @param options {Object} Possible values:
     *                         - className {String} Name of additional css class to be added to the overlay element's.
     *                                              Could be useful if one need to cusomize the presentation of overlay.
     *                         - isTabOverlay {boolean}  Whether overlay should be placed over tab.
     *                                                   Special 'tab-overlay' css class will be added if True.
     *                         - isPageOverlay {boolean} Whether overlay should should be placed over tab.
     *                                                   Special 'page-overlay' css class will be added if True.
     * 
     * @protected Only for internal use!
     * 
     * @return void
     */
    createDisabledOverlay: function($element, options) {
        var overlay = $('<div class="showDisabledBackDrop no-highlighting"></div>');

        // Set position styles
        overlay.css({
            'top': $element.offset().top + 'px',
            'left': $element.offset().left + 'px',
            'width': $element.innerWidth(),
            'height': $element.innerHeight(),
            'display': 'block'
        });

        // Set custom properties:
        if (!_.isEmpty(options)) {
            if (!_.isEmpty(options.className)) {
                overlay.addClass(options.className);
            }

            if (options.isTabOverlay) {
                overlay.addClass("tab-overlay");
            }

            if (options.isPageOverlay) {
                overlay.addClass("page-overlay");
            }
        }

        overlay.appendTo($('#current-page'));
    },

    /**
     * Remove all disabled overlays from the page (based on arguments)
     * @param options
     */
    removeDisabledOverlay: function(options) {
        var overlays = $('#current-page').find('.showDisabledBackDrop');

        _.each(overlays, function(overlay, key) {
            var $overlay = $(overlay);

            if (!_.isUndefined(options)) {
                if (options.removeTabsOverlay === true) {
                    if ($overlay.hasClass("tab-overlay")) {
                        $overlay.remove();
                    }
                }

                if (options.removePageOverlay === true) {
                    if (!_.isUndefined(options['disabledBlockClass']) &&
                        $overlay.hasClass(options['disabledBlockClass'])) {
                        $overlay.remove();
                    } else if ($overlay.hasClass("page-overlay")) {
                        $overlay.remove();
                    }
                }
            } else {
                $overlay.remove();
            }
        });
    },

    /**
     * Display corresponding message, when the page is disabled
     * @param options {Object} -> { className: <string>, ... }
     */
    showDisabledMessage: function(options) {
        var element,
            $disabledMessage = $('<div class="showDisabledBackDropMessage no-highlighting"></div>'),
            $disabledMessageContainer = $('.tabs-content').size() ? $('.tabs-content') : $('.base-content'),
            disabledMessagePosition = {};

        // Check if options container is passed:
        if (!_.isUndefined(options) && !_.isUndefined(options.container)) {
            element = this.$el.find(options.container);
        }

        // Check if element exists on the content container:
        if (_.isUndefined(element) || !element.size()) {
            return;
        }

        // Hide previosly shown message:
        this.hideDisabledMessage(options);

        // Copy html from message in container to disabled message:
        $disabledMessage.html(element.html());

        // Append element to message container:
        $disabledMessage.appendTo($('#current-page'));

        // Set correct styles to the disabled message:
        $disabledMessage.css({
            top: $disabledMessageContainer.offset().top + ($disabledMessageContainer.outerHeight() / 2),
            left: $disabledMessageContainer.offset().left + ($disabledMessageContainer.outerWidth() / 2),
            marginTop: -($disabledMessage.outerHeight() / 2 + 50) + 'px',
            marginLeft: -($disabledMessage.outerWidth() / 2) + 'px'
        });

        // Check if additional options passed:
        if (!_.isUndefined(options.className)) {
            $disabledMessage.addClass(options.className);
        }

        // Display message after everything is set:
        $disabledMessage.show();
    },

    /**
     * Hide corresponding message, when the page is disabled
     * @param options {Object}
     */
    hideDisabledMessage: function(options) {
        $('#current-page').find('.showDisabledBackDropMessage').remove();
    },

    /**
     * Save changes button handler:
     */
    saveChanges: function() {
        var self = this,
            validation = this.pageValidation(null, true);

        // Check if page validation errors exists on the page:
        $.when(validation)
            .fail(function () {
                // validation is not passed - do nothing
            })
            .done(function () {
                // Check if necessary methods existing:
                if ($.isFunction(self.save)) {
                    // Show page loading status:
                    self.showPageLoading("Saving page data..");

                    // Wait till page saving is completed
                    $.when(self.save())
                        .done(function(message) {
                            self.setRenderFinishedHook("success", message);
                        })
                        .fail(function(message) {
                            self.setRenderFinishedHook("error", message);
                        })
                        .always(function () {
                            // Define what view has to be re-rendered:
                            if (self.parentView) {
                                self.parentView.render();
                            } else {
                                self.render();
                            }
                        });
                }
            });
    },

    setRenderFinishedHook: function (type, message) {
        var self = this;

        this.on('render:finished', function() {
            if (self.tabView) {
                self.tabView.showSaveSuccess(type, message);
            } else {
                self.showSaveSuccess(type, message);
            }
            self.off('render:finished');
        });
    },

    /**
     * Cancel changes button handler:
     */
    cancelChanges: function(e) {
        this.$('.validation-message').hide();
        this.$('.save-validation-errors').hide();
        this.$('.validation-error').removeClass('validation-error');

        if (_.isFunction(this.onCancel)) {
            this.onCancel.apply(this, arguments);
        }
        if (this.parentView) {
            if (_.isFunction(this.parentView.onCancel)) {
                this.parentView.onCancel.apply(this, arguments);
            }
            this.parentView.render();
        } else {
            this.render();
        }
        // This prevents passing Event if "Cancel" pressed in Child (TabView).
        // Otherwise, if "onCancel" defined in Parent View, it will be called twice
        if (!_.isUndefined(e)) {
            e.stopPropagation();
        }
    },

    /**
     * Show successfull page save result:
     */
    showSaveSuccess: function(type, message) {
        var container;

        container = this.$('.buttons-container-message .save-' + (type || 'success'));
        container.show();

        if (!_.isUndefined(message)) {
            container.find('.error-message').hide();
            container.find('.error-message.' + message + '-message').show();
        }
    },

    /**
     * Set buttons state on the page:
     * @param e {Object}
     * @param toDisable {Boolean}
     */
    setButtonsState: function(e, toDisable) {
        var container = this.$el.find('.buttons-container-message'),
            buttonSave = container.find('.button.save-changes'),
            buttonCancel = container.find('.button.cancel-changes'),
            allValuesDefault = this.pageCheckDefaultValues();

        // Prevent method from the double call. TabView has the same method and some times it make collisions
        if (this.hasTabs) {
            return;
        }

        if (container.size()) {
            if (toDisable || allValuesDefault) {
                buttonSave.addClass('disabled');
                buttonCancel.addClass('disabled');
            } else {
                buttonSave.removeClass('disabled');
                buttonCancel.removeClass('disabled');
            }
        }
    },

    /**
     * Get all elements on the page, which can be changed by user:
     *
     * @return Array
     */
    getElements: function() {
        var possibleElements = [
                '.swc-input', '.swc-checkbox', '.swc-extended', '.swc-radio-buttons', '.swc-dropdown'
            ];

        return this.$el.find(possibleElements.join(':not(.skip-validation),') + ':not(.skip-validation)');
    },

    /**
     * Check if after changes on the page all values are set as they were
     * before the page has loaded
     *
     * @returns {boolean}
     */
    pageCheckDefaultValues: function() {
        var isDefault = true;

        $.each(this.getElements(), function(key, element) {
            var dataParameters = getParameter($(element)),
                dataDefaultValue = $(element).data('default-value');

            if (_.isBoolean(dataDefaultValue)) {
                dataDefaultValue = dataDefaultValue.toString();
            }

            if (!_.isUndefined(dataDefaultValue)) {
                if (!_.isUndefined(dataParameters.parameterValue) && dataParameters.parameterValue.toString() !== dataDefaultValue.toString()) {
                    isDefault = false;
                }
            } else {
                if (!_.isUndefined(dataParameters.parameterValue)) {
                    isDefault = false;
                }
            }
        });

        return isDefault;
    },

    /**
     * Check if current page can be viewed in selected mode
     */
    checkPermissions: function(notFromView) {
        var currentMode = swc.settings.application.get('expertMode') ? 'expert' : 'standard',
            fixedRoute;

        if (localStorage.userMode) {
            currentMode = localStorage.userMode;
        }

        if (!notFromView) {
            swc.views.currentView = this;
        }

        if ($.inArray(currentMode, swc.views.currentView.allowedMods) === -1) {
            if (!$('ul.page-tabs').size()) {
                fixedRoute = 'overview';
            } else {
                fixedRoute = ($(swc.views.currentView.el).closest('ul.page-tabs')
                    .find('li:first a').attr('href').replace('#',''));
            }
            
            swc.router.navigate(fixedRoute, { trigger: false, skipUnsavedChanges: true });
        }
    },
    
    /**
     * Page validation called when Apply button pressed, or editable field left focus
     * Implements FIFO buffer for validation requests that they may start and finish 
     * in chronological order.
     * Normall call: stacks the parameters in buffer and then starts the processing
     * (if it is not started yet) of the first item from buffer.
     * Recursive call: processing next item from buffer. Is called each time when 
     * a validation process is finished and the buffer is not empty.
     * @params e,validateAll - the parameters for validation call.
     * @param processNext usually undefined. Is set to true in recursive call only.
     * @return Deferred object
     */
    pageValidation: function(e, validateAll, processNext) {
        var self = this,
            deferred = new $.Deferred(),
            current;

        if (this.hasTabs) {
            return deferred.resolve();
        }

        if (_.isUndefined(swc.validation)) {
            swc.validation = {};
        }

        if (_.isUndefined(swc.validation.buffer)) {
            swc.validation.buffer = [];
        }

        if (_.isUndefined(swc.validation.isInProgress)) {
            swc.validation.isInProgress = false;
        }

        if (_.isUndefined(processNext)) {
            swc.validation.buffer.push({e: e, validateAll: validateAll, deferred: deferred});
        }

        if (!swc.validation.isInProgress && !_.isEmpty(swc.validation.buffer)) {
            swc.validation.isInProgress = true;
            current = swc.validation.buffer.shift();
            $.when(self.pageValidationCore(current.e, current.validateAll))
                .done(function (result) {
                    current.deferred.resolve(result);
                })
                .fail(function (result) {
                    current.deferred.reject(result);
                })
                .always(function () {
                    swc.validation.isInProgress = false;
                    if (!_.isEmpty(swc.validation.buffer)) {
                        self.pageValidation(null, null, true);
                    }
                });
        }

        return deferred.promise();
    },

    pageValidationCore: function(e, validateAll) {
        var self = this,
            deferred = new $.Deferred(),
            promises = [],
            elementsValues = [],
            validationArray = this.tabView ? this.tabView.validation : this.validation,
            elementsToValidate = this.$el.find('.validatable:not(.skip-validation)'),
            messageValidation = this.$el.find('.buttons-container-message .save-validation-errors'),
            messageSuccess = this.$el.find('.buttons-container-message .save-success'),
            messageError = this.$el.find('.buttons-container-message .save-error'),
            allValuesDefault = this.pageCheckDefaultValues();

        // Get elements name => value map
        $.each(this.getElements(), function(key, element) {
            elementsValues.push(getParameter($(element)));
        });

        // Set buttons state to disabled if all values are default
        this.setButtonsState(null, !!allValuesDefault);

        // If single element validation happens
        if (e) {
            var validatableElement = $(e).hasClass('validatable') ? $(e) : $(e.target);
            elementsToValidate = validatableElement.is('.validatable:not(.skip-validation)') ? [validatableElement] : [];
        }
        
        // Validate all elements from array and show errors:
        $.each(elementsToValidate, function(key, element) {
            var validationElement = $(element),
                promise,
                parameters = getParameter(validationElement),
                validationElements = $('.validatable[name="' + parameters.parameterName + '"], .validatable[data-name="' + parameters.parameterName + '"]'),
                validationMessage = self.$el.find('.validation-message[data-parameter-name="' + parameters.parameterName + '"]'),
                validationToDo,
                validationStatus;

            // Call single validation method:
            if (validationArray[parameters.parameterName]) {
                validationToDo = validationArray[parameters.parameterName].split(':');
                
                var aModel = swc.models[validationToDo[0]];
                // It could happen that model is not initiated yet, try find object in swc.constructors next 
                if (_.isUndefined(aModel)) {
                    aModel = new swc.constructors[validationToDo[0]]();
                    if (_.isUndefined(aModel)) {
                        throw "Called validation from undefined Model - " + validationToDo[0];
                    }
                }
                validationStatus = aModel.validation[validationToDo[1]](elementsValues);

                // Check if validation is failed or success:
                // (validator can return true, false or Deferred object)
                if (_.isFunction(validationStatus.then)) { // probably a promise
                    promise = validationStatus;
                } else if (validationStatus.status === true) {
                    promise = (new $.Deferred()).resolve(validationStatus);
                } else {
                    promise = (new $.Deferred()).reject(validationStatus);
                }
            } else {
                promise = (new $.Deferred()).resolve();
            }

            promises.push(promise);

            $.when(promise)
                .done(function () {
                    // Go through all validatable group
                    $.each(validationElements, function(key, element) {
                        var currentElement = $(element);
                        self.changeValidationMessageState(currentElement, false);
                    });

                    // Display message
                    if (validationMessage.size()) {
                        validationMessage.find('.error-message').hide();
                        validationMessage.hide();
                    }
                })
                .fail(function (validationStatus) {
                    var wasShown;
                    // Go through all validateable group
                    $.each(validationElements, function(key, element) {
                        var currentElement = $(element);
                        if (currentElement.data('validation-was-shown')) {
                            wasShown = true;
                        } else {
                            currentElement.data('validation-was-shown', true);
                        }

                        self.changeValidationMessageState(currentElement, true);
                    });
    
                    // Display message
                    if (validationMessage.size()) {
                        var globalMessageContainer = validationMessage.find('.error-message[data-error="global"]');
                        validationMessage.show();
                        validationMessage.find('.error-message').hide();

                        if (wasShown || !globalMessageContainer.size()) {
                            validationMessage
                                .find('.error-message[data-error="' + validationStatus.messages[0] + '"]').show();
                        } else {
                            globalMessageContainer.show();
                        }
                    }
                });
        });

        // Hide messages about successfull and error saves
        messageSuccess.hide();
        messageError.hide();

        // Hide "Fix errors on the page" message
        $.when.apply(null, promises)
            .done(function () {
                messageValidation.hide();
                deferred.resolve();
            })
            .fail(function () {
                if (validateAll) {
                    messageValidation.show();
                } else {
                    messageValidation.hide();
                }
                deferred.reject();
            });

        return deferred.promise();
    },

    changeValidationMessageState: function (element, on) {
        if (on) {
            element.addClass('validation-error');
            element.data('validation-error', true);
            element.trigger('validation-fail');
        } else {
            element.removeClass('validation-error');
            element.data('validation-error', false);
            element.trigger('validation-success');
        }
    },

    /**
     * Remove all existing validation messages on the page:
     */
    clearValidationMessages: function(input){
        var inputName,
            validationMessages;

        // If an input is passes then we have to clear
        // the validation messages for the input only:
        if (input) {
            inputName = $(input).data('name');
            validationMessages = $('.validation-message[data-parameter-name="' + inputName + '"]');
            validationMessages.hide();
            validationMessages.find('.error-message').hide();
            $(input).filter('.validation-error').removeClass('validation-error');
            return;
        }

        $('.validation-message').hide();
        $('.save-validation-errors').hide();
        $('.validation-message .error-message').hide();
        $('.validation-error').removeClass('validation-error');
    },

    /**
     * Init listener for current view:
     */
    startListener: function() {
        var pagesMatch = false,
            pagesRoutes = swc.settings.application.get('navigation'),
            self = this,
            page = Backbone.history.fragment;

        // Check if listener available on current page
        if (!this.listenerPages.length) {
            pagesMatch = true;
        } else {
            $.each(this.listenerPages, function(key, value) {
                if (pagesRoutes[value] === page) {
                    pagesMatch = true;
                }
            });
        }

        // Check if everything is correct:
        if (pagesMatch && this.listenerEnabled) {
            if (!this.listenerWorking) {

                // Set flag to listener to be in process:
                this.listenerWorking = true;

                this.listenerIntervalMethod = setInterval(function() {
                    var pagesMatch = false,
                        pagesRoutes = swc.settings.application.get('navigation'),
                        page = Backbone.history.fragment;

                    // Check if listener available on current page
                    if (!self.listenerPages.length) {
                        pagesMatch = true;
                    } else {
                        $.each(self.listenerPages, function(key, value) {
                            if (pagesRoutes[value] === page) {
                                pagesMatch = true;
                            }
                        });
                    }

                    // Check if no restrictions
                    if (pagesMatch) {
                        $.when(self.loadModels(true)).done(function() {
                            if (_.isFunction(self.onListenerComplete)) {
                                self.onListenerComplete();
                            }
                        });
                    } else {
                        self.stopListener();
                    }
                }, self.listenerInterval * 1000);
            }
        } else {
            this.stopListener();
        }
    },

    /**
     * Stop current view listener working:
     */
    stopListener: function() {
        // Set listening flag to false:
        this.listenerWorking = false;

        // Remove current interval variable:
        clearInterval(this.listenerIntervalMethod);
    },

    customizeDevice: function(options) {
        var pages,
            devicesNames = swc.models.Locale.getLocaleStrings('overview/modal-windows/customize-device'),
            pageActive = 1,
            pageItems = 6, // Number of devices on one row
            // the list of supported devices which has device's type as a key
            allDeviceTypes = swc.settings.application.get('supported' + options.type.capitalize() + 'Devices'),
            // the array of device's types which can be associated with device only automatically
            nonCustomizableDevices = ['unrecognized', 'multisensor', 'switch'],
            supportedDeviceTypes = [],
            sahDeviceTypes;
        
        if (options.type !== 'dect'){
            // Check if this is unsupported device and overwrite current device's type with default value
            if (!allDeviceTypes[options.device.type]) {
                options.device.type = allDeviceTypes['unrecognized'].deviceTypeDefault;
            }

            // Pass translation for each device
            $.each(allDeviceTypes, function(supportedKey, supportedDevice) {
                var deviceText = devicesNames[supportedDevice.deviceTypeLocalized];

                // Set device type to be localised
                if (!_.isUndefined(deviceText) && !_.isEmpty(deviceText)) {
                    supportedDevice.deviceTypeText = deviceText;
                } else {
                    supportedDevice.deviceTypeText = supportedDevice.deviceTypeLocalized;
                }

                if ($.inArray(supportedKey, nonCustomizableDevices) === -1) {
                    supportedDeviceTypes[supportedKey] = supportedDevice;
                }
            });

            // In order to implement simple pagination, let's pass additional array consisting of keys of supported devices.
            // SAH returns some device type as several words separated with space
            sahDeviceTypes = _.keys(supportedDeviceTypes);
            
            pages = Math.ceil(getObjectLength(sahDeviceTypes) / pageItems);
        
        }

        /**
         * Validates device name, also triggers validation failed handlers,
         * like show error message
         * 
         * NOTE: "During the HPQC review meeting Simon Maurer approved to "allow basically everything" here"
         * 
         * @param modal {PopoverWindow} Instance of current modal popover window
         * @param name {String} Name of device to be validated
         * 
         * @return {boolean} true if name is valid
         */
        var validateDeviceName = function (modal, name) {

            var validationMessage = modal.find('.validation-message'),
                validationError = false;

            // allowed basically anything, from 1 to 32 characters length
            if (!/^.{1,32}$/.test(name)) {
                validationError = true;
            }

            if (validationError) {
                validationMessage.show();
                validationMessage.find('.error-message[data-error="global"]').show();
            } else {
                validationMessage.hide();
                validationMessage.find('.error-message[data-error="global"]').hide();
            }

            modal.find('input[name="device-name"]').toggleClass('validation-error', validationError);

            return !validationError;
        };

        SWCElements.modalWindow.show({
            templateID: options.template || 'overview:modal-windows:customize-device',
            templateData: {
                device: options.device,
                pages: pages,
                pagesArray: new Array(pages),
                pageItems: pageItems,
                pageActive: pageActive,
                supported: supportedDeviceTypes,
                sahDeviceTypes: sahDeviceTypes
            },
            className: 'customize-device',

            onShow: function() {
                var modal = $('.modalWindow'),
                    rows = modal.find('.devices-list-scroll'),
                    devices = modal.find('.device'),
                    switchPage = function(page) {
                        var pages = modal.find('.switch-page');

                        // Set active class to current page:
                        pages.removeClass('active');
                        modal.find('[data-page="' + page + '"]').addClass('active');

                        // Set active class to current row:
                        rows.removeClass('active');
                        modal.find('[data-page="' + page + '"]').addClass('active');
                    };

                // Display necessary row:
                rows.removeClass('active');
                modal.find('[data-page="' + pageActive + '"]').addClass('active');

                // Handle click on devices:
                modal.on('click', '.device', function(e) {
                    var hasChanges = false,
                        element = $('input[name="device-name"]'),
                        newType = $(this).data('type'),
                        newName = modal.find('input[name="device-name"]').val();

                    devices.removeClass('selected');
                    $(this).addClass('selected');

                    if (newType !== element.data('type') || newName !== element.data('name')) {
                        hasChanges = true;
                    }

                    modal.find('.button.apply-changes').toggleClass('disabled', !hasChanges);
                });

                modal.on('click', '.navigation.left:not(.disabled)', function(e) {
                    var currentPageElement = modal.find('.page.active'),
                        currentPageNumber = currentPageElement.data('page') || 1,
                        nextPageNumber = currentPageNumber - 1,
                        nextPageElement = modal.find('.page[data-page="' + nextPageNumber + '"]');

                    if (nextPageElement.size()) {
                        switchPage(nextPageNumber);
                    } else {
                        switchPage(currentPageNumber);
                    }
                });

                modal.on('click', '.navigation.right:not(.disabled)', function(e) {
                    var currentPageElement = modal.find('.page.active'),
                        currentPageNumber = currentPageElement.data('page'),
                        nextPageNumber = currentPageNumber +  1,
                        nextPageElement = modal.find('.page[data-page="' + nextPageNumber + '"]');

                    if (nextPageElement.size()) {
                        switchPage(nextPageNumber);
                    } else {
                        switchPage(currentPageNumber);
                    }
                });

                modal.on('click', '.switch-page', function(e) {
                    var pageElement = $(e.target).closest('.switch-page'),
                        pageNumber = pageElement.data('page');

                    switchPage(pageNumber);
                });

                modal.on('keyup', 'input[name="device-name"]', function(e) {
                    var hasChanges = false,
                        element = $(e.target),
                        newType = modal.find('.device').filter('.selected').data('type'),
                        newName = modal.find('input[name="device-name"]').val();

                    if (newType !== element.data('type') || newName !== element.data('name')) {
                        hasChanges = true;
                    }

                    modal.find('.button.apply-changes').toggleClass('disabled', !hasChanges);
                });

                modal.on('change', 'input[name="device-name"]', function(e) {
                    var deviceName = $.trim($(this).val());
                    validateDeviceName(modal, deviceName);
                });
            },

            onCancel: function() {
                if (options.onCancel && $.isFunction(options.onCancel)) {
                    options.onCancel();
                }
            },

            onApply: function() {
                var modal = $('.modalWindow'),
                    $input = modal.find('input[name="device-name"]'),
                    deviceType = modal.find('.device').filter('.selected').data('type'),
                    deviceName = $.trim($input.val()),
                    deviceMac = $input.data('mac');

                if (options.onApply && $.isFunction(options.onApply) && validateDeviceName(modal, deviceName)) {
                    if (options.type==='dect') {
                        options.onApply({
                            line: $input.data('line'),
                            name: deviceName
                        });
                    } else {
                        options.onApply({
                            deviceMac: deviceMac,
                            deviceType: deviceType,
                            deviceName: deviceName
                        });
                    }

                    SWCElements.modalWindow.hide();
                }
            }
        });
    },

    /**
     * Generate Expandable element
     * @param options {Object}
     */
    generateExpandableList: function(options) {
        var template = $.template("expandable-list", swc.Templates.get("components:expandable-list").get('content')),
            objectToTemplate = {
                titleBlocks: {},
                itemsBlocks: {}
            };

        // Generate Title Block
        $.each(options.titleBlocks, function(key, block) {
            if (block.templateID && block.templateValues) {
                var html,
                    plainHTML = '';

                // Upgrade basic passed values:
                block.templateValues.localeStrings = options.locales;
                block.templateValues.localeString = getTranslationStringsjQuery;
                block.templateValues.formatDate = swc.models.Locale.formatDate;

                html = $.tmpl(
                    $.template("component-" + key, swc.Templates.get(block.templateID).get('content')), block.templateValues
                );

                $.each(html, function(key, value) {
                    if (value.outerHTML) {
                        plainHTML = plainHTML + value.outerHTML;
                    }
                });

                objectToTemplate.titleBlocks[key] = plainHTML;
            }

            if (block.translationString) {
                if (options.locales[block.translationString]) {
                    objectToTemplate.titleBlocks[key] = options.locales[block.translationString];
                } else {
                    objectToTemplate.titleBlocks[key] = "<span class='missed-string'>" + block.translationString + "</span>";
                }
            }

            if (block.plaintext) {
                objectToTemplate.titleBlocks[key] =  block.plaintext;
            }
        });

        // Generate Items blocks
        $.each(options.itemsBlocks, function(listItemID, listItem) {
            objectToTemplate.itemsBlocks[listItemID] = {
                blocks: {},

                isEditAble: listItem.isEditAble ? true : false,
                isDeleteAble: listItem.isDeleteAble ? true : false
            };

            $.each(listItem.blocks, function(key, block) {
                if (block.templateID && block.templateValues) {
                    var html,
                        plainHTML = '';

                    // Upgrade basic passed values:
                    block.templateValues.localeStrings = options.locales;
                    block.templateValues.localeString = getTranslationStringsjQuery;
                    block.templateValues.formatDate = swc.models.Locale.formatDate;

                    html = $.tmpl(
                        $.template("component-" + key, swc.Templates.get(block.templateID).get('content')), block.templateValues
                    );

                    $.each(html, function(key, value) {
                        if (value.outerHTML) {
                            plainHTML = plainHTML + value.outerHTML;
                        }
                    });

                    objectToTemplate.itemsBlocks[listItemID].blocks[key] = plainHTML;
                }

                if (block.translationString) {
                    if (options.locales[block.translationString]) {
                        objectToTemplate.itemsBlocks[listItemID].blocks[key] = options.locales[block.translationString];
                    } else {
                        objectToTemplate.itemsBlocks[listItemID].blocks[key] = "<span class='missed-string'>" + block.translationString + "</span>";
                    }
                }

                if (block.plaintext) {
                    objectToTemplate.itemsBlocks[listItemID].blocks[key] =  block.plaintext;
                }
            });

            if (listItem.hiddenPart.templateID && listItem.hiddenPart.templateValues) {
                var html,
                    plainHTML = '';

                // Upgrade basic passed values:
                listItem.hiddenPart.templateValues.localeStrings = options.locales;
                listItem.hiddenPart.templateValues.localeString = getTranslationStringsjQuery;
                listItem.hiddenPart.templateValues.formatDate = swc.models.Locale.formatDate;

                html = $.tmpl(
                    $.template("component-" + listItemID, swc.Templates.get(listItem.hiddenPart.templateID).get('content')), listItem.hiddenPart.templateValues
                );

                $.each(html, function(key, value) {
                    if (value.outerHTML) {
                        plainHTML = plainHTML + value.outerHTML;
                    }
                });

                objectToTemplate.itemsBlocks[listItemID].hiddenPart = plainHTML;
            }
        });

        var domElement = $.tmpl(template, {
            titleBlocks: objectToTemplate.titleBlocks,
            items: objectToTemplate.itemsBlocks,
            localeStrings: options.locales,
            localeString: getTranslationStringsjQuery,
            formatDate: swc.models.Locale.formatDate
        });

        if(options.openedTab){
            domElement.find('.expandable-list-item[data-key='+options.openedTab+']').addClass('expanded');
        }

        return domElement;
    }
});

/**
 * Base BackBone view for displaying tabs in pages
 *
 * @type {*}
 *
 */
swc.base.TabView = swc.base.PageView.extend({

    /**
     * Initialise view
     */
    initialize: function() {
        var pagesArray = Backbone.history.fragment.split("/");

        // Define tab template and locale ID:
        this.pageTemplateID = pagesArray[0] + ':' + pagesArray[1] + ':' + pagesArray[2];

        // Prepare template for current tab:
        this.template = $.template("pageContent", swc.Templates.get(this.pageTemplateID).get('content'));

        // Extend global events:
        this.events = _.extend({}, swc.base.PageView.prototype.events, this.events);
    }
});

/**
 *
 * Backbone view for Scheduler line
 *
 * @type {*}
 *
 */
swc.constructors.SchedulerLiner = swc.base.PageView.extend({
    className: 'ranges-block',

    events: {
        'mousedown .diap_container':'new_diapazon',
        'mousedown .left_part': 'modify_start',
        'mousedown .right_part': 'modify_start',
        'mousedown .half': 'modify_start',
        'mousemove': 'modify',
        'mouseleave': 'modify_stop',
        'mouseup': 'modify_stop',
        'click .diapazon': 'prevent_creating',
        'touchstart': 'mobilePreventScroll',
        'touchstart .mobile-block': 'mobileNewDiap',
        'touchmove .mobile-block': 'mobileModify',
        'touchleave .mobile-block': 'mobileModifyStop',
        'touchend .mobile-block': 'mobileModifyStop',
        'touchcancel .mobile-block': 'mobileModifyStop'
    },

    prevent_creating: function(e){
        e.stopPropagation();
    },

    mobilePreventScroll: function(e){
        e.preventDefault();
    },

    mobileNewDiap: function(e){
        e.pageX = e.originalEvent.pageX;
        $(e.target).hide();
        var targetElement = document.elementFromPoint(e.originalEvent.targetTouches[0].clientX, e.originalEvent.targetTouches[0].clientY);
        $(e.target).show();
        if($(targetElement).hasClass('diap_container')){
            e.target = targetElement;
            this.new_diapazon(e);
        } else if($(targetElement).hasClass('left_part') || $(targetElement).hasClass('right_part')){
            e.target = targetElement;
            this.modify_start(e);
        }
    },

    mobileModify: function(e){
        e.pageX = e.originalEvent.pageX;
        this.modify(e);
    },

    mobileModifyStop: function(e){
        this.modify_stop(e);
    },

    /**
     * creates new range
     * @param e
     */
    new_diapazon: function(e){
        var self = this, target,
            element = $(e.target);

        if (!this.isDisabled && !element.closest('.diapazon')[0]) {
            var schedulerIndex = $('.schedulers').index(element.closest('.schedulers')),
                linerIndex = element.closest('.schedulers').find('.liner').index(element.closest('.liner'));

            e.stopPropagation();
            e.preventDefault();

            if(typeof e.offsetX === "undefined") {
                var targetOffset = element.offset();
                e.offsetX = e.pageX - targetOffset.left;
            }

            this.modifyProcess = true;
            this.markForDelete = false;

            var startCoord = this.stepWidth*(Math.floor(e.offsetX/this.stepWidth));

            if (e.offsetX>=this.scaleWidth-this.stepWidth) {
                startCoord = this.scaleWidth-this.stepWidth;
            }

            var endCoord = startCoord + this.stepWidth;
            var conflictsDiapazon = false;

            var currentDay = 'monday';
            _.each(element.closest('.liner').attr('class').split(' '), function(key){
                if(key.indexOf('day') > -1){
                    currentDay = key;
                }
            });
            this.currentDiapazon = this.diapazones[currentDay];

            _.each(this.currentDiapazon, function(diapazon){
                if(startCoord === diapazon.end || (startCoord >= diapazon.begin-self.stepWidth && startCoord <= diapazon.begin)){
                    conflictsDiapazon = true;
                }
            });

            if(!conflictsDiapazon){

                this.currentDiapazon.push({
                    begin: startCoord,
                    end: endCoord
                });
                swc.constructors.dispatcher.trigger('scheduler:change');
                this.options.model.trigger('change:pixelSchedule', this.options.model);
                this.render();

                $('.schedulers').eq(schedulerIndex).find('.liner').eq(linerIndex).find('.diapazon').each(function(index, el){
                    if($(el).css('left') === startCoord+'px'){
                        target = $(el).find('.half')[0];
                    }
                });

                if(target){
                    self.modify_start({target: target, preventDefault: e.preventDefault, stopPropagation: e.stopPropagation});
                }
            } else {
                this.modifyProcess = false;
            }
        }
    },

    /**
     * Starts the modify of some range - calls when mousedown on left or right part of range
     * @param e
     */
    modify_start: function(e){
        e.preventDefault();
        e.stopPropagation();
        var self = this;

        if(!this.isDisabled){
            this.modifyProcess = true;
            this.mouseStart = e.target.offsetLeft + e.target.parentNode.offsetLeft;


            var element = $(e.target);

            // info for modified element searching - needed for popover
            this.elementInfo = {};
            this.elementInfo.operatedLineIndex = $('.schedulers .liner').index(element.closest('.liner'));

            var currentDay = 'monday';
            _.each($(e.target).closest('.liner').attr('class').split(' '), function(key){
                if(key.indexOf('day') > -1){
                    currentDay = key;
                }
            });
            this.currentDiapazon = this.diapazones[currentDay];

            this.operatedDiapazon = {};
            $.each(this.currentDiapazon, function(index, diapazon){
                if(diapazon.begin <= self.mouseStart && diapazon.end + self.stepWidth > self.mouseStart){
                    self.operatedDiapazon = diapazon;
                    self.currentDiapazon.splice(index, 1);
                    self.currentDiapazon.push(self.operatedDiapazon);
                    return false;
                }
            });
            var maxRightAlpha = this.scaleWidth;
            var maxLeftAlpha = this.scaleWidth;
            this.nearestDiapazonStart = 0;
            this.nearestDiapazonFinish = 0;

            $.each(this.currentDiapazon, function(index, diapazon){
                if(diapazon.begin !== self.operatedDiapazon.begin){

                    if(diapazon.begin > self.operatedDiapazon.end){
                        if((diapazon.begin - self.operatedDiapazon.end)<maxRightAlpha){
                            maxRightAlpha = (diapazon.begin - self.operatedDiapazon.end)-self.stepWidth;
                        }
                    }
                    if(diapazon.end < self.operatedDiapazon.begin){
                        if((self.operatedDiapazon.begin - diapazon.end)<maxLeftAlpha){
                            maxLeftAlpha = (self.operatedDiapazon.begin - diapazon.end)-self.stepWidth;
                        }
                    }
                }
            });


            if(maxRightAlpha >= this.scaleWidth){
                maxRightAlpha = this.scaleWidth - this.operatedDiapazon.end;
            }
            if(maxLeftAlpha >= this.scaleWidth){
                maxLeftAlpha = this.operatedDiapazon.begin;
            }

            this.nearestDiapazonStart = this.operatedDiapazon.end + maxRightAlpha;
            this.nearestDiapazonFinish = this.operatedDiapazon.begin - maxLeftAlpha;

            $('body').trigger('popover:close');

            if(element.hasClass('left_part')){
                this.partModified = 'left';
            } else if(element.hasClass('right_part')) {
                this.partModified = 'right';
            } else {
                this.partModified = 'both';
            }

            if(element.hasClass('left_part') || element.hasClass('right_part')){
                element.trigger('popover:toggle');
            }

            this.allowModify = true;
        }
    },

    /**
     * Proceed with modify - calls on mousemove with mousedown
     * @param e
     */
    modify: function(e){
        e.preventDefault();
        e.stopPropagation();

        if(this.modifyProcess && this.allowModify && !this.isDisabled){
            var self = this, element;
            this.allowModify = false;

            setTimeout(function(){
                self.allowModify = true;
            }, 20);


            var currentCoord = e.pageX - $(this.el).find('.diap_container').eq(0).offset().left;
            var coordsAlpha = this.stepWidth*(Math.round((currentCoord - this.mouseStart)/this.stepWidth));

            if(coordsAlpha!==0){
                var diapazonBackup = _.clone(this.operatedDiapazon);
                this.currentDiapazon.pop();

                if(this.partModified === 'right'){
                    this.operatedDiapazon.end += coordsAlpha;
                } else if(this.partModified === 'left') {
                    this.operatedDiapazon.begin += coordsAlpha;
                } else {
                    if(coordsAlpha > 0){
                        this.partModified = 'right';
                        this.operatedDiapazon.end += this.stepWidth;
                    } else {
                        this.partModified = 'left';
                        this.operatedDiapazon.begin -= this.stepWidth;
                    }
                }


                if(this.operatedDiapazon.end < this.operatedDiapazon.begin + this.stepWidth){
                    if(this.partModified === 'right'){
                        this.mouseStart = diapazonBackup.end;
                    } else {
                        this.mouseStart = diapazonBackup.begin;
                    }
                    this.currentDiapazon.push(diapazonBackup);
                    this.markForDelete = true;
                    if(this.partModified === 'right'){
                        this.operatedDiapazon.end = this.operatedDiapazon.begin + this.stepWidth;
                    } else {
                        this.operatedDiapazon.begin = this.operatedDiapazon.end - this.stepWidth;
                    }

                } else if(this.operatedDiapazon.begin < this.nearestDiapazonFinish){
                    this.operatedDiapazon.begin = this.nearestDiapazonFinish;
                    this.currentDiapazon.push(this.operatedDiapazon);
                } else if(this.operatedDiapazon.end > this.nearestDiapazonStart){
                    this.operatedDiapazon.end = this.nearestDiapazonStart;
                    this.currentDiapazon.push(this.operatedDiapazon);
                } else {
                    this.mouseStart += coordsAlpha;
                    this.currentDiapazon.push(this.operatedDiapazon);
                    this.markForDelete = false;

                    $('body').trigger('popover:close');
                    this.render();

                    /**
                     * Search for modified element
                     */
                    $('.schedulers .liner').eq(this.elementInfo.operatedLineIndex).find('.diapazon').each(function(index, el){
                        if($(el).css('left') === self.operatedDiapazon.begin+'px'){
                            element = $(el).find('.'+self.partModified+'_part');
                        }
                    });

                    if((this.operatedDiapazon.end - this.operatedDiapazon.begin) > 9){
                        element.trigger('popover:toggle');
                    }
                }
            }
        }
    },

    /**
     * Stops the modifications and triggers 'change' event on model
     * calls when mouseup or mouseleave the view container
     * @param e
     */
    modify_stop: function(e){
        $('body').trigger('popover:close');

        if(this.modifyProcess){
            this.modifyProcess = false;
            if(this.markForDelete){
                this.currentDiapazon.pop();
            }
            swc.constructors.dispatcher.trigger('scheduler:change');
            this.options.model.trigger('change:pixelSchedule', this.options.model);
            this.render();
        }
    },

    initialize: function(){
        /**
         * when true, scheduler is grey and could not be modified
         * @type {*|Boolean}
         */
        this.isDisabled = this.options.isDisabled || false;
        /**
         * width of scheduler line - usually is 432
         * @type {number}
         */
        this.scaleWidth = this.options.scaleWidth || 432;
        /**
         * width of one part on liner, usually is 432/48 = 9, 48 - number of 30-minutes pieces in one day
         * @type {number}
         */
        this.stepWidth = this.scaleWidth/48;
        /**
         * object with ranges
         * @type {*}
         */
        this.diapazones = this.options.diapazones;
        /**
         * What will be scheduled?
         * @type: string
         */
        this.schedulerType = this.options.schedulerType;

        /**
         *  block for mobile manipulations
         */
        if(detectMobileBrowser()){
            var mobileBlock = $('<div/>', {'class': 'mobile-block'}).css({'width':'100%', 'height':'100%'});
            this.$el.append(mobileBlock);
        }

        this.render();
    },

    render: function(){
        var self = this;
        var element = $(this.el);

        var template = $.tmpl(swc.Templates.get('core:scheduler:table').get('content'),{
            'isDisabled': this.isDisabled,
            'stepWidth': this.stepWidth,
            'scaleWidth' : this.scaleWidth,
            'schedulers' : this.diapazones,
            'type': this.schedulerType,
            'localeStrings': swc.models.Locale.getLocaleStrings('core'),
            'localeString': getTranslationStringsjQuery,
            formatDate: swc.models.Locale.formatDate
        })[0];

        element.find('.' + $(template).attr('class')).remove();
        element.append(template);
        return this;
    }
});


/**
 * A view based on models collection 
 */
swc.base.listView = Backbone.View.extend({
    // model: colection

    itemView: '',

    initialize: function (options) {
        var self = this;
        this.options = options;
        Backbone.View.prototype.initialize.call(this, options);
        this.itemView = options.itemView;
        this.model.on('change', function () { self.render();});
    },

    renderComplete: function () {
        // to be overriden
        if (_.isFunction(this.options.renderComplete)) {
            return this.options.renderComplete.apply(this, arguments);
        }
    },

    preRender: function () {
        if (_.isFunction(this.options.preRender)) {
            return this.options.preRender.apply(this, arguments);
        }
        return (new $.Deferred()).resolve();
    },

    render: function (){
        var self = this,
            promise;

        if (self.hasTabs) {
            promise = self.tabView.preRender();
        } else {
            promise = self.preRender();
        }

        $.when(promise)
            .done(function() {
                // render each collection item:
                self.$el.empty();

                _.each(self.model.models, function (model, idx) {
                    var view = new swc.constructors[self.itemView]({model: model});
                    self.$el.append(view.render().el);
                });

                self.renderComplete();
            });

        return this;
    }
});
