swc.constructors.Router = Backbone.Router.extend({

    /**
     * This stores history to allow navigate back
     */
    history: [],

    /**
     * Start history tracking:
     */
    initialize: function() {
        this.on('route', this.storeHistory);
        Backbone.history.start();
    },
    
    /**
     * Define router settings:
     */
    routes: {
        '': 'login',
        'login': 'login',
        'remote-login': 'remoteLogin',
        '*': 'loadModule'
    },

    login: function() {
        this._doLogin();
    },


    remoteLogin: function() {
        // We add global css class here to the body - "disabled-for-superadmin"
        // Needed to hide all controls which should not be visible for superuser
        $("body").addClass("disabled-for-superadmin");
        this._doLogin(true);
    },

    /**
     * General login method which has some specific logic if login done remotely through WAN port
     * @param isRemote {boolean} Optional. Just pass TRUE in case if this is remote login 
     * @private
     */
    _doLogin: function(isRemote) {
        var userIsLoggedIn = swc.models.Login.checkUserLogin(),
            route = 'login';
        
        if (isRemote) {
            // Set this value to let login view know about remote login
            localStorage.setItem('remoteLogin', true);
            route = 'remote-login';
        }

        if (userIsLoggedIn && !localStorage.getItem("changePassIncomplete")) {
            this.navigate('overview', {trigger: true});
        } else {
            this.navigatePage(route);
        }
    },

    /**
     * General controller - loads module
     * @see this.navigatePage method
     * @param path
     */
    loadModule: function(path) {
        var userIsLoggedIn = swc.models.Login.checkUserLogin();
        if (!userIsLoggedIn) {
            swc.models.Login.processLogout();
        } else {
            // don't allow user to see application if "set new password" procedure was not finished
            if (localStorage.getItem("changePassIncomplete")) {
                this.navigate("login", {trigger: true});
            } else {
                this.navigate(path, {trigger: true});
            }
        }
    },

    /**
     * Save current fragment into history
     */
    storeHistory: function() {
        swc.constructors.dispatcher.trigger('route:change');
        this.history.push(Backbone.history.fragment);
    },

    /**
     * Changes url to the previous one
     */
    navigateBack: function() {
        if (this.history.length > 1) {
            Backbone.history.navigate(this.history[this.history.length - 2], {trigger: true});
        } else {
            this.navigate(Backbone.history.fragment, {trigger: true});
        }
    },

    saveChangesDialog: function(confirmLeave) {
        var self = this;
        
        SWCElements.modalWindow.show({
            templateID: 'application:modal:changes-confirm',
            templateData: {
                formatDate: swc.models.Locale.formatDate
            },
            localeStrings: swc.models.Locale.getLocaleStrings('application'), // TODO :: no time to identify. Refactor this
            className: 'confirm-changes',
            
            onCancel: function() {
                // resolving deferred will allow us navigate to requested page
                confirmLeave.resolve();
            },
            
            onApply: function() {
                self.navigateBack();
                SWCElements.modalWindow.hide();
            }
        });
    },

    navigatePage: function(path) {
        var pathComponents = path ? path.split('/') : Backbone.history.fragment.split('/'),
            area = $('#current-page'),
            constructor, view, page, subPage, tab;
        
        // Check if application is already rendered:
        if (!area.size()) {
            swc.views.Application.showPageLoading('Initializing application');
        }

        if (pathComponents.length) {
            page = pathComponents[0];
        }

        if (pathComponents.length > 1) {
            subPage = pathComponents[1];
        }

        if (pathComponents.length > 2) {
            tab = pathComponents[2];
        }
        
        // Generate constructor and view name:
        if (page && subPage) {
            constructor = page.capitalize().replaceAll('-', '') + subPage.capitalize().replaceAll('-', '') + 'View';
        } else {
            constructor = page.capitalize().replaceAll('-', '') + 'View';
        }
        
        // Select correct name for view constructor:
        view = constructor.deCapitalize();

        // Check if constructor exists:
        if (!swc.views[view]) {
            swc.views[view] = new swc.constructors[constructor]();
        }

        // Define if current page has tabs:
        if (tab) {
            swc.views[view].hasTabs = true;
        } else {
            swc.views[view].hasTabs = false;
        }

        this.bindChangeRouteEvent(swc.views[view]);

        // Render current view:
        swc.views[view].render();
    },

    /**
     * Bind event "route:change"
     *
     * @description:
     *
     * This method binds event "route:change" which is fired in the this.storeHistory method
     * and invoke this.beforeRouteChangeHook.
     * The event unbinds after invoking of the hook
     */
    bindChangeRouteEvent: function(view) {
        swc.constructors.dispatcher.on('route:change', function() {
            if (_.isFunction(view.beforeRouteChangeHook)) {
                view.beforeRouteChangeHook();
            }
            swc.constructors.dispatcher.off('route:change');
        });
    },

    /**
     * This customized method will check if there are any changes made on page and Apply button is enabled.
     * Will show "confirm leave" dialog window if have unsaved pages. 
     * 
     * @override
     * @param fragment
     * @param options - standard Backbone navigation options + additional parameter {skipUnsavedChanges: true/false}, 
     *                  which defines weather to skip check for unsaved changes 
     * @returns {defer.promise|*|Promise.promise}
     */
    navigate: function(fragment, options) {
        var currentView = swc.views.currentView,
            self = this,
            confirmLeave = new $.Deferred();
        
        // Only go ahead if user confirmed their decision
        confirmLeave.promise().then(function(){
            // do the actual navigation
            Backbone.Router.prototype.navigate(fragment, options);
            self.navigatePage(fragment);
        });

        // Check for not saved changes on page
        // temporally canceled checker on some pages
        if (currentView && currentView.hasChanged() && !_.isUndefined(options) && (!options.skipUnsavedChanges)) {
            this.saveChangesDialog(confirmLeave);
            return confirmLeave.promise();
        } else {
            confirmLeave.resolve();
        }
    }
});
