/**
 *
 * Backbone model for Localisation
 *
 * @type {*}
 *
 */
swc.constructors.LocaleModel = Backbone.Model.extend({

    /**
     * Current application locale
     *
     * @param locale {String}
     *
     */
    locale: "",

    /**
     * Default application locale defined in settings
     * 
     * @var defaultLocale {String}
     */
    defaultLocale: "",

    /**
     * List of available locales:
     *
     * @param available {Array}
     *
     */
    available: [],

    initialize: function() {
        this.defaultLocale = swc.settings.application.get('locales')['default'];
        this.available = swc.settings.application.get('locales').available;
    },

    /**
     * Get localised strings for selected component:
     *
     * @param component {String}
     *
     */
    getLocaleStrings: function(componentTrigger) {
        var component = Backbone.history.fragment ? Backbone.history.fragment : 'application';

        if (componentTrigger) {
            component = componentTrigger;
        }

        var componentsArray = component.split('/'),
            strings = swc.locales;

        return this.getLocaleStringsRecursively(strings[componentsArray[0]], {});
    },

    getLocaleStringsRecursively: function(strings, result) {
        var self = this;

        if (!getObjectLength(strings)) {
            return result;
        }

        $.each(strings, function(id, string) {
            if (typeof string !== "object") {
                result[id] = string;
            } else {
                self.getLocaleStringsRecursively(string, result);
            }
        });

        return result;
    },

    /**
     * Insert locale strings to the collection when application is loaded
     *
     * @param strings {Object}
     *
     */
    setLocaleStrings: function(strings) {
        swc.locales = strings;
    },

    /**
     * Get saved UI language in a deferred way
     * 
     * @return Promise
     */
    getLocale: function(){
        var deferred = new $.Deferred();
        
        // Request language from the server
        // Save current UI language on NP side
        var restClient = new swc.constructors.Rest();
        restClient.sendRequest({
            url: '/sysbus/UserInterface:getLanguage',
            method: 'POST',
            contentType: 'application/x-sah-ws-4-call+json',
            data: {
                parameters: {}
            },

            success: function(response) {
                if(response.status && response.status !== false){
                    deferred.resolve(response);
                }
                else {
                    deferred.reject();
                }
            },
            
            error: function(){
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    /**
     * Set application locale file:
     *
     * @param locale {String}
     *
     */
    setLocale: function(locale) {
        var self = this,
            deferred = new $.Deferred();

        // Update cookie locale
        localStorage.setItem("locale", locale);
        // save into self property
        self.locale = locale;

        /**
         * Load localization strings
         * 
         * @return Deferred
         */
        function loadLocalizationData() {
            var deferred = new $.Deferred();
            
            $.ajax({
                url: 'static/translations/' + locale + '.json',
                type: 'get',
                dataType:'json'
            }).done(function(response) {
                self.setLocaleStrings(response);
                deferred.resolve();
            }).fail(function(){
                deferred.reject();
            });
            
            return deferred.promise();
        }

        /**
         * Save selected UI language on NP side
         * 
         * @return Promise
         */
        function saveLanguageOnServer(){
            var deferred = new $.Deferred();
            // Save current UI language on NP side
            var restClient = new swc.constructors.Rest();
            restClient.sendRequest({
                url: '/sysbus/UserInterface:setLanguage',
                method: 'POST',
                contentType: 'application/x-sah-ws-4-call+json',
                data: {
                    "parameters": {
                        "currentLanguage": locale
                    }
                },

                success: function(response) {
                    deferred.resolve();
                },
                
                error: function(){
                    deferred.reject();
                }
            });
            
            return deferred.promise();
        }
        
        $.when(loadLocalizationData(), saveLanguageOnServer())
            .done(function() { deferred.resolve(); })
            .fail(function() { deferred.reject(); });

        return deferred.promise();
    },

    /**
     * Get locale options for dropdown:
     */
    getLocaleOptions: function() {
        var self = this,
            options = [];

        $.each(self.available, function(key, locale) {
            options.push({
                name: locale.text,
                value: locale.key
            });
        });
        
        return options;
    },
    
    getStaticPagesLinks: function(){
        var locales = swc.settings.application.get('locales').available;
        var locale = _.findWhere(locales, {key: this.locale});
        return locale.staticPages;
    },

    formatDate: function (date, format) {
        var d = new Date(date).getTime();
        if (_.isNaN(d)) {
            return date;
        }
        moment.lang(swc.models.Locale.locale);
        return moment(d).format(format || "D MMM YYYY, HH:mm");
    }
});
