swc.constructors.TelephonyDeviceView = swc.base.PageView.extend({

    className: 'dect-and-phones',

    models: ['Telephony', 'System'],

    validation: {
        'dect-name-editable':       'Telephony:dectName',
        'internal-number-editable': 'Telephony:internalNumber'
    },

    events: {
        'click .pair-dect': 'pairDect',
        'click .unpair-phone': 'unpairPhone',
        'click .ring-phone': 'ringPhone',

        'keyup .dect-name': 'renameDect',

        'swc-checkbox:change .incoming-line': 'changeIncomingNumber',

        'swc-dropdown:change .swc-dropdown.internal-number': 'changeInternalNumber',
        'swc-dropdown:change .swc-dropdown.external-number': 'changeExternalNumber',

        'expandable:edit .expandable-list': 'listEditMode',
        'expandable:no-edit .expandable-list': 'listDefaultMode'
    },

    /**
     * List of expandable blocks for different telephony types
     */
    phoneTypes: ['wired', 'dect'],

    /* Indicates validation status */
    isValid : true,

    /**
     * Set data to be passed to template
     */
    setTemplateData: function() {
        var phones = swc.models.Telephony.getPhonesList(),
            phoneLines = swc.models.Telephony.get('voip').where({'enable': 'Enabled'});

        this.templateData = {
            dectPhones: phones.dect,
            wiredPhones: phones.cord,
            manyVoips: phoneLines.length >= 2,
            dectStatus: swc.models.Telephony.get('dectStatus'),
            voipStatus: swc.models.Telephony.get('voipStatus')
        };

        this.wiredEditing = null;
        this.dectEditing = null;
    },

    /**
     * Redefine render complete function()
     */
    renderComplete: function() {
        var self = this;

        this.setTemplateData();
        this.getTemplateContent();
        this.displayPage();
        this.hideSingleIncomingNumbers();

        _.defer(this.fillDropDowns, this);

        if (self.wiredEditing) {
            this.$('.expandable-wired-phones .expandable-list').trigger('expandable:swc-edit', self.wiredEditing);
        }

        if (self.dectEditing) {
            this.$('.expandable-dect-phones .expandable-list').trigger('expandable:swc-edit', self.dectEditing);
        }

        if (this.showSuccessMessage) {
            this.$('.buttons-container-message .save-success').show();
            this.showSuccessMessage = false;
        }
    },

    /**
     * Reset view variables
     */
    emptyEditing: function() {
        this.dectEditing = "";
        this.wiredEditing = "";
    },

    /**
     * Event handler: is fired on click on the checkbox next to the incoming number
     *
     * @param e
     * @param value
     */
    changeIncomingNumber: function(e, value){
        var element = $(e.target).closest('.swc-checkbox'),
            line = element.data('line'),
            device = element.closest('.expandable-list-item').data('key'),
            deviceModel = swc.models.Telephony.get('phones').get(device),
            deviceOptions = JSON.stringify(deviceModel.get('incomingLines')),
            deviceOptionsClone = JSON.parse(deviceOptions);

        _.each(deviceOptionsClone, function(obj) {
            if (obj.line === line) {
                obj.value = value;
            }
        });

        deviceModel.set('incomingLines', deviceOptionsClone);

        this.setButtonsState();
    },

    /**
     * Show all DOM elements of telephone incoming/outgoing numbers
     * for edit mode of the expandable-list-item
     *
     * @param phoneID {Int} Phone model ID
     */
    showSingleIncomingNumbers: function(phoneID) {
        this.$('.expandable-list-item.' + phoneID + ' .outgoing-number .state-default').toggleClass('hidden', false);
        this.$('.expandable-list-item.' + phoneID + ' .incoming-number').toggleClass('hidden', false);
    },

    /**
     * Switch visibility for DOM elements of telephone incoming/outgoing numbers
     * for default mode of the expandable-list-item
     *
     * @description:
     * The only one number should be displaied
     * if outgoing number equals to incoming number
     */
    hideSingleIncomingNumbers: function() {
        var self = this,
            phonesList = swc.models.Telephony.getPhonesList(),
            phones = _.union(phonesList.dect, phonesList.cord);

        _.each(phones, function(phone) {
            var activeIncomingLines = _.where(phone.incomingLines, {'value': true}),
                activeIncoming = (activeIncomingLines.length > 0),
                singleNumber = (activeIncomingLines.length === 1) && (phone.externalNumber.toString() === activeIncomingLines[0].number.toString());

            self.$('.expandable-list-item.' + phone.id + ' .outgoing-number .left-part .state-default').toggleClass('hidden', singleNumber);
            self.$('.expandable-list-item.' + phone.id + ' .incoming-number .left-part .state-default').toggleClass('hidden', singleNumber );
            self.$('.expandable-list-item.' + phone.id + ' .incoming-number').toggleClass('hidden', !activeIncoming || singleNumber);

            /*
               if more than one incoming line are active
               then visibility for its DOM element values should
               be set according to its model values
            */
            if (activeIncoming && !singleNumber) {
                _.each(phone.incomingLines, function(phoneLine) {
                    self.$('.expandable-list-item.' + phone.id + ' .incoming-number .right-part .state-default.' + phoneLine.line).toggleClass('hidden', !phoneLine.value);
                });
            }
        });
    },

    /*
    * Start pairing process.
    * Before start the session should be checked(just any AJAX call will do)
    */
    pairDect: function() {
        var self = this;

        swc.models.System.checkSessionState(function () {
            self._pairDect();
        });
    },

    /**
     * Pairing process handler: if the amount of connected DECT phones less then allowable limit(4 devices)
     * then open modal window with progress bar and start the connection process themself
     * else open modal window with alert
     *
     * @private
     */
    _pairDect: function() {
        var self = this;

        if (((swc.models.Telephony.get('phones').length - 2) < swc.models.Telephony.dectInternalNumbers.length) &&
            (swc.models.Telephony.get('phones').where({ deviceType:'dect' }).length < 4)) {

            var pairingModal = new swc.constructors.DectPairingModalView();

            $(pairingModal.el).on('hidden', function(){
                $(pairingModal.el).remove();
                self.render();
            });
        } else {
            var modal = swc.Templates.get('telephony:device:pair:modal:manyDects').get('content'),
                div = $('<div/>', {'class':'modalWindow modal manyDects-modal'});

            div.html($.tmpl(modal, {
                localeStrings: swc.models.Locale.getLocaleStrings("telephony"),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            }));

            div.on('hidden', function(){
                div.remove();
            });

            div.find('.close-modal').on('click', function() {
                $('.manyDects-modal').modal('hide');
            });

            div.modal();
        }
    },

    /**
     * Unpair selected DECT phone
     *
     * @param e
     */
    unpairPhone: function(e) {
        var self = this,
            dataKey = $(e.target).closest('.expandable-list-item').attr('data-key');

        $.when(swc.models.Telephony.unpairDect(dataKey)).done(function() {
            self.render();
        });
    },

    /**
     * Make test call
     *
     * @param e
     */
    ringPhone: function(e) {
        var dataKey = $(e.target).closest('.expandable-list-item').attr('data-key');

        swc.models.Telephony.ringPhone(dataKey);
    },

    /**
     * Rename selected DECT phone
     *
     * @param e
     */
    renameDect: function(e) {
        var self = this,
            item = $(e.target).closest('.expandable-list-item'),
            dataKey = item.attr('data-key');

        // update the default element with current value
        item.find('.device-name .state-default .text').text(e.target.value);

        swc.models.Telephony.get('phones').get(dataKey).set({
            'name': e.target.value,
            'is_changed': true
        });
    },

    /**
     * Event handler: is fired on change of internal number
     *
     * @param e {Event}
     * @param value {String}
     */
    changeInternalNumber: function(e, value) {
        var element = $(e.target).closest('.expandable-list-item'),
            deviceId = element.attr('data-key');

        // update the default element with current value
        element.find('.default-internal').text(value);

        swc.models.Telephony.get('phones').get(deviceId).set({
            'directoryNumber': value,
            'is_changed': true
        }, { silent: true });

        // refresh phone numbers in the dropdown list
        _.defer(this.fillDropDowns, this);

        this.setButtonsState();
    },

    /**
     * Event handler: is fired on change of external number
     *
     * @param e {Event}
     * @param value {String}
     */
    changeExternalNumber: function(e, value) {
        var element = $(e.target).closest('.expandable-list-item'),
            deviceId = element.attr('data-key');

        // update the default element with current value
        element.find('.default-external').text(value);

        swc.models.Telephony.get('phones').get(deviceId).set({
            externalNumber: value,
            is_changed: true
        }, { silent: true });

        // refresh phone numbers in the dropdown list
        _.defer(this.fillDropDowns, this);

        this.setButtonsState();
    },

    /**
     * This method do the following actions:
     * - closes all sibling item in the current expandable list
     * (i.e. expandable list which contains editable item)
     * - closes all items in the sibling expandable lists
     *
     * @param item {jQquery element}
     * @param phoneType {String} dect | wired
     */
    closeSiblingItems: function(item, phoneType) {
        var self = this,
            restPhoneTypes = _.without(this.phoneTypes, phoneType);

        // Close edit mode of sibling tabs in the expandable list which contains editable item
        _.each(item.siblings('.expandable-list-item'), function(item) {
            var dataKey = $(item).data('key');
            self.$('.expandable-' + phoneType + '-phones .expandable-list').trigger('expandable:swc-no-edit', dataKey);
        });

        // Close edit mode of all tabs in the sibling expandable lists
        // (i.e. lists which don't contain editable item)
        _.each(restPhoneTypes, function(restPhoneType) {
            var expandableList = self.$('.expandable-' + restPhoneType + '-phones .expandable-list');

            _.each(expandableList.find('.expandable-list-item'), function(item) {
                var dataKey = $(item).data('key');
                expandableList.trigger('expandable:swc-no-edit', dataKey);
            });
        });
    },

    /**
     * Event handler: is fired on the end of edit of the list item
     *
     * @param e {Event}
     * @param value {String}
     */
    listDefaultMode: function(e, value) {
        var self = this,
            target = this.$('.expandable-list-item[class~="' + value + '"]'),
            validation = this.pageValidation(null, true);

        $.when(validation)
            .done(function() {
                self.revertItemFromEditMode($(target));
                self.hideSingleIncomingNumbers();
            })
            .fail(function() {
                target.addClass('mode-editing');
            });
    },

    /**
     * Event handler: is fired on the start of edit of the list item
     *
     * @param e {Event}
     * @param value {String}
     */
    listEditMode: function(e, value) {
        var self = this,
            target = this.$('.expandable-list-item[class~="' + value + '"]'),
            validation = this.pageValidation(null, true),
            targetList;

        // if some of the elements won't pass validation the opportunity
        // to change editable element should be interrupted
        $.when(validation)
            .done(function() {
                if (target.closest('.expandable-list').hasClass('dect')) {
                    targetList = 'dect';
                    self.dectEditing = value;
                } else {
                    targetList = 'wired';
                    self.wiredEditing = value;
                }

                self.setItemToEditMode(target);
                self.closeSiblingItems(target, targetList);
                self.showSingleIncomingNumbers(value);
            })
            .fail(function() {
                target.removeClass('mode-editing');
            });
    },

    /**
     * Add suffix '-editable' to elements and appropriate error blocks
     * which are located inside the jQuery element item
     *
     * This manipulation is required to have validation only for elements which
     * are edited at this moment
     *
     * @param item {jQuery element} e.g. $('.expandable-list-item[class~="' + value + '"]')
     */
    setItemToEditMode: function(item) {
        // add '-editable' to dect-name
        item.find('.dect-name').attr('name', 'dect-name-editable');
        item.find('.validation-message[data-parameter-name="dect-name"]').attr('data-parameter-name', 'dect-name-editable');
        // add '-editable' to internal-number
        item.find('.internal-number').attr('data-name', 'internal-number-editable');
        item.find('.internal-number').data('name', 'internal-number-editable');
        item.find('.validation-message[data-parameter-name="internal-number"]').attr('data-parameter-name', 'internal-number-editable');
    },

    /**
     * Remove suffix '-editable' from elements and appropriate error blocks
     * which are located inside the jQuery element item
     *
     * This manipulation is required to have validation only for elements which
     * are edited at this moment
     *
     * @param item {jQuery element} e.g. $('.expandable-list-item[class~="' + value + '"]')
     */
    revertItemFromEditMode: function(item) {
        // remove '-editable' from dect-name
        item.find('.dect-name').attr('name', 'dect-name');
        item.find('.validation-message[data-parameter-name="dect-name-editable"]').attr('data-parameter-name', 'dect-name');
        // remove '-editable' from internal-number
        item.find('.internal-number').attr('data-name', 'internal-number');
        item.find('.internal-number').data('name', 'internal-number');
        item.find('.validation-message[data-parameter-name="internal-number-editable"]').attr('data-parameter-name', 'internal-number');
    },

    /**
     * Fill dropdown menus with phone numbers
     */
    fillDropDowns: function(self) {
        var internalNumbersCopy,
            dectExternalNumbers = [],
            // Possible internal numbers for DECT phones
            dectInternalNumbers = _.clone(swc.models.Telephony.dectInternalNumbers),
            phones = swc.models.Telephony.get('phones'),
            phoneLines = swc.models.Telephony.get('voip').where({'enable': 'Enabled'}),
            usedInternalNumbers = _.map(phones.pluck('directoryNumber'), function(v) { return +v; }),
            // internal numbers which are not occupied by any DECT phone
            freeInternalNumbers = _.difference(dectInternalNumbers, usedInternalNumbers),

            numberDropdownConstructor = function(array) {
                var obj = {};
                _.each(array, function(number) {
                    obj[number] = {
                        'name': number,
                        'value': number
                    };
                });
                return obj;
            };

        _.each(phoneLines, function(model) {
            dectExternalNumbers.push(model.get('directoryNumber'));
        });

        phones.each(function(model) {
            // Reset current list of internal numbers up to free numbers
            internalNumbersCopy = _.clone(freeInternalNumbers);
            // Add currently used number to array
            internalNumbersCopy.push(parseInt(model.get('directoryNumber'), 10));
            internalNumbersCopy.sort();

            self.$('.swc-dropdown.internal-number[data-class="internal-number-' + model.id + '"]').data({
                "options": numberDropdownConstructor(internalNumbersCopy)
            })
            .trigger('swc-dropdown:swc-change', model.get('directoryNumber'));

            // if there is more than one external number
            if (dectExternalNumbers[1]) {
                self.$('.swc-dropdown.external-number[data-class="external-number-' + model.id + '"]').data({
                    "options": numberDropdownConstructor(dectExternalNumbers)
                })
                .trigger('swc-dropdown:swc-change', model.get('externalNumber'));
            }
        });
    },

    onCancel: function() {
        this.showSuccessMessage = false;
        this.emptyEditing();
    },

    save: function() {
        var self = this,
            deferred = new $.Deferred();

        // Before doing heavy edit operation, which even has a callback delayed for 2 seconds (see phoneEdit() method),
        // let's check if session is still valid.
        swc.models.System.checkSessionState(function(){
            self.$('.expandable-list-item .validation-message').hide();
            self.$('.swc-input').removeClass('validation-error');

            self.emptyEditing();
            self.showPageLoading("Saving page data..");
            
            $.when(swc.models.Telephony.phoneEdit())
                .done(function () {
                    self.showSuccessMessage = true;
                    deferred.resolve();
                })
                .fail(function () {
                    deferred.reject();
                });
        });

        return deferred.promise();
    }
});
