swc.constructors.TelephonyPhonebookEditView = swc.base.PageView.extend({
    className: 'phonebook-new-contact',

    models: ['Phonebook'],

    validation: {
        'firstName':  'Phonebook:firstName',
        'lastName':   'Phonebook:lastName',
        'homeNumber': 'Phonebook:homeNumber',
        'cellNumber': 'Phonebook:cellNumber',
        'workNumber': 'Phonebook:workNumber'
    },

    events: {
        'keyup .contact-number': 'onChangeContact',
        'keyup .contact-name': 'onChangeContact'
    },

    phoneSet: ['HOME', 'CELL', 'WORK'],

    model: null,

    initialize: function() {
        swc.base.PageView.prototype.initialize.call(this, arguments);
        this.template = $.template("pageContent", swc.Templates.get('telephony:phonebook:new-number').get('content'));
    },

    /**
     * Event handler: is fired on changing value in the input field
     *
     * @description:
     *
     * this wrapper is used to defer validation check until all events will be triggered
     * this approach is similar to use setTimeout(func, 0)
     */
    onChangeContact: function() {
        _.defer(this.changeContact, this);
    },

    /**
     * Handle data during user input in the edit form of items in the phonebook
     *
     * @param e {Object} -> Event
     */
    changeContact: function(self) {
        var model = self.getContactModel(),
            elements = self.getElements(),
            elementParams = [];

        _.each(elements, function(element) {
            elementParams.push(getParameter($(element)));
        });

        model.set({'modified': !self.pageCheckDefaultValues()});

        // fill contact model by first, last names
        model.set({
            firstName: $.trim(_.where(elementParams, { parameterName: 'firstName' })[0].parameterValue),
            lastName: $.trim(_.where(elementParams, { parameterName: 'lastName' })[0].parameterValue)
        });

        // fill contact model by phone numbers
        _.each(self.phoneSet, function(phoneType) {
            var numberObj = _.where(model.get('telephoneNumbers'), { type: phoneType })[0],
                numberValue = $.trim(_.where(elementParams, { parameterName: phoneType.toLowerCase() + 'Number'})[0].parameterValue);

            if (!numberObj) {
                model.get('telephoneNumbers').push({
                    name: numberValue,
                    preferred: false,
                    type: phoneType
                });
            } else {
                numberObj.name = numberValue;
            }
        });
    },

    /**
     * Validation for form 'Add new contact' checks if the first or last name is empty
     *
     * @returns {boolean}
     */
    onSaveValidation: function() {
        var model = this.getContactModel(),
            fullName = model.get('firstName') + model.get('lastName'),
            emptyNumbers = _.where(model.get('telephoneNumbers'), { name: '' }),
            allNamesAreEmpty = _.isEmpty(fullName),
            allNumbersAreEmpty = (emptyNumbers.length === 3),
            validation = {};

        if (allNamesAreEmpty && allNumbersAreEmpty) {
            validation['allfields'] = {
                status: false,
                message: 'at least one field should be filled'
            };
            return validation;
        }

        if (allNamesAreEmpty) {
            validation['names'] = {
                status: false,
                message: 'at least one field should be filled'
            };
        }

        if (allNumbersAreEmpty) {
            validation['numbers'] = {
                status: false,
                message: 'at least one field should be filled'
            };
        }

        return validation;
    },

    /**
     * reset temporary data
     */
    resetContactModel: function() {
        this.model = null;
        sessionStorage.removeItem(swc.models.Phonebook.requestParamNameID);
    },

    onCancel: function() {
        this.resetContactModel();
        swc.router.navigate('telephony/phonebook');
    },

    save: function() {
        var self = this,
            model = this.getContactModel(),
            deferred = new $.Deferred(),
            onSaveErrors = this.onSaveValidation();

        if (_.isEmpty(onSaveErrors)) {
            $.when(model.save())
                .done(function() {
                    self.resetContactModel();
                    deferred.resolve();
                    swc.router.navigate('telephony/phonebook');
                });
        } else {
            deferred.reject(onSaveErrors);
        }

        return deferred.promise();
    },

    setButtonsState: function() {
        this.$('.button.save-changes').toggleClass('disabled', this.pageCheckDefaultValues());
    },

    /**
     * Show unsuccessfull page save result
     */
    showSaveSuccess: function(type, onSaveErrors) {
        var self = this;

        // if no error do nothing
        if (_.isEmpty(onSaveErrors)) {
            return;
        }

        this.clearValidationMessages();

        _.each(onSaveErrors, function(error, errorName) {
            var container = this.$('.validation-message[data-parameter-name="' + errorName + '"]');

            container.show();
            container.find('.error-message').hide();
            container.find('.error-message[data-error="' + error.message + '"]').show();
            self.highlightField(errorName);
        });
    },

    highlightField: function(errorName) {
        if (errorName === 'allfields') {
            this.$('.contact-name.validatable').addClass('validation-error');
            this.$('.contact-number.validatable:first').addClass('validation-error');
            this.$('.validation-message[data-parameter-name="allfields"]').show();
        }
        if (errorName === 'names') {
            this.$('.contact-name.validatable').addClass('validation-error');
            this.$('.validation-message[data-parameter-name="names"]').show();
        }
        if (errorName === 'numbers') {
            this.$('.contact-number.validatable:first').addClass('validation-error');
            this.$('.validation-message[data-parameter-name="numbers"]').show();
        }
    },

    /**
     * Set template data for output phone's types in the cycle
     */
    setTemplateData: function() {
        var model = this.getContactModel(),
            numbers = model.get('telephoneNumbers');

        // make object for editable number fields
        model.set('phoneNumbers', {
            'HOME': _.where(numbers, { type: 'HOME' })[0] ? _.where(numbers, { type: 'HOME' })[0].name : '',
            'CELL': _.where(numbers, { type: 'CELL' })[0] ? _.where(numbers, { type: 'CELL' })[0].name : '',
            'WORK': _.where(numbers, { type: 'WORK' })[0] ? _.where(numbers, { type: 'WORK' })[0].name : ''
        });

        this.templateData = {
            nameTypes: {
                'firstName': 'First Name',
                'lastName': 'Last Name'
            },
            phoneTypes: {
                'HOME': 'Home',
                'CELL': 'Mobile',
                'WORK': 'Office'
            },
            localeStrings: swc.models.Locale.getLocaleStrings("telephony"),
            localeString: getTranslationStringsjQuery,
            isNew: model.isNew(),
            contact: model
        };
    },

    getContactModel: function() {
        if (this.model === null) {
            var contactID;

            if (contactID = sessionStorage.getItem(swc.models.Phonebook.requestParamNameID)) {
                this.model = swc.models.Phonebook.get(contactID);
            } else {
                this.model = new swc.constructors.PhonebookContact();
            }
        }

        return this.model;
    }

});