swc.constructors.SystemSettingsResetView = swc.base.TabView.extend({

    className: 'reset',

    models: ['System', 'Telephony', 'apServiceState'],

    events: {
        'click .factory-reset':  'onFactoryReset',
        'click .save-config':    'onSaveConfig',
        'click .restore-config': 'onRestoreConfig'
    },
    
    preRender: function() {
        // We have to load model `apServiceLoadingState` only if AP is enabled
        if (swc.models.apServiceState.isEnabled()) {
            swc.models.apServiceLoadingState = new swc.constructors.apServiceLoadingState();
        }
    },

    /**
     * Handler when user clicks on backup button:
     *
     * @description:
     *
     *  When user clicks on backup button, he must be shown modal window, where he will choose backup option. If user will
     *  choose `backup to acs network` -> new RPC call will be invoked, else he will go through default procedure of backup
     */
    onSaveConfig: function(e) {
        var self = this;

        e.preventDefault();

        // before show modal window the session state should be checked
        swc.models.System.checkSessionState(function() {
            self.showModalSaveConfig();
        });
    },

    /**
     * Show save config modal window
     *
     * @description: show modal window with switcher:
     *  - save configuration in the Swisscom Customer Center (backupPlace = acs)
     *  - Save configuration in the file locally on your computer (backupPlace = local)
     *
     * After pushing button "Apply" the save process starts
     *
     * @localParam backupPlace {String} Place where the configuration
     *             settings will be saved (local | acs)
     */
    showModalSaveConfig: function() {
        var self = this;

        SWCElements.modalWindow.show({
            templateID: 'system:settings:modal-windows:backup-settings',
            tempalteData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery
            },
            className: 'backup-settings',

            onApply: function() {
                var backupPlace = $('.modalWindow.backup-settings').find('.backup-options').data('value');

                SWCElements.modalWindow.hide();
                self.processSaveConfig(backupPlace);
            }
        });
    },

    /**
     * Start process of data backup according to selected
     * place for saving configuration
     *
     * @param backupPlace {String} local | acs
     */
    processSaveConfig: function(backupPlace) {
        if (backupPlace === "local") {
            this.doBackupLocal();
        } else {
            this.doBackupAcs();
        }
    },

    /**
     * Handler when backup finishes
     *
     * @param status {Boolean} -> status of backup process (true - succes, false - fail)
     */
    onBackupComplete: function(status) {
        var messagesContainer = this.$('.buttons-container-message.save-config-message');

        // Remove loading window:
        swc.views.Application.stopPageLoading();

        // Display message about success / fail saving:
        if (status === true) {
            messagesContainer.find('.save-error').hide();
            messagesContainer.find('.save-success').show();
        } else {
            messagesContainer.find('.save-success').hide();
            messagesContainer.find('.save-error').show();
        }
    },

    /**
     * Create local backup file
     *
     * @returns {*}
     */
    doBackupLocal: function() {
        var self = this,
            deferred = new $.Deferred();

        // Create backup iframe and hide it by default:
        this.backupIframe = $.tmpl(swc.Templates.get('system:settings:backup:iframe').get('content'),
            { context: $.cookie(getDeviceID() + '/context') }
        );

        // Append it to body:
        $('body').append(this.backupIframe);

        // Display loading message:
        swc.views.Application.showPageLoading('Downloading Backup File');

        // Call RPC endpoint to create a backup file:
        $.when(swc.models.System.configBackup())
            .done(function() {
                self.onBackupComplete(true);
            })
            .fail(function() {
                self.onBackupComplete(false);
            });

        return deferred.promise();
    },

    /**
     * Save device settings to swisscom ACS network:
     */
    doBackupAcs: function() {
        var self = this;

        // Display loading message:
        swc.views.Application.showPageLoading('Downloading Backup File');

        // Call RPC endpoint to save settings to Swisscom ACS network:
        $.when(swc.models.System.configBackupACS())
            .done(function() {
                self.onBackupComplete(true);
            })
            .fail(function() {
                self.onBackupComplete(false);
            });
    },

    wrongFileModal: function() {
        var self = this;
        SWCElements.modalWindow.show({
            templateID: 'system:settings:restore:modal:restore-wrongfile',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            },
            className: 'system-restore',
            onCancel: function() {
                self.removeForm();
            }
        });
    },

    confirmModal: function() {
        var self = this;

        SWCElements.modalWindow.show({
            templateID: 'system:settings:restore:modal:restore-confirm',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate,
                filename: self.fileName
            },
            onShow: function(){
                var modal = $('.modalWindow');
                modal.find('.apply-changes').on('click', function (e) {
                    SWCElements.modalWindow.hide();
                    self.processRestoreConfig();
                    self.form.submit();
                });


            },
            className: 'system-restore',
            onCancel: function() {
                self.removeForm();
            }
        });
    },

    processRestoreConfig: function() {
        var self = this;
        
        SWCElements.modalWindow.show({
            templateID: 'system:settings:restore:modal:restore-process',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            },
            className: 'system-restore',
            onShow: function(){
                var element = $('.modalWindow'),
                    isSuccess = false,
                    restoreTime = swc.models.System.minExpectedRebootTime,
                    resetRuler;
                
                element.find('.ruler-block').html($.tmpl(swc.Templates.get('ruler').get('content'), {
                    'className': 'restore-ruler'
                }));
                
                // NOTE:
                // Don't try to move it into `var` initialization section,
                // because element will be present only if previous LOC executed 
                resetRuler = element.find('.restore-ruler');

                // NP throws custom event to document if upload was successful.
                // If we haven't received event for 2 minutes, or event was not successful, throw error
                $(document).off('sah:models:System:state');
                $(document).on('sah:models:System:state', function(e, options){
                    if(options === 'restore-success'){
                        isSuccess = true;
                    } else {
                        setTimeout(function(){
                            resetRuler.trigger('swc-ruler:finish');
                        }, 50);
                    }
                });

                var postRestoreCallback = function() {
                    self.removeForm();
                    // clean local storage with saved language
                    localStorage.removeItem('locale');
                    swc.models.System.doLogout();
                };

                resetRuler.on('swc-ruler:finish', function(){
                    if(isSuccess){
                        if (swc.models.apServiceState.isEnabled()) {
                            swc.models.System.whenApUp(postRestoreCallback);
                        } else {
                            swc.models.System.whenDeviceUp(postRestoreCallback);
                        }
                    } else {
                        SWCElements.modalWindow.hide();
                        self.errorModal();
                    }
                });

                setTimeout(function(){
                    resetRuler.trigger('swc-ruler:start', {time: restoreTime});
                }, 0);
            }
        });
    },

    errorModal: function() {
        var self = this;
        SWCElements.modalWindow.show({
            templateID: 'system:settings:restore:modal:restore-error',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings(self.pageTemplateID),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            },
            className: 'system-restore',
            onCancel: function() {
                self.removeForm();
            }
            /*
             onApply: function(){
             SWCElements.modalWindow.hide();
             self.confirmModal();
             }
             */
        });
    },

    removeForm: function(){
        if(this.restoreIframe){
            this.restoreIframe.remove();
        }
        if(this.saveIframe){
            this.saveIframe.remove();
        }
    },

    onRestoreConfig: function() {
        this.restoreConfig();
    },

    restoreConfig: function() {
        var self = this;

        this.restoreIframe = $.tmpl(swc.Templates.get('system:settings:restore:iframe').get('content'), {});
        this.form = $('#restore-form');

        this.restoreIframe.hide();
        $('body').append(this.restoreIframe);

        this.form.find('input[name="context"]').val($.cookie(getDeviceID() + '/context'));

        this.form.find('#config-restore').on('change', function(){
            var filePath = self.form.find('input[type=file]').val(), fileExtension;

            if(filePath.indexOf('\\') < 0){
                self.fileName = filePath;
            } else {
                self.fileName = filePath.slice(filePath.lastIndexOf("\\")+1, filePath.length);
            }

            fileExtension = self.fileName.slice(self.fileName.lastIndexOf(".")+1, self.fileName.length);

            if(self.fileName && fileExtension === 'cfg') {
                // before show modal window the session state should be checked
                swc.models.System.checkSessionState(function() {
                    self.confirmModal();
                });
            } else {
                self.wrongFileModal();
            }
        });
    },

    /**
     * Event handler: is fired on click button 'Reset to factory settings'
     *
     * @param e
     */
    onFactoryReset: function(e) {
        var self = this;

        e.preventDefault();

        // before show modal window the session state should be checked
        swc.models.System.checkSessionState(function() {
            self.showModalFactoryReset();
        });
    },

    /**
     * Show factory reset modal window
     *
     * @description: show modal window with switcher
     *  - reset with saving DECT pairing
     *  - complete reset
     *
     *  After pushing button "Apply" this modal window closes and
     *  the modal window with progress bar appears. The reset process
     *  is started in background
     */
    showModalFactoryReset: function() {
        var self = this;

        SWCElements.modalWindow.show({
            templateID: 'system:settings:reset:modal',
            tempalteData: {
                localeStrings: swc.models.Locale.getLocaleStrings("system"),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            },
            className: 'reset-gateway',

            onApply: function() {
                var resetType = $('.modalWindow.reset-gateway').find('.reset-type').data('value'),
                    resetDect = resetType === 'with';

                SWCElements.modalWindow.hide();
                self.processFactoryReset(resetDect);
            }
        });
    },

    /**
     * Show modal window with progress bar and start reset device process.
     * When the device is up then modal window disappears
     *
     * @param resetDect
     */
    processFactoryReset: function(resetDect) {
        var self = this;

        SWCElements.modalWindow.show({
            templateID: 'system:settings:reset-process:modal',
            templateData: {
                localeStrings: swc.models.Locale.getLocaleStrings("system"),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate
            },
            className: 'system-restore',
            onShow: function(){
                var element = $('.modalWindow'),
                    resetTime = swc.models.System.minExpectedRebootTime,
                    resetRuler;

                element.find('.ruler-block').html($.tmpl(swc.Templates.get('ruler').get('content'), {
                    'className': 'reset-ruler'
                }));

                resetRuler = element.find('.reset-ruler');

                // hide modal window only when device is up
                swc.models.System.on('device:up', function () {
                    SWCElements.modalWindow.hide();
                });

                setTimeout(function(){
                    resetRuler.trigger('swc-ruler:start', {time: resetTime});
                }, 10);

                // if resetDect is true, we reset dect at first. Otherwise, start reset gateway immediately
                if(resetDect){
                    $.when(swc.models.Telephony.dectReset(), swc.models.Telephony.dectRemoveDects()).done(function(){
                        swc.models.System.resetGateway(resetTime);
                    });
                } else {
                    swc.models.System.resetGateway(resetTime);
                }
            }
        });
    }

});
