swc.constructors.NetworkSettingsPortforwardingAddView = swc.base.TabView.extend({

    className: 'port-forwarding-add',

    allowedMods: [
        'expert'
    ],

    models: [
        'PortForwarding',
        'NetworkDevices'
    ],

    validation: {
        'deviceIP': 'PortForwarding:deviceIP',
        'serviceName': 'PortForwarding:serviceName',
        'predefinedServiceName': 'PortForwarding:predefinedServiceName',
        'entryPort': 'PortForwarding:entryPort',
        'destinationPort': 'PortForwarding:destinationPort'
    },

    events: {
        'swc-radio-buttons:change .swc-radio-buttons.service-mode': 'setServiceAddingMode',
        'swc-radio-buttons:change .swc-radio-buttons.entry-mode': 'setEntryPortMode',
        'swc-radio-buttons:change .swc-radio-buttons.destination-mode': 'setDestinationPortMode',

        'keyup input[name="entryPort"].from': 'updateDestinationPorts',
        'keyup input[name="entryPort"].to': 'updateDestinationPorts',

        'swc-dropdown:change .swc-dropdown.service-name-selection': 'setPredefinedValues',
        'swc-dropdown:change .swc-dropdown.device-selection': 'postDeviceSelectionValidation'
    },

    setTemplateData: function() {
        this.templateData = {
            serviceMode: "custom", // Default option {'Create your own'}
            servicePred: "FTP",    // Default option {'Predefined rule: FTP'}
            name: "",
            port: {
                entry: [],
                destination: []
            },
            portMode: {
                entry: "single", // Default option {'Single'}
                destination: "single"  // Default option {'Single'}
            },
            device: "0.0.0.0",      // Default option {'Select a device'}
            protocol: "6,17"        // Default option {'TCP/UDP'}
        };
    },

    /**
     * @override on {'swc.BaseView.initialize()'} method
     *
     * Need to set new template in order to support /1/2/3/4/.../n/ lvls of views including.
     *
     * @returns {boolean}
     */
    initialize: function() {
        // Prepare template for current page:
        this.template = $.template("pageContent", swc.Templates.get('network:settings:port-forwarding:rules-add').get('content'));

        // Extend global events:
        this.events = _.extend({}, swc.base.PageView.prototype.events, this.events);
    },

    /**
     * @override on {'swc.BaseView.pageCheckDefaultValues()'} method
     *
     * Need to return always true, becuase this page must not check for editing
     *
     * @returns {boolean}
     */
    pageCheckDefaultValues: function() {
        return true;
    },

    /**
     * Override from base class - never mark this page as changed.
     * FIXME: After implementation of 2-way binding we have to fix this method and check if any of model changed
     * 
     * Possible workarounds: remove and global hasChanged() will handle case,
     * but for that we need to FIXME: implement disabling/enabling of "Save" button. 
     * @returns {boolean}
     */
    hasChanged: function(){
        return false;
    },

    /**
     * @override
     *
     * @description
     *
     *  Need to keep buttons enabled all time
     *
     * @param e
     * @param toDisable
     */
    setButtonsState: function(e, toDisable) {
        var container = this.$('.buttons-container-message'),
            buttonSave = container.find('.button.save-changes'),
            buttonCancel = container.find('.button.cancel-changes');

        buttonSave.removeClass('disabled');
        buttonCancel.removeClass('disabled');
    },

    /**
     * @override on {'swc.BaseView.renderComplete()'} method
     *
     * Need to setup some values when page is ready to be rendered
     *
     * @returns {boolean}
     */
    renderComplete: function() {
        this.setDefaultValues();
    },

    /**
     * Get values from Ports inputs. Convert undefined values to 0
     *
     * @param type {String}
     * @returns {Array of Strings}
     */
    getPortsValuesFromView: function(type) {
        var inputEntryFrom = this.$('input[name="entryPort"].from'),
            inputEntryTo = this.$('input[name="entryPort"].to'),
            inputDestinationPortFrom = this.$('input[name="destinationPort"].from'),
            inputDestinationPortTo = this.$('input[name="destinationPort"].to'),
            portValues = {
                entry: [
                    !$.trim(inputEntryFrom.val()) ? "" : +inputEntryFrom.val(),
                    !$.trim(inputEntryTo.val()) ? "" : +inputEntryTo.val()
                ],
                destination: [
                    !$.trim(inputDestinationPortFrom.val()) ? "" : +inputDestinationPortFrom.val(),
                    !$.trim(inputDestinationPortTo.val()) ? "" : +inputDestinationPortTo.val()
                ]
            };

        return !_.isUndefined(portValues[type]) ? portValues[type] : [];
    },

    /**
     * Reset form elements to default values.
     *
     * @description
     *
     *      Default value for fields when adding rule are following:
     *
     *          Service - Create Your Own
     *          Entry Port - Single
     *          Protocol - TCP/UDP
     *          To Device - Choose a device
     *          Destination Port - Single
     *
     *      All input fields have to be empty
     */
    setDefaultValues: function() {
        this.setPredefinedRulesOptions();
        this.setAssignedDeviceOptions();
        this.setProtocolOptions();

        // Set service name field to default value:
        this.$('input[name="serviceName"]').val(this.templateData.name);

        // Set entry ports fields to default values:
        if (!_.isUndefined(this.templateData.port.entry[0])) {
            this.$('input[name="entryPort"].from').val(this.templateData.port.entry[0]);
        } else {
            this.$('input[name="entryPort"].from').val("");
        }

        if (!_.isUndefined(this.templateData.port.entry[1])) {
            this.templateData.portMode.entry = "range";

            this.$('input[name="entryPort"].to').val(this.templateData.port.entry[1]);
            this.$('.entry-port-values-section').addClass("range-type");
        } else {
            this.$('input[name="entryPort"].to').val("");
            this.$('.entry-port-values-section').removeClass("range-type");
            this.$('.swc-radio-buttons.destination-mode .mode[data-value="range"]').addClass('disabled');
        }

        // Set destination ports fields to default values:
        if (!_.isUndefined(this.templateData.port.destination[0])) {
            this.$('input[name="destinationPort"].from').val(this.templateData.port.destination[0]);
            this.$('.text-block-destination-from').text(this.templateData.port.destination[0]);
        } else {
            this.$('input[name="destinationPort"].from').val("");
            this.$('.text-block-destination-from').text("");
        }

        if (!_.isUndefined(this.templateData.port.destination[1])) {
            this.templateData.portMode.destination = "range";

            this.$('input[name="destinationPort"].to').val(this.templateData.port.destination[1]);
            this.$('.text-block-destination-to').text(this.templateData.port.destination[1]);
            this.$('.destination-port-values-section').addClass("range-type");
        } else {
            this.$('input[name="destinationPort"].to').val("");
            this.$('.text-block-destination-to').text("");
            this.$('.destination-port-values-section').removeClass("range-type");
        }

        // Reset radio buttons state to default
        this.$('.swc-radio-buttons.entry-mode')
            .trigger('swc-radio-buttons:swc-change', this.templateData.portMode.entry);

        this.$('.swc-radio-buttons.destination-mode')
            .trigger('swc-radio-buttons:swc-change', this.templateData.portMode.destination);

        // Reset dropdowns state to default
        this.$('.swc-dropdown.protocol-selection')
            .trigger('swc-dropdown:swc-change', this.templateData.protocol);

        this.$('.swc-dropdown.device-selection')
            .trigger('swc-dropdown:swc-change', this.templateData.device);

        // Remove all previous validation messages:
        this.clearValidationMessages();
    },

    /**
     * Handler for resetting fields to predefined rule values:
     *
     * @description
     *
     *      After selecting predefined options, all the fields and inputs in the page MUST be filled with port's values
     *
     * @param e {Object} Event
     * @param value {String}
     */
    setPredefinedValues: function(e, value) {
        var serviceName = !_.isUndefined(value) ? value : this.templateData.servicePred,
            service = swc.models.PortForwarding.predefinedServices[serviceName],
            servicePort = service.port.split('-'),
            entryRadioButtons = this.$('.swc-radio-buttons.entry-mode'),
            destinationRadioButtons = this.$('.swc-radio-buttons.destination-mode');

        // Update inputs:
        this.$('input[name="entryPort"].from').val(servicePort[0]);
        this.$('input[name="destinationPort"].from').val(servicePort[0]);
        this.$('.text-block-destination-from').text(servicePort[0]);

        if (!_.isUndefined(servicePort[1])) {
            this.$('.entry-port-values-section').addClass("range-type");
            this.$('.destination-port-values-section').addClass("range-type");

            this.$('input[name="entryPort"].to').val(servicePort[1]);
            this.$('input[name="destinationPort"].to').val(servicePort[1]);
            this.$('.text-block-destination-to').text(servicePort[1]);
        }

        // Set correct radio buttons position for entry ports:
        if (servicePort.length === 2) {
            entryRadioButtons.trigger('swc-radio-buttons:swc-change', "range");
            destinationRadioButtons.trigger('swc-radio-buttons:swc-change', "range");

            this.setEntryPortMode(null, "range");
            this.setDestinationPortMode(null, "range");
        } else {
            entryRadioButtons.trigger('swc-radio-buttons:swc-change', "single");
            destinationRadioButtons.trigger('swc-radio-buttons:swc-change', "single");

            this.setEntryPortMode(null, "single");
            this.setDestinationPortMode(null, "single");
        }

        // Update protocol:
        this.$('.swc-dropdown.protocol-selection').trigger('swc-dropdown:swc-change', service.protocol);

        // Remove all previous validation messages:
        this.clearValidationMessages();
    },

    /**
     * Handler for switching between Create your own / Predefined modes
     *
     * @descriptions
     *
     *      IF on add rule page the User will chose "Create Your Own" option THEN all field MUST have either default value
     *      (if any) or be empty, user MUST be able to change any field on the page
     *
     *      IF on add rule page the User will chose "Predefned Rule" option THEN in Service dropdown menu he MUST be able
     *      to select it from the list (see {'swc.collections.PortForwarding.defaultServices'}).
     *
     * @param e {Object} Event
     * @param value {String}
     */
    setServiceAddingMode: function(e, value) {
        var type = !_.isUndefined(value) ? value : this.templateData.serviceMode;

        if (type === "predefined") {
            this.setDefaultValues();
            this.setPredefinedValues();
        } else {
            this.setDefaultValues();
        }

        this.$('.rule-name-section').toggleClass("type-predefined", value === "predefined");
    },

    /**
     * Handler for switching between Single / Range modes for entry ports:
     *
     * @description
     *
     *      IF in Entry Port user will choose {'Single'} option THEN User MUST be able to select ONLY Single option in
     *      Destination Port, Range option MUST be disabled, the User MUST be able to edit port number
     *
     *      IF in Entry Port user will choose Range option THEN user MUST be able to select BOTH Single and Range option
     *      in Destination Port
     *
     *      IF in Entry Port user will choose Range option AND choose Single option in Destination Port THEN ports number
     *      from Entry Ports MUST be copied to Destination Port, the User MUST be able to edit this port number
     *
     *      IF in Entry Port user will choose Range option AND choose Range option in Destination Port THEN ports range
     *      from Entry Ports MUST be copied to Destination Port range, the User MUST NOT be able to edit this ports numbers
     *
     *      IF the User will enter port number in Entry port field with Single option chosen and after will switch to Range
     *      option THEN second field MUST be empty in order the User will be able to enter port number
     *
     * @param e {Object} Event
     * @param value {String}
     */
    setEntryPortMode: function(e, value) {
        var entryPorts = this.getPortsValuesFromView('entry'),
            inputEntryPortFrom = this.$('input[name="entryPort"].from'),
            inputEntryPortTo = this.$('input[name="entryPort"].to'),
            inputDestinationPortFrom = this.$('input[name="destinationPort"].from'),
            inputDestinationPortTo = this.$('input[name="destinationPort"].to'),
            spanDestinationFrom = this.$('.text-block-destination-from'),
            spanDestinationTo = this.$('.text-block-destination-to'),
            destinationRadioButtons = this.$('.swc-radio-buttons.destination-mode'),
            destinationValuesSection = this.$('.destination-port-values-section'),
            destinationRadioButtonsRangeMode = this.$('.swc-radio-buttons.destination-mode .mode[data-value="range"]');

        if (value === "range") {
            // Enable if was disabled range mode for destination ports:
            destinationRadioButtonsRangeMode.removeClass('disabled');
            destinationRadioButtons.trigger('swc-radio-buttons:swc-change', "range");
            destinationValuesSection.addClass("range-type");

            // When changed by user
            if (e) {
                // Copy entry port {'from'} and {'to'} to destination ports:
                inputDestinationPortFrom.val(entryPorts[0]);
                inputDestinationPortTo.val(entryPorts[1]);

                // Copy values to spans:
                spanDestinationFrom.text(entryPorts[0]);
                spanDestinationTo.text(entryPorts[1]);
            }

        } else {
            // Set range mode to single for destination ports:
            destinationRadioButtons.trigger('swc-radio-buttons:swc-change', "single");
            destinationRadioButtonsRangeMode.addClass('disabled');
            destinationValuesSection.removeClass("range-type");

            // Copy entry port {'from'} to destination port (when changed by user):
            if (e) {
                inputDestinationPortFrom.val(entryPorts[0]);

                // Blink field on update
                this.notifyPortsUpdate(inputDestinationPortFrom);
            }

            // Re-validate field to clear message from range ports:
            if (!_.isEmpty(inputEntryPortFrom.val())) {
                this.pageValidation(inputEntryPortFrom, false);
                this.pageValidation(inputEntryPortTo, false);
                this.pageValidation(inputDestinationPortFrom, false);
            }
        }

        this.$('.entry-port-values-section').toggleClass("range-type", value === "range");
    },

    /**
     * Handler for switching between Single / Range modes for destination ports:
     *
     * @description
     *
     *      There are no strict conditions for this handler, if user will select single - he will be able only to edit
     *      {'from'} destination port value. Either if he will select Range - he will be able to edit both.
     *
     * @param e {Object} Event
     * @param value {String}
     */
    setDestinationPortMode: function(e, value) {
        var entryPorts = this.getPortsValuesFromView('entry'),
            inputDestinationPortFrom = this.$('input[name="destinationPort"].from'),
            inputDestinationPortTo = this.$('input[name="destinationPort"].to'),
            spanDestinationFrom = this.$('.text-block-destination-from'),
            spanDestinationTo = this.$('.text-block-destination-to');

        if (value === "range") {

            // when changed by user
            if (e) {
                // Copy entry port {'from'} and {'to'} to destination ports:
                inputDestinationPortFrom.val(entryPorts[0]);
                inputDestinationPortTo.val(entryPorts[1]);

                // Copy values to spans:
                spanDestinationFrom.text(entryPorts[0]);
                spanDestinationTo.text(entryPorts[1]);
            }
        } else {

            // when changed by user
            if (e) {
                // Copy entry port {'from'} to destination ports:
                inputDestinationPortFrom.val(entryPorts[0]);

                // Blink field on update
                this.notifyPortsUpdate(inputDestinationPortFrom);
            }

            // Re-validate field to clear message from range ports:
            this.pageValidation(inputDestinationPortFrom, false);
        }

        this.$('.destination-port-values-section').toggleClass("range-type", value === "range");
    },

    /**
     * Handler for settings values for {'Select device'} dropdown
     *
     * @description
     *
     *      Default value: Select Device
     *
     *      MUST contain the list of currently connected and recently connected devices
     */
    setAssignedDeviceOptions: function() {
        var dropdown = this.$('.swc-dropdown.device-selection'),
            options = {
                "0.0.0.0": {
                    name: getTranslationStrings("Choose a device"),
                    value: "0.0.0.0",
                    isDefault: true
                }
            };

        // Define which devices to show in dropdown:
        _.each(swc.models.NetworkDevices.models, function(model, key) {
            var ipAddress = model.get('address')['IPv4'];

            // In this list can be devices which have IPv4 Address:
            if (!_.isEmpty(ipAddress) && !_.isUndefined(ipAddress)) {
                options[ipAddress] = {
                    name: model.get('name'),
                    value: ipAddress,
                    additionalLabel: ipAddress
                };
            }
        });

        // Set dropdown data:
        dropdown.data('options', options);
        dropdown.data('hide-default', true);
        dropdown.trigger('swc-dropdown:swc-change', this.templateData.device);
    },

    /**
     * Handler for settings values for {'Protocol'} dropdown
     *
     * @description
     *
     *      Default value: TCP/UDP
     *
     *      Values of Protocol dropdown menu
     *          - TCP/UDP
     *          - TCP
     *          - UDP
     */
    setProtocolOptions: function() {
        var dropdown = this.$('.swc-dropdown.protocol-selection'),
            options = [
                { name: getTranslationStrings("TCP/UDP"), value: "6,17" },
                { name: getTranslationStrings("TCP"),     value: "6"    },
                { name: getTranslationStrings("UDP"),     value: "17"   }
            ];

        // Set dropdown data:
        dropdown.data('options', options);
        dropdown.trigger('swc-dropdown:swc-change', this.templateData.protocol);
    },

    /**
     * Handler for settings values for {'Predefined Rules'} dropdown
     *
     * @description
     *
     *      Default value: FTP
     *
     *      See list of rules in
     */
    setPredefinedRulesOptions: function() {
        var dropdown = this.$('.swc-dropdown.service-name-selection'),
            services = swc.models.PortForwarding.predefinedServices,
            options = [];

        // Go through each service and create options for dropdown
        _.each(services, function(service, key) {
            options.push({name: key, value: key});
        });

        // Set dropdown data:
        dropdown.data('options', options);
        dropdown.trigger('swc-dropdown:swc-change', this.templateData.servicePred);
    },

    /**
     * Handle for updating Destination ports, when Entry ports inputs are changing:
     *
     * @param e {Object}
     * @param value {String}
     */
    updateDestinationPorts: function(e) {
        var entryInput = $(e.target),
            entryPorts = this.getPortsValuesFromView('entry'),
            inputDestinationPortFrom = this.$('input[name="destinationPort"].from'),
            inputDestinationPortTo = this.$('input[name="destinationPort"].to'),
            spanDestinationFrom = this.$('.text-block-destination-from'),
            spanDestinationTo = this.$('.text-block-destination-to');

        // Blink field on update
        this.notifyPortsUpdate(inputDestinationPortFrom);

        // Update correct parts of inputs and text blocks
        if (entryInput.hasClass("from")) {
            inputDestinationPortFrom.val(entryPorts[0]);
            spanDestinationFrom.text(entryPorts[0]);
        } else {
            inputDestinationPortTo.val(entryPorts[1]);
            spanDestinationTo.text(entryPorts[1]);
        }

        this.pageValidation(inputDestinationPortFrom, false);
    },

    /**
     * Validate destionation ports after device has changed, because they are assigned to correct device
     */
    postDeviceSelectionValidation: function() {
        var inputDestinationPortFrom = this.$('input[name="destinationPort"].from'),
            inputDestinationPortTo = this.$('input[name="destinationPort"].to');

        if (!_.isEmpty(inputDestinationPortFrom.val())) {
            this.pageValidation(inputDestinationPortFrom, false);
            this.pageValidation(inputDestinationPortTo, false);
        }
    },

    /**
     * Notify that one of ports fields will be updated on the page
     * @param element
     */
    notifyPortsUpdate: function(element) {
        if (!_.isUndefined(element) && element.size()) {
            element.addClass('input-attention');

            setTimeout(function() {
                element.removeClass('input-attention');
            }, 250);
        }
    },

    /**
     * Update model which stores information about current rule before synching it with server
     * @returns {swc.constructors.PortForwardingRule}
     */
    updateRuleModel: function() {
        var self = this,
            ruleModel = _.isUndefined(this.model) ? new swc.constructors.PortForwardingRule() : this.model,
            data = {
                name: { elementName: "serviceName" },
                portsEntryMode: { elementName: "entry-ports-mode" },
                entryPortTo: { elementName: "entryPort", elementClass: "to" },
                entryPortFrom: { elementName: "entryPort", elementClass: "from" },
                protocol: { elementName: "protocol" },
                assignedDeviceIP: { elementName: "deviceIP" },
                portsDestinationMode: { elementName: "destination-ports-mode" },
                desinationPortTo: { elementName: "destinationPort", elementClass: "to" },
                desinationPortFrom: { elementName: "destinationPort", elementClass: "from" },
                predefinedMode: { elementName: "service-mode" },
                predefinedName: { elementName: "predefinedServiceName" }
            },
            elementsMap = [],
            ports = {
                entry: [],
                destination: []
            }, id, name, status;

        // Collect elements map:
        _.each(this.getElements(), function(element) {
            elementsMap.push(getParameter($(element)));
        });

        // Get data values
        data = swc.Utils.getDataToValidate(data, elementsMap);

        // Update rule model values:
        if (data.predefinedMode === "custom") {
            id = swc.Utils.generateId();
            name = data.name;
            status = true;
        } else {
            // This is a case when rule is editing (some elements are absent)
            id = _.isString(data.predefinedMode) ? swc.Utils.generateId() : ruleModel.get('id');
            name = _.isString(data.predefinedMode) ? data.predefinedName : data.name;
            status = _.isString(data.predefinedMode) ? true : ruleModel.get('status');
        }

        ruleModel.set('id', $.trim(id));
        ruleModel.set('name', $.trim(name));
        ruleModel.set('status', status);

        ruleModel.set('protocol', data.protocol);
        ruleModel.set('deviceIP', data.assignedDeviceIP);

        // destination port -> port on stargate device
        if (data.portsEntryMode === "range") {
            ports.entry = [ data.entryPortFrom, data.entryPortTo ];
        } else {
            ports.entry = [ data.entryPortFrom ];
        }

        // destination port -> port on assigned device
        if (data.portsDestinationMode === "range") {
            ports.destination = [ data.desinationPortFrom, data.desinationPortTo ];
        } else {
            ports.destination = [ data.desinationPortFrom ];
        }

        ruleModel.set('ports', ports);

        return ruleModel;
    },

    /**
     * Handler for saving page. Will trigger, when user will press "Save" button
     *
     * @description
     *
     *      After user will press "Save" button, page validation should trigger. If validation is success, rule has to
     *      be added to the NP, collection of rules has to be resynced and user has to be navigated to the Port Forwarding
     *      rules page, in order to see his rule.
     */
    save: function() {
        var updatedModel = this.updateRuleModel();

        // NP requires commit action after any changes done to rules:
        $.when(updatedModel.sync('create'), updatedModel.sync('commit'))
            .done(function() {
                swc.router.navigate('network/settings/port-forwarding', { skipUnsavedChanges: true });
            })
            .fail(function() {
                // do nothing, maybe show some error message
            });
    },

    /**
     * @override on {'swc.BaseView.cancelChanges()'} method
     *
     * Need to navigate user to Rules List page when clicking cancel
     *
     * @returns {boolean}
     */
    cancelChanges: function() {
        swc.router.navigate('network/settings/port-forwarding', { skipUnsavedChanges: true });
    }

});
