swc.constructors.NetworkSettingsIpsettingsView = swc.base.TabView.extend({

    className: 'ip-settings',

    models: [
        'NetworkDevices',
        'DHCPLeases'
    ],

    validation: {
        'Address': 'Network:validateIPAddress',
        'Netmask': 'Network:validateNetmask',
        'DHCPRange': 'Network:validateDHCPSettings',
        'DHCPLeaseIPAddress': 'DHCPLeases:validateIP',
        'DHCPLeaseNewDevice': 'DHCPLeases:validateNewDevice',
        'DMZIPAddress': 'Network:validateDMZ'
    },

    events: {
        'swc-checkbox:change .swc-checkbox.DHCPEnable': 'setDHCPState',
        'swc-checkbox:change .swc-checkbox.enable-dmz': 'setDMZState',
        'swc-dropdown:change .swc-dropdown.dhcplease-device-selection': 'selectDHCPLeaseDevice',
        'change input[name="Address"],input[name="DHCPRange"],input[name="Netmask"]': 'revalidateForm',
        'keyup input[name="Address"],input[name="Netmask"]': 'updateSubnetInfo',

        'expandable:edit .expandable-list.dhcp-leases-list': 'setEditLease',
        'expandable:delete .expandable-list.dhcp-leases-list': 'deleteDHCPLease'
    },

    setTemplateData: function() {
        var validation = swc.models.Network.validation,
            Address = swc.models.Network.get('ip-settings').get('LocalIPAddress'),
            Netmask = swc.models.Network.get('ip-settings').get('Netmask');

        this.templateData = {
            'Address': Address,
            'Netmask': Netmask,
            'DHCPMin': swc.models.Network.get('ip-settings').get('DHCPStart'),
            'DHCPMax': swc.models.Network.get('ip-settings').get('DHCPFinish'),
            'DHCPState': swc.models.Network.get('ip-settings').get('DHCPStatus'),
            'DHCPLeases': swc.models.DHCPLeases.getActiveLeases(),
            'AddressRange' : validation.getAllowedHostRange(validation.ipStringToDecimal(Address), validation.ipStringToDecimal(Netmask))
        };
    },

    revalidateForm: function(e) {
        var elem = $(e.currentTarget);

        if (!elem.hasClass('validation-error')) {
            this.pageValidation();
        }
    },

    renderComplete: function() {
        var self = this,
            ipAddressBox = this.$el.find('input[name="Address"]'),
            DHCPMinBox = this.$el.find('input[name="DHCPRange"].range-from'),
            DHCPMaxBox = this.$el.find('input[name="DHCPRange"].range-to');

        // Listener for IP Address field change
        ipAddressBox.on('validation-success', function() {
            self.updateDHCPSettings();
            self.pageValidation(DHCPMinBox, false);
            self.pageValidation(DHCPMaxBox, false);
        });

        this.setDevicesDropdown();
        this.setDMZData();

        $('input, textarea').placeholder();
    },

    pageCheckDefaultValues: function() {
        var validation = swc.base.PageView.prototype.pageCheckDefaultValues.apply(this, arguments);

        if (this.$('.expandable-list-item.deleted').length > 0) {
            return false;
        }

        return validation;
    },

    setDevicesDropdown: function() {
        var dropdown = this.$('.dhcplease-device-selection'),
            selectedDevices = swc.models.DHCPLeases.toJSON(),
            options = swc.models.NetworkDevices.getDeviceDropdownOptions({'filter': 'DHCPLeases'}),
            defaultDevice = options[0];

        dropdown.data('options', options);
        dropdown.trigger('swc-dropdown:swc-change', defaultDevice);
    },

    /**
     * Event handler: is fired on changing device in the dropdown list
     *
     * @param e
     * @param value
     */
    selectDHCPLeaseDevice: function(e, value) {
        var $select = $(e.currentTarget),
            $item = $select.parents('.expandable-list-item'),
            $mac = $item.find('.lease-mac-address input'),
            $ip = $item.find('.lease-ip-address input'),
            device = swc.models.NetworkDevices.getDevice(value);

        $mac.val(value);

        if (device && device.address && device.status) {
            $ip.val(device.address.IPv4);
        } else {
            $ip.val('');
        }

        // Set editing flag to let view know that page changed:
        this.$('.edit-form-flag').val(value);

        // if it's an item for new lease then trigger its edit event
        if ($item.is('.new-dhcplease-form')) {
            this.$('.expandable-static-ip .expandable-list').trigger('expandable:edit', 'newItem');
        }

        this.setButtonsState();
    },

    /**
     * Close sibling items in the expandable list
     *
     * @param $item {jQuery element}
     */
    closeSiblingItems: function($item) {
        var self = this;

        $item.siblings('.expandable-list-item.mode-editing').each(function() {
            var dataKey = $(this).data('key');
            self.$('.expandable-list').trigger('expandable:swc-no-edit', dataKey);
        });
    },

    /**
     * Switch validation of elements in the sibling items
     * by adding/removing class 'skip-validation' to them
     *
     * if on === true then add validation (remove class 'skip-validation')
     * if on === false then remove validation (add class 'skip-validation')
     *
     * @param $item {jQuery element}
     * @param on {Boolean}
     */
    setValidationOnSiblingItems: function($item, on) {
        $item.siblings('.expandable-list-item').each(function() {
            $(this).find('input, .swc-dropdown').toggleClass('skip-validation', !on);
        });
    },

    /**
     * Switch validation of elements in the currently edit item
     * by adding/removing class 'skip-validation' on them
     *
     * if on === true then add validation (remove class 'skip-validation')
     * if on === false then remove validation (add class 'skip-validation')
     *
     * @param $item {jQuery element}
     * @param on {Boolean}
     */
    setValidationOnItem: function($item, on) {
        $item.find('input, .swc-dropdown').toggleClass('skip-validation', !on);
    },

    /**
     * Hide list item for adding new lease
     * and switch off validation on its elements
     */
    hideNewLeaseItem: function() {
        var $item = this.$('.new-dhcplease-form');

        $item.addClass('hidden');
        this.setValidationOnItem($item, false);
    },

    /**
     * Set enabled/disabled mode of sibling items by removing/adding
     * class 'disabled' to buttons with classes 'do-edit', 'do-delete'
     *
     * if on === true then enable sibling items (remove class 'disabled')
     * if on === false then disable sibling items (add class 'disabled')
     *
     * @param $item {jQuery element}
     * @param show {Boolean}
     */
    setEnabledModeForSiblingItems: function($item, on) {
        $item.siblings('.expandable-list-item').each(function() {
            $(this).find('.do-edit, .do-delete').toggleClass('disabled', !on);
        });
    },

    /**
     * Event handler: is fired on start edit static IP-address item
     *
     * @param e {Event}
     * @param value {String}
     */
    setEditLease: function(e, value) {
        var $item = $(e.currentTarget).find('[data-key="' + value + '"]');

        this.closeSiblingItems($item);
        this.setEnabledModeForSiblingItems($item, false);
        this.setValidationOnSiblingItems($item, false);
        this.setValidationOnItem($item, true);

        if (value !== 'newItem') {
            this.hideNewLeaseItem();
        }

        $item.find('.do-edit').addClass('disabled');

        // Set editing flag to let view know that page changed:
        this.$('.edit-form-flag').val(value);

        this.setButtonsState();
    },

    /**
     * Enable table with static IP addresses:
     *  - remove opacity from table
     *  - remove hover popup with info message above table
     */
    enableDHCPLeaseTable: function() {
        var dhcpTable = $(".dhcp-settings"),
            inputs = dhcpTable.find(".swc-input");

        _.each(inputs, function(input) {
            $(input).removeClass("skip-validation").removeClass("disabled").removeAttr("disabled", "disabled");
        });

        this.$('.dhcp-leases-list').removeClass('disabled');
    },

    /**
     * Disable table with static IP addresses:
     *  - add opacity to table
     *  - add hover popup with info message above table
     */
    disableDHCPLeaseTable: function() {
        var dhcpTable = $(".dhcp-settings"),
            inputs = dhcpTable.find(".swc-input");

        _.each(inputs, function(input) {
            $(input).addClass("skip-validation").addClass("disabled").attr("disabled", "disabled");
        });

        this.$('.dhcp-leases-list').addClass('disabled');
    },

    /**
     * Event handler: is fired on changing state of DHCP
     *
     * @param e {Event}
     * @param value {String}
     */
    setDHCPState: function(e, value) {
        if (value) {
            this.enableDHCPLeaseTable();
        } else {
            this.disableDHCPLeaseTable();
        }

        this.pageValidation(); //Revalidate DHCP and so on
    },

    /**
     * Event handler: is fired on delete static IP-address item
     *
     * @param e {Event}
     * @param value {String}
     */
    deleteDHCPLease: function(e, value){
        var $item = $(e.currentTarget).find('[data-key="' + value + '"]');

        this.setValidationOnItem($item, false);
        this.setEnabledModeForSiblingItems($item, true);

        $item.addClass('deleted').hide();

        this.setButtonsState();
    },

    setDMZState: function(e, value) {
        var DMZIPAddress = this.$('[data-name="DMZIPAddress"]');

        $(".dmz-device").toggleClass("disabled", !value);
        
        if (!value) {
            this.pageValidation(DMZIPAddress, false);
        }
    },

    setDMZData: function() {
        var options = swc.models.NetworkDevices.getDeviceDropdownOptions({'filter': 'status'}),
            deviceIP = swc.models.Network.getParameter('deviceIP', 'dmz'),
            selected = swc.models.NetworkDevices.getDeviceMac(deviceIP),
            dmzCheckbox = this.$el.find('.swc-checkbox.enable-dmz'),
            dmzDropDown = this.$el.find('.dmz-device');

        dmzDropDown.data('options', options);
        dmzDropDown.toggleClass("disabled", !deviceIP.length);

        if (!deviceIP.length) {
            dmzCheckbox.trigger('swc-checkbox:swc-change', false);

            dmzCheckbox.data("default-value", false);
            dmzDropDown.data("default-value", "");
        } else {
            dmzDropDown.data("default-value", selected);
            dmzCheckbox.data("default-value", true);

            dmzCheckbox.trigger('swc-checkbox:swc-change', true);
            dmzDropDown.trigger('swc-dropdown:swc-change', selected);
        }
    },

    //Change DHCP range subnet
    updateDHCPSettings: function() {
        var validation = swc.models.Network.validation,
            Address = this.$('input[name="Address"]').val(),
            Netmask = this.$('input[name="Netmask"]').val(),
            DHCPMin = this.$('input[name="DHCPRange"].range-from'),
            DHCPMax = this.$('input[name="DHCPRange"].range-to'),

            AddressDecimal = validation.ipStringToDecimal(Address),
            NetmaskDecimal = validation.ipStringToDecimal(Netmask);

        if (validation.validateNetmaskDecimal(AddressDecimal, NetmaskDecimal, false, false).status !== false) {
         
            var minVal = validation.replaceSubnet(DHCPMin.val(), Address, Netmask);
            var maxVal = validation.replaceSubnet(DHCPMax.val(), Address, Netmask);
            //Check that DHCPMin is less then DCHPMax
            var valsAreValid = (minVal !== false) && (maxVal !== false) &&
             (validation.ipStringToDecimal(minVal) < validation.ipStringToDecimal(maxVal));
            
            if (minVal && (!maxVal || valsAreValid)) {
                DHCPMin.val(minVal);
            }
            if (maxVal && (!minVal || valsAreValid)) {
                DHCPMax.val(maxVal);
            }
        }
    },

    //In case if subnet have changed, update, DHCP range, and hint for network range
    updateSubnetInfo: function() {
        var validation = swc.models.Network.validation,
            Address = validation.ipStringToDecimal(this.$('input[name="Address"]').val()),
            Netmask = validation.ipStringToDecimal(this.$('input[name="Netmask"]').val());

        if (validation.validateNetmaskDecimal(Address, Netmask, false, false).status !== false &&
            validation.validateIPAddressDecimal(Address, Netmask, false, false).status !== false) {
            
            
            var popoverData = this.$('.netmask-settings-info').data('popover-template-data');
            popoverData.subnetMask = validation.decimalToIpString(Netmask);
            popoverData.subnetRange = validation.getAllowedHostRange(Address, Netmask);
            this.$('.netmask-settings-info').data('popover-template-data', popoverData);
        }
        

        this.updateDHCPSettings();
    },


    getDHCPLeasesData: function () {
        var possibleElements = [
                '.swc-input', '.swc-checkbox', '.swc-extended', '.swc-radio-buttons', '.swc-dropdown'
            ],
            selector = possibleElements.join(','),
            pool = [];

        this.$('.expandable-static-ip .expandable-list-item')
            .filter('.mode-editing:not(.skip-validation), .edited:not(.skip-validation), .deleted, .new-dhcplease-form:visible:not(.skip-validation)').each(function () {
                var self = $(this),
                    item = self.find(selector).map(function () {
                        return getParameter($(this));
                    });
                if (self.is('.deleted')) {
                    item.deleted = true;
                }
                pool.push(item);
            });

        return pool;
    },

    save: function() {
        var self = this,
            deferred = new $.Deferred(),
            deferred2 = new $.Deferred(),
            pageData = [],
            warningMessages = [],
            actionsToDo,
            elements = this.getElements(),
            DHCPLeasesData = this.getDHCPLeasesData();

        // Get elements name => value map
        $.each(elements, function(key, element) {
            pageData.push(getParameter($(element)));
        });

        // warningMessages can be filled when IP setting are saved correctly already
        warningMessages = self.checkNetworkChangeWarnings(pageData);

        /*
         * if there are some warning messages then hide page loading modal window
         * and show information modal window
         */
        if (!_.isEmpty(warningMessages)) {
            self.stopPageLoading();
        }

        // Define what has to be done on save
        actionsToDo = [
            swc.models.Network.saveDMZSettings(pageData),
            swc.models.DHCPLeases.saveDHCPLeases(DHCPLeasesData)
        ];

        $.when.apply(null, actionsToDo)
            .done(function() {
                // IP settings are saved the last because we can't save anything if IP is changed: 
                $.when(swc.models.Network.saveIPSettings(pageData))
                    .done(function() {
                        deferred.resolve();
                    })
                    .fail(function() {
                        deferred.reject();
                    });
            }).fail(function() {
                deferred.reject();
            });

        function afterSave(rejected) {
            if (!_.isEmpty(warningMessages) && !rejected) {
                self.showWarningModal(deferred2, rejected, warningMessages[0]);
            } else {
                if (rejected) {
                    deferred2.reject();
                } else {
                    deferred2.resolve();
                }
            }
        }

        function afterFail() {
            afterSave(true);
        }
    
        deferred
            .done(afterSave)
            .fail(afterFail);

        return deferred2.promise();
    },

    /**
     * Get an array of warning messages if the net is changed/reduced so that some devices may need reconnection.
     * The only first message will be shown to user, that is why all messages are sorted by priority.
     * E.g. in case of editing "InternetBox IP address" and "Static IP address" only message about changing
     * "InternetBox IP address" should be shown
     *
     * @params data {Array} Array of form values
     * @return {Array} Warning messages
     */
    checkNetworkChangeWarnings: function(data) {
        var oldMask,
            newMask,
            warningMessages = [];

        function ipToDecimal(ip) {
            return _.reduce(ip.split('.'), function(m, n) { return m*256 + parseInt(n, 10); }) >>> 0;
        }

        _.each(data, function(element, key) {
            if (element.parameterName === 'Address') {
                if (element.parameterValue !== element.parameterData.defaultValue) {
                    warningMessages.push('1_ip');
                }
            }

            if (element.parameterName === 'Netmask') {
                oldMask = ipToDecimal(element.parameterData.defaultValue);
                newMask = ipToDecimal(element.parameterValue);

                if (oldMask < newMask) {
                    warningMessages.push('2_mask');
                }
            }

            if (element.parameterName === 'DHCPLeaseIPAddress') {
                if (element.parameterValue !== element.parameterData.defaultValue) {
                    warningMessages.push('3_lease_ip');
                }
            }
        });
        
        /*
         * only first message will be shown to user sort messages by priority
         * and all messages should sorted by priority
         */
        warningMessages.sort();

        return warningMessages;
    },

    showWarningModal: function (deferred, rejected, messageID) {
        var self = this,
            ip;

        SWCElements.modalWindow.show({
            templateID: 'network:settings:ip-settings:warning-modal',
            templateData: {
                messageID: messageID
            },
            className: 'network-warning-modal',

            onApply: function() {
                SWCElements.modalWindow.hide();

                // if we came by IP then we have to be redirected to the new IP:
                if (window.location.hostname.match(/^\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}$/)) {
                    self.showPageLoading('Loading page data..');
                    document.location.href = window.location.origin.replace(/\d{1,3}\.\d{1,3}\.\d{1,3}\.\d{1,3}/,
                        self.$('input[name=Address]').val()) ;
                }

                deferred.resolve();
            }
        });
    }
});
