swc.constructors.NetworkSettingsIpv6firewallAddView = swc.base.TabView.extend({

    className: 'ipv6-firewall-rules-add',

    allowedMods: [ 'expert' ],

    events: {
        'swc-radio-buttons:change .swc-radio-buttons.service-mode': 'onRulesModeChange',
        'swc-radio-buttons:change .swc-radio-buttons.ports-mode': 'onPortsModeChange',
        'swc-radio-buttons:swc-change .swc-radio-buttons.ports-mode': 'onPortsModeChange',

        'swc-dropdown:change .swc-dropdown.policy-selection': 'onPolicyChange',
        'swc-dropdown:change .swc-dropdown.service-name-selection': 'onPredefinedRuleSelection'
    },

    models:[
        'FirewallPolicy',
        'FirewallRules'
    ],

    validation: {
        'RuleName': 'FirewallRules:RuleName',
        'RulePort': 'FirewallRules:RulePort',
        'PredefinedRuleName': 'FirewallRules:PredefinedRuleName'
    },

    setTemplateData: function() {
        this.templateData = {
            isAdding: true,
            predefinedRulesList: swc.models.FirewallRules.predefined,
            ruleType: "custom",
            ruleName: "",
            ruleProtocol: "6,17",
            rulePortsMode: "single",
            rulePorts: [ "" ],
            rulePolicy: "Drop_inbound"
        };
    },

    initialize: function() {
        this.template = $.template("pageContent", swc.Templates.get('network:settings:ipv6-firewall:rules-add').get('content'));
        this.events = _.extend({}, swc.base.PageView.prototype.events, this.events);
    },

    renderComplete: function() {
        this.setPredefinedRulesDropdown();
        this.resetFormValues();
    },

    /**
     * Reset rule adding / editing form to its default values
     *
     * @description:
     *
     * If values are passed:
     *
     *  Page will be filled in with values which are passed as arguments to the function.
     *
     * If values are not passed:
     *
     *  Page will be filled in with values which are set in <this.templateData>
     *
     * @param values {Object} same structure as <this.templateData>
     */
    resetFormValues: function(values) {
        var formValues = !_.isUndefined(values) ? values : this.templateData,
            $ruleNameInput = this.$('input[name="RuleName"]'),
            $rulePortFromInput = this.$('input[name="RulePort"].from'),
            $rulePortToInput = this.$('input[name="RulePort"].to'),
            $ruleTypeRadioGroup = this.$('.swc-radio-buttons.service-mode'),
            $rulePortRadioGroup = this.$('.swc-radio-buttons.ports-mode'),
            $ruleNameDropdown = this.$('.swc-dropdown.service-name-selection'),
            $ruleProtocolDropdown = this.$('.swc-dropdown.protocol-selection'),
            $rulePolicyDropdown = this.$('.swc-dropdown.policy-selection');

        // Update page dropdown values:
        this.setProtocolDropdown(formValues.ruleProtocol);
        this.setPolicyDropdown(formValues.rulePolicy);

        // Remove all previous validation messages:
        this.clearValidationMessages();

        // Reset rule name:
        $ruleNameInput.val(formValues.ruleName);

        // Reset rule protocol value:
        $ruleProtocolDropdown.trigger('swc-dropdown:swc-change', formValues.ruleProtocol);

        // Reset rule ports and rule ports mode:
        $rulePortFromInput.val(formValues.rulePorts[0]);
        $rulePortRadioGroup.trigger('swc-radio-buttons:swc-change', formValues.rulePortsMode);

        if (formValues.rulePortsMode === "range") {
            $rulePortToInput.val(formValues.rulePorts[1]);
        } else {
            $rulePortToInput.val("");
        }

        // Reset rule policy value:
        $rulePolicyDropdown.trigger('swc-dropdown:swc-change', formValues.rulePolicy);
    },

    /**
     * Handler for changing rule adding type:
     *
     * @param e {Object}
     * @param value {String}
     */
    onRulesModeChange: function(e, value) {
        var type = !_.isUndefined(value) ? value : this.templateData.ruleType;

        if (type === "predefined") {
            this.onPredefinedRuleSelection();
        } else {
            this.resetFormValues();
        }

        this.$('.rule-name-section').toggleClass("type-predefined", value === "predefined");

        this.setButtonsState();
    },

    /**
     * Handler for changing rule ports mode (single / range):
     *
     * @param e {Object}
     * @param value {String}
     */
    onPortsModeChange: function(e, value) {
        var type = !_.isUndefined(value) ? value : this.templateData.rulePortsMode;

        if (value !== "range") {
            this.$('input[name="RulePort"].to').val("");
        }

        this.$('.ports-values-section').toggleClass("range-type", value === "range");
    },

    /**
     * Handler for changing predefined rules dropdown:
     *
     * @param e {Object}
     * @param value {Value}
     */
    onPredefinedRuleSelection: function(e, value) {
        var dropdown = this.$('.swc-dropdown.service-name-selection'),
            ruleKey = dropdown.data('value'),
            ruleData = {};

        // Override dropdown value with new value:
        if (!_.isUndefined(value)) {
            ruleKey = value;
        }

        // Get rule data from list of prdefined rules:
        ruleData = _.findWhere(this.templateData.predefinedRulesList, { name: ruleKey });

        // Reset form values with predefined rule:
        this.resetFormValues({
            ruleType: "predefined",
            ruleName: ruleData.name,
            ruleProtocol: ruleData.protocol,
            rulePortsMode: ruleData.port.length === 1 ? "single" : "range",
            rulePorts: ruleData.port,
            rulePolicy: 'Drop_inbound'
        });
    },

    /**
     * Handler for changing policy mode dropdown:
     *
     * @param e {Object}
     * @param value {Value}
     */
    onPolicyChange: function(e, value) {
        this.pageValidation(true);
    },

    /**
     * Add options to protocol dropdown
     *
     * @param value {String} (optional)
     */
    setProtocolDropdown: function(value) {
        var dropdown = this.$('.swc-dropdown.protocol-selection'),
            dropdownValue = !_.isUndefined(value) ? value : this.templateData.ruleProtocol,
            options = [
                { name: getTranslationStrings("TCP/UDP"), value: "6,17" },
                { name: getTranslationStrings("TCP"),     value: "6"    },
                { name: getTranslationStrings("UDP"),     value: "17"   }
            ];

        // Set dropdown data:
        dropdown.data('options', options);
        dropdown.trigger('swc-dropdown:swc-change', dropdownValue);
    },

    /**
     * Add options to policy dropdown
     *
     * @param value {String} (optional)
     */
    setPolicyDropdown: function(value) {
        var dropdown = this.$('.swc-dropdown.policy-selection'),
            dropdownValue = !_.isUndefined(value) ? value : this.templateData.rulePolicy,
            options = [
                { value: "Drop_inbound", name: getTranslationStrings("Block inbound") },
                { value: "Drop_outbound", name: getTranslationStrings("Block outbound") },
                { value: "Accept_inbound", name: getTranslationStrings("Allow inbound") },
                { value: "Accept_outbound", name: getTranslationStrings("Allow outbound") }
            ];

        // Set dropdown data:
        dropdown.data('options', options);
        dropdown.trigger('swc-dropdown:swc-change', dropdownValue);
    },

    /**
     * Add options to predefined rules dropdown
     */
    setPredefinedRulesDropdown: function() {
        var rules = this.templateData.predefinedRulesList,
            dropdownValue = rules[0].name,
            dropdown = this.$('.swc-dropdown.service-name-selection'),
            options = [];

        // Create dropdown options object:
        _.each(rules, function(rule, key) {
            options.push({
                name: rule.name,
                value: rule.name
            });
        });

        // Set dropdown data:
        dropdown.data('options', options);
        dropdown.trigger('swc-dropdown:swc-change', dropdownValue);
    },

    /**
     * Serialize all page data to JSON object
     *
     * @returns {Object}
     */
    serializePageData: function() {
        var jsonData = {},
            $ruleKeyInput = this.$('input[name="RuleKey"]'),
            $ruleNameInput = this.$('input[name="RuleName"]'),
            $rulePortFromInput = this.$('input[name="RulePort"].from'),
            $rulePortToInput = this.$('input[name="RulePort"].to'),
            $ruleTypeRadioGroup = this.$('.swc-radio-buttons.service-mode'),
            $rulePortRadioGroup = this.$('.swc-radio-buttons.ports-mode'),
            $ruleNameDropdown = this.$('.swc-dropdown.service-name-selection'),
            $ruleProtocolDropdown = this.$('.swc-dropdown.protocol-selection'),
            $rulePolicyDropdown = this.$('.swc-dropdown.policy-selection');

        // Handle edit / add mode
        if ($ruleTypeRadioGroup.data('value') === 'custom') {
            if ($ruleKeyInput.size()) {
                jsonData.id = $ruleKeyInput.val();
            } else {
                jsonData.id = $ruleNameInput.val();
            }

            jsonData.name = $ruleNameInput.val();
        } else {
            jsonData.id = $ruleNameDropdown.data('value');
            jsonData.name = $ruleNameDropdown.data('value');
        }

        // Protocol dropdownd handling:
        jsonData.protocol = $ruleProtocolDropdown.data('value');

        // Policy dropdownd handling:
        jsonData.policy = $rulePolicyDropdown.data('value').split('_')[0];
        jsonData.policyTarget = $rulePolicyDropdown.data('value').split('_')[1];

        // Add hash to rule id in adding mode:
        if (!_.isUndefined(this.templateData.isAdding)) {
            if (jsonData.policyTarget === 'inbound') {
                jsonData.id = swc.Utils.generateId() + '_inbound';
            } else {
                jsonData.id = swc.Utils.generateId() + '_outbound';
            }
        }

        // Add chain target to rule:
        if (jsonData.policyTarget === 'inbound') {
            jsonData.chain = 'Custom_V6In';
        } else {
            jsonData.chain = 'Custom_V6Out';
        }

        // Handle edit / add mode
        jsonData.status = !_.isUndefined(this.templateData.ruleState) ? this.templateData.ruleState : true;

        // Single / range port mode handling:
        if ($rulePortRadioGroup.data('value') === "single") {
            jsonData.port = [ $rulePortFromInput.val() ];
        } else {
            jsonData.port = [ $rulePortFromInput.val(), $rulePortToInput.val() ];
        }

        return jsonData;
    },

    /**
     * @override
     *
     * On this page it's needed to handle only "save" button state:
     *
     * @param e {Object}
     * @param toDisable {Boolean}
     */
    setButtonsState: function(e, toDisable) {
        var container = this.$('.buttons-container-message'),
            buttonSave = container.find('.button.save-changes'),
            allValuesDefault = this.pageCheckDefaultValues();

        buttonSave.toggleClass('disabled', toDisable || allValuesDefault);
    },

    /**
     * @override
     *
     * On this page cancel changes button should redirect user to other page:
     */
    cancelChanges: function() {
        swc.router.navigate('network/settings/ipv6-firewall/customize', { skipUnsavedChanges: true });
    },

    /**
     * @override
     *
     * On this page save changes button should redirect user to other page after saving complete:
     */
    save: function() {
        var ruleData = this.serializePageData(),
            deferred = new $.Deferred();


        $.when(swc.models.FirewallRules.createRule(ruleData))
            .done(function() {
                localStorage.setItem('firewall:change-rule', 'done');
                swc.router.navigate('network/settings/ipv6-firewall/customize', { skipUnsavedChanges: true });
            })
            .fail(function() {
                localStorage.setItem('firewall:change-rule', 'fail');
                swc.router.navigate('network/settings/ipv6-firewall/customize', { skipUnsavedChanges: true });
            });

        return deferred.promise();
    }
});
