swc.constructors.NetworkSettingsDyndnsView = swc.base.TabView.extend({

    className: 'dyndns',

    models: [
        'DynDNS',
        'DynDNSProviderCollection'
    ],
    
    allowedMods: ['expert'],
    
    settingsFormTemplate: '',
    
    events: {
        'keyup input.password-original': 'updatePasswordsChange',
        'keyup input.password-copy': 'updatePasswordsChange',
        'swc-checkbox:change .swc-checkbox.show-password': 'setPasswordState',
        'swc-dropdown:change .swc-dropdown.provider-selection': 'changeService',
        'swc-checkbox:change .swc-checkbox.DynDNSEnable': 'setGlobalState'
    },
    
    validation: {
        'Service': 'DynDNSProvider:service',
        'hostname': 'DynDNSProvider:hostname',
        'username': 'DynDNSProvider:username',
        'password': 'DynDNSProvider:password'
    },

    setTemplateData: function() {
        this.templateData = {
            dyndnsStatus: swc.models.DynDNS.get('enable')
        };
    },
    
    initialize: function () {
        swc.base.TabView.prototype.initialize.apply(this, arguments);
        
        // Prepare template for the settings form:
        this.settingsFormTemplate = $.template("settingsForm",
            swc.Templates.get("network:settings:dyndns:service-settings-form").get('content'));
    },

    renderComplete: function() {
        var self = this;
        
        if (!swc.models.DynDNSServiceCollection) {
            swc.models.DynDNSServiceCollection = new swc.constructors.DynDNSServiceCollection();
        }
        
        $.when(swc.models.DynDNSServiceCollection.fetch())
            .done(function() {
                // Find currently active provider and set it as selected
                var provider = swc.models.DynDNSProviderCollection.findWhere({enable: true}),
                    defaultProvider = swc.models.DynDNSServiceCollection.at(0),
                    providerId = provider ? provider.get('service') : '';
                
                // According to requirements, when no providers configured yet - 
                // first element in option list should be selected 
                if (!providerId) {
                    providerId = defaultProvider ? defaultProvider.get('id') : '';
                }
                
                self.changeService(null, providerId);
                self.setProvidersData(providerId);
                self.showState();
            });
    },

    showState: function() {
        var state = swc.models.DynDNS.get('enable');

        this.$('.swc-dropdown, .swc-checkbox.show-password').toggleClass('disabled', !state);
        
        this.setPageState(state, {disabledBlockClass: "faded"});

        if (state === false) {
            this.showDisabledMessage({
                container: '.dynDnsOffMessage',
                className: 'above'
            });
            this.$('input').attr('disabled', 'disabled');
        } else {
            this.hideDisabledMessage();
            this.$('input').removeAttr('disabled');
        }
    },

    /**
     * Set active provider as selected in dropdown list
     * 
     * @param {String} providerId
     */
    setProvidersData: function(providerId) {
        var options = swc.models.DynDNSServiceCollection.formatProvidersOptions(),
            dropdown = this.$('.provider-selection');
        
        dropdown.data('options', options);
        dropdown.data('default-value', providerId);
        dropdown.trigger('swc-dropdown:swc-change', providerId);
    },

    /**
     * Show form with auth data for the selected service 
     * @param e
     * @param {String} value Provider to select
     */
    changeService: function(e, value) {
        moment.lang(swc.models.Locale.locale);

        var provider = swc.models.DynDNSProviderCollection.findWhere({service: value}),
            dnsProviderData;
        
        if (!_.isEmpty(provider)) {
            var lastUpdate = provider.get('last_update');

            if (lastUpdate && lastUpdate !== "0001-01-01T00:00:00Z") {
                lastUpdate = moment(lastUpdate).zone(lastUpdate).format("DD.MM.YYYY, HH:mm");
            } else {
                lastUpdate = '';
            }

            dnsProviderData = {
                service: provider.get('service').capitalize(),
                status: provider.get('status'),
                lastUpdate: lastUpdate
            };
        }

        // Render template with data
        var tmpl = $.tmpl(this.settingsFormTemplate, {
                service: provider ? provider.attributes : (new swc.constructors.DynDNSProvider()).attributes,
                localeString: getTranslationStringsjQuery,
                localeStrings: swc.models.Locale.getLocaleStrings(),
                dyndnsStatus: swc.models.DynDNS.get('enable'),
                dnsProviderData: dnsProviderData
            }),
            settingsFormContainer = this.$(".table.provider-data");
        
        // Render template
        settingsFormContainer.html(tmpl);
    },

    /**
     * Show textual representation of a password.
     * Uses addition hidden field to store clone of value been input into password field.
     * 
     * @param e
     * @param value
     */
    setPasswordState: function(e, value) {
        var providerSettingsFormContainer = this.$('.table.provider-data'),
            passwordOrigBox = providerSettingsFormContainer.find('input.password-original'),
            passwordCopyBox = providerSettingsFormContainer.find('input.password-copy');

        if (value) {
            passwordOrigBox.hide();
            passwordCopyBox.show();
        } else {
            passwordCopyBox.hide();
            passwordOrigBox.show();
        }
    },

    /**
     * Copies value from password field into hiddent text field char-by-char. 
     * @param e
     */
    updatePasswordsChange: function(e) {
        var providerSettingsFormContainer = this.$('.table.provider-data'),
            passwordOrigBox = providerSettingsFormContainer.find('input.password-original'),
            passwordCopyBox = providerSettingsFormContainer.find('input.password-copy');

        // Copy values from one to another and vice-versa
        if (passwordOrigBox.is(":visible")) {
            passwordCopyBox.val(passwordOrigBox.val());
        } else {
            passwordOrigBox.val(passwordCopyBox.val());
        }
    },

    setGlobalState: function (e, value) {
        swc.models.DynDNS.set('enable', value);
        this.showState();
    },
    
    save: function() {
        var self = this,
            deferred = new $.Deferred(),
            pageData = {},
            dynDnsStatus = swc.models.DynDNS.get('enable');

        // Get elements name => value map
        $.each(this.getElements(), function(key, element) {
            var parameter = getParameter($(element));
            pageData[parameter['parameterName']] = parameter['parameterValue'];
        });
        
        // Fill in data for selected service provider
        var provider = new swc.constructors.DynDNSProvider({
            'service': pageData['Service'],
            'hostname': pageData['hostname'],
            'username': pageData['username'],
            'password': pageData['password'],
            'enable': true
        });
        
        /**
         * Add newly created provider
         * 
         * @private
         * 
         * @return Deferre.Promise
         */
        var addNewProvider = function() {
            // Add newly created provider 
            $.when(provider.sync("create"))
                .done(function (response) {
                    if (response.status === true) {
                        // Re-read collection of added hosts
                        swc.models.DynDNSProviderCollection.fetch();
                        deferred.resolve();
                    } else {
                        deferred.reject("Fill in all fields");
                    }
                })
                .fail(function (xhr, error, status) {
                    deferred.reject(status);
                });
        };
        
        // Update global DynDNS status & save new DynDNS provider marked as 'enabled'
        $.when(swc.models.DynDNS.sync("update"))
            .done(function (response) {
                // Do not re-add services if we deactivated DynDNS globally
                if (dynDnsStatus) {
                    // we actually always will have only one item in collection
                    var todo = [];

                    swc.models.DynDNSProviderCollection.forEach(function (aProvider) {
                        todo.push(aProvider.destroy());
                    });

                    $.when.apply(this, todo)
                        .done(function () {
                            addNewProvider();
                        })
                        .fail(function (xhr, error, status) {
                            deferred.reject();
                        });
                } else {
                    deferred.resolve();
                }
            })
            .fail(function() {
                deferred.reject();
            });

        return deferred.promise();
    }
});
