swc.constructors.LoginView = swc.base.PageView.extend({
    
    models: ['PasswordRecovery'],

    /**
     * Current view class name:
     */
    className: 'login',

    /**
     * List of event handlers for current view
     */
    events: {
        'click a[name="login-button"]:not(.disabled)': 'processLogin',
        'click a[name="restore-button"]': 'displayRestoreForm',

        'keyup input': 'updateFormState',

        // To update form state even if 'backspace' button holding
        'keydown input': 'updateFormState',

        // Submit login form with enter button
        'keydown .sign-in-form input': 'processLogin',

        'swc-checkbox:change .display-password': 'displayPassword',
        'swc-dropdown:change .locale-selector': 'setLocale',

        'keyup input[name*=login-password]': 'copyPassword',
        'keyup input[name*=new-password-]': 'copyPassword',

        'click a[name="back"]:not(.disabled)': 'backToLogin',

        'click .save-password-button:not(.disabled)': 'applyPassword',
        'swc-checkbox:change .display-new-password': 'showPassword',
        'swc-checkbox:change .enable-recovery': 'enableRecovery',
        'keypress input[name*=new-password-]': 'activateButtons',
        'change input[name*=new-password-]': 'activateButtons',
        'change input[name=new-password-1]': 'validatePassword'
    },

    /**
     * Initialize function has to be overrided from the base page view because of routing address mismatch
     */
    initialize: function() {
        this.template = swc.Templates.get(this.className).get('content');
    },

    /**
     * Render function has to be overrided from the base page view because of it's specific
     */
    render: function() {
        var self = this;

        setTimeout(function() {
            self.renderComplete();
        }, 0);
    },
    
    renderComplete: function(trigger) {
        var self = this,
            localesOptions = swc.models.Locale.getLocaleOptions(),
            currentPage = Backbone.history.fragment,
            isRemoteLogin = currentPage === 'remote-login' || !_.isNull(localStorage.getItem('remoteLogin')),
            loginArea = $('div.login');

        // No need to re-render this page if already on login page
        if (loginArea.size() && _.isUndefined(trigger)) {
            // remove application env variables
            this.clearLogoutState();
            return;
        }
        
        $('#application').html(this.$el.html(
            $.tmpl(self.template, {
                localeStrings: swc.models.Locale.getLocaleStrings('login'),
                localeString: getTranslationStringsjQuery,
                formatDate: swc.models.Locale.formatDate,
                recoveryEnable: swc.models.PasswordRecovery.get('recoveryEnable'),
                staticPagesLinks: swc.models.Locale.getStaticPagesLinks()
            })
        ));

        // Add supper admin class to body to handle some features with CSS styles:
        $('body').toggleClass('superAdmin', isRemoteLogin);

        // Set locale selector values:
        $('.locale-selector').data('options', localesOptions);
        $('.locale-selector').trigger('swc-dropdown:swc-change', swc.models.Locale.locale);

        // Check if current login for remote admin:
        if (isRemoteLogin) {
            this.$el.find('input[name="login-name"]').val('superadmin');
        } else {
            this.$el.find('input[name="login-name"]').val('admin');
        }

        // Remove local storage items:
        localStorage.removeItem('remoteLogin');
        localStorage.removeItem('userSuperAdmin');

        // Focus to password input
        $('input[name="login-password"]').val('').focus();
        $('input[name="login-password-copy"]').val('');

        swc.views.currentView = this;

        // Update form/login button state
        self.updateFormState();

        // Remember all events on current view
        self.delegateEvents();
        
        // Here we should display one of three steps located on #login page:
        // either "Sign-in" form, or "Password Recovery Hint" page or "Change password" form
        // if process is started already, but not finished.
        if (swc.models.PasswordRecovery.isRecoveryRunning()) {
            this.$('.sign-in-form').hide(); // Don't show login form if Recovery Process is running
            this.displayRestoreForm();
        } else {
            //-----------------------------------------
            // THIS BLOCK DISPLAYS REGULAR LOGIN FORM
            //-----------------------------------------
            $('input, textarea').placeholder();
            
            // remove application env variables
            this.clearLogoutState();

            // Remove loading window:
            self.stopPageLoading();
        }
    },

    /**
     * Clear application state after logout
     *
     * @description:
     *
     * Remove application env variables(action-logout, session-logout,
     * reset-logout) from the localStorage
     */
    clearLogoutState: function() {
        if (_.isNull(localStorage.getItem('reset-logout'))) {
            if (!_.isNull(localStorage.getItem('action-logout'))) {
                this.displayWarningMessage('logout');
                localStorage.removeItem('action-logout');
                localStorage.removeItem('session-logout');
            }

            if (!_.isNull(localStorage.getItem('session-logout'))) {
                this.displayWarningMessage('session-expired');
                localStorage.removeItem('action-logout');
                localStorage.removeItem('session-logout');
            }
        } else {
            // Clean-up flag that logout done via reset
            localStorage.removeItem('reset-logout');
            /*
             Usually we are logged out by call to LoginModel.processLogout()
             even if it is reset/upgrade/etc. process;
             thus clean-up the flag been set in order to not show "Successfully logged out" when <F5> button pressed
             */
            localStorage.removeItem('action-logout');
            localStorage.removeItem('session-logout');
        }
    },
   

    /**
     * prevent unsaved changes warning message
     * @returns {boolean}
     */
    pageCheckDefaultValues: function(){
        return true;
    },

    /**
     * Display restore form
     */
    displayRestoreForm: function() {
        var formRestore = $('.restore-form'),
            formSignIn = $('.sign-in-form'),
            deferred = new $.Deferred(),
            self = this;
        
        // If process is running - allow set new password
        if (swc.models.PasswordRecovery.isRecoveryRunning() && sessionStorage.getItem("changePassIncomplete")) {
            this.showChangePasswordForm();
            // Remove loading window:
            this.stopPageLoading();
            return;
        }

        /**
         * Start recovery process - after executing the confirmation code will appear on LCD display
         * @returns {*}
         */
        swc.models.EventManager = new swc.constructors.EventManager({
            events: [{"handler": "Screen"}]
        });

        this.showPageLoading("Starting Password Recovery Process...");
        
        // First, have to open channel to listen to password-recovery
        $.when(swc.models.EventManager.openChannel())
            .done(function() {
                // Create instance of EM listener
                swc.models.EventManagerListener = new swc.constructors.EventManagerListener();
                swc.models.EventManagerListener.setAjaxParameters(JSON.stringify({
                    "channelid": swc.models.EventManager.get("channelid"),
                    "events": [{"handler": "Screen"}]
                }));
                
                // Subscribe to listen if smth.changed
                swc.models.EventManagerListener.on("change", function (event, data, context) {
                    var ScreenEvent = swc.models.EventManagerListener.pop();
                    if (ScreenEvent) {
                        // stop listening on collection changes
                        swc.models.EventManagerListener.off("change");

                        // stop background event listener
                        clearInterval(self.eventsListenerInterval);
                        self.eventsListenerInterval = null;
                        
                        // User pressed "BACK" on device, we do the same - 
                        if (ScreenEvent.get('attributes').response === "Denied") {
                            // simulate click on back button
                            self.$('a[name="back"]:not(.disabled)').click();
                        } else if (ScreenEvent.get('attributes').response === "Allowed") {
                            sessionStorage.setItem("changePassIncomplete", true);
                            self.showChangePasswordForm();
                        }
                    }
                });

                $.when(swc.models.PasswordRecovery.startProcess())
                    .done(function() {
                        // Although listenChannel is a deferred, we don't need to resolve it
                        swc.models.EventManagerListener.listenChannel();
                        
                        // Check current recovery status
                        $.when(swc.models.PasswordRecovery.sync("read")).done(function(){
                            swc.views.Application.stopPageLoading();

                            // Start polling for events
                            if (!self.eventsListenerInterval) {
                                self.eventsListenerInterval = setInterval(function () {
                                    // although listenChannel is a deferred, we don't need to resolve it 
                                    swc.models.EventManagerListener.listenChannel();
                                    $.when(swc.models.PasswordRecovery.sync("read"))
                                        .done(function () {
                                            if (!swc.models.PasswordRecovery.isRecoveryRunning()) {
                                                // simulate click on back button
                                                self.$('a[name="back"]:not(.disabled)').click();
                                            }
                                        })
                                        .fail(function() {
                                            // TODO: do nothing or what?
                                        })
                                    ;
                                }, 15000);
                            }
                        });
                    });
                
                // Show page with hint about Password Recovery process
                self.showPassRecoveryHint();
                self.stopPageLoading();
                deferred.resolve();
            });
        
        return deferred.promise();
    },

    /**
     * Display Sign In form
     */
    displaySignInForm: function() {
        var formRestore = $('.restore-form'),
            formSignIn = $('.sign-in-form');

        formRestore.hide();
        formRestore.find('.info-messages .message').hide();
        formRestore.find('input').val('');

        formSignIn.show();
    },

    /**
     * Display password input as text type
     *
     * @param e {Object}
     *
     */
    displayPassword: function(e, value) {
        var inputPasswordVisible = $(this.el).find('input[name="login-password"]'),
            inputPasswordHidden = $(this.el).find('input[name="login-password-copy"]');

        inputPasswordHidden
            .val(inputPasswordVisible.val())
            .show();

        inputPasswordVisible.hide();

        inputPasswordVisible.attr('name', 'login-password-copy');
        inputPasswordHidden.attr('name', 'login-password');
    },

    copyPassword: function(e) {
        var element = $(e.target), visibleField, hiddenField;
        if(element.filter('[class*=login-password]')[0]){
            visibleField = $('input[name=login-password]');
            hiddenField = $('input[name=login-password-copy]');
        } else if(element.filter('[name*=new-password-1]')[0]){
            visibleField = $('input[name=new-password-1]');
            hiddenField = $('input[name=new-password-1-copy]');
        } else {
            visibleField = $('input[name=new-password-2]');
            hiddenField = $('input[name=new-password-2-copy]');
        }

        hiddenField.val(visibleField.val());

    },

    /**
     * Start login process
     *
     * @param e {String}
     *
     */
    processLogin: function(e) {
        var inputPasswordField = $(this.el).find('input[name="login-password"]'),
            inputLoginField = $(this.el).find('input[name="login-name"]'),
            self = this;

        if (e.type !== "keydown") {
            e.preventDefault();
        }

        if (e.which === 13 || e.type !== 'keydown') {
            if (!this.isValidationError()) {
                $.when(swc.models.Login.processLogin(inputLoginField.val(), inputPasswordField.val()))
                    .done(function(response) {
                        if (response.status) {
                            swc.router.navigate(self._determineNextPage(), { trigger: true });
                        } else {
                            self.displayWarningMessage('login');
                        }
                    })
                    .fail(function() {
                        self.displayWarningMessage('login');
                    });
            } else {
                this.displayWarningMessage('both-fields-required');
            }
        }
    },

    /**
     * Change login form (login button) state, according to inputs value
     *
     * @param e {String}
     *
     */
    updateFormState: function(e){
        var target = e ? $(e.target) : null,
            loginButton = this.$('a[name="login-button"]');

        if (target) {
            // Hiding validation if form is edited:
            if (target.val() !== target.data('old-val')) {
                $('.info-messages .message').hide();
                target.removeClass("validation-error");
            }

            target.data('old-val', target.val());
        }

        if(this.isValidationError()) {
            loginButton.addClass('disabled');
        } else {
            loginButton.removeClass('disabled');
        }
    },

    /**
     * Helper method to do all at once:
     * 1. SHOW "Change Password' form
     * 2. HIDE all other forms
     */
    showChangePasswordForm: function(){
        this.$('.restore-form').hide();
        this.$('.sign-in-form').hide();
        this.$('.change-pass-form').show();
    },

    /**
     * Helper method to do all at once:
     * 1. SHOW "Enter Activation Code" form
     * 2. HIDE all other forms
     */
    showPassRecoveryHint: function(){
        this.$('.restore-form').show();
        this.$('.sign-in-form').hide();
        this.$('.change-pass-form').hide();
    },

    /**
     * Validate login form:
     *
     * @return {Boolean}
     *
     */
    isValidationError: function() {
        var error = false,
            checkbox = this.$el.find('.swc-checkbox.display-password input'),
            inputPasswordOriginal = this.$el.find('input[name="login-password"]'),
            inputPasswordCopy = this.$el.find('input[name="login-password-copy"]'),
            inputLoginField = this.$el.find('input[name="login-name"]');

        // Copy password to original input if checkbox "Show Password" is active
        if (checkbox.prop('checked')) {
            inputPasswordOriginal.val(inputPasswordCopy.val());
        }

        if (!inputLoginField.val().length) {
            error = true;
        }

        if (!inputPasswordOriginal.val().length) {
            error = true;
        }

        return error;
    },

    /**
     * Display warning message to user:
     */
    displayWarningMessage: function(message) {
        var dialog = $('.info-messages').find('.' + message);

        dialog.show();

        // setTimeout(function() {
        //     dialog.fadeOut(function() {
        //         dialog.hide();
        //     });
        // }, 2500);

        if (message === "login"){
            $("input[name='login-password']").addClass("validation-error");
        }
    },

    // Called when user pressed "OK" on device and we go to next page
    afterOkPressed: function () {
        sessionStorage.setItem("changePassIncomplete", true);
        this.showChangePasswordForm();
    },

    /**
     * Handler for "Back" button
     * @callback
     */
    backToLogin: function () {
        var self = this;
        
        $.when(swc.models.PasswordRecovery.stopProcess())
            .done(function () {
                // We have to check current state of Password Recovery.
                // Should be "Enable", not "Processing" when "Back" is pressed.
                $.when(swc.models.PasswordRecovery.sync("read"))
                    .done(function () {
                        // stop listener
                        if (self.eventsListenerInterval) {
                            clearInterval(self.eventsListenerInterval);
                            self.eventsListenerInterval = null;
                        }

                        self.displaySignInForm();
                    });
            });
    },

    enableRecovery: function(e, value){
        swc.models.PasswordRecovery.set('recoveryEnable', value);
        this.$el.find('.save-password-button').removeClass('disabled');
    },

    /**
     * make the recovery field uppercase if you type smth in it.
     * we cannot do this in css file, because placeholder has lowercase symbols.
     * keydown + timeout works faster then keyup
     * @param e
     */
    makeUpperCase: function(e){
        var element = $(e.target);
        setTimeout(function(){
            element.css('text-transform', element.val() ? 'uppercase' : 'none');
        }, 1);
    },
    
    /**
     * Shows new password on "change password  page
     */
    showPassword: function(){
        var self = this,
            visibleField1 = this.$el.find('input[name=new-password-1]'),
            invisibleField1 = this.$el.find('input[name=new-password-1-copy]'),
            visibleField2 = this.$el.find('input[name=new-password-2]'),
            invisibleField2 = this.$el.find('input[name=new-password-2-copy]');

        invisibleField1.val(visibleField1.val());
        invisibleField2.val(visibleField2.val());

        visibleField1.attr('name', 'new-password-1-copy')
            .hide();

        invisibleField1.attr('name', 'new-password-1')
            .show();

        visibleField2.attr('name', 'new-password-2-copy')
            .hide();

        invisibleField2.attr('name', 'new-password-2')
            .show();

        self.delegateEvents();

    },

    /**
     * New password validation
     * @returns {boolean}
     */
    validatePassword: function(){
        var self = this,
            newPass1 = this.$('input[name=new-password-1]'),
            newPass1Val = $.trim(newPass1.val()),
            validationMessages = [];

        this.$('.validation-message').hide();
        this.$('input').removeClass('error valid');

        // Check initial password
        if (newPass1Val) {
            if (!window.validatePassword(newPass1Val)) {
                validationMessages.push('invalid password');
            }
        } else if (!newPass1Val) {
            validationMessages.push('invalid password');
        }
        
        if (validationMessages.length > 0) {
            newPass1.addClass('error').siblings('input').addClass('error');
            this.$('.pass1.validation-message, .pass1 .validation-message, .pass1 .error-message').show();
        } else {
            newPass1.addClass('valid').siblings('input').addClass('valid');
        }

        return validationMessages.length === 0;
    },

    /**
     * Check if password confirmed right
     * @returns {boolean}
     */
    validatePasswordsMatch: function () {
        var self = this,
            newPass1 = this.$('input[name=new-password-1]'),
            newPass1Val = $.trim(newPass1.val()),
            newPass2 = this.$('input[name=new-password-2]'),
            newPass2Val = $.trim(newPass2.val()),
            validationMessages = [];

        // Check confirmation password
        if (newPass2Val) {
            if (newPass1Val !== newPass2Val) {
                validationMessages.push('passwords do not match');
            }
        } else {
            validationMessages.push('passwords do not match');
        }

        if (validationMessages.length > 0) {
            newPass2.addClass('error').siblings('input').addClass('error');
            this.$('.pass2.validation-message, .pass2 .validation-message, .pass2 .error-message').show();
        } else {
            newPass2.addClass('valid').siblings('input').addClass('valid');
        }

        return validationMessages.length === 0;
    },

    /**
     * activate apply button on "change password" page
     */
    activateButtons: function () {
        this.$('.save-password-button').removeClass('disabled');
    },

    /**
     * Executes when we change the device password
     */
    applyPassword: function () {
        var self = this,
            currentPass = this.$('input[name=current-password]').val(),
            newPass1 = this.$('input[name=new-password-1]').val();

        this.$('.save-success, .save-error').hide();
        
        if (this.validatePasswordsMatch()) {
            this.showPageLoading("Applying changes...");
            $.when(swc.models.PasswordRecovery.setNewPassword(newPass1))
                .done(function(result){
                    // When no response, that means that everythins went fine, aka "Linux-style-error-handling" :-) 
                    if (result) {
                        self.stopPageLoading();
                        this.$('.info-messages .internal-error').show();
                    } else {
                        // Mark "change password" procedure as complete (finished)
                        sessionStorage.removeItem("changePassIncomplete");
                        
                        // NP replies immediately, but new password is not applied yet - wait for a while
                        setTimeout(function () {
                            // Now we have to automatically sign in
                            $.when(swc.models.Login.processLogin(swc.models.Login.adminName, newPass1))
                                .done(function() {
                                    // We only able to set password recovery after successfully signed it
                                    $.when(swc.models.PasswordRecovery.sync("update")).done(function () {
                                        // re-read status of password-recovery; otherwise will miss it after update
                                        swc.models.PasswordRecovery.sync("read");
                                    });
                                    // When everything done - move 
                                    swc.router.navigate(self._determineNextPage(), { trigger: true });
                                });
                        }, 1000);
                    }
                });
        }
    },

    /**
     * Update application locale based on dropdown value
     */
    setLocale: function(e, value) {
        $.when(swc.models.Locale.setLocale(value)).done(function() {
            swc.views.Login.renderComplete(true);
        });
    },

    /**
     * Determines to where redirect user after they successfully logged in - either overview or previous page
     * @private
     */
    _determineNextPage: function() {
        var previousPage = localStorage.getItem('previous-page'),
            location = 'overview';

        // Define page where user has to be navigated to:
        if (previousPage && _.indexOf(['login', 'remote-login'], previousPage) === -1) {
            location = previousPage;
        }
        
        // clear information about previous page 
        localStorage.removeItem('previous-page');
        
        return location;
    }

});
