swc.constructors.cloudAccountItem = Backbone.View.extend({

    pageTemplateID: 'storage:cloud:account-item',

    events: {
        'swc-checkbox:change .activate-cloud': 'toggleActivate',
        'click .do-custom-edit': 'edit',
        'click .do-disconnect': 'disconnect',
        'click .login-cloud': 'login',
        'click .sync-cloud': 'sync',
        'click .show-unsynced-files': 'showUnsyncedFiles'
    },

    render: function () {
        this.$el.empty();
        this.setTemplateData();
        this.$el.append($.tmpl(swc.Templates.get(this.pageTemplateID).get('content'), {
            data: this.templateData,
            localeStrings: swc.models.Locale.getLocaleStrings('storage'),
            localeString: getTranslationStringsjQuery,
            formatDate: swc.models.Locale.formatDate,
            GlobalEnable: swc.models.Application.get('GlobalEnable')
        }));

        return this;
    },

    showUnsyncedFiles: function() {
        var self = this,
            obj = this.model.getExtendedJSON(),
            data = _.extend(obj, {
                syncAction: false,
                loginAction: false
            });

        if (_.isEmpty(data.inaccessibleFiles)) {
            return;
        }

        data.inaccessibleFiles = _.map(data.inaccessibleFiles, function(file) {
            var list = file.split('/');

            list.splice(-1, 1, '<strong>' + list.slice(-1) + '</strong>');

            return list.join('/');
        });

        SWCElements.modalWindow.show({
            templateID: 'storage:cloud-backup:unsynced-files:modal-window',
            templateData: {
                syncDetails: data
            },
            onApply: function() {
                SWCElements.modalWindow.hide();
            }
        });
    },

    setTemplateData: function(){
        var self = this,
            obj = this.model.getExtendedJSON(),
            data;

        data = _.extend(obj, {
                syncAction: false,
                loginAction: false
            });

        switch (data.status) {

            case "IN_PROGRESS":
                data.tooltip = "Sync in progress";
                break;

            case "DISABLED":
                data.tooltip = "Sync stopped";
                break;

            case "COMPLETE":
                if (_.isEmpty(data.inaccessibleFiles)) {
                    data.tooltip = "Synced";
                } else {
                    data.status = 'INCOMPLETE';
                    data.tooltip = "Some files could not be synced";
                }
                break;

            case "ERROR":

                switch (data.error) {

                    case "NOT_ENOUGH_LOCAL_SPACE":
                        data.tooltip = "There is not enough space on your Central Storage to sync data. " +
                            "Please free some space.";
                        data.status = "WARNING";
                        break;

                    case "NOT_ENOUGH_REMOTE_SPACE":
                        data.tooltip = "There is not enough space on your Cloud Account %cloudService% " +
                            "(%accountName%) to sync data.";
                        data.status = "WARNING";
                        break;

                    case "INVALID_CREDENTIALS":
                        data.tooltip = "Could not connect to %cloudService%. Last login attempt on %lastLogin% has " +
                            "failed. Please try to login again later.";
                        data.loginAction = true;
                        break;

                    case "COMMUNICATION_ERROR":
                        data.tooltip = "There are troubles syncing some content. Try again by " +
                            "clicking \"Sync\" button or wait for 1 min for next sync attempt.";
                        data.status = "WARNING";
                        data.syncAction = true;
                        break;

                    default:
                        data.tooltip = "Unknown";
                }

                break;
            case "STORAGE_DISCONNECTED":
                data.tooltip = "Unable to connect to Cloud Account folder. Please check" +
                                " if your Central Storage device is connected to Internet-Box";
                data.deviceDisconnected = true;
                data.status = "ERROR";
                break;
            default:
                data.tooltip = "Unknown";
                break;
        }
        data.status = data.status.toLowerCase();

        this.templateData = data;
    },

    toggleActivate: function (e) {
        var value = $(e.currentTarget).data('value') ? 'COMPLETE' : 'DISABLED';
        this.model.set({status: value}, {silent: true});
    },

    edit: function (e) {
        var self = this;

        SWCElements.modalWindow.show({
            templateID: 'storage:cloud:cloud-accounts:edit-modal',
            templateData: {
                cloudServiceName: self.model.get('cloudServiceName'),
                accountName: self.model.get('accountName'),
                location: self.model.get('deviceName'),
                folderName: self.model.get('folderName')
            },
            className: 'cloud-edit-modal',

            onCancel: function() {
                SWCElements.modalWindow.hide();
                // Hide dropdown overlay if user presses "Cancel" when DropDown list is opened
                SWCElements.dropDown.closeAll();
                // Stop populating deviceDropDownValues
                swc.models.CentralStorage.off("central-storage-update");
            },

            onApply: function() {
                var deviceId = $('.modalWindow .device-selection').data('value'),
                    folderName = $('.modalWindow input[name=folderName]').val();
                $.when(self.folderNameValidation(deviceId, folderName))
                    .done(function () {
                        SWCElements.modalWindow.hide();
                        // Hide dropdown overlay if user presses "Apply" when DropDown list is opened
                        SWCElements.dropDown.closeAll();
                        swc.models.CloudAccountCollection.sync('read');
                        // Stop populating deviceDropDownValues
                        swc.models.CentralStorage.off("central-storage-update");
                    });
            },

            onShow: function () {
                // Set existing values to elements
                $('.modalWindow .device-selection').data("value", self.model.get('deviceId'));
                $('.modalWindow input[name=folderName]').val(self.model.get('folderName'));
                self.setDevicesDropdownValues();
                swc.models.CentralStorage.on("central-storage-update", self.setDevicesDropdownValues);
            }
        });
    },

    /**
     * Update usb devices dropdown with list of usb devices
     */
    setDevicesDropdownValues: function() {
        var dropdown = $('.modalWindow .swc-dropdown.device-selection'),
            validationMessages = $('.modalWindow .validation-message[data-parameter-name="usbUID"]'),
            validationMessage = validationMessages.find('.error-message[data-error="no connected devices"]'),
            oldValue = dropdown.data('value'),
            options = swc.models.CentralStorage.getDeviceDropdownOptions(),
            defaultDevice = swc.models.CentralStorage.getDefaultDeviceOption(options, oldValue);
        
        // Set dropdown data:
        dropdown.data('options', options);
        dropdown.data('hide-default', true);
        dropdown.trigger('swc-dropdown:swc-change', defaultDevice);
        
        // Validate dropdown to show message to the user
        if (defaultDevice === "no-devices-connected") {
            validationMessages.show();
            validationMessage.show();
            dropdown.addClass('validation-error');
        } else {
            validationMessages.hide();
            validationMessage.hide();
            dropdown.removeClass('validation-error');
        }
    },

    folderNameValidation: function(partitionUID, folderName) {
        var self = this,
            deferred = new $.Deferred();

        if (partitionUID === this.model.get('deviceId') && folderName === this.model.get('folderName')) {
            var validationMessages = [];
            validationMessages.push("already exists");
            this.setFolderNameValidationErrors(validationMessages);
            return deferred.reject();
        }

        // Try to create a folder
        $.when(swc.models.CloudServices.validateFolder(partitionUID, folderName))
            .done(function () {
                $.when(swc.models.CloudAccountCollection.moveFolder(self.model, partitionUID, folderName))
                    .done(function() {
                        var folderNameValidationContainer = $('.modalWindow .validation-message[data-parameter-name="folderName"]');

                        folderNameValidationContainer.find('.error-message').hide();
                        deferred.resolve();
                    })
                    .fail(function(error) {
                        var validationMessages = [];

                        switch (error.message) {
                            case 'invalid name':
                                validationMessages.push("unable to create folder");
                            break;

                            case 'already exists':
                                validationMessages.push("already exists");
                            break;

                            default:
                                validationMessages.push("unable to create folder");
                        }

                        self.setFolderNameValidationErrors(validationMessages);
                        deferred.reject();
                    });
            })
            .fail(function (status) {
                var validationMessages = [];
                switch (status.failure) {
                    case 'SYNC_FOLDER_ALREADY_EXISTS':
                        validationMessages.push("already exists");
                    break;

                    default:
                        validationMessages.push("unable to create folder");
                }
                self.setFolderNameValidationErrors(validationMessages);
                deferred.reject();
            });

        return deferred.promise();
    },

    setFolderNameValidationErrors: function(messsages) {
        var folderNameInput = $('.modalWindow input[name="folderName"]'),
            folderNameValidationContainer = $('.modalWindow .validation-message[data-parameter-name="folderName"]');

        // Add validation error to the field:
        folderNameInput.addClass('validation-error');
        folderNameInput.data('validation-error', true);

        // Show validation error in UI:
        folderNameValidationContainer.show();
        folderNameValidationContainer.find('.error-message').hide();
        folderNameValidationContainer.find('.error-message[data-error="' + messsages[0] + '"]').show();
    },

    disconnect: function (e) {
        var self = this;
        swc.views.Application.showPageLoading('Loading page data..');
        $.when(swc.models.CentralStorage.sync('read'))
            .done(function () {
                var devices = swc.models.CentralStorage.get('usb-devices').get('devices'),
                    device = _.find(devices, function (obj){
                        return obj.id === self.model.get('deviceId');
                    });
                if (device) {
                    self.disconnectModal();
                } else {
                    self.doDisconnect(true);
                }
            }).always(function () {
                swc.views.Application.stopPageLoading();
            });
    },

    disconnectModal: function () {
        var self = this;
        SWCElements.modalWindow.show({
            templateID: 'storage:cloud:cloud-accounts:disconnect-modal',
            templateData: {
                cloudServiceName: getTranslationStrings(self.model.get('cloudServiceName'), true),
                accountName: self.model.get('accountName')
            },
            className: 'cloud-disconnect-modal',

            onCancel: function() {
                SWCElements.modalWindow.hide();
            },

            onApply: function() {
                var keepData = getParameter($('.cloud-disconnect-modal .keep-data-selection'));
                self.doDisconnect(keepData.parameterValue === 'keep');
            }
        });
    },

    doDisconnect: function (keepData) {
        var self = this,
            validationMessageContainer = $(".modalWindow .save-validation-errors");
        
        $.when(this.model.logout(keepData))
            .done(function () {
                self.$el.remove();
                validationMessageContainer.hide();
                SWCElements.modalWindow.hide();
                
                // Remove cloud service id on logout process
                sessionStorage.removeItem("cloudService");
                
                // If no accounts left - redirect user to Cloud Login page
                if (swc.models.CloudAccountCollection.models.length === 0) {
                    swc.router.navigate('storage/settings/cloud-backup', { trigger: false, skipUnsavedChanges: true});
                    return;
                }
            })
            .fail(function (result) {
                validationMessageContainer.show();
                validationMessageContainer.find("div[data-error='can not remove folder']").show();
            });
    },

    login: function (e) {
        swc.models.CloudServices.checkCredentials();
    },

    forceSyncDevice: function(deviceId){
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                "service": "com.swisscom.stargate/ws/cloud/backup.com.swisscom.stargate.cloud.backup",
                "method": "forceFailedTasksForPartition",
                "parameters": {
                    "partitionUUID": deviceId
                }
            },

            success: function(response) {
                // Timeout is used because the board responds immediately
                // but actually needs some time to change the sync state.
                // sad but true :(
                setTimeout(function(){
                    deferred.resolve();
                }, 3000);
            }
        });

        return deferred.promise();
    },

    /**
     * Starts sync for the service
     * @deprecated
     * @param e
     */
    sync: function (e) {
        var self = this;

        self.showPageLoading('Saving page data..');
        $.when(this.forceSyncDevice(this.model.get('deviceId')))
            .done(function(){
                swc.models.CloudAccountCollection.sync('read')
                    .done(function(){
                        self.stopPageLoading();
                    });
            });
    }
});
