swc.constructors.StorageSettingsCloudbackupView  = swc.base.TabView.extend({

    className: 'cloud-backup',

    listenerEnabled: true,

    listenerPages: [ 'cloud-backup' ],

    models: [
        'СloudServicesStatus',
        'CloudServices',
        'CentralStorage',
        'Network'
    ],

    validation: {
        usbUID: 'CloudAccountCollection:usbUID',
        folderName: 'CloudAccountCollection:folderName'
    },

    events: {
        "swc-radio-buttons:change .select-cloud-service": "onChangeCloudService",
        "swc-dropdown:change .device-selection": "onChangeDevice"
    },

    setTemplateData: function() {
        // TODO :: change <cloudService> and <cloudServicesList> calls to the collection:
        this.templateData = {
            centralStorageState: swc.models.CentralStorage.isEnabled(),
            cloudService: sessionStorage.getItem('cloudService') || "dropbox", // Default service - 'Dropbox'
            cloudServicesList: [
                { id: "dropbox", name: "Dropbox" },
                { id: "googledrive", name: "Google Drive" }
            ]
        };
    },

    preRender: function () {
        var deferred = new $.Deferred(),
            self = this,
            oauthCallbackParams,
            oauthCallbackParamsString = sessionStorage.getItem("oauthCallbackParams"),
            syncFolderDataString = sessionStorage.getItem('syncFolderData');

        sessionStorage.removeItem("oauthCallbackParams");
        sessionStorage.removeItem("syncFolderData");

        if (syncFolderDataString) {
            try {
                this.syncFolderData = JSON.parse(syncFolderDataString);
            } catch (e) {
                return;
            }
        }

        this.errorMessages = [];
        this.folderValidationMessages = [];

        if (!swc.models.CloudAccountCollection) {
            swc.models.CloudAccountCollection = new swc.constructors.CloudAccountCollection();
        }

        if (oauthCallbackParamsString) {
            // authentication is in progress (redirected from oauthHandler):
            this.stopPageLoading();
            this.showPageLoading("Processing login to Cloud Service");

            try {
                oauthCallbackParams = JSON.parse(oauthCallbackParamsString);
                if (!_.isUndefined(oauthCallbackParams.error) && oauthCallbackParams.error === "access_denied") {
                    throw 'access denied';
                }
                if (oauthCallbackParams.not_approved) {
                    throw 'not approved';
                }

                $.when(swc.models.CloudAccountCollection.handleOauthCallback(oauthCallbackParams))
                    .done(function(response){
                        var folderPromise = self.syncFolderData ? self.folderNameValidation(self.syncFolderData) : true;
                        // Create folder used saved data from sessionStorage 
                        $.when(folderPromise)
                            .done(function() {
                                self.syncFolderData = null; // remove folder data before navigate
                                self.navigateToList(); // switch to another view
                                deferred.reject(); // stop rendering this view
                            })
                            .fail(function(validationMessages){
                                // proceed rendering with validation messages:
                                self.folderValidationMessages = validationMessages;
                                deferred.resolve();
                            });
                    })
                    .fail(function(){
                        // proceed rendering with validation messages:
                        self.errorMessages.push('could-not-connect');
                        deferred.resolve();
                    });
            } catch(e) {
                // oauth failed
                // proceed rendering with validation messages 
                // (currently the same error for all cases):
                this.errorMessages.push('could-not-connect');
                deferred.resolve();
            }
        } else {
            // authentication is not in progress:

            if (!swc.models.CentralStorage.isEnabled()) {
                // proceed rendering:
                deferred.resolve();
            } else {
                $.when(swc.models.CloudAccountCollection.sync('read'))
                    .done(function () {
                        if (swc.models.CloudAccountCollection.models.length > 0) {
                            self.navigateToList(); // switch to another view
                            deferred.reject(); // stop rendering this view
                        } else {
                            // proceed rendering:
                            deferred.resolve();
                        }
                    }).fail(function () {
                        // can't load collection but we don't really need it to render the page, so
                        // proceed rendering:
                        deferred.resolve();
                    });
            }
        }

        return deferred.promise();
    },

    onListenerComplete: function() {
        var apServiceLoadingState = swc.models.apServiceLoadingState.get('isLoading');

        // Check if AP turned off or DBUS crashed
        if (apServiceLoadingState === true) {
            swc.router.navigate('storage/settings/data-sharing', { trigger: false, skipUnsavedChanges: true });
        } else {
            if (!swc.models.СloudServicesStatus.get('isEnabled')) {
                this.onCloudServicesDisable();
            }
        }
    },

    /**
     * Listen to cloud services `disable` event
     *
     * @description:
     *
     * <СloudServicesStatus> model is listening to TR-069 parameter, which has a flag if cloud services are enabled to
     * the user. This parameter can be set to false during user work with cloud services. This method will prevent all
     * user actions and show modal window with redirection to other page
     */
    onCloudServicesDisable: function() {
        this.stopListener();

        SWCElements.modalWindow.show({
            templateID: 'storage:settings:modal-windows:disable-cloud-services',
            templateData: {},
            className: 'disable-cloud-services',

            onApply: function() {
                SWCElements.modalWindow.hide();
                swc.router.navigate('storage/settings/data-sharing', { trigger: false, skipUnsavedChanges: true });
            }
        });
    },

    navigateToList: function () {
        swc.router.navigate('storage/settings/cloud-backup/list', { trigger: false, skipUnsavedChanges: true});
    },

    setDefaults: function () {
        // Set default value for service
        this.$(".swc-radio-buttons.select-cloud-service").data("default-value", this.templateData.cloudService);

        if (this.syncFolderData) {
            // for storage
            this.$(".swc-dropdown.device-selection").data("default-value", this.syncFolderData.partitionUID);
            // for folder name
            this.$("input[name=folderName]").data("default-value", this.syncFolderData.folderName);
        } else {
            this.$(".swc-dropdown.device-selection").data("default-value", "null");
        }
    },

    /**
     * Set predefined values after page has been rendered:
     */
    renderComplete: function() {
        var self = this;

        // Check if Central Storage is Enabled, else -> disable page
        if (!swc.models.CentralStorage.isEnabled()) {
            return;
        }

        if (!swc.models.СloudServicesStatus.get('isEnabled')) {
            this.onCloudServicesDisable();
        }

        // Call default methods to (re)set values on the page:
        this.setDefaults();
        this.onChangeCloudService(null, this.templateData.cloudService, true);
        this.setDevicesDropdownValues(true);

        // Populate folder data
        if (this.syncFolderData) {
            this.$el.find(".device-selection").trigger("swc-dropdown:swc-change", this.syncFolderData.partitionUID);
            this.syncFolderData = null;
        }

        if (this.showSaveError) {
            this.showSaveError = false;
            this.errorMessages.push('can-not-receive-url');
        }

        if (!_.isEmpty(this.folderValidationMessages)) {
            this.setFolderNameValidationErrors(this.folderValidationMessages);
            this.folderValidationMessages = [];
        }

        if (!_.isEmpty(this.errorMessages)) {
            this.showSaveSuccess('error', this.errorMessages);
        }

        if (!this.devicesDropdownInterval) {
            this.devicesDropdownInterval = setInterval(function () {
                if (self.$el.parents('body').length > 0) {
                    $.when(swc.models.CentralStorage.sync('read', {silent: true}))
                        .done(function () {
                            self.setDevicesDropdownValues();
                        });
                } else {
                    // if the current view is not in the DOM
                    // we don't need the interval anymore:
                    clearInterval(self.devicesDropdownInterval);
                }
            }, 10000);
        }

        this.stopPageLoading();
    },

    /**
     * Change cloud service by user / application. Update correct model in cloud services collection:
     * @param e {Object} Event
     * @param cloudServiceID {String}
     */
    onChangeCloudService: function(e, cloudServiceID) {
        var serviceData = _.findWhere(this.templateData.cloudServicesList, { id: cloudServiceID }),
            serviceName = serviceData ? serviceData.name : '',
            input = this.$("input[name=folderName]"),
            isCalledFromRenderComplete = _.isNull(e);

        // TODO :: change service in collection

        if (this.syncFolderData && this.syncFolderData.folderName) {
            input.val(this.syncFolderData.folderName);
        } else if (input.data('default-value') === input.val() || isCalledFromRenderComplete) {
            input.val(serviceName);
        }

        input.data('default-value', serviceName);

        if (!isCalledFromRenderComplete) {
            input.trigger('validate');
        }
    },

    onChangeDevice: function (e) {
        var input = this.$("input[name=folderName]");

        if (e) {
            input.trigger('validate');
        }
    },

    /**
     * Update usb devices dropdown with list of usb devices
     * @param isSkipValidation {Boolean} skip / not skip validation for folder name
     */
    setDevicesDropdownValues: function(isSkipValidation) {
        var dropdown = this.$el.find('.swc-dropdown.device-selection'),
            oldValue = dropdown.data('value'),
            input = this.$("input[name=folderName]"),
            options = swc.models.CentralStorage.getDeviceDropdownOptions(),
            defaultDevice = swc.models.CentralStorage.getDefaultDeviceOption(options, oldValue),
            defaultDeviceValue = dropdown.data('default-value');

        // Set default value to the dropdown if not set:
        dropdown.data('default-value', defaultDevice);

        // Set dropdown data:
        dropdown.data('options', options);
        dropdown.data('hide-default', true);
        dropdown.trigger('swc-dropdown:swc-change', defaultDevice);

        // Validate dropdown to show message to the user
        if (defaultDevice === "no-devices-connected") {
            this.pageValidation(dropdown, false);
        } else {
            this.clearValidationMessages(dropdown);

            // Validate folder name only when user did some changes on the page:
            if (!isSkipValidation && !this.pageCheckDefaultValues()) {
                input.trigger('validate');
            }
        }
    },

    folderNameValidation: function(syncFolderData) {
        var self = this,
            deferred = new $.Deferred();

        // Try to create a folder before login to CS
        $.when(swc.models.CloudAccountCollection.createFolder(syncFolderData.partitionUID, syncFolderData.folderName))
            .done(function() {
                deferred.resolve();
            })
            .fail(function(error) {
                var validationMessages = [];

                switch (error.message) {
                    case 'invalid name':
                        validationMessages.push("unable to create folder");
                    break;

                    case 'already exists':
                        validationMessages.push("already exists");
                    break;

                    default:
                        validationMessages.push("unable to create folder");
                }

                deferred.reject(validationMessages);
            });

        return deferred.promise();
    },

    /**
     * Show validation errors of folder name in UI, after validation was not completed on AP:
     * @param messsages
     */
    setFolderNameValidationErrors: function(messsages) {
        var folderNameInput = this.$el.find('input[name="folderName"]'),
            folderNameValidationContainer = this.$el.find('.validation-message[data-parameter-name="folderName"]');

        folderNameValidationContainer.find('.error-message').hide();

        if (!_.isEmpty(messsages)) {
            // Add validation error to the field:
            folderNameInput.addClass('validation-error');
            folderNameInput.data('validation-error', true);

            // Show validation error in UI:
            folderNameValidationContainer.show();
            folderNameValidationContainer.find('.error-message[data-error="' + messsages[0] + '"]').show();
        } else {
            folderNameInput.removeClass('validation-error');
            folderNameInput.data('validation-error', false);
            folderNameValidationContainer.hide();
        }
    },

    /**
     * Save method will be called when user will press "Log In" button;
     * @override
     */
    save: function() {
        var self = this,
            cloudService = this.$el.find(".select-cloud-service").data('value'),
            syncFolderData = {
                partitionUID: this.$el.find(".device-selection").data('value'),
                folderName: this.$el.find("input[name='folderName']").val()
            },
            deferred = new $.Deferred(),
            validationContainer = this.$('.validation-message[data-parameter-name="global"]');
        
        // Show user login message (not "saving changes")
        this.stopPageLoading();
        this.showPageLoading("Processing login to Cloud Service");

        validationContainer.hide();
        validationContainer.find('.error-message').hide();

        $.when(swc.models.Network.sync())
            .done(function() {
                // Check if device connected to the internet TODO :: remove after implementation on AP side
                if (!swc.models.Network.getParameter('ConnectionStatus', 'status')) {
                    self.stopPageLoading();
                    self.showSaveError = true;
                    
                    return deferred.reject("can-not-receive-url");
                }
                $.when(swc.models.CloudServices.validateFolder(syncFolderData.partitionUID, syncFolderData.folderName))
                    .done(function() {
                        // Steps required to login:
                        // 1. getAuthURL()
                        // 2. handleOAtuhCallback()
                        // 3. syncFSResources() == create a folder
                        $.when(swc.models.CloudAccountCollection.getLoginURL(cloudService))
                            .done(function(url) {
                                // save folder name into local storage (or SessionStorage)
                                sessionStorage.setItem('syncFolderData', JSON.stringify(syncFolderData));
                                sessionStorage.setItem('cloudService', cloudService);
                                document.location.href = url;
                            })
                            .fail(function() {
                                self.showSaveError = true;
                                self.stopPageLoading();
                                deferred.reject();
                            });
                    })
                    .fail(function() {
                        self.stopPageLoading();
                        deferred.reject();
                    });
            });

        return deferred.promise();
    }
});
