swc.constructors.Telephony = Backbone.Model.extend({

    dectInternalNumbers: [ 203, 204, 205, 206, 207, 208, 209 ],

    sync: function(fromListener){
        var self = this,
            deferred = new $.Deferred();

        $.when(
                self.getPin(fromListener),
                self.getPhones(fromListener),
                self.getDectStatus(fromListener),
                self.getDectEco(fromListener)
            ).done(function(){
                self.trigger('change');
                deferred.resolve();
            });

        return deferred.promise();
    },

    initialize: function(){
        var self = this,
            PhonesCollectionConstructor = Backbone.Collection.extend({
                comparator: function(model){
                    return model.id;
                }
            }),
            phonesCollection = new PhonesCollectionConstructor(),
            groupsCollection = new Backbone.Collection();

        this.set('groups', groupsCollection);
        this.set('phones', phonesCollection);
        this.get('phones').on('change:externalNumber', function(model, values, options){
            if(!options || (options && !options.notProceed)){
                var newNumber = model.get('externalNumber');
                var numberFound = false;
                self.get('voip').each(function(voipModel){
                    if(voipModel.get('directoryNumber') === newNumber){
                        numberFound = true;
                        model.set({'outgoingTrunkLine': voipModel.id}, {notProceed: true});
                    }
                });
                if(!numberFound){
                    model.set({'externalNumber': self.get('voip').get(model.get('outgoingTrunkLine')).get('directoryNumber')}, {notProceed: true});
                }
            }

        });

        this.get('phones').on('change:outgoingTrunkLine', function(model, values, options){
            if(!options || (options && !options.notProceed)){
                model.set({'externalNumber': self.get('voip').get(model.get('outgoingTrunkLine')).get('directoryNumber')}, {notProceed: true});
            }
        });

        this.get('phones').on('change:incomingLines', function(model, values, options){
            var activeLines = self.get('voip').where({'enable': 'Enabled'});

            _.each(activeLines, function(line){
                var lineGroup = self.get('groups').get(line.get('groupId')),
                    devices = lineGroup.get('ep_names'),
                    deviceOptions = _.where(values, {'line': line.id})[0],
                    changed = false;

                if(deviceOptions.value){
                    if(!_.contains(devices, model.id)){
                        devices.push(model.id);
                        changed = true;
                    }
                } else {
                    if(_.contains(devices, model.id)){
                        devices = _.without(devices, model.id);
                        changed = true;
                    }
                }

                if(changed){
                    lineGroup.set({'ep_names': devices, 'changed': true});
                }
            });
        });
    },

    validation: {

        dectPin: function(data) {
            var validationMessages = [],
                validationData = swc.Utils.getDataToValidate({
                    'dectPin' : { elementName: 'pin', elementClass: 'pin-original' }
                }, data);

            if ($.trim(validationData['dectPin']).length < 4) {
                validationMessages.push('must contain 4 symbols');
            }

            if (!/^[\d]*$/.test(validationData['dectPin'])) {
                validationMessages.push('must contain 4 digits from 0 to 9');
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        dectName: function(data) {
            var validationMessages = [],
                validationData = swc.Utils.getDataToValidate({
                    'dectName' : { elementName: 'dect-name-editable' }
                }, data);

            if (_.isEmpty($.trim(validationData['dectName']))) {
                validationMessages.push('must not be empty');
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        },

        internalNumber: function(data) {
            var validationMessages = [],
                validationData = swc.Utils.getDataToValidate({
                    'internalNumber' : { elementName: 'internal-number-editable' }
                }, data);

            if (_.isUndefined(swc.models.Telephony)) {
                return {
                    status: _.isEmpty(validationMessages),
                    messages: validationMessages
                };
            }

            var phones = swc.models.Telephony.get('phones'),
            // get integer array of used phone numbers
                usedInternalNumbers = _.map(phones.pluck('directoryNumber'), function(v) { return +v; }),
            // get integer array of similar phone numbers
                similarValues = _.filter(usedInternalNumbers, function(v) { return v === validationData['internalNumber']; });

            // if the length of similar phone numbers > 1 then error should be shown
            if (similarValues.length > 1) {
                validationMessages.push('number already is used');
            }

            return {
                status: _.isEmpty(validationMessages),
                messages: validationMessages
            };
        }
    },

    /**
     * Get voip status
     * @returns json
     * {
     *  "status":[
     *      {
     *          "name":"SIP-Trunk",
     *          "signalingProtocol":"SIP",
     *          "enable":"Disabled",
     *          "trunk_lines":[
     *              {
     *                  "name":"LINE1",
     *                  "groupId":"Group1",
     *                  "enable":"Disabled",
     *                  "status":"Disabled",
     *                  "statusInfo":"",
     *                  "directoryNumber":"",
     *                  "uri":"",
     *                  "authUserName":"",
     *                  "authPassword":"",
     *                  "event_subscribe_lines":[
     *                      {
     *                          "eventSubscribeEvent":"message-summary",
     *                          "eventSubscribeAuthUserName":"",
     *                          "eventSubscribeAuthPassword":""
     *                      }
     *                  ]
     *              },
     *              {
     *                  "name":"LINE2",
     *                  "groupId":"Group2",
     *                  "enable":"Disabled",
     *                  "status":"Disabled",
     *                  "statusInfo":"",
     *                  "directoryNumber":"",
     *                  "uri":"",
     *                  "authUserName":"",
     *                  "authPassword":"",
     *                  "event_subscribe_lines":[
     *                      {
     *                          "eventSubscribeEvent":"message-summary",
     *                          "eventSubscribeAuthUserName":"",
     *                          "eventSubscribeAuthPassword":""
     *                      }
     *                  ]
     *              }
     *          ],
     *          "sip":{
     *              "proxyServer":"imst.swisscom.ch",
     *              "proxyServerPort":5060,
     *              "registrarServer":"imst.swisscom.ch",
     *              "registrarServerPort":5060,
     *              "outboundProxyServer":"",
     *              "outboundProxyServerPort":5060,
     *              "userAgentDomain":"imst.swisscom.ch",
     *              "userAgentPort":5060,
     *              "subscriptionInfo":[
     *                  {
     *                      "event":"message-summary",
     *                      "notifyServer":"",
     *                      "notifyServerPort":5060
     *                  }
     *              ]
     *          },
     *          "h323":{}
     *      },
     *      {
     *          "name":"H323-Trunk",
     *          "signalingProtocol":"H.323",
     *          "enable":"Disabled",
     *          "trunk_lines":[
     *              {
     *                  "name":"LINE3",
     *                  "groupId":"Group3",
     *                  "enable":"Disabled",
     *                  "status":"Disabled",
     *                  "statusInfo":"",
     *                  "directoryNumber":"1011",
     *                  "uri":"",
     *                  "authUserName":"",
     *                  "authPassword":"",
     *                  "event_subscribe_lines":[]
     *              }
     *          ],
     *          "sip":{},
     *          "h323":{
     *              "gatekeeper":"172.16.1.1",
     *              "gatekeeperPort":1719
     *          }
     *      }
     *  ]}
     */
    getVoipStatus: function(fromListener){
        var self = this, deferred = new $.Deferred();
        swc.models.Rest.sendRequest({
            url: '/sysbus/VoiceService/VoiceApplication:listTrunks',
            fromListener: fromListener,
            data: {
                "parameters": {}
            },

            success: function(response) {
                var swisscomSip = _.where(response.status, {signalingProtocol: "SIP"})[0],
                    Lines = Backbone.Collection,
                    lines = new Lines(),
                    voipActivated;

                // NOTE: we do not take global SIP status - swisscomSip.enable. We take it from the lines status

                if (swisscomSip) {
                    _.each(swisscomSip.trunk_lines, function(line){
                        var sipLine = new Backbone.Model();
                        sipLine.set({
                            id: line.name,
                            directoryNumber: line.directoryNumber,
                            enable: line.enable,
                            groupId: line.groupId,
                            name: line.name,
                            status: line.status,
                            statusInfo: line.statusInfo,
                            uri: line.uri
                        });

                        if(sipLine.get('enable') === 'Enabled'){
                            lines.add(sipLine);
                        }
                    });
                }

                // Check if lines collection is not empty
                voipActivated = (lines.length > 0);

                // voipStatus means that at least one line is activated
                self.set({
                    'voip': lines,
                    "voipStatus": voipActivated
                });

                deferred.resolve();
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    /**
     * Get groups.
     * Used for incoming numbers - Calls to related lines(LINE1 is related to Group1) will redirect to devices listed in "ep_names" array
     * @returns {*}
     *
     * @example
     * {"status":[
     *      {
     *          "group_id":"Group1",           – Group name, related to line name. Group1 = LINE1 etc...
     *          "ep_names":["FXS1","FXS2"]     – List of devices in group
     *      },{
     *          "group_id":"Group2",
     *          "ep_names":["FXS2"]
     *      }
     * ]}
     */
    getGroups: function(fromListener){
        var self = this, deferred = new $.Deferred();
        swc.models.Rest.sendRequest({
            url: '/sysbus/VoiceService/VoiceApplication:listGroups',
            fromListener: fromListener,
            data: {
                "parameters": {}
            },

            success: function(response) {
                var groupCol = [];
                _.each(response.status, function(group){
                    var model = {
                        id: group.group_id,
                        ep_names: group.ep_names,
                        changed: false
                    };
                    groupCol.push(model);
                });
                self.get('groups').reset(groupCol);
                deferred.resolve();
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    setGroups: function(){
        var self = this, deferred = new $.Deferred(), groupArray = [];

        self.get('groups').each(function(model){
            if(model.get('changed')){
                var obj = {
                    "group_id": model.id,
                    "ep_names": model.get('ep_names')
                };
                groupArray.push(obj);
            }
        });

        swc.models.Rest.sendRequest({
            url: '/sysbus/VoiceService/VoiceApplication:setGroups',
            data: {
                "parameters": {
                    "groups": groupArray

                }
            },

            success: function(response) {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    /**
     * Get voip lines
     * @returns json
     * {
     *      "status":[
     *          {
     *              "line":"FXS1",
     *              "name":"FXS1",
     *              "enable":"Enabled",
     *              "directoryNumber":"201",
     *              "endpointType":"FXS",
     *              "outgoingTrunkLine":"LINE1"
     *          },
     *          {
     *              "line":"HS0106EA54BB",
     *              "name":"HS0106EA54BB",
     *              "enable":"Enabled",
     *              "directoryNumber":"204",
     *              "endpointType":"DECT GAP",
     *              "outgoingTrunkLine":"LINE1"
     *          },
     *          {
     *              "line":"HS024270E3C1",
     *              "name":"HS024270E3C1",
     *              "enable":"Enabled",
     *              "directoryNumber":"205",
     *              "endpointType":"DECT CAT-iq 1.0",
     *              "outgoingTrunkLine":"LINE1"
     *          },
     *          {
     *              "line":"Account311",
     *              "name":"Account311",
     *              "enable":"Disabled",
     *              "directoryNumber":"311",
     *              "endpointType":"SIP",
     *              "outgoingTrunkLine":"LINE1"
     *          }
     *      ]
     *  }
     */
    getPhones: function(fromListener){
        var self = this, deferred = new $.Deferred();

        function getLines(){
            var deferred = new $.Deferred(),
                activeLines = self.get('voip');

            swc.models.Rest.sendRequest({
                url: '/sysbus/VoiceService/VoiceApplication:listHandsets',
                fromListener: fromListener,
                data: {
                    "parameters": {}
                },

                success: function(response) {
                    var lines = [];

                    _.each(response.status, function(obj){
                        if(obj.endpointType.indexOf('DECT')>=0 || obj.endpointType === "FXS"){
                            var line = new Backbone.Model(), deviceId = obj.line, incomingLines = [], deviceType = 'dect';

                            activeLines.each(function(line){
                                var key = line.id, value = false;
                                var lineGroup = self.get('groups').get(line.get('groupId'));
                                if(lineGroup){
                                    if(_.contains(lineGroup.get('ep_names'), deviceId)){
                                        value = true;
                                    }
                                    incomingLines.push({'line': key, 'number': line.get('directoryNumber'), 'value': value});
                                } else {
                                    /*
                                     If groupId in lines and groupId in groups are not the same, the gateway setup is wrong. So we will show this modal window
                                     */
                                    if(!$.cookie('noGroups')){
                                        $.cookie('noGroups', true);
                                        new swc.constructors.NoGroupsModal();
                                    }
                                }

                            });

                            if(obj.endpointType === "FXS"){
                                deviceType = 'wired';
                            }

                            line.set({
                                id: deviceId,
                                line: deviceId,
                                deviceType: deviceType,
                                name: obj.name,
                                enable: (obj.enable === "Enabled"),
                                directoryNumber: obj.directoryNumber,
                                externalNumber: self.get('voip').get(obj.outgoingTrunkLine) ? self.get('voip').get(obj.outgoingTrunkLine).get('directoryNumber') : "",
                                incomingLines: incomingLines,
                                type: 'DECT',
                                outgoingTrunkLine: obj.outgoingTrunkLine,
                                is_changed: false
                            });



                            lines.push(line);
                        }
                    });

                    self.get('phones').reset(lines);
                    deferred.resolve();
                },
                error: function() {
                    deferred.reject();
                }
            });

            return deferred.promise();
        }

        $.when(self.getVoipStatus(fromListener), self.getGroups(fromListener)).done(function(){
            $.when(getLines(fromListener)).done(function(){
                deferred.resolve();
            });
        });

        return deferred.promise();
    },

    ringPhone: function(phone){
        var self = this, deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/VoiceService/VoiceApplication:ring',
            method: 'POST',
            contentType: 'application/x-sah-ws-4-call+json',

            data: {
                "parameters": {
                    "PhoneToUse":phone,
                    "Duration":10000,
                    "Ringtone":""
                }
            },

            success: function(response) {
                // TODO: put this error handler in the common place
                // in this case response can be successful and
                // contain error message. That is why it should be
                //  checked on it here
                if (_.isUndefined(response.errors)) {
                    setTimeout(function() { deferred.resolve(); }, 1000);
                } else {
                    deferred.reject();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();

    },

    startPairing: function(){
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/DECT:startPairing',

            data: {
                "parameters": {
                }
            },

            success: function(response) {
                deferred.resolve(response);
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    stopPairing: function() {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/DECT:stopPairing',

            data: {
                "parameters": {
                }
            },

            success: function(response) {
                deferred.resolve(response);
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    unpairDect: function(id){
        var self = this, deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/VoiceService/VoiceApplication:deleteHandset',
            data: {
                "parameters": {
                    "line": id
                }
            },

            success: function(response) {
                deferred.resolve(response);
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    /**
     * Get PIN for device pairing
     * @returns json
     * {"status":"0000"}
     */

    getPin: function(fromListener){
        var self = this, deferred = new $.Deferred();
        swc.models.Rest.sendRequest({
            url: '/sysbus/DECT:getPIN',
            fromListener: fromListener,

            data: {
                "parameters": {
                }
            },

            success: function(response) {
                self.set('dectPIN', response.status);
                deferred.resolve();
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    setPin: function(){
        var self = this, deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/DECT:setPIN',
            data: {
                "parameters": {
                    "pin": self.get('dectPIN')
                }
            },

            success: function(response) {
                setTimeout(function(){
                    deferred.resolve();
                }, 1000);
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    phoneEdit: function(){

        var self = this, deferred = new $.Deferred(),
            changedModels = this.get('phones').filter(function(model){
                return model.get('is_changed');
            });

        var editsArray = [];

        if(changedModels.length === 0){
            deferred.resolve();
        }

        _.each(changedModels, function(model) {
            var editRequest = function() {
                var deferred = new $.Deferred(),
                    extNumber = "" + model.get('externalNumber'),
                    incomingLines = model.get('incomingLines'),
                    lines = _.where(incomingLines, {number: extNumber}),
                    outgoingTrunkLine = !_.isEmpty(lines) ? lines[0].line : model.get('outgoingTrunkLine');

                swc.models.Rest.sendRequest({
                    url: '/sysbus/VoiceService/VoiceApplication:setHandset',
                    data: {
                        "parameters": {
                            "line": {
                                "line": model.get('line'),
                                "directoryNumber": model.get('directoryNumber'),
                                "outgoingTrunkLine": outgoingTrunkLine,
                                "name": model.get('name')
                            }
                        }
                    },

                    success: function(response) {
                        deferred.resolve();
                    },
                    error: function() {
                        deferred.reject();
                    }
                });

                return deferred.promise();
            };

            editsArray.push(editRequest());
        });

        $.when.apply(this, editsArray).done(function(){
            $.when(self.setGroups()).done(function(){
                setTimeout(function(){ deferred.resolve(); }, 2000);
            });
        });

        return deferred.promise();
    },

    dectReset: function(){
        var self = this, deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/DECT:reset',
            data: {
                "parameters": {
                }
            },

            success: function(response) {
                deferred.resolve();
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    dectRemoveDects: function(){
        var self = this, deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/DECT:removeAllHandsets',
            data: {
                "parameters": {
                }
            },

            success: function(response) {
                deferred.resolve();
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    setDectStatus: function(){
        var self = this, deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/DECT:setBaseState',
            data: {
                "parameters": {
                    "state": self.get('dectStatus')
                }
            },

            success: function(response) {
                // Give DECT a little time to turn on
                setTimeout(function(){
                    deferred.resolve();
                }, self.get('dectStatus')?2000:1);
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();

    },

    getDectStatus: function(fromListener){
        var self = this, deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/DECT:getBaseState',
            fromListener: fromListener,

            data: {
                "parameters": {}
            },

            success: function(response) {
                self.set('dectStatus', response.status);
                deferred.resolve();
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    setDectEco: function(){
        var self = this, deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/DECT:setNEMoState',
            data: {
                "parameters": {
                    "state": self.get('dectEco')
                }
            },

            success: function(response) {
                setTimeout(function(){
                    deferred.resolve();
                }, 1500);
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();

    },

    getDectEco: function(fromListener){
        var self = this, deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/DECT:getNEMoState',
            fromListener: fromListener,

            data: {
                "parameters": {}
            },

            success: function(response) {
                self.set('dectEco', response.status);
                deferred.resolve();
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();

    },

    getPhonesList: function() {
        var dectPhonesArray = [],
            wiredPhonesArray = [],
            self = this;

        this.get('phones').each(function(model, index) {
            var obj = {
                key: index+1,
                id: model.id,
                isDisabled: !model.get('enable'),
                name: model.get('name'),
                internalNumber: model.get('directoryNumber'),
                externalNumber: model.get('externalNumber'),
                incomingLines: model.get('incomingLines')
            };

            if (model.get('deviceType') === "dect") {
                dectPhonesArray.push(obj);
            } else {
                wiredPhonesArray.push(obj);
            }

            dectPhonesArray.sort(function (a, b) {
                if (b.internalNumber < a.internalNumber) {
                    return 1;
                }

                return -1;
            });
        });

        return {
            dect: dectPhonesArray,
            cord: wiredPhonesArray
        };
    }
});

