swc.constructors.System = Backbone.Model.extend({

    adminName: "admin",

    groups: [
        "http",
        "admin"
    ],

    minExpectedRebootTime: 160000, // 2 minutes 40 seconds
    maxExpectedRebootTime: 240000, // 4 minutes

    sync: function(fromListener){
        var self = this,
            deferred = new $.Deferred();

        // As there are no request in the $.when() function which checks
        // session state the checkSessionState() method should wrap them
        self.checkSessionState(function() {
            $.when(self.identifyGateway(fromListener), self.getFirmware(fromListener))
                .done(function () {
                    deferred.resolve();
                });
        });

        return deferred.promise();
    },

    /**
     * Calculates quantity of days, hours and minutes from provided seconds
     * This method used to display smth. like "up" time, or "connected for" time
     *
     * @param seconds
     *
     * @returns {Object} {d - days, h - hours, m - minutes}
     */
    makeUpTime: function(seconds){
        var timeObj = {},
            leftForHours = seconds % 86400,
            leftForMinutes = leftForHours % 3600;

        timeObj.d = Math.floor(seconds / 86400);
        timeObj.h = Math.floor(leftForHours / 3600);
        timeObj.m = Math.floor(leftForMinutes / 60);

        return timeObj;
    },

    identifyGateway: function(fromListener){
        var self = this,
            deferred = new $.Deferred();
        
        swc.models.Rest.sendRequest({
            url: '/sysbus/DeviceInfo:get',
            data: {
                parameters: {}
            },
            fromListener: fromListener,

            success: function(response) {
                var data = response['status'];

                if (!data) {
                    return;
                }

                self.set({
                    'manufacturer': data.Manufacturer,
                    'NPVersion': data.SoftwareVersion,
                    'externalIP': data.ExternalIPAddress,
                    'upTime': self.makeUpTime(data.UpTime),
                    'serialNumber': data.SerialNumber,
                    'model': data.ModelName
                });
                
                deferred.resolve();
            },

            error: function() {
                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    rebootGateway: function(timeout){
        var self = this, deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC:reboot',
            data: {"parameters":{}},
            success: function(response) {
                deferred.resolve();
                setTimeout(function(){
                    if (swc.models.apServiceState.isEnabled()) {
                        self.whenApUp(self.doLogout);
                    } else {
                        self.whenDeviceUp(self.doLogout);
                    }
                }, timeout || self.minExpectedRebootTime);
            },
            error: function () {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    resetGateway: function(timeout){
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC:reset',
            method: 'POST',
            formatRule: 'jsonType',
            contentType: 'application/x-sah-ws-4-call+json',
            data: {"parameters":{}},

            success: function(response) {
                if(!response.status){
                    deferred.reject();
                } else {
                    deferred.resolve();
                    setTimeout(function () {
                        if (swc.models.apServiceState.isEnabled()) {
                            self.whenApUp(self.doReset);
                        } else {
                            self.whenDeviceUp(self.doReset);
                        }
                    }, timeout || self.minExpectedRebootTime);
                }
            },
            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    whenDeviceUp: function (callback) {
        var self = this;

        if (self.rebootCheckInterval) {
            return;
        }
        this.checkInProcess = false;

        this.doCheckDeviceUp(callback);
        this.rebootCheckInterval = setInterval(function () {
            self.doCheckDeviceUp(callback);
        }, 5000);
    },

    doCheckDeviceUp: function (callback) {
        var self = this;

        if (this.checkInProcess) {
            return;
        }
        this.checkInProcess = true;

        $.ajax({
            url: '/sysbus/DeviceInfo:get',
            type: 'post',
            data: {"parameters":{}},
            timeout: 5000,

            success: function (response) {
                var data = response['status'];
                if (data && data.DeviceStatus === 'Up') {
                    clearInterval(self.rebootCheckInterval);
                    self.rebootCheckInterval = null;
                    self.trigger('device:up');
                    callback();
                }
            },

            complete: function () {
                self.checkInProcess = false;
            }
        });
    },

    whenApUp: function (callback) {
        var self = this;

        if (self.rebootCheckInterval) {
            return;
        }
        this.checkInProcess = false;

        this.doCheckApUp(callback);
        this.rebootCheckInterval = setInterval(function () {
            self.doCheckApUp(callback);
        }, 5000);
    },

    /**
     * Polls NP and check if AP is "Up" and running.
     * Calls callback if AP is "Up"
     *
     * @param callback
     */
    doCheckApUp: function (callback) {
        var self = this;

        if (this.checkInProcess) {
            return;
        }

        this.checkInProcess = true;

        $.when(swc.models.apServiceLoadingState.fetch())
            .done(function(){
                if (swc.models.apServiceLoadingState.isApStarted()) {
                    clearInterval(self.rebootCheckInterval);
                    self.rebootCheckInterval = null;
                    self.trigger('ap:up'); // maybe we will need it? Copied from whenDeviceUp(), not sure if needed.
                    callback();
                }
            })
            .fail(function() {
                // TODO: What?
            })
            .always(function(){
                self.checkInProcess = false;
            });
    },

    doLogout: function() {
        swc.models.Login.processLogout({ 'redirectUrl': swc.settings.application.get("url"), action: 'reset-logout' });
    },

    doReset: function() {
        // clean local storage with saved language
        localStorage.removeItem('locale');
        swc.models.Login.processLogout({ 'redirectUrl': swc.settings.application.get("url"), action: 'reset-logout' });
    },

    upgradeGateway: function(formdata){
        var self = this, deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/webuiUpgrade',
            timeout: 360000,
            contentType: false,
            processData: false,
            data: formdata,
            headers: {
                "X-Context": $.cookie(getDeviceID() + '/context')
            },

            success: function(response) {
                var resp = $(response).text();
                if(resp.indexOf('200') >= 0){
                    deferred.resolve();
                } else {
                    deferred.reject();
                }
            },

            error: function(){
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    getLogs: function(){
        var self = this, deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',
            method: 'POST',
            contentType: 'application/x-sah-ws-4-call+json',
            data: {
                "service":"com.swisscom.stargate/ws/system.com.swisscom.stargate.system",
                "method":"getLogsURL",
                "parameters": {}
            },

            success: function(response) {
                deferred.resolve();
                var data = JSON.parse(response.status);
                document.location.href = data.data.url;
            }
        });

        return deferred.promise();
    },

    getFirmware: function(fromListener){
        var self = this, deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',
            fromListener: fromListener,
            data: {
                "service":"com.swisscom.stargate/ws/system.com.swisscom.stargate.system",
                "method":"getFirmwareVersion",
                "parameters": {}
            },

            success: function(response) {
                if(response.status[0] === "{"){
                    var data = JSON.parse(response.status);
                    self.set('APVersion', data.data.firmwareVersion);
                } else {
                    self.set('APVersion', false);
                }

                deferred.resolve();
            },

            error: function(response) {
                self.set('APVersion', false);
                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    checkPassword: function(password){
        var self = this, deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/UserManagement:authenticate',
            method: 'POST',
            contentType: 'application/x-sah-ws-1-call+json',
            data: {
                "parameters": {
                    "name": self.adminName,
                    "password": password,
                    "groups": self.groups
                }
            },

            success: function(response) {
                deferred.resolve(response.result.status);
            }
        });

        return deferred.promise();

    },

    configBackup: function(){
        var self = this, deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/backup?nocache',
            method: 'GET',

            success: function() {
                deferred.resolve();
            }
        });

        return deferred.promise();

    },

    configBackupACS: function(){
        var self = this, deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC/NetworkConfig:launchNetworkBackup',

            data: {
                "parameters": {}
            },

            success: function() {
                deferred.resolve();
            },

            error: function () {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    /**
     * Updates the session cookies from the server.
     * Is used to check if the session is still valid at the server side.
     * Calls callback function in case of success.
     * Initiates client logout in case of fail.
     *
     * @param callback
     *
     * @returns void
     */
    checkSessionState: function(callback) {
        swc.models.Rest.sendRequest({
            url: '/sysbus/Time:getTime',
            fromListener: true,
            timeout: 1000,

            data: {
                "parameters": {}
            },

            success: function(response) {
                swc.models.Login.checkAccessToDevice(response);
                if (swc.models.Login.checkUserLogin() && _.isFunction(callback)) {
                    callback();
                }
            },

            error: function(xhr) {
                swc.models.Login.checkAccessToDevice(xhr);
            }
        });
    }
});

