/**
 * Model with backup services states
 * @type {*}
 * @returns deferred
 */
swc.constructors.CloudServices = Backbone.Model.extend({

    sync: function(fromListener){
        var self = this, deferred = new $.Deferred();
        $.when(self.getServices(fromListener)).done(function() {
            deferred.resolve();
        });

        return deferred.promise();
    },

    /**
     * activates/deactivates cloud backup feature
     *
     * @param state - boolean
     */
    setBackupState: function(state){
        var deferred = new $.Deferred();
        
        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                service: "com.swisscom.stargate/ws/cloud/backup.com.swisscom.stargate.cloud.backup",
                method: "setCloudSyncState",
                parameters: {
                    isEnabled: state
                }
            },

            success: function(response) {
                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    /**
     * gets list of cloud services and state of backup feature
     * model.get('services') returns collection of backup services
     * @returns {*}
     *
     {"status": "200"
      "data": {"syncActivated":"true",
               "syncServices": [
                  {"id":"dropbox",
                   "displayName": "Drop Box",
                   "authStatusCode": "200",
                   "default":"true",
                   "totalSpace" : "200000", //in KB
                   "usedSpace" : "1500", //in KB
                   "lastSyncDate" : "10.12.2012, 15:45"
                  },
                  {"id":"googledrive",
                   "displayName": "Google Drive",
                   "authStatusCode": "503",
                   "default": "false",
                   "totalSpace" : "", //in KB
                   "usedSpace" : "", //in KB
                   "lastSyncDate" : ""
                   }
               ]
      }
     }
     * status codes:
     * 200 - authorized
     * 403 - invalid login/password - connection error
     * 503 - deauthorized
     */
    getServices: function(fromListener) {
        var self = this,
            deferred = new $.Deferred(),
            defaultData = {data: {
                syncActivated: false,
                syncServices: [
                    {
                        "id": "dropbox",
                        "displayName": "Drop Box",
                        "authStatusCode": "503",
                        "default": "true",
                        "totalSpace": "200000000",
                        "usedSpace": "1500000",
                        "usedPercents": 1,
                        "lastSyncDate": "10.12.2012, 15:45"
                    }
                ]
            }};

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                service: "com.swisscom.stargate/ws/cloud/backup.com.swisscom.stargate.cloud.backup",
                method: "getCloudSyncInfo",
                parameters: {}
            },
            fromListener: fromListener,

            success: function(response) {
                var services =  new Backbone.Collection(),
                    resp;

                try {
                    resp = JSON.parse(response.status);
                } catch (e) {
                    resp = defaultData;
                }

                self.set('services', services);

                if (!_.isUndefined(resp.data)) {
                    self.set('active', resp.data.syncActivated);

                    _.each(resp.data.syncServices, function(obj, index) {
                        var serviceModel = new Backbone.Model(obj);

                        serviceModel.set({
                            usedSpace: obj.usedSpace * 1024,
                            totalSpace : obj.totalSpace * 1024,
                            usedPercents: Math.floor(parseFloat(obj.usedSpace) / parseFloat(obj.totalSpace ? obj.totalSpace : 1) * 100)
                        });

                        self.get('services').add(serviceModel);
                    });
                } else {
                    self.set('active', false);
                }

                deferred.resolve();
            },

            error: function(e) {
                var services =  new Backbone.Collection(),
                    resp = defaultData;

                self.set('services', services);

                self.set('active', resp.data.syncActivated);

                _.each(resp.data.syncServices, function(obj, index) {
                    var serviceModel = new Backbone.Model(obj);
                    self.get('services').add(serviceModel);
                });

                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    /**
     * Get ID of service in the list which was synced most recently. Although at this moment there is only one service.
     * If collection of services is empty use ID of default service - "dropbox"
     *
     * @return {String} ID of default service, e.g. 'dropbox', 'googledrive'
     */
    getServiceID: function() {
        var serviceID = 'dropbox';
        
        // Sort services by LastSyncDate parameter Descending
        var sortedServices = _.extend({},
            new Backbone.Collection({comparator: function (service1, service2) {
                // lsd = Last Sync Date :)
                // "10.12.2012, 15:45" <-> "DD.MM.YYYY, H:m"
                var dateFormat = "DD.MM.YYYY, H:m";
                
                var lsd1 = +moment(service1.get("lastSyncDate"), dateFormat).valueOf(),
                    lsd2 = +moment(service2.get("lastSyncDate"), dateFormat).valueOf();
                
                return lsd2 - lsd1; // we have to have recently synced at the top, thus such sequence of operands 
            }}),
            this.get('services')
        );
        
        sortedServices.each(function(model) {
            serviceID = model.get('id');
        });
        
        return serviceID;
    },

    /**
     * Authorizes backup service that have 'default' value as true.
     * Sends user to authorization page
     */
    authorize: function() {
        var self = this;
        
        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                service: "com.swisscom.stargate/ws/cloud/auth.com.swisscom.stargate.cloud.authorization",
                method: "getServiceAuthUrl",
                parameters: {
                    data: JSON.stringify({
                        serviceId: self.getServiceID(),
                        domain: window.location.hostname
                    })
                }
            },

            success: function (response) {
                response = JSON.parse(response.status);
                if (response.status !== 200) {
                    var alertBox = $('<div/>', {
                        'class': 'modal cloud-error'
                    });
                    alertBox.html(response.error);
                    alertBox.modal('show');
                    alertBox.on('hidden', function() {
                        alertBox.remove();
                    });
                } else {
                    document.location.href = response.data.authUrl;
                }
            }
        });
    },

    /**
     * Check oauth credentials on access to cloud service
     */
    checkCredentials: function(){
        var self = this;
        
        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                service: "com.swisscom.stargate/ws/cloud/auth.com.swisscom.stargate.cloud.authorization",
                method: "validateDefaultServiceCredentials",
                parameters: {}
            },
            
            success: function(response) {
                response = JSON.parse(response.status);
                if (response.status === 200) {
                    window.location.reload();
                } else {
                    self.authorize();
                }
            }
        });
    },
    
    validateFolder: function (partitionUUID, folderName) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',
            data: {
                service: "com.swisscom.stargate/ws/cloud/backup.com.swisscom.stargate.cloud.backup",
                method: "validateSyncFolder",
                parameters: {
                    data: JSON.stringify({
                        partitionUUID: partitionUUID,
                        folderName: folderName
                    })
                }
            },

            success: function(response) {
                var status;

                // Check if AP has successfully started
                if (response && response.status) {
                    try {
                        status = JSON.parse(response.status);
                        if (status && status.success === true) {
                            deferred.resolve(status);
                            return;
                        }
                    } catch (e) {
                        // nothing to do
                    }
                }
                deferred.reject(status);
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    }
});
