swc.constructors.LoginModel = Backbone.Model.extend({

    adminName: "admin",

    /**
     * Shows current login state
     */
    isLoggedIn: false,

    /**
     * Check if user is logged in:
     *
     * @description user login is stored in two cookies {contextID} and {.../sessid}
     *
     * @return {Boolean}
     *
     */
    checkUserLogin: function() {
        var deviceID = getDeviceID();

        if (!deviceID || !$.cookie(deviceID + '/sessid')) {
            return false;
        } else {
            return true;
        }
    },

    /**
     * Check if the user is logged in via checking response message
     *
     * @description:
     *
     * if the response from device contains text "Permission denied" then
     * user is not logged in and the logout process should be started
     *
     * @param xhr {xmlHttpRequest} Response from device
     */
    checkAccessToDevice: function(xhr) {
        var errorText = 'Permission denied';

        if (_.isUndefined(xhr)) {
            swc.models.Login.processLogout({ action: 'session-logout' });
            return;
        }

        if (!_.isUndefined(xhr.responseText) && _.isString(xhr.responseText) && xhr.responseText.indexOf(errorText) !== -1) {
            swc.models.Login.processLogout({ action: 'session-logout' });
            return;
        }

        if (!_.isUndefined(xhr.errors) && !_.isEmpty(xhr.errors) && !_.isEmpty(_.where(xhr.errors, { description: errorText }))) {
            swc.models.Login.processLogout({ action: 'session-logout' });
        }
    },

    /**
     * Check if user is super admin
     * @returns {boolean}
     */
    checkUserSuperAdmin: function() {
        var isSuperAdmin = false;

        if (!_.isNull(localStorage.getItem('userSuperAdmin')) && localStorage.getItem('userSuperAdmin')) {
            isSuperAdmin = true;
        }

        return isSuperAdmin;
    },

    /**
     * Change password. It is needed in System and Login
     * @param password
     * @returns {*}
     */

    changePassword: function(password){
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/UserManagement:changePassword',
            method: 'POST',
            contentType: 'application/x-sah-ws-1-call+json',
            data: {
                parameters: {
                    name: self.adminName,
                    password: password
                }
            },

            success: function(response) {
                deferred.resolve(response.result.status);
            }
        });

        return deferred.promise();

    },

    /**
     * Remove application cookies
     */
    removeCookies: function(){
        var deviceID = getDeviceID();

        // Expire all SAH cookies
        $.cookie(deviceID + '/context', '', { expires: -1 });
        $.cookie(deviceID + '/sessid', '', { expires: -1 });

        // Expire all SWC cookies
        $.cookie('swc/deviceID', '', { expires: -1 });
    },

    /**
     * Sign-out user and release context on NP side
     * @param options {Object|null} Possible values:
     *                              - action : defines how signing out was performed - by session timeout or manually
     *                              - redirectUrl : if we have to redirect user to some url after have signed them out
     * @returns {*|Promise.promise}
     */
    processLogout: function(options) {
        var self = this;

        // preventing multiple logout processing
        if (this.isLoggedIn) {
            // Only do local sign-out post-processing if successfully signed-off on NP side
            // Release current context on NP side
            $.when(swc.models.Rest.sendRequest({
                    url: '/ws',
                    method: 'POST',
                    headers: {
                        "Authorization": "X-Sah-Logout " + $.cookie(getDeviceID() + '/context'),
                        'X-Context': $.cookie(getDeviceID() + '/context')
                    },
                    data: {
                        service: "sah.Device.Information",
                        method: "releaseContext",
                        parameters: {
                            applicationName: "webui"
                        }
                    }
                }))
                .always(function () {
                    self.isLoggedIn = false;
                    self.setLogoutState(options);
                });
        } else {
            this.setLogoutState(options);
        }
    },

    /**
     * Set application state after logout
     *
     * @description:
     *
     * This method perform following actions:
     * 1.Sets application env variables to appropriate state(action-logout, session-logout,
     *   reset-logout) in the localStorage
     * 2.Removes session cookies
     * 3.Triggers events: beforeLogout, afterLogout
     * 4.Redirects on the login/remoteLogin page
     *
     * @localParam router {swc.constructors.Router | Backbone.history} When user is not logged in
     *             and tries to navigate on the some inside page, e.g. http://localhost:3000/#storage/settings,
     *             the swc.constructors.Router is not exist and it's not possible to use its method
     *             navigate(). In this case should be used the method navigate() of the Backbone.history object
     *
     * @param options {Object} addition logout options
     * Param <options> can accept two keys:
     *   - action - defines the source which has initialized logout action.
     *              Can accept the following values:
     *              * user-logout - action initiated by user themself by clicking on "Logout" button
     *              * session-logout - session expired on server
     *              * reset-logout - when i-Box redirected after 'Upgrade'/'Restore Config'/'FactoryReset' operations
     *
     *   - redirectUrl - if after logout user should be redirected to some page
     *
     * @return void
     */
    setLogoutState: function(options) {
        var self = this,
            currentPage = Backbone.history.fragment,
            userIsSuperAdmin = this.checkUserSuperAdmin(),
            router = _.isUndefined(swc.router) ? Backbone.history : swc.router,
            doRouterTrigger = _.isUndefined(swc.router) ? true : false;

        // Local logout procedure starts here
        self.removeCookies();

        // WARNING!!!
        // DO NOT REMOVE THIS BLOCK OF CODE!
        // Sometimes we just must redirect user's browser to some url in order to refresh index.html page
        if (options && options.redirectUrl) {
            localStorage.setItem('previous-page', 'overview');
            // Set flag back that this user was logged in remotely
            var loginPageUrl = "/";
            if (userIsSuperAdmin) {
                localStorage.setItem('remoteLogin', true);
                loginPageUrl = loginPageUrl + "#remote-login";
            }
            document.location.href = options.redirectUrl + loginPageUrl;
            return;
        }

        // Let listeners know that we are about to sign out
        self.trigger('beforeLogout');

        // Hide all modals and popovers
        SWCElements.modalWindow.hide();
        SWCElements.popovers.closeAll();

        // We have to navigate user to correct page after logout / session logout / not logged in access
        if (options && options.action) {
            switch (options.action) {
                case 'user-logout':
                    localStorage.setItem('action-logout', 'true');
                    break;
                case 'session-logout':
                    localStorage.setItem('session-logout', 'true');
                    break;
                case 'reset-logout':
                    localStorage.setItem("reset-logout", 'true');
                    break;
            }

            localStorage.setItem('previous-page', 'overview');
        } else {
            if (!localStorage.getItem('previous-page') || localStorage.getItem('previous-page') === "overview") {
                localStorage.setItem('previous-page', Backbone.history.fragment);
            }
        }

        // Set flag back that this user was logged in remotely
        if (userIsSuperAdmin) {
            localStorage.setItem('remoteLogin', true);
        }

        // Update current application page:
        if (currentPage !== 'login' && currentPage !== 'remote-login') {
            if (!_.isNull(localStorage.getItem('remoteLogin'))) {
                if (currentPage === 'remote-login') {
                    swc.views.Login.renderComplete();
                } else {
                    router.navigate('remote-login', { trigger: doRouterTrigger, skipUnsavedChanges: true });
                }
            } else {
                if (currentPage === 'login') {
                    swc.views.Login.renderComplete();
                } else {
                    router.navigate('login', { trigger: doRouterTrigger, skipUnsavedChanges: true });
                }
            }
        } else {
            swc.views.Login.renderComplete();
        }

        self.trigger('afterLogout');
    },
    

    /**
     * Process user login:
     *
     * @param login {String}
     * @param password {String}
     *
     */
    processLogin: function(login, password) {
        var self = this, deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/ws',
            method: 'POST',
            contentType: 'application/x-sah-ws-4-call+json',

            headers: {
                Authorization: 'X-Sah-Login'
            },

            data: {
                service: "sah.Device.Information",
                method: "createContext",
                parameters: {
                    applicationName: "webui",
                    username: login,
                    password: password
                }
            },

            success: function(response) {
                if (response.status === 0) {

                    self.isLoggedIn = true;

                    var deviceID = getDeviceID();

                    // Set Device ID Cookie
                    // In order to define not deleted cookies from previous logins by old device ID
                    $.cookie('swc/deviceID', deviceID);

                    // Set Context Cookie
                    $.cookie(deviceID + '/context', response.data.contextID);

                    //self.loginStatusListener();

                    // Let application know if user is super admin
                    if (login === "superadmin") {
                        localStorage.setItem('userSuperAdmin', true);
                    } else {
                        localStorage.removeItem('userSuperAdmin');
                    }

                    deferred.resolve({status: true, response: response});
                } else {
                    deferred.resolve({status: false, response: response});
                }
            },

            error: function(xhr, ajaxOptions, thrownError) {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    /**
     * Listener for user login status:
     */
    loginStatusListener: function() {
        var self = this,
            interval = 10,  // seconds
            onRequestComplete = function() {
                var userLogin = self.checkUserLogin();

                if (!userLogin) {
                    self.processLogout({ action: 'session-logout' });
                } else {
                    setTimeout(function() {
                        self.loginStatusListener();
                    }, interval * 1000);
                }
            };

        swc.models.Rest.sendRequest({
            url: '/sysbus/Time:getTime',
            fromListener: true,
            timeout: 2000,
            data: {
                "parameters": {}
            },
            success: onRequestComplete,
            error: onRequestComplete
        });
    },

    /**
     * Get device recovery status - true or false
     * @returns {*}
     */
    getRecoveryStatus: function(){
        var self = this, deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/PasswordRecovery:getStatus',
            method: 'POST',
            contentType: 'application/x-sah-ws-1-call+json',
            data: {
                parameters: {}
            },

            /**
             * NOTE:
             * NP returns response as such: {result: {status: [1, "Enable"]}},
             * where first element of array is true|false indicating either feature is enabled at all.
             * Second element indicates what is the current status of recovery:
             * "Enable" - we can start a new process; "Processing" - means there is a process in-progress.
             */
            success: function(response) {
                if (!_.isUndefined(response.result) && !_.isUndefined(response.result.status)) {
                    self.set('recoveryEnable', response.result.status[0]);
                    self.set('recoveryStatus', response.result.status[1]);
                } else {
                    self.set('recoveryEnable', false);
                    self.set('recoveryStatus', null);
                }
                deferred.resolve();
            },

            error: function(response) {
                self.set('recoveryEnable', false);
                self.set('recoveryStatus', null);
                deferred.resolve();
            }
        });

        return deferred.promise();
    },

    /**
     * Set device recovery status
     * 
     * @param {Boolean} isEnable
     * 
     * @returns {*}
     */
    setRecoveryStatus: function(isEnable){
        var self = this,
            deferred = new $.Deferred();

        swc.models.PasswordRecovery.set("recoveryEnable", isEnable);
        $.when(swc.models.PasswordRecovery.sync("update"))
            .done(function () {
                deferred.resolve();
            })
            .fail(function () {
                deferred.reject();
            });
        
        return deferred.promise();
    }
    
});
