/**
 * Retrieve the collection of files on the device with concrete ID
 *
 * Usage:
 * Device ID should be passed in on the constructor as option like
 * 
 * fileManager = new swc.constructors.FileManager([], { deviceID: deviceModel.get('id') });
 * fileManager.fetch();
 * 
 * The result collection has the models with the following attributes: 
 * {
 *    name : "picture.jpg",     // File or folder name
 *    type : "file",            // Either "file" or "folder"
 *    size : 183984,            // Size in bytes - valid only for files
 *    isSync : true,            // Is synchronized - either true or false.
 *    virtual: true,            // True means that file is missing on NAS but is in cloud - actual for google drive
 *    syncState : "COMPLETE",   // One of: "COMPLETE", "ERROR", "DELETE_IN_PROGRESS", "UPLOAD_IN_PROGRESS", null
 *    syncError : 7             // Error ID, that can be translated to the meaningful message (either number, or null).
 * }
 */
swc.constructors.FileManager = swc.BaseCollection.extend({

    url: {
        read: '/ws'
    },

    ajaxParameters: {
        service: "com.swisscom.stargate/ws/nas/filemanager.com.swisscom.stargate.nas.filemanager",
        method: "listFiles",
        parameters: {
            id: 0,
            path: "/"
        }
    },

    initialize: function() {
        var args = Array.prototype.slice.call(arguments),
            errorMessage = 'For retrieving list of files the device ID should be set';

        if (args.length === 0 || _.isUndefined(args[1])) {
            throw new Error(errorMessage);
        }

        if (_.isUndefined(args[1].deviceID)) {
            throw new Error(errorMessage);
        }

        this.ajaxParameters.parameters.id = args[1].deviceID;
        swc.BaseCollection.prototype.initialize.apply(this, arguments);
    },

    /**
     * Comparator sorts elements, so folders will go first
     *
     * @return {String}
     */
    comparator: function(model) {
        var ftype = model.get('type');

        ftype = ftype.toLowerCase();
        ftype = ftype.split('');
        ftype = _.map(ftype, function(letter) {
            return String.fromCharCode(-(letter.charCodeAt(0)));
        });

        return ftype;
    },

    apiToJSON: function(response) {
        var self = this,
            parsedResponse = {},
            parsedData = [];

        try {
            parsedResponse = JSON.parse(response.status);

            if (parsedResponse.status === 200) {
                _.each(parsedResponse.data, function(file) {
                    parsedData.push({
                        isSync: file.syncEnabled,
                        id: file.name,
                        name: file.name,
                        size: file.size * 1024,
                        syncState: self.getSyncState(file.syncState),
                        type: file.type,
                        virtual: file.virtual
                    });
                });
            }
        } catch (e) {}

        return parsedData;
    },

    /**
     * Set cloud backup state
     *
     * @param responseSyncState
     * @returns {*}
     */
    getSyncState: function(responseSyncState) {
        var map = {
            'IN_PROGRESS': 'progress',
            'UPLOAD_IN_PROGRESS': 'progress',
            'DELETE_IN_PROGRESS': 'progress',
            'COMPLETE': 'complete',
            'ERROR': 'error'
        };

        if (_.isUndefined(map[responseSyncState])) {
            return 'not_sync';
        }

        return map[responseSyncState];
    }

});