/**
 * WiFi Data Storage
 *
 * @type {Interface | Collection}
 *
 * @description Stores information about WiFi 2.4GHz, WiFi 5.0GHz, WiFi Guest, WiFi Scheduler
 */
swc.constructors.Wireless = Backbone.Collection.extend({

    validation: {

        ssid: function(data) {
            var ssid,
                validationStatus = true,
                validationMessages = [];

            // Get neccessary params from array of page data:
            $.each(data, function(key, value) {
                if (value.parameterName === "ssid") {
                    ssid = value.parameterValue;
                }
            });

            if (!ssid.length) {
                validationStatus = false;
                validationMessages.push('name must not be empty');
            }

            if (ssid.length < 3 || ssid.length > 32) {
                validationStatus = false;
                validationMessages.push('name must be from 3 to 32 symbols');
            }

            if (!/^(?=.*\S)[-\w\s]+$/.test(ssid)) {
                validationStatus = false;
                validationMessages.push('name must not contain spaces');
            }

            return {
                status: validationStatus,
                messages: validationMessages
            };
        },

        password: function(data) {
            var password = '',
                encryption = '',
                userIsSuperAdmin = swc.models.Login.checkUserSuperAdmin(),
                validationStatus = true,
                validationMessages = [];

            $.each(data, function(key, value) {
                if (value.parameterName === "password") {
                    password = value.parameterValue;
                }

                if (value.parameterName === "encryption") {
                    encryption = value.parameterValue;
                }
            });

            // Skip check for superadmin and if encryption is none
            if (encryption === "None" || userIsSuperAdmin) {
                return {
                    status: validationStatus,
                    messages: validationMessages
                };
            }

            if ( !/^([A-Za-z0-9]|\-|\s|\_)*$/.test(password) ) {
                validationStatus = false;
                validationMessages.push('password contain not allowed characters');
            }

            if ( !/^(\w|\W){8,64}$/.test(password) ) {
                validationStatus = false;
                validationMessages.push('Password WPA length error');
            }

            return {
                status: validationStatus,
                messages: validationMessages
            };
        }

    },

    sync: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        // Delete deprecated models from collection
        this.reset();

        // Load new models to the collection
        $.when(
                self.getSystemTime(fromListener),
                self.getGlobalStatus(fromListener),
                self.getGuestStatus(fromListener),
                self.getGuestActivationTimer(fromListener),
                self.loadSettings(fromListener),
                self.loadSettingsGuest(fromListener)
            ).done(function() {
                self.trigger('change');
                deferred.resolve();
            });

        return deferred.promise();
    },

    getSystemTime: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/Time:getTime',
            fromListener: fromListener,

            data: {
                "parameters": {}
            },

            success: function(response) {
                var model = new swc.constructors.SimpleModel(),
                    time,
                    mode;

                model.set('accessPoint', 'systemTime');
                model.set('time', response.data.time.split('GMT')[0]); // DATE is coming from server with TimeZone.

                self.add(model);

                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    getGlobalStatus: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC/Wifi:get',
            fromListener: fromListener,

            data: {
                "parameters": {}
            },

            success: function(response) {
                var model = new swc.constructors.SimpleModel(),
                    status,
                    mode;

                model.set('accessPoint', 'status');

                // Define responce existence
                if (!response.data) {
                    return deferred.resolve();
                }

                // Define configurationg mode existence
                mode = response.data.ConfigurationMode;

                // Global status - wifi is on:
                if (response.data.Enable && response.data.Status) {
                    status = "on";
                }

                // Global status - wifi is off:
                if (!response.data.Enable && !response.data.Status) {
                    status = "off";
                }

                // Global status - wifi is off by scheduler:
                if (response.data.Enable && !response.data.Status) {
                    status = "off-sheduler";
                }

                // Global status - wifi is on till reboot of device:
                if (!response.data.Enable && response.data.Status) {
                    status = "on";
                }

                model.set('status', status);
                model.set('configurationMode', mode);

                self.add(model);

                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    getGuestStatus: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC/Guest:get',
            fromListener: fromListener,

            data: {
                "parameters": {}
            },

            success: function(response) {
                var model = new swc.constructors.SimpleModel(),
                    status = false;

                if (response.data) {
                    status = response.data.Enable;
                }

                model.set('accessPoint', 'guestStatus');
                model.set('status', status);

                self.add(model);

                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    getGuestActivationTimer: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC/WlanTimer:getActivationTimer',
            fromListener: fromListener,

            data: {
                "parameters": {
                    "InterfaceName": "guest"
                }
            },

            success: function(response) {
                var model = new swc.constructors.SimpleModel();

                model.set('accessPoint', 'guestTimer');
                model.set('timeout', response.status);

                // Add model to the collection
                self.add(model);

                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    loadSettings: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NeMo/Intf/lan:getMIBs',
            fromListener: fromListener,

            data: {
                "parameters": {
                    "mibs": "wlanradio || wlanvap || penable",
                    "flag": "",
                    "traverse": "down"
                }
            },

            success: function(response) {
                if (response.status) {
                    if (response.status.wlanradio && response.status.wlanvap && response.status.penable) {

                        // Create models from response
                        self.createModels(response);

                        deferred.resolve();
                    } else {
                        deferred.reject();
                    }
                } else {
                    deferred.reject();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    loadSettingsGuest: function(fromListener) {
        var self = this,
            deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NeMo/Intf/guest:getMIBs',
            fromListener: fromListener,

            data: {
                "parameters": {
                    "mibs": "wlanradio || wlanvap || penable",
                    "flag": "",
                    "traverse": "down"
                }
            },

            success: function(response) {
                if (response.status) {
                    if (response.status.wlanradio && response.status.wlanvap && response.status.penable) {

                        // Create models from response
                        self.createModels(response, true);

                        deferred.resolve();
                    } else {
                        deferred.reject();
                    }
                } else {
                    deferred.reject();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    setGlobalStatus: function(status) {
        var deferred = new $.Deferred(),
            configurationMode = swc.models.Wireless.getParameter("configurationMode", "status");

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC/Wifi:set',
            data: {
                "parameters": {
                    "Enable": status,
                    "ConfigurationMode": configurationMode,
                    "Status": status
                }
            },

            success: function(response) {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    setGuestStatus: function(status) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC/Guest:set',
            data: {
                "parameters": {
                    "Enable": status
                }
            },

            success: function(response) {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    setGlobalConfigurationMode: function(configurationMode) {
        var deferred = new $.Deferred(),
            status = swc.models.Wireless.getParameter("status", "status");

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC/Wifi:set',
            data: {
                "parameters": {
                    "Enable": status === 'on',
                    "ConfigurationMode": configurationMode,
                    "Status": status === 'on'
                }
            },

            success: function(response) {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    setGuestActivationTimer: function(timeout) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC/WlanTimer:setActivationTimer',
            data: {
                "parameters": {
                    "Timeout": timeout,
                    "InterfaceName": "guest"
                }
            },

            success: function(response) {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    setVAPStatus: function(accessPoint, data) {
        var deferred = new $.Deferred(),
            status = '',
            wlDriver = this.getParameter("driver", accessPoint),
            saveData = {
                "parameters": {
                    "mibs": {
                        "wlanradio":{}
                    }
                }
            };

        // Get neccessary params from array of page data:
        $.each(data, function(key, value) {
            if (value.parameterName === "status") {
                status = value.parameterValue;
            }
        });

        // SSID Availability selection
        saveData.parameters.mibs.wlanradio[wlDriver] = {
            "PersistentEnable": status
        };

        swc.models.Rest.sendRequest({
            url: '/sysbus/NeMo/Intf/lan:setWLANConfig',
            data: saveData,

            success: function(response) {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    startWPSPairing: function(timeout) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC/Wifi:startPairing',
            data: {
                "parameters": {}
            },

            success: function(response) {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },
    
    stopWPSPairing: function(timeout) {
        var deferred = new $.Deferred();

        swc.models.Rest.sendRequest({
            url: '/sysbus/NMC/Wifi:stopPairing',
            data: {
                "parameters": {}
            },

            success: function(response) {
                deferred.resolve();
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    saveSettings: function(accessPoint, data) {
        var ssid = '', password = '', encryption = '',
            visibility = '', channel = '',
            operating = this.getParameter('operating', accessPoint),
            wlID = this.getParameter("accessPointID", accessPoint),
            wlDriver = this.getParameter("driver", accessPoint),
            saveData = {
                "parameters": {
                    "mibs": {
                        "wlanvap": {},
                        "wlanradio": {},
                        "penable": {}
                    }
                }
            },
            deferred = new $.Deferred();

        // Get neccessary params from array of page data:
        $.each(data, function(key, value) {
            if (value.parameterName === "ssid") {
                ssid = value.parameterValue;
            }

            if (value.parameterName === "password") {
                password = value.parameterValue;
            }

            if (value.parameterName === "encryption") {
                encryption = value.parameterValue;
            }

            if (value.parameterName === "channel") {
                channel = value.parameterValue;
            }

            if (value.parameterName === "visibility") {
                visibility = value.parameterValue;
            }
        });

        // Main data selection
        saveData.parameters.mibs.wlanvap[wlID] = {
            "SSID": ssid,
            "SSIDAdvertisementEnabled": visibility === "visible" ? true : false,
            "Security": {
                "KeyPassPhrase": password,
                "WEPKey": this.getParameter("wepkey", accessPoint),
                "ModeEnabled": encryption
            }
        };

        // Hack for SAH not saving password if security set to No Password
        if (encryption === "None") {
            saveData.parameters.mibs.wlanvap[wlID].Security.WEPKey = this.getParameter('wepkey', accessPoint);
            saveData.parameters.mibs.wlanvap[wlID].Security.KeyPassPhrase = this.getParameter('password', accessPoint);
        }

        // Channel and Operating mode selections options
        if (channel === "auto") {
            saveData.parameters.mibs.wlanradio[wlDriver] = {
                "AutoChannelEnable": true,
                "OperatingStandards": operating
            };
        } else {
            saveData.parameters.mibs.wlanradio[wlDriver] = {
                "Channel": channel,
                "AutoChannelEnable": false,
                "OperatingStandards": operating
            };
        }

        // Send saving request:
        swc.models.Rest.sendRequest({
            url: '/sysbus/NeMo/Intf/' + wlID + ':setWLANConfig',
            data: saveData,

            success: function(response) {
                if (response.status === null) {
                    deferred.resolve();
                } else {
                    deferred.reject();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    saveSettingsCombined: function(accessPoint, data) {
        var ssid = '', password = '', encryption = '', visibility = '',
            wlID = this.getParameter("accessPointID", accessPoint),
            saveData = {
                "parameters": {
                    "mibs": {
                        "wlanvap": {}
                    }
                }
            },
            deferred = new $.Deferred();

        // Get neccessary params from array of page data:
        $.each(data, function(key, value) {
            if (value.parameterName === "ssid") {
                ssid = value.parameterValue;
            }

            if (value.parameterName === "password") {
                password = value.parameterValue;
            }

            if (value.parameterName === "encryption") {
                encryption = value.parameterValue;
            }

            if (value.parameterName === "visibility") {
                visibility = value.parameterValue;
            }
        });

        // Main data selection
        saveData.parameters.mibs.wlanvap[wlID] = {
            "SSID": ssid,
            "SSIDAdvertisementEnabled": visibility === "visible" ? true : false,
            "Security": {
                "KeyPassPhrase": password,
                "WEPKey": this.getParameter("wepkey", accessPoint),
                "ModeEnabled": encryption
            }
        };

        // Hack for SAH not saving password if security set to No Password
        if (encryption === "None") {
            saveData.parameters.mibs.wlanvap[wlID].Security.WEPKey = this.getParameter('wepkey', accessPoint);
            saveData.parameters.mibs.wlanvap[wlID].Security.KeyPassPhrase = this.getParameter('password', accessPoint);
        }

        // Send saving request:
        swc.models.Rest.sendRequest({
            url: '/sysbus/NeMo/Intf/' + wlID + ':setWLANConfig',
            data: saveData,

            success: function(response) {
                if (response.status === null) {
                    deferred.resolve();
                } else {
                    deferred.reject();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    switchToCombinedMode: function(accessPoint) {
        var ssid = this.getParameter('name', '5GHz'),
            password = this.getParameter('password', '5GHz'),
            encryption = this.getParameter('encryption', '5GHz'),
            visibility = this.getParameter('visibility', '5GHz'),
            wlID = this.getParameter("accessPointID", accessPoint),
            saveData = {
                "parameters": {
                    "mibs": {
                        "wlanvap": {}
                    }
                }
            },
            deferred = new $.Deferred();

        // Main data selection
        saveData.parameters.mibs.wlanvap[wlID] = {
            "SSID": ssid,
            "SSIDAdvertisementEnabled": visibility,
            "Security": {
                "KeyPassPhrase": password,
                "WEPKey": this.getParameter("wepkey", accessPoint),
                "ModeEnabled": encryption
            }
        };

        // Send saving request:
        swc.models.Rest.sendRequest({
            url: '/sysbus/NeMo/Intf/' + wlID + ':setWLANConfig',
            data: saveData,

            success: function(response) {
                if (response.status === null) {
                    deferred.resolve();
                } else {
                    deferred.reject();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    saveSettingsGuest: function(accessPoint, data) {
        var ssid = '',
            password = '',
            encryption = '',
            wlID = this.getParameter("accessPointID", "Guest-" + accessPoint),
            saveData = {
                "parameters": {
                    "mibs": {
                        "wlanvap": {}
                    }
                }
            },
            deferred = new $.Deferred();

        // Get neccessary params from array of page data:
        $.each(data, function(key, value) {
            if (value.parameterName === "ssid") {
                ssid = value.parameterValue;
            }

            if (value.parameterName === "password") {
                password = value.parameterValue;
            }

            if (value.parameterName === "encryption") {
                encryption = value.parameterValue;
            }
        });

        // Prepare data to be saved:
        saveData.parameters.mibs.wlanvap[wlID] = {
            "SSID": ssid,
            "Security": {
                "KeyPassPhrase": password,
                "WEPKey": this.getParameter("wepkey", "Guest-" + accessPoint),
                "ModeEnabled": encryption
            }
        };

        // Hack for SAH not saving password if security set to No Password
        if (encryption === "None") {
            saveData.parameters.mibs.wlanvap[wlID].Security.WEPKey = this.getParameter('wepkey', "Guest-" + accessPoint);
            saveData.parameters.mibs.wlanvap[wlID].Security.KeyPassPhrase = this.getParameter('password', "Guest-" + accessPoint);
        }

        // Send saving request:
        swc.models.Rest.sendRequest({
            url: '/sysbus/NeMo/Intf/' + wlID + ':setWLANConfig',
            data: saveData,

            success: function(response) {
                if (response.status === null) {
                    deferred.resolve();
                } else {
                    deferred.reject();
                }
            },

            error: function() {
                deferred.reject();
            }
        });

        return deferred.promise();
    },

    createModels: function(response, isGuest) {

        var self = this,
            accessPoints = response.status.wlanradio,
            accessPointData = response.status.wlanvap,
            accessPEnableData = response.status.penable;


        // Create models:
        $.each(accessPoints, function(accessPoint, data) {

            var model = new swc.constructors.SimpleModel(),
                accessPointName = data.OperatingFrequencyBand,
                accessPointID,
                baseData;

            // Check if guest Access Point
            if (isGuest) {
                baseData = accessPointName === '2.4GHz' ? accessPointData.wlguest2 : accessPointData.wlguest5;
                accessPointID =  accessPointName === '2.4GHz' ? 'wlguest2' : 'wlguest5';
                accessPointName = 'Guest-' + data.OperatingFrequencyBand;
            } else {
                baseData = accessPointName === '2.4GHz' ? accessPointData.wl0 : accessPointData.wl1;
                accessPointID =  accessPointName === '2.4GHz' ? 'wl0' : 'wl1';
            }

            // Set main options
            model.set('id', accessPointName);
            model.set('accessPoint', accessPointName);
            model.set('driver', accessPoint);
            model.set('accessPointID', accessPointID);

            // Main data:
            model.set('name', baseData.SSID);
            // FIX SA-1319: Incorrect data used to get Activation status of radio access points 
            model.set('status', accessPEnableData[accessPoint].PersistentEnable);
            model.set('visibility', baseData.SSIDAdvertisementEnabled);

            // Security data:
            model.set('password', baseData.Security.KeyPassPhrase);
            model.set('wepkey', baseData.Security.WEPKey);
            model.set('encryption', baseData.Security.ModeEnabled);
            model.set('encryptionsSupported', baseData.Security.ModesSupported.split(','));
            
            // Set driver data:
            model.set('channel', data.Channel);
            model.set('channelAutoEnable', data.AutoChannelEnable);
            model.set('channelsSupported', data.PossibleChannels.split(','));
            model.set('operating', data.OperatingStandards);
            model.set('operatingsSupported', data.SupportedStandards.split(','));
            model.set('OperatingStandards', data.OperatingStandards);
            model.set('OperatingChannelBandwidth', data.OperatingChannelBandwidth);

            // Add model to the collection;
            self.add(model);
        });
    },

    getParameter: function(parameter, accessPoint) {
        var search = this.where({ accessPoint: accessPoint });

        if (search.length) {
            return search[0].get(parameter);
        } else {
            return '';
        }
    },

    isAccessPointsEquals: function(interfaceOne, interfaceTwo, keys) {
        var self = this,
            isEquals = true;

        if (keys.length) {
            $.each(keys, function(key, value) {
                if (self.getParameter(value, interfaceOne) !== self.getParameter(value, interfaceTwo)) {
                    isEquals = false;
                }
            });
        }

        return isEquals;
    },

    formatModeOptions: function() {
        return {
            "combined": {
                name: getTranslationStrings("Combined"),
                value: "combined"
            },
            "separate": {
                name: getTranslationStrings("Separate"),
                value: "separate"
            }
        };
    },

    formatEncryptionOptions: function(accessPoint) {
        var options = [],
            enabled = {
                "WPA2-Personal": getTranslationStrings("WPA2"),
                "WPA-WPA2-Personal": getTranslationStrings("WPA/WPA2"),
                "None": getTranslationStrings("No encryption")
            };

        $.each(enabled, function(encryption, value) {
            options.push({
                name:  value,
                value: encryption
            });
        });

        return options;
    },

    formatOperatingOptions: function(accessPoint) {
        var accessPointOptions = this.getParameter('operatingsSupported', accessPoint),
            options = {},
            enabled = {
                "n": getTranslationStrings("802.11 n"),
                "ac": getTranslationStrings("802.11 ac"),
                "an": getTranslationStrings("Auto (802.11 a/n/ac)"),
                "bgn": getTranslationStrings("Auto (802.11 b/g/n)")
            };

        // Walk throuh all supported and enabled options
        $.each(accessPointOptions, function(key, value) {
            if (enabled[value]) {
                options[value] = {
                    name:  enabled[value],
                    value: value
                };
            }
        });

        return options;
    },

    formatChannelsOptions: function(accessPoint) {
        var search = this.where({ accessPoint: accessPoint }),
            channelsSupported,
            model,
            options = {
                "auto": {
                    name:  getTranslationStrings("Auto channel"),
                    value: "auto"
                }
            };

        if (search.length) {
            model = search[0];
            channelsSupported = model.get("channelsSupported");

            if (channelsSupported.length) {
                $.each(channelsSupported, function(key, channel) {
                    options[channel] = {
                        name:  channel,
                        value: channel
                    };
                });
            } else {
                return options;
            }
        } else {
            return options;
        }

        return options;
    },

    formatDeactivationOptions: function(accessPoint) {
        return [
            { name: getTranslationStrings("No auto disabling"), value: "0", isDefault: true },
            { name: getTranslationStrings("6 hours"), value: "21600" },
            { name: getTranslationStrings("12 hours"), value: "43200" },
            { name: getTranslationStrings("24 hours"), value: "86400" }
        ];
    }
});
